#!/usr/bin/perl -w 
#
# Script to parse the VNIC configuration file (/etc/infiniband/qlgc_vnic.cfg
# or /etc/sysconfig/ics_inic.cfg) and create VNIC interfaces by writing
# the required string to the /sys/class/infiniband_qlgc_vnic/ files of the QLogic VNIC
# kernel driver
#
# Copyright (c) 2007 QLogic, Inc.  All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#

use strict;
use Fcntl;

our $createfound = 0;
our (@lines, $primary_path, $secondary_path);
our ($line, $pstring , $sstring , $pname);
our ($path, $primary, $secondary, $create);
our ($instance, $port, $hca, $rx_csum, $heartbeat, $ioc_guid, $ib_multicast,
	$name, $name_token, $error_flag, $discover_tool, $log, $DGID);
our ($primary_file, $secondary_file, $hex, $prim_term, %parray,
	$sec_term, $c_term, $block_start);
our ($block_found, $ioc_string, $line_count, $err_flag,
	$prim_start, $sec_start, $create_start);
our ($verbose, $PARSER_VERSION, $tempfile, $ioc_guid_found, $ioc_string_found,
	$prim, $sec, $file, $file_ics_inic, $file_old, $crt, $val_hca, $val_port, $val_ioc );
our ($prim_seen, $sec_seen,$prim_start_seen,$sec_start_seen, $found,$dgid_found_port,
	$ibvexdm_output_in_file);
our ($MAX_INIT_TIME, $MAX_WAIT_FOR_ENTRY);
our (@pid, $pid_index, $command, $filename, $index);
our ($dgid_found, $PKEY, $port_guid, $val_port_guid, $port_guid_found, $port_found, $row);
our ($hca_information_command, @hca_info_array, @hca_info, $val_hca_name, $local_hca_info_flag);
# prim_term =1 means term found, 0 means terminating brace not found.
# prim_seen = 1 means PRIMARY keyword followed by opening brace is seen
$PARSER_VERSION = "1.0.0.1";
$prim_seen = 0;	
$sec_seen = 0;
$sec_start_seen = 0;
$prim_start_seen = 0;
$verbose = 0;
$crt = 0;
$ioc_string_found = 0;
$ioc_guid_found = 0;
$block_found = 0;
%parray = ("name=", "", "ioc_guid=", 
	"", "dgid=", "", "pkey=", "", "instance=", "", "rx_csum=", "", "heartbeat=",
	 "","ioc_string=","","ib_multicast=","");
$pstring = "";
$sstring = "";
$primary = "PRIMARY";
$secondary = "SECONDARY";
$create = "CREATE";
$instance = "INSTANCE";
$port = "PORT";	
$hca ="HCA";
$rx_csum ="RX_CSUM";
$heartbeat ="HEARTBEAT";
$ib_multicast="IB_MULTICAST";
$ioc_guid = "IOCGUID";
$ioc_string = "IOCSTRING";
$DGID="DGID";
$port_guid="PORTGUID";
$path = "/sys/class/infiniband_qlgc_vnic/";
$name = "NAME";
$name_token = "name";
$error_flag = 0;
$hca_information_command = "ib_qlgc_vnic_query -L";
$tempfile =`mktemp /tmp/qlvnic.XXXXXXXXXX`;
$discover_tool = "ib_qlgc_vnic_query";
$hex = "0x";
$primary_file = "/create_primary";
$secondary_file = "/create_secondary";
$prim_term = 1;
$sec_term = 1;
$c_term = 1;
$err_flag = 0;
$line_count = 0;
$prim_start = 0;
$create_start = 0;
$sec_start = 0;
$block_start = 0;
$prim = 1;
$sec = 2;
$dgid_found_port = 0;
$port_guid_found = 0;
$port_found = 0;
$val_port_guid = "";
$row = 0;
$val_hca = 0;
$val_hca_name = "";
$val_port = 1;
$val_ioc = "";
$file_ics_inic = "/etc/sysconfig/ics_inic.cfg";
$file_old = "/etc/infiniband/qlogic_vnic.cfg";
$file = "/etc/infiniband/qlgc_vnic.cfg";
$log = "/var/log/messages";
our $path_desc = "primary";
our $buff = "";
$ibvexdm_output_in_file=0;
$MAX_INIT_TIME = 45;
$MAX_WAIT_FOR_ENTRY = 10;
$pid_index = 0;
$dgid_found = 0;
$PKEY="ffff";

$command="";
$filename="";
$index=0;
$local_hca_info_flag = 0;

sub print_err{
	my $temp;
	$temp = $0. ": ". $_[0];
	`logger $temp`;

}

sub get_hca_name{
	my $hca_no = $_[0];
	my $port_no = $_[1];
	
	foreach $row (0..$#hca_info_array) {
		if (($hca_info_array[$row][0] eq $hca_no) && ($hca_info_array[$row][2] eq $port_no)) {
			$_[2] = $row;
			return $hca_info_array[$row][1];
		}
	}
	print_err("Invalid HCA No or Port No Specified. HCA No - $hca_no Port No - $port_no not present.\n");
	exit_check();
}

sub search_port_guid{
	my $guid = $_[0];
	for $row (0..$#hca_info_array) {
		if ( $hca_info_array[$row][3] eq $guid ) {
			$_[1] = $hca_info_array[$row][0]; # HCA No
			return $hca_info_array[$row][2];  # Port No
		}	
	}
	print_err("Invalid Port GUID entered - $guid.\n");
	exit_check();
}

# This function parses the line to find the parameters in array parray

sub get_params{
	my $line1;
	my $val_ioc_guid;
	my $rem;
	my $val_pkey;
	my $val_instance;
	my $val_name;
	my $val_dgid = 0;
	my ($path, $val_csum, $val_heartbeat, $val_ioc_string, $val_ib_multicast);
	my @parray1;
	my $string;
	my $dgid;
	$line1 = $_[0];

	$port_found=0;
	$port_guid_found=0;

	if( $line1 =~ /$create/){
		print_err("Error: Keyword CREATE specified multiple times ");
		print_err("on line $line_count\n");
		exit_check();
	}
	if($_[2] == 1){
		if( $line1 =~ /$primary/){
			print_err( "Error: Keyword PRIMARY specified multiple times on line $line_count\n");
		exit_check();
		}
	}
	if($_[2] == 2){
		if( $line1 =~ /$secondary/){
			print_err("Error: Keyword SECONDARY specified multiple times on line $line_count\n");
			exit_check();
		}
	}
	if( $line1 =~ /$name/){
		print_err( "Error: NAME specified multiple times on line $line_count\n");
		exit_check();
	}		
			
	if( $line1 =~ /$ioc_string\s*=\s*/ ){
		$ioc_string_found = 1;
		($rem, $val_ioc_string) = split(/"/, $');
		$val_ioc = $val_ioc_string;
		$parray{"ioc_string="} = $val_ioc_string;
	}
	if( $line1 =~ /$DGID\s*=\s*/ ){
		$dgid_found = 1;
		if( $' =~ /\w*/ ){
			$dgid = $&;
			$dgid =~ s/^(0x)?0*//;
		}
		$parray{"dgid="} = lc($dgid);
		$parray{"pkey="} = $PKEY;
	}
	if( $line1 =~ /$ioc_guid\s*=\s*/ ){
		$ioc_guid_found = 1;
		if( $' =~ /\w*/ ){
			$val_ioc_guid = $&;
		}
		$parray{"ioc_guid="} = lc($val_ioc_guid);
	}
	if( $line1 =~ /$instance\s*=\s*/ ){
		if( $' =~ /-*\w*/ ){
			$val_instance = $&;
		}
		$parray{"instance="} = $val_instance;
	}
	if( $line1 =~ /$port\s*=\s*/ ){
		if ($port_guid_found eq 1){
			print_err("PORT and PORTGUID should not be specified simultaneously.\n");
			exit_check();
		}
		if( $' =~ /-*\w*/ ){
			$port_found=1;
			$val_port = $&;
		}
		if ($val_port =~ /-/){
			print_err("Negative  values not allowed for port");
			exit_check();
		}
	}
	if($line1 =~ /$hca\s*=\s*/){
		if ($port_guid_found eq 1){
			print_err("Both HCA and PORTGUID parameters specified. Defaulting to PORTGUID.\n");
		}
		else{
			if( $' =~ /-*\w*/ ){
				$val_hca = $&;
			}
			if ($val_hca =~ /-/){
				print_err("Negative values not allowed for hca");
				exit_check();
			}
		}
	}
	if( $line1 =~ /$rx_csum\s*=\s*/ ){
		if( $' =~ /\w*/ ){
			$val_csum = $&;
		}
		$parray{"rx_csum="} = lc($val_csum);
	}
	if( $line1 =~ /$heartbeat\s*=\s*/ ){
		if( $' =~ /-*\w*/ ){
			$val_heartbeat = $&;
		}
		$parray{"heartbeat="} = lc($val_heartbeat);
	}
	if( $line1 =~ /$ib_multicast\s*=\s*/ ){
		if( $' =~ /\w*/ ){
			$val_ib_multicast = $&;
		}
		$parray{"ib_multicast="} = lc($val_ib_multicast);
	}
	if( $line1 =~ /$port_guid\s*=\s*/ ){
		$port_guid_found=1;
		if ($port_found) {
			print_err("PORT and PORTGUID should not be specified simultaneously.\n");
			exit_check();
		}
		if ($' =~ /-*\w*/ ) {
			$val_port_guid=$&;
		}
	}
	$parray{"name="} = $pname;
	if($line1 =~ /}/){
		$_[1] = 1;
		if ($prim_start_seen eq 1){
                     $path_desc = "primary";
                } else{
			if($sec_start_seen eq 1){
	                     $path_desc = "secondary";
			}
                }
		# If dgid is specified do not execute ib_qlgc_vnic_query
		# In case it is not specified run ib_qlgc_vnic_query
		# If -f option is set execute ib_qlgc_vnic_query in the background
		# and collect its output in a file.	
		if (!$dgid_found){
			if ( $local_hca_info_flag == 0 ) {
				&prepare_local_hca_information_table();
				$local_hca_info_flag = 1;
				if ($port_guid_found) {
					$val_port = &search_port_guid($val_port_guid, $val_hca);
				}
			}
			$val_hca_name = &get_hca_name($val_hca, $val_port, $index);
			$command = $discover_tool." -es -C ".$val_hca_name." -P ".$val_port;
			if ($ioc_string_found){
				$val_ioc_string = $parray{"ioc_string="};
				if ($ibvexdm_output_in_file){
					$filename = "/tmp/qlgc_vnic_".$val_hca_name."-".$val_port;
					$command = $command." > ".$filename;
					$val_dgid = &parse_iocstring_from_file($val_ioc_string, $command, 
									       $filename, $hca_info_array[$index][4]);
				}
				else{
					$val_dgid = &parse_iocstring($val_ioc_string, $command);
				}
				if($val_dgid eq "0"){
					$parray{"dgid="} = "00000000000000000000000000000000";
					$parray{"pkey="} = "0000";
					$parray{"ioc_guid="} = "0000000000000000";
					print_err( "$path_desc path will not be active since ioc_string=$val_ioc_string is unreachable.");
					print_err( "Whenever $path_desc path becomes available, Dynamic Update Daemon will update path parameters.");
				}
				$parray{"ioc_string="} = $val_ioc_string;
			}
			elsif ($ioc_guid_found) {
				$val_ioc_guid = $parray{"ioc_guid="};
				if ($ibvexdm_output_in_file){
					$filename = "/tmp/qlgc_vnic_".$val_hca_name."-".$val_port;
					$command = $command." > ".$filename;
					$val_pkey = &parse_dgid_from_file(lc($val_ioc_guid),$val_dgid, 
									  $val_ioc_string, $command, 
									  $filename, $hca_info_array[$index][4]);
				}
				else{
					$val_pkey = &parse_dgid(lc($val_ioc_guid), $val_dgid, 
								$val_ioc_string, $command);
				}
				if($val_dgid eq "0"){
					$val_dgid = "00000000000000000000000000000000";
					$val_pkey = "0000";
					$val_ioc_string = "";
					print_err( "$path_desc path will not be active since ioc_guid=$val_ioc_guid is unreachable.");
					print_err( "Whenever $path_desc path becomes available, Dynamic Update Daemon will update path parameters.");
				}
				$parray{"pkey="} = $val_pkey;
				$parray{"dgid="} = $val_dgid;
				$parray{"ioc_string="} = $val_ioc_string;
			}
		} 
		$path_desc = "primary";
		@parray1 = %parray;
		$string = &join_each(@parray1);			 
		chop($string);
		&write_to_file($string, $val_hca, $val_port, $_[2]);
		$prim_seen = 0;	
		$prim_start_seen = 0;
		$sec_start_seen = 0;
		$sec_seen = 0;
		if ($_[3] == 0){
			$c_term = 1;
			$createfound = 0;
			$block_found = 0;
			$pname="";
			$found = 0;
			if ($primary_path){
				system("echo \"$primary_path:$string\" >>$tempfile");
			}

		}
		#If terminating brace of create is on same line or terminating brace of create is found
		if ( ( $' =~ /}/ ) ){
			if ($_[2] == 1){
				if ($line1 =~ /$secondary/){
					return $string;	
				}
			}
						
			if ( $' =~ /}/) {
				print_err("Error: Extra terminating brace"); 
				print_err("on line $line_count\n");
				exit_check();
			}
			$c_term = 1;
			$createfound = 0;
			$block_found = 0;
			$found = 0;
			if ($_[2] == 1){
				$pstring = $string;
			}
			elsif ($_[2] == 2){
				$sstring = $string;
			}
		
			if ($primary_path){
				system("echo \"$primary_path:$pstring\" >>$tempfile");
			}
			if ($secondary_path){
				system("echo \"$secondary_path:$sstring\" >>$tempfile");
			}
		}
		return $string;	
	}
	else{
		$_[1] = 0;
	}

}	

	
# This function gets the dgid from the output of the tool , given iocguid 
sub parse_dgid_from_file{
	my @gline;
	my $line;
	my $dgid;
	my $iocstring;
	my $rem;
	my $guid;
	my $pkey;
	my $err;
	my $tool;
	my $print_ioc_guid;
	$err = -1;
	$tool = $_[3];
	$print_ioc_guid = $_[0];
	my $file_name =$_[4];
	my $started_ibvexdm = $_[5];
	my $start_time =0;
	my $now = 0;
	my $duration = 0;
	my $handle;
	my $ret;

	print_err( "For interface $pname trying to find dgid for $path_desc path using ioc_guid=$print_ioc_guid through port=$val_port \n" );
	if (!$started_ibvexdm){
		print_err("Running ib_qlgc_vnic_query through port $val_port");
		$pid[$pid_index++]= `$tool & echo \$! 2>&1`;
		$_[5]= 1;
	}
	else {
		print_err("Found it running. Not rerunning it again. Using output from file - $file_name.\n");
	}
		
	$ret = sysopen(QUERY_OP_FILE_HANDLE,"$file_name",O_NONBLOCK|O_RDONLY);
	if (!$ret){
		print_err("Cannot open file $file_name");
		exit_check();
	}
	$handle = *QUERY_OP_FILE_HANDLE;
	# Wait for MAX_INIT_TIME seconds for some output to 
	# be written to file in intervals of 1 second
		
	@gline=<$handle>;
	$start_time = `date +%s`;
	while ( $duration < $MAX_INIT_TIME){
		if ($#gline < 0){
			sleep 1;
			@gline=<$handle>;
		}
		else {
			last;
		}
		$now = `date +%s`;
		$duration = $now - $start_time; 
	}
	if ($#gline >= 0){ 
		print_err("Output from ib_qlgc_vnic_query: ");
		foreach $line(@gline) {
			print_err("ib_qlgc_vnic_query: $line");
		}
	}
	$duration = 0;
	$start_time = `date +%s`;
	# Wait for MAX_WAIT_FOR_ENTRY seconds till we get the required value from output given by ib_qlgc_vnic_query
	# in intervals of 1 second
	while ( $duration < $MAX_WAIT_FOR_ENTRY){
		foreach $line( @gline ) {
			if($line =~ /ioc_guid=/) {
				($guid, $rem) = split( /,/, $' );
				#Strip off the leading 0's and 0x from iocguid 
				#values from config file and ib_qlgc_vnic_query output
				$guid =~ s/^(0x)?0*//;
				$print_ioc_guid =~ s/^(0x)?0*//;
				if( $guid eq $print_ioc_guid){
					if($line =~ /dgid=/) {
						($dgid, $rem) = split( /,/, $' );
						$line = $';
						$_[1] = $dgid;
						
						if($line =~ /pkey=/){
							($pkey, $rem) = split( /,/, $');
						    if($line =~ /"/) {
								($iocstring, $rem) = split( /"/, $' );
								$_[2] = $iocstring;
							}
							close $handle;
							return $pkey;
						}
					}
				}

			}
		}
		sleep (1);
		@gline=<$handle>;
		if ($#gline >= 0){ 
			print_err("Output from ib_qlgc_vnic_query: ");
			foreach $line(@gline) {
				print_err("ib_qlgc_vnic_query: $line");
			}
		}
		$now = `date +%s`;
		$duration = $now - $start_time; 
	}
	close $handle;	
	$_[1] = 0;

}
# This function gets the dgid from the output of the tool , given iocguid 
sub parse_dgid {
	my @gline;
	my $line;
	my $dgid;
	my $iocstring;
	my $rem;
	my $guid;
	my $pkey;
	my $err;
	my $tool;
	my $print_ioc_guid;
	$err = -1;
	$tool = $_[3];
	$print_ioc_guid = $_[0];

	print_err( "For interface $pname trying to find dgid for $path_desc path using ioc_guid=$print_ioc_guid through port=$val_port using ib_qlgc_vnic_query..........\n" );
	@gline=`$tool`;
	if (!$#gline){
		print_err("Error in executing $discover_tool\n");
		$_[1] = 0;
		return;
	}

	print_err("Output from ib_qlgc_vnic_query: ");
	foreach $line(@gline) {
		print_err("ib_qlgc_vnic_query: $line");
	}

	foreach $line( @gline ) {
		if($line =~ /ioc_guid=/) {
			($guid, $rem) = split( /,/, $' );
			#Strip off the leading 0's and 0x from iocguid 
			#values from config file and ib_qlgc_vnic_query output

			$guid =~ s/^(0x)?0*//;
			$print_ioc_guid =~ s/^(0x)?0*//;
			if( $guid eq $print_ioc_guid ){
		     		if($line =~ /dgid=/) {
					($dgid, $rem) = split( /,/, $' );
					$line = $';
					$_[1] = $dgid;
					
					if($line =~ /pkey=/){
						($pkey, $rem) = split( /,/, $');
					    if($line =~ /"/) {
							($iocstring, $rem) = split( /"/, $' );
							$_[2] = $iocstring;
						}
						return $pkey;
					}
				}
			}

		}
	}
	$_[1] = 0;

}

# This function gets the dgid with IOCSTRING as input
sub parse_iocstring {
	my @gline;
	my $line;
	my $dgid;
	my $rem;
	my $guid;
	my $pkey;
	my $iocstring;
	my $temp1;
	my ($temp2, $chk);
	my $input_string;
	my $tool;
	$input_string = $_[0];
	$tool = $_[1];

	print_err( "For interface $pname trying to find dgid for $path_desc path using ioc_string=$input_string through port=$val_port using ib_qlgc_vnic_query..........\n" );
	@gline = `$tool`;
	if (!$#gline){
		print_err("Error in executing $discover_tool\n");
		return "0";
	}

	print_err("Output from ib_qlgc_vnic_query: ");
	foreach $line(@gline) {
		print_err("ib_qlgc_vnic_query: $line");
	}

	foreach $line(@gline) {
		if ($line =~ /".*"/){
			$iocstring = $&;
			($rem, $iocstring) = split( /"/, $iocstring );
			$chk = &chk_iostring($iocstring, $input_string);
			if ($chk == 1){
				($guid, $dgid, $pkey)=split( /,/, $line );
				if ($guid =~ "ioc_guid="){
					$guid = $';
				}
				if ($dgid =~ "dgid="){
					$dgid = $';
				}
				if ($pkey =~ "pkey="){
					$pkey = $';
				}
				$parray{"ioc_guid="} = $guid;
				if ($dgid){
					$parray{"dgid="} = $dgid;
					$parray{"pkey="} = $pkey;
					return $dgid;
				}
			}
		}
	}
	return "0";
}
# This function gets the dgid with IOCSTRING as input
sub parse_iocstring_from_file {
	my @gline;
	my $line;
	my $dgid;
	my $rem;
	my $guid;
	my $pkey;
	my $iocstring;
	my $temp1;
	my ($temp2, $chk);
	my $input_string;
	my $tool;
	$input_string = $_[0];
	$tool = $_[1];
	my $file_name =$_[2];
	my $started_ibvexdm = $_[3];
	my $start_time =0;
	my $now = 0;
	my $duration = 0;
	my $handle;
	my $ret;

	print_err( "For interface $pname trying to find dgid for $path_desc path using ioc_string=$input_string through port=$val_port" );
	if (!$started_ibvexdm){
		print_err("Running ib_qlgc_vnic_query through port $val_port");
		$pid[$pid_index++]= `$tool & echo \$! 2>&1`;
		$_[3]= 1;
	}
	else {
		print_err("Found it running. Not rerunning it again. Using output from file - $file_name.\n");
	}

	$ret = sysopen(QUERY_OP_FILE_HANDLE,"$file_name",O_NONBLOCK|O_RDONLY);
	if (!$ret){
		print_err("Cannot open file $file_name");
		exit_check();
	}
	$handle = *QUERY_OP_FILE_HANDLE;

	@gline=<$handle>;
	$start_time = `date +%s`;
	while ( $duration < $MAX_INIT_TIME){
		if ($#gline < 0){
			sleep 1;
			@gline=<$handle>;
		}
		else {
			last;
		}
		$now = `date +%s`;
		$duration = $now - $start_time; 
	}

	if ($#gline >= 0){ 
		print_err("Output from ib_qlgc_vnic_query: ");
		foreach $line(@gline) {
			print_err("ib_qlgc_vnic_query: $line");
		}
	}
	$duration = 0;
	$start_time = `date +%s`;
	while ( $duration < $MAX_WAIT_FOR_ENTRY){

		foreach $line(@gline) {
			if ($line =~ /".*"/){
				$iocstring = $&;
				($rem, $iocstring) = split( /"/, $iocstring );
				$chk = &chk_iostring($iocstring, $input_string);
				if ($chk == 1){
					($guid, $dgid, $pkey)=split( /,/, $line );
					if ($guid =~ "ioc_guid="){
						$guid = $';
					}
					if ($dgid =~ "dgid="){
						$dgid = $';
					}
					if ($pkey =~ "pkey="){
						$pkey = $';
					}
					$parray{"ioc_guid="} = $guid;
					if ($dgid){
						$parray{"dgid="} = $dgid;
						$parray{"pkey="} = $pkey;
						close $handle;
						return $dgid;
					}
				}
			}
		}
		sleep (1);
		@gline=<$handle>;

		if ($#gline >= 0){ 
			print_err("Output from ib_qlgc_vnic_query: ");
			foreach $line(@gline) {
				print_err("ib_qlgc_vnic_query: $line");
			}
		}		
		$now = `date +%s`;
		$duration = $now - $start_time; 
	}
	close $handle;	
	return "0";
}

# This function gets the matching iocstring from output of the tool
sub chk_iostring{
	my ($chassis1, $chassis2, $slot1, $slot2, $ioc1, $ioc2);
	my ($temp1, $temp2);
		$temp1 = $_[0];
		$temp2 = $_[1];
		($chassis1, $slot1, $ioc1) = split(/,/, $temp1);
		if ($chassis1 =~ /\s*Chassis\s*/){
			$chassis1 = $';
			$chassis1 = lc($chassis1);
		}	
		if ($slot1 =~ /\s*Slot\s*/){
			$slot1 = $';
		}	
		if ($ioc1 =~ /\s*IOC\s*/){
			$ioc1 = $';
		}

		($chassis2, $slot2, $ioc2) = split(/,/, $temp2);
		if ($chassis2 =~ /\s*Chassis\s*/){
			$chassis2 = $';
			$chassis2 = lc($chassis2);
		}	
		if ($slot2 =~ /\s*Slot\s*/){
			$slot2 = $';
		}	
		if ($ioc2 =~ /\s*IOC\s*/){
			$ioc2 = $';
		}
			
		if (($chassis1 eq $chassis2) && ($slot1 eq $slot2) && ($ioc1 eq $ioc2)){
			return 1;
		}
		else{
			return 0;
		}

}
	

sub write_to_file{
	my ($list, $filename, $hca, $port, %array);
	$list =`ls $path`;
	$hca = $_[1];
	$port = $_[2];
	$ioc_string_found = 0;
	$ioc_guid_found = 0;
	$dgid_found = 0;
	if ($list =~ /.*$hca-$port/ ) {
		$filename = $path.$&;
		if ( -d $filename){
			if( $_[3] == 1 ){
				$filename = $filename.$primary_file;
				$primary_path = $filename;
			}
			elsif ( $_[3] == 2 ){
				$filename = $filename.$secondary_file;
				$secondary_path = $filename;
			}
		}
		else{
			print_err("Error: Directory $filename not present");
			exit_check();
		}
	}
	else{
		print_err("Error: HCA $hca Port $port not present");
		exit_check();
	}
	init_array();
}

sub join_each{
	my @array = @_;
	my( $size, $n, $val, $temp);
	my $string = "";
      	$n = 0;
	$val = 0;	
	$size = @array;
	while ($n < $size){
		$val = length($array[$n+1]);	
		if ($val){
			chomp($array[$n]);
			chomp($array[$n+1]);
			if (($array[$n] ne "ioc_string=")){
				$string = $string.$array[$n];
				$string = $string.$array[$n+1].",";
				chomp($string);
			}
# To have the ioc_string echoed to the driver in "quotes" and ioc_string is being
# 'echo'ed twice, you need to have these many escapes.
# first echo to tempfile : \\\"<ioc_string>\\\" - escaping \ and "
# second echo to driver  : \"<ioc_string>\"     - escaping "
# and other 3 are for escaping \\\" in perl program.
			else {
				$buff = "\\\\\\\"" . $array[$n+1] . "\\\\\\\"";
				$string = $string.$array[$n].$buff.",";
			}
		}
		else{
			if ($array[$n] eq "name="){
				print_err("Error: Missing NAME in block");
				print_err("on line $create_start\n");
				exit_check();
			}
			if ($array[$n] eq "ioc_guid="){
				print_err("Error: Missing IOCGUID in block ");
				print_err("on line $create_start\n");
				exit_check();
			}
		}
		$n = $n+2;
	}
	return $string;
}
	
sub init_array{
	$parray{"ib_multicast="}="";
	$parray{"heartbeat="}="";
	$parray{"rx_csum="}="";
	$parray{"instance="}="";
	$parray{"name="}="";
	$parray{"pkey="}="";
	$parray{"ioc_guid="}="";
	$parray{"ioc_string="}="";
	$parray{"dgid="}="";
	$ioc_string_found = 0;
	$ioc_guid_found = 0;
	$val_port = 1;
	$val_hca = 0;
	$dgid_found_port = 0;
	$dgid_found = 0;
}

sub write_strings{
	my $file_handle;
	my @records;
	my $file_name;
	my $string;
	my $record;

	open(INFO,$tempfile );
	@records = <INFO>;
	close($tempfile);
	foreach $record (@records){
		($file_name, $string) = split(/:/, $record);
		chomp ($file_name);
		chomp ($string);
		if ($verbose){
			print_err("Writing $string to $file_name");
		}
		`echo -n $string > $file_name`;
		
		if ($?){
			$error_flag = 1;
			if ($string =~ /$name_token\s*=\s*/) {
				if( $' =~ /\w*/ ){
					print_err("Error: QLogic VNIC interface $& could not be created.\n");
				}
			}
		}
	}
}		
 
sub exit_check{
	chomp($tempfile);
	if (-e $tempfile){
		`rm -f $tempfile`;
	}
	`rm -rf /tmp/qlgc_vnic_*-*`;
	exit(1);
}	

sub check_proper{
	my $line = $_[0];
	if ($line =~ /$secondary/){
		if (!($` =~ /{/)){
			print_err("Error: Missing opening brace before PRIMARY block on line $line_count\n");
			exit_check();
		}
	}
}

sub prepare_local_hca_information_table{
	my @hca;
	my $buff;

	@hca_info=`$hca_information_command`;
	if ($?) {
		print_err("Failed to obtain local HCA information.\n");
		foreach $line(@hca_info) {
			$buff = "\"".$line."\"";
			print_err("$buff\n");
		}
		exit_check();
	}

	foreach $line(@hca_info) {
        	if ($line ne "\n") {
			chomp($line);
                	@hca=split(/,/, $line);
	                push @hca_info_array, [ @hca, "0" ]; # This zero is being added to array as a flag variable
							     # to indicate if ib_qlgc_vnic_query has been already
							     # started on the given IB port or not.
        	}
	}
}

# Main routine 	
	my $ret;
	my $val_name;
	my $rem;
	$found=0;
	my $temp;
	my $argnum;
	my $option;
	my $pid; 
	my $exists;

	print_err("QLGC_VNIC - Running QLogic Corp. Virtual NIC [VNIC] Parser Version $PARSER_VERSION\n");
	if ($#ARGV >= 0 ){
		foreach $argnum (0 .. $#ARGV) {
			$option = $ARGV[$argnum];
			if ( $option eq "-v") {
				$verbose = 1;	
			}
			if ( $option eq "-f") {
				$ibvexdm_output_in_file = 1;	
			}
			if ( $option eq "-i") {
				$MAX_INIT_TIME = $ARGV[$argnum + 1];
			}
			if ( $option eq "-p") {
				$MAX_WAIT_FOR_ENTRY = $ARGV[$argnum + 1];
			}
			if ( $option eq "-V") {
				print "QLogic VNIC Parser. Version : $PARSER_VERSION\n";
				chomp($tempfile); 
		 	    if (-e $tempfile){
			        `rm -f $tempfile`; 
			    }
			    exit(0);
			}
		}
	}	
	if (!(-e $file)){
		if (!(-e $file_old)){
		
			if( !( -e $file_ics_inic)){
				print_err("Error: No configuration file found\n");
				exit_check();
			}
			else{
				print_err("Configuration file $file does not exist\n");
				$file = $file_ics_inic;
				print_err("Using configuration file $file_ics_inic\n");
			}
		}
		else{
				print_err("Configuration file $file does not exist\n");
				$file = $file_old;
				print_err("Using configuration file $file_old\n");
		}		
		
	}	

	open(INFO, $file );
	@lines = <INFO>;
	close(INFO);
	`lsmod | grep \"qlgc_vnic\" `;
	if ($?){
		print_err("Error: QLogic VNIC module not loaded\n");
		exit_check();
	}

	foreach $line (@lines) {
		$line_count = $line_count + 1 ;
		chomp( $line );
		# If line is blank skip it
		if ($line =~ /^$/ ) {
		}	
		# If line is a comment skip it
		elsif ($line =~ /\s*#/){
			next;
		}
		# First opening brace found
		elsif (($line =~ /^\s*{/)){
			$line = $';	
			$block_found = 1;
			$block_start = $line_count;
			# Found CREATE block
			if (($line =~ /\s*$create/)){
				if ($' =~ /$create/){
					print_err("Error: CREATE keyword specified twice on line $line_count\n");
					exit_check();
				}
				if (!$sec_term){
					print_err("Error: Missing terminating brace in SECONDARY block");
					print_err("on line $sec_start\n");
					exit_check();
				}
				if (!$c_term){
					print_err("Error: Missing terminating brace in CREATE ");
					print_err("block on line $create_start");
					exit_check();
				}

				$create_start = $line_count;
				$createfound = 1;
				$found = 0;
				$c_term = 0;
				$pname = "";
				if( $line =~ /$name/){
					if ($' =~ /=/){
						($val_name,$rem) = split( /"/, $');
						$pname = $rem;
						$parray{"name="} = $pname;
					}
					else {
						print_err("Error: Syntax error in keyword NAME on line $line_count\n");
						exit_check();
					}
				}
				# Found PRIMARY within create  block
				if ($' =~ /\s*$primary/){
					$prim_start = $line_count;
					$temp = $';
					check_proper($');
					if ($temp =~ /{/){
						$prim_start_seen = 1;
						$pstring = get_params($', $prim_term, $prim, $prim);
						if ($prim_term == 1){
							init_array();
						}
					}
					else {
						$prim_start_seen = 0;
						$prim_term = 0;
					}
					$prim_seen = 1;
					$found=1;
				}
				# Found SECONDARY within create block
				if ($' =~ /\s*$secondary/){
					if (!$found){
						print_err("Error: Missing keyword PRIMARY on block on line $block_start\n");
						exit_check();
					}
					if(!$prim_term){
						print_err("Error: Missing terminating brace in PRIMARY block on line $line_count");
						exit_check();
					}
					$sec_start = $line_count;
					if ($' =~ /{/){
						$sec_start_seen = 1;
						$sstring = get_params($', $sec_term, $sec, $sec);
						if ($sec_term == 1){
							init_array();
						}
					}
					else {
						$sec_start_seen = 0;
						$sec_term = 0;
					}
					$sec_seen = 1;
					$found=1;
				}
				if (!$found){
					$pstring = get_params($',$c_term, $prim, $crt);
				}
			}
			else {
				if ((!$prim_term) && ($prim_seen)){
					$pstring = get_params($', $prim_term, $prim, $prim);
					if ($prim_term == 1){
						init_array();
					}
				}
				else {
					if ((!$c_term)){
						print_err("Missing terminating brace of CREATE block on line $create_start");
						exit_check();
					}
					if ($createfound){
						print_err("Error: Keyword PRIMARY/SECONDARY not specified on line $line_count\n");
						exit_check();
					}
				}

				if ((!$sec_term) && ($sec_seen)){
					$sstring = get_params($', $sec_term, $sec);
					if ($sec_term == 1){
						init_array();
					}
				}
				else {

					if ($createfound){
						print_err("Error: Keyword SECONDARY not specified on line $line_count\n");
						exit_check();
					}
				}
			}
		}
		elsif (($line =~ /$create/)){
			if (!$block_found) {
				print_err("Error: Missing opening brace before CREATE on line $line_count\n");
				exit_check();
			}
			if($createfound){
				print_err("Error: Missing terminating brace in CREATE block on line $create_start\n");
				exit_check();
			}
			$createfound = 1;
			$found =0;
			$create_start = $line_count;
			$pname = "";
			if( $line =~ /$name/){
				if ($' =~ /=/){
					($val_name,$rem) = split( /"/, $');
					$pname = $rem;
					$parray{"name="} = $pname;
					$pstring = get_params( $', $c_term, $prim, $crt);
				}
				else {
					print_err("Error: Syntax error in keyword NAME on line $line_count\n");
					exit_check();
				}
			}
		}
		# Found PRIMARY block
		elsif ( (($line =~ /\s*$primary/) || (!$prim_term)) && 
				( $createfound == 1 )){
			$found = 2;
		# If Primary block has not terminated
			if (!$prim_term){
				if (!$prim_start_seen){
					print_err("Error :Missing opening brace in PRIMARY block on line $line_count\n");
					exit_check();
				}	
				if ($line =~ /\s*$primary/){
					print_err("Error: Keyword PRIMARY");
					print_err(" specified multiple times on line $line_count\n");
					exit_check();
				}	
				$pstring = get_params($line, $prim_term, $prim, $prim);
				
			}
		# Start of primary block	
			else{
				$prim_term = 0;
				$prim_start = $line_count;
				if ($' =~ /\s*$primary/){
					print_err("Error: Keyword PRIMARY");
					print_err(" specified multiple times on line $line_count\n");
					exit_check();
				}
				if ($line =~ /\s*{\s*/ ){	
					$prim_start_seen = 1;
					$pstring = get_params($', $prim_term, $prim, $prim);
			
				}
			}
			if ($prim_term == 1){
				init_array();
			}
			if ((($line =~ /\s*$secondary/) || (!$sec_term))
			&& ($line !~ /\s*$create\s*/)){
				if (!$prim_term){
					print_err("Error: Missing terminating"); 
					print_err("brace in PRIMARY block"); 
					print_err("on line $prim_start\n");
					exit_check();
				}
				$found = 1;
				if (!$sec_term){
					if (!$sec_start_seen){
						print_err("Error :Missing opening brace ");
						print_err("in SECONDARY block on line $sec_start\n");
						exit_check();
					}	
					$sstring = get_params($line, $sec_term, $sec, $sec);
				}
				else{
					$sec_term = 0;
					$sec_start = $line_count;
					if($' =~ /\s*{\s*/ ){
						$sec_start_seen = 1;
						$sstring = get_params($', $sec_term, $sec, $sec);
					}
					if ($sec_term == 1){
					init_array();
                			}	
				}
			}

		}
		# If SECONDARY block found
		elsif ((($line =~ /\s*$secondary/) || (!$sec_term)) && ($line !~ /\s*$primary/) 
		&& ($line !~ /\s*$create\s*/)){
			if (!$found){
				print_err("Error: Missing keyword PRIMARY on block on $block_start\n");
				exit_check();
			} 
			if (!$prim_term){
				print_err("Error: Missing terminating brace ");
				print_err("in PRIMARY block on line $prim_start\n");
				exit_check();
			}
			$found = 2;
			# If SECONDARY block has not terminated
			if (!$sec_term){
				if (!$sec_start_seen){
					print_err("Error :Missing opening brace ");
					print_err("in SECONDARY block on line $sec_start\n");
					exit_check();
				}	
				$sstring = get_params($line, $sec_term, $sec, $sec);
			}
			# Start of SECONDARY block
			else{
				$sec_term = 0;
				$sec_start = $line_count;
				if($line =~ /\s*{\s*/ ){
					$sec_start_seen = 1;
					$sstring = get_params($', $sec_term, $sec, $sec);
				}
			}
			if ($sec_term == 1){
				
				init_array();
                	}
		}
		# If the CREATE block doesnt contain PRIMARY or SECONDARY blocks
		elsif (( $createfound == 1) && (!$found)){
			if( $line =~ /$name/){
				if ($' =~ /=/){
					($val_name,$rem) = split( /"/, $');
					$pname = $rem;
					$parray{"name="} = $pname;
					$pstring = get_params( $', $c_term, $prim, $crt);
				}
				else {
					print_err("Error: Syntax error in keyword NAME on line $line_count\n");
					exit_check();
				}
			}
			else{
				$pstring = get_params( $line, $c_term, $prim, $crt);
			}
		}
		# Closing brace found
		elsif (($line =~ /}/) ){
			if ((!$createfound)){
				if ($block_start){
					print_err("Error: Missing keyword CREATE ");
					print_err("on line $block_start\n");
				}
				else {
					print_err("Error: Missing keyword CREATE \n");
				}
				exit_check();
			}
			if (!$block_found){
				print_err("Error: Extra closing brace on line $line_count\n");
				exit_check();
			}
			$prim_seen = 0;		
			$sec_seen = 0;
			$prim_start_seen =0;
			$sec_start_seen = 0;
			$c_term = 1;
			$createfound = 0;
			$block_found = 0;
			if ($primary_path){
				$ret = system("echo \"$primary_path:$pstring\" >>$tempfile");
			}
			if ($secondary_path){
				$ret = system("echo \"$secondary_path:$sstring\" >>$tempfile");
			}
			$found = 0;
			$primary_path = "";
			$secondary_path = "";
			init_array();

		}

	}	
	if (!$prim_term){
		print_err("Error: Missing terminating brace in PRIMARY block");
		print_err("on line $prim_start \n");
		exit_check();
	}
	if (!$sec_term){
		print_err("Error: Missing terminating brace in SECONDARY block ");
		print_err("on line $sec_start\n");
		exit_check();
	}
	if (!$c_term){
		print_err("Error: Missing terminating brace in CREATE block ");
		print_err("on line $create_start\n");
		exit_check();
	}
	&write_strings();
	`rm -f $tempfile`;
	`rm -rf /tmp/qlgc_vnic_*-*`;
	if ($ibvexdm_output_in_file){	
		foreach $pid (@pid){
			$exists = kill 0,$pid;
			if ($exists){
				`kill -9 $pid`;	
			}
		}	
	}
	if ($error_flag) {
		print_err("Error: Some of the configured VNIC interfaces COULD NOT be created. ");
		print_err("Please refer the log above for configuration errors.\n");
		exit(1);
	}
