/*
 *      addressbooks.c
 *      
 *      Copyright 2013 Alex <alex@linuxonly.ru>
 *      
 *      This program is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 3 of the License, or
 *      (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <gmodule.h>

#include "addressbooks.h"


static void mmgui_addressbooks_get_gnome_contacts_foreach(gpointer data, gpointer user_data)
{
	mmgui_addressbooks_t addressbooks;
	EContact *econtact;
	mmgui_contact_t contact;
	const gchar *fullname, *nickname, *primaryphone, *mobilephone, *email, *group;
	GList *emails;
	
	addressbooks = (mmgui_addressbooks_t)user_data;
	econtact = (EContact *)data;
	
	if ((addressbooks == NULL) || (econtact == NULL)) return;
	
	fullname = (addressbooks->e_contact_get_const)(econtact, E_CONTACT_FULL_NAME);
	nickname = (addressbooks->e_contact_get_const)(econtact, E_CONTACT_NICKNAME);
	primaryphone = (addressbooks->e_contact_get_const)(econtact, E_CONTACT_PHONE_HOME);
	mobilephone = (addressbooks->e_contact_get_const)(econtact, E_CONTACT_PHONE_MOBILE);
	group = (addressbooks->e_contact_get_const)(econtact, E_CONTACT_CATEGORIES);
	emails = (addressbooks->e_contact_get)(econtact, E_CONTACT_EMAIL);
	email = g_list_nth_data(emails, 0);
	
	contact = g_new0(struct _mmgui_contact, 1);
	
	contact->name = g_strdup(fullname);
	contact->number = g_strdup(primaryphone);
	contact->email = g_strdup(email);
	contact->group = g_strdup(group);
	contact->name2 = g_strdup(nickname);
	contact->number2 = g_strdup(mobilephone);
	contact->id = addressbooks->counter;
	contact->hidden = FALSE;
	contact->storage = MMGUI_CONTACTS_STORAGE_UNKNOWN;
	
	addressbooks->gnomecontacts = g_slist_prepend(addressbooks->gnomecontacts, contact);
	
	addressbooks->counter++;
	
	g_list_foreach(emails, (GFunc)g_free, NULL);
	g_list_free(emails);
}

static gboolean mmgui_addressbooks_get_gnome_contacts(mmgui_addressbooks_t addressbooks)
{
	EBookQuery *queryelements[2];
	EBookQuery *query;
	GError *error;
	gchar *s;
	/*New API*/
	ESourceRegistry *registry;
	ESource *source, *addressbook;
	EBookClient *client;
	GSList *scontacts;
	/*Old API*/
	EBook *book;
	GList *dcontacts;
	
	if (addressbooks == NULL) return FALSE;
	if (!addressbooks->gnomesupported) return FALSE;
	if (addressbooks->ebookmodule == NULL) return FALSE;
	
	error = NULL;
	
	queryelements[0] = (addressbooks->e_book_query_field_exists)(E_CONTACT_PHONE_HOME);
	queryelements[1] = (addressbooks->e_book_query_field_exists)(E_CONTACT_PHONE_MOBILE);
	
	query = (addressbooks->e_book_query_or)(2, queryelements, TRUE);
	if (query == NULL) {
		g_debug("Failed to form GNOME contacts query\n");
		return FALSE;
	}
	
	if (addressbooks->e_book_new_system_addressbook != NULL) {
		g_debug("Using old GNOME contacts API\n");
		book = (addressbooks->e_book_new_system_addressbook)(&error);
		if (book == NULL) {
			(addressbooks->e_book_query_unref)(query);
			g_debug("Failed to load GNOME system addressbook: %s\n", error->message);
			g_error_free(error);
			return FALSE;
		}
		
		if (!(addressbooks->e_book_open)(book, TRUE, &error)) {
			(addressbooks->e_book_query_unref)(query);
			g_debug("Failed to load GNOME system addressbook: %s\n", error->message);
			g_error_free(error);
			return FALSE;
		}
		
		if (!(addressbooks->e_book_get_contacts)(book, query, &dcontacts, &error)) {
			(addressbooks->e_book_query_unref)(query);
			g_debug("Failed to get query GNOME addressbook results: %s\n", error->message);
			g_error_free(error);
			return FALSE;
		}
	} else {
		g_debug("Using new GNOME contacts API\n");
		registry = (addressbooks->e_source_registry_new_sync)(NULL, &error);
		if (registry == NULL) {
			(addressbooks->e_book_query_unref)(query);
			g_debug("Failed to get ESourceRegistry: %s\n", error->message);
			g_error_free(error);
			return FALSE;
		}
		
		source = (addressbooks->e_source_registry_ref_builtin_address_book)(registry);
		if (source == NULL) {
			(addressbooks->e_book_query_unref)(query);
			g_debug("Failed to get ESource\n");
			return FALSE;
		}
		
		if (addressbooks->e_book_client_connect_sync != NULL) {
			/*Version 3.8*/
			client = (EBookClient *)(addressbooks->e_book_client_connect_sync)(source, NULL, &error);
			if (client == NULL) {
				(addressbooks->e_book_query_unref)(query);
				g_debug("Failed to get EBookClient: %s\n", error->message);
				g_error_free(error);
				return FALSE;
			}
		} else {
			/*Versions 3.2 ... 3.6*/
			client = (addressbooks->e_book_client_new)(source, &error);
			if (client == NULL) {
				(addressbooks->e_book_query_unref)(query);
				g_debug("Failed to get EBookClient: %s\n", error->message);
				g_error_free(error);
				return FALSE;
			}
			
			if (!(addressbooks->e_client_open_sync)((EClient *)client, TRUE, NULL, &error)) {
				(addressbooks->e_book_query_unref)(query);
				g_debug("Failed to open EBookClient: %s\n", error->message);
				g_error_free(error);
				return FALSE;
			}
		}
		
		s = (addressbooks->e_book_query_to_string)(query);
		if (s == NULL) {
			(addressbooks->e_book_query_unref)(query);
			g_debug("Failed to get GNOME addressbook request in string format\n");
			return FALSE;
		}
		
		g_debug("GNOME addressbook request: %s\n", s);
		
		if (!(addressbooks->e_book_client_get_contacts_sync)(client, s, &scontacts, NULL, &error)) {
			(addressbooks->e_book_query_unref)(query);
			g_debug("Failed to get GNOME addressbook query results: %s\n", error->message);
			g_error_free(error);
			return FALSE;
		}
	}
	
	(addressbooks->e_book_query_unref)(query);
	
	if (addressbooks->e_book_new_system_addressbook != NULL) {
		if (dcontacts != NULL) {
			addressbooks->counter = 0;
			g_list_foreach(dcontacts, (GFunc)mmgui_addressbooks_get_gnome_contacts_foreach, addressbooks);
		} else {
			g_debug("No suitable contacts found\n");
		}
	} else {
		if (scontacts != NULL) {
			addressbooks->counter = 0;
			g_slist_foreach(scontacts, (GFunc)mmgui_addressbooks_get_gnome_contacts_foreach, addressbooks);
		} else {
			g_debug("No suitable contacts found\n");
		}
	}
	
	addressbooks->counter = 0;
	
	return TRUE;
}

mmgui_addressbooks_t mmgui_addressbooks_new(mmgui_libpaths_cache_t libcache)
{
	mmgui_addressbooks_t addressbooks;
	gboolean libopened;
	
	addressbooks = g_new0(struct _mmgui_addressbooks, 1);
	
	/*libebook*/
	addressbooks->ebookmodule = NULL;
	addressbooks->gnomesupported = FALSE;
	addressbooks->gnomecontacts = NULL;
	
	/*Open module*/
	addressbooks->ebookmodule = g_module_open(mmgui_libpaths_cache_get_library_name(libcache, "libebook-1.2"), G_MODULE_BIND_LAZY);
	
	if (addressbooks->ebookmodule != NULL) {
		libopened = TRUE;
		libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_book_query_field_exists", (gpointer *)&(addressbooks->e_book_query_field_exists));
		libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_book_query_or", (gpointer *)&(addressbooks->e_book_query_or));
		libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_book_query_unref", (gpointer *)&(addressbooks->e_book_query_unref));
		libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_contact_get_const", (gpointer *)&(addressbooks->e_contact_get_const));
		libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_contact_get", (gpointer *)&(addressbooks->e_contact_get));
		if (g_module_symbol(addressbooks->ebookmodule, "e_book_new_system_addressbook", (gpointer *)&(addressbooks->e_book_new_system_addressbook))) {
			libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_book_open", (gpointer *)&(addressbooks->e_book_open));
			libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_book_get_contacts", (gpointer *)&(addressbooks->e_book_get_contacts));
			/*Unused functions*/
			addressbooks->e_source_registry_new_sync = NULL;
			addressbooks->e_source_registry_ref_builtin_address_book = NULL;
			addressbooks->e_book_client_new = NULL;
			addressbooks->e_client_open_sync = NULL;
			addressbooks->e_book_client_connect_sync = NULL;
			addressbooks->e_book_query_to_string = NULL;
			addressbooks->e_book_client_get_contacts_sync = NULL;
		} else {
			if (!g_module_symbol(addressbooks->ebookmodule, "e_book_client_connect_sync", (gpointer *)&(addressbooks->e_book_client_connect_sync))) {
				/*Since version 3.2 used these functions, but in 3.8 they are deprecated*/
				libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_book_client_new", (gpointer *)&(addressbooks->e_book_client_new));
				libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_client_open_sync", (gpointer *)&(addressbooks->e_client_open_sync));
				addressbooks->e_book_client_connect_sync = NULL;
			}
			libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_source_registry_new_sync", (gpointer *)&(addressbooks->e_source_registry_new_sync));
			libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_source_registry_ref_builtin_address_book", (gpointer *)&(addressbooks->e_source_registry_ref_builtin_address_book));
			libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_book_query_to_string", (gpointer *)&(addressbooks->e_book_query_to_string));
			libopened = libopened && g_module_symbol(addressbooks->ebookmodule, "e_book_client_get_contacts_sync", (gpointer *)&(addressbooks->e_book_client_get_contacts_sync));
			
			/*Unused functions*/
			addressbooks->e_book_get_contacts = NULL;
			addressbooks->e_book_open = NULL;
			addressbooks->e_book_get_contacts = NULL;
		}
		/*If some functions not exported, close library*/
		if (!libopened) {
			addressbooks->e_book_query_field_exists = NULL;
			addressbooks->e_book_query_or = NULL;
			addressbooks->e_source_registry_new_sync = NULL;
			addressbooks->e_source_registry_ref_builtin_address_book = NULL;
			addressbooks->e_book_client_new = NULL;
			addressbooks->e_client_open_sync = NULL;
			addressbooks->e_book_query_to_string = NULL;
			addressbooks->e_book_client_get_contacts_sync = NULL;
			addressbooks->e_book_new_system_addressbook = NULL;
			addressbooks->e_book_open = NULL;
			addressbooks->e_book_get_contacts = NULL;
			addressbooks->e_book_query_unref = NULL;
			addressbooks->e_contact_get_const = NULL;
			addressbooks->e_contact_get = NULL;
			/*Close module*/
			g_module_close(addressbooks->ebookmodule);
			addressbooks->ebookmodule = NULL;
			addressbooks->gnomesupported = FALSE;
		} else {
			/*Get contacts*/
			addressbooks->gnomesupported = TRUE;
			mmgui_addressbooks_get_gnome_contacts(addressbooks);
		}
	}
	
	/*KDE addressbook is not supported now*/
	addressbooks->kdecontacts = NULL;
	addressbooks->kdesupported = FALSE;
	
	return addressbooks;
}

gboolean mmgui_addressbooks_get_gnome_contacts_available(mmgui_addressbooks_t addressbooks)
{
	if (addressbooks == NULL) return FALSE;
	
	return addressbooks->gnomesupported;
}

gboolean mmgui_addressbooks_get_kde_contacts_available(mmgui_addressbooks_t addressbooks)
{
	if (addressbooks == NULL) return FALSE;
	
	return addressbooks->kdesupported;
}

GSList *mmgui_addressbooks_get_gnome_contacts_list(mmgui_addressbooks_t addressbooks)
{
	if (addressbooks == NULL) return NULL;
	if (!addressbooks->gnomesupported) return NULL;
	
	return addressbooks->gnomecontacts;
}

GSList *mmgui_addressbooks_get_kde_contacts_list(mmgui_addressbooks_t addressbooks)
{
	if (addressbooks == NULL) return NULL;
	if (!addressbooks->kdesupported) return NULL;
	
	return addressbooks->kdecontacts;
}


static gint mmgui_addressbooks_get_contact_compare(gconstpointer a, gconstpointer b)
{
	mmgui_contact_t contact;
	guint id;
	
	contact = (mmgui_contact_t)a;
	id = GPOINTER_TO_UINT(b);
	
	if (contact->id < id) {
		return 1;
	} else if (contact->id > id) {
		return -1;
	} else {
		return 0;
	}
}

mmgui_contact_t mmgui_addressbooks_get_gnome_contact(mmgui_addressbooks_t addressbooks, guint index)
{
	GSList *contactptr;
	mmgui_contact_t contact;
	
	if (addressbooks == NULL) return NULL;
	if (!addressbooks->gnomesupported) return NULL;
	if (addressbooks->gnomecontacts == NULL) return NULL;
	
	contactptr = g_slist_find_custom(addressbooks->gnomecontacts, GUINT_TO_POINTER(index), mmgui_addressbooks_get_contact_compare);
	
	if (contactptr != NULL) {
		contact = (mmgui_contact_t)contactptr->data;
		return contact;
	} else {
		return NULL;
	}
}

mmgui_contact_t mmgui_addressbooks_get_kde_contact(mmgui_addressbooks_t addressbooks, guint index)
{
	GSList *contactptr;
	mmgui_contact_t contact;
	
	if (addressbooks == NULL) return NULL;
	if (!addressbooks->kdesupported) return NULL;
	if (addressbooks->kdecontacts == NULL) return NULL;
	
	contactptr = g_slist_find_custom(addressbooks->kdecontacts, GUINT_TO_POINTER(index), mmgui_addressbooks_get_contact_compare);
	
	if (contactptr != NULL) {
		contact = (mmgui_contact_t)contactptr->data;
		return contact;
	} else {
		return NULL;
	}
}

static void mmgui_addressbooks_free_contacts_list_foreach(gpointer data, gpointer user_data)
{
	mmgui_contact_t contact;
	
	if (data == NULL) return;
	
	contact = (mmgui_contact_t)data;
	
	if (contact->name != NULL) {
		g_free(contact->name);
	}
	if (contact->number != NULL) {
		g_free(contact->number);
	}
	if (contact->email != NULL) {
		g_free(contact->email);
	}
	if (contact->group != NULL) {
		g_free(contact->group);
	}
	if (contact->name2 != NULL) {
		g_free(contact->name2);
	}
	if (contact->number2 != NULL) {
		g_free(contact->number2);
	}
}

void mmgui_addressbooks_close(mmgui_addressbooks_t addressbooks)
{
	if (addressbooks == NULL) return;
	
	/*GNOME addressbook*/
	addressbooks->gnomesupported = FALSE;
	if (addressbooks->ebookmodule != NULL) {
		/*First free contacts list*/
		if (addressbooks->gnomecontacts != NULL) {
			g_slist_foreach(addressbooks->gnomecontacts, mmgui_addressbooks_free_contacts_list_foreach, NULL);
			g_slist_free(addressbooks->gnomecontacts);
			addressbooks->gnomecontacts = NULL;
		}
		/*Then unload module*/
		g_module_close(addressbooks->ebookmodule);
		addressbooks->ebookmodule = NULL;
	}
	
	/*KDE addressbook*/
	addressbooks->kdesupported = FALSE;
	if (addressbooks->kdecontacts != NULL) {
		/*Only free contacts list*/
		g_slist_foreach(addressbooks->kdecontacts, mmgui_addressbooks_free_contacts_list_foreach, NULL);
		g_slist_free(addressbooks->kdecontacts);
		addressbooks->kdecontacts = NULL;
	}
	
	g_free(addressbooks);
}
