// SPDX-License-Identifier: Apache-2.0
/*
 * Copyright IBM Corp. 2021, 2024
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdlib.h>
#include <string.h>

#include "testsupport.h"

void setUp(void) {

  tol_bfloat.ulps = 64;
  tol_bfloat.epsilon_mult = (0.1 / EPSILON_BFLOAT) + 1;

  tol_fp16.ulps = 64;
  tol_fp16.epsilon_mult = (0.1 / EPSILON_FP16) + 1;

  tol_fp32.ulps = 64 * 16384;
  tol_fp32.epsilon_mult = (0.1 / EPSILON_FLOAT) + 1;

  VERIFY_HW_ENV;
}

void tearDown(void) {}

// convenience debug macro
#define PRINT_DIMS(x)                                                          \
  printf(#x " pre: %u %u %u %u\n", (x)->pre_transformed_desc->dim4,            \
         (x)->pre_transformed_desc->dim3, (x)->pre_transformed_desc->dim2,     \
         (x)->pre_transformed_desc->dim1);                                     \
  printf(#x ": %u %u %u %u\n", (x)->transformed_desc->dim4,                    \
         (x)->transformed_desc->dim3, (x)->transformed_desc->dim2,             \
         (x)->transformed_desc->dim1);

typedef struct input_set {
  uint32_t n;
  uint32_t height_in;
  uint32_t width_in;
  uint32_t channel_in;
  uint32_t kernel_size[2];
  uint32_t channel_out;
} input_set;

typedef struct strides_input_set {
  uint32_t height;
  uint32_t width;
} strides_input_set;

void test_conv2d(input_set *set, strides_input_set *strides, void *input_vals,
                 void *kernel_vals, void *bias_vals, void *output_exp_vals,
                 void *output_relu_exp_vals, zdnn_pool_padding padding,
                 void *clipping_value) {

  zdnn_status status;

  /****************************************************************************

  ZDNN dims requirements

  if stride_height > 0, stride_width > 0, padding = SAME

  input:  (n, x, y, z)
  kernel: (i, j, z, a)
  bias:   (a)
  output: (n, e, f, a)   (e = ceil(x/stride_height), f = ceil(y/stride_width))

  if stride_height > 0, stride_width > 0, padding = VALID

  input:  (n, x, y, z)   (x > i, y > j)
  kernel: (i, j, z, a)
  bias:   (a)
  output: (n, e, f, a)   (e = ceil((x - i + 1)/stride_height),
                          f = ceil((y - j + 1)/stride_width))

  if stride_height = 0, stride_width = 0, padding = VALID

  input:  (n, x, y, z)
  kernel: (x, y, z, a)
  bias:   (a)
  output: (n, 1, 1, a)

  n                             = n
  x                             = height_in
  y                             = width_in
  z                             = channel_in
  (i, j)                        = kernel_size
  a                             = channel_out
  (stride_height, stride_width) = strides

  ****************************************************************************/

  uint32_t input_dims[4] = {set->n, set->height_in, set->width_in,
                            set->channel_in};
  uint32_t kernel_dims[4] = {0, 0, // 0s are placeholders
                             set->channel_in, set->channel_out};
  uint32_t bias_dims[1] = {set->channel_out};
  uint32_t output_dims[4] = {set->n, 0, 0,
                             set->channel_out}; // 0s are placeholders

  // zero-strides + VALID_PADDING is special case, so ignore kernel_size[0] &
  // [1] and set kernel_dims[0] & [1] to what zAIU wants
  if (padding == VALID_PADDING && strides->height == 0 && strides->width == 0) {
    kernel_dims[0] = set->height_in;
    kernel_dims[1] = set->width_in;
  } else {
    kernel_dims[0] = set->kernel_size[0];
    kernel_dims[1] = set->kernel_size[1];
  }

  // output_dims[1] & [2] are exactly what the zAIU wants
  if (padding == VALID_PADDING) {
    // output dim3
    output_dims[1] =
        (strides->height == 0 && strides->width == 0)
            ? 1
            : CEIL((set->height_in - kernel_dims[0] + 1), strides->height);
    // output dim2
    output_dims[2] =
        (strides->height == 0 && strides->width == 0)
            ? 1
            : CEIL((set->width_in - kernel_dims[1] + 1), strides->width);
  } else {
    // not bother a switch statement, gotta be SAME_PADDING
    output_dims[1] = CEIL(set->height_in, strides->height); // output dim3
    output_dims[2] = CEIL(set->width_in, strides->width);   // output dim2
  }

  zdnn_ztensor *input_ztensor = alloc_ztensor_with_values(
      input_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, false, input_vals);

  zdnn_ztensor *kernel_ztensor = alloc_ztensor_with_values(
      kernel_dims, ZDNN_HWCK, test_datatype, NO_CONCAT, false, kernel_vals);

  zdnn_ztensor *bias_ztensor = alloc_ztensor_with_values(
      bias_dims, ZDNN_1D, test_datatype, NO_CONCAT, false, bias_vals);

  zdnn_ztensor *output_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *output_relu_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  BEGIN_BLOCK_IF_LOGLEVEL_TRACE {
    PRINT_DIMS(input_ztensor);
    PRINT_DIMS(kernel_ztensor);
    PRINT_DIMS(bias_ztensor);
    PRINT_DIMS(output_ztensor);
    PRINT_DIMS(output_relu_ztensor);
    printf("strides = height %u width %u\n", strides->height, strides->width);
  }

  status = zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                       strides->height, strides->width, CONV2D_ACT_NONE,
                       clipping_value, output_ztensor);

  TEST_ASSERT_MESSAGE_FORMATTED(
      status == ZDNN_OK,
      "zdnn_conv2d() (CONV2D_ACT_NONE) failed, status = %08x", status);

  status = zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                       strides->height, strides->width, CONV2D_ACT_RELU,
                       clipping_value, output_relu_ztensor);

  TEST_ASSERT_MESSAGE_FORMATTED(
      status == ZDNN_OK,
      "zdnn_conv2d() (CONV2D_ACT_RELU) failed, status = %08x", status);

  BEGIN_BLOCK_IF_LOGLEVEL_TRACE {
    dumpdata_ztensor(input_ztensor, AS_FLOAT, true);
    dumpdata_ztensor(kernel_ztensor, AS_FLOAT, true);
    dumpdata_ztensor(bias_ztensor, AS_FLOAT, true);
    dumpdata_ztensor(output_ztensor, AS_FLOAT, true);
    dumpdata_ztensor(output_relu_ztensor, AS_FLOAT, true);
  }

  switch (output_ztensor->pre_transformed_desc->type) {
  case (BFLOAT):
    assert_ztensor_values_adv(output_ztensor, false, output_exp_vals,
                              tol_bfloat);
    assert_ztensor_values_adv(output_relu_ztensor, false, output_relu_exp_vals,
                              tol_bfloat);
    break;
  case (FP16):
    assert_ztensor_values_adv(output_ztensor, false, output_exp_vals, tol_fp16);
    assert_ztensor_values_adv(output_relu_ztensor, false, output_relu_exp_vals,
                              tol_fp16);
    break;
  case (FP32):
    assert_ztensor_values_adv(output_ztensor, false, output_exp_vals, tol_fp32);
    assert_ztensor_values_adv(output_relu_ztensor, false, output_relu_exp_vals,
                              tol_fp32);
    break;
  default:
    break;
  }

  free_ztensor_buffers(5, input_ztensor, kernel_ztensor, bias_ztensor,
                       output_ztensor, output_relu_ztensor);
}

/*******************************************************************

 small:

  n = 1
  height_in = 4
  width_in = 3
  channel_in = 5
  kernel_size = (2, 2)
  channel_out = 2

 medium:

  n = 3
  height_in = 10
  width_in = 8
  channel_in = 5
  kernel_size = (2, 3)
  channel_out = 5

 large:

  n = 4
  height_in = 15
  width_in = 10
  channel_in = 6
  kernel_size = (4, 6)
  channel_out = 7

 non-zero strides:
  small: (1, 1)
  medium: (2, 3)
  large: (3, 4)

*******************************************************************/

input_set small_input = {1, 4, 3, 5, {2, 2}, 2};
input_set medium_input = {3, 10, 8, 5, {2, 3}, 5};
input_set large_input = {4, 15, 10, 6, {4, 6}, 7};

strides_input_set small_non0_strides = {1, 1};
strides_input_set medium_non0_strides = {2, 3};
strides_input_set large_non0_strides = {3, 4};
strides_input_set zero_strides = {0, 0};

void test_valid_padding_non_zero_strides_small() {
  input_set *set = &small_input;
  strides_input_set *strides = &small_non0_strides;

  // 1,4,3,5
  uint32_t input_vals[] = {
      0x3f15a7d8, 0x3f55d27f, 0x3f04e322, 0x3f0bc669, 0x3f2eec77, 0x3d07cb10,
      0x3e528880, 0x3f1da880, 0x3e3fb0a8, 0x3f40bb0b, 0x3e0e0bcc, 0x3f17da1f,
      0x3f4f6880, 0x3d0b5ba0, 0x3f5ba1ae, 0x3e780c74, 0x3de61650, 0x3f7ae7a4,
      0x3f71ba1f, 0x3f2fdc52, 0x3f50c293, 0x3e1d23d0, 0x3deae1a8, 0x3f615378,
      0x3ba82d80, 0x3f4b0c93, 0x3e77825c, 0x3ea22f0a, 0x3aa20200, 0x3e33de00,
      0x3e8e771c, 0x3f39eaa3, 0x3f324e26, 0x3f17f541, 0x3f3fe98e, 0x3ef6c34e,
      0x3f3379fe, 0x3f6a0de8, 0x3ed4dfce, 0x3f1aca63, 0x3f51dd20, 0x3e50b72c,
      0x3f6f62f4, 0x3ed5df52, 0x3de131a8, 0x3f7f3fc1, 0x3f26ab72, 0x3f70f111,
      0x3e8ad072, 0x3e592e30, 0x3f32cd09, 0x3f4644b7, 0x3f19794f, 0x3f313923,
      0x3f786a79, 0x3f114ab9, 0x3edfb038, 0x3d858c20, 0x3e50bd98, 0x3f563faa,
  };

  // 2,2,5,2
  uint32_t kernel_vals[] = {
      0xbadb6d80, 0x3da65588, 0xbdb06154, 0xbd13a074, 0xbca28f68, 0xbe1025bd,
      0xbe5abfd6, 0xbe146a76, 0x3db50ae0, 0xbe05a938, 0x3df5a9a0, 0x3e3e2300,
      0x3d759a80, 0x3cc1d430, 0x3d725b60, 0xbd67df54, 0xbe17a51c, 0xbdd2ced4,
      0x3e0963a4, 0x3e44d336, 0xbe3c7496, 0xbd3c7db4, 0xbb912ca0, 0x3e4f5476,
      0xbd8a65c2, 0xbdb281f2, 0xbdbcc6b6, 0xbe1ff856, 0xbe1b3afe, 0x3dcc5820,
      0xbe2882ac, 0x3e2b57ec, 0x3e358cf2, 0xbe54696c, 0xbd340870, 0x3e45d54a,
      0x3c07b640, 0xbe567290, 0xbdc76b34, 0x3dddf448,
  };

  // 2
  uint32_t bias_vals[] = {
      0x3c40ac50,
      0xbdeac81f,
  };

  // 1,3,2,2
  uint32_t output_exp_vals[] = {
      0xbed199d0, 0xbee4ca5f, 0xbe0febf4, 0xbe2b0406, 0xbefd7c85, 0xbe846eb2,
      0xbef1750b, 0x3e52c1ce, 0xbeb43421, 0xbe83d2bf, 0xbeeeaca6, 0xbe243821,
  };

  // 1,3,2,2
  uint32_t output_relu_exp_vals[] = {
      0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3e52c1ce, 0x0, 0x0, 0x0, 0x0,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, VALID_PADDING, NULL);
}

void test_valid_padding_non_zero_strides_small_with_clip() {
  input_set *set = &small_input;
  strides_input_set *strides = &small_non0_strides;

  // 1,4,3,5
  uint32_t input_vals[] = {
      0x3f15a7d8, 0x3f55d27f, 0x3f04e322, 0x3f0bc669, 0x3f2eec77, 0x3d07cb10,
      0x3e528880, 0x3f1da880, 0x3e3fb0a8, 0x3f40bb0b, 0x3e0e0bcc, 0x3f17da1f,
      0x3f4f6880, 0x3d0b5ba0, 0x3f5ba1ae, 0x3e780c74, 0x3de61650, 0x3f7ae7a4,
      0x3f71ba1f, 0x3f2fdc52, 0x3f50c293, 0x3e1d23d0, 0x3deae1a8, 0x3f615378,
      0x3ba82d80, 0x3f4b0c93, 0x3e77825c, 0x3ea22f0a, 0x3aa20200, 0x3e33de00,
      0x3e8e771c, 0x3f39eaa3, 0x3f324e26, 0x3f17f541, 0x3f3fe98e, 0x3ef6c34e,
      0x3f3379fe, 0x3f6a0de8, 0x3ed4dfce, 0x3f1aca63, 0x3f51dd20, 0x3e50b72c,
      0x3f6f62f4, 0x3ed5df52, 0x3de131a8, 0x3f7f3fc1, 0x3f26ab72, 0x3f70f111,
      0x3e8ad072, 0x3e592e30, 0x3f32cd09, 0x3f4644b7, 0x3f19794f, 0x3f313923,
      0x3f786a79, 0x3f114ab9, 0x3edfb038, 0x3d858c20, 0x3e50bd98, 0x3f563faa,
  };

  // 2,2,5,2
  uint32_t kernel_vals[] = {
      0xbadb6d80, 0x3da65588, 0xbdb06154, 0xbd13a074, 0xbca28f68, 0xbe1025bd,
      0xbe5abfd6, 0xbe146a76, 0x3db50ae0, 0xbe05a938, 0x3df5a9a0, 0x3e3e2300,
      0x3d759a80, 0x3cc1d430, 0x3d725b60, 0xbd67df54, 0xbe17a51c, 0xbdd2ced4,
      0x3e0963a4, 0x3e44d336, 0xbe3c7496, 0xbd3c7db4, 0xbb912ca0, 0x3e4f5476,
      0xbd8a65c2, 0xbdb281f2, 0xbdbcc6b6, 0xbe1ff856, 0xbe1b3afe, 0x3dcc5820,
      0xbe2882ac, 0x3e2b57ec, 0x3e358cf2, 0xbe54696c, 0xbd340870, 0x3e45d54a,
      0x3c07b640, 0xbe567290, 0xbdc76b34, 0x3dddf448,
  };

  // 2
  uint32_t bias_vals[] = {
      0x3c40ac50,
      0xbdeac81f,
  };

  // 1,3,2,2
  uint32_t output_exp_vals[] = {
      0xbed199d0, 0xbee4ca5f, 0xbe0febf4, 0xbe2b0406, 0xbefd7c85, 0xbe846eb2,
      0xbef1750b, 0x3e52c1ce, 0xbeb43421, 0xbe83d2bf, 0xbeeeaca6, 0xbe243821,
  };

  uint32_t clip_value = 0x3e4ccccd;
  // 1,3,2,2
  uint32_t output_relu_exp_vals[] = {
      0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3e4ccccd, 0x0, 0x0, 0x0, 0x0,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, VALID_PADDING, (void *)&clip_value);
}

void test_valid_padding_zero_strides_small() {
  input_set *set = &small_input;
  strides_input_set *strides = &zero_strides;

  // 1,4,3,5
  uint32_t input_vals[] = {
      0x3e83171a, 0x3f5ba100, 0x3f5727e7, 0x3ec9e104, 0x3f7090d1, 0x3f3c9175,
      0x3ee5aee2, 0x3efe572a, 0x3e9832f2, 0x3e979464, 0x3ca6eaa0, 0x3f038eb2,
      0x3e80c192, 0x3eceb0c4, 0x3f73959b, 0x3e5088c0, 0x3ee9ea48, 0x3def4bf8,
      0x3ecdbc7c, 0x3f2e5b0e, 0x3ebfea62, 0x3f19e7a2, 0x3d036000, 0x3d523a30,
      0x3d47c290, 0x3e834c8a, 0x3f516330, 0x3d9fbc50, 0x3d8bd6f0, 0x3f50fc3a,
      0x3ea67152, 0x3f793f5d, 0x3ecc12e4, 0x3ed7f0f6, 0x3ee27582, 0x3db8b518,
      0x3f514983, 0x3f46322b, 0x3f01a7c3, 0x3e8db848, 0x3ec98b80, 0x3f798662,
      0x3f4533ba, 0x3e8bb4a4, 0x3f665b0a, 0x3f1b5c5f, 0x3eed7c54, 0x3c91afe0,
      0x3f3037ae, 0x3dc482b0, 0x3e327314, 0x3d503cd0, 0x3f4fd817, 0x3d4cae30,
      0x3f0dbc2b, 0x3ef2dcbc, 0x3e86a2e0, 0x3ddeb5d8, 0x3eeeb928, 0x3eca66d0,
  };

  // 4,3,5,2
  uint32_t kernel_vals[] = {
      0xba7d6000, 0x3d6f4388, 0xbdfc970a, 0x3d4fd1fc, 0xbcaa76f8, 0x3cc5f330,
      0xbd9a56a4, 0x3d840448, 0xbd6a842e, 0xbbe916f0, 0x3d8dd556, 0x3d54a8b0,
      0xbdaf1ac2, 0x3ddb109a, 0x3c5fd070, 0xbcb22910, 0x3de34624, 0x3cdd5a80,
      0xbb98a3f0, 0x3c9eca70, 0xbdd99c0c, 0xbd9c9d8d, 0xbd59fae2, 0x3de6143c,
      0x3def676c, 0xbcae097c, 0x3d6bf4d8, 0xbdce4036, 0x3da918da, 0xbd588e5a,
      0xbdc2943f, 0x3d1fc068, 0x3b9cb4e0, 0x3de94a92, 0xbdf545a3, 0xbd720ed0,
      0x3d802534, 0xbd983e82, 0x3e036c17, 0x3d9ce2f0, 0xbd4baa92, 0x3d912554,
      0x3d510cc0, 0x3cd27970, 0xbda67275, 0x3d4b5998, 0x3bde7ac0, 0x3df8bae2,
      0x3d51de2c, 0xbda0e525, 0x3d0dcca0, 0xbc857c3c, 0x3d9ea4b8, 0xbde5e9f4,
      0xbd05df20, 0xbd9c9a3c, 0xbd878d22, 0x3db2278e, 0x3c8edf30, 0x3defe79c,
      0xbb27a200, 0x3ba18540, 0xbdb33ea4, 0x3c429450, 0xbd4e1f72, 0x3d41d9b0,
      0x3cb2ed40, 0xbd6690fa, 0x3db14610, 0x3db530b8, 0x3dd1a2f4, 0x3cf50c00,
      0xbd6644da, 0xbda56848, 0x3de47024, 0x3d85b500, 0xbdf6dd66, 0xbd0516b6,
      0xbd9bad42, 0xbddef5da, 0xbc3bb558, 0x3da4065e, 0x3d4010e8, 0xbdf66796,
      0xbdab603a, 0x3dbd45fa, 0xbdbf0b1b, 0xbdfbdce1, 0x3cc51300, 0xbd039b9e,
      0x3d0becc0, 0x3dfddf06, 0x3ddb8d0c, 0x3deae9de, 0xbd736b86, 0x3d45b890,
      0xbd884dcc, 0xbb1812c0, 0xbdccfcc7, 0x3bbee240, 0xbd1fcec2, 0xbdcbbfe1,
      0xbcd9a694, 0x3dc9092c, 0xbdb8b7ae, 0xbe02d742, 0x3da6d138, 0xbdc71b02,
      0xbdb73fbc, 0x3c9a4d70, 0xbccfe258, 0xbd41c938, 0x3dc5d9b2, 0xbe03defc,
      0xbdf79f67, 0xbdab833c, 0x3d9b0552, 0xbdcbf180, 0xbd88951c, 0x3cfad4d8,
  };

  // 2
  uint32_t bias_vals[] = {
      0x3d4f8340,
      0x3bfd65c0,
  };

  // 1,1,1,2
  uint32_t output_exp_vals[] = {
      0xbd56b2c9,
      0x3dd5aea1,
  };

  // 1,1,1,2
  uint32_t output_relu_exp_vals[] = {
      0x0,
      0x3dd5aea1,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, VALID_PADDING, NULL);
}

void test_same_padding_non_zero_strides_small() {
  input_set *set = &small_input;
  strides_input_set *strides = &small_non0_strides;

  // 1,4,3,5
  uint32_t input_vals[] = {
      0x3f15a7d8, 0x3f55d27f, 0x3f04e322, 0x3f0bc669, 0x3f2eec77, 0x3d07cb10,
      0x3e528880, 0x3f1da880, 0x3e3fb0a8, 0x3f40bb0b, 0x3e0e0bcc, 0x3f17da1f,
      0x3f4f6880, 0x3d0b5ba0, 0x3f5ba1ae, 0x3e780c74, 0x3de61650, 0x3f7ae7a4,
      0x3f71ba1f, 0x3f2fdc52, 0x3f50c293, 0x3e1d23d0, 0x3deae1a8, 0x3f615378,
      0x3ba82d80, 0x3f4b0c93, 0x3e77825c, 0x3ea22f0a, 0x3aa20200, 0x3e33de00,
      0x3e8e771c, 0x3f39eaa3, 0x3f324e26, 0x3f17f541, 0x3f3fe98e, 0x3ef6c34e,
      0x3f3379fe, 0x3f6a0de8, 0x3ed4dfce, 0x3f1aca63, 0x3f51dd20, 0x3e50b72c,
      0x3f6f62f4, 0x3ed5df52, 0x3de131a8, 0x3f7f3fc1, 0x3f26ab72, 0x3f70f111,
      0x3e8ad072, 0x3e592e30, 0x3f32cd09, 0x3f4644b7, 0x3f19794f, 0x3f313923,
      0x3f786a79, 0x3f114ab9, 0x3edfb038, 0x3d858c20, 0x3e50bd98, 0x3f563faa,
  };

  // 2,2,5,2
  uint32_t kernel_vals[] = {
      0xbadb6d80, 0x3da65588, 0xbdb06154, 0xbd13a074, 0xbca28f68, 0xbe1025bd,
      0xbe5abfd6, 0xbe146a76, 0x3db50ae0, 0xbe05a938, 0x3df5a9a0, 0x3e3e2300,
      0x3d759a80, 0x3cc1d430, 0x3d725b60, 0xbd67df54, 0xbe17a51c, 0xbdd2ced4,
      0x3e0963a4, 0x3e44d336, 0xbe3c7496, 0xbd3c7db4, 0xbb912ca0, 0x3e4f5476,
      0xbd8a65c2, 0xbdb281f2, 0xbdbcc6b6, 0xbe1ff856, 0xbe1b3afe, 0x3dcc5820,
      0xbe2882ac, 0x3e2b57ec, 0x3e358cf2, 0xbe54696c, 0xbd340870, 0x3e45d54a,
      0x3c07b640, 0xbe567290, 0xbdc76b34, 0x3dddf448,
  };

  // 2
  uint32_t bias_vals[] = {
      0x3c40ac50,
      0xbdeac81f,
  };

  // 1,4,3,2
  uint32_t output_exp_vals[] = {
      0xbed199d0, 0xbee4ca5f, 0xbe0febf4, 0xbe2b0406, 0xbe3aa069, 0xbeb5053e,
      0xbefd7c85, 0xbe846eb2, 0xbef1750b, 0x3e52c1ce, 0xbe8b07f1, 0xbe848b7f,
      0xbeb43421, 0xbe83d2bf, 0xbeeeaca6, 0xbe243821, 0xbeb92f25, 0xbe1d4e2d,
      0x3db760f2, 0xbcdf50da, 0x3d4d4b44, 0xbe14cbd2, 0x3b1861e5, 0xbe6dd479,
  };

  // 1,4,3,2
  uint32_t output_relu_exp_vals[] = {
      0x0, 0x0,        0x0,        0x0, 0x0,        0x0, 0x0,        0x0,
      0x0, 0x3e52c1ce, 0x0,        0x0, 0x0,        0x0, 0x0,        0x0,
      0x0, 0x0,        0x3db760f2, 0x0, 0x3d4d4b44, 0x0, 0x3b1861e5, 0x0,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, SAME_PADDING, NULL);
}

void test_valid_padding_non_zero_strides_medium() {
  input_set *set = &medium_input;
  strides_input_set *strides = &medium_non0_strides;

  // 3,10,8,5
  uint32_t input_vals[] = {
      0x3f1c5cd1, 0x3f791683, 0x3f2c91ac, 0x3f4e9dfc, 0x3f7ea33e, 0x3ec54b8e,
      0x3f1e4b8c, 0x3d5833b0, 0x3f0d60ae, 0x3f19e648, 0x3e83f3a8, 0x3c473900,
      0x3f1d2739, 0x3f07625d, 0x3eeef18e, 0x3f1195d9, 0x3eb71e74, 0x3e3b6ac8,
      0x3e6479c8, 0x3f7dff2d, 0x3f693fc2, 0x3e233e04, 0x3ee4515a, 0x3e949d42,
      0x3e92990a, 0x3e25d8f0, 0x3f703971, 0x3f1076e9, 0x3eb52618, 0x3eeb7c5e,
      0x3f05f190, 0x3ed5b900, 0x3f6d0c84, 0x3c53b800, 0x3e8fe576, 0x3ea19870,
      0x3d353270, 0x3e85db72, 0x3f06a6e9, 0x3edca954, 0x3f7d9b72, 0x3eefe830,
      0x3f52005d, 0x3f16a350, 0x3f1bd1c3, 0x3cd1dda0, 0x3f5062ba, 0x3edf7ab6,
      0x3effc884, 0x3f11188a, 0x3ca952e0, 0x3f214124, 0x3e866b22, 0x3f2955c8,
      0x3ecfa5e4, 0x3f7e2118, 0x3f287963, 0x3d844d18, 0x3f7973f3, 0x38a72000,
      0x3e3c1010, 0x3f0be345, 0x3d28ec20, 0x3f22087d, 0x3f068ddb, 0x3f188a64,
      0x3f2fb886, 0x3f7d0194, 0x3ea24914, 0x3ef86ab2, 0x3ee9e7d8, 0x3ec1a006,
      0x3ec0275a, 0x3e96a4e0, 0x3f134183, 0x3eca1262, 0x3e9a29c4, 0x3f0664b9,
      0x3e386b68, 0x3f62147d, 0x3ec6d2ae, 0x3d05b790, 0x3f22b3cf, 0x3e1d126c,
      0x3f7c6ace, 0x3f51561e, 0x3dfc9540, 0x3f570177, 0x3ed6c570, 0x3e503d48,
      0x3f061e36, 0x3f3779bc, 0x3f6d3ea4, 0x3ed2b8f0, 0x3ef0098a, 0x3c582640,
      0x3e50d684, 0x3f67cf69, 0x3f38f141, 0x3e996d74, 0x3e51bc50, 0x3f125d1c,
      0x3e04aaf8, 0x3e929380, 0x3d207840, 0x3ea8bac2, 0x3f28db44, 0x3ed6fc08,
      0x3f660ac5, 0x3e0b7044, 0x3f406965, 0x3f09101a, 0x3e5107c4, 0x3e18bd60,
      0x3f7822c5, 0x3e34aae4, 0x3e340b80, 0x3e5b77c8, 0x3f005d22, 0x3f4aa1bf,
      0x3f78b197, 0x3f7a68f8, 0x3f1e5ccd, 0x3f730fcf, 0x3f44c15e, 0x3ec6fb36,
      0x3e5645b4, 0x3f78259e, 0x3f7f4803, 0x3f472298, 0x3ed2083a, 0x3f6947b3,
      0x3dcbab68, 0x3e576af0, 0x3dd27418, 0x3f644ba1, 0x3dd16648, 0x3f4d7757,
      0x3f16ccee, 0x3f0d7145, 0x3f405573, 0x3ec26b6a, 0x3e768b88, 0x3f0f1ea5,
      0x3f7721a8, 0x3f6c8f35, 0x3f45a0c5, 0x3ece2346, 0x3e87f956, 0x3e624290,
      0x3f4a0a77, 0x3e976742, 0x3f659a49, 0x3ea78da6, 0x3f71de3a, 0x3eb25766,
      0x3f6b8687, 0x3ec619b4, 0x3f22ab58, 0x3f21a4b1, 0x3e2c4d58, 0x3e1fa9f4,
      0x3f0ba7b7, 0x3e1bf918, 0x3f5a0db7, 0x3eecf3ec, 0x3da412f0, 0x3e1a1868,
      0x3f151e94, 0x3e925530, 0x3f69f007, 0x3e8c9090, 0x3f6cf576, 0x3f376386,
      0x3f3b02a1, 0x3eaa14d2, 0x3f14b0f6, 0x3edee82c, 0x3e9b1e38, 0x3d687920,
      0x3d14bf10, 0x3f75db54, 0x3e098730, 0x3f6a5273, 0x3eefbec2, 0x3f347889,
      0x3e85c90e, 0x3f257acf, 0x3eef270c, 0x3f2ab1ce, 0x3f7c9ab6, 0x3f2dc6c3,
      0x3e13f588, 0x3f3a1342, 0x3f2662f1, 0x3eb71f00, 0x3ebfd1aa, 0x3dd372d0,
      0x3f7ec25c, 0x3f6b1c87, 0x3db01a98, 0x3ec87bb6, 0x3f07cc33, 0x3eb1bfb8,
      0x3f7dc7b5, 0x3d3e5780, 0x3f5e1e6c, 0x3f657be6, 0x3f458e66, 0x3f7472eb,
      0x3f2013a4, 0x3de66a18, 0x3eb78e5c, 0x3eb6c6ca, 0x3f55f02e, 0x3eec9da2,
      0x3f0d9866, 0x3f3c3ed0, 0x3eda9af6, 0x3f59fdf0, 0x3e7d7250, 0x3f785f54,
      0x3f6dfe3d, 0x3ed49d8a, 0x3edef7e8, 0x3f19df75, 0x3edcd28e, 0x3f54e5bc,
      0x3efe68da, 0x3de553f8, 0x3f309bf0, 0x3f636253, 0x3e736cbc, 0x3e9f4a92,
      0x3ea0f742, 0x3f65cd03, 0x3eb129be, 0x3ee3eff4, 0x3f1e0064, 0x3ee342a2,
      0x3f63607e, 0x3f670900, 0x3eaf70d0, 0x3f04d2b6, 0x3e6b5274, 0x3ed9ae54,
      0x3c85d0a0, 0x3dc89400, 0x3f51234d, 0x3f416e0c, 0x3d722410, 0x3edb1ab6,
      0x3f000d8a, 0x3ecc2f2a, 0x3f2cee76, 0x3d456140, 0x3ed308d6, 0x3f604fcb,
      0x3f0cd5fb, 0x3f14a72e, 0x3f778324, 0x3f297b9e, 0x3f6bd78d, 0x3ea0c53e,
      0x3f1eeb76, 0x3f389544, 0x3f3237c9, 0x3f0bf147, 0x3da67288, 0x3f707120,
      0x3f37cbd0, 0x3f6247d3, 0x3f1d125e, 0x3f33cc23, 0x3f2e9f22, 0x3d8a0380,
      0x3ed9dc0e, 0x3e912c0c, 0x3f1060ed, 0x3d7c0730, 0x3f7687af, 0x3ef5d92a,
      0x3f609ead, 0x3e9979e6, 0x3e0c595c, 0x3f79428a, 0x3f57a7e5, 0x3e9579ea,
      0x3ea971f4, 0x3f386a7c, 0x3f7390ed, 0x3ebabbb8, 0x3e1c5678, 0x3f217988,
      0x3f117199, 0x3da02b40, 0x3f703588, 0x3f13b60f, 0x3ed1b5fc, 0x3f3e6cae,
      0x3e9f67bc, 0x3e2b1aac, 0x3f4cb8a9, 0x3f53a519, 0x3a27c800, 0x3e1fdd6c,
      0x3ee44c5c, 0x3d498a60, 0x3f074015, 0x3d1df6b0, 0x3f79362b, 0x3ef24000,
      0x3f73b535, 0x3f3059ad, 0x3f4e7e34, 0x3e92007e, 0x3f391d68, 0x3f2e0bab,
      0x3f37c4e5, 0x3f51dfe9, 0x3e8b1f3e, 0x3f5782b3, 0x3e194948, 0x3ec12fac,
      0x3d4326e0, 0x3f430f94, 0x3ed76c4c, 0x3ec8db0e, 0x3f376190, 0x3f309640,
      0x3e89938c, 0x3dafa2c8, 0x3f6f0cdc, 0x3f5e7499, 0x3de06a60, 0x3e81ea12,
      0x3f3f6c4a, 0x3dee5a08, 0x3f05e3ee, 0x3f60ef76, 0x3ee99b78, 0x3f2649f3,
      0x3eb52d52, 0x3f190c77, 0x3f282a49, 0x3ee764b6, 0x3f336ae2, 0x3f29fb42,
      0x3f04a6a5, 0x3f7e9092, 0x3de25378, 0x3e4429ec, 0x3d7dc720, 0x3f589f7f,
      0x3be58200, 0x3f6ab5a3, 0x3f526157, 0x3f462b5c, 0x3e93eb02, 0x3c98e0e0,
      0x3e8f12a6, 0x3f793c60, 0x3f42a14e, 0x3e6fd614, 0x3f568a83, 0x3f2d674c,
      0x3f0b1b53, 0x3f4f62e5, 0x3f37c6c6, 0x3f71a430, 0x3f6f5512, 0x3d0711a0,
      0x3e39936c, 0x3d84cc60, 0x3f4e6c4c, 0x3f408d71, 0x3f59dace, 0x3f7f7cf6,
      0x3f020d5a, 0x3f551925, 0x3f121f4a, 0x3e046080, 0x3e508190, 0x3e8a4494,
      0x3da183a0, 0x3f6ce8e2, 0x3f1d6c73, 0x3f7fc99f, 0x3f547bfb, 0x3f5ca5ed,
      0x3f1134e1, 0x3f1299c6, 0x3ca51ba0, 0x3d0d3b60, 0x3ce4c600, 0x3e898f56,
      0x3e882ec2, 0x3d5f4b20, 0x3f4d9c4e, 0x3f1c986d, 0x3e309f48, 0x3e4020c8,
      0x3eccd314, 0x3e98f244, 0x3f0d6465, 0x3e9a5166, 0x3e56e3c8, 0x3f64e98c,
      0x3f676a36, 0x3f6d4529, 0x3f16aa31, 0x3f389366, 0x3f5e765d, 0x3c59d440,
      0x3e0b9e44, 0x3e9b588e, 0x3eee1028, 0x3e24fd00, 0x3e396a8c, 0x3f29ffd0,
      0x3ed77eb4, 0x3ce35560, 0x3f3b759d, 0x3dbee238, 0x3f388f70, 0x3cd5b620,
      0x3e58c9d8, 0x3ddccad0, 0x3ed136da, 0x3f218c2b, 0x3ce9fdc0, 0x3f33e880,
      0x3d983f38, 0x3f63bf3b, 0x3f7f446e, 0x3f31951d, 0x3ea92bc8, 0x3f272096,
      0x3f703270, 0x3f68cdc5, 0x3e93a406, 0x3f58596d, 0x3f01e193, 0x3d848d28,
      0x3f6dd176, 0x3e488128, 0x3f653069, 0x3e88930a, 0x3ef4e888, 0x3ee01726,
      0x3e9afca8, 0x3f129ec2, 0x3c916480, 0x3f4b21b5, 0x3f6b5aef, 0x3dbacf50,
      0x3eebc1c2, 0x3e9ce4a6, 0x3f386c0d, 0x3d0b7160, 0x3dcd6a08, 0x3eaebd18,
      0x3f2ac07f, 0x3f68d865, 0x3e6bb5b4, 0x3f4c9777, 0x3ef30e1e, 0x3d185e40,
      0x3ef8e626, 0x3ec93f16, 0x3f410e14, 0x3f174c76, 0x3c69c140, 0x3e27fdac,
      0x3eb3f416, 0x3f110cd0, 0x3df1d108, 0x3f103d5c, 0x3f54ad08, 0x3f763ba1,
      0x3e6850a8, 0x3ec2d12e, 0x3e0f2730, 0x3ea836c0, 0x3f3c5f87, 0x3f2a95f2,
      0x3dac3c60, 0x3d7dc9a0, 0x3f4246d1, 0x3f526877, 0x3f05c01c, 0x3f4e8f85,
      0x3e9d5de2, 0x3ec0e98a, 0x3f57111a, 0x3e121668, 0x3e39f098, 0x3ed6c8a6,
      0x3f36697c, 0x3e0188c0, 0x3f1b8b57, 0x3f74ef01, 0x3d3af8c0, 0x3f732064,
      0x3ee2598e, 0x3f748cb4, 0x3ebe0214, 0x3e2a1f98, 0x3f770581, 0x3f12b0ef,
      0x3c064c00, 0x3e5ab018, 0x3f086487, 0x3d382fd0, 0x3f214ab6, 0x3b37c200,
      0x3f3d7f7d, 0x3ce99680, 0x3e2d5274, 0x3f41d260, 0x3f106666, 0x3f131506,
      0x3edc04a0, 0x3e96270a, 0x3e9a22c8, 0x3f606309, 0x3f58a266, 0x3e356674,
      0x3e397f40, 0x3eb2cea4, 0x3e487408, 0x3f356fa6, 0x3e1d1000, 0x3f7ad0e6,
      0x3f6b1a57, 0x3f1c2d74, 0x3c962960, 0x3f324ed7, 0x3ee963f0, 0x3ebb4b2a,
      0x3f59fb6c, 0x3f50f572, 0x3debd678, 0x3e2c5ea0, 0x3e1c57f8, 0x3f2776e9,
      0x3ecfcaaa, 0x3df7fa50, 0x3e8e152c, 0x3f56ca02, 0x3f1bb359, 0x3e8edafc,
      0x3f48fefc, 0x3e653078, 0x3f56ad08, 0x3f7fb57c, 0x3f514972, 0x3f17ee83,
      0x3f3438ff, 0x3eafba14, 0x3f536375, 0x3f5d5ab2, 0x3f74923d, 0x3d82d098,
      0x3eccd20a, 0x3e600990, 0x3d77f330, 0x3ebb9286, 0x3d0a7040, 0x3e063498,
      0x3f0b105a, 0x3ee8da04, 0x3e845920, 0x3be94a00, 0x3da8dd30, 0x3f416444,
      0x3f691d4c, 0x3ee84ee0, 0x3f371b6a, 0x3f302264, 0x3f70e269, 0x3f31929d,
      0x3f00a5c9, 0x3f3f4a4a, 0x3e92a382, 0x3b260a00, 0x3f6bd609, 0x3f619e54,
      0x3e4201ec, 0x3f65e675, 0x3f45500c, 0x3ea844e6, 0x3b007400, 0x3f1fbdc3,
      0x3dcc11d8, 0x3eb5acd0, 0x3e658f20, 0x3eedc5ca, 0x3f5e7cd6, 0x3edc9d48,
      0x3c2a3840, 0x3f6e1185, 0x3f389eff, 0x3f12993c, 0x3f02c385, 0x3f1c69a5,
      0x3f355a74, 0x3e660ce0, 0x3ec9d5ca, 0x3f41c905, 0x3e69429c, 0x3f7ae0b5,
      0x3e7bd4f8, 0x3f0974fa, 0x3f679826, 0x3f1223a0, 0x3e9511d6, 0x3f3ddbb8,
      0x3ee48cba, 0x3f56aea3, 0x3e3395e8, 0x3e374304, 0x3cab4600, 0x3f4072a9,
      0x3f385858, 0x3dfd6908, 0x3ee1fe56, 0x3f1e5491, 0x3e2458a0, 0x3e8a7f62,
      0x3eea4734, 0x3cef49e0, 0x3de73ba8, 0x3f40e8bc, 0x3e125b3c, 0x3eaa00c6,
      0x3f322c3e, 0x3f3f9fb9, 0x3f4824d1, 0x3d57d540, 0x3f4adcd9, 0x3f5e694c,
      0x3d947048, 0x3e49cacc, 0x3f065972, 0x3f0b0932, 0x3f1eba2f, 0x3f4eb1ec,
      0x3f688528, 0x3f7c2509, 0x3f63e958, 0x3ee6b498, 0x3cda97c0, 0x3f7db83b,
      0x3ecf5fee, 0x3f17fcd2, 0x3f3f7695, 0x3da34278, 0x3e9b13d4, 0x3ef53908,
      0x3eadae28, 0x3e3b01e0, 0x3f01e2e6, 0x3eb84bbc, 0x3e0beec4, 0x3f56b2de,
      0x3f7d3712, 0x3f64dcce, 0x3f3fca1d, 0x3f274c18, 0x3f06fe9c, 0x3b395900,
      0x3d930d08, 0x3dddc708, 0x3f51265e, 0x3f180648, 0x3cac41a0, 0x3e8dcf08,
      0x3f329e5e, 0x3be6c100, 0x3eadaa24, 0x3ec381bc, 0x3f706b90, 0x3f1ddcee,
      0x3e177d4c, 0x3f007599, 0x3f688e5c, 0x3ed511c8, 0x3f50aea5, 0x3f5ecf83,
      0x3ecec730, 0x3f0cd0ec, 0x3f350747, 0x3d868bf0, 0x3f1e4e4d, 0x3eea3c4e,
      0x3f312869, 0x3c9ba7c0, 0x3dad46c0, 0x3ef00a16, 0x3de16750, 0x3e08b354,
      0x3eaf7e2e, 0x3f397d10, 0x3f16e18d, 0x3efe6cf6, 0x3f2eae06, 0x3e7bfd6c,
      0x3f2e1b63, 0x3ed913c8, 0x3dae08e0, 0x3ee364c8, 0x3f6be96e, 0x3d0dca70,
      0x3e1a2144, 0x3ec8ed30, 0x3f334e30, 0x3f352007, 0x3e627a74, 0x3d0649a0,
      0x3f4a2e92, 0x3f5a6628, 0x3e0a67f4, 0x3d2e1fc0, 0x3f542fc6, 0x3e1bfb0c,
      0x3f37a39a, 0x3f14a316, 0x3e113104, 0x3ee18d5e, 0x3e26e2b0, 0x3eea393c,
      0x3efd0ec4, 0x3e8c7e2e, 0x3ea0ecc4, 0x3ec17da6, 0x3ef03b0a, 0x3e96d13c,
      0x3f69ecc8, 0x3ebdec2a, 0x3eb9d25a, 0x3e3ee23c, 0x3e5b9bd8, 0x3ecd0b00,
      0x3e68be50, 0x3f15ab36, 0x3ea3677e, 0x3eab0ca0, 0x3f498457, 0x3eb42d54,
      0x3f154000, 0x3f532de5, 0x3f64e49b, 0x3f798f25, 0x3d90fbd0, 0x3f40e88a,
      0x3e99584a, 0x3e10f8bc, 0x3f3a556f, 0x3f5fc9b9, 0x3e1f4b0c, 0x3f4f7e3b,
      0x3e1347bc, 0x3d569bf0, 0x3f344491, 0x3eedef0c, 0x3e9a7ce2, 0x3eb104e4,
      0x3f40699a, 0x3e7dc994, 0x3f4796db, 0x3ec460be, 0x3f023a0b, 0x3f30d2d9,
      0x3ea89fb2, 0x3f6eb763, 0x3f02b960, 0x3f22d7d7, 0x3f633559, 0x3f396cf4,
      0x3eed5372, 0x3d289f00, 0x3f03d74e, 0x3bde4300, 0x3ea133aa, 0x3e57bf18,
      0x3efd71a0, 0x3ec25fce, 0x3f4eb8d8, 0x3f21c0b7, 0x3f24cb93, 0x3ee63240,
      0x3e498c78, 0x3f68c5f7, 0x3f233e64, 0x3f48e4a2, 0x3ee9007c, 0x3ed210b4,
      0x3d0e9cf0, 0x3e3db310, 0x3f20106f, 0x3ddc15d8, 0x3ddf9fd8, 0x3f172511,
      0x3e343374, 0x3f07d4e2, 0x3e8eb102, 0x3f2c11d2, 0x3df2d818, 0x3f1c0428,
      0x3f6fcb0b, 0x3e886f24, 0x3e9f92a4, 0x3f39de65, 0x3f4f3400, 0x3ea35fc4,
      0x3b975a80, 0x3e944572, 0x3f2fd65e, 0x3eadc1a8, 0x3ebe30c2, 0x3f0c351c,
      0x3eca9b2c, 0x3edfe3ae, 0x3f3de46d, 0x3f6e48ee, 0x3f5741c0, 0x3ea13e1e,
      0x3db148e8, 0x3e42e970, 0x3dbbcc98, 0x3f33f355, 0x3f46238f, 0x3f333266,
      0x3f7a7ccd, 0x3f093f70, 0x3e865db0, 0x3ecf5c38, 0x3f53867b, 0x3f5b34a5,
      0x3f7b540b, 0x3d276490, 0x3f610d47, 0x3ed3393c, 0x3f17f9af, 0x3e8d37c2,
      0x3dc5dc40, 0x3f1d351b, 0x3f3c54cb, 0x3f38f23d, 0x3f7ab619, 0x3f025fd2,
      0x3ed871b0, 0x3f2f71c2, 0x3f2c9067, 0x3f719461, 0x3f580d8b, 0x3e5844e4,
      0x3e999bba, 0x3e51dec8, 0x3f77155f, 0x3e9853b8, 0x3d2cb3a0, 0x3eee3a78,
      0x3efdc488, 0x3ee7925a, 0x3e6965d4, 0x3eb551b0, 0x3f7f6a87, 0x3f2221ee,
      0x3f73cb06, 0x3e6edecc, 0x3d177cc0, 0x3f009ab3, 0x3ec63798, 0x3e9074b2,
      0x3ee9d6a2, 0x3ed0921a, 0x3dd0c7a0, 0x3f7abeb9, 0x3ea85eb0, 0x3cce36e0,
      0x3f021792, 0x3eed5492, 0x3e9d46ee, 0x3f3069ed, 0x3d70c0f0, 0x3eef809c,
      0x3f3fadd7, 0x3d5afe50, 0x3f1b0a26, 0x3f7e0083, 0x3e823f9c, 0x3eac6f80,
      0x3e861fe0, 0x3f469033, 0x3e85e8d4, 0x3f58d714, 0x3e73dfc0, 0x3d134840,
      0x3f79a410, 0x3ed1d3b0, 0x3f481647, 0x3f697191, 0x3f697a69, 0x3e3f7e5c,
      0x3ebb81aa, 0x3ec0ed6e, 0x3f43b6a8, 0x3e8d1448, 0x3f43eeda, 0x3d78d1c0,
      0x3e9ada58, 0x3ed481e6, 0x3e17c540, 0x3f2ad303, 0x3ed2de74, 0x3f5078eb,
      0x3f068a15, 0x3ee453a4, 0x3f43875a, 0x3f272ed5, 0x3f6f5cf5, 0x3ee4ecbc,
      0x3e270294, 0x3f3cb1ae, 0x3e980c9c, 0x3d2cc5b0, 0x3f1db79a, 0x3e748b78,
      0x3ea03cf4, 0x3ef516de, 0x3ec8b058, 0x3f048c8f, 0x3f414175, 0x3dd6ace8,
      0x3efec682, 0x3ee83e1e, 0x3e4a2bc0, 0x3dff2f70, 0x3f667302, 0x3b134c00,
      0x3ea34ffa, 0x3f17b711, 0x3f3c5d23, 0x3e5c0a14, 0x3f5d050b, 0x3f6d4657,
      0x3f03590a, 0x3f5d57b0, 0x3f51d579, 0x3f676519, 0x3eda2f96, 0x3f084691,
      0x3eff6d00, 0x3cc1f8e0, 0x3f574567, 0x3f6c864d, 0x3e9e7106, 0x3f38d631,
      0x3ed3b61e, 0x3eabee16, 0x3f00a1fe, 0x3f20cbe7, 0x3e6132e4, 0x3f015b4e,
      0x3f42e35a, 0x3ee43b3a, 0x3dc3f248, 0x3f63d109, 0x3f1625a3, 0x3d012ed0,
      0x3e55597c, 0x3f26431d, 0x3f7c5832, 0x3f0a58b4, 0x3ec0ab44, 0x3eaae0dc,
      0x3f455cc3, 0x3f1e6ae1, 0x3f285567, 0x3ecbef02, 0x3f293ed2, 0x3ee3d38e,
      0x3f74a4b0, 0x3ee37eec, 0x3ea597fc, 0x3f389e84, 0x3de48e80, 0x3b5ff100,
      0x3f1f5907, 0x3f35c290, 0x3ef6e438, 0x3eff11c4, 0x3ee3e418, 0x3f7eaab3,
      0x3e8b5a02, 0x3f79f6c1, 0x3f002e84, 0x3f196ec6, 0x3e34b598, 0x3ee25420,
      0x3e1da0c4, 0x3d8b0de8, 0x3d85f3e0, 0x3ef205f2, 0x3eb61442, 0x3f3113b1,
      0x3e15dd68, 0x3f3e49f3, 0x3e9e4766, 0x3cf98420, 0x3f734957, 0x3f766464,
      0x3e9e6f48, 0x3f704f9d, 0x3ed5b958, 0x3ed37c72, 0x3f5fb886, 0x3ea368c6,
      0x3f6d92b7, 0x3d9c2f30, 0x3f659ed0, 0x3f46f020, 0x3dd38b30, 0x3ef904e2,
      0x3f4aa328, 0x3f021c29, 0x3f58ac51, 0x3f345424, 0x3ea74316, 0x3f446948,
      0x3e009768, 0x3f314677, 0x3e0b7478, 0x3e43a70c, 0x3f77e8cd, 0x3e671bc0,
      0x3f480976, 0x3ec9ace4, 0x3f5b96cd, 0x3eaa08d4, 0x3f2fa4c2, 0x3f54c6f5,
      0x3f38dc93, 0x3eee5ef4, 0x3f068253, 0x3e91375e, 0x3ec4324a, 0x3f606da6,
      0x3f788d37, 0x3f1912f2, 0x3f1ec911, 0x3edf74ee, 0x3f1cf1ac, 0x3f48fe58,
      0x3f7fc5a0, 0x3f7dfc14, 0x3f5b451b, 0x3ea08046, 0x3f7009fb, 0x3f755397,
      0x3f2b7aa6, 0x3effe866, 0x3e90a61c, 0x3f653cfc, 0x3e770510, 0x3ee80a80,
      0x3e32d938, 0x3f47d81d, 0x3e3347e0, 0x3d98acf8, 0x3f1ffe86, 0x3de5fdd8,
      0x3f17bfb3, 0x3f257744, 0x3d4a2960, 0x3e5903b0, 0x3f479656, 0x3f6bc03c,
      0x3e95d8ea, 0x3e84b880, 0x3da394b0, 0x3f6a065f, 0x3f3033f8, 0x3f403973,
      0x3f6b9ee5, 0x3dfe4278, 0x3f636808, 0x3f671d51, 0x3ee8fbac, 0x3f2afd1f,
      0x3da4c250, 0x3f154d38, 0x3dc6da18, 0x3f0b14b9, 0x3f7b804f, 0x3f1a44a2,
      0x3f585b1d, 0x3f66e71d, 0x3e1de574, 0x3f5ee053, 0x3f18eea8, 0x3d27a9c0,
      0x3e8a6136, 0x3d59a660, 0x3f01db0f, 0x3e93354c, 0x3e0f1f7c, 0x3ebb7906,
      0x3f30338b, 0x3e42accc, 0x3c32fd80, 0x3c9ab440, 0x3f00c421, 0x3f1dbece,
      0x3f7cfdf9, 0x3f455695, 0x3f6720ba, 0x3ecffdea, 0x3f0b1e52, 0x3f66ad8c,
      0x3e5d7c88, 0x3efdaefe, 0x3f1951b3, 0x3f56efd9, 0x3f376aad, 0x3ed542ea,
      0x3e8a1efc, 0x3f720f76, 0x3e9519d8, 0x3e685028, 0x3da56500, 0x3f2b4fd1,
      0x3e4e166c, 0x3f2d7003, 0x3f64d5f4, 0x3e5f26a8, 0x3ea02bb8, 0x3f278ce4,
      0x3e660294, 0x3f0ba669, 0x3f35657f, 0x3e37a1bc, 0x3dc901c0, 0x3f1a8587,
      0x3f54a350, 0x3d0bd2c0, 0x3f315e90, 0x3f141915, 0x3ec58e3a, 0x3f356c46,
      0x3e5c098c, 0x3f176fc8, 0x3f6972a7, 0x3f59a44e, 0x3f43f815, 0x3f5ff729,
      0x3e1158ec, 0x3f564eea, 0x3f6fef0b, 0x3d7aede0, 0x3e796be4, 0x3e2ec3dc,
      0x3f6d9d4a, 0x3f6f2bc9, 0x3e641b0c, 0x3e9576ce, 0x3f3c03ec, 0x3f454a1d,
      0x3f05c20b, 0x3f4cc550, 0x3eb81f36, 0x3f615407, 0x3eead1e2, 0x3f4b7b10,
      0x3ecaaa80, 0x3f36b6fb, 0x3f0a6b8c, 0x3f216b9d, 0x39e1e000, 0x3d852408,
      0x3e3024c0, 0x3e7ed480, 0x3e95077a, 0x3e53a9ec, 0x3f746732, 0x3f0cd57f,
      0x3f3dced4, 0x3f6ca63a, 0x3f16e067, 0x3efa76c0, 0x3f34468a, 0x3f1652eb,
      0x3f401718, 0x3f00b6bf, 0x3f67416b, 0x3eed6a88, 0x3d7f04b0, 0x3e0941c8,
  };

  // 2,3,5,5
  uint32_t kernel_vals[] = {
      0x3dc1ef2c, 0xbd4afe40, 0xbd9592c5, 0xbe1fade9, 0x3d6c2508, 0xbdd2c720,
      0x3c195110, 0x3e39d55a, 0xbe19fe5a, 0xbd4f7cb8, 0xbdd860aa, 0xbd9f8af7,
      0xbe15e820, 0x3bc06080, 0x3dd8b200, 0x3d98e864, 0xbdf9352c, 0x3d6e6124,
      0x3e1ce872, 0xbddbdef8, 0xbdd473e1, 0xbe33b833, 0x3da421b8, 0xbe2d2c6b,
      0x3e3aa2e4, 0x3de87114, 0xbd1046d8, 0x3d5b0694, 0x3dae7864, 0x3d990464,
      0xbdd1cee5, 0x3db34934, 0x3dbce7a4, 0x3dc31f58, 0xbd8b648c, 0x3e107fd2,
      0x3d86b7ce, 0xbd57180c, 0xbd6735f0, 0xbc94cfe0, 0xbe0ded40, 0xbd85f78e,
      0x3dc8b7ec, 0xbca29f38, 0xbdbdcb4e, 0xbcea63b0, 0xbe35a862, 0x3e3869a6,
      0x3e03dc16, 0x3e369824, 0xbd82ddc2, 0x3c8f2268, 0x3c086970, 0xbdd3ca16,
      0xbd5cfa1c, 0xbe0fdbe9, 0x3db979b0, 0xbdf6cec4, 0xbc986630, 0x3d3c4390,
      0x3def1cd0, 0xbe04c359, 0xbe379160, 0xbd2299c0, 0x3d639694, 0x3e32eeae,
      0x3d234c44, 0x3cd05b90, 0xbe1ca8af, 0x3e2bd088, 0x3e1ebe9a, 0xbd17b4c0,
      0x3dd7029c, 0x3dbb63b0, 0xbe0cffec, 0x3db369c8, 0x3e2ce704, 0xbe2894ee,
      0x3db3e32c, 0x3d7a42f8, 0x3dfb04b8, 0xbe2addc3, 0x3e1c07c8, 0x3e0fe1ac,
      0x3d849fca, 0x3c5e5ae0, 0x3d8afe3c, 0xbd91a2ca, 0x3cc6c5c0, 0xbd345d88,
      0x3e0df82e, 0xbe0e8518, 0xbdfa2da2, 0xbdca4ed7, 0xbd493a3c, 0x3bf73200,
      0x3dfb6e40, 0xbca86c80, 0x3ddd534c, 0xbe178f40, 0x3cabdc18, 0x3c2027b0,
      0x3e1b0c4a, 0xbe03fa6d, 0xbd5728b0, 0x3e04e9ae, 0xbd05d144, 0x3c20c3f0,
      0x3de76758, 0xbdb6aeca, 0xbdb0897a, 0x3e1eaf24, 0x3d215434, 0x3db95a08,
      0x3d830e56, 0xbd884998, 0x3db9c368, 0xbe10f35d, 0x3e256c04, 0xba02ad00,
      0xbe0437fa, 0xbe30f32c, 0x3d831984, 0x3e1e4382, 0xbe0db618, 0xbd3297e4,
      0xbde6de41, 0x3dc95498, 0xbdb933e1, 0x3de72ce0, 0xbcaa3940, 0xbe0aef70,
      0x3db150b0, 0x3dcee6c0, 0x3c843678, 0x3e2bd24c, 0x3c4431b0, 0xbe12ca69,
      0xbe02fabe, 0xbe076e08, 0x3dd23240, 0xbe29bb4c, 0xbd66b750, 0x3e2af9f0,
      0xbdb5289e, 0xbd79f286, 0x3e1b6538, 0xbe376a62, 0xbdc9851c, 0xba66d200,
  };

  // 5
  uint32_t bias_vals[] = {
      0xbd3e9f8c, 0xbdf4be10, 0xbd861c9e, 0x3d0d4978, 0xbdc35d93,
  };

  // 3,5,2,5
  uint32_t output_exp_vals[] = {
      0x3e3eb29f, 0xbf2431cf, 0xbd3c2cb9, 0x3ec9941a, 0xbe019504, 0x3ed55520,
      0xbf268f78, 0xbeb25c44, 0xbbc1b02b, 0x3d9680fc, 0x3e5a57a9, 0xbf0861e9,
      0xbe980c59, 0x3ec9af1f, 0x3e057cc4, 0x3c397765, 0xbec5d321, 0xbe7106e9,
      0x3e6b25bf, 0xbe162702, 0x3d285bae, 0xbee24ed1, 0xbe5f61c6, 0x3e6d9bf1,
      0xbe21e5ce, 0x3e2257db, 0xbeae2fd2, 0x3d2453d3, 0x3ec62915, 0xbee4c828,
      0x3e58d564, 0xbf232884, 0x3e1c46c1, 0x3e3baa73, 0xbded1d5b, 0x3e09ce53,
      0xbeebf081, 0x3dae4215, 0x3f12fb01, 0xbd9626b6, 0x3de73a84, 0xbec044c7,
      0x3e517516, 0x3e892149, 0xbd851637, 0x3eb5d71c, 0xbf076e93, 0x3c6657e7,
      0x3d81414f, 0xbe3d5ca7, 0x3c279fa2, 0xbf1065b3, 0xbe00250d, 0x3efef655,
      0xbe071035, 0x3e3a4085, 0xbf0e87c2, 0x3db127ea, 0x3e4d846b, 0x3e1ecce4,
      0x3e3107ba, 0xbf23824b, 0x3e269cf3, 0x3e95ad70, 0xbe098bb7, 0x3dec94ee,
      0xbe65fde6, 0x3db798b1, 0x3db34030, 0xbe7429ad, 0xbbee7768, 0xbf19b3e7,
      0xbe166291, 0x3d948b67, 0xbe4f7ce2, 0x3e108942, 0xbeea1e35, 0xbe3bb4b7,
      0x3e84725a, 0xbea914ea, 0x3e041cec, 0xbed55994, 0x3e31f984, 0xbd3233e6,
      0xbe1658bb, 0x3e09a5ee, 0xbee9f260, 0xbde464c0, 0x3de69ade, 0xbd79033a,
      0x3d9a3ba7, 0xbef017f5, 0x3df0f795, 0x3d436760, 0xbcb32c93, 0x3ec1bc0a,
      0xbec3a50b, 0xbe3b3a62, 0x3e0529ee, 0xbe5c712e, 0x3eab528d, 0xbe7880c5,
      0xbe7cc267, 0x3e9789ef, 0xbe1f75ee, 0x3dfd3bff, 0xbf0529de, 0xbdb2af09,
      0x3ebbc0cb, 0xbe1f469c, 0x3e29a963, 0xbe893841, 0xbec59110, 0x3e03a680,
      0xbe0fe3fc, 0x3e4ab203, 0xbf1040a7, 0xbd4d9423, 0x3de26c26, 0xbd11f7e5,
      0x3ec20296, 0xbf17732c, 0x3e541b46, 0x3eb4273c, 0xbe80ea72, 0x3e6f3363,
      0xbf060cb4, 0xbe42defb, 0x3e723a44, 0x3d6ffd66, 0x3e162cfb, 0xbf6b8da8,
      0x3d56a599, 0x3eb102a0, 0xbe420cb5, 0x3eb0a5be, 0xbef142c6, 0x3d937133,
      0x3ea7bcdf, 0xbeb35b92, 0x3d8e0431, 0xbed46437, 0x3e651ef5, 0x3eae0d1e,
      0xbdd41583, 0x3e4231b1, 0xbe3b758c, 0xbe89c7f5, 0xbd5ab5c6, 0xbe467e60,
  };

  // 3,5,2,5
  uint32_t output_relu_exp_vals[] = {
      0x3e3eb29f, 0x0,        0x0,        0x3ec9941a, 0x0,        0x3ed55520,
      0x0,        0x0,        0x0,        0x3d9680fc, 0x3e5a57a9, 0x0,
      0x0,        0x3ec9af1f, 0x3e057cc4, 0x3c397765, 0x0,        0x0,
      0x3e6b25bf, 0x0,        0x3d285bae, 0x0,        0x0,        0x3e6d9bf1,
      0x0,        0x3e2257db, 0x0,        0x3d2453d3, 0x3ec62915, 0x0,
      0x3e58d564, 0x0,        0x3e1c46c1, 0x3e3baa73, 0x0,        0x3e09ce53,
      0x0,        0x3dae4215, 0x3f12fb01, 0x0,        0x3de73a84, 0x0,
      0x3e517516, 0x3e892149, 0x0,        0x3eb5d71c, 0x0,        0x3c6657e7,
      0x3d81414f, 0x0,        0x3c279fa2, 0x0,        0x0,        0x3efef655,
      0x0,        0x3e3a4085, 0x0,        0x3db127ea, 0x3e4d846b, 0x3e1ecce4,
      0x3e3107ba, 0x0,        0x3e269cf3, 0x3e95ad70, 0x0,        0x3dec94ee,
      0x0,        0x3db798b1, 0x3db34030, 0x0,        0x0,        0x0,
      0x0,        0x3d948b67, 0x0,        0x3e108942, 0x0,        0x0,
      0x3e84725a, 0x0,        0x3e041cec, 0x0,        0x3e31f984, 0x0,
      0x0,        0x3e09a5ee, 0x0,        0x0,        0x3de69ade, 0x0,
      0x3d9a3ba7, 0x0,        0x3df0f795, 0x3d436760, 0x0,        0x3ec1bc0a,
      0x0,        0x0,        0x3e0529ee, 0x0,        0x3eab528d, 0x0,
      0x0,        0x3e9789ef, 0x0,        0x3dfd3bff, 0x0,        0x0,
      0x3ebbc0cb, 0x0,        0x3e29a963, 0x0,        0x0,        0x3e03a680,
      0x0,        0x3e4ab203, 0x0,        0x0,        0x3de26c26, 0x0,
      0x3ec20296, 0x0,        0x3e541b46, 0x3eb4273c, 0x0,        0x3e6f3363,
      0x0,        0x0,        0x3e723a44, 0x3d6ffd66, 0x3e162cfb, 0x0,
      0x3d56a599, 0x3eb102a0, 0x0,        0x3eb0a5be, 0x0,        0x3d937133,
      0x3ea7bcdf, 0x0,        0x3d8e0431, 0x0,        0x3e651ef5, 0x3eae0d1e,
      0x0,        0x3e4231b1, 0x0,        0x0,        0x0,        0x0,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, VALID_PADDING, NULL);
}

void test_valid_padding_zero_strides_medium() {
  input_set *set = &medium_input;
  strides_input_set *strides = &zero_strides;

  // 3,10,8,5
  uint32_t input_vals[] = {
      0x3f36b631, 0x3b9ca600, 0x3f76ac7f, 0x3d3f3570, 0x3e70e1b4, 0x3e3830f8,
      0x3f3f9678, 0x3e1fb9dc, 0x3dd55dd0, 0x3aef4000, 0x3f433347, 0x3df08b90,
      0x3eb33e84, 0x3dc553c8, 0x3ee7a2f2, 0x3e377d78, 0x3f3e7fdb, 0x3f1236d7,
      0x3ea73b60, 0x3e0cc410, 0x3f57cffe, 0x3f1253af, 0x3eeea1d6, 0x3f116d9b,
      0x3de6adb0, 0x3f335bd0, 0x3f459139, 0x3f221418, 0x3e089dac, 0x3f208980,
      0x3ec148ac, 0x3dfe44a8, 0x3f434e54, 0x3f207099, 0x3f6d610b, 0x3f4929fb,
      0x3f6fef6c, 0x3e96ad46, 0x3f4d17d4, 0x3e9b4166, 0x3eabb14a, 0x3b020c00,
      0x3f09f77a, 0x3f68586f, 0x3ef184c6, 0x3f1b4509, 0x3df6bf78, 0x3f73cdae,
      0x3f71564c, 0x3e0d0f10, 0x3f25afae, 0x3f4cc291, 0x3e5e4880, 0x3f544618,
      0x3f165f1a, 0x3f0d128c, 0x3f0459a7, 0x3e24cc88, 0x3eb49e0a, 0x3f247e0f,
      0x3f19b458, 0x3e607e7c, 0x3deeb150, 0x3ee86844, 0x3ed8b3e2, 0x3efaeafa,
      0x3eca4cde, 0x3f4c835c, 0x3f71404c, 0x3f6162dc, 0x3eb60aea, 0x3d6169d0,
      0x3d298250, 0x3f3facbb, 0x3dc7db50, 0x3f476996, 0x3d84cdb8, 0x3e8381c8,
      0x3ebc1c56, 0x3edda720, 0x3f327cb8, 0x3ec91382, 0x3f567352, 0x3f60a3cf,
      0x3e63cdd0, 0x3f2fbe4e, 0x3f61a525, 0x3e4d10dc, 0x3d8119d8, 0x3f706446,
      0x3f566a2d, 0x3f263a00, 0x3f68176a, 0x3f6d8d15, 0x3f6c6af8, 0x3e741c84,
      0x3f182317, 0x3e885446, 0x3f479de4, 0x3f7de85f, 0x3ec4857c, 0x3ebc271a,
      0x3f143fe7, 0x3e0877c4, 0x3f75c402, 0x3ef8b408, 0x3f748088, 0x3f10c163,
      0x3dbccff0, 0x3f32b1fc, 0x3f674b45, 0x3f0139d4, 0x3ef9b882, 0x3f64196c,
      0x3defe888, 0x3dd093e8, 0x3f2b3835, 0x3f4fb95d, 0x3f7e460c, 0x3e24b3c8,
      0x3e5a5130, 0x3f0208d2, 0x3f6cb694, 0x3f0b2b70, 0x3f1eaf41, 0x3ec30552,
      0x3f13f950, 0x3e405394, 0x3e9bf8c4, 0x3d5b8a00, 0x3d954c60, 0x3f2aaf0d,
      0x3f47c963, 0x3f48e285, 0x3f5108e1, 0x3dd6f800, 0x3f623988, 0x3f22332e,
      0x3f39bf14, 0x3f37a5a6, 0x3f57aec5, 0x3f49ad86, 0x3efb3ac4, 0x3ee00cf8,
      0x3f1e8a44, 0x3f05f896, 0x3f10619b, 0x3e1a016c, 0x3f667a9f, 0x3d823af0,
      0x3dc83748, 0x3f332ca5, 0x3f25f83e, 0x3f52f762, 0x3f5d1d04, 0x3ed5bac2,
      0x3eb30346, 0x3f5ce83e, 0x3f1fe917, 0x3ee4e7f6, 0x3ef79562, 0x3f577e28,
      0x3d910f88, 0x3f427613, 0x3e94929e, 0x3ea4fce6, 0x3f23ee9a, 0x3d1c8fb0,
      0x3d01a550, 0x3f42d47f, 0x3da77a08, 0x3f60995e, 0x3c0a9380, 0x3e88165e,
      0x3f0ad115, 0x3f45a98a, 0x3ed80798, 0x3f1d936a, 0x3f0e07fa, 0x3e8e1ef2,
      0x3ee0618c, 0x3dbbedc8, 0x3eb6713c, 0x3f22cf06, 0x3f1df573, 0x3f7cf87a,
      0x3f19492d, 0x3eb07ab6, 0x3edd0088, 0x3e2a9198, 0x3defc080, 0x3f340f50,
      0x3f773033, 0x3f3054c9, 0x3f76f70a, 0x3e66eb1c, 0x3f676d3a, 0x3ebeb408,
      0x3f26fe18, 0x3f703110, 0x3ef2b336, 0x3f319c97, 0x3e3adb10, 0x3d1b3af0,
      0x3f3b0802, 0x3e1b7498, 0x3f2f4afe, 0x3f0b8386, 0x3f285fa0, 0x3f43b72e,
      0x3f5d486b, 0x3f65dcf2, 0x3e66fba4, 0x3eeb2f94, 0x3b42bd00, 0x3f62fc54,
      0x3ea805d4, 0x3eb4fe52, 0x3e1fab68, 0x3e6362b8, 0x3f73f5a9, 0x3f7103c8,
      0x3f439d03, 0x3f0451db, 0x3c9a8dc0, 0x3e3ccb48, 0x3f419ff7, 0x3f1d8aa7,
      0x3ec10a20, 0x3e752480, 0x3f5b1273, 0x3f557bf6, 0x3e20a7b0, 0x3e701b88,
      0x3daf0738, 0x3f58dfea, 0x3f5b49d0, 0x3f79dc64, 0x3d35c960, 0x3f5e3797,
      0x3e9a238c, 0x3dbb67a8, 0x3f56999b, 0x3f6b147d, 0x3f450ebd, 0x3f0e0ba6,
      0x3f42ab7d, 0x3e22fa34, 0x3e491070, 0x3f4b6181, 0x3f3cfea2, 0x3f773612,
      0x3f37c12c, 0x3f11311c, 0x3e8e5a48, 0x3f5b9871, 0x3e184f28, 0x3f349900,
      0x3f668a71, 0x3f618f59, 0x3d072ca0, 0x3eea0f1e, 0x3e109d28, 0x3edfbf9c,
      0x3edb0210, 0x3f7f6eff, 0x3e478798, 0x3e182b74, 0x3e306d94, 0x3ebdac5e,
      0x3ebfa2ce, 0x3ec22efa, 0x3db01a70, 0x3c854b20, 0x3f44eb5b, 0x3f342728,
      0x3f131358, 0x3f390480, 0x3f154485, 0x3f54c400, 0x3e4200a0, 0x3f4477e4,
      0x3f164d00, 0x3e7f1908, 0x3e10028c, 0x3debfcd8, 0x3eb5adde, 0x3f361860,
      0x3f0955a3, 0x3f79dc1a, 0x3e254178, 0x3cd96800, 0x3e8f8ee8, 0x3f28e2cb,
      0x3ee6297c, 0x3f4b6ffb, 0x3f451d83, 0x3f603f0b, 0x3eb77766, 0x3f22176f,
      0x3e81b948, 0x3f47cfda, 0x3f10581d, 0x3f56ca4d, 0x3b0bb400, 0x3f4a6862,
      0x3ed19728, 0x3dceec28, 0x3edeafc6, 0x3f0e3fea, 0x3f26b23b, 0x3f6d93ef,
      0x3ebf908a, 0x3e99d570, 0x3e814d2c, 0x3f32ea59, 0x3f62acaf, 0x3f4152b9,
      0x3e2d777c, 0x3e3f6a7c, 0x3ec70f36, 0x3ec7937c, 0x3f1c251e, 0x3f26d5de,
      0x3e951d02, 0x3eaf0570, 0x3f04f161, 0x3ed8d624, 0x3e437414, 0x3efeceba,
      0x3e8838fe, 0x3f168c18, 0x3e1a49a8, 0x3f1ae81d, 0x3eb71206, 0x3e61cee8,
      0x3ecdd5a4, 0x3e418040, 0x3edfb0b6, 0x3e38040c, 0x3f76051f, 0x3ec0ee50,
      0x3f6e6777, 0x3f57f7f9, 0x3f5ba562, 0x3ef21660, 0x3e8c7112, 0x3e5ab544,
      0x3e827f6a, 0x3f261474, 0x3e9237d6, 0x3e5cd520, 0x3f5ab4cb, 0x3e81e382,
      0x3d0e0a40, 0x3d0e87e0, 0x3f3fecc7, 0x3eaa5012, 0x3edeea7a, 0x3f1ad9aa,
      0x3f059183, 0x3eb3fbb8, 0x3f014789, 0x3e4747a0, 0x3f1f716e, 0x3b714000,
      0x3e852fde, 0x3c56b680, 0x3f30cc9a, 0x3f04b414, 0x3efcfc4a, 0x3f70c129,
      0x3eacc898, 0x3ea23a38, 0x3f1f14e9, 0x3f50d1f7, 0x3f147713, 0x3f6be6c0,
      0x3f3483e7, 0x3eea910a, 0x3e591a9c, 0x3f5c5ad6, 0x3d514e70, 0x3f4a49a7,
      0x3f37fad8, 0x3f46540c, 0x3f74a543, 0x3f40679c, 0x3f4abe61, 0x3e178ffc,
      0x3eae56a8, 0x3edbefee, 0x3da237b0, 0x3ed100d2, 0x3e92d8a6, 0x3dec6e58,
      0x3f1317ce, 0x3f12ab4d, 0x3f45daaf, 0x3f5fe2b6, 0x3e42cb94, 0x3f7a260c,
      0x3f0d94d4, 0x3e2f4678, 0x3f5da986, 0x3f3ecd03, 0x3f6e32c6, 0x3eb08d0e,
      0x3ea099be, 0x3e11011c, 0x3f766f53, 0x3e6611c4, 0x3f5d2b28, 0x3ee70ede,
      0x3f658ee8, 0x3d2f4970, 0x3f1d4a31, 0x3d9109c0, 0x3ce695e0, 0x3f7f63bf,
      0x3e8fd860, 0x3e3e820c, 0x3f0d6155, 0x3f032f81, 0x3f10d55d, 0x3f6d3b5e,
      0x3f02d3a0, 0x3f56b825, 0x3f146fa0, 0x3de117f0, 0x3ef0853c, 0x3f6a0c73,
      0x3de4ad70, 0x3e9a66fc, 0x3f18cfc5, 0x3f1944d9, 0x3f3f06b1, 0x3ef8a9a6,
      0x3f298468, 0x3ecc4c4c, 0x3f123787, 0x3d1cee30, 0x3f3b25da, 0x3f6fd971,
      0x3f5c07c3, 0x3f34bfaa, 0x3e91d31a, 0x3f33a6b7, 0x3ed4457e, 0x3ca0a0e0,
      0x3e968518, 0x3da00258, 0x3f0f0d86, 0x3e6f51dc, 0x3f20c172, 0x3f109e04,
      0x3f3c4c11, 0x3f33be00, 0x3d69d2d0, 0x3efcff56, 0x3f5730f0, 0x3edf8088,
      0x3e1b1620, 0x3e0a22dc, 0x3f0a1397, 0x3eea6736, 0x3e09b75c, 0x3f6a9bfc,
      0x3e64d1b8, 0x3f580cd8, 0x3d290cc0, 0x3eae325e, 0x3f02dad5, 0x3e057dd8,
      0x3f024ff2, 0x3f2c3e6f, 0x3f2a014a, 0x3f52bd7b, 0x3e9b5a80, 0x3dc798e8,
      0x3ebea594, 0x3f0bd9a6, 0x3f1c50ef, 0x3f468365, 0x3e5554cc, 0x3ec9fdf8,
      0x3f781a74, 0x3ee3854a, 0x3f280dd5, 0x3e997f22, 0x3f42c1ab, 0x3efc99ea,
      0x3f36d3c3, 0x3f735e96, 0x3f0da6b8, 0x3f166eb5, 0x3f284366, 0x3f7982ae,
      0x3f7f159b, 0x3f31652b, 0x3e4a43b8, 0x3eb27360, 0x3f7e42e8, 0x3ea3d00a,
      0x3ed03742, 0x3e89dd14, 0x3f0f4624, 0x3da20d48, 0x3e9bbeda, 0x3e9af660,
      0x3ed915e6, 0x3f0495dd, 0x3f07c35c, 0x3f118423, 0x3f382ace, 0x3ec0c056,
      0x3f6df319, 0x3e1030fc, 0x3dd09b68, 0x3c62da80, 0x3ec9ce0c, 0x3f35a223,
      0x3f4a9439, 0x3f459f23, 0x3f5971aa, 0x3e220b1c, 0x3e5c371c, 0x3f5be3f7,
      0x3ed1ecf8, 0x3ee23da0, 0x3f5cba00, 0x3d643c00, 0x3e187990, 0x3da69340,
      0x3f5910ef, 0x3f426233, 0x3cc6e800, 0x3ef1344e, 0x3dfae618, 0x3f60b7f5,
      0x3f1942a2, 0x3ea30810, 0x3f5892c1, 0x3e1f3270, 0x3f0dac81, 0x3f1cfb60,
      0x3f108bc3, 0x3f54a5b3, 0x3f232e5b, 0x3f287d6a, 0x3f0efc17, 0x3f786b06,
      0x3dfe15e8, 0x3f621efb, 0x3e5e68d4, 0x3f5fb5c0, 0x3ee6d8ca, 0x3f55f16d,
      0x3f656889, 0x3f5bb5c0, 0x3f1cf6c7, 0x3f320bfc, 0x3ed2ad8a, 0x3eb98256,
      0x3f4aeaf3, 0x3f137f9e, 0x3f410d87, 0x3d991dd0, 0x3e569a0c, 0x3ed7cafe,
      0x3e668c48, 0x3def9d98, 0x3f197f5b, 0x3e8e60d6, 0x3f31e438, 0x3f2ab320,
      0x3db0ee68, 0x3f3ba15f, 0x3e223e2c, 0x3ef1a91e, 0x3d040020, 0x3f308052,
      0x3f4cc3a8, 0x3ebdff66, 0x3f640150, 0x3f285ea6, 0x3f3ba978, 0x3e23e124,
      0x3d465d60, 0x3f34b0c8, 0x3d886860, 0x3e88cc22, 0x3e0d2ad4, 0x3f446a6a,
      0x3f5d5169, 0x3e960c80, 0x3f140f03, 0x3e27fde0, 0x3cf0dd40, 0x3d3a6940,
      0x3f43937e, 0x3c2b1f80, 0x3f488a07, 0x3e418210, 0x3f7cb6bd, 0x3e4f0bf0,
      0x3f019d08, 0x3d810b18, 0x3f55e69d, 0x3f2fb0bf, 0x3f65ec48, 0x3f1f602f,
      0x3efdd75e, 0x3e0104f0, 0x3f3e3b36, 0x3e651c0c, 0x3f01362b, 0x3eb4c58a,
      0x3f1e99ac, 0x3f1c85ab, 0x3f793d8d, 0x3ed75d7a, 0x3f37e2a9, 0x3f5fe002,
      0x3f541199, 0x3f27737e, 0x3f354703, 0x3f19231c, 0x3f7e2bde, 0x3f788080,
      0x3e910fb4, 0x3eb9a258, 0x3e7d73d8, 0x3f40c445, 0x3eea30f6, 0x3f4e1083,
      0x3f5f484b, 0x3f165b7a, 0x3ebb4c5e, 0x3db7c988, 0x3d6a4e60, 0x3d8dffa8,
      0x3f75cb55, 0x3f4924cc, 0x3f7589a7, 0x3e3ba718, 0x3e5b64c4, 0x3f15130b,
      0x3ef78d22, 0x3eca0304, 0x3e3f3a18, 0x3da92190, 0x3e812406, 0x3eb1109e,
      0x3e84898c, 0x3f10d994, 0x3d43c8a0, 0x3f044912, 0x3f006ab8, 0x3f4ecb83,
      0x3f0c933a, 0x3f5ee4ab, 0x3d297a30, 0x3f1cb629, 0x3f476f1e, 0x3d8f4010,
      0x3ec0a59e, 0x3e3780b8, 0x3f55d398, 0x3f11230d, 0x3f7b83d5, 0x3ddb9ed8,
      0x3ed072d2, 0x3f35bad5, 0x3eefee28, 0x3effc15e, 0x3f1a8c66, 0x3e40f244,
      0x3f3d1e68, 0x3ecf06e8, 0x3e6e97fc, 0x3df891f0, 0x3f6c646f, 0x3f132603,
      0x3f755d4b, 0x3f030eb4, 0x3f069de3, 0x3f18d89b, 0x3ef31d78, 0x3f4dfd0b,
      0x3e921c74, 0x3f3cd952, 0x3f632436, 0x3f35bf2e, 0x3f4c3a85, 0x3f23506e,
      0x3ec9ddfa, 0x3ec5b4de, 0x3f1ec970, 0x3f519f70, 0x3f2e5652, 0x3f41cd3e,
      0x3edc6592, 0x3cb093e0, 0x3e740a40, 0x3eb761fc, 0x3f1a4575, 0x3f204993,
      0x3f14b1c6, 0x3f1ab9cd, 0x3f077650, 0x3dca4d18, 0x3eb3e18e, 0x3e8e4ffe,
      0x3d8a25f0, 0x3f4c2939, 0x3f193155, 0x3ea8bc8e, 0x3e707ccc, 0x3f4a5d88,
      0x3e32faac, 0x3f52839c, 0x3c1bd940, 0x3f318061, 0x3b900d80, 0x3ec6ea42,
      0x3ec4a776, 0x3f606dab, 0x3f1a2caf, 0x399c3000, 0x3eda3bd4, 0x3f0c18e1,
      0x3f73d7fe, 0x3f15ad0b, 0x3bbcb600, 0x3e2d4114, 0x3f105459, 0x3ba1d600,
      0x3e868ece, 0x3f4eeef6, 0x3f0a6cdd, 0x3f0b01ca, 0x3eab4cdc, 0x3f3b253e,
      0x3f0d522a, 0x3f04b44f, 0x3f214601, 0x3df97570, 0x3f781e88, 0x3f004a84,
      0x3dc4e2c8, 0x3cae3f40, 0x3e981544, 0x3f0fa8be, 0x3f145be8, 0x3f1a41ed,
      0x3f611d95, 0x3f6172a1, 0x3f349c83, 0x3e9f1dd2, 0x3f7a2bf5, 0x3f37d399,
      0x3f44784a, 0x3d7b3b70, 0x3eb2431e, 0x3f441518, 0x3f0fce8a, 0x3eec22ee,
      0x3f4c186b, 0x3e271ef0, 0x3e6a4590, 0x3f174a78, 0x3f36ab45, 0x3f2c7736,
      0x3ec17b00, 0x3f752abc, 0x3dc62d48, 0x3f2639dc, 0x3f39aae7, 0x3e44f29c,
      0x3e200de4, 0x3eb6e81a, 0x3f45e72e, 0x3f3560a9, 0x3f035e25, 0x3e7bad00,
      0x3ef4dd66, 0x3dc91d00, 0x3ebcaa6c, 0x3f649206, 0x3f3875bd, 0x3f1e6d42,
      0x3f2790d3, 0x3ef0a232, 0x3db17798, 0x3f65d44d, 0x3e672348, 0x3cd8b4e0,
      0x3f7bcf6d, 0x3f3ef25d, 0x3e032ce8, 0x3e938888, 0x3ec79684, 0x3f2f7936,
      0x3dda4f60, 0x3e95eede, 0x3f349424, 0x3d9caff0, 0x3f1c2c7e, 0x3f3398e5,
      0x3f190799, 0x3e05dd44, 0x3edc72e0, 0x3f3675c1, 0x3f5e5a3c, 0x3f16064c,
      0x3f662418, 0x3f3f4247, 0x3f0f3b48, 0x3f25daa2, 0x3f68f840, 0x3f6ffad9,
      0x3c630280, 0x3f2b903a, 0x3f2c223d, 0x3effe77c, 0x3f2e2f6b, 0x3eecd0da,
      0x3f342867, 0x3f5363b6, 0x3e9e7f68, 0x3f63ad4c, 0x3f3414bf, 0x3e6bcdb0,
      0x3f578426, 0x3e962064, 0x3f4db122, 0x3e9c3af6, 0x3ef24892, 0x3f4c6ed5,
      0x3f03c0b1, 0x3e9e3aaa, 0x3f0ba870, 0x3f134a31, 0x3f4793dc, 0x3e001670,
      0x3d87d9c0, 0x3e9baaf2, 0x3ed92222, 0x3f22a2b0, 0x3ddd1fd0, 0x3f10d7f2,
      0x3e288014, 0x3da30988, 0x3f4211a2, 0x3e92778e, 0x3f505e3f, 0x3f643df4,
      0x3f0c2459, 0x3e20aef4, 0x3f0349c1, 0x3ee5cf92, 0x3e1474a0, 0x3f4e39d7,
      0x3e349b04, 0x3e2639b0, 0x3f0ac524, 0x3efcffac, 0x3e6e2ee4, 0x3f226cb6,
      0x3e5bbbc0, 0x3f5edf6b, 0x3f2aa1b4, 0x3f7dabe8, 0x3f13eb6d, 0x3f5b6432,
      0x3f34d9a9, 0x3e61deac, 0x3f1d30f9, 0x3d778ee0, 0x3f764987, 0x3f5fb106,
      0x3f3dcdc1, 0x3f367073, 0x3e0b0244, 0x3eafcfd0, 0x3f12cb91, 0x3dad9c30,
      0x3e9d2594, 0x3dd0d1a0, 0x3d9178d0, 0x3f0e591f, 0x3f7a3f1c, 0x3dba5ad0,
      0x3f65a922, 0x3e9dfd50, 0x3ec2c3f4, 0x3f390185, 0x3ea0ba1a, 0x3f53afab,
      0x3f33a93d, 0x3ed4cce4, 0x3edbc1d0, 0x3f03e91f, 0x3e728234, 0x3f298ab0,
      0x3e6fe3bc, 0x3dac3250, 0x3e8b58f8, 0x3f135011, 0x3eb0a3dc, 0x3f5593ec,
      0x3f03f287, 0x3f71fe15, 0x3ec51e3e, 0x3d744960, 0x3f74be06, 0x3f2d671d,
      0x3e384d84, 0x3f6f4615, 0x3eb3061c, 0x3f2f80b2, 0x3e69ed90, 0x3e2727b4,
      0x3d410e20, 0x3ec6c27e, 0x3f03a310, 0x3edfc2a4, 0x3ea53202, 0x3e0a79b4,
      0x3f49b5df, 0x3eab7e3c, 0x3ae05e00, 0x3f72d5e0, 0x3dab7408, 0x3e810eaa,
      0x3bcf8e00, 0x391bc000, 0x3e8af156, 0x3f4a7e45, 0x3f4cd234, 0x3ed098fe,
      0x3ee4327e, 0x3f766ab7, 0x3e80075e, 0x3f11c187, 0x3e14872c, 0x3f6b1a7b,
      0x3f3d8346, 0x3f4c3cf2, 0x3ed51848, 0x3e060280, 0x3ef3f384, 0x3ed91a44,
      0x3f2b44a6, 0x3d5edd00, 0x3ecd4ecc, 0x3f282129, 0x3f4233de, 0x3e82de7e,
      0x3f7e5ec6, 0x3ecda3b2, 0x3e99ac3a, 0x3f3cda2e, 0x3ebb093e, 0x3ed259d4,
      0x3f19d226, 0x3eac68c0, 0x3e7b6568, 0x3e9866b8, 0x3f508994, 0x3e0ce474,
      0x3ee3dc62, 0x3ecb4bcc, 0x3f387866, 0x3e772898, 0x3f5bfd02, 0x3e7d2af0,
      0x3f7bfbe4, 0x3e9955f4, 0x3c566100, 0x3d376530, 0x3f4e5334, 0x3f496291,
      0x3f2b7072, 0x3dcfc480, 0x3e86c298, 0x3f4fd219, 0x3e82f6e0, 0x3f0924f0,
      0x3e799b88, 0x3e9ecf24, 0x3de3f120, 0x3ebb2a18, 0x3dce3a20, 0x3eae78de,
      0x3f07e440, 0x3f274293, 0x3f47be34, 0x3e3e2694, 0x3edbd970, 0x3f34e9a9,
      0x3e963ec0, 0x3f40fb2f, 0x3e1cfe1c, 0x3f6ac0fb, 0x3ebc892c, 0x3e5993d4,
      0x3f0a2574, 0x3f431243, 0x3d1f36f0, 0x3f13fa11, 0x3ee0c4b6, 0x3e368e9c,
      0x3f1bbc36, 0x3f3fe30e, 0x3ec36554, 0x3eb09934, 0x3e065dc0, 0x3ef9902a,
      0x3f6bb3ef, 0x3e9b5008, 0x3e5c78e0, 0x3ec9f760, 0x3e4f2254, 0x3f00d2e7,
      0x3edc3108, 0x3f12dd38, 0x3f2f0fb7, 0x3b969c80, 0x3e40b1fc, 0x3f19c592,
      0x3f490f1c, 0x3ee4b532, 0x3ea567b4, 0x3f04180f, 0x3c6ab140, 0x3ed87f5a,
      0x3da7da00, 0x3df04a10, 0x3f1e5dd8, 0x3e1ba038, 0x3eee17ae, 0x3ec14278,
      0x3b2edc00, 0x3f38a28e, 0x3f64000a, 0x3f09a1d2, 0x3f7fc3e6, 0x3cbd65e0,
      0x3eb22f14, 0x3f39e132, 0x3e91fa0a, 0x3db1a4f0, 0x3e319080, 0x3ea825ec,
      0x3ef927ba, 0x3f43c3c1, 0x3de54748, 0x3c643940, 0x3e873a08, 0x3f040ad3,
      0x3f761a55, 0x3f5076ef, 0x3f27e6a1, 0x3c864a80, 0x3f11ab22, 0x3ee89a44,
      0x3f7f5c64, 0x3f2fe4e3, 0x3f2cf469, 0x3f1dc003, 0x3f08b381, 0x3c9bccc0,
      0x3f3397ba, 0x3df9d740, 0x3e227020, 0x3f67225d, 0x3d43b5c0, 0x3b13a000,
      0x3e850468, 0x3eba37b4, 0x3f68d6ed, 0x3eaff3d4, 0x3f4f2667, 0x3efd5d96,
      0x3d286520, 0x3e031f88, 0x3f7f9ed7, 0x3f18f491, 0x3f62acfc, 0x3ddaf220,
      0x3e89721c, 0x3e5d693c, 0x3f417389, 0x3f71cf35, 0x3ecfd50e, 0x3c1c3a80,
      0x3f0af1a6, 0x3ec114c6, 0x3e0b9774, 0x3ede32d4, 0x3f4e661e, 0x3f687d66,
      0x3f3d77ac, 0x3f6f0343, 0x3f641d5d, 0x3f63e789, 0x3ea55fac, 0x3cd34ac0,
      0x3f72a8f4, 0x3f160c23, 0x3ea2d36c, 0x3eb17e60, 0x3f2b9905, 0x3f67672c,
      0x3ed67d9e, 0x3f290f1b, 0x3e28541c, 0x3f657fd6, 0x3f0b24da, 0x3ef8e85c,
      0x3f1f0187, 0x3f04ea25, 0x3f35ce89, 0x3f66cd7c, 0x3f477cad, 0x3d8efe60,
      0x3f7778f2, 0x3ceb0ec0, 0x3f3e30db, 0x3eb27386, 0x3f331d5d, 0x3f345742,
      0x3eb82282, 0x3e109c08, 0x3ed02474, 0x3f3ecea4, 0x3f59f84d, 0x3ee0357a,
      0x3e7222f0, 0x3ecca506, 0x3f02d0a6, 0x3eef0a28, 0x3f4310eb, 0x3f026842,
      0x3dc64630, 0x3f0d205c, 0x3ed7565c, 0x3e9f210a, 0x3e1527f4, 0x3edc3884,
      0x3f1dbe71, 0x3ed243ba, 0x3f1268ea, 0x3eabb42e, 0x3d1abe50, 0x3f5c3655,
      0x3f490120, 0x3ea62188, 0x3f6732ed, 0x3e6cca60, 0x3f7118d7, 0x3ede6e68,
      0x3ed7550e, 0x3f3b981c, 0x3e3a1adc, 0x3f1265a5, 0x3d954db8, 0x3efae76c,
      0x3f6f78f9, 0x3f5f59bb, 0x3f368f94, 0x3ea61b26, 0x3e5a004c, 0x3f00fb35,
      0x3d216a30, 0x3ebd061a, 0x3f3217ec, 0x3e1c9fc8, 0x3e7aee48, 0x3f7b5503,
      0x3f4c506d, 0x3f7ccbec, 0x3f69ab33, 0x3f54b76f, 0x3f2ffe47, 0x3e9b4d50,
      0x3ed17370, 0x3f6b8ba6, 0x3f096fbb, 0x3e25fe6c, 0x3db3ed90, 0x3d8470b0,
      0x3ef83fea, 0x3f158e41, 0x3f6a0bf2, 0x3f3c0de9, 0x3e070158, 0x3f6ecaf7,
  };

  // 10,8,5,5
  uint32_t kernel_vals[] = {
      0x3d3114e7, 0x3d4407ad, 0xbd35d912, 0x3ca7c94a, 0xbc56a7e8, 0x3b948e90,
      0xbccbb9a0, 0x3c2b9b28, 0x3ca02e4e, 0xbcceb383, 0x3c6a04fc, 0xbd37c660,
      0xb799d800, 0xbc5c8848, 0xbc4ae274, 0xbcf0a620, 0x3cb33d9e, 0x3d261659,
      0x3cc7aeb6, 0x3d326067, 0x3c9c9e26, 0xbbcc0050, 0x3cd0ac2a, 0xbc893ff4,
      0x3b8b1050, 0xba428000, 0xbd315ffa, 0xbd0f4ef5, 0x3bbcf490, 0xbc2ab878,
      0x3bc68180, 0xbbc9bb68, 0x3cd18a86, 0x3c96670e, 0x3c22f178, 0xbca5d14a,
      0xbca34e20, 0x3c69da2c, 0x3c012fc8, 0xbc4e8c78, 0x3c6c85a4, 0xbc8a1926,
      0xbc54d694, 0xbd031dd0, 0xbc5f05c0, 0xbbdf5d98, 0x3cfff456, 0xbc9b11c7,
      0xbd0435ce, 0xbd0479da, 0xbb11a930, 0xbd09e01a, 0xbcae6513, 0x3c897392,
      0xbd33a047, 0xbc90b650, 0xbbfc8990, 0x3c8228ee, 0xbca793ea, 0xbd149155,
      0xba0b0b40, 0x3cf9af0e, 0xbd20aafd, 0x3b9c4c68, 0xbd08876d, 0x3c3bf5c0,
      0xbc85b67a, 0x3c955286, 0x3c4ab648, 0xbca8e4b7, 0x3c4cdf44, 0xbccb04c3,
      0x3c22b794, 0xbd0e93a0, 0x3d2b04dd, 0xbc6033f4, 0xbccbc0f7, 0xbd0e3688,
      0xbc4bfcd8, 0xbd37700a, 0xbd4b06a7, 0x3c0ceeec, 0xbbdb7928, 0x3c47f720,
      0x3d3832a9, 0x3bd083d8, 0xbd420c63, 0xbd20b7cd, 0x3d284029, 0xbd2f3a1d,
      0x3cdc94ea, 0x3cc68052, 0xbc0ab8e0, 0x394d6e00, 0xbd1fc3aa, 0x3c4e2404,
      0x3d0adb4d, 0x3c6f5e74, 0x3d373d99, 0xbcd89817, 0xbc582354, 0xbb25eea0,
      0xbd33a903, 0xbcc14be7, 0x3b5d7630, 0xbc550a98, 0xbd280dfd, 0xbd412b6f,
      0xbcda4e57, 0xbb931290, 0xbcd13840, 0xbd378128, 0xbb4bacb0, 0xbc816b44,
      0x3cc4982e, 0xbbf372f0, 0xbc1ece18, 0xbc8989d0, 0x3d2dbdf9, 0xbd2d3ab0,
      0x3d4754e3, 0x3c4187f8, 0xbcbd2fdd, 0x3c945352, 0x3d080845, 0x3b240150,
      0x3c131a98, 0x3b7fc8a0, 0x3d282079, 0x3c047518, 0x3c9ccfca, 0x3d252367,
      0x3d14eb05, 0x3d2ee1b1, 0xbc832ce6, 0xbb9290b0, 0x3ced2af6, 0xbbcd5880,
      0xbd237b88, 0xbc38d38c, 0x3cd2775a, 0x3c209b68, 0xbcc059b3, 0xbc2d7688,
      0x3c3664a8, 0xbd444938, 0x3bb62998, 0x3cfce4ea, 0xbd2647d2, 0x3c4f8f54,
      0xbcc7f663, 0xbc706940, 0x3cf03666, 0x3c894e02, 0x3cdd4b22, 0x3d3058e5,
      0xbd178a16, 0xbd33a122, 0xbcaf84fa, 0x3d2b357f, 0xbbcc8510, 0xbcf1e24d,
      0x3d1811bb, 0x3d07983b, 0x3d00c77d, 0xbd367605, 0xbd4672e3, 0x3d0419c7,
      0x39b31800, 0xbd492abb, 0xbc9b6eea, 0x3be18d70, 0xbd41a34a, 0xbcfcf530,
      0x3cfcab42, 0xbd3e81a2, 0xbd421e7f, 0xbcc11efd, 0xbca63d6d, 0xbd331545,
      0xbd38f0bd, 0x3d496ed7, 0xbc17b734, 0x3c3b45f4, 0x3c64196c, 0xbd417f67,
      0x3d15ae6f, 0x3d14b5f5, 0x3c64e8bc, 0x3b57aae0, 0x3c5c3774, 0xbcca7973,
      0xbcded7b3, 0xbcb2267d, 0x3ca850b6, 0xbd09ca34, 0xbcfc9c53, 0xbc99dc4d,
      0xbd2dda8b, 0xbd104bc0, 0xbcd2fcc7, 0xbbbd1f80, 0x3ba3d618, 0x3b924eb0,
      0x3c0f8a6c, 0x3cc38ea2, 0xbca04520, 0x3b4b43d0, 0xbc6d4e08, 0x3c1c136c,
      0x3d0ad6ab, 0x3c7f40fc, 0x3d0add39, 0x3d06e91b, 0xb8853000, 0x3d46d18b,
      0x3c98251a, 0xbc107654, 0xbc49e4ec, 0xbc4a6e8c, 0xbcc6af4d, 0x3d181b39,
      0xbcf100ed, 0x3bed0c00, 0xbacbcf40, 0xbc2304c0, 0x3d1b6291, 0xba2194c0,
      0xbc3212ec, 0xbbecaeb0, 0xbd425452, 0xbcb6dac3, 0xbc86e604, 0x3cccd70a,
      0xbcc3d7aa, 0xbba5a570, 0x3c4da1fc, 0xbcbb9c3d, 0xbcf26c8d, 0xbd38e4c7,
      0xbd4ab0b3, 0xbb218ae0, 0x3cce9f6e, 0x3c6a84a4, 0x3c8fbf5a, 0x3c20d718,
      0x3cd7200a, 0xbcf3275d, 0xbca6530a, 0x3cd43cfe, 0x3d1aa751, 0x3d1daee3,
      0x3cbf75f2, 0xbb8c1c50, 0x3cf04506, 0xbd43d9c2, 0xbbe133c0, 0xbc95d02a,
      0x3a580cc0, 0x3d433091, 0xbd310a97, 0x3d22b219, 0xbd20c68d, 0xbcf093a3,
      0x3a90b0c0, 0xbcd4a277, 0xbcc4ea5d, 0x3ba52110, 0xbd4584b0, 0xbc4892e0,
      0x3cf9cef2, 0xbd202d7b, 0xbcf8329d, 0xb9317b00, 0xbb02cb60, 0x3d16a987,
      0x3ccd0ae2, 0xbd0e07bb, 0x3ce5afe2, 0xbcba3e53, 0xbd004140, 0x3c727284,
      0xbd3100aa, 0x3ce1384a, 0xbc7980ac, 0x3d220849, 0xbd3db48b, 0xbd401a28,
      0xbca574ea, 0xbc3922f4, 0x3d031b4f, 0xbd32a3f0, 0xbd2c5190, 0x3d1b5ce1,
      0x3c8da5b2, 0xbd1adf65, 0xbd3eaf7f, 0xbd40fb2d, 0xbc019894, 0xba3c1140,
      0xbcf569ad, 0x3bede0a8, 0x3b1b9230, 0xbd23010b, 0x3c740fcc, 0xbbd867c0,
      0xbc17c908, 0x3b348ca0, 0xbc5dd360, 0x3d2a569d, 0xbcdc6527, 0x3d15f95b,
      0x3c943d1a, 0x3b68f8d0, 0xbce9bb5a, 0xbc0014b4, 0x3d0229a5, 0xbd4ba5e0,
      0x3d13459b, 0xbab304c0, 0x3d053451, 0xbc52e2cc, 0x3c0c96a8, 0xbd334520,
      0x3cc7999a, 0xbafba400, 0x3c4b8ce8, 0x3d3f28c9, 0x3d3959cd, 0x3ca50e6e,
      0x3c64cc2c, 0xbd4c667b, 0xbbba0840, 0xbcf05baa, 0xbb70df60, 0x3c910432,
      0x3c84d512, 0xbd388aaa, 0x3c8acbf6, 0xbc3d9808, 0xbcda55a7, 0xbc24b518,
      0xbcc722f0, 0x3cad76be, 0x3c70c6dc, 0x3d2b11e3, 0x3d080f31, 0xbc220d2c,
      0xbd3703ba, 0xbd191162, 0xbc6c6f40, 0xbd1de1dd, 0x3d1235e5, 0x3d09d783,
      0x3ccdc1ee, 0xbd1bc0b0, 0x3d100d91, 0x3d328b8f, 0x3c9d09ae, 0x3ccd7882,
      0x3d4b1a4d, 0xbd093d0c, 0xbd4c717f, 0xbceb60ea, 0x3b2b4ea0, 0x3cf9e1ea,
      0xbd493907, 0x3d3ce3f1, 0x3d195011, 0xbca6a497, 0xbcc9e50d, 0xbcc9a8b7,
      0xbd2c719d, 0xbd1ed948, 0xbc243d94, 0xbcdb1f83, 0x3ca5dcfe, 0xbd4afb10,
      0x38343400, 0xbc8c7d06, 0x3d1dc93f, 0x3d4ada1d, 0xbc86d956, 0xbce683e3,
      0x3d0fffe1, 0x3b17b100, 0x3c475238, 0xbccf00f3, 0xbb9a41d0, 0xbd1a502d,
      0x3b5ba7d0, 0x3d45967d, 0xbd119e3b, 0xbc7f0188, 0xbd0cdef0, 0x3c0efb68,
      0x3d3dd0f3, 0xb7ac8000, 0xbcab8b77, 0x3cba91c6, 0xbc100de0, 0xbd4bd305,
      0xbbf6a4d8, 0xbca78a53, 0x3c83d052, 0x3d393393, 0x3ccea7ae, 0x3d1e4b01,
      0xbd2825a6, 0xbd18795e, 0x3c6bafd4, 0xbc644f88, 0xbd2ce9d7, 0xbc0d95d4,
      0x3c083834, 0x3b0057b0, 0x3cc75282, 0x3ce1beba, 0x3c3a97ec, 0x3bd0a898,
      0xbcd2478a, 0xbccdefdd, 0xbc0876a8, 0x3bfed400, 0x3cc8e346, 0xbc8e1f0a,
      0xbca92707, 0x39f45d00, 0x3c270728, 0xbc208c78, 0x3b499c00, 0x3d4866f5,
      0x3b1b1fb0, 0x3c9e40d2, 0xbd087ff6, 0x3ca2bef2, 0xbca468d3, 0xbca16b1a,
      0x3d3addf5, 0x3d0e80bf, 0xbc78d1ac, 0xbcf4ff6d, 0x3d12995b, 0x3b26b4d0,
      0xbd02b830, 0x3c2f7634, 0xbd38ff10, 0x3ca8f88e, 0xbcc0a01a, 0x3d3e36f3,
      0x3ce4f236, 0xbc57488c, 0xbc873f94, 0xbd078f10, 0x3c5c97fc, 0x3d26b433,
      0x3c5f45f4, 0xbcb806a7, 0xbcf658aa, 0xbd4a8470, 0x3d1ac939, 0xbbb171c0,
      0xbd00ee5e, 0xbc93b7e4, 0x3c21d4a8, 0x3d1a4def, 0xbd15782e, 0xbca9c733,
      0xbd0d9e3b, 0xbcfdea43, 0xbcbde660, 0x3cb42d8e, 0xbd206ac0, 0xbae99a00,
      0xbc220d0c, 0xbccb22e0, 0x3d166429, 0xbd068cfd, 0x3d05072b, 0xbcfbdd43,
      0xbcb96ea7, 0xbb806270, 0xbc42d22c, 0xbc99f550, 0x3d13b6ef, 0xbc7b5968,
      0xbcc11cb0, 0xbcd22397, 0x3d467733, 0x3d437e0f, 0x3ce33436, 0x3d45e69f,
      0xbcb4e1d3, 0xbc9d780d, 0xbd44eddb, 0xbc9f8fca, 0xbcf78a10, 0xbc667634,
      0xbbc440b0, 0x3c4219ac, 0x3bfc1290, 0xbabf0aa0, 0xbd0e8156, 0xbcd89f10,
      0xbd22bc6a, 0xbca2091d, 0xbd231f4b, 0xbbb9ed70, 0xbc4c8ce8, 0x3d302005,
      0xbce67d5d, 0x3d3315ab, 0x3d42b557, 0xbcfb3853, 0x3cbf22fa, 0x3c12c0b8,
      0x395ae800, 0xbd13572e, 0xbc916986, 0xbc828f20, 0xbd0918b5, 0xbc012328,
      0x3c289e98, 0x3d3b4c3b, 0xbcc988c0, 0xbce724a7, 0xbcba939d, 0x3d081539,
      0x3c1c8748, 0xbd27860b, 0xbbd36d68, 0xbd32ff08, 0x3a07c480, 0x3b68ad60,
      0xbc95b244, 0xbb803750, 0x3d304595, 0xbc1a6028, 0xbca8c7c3, 0xbd2183eb,
      0x3bfa09e8, 0xbcf657b7, 0x3bff8f70, 0xbc4a8ccc, 0xbd08d850, 0xbd2ac862,
      0xbc7f8300, 0x3cad9fc2, 0xbcbab96d, 0xbc097d78, 0xbc7fad2c, 0x3c0f1f14,
      0xbc849b46, 0xbd497d13, 0xbd00be2c, 0x3bb30530, 0xbd0d0112, 0xbc06f720,
      0xbc8ddc4c, 0xbcc89d13, 0x3d202a01, 0xbbaec7d8, 0x3d29e3b7, 0xbd1a09f5,
      0xbca13973, 0x3cd3cd26, 0x3cebb3f6, 0xbbe50af0, 0xbd35d98f, 0x3d1f7d17,
      0x3d236eef, 0xbb822f98, 0x3b77e3b0, 0x3d406aa1, 0xbccda04d, 0x3d213933,
      0xbd29efdd, 0xbb52e030, 0x3cc425a6, 0xbcad5aa3, 0xbd0edd9d, 0xbc4fd994,
      0x3c731dd4, 0xbc936a74, 0x3c092048, 0x3b8cdf68, 0xbd359ca3, 0x3a916860,
      0x3d16e051, 0xbc452278, 0x3cff2f52, 0xbc2aa378, 0x3b1f33e0, 0xbd1008a5,
      0x3d1396bd, 0xbbcee730, 0xbd32750b, 0x3c5e0074, 0xbd1d38e3, 0x3d17c565,
      0xbcc91663, 0xbc58e3a8, 0x3c7060e4, 0x3d0aa399, 0x3bf3e110, 0xbd23fdb5,
      0x3cee8352, 0x3d28a7f7, 0xbc4de580, 0x3cdc852a, 0x3d0e4c21, 0xbb4875b0,
      0xbbd2b018, 0xbd0cd62a, 0x3c750ec4, 0xbca804bd, 0x3b607880, 0x3cb1ab1a,
      0xbb9b9640, 0x3c425e0c, 0xbab5cfa0, 0xbd3bd7c3, 0x3d4cb99d, 0xbd2adf2d,
      0x399faf00, 0xbb450930, 0x3c62e114, 0xbcfb6890, 0x3d19b807, 0xbc333088,
      0x3ca1ce42, 0xbca8fe90, 0x3c00c3c8, 0x3d0f85ad, 0x3c4a3528, 0xb9c2e680,
      0x3b587fe0, 0xbc6fd8e0, 0x3ca98c0a, 0xbabaeb40, 0xbbb38168, 0xbcd55fda,
      0x3cabf766, 0xbbf93d10, 0x3d2666ab, 0xbccbd870, 0x3cb013da, 0xbc8de3f0,
      0x3c853306, 0x3ca6a16e, 0x3d439811, 0xbb590460, 0x3b920898, 0x3b85bc10,
      0xbce92ce7, 0x3c6c3284, 0xbafe8960, 0x3c945cc2, 0x3c754a7c, 0xbc2abab8,
      0x3c7b58dc, 0x3d08e483, 0xbd126588, 0xbc968340, 0x3d24cd49, 0x3cb3d2da,
      0xbd2d76eb, 0xbc813a44, 0xbd39e80d, 0x3cc53a6a, 0x3d0ebf09, 0xbbb9a7f0,
      0x3d0b9495, 0xbcee629d, 0x3ce14c82, 0x3c8c3152, 0xbbac1070, 0x3cf3a29e,
      0x3cf1d7da, 0x39dc3700, 0x3d485977, 0xba38fb80, 0x3cfcefb2, 0xbcc5326d,
      0xbd0244a4, 0x3ae3e240, 0x3ad2db40, 0xbd248bd0, 0x3d4c15c9, 0x3bbe53a8,
      0xbcc67bc0, 0xbd080328, 0x3b610de0, 0x3c2f094c, 0xbd40ed1d, 0xbcea71b3,
      0xbcf7154d, 0x3d30698f, 0x3cd21802, 0x3c18a814, 0xbcd07c67, 0x3cfa565e,
      0xbcef7d00, 0x3c8ba85e, 0xbc8159b0, 0xbca6ffcd, 0xbd05df9a, 0x3c309480,
      0xbd0d905e, 0x3d2f28ab, 0x3ab1e760, 0x3c6e6cc4, 0x3d0dced9, 0x3be71b70,
      0xbd01b3b6, 0x3d3f7f8b, 0xbbb3e6b0, 0x3c429918, 0x3cdf0662, 0xbba3ee28,
      0xbca5aaed, 0xbaa6f360, 0xbd352b5f, 0xbce29c30, 0x3bae5b50, 0xbcf5ecd3,
      0xbd1b9263, 0x3c6e55fc, 0x3d095799, 0x3cfc7d6a, 0x3c90a572, 0xbab16840,
      0x3cbcd04a, 0x3a97d940, 0xbd04a19c, 0xbd42e445, 0x3c595cd4, 0xbc7c71c0,
      0xbd31da0d, 0xbc962a74, 0xbd0c49b0, 0xbd1443b5, 0x3a8b8060, 0x3d2a8f6d,
      0xbc04f974, 0xbd1fdeb0, 0xbd3aed78, 0x3c4628e8, 0x3d2145d5, 0xbb6fd580,
      0xbc8fa2da, 0xbcced14a, 0xbadfd860, 0x3ce723f6, 0xbd28aca5, 0xbca54a13,
      0x3d45bed1, 0x3cd6db22, 0x3c8338ba, 0xbd45e5e7, 0xbd330b0d, 0xbce8685d,
      0xbd47ad03, 0x3c0cfcc0, 0xbd2a62ba, 0x3cbd023a, 0x3d49da49, 0x3c23ee28,
      0x3d2c5c47, 0xbcf8b1b0, 0xbd2c365b, 0x3c59734c, 0x3ce80486, 0x3d464e63,
      0xbd2d7b1f, 0xbc804414, 0x3d463d95, 0x3ce1367a, 0xbd332f6f, 0xbc972fda,
      0x3cca32e6, 0x3d23aff5, 0x3d3fb20d, 0xba892400, 0xbca38ac3, 0x3b883350,
      0xbcfe11c7, 0x3d3bf377, 0x3bc73210, 0xbc61e0ac, 0xbd131c43, 0x3a0ddc80,
      0xbca5ecbd, 0xbd0f1b78, 0x3c69512c, 0x3d35d1f1, 0x3cc28532, 0xbbff91c0,
      0x3b51c780, 0x3c03fcc8, 0x3cb255a2, 0x3c230300, 0x3d0815e7, 0x3bacb8c0,
      0xbd039c7a, 0xbb3584d0, 0x3d1bfac9, 0xbd3ae958, 0x3cefc6a2, 0x3c235ae8,
      0x3ccab992, 0xbd370b4b, 0x3a732200, 0xbd461592, 0x3cc961f6, 0x3c838242,
      0xbc9cced3, 0x3d27de81, 0xbc8344fc, 0xbc7faee8, 0xbd1e254b, 0x3d469e51,
      0x3ce20ebe, 0x3c2f144c, 0xbc357d2c, 0xbc3620e8, 0xbc04a334, 0x3c5956a4,
      0xbc8ba3c4, 0x3bca29e8, 0x3d17d1e3, 0xbba196e8, 0x3c8c295e, 0x3d2c4267,
      0x3c983e9e, 0x3d09932f, 0xb9cddb00, 0xbd090ac2, 0x3c2467e0, 0x39fd2400,
      0x3d0f0b43, 0x3ca1e1d6, 0xbba80d18, 0xbcc25020, 0xbcc3dcb0, 0xbbe231e8,
      0xbd26d855, 0x3adee9c0, 0x3d3ef06f, 0xbd2c23e5, 0x3d2cba01, 0x3cd42aca,
      0x3ac605a0, 0xbcc3951a, 0x3b32c4b0, 0x3ce38f9a, 0x3a6874c0, 0xbb147a00,
      0x3c7019a4, 0x3c9e6102, 0x3b0e2d80, 0x3c7dbafc, 0xbd20fbd8, 0x3d436619,
      0xbd434c55, 0x3bc58228, 0xbd3591bd, 0xbbd1a028, 0x3c163ff8, 0xba18cb80,
      0xbc6d2034, 0xbbc6aaf0, 0x3d1be929, 0x3cf2d14e, 0x3d3ecf11, 0xbce0bd70,
      0x3cf668b2, 0xbd304c52, 0x3d0f5a29, 0xbb3c8050, 0x3d2a76fd, 0x3cdfec42,
      0xbc131ed4, 0x3c8715da, 0xbced47e0, 0x3caca7c2, 0xbb68ff00, 0xbd2bfced,
      0x3c6bbf0c, 0xbd313687, 0xbba436a8, 0xbcd181d7, 0xbd37cf83, 0x3c5b8504,
      0xbd082a58, 0x3c96080e, 0x3cde49b2, 0x3a8d1bc0, 0xbd32c9b7, 0xbbaeaad0,
      0xbc80155c, 0xbc08e3a8, 0x3ca31582, 0xbbea7eb0, 0x3d4b33a9, 0x3cd27dda,
      0xbc883e6c, 0xbc9deb03, 0x3ceda292, 0xbc9d334a, 0x3cab4f56, 0x3d46cadd,
      0xbd339477, 0xb98b6900, 0x3c947fb6, 0x3d023c31, 0x3c99d8a2, 0xbd1473f8,
      0x3c3642c8, 0x3d2980c5, 0x3c5b1c54, 0x3d3bb0f1, 0xbd031e18, 0xbad1c9a0,
      0xbccc6d0a, 0x3c952096, 0xbcaa9d87, 0x3cf9b81e, 0x3bfe83a8, 0xbc9c417a,
      0x3af637c0, 0xbca5ffc3, 0x3cf64072, 0xbc8c5214, 0xbcb6240d, 0xbd30cb48,
      0xbc1c45cc, 0x3d3953f1, 0xbc29d26c, 0xbd33c0e5, 0xbd130e08, 0xbd2e02cb,
      0x3acbdc60, 0x3cef5bae, 0x3d0197ed, 0xbd1cff72, 0xbd11b5a0, 0x3d1b8873,
      0xbd38de4d, 0xbd476057, 0x3d239081, 0xbc05e78c, 0xbc94c6f0, 0x3d00f2b7,
      0xbbeb7c68, 0x3d307db1, 0x3d2f397f, 0x3d3b5935, 0x3c114f98, 0xbcc65a4a,
      0xbd34016d, 0xbd05a335, 0x3d0d3551, 0x3c59b1c4, 0xbd235a40, 0xbd0a2bea,
      0x3ccc2556, 0xbbfd6258, 0x3cd81886, 0x3d41dcc5, 0x3d37ecf7, 0x3cae1086,
      0x3c73a234, 0x3d1c71a9, 0xbd3ca15d, 0x3d43e907, 0x3c94baae, 0xbd4b5aca,
      0x3d09daff, 0x3c53a574, 0xbcf09773, 0x3b3b13b0, 0xbd27229d, 0x3d2593df,
      0xbd2c7f62, 0xbd1eca76, 0x3c0888c8, 0x3b860140, 0xbcb67bb0, 0xbcf435aa,
      0xbd2e8ce2, 0x3b89b750, 0xbccdf04a, 0xbcdbd9fd, 0xbc1118c0, 0xbd4c0207,
      0x3ca91bf2, 0x3d2e3cd1, 0xbc160cac, 0x3c9bfa22, 0x3c031e94, 0xbbd129b0,
      0x3d25f675, 0x3cda9792, 0x3d2aedb3, 0x3d412a1f, 0xbd0a4846, 0x3cdd4c76,
      0xbcc4248a, 0x3c27b0a0, 0x3a615940, 0xbc66b220, 0xbd2e8bb8, 0x3d49ae11,
      0x3d4332d9, 0xbcfc2100, 0xbd2ac383, 0x3cd667c6, 0x3d0c976d, 0x3c85c5fa,
      0x3ba20c28, 0x3cf6ef96, 0x3c4b5c68, 0xb9b6ba80, 0xbcbafbf7, 0x3b0a1ee0,
      0x3cee6332, 0xbc404a0c, 0xbc0f05f8, 0x3d1b3bcb, 0x3d4820bf, 0x3d2c90c9,
      0x3d0d3843, 0x3b7f07d0, 0xbc6e3cd4, 0xbd017f98, 0xbbe09b70, 0xbc564360,
      0x3d310a81, 0xbc68efa0, 0x3aaa1800, 0xbd4b4008, 0xbb92add0, 0x3d0a26d3,
      0xbb03ccb0, 0xbb88e0d8, 0xbd0d3143, 0x3cd98022, 0xbcfba76a, 0xbcb0efaa,
      0xbcb783ed, 0xbd2702ea, 0x3c23e634, 0xbd368ec2, 0x3bbb2b18, 0x3d43a38b,
      0x3c07f7f4, 0x3c0f2cc0, 0xbca0230a, 0xbd451f0a, 0xbc8313cc, 0x3d4670e1,
      0xbd406357, 0x3cbf59fe, 0xbca8e0ed, 0xbcb9bb3d, 0x3c817452, 0x3c900d2e,
      0x3bd8d158, 0xbd2977c3, 0xbc3dd788, 0x3d12260f, 0x3cff63ea, 0xbcdeb8c3,
      0xbced00da, 0x3ce76e82, 0xbcc8f677, 0xbc6648b4, 0xbd449ada, 0xbc9af66d,
      0xbcbf552d, 0x3cdb28da, 0x3a1a6680, 0xbd1d79c0, 0xbcef2c2a, 0xbbf520b0,
      0xbabc0a00, 0x3c8d280a, 0xbc989136, 0xbd0a489a, 0x3c368168, 0x3cc19ade,
      0x3d2c7f03, 0xbd322e52, 0x3cb94f62, 0x3d0b907d, 0xbcb2682a, 0x3c09f140,
      0x3bd4a1e8, 0x3d2550e5, 0xbced6c9d, 0x3d1c208f, 0x3d029b61, 0x3c80bfd6,
      0x3c868faa, 0xbcd907aa, 0xbd31def2, 0x3d1d9951, 0x3cd8f40a, 0xbcf5fbd0,
      0x3c9fcf6e, 0x3d32e6bf, 0xbc598380, 0xbd404c47, 0x3d030313, 0x3add26a0,
      0xbc23c368, 0xbcbc4ff7, 0xbcfb37d7, 0xbd0f0d1a, 0x3d2cea83, 0xbcfc20f7,
      0xbc3e6fa0, 0x3d28f981, 0xbc44ed28, 0xbc5752c0, 0x3bd6f0a8, 0x3d47bcb9,
      0xba1b8b80, 0x3d00db71, 0x3b4f5150, 0x3c180534, 0x3ac24e00, 0x3d23a575,
      0xbcb0afaa, 0x3c3df058, 0x3bdacd10, 0xbc2f4de8, 0xbcebcbad, 0xbc044674,
      0x3d2a7241, 0xbd351873, 0xbcc99800, 0x3c644aa4, 0xbc93dba0, 0x3bd56c70,
      0x3c22a874, 0x3c29316c, 0xbccde2fa, 0x3d04bf69, 0xbd2b2bd2, 0x3c24f6b4,
      0x3d006067, 0xbd016525, 0xba8bdcc0, 0x3c7f18dc, 0x3cfa8832, 0xbc4c5414,
      0xbcdd47ca, 0xbcfd17f3, 0x3d3dfcef, 0xbc986150, 0xbc7f99f8, 0x3d47203f,
      0x3c1df868, 0xbcb19b1d, 0xbcec124d, 0xbc249dac, 0x3c8d9db2, 0xbcb76dc7,
      0xbc90ab9a, 0x3d2d7e8b, 0xbd0ecbfb, 0x3b9ad180, 0x3d229639, 0xbd44e212,
      0x3c86b72e, 0xbc825a46, 0x3cb2e2c2, 0x3ce0e25a, 0x3ccd776a, 0xbbec5d28,
      0xbb71f950, 0x3c998342, 0xbc0e10a8, 0x3d38ba4b, 0x3d1626a9, 0x3cc00aa2,
      0xbd3bfb45, 0x3c43b2d8, 0xbc601b14, 0x3bae2280, 0xbb8abdd0, 0x3d3ef73d,
      0xbd47cbeb, 0x3d18422b, 0xbd079f7c, 0x3adfe460, 0x3d3962e7, 0xbd1ec823,
      0x3ce4f25a, 0xbc419248, 0x3d0f8593, 0x3d39e519, 0x3d279cd7, 0x3ca695e2,
      0xbce8d18a, 0x3c8369fe, 0x3c7b33dc, 0x3c92c912, 0xbd02a74e, 0xbce951ea,
      0x3cddb652, 0xbd438bb0, 0x3c670944, 0x3d077419, 0x3aee6d40, 0xbcccddda,
      0x3cd07792, 0x3aeb1140, 0x3d00ab6d, 0x3cdab052, 0xbc83a6c0, 0x3d378b65,
      0x3d18ca3f, 0x3b5e9dd0, 0xbca3cd5d, 0x3cc7db5e, 0x3cece702, 0xbcdb7367,
      0x3d2e6291, 0x3d23da33, 0x3c4d13ec, 0x3c9fef32, 0xbd25bf5b, 0xb9252b00,
      0x3ad8d6a0, 0x3c337420, 0xbb6c2bd0, 0x3d031713, 0xbc4f236c, 0x3c51b244,
      0xbc44ad4c, 0x3c9474f2, 0x3c063458, 0x3c13228c, 0xbbbba390, 0x3b4f2c60,
      0xbc20288c, 0xbc1c6ec0, 0x3d2342e9, 0x3c6b03fc, 0x3b9fd890, 0xbbe72070,
      0xbc351b0c, 0xbc4d3e14, 0x3cbe837a, 0xbb30cb30, 0x3ce17856, 0xbb8c5a58,
      0x3c074738, 0x3c382288, 0xbbcd2b28, 0x3c82507a, 0x3b2a0b60, 0x3d44130f,
      0x3c10d9e0, 0xbabce6a0, 0xbadf8600, 0xbb739c80, 0xbc600f80, 0x3c82c276,
      0xbd2226f5, 0x3d3ff37f, 0x3d4426ad, 0x3d22f737, 0xbc591d08, 0xbd24f663,
      0x3bdd6390, 0xbd386275, 0xbc866100, 0x3c695014, 0x3c814c0a, 0x3d3f3311,
      0xbc6bc1e0, 0x3d32ca43, 0x3cb7d7ae, 0xbba1e9d0, 0xbd4bd5fa, 0x3ba978d8,
      0xbca2af5a, 0xbb2bc200, 0x3cb7bb0a, 0x3d0ba59f, 0x3d169ef1, 0x3b0a2650,
      0x3d1fc229, 0x3cfa4662, 0x3c9529de, 0x3cd13772, 0x3cd6f05e, 0xbca93473,
      0xbcafe123, 0xbd02a278, 0x3c3c0cd4, 0x3c894c4a, 0x3c41bd00, 0x3c5ca0a4,
      0x3d1b717b, 0xbcd16950, 0xbc7de328, 0xbd3cf5ef, 0xba650800, 0xbd3e2408,
      0xbb54cbb0, 0xba0f8cc0, 0x3cd82822, 0x3d3d792f, 0xbc9516b4, 0x3b1d1d50,
      0x3d368979, 0x3c5e6dec, 0xbd3cf378, 0x3d3a8635, 0xbd4662e0, 0x3ca3eb6a,
      0x3bd87628, 0xbd4aa05b, 0x3cc1540a, 0x3d11f57f, 0x3c6448c4, 0x3a90a600,
      0xbd25e66b, 0xbd3333bf, 0xbc35e6e8, 0xbca0f943, 0x3b20bee0, 0xbd1881d8,
      0x398e8580, 0xbd1f24b5, 0xbc42176c, 0x3d46a8a7, 0x3d17a7fd, 0x3ca6c69a,
      0xbc153748, 0x3bb1acf0, 0xbd2a041b, 0x3caf685e, 0x3ac27160, 0x3c1830a0,
      0xbc5498b8, 0x3c462634, 0x3d08fa25, 0xbd1eb5a2, 0xbc1f14d4, 0xbceeee57,
      0x3ceae45a, 0x3c4c2028, 0xbca0930a, 0xbcad99ed, 0xbd01bb5a, 0xbc541b68,
      0x3d47d671, 0xbc8964d0, 0xbc3b78f8, 0x3cbf18fa, 0x3d2a8f6d, 0x3bad6668,
      0xbcc31657, 0xbcdf69d7, 0xbc216f8c, 0xbc76a434, 0x3d06df89, 0xbd2d9123,
      0x3c8ffc22, 0x3cd98b1a, 0xbb93ef10, 0x3d4a7163, 0x3d0d6471, 0x3c02b808,
      0x3b9e7940, 0xbc331560, 0x3cfa9c82, 0x3cd98a2a, 0x3ad2af00, 0x3d16e8bf,
      0x3d04c911, 0xbcb0a740, 0x3d0eae19, 0x3d42eb55, 0x3c9cf206, 0x3d3a18c9,
      0xbb4e7e50, 0xb9f4ad00, 0xbcf3437a, 0xbd2d651f, 0x3c2297ac, 0xbd3bb2c8,
      0xbc5efd4c, 0xbc949774, 0x3cc4f6a2, 0xbd0a815a, 0x3cee9902, 0xbcbb15a0,
      0x3c82e192, 0xbd1b7e8e, 0xbcf11be0, 0x3bbbe510, 0xbce9d433, 0xbd13d5bb,
      0xbc6815ec, 0x3c89ceb2, 0x3cee4ede, 0x3c6b3384, 0xbd112576, 0xbcda1fa3,
      0xbc8a3dca, 0x3c51d724, 0x3cf2124a, 0xbbe8eeb0, 0xbcdb7f5d, 0xbd2cc46e,
      0x3d3909f3, 0x3c75b3fc, 0x3d1b4d4f, 0x3c8dcb66, 0xbbf7bad0, 0x3c82e00a,
      0xbca273e7, 0xba8bc8a0, 0xbc7053f8, 0x3c9c67ae, 0xbb958c40, 0x3c20db00,
      0x3c1b5a28, 0xbc9967d0, 0x3ca42a9e, 0xbce59ef3, 0xbd31c562, 0xbd01404c,
      0x3d06f385, 0xbc8bcd74, 0xbb05c3b0, 0x3cbbf1f6, 0xbcf06560, 0x3d13e9e9,
      0x3c083118, 0xbd183ebb, 0x3cda6dd6, 0xbd29999b, 0xbabd2ea0, 0xbce821b0,
      0x3c419c60, 0xbd2b8af8, 0x3d1f3849, 0xbca0c1ca, 0x3c5a8f1c, 0x3d1ce21b,
      0xbcaf98e0, 0x3d3c0893, 0x3d0a853f, 0x3cf646aa, 0x39affb00, 0xbd389690,
      0xbd4b39d3, 0xbb503720, 0xbbb53590, 0xbbd704b0, 0xbc37d514, 0xbd0719dd,
      0xbae6c6a0, 0xbcdbf147, 0xbc20dd08, 0xbd4c05fd, 0xbc81f7f0, 0x3bf4ba30,
      0x3cd79452, 0x3d452637, 0xbc461978, 0x3beec000, 0x3d338637, 0x3c9bf462,
      0xbd32ee0f, 0x3c22b3a0, 0x3d29b317, 0x3d3c7313, 0xbc376740, 0x3c8c37a2,
      0x3d0ca591, 0x3b46b2a0, 0xbc4f2848, 0x3c721f2c, 0x3c8cd96e, 0xba25f740,
      0xbbd8b2e8, 0xbb5a3650, 0xbc22d698, 0x3cd440fe, 0x3d1f4db9, 0x3d4323b9,
      0x39689e00, 0xbd07b34e, 0xbccfa89a, 0xbb9e7b28, 0xbd494eaa, 0xbd385b07,
      0xbbb5fa98, 0xbcbaf4d7, 0x3cc7dc46, 0xbcb7a5dd, 0xbb0a16b0, 0xbb51f160,
      0xbd3c0b1a, 0xbc1142ec, 0xbd3f8dd5, 0xba843260, 0x3ca5cc22, 0xbd26a015,
      0xbce361f0, 0xbc10a48c, 0x3c9f7b6e, 0x3c9287de, 0xbc81e2a4, 0xbd37b89b,
      0x3d480471, 0xbd14a0eb, 0x3d234b61, 0xbc89835c, 0xbcbccc1d, 0xbd291efa,
      0xbcf1d68d, 0xbbd96c40, 0xbcb922aa, 0x3c80bdfe, 0x3c7c8024, 0xbd105d62,
      0x3d244d31, 0x3cbbe22a, 0xbcb32eb7, 0xbcd1cb73, 0x3d0e8799, 0xbb920a68,
      0xbd2e2b60, 0x3cbdb9e2, 0xbcfa0777, 0xbd06be54, 0xbd24d3bb, 0x3d3683c3,
      0x3ceffe3a, 0x3ccc9cca, 0x3c3e2b00, 0x3ca3238e, 0xbd37e2b0, 0x3d11c961,
      0xbd4ae8a3, 0xbd486c65, 0xbb8237e8, 0x3d30f539, 0x3d14c629, 0xbd4193eb,
      0x3d26de35, 0xbd25110b, 0xbd1cc35a, 0x3c810422, 0x3d3cb60d, 0x3d48e591,
      0xbd044924, 0x39545e00, 0x3d09ce5f, 0x3cef5336, 0xbb5d5b50, 0xbd037c0c,
      0xbcb4b237, 0x3d4a11b9, 0xbcf4825a, 0xbd168eca, 0xbd2f5fad, 0xbba23d80,
      0x3ceb122e, 0x3b070ed0, 0x3c4e9b4c, 0x3c580244, 0xbd461647, 0xbbc52830,
      0x3d2c6e15, 0xbc8c15cc, 0xbd0d8fd2, 0x3be4a1f0, 0xbc210068, 0x3ca9a456,
      0x3cc74eba, 0xbd1a8588, 0xbc784c48, 0x3c8cfe52, 0x3d2dafa9, 0xbc666754,
      0x3cbad202, 0xbbdb5b28, 0x3c49e0f8, 0xbd3035cf, 0x3cc6bd0e, 0x3d17fb77,
      0x3b60c620, 0xbd34bfc3, 0x3cdd6aa6, 0xbd1da1de, 0xbd1d27b2, 0x3ba27e28,
      0x3cde5c2a, 0xbd4c18b2, 0xbcbcc0fd, 0x3b6fb6e0, 0xbc227260, 0x3cc3e3e2,
      0x3cda3926, 0x3c0f5880, 0x3d452a2f, 0xbcca98d0, 0xbd462d60, 0xbd0ba370,
      0x3cd64fb2, 0xbd4a8e37, 0xbd05dfee, 0xbc1a9bd4, 0xbd268438, 0xbcf40b2a,
      0xbd4a88bd, 0x3c603f74, 0xbba3e3f0, 0xbbd827a8, 0x3c8485b2, 0xbd3ee2c2,
      0xbd466335, 0x3c846b4a, 0xbd3703c0, 0xbd0ffab3, 0xbca240fd, 0x3ceacad2,
      0x3c4fbdb4, 0x3c0c45c8, 0x3d05a8d5, 0xbc5c3f28, 0xbd3ea837, 0xbd129b55,
      0x3cb3689a, 0x3d26abd1, 0x3d0cf0e3, 0xbcbe0683, 0x3ce1872a, 0xbc4cca28,
      0xbc85cbca, 0xbb3e8460, 0xbd0e79e3, 0x3c89b682, 0x3d382369, 0xbd0e41a0,
      0x3c99454a, 0xbad781c0, 0xbc811614, 0xbd37d59f, 0xbcc4fdb3, 0x3b3baa60,
      0x3d470b9b, 0xbcb15893, 0xbd2e08ef, 0xbcab4813, 0xbbdd75e8, 0x3d092ff3,
      0x3d091ac5, 0xbcbe0f03, 0x3d009871, 0xbd1deac2, 0x3d47da6f, 0xbc7323f8,
      0x3ce8096e, 0xbcc2410d, 0xbcffbc97, 0xbbbd9830, 0x3d459729, 0xbc136060,
      0xbd0330e4, 0xbce041ed, 0x3c98ac5a, 0xbd10a4b2, 0xbd3e3037, 0xbd206468,
      0x3d34e981, 0x3c389ea0, 0xbd242522, 0xbcbe9850, 0xbcd60ee7, 0xbcfb070d,
      0xbb028f80, 0xbbea97e8, 0xbbaa1f28, 0x3d18b097, 0xba530cc0, 0x3d1a05c9,
      0xbd17b3ba, 0x3c81adf2, 0x3d21a6a3, 0xbd302f33, 0xbd28c162, 0xbc43e194,
      0x3c277c58, 0xbcd14130, 0xbb89d3a8, 0xbc3f92d8, 0x3d3b5e07, 0x3bdde368,
      0xbcec6d4d, 0xbbbdede8, 0xbabb21c0, 0x3cddbbd6, 0xbd25cc2e, 0xbc6c92c8,
      0xbccb1030, 0xbcdc1163, 0x3cfb8c12, 0x3d3f2e85, 0xbd3707b8, 0x3c282b20,
      0x3b7145d0, 0xbd115813, 0xbbc6f800, 0xbd103956, 0x3ba25528, 0xbd2697ab,
      0x3cfb773a, 0x3d38ad2f, 0x3bf5df80, 0x3c631b0c, 0x3d46ce7d, 0xbc743eec,
      0xbc589f8c, 0xbd3a9070, 0xbd2e9e9b, 0xbccaef27, 0xbcf61793, 0xbcfd47a0,
      0xbd048d2d, 0x3c33edc8, 0xbca6d920, 0x3d16f5a3, 0x3bd1a650, 0xbc916a34,
      0x3ca1a002, 0x3b86b698, 0x3cc09626, 0x3d382fdf, 0x3cd125ba, 0xbcc69920,
      0x3bd58e18, 0xbb379360, 0x3ccf4b92, 0x3d3c2fd1, 0x3be5cd10, 0x3926e600,
      0x3d1a42b1, 0x3c4412cc, 0xbc251cac, 0xbcba31ea, 0x3c98b6b2, 0xbbb536b0,
      0x3c8b7ca6, 0x3cb01d82, 0x3cac849a, 0x3c575ec4, 0xbc6ff768, 0xbd43457b,
      0x3bc20340, 0xbcfe39ba, 0xbd2dcad5, 0x3d1c6923, 0x3d20d2a9, 0x3ccd6d42,
      0x3d140969, 0xbd47ea7f, 0xbc9d1967, 0xbad11440, 0x3d3fd6b3, 0x3d0406db,
      0xbcd0d390, 0x3d0117c9, 0xbb4abfd0, 0x3ca4b0a2, 0x3d3c14df, 0xbcc52653,
      0x3be00400, 0xbc633560, 0x3b9ba198, 0xbca1ecad, 0xbd148732, 0xbcf05240,
      0x3d3c4535, 0xbd2df2a7, 0x3bdc7bf0, 0x3d1a9d01, 0x3b04afd0, 0xbcbf7093,
      0x3d10cf11, 0xbd20fecb, 0x3c5a2294, 0x3cfaa8c2, 0x3d4544a1, 0xbb778fb0,
      0x3bd6c468, 0x3c533e64, 0xbb03f380, 0xba8cc760, 0xbd1b780a, 0xbc33f834,
      0x3ca93136, 0xbcee5fa7, 0x3d4824bd, 0xbc8c2364, 0xbc96c32c, 0x3b5274b0,
      0xbd40acb2, 0xbb0aa3b0, 0x3c5e3a04, 0x3cb05e5a, 0xbbf5a490, 0xbd469270,
      0xbcb1613d, 0x3c4d4104, 0x3d29fd19, 0xbd3ca957, 0xbd367eca, 0xbcf4b8b0,
      0xbd4899d8, 0x3c4ad04c, 0x3cd504aa, 0xbd292aa0, 0xbc93fb1a, 0xb8927000,
      0xbcb399bd, 0xbcb1882d, 0x3cdf1e82, 0xbd154a58, 0xbba65590, 0x3d223bf5,
      0xba21a2c0, 0x3c9cadfe, 0xbccd19c3, 0xbd063e1e, 0x3d2fa8af, 0xbcaad777,
      0xbd493cf5, 0xba19c780, 0x3cdf4afe, 0x3cf71c46, 0xbd0e8150, 0x3d2b94df,
      0x3c9890e6, 0xbc875256, 0xbb92a798, 0xb8d05400, 0x3b83e610, 0xbcf30377,
      0xbc970b7a, 0x3cb85f32, 0x3d0aeb31, 0xbd100dc5, 0xbd2ec743, 0xba81f1a0,
      0xbcd2f36a, 0x3c8b8912, 0x3cd213ce, 0xbcd8505a, 0x3caf84ca, 0xbd1a1f43,
      0xbd22fc05, 0xbc38fb40, 0x3c29ffa8, 0x3d21e4f9, 0x3d336049, 0xbc29fb14,
      0x3d4c8f65, 0x3d0156b9, 0xbc9c1a63, 0x3bf1d810, 0x3d2f3379, 0xbcc6024d,
      0xbd2b784e, 0x3cc61f72, 0x3bcad3e8, 0x3d1d16c7, 0x3c493368, 0x3d4a3853,
      0x3d2f9a0f, 0xbd18cc55, 0x3ca27c92, 0xbc0e0578, 0x3d2f9f6b, 0x3d25c15f,
      0xbccba443, 0x3d2861f9, 0x3cdd1c26, 0xb9bba980, 0x3c215ce8, 0xbc6fe358,
      0xbd436fd3, 0xbc5fa958, 0xbcfd9ef3, 0xbc2e3d88, 0x3c9630be, 0xbd019f08,
      0x3c552b0c, 0x3ccead72, 0x3d3161b5, 0xbd349167, 0x3cfb291a, 0x3baf3a70,
      0xbd30eaef, 0x3d36d16d, 0xbbff9db0, 0xbd05cfe5, 0xbd46d333, 0x3a5d36c0,
      0xbd2f322b, 0x3c6ea574, 0x3cc23a2a, 0xbd087a4d, 0x3c9e21b6, 0x3c8b4572,
      0xbcfb10fd, 0x3d256731, 0x3ca1cd0e, 0xbd4060a8, 0x3c9c80e2, 0x3d0bb7b1,
      0x3caec47a, 0xbca2cfaa, 0xbcd33083, 0xbbd930f0, 0x3d2a8e01, 0x3a034b80,
      0x3c964966, 0x3d2e454f, 0xbd1daa35, 0x3d42e051, 0x3cb0dc8e, 0xbd03e9f0,
      0x3ce23c82, 0x3d2b9c51, 0xbad26360, 0x3cf6b6c2, 0x3c5ccecc, 0x3d0d4d23,
      0xbd2023dd, 0xbd080fdd, 0x3d27cddf, 0x3d4c3a39, 0x3c8303fa, 0x3cce2002,
      0xbd420ceb, 0x3ce895e2, 0x3d1dd9a3, 0xbc269ba0, 0xbcce26cd, 0x3ce6a7ea,
      0x3cbdf30e, 0xbd48fe87, 0x3c5c97a4, 0x3c961dfa, 0x3c323fb4, 0x3d1aa5ef,
      0xbb308e50, 0x3d0699af, 0x3cbf1eb2, 0xbd0a3460, 0x3ba9a618, 0xbcdfe007,
      0xbc13b634, 0xbc5bbbe0, 0x3d2a4e3f, 0xbcd5f22a, 0x3c76f9f4, 0xbc9b65cd,
      0x3cb59b36, 0xbcaa9fd0, 0x3ccb71da, 0xbd38c728, 0x3cc6f0ca, 0xbd1d5c6a,
      0x3d320255, 0xbd3a9ed5, 0x3b3d4930, 0xbd3aaa4d, 0x3c9e2a82, 0x3be26210,
      0x3b52f560, 0x3cbaf15a, 0xbc9efa8a, 0xbd0726e6, 0xbd2c5ebd, 0xbd0af8a2,
      0x3d26a0d7, 0x3cc926b6,
  };

  // 5
  uint32_t bias_vals[] = {
      0x3bded4d8, 0x3c9d39d2, 0x3ca89fd2, 0xbc5af538, 0xbcb69fcd,
  };

  // 3,1,1,5
  uint32_t output_exp_vals[] = {
      0x3c0f5041, 0xbd5feb0d, 0xbe2ac302, 0x3e4629df, 0xbf31fe38,
      0x3e5c01b4, 0x3e7c96f6, 0xbce63e5a, 0x3e379fba, 0xbf3027ad,
      0xbdb021b6, 0xbe97d08d, 0xbef57ffa, 0xbdfbe7fc, 0xbf1bf24c,
  };

  // 3,1,1,5
  uint32_t output_relu_exp_vals[] = {
      0x3c0f5041, 0x0, 0x0, 0x3e4629df, 0x0, 0x3e5c01b4, 0x3e7c96f6, 0x0,
      0x3e379fba, 0x0, 0x0, 0x0,        0x0, 0x0,        0x0,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, VALID_PADDING, NULL);
}

void test_valid_padding_zero_strides_medium_with_clip() {
  input_set *set = &medium_input;
  strides_input_set *strides = &zero_strides;

  // 3,10,8,5
  uint32_t input_vals[] = {
      0x3f36b631, 0x3b9ca600, 0x3f76ac7f, 0x3d3f3570, 0x3e70e1b4, 0x3e3830f8,
      0x3f3f9678, 0x3e1fb9dc, 0x3dd55dd0, 0x3aef4000, 0x3f433347, 0x3df08b90,
      0x3eb33e84, 0x3dc553c8, 0x3ee7a2f2, 0x3e377d78, 0x3f3e7fdb, 0x3f1236d7,
      0x3ea73b60, 0x3e0cc410, 0x3f57cffe, 0x3f1253af, 0x3eeea1d6, 0x3f116d9b,
      0x3de6adb0, 0x3f335bd0, 0x3f459139, 0x3f221418, 0x3e089dac, 0x3f208980,
      0x3ec148ac, 0x3dfe44a8, 0x3f434e54, 0x3f207099, 0x3f6d610b, 0x3f4929fb,
      0x3f6fef6c, 0x3e96ad46, 0x3f4d17d4, 0x3e9b4166, 0x3eabb14a, 0x3b020c00,
      0x3f09f77a, 0x3f68586f, 0x3ef184c6, 0x3f1b4509, 0x3df6bf78, 0x3f73cdae,
      0x3f71564c, 0x3e0d0f10, 0x3f25afae, 0x3f4cc291, 0x3e5e4880, 0x3f544618,
      0x3f165f1a, 0x3f0d128c, 0x3f0459a7, 0x3e24cc88, 0x3eb49e0a, 0x3f247e0f,
      0x3f19b458, 0x3e607e7c, 0x3deeb150, 0x3ee86844, 0x3ed8b3e2, 0x3efaeafa,
      0x3eca4cde, 0x3f4c835c, 0x3f71404c, 0x3f6162dc, 0x3eb60aea, 0x3d6169d0,
      0x3d298250, 0x3f3facbb, 0x3dc7db50, 0x3f476996, 0x3d84cdb8, 0x3e8381c8,
      0x3ebc1c56, 0x3edda720, 0x3f327cb8, 0x3ec91382, 0x3f567352, 0x3f60a3cf,
      0x3e63cdd0, 0x3f2fbe4e, 0x3f61a525, 0x3e4d10dc, 0x3d8119d8, 0x3f706446,
      0x3f566a2d, 0x3f263a00, 0x3f68176a, 0x3f6d8d15, 0x3f6c6af8, 0x3e741c84,
      0x3f182317, 0x3e885446, 0x3f479de4, 0x3f7de85f, 0x3ec4857c, 0x3ebc271a,
      0x3f143fe7, 0x3e0877c4, 0x3f75c402, 0x3ef8b408, 0x3f748088, 0x3f10c163,
      0x3dbccff0, 0x3f32b1fc, 0x3f674b45, 0x3f0139d4, 0x3ef9b882, 0x3f64196c,
      0x3defe888, 0x3dd093e8, 0x3f2b3835, 0x3f4fb95d, 0x3f7e460c, 0x3e24b3c8,
      0x3e5a5130, 0x3f0208d2, 0x3f6cb694, 0x3f0b2b70, 0x3f1eaf41, 0x3ec30552,
      0x3f13f950, 0x3e405394, 0x3e9bf8c4, 0x3d5b8a00, 0x3d954c60, 0x3f2aaf0d,
      0x3f47c963, 0x3f48e285, 0x3f5108e1, 0x3dd6f800, 0x3f623988, 0x3f22332e,
      0x3f39bf14, 0x3f37a5a6, 0x3f57aec5, 0x3f49ad86, 0x3efb3ac4, 0x3ee00cf8,
      0x3f1e8a44, 0x3f05f896, 0x3f10619b, 0x3e1a016c, 0x3f667a9f, 0x3d823af0,
      0x3dc83748, 0x3f332ca5, 0x3f25f83e, 0x3f52f762, 0x3f5d1d04, 0x3ed5bac2,
      0x3eb30346, 0x3f5ce83e, 0x3f1fe917, 0x3ee4e7f6, 0x3ef79562, 0x3f577e28,
      0x3d910f88, 0x3f427613, 0x3e94929e, 0x3ea4fce6, 0x3f23ee9a, 0x3d1c8fb0,
      0x3d01a550, 0x3f42d47f, 0x3da77a08, 0x3f60995e, 0x3c0a9380, 0x3e88165e,
      0x3f0ad115, 0x3f45a98a, 0x3ed80798, 0x3f1d936a, 0x3f0e07fa, 0x3e8e1ef2,
      0x3ee0618c, 0x3dbbedc8, 0x3eb6713c, 0x3f22cf06, 0x3f1df573, 0x3f7cf87a,
      0x3f19492d, 0x3eb07ab6, 0x3edd0088, 0x3e2a9198, 0x3defc080, 0x3f340f50,
      0x3f773033, 0x3f3054c9, 0x3f76f70a, 0x3e66eb1c, 0x3f676d3a, 0x3ebeb408,
      0x3f26fe18, 0x3f703110, 0x3ef2b336, 0x3f319c97, 0x3e3adb10, 0x3d1b3af0,
      0x3f3b0802, 0x3e1b7498, 0x3f2f4afe, 0x3f0b8386, 0x3f285fa0, 0x3f43b72e,
      0x3f5d486b, 0x3f65dcf2, 0x3e66fba4, 0x3eeb2f94, 0x3b42bd00, 0x3f62fc54,
      0x3ea805d4, 0x3eb4fe52, 0x3e1fab68, 0x3e6362b8, 0x3f73f5a9, 0x3f7103c8,
      0x3f439d03, 0x3f0451db, 0x3c9a8dc0, 0x3e3ccb48, 0x3f419ff7, 0x3f1d8aa7,
      0x3ec10a20, 0x3e752480, 0x3f5b1273, 0x3f557bf6, 0x3e20a7b0, 0x3e701b88,
      0x3daf0738, 0x3f58dfea, 0x3f5b49d0, 0x3f79dc64, 0x3d35c960, 0x3f5e3797,
      0x3e9a238c, 0x3dbb67a8, 0x3f56999b, 0x3f6b147d, 0x3f450ebd, 0x3f0e0ba6,
      0x3f42ab7d, 0x3e22fa34, 0x3e491070, 0x3f4b6181, 0x3f3cfea2, 0x3f773612,
      0x3f37c12c, 0x3f11311c, 0x3e8e5a48, 0x3f5b9871, 0x3e184f28, 0x3f349900,
      0x3f668a71, 0x3f618f59, 0x3d072ca0, 0x3eea0f1e, 0x3e109d28, 0x3edfbf9c,
      0x3edb0210, 0x3f7f6eff, 0x3e478798, 0x3e182b74, 0x3e306d94, 0x3ebdac5e,
      0x3ebfa2ce, 0x3ec22efa, 0x3db01a70, 0x3c854b20, 0x3f44eb5b, 0x3f342728,
      0x3f131358, 0x3f390480, 0x3f154485, 0x3f54c400, 0x3e4200a0, 0x3f4477e4,
      0x3f164d00, 0x3e7f1908, 0x3e10028c, 0x3debfcd8, 0x3eb5adde, 0x3f361860,
      0x3f0955a3, 0x3f79dc1a, 0x3e254178, 0x3cd96800, 0x3e8f8ee8, 0x3f28e2cb,
      0x3ee6297c, 0x3f4b6ffb, 0x3f451d83, 0x3f603f0b, 0x3eb77766, 0x3f22176f,
      0x3e81b948, 0x3f47cfda, 0x3f10581d, 0x3f56ca4d, 0x3b0bb400, 0x3f4a6862,
      0x3ed19728, 0x3dceec28, 0x3edeafc6, 0x3f0e3fea, 0x3f26b23b, 0x3f6d93ef,
      0x3ebf908a, 0x3e99d570, 0x3e814d2c, 0x3f32ea59, 0x3f62acaf, 0x3f4152b9,
      0x3e2d777c, 0x3e3f6a7c, 0x3ec70f36, 0x3ec7937c, 0x3f1c251e, 0x3f26d5de,
      0x3e951d02, 0x3eaf0570, 0x3f04f161, 0x3ed8d624, 0x3e437414, 0x3efeceba,
      0x3e8838fe, 0x3f168c18, 0x3e1a49a8, 0x3f1ae81d, 0x3eb71206, 0x3e61cee8,
      0x3ecdd5a4, 0x3e418040, 0x3edfb0b6, 0x3e38040c, 0x3f76051f, 0x3ec0ee50,
      0x3f6e6777, 0x3f57f7f9, 0x3f5ba562, 0x3ef21660, 0x3e8c7112, 0x3e5ab544,
      0x3e827f6a, 0x3f261474, 0x3e9237d6, 0x3e5cd520, 0x3f5ab4cb, 0x3e81e382,
      0x3d0e0a40, 0x3d0e87e0, 0x3f3fecc7, 0x3eaa5012, 0x3edeea7a, 0x3f1ad9aa,
      0x3f059183, 0x3eb3fbb8, 0x3f014789, 0x3e4747a0, 0x3f1f716e, 0x3b714000,
      0x3e852fde, 0x3c56b680, 0x3f30cc9a, 0x3f04b414, 0x3efcfc4a, 0x3f70c129,
      0x3eacc898, 0x3ea23a38, 0x3f1f14e9, 0x3f50d1f7, 0x3f147713, 0x3f6be6c0,
      0x3f3483e7, 0x3eea910a, 0x3e591a9c, 0x3f5c5ad6, 0x3d514e70, 0x3f4a49a7,
      0x3f37fad8, 0x3f46540c, 0x3f74a543, 0x3f40679c, 0x3f4abe61, 0x3e178ffc,
      0x3eae56a8, 0x3edbefee, 0x3da237b0, 0x3ed100d2, 0x3e92d8a6, 0x3dec6e58,
      0x3f1317ce, 0x3f12ab4d, 0x3f45daaf, 0x3f5fe2b6, 0x3e42cb94, 0x3f7a260c,
      0x3f0d94d4, 0x3e2f4678, 0x3f5da986, 0x3f3ecd03, 0x3f6e32c6, 0x3eb08d0e,
      0x3ea099be, 0x3e11011c, 0x3f766f53, 0x3e6611c4, 0x3f5d2b28, 0x3ee70ede,
      0x3f658ee8, 0x3d2f4970, 0x3f1d4a31, 0x3d9109c0, 0x3ce695e0, 0x3f7f63bf,
      0x3e8fd860, 0x3e3e820c, 0x3f0d6155, 0x3f032f81, 0x3f10d55d, 0x3f6d3b5e,
      0x3f02d3a0, 0x3f56b825, 0x3f146fa0, 0x3de117f0, 0x3ef0853c, 0x3f6a0c73,
      0x3de4ad70, 0x3e9a66fc, 0x3f18cfc5, 0x3f1944d9, 0x3f3f06b1, 0x3ef8a9a6,
      0x3f298468, 0x3ecc4c4c, 0x3f123787, 0x3d1cee30, 0x3f3b25da, 0x3f6fd971,
      0x3f5c07c3, 0x3f34bfaa, 0x3e91d31a, 0x3f33a6b7, 0x3ed4457e, 0x3ca0a0e0,
      0x3e968518, 0x3da00258, 0x3f0f0d86, 0x3e6f51dc, 0x3f20c172, 0x3f109e04,
      0x3f3c4c11, 0x3f33be00, 0x3d69d2d0, 0x3efcff56, 0x3f5730f0, 0x3edf8088,
      0x3e1b1620, 0x3e0a22dc, 0x3f0a1397, 0x3eea6736, 0x3e09b75c, 0x3f6a9bfc,
      0x3e64d1b8, 0x3f580cd8, 0x3d290cc0, 0x3eae325e, 0x3f02dad5, 0x3e057dd8,
      0x3f024ff2, 0x3f2c3e6f, 0x3f2a014a, 0x3f52bd7b, 0x3e9b5a80, 0x3dc798e8,
      0x3ebea594, 0x3f0bd9a6, 0x3f1c50ef, 0x3f468365, 0x3e5554cc, 0x3ec9fdf8,
      0x3f781a74, 0x3ee3854a, 0x3f280dd5, 0x3e997f22, 0x3f42c1ab, 0x3efc99ea,
      0x3f36d3c3, 0x3f735e96, 0x3f0da6b8, 0x3f166eb5, 0x3f284366, 0x3f7982ae,
      0x3f7f159b, 0x3f31652b, 0x3e4a43b8, 0x3eb27360, 0x3f7e42e8, 0x3ea3d00a,
      0x3ed03742, 0x3e89dd14, 0x3f0f4624, 0x3da20d48, 0x3e9bbeda, 0x3e9af660,
      0x3ed915e6, 0x3f0495dd, 0x3f07c35c, 0x3f118423, 0x3f382ace, 0x3ec0c056,
      0x3f6df319, 0x3e1030fc, 0x3dd09b68, 0x3c62da80, 0x3ec9ce0c, 0x3f35a223,
      0x3f4a9439, 0x3f459f23, 0x3f5971aa, 0x3e220b1c, 0x3e5c371c, 0x3f5be3f7,
      0x3ed1ecf8, 0x3ee23da0, 0x3f5cba00, 0x3d643c00, 0x3e187990, 0x3da69340,
      0x3f5910ef, 0x3f426233, 0x3cc6e800, 0x3ef1344e, 0x3dfae618, 0x3f60b7f5,
      0x3f1942a2, 0x3ea30810, 0x3f5892c1, 0x3e1f3270, 0x3f0dac81, 0x3f1cfb60,
      0x3f108bc3, 0x3f54a5b3, 0x3f232e5b, 0x3f287d6a, 0x3f0efc17, 0x3f786b06,
      0x3dfe15e8, 0x3f621efb, 0x3e5e68d4, 0x3f5fb5c0, 0x3ee6d8ca, 0x3f55f16d,
      0x3f656889, 0x3f5bb5c0, 0x3f1cf6c7, 0x3f320bfc, 0x3ed2ad8a, 0x3eb98256,
      0x3f4aeaf3, 0x3f137f9e, 0x3f410d87, 0x3d991dd0, 0x3e569a0c, 0x3ed7cafe,
      0x3e668c48, 0x3def9d98, 0x3f197f5b, 0x3e8e60d6, 0x3f31e438, 0x3f2ab320,
      0x3db0ee68, 0x3f3ba15f, 0x3e223e2c, 0x3ef1a91e, 0x3d040020, 0x3f308052,
      0x3f4cc3a8, 0x3ebdff66, 0x3f640150, 0x3f285ea6, 0x3f3ba978, 0x3e23e124,
      0x3d465d60, 0x3f34b0c8, 0x3d886860, 0x3e88cc22, 0x3e0d2ad4, 0x3f446a6a,
      0x3f5d5169, 0x3e960c80, 0x3f140f03, 0x3e27fde0, 0x3cf0dd40, 0x3d3a6940,
      0x3f43937e, 0x3c2b1f80, 0x3f488a07, 0x3e418210, 0x3f7cb6bd, 0x3e4f0bf0,
      0x3f019d08, 0x3d810b18, 0x3f55e69d, 0x3f2fb0bf, 0x3f65ec48, 0x3f1f602f,
      0x3efdd75e, 0x3e0104f0, 0x3f3e3b36, 0x3e651c0c, 0x3f01362b, 0x3eb4c58a,
      0x3f1e99ac, 0x3f1c85ab, 0x3f793d8d, 0x3ed75d7a, 0x3f37e2a9, 0x3f5fe002,
      0x3f541199, 0x3f27737e, 0x3f354703, 0x3f19231c, 0x3f7e2bde, 0x3f788080,
      0x3e910fb4, 0x3eb9a258, 0x3e7d73d8, 0x3f40c445, 0x3eea30f6, 0x3f4e1083,
      0x3f5f484b, 0x3f165b7a, 0x3ebb4c5e, 0x3db7c988, 0x3d6a4e60, 0x3d8dffa8,
      0x3f75cb55, 0x3f4924cc, 0x3f7589a7, 0x3e3ba718, 0x3e5b64c4, 0x3f15130b,
      0x3ef78d22, 0x3eca0304, 0x3e3f3a18, 0x3da92190, 0x3e812406, 0x3eb1109e,
      0x3e84898c, 0x3f10d994, 0x3d43c8a0, 0x3f044912, 0x3f006ab8, 0x3f4ecb83,
      0x3f0c933a, 0x3f5ee4ab, 0x3d297a30, 0x3f1cb629, 0x3f476f1e, 0x3d8f4010,
      0x3ec0a59e, 0x3e3780b8, 0x3f55d398, 0x3f11230d, 0x3f7b83d5, 0x3ddb9ed8,
      0x3ed072d2, 0x3f35bad5, 0x3eefee28, 0x3effc15e, 0x3f1a8c66, 0x3e40f244,
      0x3f3d1e68, 0x3ecf06e8, 0x3e6e97fc, 0x3df891f0, 0x3f6c646f, 0x3f132603,
      0x3f755d4b, 0x3f030eb4, 0x3f069de3, 0x3f18d89b, 0x3ef31d78, 0x3f4dfd0b,
      0x3e921c74, 0x3f3cd952, 0x3f632436, 0x3f35bf2e, 0x3f4c3a85, 0x3f23506e,
      0x3ec9ddfa, 0x3ec5b4de, 0x3f1ec970, 0x3f519f70, 0x3f2e5652, 0x3f41cd3e,
      0x3edc6592, 0x3cb093e0, 0x3e740a40, 0x3eb761fc, 0x3f1a4575, 0x3f204993,
      0x3f14b1c6, 0x3f1ab9cd, 0x3f077650, 0x3dca4d18, 0x3eb3e18e, 0x3e8e4ffe,
      0x3d8a25f0, 0x3f4c2939, 0x3f193155, 0x3ea8bc8e, 0x3e707ccc, 0x3f4a5d88,
      0x3e32faac, 0x3f52839c, 0x3c1bd940, 0x3f318061, 0x3b900d80, 0x3ec6ea42,
      0x3ec4a776, 0x3f606dab, 0x3f1a2caf, 0x399c3000, 0x3eda3bd4, 0x3f0c18e1,
      0x3f73d7fe, 0x3f15ad0b, 0x3bbcb600, 0x3e2d4114, 0x3f105459, 0x3ba1d600,
      0x3e868ece, 0x3f4eeef6, 0x3f0a6cdd, 0x3f0b01ca, 0x3eab4cdc, 0x3f3b253e,
      0x3f0d522a, 0x3f04b44f, 0x3f214601, 0x3df97570, 0x3f781e88, 0x3f004a84,
      0x3dc4e2c8, 0x3cae3f40, 0x3e981544, 0x3f0fa8be, 0x3f145be8, 0x3f1a41ed,
      0x3f611d95, 0x3f6172a1, 0x3f349c83, 0x3e9f1dd2, 0x3f7a2bf5, 0x3f37d399,
      0x3f44784a, 0x3d7b3b70, 0x3eb2431e, 0x3f441518, 0x3f0fce8a, 0x3eec22ee,
      0x3f4c186b, 0x3e271ef0, 0x3e6a4590, 0x3f174a78, 0x3f36ab45, 0x3f2c7736,
      0x3ec17b00, 0x3f752abc, 0x3dc62d48, 0x3f2639dc, 0x3f39aae7, 0x3e44f29c,
      0x3e200de4, 0x3eb6e81a, 0x3f45e72e, 0x3f3560a9, 0x3f035e25, 0x3e7bad00,
      0x3ef4dd66, 0x3dc91d00, 0x3ebcaa6c, 0x3f649206, 0x3f3875bd, 0x3f1e6d42,
      0x3f2790d3, 0x3ef0a232, 0x3db17798, 0x3f65d44d, 0x3e672348, 0x3cd8b4e0,
      0x3f7bcf6d, 0x3f3ef25d, 0x3e032ce8, 0x3e938888, 0x3ec79684, 0x3f2f7936,
      0x3dda4f60, 0x3e95eede, 0x3f349424, 0x3d9caff0, 0x3f1c2c7e, 0x3f3398e5,
      0x3f190799, 0x3e05dd44, 0x3edc72e0, 0x3f3675c1, 0x3f5e5a3c, 0x3f16064c,
      0x3f662418, 0x3f3f4247, 0x3f0f3b48, 0x3f25daa2, 0x3f68f840, 0x3f6ffad9,
      0x3c630280, 0x3f2b903a, 0x3f2c223d, 0x3effe77c, 0x3f2e2f6b, 0x3eecd0da,
      0x3f342867, 0x3f5363b6, 0x3e9e7f68, 0x3f63ad4c, 0x3f3414bf, 0x3e6bcdb0,
      0x3f578426, 0x3e962064, 0x3f4db122, 0x3e9c3af6, 0x3ef24892, 0x3f4c6ed5,
      0x3f03c0b1, 0x3e9e3aaa, 0x3f0ba870, 0x3f134a31, 0x3f4793dc, 0x3e001670,
      0x3d87d9c0, 0x3e9baaf2, 0x3ed92222, 0x3f22a2b0, 0x3ddd1fd0, 0x3f10d7f2,
      0x3e288014, 0x3da30988, 0x3f4211a2, 0x3e92778e, 0x3f505e3f, 0x3f643df4,
      0x3f0c2459, 0x3e20aef4, 0x3f0349c1, 0x3ee5cf92, 0x3e1474a0, 0x3f4e39d7,
      0x3e349b04, 0x3e2639b0, 0x3f0ac524, 0x3efcffac, 0x3e6e2ee4, 0x3f226cb6,
      0x3e5bbbc0, 0x3f5edf6b, 0x3f2aa1b4, 0x3f7dabe8, 0x3f13eb6d, 0x3f5b6432,
      0x3f34d9a9, 0x3e61deac, 0x3f1d30f9, 0x3d778ee0, 0x3f764987, 0x3f5fb106,
      0x3f3dcdc1, 0x3f367073, 0x3e0b0244, 0x3eafcfd0, 0x3f12cb91, 0x3dad9c30,
      0x3e9d2594, 0x3dd0d1a0, 0x3d9178d0, 0x3f0e591f, 0x3f7a3f1c, 0x3dba5ad0,
      0x3f65a922, 0x3e9dfd50, 0x3ec2c3f4, 0x3f390185, 0x3ea0ba1a, 0x3f53afab,
      0x3f33a93d, 0x3ed4cce4, 0x3edbc1d0, 0x3f03e91f, 0x3e728234, 0x3f298ab0,
      0x3e6fe3bc, 0x3dac3250, 0x3e8b58f8, 0x3f135011, 0x3eb0a3dc, 0x3f5593ec,
      0x3f03f287, 0x3f71fe15, 0x3ec51e3e, 0x3d744960, 0x3f74be06, 0x3f2d671d,
      0x3e384d84, 0x3f6f4615, 0x3eb3061c, 0x3f2f80b2, 0x3e69ed90, 0x3e2727b4,
      0x3d410e20, 0x3ec6c27e, 0x3f03a310, 0x3edfc2a4, 0x3ea53202, 0x3e0a79b4,
      0x3f49b5df, 0x3eab7e3c, 0x3ae05e00, 0x3f72d5e0, 0x3dab7408, 0x3e810eaa,
      0x3bcf8e00, 0x391bc000, 0x3e8af156, 0x3f4a7e45, 0x3f4cd234, 0x3ed098fe,
      0x3ee4327e, 0x3f766ab7, 0x3e80075e, 0x3f11c187, 0x3e14872c, 0x3f6b1a7b,
      0x3f3d8346, 0x3f4c3cf2, 0x3ed51848, 0x3e060280, 0x3ef3f384, 0x3ed91a44,
      0x3f2b44a6, 0x3d5edd00, 0x3ecd4ecc, 0x3f282129, 0x3f4233de, 0x3e82de7e,
      0x3f7e5ec6, 0x3ecda3b2, 0x3e99ac3a, 0x3f3cda2e, 0x3ebb093e, 0x3ed259d4,
      0x3f19d226, 0x3eac68c0, 0x3e7b6568, 0x3e9866b8, 0x3f508994, 0x3e0ce474,
      0x3ee3dc62, 0x3ecb4bcc, 0x3f387866, 0x3e772898, 0x3f5bfd02, 0x3e7d2af0,
      0x3f7bfbe4, 0x3e9955f4, 0x3c566100, 0x3d376530, 0x3f4e5334, 0x3f496291,
      0x3f2b7072, 0x3dcfc480, 0x3e86c298, 0x3f4fd219, 0x3e82f6e0, 0x3f0924f0,
      0x3e799b88, 0x3e9ecf24, 0x3de3f120, 0x3ebb2a18, 0x3dce3a20, 0x3eae78de,
      0x3f07e440, 0x3f274293, 0x3f47be34, 0x3e3e2694, 0x3edbd970, 0x3f34e9a9,
      0x3e963ec0, 0x3f40fb2f, 0x3e1cfe1c, 0x3f6ac0fb, 0x3ebc892c, 0x3e5993d4,
      0x3f0a2574, 0x3f431243, 0x3d1f36f0, 0x3f13fa11, 0x3ee0c4b6, 0x3e368e9c,
      0x3f1bbc36, 0x3f3fe30e, 0x3ec36554, 0x3eb09934, 0x3e065dc0, 0x3ef9902a,
      0x3f6bb3ef, 0x3e9b5008, 0x3e5c78e0, 0x3ec9f760, 0x3e4f2254, 0x3f00d2e7,
      0x3edc3108, 0x3f12dd38, 0x3f2f0fb7, 0x3b969c80, 0x3e40b1fc, 0x3f19c592,
      0x3f490f1c, 0x3ee4b532, 0x3ea567b4, 0x3f04180f, 0x3c6ab140, 0x3ed87f5a,
      0x3da7da00, 0x3df04a10, 0x3f1e5dd8, 0x3e1ba038, 0x3eee17ae, 0x3ec14278,
      0x3b2edc00, 0x3f38a28e, 0x3f64000a, 0x3f09a1d2, 0x3f7fc3e6, 0x3cbd65e0,
      0x3eb22f14, 0x3f39e132, 0x3e91fa0a, 0x3db1a4f0, 0x3e319080, 0x3ea825ec,
      0x3ef927ba, 0x3f43c3c1, 0x3de54748, 0x3c643940, 0x3e873a08, 0x3f040ad3,
      0x3f761a55, 0x3f5076ef, 0x3f27e6a1, 0x3c864a80, 0x3f11ab22, 0x3ee89a44,
      0x3f7f5c64, 0x3f2fe4e3, 0x3f2cf469, 0x3f1dc003, 0x3f08b381, 0x3c9bccc0,
      0x3f3397ba, 0x3df9d740, 0x3e227020, 0x3f67225d, 0x3d43b5c0, 0x3b13a000,
      0x3e850468, 0x3eba37b4, 0x3f68d6ed, 0x3eaff3d4, 0x3f4f2667, 0x3efd5d96,
      0x3d286520, 0x3e031f88, 0x3f7f9ed7, 0x3f18f491, 0x3f62acfc, 0x3ddaf220,
      0x3e89721c, 0x3e5d693c, 0x3f417389, 0x3f71cf35, 0x3ecfd50e, 0x3c1c3a80,
      0x3f0af1a6, 0x3ec114c6, 0x3e0b9774, 0x3ede32d4, 0x3f4e661e, 0x3f687d66,
      0x3f3d77ac, 0x3f6f0343, 0x3f641d5d, 0x3f63e789, 0x3ea55fac, 0x3cd34ac0,
      0x3f72a8f4, 0x3f160c23, 0x3ea2d36c, 0x3eb17e60, 0x3f2b9905, 0x3f67672c,
      0x3ed67d9e, 0x3f290f1b, 0x3e28541c, 0x3f657fd6, 0x3f0b24da, 0x3ef8e85c,
      0x3f1f0187, 0x3f04ea25, 0x3f35ce89, 0x3f66cd7c, 0x3f477cad, 0x3d8efe60,
      0x3f7778f2, 0x3ceb0ec0, 0x3f3e30db, 0x3eb27386, 0x3f331d5d, 0x3f345742,
      0x3eb82282, 0x3e109c08, 0x3ed02474, 0x3f3ecea4, 0x3f59f84d, 0x3ee0357a,
      0x3e7222f0, 0x3ecca506, 0x3f02d0a6, 0x3eef0a28, 0x3f4310eb, 0x3f026842,
      0x3dc64630, 0x3f0d205c, 0x3ed7565c, 0x3e9f210a, 0x3e1527f4, 0x3edc3884,
      0x3f1dbe71, 0x3ed243ba, 0x3f1268ea, 0x3eabb42e, 0x3d1abe50, 0x3f5c3655,
      0x3f490120, 0x3ea62188, 0x3f6732ed, 0x3e6cca60, 0x3f7118d7, 0x3ede6e68,
      0x3ed7550e, 0x3f3b981c, 0x3e3a1adc, 0x3f1265a5, 0x3d954db8, 0x3efae76c,
      0x3f6f78f9, 0x3f5f59bb, 0x3f368f94, 0x3ea61b26, 0x3e5a004c, 0x3f00fb35,
      0x3d216a30, 0x3ebd061a, 0x3f3217ec, 0x3e1c9fc8, 0x3e7aee48, 0x3f7b5503,
      0x3f4c506d, 0x3f7ccbec, 0x3f69ab33, 0x3f54b76f, 0x3f2ffe47, 0x3e9b4d50,
      0x3ed17370, 0x3f6b8ba6, 0x3f096fbb, 0x3e25fe6c, 0x3db3ed90, 0x3d8470b0,
      0x3ef83fea, 0x3f158e41, 0x3f6a0bf2, 0x3f3c0de9, 0x3e070158, 0x3f6ecaf7,
  };

  // 10,8,5,5
  uint32_t kernel_vals[] = {
      0x3d3114e7, 0x3d4407ad, 0xbd35d912, 0x3ca7c94a, 0xbc56a7e8, 0x3b948e90,
      0xbccbb9a0, 0x3c2b9b28, 0x3ca02e4e, 0xbcceb383, 0x3c6a04fc, 0xbd37c660,
      0xb799d800, 0xbc5c8848, 0xbc4ae274, 0xbcf0a620, 0x3cb33d9e, 0x3d261659,
      0x3cc7aeb6, 0x3d326067, 0x3c9c9e26, 0xbbcc0050, 0x3cd0ac2a, 0xbc893ff4,
      0x3b8b1050, 0xba428000, 0xbd315ffa, 0xbd0f4ef5, 0x3bbcf490, 0xbc2ab878,
      0x3bc68180, 0xbbc9bb68, 0x3cd18a86, 0x3c96670e, 0x3c22f178, 0xbca5d14a,
      0xbca34e20, 0x3c69da2c, 0x3c012fc8, 0xbc4e8c78, 0x3c6c85a4, 0xbc8a1926,
      0xbc54d694, 0xbd031dd0, 0xbc5f05c0, 0xbbdf5d98, 0x3cfff456, 0xbc9b11c7,
      0xbd0435ce, 0xbd0479da, 0xbb11a930, 0xbd09e01a, 0xbcae6513, 0x3c897392,
      0xbd33a047, 0xbc90b650, 0xbbfc8990, 0x3c8228ee, 0xbca793ea, 0xbd149155,
      0xba0b0b40, 0x3cf9af0e, 0xbd20aafd, 0x3b9c4c68, 0xbd08876d, 0x3c3bf5c0,
      0xbc85b67a, 0x3c955286, 0x3c4ab648, 0xbca8e4b7, 0x3c4cdf44, 0xbccb04c3,
      0x3c22b794, 0xbd0e93a0, 0x3d2b04dd, 0xbc6033f4, 0xbccbc0f7, 0xbd0e3688,
      0xbc4bfcd8, 0xbd37700a, 0xbd4b06a7, 0x3c0ceeec, 0xbbdb7928, 0x3c47f720,
      0x3d3832a9, 0x3bd083d8, 0xbd420c63, 0xbd20b7cd, 0x3d284029, 0xbd2f3a1d,
      0x3cdc94ea, 0x3cc68052, 0xbc0ab8e0, 0x394d6e00, 0xbd1fc3aa, 0x3c4e2404,
      0x3d0adb4d, 0x3c6f5e74, 0x3d373d99, 0xbcd89817, 0xbc582354, 0xbb25eea0,
      0xbd33a903, 0xbcc14be7, 0x3b5d7630, 0xbc550a98, 0xbd280dfd, 0xbd412b6f,
      0xbcda4e57, 0xbb931290, 0xbcd13840, 0xbd378128, 0xbb4bacb0, 0xbc816b44,
      0x3cc4982e, 0xbbf372f0, 0xbc1ece18, 0xbc8989d0, 0x3d2dbdf9, 0xbd2d3ab0,
      0x3d4754e3, 0x3c4187f8, 0xbcbd2fdd, 0x3c945352, 0x3d080845, 0x3b240150,
      0x3c131a98, 0x3b7fc8a0, 0x3d282079, 0x3c047518, 0x3c9ccfca, 0x3d252367,
      0x3d14eb05, 0x3d2ee1b1, 0xbc832ce6, 0xbb9290b0, 0x3ced2af6, 0xbbcd5880,
      0xbd237b88, 0xbc38d38c, 0x3cd2775a, 0x3c209b68, 0xbcc059b3, 0xbc2d7688,
      0x3c3664a8, 0xbd444938, 0x3bb62998, 0x3cfce4ea, 0xbd2647d2, 0x3c4f8f54,
      0xbcc7f663, 0xbc706940, 0x3cf03666, 0x3c894e02, 0x3cdd4b22, 0x3d3058e5,
      0xbd178a16, 0xbd33a122, 0xbcaf84fa, 0x3d2b357f, 0xbbcc8510, 0xbcf1e24d,
      0x3d1811bb, 0x3d07983b, 0x3d00c77d, 0xbd367605, 0xbd4672e3, 0x3d0419c7,
      0x39b31800, 0xbd492abb, 0xbc9b6eea, 0x3be18d70, 0xbd41a34a, 0xbcfcf530,
      0x3cfcab42, 0xbd3e81a2, 0xbd421e7f, 0xbcc11efd, 0xbca63d6d, 0xbd331545,
      0xbd38f0bd, 0x3d496ed7, 0xbc17b734, 0x3c3b45f4, 0x3c64196c, 0xbd417f67,
      0x3d15ae6f, 0x3d14b5f5, 0x3c64e8bc, 0x3b57aae0, 0x3c5c3774, 0xbcca7973,
      0xbcded7b3, 0xbcb2267d, 0x3ca850b6, 0xbd09ca34, 0xbcfc9c53, 0xbc99dc4d,
      0xbd2dda8b, 0xbd104bc0, 0xbcd2fcc7, 0xbbbd1f80, 0x3ba3d618, 0x3b924eb0,
      0x3c0f8a6c, 0x3cc38ea2, 0xbca04520, 0x3b4b43d0, 0xbc6d4e08, 0x3c1c136c,
      0x3d0ad6ab, 0x3c7f40fc, 0x3d0add39, 0x3d06e91b, 0xb8853000, 0x3d46d18b,
      0x3c98251a, 0xbc107654, 0xbc49e4ec, 0xbc4a6e8c, 0xbcc6af4d, 0x3d181b39,
      0xbcf100ed, 0x3bed0c00, 0xbacbcf40, 0xbc2304c0, 0x3d1b6291, 0xba2194c0,
      0xbc3212ec, 0xbbecaeb0, 0xbd425452, 0xbcb6dac3, 0xbc86e604, 0x3cccd70a,
      0xbcc3d7aa, 0xbba5a570, 0x3c4da1fc, 0xbcbb9c3d, 0xbcf26c8d, 0xbd38e4c7,
      0xbd4ab0b3, 0xbb218ae0, 0x3cce9f6e, 0x3c6a84a4, 0x3c8fbf5a, 0x3c20d718,
      0x3cd7200a, 0xbcf3275d, 0xbca6530a, 0x3cd43cfe, 0x3d1aa751, 0x3d1daee3,
      0x3cbf75f2, 0xbb8c1c50, 0x3cf04506, 0xbd43d9c2, 0xbbe133c0, 0xbc95d02a,
      0x3a580cc0, 0x3d433091, 0xbd310a97, 0x3d22b219, 0xbd20c68d, 0xbcf093a3,
      0x3a90b0c0, 0xbcd4a277, 0xbcc4ea5d, 0x3ba52110, 0xbd4584b0, 0xbc4892e0,
      0x3cf9cef2, 0xbd202d7b, 0xbcf8329d, 0xb9317b00, 0xbb02cb60, 0x3d16a987,
      0x3ccd0ae2, 0xbd0e07bb, 0x3ce5afe2, 0xbcba3e53, 0xbd004140, 0x3c727284,
      0xbd3100aa, 0x3ce1384a, 0xbc7980ac, 0x3d220849, 0xbd3db48b, 0xbd401a28,
      0xbca574ea, 0xbc3922f4, 0x3d031b4f, 0xbd32a3f0, 0xbd2c5190, 0x3d1b5ce1,
      0x3c8da5b2, 0xbd1adf65, 0xbd3eaf7f, 0xbd40fb2d, 0xbc019894, 0xba3c1140,
      0xbcf569ad, 0x3bede0a8, 0x3b1b9230, 0xbd23010b, 0x3c740fcc, 0xbbd867c0,
      0xbc17c908, 0x3b348ca0, 0xbc5dd360, 0x3d2a569d, 0xbcdc6527, 0x3d15f95b,
      0x3c943d1a, 0x3b68f8d0, 0xbce9bb5a, 0xbc0014b4, 0x3d0229a5, 0xbd4ba5e0,
      0x3d13459b, 0xbab304c0, 0x3d053451, 0xbc52e2cc, 0x3c0c96a8, 0xbd334520,
      0x3cc7999a, 0xbafba400, 0x3c4b8ce8, 0x3d3f28c9, 0x3d3959cd, 0x3ca50e6e,
      0x3c64cc2c, 0xbd4c667b, 0xbbba0840, 0xbcf05baa, 0xbb70df60, 0x3c910432,
      0x3c84d512, 0xbd388aaa, 0x3c8acbf6, 0xbc3d9808, 0xbcda55a7, 0xbc24b518,
      0xbcc722f0, 0x3cad76be, 0x3c70c6dc, 0x3d2b11e3, 0x3d080f31, 0xbc220d2c,
      0xbd3703ba, 0xbd191162, 0xbc6c6f40, 0xbd1de1dd, 0x3d1235e5, 0x3d09d783,
      0x3ccdc1ee, 0xbd1bc0b0, 0x3d100d91, 0x3d328b8f, 0x3c9d09ae, 0x3ccd7882,
      0x3d4b1a4d, 0xbd093d0c, 0xbd4c717f, 0xbceb60ea, 0x3b2b4ea0, 0x3cf9e1ea,
      0xbd493907, 0x3d3ce3f1, 0x3d195011, 0xbca6a497, 0xbcc9e50d, 0xbcc9a8b7,
      0xbd2c719d, 0xbd1ed948, 0xbc243d94, 0xbcdb1f83, 0x3ca5dcfe, 0xbd4afb10,
      0x38343400, 0xbc8c7d06, 0x3d1dc93f, 0x3d4ada1d, 0xbc86d956, 0xbce683e3,
      0x3d0fffe1, 0x3b17b100, 0x3c475238, 0xbccf00f3, 0xbb9a41d0, 0xbd1a502d,
      0x3b5ba7d0, 0x3d45967d, 0xbd119e3b, 0xbc7f0188, 0xbd0cdef0, 0x3c0efb68,
      0x3d3dd0f3, 0xb7ac8000, 0xbcab8b77, 0x3cba91c6, 0xbc100de0, 0xbd4bd305,
      0xbbf6a4d8, 0xbca78a53, 0x3c83d052, 0x3d393393, 0x3ccea7ae, 0x3d1e4b01,
      0xbd2825a6, 0xbd18795e, 0x3c6bafd4, 0xbc644f88, 0xbd2ce9d7, 0xbc0d95d4,
      0x3c083834, 0x3b0057b0, 0x3cc75282, 0x3ce1beba, 0x3c3a97ec, 0x3bd0a898,
      0xbcd2478a, 0xbccdefdd, 0xbc0876a8, 0x3bfed400, 0x3cc8e346, 0xbc8e1f0a,
      0xbca92707, 0x39f45d00, 0x3c270728, 0xbc208c78, 0x3b499c00, 0x3d4866f5,
      0x3b1b1fb0, 0x3c9e40d2, 0xbd087ff6, 0x3ca2bef2, 0xbca468d3, 0xbca16b1a,
      0x3d3addf5, 0x3d0e80bf, 0xbc78d1ac, 0xbcf4ff6d, 0x3d12995b, 0x3b26b4d0,
      0xbd02b830, 0x3c2f7634, 0xbd38ff10, 0x3ca8f88e, 0xbcc0a01a, 0x3d3e36f3,
      0x3ce4f236, 0xbc57488c, 0xbc873f94, 0xbd078f10, 0x3c5c97fc, 0x3d26b433,
      0x3c5f45f4, 0xbcb806a7, 0xbcf658aa, 0xbd4a8470, 0x3d1ac939, 0xbbb171c0,
      0xbd00ee5e, 0xbc93b7e4, 0x3c21d4a8, 0x3d1a4def, 0xbd15782e, 0xbca9c733,
      0xbd0d9e3b, 0xbcfdea43, 0xbcbde660, 0x3cb42d8e, 0xbd206ac0, 0xbae99a00,
      0xbc220d0c, 0xbccb22e0, 0x3d166429, 0xbd068cfd, 0x3d05072b, 0xbcfbdd43,
      0xbcb96ea7, 0xbb806270, 0xbc42d22c, 0xbc99f550, 0x3d13b6ef, 0xbc7b5968,
      0xbcc11cb0, 0xbcd22397, 0x3d467733, 0x3d437e0f, 0x3ce33436, 0x3d45e69f,
      0xbcb4e1d3, 0xbc9d780d, 0xbd44eddb, 0xbc9f8fca, 0xbcf78a10, 0xbc667634,
      0xbbc440b0, 0x3c4219ac, 0x3bfc1290, 0xbabf0aa0, 0xbd0e8156, 0xbcd89f10,
      0xbd22bc6a, 0xbca2091d, 0xbd231f4b, 0xbbb9ed70, 0xbc4c8ce8, 0x3d302005,
      0xbce67d5d, 0x3d3315ab, 0x3d42b557, 0xbcfb3853, 0x3cbf22fa, 0x3c12c0b8,
      0x395ae800, 0xbd13572e, 0xbc916986, 0xbc828f20, 0xbd0918b5, 0xbc012328,
      0x3c289e98, 0x3d3b4c3b, 0xbcc988c0, 0xbce724a7, 0xbcba939d, 0x3d081539,
      0x3c1c8748, 0xbd27860b, 0xbbd36d68, 0xbd32ff08, 0x3a07c480, 0x3b68ad60,
      0xbc95b244, 0xbb803750, 0x3d304595, 0xbc1a6028, 0xbca8c7c3, 0xbd2183eb,
      0x3bfa09e8, 0xbcf657b7, 0x3bff8f70, 0xbc4a8ccc, 0xbd08d850, 0xbd2ac862,
      0xbc7f8300, 0x3cad9fc2, 0xbcbab96d, 0xbc097d78, 0xbc7fad2c, 0x3c0f1f14,
      0xbc849b46, 0xbd497d13, 0xbd00be2c, 0x3bb30530, 0xbd0d0112, 0xbc06f720,
      0xbc8ddc4c, 0xbcc89d13, 0x3d202a01, 0xbbaec7d8, 0x3d29e3b7, 0xbd1a09f5,
      0xbca13973, 0x3cd3cd26, 0x3cebb3f6, 0xbbe50af0, 0xbd35d98f, 0x3d1f7d17,
      0x3d236eef, 0xbb822f98, 0x3b77e3b0, 0x3d406aa1, 0xbccda04d, 0x3d213933,
      0xbd29efdd, 0xbb52e030, 0x3cc425a6, 0xbcad5aa3, 0xbd0edd9d, 0xbc4fd994,
      0x3c731dd4, 0xbc936a74, 0x3c092048, 0x3b8cdf68, 0xbd359ca3, 0x3a916860,
      0x3d16e051, 0xbc452278, 0x3cff2f52, 0xbc2aa378, 0x3b1f33e0, 0xbd1008a5,
      0x3d1396bd, 0xbbcee730, 0xbd32750b, 0x3c5e0074, 0xbd1d38e3, 0x3d17c565,
      0xbcc91663, 0xbc58e3a8, 0x3c7060e4, 0x3d0aa399, 0x3bf3e110, 0xbd23fdb5,
      0x3cee8352, 0x3d28a7f7, 0xbc4de580, 0x3cdc852a, 0x3d0e4c21, 0xbb4875b0,
      0xbbd2b018, 0xbd0cd62a, 0x3c750ec4, 0xbca804bd, 0x3b607880, 0x3cb1ab1a,
      0xbb9b9640, 0x3c425e0c, 0xbab5cfa0, 0xbd3bd7c3, 0x3d4cb99d, 0xbd2adf2d,
      0x399faf00, 0xbb450930, 0x3c62e114, 0xbcfb6890, 0x3d19b807, 0xbc333088,
      0x3ca1ce42, 0xbca8fe90, 0x3c00c3c8, 0x3d0f85ad, 0x3c4a3528, 0xb9c2e680,
      0x3b587fe0, 0xbc6fd8e0, 0x3ca98c0a, 0xbabaeb40, 0xbbb38168, 0xbcd55fda,
      0x3cabf766, 0xbbf93d10, 0x3d2666ab, 0xbccbd870, 0x3cb013da, 0xbc8de3f0,
      0x3c853306, 0x3ca6a16e, 0x3d439811, 0xbb590460, 0x3b920898, 0x3b85bc10,
      0xbce92ce7, 0x3c6c3284, 0xbafe8960, 0x3c945cc2, 0x3c754a7c, 0xbc2abab8,
      0x3c7b58dc, 0x3d08e483, 0xbd126588, 0xbc968340, 0x3d24cd49, 0x3cb3d2da,
      0xbd2d76eb, 0xbc813a44, 0xbd39e80d, 0x3cc53a6a, 0x3d0ebf09, 0xbbb9a7f0,
      0x3d0b9495, 0xbcee629d, 0x3ce14c82, 0x3c8c3152, 0xbbac1070, 0x3cf3a29e,
      0x3cf1d7da, 0x39dc3700, 0x3d485977, 0xba38fb80, 0x3cfcefb2, 0xbcc5326d,
      0xbd0244a4, 0x3ae3e240, 0x3ad2db40, 0xbd248bd0, 0x3d4c15c9, 0x3bbe53a8,
      0xbcc67bc0, 0xbd080328, 0x3b610de0, 0x3c2f094c, 0xbd40ed1d, 0xbcea71b3,
      0xbcf7154d, 0x3d30698f, 0x3cd21802, 0x3c18a814, 0xbcd07c67, 0x3cfa565e,
      0xbcef7d00, 0x3c8ba85e, 0xbc8159b0, 0xbca6ffcd, 0xbd05df9a, 0x3c309480,
      0xbd0d905e, 0x3d2f28ab, 0x3ab1e760, 0x3c6e6cc4, 0x3d0dced9, 0x3be71b70,
      0xbd01b3b6, 0x3d3f7f8b, 0xbbb3e6b0, 0x3c429918, 0x3cdf0662, 0xbba3ee28,
      0xbca5aaed, 0xbaa6f360, 0xbd352b5f, 0xbce29c30, 0x3bae5b50, 0xbcf5ecd3,
      0xbd1b9263, 0x3c6e55fc, 0x3d095799, 0x3cfc7d6a, 0x3c90a572, 0xbab16840,
      0x3cbcd04a, 0x3a97d940, 0xbd04a19c, 0xbd42e445, 0x3c595cd4, 0xbc7c71c0,
      0xbd31da0d, 0xbc962a74, 0xbd0c49b0, 0xbd1443b5, 0x3a8b8060, 0x3d2a8f6d,
      0xbc04f974, 0xbd1fdeb0, 0xbd3aed78, 0x3c4628e8, 0x3d2145d5, 0xbb6fd580,
      0xbc8fa2da, 0xbcced14a, 0xbadfd860, 0x3ce723f6, 0xbd28aca5, 0xbca54a13,
      0x3d45bed1, 0x3cd6db22, 0x3c8338ba, 0xbd45e5e7, 0xbd330b0d, 0xbce8685d,
      0xbd47ad03, 0x3c0cfcc0, 0xbd2a62ba, 0x3cbd023a, 0x3d49da49, 0x3c23ee28,
      0x3d2c5c47, 0xbcf8b1b0, 0xbd2c365b, 0x3c59734c, 0x3ce80486, 0x3d464e63,
      0xbd2d7b1f, 0xbc804414, 0x3d463d95, 0x3ce1367a, 0xbd332f6f, 0xbc972fda,
      0x3cca32e6, 0x3d23aff5, 0x3d3fb20d, 0xba892400, 0xbca38ac3, 0x3b883350,
      0xbcfe11c7, 0x3d3bf377, 0x3bc73210, 0xbc61e0ac, 0xbd131c43, 0x3a0ddc80,
      0xbca5ecbd, 0xbd0f1b78, 0x3c69512c, 0x3d35d1f1, 0x3cc28532, 0xbbff91c0,
      0x3b51c780, 0x3c03fcc8, 0x3cb255a2, 0x3c230300, 0x3d0815e7, 0x3bacb8c0,
      0xbd039c7a, 0xbb3584d0, 0x3d1bfac9, 0xbd3ae958, 0x3cefc6a2, 0x3c235ae8,
      0x3ccab992, 0xbd370b4b, 0x3a732200, 0xbd461592, 0x3cc961f6, 0x3c838242,
      0xbc9cced3, 0x3d27de81, 0xbc8344fc, 0xbc7faee8, 0xbd1e254b, 0x3d469e51,
      0x3ce20ebe, 0x3c2f144c, 0xbc357d2c, 0xbc3620e8, 0xbc04a334, 0x3c5956a4,
      0xbc8ba3c4, 0x3bca29e8, 0x3d17d1e3, 0xbba196e8, 0x3c8c295e, 0x3d2c4267,
      0x3c983e9e, 0x3d09932f, 0xb9cddb00, 0xbd090ac2, 0x3c2467e0, 0x39fd2400,
      0x3d0f0b43, 0x3ca1e1d6, 0xbba80d18, 0xbcc25020, 0xbcc3dcb0, 0xbbe231e8,
      0xbd26d855, 0x3adee9c0, 0x3d3ef06f, 0xbd2c23e5, 0x3d2cba01, 0x3cd42aca,
      0x3ac605a0, 0xbcc3951a, 0x3b32c4b0, 0x3ce38f9a, 0x3a6874c0, 0xbb147a00,
      0x3c7019a4, 0x3c9e6102, 0x3b0e2d80, 0x3c7dbafc, 0xbd20fbd8, 0x3d436619,
      0xbd434c55, 0x3bc58228, 0xbd3591bd, 0xbbd1a028, 0x3c163ff8, 0xba18cb80,
      0xbc6d2034, 0xbbc6aaf0, 0x3d1be929, 0x3cf2d14e, 0x3d3ecf11, 0xbce0bd70,
      0x3cf668b2, 0xbd304c52, 0x3d0f5a29, 0xbb3c8050, 0x3d2a76fd, 0x3cdfec42,
      0xbc131ed4, 0x3c8715da, 0xbced47e0, 0x3caca7c2, 0xbb68ff00, 0xbd2bfced,
      0x3c6bbf0c, 0xbd313687, 0xbba436a8, 0xbcd181d7, 0xbd37cf83, 0x3c5b8504,
      0xbd082a58, 0x3c96080e, 0x3cde49b2, 0x3a8d1bc0, 0xbd32c9b7, 0xbbaeaad0,
      0xbc80155c, 0xbc08e3a8, 0x3ca31582, 0xbbea7eb0, 0x3d4b33a9, 0x3cd27dda,
      0xbc883e6c, 0xbc9deb03, 0x3ceda292, 0xbc9d334a, 0x3cab4f56, 0x3d46cadd,
      0xbd339477, 0xb98b6900, 0x3c947fb6, 0x3d023c31, 0x3c99d8a2, 0xbd1473f8,
      0x3c3642c8, 0x3d2980c5, 0x3c5b1c54, 0x3d3bb0f1, 0xbd031e18, 0xbad1c9a0,
      0xbccc6d0a, 0x3c952096, 0xbcaa9d87, 0x3cf9b81e, 0x3bfe83a8, 0xbc9c417a,
      0x3af637c0, 0xbca5ffc3, 0x3cf64072, 0xbc8c5214, 0xbcb6240d, 0xbd30cb48,
      0xbc1c45cc, 0x3d3953f1, 0xbc29d26c, 0xbd33c0e5, 0xbd130e08, 0xbd2e02cb,
      0x3acbdc60, 0x3cef5bae, 0x3d0197ed, 0xbd1cff72, 0xbd11b5a0, 0x3d1b8873,
      0xbd38de4d, 0xbd476057, 0x3d239081, 0xbc05e78c, 0xbc94c6f0, 0x3d00f2b7,
      0xbbeb7c68, 0x3d307db1, 0x3d2f397f, 0x3d3b5935, 0x3c114f98, 0xbcc65a4a,
      0xbd34016d, 0xbd05a335, 0x3d0d3551, 0x3c59b1c4, 0xbd235a40, 0xbd0a2bea,
      0x3ccc2556, 0xbbfd6258, 0x3cd81886, 0x3d41dcc5, 0x3d37ecf7, 0x3cae1086,
      0x3c73a234, 0x3d1c71a9, 0xbd3ca15d, 0x3d43e907, 0x3c94baae, 0xbd4b5aca,
      0x3d09daff, 0x3c53a574, 0xbcf09773, 0x3b3b13b0, 0xbd27229d, 0x3d2593df,
      0xbd2c7f62, 0xbd1eca76, 0x3c0888c8, 0x3b860140, 0xbcb67bb0, 0xbcf435aa,
      0xbd2e8ce2, 0x3b89b750, 0xbccdf04a, 0xbcdbd9fd, 0xbc1118c0, 0xbd4c0207,
      0x3ca91bf2, 0x3d2e3cd1, 0xbc160cac, 0x3c9bfa22, 0x3c031e94, 0xbbd129b0,
      0x3d25f675, 0x3cda9792, 0x3d2aedb3, 0x3d412a1f, 0xbd0a4846, 0x3cdd4c76,
      0xbcc4248a, 0x3c27b0a0, 0x3a615940, 0xbc66b220, 0xbd2e8bb8, 0x3d49ae11,
      0x3d4332d9, 0xbcfc2100, 0xbd2ac383, 0x3cd667c6, 0x3d0c976d, 0x3c85c5fa,
      0x3ba20c28, 0x3cf6ef96, 0x3c4b5c68, 0xb9b6ba80, 0xbcbafbf7, 0x3b0a1ee0,
      0x3cee6332, 0xbc404a0c, 0xbc0f05f8, 0x3d1b3bcb, 0x3d4820bf, 0x3d2c90c9,
      0x3d0d3843, 0x3b7f07d0, 0xbc6e3cd4, 0xbd017f98, 0xbbe09b70, 0xbc564360,
      0x3d310a81, 0xbc68efa0, 0x3aaa1800, 0xbd4b4008, 0xbb92add0, 0x3d0a26d3,
      0xbb03ccb0, 0xbb88e0d8, 0xbd0d3143, 0x3cd98022, 0xbcfba76a, 0xbcb0efaa,
      0xbcb783ed, 0xbd2702ea, 0x3c23e634, 0xbd368ec2, 0x3bbb2b18, 0x3d43a38b,
      0x3c07f7f4, 0x3c0f2cc0, 0xbca0230a, 0xbd451f0a, 0xbc8313cc, 0x3d4670e1,
      0xbd406357, 0x3cbf59fe, 0xbca8e0ed, 0xbcb9bb3d, 0x3c817452, 0x3c900d2e,
      0x3bd8d158, 0xbd2977c3, 0xbc3dd788, 0x3d12260f, 0x3cff63ea, 0xbcdeb8c3,
      0xbced00da, 0x3ce76e82, 0xbcc8f677, 0xbc6648b4, 0xbd449ada, 0xbc9af66d,
      0xbcbf552d, 0x3cdb28da, 0x3a1a6680, 0xbd1d79c0, 0xbcef2c2a, 0xbbf520b0,
      0xbabc0a00, 0x3c8d280a, 0xbc989136, 0xbd0a489a, 0x3c368168, 0x3cc19ade,
      0x3d2c7f03, 0xbd322e52, 0x3cb94f62, 0x3d0b907d, 0xbcb2682a, 0x3c09f140,
      0x3bd4a1e8, 0x3d2550e5, 0xbced6c9d, 0x3d1c208f, 0x3d029b61, 0x3c80bfd6,
      0x3c868faa, 0xbcd907aa, 0xbd31def2, 0x3d1d9951, 0x3cd8f40a, 0xbcf5fbd0,
      0x3c9fcf6e, 0x3d32e6bf, 0xbc598380, 0xbd404c47, 0x3d030313, 0x3add26a0,
      0xbc23c368, 0xbcbc4ff7, 0xbcfb37d7, 0xbd0f0d1a, 0x3d2cea83, 0xbcfc20f7,
      0xbc3e6fa0, 0x3d28f981, 0xbc44ed28, 0xbc5752c0, 0x3bd6f0a8, 0x3d47bcb9,
      0xba1b8b80, 0x3d00db71, 0x3b4f5150, 0x3c180534, 0x3ac24e00, 0x3d23a575,
      0xbcb0afaa, 0x3c3df058, 0x3bdacd10, 0xbc2f4de8, 0xbcebcbad, 0xbc044674,
      0x3d2a7241, 0xbd351873, 0xbcc99800, 0x3c644aa4, 0xbc93dba0, 0x3bd56c70,
      0x3c22a874, 0x3c29316c, 0xbccde2fa, 0x3d04bf69, 0xbd2b2bd2, 0x3c24f6b4,
      0x3d006067, 0xbd016525, 0xba8bdcc0, 0x3c7f18dc, 0x3cfa8832, 0xbc4c5414,
      0xbcdd47ca, 0xbcfd17f3, 0x3d3dfcef, 0xbc986150, 0xbc7f99f8, 0x3d47203f,
      0x3c1df868, 0xbcb19b1d, 0xbcec124d, 0xbc249dac, 0x3c8d9db2, 0xbcb76dc7,
      0xbc90ab9a, 0x3d2d7e8b, 0xbd0ecbfb, 0x3b9ad180, 0x3d229639, 0xbd44e212,
      0x3c86b72e, 0xbc825a46, 0x3cb2e2c2, 0x3ce0e25a, 0x3ccd776a, 0xbbec5d28,
      0xbb71f950, 0x3c998342, 0xbc0e10a8, 0x3d38ba4b, 0x3d1626a9, 0x3cc00aa2,
      0xbd3bfb45, 0x3c43b2d8, 0xbc601b14, 0x3bae2280, 0xbb8abdd0, 0x3d3ef73d,
      0xbd47cbeb, 0x3d18422b, 0xbd079f7c, 0x3adfe460, 0x3d3962e7, 0xbd1ec823,
      0x3ce4f25a, 0xbc419248, 0x3d0f8593, 0x3d39e519, 0x3d279cd7, 0x3ca695e2,
      0xbce8d18a, 0x3c8369fe, 0x3c7b33dc, 0x3c92c912, 0xbd02a74e, 0xbce951ea,
      0x3cddb652, 0xbd438bb0, 0x3c670944, 0x3d077419, 0x3aee6d40, 0xbcccddda,
      0x3cd07792, 0x3aeb1140, 0x3d00ab6d, 0x3cdab052, 0xbc83a6c0, 0x3d378b65,
      0x3d18ca3f, 0x3b5e9dd0, 0xbca3cd5d, 0x3cc7db5e, 0x3cece702, 0xbcdb7367,
      0x3d2e6291, 0x3d23da33, 0x3c4d13ec, 0x3c9fef32, 0xbd25bf5b, 0xb9252b00,
      0x3ad8d6a0, 0x3c337420, 0xbb6c2bd0, 0x3d031713, 0xbc4f236c, 0x3c51b244,
      0xbc44ad4c, 0x3c9474f2, 0x3c063458, 0x3c13228c, 0xbbbba390, 0x3b4f2c60,
      0xbc20288c, 0xbc1c6ec0, 0x3d2342e9, 0x3c6b03fc, 0x3b9fd890, 0xbbe72070,
      0xbc351b0c, 0xbc4d3e14, 0x3cbe837a, 0xbb30cb30, 0x3ce17856, 0xbb8c5a58,
      0x3c074738, 0x3c382288, 0xbbcd2b28, 0x3c82507a, 0x3b2a0b60, 0x3d44130f,
      0x3c10d9e0, 0xbabce6a0, 0xbadf8600, 0xbb739c80, 0xbc600f80, 0x3c82c276,
      0xbd2226f5, 0x3d3ff37f, 0x3d4426ad, 0x3d22f737, 0xbc591d08, 0xbd24f663,
      0x3bdd6390, 0xbd386275, 0xbc866100, 0x3c695014, 0x3c814c0a, 0x3d3f3311,
      0xbc6bc1e0, 0x3d32ca43, 0x3cb7d7ae, 0xbba1e9d0, 0xbd4bd5fa, 0x3ba978d8,
      0xbca2af5a, 0xbb2bc200, 0x3cb7bb0a, 0x3d0ba59f, 0x3d169ef1, 0x3b0a2650,
      0x3d1fc229, 0x3cfa4662, 0x3c9529de, 0x3cd13772, 0x3cd6f05e, 0xbca93473,
      0xbcafe123, 0xbd02a278, 0x3c3c0cd4, 0x3c894c4a, 0x3c41bd00, 0x3c5ca0a4,
      0x3d1b717b, 0xbcd16950, 0xbc7de328, 0xbd3cf5ef, 0xba650800, 0xbd3e2408,
      0xbb54cbb0, 0xba0f8cc0, 0x3cd82822, 0x3d3d792f, 0xbc9516b4, 0x3b1d1d50,
      0x3d368979, 0x3c5e6dec, 0xbd3cf378, 0x3d3a8635, 0xbd4662e0, 0x3ca3eb6a,
      0x3bd87628, 0xbd4aa05b, 0x3cc1540a, 0x3d11f57f, 0x3c6448c4, 0x3a90a600,
      0xbd25e66b, 0xbd3333bf, 0xbc35e6e8, 0xbca0f943, 0x3b20bee0, 0xbd1881d8,
      0x398e8580, 0xbd1f24b5, 0xbc42176c, 0x3d46a8a7, 0x3d17a7fd, 0x3ca6c69a,
      0xbc153748, 0x3bb1acf0, 0xbd2a041b, 0x3caf685e, 0x3ac27160, 0x3c1830a0,
      0xbc5498b8, 0x3c462634, 0x3d08fa25, 0xbd1eb5a2, 0xbc1f14d4, 0xbceeee57,
      0x3ceae45a, 0x3c4c2028, 0xbca0930a, 0xbcad99ed, 0xbd01bb5a, 0xbc541b68,
      0x3d47d671, 0xbc8964d0, 0xbc3b78f8, 0x3cbf18fa, 0x3d2a8f6d, 0x3bad6668,
      0xbcc31657, 0xbcdf69d7, 0xbc216f8c, 0xbc76a434, 0x3d06df89, 0xbd2d9123,
      0x3c8ffc22, 0x3cd98b1a, 0xbb93ef10, 0x3d4a7163, 0x3d0d6471, 0x3c02b808,
      0x3b9e7940, 0xbc331560, 0x3cfa9c82, 0x3cd98a2a, 0x3ad2af00, 0x3d16e8bf,
      0x3d04c911, 0xbcb0a740, 0x3d0eae19, 0x3d42eb55, 0x3c9cf206, 0x3d3a18c9,
      0xbb4e7e50, 0xb9f4ad00, 0xbcf3437a, 0xbd2d651f, 0x3c2297ac, 0xbd3bb2c8,
      0xbc5efd4c, 0xbc949774, 0x3cc4f6a2, 0xbd0a815a, 0x3cee9902, 0xbcbb15a0,
      0x3c82e192, 0xbd1b7e8e, 0xbcf11be0, 0x3bbbe510, 0xbce9d433, 0xbd13d5bb,
      0xbc6815ec, 0x3c89ceb2, 0x3cee4ede, 0x3c6b3384, 0xbd112576, 0xbcda1fa3,
      0xbc8a3dca, 0x3c51d724, 0x3cf2124a, 0xbbe8eeb0, 0xbcdb7f5d, 0xbd2cc46e,
      0x3d3909f3, 0x3c75b3fc, 0x3d1b4d4f, 0x3c8dcb66, 0xbbf7bad0, 0x3c82e00a,
      0xbca273e7, 0xba8bc8a0, 0xbc7053f8, 0x3c9c67ae, 0xbb958c40, 0x3c20db00,
      0x3c1b5a28, 0xbc9967d0, 0x3ca42a9e, 0xbce59ef3, 0xbd31c562, 0xbd01404c,
      0x3d06f385, 0xbc8bcd74, 0xbb05c3b0, 0x3cbbf1f6, 0xbcf06560, 0x3d13e9e9,
      0x3c083118, 0xbd183ebb, 0x3cda6dd6, 0xbd29999b, 0xbabd2ea0, 0xbce821b0,
      0x3c419c60, 0xbd2b8af8, 0x3d1f3849, 0xbca0c1ca, 0x3c5a8f1c, 0x3d1ce21b,
      0xbcaf98e0, 0x3d3c0893, 0x3d0a853f, 0x3cf646aa, 0x39affb00, 0xbd389690,
      0xbd4b39d3, 0xbb503720, 0xbbb53590, 0xbbd704b0, 0xbc37d514, 0xbd0719dd,
      0xbae6c6a0, 0xbcdbf147, 0xbc20dd08, 0xbd4c05fd, 0xbc81f7f0, 0x3bf4ba30,
      0x3cd79452, 0x3d452637, 0xbc461978, 0x3beec000, 0x3d338637, 0x3c9bf462,
      0xbd32ee0f, 0x3c22b3a0, 0x3d29b317, 0x3d3c7313, 0xbc376740, 0x3c8c37a2,
      0x3d0ca591, 0x3b46b2a0, 0xbc4f2848, 0x3c721f2c, 0x3c8cd96e, 0xba25f740,
      0xbbd8b2e8, 0xbb5a3650, 0xbc22d698, 0x3cd440fe, 0x3d1f4db9, 0x3d4323b9,
      0x39689e00, 0xbd07b34e, 0xbccfa89a, 0xbb9e7b28, 0xbd494eaa, 0xbd385b07,
      0xbbb5fa98, 0xbcbaf4d7, 0x3cc7dc46, 0xbcb7a5dd, 0xbb0a16b0, 0xbb51f160,
      0xbd3c0b1a, 0xbc1142ec, 0xbd3f8dd5, 0xba843260, 0x3ca5cc22, 0xbd26a015,
      0xbce361f0, 0xbc10a48c, 0x3c9f7b6e, 0x3c9287de, 0xbc81e2a4, 0xbd37b89b,
      0x3d480471, 0xbd14a0eb, 0x3d234b61, 0xbc89835c, 0xbcbccc1d, 0xbd291efa,
      0xbcf1d68d, 0xbbd96c40, 0xbcb922aa, 0x3c80bdfe, 0x3c7c8024, 0xbd105d62,
      0x3d244d31, 0x3cbbe22a, 0xbcb32eb7, 0xbcd1cb73, 0x3d0e8799, 0xbb920a68,
      0xbd2e2b60, 0x3cbdb9e2, 0xbcfa0777, 0xbd06be54, 0xbd24d3bb, 0x3d3683c3,
      0x3ceffe3a, 0x3ccc9cca, 0x3c3e2b00, 0x3ca3238e, 0xbd37e2b0, 0x3d11c961,
      0xbd4ae8a3, 0xbd486c65, 0xbb8237e8, 0x3d30f539, 0x3d14c629, 0xbd4193eb,
      0x3d26de35, 0xbd25110b, 0xbd1cc35a, 0x3c810422, 0x3d3cb60d, 0x3d48e591,
      0xbd044924, 0x39545e00, 0x3d09ce5f, 0x3cef5336, 0xbb5d5b50, 0xbd037c0c,
      0xbcb4b237, 0x3d4a11b9, 0xbcf4825a, 0xbd168eca, 0xbd2f5fad, 0xbba23d80,
      0x3ceb122e, 0x3b070ed0, 0x3c4e9b4c, 0x3c580244, 0xbd461647, 0xbbc52830,
      0x3d2c6e15, 0xbc8c15cc, 0xbd0d8fd2, 0x3be4a1f0, 0xbc210068, 0x3ca9a456,
      0x3cc74eba, 0xbd1a8588, 0xbc784c48, 0x3c8cfe52, 0x3d2dafa9, 0xbc666754,
      0x3cbad202, 0xbbdb5b28, 0x3c49e0f8, 0xbd3035cf, 0x3cc6bd0e, 0x3d17fb77,
      0x3b60c620, 0xbd34bfc3, 0x3cdd6aa6, 0xbd1da1de, 0xbd1d27b2, 0x3ba27e28,
      0x3cde5c2a, 0xbd4c18b2, 0xbcbcc0fd, 0x3b6fb6e0, 0xbc227260, 0x3cc3e3e2,
      0x3cda3926, 0x3c0f5880, 0x3d452a2f, 0xbcca98d0, 0xbd462d60, 0xbd0ba370,
      0x3cd64fb2, 0xbd4a8e37, 0xbd05dfee, 0xbc1a9bd4, 0xbd268438, 0xbcf40b2a,
      0xbd4a88bd, 0x3c603f74, 0xbba3e3f0, 0xbbd827a8, 0x3c8485b2, 0xbd3ee2c2,
      0xbd466335, 0x3c846b4a, 0xbd3703c0, 0xbd0ffab3, 0xbca240fd, 0x3ceacad2,
      0x3c4fbdb4, 0x3c0c45c8, 0x3d05a8d5, 0xbc5c3f28, 0xbd3ea837, 0xbd129b55,
      0x3cb3689a, 0x3d26abd1, 0x3d0cf0e3, 0xbcbe0683, 0x3ce1872a, 0xbc4cca28,
      0xbc85cbca, 0xbb3e8460, 0xbd0e79e3, 0x3c89b682, 0x3d382369, 0xbd0e41a0,
      0x3c99454a, 0xbad781c0, 0xbc811614, 0xbd37d59f, 0xbcc4fdb3, 0x3b3baa60,
      0x3d470b9b, 0xbcb15893, 0xbd2e08ef, 0xbcab4813, 0xbbdd75e8, 0x3d092ff3,
      0x3d091ac5, 0xbcbe0f03, 0x3d009871, 0xbd1deac2, 0x3d47da6f, 0xbc7323f8,
      0x3ce8096e, 0xbcc2410d, 0xbcffbc97, 0xbbbd9830, 0x3d459729, 0xbc136060,
      0xbd0330e4, 0xbce041ed, 0x3c98ac5a, 0xbd10a4b2, 0xbd3e3037, 0xbd206468,
      0x3d34e981, 0x3c389ea0, 0xbd242522, 0xbcbe9850, 0xbcd60ee7, 0xbcfb070d,
      0xbb028f80, 0xbbea97e8, 0xbbaa1f28, 0x3d18b097, 0xba530cc0, 0x3d1a05c9,
      0xbd17b3ba, 0x3c81adf2, 0x3d21a6a3, 0xbd302f33, 0xbd28c162, 0xbc43e194,
      0x3c277c58, 0xbcd14130, 0xbb89d3a8, 0xbc3f92d8, 0x3d3b5e07, 0x3bdde368,
      0xbcec6d4d, 0xbbbdede8, 0xbabb21c0, 0x3cddbbd6, 0xbd25cc2e, 0xbc6c92c8,
      0xbccb1030, 0xbcdc1163, 0x3cfb8c12, 0x3d3f2e85, 0xbd3707b8, 0x3c282b20,
      0x3b7145d0, 0xbd115813, 0xbbc6f800, 0xbd103956, 0x3ba25528, 0xbd2697ab,
      0x3cfb773a, 0x3d38ad2f, 0x3bf5df80, 0x3c631b0c, 0x3d46ce7d, 0xbc743eec,
      0xbc589f8c, 0xbd3a9070, 0xbd2e9e9b, 0xbccaef27, 0xbcf61793, 0xbcfd47a0,
      0xbd048d2d, 0x3c33edc8, 0xbca6d920, 0x3d16f5a3, 0x3bd1a650, 0xbc916a34,
      0x3ca1a002, 0x3b86b698, 0x3cc09626, 0x3d382fdf, 0x3cd125ba, 0xbcc69920,
      0x3bd58e18, 0xbb379360, 0x3ccf4b92, 0x3d3c2fd1, 0x3be5cd10, 0x3926e600,
      0x3d1a42b1, 0x3c4412cc, 0xbc251cac, 0xbcba31ea, 0x3c98b6b2, 0xbbb536b0,
      0x3c8b7ca6, 0x3cb01d82, 0x3cac849a, 0x3c575ec4, 0xbc6ff768, 0xbd43457b,
      0x3bc20340, 0xbcfe39ba, 0xbd2dcad5, 0x3d1c6923, 0x3d20d2a9, 0x3ccd6d42,
      0x3d140969, 0xbd47ea7f, 0xbc9d1967, 0xbad11440, 0x3d3fd6b3, 0x3d0406db,
      0xbcd0d390, 0x3d0117c9, 0xbb4abfd0, 0x3ca4b0a2, 0x3d3c14df, 0xbcc52653,
      0x3be00400, 0xbc633560, 0x3b9ba198, 0xbca1ecad, 0xbd148732, 0xbcf05240,
      0x3d3c4535, 0xbd2df2a7, 0x3bdc7bf0, 0x3d1a9d01, 0x3b04afd0, 0xbcbf7093,
      0x3d10cf11, 0xbd20fecb, 0x3c5a2294, 0x3cfaa8c2, 0x3d4544a1, 0xbb778fb0,
      0x3bd6c468, 0x3c533e64, 0xbb03f380, 0xba8cc760, 0xbd1b780a, 0xbc33f834,
      0x3ca93136, 0xbcee5fa7, 0x3d4824bd, 0xbc8c2364, 0xbc96c32c, 0x3b5274b0,
      0xbd40acb2, 0xbb0aa3b0, 0x3c5e3a04, 0x3cb05e5a, 0xbbf5a490, 0xbd469270,
      0xbcb1613d, 0x3c4d4104, 0x3d29fd19, 0xbd3ca957, 0xbd367eca, 0xbcf4b8b0,
      0xbd4899d8, 0x3c4ad04c, 0x3cd504aa, 0xbd292aa0, 0xbc93fb1a, 0xb8927000,
      0xbcb399bd, 0xbcb1882d, 0x3cdf1e82, 0xbd154a58, 0xbba65590, 0x3d223bf5,
      0xba21a2c0, 0x3c9cadfe, 0xbccd19c3, 0xbd063e1e, 0x3d2fa8af, 0xbcaad777,
      0xbd493cf5, 0xba19c780, 0x3cdf4afe, 0x3cf71c46, 0xbd0e8150, 0x3d2b94df,
      0x3c9890e6, 0xbc875256, 0xbb92a798, 0xb8d05400, 0x3b83e610, 0xbcf30377,
      0xbc970b7a, 0x3cb85f32, 0x3d0aeb31, 0xbd100dc5, 0xbd2ec743, 0xba81f1a0,
      0xbcd2f36a, 0x3c8b8912, 0x3cd213ce, 0xbcd8505a, 0x3caf84ca, 0xbd1a1f43,
      0xbd22fc05, 0xbc38fb40, 0x3c29ffa8, 0x3d21e4f9, 0x3d336049, 0xbc29fb14,
      0x3d4c8f65, 0x3d0156b9, 0xbc9c1a63, 0x3bf1d810, 0x3d2f3379, 0xbcc6024d,
      0xbd2b784e, 0x3cc61f72, 0x3bcad3e8, 0x3d1d16c7, 0x3c493368, 0x3d4a3853,
      0x3d2f9a0f, 0xbd18cc55, 0x3ca27c92, 0xbc0e0578, 0x3d2f9f6b, 0x3d25c15f,
      0xbccba443, 0x3d2861f9, 0x3cdd1c26, 0xb9bba980, 0x3c215ce8, 0xbc6fe358,
      0xbd436fd3, 0xbc5fa958, 0xbcfd9ef3, 0xbc2e3d88, 0x3c9630be, 0xbd019f08,
      0x3c552b0c, 0x3ccead72, 0x3d3161b5, 0xbd349167, 0x3cfb291a, 0x3baf3a70,
      0xbd30eaef, 0x3d36d16d, 0xbbff9db0, 0xbd05cfe5, 0xbd46d333, 0x3a5d36c0,
      0xbd2f322b, 0x3c6ea574, 0x3cc23a2a, 0xbd087a4d, 0x3c9e21b6, 0x3c8b4572,
      0xbcfb10fd, 0x3d256731, 0x3ca1cd0e, 0xbd4060a8, 0x3c9c80e2, 0x3d0bb7b1,
      0x3caec47a, 0xbca2cfaa, 0xbcd33083, 0xbbd930f0, 0x3d2a8e01, 0x3a034b80,
      0x3c964966, 0x3d2e454f, 0xbd1daa35, 0x3d42e051, 0x3cb0dc8e, 0xbd03e9f0,
      0x3ce23c82, 0x3d2b9c51, 0xbad26360, 0x3cf6b6c2, 0x3c5ccecc, 0x3d0d4d23,
      0xbd2023dd, 0xbd080fdd, 0x3d27cddf, 0x3d4c3a39, 0x3c8303fa, 0x3cce2002,
      0xbd420ceb, 0x3ce895e2, 0x3d1dd9a3, 0xbc269ba0, 0xbcce26cd, 0x3ce6a7ea,
      0x3cbdf30e, 0xbd48fe87, 0x3c5c97a4, 0x3c961dfa, 0x3c323fb4, 0x3d1aa5ef,
      0xbb308e50, 0x3d0699af, 0x3cbf1eb2, 0xbd0a3460, 0x3ba9a618, 0xbcdfe007,
      0xbc13b634, 0xbc5bbbe0, 0x3d2a4e3f, 0xbcd5f22a, 0x3c76f9f4, 0xbc9b65cd,
      0x3cb59b36, 0xbcaa9fd0, 0x3ccb71da, 0xbd38c728, 0x3cc6f0ca, 0xbd1d5c6a,
      0x3d320255, 0xbd3a9ed5, 0x3b3d4930, 0xbd3aaa4d, 0x3c9e2a82, 0x3be26210,
      0x3b52f560, 0x3cbaf15a, 0xbc9efa8a, 0xbd0726e6, 0xbd2c5ebd, 0xbd0af8a2,
      0x3d26a0d7, 0x3cc926b6,
  };

  // 5
  uint32_t bias_vals[] = {
      0x3bded4d8, 0x3c9d39d2, 0x3ca89fd2, 0xbc5af538, 0xbcb69fcd,
  };

  // 3,1,1,5
  uint32_t output_exp_vals[] = {
      0x3c0f5041, 0xbd5feb0d, 0xbe2ac302, 0x3e4629df, 0xbf31fe38,
      0x3e5c01b4, 0x3e7c96f6, 0xbce63e5a, 0x3e379fba, 0xbf3027ad,
      0xbdb021b6, 0xbe97d08d, 0xbef57ffa, 0xbdfbe7fc, 0xbf1bf24c,
  };

  uint32_t clip_value = 0x3e4ccccd;

  // 3,1,1,5
  uint32_t output_relu_exp_vals[] = {
      0x3c0f5041, 0x0, 0x0, 0x3e4629df, 0x0, 0x3e4ccccd, 0x3e4ccccd, 0x0,
      0x3e379fba, 0x0, 0x0, 0x0,        0x0, 0x0,        0x0,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, VALID_PADDING, (void *)&clip_value);
}

void test_same_padding_non_zero_strides_medium() {
  input_set *set = &medium_input;
  strides_input_set *strides = &medium_non0_strides;

  // 3,10,8,5
  uint32_t input_vals[] = {
      0x3dedbd80, 0x3f3c7b79, 0x3e131b74, 0x3f07e9cb, 0x3e936dc0, 0x3f2a452c,
      0x3f63ab13, 0x3f64a605, 0x3f17672a, 0x3ed1273a, 0x3ebebdd8, 0x3f0c947e,
      0x3f5af4f0, 0x3e672280, 0x3f222b95, 0x3e84c4d4, 0x3f481888, 0x3eb31760,
      0x3f282381, 0x3d832688, 0x3f50d901, 0x3e87184e, 0x3f130b99, 0x3ef625d8,
      0x3f039d45, 0x3ef74f90, 0x3f47739d, 0x3f0f95c4, 0x3eb4895e, 0x3f3d0eef,
      0x3e5b5400, 0x3f051d17, 0x3edcaddc, 0x3eff1d44, 0x3f1009db, 0x3e4da22c,
      0x3f550f4d, 0x3e286098, 0x3c7b4a80, 0x3ee87202, 0x3f788b9a, 0x3f7a6028,
      0x3f697980, 0x3e1754a8, 0x3f524aa1, 0x3f5341de, 0x3df96408, 0x3f3efd3b,
      0x3f10f16c, 0x3f37ada8, 0x3e10ddac, 0x3f798359, 0x3da84670, 0x3f7b9cca,
      0x3d6356b0, 0x3f000faa, 0x3f61b5d8, 0x3de66248, 0x3e5086a8, 0x3f4b5467,
      0x3efa7626, 0x3dc718f0, 0x3f79b8e4, 0x3f520efc, 0x3dd857c8, 0x3f3d0355,
      0x3f2585a9, 0x3f146def, 0x3f0bb6e8, 0x3f47e6ab, 0x3d92fb78, 0x3dbe5a80,
      0x3ec52f64, 0x3e91dc5c, 0x3f6bc833, 0x3f01e701, 0x3f29857c, 0x3f56c01c,
      0x3f0023e7, 0x3f0bcbe5, 0x3f2aa5fa, 0x3f13cb3e, 0x3f24c58f, 0x3f3d5dcf,
      0x3ea97abc, 0x3f4123bc, 0x3cb9e960, 0x3f2df2c2, 0x3f0f3a6d, 0x3e778748,
      0x3f0d6252, 0x3f2e7767, 0x3f658c24, 0x3f714eda, 0x3e8e536c, 0x3e811bb8,
      0x3eb9ae8e, 0x3eec050e, 0x3ecf8cd2, 0x3ed0cf9e, 0x3ca3c7c0, 0x3f7f9666,
      0x3f79f7d1, 0x3e2a18a0, 0x3e95ddf8, 0x3f452f8b, 0x3f7cedd5, 0x3e185a7c,
      0x3f73aa6c, 0x3f7f1133, 0x3f74dbfb, 0x3f2c2fd8, 0x3e4872ac, 0x3eb6e2e0,
      0x3e91935a, 0x3eb2b562, 0x3f64e525, 0x3f15b029, 0x3f513750, 0x3f3524e8,
      0x3d59cbf0, 0x3f2e3b2b, 0x3ee0eb9e, 0x3f546dab, 0x3d14d5d0, 0x3f257374,
      0x3f7919bb, 0x3f490b1c, 0x3f16ab5c, 0x3f40e084, 0x3e19099c, 0x3ee85d48,
      0x3eba62fc, 0x3db78e00, 0x3f6ceab8, 0x3e317a24, 0x3f1fa2a0, 0x3f1420b6,
      0x3c158fc0, 0x3e10f6c0, 0x3f22e418, 0x3ed5b692, 0x3ecace68, 0x3f3cc42c,
      0x3ebeaef0, 0x3ee7b088, 0x3efb85e2, 0x3f601876, 0x3f1f4b7d, 0x3f1e81ea,
      0x3eee9634, 0x3eb2c5fe, 0x3eef3554, 0x3f4a7a82, 0x3caf3280, 0x3f72cf34,
      0x3e9aafd0, 0x3f01f6f8, 0x3d72e160, 0x3e3c0c40, 0x3ed318d6, 0x3f4a0b90,
      0x3f5dc59f, 0x3edcbaf8, 0x3f2c8435, 0x3f4dea36, 0x3f36eda3, 0x3f7c7056,
      0x3f6e9798, 0x3f7887d5, 0x3eb053aa, 0x3e8efdee, 0x3f7a7a39, 0x3f433fac,
      0x3eb38584, 0x3f5f3ece, 0x3d731830, 0x3c367c80, 0x3f6a5f53, 0x3efdab28,
      0x3f270c9d, 0x3f3e9627, 0x3e5eb05c, 0x3f647ede, 0x3c8dfda0, 0x3f720f81,
      0x3d241a50, 0x3ec868f8, 0x3ec12f9a, 0x3ca47700, 0x3f2ca4ce, 0x3ef4abf4,
      0x3ecb2688, 0x3e23561c, 0x3f6f5198, 0x3ea55200, 0x3d8ee818, 0x3f4a0a40,
      0x3e4eb34c, 0x3f6c9f2b, 0x3f37ca0f, 0x3f6b2da8, 0x3f39cf91, 0x3f2b78e3,
      0x3eb1faa8, 0x3f6d0da1, 0x3f683a3a, 0x3eeaf18c, 0x3e9a626a, 0x3e9ac5da,
      0x3f1842a0, 0x3db66f18, 0x3f030844, 0x3f0081f5, 0x3eef6d7e, 0x3f3b02fd,
      0x3d0fe810, 0x3f1b6ccb, 0x3f146d3e, 0x3f3b640f, 0x3e17cbcc, 0x3e51480c,
      0x3f6e57e0, 0x3f4f759d, 0x3c89ac20, 0x3f455ae5, 0x3f23520b, 0x3f4254d0,
      0x3f4762d4, 0x3e646460, 0x3ce018a0, 0x3e7ff60c, 0x3e7dbe08, 0x3f490e08,
      0x3eab9ab0, 0x3f71279e, 0x3ec308dc, 0x3e79b7e8, 0x3f0533ca, 0x3e9092d0,
      0x3f34ddf6, 0x3ebf1bfc, 0x3f287aca, 0x3f484329, 0x3ef93cb6, 0x3f086c69,
      0x3f177abb, 0x3ee61b4e, 0x3e882744, 0x3f5d5d3d, 0x3d032af0, 0x3e86e918,
      0x3e3787f8, 0x3e99895c, 0x3f1bf283, 0x3f7465f2, 0x3f3a3760, 0x3c8394c0,
      0x3ed7777a, 0x3d429930, 0x3f519e3a, 0x3e119e9c, 0x3f1255b6, 0x3e56ab2c,
      0x3f76628c, 0x3f75ffc9, 0x3ee93664, 0x3eb53df8, 0x3f722cfd, 0x3f671072,
      0x3f325175, 0x3f6b4b00, 0x3e56199c, 0x3ed8c590, 0x3f284d2c, 0x3f2b1b87,
      0x3e6548d8, 0x3f1813fd, 0x3f253675, 0x3f3ea74a, 0x3ebdaffc, 0x3f623436,
      0x3f7ca4f7, 0x3ef80a6e, 0x3f2fd989, 0x3e97173c, 0x3f429e6f, 0x3f6a5080,
      0x3efc7faa, 0x3f68eabf, 0x3f32bd55, 0x3ead4eae, 0x3f48fb09, 0x3e6f1bd0,
      0x3c9c7b80, 0x3f2c641e, 0x3f2c62c7, 0x3dfbb258, 0x3eed3488, 0x3e4145d4,
      0x3ef7b352, 0x3e261aec, 0x3f490dac, 0x3d5812f0, 0x3edf7d5c, 0x3f145c08,
      0x3f0a4d41, 0x3f423434, 0x3e85940e, 0x3ecf4dbe, 0x3f2bf841, 0x3f2441e2,
      0x3f2ab4d1, 0x3f6280f8, 0x3e2bf75c, 0x3c89cda0, 0x3f6eb70d, 0x3ec03ac8,
      0x3cfe2700, 0x3ca7f1a0, 0x3d6cdd50, 0x3e617ee0, 0x3efb8ad2, 0x3f151055,
      0x3df1f228, 0x3ebbda10, 0x3e5cd078, 0x3e6a4abc, 0x3f146456, 0x3ed83d5a,
      0x3f572cf9, 0x3e803aaa, 0x3ec6558a, 0x3cafa8c0, 0x3f6ca7e8, 0x3f3e72ed,
      0x3c539440, 0x3ed5dbd2, 0x3f190013, 0x3f54dc87, 0x3cba8620, 0x3f19e297,
      0x3edbbc60, 0x3f3b6eab, 0x3f24e529, 0x3eb322e8, 0x3db05d18, 0x3b1da600,
      0x3f5cb546, 0x3e4589b8, 0x3ee4f080, 0x3ebb4742, 0x3f2d2824, 0x3f17f560,
      0x3e990320, 0x3f2a0a5b, 0x3e8dc70c, 0x3dba2c20, 0x3f640dfb, 0x3f0c54b2,
      0x3f68b832, 0x3f072818, 0x3e011bec, 0x3f04d165, 0x3d15b270, 0x3f02ccb0,
      0x3f26d422, 0x3f3403f1, 0x3f30c556, 0x3ef45046, 0x3de2eec8, 0x3e0befdc,
      0x3db9c658, 0x3f25c969, 0x3f1a076a, 0x3f07da9d, 0x3eb46cd0, 0x3f7a1c27,
      0x3e067c54, 0x3f73b91c, 0x3e9d248e, 0x3ee76372, 0x3f7a90dd, 0x3ec7cde8,
      0x3ebcccea, 0x3e471868, 0x3e580598, 0x3eaea5d6, 0x3e17722c, 0x3f7da451,
      0x3f46e5df, 0x3f1d91c9, 0x3ed3e17c, 0x3f6374ab, 0x3f09a825, 0x3edd9b1c,
      0x3f2f6298, 0x3df09568, 0x3f3d1043, 0x3e1cc804, 0x3f78f9d1, 0x3f2ba9e8,
      0x3f5b78d7, 0x3f334558, 0x3f316de8, 0x3f6ef91e, 0x3f31afd0, 0x3f0b2158,
      0x3e8e10b0, 0x3edf2d64, 0x3dd6e110, 0x3f225347, 0x3f6d9959, 0x3ed320c6,
      0x3e70d344, 0x3e56fdd0, 0x3f5d77a8, 0x3ea6728a, 0x3ee569e4, 0x3cb04d00,
      0x3f3d3403, 0x3f432b18, 0x3cd6f2c0, 0x3f2d2f41, 0x3f3d8e67, 0x3e44f684,
      0x3f25c4d7, 0x3f27bff1, 0x3f63e0cc, 0x3f63d261, 0x3f5708ab, 0x3ea6ad68,
      0x3e92e8a0, 0x3ec7b2a8, 0x3e9e2806, 0x3e85dab2, 0x3e841af4, 0x3e4de57c,
      0x3f533c07, 0x3e8cf034, 0x3eb08c02, 0x3ec0b52c, 0x3f4fccf0, 0x3f450456,
      0x3f1611b5, 0x3eb682c4, 0x3f3d1629, 0x3f08e6c0, 0x3d88b960, 0x3ee858de,
      0x3f7a1db8, 0x3e694c18, 0x3eea23e4, 0x3f5d92f8, 0x3f6247d8, 0x3ea96e62,
      0x3f22f878, 0x3f7ca7c3, 0x3ef5a19e, 0x3efb917e, 0x3ece38ce, 0x3f6ffcce,
      0x3f6f8161, 0x3f62cc20, 0x3ef849fe, 0x3f4d7328, 0x3f6c6233, 0x3dbb78f8,
      0x3f63b421, 0x3d0ffff0, 0x3e21b148, 0x3e070a9c, 0x3d255c60, 0x3c81a9a0,
      0x3f271171, 0x3efec452, 0x3f247bff, 0x3e816868, 0x3dab3408, 0x3f0071de,
      0x3de909b0, 0x3f53b218, 0x3de4f200, 0x3ee2dc06, 0x3f1e3273, 0x3d1dbd50,
      0x3d3bc1f0, 0x3e72dc20, 0x3f2921a7, 0x3f4e41b9, 0x3ec58d5e, 0x3f51ce94,
      0x3f565b18, 0x3ed741f0, 0x3f367441, 0x3f0e265c, 0x3e40fa48, 0x3e2b5118,
      0x3c98b7c0, 0x3f003e6a, 0x3f6d277f, 0x3e0dcc8c, 0x3f75654b, 0x3f45a03e,
      0x3eec46a4, 0x3ca7f820, 0x3e937bc0, 0x3f27efc4, 0x3e5b3990, 0x3e7d651c,
      0x3eb65652, 0x3e9fb5a8, 0x3f058f3e, 0x3f3a1be6, 0x3f7396ab, 0x3f21af26,
      0x3e9628b0, 0x3f15e4f5, 0x3f5a691b, 0x3f6a8a45, 0x3f30558c, 0x3f24d9e0,
      0x3f1b0465, 0x3ea9a774, 0x3efef674, 0x3f4ac8cf, 0x3f31965b, 0x3f292f47,
      0x3f4d1662, 0x3f7fd9a8, 0x3dc89ce0, 0x3f31bf2c, 0x3dbf2fb8, 0x3f11056d,
      0x3f706665, 0x3ec16a32, 0x3f585ffa, 0x3efb8c32, 0x3eafa1cc, 0x3f499693,
      0x3f1d98fb, 0x3e3c0d90, 0x3f54e9b3, 0x3f084daf, 0x3f41cc22, 0x3dcb2790,
      0x3cceba40, 0x3f7f2b2a, 0x3f2e0799, 0x3f32acd5, 0x3f408035, 0x3dca84b0,
      0x3ef2750a, 0x3f418550, 0x3ed317c8, 0x3cbbbac0, 0x3e669788, 0x3e123e94,
      0x3e5a33c0, 0x3f0cb858, 0x3f4fdebf, 0x3e8ae73e, 0x3f3ad384, 0x3f5fac72,
      0x3f68225e, 0x3f299d5b, 0x3f383eb2, 0x3d2e2fc0, 0x3e86ebfe, 0x3ed4dd82,
      0x3eb7489a, 0x3e0d7c48, 0x3f7216ff, 0x3d8de648, 0x3f27f280, 0x3f2986a1,
      0x3e09a7f0, 0x3f4eabbe, 0x3e915532, 0x3ead61b0, 0x3f6c69ce, 0x3f448c4e,
      0x3f39a72a, 0x3d6adc00, 0x3f15016d, 0x3f5e2975, 0x3e3a381c, 0x3f117bf9,
      0x3f23d70f, 0x3f4a572f, 0x3f4da388, 0x3f18020b, 0x3f5d0c0c, 0x3f089df5,
      0x3f4f9b78, 0x3e55f310, 0x3df335f8, 0x3e989620, 0x3ef388a0, 0x3f3ea204,
      0x3f24c82e, 0x3f30d6ca, 0x3da9e5c8, 0x3e257ba8, 0x3f3ce840, 0x3f600e76,
      0x3f4ffbb9, 0x3e64f914, 0x3f4379fd, 0x3ec28b48, 0x3f169355, 0x3da2a9d0,
      0x3f5f400d, 0x3f14f664, 0x3f748e3c, 0x3f54bead, 0x3f2b8a26, 0x3e800e16,
      0x3ebfc00e, 0x3f59eeb2, 0x3e427770, 0x3e7392a4, 0x3dcf7eb0, 0x3f16ad97,
      0x3f792cb7, 0x3f59f716, 0x3f513bd6, 0x3f1c096a, 0x3f2d24f1, 0x3dc40ea0,
      0x3ef7821c, 0x3f731fc0, 0x3e41f49c, 0x3e4c64a0, 0x3e8bfe88, 0x3d529000,
      0x3da02110, 0x3ef52a00, 0x3eb6444c, 0x3dadb828, 0x3ebda364, 0x3f2fe0d1,
      0x3e3b0660, 0x3e8bbeda, 0x3f0b56bd, 0x3f7121e5, 0x3f5671aa, 0x3f1b0b70,
      0x3f714a25, 0x3f61cdce, 0x3ea6916c, 0x3f745fb4, 0x3ed63536, 0x3f1fa671,
      0x3e221058, 0x3ed3a964, 0x3f60ac46, 0x3f10e377, 0x3dfb3328, 0x3f217fcf,
      0x3e999712, 0x3f192c83, 0x3f3b2a9e, 0x3f5a65c4, 0x3efae0a6, 0x3e699c84,
      0x3f077ba8, 0x3e365360, 0x3ed759c4, 0x3d0d34c0, 0x3f162cf4, 0x3f1a5f7b,
      0x3ea8b7b8, 0x3f08cf56, 0x3ef5f27a, 0x3f4da628, 0x3e4ec16c, 0x3e2a9660,
      0x3f108fbc, 0x3ee1f14e, 0x3f0f22a8, 0x3ecdd798, 0x3f6a96ff, 0x3ee219f4,
      0x3e83d640, 0x3ef035e6, 0x3efc0d4e, 0x3ea456fc, 0x3f792d31, 0x3e407ae4,
      0x3f076787, 0x3e3c0354, 0x3f30345e, 0x3f37721e, 0x3f4bac1e, 0x3e000e68,
      0x3ef4a632, 0x3f1d8a79, 0x3f1b8efb, 0x3ee5baa8, 0x3e723a74, 0x3eddfe00,
      0x3bdbda00, 0x3ea2edc2, 0x3ecd4328, 0x3f0667b2, 0x3ea660d6, 0x3e1485b8,
      0x3f1d75db, 0x3eb8d374, 0x3dcd85c8, 0x3eb15f44, 0x3ebe3540, 0x3e0b351c,
      0x3edf448a, 0x3f3f6611, 0x3b057d00, 0x3deefdf0, 0x3ea77d9a, 0x3eb96394,
      0x3f13827e, 0x3f79fbc1, 0x3f43df3d, 0x3f40e6d9, 0x3e4d4560, 0x3f01590a,
      0x3f3f1770, 0x3f5075ee, 0x3c016940, 0x3f369ccb, 0x3eef9256, 0x3f5c8a2b,
      0x3eff87c6, 0x3ef98f3e, 0x3efc372a, 0x3f6c1bbe, 0x3d985528, 0x3f0fac56,
      0x3ea1a5c8, 0x3d588fc0, 0x3f74d0e1, 0x3c2bfb80, 0x3f21bb2f, 0x3f4164fa,
      0x3e7ee518, 0x3f3e7fc1, 0x3f5f306b, 0x3f2a9288, 0x3e402ec4, 0x3ecbc582,
      0x3f3f92e6, 0x3ed30bb2, 0x3ece7c06, 0x3f1627af, 0x3e9e5b68, 0x3e83defa,
      0x3f13f3d7, 0x3e9b44c8, 0x3eb6326a, 0x3f1e320d, 0x3f345313, 0x3f402f2a,
      0x3e436ad0, 0x3f6b6267, 0x3f77bf4f, 0x3e25b374, 0x3f082df3, 0x3f0518ce,
      0x3ed9cd10, 0x3e0e79bc, 0x3f375186, 0x3ef78588, 0x3f65373c, 0x3e0a1c48,
      0x3efc3e56, 0x3e147014, 0x3f4c0eb4, 0x3ee02dc8, 0x3eb077a8, 0x3f6f37d5,
      0x3efe9988, 0x3f6641c7, 0x3f6e233b, 0x3f5a3c19, 0x3e558d0c, 0x3f5cc5c2,
      0x3eb8a8ba, 0x3f2cc5d1, 0x3f3101a7, 0x3de0c6e8, 0x3f54fef7, 0x3d95c008,
      0x3e788eec, 0x3f1b67dd, 0x3f222409, 0x3f712da8, 0x3e67405c, 0x3f31b7d3,
      0x3f4fd102, 0x3efd046c, 0x3efca6b0, 0x3f6d19a5, 0x3df793e8, 0x3daa5070,
      0x3f7702cb, 0x3f506ab7, 0x3f27a076, 0x3f55bb1e, 0x3f72bd64, 0x3f27c240,
      0x3d93cf20, 0x3f10d198, 0x3f2cc756, 0x3d59c870, 0x3f111e75, 0x3e4daff0,
      0x3f0c06ad, 0x3f5da51d, 0x3e9dc660, 0x3eceaa4c, 0x3f58d67a, 0x3f44e0b4,
      0x3daa4100, 0x3f47d049, 0x3e9400d8, 0x3e188cdc, 0x3e4b3d50, 0x3e8a3384,
      0x3f3cbc6c, 0x3d554a90, 0x3edda170, 0x3ef5c6f4, 0x3ef7e32c, 0x3f468a31,
      0x3d316110, 0x3f3b6d31, 0x3f70fabc, 0x3e7612bc, 0x3ec31408, 0x3f3676ac,
      0x3e686450, 0x3ed8430e, 0x3f4ff828, 0x3d7e8100, 0x3e3bea48, 0x3de5ed28,
      0x3d933b70, 0x3f65a8ef, 0x3c2b2400, 0x3e96fb7e, 0x3e4e3ed4, 0x3e4e905c,
      0x3f6ef198, 0x3f0a7cf9, 0x3f3bd186, 0x3e02fcf4, 0x3f4f5b14, 0x3f11f976,
      0x3c8567e0, 0x3e437f3c, 0x3d927f70, 0x3f436845, 0x3d3c3f30, 0x3e2e7e40,
      0x3f2f5336, 0x3ed60ba6, 0x3f585907, 0x3ea42d1c, 0x3f53a5c3, 0x3ef1d578,
      0x3d12f280, 0x3ea58648, 0x3f1db202, 0x3ed896d0, 0x3f1a69a7, 0x3e4895f8,
      0x3e88f8de, 0x3ebae986, 0x3f2cacb6, 0x3e8b6300, 0x3f097243, 0x3d6d6890,
      0x3f36aa21, 0x3dda6810, 0x3f5d5525, 0x3f773125, 0x3f08d4c5, 0x3ebb9654,
      0x3f144a20, 0x3dba7120, 0x3f7e3683, 0x3ec8aa32, 0x3f69c524, 0x3f7356b9,
      0x3d080870, 0x3f6afdd4, 0x3f797221, 0x3f6ee080, 0x3f174546, 0x3f63de2d,
      0x3f0e2ba6, 0x3e8caba8, 0x3f7ed398, 0x3f1693d5, 0x3e2617fc, 0x3f4fb2de,
      0x3e3aa900, 0x3f2a6432, 0x3e440a70, 0x3f7adf8b, 0x3eaff642, 0x3f33e8a1,
      0x3e256498, 0x3e0dcc58, 0x3f38da84, 0x3e36d164, 0x3ea298a6, 0x3ebdbe6e,
      0x3f4a9e9b, 0x3f3166c6, 0x3f403ca9, 0x3ee83b92, 0x3e6ec430, 0x3d567790,
      0x3ee04c4c, 0x3f327eec, 0x3f7fb102, 0x3f39927d, 0x3ef05162, 0x3f18137c,
      0x3f2d0df7, 0x3f4fd610, 0x3ee98a1e, 0x3dcb7bf0, 0x3f20d328, 0x3f3b9016,
      0x3ad8fc00, 0x3f33401a, 0x3f08ff04, 0x3e879e7c, 0x3ebfabca, 0x3f062c57,
      0x3f252797, 0x3c24c100, 0x3f5e3746, 0x3f5e2942, 0x3f04073d, 0x3eb16d70,
      0x3f2a5fdf, 0x3f4eabd0, 0x3f45c239, 0x3f7d2f3a, 0x3f11bbdd, 0x3f0037c2,
      0x3f0786a2, 0x3e909d36, 0x3d8ab390, 0x3ed8ae20, 0x3e2f5364, 0x3f24b350,
      0x3f5e11ee, 0x3f4ecc19, 0x3f162952, 0x3ea0adbc, 0x3ee0edc0, 0x3f2e20f3,
      0x3c141e80, 0x3e2d0eb4, 0x3e99b296, 0x3ea387f0, 0x3f045ce5, 0x3c0aea80,
      0x3f025ea1, 0x3e8d1594, 0x3deaace0, 0x3f01ec3a, 0x3e798b40, 0x3f6190f6,
      0x3ec02984, 0x3e9da7c8, 0x3f6e0478, 0x3cb27220, 0x3f423983, 0x3f17954b,
      0x3cad7400, 0x3f04289c, 0x3ef04050, 0x3e41f8e8, 0x3e596930, 0x3eaf7e62,
      0x3f2d03bc, 0x3f675d3c, 0x3f6acb66, 0x3f204dd4, 0x3f42f7f0, 0x3f5d68e3,
      0x3eb8faa2, 0x3e9e208e, 0x3db43448, 0x3f7c3108, 0x3f06eb2f, 0x3df76fd0,
      0x3f726839, 0x3f788afa, 0x3d2df150, 0x3f73f521, 0x3ee9f9fa, 0x3ecbcd28,
      0x3e4d208c, 0x3dacc190, 0x3e50ee64, 0x3dbe8f88, 0x3f7b7970, 0x3f366db3,
      0x3f69a598, 0x3ee6fcec, 0x3f56d6e5, 0x3f4b45d8, 0x3e218a18, 0x3ea5a320,
      0x3f0a59ca, 0x3f1281a9, 0x3eead684, 0x3e2287f8, 0x3f002c55, 0x3f06f0e7,
      0x3f7185a6, 0x3f563dfe, 0x3eda6c3e, 0x3f50510a, 0x3e0bd720, 0x3f364ccc,
      0x3f6cb83c, 0x3e5c546c, 0x3f7160a0, 0x3f7a25d1, 0x3f2dc4c6, 0x3ddf1050,
      0x3eb86014, 0x3f307988, 0x3e60b364, 0x3f2b216f, 0x3f170233, 0x3e90a560,
      0x3f1fb4bb, 0x3f5c6307, 0x3f76e518, 0x3f579f62, 0x3eb86790, 0x3ef9d3ca,
      0x3e99df66, 0x3e3fb484, 0x3ee57db0, 0x3de866c8, 0x3f0339c6, 0x3f32e51d,
      0x3e46e738, 0x3f510500, 0x3f7cd665, 0x3f33a5da, 0x3e65e29c, 0x3f6270c5,
      0x3f63d846, 0x3f48314f, 0x3cf30d40, 0x3e908352, 0x3f2e7409, 0x3dd3d868,
      0x3ea48c34, 0x3c82db60, 0x3f21bf14, 0x3f756025, 0x3f45cae6, 0x3f5cf345,
      0x3f564efa, 0x3f580521, 0x3f1dd566, 0x3d264190, 0x3e8df1a4, 0x3c2d0b40,
      0x3eeb6ce2, 0x3f2e578b, 0x3f50245b, 0x3dbb7e00, 0x3f4343d9, 0x3f300078,
      0x3f4ee399, 0x3e64e4bc, 0x3f28ffa4, 0x3f5ca694, 0x3e6593f0, 0x3f536710,
      0x3eb0528a, 0x3ea7e840, 0x3f54e1d4, 0x3f28760a, 0x3f463c53, 0x3f1f4ded,
      0x3f604425, 0x3f270acd, 0x3ee33c98, 0x3ddcf9e8, 0x3d8e3fe8, 0x3ebb1ca4,
      0x3f0e38a2, 0x3ec5ac5e, 0x3f250d7a, 0x3ef72cd2, 0x3dbe4268, 0x3f3b1d04,
      0x3f3f0582, 0x3e219e5c, 0x3ea6702e, 0x3f6beed3, 0x3f4c4630, 0x3eab05c0,
      0x3f1085b2, 0x3f287fb3, 0x3f1cbb42, 0x3f0407ce, 0x3e474f60, 0x3f461279,
      0x3ef0d36c, 0x3f26d8ef, 0x3e0f9860, 0x3eb7626a, 0x3eaa6ee0, 0x3d3ad180,
      0x3ee73df2, 0x3f4fa2c1, 0x3f3fd545, 0x3f6f0e09, 0x3ec0cb9a, 0x3dd1c540,
      0x3f0e4267, 0x3f7a8993, 0x3f71f72a, 0x3f13f715, 0x3f2934e9, 0x3f4d1b2b,
      0x3f73b580, 0x3f24d9a9, 0x3f46160d, 0x3f005880, 0x3e28e334, 0x3f596991,
      0x3e0d9ba4, 0x3e5bb108, 0x3e2d0298, 0x3ec001c8, 0x3e44075c, 0x3f02abe7,
      0x3e48b7e0, 0x3f48f140, 0x3e371784, 0x3f724e6e, 0x3f63981a, 0x3f39aaa9,
      0x3d023890, 0x3f0754cc, 0x3f341bd4, 0x3f16fd77, 0x3e4caa04, 0x3e423cf8,
      0x3f6cc610, 0x3f16c03a, 0x3f6f8276, 0x3e9ad074, 0x3ec44f62, 0x3f185439,
      0x3e572f90, 0x3f09432a, 0x397ed000, 0x3d006030, 0x3f6cce05, 0x3e6ffa90,
      0x3e89707c, 0x3ec25468, 0x3d4c4850, 0x3f5e0be1, 0x3f21058c, 0x3f53870b,
      0x3dcf7c00, 0x3eb316a2, 0x3e1fd5ec, 0x3ee94d58, 0x3f2a5bc7, 0x3f140db9,
      0x3ecfbdf8, 0x3f7adedf, 0x3eb7adc8, 0x3f46abdb, 0x3f67a77e, 0x3f5de512,
      0x3f071c08, 0x3df25a78, 0x3e4e1280, 0x3f38d7fc, 0x3f3bbb2d, 0x3f6b2c21,
  };

  // 2,3,5,5
  uint32_t kernel_vals[] = {
      0x3e1010ca, 0xbd9de6bf, 0xbdb40e03, 0x3d07c460, 0x3dafaf78, 0xbccd1bd8,
      0xbe1e0e30, 0xbaea0500, 0x3e05b604, 0x3c917808, 0x3e34318a, 0x3cf2cc70,
      0x3dba5928, 0xbdef8b99, 0x3deeaf4c, 0x3e0999e4, 0xbcd53068, 0xbd1ff43c,
      0x3e1fe34a, 0x3cfddc80, 0x3d891584, 0x3e2412a0, 0xbe18e584, 0x3e276c82,
      0x3c90c168, 0xbdeb852c, 0x3e2ee1e8, 0xbe1c7d4b, 0xbe019dbd, 0x3bad51c0,
      0x3e0acf02, 0x3cb43390, 0xbd828287, 0x3d279350, 0x3d8108b0, 0xbdb5cade,
      0xbe291e11, 0xbddf82e9, 0xbda02529, 0xbd61ae8c, 0xbd8f694b, 0xbd358f20,
      0xbe181e84, 0x3df22f4c, 0xbe364531, 0xbe1026d4, 0xbdf32df6, 0xbe0e3a40,
      0x3e17ef54, 0xbca39b48, 0xbd9493c8, 0xbe342924, 0x3b92be80, 0xbdb81fab,
      0x3d306e5c, 0xbd9226b0, 0x3e38f870, 0x3e16adbc, 0x3e12bdde, 0xbe2336c5,
      0xbd4dc424, 0x3db7f748, 0x3d839a8a, 0xbe34d672, 0x3e1c0870, 0xbdc20690,
      0xbe2b2f41, 0x3d374784, 0x3d9a9b28, 0xbe2067f5, 0x3e29e600, 0xbd91ac51,
      0xbdc2de89, 0x3d43c5a8, 0x3e1d16f6, 0xbe0068ff, 0x3c4c4b20, 0x3e25828c,
      0x3e298766, 0xbdf0dabf, 0xbdcca37b, 0xbe3720f9, 0xbdc71d63, 0xbe24b4d1,
      0xbda01e6d, 0x3da88708, 0xbd25e58c, 0xbd84a85c, 0x3e0154de, 0xbd10a28c,
      0xbcc18120, 0xbd9ed2bc, 0xbd44b5dc, 0x3e36fabe, 0xbdef4e3b, 0xbe31bfbd,
      0xbcc802e0, 0x3db30ed0, 0xbc7a8200, 0x3e372460, 0x3d71c758, 0x3def98d4,
      0x3d3826b0, 0xbdc5af54, 0xbd8cc9db, 0xbe04e70c, 0x3d36a850, 0xbe2cf434,
      0x3dbae5f0, 0xbb9842a0, 0xbe0f969d, 0x3ddf7740, 0xbcb578d0, 0xbd0f7e4c,
      0x3d916ec0, 0xbcd51e38, 0x3bc6af20, 0x3e0ef404, 0x3e28d4d4, 0x3d515bc4,
      0x3d7d46cc, 0xbe177230, 0xbdf16a23, 0x3be38380, 0x3d9ecbcc, 0x3b974ce0,
      0x3e272334, 0x3e1a3306, 0xbd79275e, 0x3e0c2950, 0xbdbe384a, 0xbdea4848,
      0x3d23abb8, 0xbce4c568, 0xbc9cf250, 0xbde2f2f6, 0xbb33ed40, 0xbd2594e8,
      0xbe375192, 0xbe039f4b, 0x3d08aae0, 0xbe10a348, 0x3cf9a400, 0x3e2eb8a6,
      0x3c67d8b0, 0xbd5300d4, 0xbabd1200, 0xbdab0df9, 0xbd9b63b2, 0x3e14ad7a,
  };

  // 5
  uint32_t bias_vals[] = {
      0xbc13d600, 0xbd897640, 0x3d116244, 0x3df5dc48, 0xbce727a0,
  };

  // 3,5,3,5
  uint32_t output_exp_vals[] = {
      0xbee9b3fc, 0xbf17b60f, 0xbe484f39, 0x3f0961db, 0x3d4c1743, 0xbf2531ad,
      0xbe9e12fb, 0xbd2cb655, 0x3e7577b7, 0x3e57e7a6, 0xbddbe7ef, 0xbd9c38f2,
      0xbe8bdcb8, 0x3f07b811, 0x3e993fa2, 0xbee4d216, 0xbed2d3c1, 0xbf018cf6,
      0x3e9ab29a, 0xbc846189, 0xbe2a2151, 0xbf0cf26c, 0xbe3d15aa, 0x3ec875d1,
      0xbce80543, 0xbd2144f7, 0xbea62a1f, 0xbf0a4bb7, 0x3f1e16ea, 0xbe6af02a,
      0xbefc6588, 0xbed44ade, 0xbf129453, 0x3ee895e4, 0xbdc1fd8c, 0xbec8491e,
      0xbe75412e, 0xbdec8108, 0x3f1ac5f4, 0x3dbea2f0, 0xbdfc22ef, 0xbe71cce2,
      0xbeee18a0, 0x3f0d220f, 0xbd4cce24, 0xbe78ddf6, 0xbec93b9a, 0xbf06ee8e,
      0x3f03fe72, 0x3e3ac3ba, 0xbf12738d, 0xbf2febbf, 0xbebc542d, 0x3e8172af,
      0xbd330054, 0xbeb90581, 0xbec3ecc7, 0xbf310a62, 0x3f4cf144, 0xbe7f9437,
      0xbe86389f, 0xbec82670, 0xbe71353a, 0x3f0802ad, 0x3df16c7b, 0xbeab9b58,
      0xbecb53ec, 0xbee876ac, 0x3f218016, 0x3e0a5d6e, 0xbe80e3b4, 0xbe6a7756,
      0xbea33ab7, 0x3f3949a0, 0xbe2f42e1, 0xbe3941bb, 0xbddc8acb, 0xbecacf4e,
      0x3ef3e4ce, 0x3eb546d3, 0xbee8c535, 0xbee112e6, 0xbe5fdadc, 0x3f58da9c,
      0x3dc93527, 0xbdb5e1ed, 0xbe3e8ebb, 0xbec5b836, 0x3ec7d54a, 0x3d819f14,
      0xbf0426fb, 0xbefe0525, 0xbe8f9be7, 0x3f4a54bd, 0x3dae38c1, 0xbf211377,
      0xbea54527, 0xbf0947c1, 0x3db8822c, 0xbb87d99d, 0xbe25102c, 0xbe8a8a9d,
      0xbf0bf3db, 0x3f5a6b25, 0xbe0e5dfa, 0xbdfbbc30, 0xbc731067, 0xbea56ac8,
      0x3e765c6e, 0x3f01f8c7, 0xbea4492e, 0xbf227ee2, 0xbe21160c, 0x3e457e8b,
      0x3eaed0ce, 0xbd9b51a3, 0xbeaefc45, 0xbeade1a5, 0x3efa07f4, 0x3c93b7b0,
      0xbf0bc708, 0xbe736c16, 0xbebb2c04, 0x3f4cffa1, 0xbe6ba6d8, 0xbe6e2e8f,
      0xbefa7f52, 0xbeb2ea70, 0x3f2b9797, 0xbdc01028, 0xbc79e4b4, 0xbe13949a,
      0xbeb5c4d9, 0x3ebf20b2, 0x3bd54764, 0xbf0d56cd, 0xbee63f7f, 0xbde73ed3,
      0x3ed4308f, 0x3dfa5af5, 0xbf22f3df, 0xbe1374eb, 0xbe34d7fa, 0x3e1a3b6d,
      0x3e1f64b5, 0xbde3cde2, 0xbe300e8a, 0xbe04bcd1, 0x3f315c43, 0x3d585df5,
      0xbec3da8a, 0xbe56fc4e, 0xbf353dcf, 0x3ebcad9c, 0x3c2c3339, 0xbe66802a,
      0xbef105fa, 0xbee068fa, 0x3eb0b458, 0xba9b0dc8, 0xbd473b6e, 0xbe8544e8,
      0xbebb0b5d, 0x3f0ce772, 0x3e0fecd2, 0xbf1b6e40, 0xbd1dd2cd, 0xbe6f7d6c,
      0x3d20971f, 0x3e0530f4, 0xbc98b77a, 0xbeee6e72, 0xbef41f95, 0x3ef63aa7,
      0x3ee45d03, 0xbea14327, 0xbef5e1a9, 0xbee27363, 0x3f1ee832, 0xbe403bbc,
      0xbe42f7ae, 0xbe932cb8, 0xbeb69872, 0x3ebeca24, 0xbd155237, 0xbecb8a26,
      0xbead579a, 0xbec12b5a, 0xbcecc13d, 0xbb50feab, 0xbd660f71, 0xbe938724,
      0xbf0ee147, 0x3efae9af, 0x3e55edd3, 0xbea91e8f, 0xbed6ee3b, 0xbf095e37,
      0x3e9da89b, 0x3e994dee, 0xbdcc19ee, 0xbed770c0, 0xbe91cccc, 0x3f06de06,
      0xbde3ada4, 0xbd40085a, 0xbd522e19, 0xbe92716c, 0x3f4a1856, 0x3e3a5b61,
      0xbf04a036, 0xbeae4b75, 0xbe462a7d, 0x3f07781f, 0x3e5d9f05, 0xbe83bdf3,
      0xbeb8640d, 0xbf061386, 0x3e8624aa, 0x3eba5fd0, 0xbe5c435e, 0xbedc4cac,
      0xbf00a447, 0x3f118d65, 0x3d7c81bf,
  };

  // 3,5,3,5
  uint32_t output_relu_exp_vals[] = {
      0x0,        0x0,        0x0,        0x3f0961db, 0x3d4c1743, 0x0,
      0x0,        0x0,        0x3e7577b7, 0x3e57e7a6, 0x0,        0x0,
      0x0,        0x3f07b811, 0x3e993fa2, 0x0,        0x0,        0x0,
      0x3e9ab29a, 0x0,        0x0,        0x0,        0x0,        0x3ec875d1,
      0x0,        0x0,        0x0,        0x0,        0x3f1e16ea, 0x0,
      0x0,        0x0,        0x0,        0x3ee895e4, 0x0,        0x0,
      0x0,        0x0,        0x3f1ac5f4, 0x3dbea2f0, 0x0,        0x0,
      0x0,        0x3f0d220f, 0x0,        0x0,        0x0,        0x0,
      0x3f03fe72, 0x3e3ac3ba, 0x0,        0x0,        0x0,        0x3e8172af,
      0x0,        0x0,        0x0,        0x0,        0x3f4cf144, 0x0,
      0x0,        0x0,        0x0,        0x3f0802ad, 0x3df16c7b, 0x0,
      0x0,        0x0,        0x3f218016, 0x3e0a5d6e, 0x0,        0x0,
      0x0,        0x3f3949a0, 0x0,        0x0,        0x0,        0x0,
      0x3ef3e4ce, 0x3eb546d3, 0x0,        0x0,        0x0,        0x3f58da9c,
      0x3dc93527, 0x0,        0x0,        0x0,        0x3ec7d54a, 0x3d819f14,
      0x0,        0x0,        0x0,        0x3f4a54bd, 0x3dae38c1, 0x0,
      0x0,        0x0,        0x3db8822c, 0x0,        0x0,        0x0,
      0x0,        0x3f5a6b25, 0x0,        0x0,        0x0,        0x0,
      0x3e765c6e, 0x3f01f8c7, 0x0,        0x0,        0x0,        0x3e457e8b,
      0x3eaed0ce, 0x0,        0x0,        0x0,        0x3efa07f4, 0x3c93b7b0,
      0x0,        0x0,        0x0,        0x3f4cffa1, 0x0,        0x0,
      0x0,        0x0,        0x3f2b9797, 0x0,        0x0,        0x0,
      0x0,        0x3ebf20b2, 0x3bd54764, 0x0,        0x0,        0x0,
      0x3ed4308f, 0x3dfa5af5, 0x0,        0x0,        0x0,        0x3e1a3b6d,
      0x3e1f64b5, 0x0,        0x0,        0x0,        0x3f315c43, 0x3d585df5,
      0x0,        0x0,        0x0,        0x3ebcad9c, 0x3c2c3339, 0x0,
      0x0,        0x0,        0x3eb0b458, 0x0,        0x0,        0x0,
      0x0,        0x3f0ce772, 0x3e0fecd2, 0x0,        0x0,        0x0,
      0x3d20971f, 0x3e0530f4, 0x0,        0x0,        0x0,        0x3ef63aa7,
      0x3ee45d03, 0x0,        0x0,        0x0,        0x3f1ee832, 0x0,
      0x0,        0x0,        0x0,        0x3ebeca24, 0x0,        0x0,
      0x0,        0x0,        0x0,        0x0,        0x0,        0x0,
      0x0,        0x3efae9af, 0x3e55edd3, 0x0,        0x0,        0x0,
      0x3e9da89b, 0x3e994dee, 0x0,        0x0,        0x0,        0x3f06de06,
      0x0,        0x0,        0x0,        0x0,        0x3f4a1856, 0x3e3a5b61,
      0x0,        0x0,        0x0,        0x3f07781f, 0x3e5d9f05, 0x0,
      0x0,        0x0,        0x3e8624aa, 0x3eba5fd0, 0x0,        0x0,
      0x0,        0x3f118d65, 0x3d7c81bf,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, SAME_PADDING, NULL);
}

void test_valid_padding_non_zero_strides_large() {
  input_set *set = &large_input;
  strides_input_set *strides = &large_non0_strides;

  // 4,15,10,6
  uint32_t input_vals[] = {
      0x3efb1fe6, 0x3f04c7af, 0x3e09e1c4, 0x3f0c288a, 0x3ec9f8a8, 0x3f69e8f8,
      0x3e3ad78c, 0x3f6d04b6, 0x3f5c45b3, 0x3ece8426, 0x3f47fb5c, 0x3ebe3d66,
      0x3f101556, 0x3f09cdc8, 0x3f6706c4, 0x3e0f1eb4, 0x3ec8bc90, 0x3dbd7ff8,
      0x3b17de00, 0x3e931fc2, 0x3e7470b4, 0x3f62874f, 0x3d194080, 0x3f54f750,
      0x3e43d79c, 0x3e659be8, 0x3f6dd296, 0x3f0ce854, 0x3eec83a2, 0x3e3a5c7c,
      0x3dbfdce8, 0x3f14bd6f, 0x3eb2de3c, 0x3e895b6a, 0x3f64c7d1, 0x3bbf4400,
      0x3f069316, 0x3eaa5b92, 0x3c677f40, 0x3f524fcf, 0x3e5915f0, 0x3eea46d2,
      0x3e66215c, 0x3f40b78b, 0x3ed69440, 0x3f2d50b6, 0x3f6bdec5, 0x3f7b4367,
      0x3efa6a62, 0x3da8c580, 0x3dfb13b8, 0x3de07918, 0x3f473fdf, 0x3df834f8,
      0x3df558d0, 0x3e27efbc, 0x3f1ab25f, 0x3ed5af5e, 0x3ee2c9ba, 0x3f1bbf99,
      0x3d8dc3b0, 0x3f6fa25c, 0x3f18bc2d, 0x3d8db638, 0x3ef80f76, 0x3de52090,
      0x3ee26e70, 0x3e9ce620, 0x3e8f10c8, 0x3e62ada0, 0x3f6a02d0, 0x3f3457d3,
      0x3f0f148b, 0x3bec7800, 0x3eb62f06, 0x3f17133c, 0x3f0211b9, 0x3f56b35c,
      0x3f5c1b82, 0x3f5b93c2, 0x3f25babc, 0x3f237624, 0x3f115d74, 0x3ed65682,
      0x3ea877aa, 0x3eba8990, 0x3e67ae8c, 0x3e9cfce2, 0x3f0a8356, 0x3d611b90,
      0x3f79c40c, 0x3f00c1a1, 0x3f34a013, 0x3ea5df38, 0x3f09a2e3, 0x3e016568,
      0x3e11df40, 0x3f4f4e7b, 0x3e04d988, 0x3d958d60, 0x3d873190, 0x3e4571cc,
      0x3f2c5002, 0x3f2abab0, 0x3f4ace91, 0x3e4378b8, 0x3e584d8c, 0x3ea0633c,
      0x3d4d85b0, 0x3ca631c0, 0x3e8631d8, 0x3eefbb12, 0x3f762250, 0x3e85832a,
      0x3f7392cc, 0x3f16f6a8, 0x3da94818, 0x3f049964, 0x3f4c4075, 0x3f385374,
      0x3e9f60ea, 0x3f7c0541, 0x3f188ebc, 0x3f480457, 0x3f06a99b, 0x3f1cdb4b,
      0x3f471879, 0x3ef1355c, 0x3f0d7dc8, 0x3f5508a3, 0x3f78fa60, 0x3dcc6918,
      0x3f062a79, 0x3f5d14df, 0x3e05e104, 0x3f6fb45c, 0x3e9c0a5e, 0x3f1047e2,
      0x3e990dd2, 0x3f702316, 0x3f70868f, 0x3f18e412, 0x3f36c1c2, 0x3d1567d0,
      0x3ec2f846, 0x3f2f2bed, 0x3ea3a6d2, 0x3e052938, 0x3e74002c, 0x3f5bacb6,
      0x3ea05d00, 0x3d1b0c00, 0x3f545908, 0x3eb79dce, 0x3f62f452, 0x3eee4680,
      0x3f4e453d, 0x3f6dcecc, 0x3ebde2f0, 0x3d75c940, 0x3f4690d8, 0x3f77b2f9,
      0x3f5d7c4d, 0x3f6518a8, 0x3f559934, 0x3eec5048, 0x3f2c1e18, 0x3f63fa3f,
      0x3f79d1ff, 0x3f265ef8, 0x3e84972c, 0x3f430748, 0x3ef0b100, 0x3e5a6b60,
      0x3ec31cca, 0x3eeaebfe, 0x3e1c2028, 0x3f0ea570, 0x3f57326d, 0x3f6310ae,
      0x3eb2d324, 0x3e1f5700, 0x3dbc2160, 0x3f3a7367, 0x3f459dbc, 0x3d951300,
      0x3e7efe8c, 0x3f1c37dc, 0x3e7aa230, 0x3f6c120d, 0x3f4234c8, 0x3d168a80,
      0x3eee7c14, 0x3ea7113e, 0x3e68b00c, 0x3f4297f7, 0x3ea0b0ae, 0x3f55be3a,
      0x3f1ef513, 0x3f72b43c, 0x3f3a6516, 0x3ecac4d6, 0x3f039201, 0x3e794510,
      0x3eeaedd8, 0x3acafc00, 0x3e2e0a30, 0x3f749de8, 0x3ec2e9f6, 0x3e08f674,
      0x3e352a6c, 0x3f1b991c, 0x3ed49d32, 0x3f46f515, 0x3e550574, 0x3e3330cc,
      0x3f097109, 0x3efa98f8, 0x3ea2fd6a, 0x3e0404ec, 0x3f1f102f, 0x3e58d0d8,
      0x3ef864e0, 0x3f0055f8, 0x3e9f58b4, 0x3f1264ef, 0x3f6e1e66, 0x3d6e0380,
      0x3dcced30, 0x3ee02d72, 0x3ee0a450, 0x3f3dbcbd, 0x3f4b58e6, 0x3f106ff4,
      0x3eba5ef4, 0x3f362c53, 0x3f46a0a9, 0x3f290de0, 0x3eb889d8, 0x3d91fa90,
      0x3ef58618, 0x3d90bf98, 0x3e966d00, 0x3f7b3920, 0x3f492bf2, 0x3eb87c68,
      0x3c6bbbc0, 0x3f1b8c60, 0x3da5f5b8, 0x3e300078, 0x3eed0364, 0x3f50a465,
      0x3f11cce6, 0x3e2b11a8, 0x3f57243b, 0x3f120649, 0x3f7923d1, 0x3dea6f38,
      0x3f2325f1, 0x3f28b92b, 0x3e567b98, 0x3f7c185b, 0x3bac6980, 0x3f570842,
      0x3ee800cc, 0x3eea43ee, 0x3f5854b4, 0x3e3ee6f4, 0x3f22787d, 0x3f7eb373,
      0x3edca08a, 0x3f103e44, 0x3ee5f136, 0x3e6c3e70, 0x3d6f6d90, 0x3ed90442,
      0x3f031156, 0x3f482df2, 0x3ee1bc50, 0x3f56bb38, 0x3f3cfe7e, 0x3ee127b2,
      0x3f20a03e, 0x3e9a35a2, 0x3f780de4, 0x3f407beb, 0x3f5bf3c5, 0x3cc97360,
      0x3e855818, 0x3dc7e2e8, 0x3f170ad5, 0x3f64bb17, 0x3f6221fb, 0x3f76cb6d,
      0x3ee82c84, 0x3e8893cc, 0x3d434150, 0x3ed38ab8, 0x3eac05dc, 0x3f668dd6,
      0x3f49803b, 0x3f68561a, 0x3ca9b8c0, 0x3f4bdfb8, 0x3f0b4755, 0x3ee48e54,
      0x3eb4ef00, 0x3dbe76c8, 0x3d0444c0, 0x3d8fea10, 0x3f218222, 0x3f4c3771,
      0x3effa3e2, 0x3f512bd2, 0x3e062a1c, 0x3e58cc50, 0x3ea83bc2, 0x3d744060,
      0x3d19c180, 0x3ea59b66, 0x3f2e4483, 0x3f148556, 0x3f77eb26, 0x3e256ed4,
      0x3df48b50, 0x3f06bb54, 0x3f3529f8, 0x3e14a980, 0x3f57a6f2, 0x3f78465d,
      0x3ee75dba, 0x3f346d2f, 0x3f4253ca, 0x3efb970e, 0x3f3f5b7f, 0x3b682000,
      0x3ea78bb8, 0x3ea5be72, 0x3d2381b0, 0x3ea07b70, 0x3e97bbb6, 0x3ee4fece,
      0x3f7c73e0, 0x3e1f50a4, 0x3f4ceab9, 0x3f7e8915, 0x3f598169, 0x3e6ef65c,
      0x3ec93908, 0x3f2b51c7, 0x3f7178f1, 0x3eb384a8, 0x3e7b2f94, 0x3ce514a0,
      0x3da97e40, 0x3f4d4ce2, 0x3f323df1, 0x3f2c8b93, 0x3dbe1988, 0x3f15068a,
      0x3f6d2592, 0x3f2fb19c, 0x3e9b12b0, 0x3f189359, 0x3eafd3f8, 0x3e906f12,
      0x3e350cb4, 0x3ef71540, 0x3d880588, 0x3f7b54af, 0x3ee86f08, 0x3f193941,
      0x3f69f1dc, 0x3f0da5f1, 0x3f7e8cae, 0x3f0155a5, 0x3f50c0d3, 0x3ea06680,
      0x3ee6018e, 0x3f35bde6, 0x3f0c8ec9, 0x3f48b858, 0x3f34cb33, 0x3f2a8203,
      0x3f192248, 0x3f28ca8c, 0x3f1b99ec, 0x3e248d4c, 0x3f2e2321, 0x3f10ea99,
      0x3c1ee400, 0x3e811cb0, 0x3f04d7ec, 0x3e0ea6dc, 0x3eb266aa, 0x3b58bd00,
      0x3f742656, 0x3f5c446e, 0x3f55f024, 0x3f4242aa, 0x3f635c84, 0x3f519668,
      0x3efd1ed4, 0x3eddbbdc, 0x3c0c8200, 0x3f4f860d, 0x3f7a2db3, 0x3f4f22cc,
      0x3e7f413c, 0x3e9b018e, 0x3f40b589, 0x3e7cb6b0, 0x3d8db6c8, 0x3df00618,
      0x3e129dfc, 0x3eb2c392, 0x3e24078c, 0x3f668a5b, 0x3f732b7d, 0x3f2626e2,
      0x3f174914, 0x3f10ffbc, 0x3f76b981, 0x3f2a3556, 0x3f1918f3, 0x3e86c1b6,
      0x3ce4c600, 0x3e77149c, 0x3de0d530, 0x3e5dd56c, 0x3f2ca047, 0x3f57ec31,
      0x3f5ee5d5, 0x3e0a74b4, 0x3ec78b28, 0x3f3dba83, 0x3f3205b8, 0x3e84e402,
      0x3f462657, 0x3f4760a1, 0x3f2a4c2d, 0x3f7f45b9, 0x3f4a967e, 0x3f504b08,
      0x3f6ef1e1, 0x3f68745e, 0x3f2a8f59, 0x3ebe6142, 0x3f01f724, 0x3e3b4198,
      0x3f71f942, 0x3b47e300, 0x3f2ad0ac, 0x3f636b1d, 0x3ebe1466, 0x3e6c4184,
      0x3e5258c0, 0x3f65dc3c, 0x3f052a59, 0x3eb0faaa, 0x3ea7c1be, 0x3f34196b,
      0x3ebaadb4, 0x3e531e88, 0x3f2323c5, 0x3cfb8840, 0x3e3760f4, 0x3f22e05f,
      0x3f4e688b, 0x3f1e3be7, 0x3e276500, 0x3dd6e130, 0x3ef6257a, 0x3e8b053a,
      0x3f51e898, 0x3f7675c2, 0x3ef0dffc, 0x3ea9eb30, 0x3e6095bc, 0x3f5e1e30,
      0x3f7d498f, 0x3e3fd274, 0x3f23b067, 0x3f7910d1, 0x3f4c62c2, 0x3f1854bc,
      0x3e9636de, 0x3e96bef0, 0x3f66ad3a, 0x3f35ca4d, 0x3e116ee4, 0x3f3d2efc,
      0x3f126699, 0x3f24e4a7, 0x3dbe5e30, 0x3f6e5dc9, 0x3f6f4499, 0x3e5065b4,
      0x3ef74efe, 0x3e3fbcfc, 0x3e9af28c, 0x3f40dee9, 0x3d6f78c0, 0x3f1ef8cd,
      0x3f49b785, 0x3e9c5d2c, 0x3e28c99c, 0x3efc5c32, 0x3f6a96f4, 0x3f1eefce,
      0x3e82942c, 0x3da59718, 0x3e776bc8, 0x3e2db094, 0x3f0e7fbd, 0x3f4a7e9c,
      0x3f73e136, 0x3f0ef035, 0x3e54c844, 0x3d3ab0e0, 0x3ceaef20, 0x3f761c50,
      0x3eb43666, 0x3f4feac0, 0x3ed8ee16, 0x3eaa473a, 0x3f26617d, 0x3f5126e3,
      0x3f3f94a2, 0x3f22481a, 0x3f2e4011, 0x3e8ef2de, 0x3e365e14, 0x3f7dd058,
      0x3f60949d, 0x3ed27088, 0x3ece364a, 0x3f733387, 0x3dd4d518, 0x3e8fd25a,
      0x3a223000, 0x3f7955f5, 0x3f19a65e, 0x3ec22a74, 0x3f06bcde, 0x3db46c30,
      0x3f17d92c, 0x3f63e82b, 0x3ec37972, 0x3ee10f36, 0x3eb729e8, 0x3f16a912,
      0x3e20ece8, 0x3e6976bc, 0x3c9a7e40, 0x3f17379c, 0x3f1f96b4, 0x3efb6a70,
      0x3f3d9f7c, 0x3f493889, 0x3f104771, 0x3f691356, 0x3f653a89, 0x3ed9e542,
      0x3f6d9d14, 0x3ec5448c, 0x3f059ac9, 0x3ee5a888, 0x3d29c310, 0x3dc77fe0,
      0x3eaea27e, 0x3f29479d, 0x3e244c30, 0x3f431fc3, 0x3f7f4ff3, 0x3ef71410,
      0x3e250eb8, 0x3f0de805, 0x3c97f220, 0x3e1ec460, 0x3e641744, 0x3f03c34e,
      0x3e7bc584, 0x3f380601, 0x3f1f9a7e, 0x3cc19ce0, 0x3ecd6970, 0x3f4e7492,
      0x3f63b24b, 0x3e7b8df0, 0x3eb98e48, 0x3ed8b934, 0x3f52f89d, 0x3f3d69e9,
      0x3ed6865c, 0x3e849cac, 0x3f19a2a0, 0x3f3aa4fa, 0x3e951196, 0x3f15a2bb,
      0x3e02e578, 0x3e5c0068, 0x3ec80afe, 0x3f502e89, 0x3e7df2f4, 0x3f5a0e27,
      0x3f68c46e, 0x3e347d74, 0x3ec0f706, 0x3f2eabbb, 0x3ec41b44, 0x3ee48d30,
      0x3f53eaf8, 0x3e919d0a, 0x3e9953f2, 0x3f610369, 0x3f3207eb, 0x3f66f78c,
      0x3ee44a16, 0x3f1eb990, 0x3f7aff76, 0x3e52f3d0, 0x3ef4e9f0, 0x3d386090,
      0x3f0ebbc6, 0x3f246f9a, 0x3d5ffcc0, 0x3f1cc8b7, 0x3f749b7a, 0x3ef87be0,
      0x3f0fb75f, 0x3f18412c, 0x3ec070ea, 0x3f6a7f47, 0x3f3743ab, 0x3e8d5ac2,
      0x3f7bb722, 0x3f5f411a, 0x3eba39de, 0x3e431950, 0x3d333210, 0x3f5220ff,
      0x3f4a3d29, 0x3f61584e, 0x3ea5e54a, 0x3f58958b, 0x3e0f6de8, 0x3eb87362,
      0x3f171095, 0x3f674b58, 0x3f40e5f4, 0x3dc28630, 0x3f0a15fe, 0x3e25d584,
      0x3f02daca, 0x3f4faa88, 0x3f71cec7, 0x3f53415c, 0x3ef63138, 0x3ed4d464,
      0x3f29385b, 0x3efdfe26, 0x3f6990ee, 0x3f42bb6d, 0x3eb2342c, 0x3f0012d8,
      0x3e05c144, 0x3f208fdd, 0x3e836d7a, 0x3f6fd102, 0x3f2a1859, 0x3f56258f,
      0x3f4565e6, 0x3e7e2ddc, 0x3e904c0a, 0x3ee4514e, 0x3f35b7db, 0x3f4684ee,
      0x3f35c739, 0x3f02253b, 0x3f32d091, 0x3ef1ec72, 0x3e56d044, 0x3d43ca70,
      0x3f0ed186, 0x3f09b44b, 0x3f44b512, 0x3e4c63e8, 0x3eccedfe, 0x3e93d28c,
      0x3f4a4b8b, 0x3dbac570, 0x3da57090, 0x3f755b9b, 0x3f2dd421, 0x3e83d67a,
      0x3ecea17a, 0x3f68c315, 0x3e8beea2, 0x3ef7233e, 0x3f2e6261, 0x3ef28322,
      0x3d301700, 0x3ec74384, 0x3e875442, 0x3ebd7746, 0x3ee907e8, 0x3f38df88,
      0x3dabbee8, 0x3f61f4bd, 0x3ee71514, 0x3f085c8c, 0x3f405a59, 0x3edac0de,
      0x3f78adcd, 0x3f3ccc39, 0x3ea60262, 0x3f428f40, 0x3f2dbdd4, 0x3f7485c1,
      0x3eab8edc, 0x3e69e668, 0x3de87048, 0x3e7c4bfc, 0x3cfbaf60, 0x3f0e1a87,
      0x3e0a3a4c, 0x3f258435, 0x3ed50534, 0x3dd094b8, 0x3cb680e0, 0x3f46d914,
      0x3f7d40f9, 0x3e8f32e0, 0x3ec4d108, 0x3ec1ad0a, 0x3f0448c0, 0x3ec27550,
      0x3e16101c, 0x3e81ffd6, 0x3f09d368, 0x3f26a6b7, 0x3f2bd10c, 0x3f4a3448,
      0x3f680539, 0x3f28a1c4, 0x3e9e6cb2, 0x3f0ce0c0, 0x3e1303c4, 0x3f323574,
      0x3f720290, 0x3f3e9879, 0x3e8880e4, 0x3ebbe646, 0x3e78551c, 0x3f04bc80,
      0x3edf7e8c, 0x3e5b8e58, 0x3f7248e5, 0x3f10a57f, 0x3cb5f580, 0x3ef776ee,
      0x3f4a88a8, 0x3e7d2548, 0x3eb63a8e, 0x3eeb3014, 0x3f183e57, 0x3f38b629,
      0x3f61f7cf, 0x3f60aaa6, 0x3f4c3d61, 0x3f632e9d, 0x3f5cc626, 0x3f121382,
      0x3cf217c0, 0x3ea1e382, 0x3e311c44, 0x3e5324fc, 0x3f716802, 0x3e3d3080,
      0x3e6df308, 0x3e285a10, 0x3e9105f8, 0x3f7b6fd1, 0x3dbe3dd8, 0x3eb093ac,
      0x3f10a60a, 0x3d9ce3e8, 0x3f3415ef, 0x3f013730, 0x3f62d649, 0x3e3025f0,
      0x3f282129, 0x3f49fce7, 0x3f1945e7, 0x3f59e7e8, 0x3e99773c, 0x3f1e920a,
      0x3e227ea8, 0x3f5c3593, 0x3dfbd380, 0x3f3388d2, 0x3d744be0, 0x3f6007de,
      0x3db4cd40, 0x3f7ee35b, 0x3f62f411, 0x3dfdd1d8, 0x3f522532, 0x3e5a0e20,
      0x3f7aab9d, 0x3dff2c38, 0x3eba4ecc, 0x3f5bd074, 0x3f6b8e9a, 0x3f2aec15,
      0x3e875894, 0x3f0f5970, 0x3effed90, 0x3ea6615e, 0x3ef2dc18, 0x3de973b8,
      0x3c495280, 0x3f18e575, 0x3edd24cc, 0x3e279ef4, 0x3eb67724, 0x3f34ba39,
      0x3f383a9f, 0x3f0ca9dd, 0x3f61c23a, 0x3f38048a, 0x3f0ffde7, 0x3cff45a0,
      0x3e78b944, 0x3f19d762, 0x3f1efc57, 0x3f183493, 0x3f3c0ef1, 0x3e9b7286,
      0x3ea6a7f6, 0x3d1c3580, 0x3f3ba3b7, 0x3e7e7b94, 0x3f3b319d, 0x3efd118a,
      0x3ea636c6, 0x3c9e6720, 0x3ec85144, 0x3e87b89a, 0x3e7e22e8, 0x3ed49236,
      0x3f7627a4, 0x3e95db98, 0x3f7f9ee5, 0x3f6fe8b9, 0x3f0443ee, 0x3f4808da,
      0x3f30b5e4, 0x3f0bc545, 0x3f31cfdb, 0x38f86000, 0x3f233ba6, 0x3e99eb18,
      0x3f7d6c3d, 0x3f09b22c, 0x3f3f2ec4, 0x3f124e24, 0x3f128af7, 0x3f2a6d40,
      0x3ea586a2, 0x3edb681a, 0x3ece5504, 0x3f029603, 0x3f1a9d46, 0x3f383803,
      0x3f5cc771, 0x3a27c000, 0x3db04f68, 0x3dce6d18, 0x3f1d2042, 0x3f22fbbd,
      0x3dc10ea8, 0x3f4feda6, 0x3f5c7f71, 0x3f37a3c7, 0x3f21ee7c, 0x3c7d6ac0,
      0x3f4d839b, 0x3e28c33c, 0x3d603190, 0x3d50af60, 0x3f454931, 0x3e641980,
      0x3f053d19, 0x3f1e6400, 0x3bbaa300, 0x3ef85e94, 0x3e7c3734, 0x3ec13450,
      0x3efbb828, 0x3f539b80, 0x3f3c920b, 0x3f40db11, 0x3caa9ae0, 0x3f7476a0,
      0x3f31270e, 0x3f34d889, 0x3f265668, 0x3f38238e, 0x3f1e0f3f, 0x3f2a804e,
      0x3f5a0e18, 0x3f7922fc, 0x3f556142, 0x3f3c373b, 0x3ec56d56, 0x3f2ebf49,
      0x3f1a6b43, 0x3f7b678f, 0x3f449ef0, 0x3e28fcf0, 0x3db670a8, 0x3f16c306,
      0x3e215480, 0x3d2edb40, 0x3eb5862c, 0x3d7800d0, 0x3defac10, 0x3f49bef3,
      0x3b4d4900, 0x3f0326ec, 0x3e93b188, 0x3b953080, 0x3f24829a, 0x3eee36de,
      0x3f72a546, 0x3f24f435, 0x3f25ed29, 0x3f644f16, 0x3f601ce4, 0x3e00230c,
      0x3f007043, 0x3ecd65fc, 0x3f50572a, 0x3e6333d0, 0x3f6d09ee, 0x3f710c46,
      0x3d63f3a0, 0x3e77e610, 0x3f55738f, 0x3ddadde8, 0x3f57d809, 0x3dfcfbb0,
      0x3eb99b38, 0x3f7b469c, 0x3f0088bb, 0x3f105d58, 0x3dc54c08, 0x3f7c7ccf,
      0x3f311603, 0x3e4cb7c4, 0x3f7ff53c, 0x3eceb33c, 0x3f6a67b8, 0x3f070edd,
      0x3ef7f39e, 0x3d19c230, 0x3e107544, 0x3f28a38a, 0x3f7a0343, 0x3f3ab755,
      0x3f545ce0, 0x3d30e140, 0x3f351b67, 0x3ebd2506, 0x3f063351, 0x3f038cb3,
      0x3ed42b4c, 0x3e693418, 0x3f6da0cd, 0x3f632c9f, 0x3ca3a020, 0x3f367f4a,
      0x3f10126c, 0x3f0d7629, 0x3f07fe5f, 0x3cd47e20, 0x3e0dd1c0, 0x3f5738ba,
      0x3f77f75c, 0x3f56639f, 0x3f4d4f63, 0x3bee1000, 0x3d6a91e0, 0x3ce679a0,
      0x3d653f50, 0x3ef009b2, 0x3f707b7c, 0x3f6e3f8d, 0x3f149e22, 0x3f48d35b,
      0x3f0cbaf2, 0x3ddbda80, 0x3de6fb88, 0x3e90e962, 0x3f6943e1, 0x3ea97734,
      0x3dbe7628, 0x3f5ef3c4, 0x3f533603, 0x3f1f5e27, 0x3f521baf, 0x3d4c61a0,
      0x3e270004, 0x3eed11dc, 0x3f3b55fa, 0x3f2ea580, 0x3e837f60, 0x3dcaf748,
      0x3d024910, 0x3ea39b1a, 0x3f4343ad, 0x3f1416ee, 0x3e909014, 0x3f0b7a44,
      0x3f3ed8be, 0x3ef86f42, 0x3f403e99, 0x3e3c9860, 0x3f1e44f9, 0x3f639f3b,
      0x3ecdc35e, 0x3f37b491, 0x3f4c9133, 0x3f0de154, 0x3e6d3a18, 0x3cdc73a0,
      0x3f2e7908, 0x3ee000de, 0x3f28e79c, 0x3e30c8ac, 0x3ee7840a, 0x3f251eb0,
      0x3f0e7d49, 0x3f326641, 0x3f470952, 0x3f6bf969, 0x3f25092a, 0x3f33a95c,
      0x3f29c1b4, 0x3f358259, 0x3d3e86c0, 0x3f13c2da, 0x3ea2418c, 0x3d20d3d0,
      0x3da3bc68, 0x3e2e98bc, 0x3af5be00, 0x3f5c74db, 0x3f2ddc69, 0x3f1c4ebc,
      0x3ea536e0, 0x3f731956, 0x3f440397, 0x3f5a6bd7, 0x3edb8d4c, 0x3e825f1e,
      0x3eb9e452, 0x3f54d22a, 0x3f7f766e, 0x3f367c5a, 0x3f75ca05, 0x3f13ff6e,
      0x3e0c2060, 0x3ef3d12c, 0x3eb39a14, 0x3f2d35c9, 0x3db6a9e0, 0x3e9ea3f2,
      0x3f367a3f, 0x3f5277d1, 0x3f62678d, 0x3f482f61, 0x3f6c0a9e, 0x3f68105b,
      0x3f48a7c4, 0x3eb94e86, 0x3ee41e9a, 0x3e399568, 0x3f56523b, 0x3d27da80,
      0x3ea3688e, 0x3ec94926, 0x3f40360d, 0x3f6095a9, 0x3d560080, 0x3f42e8a0,
      0x3e221114, 0x3f357b2e, 0x3eeb2fa6, 0x3ec8f7de, 0x3f1e14bf, 0x3e9fb55a,
      0x3e75799c, 0x3e043474, 0x3f3e3f41, 0x3f62ec9f, 0x3edf39a2, 0x3f240071,
      0x3d0aa940, 0x3ec205ec, 0x3e90d806, 0x3eeedc50, 0x3f3c2177, 0x3f0cd18e,
      0x3e32bd0c, 0x3e2b5510, 0x3db62460, 0x3f3b50df, 0x3f5c8656, 0x3d0cb440,
      0x3eb97b82, 0x3eea0ba8, 0x3f445c08, 0x3f5e8508, 0x3e6ce20c, 0x3e8652b2,
      0x3f151cc9, 0x3e9ae2da, 0x3e948fe6, 0x3f1851b1, 0x3e5ccc3c, 0x3f0565e0,
      0x3e8522b4, 0x3eab573c, 0x3f5f7f0a, 0x3ece5d48, 0x3e0bc998, 0x3f03ed0c,
      0x3db93a08, 0x3ec80d58, 0x3f114b7d, 0x3f343551, 0x3f5ce96e, 0x3e7cfab0,
      0x3f40a8a8, 0x3f6285cf, 0x3e97a60c, 0x3ae6bc00, 0x3da64fa0, 0x3f492f69,
      0x3f665170, 0x3f1f7ba1, 0x3f7b2b34, 0x3ee5bb02, 0x3e413d78, 0x3e854a76,
      0x3f061166, 0x3c8bff80, 0x3f7a459f, 0x3ecbe174, 0x3f735006, 0x3e7056a8,
      0x3edb975c, 0x3e1d7c10, 0x3e768ddc, 0x3defc8e8, 0x3f41f65d, 0x3f37fbbf,
      0x3e8ec32a, 0x3eec0fa6, 0x3f38ed2b, 0x3ef3e1ea, 0x3f57dc16, 0x3ee48668,
      0x3f29bd4c, 0x3f2a4ba5, 0x3f24b7df, 0x3f1dc4d0, 0x3f6e480a, 0x3f27fd8d,
      0x3f0a3846, 0x3f4fa834, 0x3e458a54, 0x3c651500, 0x3d8294e0, 0x3f4f79b4,
      0x3e8af340, 0x3f750540, 0x3f10410d, 0x3f68d688, 0x3ec04f0e, 0x3e9b4964,
      0x3f57d1aa, 0x3e269844, 0x3d7dd470, 0x3f487e77, 0x3eea6ab6, 0x3f246078,
      0x3f4b6754, 0x3f78f291, 0x3f09c52d, 0x3f7e4924, 0x3f7665f0, 0x3f6613b2,
      0x3f5e8ecc, 0x3f55f7f1, 0x3d8810e0, 0x3e91f6ba, 0x3eb7a8a6, 0x3f145771,
      0x3f584d56, 0x3dd12d38, 0x3f6d7552, 0x3e151aec, 0x3eeedc04, 0x3f6f1fc5,
      0x3e503050, 0x3db59340, 0x3f403f9d, 0x3f4e20eb, 0x3f489973, 0x3f3799d7,
      0x3f75306d, 0x3de0a890, 0x3f65653d, 0x3f78506c, 0x3f42ad47, 0x3f5dfae5,
      0x3efaef42, 0x3f027d85, 0x3f18d19a, 0x3e55943c, 0x3e31d730, 0x3f63ae74,
      0x3d23fa00, 0x3ec87ec4, 0x3e930264, 0x3e53a034, 0x3d868598, 0x3f4cfbde,
      0x3f512309, 0x3c3be0c0, 0x3f41660f, 0x3eed1d32, 0x3e9f5226, 0x3f22bddc,
      0x3e36b2ec, 0x3eb74a74, 0x3f385fae, 0x3f6cd6fe, 0x3f28d8f3, 0x3f4034b6,
      0x3e87c4c8, 0x3f502e00, 0x3f463b41, 0x3ed869ca, 0x3f79ce5e, 0x3f4e5bd5,
      0x3f18efff, 0x3eedb4ee, 0x3ed8c8da, 0x3cacda00, 0x3f39b71e, 0x3f08a753,
      0x3f5adc83, 0x3dced118, 0x3f7f76b3, 0x3e961664, 0x3f4a107e, 0x3f15cb58,
      0x3f1482e9, 0x3e2fa464, 0x3e396c64, 0x3e5afe2c, 0x3f27651b, 0x3e9dd2f2,
      0x3d9471f8, 0x3f2fc954, 0x3d5d6120, 0x3f60812e, 0x3ebeee1e, 0x3ef74d5e,
      0x3e2360f8, 0x3f280e5d, 0x3f223bf5, 0x3e098208, 0x3ed5c05e, 0x3f419257,
      0x3f2f7988, 0x3e90e382, 0x3f73aa27, 0x3f5f8ca1, 0x3f6baa17, 0x3eebf75a,
      0x3eb9d238, 0x3db97c70, 0x3f13beb0, 0x3f2fd1dc, 0x3f1b43ba, 0x3eadb87c,
      0x3f22f4f3, 0x3f365e2c, 0x3eb67e5a, 0x3ee924b4, 0x3f19935e, 0x3f1fa90a,
      0x3f6f9c0d, 0x3e6e8b98, 0x3d969cb0, 0x3ebfbd52, 0x3f4ae7d3, 0x3ef7b4bc,
      0x3e32575c, 0x3d7e1040, 0x3f55f21b, 0x3f46b501, 0x3f5c4488, 0x3f0ad460,
      0x3d8bba38, 0x3f755ec6, 0x3c0772c0, 0x3e5e1668, 0x3f24868d, 0x3ec90faa,
      0x3eaa22fc, 0x3f337691, 0x3e967858, 0x3e9d1b90, 0x3ee62a4c, 0x3efd76d0,
      0x3eec2bfe, 0x3ed491ba, 0x3f470aa6, 0x3f70f153, 0x3e8ec54c, 0x3ec2486c,
      0x3ea9d4e6, 0x3f12c9bd, 0x3e5e73c4, 0x3e9d0336, 0x3e48e408, 0x3f6ba35e,
      0x3f7ce98c, 0x3f67f738, 0x3e30e000, 0x3f0cfef9, 0x3e800654, 0x3e24da18,
      0x3e4dde08, 0x3d63f810, 0x3e7dda24, 0x3f67f663, 0x3f682c33, 0x3df33cc8,
      0x3f4a26e4, 0x3ecd1220, 0x3f3d2216, 0x3ddce4b0, 0x3f65e844, 0x3ece340a,
      0x3f31a06d, 0x3ea55984, 0x3e92de8a, 0x3f152ad9, 0x3e714c98, 0x3f0982b7,
      0x3f75ba4d, 0x3c6a9380, 0x3ec5acfa, 0x3f038cd8, 0x3f6bfac5, 0x3f542edf,
      0x3c99a2e0, 0x3f6c8d6a, 0x3ec26f2e, 0x3e7785b0, 0x3ebddc76, 0x3f236c2f,
      0x3f5d7b7e, 0x3ce05620, 0x3f457b46, 0x3f675754, 0x3ebf4f7e, 0x3efbcbdc,
      0x3eae71ce, 0x3e990030, 0x3f706f56, 0x3f5827a4, 0x3f28a1b5, 0x3cdc3360,
      0x3ea5c976, 0x3f40fb45, 0x3f07fbd7, 0x3eef39ea, 0x3dbfa1f0, 0x3e871ab2,
      0x3f0444f9, 0x3f73d779, 0x3f008f41, 0x3ee47730, 0x3e5115b8, 0x3f2159a1,
      0x3f56c876, 0x3ebc632a, 0x3ee5cf6e, 0x3f4fba1e, 0x3eda8ce4, 0x3eae05b6,
      0x3f41a8c9, 0x3f6a0608, 0x3f3cf8b7, 0x3eab4d18, 0x3eaf8b42, 0x3e467e40,
      0x3e57039c, 0x3efac590, 0x3f5c5b9a, 0x3f487734, 0x3f40a058, 0x3f204690,
      0x3f1e977e, 0x3d1b5210, 0x3f1e529f, 0x3e2a73dc, 0x3ec374d8, 0x3f5b594f,
      0x3ef58e9e, 0x3d0b0450, 0x3f4f1793, 0x3db6d750, 0x3d9766e0, 0x3f0fbede,
      0x3f2a200b, 0x3e890cc0, 0x3eaf877a, 0x3f557638, 0x3d8476a8, 0x3f5fdda2,
      0x3effd012, 0x3efe6fea, 0x3ef36232, 0x3f5750dc, 0x3f7b98fd, 0x3f6e69bb,
      0x3ec07e7c, 0x3f67d100, 0x3dab3f40, 0x3de3f790, 0x3f39b8ac, 0x3f032c45,
      0x3f3b3a41, 0x3e92a656, 0x3f37ab47, 0x3ec76088, 0x3b168e00, 0x3f7ffc48,
      0x3f6d40da, 0x3ed852e6, 0x3e78782c, 0x3f365fc1, 0x3db7e790, 0x3f2a4690,
      0x3f44258b, 0x3f578106, 0x3ed8b2f6, 0x3f446045, 0x3e3ca5d0, 0x3f2a38e6,
      0x3eabb6be, 0x3ec0e31c, 0x3c312e40, 0x3f4c44ac, 0x3f73eac4, 0x3f2e6733,
      0x3f01bf30, 0x3f3774a5, 0x3f0af237, 0x3f59b555, 0x3f45a569, 0x3d17de70,
      0x3f185d3a, 0x3eea4ab8, 0x3f284b11, 0x3e8ad5f4, 0x3f29c851, 0x3f34265c,
      0x3e56612c, 0x3e9c3ba0, 0x3f4d67b6, 0x3f1b4671, 0x3f4876cf, 0x3f6b7e85,
      0x3f6c9081, 0x3e3444d4, 0x3ec7a8d0, 0x3f6df398, 0x3f15ae99, 0x3f17e000,
      0x3f439479, 0x3f7da41d, 0x3f062d00, 0x3db7a440, 0x3f5b43bf, 0x3f07b2d3,
      0x3f3777c8, 0x3e6cffd0, 0x3f78570b, 0x3f5306f8, 0x3dc78520, 0x3ed4442e,
      0x3f768b99, 0x3f0d2fd2, 0x3f019e24, 0x3f2061ec, 0x3f64958c, 0x3ddbafe0,
      0x3ede4ff0, 0x3f2c735d, 0x3f5c2ae3, 0x3f05987a, 0x3f3a8226, 0x3f103c9e,
      0x3f60991e, 0x3f17be78, 0x3f7d64cd, 0x3f7ecd70, 0x3f2a7cbb, 0x3f3392f4,
      0x3ee1fb42, 0x3eeb50f4, 0x3f6ceafc, 0x3e95aaf4, 0x3f51d9dd, 0x3f30d26e,
      0x3f672cec, 0x3ece85ca, 0x3f3a1108, 0x3eb9833c, 0x3ecdfc3c, 0x3f744326,
      0x3e4c115c, 0x3e944106, 0x3f51c13f, 0x3e038480, 0x3e8662c4, 0x3e4daaa0,
      0x3c34c500, 0x3f2c3bea, 0x3f74d7e6, 0x3f6bc555, 0x3eba5ba6, 0x3f70b2e2,
      0x3f421331, 0x3eb55894, 0x3ee6db14, 0x3e09bf0c, 0x3dffdf50, 0x3ec6aca2,
      0x3f2c1388, 0x3f2fb529, 0x3f66b2d9, 0x3f7278b9, 0x3f00df53, 0x3eb554fe,
      0x3cd22800, 0x3f41eda7, 0x3e96add2, 0x3e073dd4, 0x3f41d61f, 0x3e7d0e3c,
      0x3f3797a2, 0x3f2d6060, 0x3eaee812, 0x3f495f54, 0x3f17d3e9, 0x3e29a128,
      0x3ec94d3c, 0x3e878340, 0x3f397a76, 0x3e802aa2, 0x3ef726c8, 0x3f1b7d12,
      0x3f341766, 0x3e1ff270, 0x3e4b8150, 0x3f707c7e, 0x3f306ce9, 0x3ec96bd0,
      0x3f743dd4, 0x3eba121c, 0x3e9de208, 0x3e293244, 0x3d3474b0, 0x3f17851b,
      0x3f6750a5, 0x3f28eddf, 0x3f06e1bb, 0x3e34752c, 0x3e2f601c, 0x3c5a4cc0,
      0x3f049716, 0x3f0dcb62, 0x3f0a8c27, 0x3eb62324, 0x3d822228, 0x3ead06f8,
      0x3f3f0a3d, 0x3f7d1f3e, 0x3f067ad0, 0x3dc8bbc8, 0x3f7f414d, 0x3f41c1aa,
      0x3d1592a0, 0x3f56773d, 0x3eddb3a4, 0x3f79aaa3, 0x3d3a7be0, 0x3e838bcc,
      0x3f097530, 0x3e883aca, 0x3f28beee, 0x3f2f533a, 0x3f3ccde2, 0x3f7642ce,
      0x3f296fd6, 0x3f5bf4e3, 0x3f7baace, 0x3e69e954, 0x3f5c01ae, 0x3e2c1cbc,
      0x3f417069, 0x3e5c1d0c, 0x3eaa39ae, 0x3df9cf68, 0x3eb76c40, 0x3e593544,
      0x3f2e3814, 0x3ea9971a, 0x3f466aee, 0x3eab6256, 0x3ed7a840, 0x3f5520fc,
      0x3f55d7ab, 0x3f5edf92, 0x3f1f5934, 0x3f1339bf, 0x3f69261c, 0x3f6e3d77,
      0x3f08ed87, 0x3e07a544, 0x3f50f963, 0x3e8cbac4, 0x3dbb0060, 0x3f38aca9,
      0x3f29aac1, 0x3eb8da7a, 0x3f68687b, 0x3f4a6173, 0x3aa8ba00, 0x3f4410ec,
      0x3c191e00, 0x3db66910, 0x3ee78460, 0x3e8250c2, 0x3eda77e6, 0x3e150e0c,
      0x3e85435a, 0x3f3b6662, 0x3ecaca3c, 0x3f2623c9, 0x3edf3894, 0x3f182df0,
      0x3f3a721e, 0x3f5bb953, 0x3e9b0806, 0x3f5c289b, 0x3e9737d4, 0x3ee818ea,
      0x3f2c3e8c, 0x3f7085f2, 0x3eb91014, 0x3e14ff5c, 0x3f09fc37, 0x3e878f32,
      0x3f36a5c5, 0x3ea6c388, 0x3e933240, 0x3de07df8, 0x3f441b49, 0x3f1b30d8,
      0x3f33b738, 0x3e05692c, 0x3ec0c92a, 0x3cb27d20, 0x3f447bf3, 0x3f33eb6c,
      0x3d9ec4a8, 0x3f438b95, 0x3f4acfa9, 0x3f181633, 0x3f254931, 0x3e1dcf7c,
      0x3f2a58c5, 0x3e34e20c, 0x3f266b7a, 0x3f647e3b, 0x3f5a65fb, 0x3ee099ae,
      0x3eb35820, 0x3eaab0e2, 0x3e8738dc, 0x3e395a90, 0x3e88a826, 0x3f2187fd,
      0x3ed68c56, 0x3db58110, 0x3f486561, 0x3edd8560, 0x3f088cd6, 0x3f7be94f,
      0x3eebcafc, 0x3f63d8d3, 0x3f0b7a92, 0x3f5020e1, 0x3f2ae997, 0x3dc740a8,
      0x3f6c0217, 0x3ee646b4, 0x3f0510e1, 0x3f63518b, 0x3f292a1e, 0x3f70fc3c,
      0x3f72a4a8, 0x3f1106c8, 0x3f316cb3, 0x3f4630ed, 0x3ea42778, 0x3ee7383a,
      0x3f3f7ffe, 0x3f0779e9, 0x3e613330, 0x3f28fe48, 0x3f67f1b4, 0x3f1c7a16,
      0x3e9c3b4a, 0x3f124dce, 0x3ee4e97e, 0x3f1636eb, 0x3df17868, 0x3d8c80c0,
      0x3f25f56f, 0x3e8fd746, 0x3f61a234, 0x3f0965ee, 0x3e868be4, 0x3d768430,
      0x3f2c07b0, 0x3f7d6acc, 0x3f52edc2, 0x3f6fbf5d, 0x3ca6e3e0, 0x3e79da24,
      0x3f620b27, 0x3d3edd10, 0x3f482759, 0x3f0491a1, 0x3f0f50ba, 0x3eed244c,
      0x3f0a1668, 0x3ede80d4, 0x3edc13d4, 0x3f31beaf, 0x3de467e0, 0x3f185740,
      0x3ceb64c0, 0x3e55df5c, 0x3f0ce9de, 0x3eed511c, 0x3f5b1690, 0x3f72794f,
      0x3f336dab, 0x3e238744, 0x3f5316b3, 0x3f655fde, 0x3e292c48, 0x3ed5a3d8,
      0x3dd0dfd0, 0x3f44c4e0, 0x3f16495c, 0x3b92b700, 0x3f21bf81, 0x3f583be9,
      0x3f109eb0, 0x3f5df0b1, 0x3f467ca9, 0x3eae953e, 0x3f71a80f, 0x3da50470,
      0x3f47aa45, 0x3ec203d4, 0x3f726be5, 0x3ea4fdee, 0x3da21d58, 0x3f294699,
      0x3d8ae288, 0x3ed194aa, 0x3f47ab15, 0x3f247150, 0x3ed1772c, 0x3f544a52,
      0x3cd05740, 0x3f315a73, 0x3e1e1b2c, 0x3f32bb5f, 0x3f0f36c4, 0x3d7816a0,
      0x3f4cd169, 0x3e9cd702, 0x3ecafbe6, 0x3d3fda60, 0x3f6f5538, 0x3e866d56,
      0x3e31c90c, 0x3ef28564, 0x3f382084, 0x3f01b0c1, 0x3f5a0649, 0x3f7d7fad,
      0x3e6503c0, 0x3f6d0c17, 0x3f4b17fc, 0x3f764541, 0x3f41fcd5, 0x3f05afb0,
      0x3e51a34c, 0x3f2a0c40, 0x3f5cb1e6, 0x3ed4cbda, 0x3f10db2b, 0x3ed5ed12,
      0x3eb950a4, 0x3f7d710e, 0x3e72b1ec, 0x3c2da780, 0x3d1e3950, 0x3f651ef4,
      0x3f3e816b, 0x3ef423b6, 0x3e98b0d8, 0x3d36b290, 0x3ead6a7e, 0x3f6f7e1b,
      0x3f4012b1, 0x3ea35be8, 0x3f4544ce, 0x3f1377ff, 0x3f721bb8, 0x3f2cf517,
      0x3dfd3388, 0x3f7094cd, 0x3f1bf9b5, 0x3eaebfe8, 0x3f0a8a99, 0x3dd93808,
      0x3f0a5181, 0x3e3e3494, 0x3e77e78c, 0x3e2ae294, 0x3ec64f1e, 0x3f7a17b6,
      0x3db13438, 0x3f3bba34, 0x3df86fc0, 0x3f67f92a, 0x3ea7af42, 0x3d41f930,
      0x3d9a9040, 0x3f6c3d62, 0x3ef11d6e, 0x3e5eb804, 0x3f3dcc72, 0x3b6d7a00,
      0x3ebe9c3e, 0x3e5e8b30, 0x3f6cc3e7, 0x3e0ecaf8, 0x3f5085d8, 0x3f136306,
      0x3eea04f8, 0x3ecc6b04, 0x3e9aa354, 0x3f56f005, 0x3ee9a98a, 0x3eb50038,
      0x3f3fdd0e, 0x3f739072, 0x3edc6df0, 0x3f61328e, 0x3f257d3e, 0x3d0ec5c0,
      0x3f3790ce, 0x3eeb52c2, 0x3f306a5f, 0x3f2bd889, 0x3f5e42c6, 0x3f637ddd,
      0x3f1fed94, 0x3d7b5f70, 0x3da97158, 0x3f7fd7d5, 0x3f55b2b7, 0x3f0f845f,
      0x3f61a892, 0x3e641338, 0x3e986a90, 0x3f033e43, 0x3f42648d, 0x3e6d3d40,
      0x3ee2adde, 0x3cef7220, 0x3f1226fb, 0x3f60081e, 0x3f5767db, 0x3f2cd931,
      0x3ee94946, 0x3ece2fd2, 0x3e8b272c, 0x3e505a5c, 0x3f2e950c, 0x3f5d3165,
      0x3f7bb32c, 0x3f46cd2c, 0x3f641a27, 0x3dbe0a88, 0x3f231482, 0x3ee4dcde,
      0x3f08c3e8, 0x3eddcd8e, 0x3edbbf50, 0x3f0e508a, 0x3c029500, 0x3f5e7667,
      0x3f572967, 0x3ed03836, 0x3f364022, 0x3e2b88bc, 0x3ecdb87c, 0x3f4f4864,
      0x3e806d3e, 0x3f0a7597, 0x3ed0a246, 0x3ef8c7fa, 0x3eb39dec, 0x3f15bcab,
      0x3f521cc1, 0x3eeb58c6, 0x3ee273e8, 0x3ee64b6e, 0x3f1cb300, 0x3e70507c,
      0x3f4c7ada, 0x3ed285de, 0x3f74d3e4, 0x3f13b2b0, 0x3eb69026, 0x3df0a208,
      0x3f64acf9, 0x3eaee482, 0x3dd0ada0, 0x3ec1330c, 0x3d33aa30, 0x3efc51f4,
      0x3e506660, 0x3f0fcc47, 0x3e1bc37c, 0x3eedba8c, 0x3f6b1b56, 0x3ebbb730,
      0x3cbca4c0, 0x3ef07072, 0x3ea0d5fc, 0x3d3c8150, 0x3e910c26, 0x3d4b8220,
      0x3f16b029, 0x3edb4bee, 0x3f1856bb, 0x3e941346, 0x3cce5f20, 0x3d0297d0,
      0x3edf75be, 0x3ebf3696, 0x3eb4db48, 0x3f3ac8eb, 0x3e958cf6, 0x3e3b0190,
      0x3f7924b8, 0x3f2158b9, 0x3f283a31, 0x3f6f451b, 0x3f1ad181, 0x3f02752a,
      0x3eaa7496, 0x3e9e61ce, 0x3e89e424, 0x3f706390, 0x3ea8fc3c, 0x3e8c4196,
      0x3e8fd3e0, 0x3ef14214, 0x3f2f7db5, 0x3f183726, 0x3a17dc00, 0x3da3a6d8,
      0x3f13fa3b, 0x3ed8d010, 0x3d09d8d0, 0x3ece6948, 0x3f2a96ed, 0x3ed3d1e2,
      0x3dcdf918, 0x3f5a2b0e, 0x3f04d936, 0x3f61fd07, 0x3f3157e3, 0x3e8f2bbe,
      0x3db68a58, 0x3f35a7dd, 0x3f43e0f6, 0x3f05c40e, 0x3f7727bc, 0x3ec23d50,
      0x3f526af1, 0x3f35a25f, 0x3f1640bf, 0x3efb444a, 0x3bbe6600, 0x3f4d6d00,
      0x3f1d3e2f, 0x3efe6e34, 0x3e8f4d72, 0x3f04a7c4, 0x3ecf7ee6, 0x3f4a3ac3,
      0x3f1935b1, 0x3e3b9ccc, 0x3ea0885e, 0x3e42c47c, 0x3e9e71e6, 0x3f537d42,
      0x3ed8f5e2, 0x3f0891a9, 0x3f1bdf8b, 0x3eb4e490, 0x3ea74d7e, 0x3f7fb442,
      0x3f476d72, 0x3ddb9e80, 0x3e1eabac, 0x3d491a30, 0x3f02c419, 0x3f08d52d,
      0x3e9f71de, 0x3f02cc29, 0x3de4c790, 0x3ee8028e, 0x3f775731, 0x3e6fc344,
      0x3f26d772, 0x3f7d9af7, 0x3efcafaa, 0x3c8e0ae0, 0x3ee53c22, 0x3ee54618,
      0x3f2ab56d, 0x3edb31fa, 0x3d963600, 0x3f13813a, 0x3f6d9550, 0x3f1473f4,
      0x3f20891d, 0x3f7a432d, 0x3f1cab7e, 0x3eff8f70, 0x3f11b90f, 0x3ea4aace,
      0x3e224f90, 0x3e41257c, 0x3e2da9e8, 0x3f2d4e43, 0x3e754fd8, 0x3f6609e9,
      0x3d173320, 0x3f64929a, 0x3f46b6a9, 0x3e408dac, 0x3ee75e32, 0x3f433d57,
      0x3e85f0f2, 0x3f300ef5, 0x3f4aaa91, 0x3efe4ba2, 0x3f41e04f, 0x3e0ca418,
      0x3d95dc50, 0x3e9078ca, 0x3f675c91, 0x3eb8a6d0, 0x3f4baf39, 0x3f0c2056,
      0x3d889150, 0x3bd57f80, 0x3e3a3bf4, 0x3e4e4df0, 0x3eab0ffe, 0x3f0c133c,
      0x3ea79c34, 0x3e8cd78a, 0x3f05f9a1, 0x3f63d135, 0x3ee0af50, 0x3f27b67f,
      0x3ee4e7f8, 0x3f3c424d, 0x3f76cc28, 0x3bdfcd80, 0x3da7c8d8, 0x3e2aa0cc,
      0x3f12e4cb, 0x3f7b9b09, 0x3ec001c6, 0x3f5a9924, 0x3e39f428, 0x3f7dea38,
      0x3e77cae4, 0x3dc868a8, 0x3f2db1a3, 0x3ec09fc8, 0x3ebc5a10, 0x3ea7699c,
      0x3e3691a4, 0x3e8ce004, 0x3f456d22, 0x3e617344, 0x3f4c46f1, 0x3f302bad,
      0x3ebc5b8e, 0x3f2d88ef, 0x3f046b5b, 0x3c6da840, 0x3f07e6d8, 0x3e815280,
      0x3f4aab7d, 0x3ec04340, 0x3ea633c6, 0x3cd27500, 0x3f43a1fe, 0x3f2b887c,
      0x3f432174, 0x3d973220, 0x3f216c73, 0x3ea7b2da, 0x3da974a0, 0x3ea73f28,
      0x3f7726e0, 0x3ea9e0f8, 0x3f22ee20, 0x3f5295a8, 0x3f1a30ec, 0x3e713490,
      0x3f3b1cdb, 0x3f28c7a8, 0x3de7f4c0, 0x3f1962d3, 0x3dc47678, 0x3d580600,
      0x3f5ba907, 0x3c5ec180, 0x3ea29cf6, 0x3e0a047c, 0x3eb521bc, 0x3f2d12e4,
      0x3f0211b7, 0x3ea0e182, 0x3da1c048, 0x3f6bf772, 0x3f687cb3, 0x3f118e12,
      0x3e447350, 0x3f4b3135, 0x3d9e52d8, 0x3f68de56, 0x3f055759, 0x3f2783b2,
      0x3f276191, 0x3f3cfd76, 0x3f5e2fad, 0x3f353203, 0x3e00a790, 0x3e66de8c,
      0x3f6f604a, 0x3e9292ce, 0x3f59fb9a, 0x3f7d4d6c, 0x3bf91c00, 0x3f1b3f19,
      0x3ea647b8, 0x3f771c34, 0x3e231834, 0x3f554200, 0x3eb29dfc, 0x3e974ebc,
      0x3ef8058c, 0x3f14963e, 0x3c889480, 0x3f122b20, 0x3e9b9d12, 0x3f2399df,
      0x3d616380, 0x3f3d3cea, 0x3dc06950, 0x3f587785, 0x3d12b6e0, 0x3f7781fe,
      0x3f62c99c, 0x3ee9ee7e, 0x3f3cbefc, 0x3e5c43a0, 0x3f460b5f, 0x3da5f5d8,
      0x3dc047f8, 0x3e9ebaee, 0x3f6aa518, 0x3eded60e, 0x3f0290a3, 0x3f3117fd,
      0x3e8dcd20, 0x3f6eeccc, 0x3ed9a660, 0x3e752314, 0x3ed8b392, 0x3e35c83c,
      0x3f01303d, 0x3d0433a0, 0x3ea7ca66, 0x3ed2f11e, 0x3f62dd89, 0x3f2736d8,
      0x3e85ce88, 0x3f2cddad, 0x3dc79ee0, 0x3dd59368, 0x3cd7e200, 0x3ec8262c,
      0x3f0088c5, 0x3e8f9eac, 0x3ef69be0, 0x3f5703d8, 0x3f28dc67, 0x3f343f42,
      0x3e694760, 0x3eaf07d0, 0x3f4701f8, 0x3e9addd4, 0x3f5ecb0d, 0x3e01e05c,
      0x3f7c0eac, 0x3f61b64c, 0x3f3dbde3, 0x3f276e4b, 0x3f1b9fe1, 0x3f5a00a1,
      0x3e56a4d8, 0x3d355100, 0x3e97513e, 0x3f7563c0, 0x3f44a370, 0x3f484d75,
      0x3f14a3f0, 0x3edd9a86, 0x3e97c540, 0x3f2b38fa, 0x3ea52450, 0x3e1ffda8,
      0x3cb4f400, 0x3ea919ac, 0x3f16e4ac, 0x3f06c521, 0x3cef2c20, 0x3e16a5d0,
      0x3f121e6e, 0x3f5dd2ff, 0x3f3e0a5a, 0x3f7f21a2, 0x3f77966f, 0x3f45f1c8,
      0x3f54eecc, 0x3df838c0, 0x3f68cc88, 0x3bad1000, 0x3b71b500, 0x3ddc5348,
      0x3e3f66dc, 0x3ecdd10a, 0x3f1b1411, 0x3f768d3e, 0x3e9790d6, 0x3f31fdaf,
      0x3ecd4b82, 0x3e99dda8, 0x3f4fe66f, 0x3d883970, 0x3f73ffad, 0x3e817a76,
      0x3ef645de, 0x3f5f42fd, 0x3f685a96, 0x3f7ee0b8, 0x3e857726, 0x3d54d6d0,
      0x3ef6b218, 0x3f64870a, 0x3efa433a, 0x3f0a2d42, 0x3ecd5ef4, 0x3f27b5b8,
      0x3f18d599, 0x3f02a468, 0x3ec6ae00, 0x3e45c4e0, 0x3e796a30, 0x3e94739e,
      0x3f675640, 0x3f3364bc, 0x3f4c8dee, 0x3ea8903e, 0x3f1a1c09, 0x3e2e204c,
      0x3eb4b28a, 0x3f7fe159, 0x3f0cca06, 0x3e791940, 0x3ea3d052, 0x3f26184e,
      0x3f2a704c, 0x3d9060d0, 0x3f7a80cf, 0x3f4cc0c6, 0x3f2fc910, 0x3f3b2d26,
      0x3db69cf0, 0x3f026699, 0x3f454095, 0x3f6df60a, 0x3f7cd023, 0x3f677425,
      0x3e504aa0, 0x3f74c4de, 0x3f1d1252, 0x3f4d6716, 0x3dfeca28, 0x3f7d2b23,
      0x3e0efe44, 0x3ebd1232, 0x3d23bc30, 0x3f5d98b6, 0x3f53cbe5, 0x3f2c4d0b,
      0x3f1d8d9c, 0x3e04e914, 0x3f4eac75, 0x3f7b688e, 0x3ebf9742, 0x3eb606cc,
      0x3f15e1f4, 0x3f715d1b, 0x3ea678e6, 0x3d730a30, 0x3adc4600, 0x3ed41538,
      0x3f7d191c, 0x3f7944c0, 0x3ed2342e, 0x3f35d0b4, 0x3f37a5c5, 0x3f60a0e3,
      0x3f2d200b, 0x3f4b7f21, 0x3f39493a, 0x3f0a7a43, 0x3e298768, 0x3de3f950,
      0x3ee894ba, 0x3f603702, 0x3f23688b, 0x3f4d266a, 0x3f4b505d, 0x3f5bf4df,
      0x3e99ec82, 0x3eea0b92, 0x3eba586c, 0x3f241f30, 0x3ed36a62, 0x3f3b0ead,
      0x3f0c6c68, 0x3e1ca338, 0x3ef19a80, 0x3f0391b3, 0x3e4ae480, 0x3f312cd1,
      0x3e882b00, 0x3d6cdbd0, 0x3df20720, 0x3f552ce6, 0x3d8457d8, 0x3f279a29,
      0x3f14306c, 0x3f352f27, 0x3e948830, 0x3e9876d4, 0x3f1b89ae, 0x3dadca70,
      0x3ec64564, 0x3eb84b2e, 0x3ebfbf88, 0x3df3d1d0, 0x3c94a980, 0x3f332e28,
      0x3e358be8, 0x3f7d27eb, 0x3f00cc29, 0x3e110758, 0x3f4659a3, 0x3ef088e4,
      0x3f07299b, 0x3f0e49b4, 0x3ebf8244, 0x3f5bdcad, 0x3f30ee6b, 0x3f01aba3,
      0x3f3168b8, 0x3f5304c5, 0x3f46c8c2, 0x3f02b22d, 0x3d9b1968, 0x3e89197c,
      0x3ed0d7f8, 0x3f084022, 0x3ef2b110, 0x3dbf97b0, 0x3f42eb04, 0x3f5c85a5,
      0x3f02584f, 0x3ec52bbe, 0x3e3eee8c, 0x3f2c42e5, 0x3dd8dbd8, 0x3ed53dbe,
      0x3b89b580, 0x3e9d8f04, 0x3f231fea, 0x3ef7aef2, 0x3f0cfdea, 0x3f010a37,
      0x3ea89d68, 0x3d95ab20, 0x3eaba508, 0x3e504740, 0x3eb6260a, 0x3d955910,
      0x3b191000, 0x3c63b380, 0x3f5b70c3, 0x3f702761, 0x3f5597a9, 0x3f624aee,
      0x3f7f215a, 0x3ecb9636, 0x3ef8b348, 0x3f4a5694, 0x3d9e2ac8, 0x3d1d91f0,
      0x3f6d324d, 0x3d8694d8, 0x3f6ed642, 0x3f6c61fe, 0x3eaf6eb2, 0x3f4f39d3,
      0x3d8870c8, 0x39c60000, 0x3f6eeb8c, 0x3ed64f4e, 0x3ea5316c, 0x3f381d93,
      0x3dec5f48, 0x3f3fcd55, 0x3f7e3b3f, 0x3e437988, 0x3ee9f06a, 0x3f137d39,
      0x3d9b9cb8, 0x3e8f6fd4, 0x3f0db959, 0x3d9dc550, 0x3f0cf803, 0x3c824ca0,
      0x3f49b93c, 0x3f0024f8, 0x3f61567c, 0x3f0ef5bd, 0x3e3fb8b0, 0x3e89f82e,
      0x3f1acbc7, 0x3ebe55ca, 0x3efcee30, 0x3e192cd8, 0x3f116e1b, 0x3eef0498,
      0x3d1bd930, 0x3dfcf888, 0x3f29be21, 0x3f4c5b60, 0x3f05cd59, 0x3e04c3b8,
      0x3eb0b3cc, 0x3eb1c690, 0x3e884f5e, 0x3d3f6610, 0x3e7c24b0, 0x3ead28a0,
      0x3f602641, 0x3e373220, 0x3dd1cd80, 0x3e1397c4, 0x3f1e61b1, 0x3ec616c2,
      0x3efd70d6, 0x3f17132d, 0x3dee0960, 0x3f129730, 0x3f5e06a9, 0x3eb00f6e,
      0x3d459be0, 0x3f783b0d, 0x3cc6fc60, 0x3f1f52b1, 0x3f41e3d2, 0x3f590e1e,
      0x3f2c68a8, 0x3ec32c28, 0x3ec5a670, 0x3ede47be, 0x3f2943ca, 0x3f75085b,
      0x3f7f4144, 0x3f123e70, 0x3d8f9718, 0x3ecab2aa, 0x3e8f3f76, 0x3e8d5bf0,
      0x3f74b88f, 0x3f490d95, 0x3f637989, 0x3e7c7200, 0x3e0d5968, 0x3f4218e6,
      0x3f0b3015, 0x3f37ed57, 0x3da88578, 0x3eb5f42c, 0x3f4c8054, 0x3f080983,
      0x3eb9a226, 0x3f0af081, 0x3f61c384, 0x3f109a3f, 0x3f13bee0, 0x3f5db356,
      0x3f0d5e9c, 0x3ea82f4c, 0x3f1bcd4b, 0x3d51b510, 0x3f09f00c, 0x3d82ec08,
      0x3f5e6ee1, 0x3ecde5dc, 0x3e0f95a0, 0x3ed39f80, 0x3f413551, 0x3e812c1c,
      0x3f37428a, 0x3edcc670, 0x3f6d241a, 0x3ed78a4a, 0x3e8a14ba, 0x3eccb736,
      0x3daea0c0, 0x3f7fca4b, 0x3f2a8309, 0x3f02ce7c, 0x3b694900, 0x3f7aee80,
      0x3f437099, 0x3f121298, 0x3eeb6eb2, 0x3f5f222d, 0x3e207f14, 0x3f654eeb,
      0x3f5354b5, 0x3f53c7f7, 0x3f65607b, 0x3da58fb0, 0x3d020c20, 0x3ebab812,
      0x3d99f038, 0x3f48fb5d, 0x3da33a48, 0x3f2c382f, 0x3f4faac1, 0x3e44648c,
      0x3e17d650, 0x3f69759f, 0x3f5c2d9c, 0x3f2f7408, 0x3f4deeb8, 0x3e80369c,
      0x3f578763, 0x3f161695, 0x3ee9a0b2, 0x3f2387f6, 0x3e8ab8c6, 0x3f0fbe20,
      0x3f43d4d7, 0x3f0236dc, 0x3f03aeed, 0x3eb6a360, 0x3ec6ae26, 0x3f1cf6ba,
      0x3d8155f0, 0x3eb5f598, 0x3f2ec944, 0x3f4a5829, 0x3eca81ce, 0x3e7cfa68,
      0x3f2349be, 0x3f51cedc, 0x3e97243c, 0x3e15ae84, 0x3e97e128, 0x3f24110c,
      0x3f4d4bce, 0x3f5f5de8, 0x3efc9c5e, 0x3e98d9ee, 0x3e20c9fc, 0x3ee26668,
      0x3f19ec8f, 0x3f637941, 0x3f5c45a1, 0x3ef920aa, 0x3f284d25, 0x3f5f43b9,
      0x3f444684, 0x3d65a5f0, 0x3f330075, 0x3eda58ca, 0x3ea54fec, 0x3e9e695a,
      0x3f29556b, 0x3f0917b6, 0x3f3dde5d, 0x3bb33000, 0x3e32842c, 0x3e607d38,
      0x3f24c664, 0x3e474a28, 0x3f6d56dd, 0x3f44add0, 0x3f17fe92, 0x3f13ac13,
      0x3f4a9cde, 0x3e2eab70, 0x3e357750, 0x3f4ba011, 0x3e9934ee, 0x3f069a5d,
      0x3ea8c8c0, 0x3d824180, 0x3ed1326a, 0x3d65a1a0, 0x3f194f22, 0x3f4847f3,
      0x3f00b367, 0x3e9e3ed4, 0x3f6f9af8, 0x3f003f05, 0x3ef6506c, 0x3f7cf576,
      0x3e98e6f6, 0x3e9a530c, 0x3ec3ac60, 0x3ca49e20, 0x3f43598f, 0x3eda0dd6,
      0x3f770751, 0x3f6db49f, 0x3f002df0, 0x3f551e6c, 0x3ec5fa6c, 0x3e91dc8a,
      0x3f3c2046, 0x3f0c31bb, 0x3eded452, 0x3d0fc900, 0x3f140fe7, 0x3f0272ac,
      0x3f6cf8ef, 0x3f752bed, 0x3f71371d, 0x3f3e06fe, 0x3ec62fd6, 0x3da0cfd0,
      0x3f2770a1, 0x3e4ec10c, 0x3f359ed9, 0x3f4a0f07, 0x3f52ab68, 0x3f5a19cb,
      0x3ed9b99a, 0x3e243410, 0x3f5aabd3, 0x3f5da3d4, 0x3f04112d, 0x3f0c9229,
      0x3f1190d6, 0x3f5d2301, 0x3d577600, 0x3f031693, 0x3f5bb257, 0x3e8fb77c,
      0x3f035d2b, 0x3e592930, 0x3e982898, 0x3ebe9a04, 0x3f5fae2f, 0x3f0c312b,
      0x3e05dc00, 0x3f18bbf3, 0x3dfa7188, 0x3f10b13c, 0x3f2799d5, 0x3e736dec,
      0x3eaac872, 0x3f794288, 0x3f1c02af, 0x3edc8d40, 0x3eeb2854, 0x3ed88fc8,
      0x3f60f296, 0x3f17ef3f, 0x3e9c10ea, 0x3cebcc60, 0x3f3b4496, 0x3e38213c,
      0x3f1914ee, 0x3e145484, 0x3f251a26, 0x3f1e3292, 0x3e522ba0, 0x3f22835e,
      0x3f22030e, 0x3e484284, 0x3f42a72c, 0x3e5380cc, 0x3f5a0420, 0x3f3e0d80,
      0x3f0c6364, 0x3e6575dc, 0x3ebc5ca2, 0x3f5ebe2f, 0x3f6327c9, 0x3eb0dd20,
      0x3f1b9df8, 0x3e24c074, 0x3f0f117f, 0x3e87cef2, 0x3f10ad68, 0x3f44e1f6,
      0x3e1b9914, 0x3f44a79d, 0x3e069a74, 0x3ece147a, 0x3ee649b8, 0x3f5ac8a1,
      0x3ecfca14, 0x3f128ed6, 0x3ee27a88, 0x3de5bc88, 0x3f1fae3b, 0x3d4c0ad0,
      0x3ec8a086, 0x3ee8ee7e, 0x3f14b396, 0x3f75e742, 0x3f70ad2f, 0x3e34e8ac,
      0x3ea88008, 0x3f23a599, 0x3f5b54ce, 0x3f4a42b6, 0x3f1e336d, 0x3f0bb247,
      0x3e3c62d0, 0x3f4ad45d, 0x3f33b8a4, 0x3f701309, 0x3e7cba70, 0x3e6e9144,
      0x3ee55078, 0x3f0b4a38, 0x3f2208b1, 0x3f4ab2aa, 0x3ec93514, 0x3f54cda5,
      0x3ef9e268, 0x3f5318c3, 0x3f4271d0, 0x3e446518, 0x3c0cc540, 0x3f6ffca9,
      0x3eac477e, 0x3edd9ad4, 0x3f64204f, 0x3f790415, 0x3ef92596, 0x3e605498,
      0x3ec16dbc, 0x3f31715b, 0x3f0c059f, 0x3edd9362, 0x3d34d790, 0x3f0fb023,
      0x3eb1630a, 0x3f1b5d28, 0x3f7b4b91, 0x3f75c52d, 0x3f1cd993, 0x3f083531,
      0x3ed72c16, 0x3f011af4, 0x3e47e100, 0x3f11ed07, 0x3f105f4f, 0x3f6b70d1,
      0x3dce81e0, 0x3f5d206c, 0x3f2b5a0f, 0x3f510dd5, 0x3f773089, 0x3dcc1078,
      0x3f68548d, 0x3f7b910d, 0x3f0069c2, 0x3e9a6492, 0x3eb39030, 0x3f2c1836,
      0x3e6e8a30, 0x3f67045a, 0x3f5ebf5d, 0x3f32486f, 0x3ef29262, 0x3e13a8a8,
      0x3f42ab87, 0x3ebf0958, 0x3cbb6ce0, 0x3efe75b2, 0x3eff1664, 0x3d968ac8,
      0x3d755380, 0x3eace430, 0x3f3e1fad, 0x3dcebc90, 0x3ea1f368, 0x3f71ad48,
      0x3ef44b0a, 0x3f695d62, 0x3d8f45d0, 0x3f6325f6, 0x3ea517a6, 0x3f4b12ad,
      0x3ee85e72, 0x3e8bb160, 0x3eb37644, 0x3e6946f0, 0x3e963a1c, 0x3bc5d680,
      0x3d991c50, 0x3f0dba99, 0x3ead6efc, 0x3e2cbb1c, 0x3e93f2c8, 0x3f4c888f,
      0x3f06a59e, 0x3e5d6af0, 0x3eaa509a, 0x3f66abce, 0x3f0eb11f, 0x3e106ef8,
      0x3e01d60c, 0x3ebd6b4a, 0x3e83a364, 0x3d291040, 0x3f3147c6, 0x3f6c54ef,
      0x3f2d46e2, 0x3ea063ee, 0x3dbd23d8, 0x3f2c3471, 0x3f198565, 0x3f14cc7a,
      0x3f50197a, 0x3f4fe81f, 0x3d7f2de0, 0x3f6817dc, 0x3f66531d, 0x3eae3468,
      0x3ed86a7a, 0x3e8ffc1e, 0x3ecbc786, 0x3ed739a6, 0x3edfb1c8, 0x3f0bed6b,
      0x3f153681, 0x3e90d1ac, 0x3f688961, 0x3cdd5920, 0x3f2553a5, 0x3ed1ddc8,
      0x3e7fbcac, 0x3f3641e1, 0x3ebb566c, 0x3dc216d8, 0x3db053e0, 0x3eff6246,
      0x3f2e5590, 0x3ef1d916, 0x3e2611d0, 0x3ef35528, 0x3f5048ad, 0x3e206988,
      0x3ce4f4e0, 0x3f0c57e6, 0x3eb6e9c6, 0x3ed75f84, 0x3f69ffdc, 0x3f3f286c,
      0x3f7f7f91, 0x3e011f98, 0x3db3a9d8, 0x3f58e30d, 0x3f310050, 0x3e1d9ce0,
      0x3ef19468, 0x3e8f5922, 0x3ef21cf8, 0x3f6651bd, 0x3e3668e8, 0x3f08cb81,
      0x3f1a8bee, 0x3f40511c, 0x3f692993, 0x3e435dd8, 0x3ef67a0a, 0x3e0cefd8,
      0x3f6c71b4, 0x3f3acd2b, 0x3c410100, 0x3ebc59c0, 0x3da99290, 0x3e6a7b8c,
      0x3dafd3d0, 0x3d2a67f0, 0x3f450ccb, 0x3ddb5478, 0x3d39eb10, 0x3f0eafb7,
      0x3e754dc0, 0x3e1c7c50, 0x3f5ee4b2, 0x3f1c7e7a, 0x3f60c0d1, 0x3f0b805c,
      0x3f16f607, 0x3f25316c, 0x3f5c4ba5, 0x3f5dcab2, 0x3f3b84a1, 0x3d3da1b0,
      0x3a912400, 0x3ddbe120, 0x3e504938, 0x3ea8786a, 0x3f60c1da, 0x3f22c4fd,
      0x3f1d1d95, 0x3e711420, 0x3f0ef8af, 0x3f043f45, 0x3f5293d7, 0x3f291b7f,
      0x3eea3d2c, 0x3e5e64bc, 0x3f4825cc, 0x3eec1d98, 0x3ed758d4, 0x3e89a952,
      0x3cfde3c0, 0x3df27828, 0x3d927b30, 0x3d87e550, 0x3f3b4132, 0x3f0eacf2,
      0x3ec04a88, 0x3e82f43c, 0x3f6252aa, 0x3f7cc3a5, 0x3e865bf2, 0x3e710808,
      0x3e094d7c, 0x3f59cebc, 0x3eed3662, 0x3f1b9e85, 0x3dbe8080, 0x3d8b4470,
      0x3f0c5d83, 0x3d23fb30, 0x3d7561c0, 0x3f781334, 0x3ed7dcaa, 0x3ed4da30,
      0x3f5e52b1, 0x3f54d0af, 0x3ebe54a8, 0x3d85b380, 0x3e5194b4, 0x3f5493d8,
      0x3f79d6f0, 0x3d887998, 0x3f03ace0, 0x3eb52c78, 0x3f7751ac, 0x3f04adea,
      0x3d050890, 0x3ebaf404, 0x3e97a6d6, 0x3ebc7e76, 0x3dbf44a8, 0x3e158ffc,
      0x3f2da489, 0x3eaa71a8, 0x3e980f0c, 0x3f2a1f1e, 0x3f7779af, 0x3f639c13,
      0x3debd510, 0x3f513358, 0x3ee5acf8, 0x3ebf290e, 0x3ed4b634, 0x3ed78f0a,
      0x3f2d7ee6, 0x3dc26f60, 0x3f2c05ce, 0x3e597350, 0x3f172c2e, 0x3d738630,
      0x3f4674f7, 0x3e80a71a, 0x3f3a06f0, 0x3f68d353, 0x3f6f7572, 0x3f2a407b,
      0x3cc1ebc0, 0x3bf95f00, 0x3c373940, 0x3f647d22, 0x3e898bd6, 0x3db94918,
      0x3e0e7130, 0x3eb4baa6, 0x3eef9370, 0x3f7f4a1a, 0x3ea125b4, 0x3f5307a1,
      0x3d539b00, 0x3f32ee21, 0x3f1fa938, 0x3f4cf527, 0x3f34f504, 0x3f19c8da,
      0x3ee7ff62, 0x3f791b26, 0x3f674aad, 0x3f5433cb, 0x3ebf244c, 0x3f2564cc,
      0x3f2ee888, 0x3f6ee3dc, 0x3f402302, 0x3e2db948, 0x3f10354d, 0x3d77e9a0,
      0x3e086dfc, 0x3ebbe8a0, 0x3f5717f8, 0x3f098e91, 0x3f50a649, 0x3f21cf6e,
      0x3eef57ee, 0x3e6f544c, 0x3f25f7b4, 0x3eafbafe, 0x3f062946, 0x3ee87e1e,
      0x3ea6e1dc, 0x3f1c66a4, 0x3ee6fe92, 0x3ee91d80, 0x3f572123, 0x3ec88b50,
      0x3f742018, 0x3f54e90f, 0x3e65086c, 0x3f0a6c6e, 0x3e8fd818, 0x3f244c65,
      0x3d2a4a40, 0x3f72aaa5, 0x3e95a6de, 0x3f1ebabd, 0x3f340a85, 0x3d4c40f0,
      0x3e63d1dc, 0x3f579876, 0x3f29635d, 0x3e5158d8, 0x3f0c63df, 0x3e11fe8c,
      0x3efd9a12, 0x3efb66cc, 0x3eca6a44, 0x3ed66508, 0x3dec61c0, 0x3d35f550,
      0x3de20030, 0x3db20850, 0x3f75d785, 0x3e89cfa0, 0x3ee36e38, 0x3ea5cf3a,
      0x3f628708, 0x3e4a2ba4, 0x3e3ef4e0, 0x3ebe1fd4, 0x3f3204d2, 0x3f388c96,
      0x3d266f80, 0x3f4e05f6, 0x3f570918, 0x3e3650e0, 0x3e44cef8, 0x3f461eba,
      0x3ead68a2, 0x3d984998, 0x3f51f8aa, 0x3e748b64, 0x3f6ca2d9, 0x3f20e07b,
      0x3e94f292, 0x3e08ffe8, 0x3f548a2d, 0x3e1a4b84, 0x3e60af70, 0x3e9aff5a,
      0x3efda772, 0x3e3a124c, 0x3f645cca, 0x3e188670, 0x3f45c4e3, 0x3f0502e6,
      0x3e865ef4, 0x3ebb3d6a, 0x3e5807b4, 0x3eef22f4, 0x3f0e776c, 0x3f325003,
      0x3f5be481, 0x3f3bada9, 0x3f49fa0c, 0x3d996b70, 0x3e56903c, 0x3e8d8cb2,
      0x3d8e7520, 0x3f58b77a, 0x3ba20200, 0x3cbe9c40, 0x3f60815f, 0x3cc5e9e0,
      0x3e71f000, 0x3f62a7ea, 0x3e6d3bd8, 0x3f59cb91, 0x3ec5af5e, 0x3ecc41b2,
      0x3e8ad474, 0x3f3b6b09, 0x3ed75dc2, 0x3e0f521c, 0x3ed84246, 0x3e9a1586,
      0x3cb52520, 0x3f1d0fcd, 0x3f6bd064, 0x3f599cab, 0x3eb66bf6, 0x3f535c3a,
      0x3b856700, 0x3ebca092, 0x3ecc6d48, 0x3f08dcc0, 0x3b897f00, 0x3f2ae8ed,
      0x3f2d2a36, 0x3e464a74, 0x3dcb02e8, 0x3eda8a62, 0x3f4b9edc, 0x3f125394,
      0x3f788ddb, 0x3e7b4104, 0x3d5c6a60, 0x3f6309c8, 0x3e8c04be, 0x3f3667b9,
      0x3ed8dc34, 0x3ef846f6, 0x3f02654a, 0x3de06ea0, 0x3f4cb88b, 0x3d6d4bc0,
      0x3e9cad44, 0x3f6ed175, 0x3f098e2a, 0x3f3d5201, 0x3f3ce099, 0x3e733bb0,
      0x3f37f4c5, 0x3ea4e26e, 0x3f1f2323, 0x3f556ee5, 0x3f455e12, 0x3dba88b8,
      0x3f4b6c7e, 0x3ed9a0a6, 0x3eb84ec4, 0x3f6619fc, 0x3f2dae60, 0x3ec23506,
      0x3d520070, 0x3ef910c0, 0x3f2ab23a, 0x3d07a9b0, 0x3f29343c, 0x3e169d5c,
      0x3e85c5e4, 0x3e8ccb44, 0x3f0d6acc, 0x3f0610ab, 0x3da08f18, 0x3ef4f6c2,
      0x3f6c30d2, 0x3e864474, 0x3ebcc42a, 0x3edf7a74, 0x3ef1a0b4, 0x3e8aaf86,
      0x3f38f8f8, 0x3f245c9e, 0x3f418e46, 0x3e041c3c, 0x3e960e66, 0x3f277689,
      0x3d12b8f0, 0x3e127ed4, 0x3e0e877c, 0x3f576bb4, 0x3f60cb7a, 0x3d9b4cf0,
      0x3f02649d, 0x3ef08648, 0x3e76ea30, 0x3ed2ec12, 0x3f3181de, 0x3e73a63c,
      0x3df88ff8, 0x3ec23b34, 0x3f1fa52a, 0x3f569318, 0x3f033044, 0x3f589280,
      0x3de7e730, 0x3f1c62a8, 0x3eb6dd84, 0x3e6c4f20, 0x3d413ab0, 0x3f5b45cf,
      0x3e80d9a0, 0x3edf4900, 0x3ef9cf66, 0x3e8e76d8, 0x3d2590e0, 0x3f0dcfa7,
      0x3f7f0b3d, 0x3e73b644, 0x3ea4492e, 0x3f302059, 0x3da70828, 0x3e8a3234,
      0x3e58c8c8, 0x3eb2b674, 0x3f48fb98, 0x3f105af2, 0x3f598be0, 0x3d7be3d0,
      0x3f57fb03, 0x3ec08bbe, 0x3f788d7f, 0x3f495be3, 0x3f501880, 0x3f594dd8,
      0x3d3d6b10, 0x3f367bd7, 0x3f6d9f4a, 0x3f369aae, 0x3d698ae0, 0x3f027daf,
      0x3eee638a, 0x3f1d3820, 0x3f2134f1, 0x3f6675db, 0x3d8e8820, 0x3ed46a7a,
      0x3f73c1e5, 0x3eb09022, 0x3e3b7b2c, 0x3e069588, 0x3f6a933d, 0x3ee4b4b4,
      0x3ec684b4, 0x3f6451c7, 0x3ea6806c, 0x3f1cec2d, 0x3f339f15, 0x3ec154e4,
      0x3ea7c544, 0x3f0b5d07, 0x3e8b78d0, 0x3f5f1241, 0x3e3d5bec, 0x3f44d002,
      0x3e8ed2c2, 0x3f45e176, 0x3ece9b6a, 0x3e8bff6c, 0x3ee269a8, 0x3e379324,
      0x3e0df360, 0x3e9ea76a, 0x3e806788, 0x3e401320, 0x3f4eea32, 0x3eefd2f2,
      0x3f2911b5, 0x3d9bb480, 0x3f4c48e2, 0x3f7df2cf, 0x3f43a17f, 0x3f534ea2,
      0x3e0ac4a8, 0x3c91cfe0, 0x3cd24a20, 0x3da674c0, 0x3ef5e8de, 0x3ec76640,
      0x3e3a5a88, 0x3eaaa6d4, 0x3f059644, 0x3e0c21f4, 0x3e9584ce, 0x3f4419cc,
      0x3f6f30ca, 0x3f4ee81e, 0x3f5c7c9a, 0x3d9d93a8, 0x3ea9e01c, 0x3f204655,
      0x3f5f3f42, 0x3f507d1e, 0x3dd3d140, 0x3f1fac08, 0x3eb3896e, 0x3d98b210,
      0x3e744904, 0x3f2701f6, 0x3ed3260e, 0x3f1f0b91, 0x3e423b14, 0x3f0ec2bc,
      0x3f02b3c0, 0x3f2433a2, 0x3f462c82, 0x3d7a6790, 0x3e2dd630, 0x3eeac03a,
      0x3ebb3756, 0x3e280664, 0x3ea68fea, 0x3f4e6127, 0x3f5a9e70, 0x3e359020,
      0x3f6171f9, 0x3eb47ea4, 0x3e86c8c4, 0x3f24b254, 0x3f153db9, 0x3e8848a8,
      0x3d5150b0, 0x3f0f86c9, 0x3f442c61, 0x3f2b5c06, 0x3f2a0203, 0x3f6ae7f9,
      0x3d4ff760, 0x3d825e90, 0x3e1d82ac, 0x3c2d3900, 0x3e0bec74, 0x3f4b4ae0,
      0x3f3ea747, 0x3d38e940, 0x3f7fb9d4, 0x3d2f52e0, 0x3f17bb21, 0x3f784439,
      0x3f0619a3, 0x3df725c8, 0x3f72d11d, 0x3db65e90, 0x3f720730, 0x3f783c9c,
      0x3eb51d22, 0x3f27b0cc, 0x3f05baec, 0x3e96f534, 0x3f461b79, 0x3f1e7da3,
      0x3f418f68, 0x3edf861e, 0x3f14a742, 0x3d644f00, 0x3ef4cd4a, 0x3eaeb200,
      0x3f3491c8, 0x3ed2cdaa, 0x3e7b3738, 0x3f40d64f, 0x3e8a2624, 0x3f3d2bb5,
      0x3e23b094, 0x3f4fa2aa, 0x3f1dd2f1, 0x3edd400e, 0x3f7b44c9, 0x3e70bbb8,
      0x3f492120, 0x3e298a98, 0x3dfa5b98, 0x3e9b0b02, 0x3e92b148, 0x3f6494f3,
      0x3f4a54c1, 0x3e435654, 0x3ebd4994, 0x3f6f391c, 0x3eb764e0, 0x3f728eca,
      0x3edbccd2, 0x3ec57116, 0x3f6654cc, 0x3f10e510, 0x3f062043, 0x3f693583,
      0x3d8a1020, 0x3f4edcf2, 0x3f2559da, 0x3f10889d, 0x3eb765e0, 0x3f55386e,
      0x3f35beee, 0x3f148d45, 0x3f5cbf2e, 0x3e607d30, 0x3ef5e90a, 0x3e9d6a82,
      0x3f4c40d1, 0x3d8ce2a8, 0x3f5477a7, 0x3eb4a2c2, 0x3ec3bf78, 0x3e994b32,
      0x3b59dc00, 0x3f5a9476, 0x3f7a4370, 0x3d9fc408, 0x3e5048fc, 0x3ea3ab18,
      0x3d191bb0, 0x3e56a758, 0x3f206535, 0x3e4637c8, 0x3f0140c8, 0x3d54a3c0,
      0x3ee116be, 0x3f63e06b, 0x3e08bfd8, 0x3f7f2a9d, 0x3f34bac9, 0x3f78b84a,
      0x3d7c0d10, 0x3f2ed9da, 0x3d4240e0, 0x3e3212b8, 0x3c426440, 0x3e738ae4,
      0x3e802148, 0x3f218f91, 0x3f4f649e, 0x3f46325e, 0x3f2cb56a, 0x3efd33e0,
      0x3f63e5c0, 0x3f6a03c8, 0x3f67f7c3, 0x3f57f7d1, 0x3f128f66, 0x3eb8db2c,
      0x3f1768ed, 0x3f2f1311, 0x3f7e06ce, 0x3f539971, 0x3e36d1bc, 0x3f21276d,
      0x3f4c7853, 0x3be14a80, 0x3e241fa8, 0x3f68cd85, 0x3f5dadd7, 0x3da7f6e8,
      0x3f1e2519, 0x3f32df33, 0x3f74f0f1, 0x3f6dd595, 0x3f539f6c, 0x3f043a4a,
      0x3f7d281d, 0x3f7c2902, 0x3ea4c72a, 0x3f44af85, 0x3f28cbf4, 0x3f6afd3b,
      0x3f34f8ff, 0x3e95bc74, 0x3f2167a4, 0x3ef247ee, 0x3e99cb86, 0x3ea68870,
      0x3f25b107, 0x3f5dbbef, 0x3f58b045, 0x3ebe5cb0, 0x3f1e7667, 0x3f186432,
      0x3f60e65a, 0x3f679bba, 0x3de9cd40, 0x3cb8fd20, 0x3f3b2e03, 0x3f35b46f,
  };

  // 4,6,6,7
  uint32_t kernel_vals[] = {
      0x3c843120, 0x3d9cb687, 0x3d5b919e, 0xbd96186c, 0x3d58fa3a, 0x3d2d416a,
      0xbd0d67ab, 0x3d843683, 0x3d71186a, 0xba010e00, 0xbd9f03ba, 0x3d06df00,
      0xbc1a3998, 0x3d79f232, 0x3daa7307, 0x3d925f53, 0xbb375d80, 0x3d134948,
      0xbd1e015e, 0x3d34a30a, 0x3d18cc42, 0xbd3504c0, 0x3cf0d59c, 0x3b31aa80,
      0x3d82c6a5, 0xbd98537c, 0x3cd30210, 0xbd32858e, 0x3d962b29, 0xbd041de8,
      0x3d5905ca, 0xbd9318dc, 0xbc822da4, 0xbd99ca93, 0xbca3e600, 0x3d8da64d,
      0xbd764cd8, 0x3c444220, 0xbd7b8ddb, 0x3d32a706, 0x3c441368, 0xbd3e4fd3,
      0xbc691d58, 0xbd96f41f, 0x3da60aeb, 0xbd4b25de, 0xbd95ae4c, 0x3d577b42,
      0x3d946765, 0x3d5dfee2, 0xbd1e98c3, 0x3d08d7f8, 0x3cda02c0, 0x3d3aa8ca,
      0x3d621622, 0x3cdba8cc, 0xbd0418ee, 0x3d98a03d, 0x3d872eed, 0xbd827dc6,
      0xbc851910, 0xba82bcc0, 0xbda376c0, 0x3d9e0c99, 0xbd82fce2, 0x3a965840,
      0xbcb0ab80, 0xbc87e814, 0x3b674180, 0x3d722b1a, 0xbd8ff94b, 0xbd24e630,
      0x3da4799d, 0x3d207270, 0xbcb18006, 0x3cc6e480, 0x3d07dcb2, 0xbd27e0b8,
      0x3ce3646c, 0x3cfd7400, 0x3b931450, 0x3c9d81c0, 0x3d827fff, 0xbcb4a356,
      0x3d90e22b, 0xbd5dc973, 0xbd93ad6e, 0x3d86e28b, 0xbcfe8596, 0x3a359380,
      0x3c8cefc0, 0x3d1e98e8, 0x3d92a301, 0xbce5a52c, 0xbc7dc138, 0xbd70686e,
      0x3d67f49a, 0x3c5fb808, 0x3d6cfd3a, 0xbd762403, 0x3d91afd3, 0xbcac63d6,
      0xbca5e2f0, 0x3da11785, 0xbc47cad8, 0x3d686e6a, 0xbd9d768b, 0xbbd3e1c0,
      0x3d29cdf8, 0x3c8e1a50, 0xbbdf52c0, 0x3d84ca75, 0x3d558672, 0x3cb35b44,
      0x3d3ae79a, 0xbba14450, 0x3c4f23e8, 0xbd92a8e0, 0xbc7e1008, 0x3d0aef56,
      0xbd1ee9c0, 0x3c1ef9e8, 0xbd816bee, 0x3d5d5a0e, 0x3cf05d90, 0x3d945d4b,
      0xbd7c9058, 0xbd84b6db, 0xbd22a31b, 0x3cab977c, 0x3d937d2f, 0xbc86fb94,
      0xbd9aefe6, 0x3d965a17, 0xbc85e5cc, 0xbd8383cf, 0x3d3ed7da, 0xbd93875c,
      0x3ca917f0, 0x3d27d858, 0xbca242f0, 0xbdaa27de, 0xbd41e7de, 0x3cd140cc,
      0x3da4c293, 0xbd9d1c4f, 0xbd8573b4, 0xbd12fe33, 0xbd4da3e8, 0xbd9c6e88,
      0xbc53e6e8, 0xbd3007c6, 0xbda06ec6, 0xbbe3b240, 0x3cc78960, 0xbb067e00,
      0xbc596918, 0xbbda3d70, 0x3d00ebbe, 0x3d104f7a, 0xbc8a715c, 0x3bb9fa70,
      0xbd9b2ed4, 0xbd29cdb6, 0xbb7bdc20, 0xbda18690, 0x3d8aa6f3, 0xbd8896d6,
      0x3af500c0, 0x3da33325, 0x3d688536, 0xbcf6ccb6, 0xbd7d58b0, 0x3cfffc2c,
      0xbc37caf8, 0xbd80b8fa, 0xbcfc3a1c, 0xbd60d596, 0x3d316982, 0xbc6309a8,
      0xbd838248, 0xbce41de6, 0xbd779b3b, 0xbd853a67, 0x3d999799, 0x3d863273,
      0xbc750968, 0xbd2b6248, 0xbbebcf40, 0xbca76e64, 0x3d714b62, 0x3bed9700,
      0xbd86f7ae, 0x3d2196fa, 0xbb24a280, 0x3ceb63fc, 0xbd0a6a16, 0xbd256e18,
      0xbd8f2f32, 0xbd2632de, 0xbd6850db, 0xbd02f30b, 0x3d4e678e, 0xbc3a5be8,
      0x3da842a5, 0xbd004ba0, 0xbc3ad7c0, 0xbbfebf80, 0xbce8a4d6, 0xbd07428e,
      0xbc3f40e8, 0x3ca5ae7c, 0x3ceca810, 0xbd2a1138, 0xbd9a6d3c, 0xbd7f63d0,
      0x3c99953c, 0xbd90eb57, 0x3d2dc97e, 0xbd69bda6, 0x3d8f2d3b, 0xbc447de8,
      0x3d6e7a9a, 0x3c0de408, 0x3c6a67e0, 0xbcb53cac, 0xbcb237ac, 0xbc6d7660,
      0xbd99aa56, 0x3ce2c874, 0x3d9807fb, 0x3ccfee9c, 0xbd55f0f3, 0x3d9c9a4d,
      0xbcae07e6, 0x3d2915ba, 0xbcda6406, 0x3c906df0, 0xbcc83906, 0xbd36f4d6,
      0x3d923683, 0x3d8667cf, 0x3da75c75, 0xbc8bf5a0, 0x3d98efe1, 0x3cdf5530,
      0x3d1c689a, 0x3d6335f6, 0xbd842942, 0xbda390a0, 0x3d52fe0e, 0xbcd93720,
      0x3b76c480, 0xbd769e0e, 0xbd5e580b, 0xbc9c6a80, 0x3d52ce0e, 0x3da14773,
      0x3cb84a70, 0xbd2e62a3, 0xbc04cf60, 0xbd89c570, 0x3bc6d770, 0xbd031beb,
      0xbd2739c3, 0x3da4af81, 0xbc8278cc, 0xbb758b60, 0x3c92de7c, 0xbd3cfcd0,
      0xbcd8cdc6, 0x3d514702, 0x3d77b8a6, 0x3d007242, 0xbd089b6b, 0x3cdfe96c,
      0x3d019e30, 0xbc45f238, 0xba8bdc40, 0x3d15f902, 0x3c68bd00, 0x3ce60520,
      0xbd35ca38, 0x3c5ca7c0, 0x3d6f2aea, 0x3d908d8d, 0x3d108622, 0x3d808301,
      0xbccbab96, 0xbd9517e8, 0xbd1a1a9e, 0xbd8656ab, 0xbc53a0e0, 0x3d9c6e5d,
      0xbcc3dbac, 0xbd946d5c, 0x3d7fea56, 0xbc807d7c, 0x3cf08e5c, 0x3c29b0e8,
      0x3d90cc99, 0x3c8a5c84, 0xbd676793, 0x3c5a74e8, 0x3bd0f9d0, 0x3cb3ad1c,
      0x3d918bc7, 0xbd13a94e, 0x3d8deccd, 0x3cbc253c, 0x3d98008b, 0xbd183573,
      0x3cfa6ed0, 0xbd0c2a7b, 0xbcf67e00, 0xbd20b236, 0xbd0a6de3, 0xbd9db21e,
      0x3d88543b, 0xbcdb5996, 0xbd51acd8, 0x3d64ed4e, 0x3c90d3d4, 0x3d287db2,
      0x3c4da880, 0xbbbddfc0, 0xbd4532e3, 0x3d9fc6c5, 0x3cca5754, 0x3cd4b7d4,
      0x3c59de38, 0xbcebf0fc, 0x3d15333a, 0x3cf11144, 0xbbc5d4d0, 0x3ba666d0,
      0x3d878075, 0x3d7cfa7a, 0xbd2e5538, 0x3d9d7023, 0x3c15b2a8, 0xbd9ea2e4,
      0x3b2f2d20, 0x3d96e0c7, 0x3c12a888, 0xbbf2f780, 0x3a1e5180, 0xbd82f792,
      0xbd5499b8, 0x3d0988e6, 0x3bbd3c40, 0x3ca2d0f4, 0xbd214ade, 0xba48f600,
      0x3d1398b2, 0x3d088956, 0xbd396bc8, 0x3d0a9f36, 0x3d7298e6, 0x3c7ee2c0,
      0xbd8b436e, 0x3d2533b0, 0x3d00c956, 0x3cc59610, 0x3d90e525, 0x3da28ad9,
      0xbd9a6c17, 0xbb412ce0, 0x3d212ece, 0xbcab2ff6, 0x3d2d7bc6, 0x3c140260,
      0xbcf299ec, 0xbd9c39df, 0xbcc3b556, 0xbca145c0, 0x3c91e144, 0x3d68b302,
      0xbb8eba80, 0x3d9cbb9f, 0x3da20deb, 0xbd4f8996, 0x3d8817e5, 0xbd451a30,
      0xbc2489b8, 0x3d615c9e, 0xbd1eeff0, 0xbd997632, 0x3d6bd0ca, 0x3da5d6ad,
      0xbbbc2c40, 0x3d8f81dd, 0xbc95bc60, 0x3c8441fc, 0x3d1f521e, 0xbd3a9aae,
      0xbd64f95b, 0xbd3c07ae, 0xbd8ba57b, 0x3c5915e8, 0x3d0441f6, 0xbc5d2880,
      0xbd46130b, 0x3b99d490, 0xbd535633, 0xbce4b076, 0x3da31483, 0xbd100a10,
      0x3d529016, 0x3d776636, 0x3c7266a8, 0xbd447ce6, 0x3c39d588, 0x3d8a88c1,
      0xbd6102e0, 0xbd4c0f96, 0xbd96a177, 0x3d63fb9a, 0x3da48ae1, 0xbc077de0,
      0x3d0d0a12, 0xbd4e8c08, 0xbd8836e0, 0x3c4003f8, 0x3b867570, 0x3d249868,
      0x3d7159fa, 0x3cd562cc, 0xbd770406, 0x3d311d2a, 0x3d80deab, 0xbc89604c,
      0xbd03fa18, 0xbc3ed040, 0xbd879210, 0xbd853ca2, 0xbc3a2820, 0x3ca8a814,
      0xbd707186, 0xbd4ebae0, 0xbd285338, 0x3d44252a, 0x3d43a6ea, 0x3cc93b7c,
      0x3d31e60a, 0xbd83f6fc, 0xbcdc9816, 0x3b56cd60, 0x3b9d9870, 0xbd498360,
      0x3c890e2c, 0x3da91cb9, 0x3d3b95aa, 0x3d3a85da, 0xbd99b3e4, 0x3be1f540,
      0xbd9dae9b, 0x3c377bb8, 0x3d8ad909, 0x3d7ea3ba, 0x39b0fb00, 0xbc559880,
      0x3d12fd0e, 0x3caeb8b0, 0x3d30ec4a, 0x3cc17f34, 0x3da66999, 0x3c9d8bd4,
      0xbc49a868, 0xbc7262c8, 0xbd575d66, 0x3d53849a, 0x3d20d3aa, 0xbd3eaa60,
      0xbd3ae560, 0x3d8c6e43, 0x3d5f4c2a, 0x3da44397, 0xbd4126fb, 0x3d1a521e,
      0xbcc23d56, 0x3d3aa852, 0x3ce978bc, 0xbd3873de, 0xbc7f1120, 0xbd9cbb92,
      0xbd2e5fc3, 0x3d44b92e, 0xbd1cecc6, 0x3c1dae28, 0x3da088e3, 0xbd572ea8,
      0x3da278a1, 0xbd427378, 0xbafe5a00, 0xbd1732bb, 0xbda94f20, 0x3c960d6c,
      0x3d5e8046, 0xbc9b8540, 0xbcd4d6bc, 0x39561200, 0xbc11d0e0, 0x3d41fe0e,
      0x3d9b803d, 0xbca97a70, 0xbb9b3cb0, 0xbd294f7b, 0xbd628373, 0x3d289706,
      0x3cdbf240, 0xbd41cdce, 0x3cecef54, 0x3d90fa4d, 0xbc888134, 0xbc5a2918,
      0xbce128e6, 0xbcb35750, 0x3d63d15a, 0x3d930435, 0xbba63e90, 0x3da11179,
      0x3bae9d40, 0xbba646f0, 0xbd4a1e16, 0xbd5c64bb, 0xbd49d440, 0x3c643d48,
      0x3b8e4b70, 0x3da8028d, 0xbd2bd5cb, 0xbd8d052a, 0xbd49def0, 0xbd673928,
      0x3d25a522, 0xbd975f9f, 0x3ce6f274, 0xbd2083b3, 0xbd9ef9ef, 0x3da66731,
      0xbd9d9aa8, 0x3ce37d8c, 0x3c248e08, 0xbccab67c, 0xbd0598d8, 0xbd03547e,
      0x3cf5786c, 0x3d4a7152, 0xbd93b99f, 0xbbaa4240, 0xbd92ad06, 0xbda84d06,
      0xbd7fab28, 0x3da9bac9, 0x3be5d250, 0x3b5ff2e0, 0xbcc6309c, 0xbd87240b,
      0x3d5bfcfe, 0xbc884a9c, 0x3c4008c8, 0x3d264cae, 0x3d91e3b3, 0x3d8f85b9,
      0xbd8c734b, 0xbd79c950, 0x3d3b9e9a, 0x3d1eb410, 0x3cf60a2c, 0xbd73f176,
      0xbd6e1b78, 0x3d921101, 0x3a9b9ec0, 0x3ce062bc, 0x3a38ee00, 0x3da89639,
      0xbda2fe80, 0xbd66eb48, 0x3cbeca5c, 0xbd805854, 0xbd961532, 0xbd771aa3,
      0xbd80bab3, 0x3d820b2d, 0x3d893ed5, 0x3d6cc3f2, 0xbd579be0, 0x3d4aff3a,
      0xbdaa21ab, 0xbd3f0063, 0x3d309d5a, 0xbc76d488, 0xbc947420, 0xb9d77d00,
      0xbcf10e30, 0xbd95628c, 0xbd0a576b, 0xbbc7eb90, 0x3d9e278d, 0xbd93adc3,
      0x3ce18a20, 0x3cae06f0, 0xbbef3980, 0x3c951ac4, 0x3d3e7aea, 0xbc8b84cc,
      0x3d1d235e, 0xbb533c80, 0xbcf53716, 0xbb5518e0, 0x3d09b736, 0x3c4245a0,
      0x3bac8070, 0xbd3c4a9b, 0xbd17e246, 0xbd5e6703, 0xbd8cf436, 0xbd26ed80,
      0x3d86dd01, 0xbd3953de, 0x3d9f7ecd, 0x3da99971, 0x3d8142d9, 0x3d77f962,
      0x3d8e694d, 0x3c8f9720, 0x3d29f5f0, 0x3da2b95d, 0x3d441176, 0x3d954c0d,
      0xbd8d7ca2, 0x3d691db2, 0xbcfa8bb6, 0x3d82379f, 0x3da5ea69, 0x3d4b5ffe,
      0xbd80ba9f, 0xbd6b7473, 0x3cf3ae6c, 0x3d48c2de, 0xbd309196, 0x3cc590e4,
      0xbbfbd930, 0xbc531940, 0xbce4b526, 0xbd3e999e, 0xbd30845e, 0x3d1114ca,
      0xbc801fc4, 0x3d963419, 0x3d9e1e59, 0xbd14bd60, 0x3d8d42f1, 0x3d4d7052,
      0xbd252866, 0x3d9e72d7, 0xbd13bfa0, 0x3d5877ce, 0x3d2382b8, 0xbd2090c3,
      0xbd89eac2, 0xbd8b6e5b, 0x3c407860, 0x3ba1f480, 0xbbc0e000, 0xbd23f85b,
      0xbd56f7c6, 0x3da21681, 0xbd89f284, 0x3c250938, 0xbc8da464, 0x3d0af776,
      0xbda13d8a, 0x3daa67cb, 0x3abe13c0, 0x3d7437f6, 0xbd91eb66, 0x3d3c8f82,
      0x3c07c488, 0x3d48e0ca, 0xbcd5b28c, 0xbd940127, 0x3c6ad8c8, 0xbcbb58a6,
      0xbd891232, 0xbcb343b0, 0x3c9e3bfc, 0x3bc78cf0, 0x3d9a7bd1, 0x3c332128,
      0x3d468eea, 0x3c9bdc54, 0xbca528ac, 0x3d85f30b, 0xbd9feb7e, 0x3d6ef656,
      0x3c987604, 0xbd0e4006, 0x3c0fe698, 0x3d98f085, 0x3c88284c, 0x3d7a6ac2,
      0xbd232328, 0xbcf65dd6, 0x3d8b1c87, 0x3c8579b0, 0xbd6c9a18, 0xbb8c4070,
      0xbce53460, 0xb966da00, 0x3d96bf7f, 0xbd8e6da4, 0xbce5f490, 0x3d52ab1a,
      0xbda75136, 0x3d4eb822, 0xbcd06aec, 0xbb83bd70, 0x3d0cd1ee, 0x3b944bc0,
      0xbd85cd7e, 0x3d852373, 0xbd83b8ee, 0x3ba8cdb0, 0x3cd6a290, 0xbd38d6cb,
      0xbd3d1808, 0xbc85171c, 0x3da3c7e5, 0xbd739646, 0x3c8ae160, 0x3d21d400,
      0x3ccb75c0, 0xbc889df0, 0xbc72fc28, 0xbc17e118, 0xbd7a7630, 0xbc61a280,
      0x3d0696ae, 0xbd8153da, 0x3d0f75ae, 0x3c39d300, 0xbd4ec36b, 0x3d87b79f,
      0x3d303cca, 0xbd040c7e, 0x3c61e480, 0xbd467ac3, 0x3c94ba84, 0xbd458a26,
      0x39c9c400, 0x3da4a66d, 0xbcbe4bb0, 0x3d0eb460, 0x3c9431b4, 0xbd8442ea,
      0xbd8dfeff, 0xbc699a60, 0xbc989a74, 0xbd2eeb46, 0x3caecdb4, 0xbda8bd30,
      0x3d820281, 0x3cb36bec, 0x3cea7350, 0x3d5f5bf6, 0x3d30c10a, 0x3d1d6ace,
      0x3d41da46, 0x3c8bb4b0, 0xbc60f2b8, 0x3d9cdd91, 0x3d91abb9, 0xbd1b2748,
      0xbc59dff8, 0xbc775560, 0x3d493bb6, 0xbd63f4ce, 0x3d5ff1ae, 0x3d979a9b,
      0x3c82ebfc, 0x3d746e3a, 0xbd2269a3, 0x3d8ee721, 0xbd82397e, 0xbd9c315f,
      0xbd39c52b, 0xba1c2e80, 0x3cdbe304, 0x3d1f1e42, 0x39e59000, 0x3d613952,
      0xbda8895f, 0x3c5c46c8, 0x3cae243c, 0xbc63e420, 0x3c691308, 0x3d9d74d5,
      0xba9f3440, 0x3c329100, 0xbcaeb7ec, 0xbd43f623, 0xbd1088e8, 0x3d9526a1,
      0xbd1756e8, 0xbabed200, 0xbce877e0, 0x3d6c38c6, 0xbd4f0708, 0x3d855191,
      0xbd1f726e, 0xbd3c8a5b, 0x3bf92500, 0xbd9791e2, 0x3da2554d, 0x3da04af7,
      0xbbf66f80, 0xbd4b6d36, 0x3d6f47ea, 0xbc42df68, 0x3cb09534, 0xbba09dc0,
      0xbd86382b, 0xbd70b530, 0x3d35f66e, 0x3aeb94c0, 0xbc878d1c, 0xbc168020,
      0x3cb30270, 0x3cc2b0f0, 0xbc180f08, 0xbda1d4ee, 0xbd4ccd98, 0xbd894473,
      0xbd9af53e, 0xbb0c2d60, 0x3d17f51e, 0xbd7b7eee, 0xbc5da258, 0xbd4dd2a3,
      0x3da186fd, 0x3d134758, 0x3d193cc2, 0xbd115248, 0xbd8bf6a3, 0x3d031b48,
      0xbd69ef46, 0xbc223768, 0xbcca49fc, 0x3c20e208, 0xbc83cb20, 0x3c1916b8,
      0xbd8399a2, 0xbd8255ca, 0xbda299d2, 0x3d914165, 0xbcc6bf0c, 0x3a114400,
      0x3cd1da4c, 0x3d4228c2, 0x3d7670c6, 0xbd1e2430, 0x3d629c22, 0xbd4473c0,
      0x3d7feb8a, 0x3d9cdef1, 0x3cca8d20, 0x3d56a59a, 0xbd00ebd6, 0x3ccd2300,
      0x3c0c9fa8, 0x3d5b419a, 0xbc144938, 0xbcc62d7c, 0xbda34f16, 0xbd173876,
      0xbd8a6fd3, 0x3cbecc6c, 0x3cee48a4, 0x3c92f5bc, 0x3d8ea19d, 0x3d4007c2,
      0xbd5ce9b0, 0xbd44c9a3, 0xbbdc8c40, 0x3d940e0f, 0x3d0eed0a, 0x3d055578,
      0x3ba45490, 0x3c31d480, 0x3d755952, 0xbbe273d0, 0xbb1f86e0, 0xbd2a17e6,
      0xbd3b3688, 0xbd85c086, 0x3cae4764, 0x3da0ae0d, 0x3d8840e1, 0x3d583022,
      0xbd31600b, 0x3ab47ec0, 0x3d7743b2, 0xbcebc4c0, 0xbcaca0a6, 0x3d1f8a70,
      0x3d8fa7f7, 0x3da7431d, 0x3d83c1fd, 0x3d2a48ee, 0xbd0b1903, 0x3c50cd78,
      0x3bc170f0, 0xbd6136a3, 0xbd80455c, 0xbc50a718, 0x3c9e07a4, 0x3c6be758,
      0xbc4dd160, 0xbd8690a4, 0x3d694d86, 0x3aff6500, 0x3d50db16, 0x3cd51a0c,
      0x3d0f99b0, 0xbd1f047b, 0x3d6936aa, 0x3d8abc69, 0x3d0001c8, 0x3d4db9ae,
      0xbd20db0e, 0xbd2bbe13, 0x3d7f21f2, 0x3cf3f014, 0xbbf91340, 0x3d39be8a,
      0xbce444ec, 0x3da7e361, 0xbd123e96, 0xbd253e5e, 0xbc4a61f8, 0x3d7d283a,
      0xbcb0f940, 0xbd3055bb, 0xbda15e84, 0xbd8c7f2f, 0x3daa2d09, 0x3b6c8920,
      0xbd990704, 0xbd6569b8, 0x3c041668, 0xbd8e3924, 0xbd94a413, 0xbd420e3e,
      0xbd396483, 0xbd7490ab, 0x3da40ca1, 0x3d48e22a, 0x3aa71100, 0xbd02a233,
      0xbcf71146, 0x3d8b2c4d, 0xbd1a098e, 0x3aa1dcc0, 0x3ca652d4, 0x3bc7ce30,
      0xbd1bbdb6, 0xbda6f36e, 0x3d09e0f0, 0x3c87be40, 0xbb0ed7e0, 0x3d49bbf2,
      0xbd4c58f8, 0x3da8bae1, 0xbda70c08, 0x3c911f2c, 0xbd927990, 0xbceb18dc,
      0xbd17b05b, 0x3b1281e0, 0xbca1db74, 0xbd9849da, 0x3c5786f8, 0xbc909ac4,
      0x3d1c371a, 0x3b9e3a50, 0x3d3c374e, 0x3d883bd5, 0x3d056576, 0xbd86a74b,
      0x3d8bc7e9, 0x3d620cb2, 0x3d8c0993, 0xbd88e19e, 0xbd5979e0, 0xbd898838,
      0x3d6ee24a, 0xbd14ad70, 0xbd38acf8, 0xbbfe6bd0, 0xbcf2441c, 0x3c12b5f8,
      0x3d7d8002, 0x3d9c2885, 0x3d7dabba, 0x3da9cb05, 0xbd21b2ab, 0x3cb0d1f4,
  };

  // 7
  uint32_t bias_vals[] = {
      0xbda771f8, 0x3da2ff01, 0xbbafffb0, 0x3d986e05,
      0x3b05b520, 0xbd0b22e0, 0x3d55a4aa,
  };

  // 4,4,2,7
  uint32_t output_exp_vals[] = {
      0x3aae9b04, 0x3d11f138, 0x3ea6c5e5, 0x3e27faa2, 0x3e1af7ae, 0xbcd55ed5,
      0x3d64578e, 0xbe34ca8f, 0xbea1a8b4, 0x3f224d3d, 0x3d8b4f5c, 0x3d5da826,
      0x3d5ba7f0, 0x3e837ddb, 0xbc3cbed6, 0xbedabd2f, 0x3e0e76a3, 0x3e5e1078,
      0x3eb6b112, 0xbbb0ae9e, 0x3ea6b36e, 0x3e7f83d2, 0xbe01a079, 0x3e566d4b,
      0x3ecdcb4c, 0x3e278bed, 0xbe1ca4ea, 0x3e033ef1, 0x3e36026f, 0xbe814275,
      0x3eda68e1, 0x3ece2c47, 0x3e99d145, 0x3de7c10a, 0x3e003eb8, 0x3ddc462f,
      0xbf128f10, 0x3ef8c6c4, 0x3e0b4fe8, 0x3e234e9f, 0xbc68b0cc, 0x3ec20083,
      0x3e2b8213, 0xbe83f3f4, 0x3f0cedbc, 0x3ecffe92, 0x3eab0eeb, 0x3a5ee371,
      0x3eaf1fcd, 0xbdfdd7b5, 0xbe333d06, 0x3e4d244d, 0x3d9c8735, 0x3da1fd6f,
      0x3dbdf320, 0x3edff713, 0x3da12ca6, 0xbe418c28, 0x3ea46203, 0x3ed77016,
      0x3d4e8755, 0x3e85bec4, 0x3ec9875d, 0x3e7c02c8, 0xbed5c916, 0x3edba911,
      0x3e9fc87d, 0x3e3d0777, 0x3d3e1d2d, 0x3dd82c20, 0x3e96fc25, 0xbf131fb2,
      0x3f6dc404, 0x3e083262, 0x3e8bcd87, 0x3dfb8de2, 0x3db91ad3, 0x3ebcac31,
      0xbe9b16af, 0x3ec8e755, 0x3f036f70, 0x3e78d3e5, 0x3cc1b424, 0x3e73c23c,
      0x3d8a27e0, 0xbea947af, 0x3f006376, 0x3e3b6c9c, 0x3e145c90, 0x3b90ce9d,
      0x3e8ef2c4, 0xbca2e459, 0xbe94037d, 0x3ee2e324, 0x3e83fadd, 0x3cdd4517,
      0x3e2722a4, 0x3e85ec31, 0x3dde5094, 0xbef14213, 0x3f1f8519, 0x3ea5cc4d,
      0x3e96bce1, 0xbbc4f4a2, 0x3e86b220, 0x3d91f714, 0xbf0b7eea, 0x3ede3416,
      0x3ebd898b, 0xbdc5e339, 0x3d645a8c, 0x3ecd54ea, 0xbe7270b7, 0xbe2a28d0,
      0x3efe2f4f, 0x3ecdd1bb, 0x3eb541ae, 0x3df78530, 0x3e2ff48b, 0x3ecf015e,
      0xbdfc62ab, 0x3f306492, 0x3e32bcf1, 0x3d46d562, 0x3e26ff13, 0x3ebb5471,
      0x3e0fdbb5, 0xbe8821f2, 0x3e9b86b6, 0x3dab64b8, 0x3e64597d, 0xbdbbcdc7,
      0x3e3f9585, 0x3de4346a, 0x3dcbc01b, 0x3e5b0824, 0x3e80aced, 0x3e3d057f,
      0xbdb1dc06, 0x3ed614bf, 0xbe4b9925, 0xbea5e903, 0x3efbac65, 0x3f0429b1,
      0x3e397d7b, 0x3ef9017d, 0x3f09e093, 0x3cee88fd, 0xbf13fa56, 0x3ea66d86,
      0x3ec4d5e4, 0x3deb3387, 0xbe3fe165, 0xbe29dbc3, 0x3e006749, 0xbe356e13,
      0x3f4c4463, 0xbd05aef1, 0x3e5cacf5, 0x3bb3f58a, 0x3f0276b4, 0x3e19ffaf,
      0xbefbb77f, 0x3f02c7a0, 0x3ed7c512, 0x3da438e7, 0x3e27f543, 0x3e0d4b61,
      0x3e462486, 0xbeece080, 0x3f2434e2, 0x3d412397, 0x3d27f66b, 0xbe8014f3,
      0x3e6c1353, 0xbe6eaff0, 0xbeba0a82, 0x3ed479f9, 0x3ea08d22, 0x3e3a5d62,
      0xbe168c35, 0xbb8818a5, 0x3eb64d64, 0xbe94a7f4, 0x3ef4553a, 0x3e1b35e2,
      0xbb866309, 0x3c700992, 0x3ee4e2bc, 0xbe081632, 0xbe209e82, 0x3ef13307,
      0x3f0aaf13, 0x3d33f762, 0x3db0d374, 0x3dac7411, 0x3de43756, 0xbea7e814,
      0x3e9bfcf1, 0x3e985384, 0x3e92c2f7, 0xbe351877, 0x3e0cf9db, 0xbea20a24,
      0xbf169121, 0x3f10234e, 0x3e807156, 0x3e6978b4, 0xbd83f065, 0x3ecb7fbb,
      0xbce91195, 0xbe653e1a, 0x3eef12cc, 0x3ded14a4, 0x3d0345ca, 0x3dbafae5,
      0x3ebe95f8, 0x3e1207d8, 0xbeea224b, 0x3eea7e97, 0x3f063448, 0x3e843290,
      0x3da4e66c, 0xbdcffdd6,
  };

  // 4,4,2,7
  uint32_t output_relu_exp_vals[] = {
      0x3aae9b04, 0x3d11f138, 0x3ea6c5e5, 0x3e27faa2, 0x3e1af7ae, 0x0,
      0x3d64578e, 0x0,        0x0,        0x3f224d3d, 0x3d8b4f5c, 0x3d5da826,
      0x3d5ba7f0, 0x3e837ddb, 0x0,        0x0,        0x3e0e76a3, 0x3e5e1078,
      0x3eb6b112, 0x0,        0x3ea6b36e, 0x3e7f83d2, 0x0,        0x3e566d4b,
      0x3ecdcb4c, 0x3e278bed, 0x0,        0x3e033ef1, 0x3e36026f, 0x0,
      0x3eda68e1, 0x3ece2c47, 0x3e99d145, 0x3de7c10a, 0x3e003eb8, 0x3ddc462f,
      0x0,        0x3ef8c6c4, 0x3e0b4fe8, 0x3e234e9f, 0x0,        0x3ec20083,
      0x3e2b8213, 0x0,        0x3f0cedbc, 0x3ecffe92, 0x3eab0eeb, 0x3a5ee371,
      0x3eaf1fcd, 0x0,        0x0,        0x3e4d244d, 0x3d9c8735, 0x3da1fd6f,
      0x3dbdf320, 0x3edff713, 0x3da12ca6, 0x0,        0x3ea46203, 0x3ed77016,
      0x3d4e8755, 0x3e85bec4, 0x3ec9875d, 0x3e7c02c8, 0x0,        0x3edba911,
      0x3e9fc87d, 0x3e3d0777, 0x3d3e1d2d, 0x3dd82c20, 0x3e96fc25, 0x0,
      0x3f6dc404, 0x3e083262, 0x3e8bcd87, 0x3dfb8de2, 0x3db91ad3, 0x3ebcac31,
      0x0,        0x3ec8e755, 0x3f036f70, 0x3e78d3e5, 0x3cc1b424, 0x3e73c23c,
      0x3d8a27e0, 0x0,        0x3f006376, 0x3e3b6c9c, 0x3e145c90, 0x3b90ce9d,
      0x3e8ef2c4, 0x0,        0x0,        0x3ee2e324, 0x3e83fadd, 0x3cdd4517,
      0x3e2722a4, 0x3e85ec31, 0x3dde5094, 0x0,        0x3f1f8519, 0x3ea5cc4d,
      0x3e96bce1, 0x0,        0x3e86b220, 0x3d91f714, 0x0,        0x3ede3416,
      0x3ebd898b, 0x0,        0x3d645a8c, 0x3ecd54ea, 0x0,        0x0,
      0x3efe2f4f, 0x3ecdd1bb, 0x3eb541ae, 0x3df78530, 0x3e2ff48b, 0x3ecf015e,
      0x0,        0x3f306492, 0x3e32bcf1, 0x3d46d562, 0x3e26ff13, 0x3ebb5471,
      0x3e0fdbb5, 0x0,        0x3e9b86b6, 0x3dab64b8, 0x3e64597d, 0x0,
      0x3e3f9585, 0x3de4346a, 0x3dcbc01b, 0x3e5b0824, 0x3e80aced, 0x3e3d057f,
      0x0,        0x3ed614bf, 0x0,        0x0,        0x3efbac65, 0x3f0429b1,
      0x3e397d7b, 0x3ef9017d, 0x3f09e093, 0x3cee88fd, 0x0,        0x3ea66d86,
      0x3ec4d5e4, 0x3deb3387, 0x0,        0x0,        0x3e006749, 0x0,
      0x3f4c4463, 0x0,        0x3e5cacf5, 0x3bb3f58a, 0x3f0276b4, 0x3e19ffaf,
      0x0,        0x3f02c7a0, 0x3ed7c512, 0x3da438e7, 0x3e27f543, 0x3e0d4b61,
      0x3e462486, 0x0,        0x3f2434e2, 0x3d412397, 0x3d27f66b, 0x0,
      0x3e6c1353, 0x0,        0x0,        0x3ed479f9, 0x3ea08d22, 0x3e3a5d62,
      0x0,        0x0,        0x3eb64d64, 0x0,        0x3ef4553a, 0x3e1b35e2,
      0x0,        0x3c700992, 0x3ee4e2bc, 0x0,        0x0,        0x3ef13307,
      0x3f0aaf13, 0x3d33f762, 0x3db0d374, 0x3dac7411, 0x3de43756, 0x0,
      0x3e9bfcf1, 0x3e985384, 0x3e92c2f7, 0x0,        0x3e0cf9db, 0x0,
      0x0,        0x3f10234e, 0x3e807156, 0x3e6978b4, 0x0,        0x3ecb7fbb,
      0x0,        0x0,        0x3eef12cc, 0x3ded14a4, 0x3d0345ca, 0x3dbafae5,
      0x3ebe95f8, 0x3e1207d8, 0x0,        0x3eea7e97, 0x3f063448, 0x3e843290,
      0x3da4e66c, 0x0,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, VALID_PADDING, NULL);
}

void test_valid_padding_zero_strides_large() {
  input_set *set = &large_input;
  strides_input_set *strides = &zero_strides;

  // 4,15,10,6
  uint32_t input_vals[] = {
      0x3dde9a10, 0x3f5666b9, 0x3e8a80fa, 0x3e0e30e4, 0x3ebd0716, 0x3f22510c,
      0x3f0f0b05, 0x3f0b527d, 0x3f46e8f1, 0x3f0e2236, 0x3f3843d9, 0x3f11a3bb,
      0x3e53223c, 0x3e9ebd48, 0x3f7f5de3, 0x3ed7b118, 0x3d040570, 0x3e91fc24,
      0x3f65f617, 0x3e86c634, 0x3d95c918, 0x3f7380cd, 0x3f234774, 0x3edabe94,
      0x3e7135f4, 0x3e350480, 0x3e5e08cc, 0x3f2f8802, 0x3f6d3d5b, 0x3ed9ee36,
      0x3eccc264, 0x3ea07fe0, 0x3f7c6112, 0x3f2b105b, 0x3ebd523c, 0x3f315182,
      0x3f39ff9c, 0x3e83e828, 0x3f62ed12, 0x3f5dfc1c, 0x3ef1d4fe, 0x3b973980,
      0x3f25010e, 0x3df4f550, 0x3f216f1a, 0x3e72ac50, 0x3f3f925f, 0x3ef34a1c,
      0x3f1514ea, 0x3f1912bd, 0x3f7518ff, 0x3f560cc0, 0x3ee8d69a, 0x3f28636f,
      0x3ef8336a, 0x3e366224, 0x3efab474, 0x3f00be28, 0x3efabf90, 0x3f268971,
      0x3ecc927e, 0x3efa2e24, 0x3f1e3dea, 0x3d25ae80, 0x3f28c692, 0x3e699f80,
      0x3e892528, 0x3f5c0fa4, 0x3f3d0b21, 0x3f504cea, 0x3eec2d20, 0x3ea48b0c,
      0x3f7c7068, 0x3e2b6d44, 0x3f7788a7, 0x3e275054, 0x3efed888, 0x3f4bfe3c,
      0x3f6120fe, 0x3f504c76, 0x3dff1a98, 0x3f4e8175, 0x3ef0831c, 0x3f1ddffc,
      0x3e82330a, 0x3dc33bc8, 0x3f394b53, 0x3f634e17, 0x3f47ebf7, 0x3f7f3c84,
      0x3f4cd033, 0x3d9d2098, 0x3f341604, 0x3eed28a2, 0x3f763a8b, 0x3ea42184,
      0x3f29d214, 0x3f7b3dc8, 0x3e94f71c, 0x3eabbd3a, 0x3f0b6fd7, 0x3f46fac2,
      0x3e276790, 0x3e82797c, 0x3e92c996, 0x3f1592e2, 0x3f12d101, 0x3edcf45a,
      0x3e86d9bc, 0x3f6d4119, 0x3f30d665, 0x3f5fad7a, 0x3e13afb0, 0x3f144cd9,
      0x3efede78, 0x3f72d999, 0x3e4f1154, 0x3f40f5ea, 0x3f474e3b, 0x3efa4892,
      0x3e5460cc, 0x3f23568b, 0x3f450c05, 0x3f61a5aa, 0x3f4859d2, 0x3cd13f40,
      0x3e3f6d04, 0x3e805646, 0x3f53dfe9, 0x3ef89136, 0x3e0add1c, 0x3f33e7df,
      0x3efae34a, 0x3f2113b1, 0x3f3ed68e, 0x3dfa3530, 0x3f4b139a, 0x3f233a7e,
      0x3d8516b0, 0x3f1aa364, 0x3e66ff18, 0x3f38dcf6, 0x3f231575, 0x3d83d8c0,
      0x3e2fe2e0, 0x3f1aa7d5, 0x3f78784e, 0x3f096b77, 0x3e45bb30, 0x3ef7329e,
      0x3f145b96, 0x3f0ff17d, 0x3f30c586, 0x3f1e8e09, 0x3ed5ce52, 0x3f17f212,
      0x3ee5824a, 0x3f7dc58f, 0x3c189280, 0x3f772b3a, 0x3d60d290, 0x3f67010e,
      0x3f3d57cf, 0x3f11a4b3, 0x3c8f9220, 0x3ec55dfa, 0x3f1152b0, 0x3f7e784b,
      0x3f5b8914, 0x3f3f87da, 0x3f2d606d, 0x3f7465f2, 0x3f4048d2, 0x3ed29954,
      0x3f51c6fa, 0x3ea0a238, 0x3f3b0cd7, 0x3e51a488, 0x3e1e8910, 0x3ed2c5de,
      0x3e8d776e, 0x3eabf5c4, 0x3f6f08c2, 0x3e34abe8, 0x3eecc686, 0x3ec6b340,
      0x3f0ef530, 0x3f6a2f92, 0x3f6312d6, 0x3f53b437, 0x3f64b769, 0x3e071134,
      0x3eaaf75e, 0x3eea6cbc, 0x3f4f7b3c, 0x3f6153a5, 0x3f621982, 0x3f3e978e,
      0x3f1f06ec, 0x3f35445c, 0x3eb2f924, 0x3e9ec55e, 0x3f51c216, 0x3f7d51ea,
      0x3d2a5290, 0x3e8f57d2, 0x3eeea1a0, 0x3f177ad9, 0x3e6167f4, 0x3f6e0812,
      0x3f682c4a, 0x3f18b998, 0x3e3f51dc, 0x3f0eb695, 0x3efa3014, 0x3d83a8f0,
      0x3f5400e4, 0x3de51a28, 0x3f044430, 0x3f28dfee, 0x3e848d58, 0x3d1c74a0,
      0x3ec975c6, 0x3f11e457, 0x3f1b0942, 0x3ed94f5a, 0x3f727868, 0x3e330edc,
      0x3e920154, 0x3f13b95a, 0x3f3a6348, 0x3c5207c0, 0x3f05a886, 0x3ed204f2,
      0x3f6a800e, 0x3eea6228, 0x3f1a60d5, 0x3f2abe0b, 0x3e4471d4, 0x3ebdc5c4,
      0x3f46891e, 0x3f339bc7, 0x3f173a63, 0x3c6fa340, 0x3edc80ca, 0x3cf6d5a0,
      0x3f7c30af, 0x3f44aa42, 0x3f001b8b, 0x3e1f0ad0, 0x3f071aa5, 0x3d830718,
      0x3e519f6c, 0x3f5af810, 0x3ddd5948, 0x3dd93078, 0x3f2b2a8f, 0x3dd55958,
      0x3e186300, 0x3d32ef30, 0x3e906c9e, 0x3f325c14, 0x3ed906a2, 0x3eb646dc,
      0x3d4eb620, 0x3eec02b6, 0x3ec49966, 0x3f37af89, 0x3f15ac66, 0x3f021455,
      0x3ed9680a, 0x3f49fa35, 0x3f223794, 0x3e55bda0, 0x3cecfce0, 0x3f7af654,
      0x3f00a73a, 0x3f55119d, 0x3f04f474, 0x3f729c90, 0x3eb28c82, 0x3ce7c2c0,
      0x3f6852b3, 0x3ddd8638, 0x3e5ff158, 0x3e189898, 0x3f46bbe7, 0x3f4e5dcf,
      0x3e769b38, 0x3effedc6, 0x3e88efca, 0x3f5d77a8, 0x3f348d05, 0x3e978342,
      0x3d546e00, 0x3ef2d14a, 0x3ec0c2b2, 0x3f38002e, 0x3f7d2946, 0x3f7ea4b8,
      0x3f056100, 0x3ed5704c, 0x3f6d2747, 0x3ec7e3f6, 0x3f663e4e, 0x3da77ed8,
      0x3f169043, 0x3f36da8a, 0x3f5562d8, 0x3f654053, 0x3f426c9a, 0x3d6bb610,
      0x3f233c4c, 0x3f359222, 0x3d8ffb60, 0x3f5e3978, 0x3e83b710, 0x3eda2fc0,
      0x3e9adcb2, 0x3d3ad540, 0x3e6a84ac, 0x3f688790, 0x3f737dc8, 0x3f34f35b,
      0x3ddcc4f8, 0x3f147b0e, 0x3f2fa60f, 0x3dc02228, 0x3f57c84d, 0x3dba3300,
      0x3d0b1030, 0x3e6c3878, 0x3ea7c9de, 0x3ea03e24, 0x3ea43e30, 0x3ec423e2,
      0x3f7f26eb, 0x3f420836, 0x3eb02a96, 0x3e159168, 0x3e896ebe, 0x3ddc4ee0,
      0x3ebd628e, 0x3efa24b2, 0x3dbdf0b8, 0x3f6638e7, 0x3e80da60, 0x3f492d7d,
      0x3ecff6b8, 0x3f163ca6, 0x3ee5d554, 0x3f1cae60, 0x3e3afd54, 0x3e86460a,
      0x3f45a3ef, 0x3edc3dc0, 0x3f2c2859, 0x3e7cf8ac, 0x3f334c75, 0x3e842c28,
      0x3e7026d0, 0x3f65fdbb, 0x3f719460, 0x3ede5fd6, 0x3e75b1ac, 0x3d8c6be8,
      0x3d2c88d0, 0x3daf1b08, 0x3eba86ea, 0x3f586ea8, 0x3f712640, 0x3f26be89,
      0x3c52abc0, 0x3f37a1b8, 0x3e9ab7ae, 0x3f778239, 0x3f1cc44e, 0x3f759a65,
      0x3ef2481a, 0x3f7c9113, 0x3f17a8e7, 0x3f010e4a, 0x3f0226dc, 0x3f20b226,
      0x3f2863ba, 0x3e9adea6, 0x3f0a25ee, 0x3d3f6df0, 0x3f284520, 0x3f320f11,
      0x3f3c65be, 0x3f20ed9d, 0x3f0f8492, 0x3e8b1e8c, 0x3f4810fe, 0x3f641106,
      0x3e8cdbce, 0x3e8199ca, 0x3d13ded0, 0x3f59d926, 0x3ed58276, 0x3eeebb88,
      0x3f095c8e, 0x3e845efc, 0x3f63f1ad, 0x3f137a99, 0x3de7ee08, 0x3efeb994,
      0x3f05beda, 0x3e688e04, 0x3efcfb46, 0x3f55867b, 0x3c3e7dc0, 0x3f6d645e,
      0x3f4e03a8, 0x3f3f44fe, 0x3eea0742, 0x3f697c49, 0x3f7706f6, 0x3eaa1804,
      0x3e4bf2f0, 0x3f528e35, 0x3ead3262, 0x3ea84e78, 0x3f77c29f, 0x3d2ab6e0,
      0x3f5e5096, 0x3ede2990, 0x3d20b2c0, 0x3f4a4c64, 0x3ea004aa, 0x3f107192,
      0x3ec62bdc, 0x3ee0dd08, 0x3eab5996, 0x3ee2688e, 0x3f70b6ca, 0x3f367c38,
      0x3f3703c0, 0x3e0e7294, 0x3d27dbd0, 0x3e8e7d26, 0x3f32af3f, 0x3f2a3f2a,
      0x3db1c370, 0x3f4519b3, 0x3f34aa2b, 0x3e1e5c14, 0x3d85acc0, 0x3f5f1e1f,
      0x3ea4b136, 0x3f3a3b66, 0x3e736488, 0x3f18ff06, 0x3ed88d1e, 0x3f6afd1d,
      0x3e64fbc0, 0x3ef46cb2, 0x3d3e21b0, 0x3f1be89b, 0x3d468400, 0x3f003634,
      0x3e842706, 0x3e3e4764, 0x3e7c9e38, 0x3f53fd99, 0x3f378a79, 0x3f4b4832,
      0x3de73ed8, 0x3f4036f6, 0x3f321383, 0x3f7d92f0, 0x3f2d9197, 0x3d9fbdd8,
      0x3f1d5f3f, 0x3e31b094, 0x3e630d20, 0x3ddcca98, 0x3f3ced3a, 0x3f0dbd5c,
      0x3f7d0a1c, 0x3f535d3c, 0x3de89e08, 0x3ed6d14e, 0x3ef5b28e, 0x3f4b3164,
      0x3f606410, 0x3dd17730, 0x3e9b5210, 0x3eb28bf4, 0x3f128966, 0x3f4e7f32,
      0x3e401670, 0x3f2cee74, 0x3f78534f, 0x3f417f95, 0x3e4ca56c, 0x3cfe2aa0,
      0x3ee6706a, 0x3e25c45c, 0x3f46bc4d, 0x3f3e6af8, 0x3dafa298, 0x3f70143c,
      0x3ebbbce6, 0x3f0f0a79, 0x3f1e9e36, 0x3f7415e6, 0x3ea5b550, 0x3f044ea3,
      0x3e902d2a, 0x3f7bbffe, 0x3ebe17f6, 0x3f58254c, 0x3f2eb2a9, 0x3f0d0d50,
      0x3e8fe4b4, 0x3f59fd71, 0x3e978a5a, 0x3f56f198, 0x3ed0adf6, 0x3f078ee9,
      0x3f220c69, 0x3e0186dc, 0x3f1a3fc2, 0x3df6e6f8, 0x3ecb2ba4, 0x3f01f111,
      0x3e914772, 0x3ec2fa9a, 0x3f5c0c34, 0x3e2ce3c4, 0x3e688ce0, 0x3e4c5c6c,
      0x3e197710, 0x3f153d72, 0x3de9fe40, 0x3dcb82d8, 0x3e802ffa, 0x3f6f96d3,
      0x3f111cab, 0x3eaa9140, 0x3f639e7d, 0x3e81c8f4, 0x3e73f658, 0x3f28144d,
      0x3f79a8bf, 0x3ed0dc28, 0x3c997ee0, 0x3e81f87c, 0x3ecb3056, 0x3e185a84,
      0x3ef76e98, 0x3f60b77c, 0x3d44ae80, 0x3f596e31, 0x3f7791bc, 0x3def36c8,
      0x3f7ba9b4, 0x3e00e470, 0x3cbd5740, 0x3e83c666, 0x3ebe7e1c, 0x3f47be8d,
      0x3e38b0f8, 0x3ddd5388, 0x3f296cd3, 0x3f6dcc4d, 0x3ede214a, 0x3f64cb7c,
      0x3f6b83d7, 0x3db74d28, 0x3f05418c, 0x3f6c030d, 0x3e72fb40, 0x3f170005,
      0x3efa00aa, 0x3efb578c, 0x3f1fea78, 0x3f4fce40, 0x3e488180, 0x3f700aa9,
      0x3f5b50a2, 0x3ed435e6, 0x3e086648, 0x3f4b174c, 0x3c2939c0, 0x3eeae46a,
      0x3f114a86, 0x3f214240, 0x3e4ac3f8, 0x3f5ae693, 0x3e39caa0, 0x3e9b7cb0,
      0x3f207954, 0x3e679794, 0x3f35f930, 0x3f14fab7, 0x3eb63b42, 0x3f3ae46c,
      0x3da38db0, 0x3e777d44, 0x3f5b4f5b, 0x3e2bd12c, 0x3ec4c640, 0x3efe0a04,
      0x3ed14470, 0x3e8b93fc, 0x3da4eed8, 0x3f5fa53c, 0x3e1f86ec, 0x3f0f9ac9,
      0x3efcee4a, 0x3f313a50, 0x3f071c15, 0x3f5d44c9, 0x3eec2ebe, 0x3f0a13cf,
      0x3ed6d21e, 0x3f7b9ed5, 0x3eef0120, 0x3f571c5b, 0x3f6befec, 0x3f607b3f,
      0x3f537d7e, 0x3d8b4a48, 0x3ec1ad4c, 0x3e1a8fa0, 0x3de922f8, 0x3f3bcb03,
      0x3ea04e1c, 0x3f466874, 0x3f2c44c4, 0x3edfbbe4, 0x3f4fe4dd, 0x3f43503d,
      0x3e99177a, 0x3f547e6b, 0x3df35af8, 0x3f52ffa7, 0x3dec08e0, 0x3f0e64c0,
      0x3e0c8588, 0x3e171508, 0x3f3edf17, 0x3ef7e06e, 0x3f1732af, 0x3eb4c858,
      0x3f2a4919, 0x3e552a04, 0x3e9c4d5a, 0x3eede6c6, 0x3f013ec9, 0x3e1b7d40,
      0x3f69938c, 0x3f636881, 0x3f5664e3, 0x3e6e669c, 0x3dc7b5f0, 0x3edf5cc4,
      0x3f16adc4, 0x3e9c66ce, 0x3ec0871e, 0x3f40173a, 0x3e79c3cc, 0x3d6829e0,
      0x3e45a1a4, 0x3f44d2d6, 0x3f59ba11, 0x3f3873af, 0x3e866832, 0x3e5f5550,
      0x3f3095ba, 0x3f0a1994, 0x3e2a01b0, 0x3ec81fee, 0x3df462b8, 0x3ec164cc,
      0x3f111a8d, 0x3d5dc7e0, 0x3d3c9cb0, 0x3f5e78ac, 0x3f0956a1, 0x3f5f80b1,
      0x3f512aba, 0x3f44f54a, 0x3f7dd77c, 0x3e52c9ec, 0x3dd14f80, 0x3b5cda00,
      0x3effc556, 0x3f0df3ec, 0x3ea90e42, 0x3f42940f, 0x3f7734a2, 0x3f4b1e5e,
      0x3f776aa0, 0x3e485e24, 0x3f011595, 0x3ef540ae, 0x3f0748a5, 0x3e46d958,
      0x3f6ec052, 0x3f166e84, 0x3f5db818, 0x3e988ebe, 0x3df37ec8, 0x3d138ce0,
      0x3e39f538, 0x3e9f145a, 0x3f72514e, 0x3f13ceab, 0x3e99a322, 0x3e262edc,
      0x3f617b26, 0x3e5d84dc, 0x3f1addd1, 0x3d824e90, 0x3f6fc8f7, 0x3e26588c,
      0x3ea986fa, 0x3eb074ae, 0x3d090e40, 0x3f688236, 0x3ef57cd8, 0x3f7bfe40,
      0x3c3dcf80, 0x3dba81f8, 0x3e42aa74, 0x3ec9e218, 0x3f3acc72, 0x3f5d4b80,
      0x3f7789e4, 0x3d112700, 0x3f4fab9f, 0x3e7883c0, 0x3f57ece7, 0x3f799bcd,
      0x3e6c8aa0, 0x3efaa290, 0x3f2899ed, 0x3f0c8eff, 0x3e7a7e68, 0x3e6b8fd8,
      0x3f145bda, 0x3e320ddc, 0x3f45866a, 0x3f49a803, 0x3f6dd94c, 0x3ca89800,
      0x3f75a2de, 0x3f2b8b90, 0x3e9fe78a, 0x3f5f2f68, 0x3e718f34, 0x3ec2cddc,
      0x3e84f64e, 0x3f569922, 0x3d141e50, 0x3f54a651, 0x3daa4730, 0x3f65103c,
      0x3f5c03bb, 0x3ee2a65a, 0x3ed6f704, 0x3ed0de16, 0x3f4d955c, 0x3f420b48,
      0x3f1e7c00, 0x3d9e5cf0, 0x3f332f34, 0x3f1d0d66, 0x3df5cb48, 0x3dd27670,
      0x3f00f0a1, 0x3d2ef880, 0x3f595d1d, 0x3f757dff, 0x3e2434c8, 0x3eb6ee10,
      0x3f3160b4, 0x3f4340a3, 0x3e2bf9d8, 0x3f44c362, 0x3df45d88, 0x3f03aaf1,
      0x3e8aedb0, 0x3f0864ba, 0x3e82663e, 0x3f7f893c, 0x3f2c9e98, 0x3f7bef69,
      0x3e92bda0, 0x3d8595e0, 0x3f6ac9e2, 0x3f162920, 0x3ef30786, 0x3e1fbfe4,
      0x3d9d8840, 0x3e8fffb8, 0x3f690843, 0x3ca13020, 0x3f6e3e98, 0x3f56cb81,
      0x3f699860, 0x3f2800b6, 0x3f7c2a76, 0x3f25f83f, 0x3e8f9a3a, 0x3f4b45c1,
      0x3e941d82, 0x3f0a6617, 0x3f5b568c, 0x3ebefe56, 0x3f062dc4, 0x3e6c3608,
      0x3f644910, 0x3f3d9385, 0x3f2248e3, 0x3f16e417, 0x3f422fc9, 0x3e02743c,
      0x3f10b997, 0x3f787f63, 0x3eff6528, 0x3f31f201, 0x3ed63288, 0x3e4dc254,
      0x3eef9eba, 0x3ea424f0, 0x3e033898, 0x3f483b3e, 0x3eee0e60, 0x3f54a3ae,
      0x3f707107, 0x3eb1d8b8, 0x3ea3f662, 0x3e86a1f0, 0x3e3a55a0, 0x3c5b4080,
      0x3f400a5d, 0x3ba1e600, 0x3c245d80, 0x3e03f8dc, 0x3f4bd525, 0x3f56d750,
      0x3f465ca5, 0x3f753a24, 0x3f53932c, 0x3f753932, 0x3f3002fa, 0x3f7573e1,
      0x3edc2d02, 0x3eea00b4, 0x3f4ef31c, 0x3f2061a2, 0x3ed42d84, 0x3cfc5a00,
      0x3f3582b0, 0x3eb5e528, 0x3f39b688, 0x3f3d6023, 0x3e93d0ac, 0x3ed99934,
      0x3ca1a700, 0x3ee8cbc6, 0x3ed281a4, 0x3f22db40, 0x3f553c37, 0x3f569447,
      0x3ef9dd52, 0x3e9119ae, 0x3d9d6450, 0x3ed21636, 0x3d7c0c80, 0x3ebf4de0,
      0x3eff1cdc, 0x3f0d44fd, 0x3f6fa051, 0x3d224dc0, 0x3f35bc2d, 0x3f36af8d,
      0x3dabfb38, 0x3f5f0742, 0x3f1d0ba1, 0x3f72de8e, 0x3f595940, 0x3efda5c6,
      0x3ed11820, 0x3f2caeca, 0x3f1b0ee9, 0x3f79485e, 0x3eaa2076, 0x3e8c2908,
      0x3f2aab2a, 0x3f1fb784, 0x3ebed2b0, 0x3efaad12, 0x3f32d9c5, 0x3f5ca5aa,
      0x3f13aac6, 0x3f33e900, 0x3e9c048a, 0x3ebeec30, 0x3de4ace8, 0x3ee1bf3e,
      0x3eda1554, 0x3e4304a0, 0x3eccea32, 0x3c185780, 0x3a62c000, 0x3f129687,
      0x3f2b6afb, 0x3ef8dd9a, 0x3efef2c8, 0x3efabb70, 0x3f31329b, 0x3eaaf4b2,
      0x3e705b30, 0x3f359ace, 0x3d755780, 0x3e03656c, 0x3f5aee42, 0x3f601ac2,
      0x3ecf12c2, 0x3f1542cb, 0x3f616c0f, 0x3f65fa63, 0x3eb25ac8, 0x3f3150f8,
      0x3e4f0330, 0x3f7f2005, 0x3c8f2020, 0x3d7e11f0, 0x3ce24b60, 0x3f4f1d9b,
      0x3e59a0f8, 0x3e95f830, 0x3e8259fc, 0x3e9af1a6, 0x3e9d9f6a, 0x3f2be0f4,
      0x3e58d5cc, 0x3c3b3ec0, 0x3db825b0, 0x3f22da62, 0x3f33d2af, 0x3f2a8f48,
      0x3eeebd58, 0x3f47df0d, 0x3e562c84, 0x3f389e47, 0x3f73c47b, 0x3f0f051d,
      0x3e09338c, 0x3f09097a, 0x3f2636c7, 0x3f7fdd98, 0x3f43b245, 0x3e887714,
      0x3ee2db9a, 0x3dc37750, 0x3e4248b8, 0x3f68cb57, 0x3e9e4ddc, 0x3ef87f02,
      0x3eae45ac, 0x3ef16d84, 0x3e9b63ba, 0x3f15d482, 0x3f7686fc, 0x3f32374b,
      0x3ecaa86e, 0x3d4eb820, 0x3e189168, 0x3f657fb2, 0x3f487ddb, 0x3e485900,
      0x3e87ec2e, 0x3f4c0789, 0x3ef47170, 0x3f05f39d, 0x3f071df5, 0x3f7ef527,
      0x3e44b92c, 0x3f741fe0, 0x3f72785a, 0x3e8d064a, 0x3f56187b, 0x3f6369db,
      0x3f763261, 0x3f20d6c1, 0x3f361d8b, 0x3f01b064, 0x3e3e8e40, 0x3e477d3c,
      0x3f30fba9, 0x3f7de0ed, 0x3f4e06a5, 0x3f120740, 0x3f0fc4b2, 0x3e124838,
      0x3f6bd6fc, 0x3e524bb8, 0x3dd3ae18, 0x3f026d51, 0x3f0c2ae3, 0x3f1f40f5,
      0x3ef1961c, 0x3ed25ff8, 0x3ea8a132, 0x3ef53324, 0x3efd0554, 0x3e04ffa0,
      0x3f5e4d19, 0x3f15b827, 0x3f3a2ce0, 0x3e509920, 0x3f578a80, 0x3f35395e,
      0x3f0202ea, 0x3ef45882, 0x3ee79d08, 0x3f311f86, 0x3f46114c, 0x3f527009,
      0x3f03040e, 0x3f67a752, 0x3f636c40, 0x3f305276, 0x3f1d0907, 0x3f375100,
      0x3d14c6c0, 0x3f6427f7, 0x3e173860, 0x3f4086d0, 0x3f251e5a, 0x3f08e8aa,
      0x3f2c5678, 0x3ebfe734, 0x3ca7c0c0, 0x3ea0cb40, 0x3e240a40, 0x3e7771dc,
      0x3f46629f, 0x3d2f2a90, 0x3de78420, 0x3f2098a2, 0x3f6036b8, 0x3d57cbd0,
      0x3ec24cb8, 0x3f1abc4f, 0x3e7fa680, 0x3f531f0a, 0x3f008460, 0x3f25578b,
      0x3d686ee0, 0x3f509a87, 0x3f59da6b, 0x3e893680, 0x3e8757ae, 0x3f5db794,
      0x3e1a2738, 0x3f0f2303, 0x3ea6bab0, 0x3eb59ca6, 0x3f21173a, 0x3ef9bf4c,
      0x3f58b072, 0x3ea3aede, 0x3e826bf6, 0x3f48612c, 0x3e9ae01e, 0x3ef38c3e,
      0x3eb5d910, 0x3f231879, 0x3ef604e6, 0x3e5c26e8, 0x3f0ce0b9, 0x3e993b1e,
      0x3f40f716, 0x3d03ba80, 0x3f46ef54, 0x3f16e89e, 0x3f0a2dab, 0x3f09c400,
      0x3f17d07c, 0x3f6a60a4, 0x3b3eb100, 0x3ec323c4, 0x3f11e57d, 0x3e9e09d0,
      0x3f6dc06e, 0x3f113b2d, 0x3ec51676, 0x3ea7c88e, 0x3f51a558, 0x3dc49d10,
      0x3ef73706, 0x3f7c9820, 0x3ea1bf14, 0x3cd7c060, 0x3f520de6, 0x3f09a730,
      0x3f1d1603, 0x3f6d2e4e, 0x3e36f550, 0x3f2b96c8, 0x3f65850c, 0x3ea4292e,
      0x3e1bf0b4, 0x3f48ff2d, 0x3eb264c2, 0x3f04938b, 0x3e04a5a4, 0x3e719f1c,
      0x3e2851e0, 0x3ea32a5c, 0x3ee7f486, 0x3d942018, 0x3edb51c2, 0x3d9b0df0,
      0x3f5f80d5, 0x3e79d294, 0x3d6368e0, 0x3ed616ec, 0x3ec07510, 0x3eb2237c,
      0x3e336bfc, 0x3f0857f4, 0x3f643c6a, 0x3f0a9bd0, 0x3ea97538, 0x3f0cc6ea,
      0x3f0349d7, 0x3f7837e4, 0x3f28e3c4, 0x3f182155, 0x3c2022c0, 0x3dcfc128,
      0x3cc5d4e0, 0x3edc633c, 0x3f049650, 0x3dc3f3d0, 0x3f554ea0, 0x3f14f358,
      0x3e9f12d0, 0x3f5e568b, 0x3efee806, 0x3ee80fbe, 0x3efa549a, 0x3eb13138,
      0x3edea102, 0x3ee5caa8, 0x3f14c6f6, 0x3f5f7be2, 0x3f6a3552, 0x3f3530d0,
      0x3f4d53c4, 0x3f328d76, 0x3f762e45, 0x3d4a4180, 0x3f78be4d, 0x3f364eb2,
      0x3f4f252a, 0x3e675ea4, 0x3f18c5b8, 0x3e16f590, 0x3f2c6606, 0x3ecca7fe,
      0x3f2c1590, 0x3f03fae4, 0x3f046c04, 0x3f5054bf, 0x3eb195a4, 0x3cbb6940,
      0x3ec22620, 0x3f6e4ef7, 0x3ef9a4e6, 0x3e97d63e, 0x3f72afeb, 0x3ec5c2dc,
      0x3f3718ee, 0x3dbd3cc0, 0x3e243758, 0x3e83a3f2, 0x3f4afacc, 0x3db27ea8,
      0x3f20dfa1, 0x3e8267e0, 0x3bf90e00, 0x3e3e79ec, 0x3f4af74f, 0x3f0b254d,
      0x3f2904d4, 0x3ed25a54, 0x3ecf2002, 0x3f216188, 0x3ed106f4, 0x3ce40b80,
      0x3f128659, 0x3d0af280, 0x3d42f880, 0x3f0c2351, 0x3f1b1295, 0x3dcd9fa0,
      0x3eef4b00, 0x3d8d4058, 0x3f34d4e6, 0x3f022e1e, 0x3dc43d48, 0x3f4a4526,
      0x3f77f9c3, 0x3f027a21, 0x3e61a9a0, 0x3f28a426, 0x3f6a5e14, 0x3e7ed790,
      0x3f595100, 0x3ee07650, 0x3f103dca, 0x3f265efc, 0x3f6d0eda, 0x3f208d06,
      0x3eee460a, 0x3f53c6ad, 0x3f35f4e4, 0x3e491a8c, 0x3edfa122, 0x3f6ae4c8,
      0x3c04b240, 0x3ee4a666, 0x3e640d3c, 0x3f057a38, 0x3ef2d08a, 0x3ddb1d30,
      0x3f1ef9bd, 0x3f59b5f6, 0x3ec9467e, 0x3b9f4e80, 0x3ef417b8, 0x3d40d7b0,
      0x3f166c75, 0x3f48c414, 0x3dc185e0, 0x3dcfb480, 0x3f29a020, 0x3dea2738,
      0x3f3c5bc2, 0x3f63d794, 0x3ecdaf12, 0x3f4092f3, 0x3f04a597, 0x3f3bf424,
      0x3cf68a80, 0x3ecaf7c4, 0x3f680a46, 0x3f0cd9b3, 0x3d9a6170, 0x3f74571d,
      0x3f189457, 0x3f0fb14c, 0x3f32a257, 0x3e37c8a8, 0x3ee3f834, 0x3f1b9abb,
      0x3f6c2eed, 0x3d81d488, 0x3f576f7a, 0x3e64d798, 0x3ea4c918, 0x3e0465f0,
      0x3f10ee31, 0x3f49a007, 0x3e672bd0, 0x3f5ac69f, 0x3f4e398c, 0x3e66ea30,
      0x3e819114, 0x3f35d445, 0x3d5f85c0, 0x3d9f6d18, 0x3e27ec0c, 0x3e95349e,
      0x3e868136, 0x3f1a4348, 0x3e8b20d6, 0x3e4497f0, 0x3de0e670, 0x3c9df120,
      0x3f266f3a, 0x3ec92d5e, 0x3e95f826, 0x3f6195ce, 0x3f6561ce, 0x3f32d7f1,
      0x3f38602a, 0x3f744f23, 0x3f4c2594, 0x3dcf7210, 0x3db90100, 0x3f5f9ba4,
      0x3de93160, 0x3db1ed90, 0x3e77e55c, 0x3e593b08, 0x3f3bd60c, 0x3f6a2099,
      0x3f71fcfd, 0x3f432777, 0x3f526fea, 0x3f496542, 0x3ed3a442, 0x3d34b0d0,
      0x3e7403f4, 0x3f6fef7c, 0x3dee3f40, 0x3dee2ba0, 0x3ce9a380, 0x3f2fbeb1,
      0x3f1b8226, 0x3f2ce7bd, 0x3f242d28, 0x3f1f15bb, 0x3d15a8d0, 0x3ecf248c,
      0x3f523e8c, 0x3f68f26e, 0x3f111dcd, 0x3f644868, 0x3f045a7c, 0x3cd52e40,
      0x3e090228, 0x3e468aac, 0x3f5683b3, 0x3f1012b0, 0x3eea4c98, 0x3f4fa265,
      0x3f599461, 0x3f72ce24, 0x3f54457a, 0x3e73f554, 0x3f31811f, 0x3f452b9e,
      0x3f00ad20, 0x3f5e239e, 0x3dea9d98, 0x3e30e688, 0x3f1663cf, 0x3ef10362,
      0x3eaf5b1e, 0x3e947300, 0x3f71bbe4, 0x3d271680, 0x3e7f71b0, 0x3f078c13,
      0x3e3a54c0, 0x3f6d6c9a, 0x3f1f63db, 0x3f446f5a, 0x3f6d72bf, 0x3e371870,
      0x3f66f42f, 0x3e8be2e4, 0x3f7bf867, 0x3ef936b6, 0x3eeeecae, 0x3f4ce3cc,
      0x3f49ffd9, 0x3d9e5af8, 0x3f351566, 0x3efacb16, 0x3f19305c, 0x3eafecf6,
      0x3ec8e93c, 0x3f77e805, 0x3d94bec0, 0x3e459a4c, 0x3e9cb5b2, 0x3eff2850,
      0x3f1f0616, 0x3f27f7ac, 0x3f2049f2, 0x3de85d90, 0x3ec46a54, 0x3f6a5bf4,
      0x3f22beb3, 0x3f476ea3, 0x3ef531ec, 0x3ecf4648, 0x3e63a008, 0x3e500980,
      0x3dc9a988, 0x3f1f2875, 0x3f3d6ff0, 0x3f26476b, 0x3e4c7368, 0x3f79d8f9,
      0x3f7163a8, 0x3f127746, 0x3e9acad6, 0x3f50416f, 0x3e0a3764, 0x3f407361,
      0x3d863c60, 0x3f2aa2d2, 0x3efaecb4, 0x3f5e64d4, 0x3f3c2367, 0x3e456730,
      0x3f2f91cc, 0x3f2f9a98, 0x3dce6770, 0x3ee98ae8, 0x3f4a12f8, 0x3e28fcdc,
      0x3f3768b4, 0x3e2b0850, 0x3ed5462a, 0x3c33c800, 0x3f34576e, 0x3d99d928,
      0x3efc18ee, 0x3f53b5cb, 0x3e6ff138, 0x3f593d23, 0x3bbd5400, 0x3f4e8810,
      0x3e798830, 0x3f088dea, 0x3f25f6f3, 0x3e9a0dd6, 0x3f08c3ec, 0x3f0472d2,
      0x3da2d0e0, 0x3f379feb, 0x3dce43b8, 0x3f0c2f75, 0x3cf2d820, 0x3edabe3a,
      0x3f5492ed, 0x3d8458a0, 0x3f0159f9, 0x3eceb838, 0x3e99b766, 0x3f616097,
      0x3ef170a4, 0x3e42ccb0, 0x3f0f828c, 0x3e5f56e4, 0x3f03cf3d, 0x3ecd7f68,
      0x3e90cd66, 0x3e8d4228, 0x3f0370cc, 0x3ea8e362, 0x3f46f665, 0x3f310e94,
      0x3ea651b8, 0x3db9f2e8, 0x3f7c709c, 0x3ed4b2f4, 0x3e9b356e, 0x3f16e8ff,
      0x3f0de0ab, 0x3f63e075, 0x3d8be170, 0x3dd62588, 0x3f4a9656, 0x3eeaa6ae,
      0x3f639f2b, 0x3f140ab8, 0x3f59faa2, 0x3e492ca8, 0x3f2e0ecf, 0x3e6a5404,
      0x3d5142a0, 0x3e095f38, 0x3b06cb00, 0x3edeb7aa, 0x3f0248c7, 0x3e2a5bcc,
      0x3f0fc87b, 0x3e43e100, 0x3e9552d4, 0x3f2c7ba7, 0x3f7ecf7c, 0x3d33aab0,
      0x3d685c60, 0x3ec53d5e, 0x3e8d6620, 0x3f7f0ab4, 0x3ec0bdcc, 0x3f63fd18,
      0x3dab59b0, 0x3f6bea4a, 0x3e7734a4, 0x3f529e4f, 0x3f1be6bb, 0x3f7435f7,
      0x3e485f64, 0x3e966afc, 0x3f2923cd, 0x3f55761b, 0x3e93a548, 0x3f7e189d,
      0x3f5e5618, 0x3f738a75, 0x3e96944e, 0x3ef4d644, 0x3f03db36, 0x3f3df7a2,
      0x3ece0e08, 0x3f4624f6, 0x3eb398aa, 0x3f1bc3bf, 0x3f287e57, 0x3ec5f0da,
      0x3f0f675e, 0x3e0fd6c8, 0x3f662cef, 0x3ea98894, 0x3b4d8000, 0x3e4c3fa4,
      0x3f31763c, 0x3d4e91b0, 0x3f263f88, 0x3e26e93c, 0x3f5a3217, 0x3ba7b900,
      0x3f1123cc, 0x3f2486d6, 0x3f251a39, 0x3f624d10, 0x3f3c0ac3, 0x3e9d338c,
      0x3f30a70e, 0x3f577929, 0x3eb9511a, 0x3f3b7363, 0x3e9a3b0e, 0x3e1fc188,
      0x3e77b2d8, 0x3ecd0964, 0x3f246d1d, 0x3f6e821f, 0x3e3afa50, 0x3eb97996,
      0x3f228232, 0x3d7b7070, 0x3e91a548, 0x3f3ffb2e, 0x3e8a35f0, 0x3e173980,
      0x3f63a28f, 0x3f2fa0c8, 0x3f1f55de, 0x3e4c82ac, 0x3f345672, 0x3e10f60c,
      0x3c5c19c0, 0x3eaaedda, 0x3d8cdfb8, 0x3d408450, 0x3bfdcb00, 0x3ef9b4e4,
      0x3f45e2b0, 0x3e86b840, 0x3d76dad0, 0x3f7e9142, 0x3e8928e4, 0x3e19d144,
      0x3ec42918, 0x3f0ae221, 0x3f43a419, 0x3e0d8408, 0x3e1dc598, 0x3f0e0ec8,
      0x3f492f98, 0x3f5fb339, 0x3f465f0e, 0x3eb6bf26, 0x3f715b5e, 0x3ef79e58,
      0x3ec3e3be, 0x3d4e4d60, 0x3f50b567, 0x3e6be678, 0x3f463e1b, 0x3f6a34da,
      0x3f1f2dd1, 0x3f1ccc19, 0x3e87800c, 0x3f305a79, 0x3b9ad580, 0x3f46943a,
      0x3def1230, 0x3ef3bcb0, 0x3ed8abc8, 0x3f560325, 0x3da2e3d8, 0x3f370cca,
      0x3eba530e, 0x3e20a1f8, 0x3e45f97c, 0x3edb8f50, 0x3e82c882, 0x3e29cd40,
      0x3e88ff12, 0x3f6b4a53, 0x3e5c3df8, 0x3f131101, 0x3f64681b, 0x3f1a765a,
      0x3dec0a60, 0x3ed95430, 0x3efe0a50, 0x3ef370ce, 0x3f5e58df, 0x3e5e6cbc,
      0x3f21e634, 0x3f7a707f, 0x3ece85b2, 0x3e86eb6a, 0x3f454746, 0x3f4c0c0f,
      0x3f774311, 0x3e13a5ac, 0x3f4b676d, 0x3f7c2b04, 0x3f2383a7, 0x3f75ea82,
      0x3e02b694, 0x3f07cbba, 0x3f66b46f, 0x3f3cba22, 0x3de50fa8, 0x3dda54d0,
      0x3f7e6bb7, 0x3ebc283e, 0x3f669e20, 0x3ebcc22a, 0x3ed48068, 0x3f6e57ba,
      0x3f0e82d9, 0x3f1adb40, 0x3eaf97b0, 0x3e25b7a8, 0x3f6fa985, 0x3e443458,
      0x3ca10900, 0x3c9317c0, 0x3f0260a1, 0x3e84a166, 0x3e4dc148, 0x3e958a98,
      0x3eb75efc, 0x3f261269, 0x3ea3b3de, 0x3ebdad98, 0x3eb79c90, 0x3e5dd240,
      0x3f53aa15, 0x3e6f350c, 0x3f17b9aa, 0x3f72c6e1, 0x3f554366, 0x3f296595,
      0x3d9c2888, 0x3f47af71, 0x3cae2320, 0x3e932fec, 0x3ebe6b28, 0x3d7dd930,
      0x3ed79088, 0x3e91fb4a, 0x3ece1b38, 0x3f0f2bb4, 0x3e5c4164, 0x3e458214,
      0x3f78239d, 0x3efddbfa, 0x3ee7f49a, 0x3f4cdf09, 0x3e337188, 0x3f1466b1,
      0x3eb1434a, 0x3de78d48, 0x3f1cc3a3, 0x3f00e0e5, 0x3e55763c, 0x3e82a3f0,
      0x3f4b8499, 0x3e3719d8, 0x3f4211a3, 0x3f3aff13, 0x3d1ca000, 0x3f668513,
      0x3ecd7990, 0x3f3ce6b0, 0x3ee89194, 0x3e825ce2, 0x3ee31ba4, 0x3ee91046,
      0x3edd47ce, 0x3f723c6c, 0x3e860fe4, 0x3f024c5f, 0x3ef94040, 0x3f3194b7,
      0x3e5c802c, 0x3f6aad09, 0x3f72d818, 0x3f0bfbed, 0x3f511cb4, 0x3f365213,
      0x3f5faa57, 0x3db5dde8, 0x3ec7a944, 0x3f4b6844, 0x3e7e6b0c, 0x3e36d818,
      0x3d70c5b0, 0x3eab8cfe, 0x3ed05dce, 0x3f691005, 0x3eb7cdfa, 0x3f6ab9f6,
      0x3e2ebe40, 0x3f57e835, 0x3e5a4adc, 0x3e30116c, 0x3e18b34c, 0x3f536553,
      0x3d58bc80, 0x3f326754, 0x3f3c8751, 0x3ec590cc, 0x3f7e3076, 0x3f2e3b1e,
      0x3f10f414, 0x3f4786ad, 0x3f481efc, 0x3e4866d4, 0x3f015d2f, 0x3d2f6710,
      0x3e4b5484, 0x3f740318, 0x3f46091e, 0x3f73b3d1, 0x3ca8cfa0, 0x3f7c01eb,
      0x3f747219, 0x3eb9ba4c, 0x3f1276f4, 0x3f6c1f4e, 0x3f3aab14, 0x3e2c8170,
      0x3f35994e, 0x3eaa01ee, 0x3e1cebb8, 0x3f611969, 0x3f4643db, 0x3f6a617f,
      0x3f514489, 0x3f0a4a23, 0x3f764ae4, 0x3f2c6748, 0x3f56c6ac, 0x3f763b87,
      0x3f0e1b99, 0x3d3e57c0, 0x3f56c4bb, 0x3eb551a2, 0x3f7fa8f0, 0x3efc0212,
      0x3f678ac4, 0x3ec8cc8a, 0x3f4f855e, 0x3f2974d0, 0x3ecaa5ce, 0x3f37e2dd,
      0x3e6ff598, 0x3f7933a6, 0x3f372480, 0x3db7be20, 0x3f1bee0e, 0x3e022cf4,
      0x3eaa78c2, 0x3eeae1c6, 0x3f14aa90, 0x3efd21da, 0x3f611664, 0x3f56e65e,
      0x3f6c08f4, 0x3ea216c8, 0x3f433e46, 0x3e5590a4, 0x3f1781a5, 0x3f099b7a,
      0x3f1ae27a, 0x3f4fd0bb, 0x3ef14684, 0x3dd89030, 0x3f4f324a, 0x3eeb0542,
      0x3edb9f22, 0x3e93ebb0, 0x3e66f1e0, 0x3f6970f1, 0x3e70e300, 0x3f3895e7,
      0x3f597387, 0x3f0a9b1b, 0x3f3cc46b, 0x3f077686, 0x3ebd35bc, 0x3ef96736,
      0x3dd0c998, 0x3dfd4988, 0x3f212a2b, 0x3eb7a320, 0x3b17b100, 0x3eb3c42a,
      0x3f1582b1, 0x3f1c11b3, 0x3e8913fa, 0x3dbdeb30, 0x3df8fcb8, 0x3dc46af0,
      0x3f352040, 0x3f589011, 0x3edc808c, 0x3e80d37c, 0x3d9236b0, 0x3e9f2e9e,
      0x3f4d1f56, 0x3f08ccde, 0x3f34f622, 0x3e3d12cc, 0x3e223134, 0x3f44ff79,
      0x3e6abc74, 0x3f144753, 0x3da02ee8, 0x3d764cb0, 0x3f2a681d, 0x3f3491f1,
      0x3f0ba337, 0x3f6bd6ac, 0x3f12be1f, 0x3eaa3680, 0x3e553a74, 0x3f3d6fe9,
      0x3f2611e3, 0x3f0463db, 0x3f1b3942, 0x3f5e14fd, 0x3c59a740, 0x3e2dd320,
      0x3e8016e8, 0x3e9dd2ea, 0x3c22b540, 0x3d66d830, 0x3d0a2720, 0x3c96bce0,
      0x3e843762, 0x3f61c34c, 0x3f127569, 0x3f64d6da, 0x3db25a28, 0x3ed2912e,
      0x3f0f72b5, 0x3e0d5b28, 0x3dddbfd0, 0x3f21c60d, 0x3f23f63b, 0x3d6dda30,
      0x3e85953a, 0x3f3c5bc9, 0x3d538620, 0x3f47704f, 0x3ea30326, 0x3f181936,
      0x3f7fd50c, 0x3f530489, 0x3f7060a9, 0x3ee2e3fc, 0x3f044180, 0x3f3b70f7,
      0x3f1f4a58, 0x3efd69da, 0x3f41cd24, 0x3e86283c, 0x3f3497e9, 0x3f3a71b3,
      0x3eaa79c8, 0x3c131f00, 0x3f5ba1a1, 0x3f398f4f, 0x3f681381, 0x3de61d90,
      0x3f1e5aae, 0x3f398ee4, 0x3f720039, 0x3f5d9df1, 0x3e442bbc, 0x3f1cc518,
      0x3e920260, 0x3f511177, 0x3e6c3b04, 0x3f242e0c, 0x3f5c1cd5, 0x3f4eed1a,
      0x3eec875c, 0x3ee452f6, 0x3e92b064, 0x3f3dfa7f, 0x3f3a4e23, 0x3ebfbc38,
      0x3f22a1c9, 0x3f560a99, 0x3efff830, 0x3f7ff95d, 0x3efeceae, 0x3f78704e,
      0x3f32e8de, 0x3f323d9c, 0x3f336afa, 0x3f178a2a, 0x3ea009d8, 0x3f483fce,
      0x3f153090, 0x3f1c58d3, 0x3f0d2b3b, 0x3e809562, 0x3f255d6e, 0x3e2041ec,
      0x3efff0f4, 0x3f7673c3, 0x3ee88ec8, 0x3f44190e, 0x3f0322d9, 0x3f1d55d4,
      0x3e948cf0, 0x3e8534be, 0x3f7fe700, 0x3e69c60c, 0x3f2e0ebe, 0x3f6c03ae,
      0x3d64f890, 0x3f146a56, 0x3e831b86, 0x3f6c3c3a, 0x3da9b8e0, 0x3eab7e4c,
      0x3f603bf8, 0x3dfd1f68, 0x3f3a7662, 0x3f14a931, 0x3e98cbac, 0x3f2c15c3,
      0x3ec342b8, 0x3f6f735d, 0x3f34e540, 0x3f3af08a, 0x3ee0eb1a, 0x3f0e338f,
      0x3c8e3bc0, 0x3f28ebe0, 0x3e84250c, 0x3f13aa39, 0x3c61c100, 0x3f320917,
      0x3f5e4614, 0x3f71c360, 0x3f48d744, 0x3f6cc4b8, 0x3e13ccb4, 0x3edf03c8,
      0x3f670c60, 0x3dd16130, 0x3f583099, 0x3f4486fa, 0x3f24565a, 0x3df547f0,
      0x3ec30c1e, 0x3f6f621d, 0x3ecd683c, 0x3f57bbd0, 0x3d2d3680, 0x3f10a4bd,
      0x3f01f7f2, 0x3f501432, 0x3e25a45c, 0x3e864886, 0x3f48fe1f, 0x3f41e4d8,
      0x3e23be98, 0x3f6f746f, 0x3f4ea9ed, 0x3f16a40d, 0x3f75d25b, 0x3eae07a2,
      0x3f51766c, 0x3e8e9d3c, 0x3f60d151, 0x3f57ec87, 0x3efa1856, 0x3f2d3b5e,
      0x3dafb860, 0x3db85348, 0x3f30155e, 0x3f59d2f3, 0x3ed13064, 0x3f029fd1,
      0x3f5fc472, 0x3ed3041c, 0x3f0fcfa3, 0x3eaccfa4, 0x3ee427fa, 0x3e76d244,
      0x3f2b864f, 0x3f654967, 0x3f3e733a, 0x3f60df6f, 0x3eac9918, 0x3e3524e8,
      0x3f6cde8b, 0x3f7d9825, 0x3f2757ca, 0x3d37ee70, 0x3f7acfa9, 0x3e77aa98,
      0x3e755d2c, 0x3ee84108, 0x3eb10a4e, 0x3f624610, 0x3e614f68, 0x3f5405b3,
      0x3eee393a, 0x3f0e8e69, 0x3e06d1f8, 0x3e999c3a, 0x3df80670, 0x3f0ab050,
      0x3efe2d26, 0x3ee8bc76, 0x3e9029bc, 0x3e1f1118, 0x3e954f74, 0x3f36d02f,
      0x3f2f7c67, 0x3eeb9028, 0x3f2d43e7, 0x3f6b49e1, 0x3ee2bb78, 0x3f1bc70a,
      0x3f21895a, 0x3f15c497, 0x3e98612c, 0x3efc8c6a, 0x3e54f3fc, 0x3e77f614,
      0x3f1cf80a, 0x3f3beb10, 0x3f30d914, 0x3e9be6bc, 0x3ebf606c, 0x3f01c4c2,
      0x3f7f623c, 0x3ec4ca68, 0x3e7636d8, 0x3f436c2b, 0x3eb0c2a2, 0x3efa261c,
      0x3f200bea, 0x3de9b0b0, 0x3dfcff90, 0x3f5b7881, 0x3f693652, 0x3f2a5ae5,
      0x3e52a3c8, 0x3ef8840c, 0x3f5ac69c, 0x3f0f4988, 0x3e8f3aae, 0x3d385340,
      0x3e85b5c2, 0x3f46a618, 0x3e3c46b4, 0x3e113e64, 0x3f0b9306, 0x3e8a0b80,
      0x3f48e8e7, 0x3dc768d8, 0x3f272abb, 0x3e8d5b7c, 0x3ef5aa78, 0x3f50361d,
      0x3f7769dc, 0x3f77e957, 0x3f6c763e, 0x3e66579c, 0x3ee5f23a, 0x3d677940,
      0x3d2f6c00, 0x3f2daff7, 0x3f077be5, 0x3f2d9ce3, 0x3e0924f0, 0x3f16ebc5,
      0x3dbe7a68, 0x3f3c5b2a, 0x3f4ffcb0, 0x3cfa0760, 0x3f2f6364, 0x3f46d903,
      0x3dcd8e50, 0x3f3df586, 0x3ebc4bca, 0x3e6d0a14, 0x3f52aa2c, 0x3f3573f4,
      0x3f633133, 0x3f58229d, 0x3d8cda18, 0x3c9352c0, 0x3efa765c, 0x3f33bfca,
      0x3e46c45c, 0x3f703e85, 0x3e8592e6, 0x3efd79f6, 0x3ec3d824, 0x3f2be1ce,
      0x3f47fee3, 0x3e8f1ba0, 0x3eb138f2, 0x3f740733, 0x3f2d475b, 0x3e862a0a,
      0x3e9a2daa, 0x3f21946d, 0x3e564fa8, 0x3f291afc, 0x3f04b722, 0x3f127996,
      0x3f4c846a, 0x3f4ced47, 0x3e4c3bd4, 0x3f0d3585, 0x3ee064ce, 0x3f3e2443,
      0x3ef5def4, 0x3ba20000, 0x3f383b0f, 0x3f3b2372, 0x3f23e5bc, 0x3e13ac30,
      0x3eaf03fa, 0x3f4a22ba, 0x3f76bc54, 0x3e389b38, 0x3f410ba1, 0x3f6a336c,
      0x3f5a8d80, 0x3ca3a940, 0x3f51a407, 0x3ec6d6c8, 0x3f3bd982, 0x3e7f5800,
      0x3f7c3988, 0x3e31fcb0, 0x3f2b0471, 0x3f63f395, 0x3e71851c, 0x3f0048a0,
      0x3e97dc4c, 0x3e53adf8, 0x3dd20ed0, 0x3da668a0, 0x3f2014d9, 0x3eddf04c,
      0x3f5ccc7b, 0x3e23011c, 0x3ed57f9c, 0x3c3ae140, 0x3e945d80, 0x3efaecf8,
      0x3dbd3af8, 0x3e1016b4, 0x3f2f15e9, 0x3eabbafc, 0x3f195dde, 0x3f7f98c4,
      0x3df5f4d8, 0x3f4b85a3, 0x3f709498, 0x3d547530, 0x3ea4f138, 0x3f008bd8,
      0x3f0238e0, 0x3d8bf808, 0x3e82c1e4, 0x3ec9f61c, 0x3eface24, 0x3f667ede,
      0x3f6be85d, 0x3e83a310, 0x3f4bd848, 0x3f54fa2e, 0x3f2297dc, 0x3e0badd0,
      0x3f05ecad, 0x3d0391b0, 0x3f498876, 0x3efd1096, 0x3ef6071e, 0x3ed7f250,
      0x3f5fa873, 0x3e9abed6, 0x3ec8276a, 0x3f21602e, 0x3da602c8, 0x3c880fc0,
      0x3f4c880d, 0x3eff0c3a, 0x3d2a8100, 0x3f116087, 0x3e0df66c, 0x3e5ea110,
      0x3f500cd7, 0x3f2b0de5, 0x3f3b00ba, 0x3e201a14, 0x3eb102ba, 0x3f65e38a,
      0x3f2cae99, 0x3e77fc98, 0x3f41727f, 0x3e7ebe48, 0x3e48ea6c, 0x3e5d3b70,
      0x3c8b90a0, 0x3f16a40f, 0x3dd3a640, 0x3f0efb14, 0x3ce40dc0, 0x3f4b2b1d,
      0x3df14440, 0x3f3e342b, 0x3f63b8bc, 0x3f6fed88, 0x3deaff38, 0x3e7f4a40,
      0x3db4b090, 0x3f67b32b, 0x3ef0d598, 0x3e091444, 0x3e93bdfa, 0x3f6921d4,
      0x3f1deac9, 0x3ece6476, 0x3dff8118, 0x3f543339, 0x3e3c701c, 0x3ee65820,
      0x3f142426, 0x3e246830, 0x3f0b1ba5, 0x3e46c48c, 0x3f3dd169, 0x3ec4c396,
      0x3f11ddf2, 0x3f4e810c, 0x3f47d649, 0x3d898da8, 0x3f352aaa, 0x3f774f54,
      0x3cf165e0, 0x3ea3cf54, 0x3f04a59b, 0x3f1823fc, 0x3c866480, 0x3ed64912,
      0x3f21c63d, 0x3cb0e280, 0x3de75608, 0x3f5b635f, 0x3eef44a4, 0x3f5bf695,
      0x3e5004ec, 0x3e9e49ac, 0x3e6cb4f0, 0x3f13e360, 0x3ee84642, 0x3f170a7e,
      0x3c98b440, 0x3ed8434e, 0x3f4b6a8d, 0x3f4e1c12, 0x3f017445, 0x3f602c52,
      0x3f1d98ab, 0x3f1c39e0, 0x3f36b91e, 0x3f099460, 0x3f3f473c, 0x3eaa309c,
      0x3ef40028, 0x3f614cb0, 0x3f51f29d, 0x3edc5382, 0x3e9817ba, 0x3eebfaee,
      0x3dfe9798, 0x3f513324, 0x3f2f18bc, 0x3f1eaee4, 0x3f7ec9cf, 0x3e9a5f6e,
      0x3f02c823, 0x3e926c4e, 0x3f52e716, 0x3f1f4e7f, 0x3f4699de, 0x3e5e306c,
      0x3f37202a, 0x3f67b6c3, 0x3ec800ec, 0x3ebc5468, 0x3c4cb380, 0x3d8713f8,
      0x3caa03c0, 0x3f2f2036, 0x3f3b96be, 0x3f1da0bb, 0x3f162c47, 0x3edf0b52,
      0x3ef739c6, 0x3f20fc82, 0x3f23c33f, 0x3f07db38, 0x3f3c6951, 0x3f7d9df3,
      0x3f6716d5, 0x3e0a1e28, 0x3f1643ac, 0x3d48e910, 0x3ef0e90c, 0x3f12d5c9,
      0x3f6e4c6f, 0x3ed4912a, 0x3f722240, 0x3f5df927, 0x3f2566e8, 0x3f2ec523,
      0x3e969378, 0x3f2db7d4, 0x3e9bedd0, 0x3d083d30, 0x3eeabfbc, 0x3ec5ff7e,
      0x3ea059f6, 0x3eb4bbae, 0x3f616a75, 0x3f28ff1f, 0x3f413800, 0x3ee24bee,
      0x3e72c494, 0x3f363b25, 0x3e9bda30, 0x3de09550, 0x3e1883f4, 0x3f5a9853,
      0x3df29198, 0x3f3bc2f9, 0x3f66ab27, 0x3ece770e, 0x3f0fa0ea, 0x3d6d73a0,
      0x3e83c22e, 0x3e1ad670, 0x3ed5d654, 0x3f3ab35e, 0x3f791af7, 0x3ef7a014,
      0x3f53a835, 0x3f67b69e, 0x3f605bb0, 0x3f2e9544, 0x3f4bc5da, 0x3ddfb1a8,
      0x3f5a50e0, 0x3f0d3923, 0x3eac19d4, 0x3f614842, 0x3ccd6160, 0x3e63bb00,
      0x3eac9742, 0x3eeef5e6, 0x3f0a7bb0, 0x3ea15e76, 0x3f41d738, 0x3f5d7a96,
      0x3f102db6, 0x3eeca428, 0x3f7019ee, 0x3f274fc1, 0x3f106608, 0x3f4187e6,
      0x3e4cb700, 0x3eec45e2, 0x3ee9acc4, 0x3f393026, 0x3eb3c5b8, 0x3e55a6ac,
      0x3ed7f8e8, 0x3ee304cc, 0x3eeaf74e, 0x3d190800, 0x3f518a7e, 0x3f342f7e,
      0x3f51e039, 0x3eaa4dde, 0x3e944224, 0x3e5737d8, 0x3f50cb33, 0x3f4ddbc8,
      0x3f4fbe94, 0x3f12d4b3, 0x3f5bbdde, 0x3f426e0c, 0x3f44b4f7, 0x3e54832c,
      0x3ed3b7a6, 0x3ebbaea0, 0x3f43590d, 0x3f16857b, 0x3e2dd944, 0x3f3a245d,
      0x3dcede78, 0x3f67a2d8, 0x3ec786ea, 0x3ed5cd6a, 0x3e0cc294, 0x3db674e0,
      0x3ef709a2, 0x3ea8623e, 0x3ebbbcfa, 0x3ebd075c, 0x3f12e024, 0x3ed05d68,
      0x3e821974, 0x3d28b000, 0x3f723c1b, 0x3f32a521, 0x3f6c2ffb, 0x3f145080,
      0x3f1e5875, 0x3f7ee410, 0x3f4da8b3, 0x3f59e2d9, 0x3e767498, 0x3eec5f82,
      0x3f74ecf3, 0x3f4b9316, 0x3f67ba98, 0x3f5ef6a7, 0x3f5726e4, 0x3e8e00a2,
      0x3f031959, 0x3e1380ac, 0x3e722d8c, 0x3f483aea, 0x3f01ca43, 0x3e207030,
      0x3ecf009e, 0x3f6513de, 0x3e9f542e, 0x3f23e5d6, 0x3ec9f980, 0x3ec0a858,
      0x3edbf588, 0x3d846290, 0x3e3065c8, 0x3ddfe360, 0x3e5e9414, 0x3e6a7f7c,
      0x3ef7a910, 0x3e809ff6, 0x3e0634e0, 0x3f49b10d, 0x3f4c9395, 0x3e257acc,
      0x3f1da635, 0x3e76b5c8, 0x3ee3c3ae, 0x3e903e0e, 0x3f2ecd34, 0x3f23b2e3,
      0x3efc3bae, 0x3f153f7e, 0x3ef5b9cc, 0x3f492e81, 0x3d7e3ba0, 0x3f27bf69,
      0x3d352820, 0x3f22bf1e, 0x3e7e2248, 0x3e19b688, 0x3f59b74c, 0x3dcf0850,
      0x3f751ea5, 0x3e9784f2, 0x3eb03d4a, 0x3f29ea13, 0x3e96fda6, 0x3e973df2,
      0x3e55e8b8, 0x3ea99e82, 0x3f0ab311, 0x3ef8b48e, 0x3d5c8aa0, 0x3e5afaf0,
      0x3f7d1f59, 0x3e6d6098, 0x3ecdbb70, 0x3f5caa96, 0x3f09f986, 0x3e91acd2,
      0x3eef4158, 0x3f621726, 0x3de9ad60, 0x3f0dfa1f, 0x3f648800, 0x3f0b34ac,
      0x3ef10064, 0x3f3e5453, 0x3f6945f9, 0x3f3f1ca2, 0x3f728858, 0x3e4e1a4c,
      0x3f3ee4a5, 0x3f0b496a, 0x3eaa8d1c, 0x3d0ac270, 0x3e65b030, 0x3f7c843d,
      0x3e8f817e, 0x3f06f535, 0x3f15fb72, 0x3f269adc, 0x3f212037, 0x3f2645b2,
      0x3f0132d6, 0x3f07f69b, 0x3f3d3e7b, 0x3c08a240, 0x3f2fcfda, 0x3f319dba,
      0x3ea14dac, 0x3ec52376, 0x3d295380, 0x3d2ec5e0, 0x3f42613f, 0x3f240e75,
      0x3e1a82d4, 0x3e8c0a1c, 0x3eded670, 0x3b6f8700, 0x3e1a18ec, 0x3e2e1cf0,
      0x3f50749b, 0x3f7344d2, 0x3decf740, 0x3e0ea45c, 0x3d6bd740, 0x3efa76f4,
      0x3f5d8683, 0x3e9bfdda, 0x3f6f9a3d, 0x3e9544ce, 0x3e499b28, 0x3d896360,
      0x3f39f9c7, 0x3e4494b8, 0x3e8ad98e, 0x3f210a7b, 0x3edff444, 0x3e6b664c,
      0x3f7d3c8a, 0x3ee675e8, 0x3f577b63, 0x3f10e622, 0x3e6e8158, 0x3f36f6e6,
      0x3f40c951, 0x3f276d71, 0x3f081dc8, 0x3eb4944c, 0x3f551eb2, 0x3de54648,
      0x3f6c21e5, 0x3d94f1a8, 0x3e931b0c, 0x3f2f6149, 0x3cf02720, 0x3f702f0d,
      0x3ec4e20c, 0x3f22fc7d, 0x3f338426, 0x3d186c70, 0x3f52b64e, 0x3f35da4e,
      0x3f600a27, 0x3ef3cd62, 0x3dceb1b8, 0x3f73e064, 0x3ebb3e40, 0x3f080b45,
      0x3f205635, 0x3e8903a2, 0x3ea2dd12, 0x3e6e3c5c, 0x3f606cdb, 0x3f03a4f3,
      0x3effbf00, 0x3f101011, 0x3ee0c556, 0x3e9500c8, 0x3cc977e0, 0x3ea7abda,
      0x3f5f2f83, 0x3f0cc395, 0x3f7b7b06, 0x3f046580, 0x3ee6f630, 0x3f1598f3,
      0x3de3f398, 0x3c9258a0, 0x3d82fcf8, 0x3e9df156, 0x3e26fc7c, 0x3f34f4f1,
      0x3ef6e51e, 0x3f3c3011, 0x3f2f7b3e, 0x3f2f77a8, 0x3c391300, 0x3e98f174,
      0x3e8bdefc, 0x3f30c783, 0x3f7e96c6, 0x3ee68bcc, 0x3f34f143, 0x3e829ce8,
      0x3f7e51ec, 0x3e599c8c, 0x3ec88726, 0x3f3e3fbc, 0x3f6fd35a, 0x3ec13da6,
      0x3ee144a0, 0x3f286fb7, 0x3e6fd0b0, 0x3ed9cc4c, 0x3ea2d060, 0x3e955556,
      0x3f2d06cf, 0x3e720e38, 0x3e2fd8ac, 0x3e9d9a9c, 0x3ea6b2ae, 0x3de723e0,
      0x3f5a8c80, 0x3f5f3173, 0x3ecc026e, 0x3f114ce9, 0x3e8c8674, 0x3df6e308,
      0x3f42c120, 0x3e8db458, 0x3f67cd9e, 0x3f30eaed, 0x3d64dc00, 0x3ea9c648,
      0x3e9b1c00, 0x3f2dd324, 0x3e3b5bbc, 0x3f4504a5, 0x3f5c1b0a, 0x3f55056c,
      0x3f031a8b, 0x3f295404, 0x3f79fe7e, 0x3f004533, 0x3f02251d, 0x3f552ee3,
      0x3e92b6e8, 0x3ebc8276, 0x3f3b09a9, 0x3d832f88, 0x3d883ec8, 0x3f39313c,
      0x3f79229e, 0x3f29359d, 0x3f2a06a1, 0x3f72c6d9, 0x3e472458, 0x3f3e730e,
      0x3e7e72e0, 0x3f4a247a, 0x3f3d4c9c, 0x3f7548eb, 0x3f06acff, 0x3f38e4a1,
      0x3f05bcf8, 0x3f4fec3e, 0x3f6ee7ac, 0x3f4ae502, 0x3f79fa3f, 0x3f74cb35,
      0x3f77b600, 0x3f0ebe64, 0x3e1dab58, 0x3f3a0bb6, 0x3f7c8f45, 0x3ec8cb28,
      0x3f1b8b9d, 0x3ed94b06, 0x3ecea096, 0x3f54fe0b, 0x3ea5ea86, 0x3f56c7ee,
      0x3e8115f6, 0x3f5cb782, 0x3e03f00c, 0x3f070772, 0x3ecff4dc, 0x3ed39aa0,
      0x3f1c8d73, 0x3f60bb81, 0x3e7fdbb8, 0x3f09dcec, 0x3e07d3f0, 0x3f645836,
      0x3da27348, 0x3ea00164, 0x3f356582, 0x3efeae18, 0x3e8f9a26, 0x3f4f505e,
      0x3f29a161, 0x3f12258e, 0x3f69c2b3, 0x3d874650, 0x3f1f6a46, 0x3f00ff2a,
      0x3de7d058, 0x3ed437ee, 0x3f2c6f5b, 0x3ed248d6, 0x3e60a4cc, 0x3f3e748e,
      0x3f600b31, 0x3f3d84e6, 0x3ec4d5cc, 0x3f4bafee, 0x3e386644, 0x3e318dc8,
      0x3eb17c54, 0x3e31e274, 0x3e8983d4, 0x3ef7f19e, 0x3f79fd37, 0x3f46ab95,
      0x3ef4fa0a, 0x3e45ff94, 0x3e29b3e8, 0x3bdda000, 0x3e4c0640, 0x3e5cbc1c,
      0x3ecfe312, 0x3ece25ca, 0x3f1ed6e3, 0x3f2912cb, 0x3b579900, 0x3e69f2e4,
      0x3d2799b0, 0x3f69ddbd, 0x3dc83ab8, 0x3f26740b, 0x3f6cc1fc, 0x3ea15e64,
      0x3f6e6eae, 0x3f1a2698, 0x3f669642, 0x3f2ffa0f, 0x3f70c1e2, 0x3f5ee0b9,
      0x3f48f8ab, 0x3f2a2697, 0x3ef3d882, 0x3f3147a7, 0x3f3024b6, 0x3e1b1a84,
      0x3f5c060f, 0x3f30a1d8, 0x3f0d2dc0, 0x3f3a2052, 0x3edd3952, 0x3f5dbf09,
      0x3f73ddbf, 0x3f796f28, 0x3e79afd8, 0x3f4303b6, 0x3e591f44, 0x3f435101,
      0x3f7b47a3, 0x3e846f08, 0x3e94f9f0, 0x3f2ab23d, 0x3f113f6c, 0x3f706bc8,
      0x3f74e40a, 0x3ef13f1c, 0x3f2a33b6, 0x3e6584e0, 0x3f032998, 0x3eb39b0a,
      0x3f364f87, 0x3f339940, 0x3f18226c, 0x3e8a80e0, 0x3e821214, 0x3f06546e,
      0x3f268e0e, 0x3f0b5f1b, 0x3f3fe2bf, 0x3f5b0d6c, 0x3f729e72, 0x3d2562c0,
      0x3d0ca500, 0x3f68fc82, 0x3f5366e4, 0x3f3568ef, 0x3e87fad4, 0x3f75729b,
      0x3f24e8a3, 0x3d6abaa0, 0x3e294464, 0x3ee48360, 0x3cd4a340, 0x3f079096,
      0x3f04b48a, 0x3eccbce8, 0x3f74479d, 0x3f5798f2, 0x3f177ffb, 0x3f468703,
      0x3ede83d8, 0x3e33576c, 0x3f079f61, 0x3ebbedf4, 0x3ef9986e, 0x3f1ee33b,
      0x3f100c86, 0x3f1d754f, 0x3dacc6d0, 0x3ea3ee3e, 0x3f6148f7, 0x3e59ebe0,
      0x3eb57436, 0x3f36b9d7, 0x3f48a837, 0x3eb09c80, 0x3f4af664, 0x3e62ef4c,
      0x3f780af1, 0x3e594e4c, 0x3e5f4328, 0x3f667cf8, 0x3f01ad7c, 0x3ed7d95e,
      0x3ee471de, 0x3f6f8cc3, 0x3f626796, 0x3e2bb88c, 0x3eb539bc, 0x3e9d1e70,
      0x3d04af60, 0x3f0dc769, 0x3e8509ea, 0x3f2d3dca, 0x3f0eaf70, 0x3f14c9cb,
      0x3f1d0796, 0x3d2e9980, 0x3e91e186, 0x3ef9dea2, 0x3eaf3da4, 0x3ee2b866,
      0x3f74d4df, 0x3c97ad20, 0x3f3501e4, 0x3eb685bc, 0x3f683e80, 0x3f3064eb,
      0x3e4850c8, 0x3f299c76, 0x3f7a2df3, 0x3d9d8918, 0x3f2e0403, 0x3ee87888,
      0x3ee1c118, 0x3f20afd2, 0x3e0a36a0, 0x3f4f002e, 0x3f2004be, 0x3ec0ea64,
      0x3ee5e8fe, 0x3e3b0098, 0x3f360ae8, 0x3f2f745a, 0x3de86310, 0x3f172288,
      0x3d0bdb90, 0x3f2dc56c, 0x3dd51778, 0x3f4c2f67, 0x3f7d1f1c, 0x3f697864,
      0x3f2b0d22, 0x3f53298b, 0x3f69653e, 0x3f65295a, 0x3e8a03fe, 0x3e0ec28c,
      0x3e2d7d60, 0x3f5fc080, 0x3ef7bc34, 0x3ddf8a60, 0x3f59eeff, 0x3f353235,
      0x3ebd4fe2, 0x3edc9062, 0x3f485eb6, 0x3f4a01a5, 0x3ef23f14, 0x3dd82e78,
      0x3eab4254, 0x3eacd2da, 0x3e483ad0, 0x3f17ab53, 0x3f794651, 0x3f5e7411,
      0x3f4a7dce, 0x3f2e1c71, 0x3f26887f, 0x3eb9448c, 0x3ec6bc90, 0x3f2d4776,
      0x3d30df00, 0x3e621c80, 0x3f674051, 0x3da442d0, 0x3f01afbd, 0x3da56ea8,
      0x3f1b26d0, 0x3d3b1de0, 0x3f463b7c, 0x3f5c5a36, 0x3e82972a, 0x3f6cb23b,
      0x3f4f466e, 0x3ec2e70a, 0x3f5856e9, 0x3f0e5043, 0x3f2429af, 0x3f4d2142,
      0x3f3882b1, 0x3f0f5c7d, 0x3f597aa7, 0x3f50529a, 0x3dab32d8, 0x3f6e5fdb,
      0x3e81cd28, 0x3f26a4f2, 0x3ec57b76, 0x3ebc7f04, 0x3f62884e, 0x3f0bea47,
      0x3f1a8299, 0x3f640dfd, 0x3db05d48, 0x3efc1a52, 0x3f753998, 0x3dc9acd0,
      0x3f2c5310, 0x3edb896c, 0x3f0f220d, 0x3f624244, 0x3e82dea4, 0x3f1cc000,
      0x3f6d70f2, 0x3e40a000, 0x3de915d0, 0x3ea79dd6, 0x3f6c3322, 0x3f3a5260,
      0x3f2fafe6, 0x3f0220ba, 0x3c442f00, 0x3ead4e94, 0x3f5e0bef, 0x3f31c108,
      0x3f0758ae, 0x3f23a01b, 0x3f357fc9, 0x3ec12ce6, 0x3f71ebfa, 0x3f54693f,
      0x3e498b00, 0x3f76fe33, 0x3f5c7f35, 0x3e1a0d8c, 0x3be7bb80, 0x3ea85f74,
      0x3f3501f1, 0x3f564dea, 0x3f1c1785, 0x3f54121f, 0x3dd2fe98, 0x3eaf108e,
      0x3f137b55, 0x3d38b7a0, 0x3d8d7ad8, 0x3f0682c8, 0x3b82b780, 0x3ed94374,
      0x3f6a6d68, 0x3f1947be, 0x3f09945e, 0x3f12a126, 0x3f211a5c, 0x3a417000,
      0x3f73859c, 0x3f45166e, 0x3f5ebe59, 0x3f4bcb13, 0x3e9c5f96, 0x3f75736f,
      0x3f51c1c5, 0x3ea9ee7a, 0x3f10e063, 0x3db5d480, 0x3f160ed2, 0x3f7cc91b,
      0x3f10cd60, 0x3f3bf56b, 0x3dd68df8, 0x3d9faa98, 0x3f51ab40, 0x3f0c3d4d,
      0x3e460d30, 0x3eed3cbc, 0x3ecc0972, 0x3d93c318, 0x3f2ee9a6, 0x3f085594,
      0x3f78f9dc, 0x3e961498, 0x3e1e79c0, 0x3f25770f, 0x3f007cea, 0x3e5cfafc,
      0x3dd02650, 0x3e91ff76, 0x3f07e6a2, 0x3eb84a80, 0x3f2cdfef, 0x3e22a594,
      0x3eea89f0, 0x3f492e26, 0x3e85078e, 0x3f5bbaef, 0x3eb3905a, 0x3f54cc91,
      0x3e93bf94, 0x3f6a45ae, 0x3cc7ade0, 0x3f549dbc, 0x3f0bb524, 0x3f418b70,
      0x3f51f4fb, 0x3f70a11f, 0x3f21af9d, 0x3e3d5918, 0x3de43008, 0x3f028458,
      0x3f6a153a, 0x3f14d2f6, 0x3ebb065e, 0x3f0f8d0e, 0x3e54b0a0, 0x3f26c7d4,
      0x3e283e68, 0x3ef86164, 0x3d2a2ac0, 0x3e24efec, 0x3f5e896c, 0x3edb8888,
      0x3f2bb5b7, 0x3f4e5a25, 0x3f7129ea, 0x3f30e0a6, 0x3e42d17c, 0x3f418e0c,
      0x3f656084, 0x3f653727, 0x3eb9f338, 0x3f0c8312, 0x3f6ad76b, 0x3c6202c0,
      0x3e21bfd0, 0x3eed70d8, 0x3d8ed008, 0x3f173534, 0x3f486671, 0x3d2b60a0,
      0x3e89b5f6, 0x3eb618e6, 0x3f2ce2fa, 0x3f1381de, 0x3f1d9160, 0x3e9a7c70,
      0x3db77628, 0x3e40f778, 0x3f69f880, 0x3f00cf19, 0x3f23b598, 0x3f08174d,
      0x3ea918a8, 0x3f37c571, 0x3e820f62, 0x3d9b0ad8, 0x3f6d08c0, 0x3f7e543a,
      0x3e8ce988, 0x3d861c00, 0x3d9b1840, 0x3eedd152, 0x3e485274, 0x3e6e882c,
      0x3f595212, 0x3f2d1eb1, 0x3ee28b92, 0x3f393f6c, 0x3e5e0ef8, 0x3e2c3a80,
      0x3f0e7d9c, 0x3d9d3400, 0x3f7c9d2b, 0x3e66c4e8, 0x3f1009f7, 0x3e7bf370,
      0x3e87f8d0, 0x3f788965, 0x3f7e73a4, 0x3f7c8526, 0x3dfb2360, 0x3f02793c,
      0x3e070eec, 0x3cdc3e20, 0x3f32d2dc, 0x3f098e7b, 0x3e4a9388, 0x3f00bbcb,
      0x3f17cf5c, 0x3e4ecfdc, 0x3ec449ba, 0x3eb8c960, 0x3ef8abde, 0x3e24469c,
      0x3f170541, 0x3edbb978, 0x3f4d6120, 0x3ef277ae, 0x3f4e43f2, 0x3ec9478a,
      0x3e6e5808, 0x3ebde8d6, 0x3f082d70, 0x3e87f052, 0x3e29bfb0, 0x3eda91e2,
      0x3f202a5d, 0x3ed63c34, 0x3f0fa677, 0x3f0775e8, 0x3ea35558, 0x3dc8b398,
      0x3e2002bc, 0x3e697b90, 0x3f19a280, 0x3ec93b6c, 0x3e44b508, 0x3ea13342,
      0x3c90ad40, 0x3ed610d2, 0x3e87b876, 0x3eda1da8, 0x3f64ea12, 0x3c29b900,
      0x3e4cb500, 0x3e294f00, 0x3e936792, 0x3f73d94b, 0x3da75668, 0x3cd397a0,
      0x3d369470, 0x3c97c5e0, 0x3e6d00a8, 0x3f4f6edb, 0x3efa83ea, 0x3f5bad61,
      0x3f514571, 0x3ca64680, 0x3eae2800, 0x3ef8ceaa, 0x3f6370c8, 0x3f440ca1,
      0x3f7a74c5, 0x3f65d024, 0x3f1767f7, 0x3dd46c08, 0x3f6f9204, 0x3f5e1606,
      0x3f6ab19b, 0x3f1f20d7, 0x3f06952e, 0x3d5fb890, 0x3ed6ba26, 0x3f0ad89e,
      0x3e740f08, 0x3f7428e9, 0x3f1b0d20, 0x3daba2b0, 0x3e172e24, 0x3a779800,
      0x3efbf780, 0x3f0fc439, 0x3f6f80fd, 0x3ed131e4, 0x3ec6e8ba, 0x3f5cb35b,
      0x3e615610, 0x3eab48c6, 0x3f0ac3db, 0x3c813180, 0x3f6a6240, 0x3f58f98e,
      0x3ea44696, 0x3edb1334, 0x3f47e4a5, 0x3f4de0e1, 0x3edbf24c, 0x3d544610,
      0x3f2426aa, 0x3cd486e0, 0x3ed9ccd4, 0x3d640b00, 0x3f620ff1, 0x3e9f4fb4,
      0x3f6bc59f, 0x3f74ab39, 0x3e0ab1f0, 0x3f11b2b5, 0x3f65e216, 0x3eb0b7c6,
      0x3f2c35dc, 0x3f5321d2, 0x3f3d4365, 0x3f6b1eee, 0x3f38c399, 0x3ec4deae,
      0x3f0f7454, 0x3e91c52a, 0x3f7bf797, 0x3f5aa345, 0x3e62886c, 0x3f3a35d4,
      0x3ecbf950, 0x3f214f6e, 0x3f4c49e8, 0x3f322e38, 0x3e3caa44, 0x3ea79586,
      0x3f360305, 0x3f5223fc, 0x3f085114, 0x3f10d3bb, 0x3e8da57a, 0x3ed097ac,
      0x3ede38c8, 0x3e898dfa, 0x3ed10910, 0x3f33ba5c, 0x3d8475d0, 0x3f4bf145,
      0x3daf4e18, 0x3e4975f8, 0x3d73ea90, 0x3f6b831d, 0x3d4b8f60, 0x3f39970d,
      0x3f0a2217, 0x3e2c9714, 0x3f62858e, 0x3d02c820, 0x3f2f9d59, 0x3eae793e,
      0x3f6cc09d, 0x3c3a98c0, 0x3f5c15c0, 0x3f17a57d, 0x3e1401d8, 0x3f2372f3,
      0x3f599774, 0x3f7d5836, 0x3f1a7a50, 0x3e487078, 0x3eda58ec, 0x3f5ed2cf,
      0x3eea7eba, 0x3f023c54, 0x3eea7fea, 0x3f6f4922, 0x3f1f446b, 0x3e9ed28a,
      0x3e38c488, 0x3edde92a, 0x3f3a1bd0, 0x3f41363a, 0x3f469bcb, 0x3eed8668,
      0x3e9badae, 0x3f0b67fc, 0x3e42a6cc, 0x3e2b63c4, 0x3f68f9c6, 0x3ba62880,
      0x3ef6f17a, 0x3e035950, 0x3f513d0f, 0x3e91a298, 0x3d7e5440, 0x3f439366,
      0x3e98604c, 0x3f6137ec, 0x3dc44c58, 0x3f199655, 0x3e90b53c, 0x3f3cf9df,
      0x3e8cdb66, 0x3c7f9cc0, 0x3f7e21b0, 0x3ec2df98, 0x3e3c6a20, 0x3f682873,
      0x3f492deb, 0x3f63bda8, 0x3e7dae2c, 0x3d8e8280, 0x3f18eb3c, 0x3f798158,
      0x3d366370, 0x3e3c25dc, 0x3e79ed0c, 0x3e8a14ae, 0x3d23e8f0, 0x3f442f56,
      0x3ee36cc0, 0x3f375a0e, 0x3f0cfd62, 0x3f41deca, 0x3f371e13, 0x3f7df3b9,
      0x3f641d47, 0x3eff708a, 0x3eb5ac70, 0x3eeb1ea0, 0x3f2dd090, 0x3f5b1711,
      0x3f2c9c50, 0x3c9b5d00, 0x3f21ce76, 0x3f0b3455, 0x3e721780, 0x3debd3d8,
      0x3ee69ad0, 0x3f6a3e92, 0x3e44370c, 0x3cd70b20, 0x3f67e35c, 0x3ebec97c,
      0x3f5f8319, 0x3f4b96a2, 0x3eaf1298, 0x3e4e204c, 0x3eaf0214, 0x3d2d3440,
      0x3e56d6b8, 0x3f217618, 0x3f047754, 0x3f3ce9e2, 0x3f791eba, 0x3f3da46b,
      0x3f0cf931, 0x3f3b2bca, 0x3eaebb44, 0x3f2c2f97, 0x3f7a789d, 0x3f498bd1,
      0x3f7e30b0, 0x3d0fb0e0, 0x3ebad65a, 0x3e0cd240, 0x3e18fb34, 0x3e99995a,
      0x3f031f8e, 0x3f2074ca, 0x3efe0d1e, 0x3e7ba5b0, 0x3f789ea7, 0x3e81216c,
      0x3d150440, 0x3f31c10f, 0x3f1c1ff7, 0x3e7aaa80, 0x3f5d13e3, 0x3f7562bf,
      0x3e002ef8, 0x3f548e97, 0x3f43d011, 0x3f6cbe41, 0x3e1dd5c4, 0x3f0455e9,
      0x3d7a8840, 0x3bffb480, 0x3f4a53cc, 0x3d6d95a0, 0x3dbf4398, 0x3e1f5d18,
      0x3e73f26c, 0x3efc6682, 0x3f49593a, 0x3f20560d, 0x3e6684ac, 0x3f1c2fec,
      0x3f2928fd, 0x3f62c2be, 0x3e86282e, 0x3f10711e, 0x3f72c1ba, 0x3cb91760,
      0x3e891758, 0x3e05b87c, 0x3e32b134, 0x3f5300a3, 0x3f72a7eb, 0x3ce5daa0,
      0x3f05dc44, 0x3e389e00, 0x3ea3a2c6, 0x3f323be9, 0x3f41aa86, 0x3e56f618,
      0x3c152180, 0x3f74b62c, 0x3d6b5cf0, 0x3d80c960, 0x3f704c6a, 0x3d903960,
      0x3ece4754, 0x3f1b0eb6, 0x3dd1c908, 0x3d4aa1f0, 0x3e6c8818, 0x3e7bac78,
      0x3f31e5fb, 0x3ecae886, 0x3f3fb539, 0x3f40c8a9, 0x3f21b7b3, 0x3f3f1634,
      0x3f35d098, 0x3e653504, 0x3e4b3124, 0x3c0bee00, 0x3e768770, 0x3f4c8996,
      0x3f6bad38, 0x3f14dcf6, 0x3ef6198e, 0x3e21a758, 0x3e981052, 0x3ea62be0,
      0x3f3b0534, 0x3f11f168, 0x3f487ab5, 0x3e47f804, 0x3f26b6cd, 0x3e791c58,
      0x3e00e8bc, 0x3f6899fd, 0x3f6541f4, 0x3d5b9fe0, 0x3e389b90, 0x3f2b13aa,
      0x3f4d9730, 0x3eca8a50, 0x3f278121, 0x3f39e2ff, 0x3f3e4aa6, 0x3f400835,
      0x3e130ddc, 0x3f35d27b, 0x3ecc9640, 0x3f69018f, 0x3f0082f1, 0x3f452131,
      0x3ed5793e, 0x3f419e18, 0x3eb9201e, 0x3e2f867c, 0x3dc4b4a0, 0x3d9042d0,
      0x3f420e18, 0x3f2a985f, 0x3f5b1303, 0x3ee870fc, 0x3eaaa0e2, 0x3f465da8,
      0x3f67e8d4, 0x3e31306c, 0x3e974a36, 0x3e9aa284, 0x3eae3e36, 0x3e56dfdc,
      0x3f1246ef, 0x3f2efa26, 0x3f47c8a5, 0x3ef8259a, 0x3f1383d5, 0x3f68cb4d,
      0x3f33b5ed, 0x3ed86a62, 0x3f41b36e, 0x3f115e00, 0x3f3af463, 0x3ee41e10,
      0x3f473c66, 0x3e261b4c, 0x3e8d4fb2, 0x3f159f00, 0x3e508c38, 0x3f32392f,
      0x3f7e21b6, 0x3ea3b734, 0x3f005931, 0x3d6700a0, 0x3e51df88, 0x3ef213fa,
      0x3f5f7884, 0x3ebac918, 0x3f2e4c2f, 0x3f25a310, 0x3e95f6e4, 0x3f384f8c,
      0x3ea1734c, 0x3ed012aa, 0x3ddd03c8, 0x3ede75f0, 0x3f441bf2, 0x3e8a57b0,
      0x3f281be8, 0x3eec1672, 0x3f432bee, 0x3f2b360b, 0x3f4c1fe8, 0x3e8a23de,
      0x3d45dba0, 0x3f65ef97, 0x3f3f60db, 0x3dca8de0, 0x3c366400, 0x3f5fe7ff,
  };

  // 15,10,6,7
  uint32_t kernel_vals[] = {
      0x3cdb73c2, 0x3c526030, 0xbc26191a, 0xbcb1ae92, 0xbc7bf83c, 0x3c2f1530,
      0xbb4ac0f0, 0x3c97fb2a, 0x3c31873c, 0x3c980e58, 0x3cd67cbe, 0x3c31b9c0,
      0xbc834918, 0xbd06e50d, 0x3cb9e752, 0xbc363616, 0x3cb73d78, 0x3c47f97c,
      0xbc82b898, 0x3c2760f0, 0x3cea7da8, 0x3c3b0884, 0x3cc7c39c, 0x3ce3a586,
      0xbb3c9b58, 0xbcab4bb2, 0xbc0cf1da, 0xbd009685, 0x3b513b20, 0x3d007089,
      0x3b707a20, 0x3d01ff61, 0xbaf24ae0, 0x3c9255d2, 0xbba7a480, 0x3bef72b8,
      0x3bb80a70, 0x3a3aeac0, 0x3cae4cfe, 0x3b9f1930, 0xbbe18824, 0xbcb4bf92,
      0xbd03344e, 0xbc1e3f8a, 0xbce4b6a9, 0x3be7c068, 0xbbe5ddf8, 0x3bd8db80,
      0x3cf41fde, 0x3c584bac, 0x3ce409a0, 0xbcbbbaa3, 0xbc93c292, 0xbc35fb5e,
      0x3ad80ba0, 0x3c9e5f78, 0x3c9dd550, 0x3cad6cec, 0xbb59bac0, 0x3c4fa608,
      0xbcd34638, 0xbc321fce, 0x3ca213be, 0x3cf9969a, 0x3cfd9836, 0xbbef3644,
      0xbd06dde2, 0x3c96f69c, 0x3cd102ec, 0xbc7d8c30, 0xbd0836d6, 0x3c0fc210,
      0xba84f760, 0xbbdf353c, 0xbb44f878, 0xbb84fa68, 0x3cd2f472, 0xbc5b2b5e,
      0x3ce85990, 0x3cf00b22, 0xbc094c40, 0x3b4ef580, 0x3ce62008, 0xbb48f8f0,
      0xbc6ffb34, 0x3cdec706, 0xbb674aa8, 0xbc8525c7, 0xbb7547c0, 0xbcff6214,
      0xbade41a0, 0x3c3b0a78, 0xbcdd38fe, 0x3c805196, 0x3cb22bbe, 0x3d086e91,
      0x3b0b5450, 0x3b3921e0, 0xbcde18f8, 0x3ce8a88e, 0xbc889949, 0xbc40e830,
      0x3cf78466, 0xb9423100, 0xbd0224d4, 0x3d0861bf, 0x3cc13370, 0x3d018ce1,
      0xbce7a2ee, 0x3c9463c2, 0x3bcf3c78, 0xbb9c0de0, 0x3ca52d6e, 0xbb45ea00,
      0xbccccf4c, 0x3c9e357a, 0xbce0f487, 0xbcf80901, 0xbc67b1da, 0x3cd12a12,
      0xbd033d54, 0x3c9cba84, 0xbcfb5fc1, 0x3be3fd08, 0xbccba260, 0x3b1da040,
      0xbcc7f0c5, 0x3c928254, 0x3cab85ee, 0x3c954e84, 0x3cc7f242, 0xbcb3b7b6,
      0x3c25e534, 0x3ccf0e5a, 0xbc91e9c9, 0xbc73e8ca, 0x3c9e0676, 0x3cc00de2,
      0x3c098bb0, 0x3c8a0e6e, 0xbb8acaac, 0x3cdfa214, 0xbc354dd6, 0xbc6ba928,
      0x3cdf1b8a, 0x3caf2c38, 0x3c249960, 0x3ce113fa, 0x3c612318, 0xbb61fb20,
      0xbccfe7f0, 0xbbd52a78, 0xbbb25e10, 0xbc956ee5, 0xbca6dd38, 0xbca1c894,
      0xbcf8c96e, 0x3cfaba02, 0xbcda72e9, 0x3b746490, 0xbc54cdce, 0x3d027979,
      0xbccb1618, 0xbce43af6, 0xbc5df28a, 0xbb6369f0, 0xbba6cd08, 0x3ca5150c,
      0xbcc54076, 0x3cae99b6, 0x3cf34a76, 0x3cc7246e, 0xbaecf3a0, 0x3cf48276,
      0x3c70c220, 0x3b89cd00, 0xba265080, 0x3cc80524, 0xbcd12c7a, 0x3c3b7b84,
      0x3cab054c, 0xbce9b3d0, 0xbcdf41a3, 0xbbedc0b4, 0xbcdc5af2, 0x3cd5c4a0,
      0xbbce6bb4, 0x3cc4958e, 0x3be1da90, 0x3bfacb08, 0xbc99371a, 0xbca37ee5,
      0xbb0455f0, 0xbc363b68, 0xbc92b249, 0xbc551568, 0xbc37cc92, 0xbc72ba9a,
      0xbcd2b25a, 0x3bd20c48, 0x3c80ff92, 0x3bdfffd0, 0x38c8a600, 0x3c070c94,
      0xbcdd1e81, 0xbaf0de40, 0x3cec0380, 0x3cd32e18, 0x3cd5deda, 0xbbdc0634,
      0xbc9f18a5, 0x3cd5b556, 0xba5fe580, 0xbbc1fa9c, 0xbc2c6ac4, 0xbb775968,
      0xbc9bee18, 0x3cf68936, 0xbc29e44a, 0xbc05fd9a, 0x3ca907c0, 0x3c823938,
      0xbbbbe134, 0x379fd800, 0x3ce5dd14, 0x3c3eff30, 0x3c86194c, 0x3cc253e2,
      0xbd067b50, 0xbc9feee5, 0xbb145248, 0x3d06162f, 0xbc848272, 0xbbf61da4,
      0x3cd025f6, 0xbca65985, 0x3c935de4, 0xbc761240, 0xbce2224c, 0xbcb8c0e7,
      0xbcca2f36, 0xbcc5dfce, 0x3c06b534, 0x3c631320, 0x3c9920ce, 0xbc37217c,
      0xbceb8db0, 0xbc738646, 0xbcde5e2e, 0xbcd6a0b4, 0x3cc074b6, 0xbc8e59cd,
      0x3c31f2b0, 0x3b113410, 0xbca06300, 0x3ca48f16, 0x3ca885fe, 0xbc827b12,
      0xbc448e00, 0xbba60d44, 0xbcbce2c0, 0x3c71c9e4, 0x3d039d41, 0x3cb6e8d4,
      0x3c680c30, 0xbcdb4bf6, 0x3b6a1710, 0xbbab9d34, 0x3cff1eae, 0xbc8b0ba7,
      0x3ca95a46, 0x3bea8810, 0x3cda3dd8, 0xbbd334a4, 0xbbc88d80, 0xbcfe6a7d,
      0x3c0d0b30, 0x3ca7866a, 0xbc298abc, 0x3cf81c36, 0x3c20ea54, 0x3ba54a70,
      0xbce0ad30, 0xbacf5e80, 0x3b4bff10, 0xbc6f5fe2, 0x3cfbeeb6, 0xbc14f49e,
      0x3b2fc1f0, 0xbac974a0, 0xbc43e8a8, 0xbb0e4de0, 0xbd04e6ea, 0xbd042bef,
      0x3cd31c08, 0xbaa9caa0, 0x3c1df88c, 0x3ce01630, 0x3bd3ca30, 0xbcc6d7b2,
      0xb979e500, 0x3c91ef76, 0xbc676540, 0x3c100668, 0xbbffb008, 0xbba2f1e0,
      0xbd041ee4, 0x3c8123ee, 0xbcbbd0b6, 0x3c26b13c, 0xbb88d7cc, 0x3c4248bc,
      0x3c332370, 0x3c8c954e, 0xbcc31360, 0xba90d1c0, 0x3c984162, 0x3ca679be,
      0xbbb18d88, 0x3d062691, 0x3b859268, 0xbb056a10, 0xbc6d34bc, 0x3d017cd3,
      0x3c800880, 0xbc9f02c7, 0x3ce1f8ce, 0xbca4cb8b, 0x3c7fb720, 0x3cce7622,
      0x3c96871a, 0x3b472a20, 0x3c68f488, 0x3bc68a60, 0xbcd24547, 0xbab6da00,
      0x3b0f53d0, 0xbbd7a9bc, 0x3ccd74f4, 0x3cbe3b2c, 0x3c600240, 0xb8fcb800,
      0x3ce7968e, 0x3cd7a0f4, 0xbcf17852, 0xbc392c00, 0x3cebb946, 0x3bc06630,
      0xbcc378fe, 0xbcacecc7, 0x3cee1276, 0x3bbfc718, 0xbc9f908e, 0xbc686c00,
      0x3caee9a0, 0x3c18c964, 0x3cff88ee, 0xbcbb91a7, 0xbb34c848, 0x3c819278,
      0x3ca7ced8, 0xbc109352, 0x3cf20012, 0x3cdb2506, 0x3b1f4d50, 0xbc36af4a,
      0xbcd04c8c, 0xbcde58ee, 0xbd087c84, 0xbc0a0b8e, 0xbc758822, 0xbbd7ae58,
      0xbcebff76, 0x3b862a48, 0xbbfc8180, 0xbc707e16, 0x3c86bc88, 0x3ce3ea2a,
      0x3b38ab00, 0x3cfe37a6, 0xbca4a3d2, 0x3ca1c802, 0xbc810747, 0xbcceb178,
      0xbb2c6dc0, 0x3ad1d640, 0x3cf32c3e, 0x3b69acc0, 0x3d018aff, 0xbc29adb4,
      0xbc2bacc0, 0xbc985f25, 0x3cb9b5ce, 0xbce45252, 0x3d079ecd, 0xbd00b00b,
      0x3bd9a998, 0xbcfacd52, 0x3cd7b432, 0x3c632aa4, 0xbb7c1c20, 0x3bc66918,
      0x3c72de5c, 0x3c7f6f48, 0x3c23d93c, 0xbcbf6e94, 0xbca65718, 0xbc62ad46,
      0x3c82db7a, 0x3cc31a50, 0x3cc49d1c, 0x3c94d1a2, 0x3b15b110, 0x3c844dd6,
      0xbba9761c, 0x3ca76c5e, 0x3d00a875, 0xbb1ea078, 0xbc56518a, 0xbcfd8b9f,
      0xbb665798, 0x3cbbba3c, 0x3bf89fe8, 0xbcfed0d0, 0xbc633e6c, 0xbc3980a2,
      0x3d06fdfd, 0x3cceb832, 0x3c994c32, 0x3c829982, 0x3c092dbc, 0xbc2b6c44,
      0x3cc4d2ce, 0xbcb16cc0, 0xbbeec568, 0x3cd72d26, 0xbc503d92, 0xbabf45a0,
      0x3c66d204, 0x3cf23036, 0x3c10ae80, 0x3c5560c0, 0x3c0b9c24, 0xbcef18d6,
      0xbcbbede3, 0x3ca1fd72, 0xbb8bb6c4, 0x3c840274, 0x3c8cd982, 0x3bf27fc8,
      0x3cb5b230, 0xbc4e3f0a, 0x3c859f14, 0x3ccb5b14, 0x3cb55d6c, 0xbcc16dbc,
      0x3c0ab0f0, 0x3cbc3fd8, 0xbc261bde, 0xbd03a753, 0x3cc23830, 0xbbdbae4c,
      0xbd03dd28, 0xbcd040c3, 0x3c985ef0, 0xbcdff2a1, 0x3c85f30e, 0x3ccf4274,
      0x3cf8077e, 0x3cada580, 0xbce5017a, 0x3cae75ae, 0x3a3cda80, 0x3aaaa060,
      0xbbfe0290, 0xbcacd3a9, 0x3c89fadc, 0xbc980c65, 0xbbd7c1bc, 0xbce83f0e,
      0xbb0a9000, 0x3c1e56dc, 0x3ce86290, 0x3c5d3720, 0x3c4cabec, 0x3c2ffcc4,
      0x3bf320e0, 0x3c3ca4e4, 0x3ce2190a, 0xbcf6bf45, 0xbcc675a9, 0xbcff935a,
      0xb90d3700, 0x3cdb6000, 0xbcd76f67, 0xbceba5da, 0x3cd9d4ca, 0xbbc4663c,
      0x3c1b1280, 0x3ae7d280, 0x3cc33eec, 0x37476000, 0x3c1e3270, 0x3cd543f8,
      0x3d0788f1, 0x3c18f048, 0xbb9aa8e8, 0x3cdab5c2, 0x3c9daad8, 0x3d052e2f,
      0x3c8ada02, 0x3c4b6438, 0xbcb6def6, 0x3bf10dd8, 0x3ce2a0f4, 0xbcb73638,
      0xbc9d7076, 0xbb67a998, 0xb9fda680, 0xbb4a26e0, 0xbcb7e2fa, 0xbc80fffc,
      0x3d06ab9f, 0xbcbd2ea3, 0x3bcc1dd8, 0x3cc49b38, 0xbbcf2934, 0x3c9b12de,
      0x3b131c10, 0xbcef8ed8, 0x3ce385b8, 0xbc33e434, 0xbce2f021, 0xbc39a216,
      0x3c016178, 0xbc9ebf52, 0x3bea4878, 0xbb81a378, 0x3cc40aaa, 0x3c7a4b30,
      0x3c6a541c, 0xbc4d1b96, 0x3c935182, 0x3ab8a8c0, 0x3cc4d618, 0x3cf00b96,
      0x3c58767c, 0xbc15fc30, 0xbcdf6170, 0x3c4d03dc, 0x3c2cbd74, 0x3ce0de8e,
      0xbcff0603, 0xbc4babb4, 0x3cf1d252, 0x3c8186f8, 0x3cbcc842, 0x3cf2abea,
      0x3cdbdcf6, 0x3c081b9c, 0xbb1efe88, 0xbca0ef98, 0x3cddd9fe, 0xbb239d20,
      0xbbc9f978, 0x3ca4763a, 0xbc6dddd2, 0x3cec2e36, 0xbc9d0eba, 0x3cea7d12,
      0x3c5e268c, 0x3cb285ca, 0xbbca0ebc, 0xbc5ef63c, 0xbd00b5aa, 0xbcbdfcf6,
      0x3caa6e92, 0x3c62437c, 0x3c240b8c, 0x3c5fd878, 0x3c383ff8, 0x3cc0b35e,
      0xbbfa7960, 0xbc87ecd8, 0xbcf85c98, 0xbbbd76a4, 0xbc7274d2, 0xbcb3d669,
      0xbce9ca70, 0x3ce726ae, 0xbc8eb800, 0xbcd6179a, 0x3c201370, 0xbc0ac07c,
      0xbc29858e, 0xbccc8e76, 0x3ce08baa, 0x3c2681b0, 0x3d01ec3b, 0xbcb64a34,
      0xbbc730f8, 0x3cbda8b2, 0x3c675478, 0xbc39a268, 0x3bb67608, 0xbcaea76c,
      0x3c7ec200, 0xbb0391f0, 0xbcb73d32, 0xbd01021b, 0x3cf52186, 0xbc66134a,
      0x3b621ff0, 0xbbd6c370, 0x3bbd1c30, 0xbcfe6af4, 0x3cedab66, 0xbcc48a90,
      0xbcf79fc5, 0xbb130f30, 0xbce03b0c, 0x3c8403b6, 0xbb8690f0, 0xbaf30d80,
      0xbc1b3292, 0x3c8d792a, 0x3d037777, 0xbd055679, 0x3b9a3a08, 0xbcedfd94,
      0x3bfdc8d8, 0xbc9d6c6b, 0xbd05151d, 0x3cea4bce, 0x3980a900, 0xbcf3e49a,
      0xbce8b7b6, 0x3c584198, 0x3bf91a08, 0x3cd96e1a, 0xbcaf1b90, 0xbcaf24fc,
      0x3c3762ac, 0x3d06627b, 0xbb3aea88, 0x3cd10bfa, 0xbd07ca4c, 0x3c9ecb7a,
      0xbd04a7ca, 0xbce6d4e1, 0x3cff3e62, 0xbc1ac1e2, 0xbc9354e7, 0x3c5bc734,
      0xbc22977c, 0xbd00844e, 0xbce5b0e7, 0xbccb86ba, 0x3c8098de, 0xbc24e4d6,
      0x3cebde02, 0xbb733238, 0x3c779648, 0x3cf6bfba, 0x3cbad862, 0x3b7b7600,
      0xbca75200, 0x3c45bc48, 0xbcab4989, 0x3c700fa4, 0xbcc7729e, 0x3d077b59,
      0x3cbbcda4, 0xbcd4a4c3, 0xba5fdbc0, 0x3b123de0, 0xbcddc521, 0x3c8ae7e0,
      0xbd0109a5, 0x3ca10d04, 0xbc64955e, 0xbcbb8752, 0x3c3a4384, 0xbb97972c,
      0x3c510624, 0xbc1a2612, 0xbc5e2b52, 0x3cc81ec2, 0x3d00660b, 0xbc219604,
      0xbb9cf3bc, 0x3bbd5930, 0xbbc07908, 0xbcda499a, 0x3cf15722, 0x3c7f94dc,
      0xbc785ada, 0xbc66224e, 0x3cfbf0da, 0xbcb7833c, 0xbcbf6432, 0x3cd409f6,
      0x3d06735d, 0x3c9db84e, 0xbc381bca, 0x3ca26d34, 0x3c371554, 0xbc192fa6,
      0xbcfc9a3d, 0x3ce755b6, 0x3ca6a260, 0xbc5f1ad2, 0xbcea7852, 0xbc82f598,
      0x3ca71ffc, 0x3bf22490, 0x3cc5f3da, 0xbcf65cda, 0x3ba4c700, 0xbc26825e,
      0xbab1c220, 0xbcf4e716, 0x3c67d8d0, 0xbc6b8b38, 0x3ca6af32, 0x3cb24316,
      0x3cff1a36, 0x3bf3a160, 0x3c221cb4, 0x3ce98b3c, 0x3ca64ea0, 0xbcd5567e,
      0x3b675320, 0xbce106ff, 0x3be95578, 0x3ccf83fc, 0xbcf785a5, 0xbb8a2578,
      0x3cd41e8c, 0xbbd38060, 0xbcec5e89, 0xbbbe17d4, 0xbc06ae16, 0x3c6859a4,
      0x3d024923, 0xbcf7403d, 0xbc723b9a, 0xbc6225a2, 0xbcec07df, 0xbc7dafe8,
      0xb9adad00, 0xbce63365, 0xbcac3d65, 0x3d015505, 0xbc38a204, 0x3c15a10c,
      0x3cf4a96e, 0xbcbaf058, 0xbb28b930, 0xbc5b50f8, 0x3c001f00, 0xbcc1f280,
      0x3d03cd45, 0xba18dc40, 0x3c26efa8, 0xb8944600, 0xbcb1c058, 0xbc871178,
      0x3bd5bf48, 0xbc686996, 0xbc35c234, 0xbced4fb4, 0x3c47efc0, 0x3bb829e8,
      0x3cdf2256, 0xbca60f07, 0xbbe60188, 0xba88d7a0, 0xbcd30e89, 0x3b44a3d0,
      0x3c2d328c, 0x3cb49e70, 0x3cb94a8c, 0xbc34637c, 0x3cd14968, 0x3c17e430,
      0xbc55323c, 0x3ce435b4, 0xbb14a148, 0x3cd41972, 0x3d05b025, 0x3d0229b3,
      0x3c288b84, 0xbcd0770c, 0xbc49c738, 0x3d015ea5, 0x3cfd81da, 0xbd07777b,
      0x3cef3a1e, 0x3cee03e8, 0x3cd07716, 0xbce633ff, 0x3c993754, 0x3c0829dc,
      0x3cc072ca, 0x37071000, 0xbc49d01e, 0x3ca89022, 0x3bb9ce68, 0xbb138a00,
      0xbb5d23c0, 0xbd027647, 0x3c9f8134, 0x3ce12816, 0x3d0704cd, 0x3c067b2c,
      0x3c938550, 0xbc9bc0a0, 0x3ca180ae, 0xbcf2cbc7, 0xbc22144a, 0x3cac96a4,
      0x3cafc46c, 0xbd04b8a2, 0xbca675b8, 0xbcfbf845, 0x3c276504, 0x3ce398d0,
      0x3ca7a224, 0xbbdff44c, 0x3c4d8db0, 0x3b8c1510, 0xbb1fe988, 0xbc67048a,
      0x3b700fd0, 0xbca368ab, 0x3c5e92ec, 0x3cebcf46, 0x3cc91a6e, 0x3c7a446c,
      0x3cde3ca6, 0x3cfe02e6, 0xbcdab10e, 0xbc3f8000, 0x3b91f660, 0xbbc1d508,
      0x3bdcfc30, 0xbbe94e80, 0x3c0086f8, 0xbd0611d8, 0xbc9450a9, 0x3c98d27e,
      0x3ca10f0e, 0xbc0b78bc, 0x3cf39c8a, 0xbcaf7d56, 0x3be738d0, 0xbc11fcda,
      0x3d04a341, 0xbc070e5e, 0x3b02dbe0, 0x3c67cd80, 0x3ca63e0a, 0x3b355fc0,
      0x3cd4b738, 0x3c0d9bbc, 0x3d06a273, 0x3bb0efd8, 0x3cf7bfae, 0x3c9f7218,
      0xbc48e062, 0x3c76a55c, 0x3c907bce, 0x3cc87c44, 0x3d029e47, 0xbc3f6b16,
      0x3cddab66, 0x3c710c24, 0xbbc3fde0, 0x3cdd1a02, 0x3cc357b6, 0xb977d900,
      0xbbe3b89c, 0xbcbec37e, 0x3d028bbf, 0xbb4e3c20, 0x3d076067, 0x3cffd22e,
      0xbcb0b545, 0x3ca03178, 0x3cd86462, 0x3c729b88, 0xbcc4cd5e, 0xbceee7d0,
      0xbb7d0d48, 0x3cd2f1dc, 0xbc86cd2b, 0xbb1e1868, 0xbc840023, 0xbcce608c,
      0x3b8a11d8, 0x3a095040, 0xbb8e70f8, 0xbca76038, 0x3c69b384, 0xbc2140b0,
      0x3c5f0e00, 0xbcd8b59e, 0xbc243cda, 0x3c6cac28, 0xbba19788, 0xb9ce1380,
      0x3ba9e828, 0xbca20ed2, 0xbd03ba43, 0xbcd568d4, 0x3ca1d658, 0x3cc785cc,
      0x3d02cf0d, 0x3cd45810, 0xbcb7acde, 0x3cd56f94, 0x3ccf021c, 0xbbcf0df8,
      0x3c13d4c8, 0xbbb87490, 0xbc182be6, 0xbb3aa1f0, 0xbb78c010, 0x3bf0c898,
      0xbcc94d5a, 0x3cea3908, 0x3cdc9df4, 0x3d0298ff, 0x3d062153, 0x3cfdc31e,
      0x3cc8462e, 0x3c4160a8, 0x3c154fc8, 0x3cd95e3c, 0xbc54367c, 0x3c730b1c,
      0xbc334584, 0xbb377010, 0xbc869a6d, 0xba21e1c0, 0x3b61d0d0, 0x3c523b9c,
      0xbcd9e6da, 0x3cc05fe8, 0xbcaf60e9, 0x3bbc00e8, 0xbb52ee48, 0x3c074ce4,
      0x3ce45c50, 0xbc8e6eab, 0x3aa5b0a0, 0xbbdeda60, 0xbb09b6f0, 0x3ce01e20,
      0x3c18bea4, 0xbb3013f0, 0x3cbcca08, 0x399cfd00, 0xbd00b20b, 0xbc87ddfa,
      0x3c3e5648, 0xbb8b2578, 0xbcb8e974, 0xbb98cab4, 0x3ca893ca, 0x3c60ff50,
      0xbc793762, 0xbc3eebd6, 0xbce48194, 0x3cb074f8, 0xbcdd5bfa, 0x3c93372a,
      0x3d04f033, 0xb96f5b00, 0x3bc81fd8, 0x3b93c8a0, 0x3c8c7340, 0xbc384022,
      0x3ae723e0, 0x3beb9540, 0x3cf7406a, 0x3c91a4e6, 0x3b8861c0, 0x3b126200,
      0x3b825998, 0xbcbc6bf6, 0xbc83fe1e, 0x3ccf744a, 0x3ccb8cb4, 0xbb96ba08,
      0x3be4d728, 0xba4a9dc0, 0xbcfe37f6, 0xbcf9004c, 0xbce94c07, 0xbc3b98bc,
      0xbc81edcb, 0xbce48a6e, 0x3c208ff0, 0x3c76d3ac, 0x3cbe305e, 0x3c6d8560,
      0xbce826ee, 0xbcfd777f, 0x3cda06ca, 0xbb5a4498, 0xbcf19732, 0x3c193cc0,
      0x3c5eb010, 0x3c5d80d8, 0x3c488e00, 0x3cf148d2, 0xbcf9ee85, 0x3c38741c,
      0xbbcf39c4, 0xbce251df, 0x3c876fe4, 0x3c44f844, 0xbc978e98, 0xbc6054a2,
      0xba43f440, 0x3c64ab2c, 0x3d0473ad, 0x3c97e1bc, 0xba3396c0, 0xbc370752,
      0x3ca3df36, 0xbcc8f15e, 0xbd051a4e, 0xbb3d8468, 0xbab439c0, 0x3c8583d8,
      0x3c47b868, 0xbc02b54e, 0x3c026d14, 0x3c8e1818, 0xbd02c07b, 0xbc4a8efc,
      0x3c85949c, 0x3badd498, 0x3a1f3040, 0xbc20b43c, 0x3cd6d7de, 0xbc5a4480,
      0x3c358af8, 0xbccec1da, 0xbbea6f68, 0x3cde5058, 0xbba4149c, 0x3c338fc8,
      0xba8371c0, 0x3ccb02ec, 0xbb051390, 0xbc4385bc, 0x3ce366ac, 0x37e63000,
      0xbc98d389, 0xbc88ae3a, 0x3a609d40, 0x3bf81c88, 0x3cb28efc, 0x3cdcec78,
      0x3c232060, 0x3cac4102, 0x3c90d24c, 0x3ca9b482, 0xbcd228da, 0xbb5d5f20,
      0xbc1a980e, 0xbc3a6640, 0xbc8951ab, 0x3cd96a1c, 0xbce888bc, 0x3bf4ebe8,
      0xbc1164c0, 0xbc5dfe38, 0xbbd0bbc4, 0x3c0d117c, 0xbc82f2fa, 0xbcc73cbc,
      0x3c6a1c30, 0xbceadbf2, 0x3ce19188, 0xba129880, 0xbc0a8d4e, 0xbc72873c,
      0xbca63de2, 0x3cfe3336, 0xbb598810, 0x3cbe8a0c, 0xbcb0355e, 0xbd032fb5,
      0xbcbba729, 0xbb2c3d78, 0x3c6f677c, 0x3b687420, 0x3ca18b34, 0x3ca06c76,
      0xbc0c9fac, 0x3c106838, 0x3a9f6720, 0xbce0f62c, 0xbc9897b2, 0xbc99cdd8,
      0x3bce3950, 0xbc5f8006, 0xbace8260, 0xba1e0b00, 0xbc276174, 0x3c9eda36,
      0x3cd0f708, 0xbc3d0ef4, 0xbcaba8c0, 0xbb0d3ad0, 0xbbf87a70, 0x3c5a1e40,
      0xbc819ea3, 0x3c93c35e, 0xbb8936e0, 0x3cc0ec64, 0x3c638138, 0x3c85e4bc,
      0xbbb3bf80, 0x3cae71ec, 0x3c5457a8, 0xbcc25292, 0xbce3c265, 0xbcac987a,
      0x3d0473f7, 0x3cc8f57e, 0x3c7c3fe0, 0xbcfa8a47, 0x38283400, 0xbb2c6a78,
      0x3c84591c, 0x3c92d74c, 0xbc83c58b, 0x3ceba726, 0x3bf58360, 0xbc1a0ab4,
      0x3cf38b7a, 0x3d00757d, 0x3c0a1d50, 0xbc7236e8, 0x3cdf5ef6, 0x3c9b80b4,
      0xbc756d7c, 0x3d07b20b, 0x3cb913e2, 0xbb54d920, 0x3bf79af8, 0x3d06fc93,
      0xbc80b485, 0xbc0fda62, 0xbc14574a, 0xbb2009f0, 0x3cca5496, 0xbcff7647,
      0x3cdec87a, 0xbb6e7f78, 0xbce2a198, 0xbb8189e8, 0x3b288340, 0xbbd2e944,
      0xbce2ca74, 0xbcccf7d0, 0x3cf8fb9e, 0xbb875f78, 0xbc15be56, 0x3cdefde0,
      0x3cba7f7e, 0xbc4207de, 0xbb8099cc, 0x3bff01f0, 0xbcf34298, 0x3c2e1740,
      0xbce95f9f, 0x3d04936f, 0xbc4fb03c, 0xbca6a58e, 0x3cc579c4, 0x3b3af630,
      0xbcf3e7d6, 0xbadcffe0, 0xbce1ff9c, 0xbc29049e, 0xbbd78180, 0x39b82e80,
      0x3c6c76c8, 0xbc22dc9a, 0x3ceb6498, 0xbac9cf40, 0xbc7c4870, 0xbc102e96,
      0xbcfee66c, 0xbce39c49, 0xbbec429c, 0xbafc4ca0, 0x3c624a9c, 0xbb940e9c,
      0xbcf671e1, 0xbc6023fc, 0x3c96c674, 0x3ccdb124, 0x3b560c10, 0xbc6148bc,
      0xbcf1117a, 0x3b9eace0, 0x371f5000, 0x3b1702f0, 0x3cd7ee40, 0x3a6fb5c0,
      0x3cf50782, 0x3c4fcb20, 0xbad398c0, 0xbc8bf7f6, 0xbc80ac9c, 0xbcace4a5,
      0x3b9b7e00, 0xbb1f1148, 0x3cd1152c, 0xbbdc57c4, 0x3a843cc0, 0x3ccb4a14,
      0x3c122e44, 0xbc9ec47a, 0x3ba24088, 0xba458680, 0x3ccedc74, 0xbbec7bc4,
      0x3b6b6c20, 0xbcfb1801, 0xbc22e970, 0xbd068d96, 0x3d0217df, 0x3cd3ee0e,
      0x3d080413, 0xbc91acae, 0x3c88e316, 0xbc5b5dda, 0xbbde2a88, 0x3c9130c4,
      0x3bbf0378, 0xbb799600, 0xbca85385, 0x3cb98a2a, 0xbcde983a, 0xbce53336,
      0x3cdfb316, 0x39c27d00, 0xbce8d374, 0xbbae52b4, 0xbc894a47, 0xbbad4d70,
      0x3ce7d2d0, 0xbbee31c4, 0x3c80c000, 0xbbce64f0, 0x3cd7d3ee, 0x3ce571fa,
      0xbc1c3284, 0xbcfea7b2, 0x3cb238b0, 0xbc723b5a, 0xbc2e711e, 0xbc984a05,
      0x3c2bff90, 0x3c76a230, 0xbd06b800, 0xb981e100, 0xbc98b220, 0xbcaa8182,
      0xbcc1c9fc, 0xbcab7792, 0xbccefab2, 0xbc421916, 0xbbba61b4, 0x3d03c6eb,
      0x3cb024e4, 0xbbe25c1c, 0xbc62eb0e, 0xbc524868, 0xbd05b88a, 0x3c4c6d64,
      0xbb533bf0, 0x3cf49a4a, 0xbb8d6af0, 0x3d034e95, 0x3c9656d8, 0xbc2337f4,
      0xbcbffbe3, 0x3c8977c6, 0x3c9f03f2, 0xbcfc0aee, 0xbc73092c, 0x3bd739f0,
      0x3cf7bd1e, 0x3baa3f00, 0x3d01db81, 0xbc56289e, 0x3c5f56ec, 0x3b6713b0,
      0x3d05d4dd, 0xbcf150b2, 0x3c28d864, 0x3cce871e, 0xbba182d4, 0x3b127200,
      0x3c8c9dee, 0x3b60fa40, 0xbb7fdac0, 0xbcae9f89, 0x3c1feb88, 0x3d041bc1,
      0x3a7af540, 0xbacc52c0, 0xbcf828d4, 0x3ccfce22, 0xbbffeb1c, 0x3ca072e4,
      0x3cdfe5fe, 0x3c8846d4, 0xbc409d16, 0x3c03b980, 0x3cc896ae, 0x3c082068,
      0xbc19d440, 0xbcdfe130, 0xbcf0ecb6, 0xbc8e4a14, 0xbc07444e, 0x3c301d40,
      0x3cf9087a, 0xbcbd2bd8, 0xbcd98945, 0x3bccadf8, 0xbc9ec78b, 0x3cb8d058,
      0xbd05ff5b, 0xbb119f20, 0xbc7365f0, 0xbc8a07b8, 0x3ccf3b88, 0x3b79d640,
      0xbcfd2ece, 0x3ce106ce, 0x3ce81996, 0x3be782e0, 0x3bcc7580, 0x3cc2b156,
      0xbcd8c070, 0x3c7ce6bc, 0x3c9e0cc0, 0x3cb48f3e, 0x3cc262da, 0x3b55d8e0,
      0xbcc15565, 0x39946780, 0xbc64f422, 0x3c8bbab2, 0xbc953aa7, 0x3c863780,
      0x3ca7e1ea, 0x3beac828, 0x3be755b0, 0xbc956369, 0xbbed2c78, 0xbc5e180a,
      0xbd0560a3, 0xbbb5c768, 0x3cc2d328, 0x3bf84868, 0x3c8f50d8, 0xbce6e03f,
      0xbce40316, 0x3c751da0, 0x3c456c98, 0xbb66bbd0, 0xbc88117a, 0xbce4e538,
      0xbce70252, 0x3c3fe14c, 0x3aa19bc0, 0xbc937442, 0x3c624c84, 0x3c3a0238,
      0x3ce65722, 0xbcb0e9a0, 0xbcc2ec34, 0xbcf4b6f2, 0xbd028280, 0x3b41ed50,
      0x3caba662, 0xbb82a71c, 0xbc86a9c0, 0x3b31ed80, 0x3bfe5ce0, 0x3b9bd280,
      0xba35e340, 0x3d021d1f, 0xbcb5b66c, 0xbcf9d47f, 0x3c8b17d8, 0xbce67927,
      0xbce8378c, 0x3ccc25aa, 0x3c04994c, 0x3c3ca310, 0xbbe559d8, 0x3cb0d90c,
      0xbca26047, 0xbbaadb78, 0x3bd805a0, 0x3c212190, 0xbcde54e7, 0x3c820ef2,
      0xbc75bc2c, 0x3a727900, 0xba00d540, 0xbc5fb1e8, 0xbc155d04, 0x3cb2bab0,
      0xbce88ee1, 0xbc9302fc, 0xbc1abb04, 0xbd070c5f, 0x3ca67796, 0xbc4e2a5a,
      0xbbcc1c1c, 0x3cfb92f2, 0x3cf43336, 0x3cc967dc, 0xbcf7c134, 0xbca300fe,
      0x3c0479c4, 0x3c9e093e, 0xbb933278, 0x3c592e64, 0xbb8d7570, 0xbcd99bda,
      0x3c158144, 0x3c8a6aec, 0xbc3b4e44, 0x3c0839d4, 0x3d03d73d, 0xbc7c628e,
      0x3c27d26c, 0xbc6b8fbc, 0xbcc1136e, 0xbcda85f4, 0x3cf80d3e, 0xbbe9d390,
      0xbc9bdc00, 0x3c84f3ee, 0x3b4c5510, 0x3cc32202, 0x3a160580, 0xbccf5010,
      0xbb3e7ef0, 0x3b8573e0, 0xbafc4e20, 0xbc5ef50a, 0x3b68db00, 0x3c7a38e0,
      0x3cd94edc, 0x3cebb86c, 0x3b009760, 0xbc508ec0, 0xbc93afa9, 0x3ca4c752,
      0xbccad740, 0x3bd251d8, 0xbd037efd, 0x3ce126ac, 0xbc444f00, 0x3bc8fef0,
      0xbc8773ab, 0x3c8f8a44, 0x3cdfea4a, 0x3c186c6c, 0xbca5cb49, 0x3cc193ca,
      0xbd0874ce, 0x3ca9d586, 0xbbe956ac, 0x3cfae08e, 0x3cf17372, 0xbc614856,
      0x3cfbe4da, 0xbbe03c70, 0xbcf0db56, 0xbb853690, 0x3c94ce26, 0xbc943114,
      0x3ba49e78, 0x3ce11a52, 0x3ccbe3d6, 0x3bd42b00, 0x3c8a4e0c, 0x3d03168f,
      0x3cfab03e, 0xbcb93b54, 0xbc72f3ca, 0xbc61ea52, 0x3c9b9678, 0x3c933c80,
      0xbc8fd13e, 0xbbbd9d54, 0x3ca6ba12, 0xbbd3a29c, 0x3c46c248, 0x3c7bf530,
      0xbc187684, 0xbcd3ff70, 0x3c850b4e, 0xbae7abe0, 0x3b77f920, 0xbbe86e34,
      0xbced401c, 0xbcb71340, 0x3c4243c8, 0xbd028d65, 0xbc7ee786, 0xb9b09b00,
      0x3cbf051e, 0x3babd340, 0xbccaa152, 0xbcd92201, 0xbcf7c16e, 0x3d04eb79,
      0xbc9ae42e, 0x3c95eee0, 0x3cedea06, 0xbb9dfe00, 0x3c28e33c, 0xbcc3013c,
      0x3c5503e0, 0xbcab76ba, 0x3b85bac0, 0xbcf9e0f0, 0x3c21e944, 0xbcbbc8c9,
      0x3bb96428, 0x3d00fed1, 0xbca0ed4e, 0x3ca3f8d0, 0x3a4dfc80, 0xbb8b93bc,
      0x3cb8b96e, 0x3d068f0b, 0x3cc4b868, 0xbcfb1207, 0xbd011463, 0x3c7661bc,
      0xbba73b70, 0x3d051491, 0x3c8f4ffa, 0x3cfb6062, 0x3cb4c542, 0xbcea0814,
      0x3b63ddf0, 0x3bf17388, 0x39867580, 0x3b977398, 0x3c531c4c, 0x3c7378b4,
      0xbc77a492, 0x3bfec698, 0x3cab60ec, 0xbb5ae458, 0xbb3e9aa8, 0x3bad48f0,
      0x3bb83bd0, 0xbc3d94f0, 0xbba027f8, 0xbc747aca, 0x3c38bea8, 0x3999c800,
      0xbcf83516, 0xbbca5a60, 0x3cbf2568, 0x3ce82c4a, 0xbd07cf83, 0x3cf883fa,
      0xbcb019fc, 0xbbc520e0, 0xbcec7c5c, 0x3c01a078, 0xbceaf723, 0xbca2f14b,
      0x3c581308, 0xbc97943e, 0xbbd7efb4, 0x3c2cc478, 0xbcc7b300, 0xbc9c64c9,
      0xbad26000, 0x3cd0a022, 0x3cc5f888, 0xbcf6a9f6, 0xbc80ab27, 0x3a2c1ac0,
      0x3ce5b2e8, 0x3bcaf110, 0x3b49afb0, 0xbd008b8e, 0x3ce1c3e2, 0x3cb933f0,
      0xbbc654f8, 0xbbd1dd70, 0x3ceed6e6, 0xbc92c338, 0xbbee5670, 0xbce82970,
      0x3c177538, 0xbd001ef6, 0x3c45dd78, 0x3cc30bce, 0x3a241040, 0xbcd79ede,
      0xbc52040a, 0xbcfe23bf, 0xbccc6e29, 0x3ba9a0e0, 0x3ce5cb08, 0xbbb8bcac,
      0x3cc72efc, 0xbc9f1c18, 0x3bfc8e18, 0xbb7d54c0, 0x3ccdcc6e, 0xbb0f8178,
      0x3d064b73, 0xbbc32110, 0xbc96b494, 0xbc9087e7, 0xbcff701d, 0xbb92b3b4,
      0x3c28f054, 0xbc1d313c, 0x3ca2a5ae, 0x3b301940, 0xbce6244c, 0x3aa48600,
      0x3cef975a, 0x3ce93b72, 0xbc7e7070, 0x3bee4e10, 0xbc990b2e, 0x3c024440,
      0xbbfa03d8, 0xbcf3b856, 0x3cef0a4e, 0xb9fb7e80, 0xbc089470, 0xbcf27ed4,
      0xbcf25630, 0xbcfb8b6c, 0xbce5cb61, 0xbcdd3307, 0x3b252f30, 0x3caad482,
      0xbd02302e, 0xbcc9f17e, 0xbb714478, 0xbc7903b4, 0xbccb3ea5, 0x3c9e1696,
      0x3bf65510, 0xbc1905b8, 0xbbb197bc, 0xbc48950e, 0x3a146f80, 0x3d0418d3,
      0xbc86a367, 0x3cc492ba, 0x3c81e09c, 0xbc1cbb74, 0x3ca1fe6e, 0x3c30e93c,
      0x3ce7f838, 0xbbc89108, 0x3cbb1f90, 0x3b89aa00, 0xbcf530ec, 0x3cb11ed8,
      0x3c43df68, 0xbce19434, 0x3b9e2490, 0x3cf89f56, 0xbc849774, 0x3c9fc5d2,
      0x3cd40aac, 0x3c7a737c, 0x3c7d2ea4, 0xbc6e32d2, 0xbcf60c58, 0xbceac681,
      0x3b5cc6f0, 0xbcb24d05, 0xbc50ea44, 0x3c825ef2, 0xbc8e625e, 0x3b913ee0,
      0x3cbabd98, 0x3c4cc200, 0xbc927f40, 0xbaa63d80, 0xbcfb9adf, 0xbb255d48,
      0xbcb4b2be, 0xbc223dec, 0xbca329c0, 0x3996f300, 0xbc4b28a8, 0xbca2024e,
      0x3bc93218, 0xbcfb261d, 0xbcc07b70, 0x3c5b7edc, 0x3c29eec0, 0xba877a60,
      0x3bcaaaa8, 0xbc985c60, 0x3cdabce2, 0x3c05e470, 0xba12b580, 0xbace2ce0,
      0x3cf43386, 0x3cc77856, 0x3cdac336, 0x3bcc9858, 0xbca85360, 0xbc7dfc74,
      0xbca8f6f2, 0xbcebb929, 0x3c54ddf4, 0x3cef30be, 0x3d086381, 0xbcc61963,
      0x3c4933a0, 0x3bd0e390, 0xbcd9286e, 0xbcf52a29, 0x3c41a090, 0xbbe75208,
      0xbcfae2f0, 0xbb6de348, 0x3c371660, 0xbca5ee80, 0xbcbd551a, 0xbc87a0cf,
      0x3d07b59b, 0x3d07c167, 0x3cf04702, 0xbcb3c79e, 0xbcd251d2, 0xbce30070,
      0xbce0f9a9, 0x3bf6c1d8, 0xbd0731b5, 0x3d0370b5, 0xbcaa443a, 0xbc56cda8,
      0x3cb7792e, 0xbc8f5829, 0xbcf93b5f, 0x3c839fac, 0xbcaae5d4, 0xbcc4e849,
      0x3cb66a44, 0x3b199630, 0x3c468f30, 0x3c77b788, 0x3b168fb0, 0x3c64b68c,
      0xbd06ecbd, 0x3b8c23c8, 0xbce2b2a3, 0x3c8dc6a0, 0x3cdd1d38, 0xbca4e910,
      0xba55c9c0, 0xbcf4d532, 0x3c2cdfc8, 0x3ca1fbc4, 0x3c86ddc6, 0x3cec0c5a,
      0xbb921d80, 0x3cca2cec, 0xbd01a764, 0x3c65cc38, 0xbd015bd0, 0xbbb9c09c,
      0x3ce1ab58, 0x3cc7f21a, 0x3cbb4f7e, 0x3d083d27, 0x3baef860, 0xbc0828bc,
      0xbcdd867c, 0x3c7b03c0, 0xbc644a06, 0x3c82be4e, 0x3b24bff0, 0xbbb0e154,
      0xbcd2baa5, 0x3c0c5f34, 0x3cf15a7e, 0x3c38b8b4, 0xbbf001cc, 0x3ceda660,
      0x3c007a10, 0x3b9db930, 0x3d064f59, 0x3c60ed70, 0xbc6bf90e, 0x3c5f42c4,
      0xbcfb25e3, 0x3a1da400, 0x3ada6b40, 0xbbd3a34c, 0xbcc02c69, 0xbc91701c,
      0xbce28e14, 0xbbdfb1bc, 0x3c7d7424, 0x3d011235, 0x3c93de26, 0xba18af80,
      0xbb9be300, 0xbb4fd198, 0xbc1ab75a, 0x3d038375, 0x3c476580, 0x3c61dba0,
      0xbaed0000, 0x391a4400, 0x3beba058, 0xbcc32eba, 0x3c35a408, 0x3c5e8790,
      0xbcffa045, 0xbac834a0, 0xbcc64ee0, 0xbcca8e32, 0x3cee84e2, 0x3c44d5e0,
      0xbc86f4c9, 0xbaf97260, 0xbc4accb4, 0x3cffb27e, 0xbcb03482, 0x3be7bc00,
      0xbbff9e78, 0x3c218ce0, 0x3c97a8b4, 0xbbc192b4, 0x3c3c5bbc, 0x3c89a4e2,
      0x3c7881e0, 0x3cfd65c2, 0x3ce6165e, 0xbcec7dae, 0x3cac24bc, 0x3ccaffde,
      0x3c8b2324, 0xbd081384, 0x3c8e6bdc, 0xbbff50f8, 0xbc8a41fc, 0x3b812fc8,
      0xbc95b9c5, 0xbc795b12, 0x3b0e21b0, 0x3cc139f2, 0x3ccdcd8a, 0x3c919a58,
      0xbadf4040, 0x3d0388bb, 0x3c126440, 0x3c73270c, 0xbce2c5c1, 0xbcd90be1,
      0x3ca06d9c, 0xbcd0e34c, 0xbcba1c89, 0x3cc59b90, 0xbc9a236b, 0xbc8a836b,
      0xbcd48874, 0xbcb74023, 0x3c268c1c, 0xbc994032, 0x3c761ec0, 0xbc192b30,
      0x3c9233c6, 0xbce39dd8, 0xbbced258, 0x3a8747e0, 0xbc4cb138, 0xbc35a752,
      0xbc40e068, 0xbcf891fd, 0x3bf3ce18, 0xbc914e4e, 0xbc210292, 0xbc601c40,
      0xbce2742c, 0xbc2e3b66, 0x3c23d840, 0xbb20b9c0, 0x3afc74c0, 0x3cb3033e,
      0x3cc42b52, 0xbc3427a2, 0x3d07da63, 0xba00c9c0, 0x3cdb947c, 0x3cf3b55a,
      0x3c9bbc24, 0xb94a5300, 0xbb8e6168, 0xbced349c, 0xbcd450c3, 0xbb34d8e0,
      0xbc17c062, 0xbcb5627a, 0x3c63403c, 0xbb5d8630, 0xbcbbb84e, 0x3d08704b,
      0x3c1b21d0, 0xbbecbbf0, 0xbcef113c, 0x3bcd7dd8, 0xbc363f2c, 0x3ca48530,
      0x3d073acf, 0xbcb2fd52, 0xbba4513c, 0x3b33cc10, 0xbc11bc84, 0x3c98d7c6,
      0x3ba25528, 0xbc456d0a, 0x3c807874, 0xbb66bd88, 0x3cf9298e, 0x3cfb2672,
      0x3c2b2e3c, 0xbc465530, 0xbc7ffd8e, 0xba7fd240, 0xbb0b10c0, 0x3bb898f8,
      0x3c69a6a8, 0x3b675ab0, 0xbc53e2ec, 0xbd0407e0, 0x3c3ff124, 0x3a8811c0,
      0x3b33baf0, 0x3b805ec8, 0x3cc979d6, 0xbcc31bb2, 0x3c93e704, 0x3bd1c500,
      0x3b2cfd80, 0x3c82f4ac, 0x3bf7a6e0, 0x3c477180, 0xbcd269e1, 0x3c4d62dc,
      0x3cff0602, 0x3c53d124, 0xbd0157b5, 0x3c655cb0, 0xbcbed309, 0x3cc7e680,
      0xbcc89f70, 0xbd01e10f, 0x3c8156c4, 0x3cc82ac0, 0x3cf3b95e, 0x3cee6b4c,
      0x3c8451d0, 0x3cf4d9d6, 0x3d01d7d3, 0x3c12fc68, 0xba812800, 0xbbf9c700,
      0x3ac36080, 0xbcfd4958, 0x3c8af64a, 0x3c9ce8f4, 0xbc9f2ac2, 0x392d5b00,
      0x3c8e7cdc, 0x3caeed1e, 0xbca5ae47, 0xbcb10f92, 0xbc128774, 0x3c4e5084,
      0xbb9ff568, 0x3bc59390, 0xbd03c71a, 0xbd05f08f, 0x3c3516fc, 0xbc11882c,
      0x3cdaec22, 0xbd04e1b9, 0x3c8bbeee, 0xbca7fd69, 0xbb944388, 0x3cc9e5cc,
      0xbcea46fa, 0x3c0981ec, 0x3cc1eac4, 0xbcd82e09, 0x3c2c7ff0, 0x3cc37514,
      0xbc102a0a, 0x3ccab560, 0x3aab6800, 0x3c7fce90, 0xbc8970e0, 0xbc857b94,
      0x3a25c480, 0xbceec3a1, 0x3b805da0, 0xbc5cbc28, 0x3c8e9366, 0xbcdc9a0e,
      0x3acb6aa0, 0x3cebd676, 0x3cd21384, 0x3b0405d0, 0x3becd640, 0x3cf9c51a,
      0x3cc07f88, 0x3b32f0a0, 0xbca56c38, 0xbcb835f8, 0x3be691e8, 0x3cb4a372,
      0x3c35ccd0, 0xbb4813f0, 0xbca5f77e, 0xbc8eee78, 0xbbcc143c, 0xbce5f749,
      0x3c90daec, 0xbca1a6e9, 0x3bf37240, 0x3cff7b2e, 0x3cf37c9e, 0x3ba8ab90,
      0xbbe27768, 0x3be9b410, 0xbcff2392, 0xbc51adbc, 0xbaa99120, 0x3cd5f0bc,
      0xbb80d9bc, 0x3ce34b16, 0x3c1c5950, 0x3cec9788, 0xbcebf9a7, 0xbb65bd98,
      0x3d07f071, 0x3be2cb90, 0xbc3ae3ac, 0xbca740a7, 0xbb080260, 0xbc48507c,
      0xbcc31098, 0x3c0e56d4, 0xbc65f9b0, 0xbc05f622, 0x3c97a35c, 0xbcd1b627,
      0xbc8863a7, 0x3bf91fc8, 0xbc77b5da, 0xbcbecc3c, 0x3c29339c, 0xbc6124ac,
      0x3d033551, 0xbccea7f0, 0x3cc14b62, 0xbceb1b27, 0xbb7bdf98, 0xbc437e1e,
      0x3c2a67e0, 0x3cb5d8da, 0x3be8e108, 0xbcae5596, 0x3bda5500, 0x3c3e70b4,
      0xbb56ae88, 0xbcaf602c, 0x3c2bc4f4, 0xbcd81a72, 0x3caa69fa, 0xbc5a8e46,
      0xbcb7a360, 0xbc8b4f8f, 0xbcfc7add, 0xbc968b49, 0x3b96a450, 0xbb516348,
      0x3c5b38f8, 0xbcf4b78e, 0x3d00afa9, 0xbb861934, 0xbc225f3c, 0xbc722068,
      0xbc52d470, 0xbc81109c, 0xbcd19e69, 0x3cf15eee, 0x3b3f7f40, 0x3c86fde4,
      0x3c841d52, 0xbc3fc934, 0xbb981878, 0x3ce9c266, 0xbd0757c2, 0xbc310934,
      0xbbba682c, 0x3c226ff8, 0x3cc1aef8, 0xbc99dfba, 0xbc51c2da, 0xbc194f70,
      0xbc94858b, 0xbc0be334, 0x3cc96dd4, 0x3cab7d3e, 0xbb4d4800, 0xbbcd1110,
      0xbc9e8d5e, 0xbc79985a, 0x3ccba72e, 0xbc49e9de, 0x3aec0ca0, 0x3cfbde2a,
      0x3cc54062, 0x3c5cc4f4, 0x3be1b058, 0xbc9d79f0, 0xbcce4198, 0x3c895c2e,
      0xbd02ae7d, 0xbc41749a, 0xbcfa24b4, 0xbcd941a9, 0xbcfca8ae, 0xbcbbb854,
      0xbc205552, 0x3d03416d, 0xbc2c3266, 0x3cd1be9c, 0xbc78dc8a, 0x3c0a7dbc,
      0x3cdcba4e, 0x3c4f7d18, 0x3ccbe278, 0xbc964cd0, 0xb957cf00, 0x3c7a4ca0,
      0x3c9bfd10, 0xbb29a4e0, 0xbab92800, 0xbbea93f0, 0x3c01c884, 0x3c4e1bfc,
      0x3ca1405e, 0xbd04a8dd, 0x3c97623c, 0x3ce4e09a, 0xbbaca234, 0xbc4bb730,
      0xbcd13356, 0x3c176238, 0x3cbcada2, 0xbc919b45, 0x3c620020, 0x3ca76d18,
      0x3c81756c, 0x3cb5b014, 0x3b8cad50, 0xbcaf6732, 0xbc2bfbb8, 0xbb82979c,
      0xbcd79ed8, 0xbba46e9c, 0xba3c0d80, 0x3bf32790, 0xbccd0363, 0xbcfeb2d0,
      0xbc7db252, 0x3ce3bbb0, 0x3cf11c6e, 0xbb10a498, 0x3cdaf408, 0xbba954f8,
      0xbbfac770, 0xbcb84c47, 0xbaa2cca0, 0x3c6ea924, 0x3c0871ac, 0xbc072378,
      0xbc5c908a, 0x3c57b600, 0xbc216d8a, 0x3c2fea2c, 0x3ba4a400, 0xbd083ef3,
      0x3d00cfb5, 0xbc657028, 0xbcc1a685, 0x3b54d820, 0xbc144db4, 0xbbbe6254,
      0x3ca97a1a, 0x3cb42dc4, 0xbc99521c, 0x3cd05c10, 0x3c367044, 0xbcb84358,
      0x3a2aeec0, 0xbd0451ab, 0xbcfd23d6, 0x3cd36c8a, 0xbcd8a67a, 0x3bf86e78,
      0xbcf56c07, 0x3a3f4d80, 0x3bdcd350, 0xbcf53e67, 0x3d00f3cb, 0x3a984440,
      0xbc4f9d8e, 0x3b366650, 0x3cb74bec, 0xbc8b6ecf, 0xbc8f8a54, 0x3c328ba0,
      0xbb32cd68, 0x3ca9a316, 0xbc8a87e3, 0xbc1eb4f8, 0x3cdd0c96, 0x3c1b1080,
      0xbcf105c9, 0x3c9712d4, 0x3c7c721c, 0xbc6e1122, 0x3c0f9f2c, 0x3be70530,
      0xbcfc1e7f, 0x3b8a9d00, 0x3cfbe32a, 0x3cd39d64, 0xbc6bc71e, 0xbcba1256,
      0xbc2a8b38, 0x3bc427a8, 0x3b356e60, 0xbc856ada, 0xbc98bf2e, 0x3c831ab8,
      0xbc87afbe, 0x3b178930, 0x3bbbddb0, 0xbc69e9b8, 0xbcc361f8, 0xbc504e2c,
      0xbcb9d2be, 0xbc9f1b30, 0x3b7bc740, 0xbbb9d234, 0x3cfd5696, 0xbcd3fec3,
      0x3b7f5a90, 0xbb764668, 0xbc55cc22, 0x3c9cc034, 0x3c54e4e8, 0x3b32f0c0,
      0x3b4eabe0, 0xbabc2e40, 0xb923a200, 0xbad81480, 0x3ca45c6c, 0x3c6c9f44,
      0x3b5d6f20, 0x3cc5c13c, 0x3cbef83e, 0x3ca1ccae, 0x3c1fced4, 0xbb840c00,
      0xbbeb5b2c, 0xbc5854ac, 0x3c3ed580, 0x3ca2d572, 0x3d001821, 0xbcfd058e,
      0xbc9c31e7, 0xbcd0513e, 0x3cbf8fd6, 0x3c141c34, 0x3cda3850, 0xbc99e2d0,
      0x3d0631af, 0x3bf1ece8, 0x3c4b2df4, 0x3cb62e62, 0x3b43c0d0, 0x3c2bc42c,
      0xbc05da80, 0x3cdc6088, 0xbce1ea56, 0xbce4b754, 0x3d0325b5, 0xbc1d286c,
      0x3d03210f, 0x3cd20ef8, 0xbb98a0e8, 0xbc724886, 0x3990fe00, 0xbca9d589,
      0xbd061335, 0x3ce563fc, 0xbcf42150, 0xbd06d273, 0xbc1a1d2c, 0xbba770a4,
      0x3ab20820, 0x3c11c7ec, 0xbc4aab00, 0xbcc23ae3, 0x3c18e3e8, 0x3cba4b1e,
      0xbcd0ad32, 0xbc82fab8, 0x3cf3a5a2, 0xbba4bfbc, 0xbc53dea2, 0xbb60a568,
      0xbb9544a4, 0x3c4482a4, 0xbd022ff9, 0xbc70062c, 0x3bc5c6f8, 0x3c70e28c,
      0xbc8fb583, 0xbcd33961, 0xba58d840, 0xbbb95e08, 0x3c781470, 0x3ca178c6,
      0xbc4cc830, 0x3c5fae0c, 0xbc70129e, 0xbc56003c, 0x3cc643e8, 0xbc0f8cc4,
      0xbc61bb06, 0x3b4bdf10, 0x3c5f44f0, 0xbc9ab350, 0x3c3f0adc, 0xbcb88bbc,
      0xbb687d30, 0xbc8b0087, 0xb9c57500, 0x3c3d2f74, 0xbac66460, 0xbc1a1ace,
      0x3c3e85bc, 0x3adb54e0, 0x3c88a206, 0xbce59fc5, 0x3c1d3fdc, 0xbc8e51b6,
      0xbc08c38a, 0xbc9c1c36, 0x3ce2ee86, 0x3b8a11f0, 0xbc41dcec, 0x3caf8bf4,
      0xbce48701, 0xbcab3300, 0xbb89681c, 0x3cb0231a, 0xbba1bdf0, 0xbc9cd35a,
      0xbcf7a6b4, 0xbc4b7e4a, 0xbcfc3da3, 0xbce375ac, 0xbc0cad5a, 0x3bd99a88,
      0xbd010fa3, 0xbc8e4f45, 0x3cf2f416, 0x39bdfb00, 0xbcf61814, 0xbc9dadd6,
      0xbcec76c7, 0x3b63d250, 0xbc3d5a68, 0xbca98b4e, 0x3c532870, 0xbc891160,
      0xbb82ab78, 0x3d05d893, 0xbcb4aafa, 0x3cabd366, 0x3cadef44, 0x3cf8e1e2,
      0xbcf69896, 0xbc6b242c, 0x3c83ab98, 0xbcd96dc9, 0xbc5cf51a, 0x3c99d3d4,
      0x3beb7cd8, 0x3ce729d2, 0xbc3bc430, 0x3a378d80, 0xbc3b44d6, 0x3c5909d8,
      0x3cff20b6, 0xbcfebc2e, 0xbce4d2ba, 0xb9f32300, 0x3b9a8ea0, 0x3c85d8f4,
      0xbd00121b, 0x3c7c43a8, 0x3cee57ca, 0xbc0c51ce, 0xbd0361fa, 0x3d0490b3,
      0x3c64c36c, 0xbcad1fac, 0xbb1534e0, 0xbc844a6f, 0xbcae8670, 0xbaa58120,
      0x3c15d028, 0xbb641f20, 0xbbdfe934, 0x3b0e7820, 0xbcff2025, 0x3cf6812a,
      0xbcc6873e, 0x3cad7ca4, 0x3ceb95dc, 0xbcd5c158, 0x3cf7feba, 0x3c55b5c0,
      0xbccfc980, 0x3cffb27a, 0x3b741d10, 0xbce142dc, 0x3d05e58b, 0xba3f8d80,
      0xbd008da3, 0x3ce25cf4, 0xbb0e5ce0, 0xbbb0b5e8, 0x3ceee2e8, 0x3d057d6f,
      0x3beed588, 0x3cc1d5b6, 0xbcf7c752, 0xbcb96b7a, 0x3cf6a0aa, 0xbc60a652,
      0xbcb54eb4, 0x3c9f3698, 0xbce42ff2, 0xb8ad1600, 0xbcdf26e5, 0x3cf2c1a6,
      0x3bcf89a0, 0xbbc7d5cc, 0x3c240b9c, 0xbcb2f600, 0x3bef1888, 0xbcd39687,
      0x3cf4c39e, 0x3c29e8b4, 0x3cb3df62, 0xbc68a530, 0x3c05442c, 0x3cd53d1a,
      0x3cc7c256, 0xbcdc8a2e, 0x3ce5b7b6, 0xbbe30b78, 0xbc8bddde, 0xbd048c9e,
      0x3ab8d040, 0xbcf1b270, 0xbca48fe2, 0x3c48db6c, 0x3c61e860, 0x3c2b1e8c,
      0xbcbe2dd4, 0xbcbaa53c, 0x3c1fdfd0, 0xbcf1ebd8, 0xbca668b6, 0xbb8c7100,
      0xbc04cc84, 0x3ca11246, 0xbc83da2b, 0xbc34d780, 0x3cc96580, 0x3bf0f480,
      0xbccb4ec0, 0xba9cfba0, 0x3ca1e24c, 0x3c5659ac, 0xbb0f6b30, 0xbcc21663,
      0xbb912b54, 0x3c90beea, 0x3c4f32f8, 0x3b765a30, 0xbcfc58e7, 0xbc318ac4,
      0xbd0644a7, 0x3c937c86, 0x3afb3fc0, 0x3cb73b0c, 0xb7c26000, 0xbb699530,
      0xbcf043b2, 0xbc2215a2, 0xbc9e713a, 0xbbabffcc, 0x3d075a2f, 0xbd005a1d,
      0x3c1f5124, 0xbc8ecbde, 0x3bc09798, 0x3cc5bb80, 0xbbd6e768, 0xbd014cd0,
      0x3c4f5c80, 0x3cb3800e, 0xbcf17241, 0x3c9a5b24, 0x3d061bdd, 0x3ba66960,
      0x3c39eedc, 0xbbb5efd4, 0xbc67a0ac, 0xbcabff78, 0x3ce71656, 0xbbdb2508,
      0x3ce9e56c, 0x3cbdb776, 0x3ab1cda0, 0xbd031962, 0x3c88b13e, 0xbcbd6a98,
      0x3c52e68c, 0xbcc5541c, 0x3c58af70, 0xbc8aa10d, 0xbaa88bc0, 0x3cf1b37e,
      0x3cefde26, 0x3ceef896, 0xbbab2624, 0x3cbec2be, 0xbcdb6b34, 0x3c89d472,
      0xbc5ab87c, 0x3cee7aae, 0x3c071668, 0x3cb1fee4, 0xbbbaf770, 0xbbb33ccc,
      0xbcfa02dd, 0x3b88b5b0, 0xbbacec08, 0x3c4c7078, 0x3a533940, 0xbb93bd88,
      0x3cdc79a8, 0x3d026b9b, 0xbc58cfca, 0xbd027e47, 0x39b43200, 0x3bb25e20,
      0x3cb83290, 0x3caa280c, 0x3d08510b, 0x3c333000, 0xbc9b5b72, 0x3c90a6e6,
      0x3c769bd0, 0x3c8f6ee0, 0xbc127ebc, 0xbce680f8, 0xbc067a3c, 0xbcda0f2e,
      0xbc13b496, 0xbc3ae8b0, 0x3ca9727c, 0x3aeed180, 0x3c3204d0, 0x3cfc8cca,
      0x3cc53674, 0xbcb47360, 0x3c5a9fdc, 0xbc691774, 0x3cab724a, 0x3ca55d6a,
      0x3ca0cbfa, 0x3c397c24, 0xbc14075a, 0x3c1a7448, 0xbc73c378, 0xbca40365,
      0xbcceed76, 0x3c83111e, 0x3c16b72c, 0x3b410060, 0xbd01924a, 0xbcaaf102,
      0xbc59517c, 0xbc565b9a, 0xbcee76c5, 0x3c4747a8, 0xbc647d56, 0x3c9787ec,
      0xbb72fea8, 0xbc4e057c, 0x39ed4500, 0x3cee9ac2, 0xbc0596e2, 0xb9f84800,
      0xbcfb4f09, 0xbc4dcb68, 0xbcc20765, 0x3c9267b8, 0x3c69add0, 0xbbe132f0,
      0x3b33e8c0, 0xbcd84309, 0x3c3adaf4, 0x3be8a278, 0xbc2a9744, 0x3b3003a0,
      0x3c26b764, 0x3a7bbac0, 0x3bb76430, 0xbc584b92, 0x3b4c1390, 0x3ce6161e,
      0x3ba96ba8, 0x3ce31d1c, 0xbc374df8, 0xbc731596, 0xb9437c00, 0x3cd9500a,
      0xbcb266d2, 0x3c9e60ac, 0xbcc69550, 0x3ce4606e, 0xbc45f5c0, 0x3c96f756,
      0x3c94550a, 0x3c53ef88, 0x3bf1d488, 0xbc46b02c, 0x3ca98d24, 0x3cf092a6,
      0x3c6d5b20, 0x3c2cd21c, 0xbc4c64c4, 0x3c3010ec, 0x3bf333f8, 0x3c7e2550,
      0x3ce84bb6, 0xbc95c87a, 0x3cefc47e, 0xbb963a34, 0x3bdfa580, 0x3c8f7bce,
      0xbc682d4e, 0xbcfc9045, 0xbbc1b02c, 0xbb6739e0, 0x3bbc8bc8, 0x3bfe00c0,
      0x3ca10ff4, 0x3bfbbf90, 0xbd076589, 0x3cc5eb02, 0xbc1d3b1a, 0x3c10a760,
      0x3c5d67fc, 0xbbe5159c, 0x3cf10b0e, 0x3cc769a4, 0x3cd60c02, 0x3cc997c2,
      0xbc8311f8, 0xbcf171e1, 0x3c4f4ba0, 0xbb8aaa3c, 0xbc326400, 0xbcf261d4,
      0xbcd6541a, 0xbd07bbe6, 0x3b6e3cd0, 0x3b0a9030, 0x3ce36768, 0xbccdda09,
      0xbcfdc752, 0xbd054829, 0x3a141cc0, 0x3c0c6b48, 0xbb949af0, 0xbc51638e,
      0xbd07759c, 0x3cba70c0, 0x3cd7013c, 0xbbc33288, 0x3cef3a36, 0xbc627328,
      0x3c918272, 0x3cc07d32, 0xbbb1064c, 0xbceea610, 0x3b189e50, 0x3cf292ca,
      0xbcad085c, 0x3c511718, 0xbcf6b363, 0xbc8c3ada, 0xbac24260, 0x3cd05774,
      0x3abc26a0, 0xbc693b56, 0x3cf0cd5a, 0x3bc29f90, 0xbc9beef4, 0x3cb18942,
      0x3cd37d4c, 0x3c3dba5c, 0x3cf0106a, 0x3c99f240, 0x3c701eec, 0x3ccb8c6c,
      0xbc579bb8, 0x3cde0450, 0x3cbf11bc, 0xbbe146ac, 0xbc2a5922, 0xbc3b8f8e,
      0x3ce13fe2, 0xbc82a743, 0x3c76e5b0, 0xbcd67aac, 0xbbcda03c, 0x3c801036,
      0x3bbf57f0, 0x39884100, 0xbcb5169e, 0xbcca37ba, 0x3c716cf0, 0xbc05b844,
      0x3cd7b4fe, 0xbc28fdf0, 0xbca14a74, 0xbc1ba43c, 0xbc520640, 0x3d031b4b,
      0xbcf8b3f6, 0x3ca639a6, 0xbcc63a49, 0x3caceaea, 0x3cc0a818, 0xbc124816,
      0xbcfe1963, 0x3cb51a1a, 0x3ccfeb04, 0xbc65d6a8, 0x3a98a3a0, 0x3ce07b8e,
      0x3ca3c49e, 0xbce333ae, 0xbc84e6be, 0xbc81e5d4, 0x3d087a8d, 0xbce9af4c,
      0x3cccf4f2, 0xbc902272, 0xbcfc35ff, 0xbc9bee87, 0x3c9b95e6, 0xbb606558,
      0x3cc2680c, 0xbbd6f708, 0x3cfe99e6, 0x3b575d40, 0x3ccd789e, 0x3c957b14,
      0xbc7f4ff8, 0x3c99c940, 0xbb9b4310, 0x3c9cfc38, 0x3baf8ca8, 0xbc7d2b9a,
      0x3bcc0388, 0x3cb47954, 0x3c9d434c, 0xbb5d5088, 0x3b830b78, 0xbc80bbe5,
      0x3cf821c2, 0x3ca7ca72, 0xbcb63518, 0xbb06adc0, 0xbc4c0184, 0x3be5da30,
      0xbcc41a1c, 0xbc23dea6, 0xbd029940, 0xbc9c843e, 0xbc9d29a0, 0x3ccd5438,
      0x3cedfcec, 0x3cc17af8, 0x3cc77452, 0x3bb6ec48, 0xbc2d0d96, 0x3cafe94e,
      0xbb8c751c, 0xbcec9fce, 0x3cc950c2, 0xbc4d007c, 0xbca16ada, 0xbbd06b24,
      0xbc42afb0, 0x3be33c68, 0x3b45c1e0, 0x3c7652e0, 0xbc927a7e, 0x3b1e6ce0,
      0x3caedb76, 0x3bcf77b0, 0xbcb5789e, 0x3cdeb7e6, 0xbc8ebc92, 0x3c1a7ad0,
      0x3cdcb63a, 0x3c7faccc, 0x3d0699c7, 0x3c217944, 0x3be637f0, 0x3bacb788,
      0x3c870712, 0xbd040b4a, 0xbb1eb6e0, 0x3cc4eb4e, 0xbc947985, 0xbca65ce5,
      0xbc38e6ac, 0x3cf7a40e, 0x3bec0ba8, 0x3c8b3542, 0xba5ac280, 0x3ce15de2,
      0x3cf57a0e, 0xbce80854, 0x3c6ef8b4, 0xbb7b8310, 0x3ce9e72e, 0xbc41c86c,
      0x3c65a2bc, 0xbca8c174, 0x3c8dedb4, 0xbcf6252e, 0xbc66e10e, 0x3cab48b2,
      0x3c5e095c, 0x3c7e8aa0, 0x3cd19b9c, 0x3ac46ae0, 0xbc7ac2de, 0xbc21edf8,
      0xbc909280, 0xbbefeb3c, 0xbc8e3a69, 0xbc1f7984, 0xbb4b6678, 0x3be1bca0,
      0x3c35876c, 0x3c48161c, 0x3cf930a2, 0xbcdc4d34, 0xbce476bf, 0xbbd82808,
      0xbcb32905, 0x3c3b3928, 0x3c92022c, 0x3c8a843e, 0xbc29f3e2, 0xbce40be3,
      0xbcb7ddae, 0x3c9f751c, 0x3c8fa44c, 0x3cb4c8be, 0x3cbeb818, 0x3a9c7160,
      0x3c965892, 0x3bc1efe8, 0xbb9d37e0, 0x3ce3048e, 0x3c00c710, 0xbcea4f90,
      0xbca42307, 0xbd0511e6, 0xbbd287ac, 0xbca2cb8e, 0x3cb39332, 0xbd0648a0,
      0x3c2bd55c, 0x3bb01130, 0x3b341990, 0xbd05ab39, 0x3b9ce9e8, 0x3ba790c8,
      0xbb7fbb58, 0x3c64f2b4, 0xbb654188, 0x3cfe6f6e, 0x3ce35eb6, 0xbcde0238,
      0xbc19c252, 0xbcda5d05, 0xbca0b232, 0x3cbdde16, 0x3cfbbeb6, 0xbcbb4e80,
      0xbcbf95be, 0x39afd500, 0x3c89411a, 0xbcc2a912, 0xbc04b97c, 0xbbecbe60,
      0xbcaa17f0, 0x3d07ec8f, 0x3c7f21b4, 0x3cea800c, 0xbc8728b6, 0x3cec756c,
      0xbc9976fa, 0x3bb3cc78, 0xbca3d17c, 0xbc625056, 0xbc99576e, 0x3b0c6000,
      0x3c31e3e0, 0x3cf2ecc2, 0xbd07c5c9, 0x3c8db84e, 0x3cfb7a6a, 0xbcba9927,
      0xbbf6c808, 0xbc880f58, 0x3caf62fc, 0xbcabf752, 0xbc7a409e, 0xbcfcae4c,
      0x3c559ad8, 0xbcb1e827, 0x3cf59b5e, 0xbcfbec1d, 0x3c495370, 0xbc895303,
      0x3cca674c, 0xbb833500, 0x3c5cbe30, 0x3c0525a4, 0x3d07908f, 0x3cf26032,
      0x3c12350c, 0x3a7728c0, 0x3ca2ef34, 0xbcc36b3e, 0x3c66bf50, 0xbcdbd674,
      0xbc30c4a2, 0x3ccdf0e4, 0xbcba54ae, 0x3c7783c4, 0xbc756bb4, 0x3c35c734,
      0xbd0082fe, 0xbbe8eabc, 0x3c8c534e, 0x3ca7e76c, 0xbc943814, 0xbce8dab2,
      0xbbf1a94c, 0xbcf752f2, 0xbaf0d860, 0x3ce16e08, 0xbce67374, 0xbcb21812,
      0x3d0856bd, 0x3cd63bae, 0xbcf4424e, 0xbd032756, 0xbc9b20e7, 0xbb29a300,
      0xbc410c12, 0x3ba1b2a0, 0xbad8b9a0, 0xbcb7e769, 0xbcb4cef2, 0xbba5801c,
      0x3bd98f00, 0x3ceb023c, 0xbc9e5f0b, 0x3b7a2040, 0xbce05b5a, 0x3cd893fc,
      0x3cb2dd1e, 0xbcecf863, 0x3c5a0464, 0x3c6fec38, 0xbbcb5268, 0x3c356314,
      0x3cf3658a, 0xbc222b80, 0x3c6f9afc, 0x3c173990, 0x3c177e50, 0x3c8ba066,
      0xbcc6d7d0, 0x3c77710c, 0xbccc4b96, 0xbb2b2068, 0xbb2e7e10, 0x3c2e8224,
      0xba3db0c0, 0xbce7baee, 0xbc3856c4, 0xbd01b3c9, 0xbcb100e2, 0x3c6930f8,
      0x3c20e47c, 0x3c9fbaf6, 0x3cd6647e, 0x3bb56368, 0xbcda525c, 0x3ce10a0a,
      0xbc1ddbf8, 0x3c51683c, 0x3d060601, 0x3c9ad062, 0x3cd984f2, 0x3cfbbdc6,
      0x3bde9258, 0x3b8f5c40, 0xbd04871d, 0x3aff46e0, 0xbc02d9fc, 0x3abc5b60,
      0x3cef1636, 0x3ccd2cee, 0xbb7329e0, 0xbc84e5da, 0x3c7c35a0, 0x3cb35d8a,
      0x3cd3a1f4, 0x3ce64e9a, 0xbc380740, 0xbcf60a67, 0x3c808e92, 0xbc87e1e9,
      0xbcf06450, 0x3cbc3f52, 0x3c9c1d54, 0xbcbac820, 0x3aac32a0, 0x3d0001bf,
      0x3c4997cc, 0x3c2571ec, 0xbc902d7c, 0xbbf935e8, 0xbcc7c245, 0xbccd7234,
      0xbcc67900, 0xbc99b096, 0xbc3ca092, 0x3c2e42a8, 0x3cb7fa88, 0x3c88c8ea,
      0x3c1c8644, 0xbca3c64e, 0xbd072028, 0xbca1fc12, 0x3c92bf5a, 0xbd012dc2,
      0x3b676500, 0xbc5a7ab0, 0x3ce5dcd4, 0xbbb3acf8, 0xba726b40, 0x3ce12880,
      0x3b897a08, 0x3c3a72f4, 0xbc5ab9e8, 0x3b3248e0, 0xbcd90201, 0x3bdb89e0,
      0x3bdb8478, 0xbcd3d9ce, 0x3b3d4f00, 0x3b4f6530, 0xbb275de0, 0x3c6d0078,
      0x3cdb8ba2, 0x3c3e2bd0, 0xbcdd07ce, 0xbc68dbb0, 0xbc20a6ec, 0xbb506fe0,
      0x3c0b0dd4, 0x3c9bd2f2, 0xbc564c4e, 0x3b43d8b0, 0xbb3cbd00, 0x3b7c7ef0,
      0x3bc4e7d0, 0x3ab0a1c0, 0xbc668d6c, 0x3c8b3f06, 0xbc880d2b, 0xba83bac0,
      0x3bcbd9c0, 0x3cde0f18, 0xbbdfac68, 0x3c637e04, 0xbc96cc89, 0xbcfd880e,
      0x3cddc174, 0x3cee9530, 0x3c813408, 0x3c4590b4, 0xb9962000, 0x3c11daa0,
      0x3c9fd4be, 0xbcea5c09, 0x3836c400, 0x3bd2de68, 0x3b877b78, 0xbd02aeb9,
      0xbcaf2c30, 0x3d05996f, 0xbc08af56, 0xbbe46e90, 0xbbfb8370, 0xbce53234,
      0xbc93e20e, 0x3be51740, 0xbc90ac34, 0xbcf8e2ee, 0x3c87d5b6, 0xbc9ef13e,
      0x3cbc9f8e, 0xbc7e5b4e, 0xbbf17aa4, 0xbc97f374, 0xbb6d7b58, 0xbb8b8cf8,
      0xbc1b6d40, 0x3cfcde36, 0xbcb75f76, 0x3ca94bb6, 0xbb88fd44, 0xbc8c2069,
      0xbcbc8194, 0xbb0c0920, 0x3b9af440, 0xbc8f0820, 0x3b1a32b0, 0x3c66eb1c,
      0xbcae0554, 0x3bba47e8, 0xbce5f709, 0x3cbe3f94, 0xbc40c856, 0x3cb44652,
      0x3c554058, 0x3ce3e866, 0x3cf63efe, 0xbcdcbb9e, 0x3cd34ae8, 0x3c67f5b4,
      0xbca49229, 0xb8b72800, 0x3b795730, 0xba269680, 0xbb0a52f0, 0x3a1e5b40,
      0x3cac9cb0, 0x3d05c0cd, 0x3b5b8a60, 0x3cb55ea8, 0x3d03d343, 0x3cccd850,
      0xbb00aa30, 0x3c834fba, 0xbc1caef4, 0x3bca8860, 0x3cada96a, 0x3bec4c00,
      0xbd01bcdb, 0xbc39132c, 0x3c624468, 0xbcb6d45a, 0x3cb188b2, 0x3be5a708,
      0xbcdb135e, 0xbcc60e3e, 0xbc7c659e, 0xbce52421, 0x3c3ac530, 0x3c2f57ec,
      0xbc4a0d2c, 0xbcb957a0, 0x3be8b768, 0xbbde881c, 0x3b112800, 0x3c6750d0,
      0x3c411de4, 0x3cd5e450, 0xbc9acca2, 0xbaeddc80, 0xbb67e758, 0x3cf3c0f6,
      0x3d05b2a3, 0xbc78e5f0, 0xbc86b3c3, 0x3c268c20, 0x3c25ff7c, 0xbc980ceb,
      0xbbf54c70, 0x3d00f9e7, 0x3c8aebb0, 0x3c53a918, 0x3c2023d0, 0x3b67ec90,
      0xbc27bcf4, 0xbb6e2138, 0xba7412c0, 0xbb703f68, 0x3c071b14, 0xba141180,
      0x3cca355e, 0xbd0719d7, 0x3cef542e, 0x3c2769fc, 0xbd02ed8f, 0x3ceff102,
      0xbc49d9ac, 0x3c201944, 0x3a8797e0, 0x3bb35010, 0x3c61fb10, 0xbb410cd0,
      0xbca7526e, 0x3aff70c0, 0x3cfabada, 0x3c6966a4, 0x3c21ce28, 0x3c8139c0,
      0xbc88b823, 0xbcf53ee1, 0xbce3d583, 0x3cbc9ab4, 0x3cdb5076, 0x3bad9208,
      0x3b8754c8, 0x3c1f79a4, 0x3c719390, 0xbc006762, 0x3cf31bbe, 0x3b4cb160,
      0xbca374a0, 0xbcbc0e5e, 0xbcfc6b8e, 0xbccdcd1e, 0x3cc65ba8, 0x3b04b310,
      0x3ccb4b2e, 0xbcbf553c, 0xbcc3ef94, 0x3bb47060, 0xbccf8e09, 0x3ab70140,
      0x3bddcdd0, 0x3cc88b4a, 0x3c8ac2e0, 0x3d049ebf, 0xb9e41e80, 0x3c88d992,
      0x3ca17420, 0xbc9e413c, 0xb95b4f00, 0x3c918796, 0xbb9e2a78, 0xbcd912a9,
      0xbcb3ae74, 0xbc7c4dc6, 0xba96bc80, 0x3c9f1104, 0x3c2418c4, 0x3c42f3a8,
      0xbcc5c938, 0xbc2270de, 0xbc3923ca, 0x3aab76e0, 0x3c2aab04, 0xbcf740a1,
      0x3b775bf0, 0x3c928e92, 0xbc27098a, 0xb90e6000, 0x3c7625e8, 0x3b5eb9b0,
      0x3b9afda0, 0x3cf5c6ae, 0x3c15adbc, 0xbc99da40, 0x3b7aa610, 0x3c8a9aea,
      0xb9eec880, 0x3c75f54c, 0x3c41dcf0, 0xbc97cca7, 0xbb1be0e0, 0x3cc6812c,
      0xbcd90949, 0x3c6238fc, 0x3c8fc6a4, 0xbc68ac5a, 0xbc7c2c1a, 0x3a26b840,
      0x3ca012b2, 0xbca76638, 0x3cbeeaf8, 0xbc9cd3b8, 0xba8667e0, 0xbc86ab8d,
      0x3c8c9b76, 0x3bf1c4e8, 0x3bbff6b0, 0x3c600800, 0x3c99e3aa, 0x3c890a4a,
      0x3d04969f, 0xbd04baf5, 0x3cd59b58, 0xbcab04d2, 0xbd029f20, 0xbbdf3eac,
      0xbc9ac0b6, 0x3acd0780, 0xbc2711c4, 0x3cd3f5fe, 0x3bb13e68, 0xbb692b48,
      0xbc1d010a, 0x3ceed050, 0xba1fb980, 0x3c9b9f40, 0xbc46bfa8, 0xbd0237b6,
      0xbce5c576, 0x3c89e490, 0xbc6e70f0, 0xbceb481a, 0xbca1088e, 0x3bcf9718,
      0x3c85c988, 0x3c8f38a4, 0xbc837a8b, 0xbd084977, 0xbcc45a25, 0x3c26d850,
      0x3d080d2f, 0x38ea2600, 0xbb8b5e60, 0xbb610a30, 0x3ca34eee, 0x3cbced10,
      0xbc728ff0, 0x3c323508, 0x3c772628, 0x39261600, 0x3bcf35d0, 0x3ceff45a,
      0x3be2bcc0, 0xbacc4800, 0x3cc3b5e8, 0x3cee2ab8, 0xbc9d042e, 0x3d0487a7,
      0xbc8b69c0, 0x3cc214ee, 0x3c582cd0, 0x3c09bed0, 0x3c355288, 0x3d07cb3d,
      0xba28efc0, 0x38e55800, 0xbc9eed30, 0x3cc53d4a, 0x3ccee898, 0xbc8325d6,
      0xbcb64d29, 0xbcfb1b89, 0xbcc3eeb6, 0xbb63f8d0, 0x3b5d0920, 0x3c7f3260,
      0xbc7eded6, 0x3b92da00, 0xbcd09772, 0xbc1aaeca, 0x3b887af8, 0x3b30f9c0,
      0xbc1271de, 0x3c7bf480, 0xbbacc7c4, 0x3bf94050, 0x39eef700, 0x3cfe6be2,
      0x3ba6fd48, 0x3cf0c08a, 0x3c864ec4, 0x3c3d8adc, 0x3c39ec04, 0x3c559018,
      0xbcf57d98, 0xbb3b4998, 0xbc8bebe5, 0x3ce85a80, 0x3cd94612, 0xbc9c54eb,
      0x3c863dd0, 0xbb1b4258, 0x3c520734, 0x3ca4d7f6, 0x3c23a534, 0xbcfd1249,
      0x3c136ff8, 0x3c363e40, 0xbc0f48ac, 0x3cacbb8c, 0x3d002fe9, 0x3c3188b0,
      0x3ca883a6, 0xbcecd078, 0xba64b400, 0x3cba0cd8, 0x3c2d21a4, 0xbcc0e6f2,
      0x3cd5764c, 0xbbaa6ba4, 0x3a229640, 0xbc6aceda, 0x3c2807fc, 0xbc0424d2,
      0xbc7c0b7c, 0x3d03326b, 0x3c9c079e, 0x3d0692e9, 0xbc815bed, 0x3bcbd930,
      0x3ce50d8c, 0x3b92e060, 0x3b7653b0, 0x3b54a100, 0xbb925abc, 0xbcae2f30,
      0xbc49f204, 0xbd041cb9, 0x3c25398c, 0x3c02e064, 0xbbb6a7b4, 0xbcd2ab4c,
      0xbb3b6810, 0xbd062bf0, 0x3c831b96, 0x3cb250f4, 0xbace1f20, 0xbd063cb8,
      0xbd04f07d, 0xbc5c98ca, 0x3cd69f0c, 0x3cab93d8, 0xbca46390, 0x3ae56760,
      0x3c82fb20, 0xbb7d0920, 0xbb598410, 0x3ced847c, 0xbcf9a16c, 0xbbf6d2a4,
      0xbb5fc100, 0xbbf5ec70, 0xbc289f9e, 0x3ccbd2b4, 0x3cb125fc, 0xbcc87a74,
      0xbcad2560, 0x3ae0a400, 0xbc46ea3c, 0x3bdbeb90, 0x3ccab62c, 0x3d085bd7,
      0x3c8defee, 0x3a4ab140, 0x3bae9858, 0x3cac64ac, 0xbc0f6640, 0x3c48f0f8,
      0x3b58d8c0, 0xbca89036, 0xbae91ca0, 0xbca5905c, 0x3d021283, 0x3cc9d1bc,
      0xbb641230, 0x3cb019ac, 0x3c3106f8, 0xbba6462c, 0xbcb789f6, 0x3b0e20c0,
      0xbd028913, 0xbc54c412, 0xbcc32ef2, 0xbc24a00e, 0xbc842a83, 0xbc560b70,
      0xbc80f1f4, 0xbcd5a765, 0xbce8c83a, 0x3cd8b576, 0x3d0865c3, 0xbc89f578,
      0xbb8213e8, 0x3c7a3ef0, 0x3c9a6d98, 0x3ccfd996, 0x3d0845b9, 0xbc81acfa,
      0xbc2001d2, 0x3ce472ec, 0x3cb135c2, 0x3b0551d0, 0x3cc4c242, 0x3cd884d6,
      0x3b3b1850, 0x3c7c5150, 0xbc7517ca, 0xbc21301a, 0x3c7bc704, 0x3d08556f,
      0x3bb73d70, 0xbb211778, 0x3cac175e, 0x3c6b1df0, 0xbd04d460, 0x3cd28a76,
      0xbc868687, 0x399dd180, 0xbc807a20, 0x3ac527c0, 0xbc811eba, 0xbcb3afc7,
      0x3cfc0b46, 0x3bb40008, 0xbc7a8480, 0x3cfa7bee, 0x3ce051c2, 0xbcc7c13c,
      0x3cc39354, 0xbccf8f92, 0x3cfec29a, 0x3ae72640, 0x3c913058, 0xbcbfbdd6,
      0xbccf56ba, 0xbc0d2c34, 0x3c181fd4, 0x3cfedab2, 0x3c32d850, 0xbc686362,
      0xba947f20, 0x3cf212ea, 0x3ce684ac, 0xbc8232b6, 0xbc611812, 0xbb1a0558,
      0x3cde538c, 0xbca671d6, 0x3c6983b0, 0x3b982af0, 0x3cd71bbe, 0x3ce264dc,
      0xbccc30b0, 0x3d07925b, 0xbb888cd4, 0x3c1a6750, 0xbc925e05, 0x39cde380,
      0xbcca592c, 0x3c993e04, 0xbbd516e0, 0xbcd39436, 0x3c3f6948, 0xbcb53bde,
      0x3c217a80, 0xbc734b9e, 0x3bc88528, 0xbd001508, 0x3c6ddefc, 0x3cc3da62,
      0x3c8492e0, 0x3baa9d08, 0xbcd78e4e, 0x3ccdddf6, 0xbd076d9f, 0xbca14969,
      0xbc89477c, 0xbc7c7d5e, 0xbb3e9b78, 0x3cba6fd8, 0xba1ed040, 0xbc9f92a2,
      0xbcdc14b8, 0xbc12d940, 0xbca06225, 0xbb73b498, 0xbcf71387, 0xbca9c7e7,
      0xbc401bde, 0x3d027ebd, 0xbce5ac21, 0x3bb2dd60, 0x39b74e80, 0x3cac36a6,
      0x3cee8dc6, 0x3ca42f80, 0xbca80fb8, 0xbca1a9ce, 0x3cd02acc, 0x3cff8f36,
      0xbcb035a7, 0xbc86df36, 0x3c895d9e, 0xbb010d50, 0x3c3fbe24, 0xbca37372,
      0xbbd679f8, 0x3c0e3900, 0xbcded5f8, 0xbc362b28, 0x3cfb7202, 0xbc224740,
      0x3c6f4b60, 0x3b249ea0, 0x3ce1555a, 0xbc5efff8, 0xbc72b468, 0xbc0aef2c,
      0x3cc4659a, 0xbaad4380, 0xbcc72a7e, 0x3b05b830, 0xbc906625, 0x3d03b25f,
      0x3c8675d2, 0xbb020940, 0x3b870b18, 0x3c6ea2ec, 0xbc839ed2, 0xbc95022b,
      0x3cdd0efe, 0x3c8e6b94, 0x3c6b2ae4, 0xbd01e45a, 0xbcd3c912, 0xbc4f0b16,
      0xbcfbd26c, 0x3d012885, 0x3cf7cc6a, 0xbcc5f110, 0x3cfb654a, 0xbceb9478,
      0xbd066a7b, 0x3cb263ce, 0xbd075302, 0xbca2b105, 0xbc8ae6b4, 0x3cdc000c,
      0x3c99a3ac, 0x3c66e8ac, 0x3b73a580, 0x3c6f9658, 0x3cb3f92c, 0xbb1ac588,
      0x3ae51580, 0x3c455478, 0x3b72a730, 0x3cc69b20, 0x3cf0bd3e, 0x3c2fc1b8,
      0x3c881b76, 0x3c228438, 0x3c8202d8, 0xbc8f8cb4, 0xbc19790e, 0xbce8c5a9,
      0xbc9bb7fa, 0xbd042ab2, 0xbc09cada, 0x3ceff31e, 0xbcc53527, 0xbc20125e,
      0x3cfb50e6, 0xbc8a1f1c, 0xbcf064d2, 0x3c0a9a44, 0x3c2ffb74, 0x3d06df61,
      0x3c85cb60, 0x3b0f1cf0, 0xbca1daa5, 0xbc626fc0, 0xbcf89569, 0x3ca82902,
      0x3cfa151a, 0x3c4d1cb8, 0x3c95953e, 0x3c75e4c0, 0xbc96a1a5, 0xbcadb9e9,
      0xbb564ad0, 0x3c9fd6ca, 0x3c11103c, 0x3c781618, 0x3d01b7cd, 0x3c9a5332,
      0x3c81b95c, 0xbca25ecb, 0xbcd0462c, 0x3ba95b00, 0x3b100a60, 0xbc167680,
      0xbc71a5b4, 0xbd05eace, 0xbc55eff8, 0x3bc1d7c8, 0xbcf8a26c, 0x3adf2040,
      0x3c973cc2, 0xbc299eac, 0x3cd58596, 0xbc4936b0, 0xbbaf8210, 0xbcded001,
      0xbd04c3fd, 0xbc5055f8, 0xbbde7dd8, 0xbcbcff9a, 0xbcd688a7, 0x3bc759e8,
      0xbc715d1a, 0xbcdfe1c9, 0x3c59054c, 0x3d00e963, 0x3c310d6c, 0xbd00c913,
      0x3ba053b0, 0xbc9ea8d2, 0x3cb64da4, 0xbc4c1f3c, 0x3c99e57e, 0xbce09e7c,
      0x3c221940, 0xbcc2b3f8, 0x3c2f0d0c, 0xbc05d274, 0xbb07ae90, 0xb9f01500,
      0xbbf8c744, 0xbc79be5a, 0xbcd1f01c, 0xbccee11a, 0xbcd6c525, 0xbcd31621,
      0x3b5daec0, 0x3ad21080, 0x3b0bd470, 0xbbcdfb60, 0xbcfa1036, 0x3af4e9c0,
      0x3c58bc58, 0x3cae1f16, 0xbb8ac8d4, 0x3cd874fc, 0xbcef4496, 0x3d077d85,
      0x3c53131c, 0x3bb24160, 0x3c8f6ba6, 0x3c8789e2, 0x3c30b3a4, 0xbc92e285,
      0xbcc14d50, 0x3c8ae082, 0x3bee3e48, 0x3b6be940, 0xbd0663dd, 0xbc26f4e2,
      0x3b98eb48, 0xbcfdd825, 0x3c1208d4, 0x3cb6d790, 0xbbc5863c, 0x3cc3db4a,
      0xbcc25c52, 0xbc8a5f9a, 0x3c512028, 0xbc88a30f, 0x3affbc80, 0xbcd9dae1,
      0xbcea2bd4, 0xbd041e89, 0xbc510d9e, 0xbc331eb4, 0xbc83865e, 0xbcbaacc3,
      0xbbe884a4, 0x3c7a9b78, 0xbc0aab38, 0xbc71dfca, 0xbcd10df4, 0xbc8911f8,
      0x3c5b59e0, 0xbb2fa448, 0xbbcf8d34, 0xbcb92fbc, 0xbccb8ca9, 0x3a8387a0,
      0x3c64f1b8, 0x3ba10cc0, 0xbc8f796d, 0xbbd9941c, 0x3d023095, 0xbaa41f80,
      0x3c718c68, 0xbbf14b08, 0xbbe2ad60, 0xb9fde580, 0xbcc5e63a, 0x3cd9545e,
      0x3ccd066e, 0xba9188a0, 0x3cb0637c, 0xbccafae9, 0x3ca6512c, 0xbcaf97b2,
      0xbc9e2b67, 0xbbbe9c00, 0x3cefd74e, 0x3c44fad8, 0xbcbfd7e3, 0xbcf8ffb8,
      0xbc0d3f80, 0x3c944bbc, 0x3d0284d5, 0x3cf31682, 0xbcf6e33a, 0x3becadb8,
      0xbcdb1f76, 0xbca29585, 0x3cf70a36, 0xbce64ce1, 0x3cbf21f6, 0xbb86accc,
      0xbb82d144, 0xbb86e2a4, 0x3cb144ee, 0x3c41b0ac, 0x3c29d500, 0xbd02ccfd,
      0x3c295bdc, 0xbbd50ff8, 0xbc20074a, 0x3c1185ec, 0xbcb8bd8c, 0x3cc662ce,
      0xbc02e3da, 0x3ba90f28, 0xbc7e4d5a, 0x3c7bb3e4, 0x3c7578c0, 0xbc8638f4,
      0x3c8932e6, 0x3bcb98b0, 0x3cffe982, 0x3bd15de8, 0x3c11511c, 0xbb74d2e0,
      0x3c397ec8, 0x3c5a8b5c, 0x3b4b3f10, 0x3c925614, 0x3bb4fdf0, 0x3b437ee0,
      0x3bdca5c0, 0xbced3e05, 0x3cf667ba, 0xbc490970, 0x3c25bd30, 0x3ae84600,
      0x3c3daa0c, 0x3cb48bba, 0x3c92b91c, 0x3ce8accc, 0x3c6806f4, 0x3c474b98,
      0xbcbda465, 0xbc282952, 0x3ba98f00, 0xbd015790, 0xbb75dac0, 0xbc8958a0,
      0x3ce9ecf8, 0xbc7ac2f8, 0xbc35c722, 0xbcd5724c, 0xbb970b70, 0xbcfac23f,
      0xbc59223c, 0x3d064b6b, 0x3cd9edf0, 0xbc996b56, 0xbc87ae27, 0xbccc0b52,
      0xbb9f8ef8, 0xbca4b58e, 0xbbe28f60, 0x3be20178, 0x3bd416c8, 0xbca49a65,
      0x3b57e510, 0xbc5c1822, 0x3cf8b3fe, 0x3c725b78, 0x3ca8bad8, 0x3becbf40,
      0xbc896c76, 0x3bcbe6c8, 0x3c882fb0, 0xbd06f482, 0xbc89658b, 0xbcf8d2b4,
      0xbc6a5ef0, 0x3bf410c0, 0xbc91a6fe, 0xbcfaa2e5, 0x3cfa6bb2, 0xbcbf90fc,
      0x3ce321b4, 0x3c8cd4e2, 0x3cb4ed6e, 0x3d07058f, 0xbb913f68, 0xbc30e016,
      0x3cfba0be, 0x3cafb562, 0xbc7afca8, 0x3c760d78, 0x3cffd0b6, 0xbbfa40a4,
      0xbb652320, 0xbca00bb8, 0x3cb2cf52, 0xbc2cb5e6, 0xbb24c730, 0x3bbf30f0,
      0x3ca9129e, 0xbc110734, 0xbc95aa5c, 0x3b5d6200, 0xbb779658, 0x3c21a1e0,
      0xbba74de8, 0xbc8114cd, 0xbc7a81f4, 0x3afc0960, 0xbbbe3368, 0x3cb80172,
      0x3c5be36c, 0x3ccbf4f6, 0x3cfe4eea, 0x3d03d2c3, 0x3d010235, 0x3bdc42d8,
      0x3cb27b2c, 0x3c97042c, 0x3bd8df90, 0xbc1548fc, 0x3cbc4482, 0x3c3557f8,
      0xbc93bfc2, 0x3cdb6014, 0xbc4df98a, 0x3cc0033c, 0x3ba6f1e0, 0xbcdd3bc9,
      0xbc0b12ce, 0x3c9b0eb2, 0xbba77be0, 0x3caffcbe, 0x3c5cad38, 0x3ccb8db2,
      0xbc431178, 0x3a33fa40, 0xbc8160d6, 0x3bbcf268, 0xbc4bef80, 0x3d07b6b9,
      0x3a0c3300, 0xbbeb6a80, 0xbcaf4190, 0xbcd16ca7, 0xbc6da080, 0x3ceb5bee,
      0xbb9f38cc, 0xbc00d192, 0x3b785300, 0x3c06b97c, 0x3b93c240, 0x3c8bd8b4,
      0x3cfa14fe, 0x3c568320, 0x3ca499da, 0x3c2a230c, 0x3c19eff8, 0x3c2482e0,
      0xbb3d2220, 0x3c5f90a8, 0xbc56b300, 0x3c15a410, 0x3cbe3234, 0xbaed7980,
      0x3ce88b92, 0xbca9491e, 0xbcee77f2, 0x3ca6fbfa, 0x3c29c8ac, 0xbbea649c,
      0x3bb58118, 0x3cdb2380, 0xbc98d760, 0x3c42f7a4, 0x3d0277ed, 0xbc16893c,
      0x3cfe39f2, 0xbcf99b8c, 0x3d030f63, 0x3cee1518, 0xbcf60b83, 0xbc48bcd6,
      0xbd0517d7, 0xbcee31c5, 0x3c5551e4, 0xbcb993e0, 0x3c0a2330, 0xbca4f350,
      0x3c697358, 0x3c3a4504, 0xbcad56dc, 0xbcdab9a9, 0xbca75e82, 0x3d06e983,
      0x3c5a71a8, 0x3c9e62de, 0x3c1dd1e0, 0x3bf23190, 0xbc8f3a9a, 0x3cf537c2,
      0x3c242040, 0x3a120040, 0xbbdd1ecc, 0xbca04b92, 0x3be20df8, 0xbc84a4b6,
      0xbb8f3ec4, 0x3c83d1da, 0x3bb453c8, 0xbc24b8fc, 0x3cc4ba92, 0xbcb09eae,
      0xbc2fc592, 0xbce635b0, 0xbcc36103, 0x3c2033fc, 0xbc80c620, 0xbc5c8a62,
      0xbd00982d, 0xbc367e52, 0x3c94e682, 0xbb18a588, 0xbcf5cf30, 0xbcf262c5,
      0x3cdd5952, 0xbc64d3e8, 0xbca6c907, 0x3cf0221e, 0xbcab3122, 0xbcc67483,
      0x3cdbee8a, 0x3ab36c60, 0xbd055b2b, 0xbd0171ad, 0x3cddc1d4, 0x3c78b6cc,
      0x3cb1c924, 0x3ce5a9b2, 0x3cf4562e, 0xbc7189ce, 0x3cc552be, 0xbc01984e,
      0xbbcc9558, 0x3c82857a, 0x3ca8b45a, 0xbc0d5038, 0x3c8a2784, 0xbcd209ac,
      0x3cbfac22, 0xbc903992, 0x3ce3dd7e, 0x3c4ade90, 0xbb8d8388, 0xbc0bf322,
      0xbce40b12, 0xbc08d2c4, 0xbbcf3f10, 0xbc9dbcd8, 0xbb757868, 0xbc133c56,
      0xbc3b3884, 0xbbc0ea44, 0xbbcc339c, 0x3bf3e970, 0x3c612c38, 0x3c9aea24,
      0xbca0e329, 0xbc8f8ce5, 0x3b391e20, 0x3c0743d4, 0xbc7100b4, 0x3c64efc0,
      0x3cbd34f2, 0xbc091d9e, 0x3b8f85d8, 0xbce1e112, 0xbcf83f38, 0x3c844576,
      0xbb518158, 0x399b3200, 0x3ca199f0, 0x3cf71c4a, 0xbbd42b10, 0x3cbde68e,
      0x3b3decf0, 0xbce11d87, 0xbb0cc078, 0xbce45c32, 0x3c43f468, 0xbb547598,
      0xbbaadf00, 0x3cab0e3c, 0x3cce26f0, 0xbbb28100, 0xbaf39ec0, 0xbccbc369,
      0xbae36720, 0xbc43dd6c, 0xbc3d8eb0, 0x3cf4d142, 0xbcbafd72, 0x3c32f6e0,
      0xbb84964c, 0x3bbd58f0, 0xbb878990, 0xbce332e1, 0x3ca67338, 0x3cdf8a1a,
      0xbcce5a47, 0x3ace26e0, 0xbb9b8b34, 0xbc85f0f8, 0xbcc4c730, 0xbca5c9d6,
      0xbc952af6, 0x3c0ab3dc, 0xbb0f0878, 0x3cbd885c, 0xbb859890, 0x3cedb1ba,
      0xbab99e80, 0xbc58f14a, 0x3bb51110, 0x3ce09932, 0xbcb5d9ac, 0xbc73df40,
      0x3cd64410, 0x3cc52fc0, 0x3b025c20, 0x3bf84140, 0xbcd92003, 0xbc7f8a78,
      0xbd04a714, 0xb8ed9800, 0xbc6b9700, 0x3c4903fc, 0xbcfe6c32, 0xba821420,
      0x3c224138, 0xbcde6e72, 0x3bfcb690, 0x3c179a88, 0xbc94245a, 0xbcd2b4f8,
      0xbc17d444, 0x3cdb2d12, 0xbd031c6b, 0xbcff77e7, 0xbbc303e8, 0xbcaef625,
      0xbbfc5058, 0x3d077697, 0xbcc22e94, 0x3c5585e8, 0xb732e000, 0x3c51abb4,
      0xbc60da30, 0xbcccd774, 0xbc90ca76, 0xbb8a7e70, 0x3cf46ba2, 0xbc615830,
      0xbc1be238, 0x39ac7580, 0x3c451fa8, 0xbcdd102e, 0xbcddcf76, 0x3ceb918a,
      0xbcedd2fc, 0xbba2954c, 0x3ce29b2a, 0x3cf9262e, 0xbcd8746e, 0x3c3c5b8c,
      0x3d00f4b5, 0xbc7566ec, 0x3cf1e40a, 0x3cb4efca, 0xbd03898a, 0xbcbb4f43,
      0xbc3fee78, 0x3cfefe7e, 0x3b98c750, 0xbd00a394, 0x3bead0d0, 0x3b8a2f80,
      0xbca7883e, 0x3d03f5c5, 0xbcb2ada9, 0x3be4b8d8, 0x3cdd4c7e, 0xbc8a4bde,
      0x3c054e94, 0x3ca5bba6, 0x3c0da66c, 0x3cb9a6da, 0xbc93ca69, 0x3ca89d66,
      0xbbed9090, 0xbd02fca0, 0xbc7b4962, 0x3c16b6f4, 0xbc105e62, 0x3c88fb06,
      0x3cee3aec, 0xbcda8176, 0x3b18c550, 0x3c6f9088, 0xbc7a4678, 0xbcdf5f03,
      0x3d048979, 0x39974780, 0x3c1bafc4, 0x3c32731c, 0x3cdb015a, 0xbd072fcb,
      0xbb909b70, 0x3caafcee, 0xbca9a702, 0x3ce50694, 0xbcb946ae, 0x3c5a4d8c,
      0x3c776370, 0x3cb47aee, 0x3b144e30, 0xba2ecbc0, 0x3cdd2cfa, 0x3c0c5e8c,
      0x3b90be48, 0x3ce13622, 0x3b60c960, 0xb9ad5200, 0xbccc3952, 0x3c839c24,
      0x3cd6cd3a, 0x3c8a5350, 0xbc90c652, 0x3c273a8c, 0xbc496e04, 0x3ccec760,
      0xbc3d8130, 0x3c9a5790, 0xbccf7927, 0x3c9850bc, 0xbc8c5036, 0xbc1998b0,
      0xbccc6cb4, 0xbcabe992, 0xbc6b5956, 0x3cfa5006, 0x3c63f6ec, 0xbc03a9c0,
      0x3c1c9514, 0xbc65e956, 0x3c811300, 0xbb8cdd08, 0x3cab34e2, 0x3ccf0678,
      0x3c60903c, 0xbce9cab0, 0x3cff91ee, 0x3b0a9290, 0x3c909fe2, 0x3c01e3ac,
      0xbcf09eb0, 0xbcc7fdd8, 0x3b25ac50, 0x3a787440, 0x3cf41662, 0x3c102f14,
      0x3a304a40, 0x3b4fb7b0, 0xbd025e8e, 0x3cd64278, 0xbc0cf13c, 0x3bf58f40,
      0xbd072855, 0x3cecd648, 0x3bbb0928, 0xbc2ce778, 0xbc309b12, 0x3b8c99f8,
      0xbba8f370, 0x3cf43a86, 0x3bd94668, 0xbcaa5e18, 0xbbb6161c, 0x3b95dbf0,
      0x3c6dbcc4, 0x3c8d5b24, 0x3c7c49f4, 0xbccf4cd0, 0x3bf82f70, 0xbce838f2,
      0xbcc86ddc, 0xbcbbf870, 0xbc9286d8, 0x3cc315b6, 0xbc0d1ab4, 0x3d041bc1,
      0xbcc2ced0, 0x3c898e6c, 0xbb01c8b0, 0x3c78fca4, 0x3c88b30a, 0xbb1d6500,
      0xbcb29607, 0xbc55b7f4, 0x3b6541a0, 0x3c3adaa4, 0xbc4b6e0a, 0xba310b00,
      0x3cad7e60, 0x3cd3bbea, 0xbd0710f3, 0x3bf99a28, 0xbcee6783, 0xbcd1fbc3,
      0xbcff8c32, 0x3b2de760, 0x3c85e2bc, 0x3c225a0c, 0x3c7089ec, 0x3cfadfee,
      0xbb39f700, 0xba9c4c40, 0x3cc2587a, 0x3ad98420, 0xbc891b3a, 0x3c929484,
      0x3c5b714c, 0x3cb77140, 0x3c81fb3e, 0x3c1ddddc, 0xbc272b74, 0xbc8004d4,
      0xbcf83f81, 0xbca40800, 0x3cee3c16, 0x3cdcf55c, 0xbc9bd5eb, 0x3b31c050,
      0x3cc192ba, 0xbad2f7e0, 0xbc8bcad2, 0xbcabec49, 0x3c3e5cbc, 0x3ca47d1e,
      0xbcb624b8, 0x3cacd350, 0x3c86b740, 0x3bcacca8, 0x3ce795fe, 0x3c865f94,
      0x3c335cc0, 0xbc48b09e, 0x3c8c6676, 0x3cf56e2e, 0x3cfad54a, 0xbbd5094c,
      0xbcb5f870, 0x3be0d910, 0xbc2003d6, 0xbcf35f85, 0x3c974afc, 0xbc11e692,
      0x3cb93bf0, 0x3c8e9842, 0xbc777a22, 0x3ce3b1fe, 0xbc9649ae, 0xbcfd9416,
      0x3cfb7b7e, 0x3c6dd6c4, 0x3ca5f416, 0xbc6846b4, 0xbc3d9430, 0x3c3a61c4,
      0x3b080ee0, 0x3c854c2c, 0x3c86e442, 0x3ce59e08, 0xbcbf9792, 0xbcb9fb6e,
      0x3c6674e8, 0xbca52d89, 0x3bef2858, 0x3c5ca048, 0x3c9328ea, 0x3c8429cc,
      0xb9b56600, 0x3d017dbb, 0x3b80d380, 0x3cd5f764, 0xbcc2c9dc, 0xbb00f2c0,
      0x3cbaf692, 0xbc4dc70a, 0xbc816274, 0xbbcffbac, 0x3bce7118, 0x3bc7da10,
      0xbc9f8512, 0xbc97a6de, 0xbc1469da, 0x3c9269aa, 0x3cb89544, 0x3c9b692e,
      0xbcb2dc30, 0x3cc03fda, 0x3ced6b48, 0x3c4be2cc, 0x3cd1fb1c, 0x3ca287bc,
      0xbb83eef8, 0xbceaba38, 0x3cc1bcec, 0x3cc20e14, 0xbc09d316, 0x3c77650c,
      0xbc87d1eb, 0xbc1040b4, 0xbc9dfc90, 0xbc46351a, 0xbcd19b52, 0xbd03ecdd,
      0x3c53e170, 0xbcc6cfde, 0x3cdf14ee, 0xb942ba00, 0xbba498bc, 0xbaf51ac0,
      0x3caf214e, 0x3c91eaca, 0x3c4d1248, 0xbb66c120, 0xbcffc134, 0x3cd3cccc,
      0xbcc0de12, 0xbc3f1452, 0x3cdf777c, 0xbc153456, 0x3a8a4280, 0x3b5c5210,
      0x3d063d21, 0x3c9819dc, 0xbba7f3e8, 0xbc98f1c9, 0xbb7237e0, 0xbc296ad2,
      0x3cf493a6, 0x3cea3096, 0xbcd7aaf0, 0x3cdc22fa, 0x3c37b5e0, 0x3bc3b6f8,
      0xbc6cef70, 0xbcbac443, 0x3cf74e7e, 0x3cbb6b94, 0xbcb887a0, 0x3caf44ca,
      0x3d0586ed, 0x3cf087be, 0xbcf1b8c3, 0x3c409c94, 0xbc2ea070, 0x3c838c5a,
      0xbcfd5a52, 0xbc9a106e, 0x3cad01e2, 0xba5887c0, 0xbcb81a8c, 0x3a28d4c0,
      0x3c93d4cc, 0xbbec2890, 0x3c14e20c, 0xbbb3e588, 0x3b901de0, 0x3cb4c838,
      0x3ce3cdaa, 0xbca2c660, 0xbca49e52, 0xbb3b1a10, 0xbc89c3a9, 0x3cc9e23a,
      0xba077a40, 0x3c8dee9c, 0xbb0f8200, 0xbcca77b6, 0x3d01b6a3, 0xbc93a207,
      0x3cd24e4a, 0x3b28fb80, 0xbcddc2c9, 0x3c81739a, 0x3c7d9378, 0xbc1b41d6,
      0x3d0482fd, 0x3cdfd30c, 0x3c1b70bc, 0x3c8c97f6, 0x3cfc96f6, 0x3ce5d134,
      0xbc86b83e, 0xbbde6800, 0xbbbc5390, 0x3c067db4, 0x3b36df80, 0x3c39c950,
      0xbc8e241e, 0x384f4800, 0xbc8cca16, 0xbcc9d752, 0x3cc3ac4e, 0x3c58509c,
      0x3cd915d8, 0xbb54c4f0, 0x3bca4118, 0xbac029c0, 0x3cfe645e, 0xbcbcafc9,
      0xbc647f3c, 0xbc13e252, 0xbceb0dd6, 0xbc7c4280, 0xbcf4ee38, 0x3c9385e6,
      0x3cc63156, 0x3c065614, 0x3bf66ce8, 0xbb219cd0, 0xbc8922e7, 0x3c2350c4,
      0x3cfd73ce, 0xbb4c2648, 0x3d06b483, 0xbcc5490c, 0xbced9a3a, 0xbc5290e8,
      0xbcc0f476, 0x3b033bd0, 0xbc2d1796, 0xbc10aed6, 0xbcf2762e, 0xbc2d2d2c,
      0x3ceb5f4a, 0xbd072fe4, 0xbcf5f3f0, 0x3bf4c740, 0x3a7e76c0, 0x3ce2cf78,
      0xb9c43800, 0x3c69976c, 0x3c07f838, 0x3ca754d8, 0xbbfd4890, 0xbcea7156,
      0x3cfa4c52, 0xbbf6dc70, 0x3bf03178, 0x3ca5242c, 0x3cd0dfb4, 0xbbd57290,
      0xb9ded580, 0xbbb8849c, 0xbc57defc, 0x3b0e5140, 0x3cf08bee, 0x3b6479b0,
      0x3c19fc50, 0xbca7996e, 0x3cb4989e, 0xbca6f685, 0x3b905770, 0x3c943ad8,
      0x3cb9a884, 0xbd02fb7f, 0x3c324520, 0xbced2467, 0x3c32c130, 0xbc45e6fc,
      0x39c59100, 0xbcc75434, 0x3c33ddfc, 0x3cdd3ac8, 0xbb7d1b98, 0xbcaab509,
      0x3ca6efb2, 0xbc932129, 0x3bacdb10, 0x3cc83ef4, 0x3c49206c, 0xbc913570,
      0xbca464ab, 0xba8fd3e0, 0x3c53e098, 0xbd01d96e, 0xbbc080ac, 0x3bd3ec18,
      0xbcdc0fac, 0xbccac752, 0x3cd04de0, 0xbcef8683, 0xbcc73d98, 0x3cd46090,
      0x3c49129c, 0xbc90860d, 0xbb620900, 0xbce16cd6, 0x3c73b068, 0x3cc1cca4,
      0x3c9f8cb0, 0x3c0ae7bc, 0xbd007e70, 0xbc6bf64a, 0x3c10ce40, 0x3c806842,
      0x3c6b8f9c, 0xbb149d30, 0xbc943165, 0xbba8c2f8, 0xbc6602ca, 0x3bf70880,
      0xbca78598, 0x3bd60950, 0x3c91b7b6, 0xbc028270, 0x3b95c258, 0x3bd31868,
      0xbcb8cf4e, 0xbadefb80, 0x3cc43e7a, 0xbc60cb92, 0x3b84f658, 0x3c9a6fdc,
      0xbc5bbd00, 0xbc0e2aca, 0xbc035cfc, 0xbc90d0c9, 0xbcaac33a, 0x3b30caf0,
      0x3c295ce8, 0x3cb869b0, 0xbbf15acc, 0x3cd54d8e, 0xbbe3bbf0, 0xbb5b04d0,
      0xbbe1b92c, 0x3c000814, 0xbceca496, 0x3b779400, 0xbbc76cb4, 0xbd00f6b4,
      0xbc8d933c, 0x3b868df0, 0x3cf0f66a, 0xbc82fc69, 0xbcf99e34, 0x3cb27ace,
      0xbcd55074, 0xbc1ab3ac, 0xbc8a19fa, 0xbce33b2c, 0xbca12f94, 0xbb80169c,
      0x3cc3a4e4, 0x3c97d420, 0xbcc23a58, 0x3cb51f92, 0x391c5e00, 0xbcd65816,
      0xbc2689ac, 0xbc8ffebc, 0xbcba6b65, 0xbcbdf2c0, 0x3c0cc57c, 0xbc5d92e2,
      0xbac82ae0, 0xbc237ea6, 0x3b7791a0, 0x3cff167a, 0x3cb9dac0, 0xba552f40,
      0xbc9a7367, 0x3b8c9528, 0x3b4f7f80, 0xbc5021b8, 0x3cf19bd2, 0xbc270cac,
      0xbc10dfec, 0x3c3118c4, 0x3cfe03a2, 0xbb152ae0, 0x3af8a1c0, 0x3bbb4c88,
      0x3c6dc43c, 0x3ca2f458, 0x3d0859ad, 0x3bd38108, 0x3caa3ba2, 0xb92fc900,
      0x3ccf9f42, 0xba820b80, 0xba6fab40, 0x3c547820, 0xbc88d685, 0x3c03badc,
      0x3cb5d768, 0x3c9820ac, 0xbca33189, 0x3c81175a, 0xbabef0c0, 0xbc8005de,
      0x3cb58ad4, 0xbc8e5374, 0xbcf907ce, 0x3c606cec, 0x3cccb070, 0xbccdf9b4,
      0x3a8ed440, 0x3bf221a8, 0x3d00eb85, 0x3cfb2316, 0x3cd031c4, 0xbc126e78,
      0xbccac8e5, 0xbc092570, 0xbc08f39e, 0x3c94d7d0, 0x3c8ed01c, 0xbb08c9c0,
      0x3c783380, 0xbca348d6, 0xbcdce90e, 0x3c296fa4, 0x3c19b4d0, 0xba72c1c0,
      0x3c463cbc, 0xbb4e4c30, 0xbc9581e0, 0xbbe894ac, 0x3cc0110e, 0x3c567dd8,
      0xbd079242, 0x3d02a0b9, 0xbcda7594, 0x3cc4a896, 0x3cd7a102, 0xbc31d4e8,
      0xbc91580b, 0x3cbc3630, 0xbcd1b3de, 0xbce0b232, 0xbcd8a7b4, 0xbbfe6b68,
      0xbc9a3e3e, 0x3c7cd780, 0x3cab28e4, 0x3bf67e00, 0x3bf45a98, 0x3b34c7b0,
      0x3cac600e, 0x3cb8c0ee, 0xbc10e84a, 0x3c93942a, 0xbc835c69, 0xbc874abe,
      0x3cfe586a, 0x3c55092c, 0x3c998860, 0xbcb7bb92, 0x3a398e40, 0xbbfb08b4,
      0x3cc4bdfa, 0xbd0179c8, 0xbbbb879c, 0x3d04662f, 0xbc81349a, 0xbcadd850,
      0x3c3bfcc4, 0x3cf4e94e, 0x3c92d9e4, 0x3cdae0e0, 0x3ccc4c10, 0x3cc2d3c2,
      0x3bc41e98, 0xbc316b74, 0x3c9f2436, 0x3cb208f4, 0xbcc2c210, 0x39d52080,
      0xbb1b35a8, 0xbc74493c, 0xbc94a0be, 0xbcac5010, 0xbc6cf9ec, 0xbd016825,
      0xbcabafee, 0x3c6b4080, 0xbcae91d0, 0xbc87876b, 0xbb0e5278, 0xbc1e43d6,
      0xbc89471e, 0x3a792040, 0x3cd39710, 0xbca856b4, 0x3ceb18ec, 0x3c2df9c0,
      0xbce2eb89, 0x3cac9d54, 0x3c529e78, 0xbc4499a8, 0xbc73cca2, 0xbc3e9200,
      0xbcb196e2, 0xbb5a73f0, 0x3c10d424, 0x3c4347e0, 0x3d052a21, 0xbbc07a10,
      0xbccd0469, 0xbbbda124, 0x3c06a188, 0xbc9fca45, 0x3cbce2be, 0xbcba9fa0,
      0x3d069fff, 0xbcdd768e, 0xbc2aabbc, 0x3d03628d, 0xbc821212, 0x3cd850ac,
      0xb9a97d00, 0x3c9b54e4, 0x3cffa576, 0xbb306020, 0xbc97c870, 0xbc092178,
      0xbc9a433c, 0x38fbba00, 0x3b6db9b0, 0x3cae3c4e, 0x3c868404, 0x3c77936c,
      0xbcf4ad76, 0xbc9519d2, 0xbce868a7, 0x3d0258ab, 0x3caf0760, 0xbcbb6727,
      0xbcefee87, 0x3bd6b558, 0x3ba51868, 0x3aa09dc0, 0x3cace854, 0x3bf1af00,
      0x3c9d4d96, 0xb9e16600, 0x3aa309e0, 0x3bb5d0f0, 0x3ba5c0e8, 0xbc2706f4,
      0xbc3c2d68, 0x3ca0c67c, 0xbcbc3712, 0x3cd7e892, 0x3c9a4ca2, 0xbbd3c9e0,
      0x3ba71b10, 0x3cf872c6, 0x3cf8262a, 0xbc5d9af4, 0xbceeb1d0, 0xbd0851d6,
      0xbcfebe38, 0xbcc19e69, 0x3c8e3404, 0xbce02aff, 0x3cf8cc2e, 0x3b927af0,
      0xbbb2cec4, 0xbcd1db7c, 0xbc1d6752, 0xbb66ca20, 0x3c3d24b8, 0xbc901a78,
      0x3d015185, 0xbba40fd4, 0xbc4b5168, 0xbc1cf6fc, 0x3c197c84, 0x3c536ea0,
      0xbcd76f8c, 0x3b0af110, 0xbc8eefb6, 0xbb7e09c0, 0x3ce6a054, 0x3ce6187e,
      0x3cb4d80c, 0x3bdaeb68, 0xbcd227d6, 0xbcb259a9, 0x3ad60f60, 0xbd0343aa,
      0x3ce7f220, 0x3bd95478, 0x3ac64a40, 0x3b2accf0, 0xbca5b6f4, 0x3ca91db2,
      0x3b36e9a0, 0x3acf0de0, 0x3cc7697c, 0x3cd4fe9c, 0x3cb96ec4, 0xbb7665e0,
      0x3cfb295e, 0x3c8b574e, 0xbb197400, 0xbc0cd8d6, 0xbc2690da, 0xbc2709ec,
      0x3c8f8f54, 0xbcdec5d4, 0x3c90eab8, 0x3cf5cd9a, 0x3c6a9ddc, 0x3cb00db0,
      0x3c8b9f66, 0xbcd01c30, 0x3bfd2208, 0xbbc3d668, 0xbc8203b2, 0xbce0bf5c,
      0x3cac1132, 0xbb484658, 0xbcbc193e, 0xbcadc67a, 0x3cb05378, 0x3b9e5580,
      0x3c62ffac, 0x3c982f4e, 0xbb4ce1e0, 0xbc3c35a2, 0x3cab481a, 0x3b239210,
      0xbbef0144, 0xbcd40950, 0xbc05a4b4, 0xbb3d0220, 0x3cc4a362, 0xbbc72970,
      0xbcae4707, 0xbcb11310, 0x3cd1d7c2, 0xbceb50fa, 0x3ce35e8e, 0x3c96304a,
      0x3c63585c, 0xbcd107c5, 0xbcc526cc, 0x3c93daae, 0xbcc40e29, 0xbc876763,
      0xbb548cf0, 0xbad0e680, 0x3cda8d8a, 0xbcdb2967, 0xbc136a8e, 0xbcc601fc,
      0x3c059438, 0x3c595b5c, 0xbc588d6c, 0x3cb2e0cc, 0xbcb73f9e, 0x3c11da08,
      0xbcce542e, 0x3ca4aa78, 0x3bcda018, 0xbcf664e3, 0x3c658e78, 0x3c998bda,
      0x3c2f8e3c, 0xbc975ec5, 0xbd013f3c, 0xbc564740, 0xbcb82ab0, 0x3d0812d3,
      0xbcc6603e, 0x3c4f3978, 0x3cdf3bc8, 0x3ce7d418, 0xbc4d583c, 0x3ce67cb8,
      0xbce87889, 0x3abff540, 0xbcf85d21, 0xbc7c30f4, 0x3c333be8, 0x3c6bec34,
      0xb68b8000, 0x39771400, 0x3c1c3ad4, 0x3cf6e4ba, 0xbcbe77a0, 0x3ab0ede0,
      0x3be31df0, 0xbd008ef5, 0x3cc83032, 0xbc6a01e2, 0xbd08206e, 0x3c0b457c,
      0xbbd61be8, 0xbc7bf32c, 0xbcccbca5, 0xbb5d0058, 0x3b9ac520, 0x3b48d520,
      0x3c5bd1d8, 0xbb9e8b08, 0x3cdefae8, 0xbce3210e, 0xbc76dede, 0x3cd2d6ae,
      0x3c9298d0, 0xbc0b8a8e, 0x3c0d3ba4, 0xbce6c883, 0x3c9f80dc, 0xbcb7cdda,
      0xbce736c9, 0xbcf4bc56, 0xbc6720fc, 0x3c9049d8, 0x3c16da64, 0x3c7d7d28,
      0x3b99c380, 0xba0c1400, 0x3c8e58de, 0x3aea2600, 0x3ce129b6, 0x3c2671b4,
      0x3ce6cb20, 0x3cf414e6, 0x3bc38258, 0xbb2cc788, 0xbcc67c90, 0xbc8b7a78,
      0x3c2e65dc, 0x3c0352f0, 0xbd06d63c, 0x3c7a6644, 0xbc4022f0, 0x3cb5c6e8,
      0xbc923efe, 0x3c479b9c, 0xbbb43554, 0xbb3e94a8, 0x3c714e5c, 0xbc82f3b6,
      0xbcd2b745, 0xbcefa4cc, 0xbc95a210, 0x3d00ac87, 0xbd04a9fd, 0xbcb1b2a2,
      0x3d01548d, 0xbc9ec1e7, 0xbcdbf316, 0x3c91ff46, 0xbc6642c6, 0xbb7ab610,
      0xbbddad78, 0xbce87b5a, 0xbbd5d360, 0xbcc499f0, 0xbcf86821, 0xbc65160e,
      0x3c4a65a0, 0x3c34d350, 0xbca790b0, 0x3cb698fe, 0x3b3b1bb0, 0x3cba0110,
      0x3c894804, 0x3cc69c4c, 0x39df2600, 0xbc02c58e, 0xbbc667f8, 0xbc7c750a,
      0xbb1f7f48, 0xb8d7cc00, 0xbc6b90fc, 0xbc965d74, 0x3cdd27e8, 0xbc22c1f0,
      0xbb181c68, 0xbb0bcf88, 0xbc821b1e, 0x3bf115a0, 0xbcb2d9b2, 0xbc7ce8e2,
      0x3c0d5f3c, 0x3ce6b0ee, 0x3c76580c, 0x3c87d3a8, 0x3cf60e7a, 0xbcc239b0,
      0x3cb97dfc, 0xbc0c2fd6, 0xbc34453c, 0x3d01b7cf, 0xbcd54bf6, 0xbbf960f8,
      0x3c7cbfa4, 0x3cd0a2ec, 0x3cde61da, 0x3c89fb82, 0xbb7a5178, 0x3cd6312a,
      0xbc990d45, 0x3cfc53d6, 0xbc8246f4, 0xbcb5ef9e, 0xbb5396a8, 0xbd0331b4,
      0x3c74ff60, 0x3c78771c, 0xbcb1e942, 0xbccdeb23, 0xbc663ec6, 0xbca7a238,
      0x3c970758, 0x3bb9fbf0, 0x3d01a921, 0xba6682c0, 0xbc3b0368, 0xbd00ea8e,
      0x3cf25042, 0x3cdd8a30, 0x3c5caf1c, 0x3d015f8f, 0x3ccdb6c8, 0xbc16039e,
      0x3c686a00, 0x3b1c0530, 0xbb78ee88, 0xbc30d322, 0x3bff3ea0, 0xbba0f880,
      0x3a8b4c80, 0xbce257ec, 0xbc801f3a, 0x3cb55980, 0xbc1f39ce, 0xbc988420,
      0xbc211ae2, 0xbca88fc7, 0xbc9628a9, 0x3cb5fac0, 0xbc7acf74, 0xbc54d4a2,
      0x3b6a6e10, 0xbc959eb8, 0x3ca89150, 0xbd017bae, 0x3cfe6956, 0xbcd49223,
      0x3c60def8, 0x3b141eb0, 0x3d05f1a5, 0x3c25c610, 0xbcc62d89, 0xbc4fa5da,
      0x3cca0248, 0xbb353698, 0x3cc27770, 0xbbf02760, 0xbc88fcf8, 0x3c4dbf28,
      0xbce4795a, 0xbc7a6e9a, 0x3c833294, 0x3cc77a1c, 0x3ca92676, 0xbc7db18e,
      0xbcca16be, 0xba0f4500, 0x3c830bdc, 0x3ca7ffb8, 0x3b89bea0, 0x3c22b0ac,
      0xbcba6938, 0x3d04a6bd, 0x3d012b07, 0x3cd724fa, 0xbc57ec4a, 0x3c75f5fc,
      0x3cc2c078, 0xbcb295f6, 0xbcfa7d8e, 0x3828f400, 0xbc845ce0, 0x3cb16880,
      0x3b2645c0, 0xbc9efdb0, 0xbcefe3cc, 0xbcf5a90c, 0x3ce8afba, 0xbd01b1b8,
      0xbc3dbaac, 0x3c96c3ac, 0x3b923e60, 0x3b9c4b90, 0xbc3ca5f4, 0x3b17cf90,
      0xbc94a080, 0x3ca88be0, 0xbc95f7d0, 0x3cbc7a42, 0xbc697880, 0xbcefdff0,
      0x3ca5127e, 0xbcb75db4, 0xbcc0d9c0, 0xbcbd1d63, 0x3a162300, 0xbc15aa44,
      0xbb7e3dc0, 0x3c5476ec, 0x3bd01d88, 0xbb93cbe0, 0x3c4cbf28, 0xbb848660,
      0xba55b1c0, 0xbd072216, 0x3b80cf00, 0xbc95fe14, 0x3cf997ca, 0xbd02b8ad,
      0x3c68c3ac, 0xbc0e8ff0, 0x3ce3c41c, 0x3ca6954e, 0xbccb68b0, 0x3cd639fe,
      0xbc9e0c2e, 0xbb0e12f0, 0xbc3295c4, 0x3bd48bf8, 0x3ce272b8, 0x3ceafbb6,
      0xbcdbec25, 0x39c08980, 0xbcd13f58, 0x3c691224, 0x3cda4486, 0xbc8c1baf,
      0x3c811d76, 0xbc941507, 0xbcee6978, 0xbcf777d2, 0x3ca78c30, 0xbc8fe2c0,
      0xbcc1247c, 0x3b0c50e0, 0x3c962eac, 0x3b0983b0, 0xbd086f5d, 0xbbfabcb4,
      0xbcbb954c, 0x3ccd797c, 0x3c3f6e40, 0xbaf64280, 0xbc4df5ce, 0xbc7ff25a,
      0xbc9ceb1a, 0x3c54e5e4, 0xbc7d635e, 0xbc60e4b8, 0x3b8901f8, 0x3d05c733,
      0xbcda7545, 0x3ce0f936, 0x3b853218, 0x3c3b9bc8, 0xbb400ba8, 0x3ca93066,
      0x3bd8f650, 0x3c976b9c, 0x3c6e5b44, 0x3c9588b2, 0x3cd7e08e, 0x3c9c0d04,
      0x3c4a9208, 0x3cdf445c, 0x3c9c475a, 0xbce6f71c, 0xbcf2c7b8, 0xbcadf7b6,
      0xbbd02480, 0xbcebd9fc, 0x3cb37056, 0xbcc0157c, 0x3cfc1d0e, 0xbc766028,
      0x3c170cf8, 0x3c33f4e8, 0x3ca2ede6, 0xbb96631c, 0xbcd56087, 0x3cdbe8be,
      0x3cf235fe, 0x3c1e8564, 0x3c989d7a, 0xbca5dda9, 0xbcc5e9fc, 0xbcbbf0b6,
      0x3cf0cd2a, 0x3b26e860, 0x3bc16ff0, 0xbc2b1f04, 0xbcbbfeb4, 0xbce34010,
      0x39b52b00, 0x3d06c2e7, 0x3c4c9f10, 0x3cc9808a, 0xbca949e7, 0x3cd4d572,
      0xbce2c890, 0x3d07fdb7, 0xbc885d69, 0x3cf9c136, 0x3cdae38a, 0x3cb90130,
      0xbb280730, 0x3c984e78, 0xbbbea2c4, 0x3ad55a20, 0xbb2a5c48, 0xbcc7a4dc,
      0xbab8ddc0, 0xbc23c838, 0xbbf56e10, 0xbcdfb5c9, 0xbd01f440, 0xbc8fe040,
      0x3cb35202, 0xbcd6d65c, 0x3cafb4d4, 0x3bf83fc8, 0x3cc0e8c4, 0x3a9cfc40,
      0xbbf5de2c, 0xbc86ba7e, 0x3c612e80, 0xbca7e7c2, 0x3c951086, 0x3a7f3100,
      0xbc222ba2, 0x3a2b3f80, 0xbd02cb75, 0xbc761a38, 0xbccb73f8, 0xbb654d58,
      0x3cf8f39a, 0xbc59d8da, 0x3bc73088, 0xbcbb77ee, 0xbcaac3c2, 0x3b2f3de0,
      0x3c99f858, 0xbcaf3ef8, 0x3cf272f6, 0x3cc7bb00, 0x3c9583b6, 0x3d000193,
      0x3c8a597a, 0x3d06ba1f, 0xbcbc1127, 0xbc716128, 0x3cff6a26, 0xbd00cef0,
      0xbcdea778, 0x3c637cd8, 0x3cba48a8, 0x3b97e3d0, 0xbc3d0756, 0x3cfada2a,
      0x3aa26620, 0xbc9ee5fe, 0xbcaea500, 0x3b922a98, 0x3b514990, 0x3cf00f8a,
      0xbcc87fbe, 0x3cfb788e, 0x3b68ca30, 0x3cb700f2, 0xbb83b5f0, 0x3ccb2974,
      0x3b1d7a20, 0xbc6a7830, 0xbb523e10, 0xbcd7564c, 0xbc88e6c7, 0xbcc10e87,
      0x3ca9ddd8, 0xbc97a34e, 0xbc657cb4, 0xbcfd32ba, 0xbcfd053a, 0xbce81c36,
      0xbca7be80, 0xbc823292, 0x3cb80ea4, 0x3d0515e7, 0xbc494468, 0xbc20589e,
      0xbcc50a50, 0xbc31deec, 0xbcc4e52e, 0xbc9a18d6, 0x3cae06e2, 0x3bb56e50,
      0x3ce89de6, 0xbce493ce, 0xbc842f60, 0xbac761a0, 0x3cd87ece, 0xbccc198e,
      0xb9db1700, 0x3cca6094, 0xbcd88987, 0x3c4f96e0, 0xbbb33800, 0xbbd6e59c,
      0xbbd72f80, 0x3c5aa5bc, 0x3d012807, 0x3d08738b, 0xbd02a798, 0x3abc77e0,
      0x3c6706dc, 0x3ca6f97c, 0xbc3bee9a, 0xbce09fd0, 0xbd0728c8, 0xbcb1f654,
      0xbc441f16, 0xbca4e430, 0x3c127260, 0xb952eb00, 0x3cb4170c, 0x3cd570a0,
      0x3cc889de, 0xbb114810, 0xbcbf652c, 0x3cddec56, 0xbc8a2027, 0x3d001313,
      0x3b4dca30, 0x3c8f457e, 0xbb2cec20, 0x3c87b5b0, 0x3c0328b0, 0xbbce5768,
      0x3ca4e6de, 0x3b9c9618, 0x3cf724c2, 0xbc73db70, 0x3bf391c0, 0x3cd0ac62,
      0x3c314338, 0x3ce7253e, 0xba952be0, 0xbca30987, 0xbcaf5902, 0xbcd52f45,
      0x3c4b986c, 0x3cf28ac2, 0xbcb18c20, 0xbd07464f, 0xbbcd4a10, 0x3d020e4f,
      0x3c3df3ec, 0x3ba9f5d0, 0xbb85d7cc, 0xbacfce60, 0xbacb3e00, 0x3cd1447a,
      0xbccaadb0, 0xbc80c71a, 0x3c645bc4, 0x3a8aede0, 0x3cc63036, 0x3b51ed40,
      0x3ce04b0e, 0xbce155b8, 0x3bca23f0, 0xbd025ade, 0x3b0541d0, 0x3b393de0,
      0x3ccbff10, 0x3cc9b2f8, 0xbc9091c3, 0xbcdf9e94, 0x3c65f884, 0x3c03063c,
      0x3c80e916, 0x3cd3157a, 0x3c864a42, 0xbcc95027, 0x3cadbf54, 0xbcb1d287,
      0xbc1a0b16, 0x3ca820ac, 0x3c3da888, 0xbbb85800, 0xbc7e76ac, 0xbbf8c3b4,
      0xbccb3947, 0x3bb0a468, 0x3b3264c0, 0x3b87fbe0, 0xbba42178, 0xbc813b4d,
      0xbcdfeffa, 0xbbcbc300, 0xbcab38f2, 0xbca1e7e2, 0x39faf280, 0xbbd2fac4,
      0x3c0dd660, 0xbd0247e8, 0x3a76b040, 0xbcf07389, 0xbba6cbac, 0xbc5724ac,
      0x3ca1835c, 0xbb485120, 0xbd045c4d, 0xbd0862e3, 0xbc4c91ca, 0x3bf601b8,
      0x3bbc7c88, 0x3c84c53c, 0xbcb74e2c, 0xbca33140, 0xbc3f1184, 0xbc4717c0,
      0xbc8ca905, 0x3cab9400, 0xbc075a00, 0x3c8eb8fa, 0xbcd7e487, 0x3cdb8d6a,
      0x3bbb5828, 0x3cc83b4e, 0x3c7895ec, 0x3b8d3d20, 0x3ce42ccc, 0x3c93a04a,
      0x3ccb2ddc, 0x3ce04e76, 0xbcf3bb07, 0x3bf4fd68, 0xbcc75da7, 0x3cfc2612,
      0xbcb04420, 0x3d008441, 0x3c7075c4, 0xbccd9065, 0xbcfe6869, 0x3c729d9c,
      0x3c831510, 0xbc7908e8, 0x3c8712ae, 0xbcdaf881, 0x3c97d9b6, 0xbba84654,
      0xbcbf7572, 0x3ccf1602, 0x3c9f1f9e, 0x3c0a65bc, 0xbcb7245e, 0xbc0d2f5a,
      0xbcc024d6, 0xbc99f849, 0x3c780564, 0x3cc912d4, 0xbcd3976c, 0x3c0949e0,
      0x3ba14aa8, 0x3c76a768, 0xbc1aea6c, 0xbcb66334, 0x3ca316ea, 0xbc345952,
      0x3cdd4074, 0x3c909030, 0xbcd1e95e, 0xbcc349e7, 0x3c81feb8, 0xbcb90a25,
      0x3bafcc58, 0x3c0026a4, 0xbccf477e, 0xbc951f02, 0xbcd82fc5, 0xbcaf1109,
      0x3bbaac40, 0xbd00e2ee, 0xba1e7800, 0x3d0535b3, 0x3bd01230, 0xbc3e3cc4,
      0xbbe0d370, 0x3bf913c8, 0xbccb8750, 0x3cc43eae, 0x3bf7e520, 0xbca03605,
      0x3cbe253e, 0x3cf76296, 0xbc07f126, 0xbcc1a187, 0x3c7020bc, 0xbc33e4bc,
      0x3c229120, 0x3c48fd20, 0xbacd2540, 0xbcb292ec, 0xbca7dad6, 0x3cc38596,
      0x3cc37a28, 0xbc3a190e, 0x3d07ba63, 0xbcb063b2, 0xbc884e5a, 0xbbb4d6cc,
      0x3bdec180, 0x3ce2c20a, 0x3ce07a62, 0x3bf6e028, 0x3b2c73b0, 0x3cda318e,
      0xbce13456, 0xbc5d567c, 0xbd03a14a, 0xbc72245a, 0xbbfc90ac, 0x3cd8f36c,
      0xbcc891bc, 0xbbb3cf08, 0x3ca0fff4, 0xbcff9263, 0xbbe7fb4c, 0x3a3f9540,
      0xbce8f9f6, 0xbc238c26, 0x3c0ec5c4, 0xbcd4159a, 0xbca68c87, 0x3c74ce08,
      0x3a03c5c0, 0x3cc5f898, 0x3c0a6bbc, 0x3c1941bc, 0x3d038b0d, 0xbc9c0280,
      0x3d0721a3, 0xbc86edb6, 0xbcacd716, 0x3cb7f5ea, 0xbc9f354b, 0x3b8f3ba0,
      0x3bcd97f0, 0x3bfc87d8, 0x3c58e8dc, 0xbb9e0460, 0xbcaa1569, 0xbcc06f03,
      0xbcc3bf3c, 0x3c1660d4, 0x3d042d93, 0xbcf2df70, 0x3cd05848, 0xbb45ba58,
      0x3cb61fca, 0x3c9f104a, 0xba885200, 0xbccb623e, 0xbce2d754, 0x3b6e8200,
      0x3c549d00, 0xbd017c4f, 0x3ca2a29c, 0xbc2b662c, 0xbbb1daf0, 0xbc8a599a,
      0x3d03e79d, 0xbd03922b, 0x3ce678ba, 0x3c956f06, 0xbba0f344, 0xbd035e33,
      0xbd037712, 0xbbb19388, 0xbc4da662, 0x3ce7433e, 0xbcf0de23, 0x3c15c75c,
      0xbc244f40, 0xbcef703f, 0x3ac81c80, 0xbc4753b4, 0x3cc77f4c, 0x3cd9fcb0,
      0xbd0676fa, 0xba851b60, 0xbcb7cbc9, 0xbd073d96, 0xbc599d62, 0x3b7bf5f0,
      0xbd0797b5, 0xbcbb0f58, 0xbb2bb658, 0xbbadcc3c, 0xbc9f687c, 0x3cd4c93e,
      0x3c2d2970, 0x3a0a3ec0, 0x3c1a9600, 0x3c149b6c, 0x3c8abef4, 0x3d07cd7d,
      0x3c10c2ec, 0x3cb67b3a, 0x3c8b5ff0, 0x3cb049a2, 0xbce69c5c, 0x3ce6ae48,
      0x3c83f6dc, 0xbb8aa460, 0xbb244e10, 0x3cbff150, 0x3c6cdaf4, 0x3c7d2c20,
      0x3cd80a72, 0xbc86b18b, 0x3ccfb030, 0x3bae4cb0, 0xbcd13558, 0x3ca1aeec,
      0xbbb59068, 0x3cebdcb4, 0xbd056059, 0xbcbab0c9, 0x3993f900, 0x3ce784c6,
      0x3cae515a, 0x3cc6f57c, 0x3cac1c6c, 0xbc2eca00, 0xbcc78d49, 0xbc59097c,
      0x3c8ab526, 0xbc93e3f2, 0x3bad1360, 0xbbfc319c, 0xbc385e00, 0xbc0b8466,
      0x3c90dfe6, 0x3badbe18, 0xbb06dd60, 0x3d02a7df, 0xbd03ddeb, 0x3d01ed87,
      0xbc549d5a, 0xbc9d4a5a, 0xbce81f3f, 0xbc8bc345, 0x3c96b0fe, 0xbd06035d,
      0xbd07efd5, 0x3cf98c2e, 0x3cfe8fee, 0xbc1cc90a, 0x39472b00, 0xbd01a0ac,
      0x3cfba286, 0x3cd6f678, 0x3b9318d8, 0xbbea0480, 0x3ca226bc, 0xbca48882,
      0x3ca03a2a, 0xbd00b593, 0x3cd8847e, 0xbb0d1548, 0xbcddff27, 0x3a8bca60,
      0x3ce2085e, 0x3c8f949e, 0xbcc2c2b0, 0x3b83e398, 0x3c227ba0, 0xbcfe997d,
      0xbd0300b7, 0x3c6f8788, 0x3bbe5cd8, 0x3c8f9312, 0x3c41b04c, 0xbcf08636,
      0xbb6d7210, 0x3cfda6ca, 0xbc5ebc12, 0x3c41d5fc, 0xbc77e18a, 0xbc3ae016,
      0xbcd7fdbe, 0x3b6a6880, 0xbc9efc12, 0x3c7d8080, 0xbb9f6b3c, 0xbad5be00,
      0x3ac05de0, 0x3befe9d8, 0xbba7284c, 0xbc80608b, 0xbae089a0, 0xbc84d0c0,
      0x3c724c9c, 0xbc1f42de, 0xbcc92d7a, 0xbc4ae99e, 0xbd006a18, 0xbc819cd6,
      0x3cb06f54, 0x3cf0367e, 0xbcf8e2d4, 0xbca7fe22, 0xbccc1327, 0x3cc44204,
      0x3bec0bb8, 0x3c15b4d0, 0xb9fb1180, 0x3c96912c, 0x3cb75af8, 0x3c860fa4,
      0x3ce33c00, 0xba891ce0, 0xbcca8127, 0xbca4d3bc, 0xbccfd952, 0xbca2e57e,
      0xbc8d7916, 0xbc875bef, 0xbc9deeb6, 0xbbb0d908, 0xbcc6b3a7, 0xbb1fbf68,
      0x3ba764d8, 0xbcf7333d, 0xbd0448a2, 0xbcc365a3, 0x3bc116c8, 0x3ce96450,
      0xbbf58fa4, 0x3c3775c4, 0xbc737f96, 0xbc292156, 0xbba41a1c, 0x3cba6c00,
      0xbcdb4a47, 0x3a995be0, 0xbd04d654, 0xbc90512d, 0xbcf84b4e, 0xbbb9d44c,
      0x3c4791d8, 0xbbeebc60, 0xbc9dd8fe, 0x3c0f4000, 0xbb16c4e0, 0xbaf28020,
      0x3ca95434, 0x3bf73158, 0xbc4db56c, 0x3a133480, 0xbcb20d1e, 0x3c35d088,
      0xbcd254b0, 0x3cf70ace, 0xbd01e9b0, 0xbc4d3f22, 0x3cf1b496, 0x3b49f3e0,
      0xbc8f9f2f, 0xbc85c9d6, 0x3a1d5040, 0x3c00c560, 0xbc422a1a, 0xbc144cb0,
      0xbc98753e, 0x3c9531d4, 0x3a89f820, 0xb9a60600, 0x3b193030, 0xbcc29280,
      0x3cfd556e, 0xbcd1cbd4, 0x3abe7ac0, 0x3cd351e4, 0x3cc08c7a, 0xbba70f80,
      0xbc0e0530, 0x3cfc223a, 0xbc67cae8, 0x3cdc2f3c, 0x3c40b67c, 0xbcff7a8e,
      0x3bd87fd0, 0xbc249352, 0xbbdcc370, 0x3cc25198, 0x3c99b5d2, 0xbcf838ee,
      0x3cbae4c4, 0x3c3384f4, 0xbd02b0bd, 0x3d01755f, 0x3c43808c, 0xbcb477b8,
      0xbd02923b, 0x3d0692df, 0x3ca8f5b6, 0x3cb1187a, 0x3cb0dc94, 0xbcc47203,
      0x3c46c2cc, 0x3d07d1af, 0xbc897874, 0xbcb1804e, 0xbd077393, 0x3bf2fca8,
      0x3d0307a3, 0xbbed9be8, 0x3c5d87b0, 0x3c4761a4, 0x3c094538, 0xbbaa94e8,
      0x3c902bce, 0xbcd29c54, 0x3cae989a, 0x3ca0159c, 0xbcc4634c, 0x3cbe1a62,
      0xbb494998, 0xbb2d35e0, 0xbb4388e0, 0xbc2aa9e6, 0xbd05a907, 0xbbadf770,
      0x3ccb3196, 0xbcbbcc92, 0xbd01dd44, 0xbccee7b2, 0x3b82bcd8, 0xbbd39008,
      0xbcd1a99c, 0x3c066588, 0x3cdf6bc8, 0xbbc89024, 0xbccac8e7, 0x3d007d01,
      0x3ce69722, 0x3ccfe0cc, 0x3cdc3dea, 0x3c905bc4, 0x3bb61460, 0x3cef284e,
      0x3cc4bef0, 0x3b6700b0, 0xbab30ba0, 0xbc8c9f89, 0xbc18ffd6, 0xbc8698b4,
      0x3cb0599c, 0x3c08f170, 0x3caac12e, 0xbc29cfb8, 0x3c86bf56, 0x3bc4a360,
      0x3c10b9fc, 0xbcb47054, 0x3a95c360, 0xbceec009, 0x3b2eed90, 0x3cd8e19a,
      0x3cf5d8c6, 0x3ccb810a, 0xbc126644, 0x3cd5ae9a, 0x3c5734b4, 0x3bf9fee8,
      0x3cff98b2, 0xbc4414c0, 0xbcebdf85, 0xbcb8cbb8, 0x3bc4d5a8, 0x3c5be984,
      0xbbed5458, 0xbba4781c, 0x3cdb8b42, 0x3c52eab0, 0x3c960820, 0xbb56f868,
      0xbc3e8592, 0xbcaabb85, 0xba614f00, 0x3cfeb4aa, 0xbcd7f9ce, 0x3ce447c8,
      0x3c325164, 0x3cdef2e0, 0x3ce12e80, 0x3c2b4f64, 0xbcb51d72, 0xbbfdd4e0,
      0x3caec252, 0x3d084b45, 0xbca264cb, 0xbcc28620, 0xbb9d03e8, 0x3d03d5af,
      0xbc21079a, 0xbc1f78c4, 0xbcbc17e7, 0xbce1c11c, 0xbcae75f6, 0xbc2b3580,
      0xbce9c285, 0xbc9cf3da, 0xbbb54f80, 0x3c1a2ca8, 0xbcc8a238, 0xbb2fdbc0,
      0xbcf3359a, 0x3bf58e78, 0xbcd241d4, 0xbbbdb780, 0xbbf903cc, 0x3ca1aede,
      0x3cb51d94, 0xbcb93220, 0x3c9f3810, 0xbcaed450, 0xbb3c5198, 0x3cb1f008,
      0xbcced540, 0xbc24ff84, 0xbced7136, 0xbbb02670, 0x3ccc6cf2, 0xbc0286f8,
      0xbc88d8c0, 0x3c5fa5f4, 0x3ca98f76, 0x3cec80a8, 0x3ca7def4, 0xbca582ba,
      0x3c7f792c, 0x3c241300, 0x3b3010a0, 0x3bd62058, 0xbc1337f8, 0x3a21b640,
      0xbcc955b2, 0xbb7cc600, 0x3ca54724, 0x3c6f1a0c, 0x3c9fb978, 0x39f92d80,
      0xbcc11f1e, 0x3a006d40, 0xbc8ce134, 0xba6e3f40, 0x3ba9b720, 0x3b1c5da0,
      0x3caebcac, 0xbc8d3cf2, 0xbcacee5e, 0x3a8fc4e0, 0xb9df7f00, 0x3c82302e,
      0xbbf8d124, 0x3c29ee5c, 0xbbd647cc, 0xbb89cdf0, 0x3b13dea0, 0xbc41da12,
      0x3cdb47d4, 0x3ce61f08, 0x3ab371c0, 0x3b165e60, 0xbc46205a, 0x3cca411e,
      0x3d070b51, 0xbc5c7cc0, 0x3c1c63a4, 0xba816c40, 0xbce42087, 0x3ca64dda,
      0x3c54efdc, 0x3cc7dfea, 0x3cc16c96, 0xbbb68768, 0xbae18de0, 0xbcdadfc1,
      0x3c2accb0, 0xbc6a1e7c, 0x3c2dbf74, 0xbcd2bfbc, 0xbb36b010, 0xbccf57e7,
      0x3c8d0a92, 0x3d07900f, 0xba42a500, 0x3cc67a20, 0xbd00b7ea, 0x3ca53984,
      0x3cce6394, 0x3cf28e46, 0x3c229e30, 0x3cec6b26, 0x3ca12fac, 0x3c1294cc,
      0xbd052988, 0x3b87db80, 0xbc4ec378, 0x3cdb7142, 0xbcd7d0f2, 0xbd074a48,
      0xbca2c434, 0x3ba4d990, 0xbcb2c86e, 0xbc98fe14, 0xbbad4670, 0x3c235d4c,
      0x3ba07a88, 0xbb21fe78, 0xbc46f41a, 0xbc2096f0, 0x3d00192b, 0x3cd7bf32,
      0xbcccf990, 0x3c8a79f2, 0x3ca5ccea, 0x3cee5d52, 0xbce28307, 0xbcbf73c5,
      0x3c5ce758, 0x3c3ac870, 0xbcbeecc0, 0xbc0c4634, 0xbca1113a, 0xbc2634fc,
      0x3c362d28, 0x382c8800, 0x3c76a94c, 0xbc130fb0, 0x3d05ba49, 0xbcbcc798,
  };

  // 7
  uint32_t bias_vals[] = {
      0xbc3266ec, 0xbbe43fa4, 0x3c6898b4, 0x3c210868,
      0xbcec5130, 0xbb382ff0, 0xbcd3e31a,
  };

  // 4,1,1,7
  uint32_t output_exp_vals[] = {
      0x3f0ae6f4, 0x3e06d0c7, 0x3e8cb149, 0xbb4830b7, 0xbe872094, 0x3ea9f6e4,
      0x3e83094a, 0x3f29eae5, 0x3e4738be, 0x3eaa75dd, 0x3d940a8e, 0xbe7e27dd,
      0xbe0ac591, 0xbd849768, 0x3eb2a525, 0x3f244889, 0x3ec81eaa, 0xbd5edf10,
      0xbe003ea4, 0xbd500669, 0x3e220519, 0x3f356aed, 0x3ee21c12, 0x3e9c80a2,
      0xbe3e4da9, 0xbe3df2ea, 0x3e3846d2, 0x3dd1579c,
  };

  // 4,1,1,7
  uint32_t output_relu_exp_vals[] = {
      0x3f0ae6f4, 0x3e06d0c7, 0x3e8cb149, 0x0,        0x0,        0x3ea9f6e4,
      0x3e83094a, 0x3f29eae5, 0x3e4738be, 0x3eaa75dd, 0x3d940a8e, 0x0,
      0x0,        0x0,        0x3eb2a525, 0x3f244889, 0x3ec81eaa, 0x0,
      0x0,        0x0,        0x3e220519, 0x3f356aed, 0x3ee21c12, 0x3e9c80a2,
      0x0,        0x0,        0x3e3846d2, 0x3dd1579c,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, VALID_PADDING, NULL);
}

void test_same_padding_non_zero_strides_large() {
  input_set *set = &large_input;
  strides_input_set *strides = &large_non0_strides;

  // 4,15,10,6
  uint32_t input_vals[] = {
      0x3b227300, 0x3f239634, 0x3e391590, 0x3f09c42a, 0x3e166940, 0x3e069194,
      0x3f5d35bc, 0x3f1db843, 0x3f50936c, 0x3e8b542a, 0x3f1a91df, 0x3e086f24,
      0x3e493b6c, 0x3e70d520, 0x3da58e98, 0x3f47d07e, 0x3f4e3621, 0x3f672084,
      0x3f08a2e8, 0x3f243d5c, 0x3ecd7c1e, 0x3e99be2a, 0x3eb2bb0a, 0x3ef2679e,
      0x3f56d025, 0x3f2783aa, 0x3f2bf3f6, 0x3f71a850, 0x3eaf2578, 0x3e63f6dc,
      0x3e8df942, 0x3f15d269, 0x3eb210a4, 0x3f700f08, 0x3f28078d, 0x3e2abb70,
      0x3f155820, 0x3e59b2f8, 0x3e98d6d2, 0x3d1d8d70, 0x3f37490b, 0x3d292ac0,
      0x3eb0129a, 0x3e294e38, 0x3f3e92bb, 0x3eb536ce, 0x3ee0aad6, 0x3edb1218,
      0x3f259a84, 0x3d9b6888, 0x3d769400, 0x3f5bbbf0, 0x3e752aac, 0x3dee2610,
      0x3f501963, 0x3e646fa4, 0x3eb97480, 0x3f61aed5, 0x3f045fc5, 0x3f6eec5f,
      0x3ee9c662, 0x3f5da091, 0x3f008237, 0x3e22af00, 0x3eac1394, 0x3f16c6a8,
      0x3e95e63a, 0x3f7cb28a, 0x3f6e5ab1, 0x3e2ff794, 0x3dda8690, 0x3eb6527e,
      0x3f545081, 0x3ecc3588, 0x3ecdda7a, 0x3f5e48ef, 0x3e1dbd08, 0x3f388474,
      0x3ec9797c, 0x3e20ddfc, 0x3e210f04, 0x3dbf1058, 0x3dfef5f0, 0x3eca4c76,
      0x3ed27c6a, 0x3f1c72dd, 0x3d8caa50, 0x3f1732a3, 0x3e55c1dc, 0x3f7849e2,
      0x3f78300a, 0x3cecc400, 0x3f2920de, 0x3f443ee2, 0x3e526a40, 0x3e2ac644,
      0x3f3fa4c6, 0x3f14a766, 0x3ecf9714, 0x3f54295a, 0x3e6a3400, 0x3f3a3c48,
      0x3f4d1a5a, 0x3f649eb8, 0x3f632da3, 0x3f248220, 0x3e4dd434, 0x3efa92c4,
      0x3f0fc029, 0x3e5a078c, 0x3e76b808, 0x3f19a861, 0x3f774430, 0x3f77569e,
      0x3de9afd8, 0x3d90e268, 0x3f1ac701, 0x3f57f86b, 0x3f66a396, 0x3ecec390,
      0x3e88fe3c, 0x3df434e8, 0x3f31c5c9, 0x3e32ebf0, 0x3f75dba1, 0x3ee0ab1c,
      0x3f16a274, 0x3ea1bf7c, 0x3ed0d7ec, 0x3e1ad758, 0x3f45ef00, 0x3f27c019,
      0x3f7b46cc, 0x3f493cd3, 0x3f3c87aa, 0x3ef91326, 0x3efd7bd4, 0x3f16e60d,
      0x3db0d428, 0x3da74760, 0x3ea228e0, 0x3f1ae3e8, 0x3ea12492, 0x3ce1b520,
      0x3e2f785c, 0x3f129c0b, 0x3f04c774, 0x3e4c3798, 0x3eeab532, 0x3f0b3116,
      0x3ef157ec, 0x3f66aea5, 0x3eaf7386, 0x3ea57c6a, 0x3e1b21a0, 0x3f310164,
      0x3ea4b1c0, 0x3d53d630, 0x3f6469ee, 0x3ead75d0, 0x3f09cd39, 0x3e95e5c8,
      0x3bd4df80, 0x3de69818, 0x3e5265ac, 0x3f2da038, 0x3f16ca3a, 0x3f1f2851,
      0x3f3483a0, 0x3f6d0642, 0x3f4d0c24, 0x3db32fe8, 0x3d860e10, 0x3f330ec4,
      0x3f2fde45, 0x3e82c0f4, 0x3f0f720d, 0x3f6daebd, 0x3f03ae0e, 0x3f0b8600,
      0x3ebf31c4, 0x3ea4aa22, 0x3f1c78a3, 0x3c716e40, 0x3e4575d8, 0x3f57ae16,
      0x3e6a4260, 0x3f5e50c1, 0x3ecf21ea, 0x3f075442, 0x3ec51e74, 0x3edb6b48,
      0x3f2d9719, 0x3ec9526c, 0x3e70fc3c, 0x3afcb800, 0x3f2f4f9c, 0x3f34ab61,
      0x3e989b0e, 0x3f1ad501, 0x3f7b4eb2, 0x3f3c5a20, 0x3f398ad5, 0x3f21b58b,
      0x3f5339d1, 0x3eb99f66, 0x3f29336e, 0x3f198507, 0x3f1ff7c2, 0x3f0f7b7c,
      0x3f7da3d8, 0x3f6d870b, 0x3f70cbc9, 0x3f2684fd, 0x3e96332c, 0x3e3078d0,
      0x3e809b9c, 0x3f67939d, 0x3f387367, 0x3f067a8f, 0x3f3ab1fd, 0x3e35a1d8,
      0x3e998322, 0x3f173882, 0x3f4547d2, 0x3f7ab26b, 0x3ec8f190, 0x3f63d684,
      0x3df099a0, 0x3f7f5118, 0x3eec1784, 0x3f3eaf5f, 0x3f35a0ef, 0x3dcb6f90,
      0x3e946d0a, 0x3f0f8eb7, 0x3dc1fd68, 0x3f518bf1, 0x3f5a11d0, 0x3f246bec,
      0x3ef26256, 0x3e0f2670, 0x3f1b8e29, 0x3f129ede, 0x3ef82a46, 0x3eb39a8e,
      0x3dab1d08, 0x3e8b823a, 0x3f1be004, 0x3cb9b580, 0x3e9293da, 0x3f30e6c2,
      0x3f50a3ac, 0x3f1d72ac, 0x3e7b6bf0, 0x3e28a754, 0x3f36df9b, 0x3ddf8260,
      0x3c81b4a0, 0x3f07fcc4, 0x3f69986c, 0x3e1de344, 0x3ec52ec4, 0x3c57bfc0,
      0x3f0b3294, 0x3f63093d, 0x3f1b7dce, 0x3ed7e19a, 0x3e27a4e4, 0x3ddf4478,
      0x3f478ae5, 0x3f3f771e, 0x3e5b9e3c, 0x3ec4f3c4, 0x3eead50e, 0x3f321718,
      0x3f1b6d81, 0x3f276718, 0x3f40d46c, 0x3f17eb30, 0x3f56db9d, 0x3f35a402,
      0x3e24e830, 0x3f404ba7, 0x3f6d17d4, 0x3ea907b6, 0x3f55b98a, 0x3f31ac61,
      0x3f27cd25, 0x3e453574, 0x3efb8bb2, 0x3f2a78c8, 0x3c219740, 0x3e9733f6,
      0x3e1b3dac, 0x3e52b850, 0x3ea19c54, 0x3f74eea4, 0x3ec0888e, 0x3ea375a2,
      0x3e7d1340, 0x3efefce8, 0x3ef7b7a4, 0x3f15e1d4, 0x3e47e7c0, 0x3ed4e0e0,
      0x3f4c78ae, 0x3e464d58, 0x3f3c8f56, 0x3db95320, 0x3f540c87, 0x3ee0a846,
      0x3eee9cd4, 0x3f679b97, 0x3eeb1728, 0x3f2eda9a, 0x3f41cb41, 0x3e6a0438,
      0x3ed619c8, 0x3f0de10b, 0x3eb85afc, 0x3f102337, 0x3f228999, 0x3f0ba4fb,
      0x3f58c2e2, 0x3f475fb4, 0x3f0285eb, 0x3ea4b84a, 0x3f686f8d, 0x3c787280,
      0x3f483440, 0x3e87dcba, 0x3e186dd4, 0x3dbbe590, 0x3f0a98cd, 0x3f4f4b31,
      0x3f3c31a3, 0x3e7ec630, 0x3dbc4970, 0x3f0e6e6c, 0x3e6c6300, 0x3f533bb5,
      0x3e5252c8, 0x3f7bbeee, 0x3e512cf0, 0x3e6eaa88, 0x3ed19e52, 0x3ef2dfd4,
      0x3f66298e, 0x3e06c284, 0x3f7913ea, 0x3e086028, 0x3e983d58, 0x3eb12cde,
      0x3f69b259, 0x3f1bd003, 0x3e378cc8, 0x3d8336f8, 0x3f39dc93, 0x3f1ec3fe,
      0x3ebb3a58, 0x3f5da147, 0x3f6f35c2, 0x3f30f554, 0x3dbbca18, 0x3f2658b0,
      0x3f6f702e, 0x3f48373a, 0x3ef3b78e, 0x3f09770f, 0x3edf1ef2, 0x3f73bce8,
      0x3f166e85, 0x3f1bf0ab, 0x3f6d0fb0, 0x3f6dbb6b, 0x3e85114e, 0x3e36911c,
      0x3a967a00, 0x3f71e565, 0x3ea6d724, 0x3ef5c85a, 0x3eb070ea, 0x3e3fb128,
      0x3f0ec81c, 0x3eff7f18, 0x3e21e134, 0x3f119379, 0x3f726941, 0x3eaa67fa,
      0x3cd05980, 0x3f646c18, 0x3f343541, 0x3efb9554, 0x3f6e7051, 0x3e5560f4,
      0x3f2e20b0, 0x3f4584f1, 0x3f57bd33, 0x3e62445c, 0x3e0db0ec, 0x3e5e4a7c,
      0x3f6e32e6, 0x3efabfb4, 0x3f68f574, 0x3e5d16f0, 0x3f1504f6, 0x3ea6319e,
      0x3dfece38, 0x3ed875ea, 0x3f7bc660, 0x3ea7df5c, 0x3f00b6e2, 0x3f7bbfd8,
      0x3f44f6d3, 0x3f06d40f, 0x3ead5386, 0x3f2ab30b, 0x3f78b86a, 0x3f1d2711,
      0x3f78b429, 0x3f7b3075, 0x3f16c7f6, 0x3f4d8f95, 0x3f5c6505, 0x3e154bd8,
      0x3dac8ea8, 0x3f29ce96, 0x3f4a8764, 0x3f65ee0a, 0x3ef75684, 0x3e4668ec,
      0x3ea73150, 0x3e78878c, 0x3ec29160, 0x3eacb782, 0x3f6bcb4b, 0x3ecfc520,
      0x3f6c98ff, 0x3cf2ac20, 0x3f04bbf0, 0x3dda9218, 0x3f17043f, 0x3dd99ac0,
      0x3f2f7419, 0x3e151de8, 0x3f3d2a1d, 0x3e4904cc, 0x3f5a5aeb, 0x3e9df2ac,
      0x3f2c2b40, 0x3dbd7280, 0x3f467843, 0x3d777790, 0x3f141bc5, 0x3f2acbac,
      0x3ebcc4ac, 0x3f49ae6c, 0x3d903228, 0x3f7f1524, 0x3edee394, 0x3f10274a,
      0x3f5fe92b, 0x3eb6e0fa, 0x3db0b4d8, 0x3f57f931, 0x3f7d17f4, 0x3f7568e6,
      0x3f169c26, 0x3f47d4f7, 0x3f60c98a, 0x3f51a1e1, 0x3f406684, 0x3e214454,
      0x3e65d3d4, 0x3f105cc0, 0x3eeb2868, 0x3f00677c, 0x3d840968, 0x3f0c7af9,
      0x3e647864, 0x3dbddbb0, 0x3f0858ee, 0x3f65bdc4, 0x3f3c397f, 0x3f171337,
      0x3cf62fe0, 0x3ea6e188, 0x3f445b06, 0x3f153ac8, 0x3ebffa40, 0x3ea58e80,
      0x3e2098e0, 0x3f355606, 0x3f1bef0c, 0x3f083143, 0x3f46cd34, 0x3f6944c6,
      0x3f122dac, 0x3ec87b64, 0x3eb02282, 0x3f3d5363, 0x3e8d3f60, 0x3f4594ef,
      0x3f27eddc, 0x3f5e8d79, 0x3f510543, 0x3ec1869e, 0x3e01bb4c, 0x3f1e21f4,
      0x3eb8f602, 0x3f46079e, 0x3f4412bf, 0x3eb54668, 0x3b115600, 0x3f46043a,
      0x3c899840, 0x3f7c32fa, 0x3f297c08, 0x3e05d578, 0x3eb9b888, 0x3f62adb0,
      0x3ebfeb6c, 0x3dc73698, 0x3f7b0564, 0x3f6ad0a8, 0x3f271995, 0x3daf2d98,
      0x3f4f6c8a, 0x3f608210, 0x3f1e818b, 0x3f21c37f, 0x3f29bbe3, 0x3ec29db8,
      0x3f048eed, 0x3e578144, 0x3f7711d1, 0x3e4f9d44, 0x3eca2398, 0x3e903ca0,
      0x3ea10eda, 0x3f7ccd9d, 0x3f78c368, 0x3d899908, 0x3f7da29f, 0x3f07e4b3,
      0x3f643cb6, 0x3efd82da, 0x3de20110, 0x3f236212, 0x3f1add60, 0x3ecf797a,
      0x3f61c12b, 0x3f6b564c, 0x3e43140c, 0x3ec424fe, 0x3f665adb, 0x3d982838,
      0x3f3ccd0d, 0x3e960a94, 0x3e845d42, 0x3f6fe543, 0x3e81547a, 0x3f30a7ee,
      0x3e440f70, 0x3eee3322, 0x3eeead26, 0x3f36b3f9, 0x3f304acd, 0x3ed1598a,
      0x3ec1edda, 0x3e5f47d4, 0x3efd6506, 0x3ec67c2a, 0x3df9b4f0, 0x3e9df502,
      0x3f45e401, 0x3e589ad8, 0x3e1a93b4, 0x3f5c46b8, 0x3e621148, 0x3f7f5600,
      0x3f72baff, 0x3f4123e2, 0x3f71a85d, 0x3ead1466, 0x3daa2a58, 0x3f3c9ae3,
      0x3f4ca2fc, 0x3f60ab28, 0x3f65251a, 0x3eb71be8, 0x3f2a8be0, 0x3ef2cf8a,
      0x3de3cfa8, 0x3f5eedd8, 0x3dbe16d8, 0x3e00f3b4, 0x3e9c556c, 0x3e8e6816,
      0x3c8fe4a0, 0x3f47981b, 0x3f647967, 0x3f2302d6, 0x3f6db572, 0x3e756f1c,
      0x3f237ae4, 0x3f260622, 0x3f37b5f3, 0x3ebaa3a0, 0x3c14c2c0, 0x3ef040e0,
      0x3ecdd6cc, 0x3f09c5ba, 0x3eafffe0, 0x3ee2ad9c, 0x3e69c148, 0x3f45a742,
      0x3eef0b64, 0x3e971e1c, 0x3e92f68a, 0x3e8c65b2, 0x3f1d2171, 0x3f465233,
      0x3f7ce279, 0x3f728308, 0x3f2cfe96, 0x3f63220d, 0x3ee210e0, 0x3f21773f,
      0x3f13a5e2, 0x3e33a0d0, 0x3cf33a80, 0x3f734583, 0x3f53a2dc, 0x3d047510,
      0x3d723680, 0x3ef45e58, 0x3c239340, 0x3f2423d9, 0x3f0375eb, 0x3e9ca4c2,
      0x3ee4a250, 0x3ed13724, 0x3f460cda, 0x3efccdf4, 0x3f62d212, 0x3f0709fc,
      0x3e6e0788, 0x3ef183b6, 0x3f040b9f, 0x3f25cdd1, 0x3eecf9aa, 0x3f4d69af,
      0x3f2474ef, 0x3e183368, 0x3f4bacf2, 0x3e38d730, 0x3f5dcdc3, 0x3ebbe596,
      0x3edd6934, 0x3f03f7f2, 0x3e185098, 0x3eac703a, 0x3d1c79d0, 0x3dd411b8,
      0x3f364ceb, 0x3f070ed7, 0x3ea6470a, 0x3d52aae0, 0x3f720e21, 0x3ea081d8,
      0x3d94d6c8, 0x3f25e9c7, 0x3e288ac4, 0x3f01a8ad, 0x3eb0175a, 0x3ec99266,
      0x3effa4b2, 0x3f3923ca, 0x3f1f7ee6, 0x3e068364, 0x3f284e78, 0x3d899a78,
      0x3f3ada39, 0x3f70a7c2, 0x3e993112, 0x3ee1528a, 0x3f63bc2a, 0x3f0281a1,
      0x3f6e80be, 0x3ed4482a, 0x3f3d10e7, 0x3f2f11ec, 0x3d82e6e8, 0x3f30d8dc,
      0x3d59a860, 0x3f3a3acc, 0x3e18b574, 0x3f683a08, 0x3eb08d96, 0x3ebc9c68,
      0x3e9cb71e, 0x3d976330, 0x3f5cf76c, 0x3edc775c, 0x3f21f7d3, 0x3f63706b,
      0x3ecaac74, 0x3e8da316, 0x3e963970, 0x3ee921f2, 0x3e5ddfcc, 0x3f731a16,
      0x3d343f70, 0x3dbb54b0, 0x3eab397a, 0x3f2fd8b0, 0x3f590838, 0x3ef754ea,
      0x3db72170, 0x3f1223bc, 0x3f40aa88, 0x3e1133e8, 0x3f2e5bd2, 0x3eecf71c,
      0x3dbaeaf0, 0x3f6dfea7, 0x3f26cbe0, 0x3dfef600, 0x3f4ea023, 0x3eb03008,
      0x3f007f87, 0x3ec6fb9a, 0x3ef3a764, 0x3f788842, 0x3f1ff8ad, 0x3f49695f,
      0x3ee9599a, 0x3f55dff9, 0x3f69a690, 0x3eaaa87a, 0x3c923cc0, 0x3f6d2d26,
      0x3e9fddae, 0x3eaaa242, 0x3f077995, 0x3e22c624, 0x3e6e3abc, 0x3f5ce1ad,
      0x3e48f3b0, 0x3d9ebf48, 0x3ecec5be, 0x3f5dd00b, 0x3ef86604, 0x3f679d61,
      0x3cdb0360, 0x3df97ec0, 0x3f25ec6b, 0x3f363d3f, 0x3e596a0c, 0x3f7f2347,
      0x3edae694, 0x3f6132f5, 0x3d800fc0, 0x3f3bc71f, 0x3e0ad0e8, 0x3f486f7d,
      0x3df0ddd0, 0x3f5430fb, 0x3dfc07c0, 0x3f2bef96, 0x3e09ee0c, 0x3c31b900,
      0x3f65a5a4, 0x3e5a7adc, 0x3dfc5d50, 0x3f688721, 0x3f4147d4, 0x3f150ac4,
      0x3e480a64, 0x3f2369d7, 0x3e5f2f14, 0x3f6459bf, 0x3dcc8530, 0x3f2128f4,
      0x3e59dfd8, 0x3f09ee57, 0x3ebc11e4, 0x3e696ebc, 0x3f7b5d10, 0x3f4f2966,
      0x3e10c538, 0x3e419370, 0x3f612083, 0x3f7c7854, 0x3f2e0e5f, 0x3eb470f2,
      0x3d9a3708, 0x3e98d21c, 0x3f56c03b, 0x3f0c82c9, 0x3d1488d0, 0x3d99d150,
      0x3f7d2b78, 0x3f51c7dd, 0x3d4ac580, 0x3f452f56, 0x3c637bc0, 0x3f73d49d,
      0x3f2a91a9, 0x3f43c4bd, 0x3d0fdde0, 0x3ea53ab6, 0x3de05858, 0x3f7a2eff,
      0x3f3adcb7, 0x3e7da6ac, 0x3f6c400b, 0x3f2cf549, 0x3f410de1, 0x3f6c1df4,
      0x3f13f3e7, 0x3ece914e, 0x3ef81cf2, 0x3f517093, 0x3df8d6e8, 0x3d0103b0,
      0x3edab6cc, 0x3e07ff04, 0x3d7ceec0, 0x3f2c77d7, 0x3f52b252, 0x3e46be24,
      0x3f3ccb1f, 0x3e2b7570, 0x3e00e6a4, 0x3e858172, 0x3e8b1e94, 0x3f423cdf,
      0x3d7b2790, 0x3ec635ea, 0x3f086a23, 0x3f3babb7, 0x3eca00da, 0x3dd86da0,
      0x3e6d554c, 0x3f22bdb0, 0x3f7e1cf7, 0x3e299cd0, 0x3eedd0b0, 0x3f154835,
      0x3e9d9c14, 0x3f2b023e, 0x3eec0434, 0x3f48c4fc, 0x3f7f2970, 0x3f2bd569,
      0x3e95596e, 0x3e7d79c8, 0x3f2e5e73, 0x3f15abd8, 0x3e889f28, 0x3ecc220c,
      0x3f241599, 0x3f71b6f6, 0x3f2b4cb0, 0x3d0c1740, 0x3f5d7ec2, 0x3f54ce68,
      0x3ea8ff4c, 0x3f60bdde, 0x3d0b0820, 0x3ee349b2, 0x3e8a28a2, 0x3f3c4767,
      0x3f5b0ac8, 0x3f0af48f, 0x3ed0230c, 0x3e86b4e4, 0x3f2b0d4a, 0x3d98d4b0,
      0x3f0698a6, 0x3dae6af8, 0x3d30d310, 0x3ded8630, 0x3f635c9c, 0x3e9c1ce4,
      0x3cc29100, 0x3f5e699c, 0x3f774a6f, 0x3f2fc333, 0x3f7ca2dc, 0x3f6b4b10,
      0x3ebff2b8, 0x3edce210, 0x3f15b40b, 0x3f414d23, 0x3f295ef3, 0x3d1255c0,
      0x3e99ec1e, 0x3e67421c, 0x3f561b2b, 0x3e35e1c4, 0x3f396694, 0x3f0a8afc,
      0x3ea9504c, 0x3f2ba5b9, 0x3f765bb2, 0x3d4c0170, 0x3edf6faa, 0x3f53c21f,
      0x3eed2712, 0x3f240319, 0x3ea34d6e, 0x3f53e9d4, 0x3ed5939e, 0x3ec683c6,
      0x3e97b538, 0x3da0cf30, 0x3f198721, 0x3bea2380, 0x3e9b28da, 0x3e9341c6,
      0x3e4414d0, 0x3e86a172, 0x3f37d055, 0x3f4f6140, 0x3f6fa7aa, 0x3f7cf6be,
      0x3f63e38d, 0x3f36b40f, 0x3f25c24a, 0x3f57dcee, 0x3ee8e10e, 0x3d951730,
      0x3f266947, 0x3e6ef9e8, 0x3db91588, 0x3dd773d8, 0x3f0a5722, 0x3ec2aab2,
      0x3f696037, 0x3e81cbcc, 0x3f71584f, 0x3f2225ce, 0x3ded1468, 0x3df3b248,
      0x3f38aa36, 0x3f4c3d00, 0x3e5385c0, 0x3eaa1294, 0x3f67406f, 0x3e01e2b8,
      0x3eab3856, 0x3e1b7a90, 0x3f29ea98, 0x3eb2ee0c, 0x3ea0b8de, 0x3f74e2a7,
      0x3f37a4ba, 0x3e63dd0c, 0x3eba263e, 0x3f142153, 0x3f54cc97, 0x3e1833ec,
      0x3ce4fae0, 0x3f4885f6, 0x3e0e6090, 0x3d37e430, 0x3d485e60, 0x3f5f9d3f,
      0x3efa6ed6, 0x3f4c4b68, 0x3f27e414, 0x3e0d2370, 0x3e57096c, 0x3e329078,
      0x3f6994b6, 0x3f2c8ea3, 0x3eb01968, 0x3f2c91df, 0x3f1b4723, 0x3f75441d,
      0x3ed08cd4, 0x3e8e4594, 0x3f3bd23f, 0x3ec0c306, 0x3d793b10, 0x3da3fc98,
      0x3ed9eee2, 0x3f2ca505, 0x3f17327d, 0x3ded1528, 0x3f7d8646, 0x3ebd4142,
      0x3f60257a, 0x3f0b7bcc, 0x3e9b6968, 0x3ef868ca, 0x3f4fb17e, 0x3f66464e,
      0x3f7a95d2, 0x3ccc43c0, 0x3f1317ba, 0x3f020e47, 0x3f772072, 0x3d677c70,
      0x3de323d8, 0x3e8718e8, 0x3f6eea63, 0x3f30b4c8, 0x3f1b58d8, 0x3efc1cca,
      0x3d02e760, 0x3f35ad69, 0x3f1d6006, 0x3f4be0f1, 0x3d00f4f0, 0x3f2fdd10,
      0x3c8f0820, 0x3ec6a986, 0x3d8c8958, 0x3dd64960, 0x3f0c7fc4, 0x3f398ce9,
      0x3f0aaddf, 0x3ea13cd6, 0x3e1a154c, 0x3ea00ee0, 0x3e3a2e18, 0x3ee4b0fc,
      0x3eac5aca, 0x3ef1e564, 0x3f4b37cc, 0x3f7ae104, 0x3f53ab47, 0x3e6aadd8,
      0x3f437532, 0x3f2c7fb9, 0x3e098bd0, 0x3dbaf628, 0x3dc9ce08, 0x3efd63f8,
      0x3e8d38b4, 0x3f20934f, 0x3f238d1f, 0x3f76fb8b, 0x3ece9daa, 0x3f3c284b,
      0x3f621b30, 0x3e93a29c, 0x3f2e7902, 0x3d809498, 0x3e047910, 0x3e9265b4,
      0x3e5703a8, 0x3cf04620, 0x3f653718, 0x3f557720, 0x3e69fd90, 0x3f12701c,
      0x3f0d5de4, 0x3ea9fa56, 0x3f60d22b, 0x3f0e7bf6, 0x3f1e16cc, 0x3e23c218,
      0x3dbd6d60, 0x3d71d370, 0x3f25f6be, 0x3e99c3a0, 0x3e4554ac, 0x3f5e6597,
      0x3eaf413a, 0x3e83ba80, 0x3f390641, 0x3ec20c52, 0x3f64022f, 0x3f1904d1,
      0x3f7aeeeb, 0x3f5f26ed, 0x3efe093a, 0x3f2c3af0, 0x3d7c5340, 0x3e771b70,
      0x3f3c9ba2, 0x3f3a0f93, 0x3f05971e, 0x3ef2ea5a, 0x3eb8253c, 0x3f7ecfbf,
      0x3d086610, 0x3eb88c36, 0x3f55e59a, 0x3f37f6c6, 0x3d0f44b0, 0x3f44576e,
      0x3e5041b4, 0x3f2796f7, 0x3e2d9614, 0x3f7879d6, 0x3f767896, 0x3df10540,
      0x3f5a5f10, 0x3ed861b0, 0x3e8155c4, 0x3e009588, 0x3efb9706, 0x3f293d23,
      0x3efc23ca, 0x3f0e3473, 0x3f1571b8, 0x3ec84a82, 0x3f490ff7, 0x3e3bf170,
      0x3f16585d, 0x3f2f5cf0, 0x3e0ecf24, 0x3e21ebe8, 0x3eb877ba, 0x3f70ba33,
      0x3f47bb80, 0x3cbc7740, 0x3f495d88, 0x3e83f83c, 0x3ea633d8, 0x3e0cf644,
      0x3d8f7018, 0x3f41d8e8, 0x3f59a178, 0x3f1b532f, 0x3f46bc83, 0x3f73f2b2,
      0x3e01f8cc, 0x3ee03a50, 0x3e55f9a8, 0x3edd34f8, 0x3f29d7b1, 0x3f4026b2,
      0x3f37a4d2, 0x3f6ce2fd, 0x3e7684e4, 0x3f046475, 0x3f0a0a96, 0x3ed47f3a,
      0x3f5bea8b, 0x3e743e34, 0x3e4ffb34, 0x3eac9ac2, 0x3d876210, 0x3f7c9b4c,
      0x3e796594, 0x3e312c50, 0x3f4fc17e, 0x3ee8d182, 0x3ef8ac4a, 0x3e87a7fa,
      0x3f0c75c5, 0x3e7cb45c, 0x3e823ed4, 0x3f3da9a1, 0x3e82d524, 0x3ef124aa,
      0x3e53f980, 0x3f129f91, 0x3ec45ebe, 0x3f1c6cea, 0x3f6b277b, 0x3f21c8ee,
      0x3ea44fd0, 0x3f49df77, 0x3e2ba6c0, 0x3f3a2841, 0x3f3b98b8, 0x3f4a1a05,
      0x3f672f6c, 0x3f4f85ff, 0x3f216157, 0x3e55c7f4, 0x3f14fbd5, 0x3f62832a,
      0x3e863746, 0x3e699c84, 0x3eefa3f2, 0x3f2377ae, 0x3cb7d760, 0x3d092840,
      0x3f024d7b, 0x3f66ed12, 0x3f27d1a8, 0x3e2e9848, 0x3f7b5743, 0x3f1c8caa,
      0x3f355965, 0x3ef1a048, 0x3edc411a, 0x3f16c4b9, 0x3f0e6027, 0x3f5a8af9,
      0x3e8ba48a, 0x3f4da2e7, 0x3f34c842, 0x3ead1d9a, 0x3dcb0c20, 0x3eaa8a30,
      0x3f082b28, 0x3ee3d874, 0x3ed7ffe4, 0x3e9e102e, 0x3ede2698, 0x3f1fcd07,
      0x3ec59d64, 0x3e9040d2, 0x3f6990d6, 0x3ee6a87e, 0x3f551534, 0x3f001e99,
      0x3f5344d8, 0x3e9ad0bc, 0x3eb64558, 0x3f35d210, 0x3f0c8331, 0x3eed874e,
      0x3f1dac4a, 0x3ea80004, 0x3f67a0b5, 0x3f7f3807, 0x3e53cd80, 0x3ee9d09c,
      0x3e079ba8, 0x3f023023, 0x3ef7fdf2, 0x3e8fa2c4, 0x3e88f4a4, 0x3d15e2c0,
      0x3f0055cb, 0x3e3edc90, 0x3f0dd291, 0x3c7efa80, 0x3f7e1978, 0x3e39a244,
      0x3f63ad73, 0x3f6de737, 0x3e93b68a, 0x3e943ad4, 0x3ecb8976, 0x3f5b8b43,
      0x3f2e4fbf, 0x3db65498, 0x3e28e8f4, 0x3f322b22, 0x3f11aea3, 0x3f7af29d,
      0x3f4a4c62, 0x3f400888, 0x3e59df3c, 0x3f53e6a2, 0x3d4a6460, 0x3edcc86c,
      0x3f2b89f4, 0x3eb6a5e4, 0x3dfa75c8, 0x3f55e704, 0x3f13b35c, 0x3f68712d,
      0x3f424599, 0x3e6b005c, 0x3f4c60ae, 0x3edf3cb8, 0x3f582657, 0x3e52bb8c,
      0x3f65c0e9, 0x3f04782b, 0x3f3df458, 0x3f113abd, 0x3f388998, 0x3f675761,
      0x3ed4f14a, 0x3ecd2304, 0x3eefb426, 0x3f6c7beb, 0x3f710a5b, 0x3f46623c,
      0x3f22e4e3, 0x3cc090e0, 0x3f3dc384, 0x3e53b8a0, 0x3f23c092, 0x3f270aff,
      0x3f11881b, 0x3ef7a828, 0x3f312f67, 0x3f26f96f, 0x3f666e90, 0x3ef226a8,
      0x3f71c6c5, 0x3f6234f2, 0x3f6cac00, 0x3e7ab140, 0x3e6965e4, 0x3ee6d92c,
      0x3e6283b8, 0x3f5e2907, 0x3f70c9c0, 0x3edfc8f4, 0x3ee9d348, 0x3f3c36b2,
      0x3dedf718, 0x3ef4d5e6, 0x3f79dee7, 0x3ef4f828, 0x3ef67c60, 0x3ed8525c,
      0x3f280f4b, 0x3f608778, 0x3e96f1d6, 0x3f147649, 0x3dea3748, 0x3e37a768,
      0x3f51ff66, 0x3f1d0a7f, 0x3f6e2bca, 0x3f53c73b, 0x3eb64b60, 0x3f501bb8,
      0x3f373852, 0x3f5184b8, 0x3bc63c00, 0x3e93e44a, 0x3e288efc, 0x3e38ba08,
      0x3f35fbbb, 0x3f5048b0, 0x3f2eee5d, 0x3edaaa46, 0x3f12a981, 0x3e874fd6,
      0x3e16a03c, 0x3f45aefb, 0x3e4f9560, 0x3ea0a1fa, 0x3e583ad0, 0x3f7cc46b,
      0x3f391376, 0x3f4a498a, 0x3f57f889, 0x3edafebc, 0x3f0cd50e, 0x3f2801ac,
      0x3eecc4b4, 0x3d93a9b0, 0x3f1dc08b, 0x3f61a9f9, 0x3e46e06c, 0x3f39f92e,
      0x3dfba050, 0x3f1f1350, 0x3d8276a0, 0x3f7e9125, 0x3f60a00d, 0x3f5f0c12,
      0x3e558830, 0x3ed37dfc, 0x3f1d01bc, 0x3f1da1fc, 0x3f4d0476, 0x3e9ca804,
      0x3f3c6250, 0x3f2228c0, 0x3e49d904, 0x3eacdd64, 0x3f5e3ab2, 0x3f7c8ff0,
      0x3e33698c, 0x3e814d86, 0x3f70bf0b, 0x3f3dc425, 0x3f445f4d, 0x3e27dc04,
      0x3ebee4bc, 0x3f7f0527, 0x3eb3740a, 0x3e48be44, 0x3deda760, 0x3f081c3a,
      0x3f5e8c2d, 0x3eb45d9e, 0x3dbf1ed0, 0x3e3e09bc, 0x3e2a2c68, 0x3e0ebd48,
      0x3f287380, 0x3f38519f, 0x3e8a551a, 0x3f5122b2, 0x3f397f94, 0x3dd3e600,
      0x3f46edca, 0x3f6dbc40, 0x3f74de03, 0x3ebfedcc, 0x3e08fbe8, 0x3f0ed371,
      0x3eb57908, 0x3da559f0, 0x3f1d54e0, 0x3a916000, 0x3f568f8e, 0x3f010d0f,
      0x3b9cc000, 0x3e9fc0d4, 0x3f220e60, 0x3e3336e4, 0x3f367ad3, 0x3f52970e,
      0x3f09b04c, 0x3dca8b40, 0x3f60d904, 0x3e6ef8d4, 0x3e04bf9c, 0x3df93a88,
      0x3f5406b2, 0x3f6fcd2b, 0x3eb29bd8, 0x3f7db523, 0x3f2764da, 0x3f2cc8ea,
      0x3e3f5ff4, 0x3e4ccfa4, 0x3f57f964, 0x3f171a31, 0x3f3ca691, 0x3ed90664,
      0x3dbc1da8, 0x3f5f132f, 0x3f3c41a4, 0x3e9ba21c, 0x3f503055, 0x3f1b07bc,
      0x3e9bb2fe, 0x3ec3c94a, 0x3f020dc1, 0x3f1fec28, 0x3d841d00, 0x3ecff25a,
      0x3ee58bde, 0x3cb74980, 0x3f6f5ba1, 0x3f388085, 0x3ec092b0, 0x3f1c390a,
      0x3f6ebccb, 0x3e664650, 0x3eef27f2, 0x3f14bfa6, 0x3ee8a0cc, 0x3f0065d0,
      0x3f1f2548, 0x3edbbb54, 0x3e4c356c, 0x3f106c5c, 0x3ea1058e, 0x3d850760,
      0x3f509fcb, 0x3f718560, 0x3b839c00, 0x3eb6e0b4, 0x3f0e8edb, 0x3d6aaea0,
      0x3ef40940, 0x3cf20bc0, 0x3e20e9d8, 0x3f753841, 0x3dfa5240, 0x3ec00f88,
      0x3e73107c, 0x3e7332e4, 0x3e3e6dac, 0x3f7233bb, 0x3f385e2e, 0x3b4f6800,
      0x3f7f377d, 0x3f5a50a9, 0x3f2cf977, 0x3e36cb98, 0x3f421c56, 0x3ec57e14,
      0x3f1a8b0f, 0x3ea27ea8, 0x3f353c2b, 0x3f308a75, 0x3f0d810d, 0x3f419fe4,
      0x3dc3b938, 0x3e8fb798, 0x3ebd9bb4, 0x3f65c159, 0x3e8824da, 0x3f5e9eb6,
      0x3f4948d5, 0x3f29f64a, 0x3f0af015, 0x3da556a0, 0x3f1a32af, 0x3f0eda76,
      0x3f2aecee, 0x3e72c01c, 0x3f706328, 0x3e95d614, 0x3ef4523c, 0x3f19afb7,
      0x3e6f10c8, 0x3e5c692c, 0x3ef7842c, 0x3e07d914, 0x3e0e2e08, 0x3f4749b4,
      0x3f0a307b, 0x3f727e45, 0x3f482963, 0x3f58d43e, 0x3f7a85ef, 0x3ee3d7b4,
      0x3c0ba340, 0x3eb9240c, 0x3f2c40ce, 0x3f7e9466, 0x3ce1c320, 0x3f0086ee,
      0x3d9ab080, 0x3ecc54d2, 0x3f776ccb, 0x3ec39e70, 0x3f5d5768, 0x3eeded96,
      0x3ede3db6, 0x3d82a150, 0x3f756354, 0x3e04d104, 0x3eb8d7f6, 0x3f490930,
      0x3f7fcece, 0x3f15aedb, 0x3e1ad2dc, 0x3f49b470, 0x3f72bde7, 0x3f73bfb6,
      0x3eb2c40e, 0x3ecc8ed8, 0x3e85e01e, 0x3f04b4bb, 0x3f7980ed, 0x3ee04240,
      0x3f1ff613, 0x3f453cd9, 0x3e670308, 0x3f06966a, 0x3f207b74, 0x3f729c4b,
      0x3cffe4c0, 0x3f108324, 0x3e9600a4, 0x3e82ce5a, 0x3ec994f6, 0x3f3fe22c,
      0x3f4e94f3, 0x3f57e3f8, 0x3df9ba48, 0x3ea2a208, 0x3f572c4d, 0x3f62cb7f,
      0x3eb9817c, 0x3f672d44, 0x3d9e86f0, 0x3f5ccda2, 0x3f5e79de, 0x3f20ef81,
      0x3f33f733, 0x3e09f4ac, 0x3f547cd7, 0x3f2e4423, 0x3e01989c, 0x3ee78190,
      0x3f7b02f9, 0x3ee02960, 0x3f39c136, 0x3ec77460, 0x3e1bb978, 0x3f629d46,
      0x3eff370a, 0x3f439684, 0x3f6097b1, 0x3f027a4f, 0x3f706aad, 0x3f23a9be,
      0x3f113c59, 0x3f69e97d, 0x3ed2ab70, 0x3f18c6a9, 0x3ecf9680, 0x3f010123,
      0x3e12f558, 0x3f5c23eb, 0x3f3ec919, 0x3ec2cbd0, 0x3c922460, 0x3f7a6e4d,
      0x3e555d10, 0x3f7b3a25, 0x3c4161c0, 0x3f6287b0, 0x3e47d0b8, 0x3eac0d38,
      0x3ddd70f0, 0x3c9b68a0, 0x3f1d0ef6, 0x3f4738c3, 0x3ebc1506, 0x3f2d3f6a,
      0x3e87be3e, 0x3ea3e890, 0x3ee21114, 0x3dce2b98, 0x3d07ead0, 0x3ee9c9b2,
      0x3f032b58, 0x3edd9a9c, 0x3f1dfc57, 0x3f3d67cc, 0x3ea22e9a, 0x3eb85d7e,
      0x3eed110c, 0x3f630656, 0x3ec4af4c, 0x3f472946, 0x3f29af7c, 0x3e23486c,
      0x3f376384, 0x3f6e03d8, 0x3f336c44, 0x3dee7be0, 0x3e2ba424, 0x3f6d9ed2,
      0x3e22b904, 0x3c49ae40, 0x3ebdf868, 0x3f00cc53, 0x3ed1d02e, 0x3bf9bf00,
      0x3f6ea249, 0x3ec91796, 0x3ea9c620, 0x3f1e9ff6, 0x3f1eb6c8, 0x3f21c816,
      0x3d24dec0, 0x3e6eee2c, 0x3ee6ef9a, 0x3ee1aa84, 0x3f77ec2d, 0x3f492f6a,
      0x3f34cbd0, 0x3e24cc90, 0x3ed787e2, 0x3f112d80, 0x3e89a4de, 0x3c542280,
      0x3ec63388, 0x3e24d3d8, 0x3e49a7f8, 0x3f21fe49, 0x3f53e9ba, 0x3f7d9a39,
      0x3f088ec4, 0x3d9867f8, 0x3f79bb20, 0x3d852288, 0x3ed24420, 0x3e0f725c,
      0x3db85c00, 0x3f5f4bc9, 0x3f5ba8c7, 0x3e01f854, 0x3f419dbc, 0x3f3e906b,
      0x3f0cd816, 0x3ef6765e, 0x3f52b96e, 0x3f31a547, 0x3f703a25, 0x3f389339,
      0x3deb1c60, 0x3f57e834, 0x3e8c26c2, 0x3f45feb1, 0x3dd6fb30, 0x3f037d1e,
      0x3d9b0c90, 0x3e057c9c, 0x3f599f1f, 0x3e0c762c, 0x3f76a82d, 0x3f24342e,
      0x3eb22cc6, 0x3f519440, 0x3e18b588, 0x3f586054, 0x3e5475d4, 0x3f3ab683,
      0x3e267038, 0x3ef4188a, 0x3f5cf39c, 0x3ed45ff0, 0x3d62cf90, 0x3f588506,
      0x3ea692da, 0x3f5ee54f, 0x3ee416da, 0x3f56b3f2, 0x3e951744, 0x3ef06706,
      0x3f664f2d, 0x3f3a3575, 0x3ed6c5fc, 0x3eac3316, 0x3ebd8dd8, 0x3f0f22e4,
      0x3f174bd5, 0x3ee9f2ae, 0x3e721ae8, 0x3f2835cf, 0x3f118f07, 0x3f0b0cd0,
      0x3f262426, 0x3f4a4512, 0x3f02b1ad, 0x3f60fcbe, 0x3ea682f4, 0x3f30fc50,
      0x3f305809, 0x3eabdaf2, 0x3eff5ea0, 0x3f423fd7, 0x3b88ef80, 0x3f2a4423,
      0x3f1d6e73, 0x3f143475, 0x3ef5d70e, 0x3cca67e0, 0x3f1e6546, 0x3f385b79,
      0x3f35ab29, 0x3f122ab6, 0x3f49c5b6, 0x3ec25e9e, 0x3e039dac, 0x3ebad9f8,
      0x3f5e02b0, 0x3f024f09, 0x3da6eed0, 0x3d9e6bb0, 0x3ec78492, 0x3f2d84d2,
      0x3f0d85f5, 0x3ee70d2c, 0x3e345bcc, 0x3f1af613, 0x3f3fed1b, 0x3ecd5254,
      0x3f4e0343, 0x3f695429, 0x3e8bb032, 0x3f018a66, 0x3f12afca, 0x3f7bec67,
      0x3f39960d, 0x3f10af91, 0x3d8e22b8, 0x3f351413, 0x3eb0020e, 0x3ad9fe00,
      0x3edc1bee, 0x3e28de40, 0x3e6b1818, 0x3ea3b442, 0x3e016fac, 0x3ed68ad2,
      0x3e99569c, 0x3f49137a, 0x3f55621e, 0x3edce8de, 0x3f7f6758, 0x3ed778de,
      0x3f7f0637, 0x3e2ae554, 0x3f202c8b, 0x3f70d27f, 0x3e12963c, 0x3f340dea,
      0x3f76765a, 0x3f29672b, 0x3e74f948, 0x3ec404a8, 0x3f469499, 0x3e6c3cd8,
      0x3e4a7b58, 0x3f78f6c8, 0x3ece42bc, 0x3f7addc1, 0x3f4fd36f, 0x3ce962c0,
      0x3f5417df, 0x3f3fff09, 0x3e6101f0, 0x3ee25768, 0x3f21652c, 0x3d82b0f8,
      0x3eef5cfc, 0x3dda8d90, 0x3ea0d042, 0x3f25d9df, 0x3f3f94de, 0x3f6a0f1b,
      0x3e8608c8, 0x3eddeadc, 0x3f42b07d, 0x3e98b09a, 0x3f1ceb72, 0x3f739382,
      0x3f0004ac, 0x3ea38fc6, 0x3f5f92cf, 0x3f5dd3ed, 0x3f3c1247, 0x3e0960d4,
      0x3f17b78e, 0x3eeafc04, 0x3e762f94, 0x3f6d5945, 0x3f3702a2, 0x3e9f64e2,
      0x3eeb56ca, 0x3f494bbf, 0x3c81b540, 0x3f6de5b1, 0x3dd375e0, 0x3f274922,
      0x3e871a1a, 0x3f307c6a, 0x3f5c5c35, 0x3e5f5ca8, 0x3e63b024, 0x3f49ce50,
      0x3f47ecbe, 0x3f0d0f5d, 0x3e871360, 0x3dedf9c0, 0x3f5057e6, 0x3eded724,
      0x3f34c89b, 0x3f2068e8, 0x3eaf5e08, 0x3d902850, 0x3e595034, 0x3f3278a9,
      0x3f31579b, 0x3f20089d, 0x3f3263be, 0x3f262b0a, 0x3e057894, 0x3dce3d30,
      0x3ef6cfe0, 0x3d9413c0, 0x3f209b96, 0x3f7a0cb1, 0x3f718876, 0x3e5a9b0c,
      0x3f1f45f7, 0x3ec7e492, 0x3f0c2fcd, 0x3e840b70, 0x3f546fb5, 0x3f600fa1,
      0x3ea64ede, 0x3f120f60, 0x3f7ef5c7, 0x3eba7854, 0x3f2cf1bd, 0x3f1ef247,
      0x3e309b54, 0x3de4fe28, 0x3dba0a50, 0x3ec256fa, 0x3f07e755, 0x3ece9524,
      0x3e255ffc, 0x3ec766c8, 0x3e9dbbd6, 0x3cdc3500, 0x3f200393, 0x3f71f2f3,
      0x3f6499b1, 0x3f470399, 0x3eef0260, 0x3f502ece, 0x3e6ab1e8, 0x3f61eef3,
      0x3ea1c83a, 0x3e2c60a8, 0x3ecae280, 0x3f01aee0, 0x3f3d6ed0, 0x3f448546,
      0x3e459bb4, 0x3f6edde2, 0x3f7d07c3, 0x3f4f7933, 0x3f23e0eb, 0x3e2d0418,
      0x3e4fcec8, 0x3f35e581, 0x3f08ab57, 0x3e9397e2, 0x3ee99216, 0x3e98b25c,
      0x3f7df9ca, 0x3df3e7d8, 0x3f1f190c, 0x3e50ec10, 0x3ec4b8b2, 0x3efb67ae,
      0x3ec1f562, 0x3f7ff33b, 0x3ec77a62, 0x3e33b8d0, 0x3eff590e, 0x3f4408d8,
      0x3e4eb4a8, 0x3eafb0d2, 0x3f166ea0, 0x3c44d680, 0x3eec12c4, 0x3efeed80,
      0x3e84fa62, 0x3f59854e, 0x3e468548, 0x3cbd9500, 0x3e4082c8, 0x3f24a064,
      0x3f72db6a, 0x3f35d480, 0x3f279af7, 0x3f5d2516, 0x3f63b61a, 0x3f6f14a3,
      0x3ef771c2, 0x3e54ac3c, 0x3ebad670, 0x3e9d4f0c, 0x3ed954ba, 0x3f2a1382,
      0x3f478ce4, 0x3e431f24, 0x3e37db34, 0x3f48fbb0, 0x3e526f24, 0x3f08827c,
      0x3f42d461, 0x3f2c2877, 0x3f7846d1, 0x3f54c2bf, 0x3f737489, 0x3f5ce33c,
      0x3f388597, 0x3f6d26e0, 0x3f0cf819, 0x3ead938c, 0x3f64a4b6, 0x3da59d88,
      0x3f5bb7b7, 0x3ec3531a, 0x3f2c7d1f, 0x3e7817c0, 0x3f40fdb2, 0x3eca7f2a,
      0x3dad5650, 0x3f451750, 0x3eb5663c, 0x3e4113e0, 0x3f27640a, 0x3f590f3e,
      0x3f47f6f7, 0x3f761643, 0x3f0a6118, 0x3f429106, 0x3e388134, 0x3def1fa0,
      0x3f5f1956, 0x3eb708cc, 0x3dce13b0, 0x3f71fb6d, 0x3f2d06f2, 0x3eef075c,
      0x3f42109d, 0x3f7c6db5, 0x3f10916f, 0x3f258d7d, 0x3f6144eb, 0x3e859e92,
      0x3e7dfa08, 0x3ea6d616, 0x3edfcbfa, 0x3e96d04e, 0x3f3860ee, 0x3ebc752a,
      0x3e8c31f8, 0x3f0182ed, 0x3ed46c60, 0x3f595434, 0x3ea8ac68, 0x3f676640,
      0x3c3af8c0, 0x3f5e2d50, 0x3f28fb88, 0x3f7621f0, 0x3f01d8b4, 0x3f02033b,
      0x3d55bfe0, 0x3e7976e4, 0x3e115b50, 0x3efd8986, 0x3d1fdb10, 0x3ef19500,
      0x3f09c718, 0x3f4d05ce, 0x3f4be92f, 0x3f1f75f6, 0x3d2dad70, 0x3eaeba52,
      0x3f24ea89, 0x3ef8bce2, 0x3f3b2776, 0x3ef9a9ae, 0x3f4ab469, 0x3f3d4feb,
      0x3f4052e2, 0x3f21b064, 0x3f0f8001, 0x3ecfe7e4, 0x3eb79db8, 0x3ed0967c,
      0x3f06e2a0, 0x3e1962f8, 0x3f464dcb, 0x3f0c8942, 0x3f7d604d, 0x3e2c508c,
      0x3e6ef3a4, 0x3ed3f7ca, 0x3e10c3e8, 0x3f6f9f37, 0x3eebbaac, 0x3f15cef3,
      0x3ef75d62, 0x3e997036, 0x3f709b61, 0x3f4b4305, 0x3f5955f7, 0x3f1c7f75,
      0x3f0eacd7, 0x3f72cec1, 0x3ebf8114, 0x3f0c8ed3, 0x3f4742ee, 0x3f464d9f,
      0x3f75dcfe, 0x3e2c5308, 0x3e5945e8, 0x3eb73a0a, 0x3e758788, 0x3eee1b88,
      0x3f2e979f, 0x3f5f59c1, 0x3f012723, 0x3e8213e6, 0x3efe2ed4, 0x3f1e4948,
      0x3f11a540, 0x3f2c9681, 0x3f047e2b, 0x3f148f4b, 0x3ef172c6, 0x3f617cf0,
      0x3ee21d74, 0x3f5a87a4, 0x3f1c876a, 0x3d98f4e0, 0x3d2e0570, 0x3ea43b50,
      0x3f3e95c5, 0x3e608390, 0x3edf1b0e, 0x3ed9c0ac, 0x3f0516ce, 0x3f6532c6,
      0x3f5d2091, 0x3ea40344, 0x3f21a37e, 0x3f7adb96, 0x3c20e5c0, 0x3e227b3c,
      0x3f11773c, 0x3d519c50, 0x3f1c6945, 0x3e11dbe4, 0x3f0d3b70, 0x3e8804e8,
      0x3dfc17f8, 0x3f669265, 0x3edb721c, 0x3eafdb3c, 0x3f5bc025, 0x3f4724fc,
      0x3f7905ff, 0x3ecc3dc6, 0x3e2f1a60, 0x3e62e3a0, 0x3eb7ea3c, 0x3f1b97e4,
      0x3ed44ac6, 0x3f5184ce, 0x3d09b140, 0x3ee1a34a, 0x3f748cae, 0x3f763296,
      0x3d5cb6e0, 0x3e9bb7f0, 0x3f6e5e2a, 0x3eeb54ec, 0x3eb429be, 0x3e64c554,
      0x3e0195d0, 0x3f6eb90a, 0x3e6dce14, 0x3e9437cc, 0x3f3407a3, 0x3ec0175a,
      0x3d7a6b40, 0x3f195874, 0x3ec5015e, 0x3e1c1db8, 0x3ef41d56, 0x3ecb2b26,
      0x3f2c8782, 0x3f1efa48, 0x3f40ca3d, 0x3f75800f, 0x3f334287, 0x3f4d3838,
      0x3f63d41c, 0x3f6a4a13, 0x3e4f668c, 0x3f1df918, 0x3f66fe12, 0x3eafd92c,
      0x3e1da3fc, 0x3eb57c2c, 0x3e5e673c, 0x3e193e68, 0x3db898f8, 0x3f74ea47,
      0x3e941964, 0x3f24994a, 0x3f0f8cc0, 0x3ecd12ca, 0x3f535c86, 0x3eb28f96,
      0x3f031a2b, 0x3e6ac940, 0x3f03c80f, 0x3f1a73a1, 0x3f1df14f, 0x3f3e98f6,
      0x3f70c78e, 0x3f268bdd, 0x3f7998c6, 0x3ee4e72a, 0x3e00818c, 0x3ecc361e,
      0x3eaeb8a0, 0x3f1c39bb, 0x3bab2f00, 0x3f17ad32, 0x3f47d9b9, 0x3f6512cb,
      0x3f3dd5dc, 0x3f129b36, 0x3ef6e4ec, 0x3f130246, 0x3f400fe5, 0x3f4cecae,
      0x3f533548, 0x3ef5ccf0, 0x3e1911d8, 0x3e2ee04c, 0x3f1c197f, 0x3f2b6231,
      0x3e9ace5e, 0x3f3e1eaf, 0x3f6ac40d, 0x3f13fd09, 0x3f03eb36, 0x3f0b0e09,
      0x3e2bf2fc, 0x3ea72f8e, 0x3e845bc6, 0x3f65f06c, 0x3e6c6440, 0x3ed45fc0,
      0x3e99bdf2, 0x3e62ea74, 0x3f503c08, 0x3f5c98d4, 0x37760000, 0x3edabe46,
      0x3f69fcc7, 0x3ebe0d90, 0x3dccd1a0, 0x3e1df22c, 0x3dae55e8, 0x3f3c2eb1,
      0x3f50f0f1, 0x3f379a5a, 0x3f7bd5df, 0x3e90ea36, 0x3f69acab, 0x3ea7fd44,
      0x3f530572, 0x3d8696f0, 0x3e147390, 0x3eb72930, 0x3f3c68c1, 0x3f7f6004,
      0x3f0871dc, 0x3e8f7f80, 0x3ed03f4a, 0x3f27246a, 0x3e1fde98, 0x3f4bd5ca,
      0x3f3d3117, 0x3e018b30, 0x3ebf3038, 0x3ec2f070, 0x3f3796c9, 0x3f5c0430,
      0x3ed1e610, 0x3f547dc5, 0x3ef7b59a, 0x3e8e159e, 0x3f6ba379, 0x3f4539b2,
      0x3f5237cc, 0x3e563bd8, 0x3f3fcef1, 0x3ea22640, 0x3f1b5c43, 0x3e18c0d4,
      0x3d9429a8, 0x3f68b0f0, 0x3dd51f88, 0x3f154c7c, 0x3ea7ebae, 0x3f496013,
      0x3f7e4d14, 0x3e179958, 0x3f2399cd, 0x3edd1a5a, 0x3eddec8a, 0x3f7a5a26,
      0x3f27b67a, 0x3ea973d4, 0x3f46095d, 0x3f00956f, 0x3f077540, 0x3ec270b0,
      0x3f192a75, 0x3e9d0352, 0x3f40a28f, 0x3e9436ce, 0x3f291746, 0x3f042d5d,
      0x3e983db2, 0x3f315562, 0x3e9644dc, 0x3f4e2e54, 0x3db32a10, 0x3f0d8a75,
      0x3e19ea78, 0x3f76b767, 0x3f0571eb, 0x3f0cff22, 0x3d00cc70, 0x3e94c912,
      0x3efac0be, 0x3f2ef1d9, 0x3ce83380, 0x3f7f51c3, 0x3f197e77, 0x3f089771,
      0x3e0e7a54, 0x3f08b53a, 0x3da52488, 0x3e474f88, 0x3e7d6fc4, 0x3e0af4f8,
      0x3f0e389e, 0x3f2fc5cc, 0x3e125a90, 0x3e50ef88, 0x3f343717, 0x3f282bb7,
      0x3f3567c3, 0x3dc3d358, 0x3f37dc10, 0x3e825f14, 0x3f3b96f0, 0x3f487608,
      0x3f6f7596, 0x3f1b5452, 0x3f69697b, 0x3f7ed8c8, 0x3f233160, 0x3ea0a56e,
      0x3da6f468, 0x3e24c4ac, 0x3e914dea, 0x3ebd8b8a, 0x3f6d1064, 0x3f5c98d4,
      0x3e68ffc0, 0x3e4b04b0, 0x3f5a1db2, 0x3e9ab326, 0x3e1218fc, 0x3dd02ba8,
      0x3f44866b, 0x3e51c74c, 0x3ebd1708, 0x3f7227b7, 0x3f64907a, 0x3ee79d14,
      0x3f777af7, 0x3f771175, 0x3ef268d6, 0x3f326ec2, 0x3e8dbe4e, 0x3f75b929,
      0x3f375dc2, 0x3f2195da, 0x3d51ccc0, 0x3f2d8eb5, 0x3f689246, 0x3f57db6e,
      0x3e5ddc70, 0x3e437958, 0x3ea17d9e, 0x3f5d2337, 0x3da0b840, 0x3ebbee8a,
      0x3ea81168, 0x3ef0c596, 0x3ec6c514, 0x3e422e20, 0x3f40c452, 0x3f2bd24e,
      0x3e442174, 0x3c9587a0, 0x3ed7ec3e, 0x3f3429fb, 0x3f6dfcf8, 0x3ebf84b2,
      0x3f4a9022, 0x3f23dafd, 0x3f774fee, 0x3f0819bd, 0x3f6a9486, 0x3f44cb26,
      0x3d82f4c8, 0x3f357278, 0x3f56480b, 0x3ecc3076, 0x3ee6a840, 0x3f591675,
      0x3da4caa8, 0x3f718358, 0x3f2f435c, 0x3f08fcb0, 0x3f6680ef, 0x3f39e1ae,
      0x3f6e36b5, 0x3f314f87, 0x3f12309a, 0x3e0262b0, 0x3ee36666, 0x3f1af004,
      0x3da00238, 0x3d93b0b0, 0x3f007353, 0x3e7a2bd8, 0x3f1d6c7f, 0x3e193548,
      0x3e887c04, 0x3ecc8254, 0x3ece8006, 0x3eb2aa5a, 0x3e9b381c, 0x3f39d3ca,
      0x3ebe38de, 0x3f75894c, 0x3f49b065, 0x3f270eaa, 0x3f7ed96a, 0x3e6ed5dc,
      0x3f4c2350, 0x3e9635ea, 0x3de0b218, 0x3e9b0c62, 0x3ea4223e, 0x3f7df066,
      0x3f7bb910, 0x3f5b64a4, 0x3e13dda0, 0x3e113c14, 0x3db82f18, 0x3e912f40,
      0x3e4f07b4, 0x3f26b861, 0x3dbeda70, 0x3f37c8de, 0x3f0ac7fa, 0x3e979360,
      0x3dfff0f0, 0x3f4fd16a, 0x3ea211aa, 0x3e5fc338, 0x3dd4b6c8, 0x3f6be5a1,
      0x3f1ddc23, 0x3f500bab, 0x3e86441c, 0x3f402a0c, 0x3c4f3cc0, 0x3e8e86bc,
      0x3f24c881, 0x3e67bcf0, 0x3cf3b500, 0x3f6f5123, 0x3f35378c, 0x3f12fdd1,
      0x3f4b46a9, 0x3f34d642, 0x3f618818, 0x3ecb9592, 0x3f7a0d98, 0x3e297414,
      0x3f49405a, 0x3f1941c8, 0x3f60ed84, 0x3f16fd3a, 0x3f7cc034, 0x3db31428,
      0x3e3d6ac0, 0x3e04cb88, 0x3e7b7324, 0x3e93eb60, 0x3f794bef, 0x3f0c8e33,
      0x3f19ebb9, 0x3f7f787c, 0x3d9427d8, 0x3f490a6b, 0x3b96de80, 0x3eb57cdc,
      0x3e4325e0, 0x3f2ca37d, 0x3ed3a328, 0x3eb401d6, 0x3f6f8932, 0x3f4eab48,
      0x3f59202a, 0x3d749bf0, 0x3f727c17, 0x3f6db717, 0x3ebe009c, 0x3e841b4e,
      0x3f0eb52a, 0x3e8021be, 0x3f196472, 0x3f764abd, 0x3f1a2d89, 0x3f2be68f,
      0x3f54858e, 0x3f0ff02c, 0x3e340fe4, 0x3f2185cd, 0x3f7cb696, 0x3f26a29f,
      0x3eafc8f0, 0x3d90c708, 0x3edaf2e0, 0x3f2521cc, 0x3ef49a98, 0x3f58dcb6,
      0x3ea7d180, 0x3e1bb02c, 0x3f6a1aad, 0x3eb394a2, 0x3f301c25, 0x3f520bd8,
      0x3e07aeb4, 0x3f0b853a, 0x3f7984f2, 0x3f497a48, 0x3f314461, 0x3e325f8c,
      0x3d051e70, 0x3ded2620, 0x3ec6e40e, 0x3d518140, 0x3f7649c4, 0x3f1936a4,
      0x3f51b52c, 0x3ee85672, 0x3e944a30, 0x3f1540d1, 0x3f18997e, 0x3ba31200,
      0x3f552ab5, 0x3f7b8b95, 0x3ec389ba, 0x3f642864, 0x3ecfd75a, 0x3f2fb0f3,
      0x3f5da11b, 0x3e56fed4, 0x3eebb204, 0x3f6e77d9, 0x3f16f32e, 0x3f3f9efa,
      0x3d98b068, 0x3f37a106, 0x3f6f2d74, 0x3e820e62, 0x3f50fc13, 0x3f0fa747,
      0x3f3f2ca1, 0x3da95338, 0x3e06afc8, 0x3e9512ea, 0x3da86140, 0x3f51e6d0,
      0x3ad07e00, 0x3efbf832, 0x3dc26948, 0x3e8d6f54, 0x3e8ede88, 0x3f79a642,
      0x3f331cd7, 0x3f700323, 0x3ef31d8e, 0x3f78050a, 0x3f71aa8a, 0x3e4414c8,
      0x3f6f6c5f, 0x3e064450, 0x3d210320, 0x3e1c088c, 0x3f0a410b, 0x3ed78a88,
      0x3e1d6094, 0x3d4ce330, 0x3f7e5f07, 0x3f63d241, 0x3f4c3674, 0x3dc16bc8,
      0x3df80890, 0x3edabcee, 0x3eb94008, 0x3da77048, 0x3e9e736e, 0x3ee674e4,
      0x3df7be78, 0x3ef96414, 0x3f09281f, 0x3e2c02b8, 0x3e74362c, 0x3f2e0ee1,
      0x3f01ff5a, 0x3f3de5d5, 0x3d8127b8, 0x3f5b9b46, 0x3f7b98f1, 0x3e993f08,
      0x3f089379, 0x3f03021d, 0x3e32cabc, 0x3f19790e, 0x3cdba020, 0x3ddea6f8,
      0x3f356643, 0x3f76f320, 0x3f51fa9b, 0x3e5c9b1c, 0x3eb22fca, 0x3e0a7a20,
      0x3ed8e80e, 0x3f51cdf9, 0x3e9b8816, 0x3ec376ac, 0x3ec8e688, 0x3f04a40b,
      0x3f3ab4b5, 0x3e94d3c4, 0x3db22238, 0x3c023640, 0x3dbb4a48, 0x3f679fb9,
      0x3e8d1bf6, 0x3e420610, 0x3e1b9004, 0x3f16b085, 0x3f6a6b6e, 0x3f2a1907,
      0x3e39c22c, 0x3f576cf2, 0x3e3b22b4, 0x3f152e47, 0x3d638720, 0x3cbcc3a0,
      0x3ef31bc2, 0x3f1fca54, 0x3f451b37, 0x3e4b8b68, 0x3f23c45b, 0x3f248fcf,
      0x3f75e088, 0x3f04151e, 0x3f4e1756, 0x3f42adec, 0x3f6571ef, 0x3f322735,
      0x3f3c11c4, 0x3ede675c, 0x3f0e2a15, 0x3eb09402, 0x3e3b6b7c, 0x3c472800,
      0x3d898160, 0x3f39b80e, 0x3f480a1e, 0x3f5deff4, 0x3e55e4d0, 0x3f21db91,
      0x3ee97c30, 0x3e8f32f8, 0x3f5bb475, 0x3dbda638, 0x3f16f12e, 0x3f52ed8d,
      0x3f0257ca, 0x3f544fdb, 0x3f280268, 0x3e9a594a, 0x3ee546e8, 0x3e1c2810,
      0x3ec87438, 0x3f5fb336, 0x3f03501b, 0x3f2b7924, 0x3f22f754, 0x3d4aea80,
      0x3f02ec34, 0x3f7f79e5, 0x3f25fc9c, 0x3c08a280, 0x3dcb3898, 0x3e7b0694,
      0x3f0afae0, 0x3f3f0133, 0x3f768fbc, 0x3ddf8810, 0x3d6f1a50, 0x3f13fe60,
      0x3ea37c1c, 0x3f1f9dcd, 0x3f40042f, 0x3dc8cb58, 0x3f712d9b, 0x3f2302c8,
      0x3ef349a6, 0x3e61a508, 0x3f6d73d4, 0x3f355fca, 0x3e30d0cc, 0x3ef5a774,
      0x3f2d61c2, 0x3f5efa8a, 0x3f705611, 0x3edc25bc, 0x3f190445, 0x3f0350b8,
      0x3e40d918, 0x3e916348, 0x3ec95924, 0x3f52e13c, 0x3ef55e82, 0x3eb2f05c,
      0x3ea39b5e, 0x3f242856, 0x3f3284b4, 0x3e94b6fa, 0x3e9fd99c, 0x3f684583,
      0x3d4b4f00, 0x3ee81350, 0x3ef81ac2, 0x3f0faca6, 0x3ef5fe10, 0x3f53eda8,
      0x3f7f633f, 0x3e54deec, 0x3f5d2e90, 0x3f14fe52, 0x3e9fe104, 0x3d0178c0,
      0x3e98eb60, 0x3f2444c8, 0x3d957178, 0x3f6bedcb, 0x3b8f6800, 0x3eb3c32a,
      0x3efcfd92, 0x3ee36d4e, 0x3ec223d0, 0x3ef54b3e, 0x3de4dc98, 0x3ef31280,
      0x3f776a3f, 0x3e0853a8, 0x3f6c49fb, 0x3e53d250, 0x3ebe0a70, 0x3f359f3b,
      0x3eafb468, 0x3e6b2b74, 0x3f02f05c, 0x3ecdf21c, 0x3d4ba150, 0x3eb68e26,
      0x3f2686fd, 0x3f19725c, 0x3f2a08d3, 0x3eeef34e, 0x3f495faf, 0x3f735e74,
      0x3e73594c, 0x3d8eb898, 0x3d8283b8, 0x3ecdba8a, 0x3e845460, 0x3ef0f7c0,
      0x3f086c16, 0x3e89c828, 0x3ea2e1be, 0x3edf4e2c, 0x3e295d7c, 0x3f7152ad,
      0x3f0002ea, 0x3f2e07dc, 0x3d9c6f88, 0x3eb81618, 0x3eee1092, 0x3e15a800,
      0x3e6d6918, 0x3e594614, 0x3f26ea9e, 0x3f3a22f5, 0x3da12e30, 0x3f54fb12,
      0x3d1c9100, 0x3f75dcac, 0x3f1702c1, 0x3cbffaa0, 0x3f5c777d, 0x3f246abf,
      0x3d8de358, 0x3e4e8564, 0x3c0ec480, 0x3f2f3d52, 0x3f4f0720, 0x3f5869b7,
      0x3d8d5068, 0x3f0ff831, 0x3ebf6a60, 0x3e356d04, 0x3f1b0841, 0x3f2a51d1,
      0x3f48756f, 0x3e902698, 0x3eba6c6e, 0x3f789b74, 0x3dfe4558, 0x3ce61320,
      0x3f192210, 0x3f55bdb5, 0x3e12caf0, 0x3f0051a6, 0x3f449833, 0x3f25bab1,
      0x3f490b43, 0x3f227569, 0x3f786323, 0x3f0c3697, 0x3dcf02e8, 0x3e831d0a,
      0x3ed29ea0, 0x3f396ad5, 0x3f2039f1, 0x3e107048, 0x3f666b9c, 0x3ea8e710,
      0x3eaafe84, 0x3f0b19e0, 0x3e727ff8, 0x3ec4a136, 0x3ec690bc, 0x3db7b210,
      0x3f4ba9b6, 0x3f70667d, 0x3f59ce18, 0x3f53bd30, 0x3ef17d8c, 0x3f207066,
      0x3dec0188, 0x3f6eaeb3, 0x3ea4d9a2, 0x3f3e6334, 0x3d62ce10, 0x3ec38d8e,
      0x3e6c502c, 0x3eec5906, 0x3f440057, 0x3e28bcd0, 0x3f29cfe6, 0x3edc2074,
      0x3f718e70, 0x3c7b0c80, 0x3ef89c8e, 0x3f66bdf1, 0x3ebfdf80, 0x3df57108,
      0x3e51b634, 0x3f7e035b, 0x3e5233dc, 0x3ee5b03a, 0x3ed29e72, 0x3f688ef0,
      0x3ee63522, 0x3e7b6be4, 0x3f32fb4e, 0x3f68a697, 0x3eaea96a, 0x3f615116,
      0x3f0965aa, 0x3e7e3d74, 0x3ca56180, 0x3f0929c4, 0x3ea08428, 0x3e9a6712,
      0x3f3b5687, 0x3f3f1e73, 0x3f0ece49, 0x3e9d3b28, 0x3f10666d, 0x3f027f65,
      0x3ef73f7a, 0x3f0d5c89, 0x3f7f0a14, 0x3f12fdaf, 0x3f1fecbd, 0x3dc6ca98,
      0x3f01277f, 0x3e443e84, 0x3f67d3ec, 0x3f48fae8, 0x3f5ad5fa, 0x3eb0656c,
      0x3e6a58dc, 0x3f0bd847, 0x3f057ab0, 0x3ca21960, 0x3ed907c2, 0x3f5b3862,
      0x3f1fedb1, 0x3f4d2730, 0x3edfab0c, 0x3e81717c, 0x3f4b6bc1, 0x3f53697c,
      0x3f2ba5cf, 0x3ea23ff8, 0x3ebd7fa0, 0x3f5a0f85, 0x3c5d5980, 0x3f279fd2,
      0x3f27bad6, 0x3e306e0c, 0x3f448674, 0x3f0b90af, 0x3f1d7452, 0x3e453514,
      0x3e1ba2e8, 0x3f2b871b, 0x3f07d16e, 0x3e7b9008, 0x3f04f1cf, 0x3f3c6147,
      0x3d9c4208, 0x3e7ecff4, 0x3f671413, 0x3eec57aa, 0x3ebcbd72, 0x3eb239a4,
      0x3ea43872, 0x3dbc7008, 0x3e5c5784, 0x3f449f13, 0x3f2ebe92, 0x3ee73112,
      0x3f03f3cb, 0x3f4ea556, 0x3df7d078, 0x3f03f37a, 0x3ef82142, 0x3f1619d0,
      0x3ed6933e, 0x3f33b79e, 0x3eaa2022, 0x3dc7ef58, 0x3ee5e184, 0x3d5be810,
      0x3da4ed10, 0x3e5fa068, 0x3f588973, 0x3c3b4740, 0x3f77de66, 0x3ef4d7bc,
      0x3ee91b02, 0x3ee58602, 0x3f5222ed, 0x3f70cbe0, 0x3d60d810, 0x3e88ceb2,
      0x3f534e68, 0x3e1a306c, 0x3f458c71, 0x3eb8ca22, 0x3e442da8, 0x3e82504a,
      0x3edc88a8, 0x3f21da37, 0x3e115da0, 0x3e01f7ec, 0x3e9c3eb6, 0x3f1bfd38,
      0x3f780f95, 0x3e404f90, 0x3f7e7462, 0x3ed94262, 0x3f6f9ccf, 0x3eb7fd7a,
      0x3f6b0471, 0x3d44c3f0, 0x3f50ae5d, 0x3f34d178, 0x3ead6a08, 0x3f27a4b6,
      0x3e914308, 0x3e90cc92, 0x3d1debe0, 0x3f61c1b9, 0x3f134823, 0x3f2a814d,
      0x3eeedd64, 0x3f1b15c8, 0x3f31a1ba, 0x3e4437f8, 0x3ecc2796, 0x3ce41b20,
      0x3f002597, 0x3f39a677, 0x3f55c778, 0x3f47d34e, 0x3f543983, 0x3f05609f,
      0x3f2bb2d6, 0x3daec9a8, 0x3ed9beae, 0x3e9859c2, 0x3f51ae0b, 0x3f6fce44,
      0x3f4566e2, 0x3e913e3e, 0x3f206163, 0x3f1ad23e, 0x3e22e3c8, 0x3e417f88,
      0x3f79ad00, 0x3f19f81d, 0x3ed73a18, 0x3eea5daa, 0x3f0b5bdb, 0x3f40cff4,
      0x3f6f8603, 0x3ec2c82e, 0x3f5c4bb2, 0x3e08c228, 0x3ca40d80, 0x3d450fe0,
      0x3eb59a7a, 0x3f3f3c4a, 0x3f617c07, 0x3ecf6f26, 0x3e913fbc, 0x3f2d4a48,
      0x3f3ababf, 0x3e2afc54, 0x3eda0a2e, 0x3d51dd00, 0x3e02f2d4, 0x3e0d2200,
      0x3e88f5a6, 0x3ebb2084, 0x3f64dc94, 0x3d86c5f0, 0x3eb21452, 0x3f63498b,
      0x3f319c3b, 0x3eb7b504, 0x3f1e95c3, 0x3e949938, 0x3effa1e4, 0x3de08088,
      0x3f4a7323, 0x3e7d6334, 0x3ebf5070, 0x3ea2d94c, 0x3ea66a3c, 0x3f04a629,
      0x3f7015cb, 0x3f2eb99a, 0x3f398026, 0x3e529a40, 0x3e9c4dde, 0x3f772f27,
      0x3f21f326, 0x3dded688, 0x3f77cb6f, 0x3f2b44a8, 0x3eadb7dc, 0x3f3e617b,
      0x3f199ddb, 0x3f7eedbe, 0x3f1bdf85, 0x3ed3f304, 0x3f13228e, 0x3ed8bd6a,
      0x3f1a097e, 0x3f32b1e2, 0x3f18417f, 0x3f68e58c, 0x3db6e728, 0x3f38846f,
      0x3e628ecc, 0x3f063374, 0x3f6076f0, 0x3f673242, 0x3f67ca8e, 0x3f1f410b,
      0x3f4ae052, 0x3f5e3eea, 0x3e957e5a, 0x3ee74f20, 0x3e89cc60, 0x3eb12bba,
      0x3e84cbb0, 0x3efd8618, 0x3f4f8b28, 0x3ed1d7ca, 0x3efb6f0e, 0x3ebd8642,
      0x3f7b414a, 0x3c8b5780, 0x3f277100, 0x3efcffaa, 0x3dffb6d0, 0x3eb2fe7a,
      0x3f406b21, 0x3e8bdc72, 0x3f0455ad, 0x3ef692cc, 0x3d683510, 0x3d1d6250,
      0x3ef59622, 0x3f477071, 0x3f58f999, 0x3f20c9a7, 0x3e40e780, 0x3e341ffc,
      0x3f085ed3, 0x3ed67fe4, 0x3db2b568, 0x3ee90dfe, 0x3da8dee0, 0x3eee5dca,
      0x3c9905e0, 0x3ec8013c, 0x3ebd557a, 0x3f6fd089, 0x3c09da80, 0x3f43fe75,
      0x3d931318, 0x3e2be0ac, 0x3f7c5cfb, 0x3f165119, 0x3f570b66, 0x3dca4978,
      0x3a9e6400, 0x3f21fa05, 0x3ed49f6a, 0x3f1d92b0, 0x3f7b7b36, 0x3f5c4291,
      0x3f2cc8b5, 0x3e87901c, 0x3eb2a546, 0x3dbc1e70, 0x3e4bf77c, 0x3f726fa6,
      0x3e5c56cc, 0x3e1ece54, 0x3e172bc0, 0x3ce629a0, 0x3f71d461, 0x3e62af64,
      0x3e911fbe, 0x3e8ee43a, 0x3bb3c580, 0x3f06e073, 0x3f0f2e42, 0x3f373e46,
      0x3eb56d8e, 0x3c9837e0, 0x3f3223cd, 0x3f552bb2, 0x3f79360c, 0x3f55cc2a,
      0x3f49cfe7, 0x3f24dfb5, 0x3ed5161c, 0x3eb6a25c, 0x3dcbc9b0, 0x3e800750,
      0x3ea240ce, 0x3ede991a, 0x3e8f8746, 0x3f2602ef, 0x3f16672c, 0x3f3f3a56,
      0x3f1a264b, 0x3f546edd, 0x3f18e677, 0x3d242890, 0x3f27d72e, 0x3f73a182,
      0x3f0dffe6, 0x3e888c62, 0x3f5c4d18, 0x3f169cbe, 0x3d7b8c40, 0x3f605ca4,
      0x3f44e1c3, 0x3f4c64c3, 0x3f3b6024, 0x3ec74bb4, 0x3f4fe7ec, 0x3d031120,
      0x3f1a9c3a, 0x3eafddd2, 0x3e2ac25c, 0x3e9922e0, 0x3ec26036, 0x3eebbab8,
      0x3f495d11, 0x3d89a9d8, 0x3f19aff9, 0x3e85fce8, 0x3f5379d2, 0x3eece884,
      0x3f53cf2e, 0x3f634ee4, 0x3e1ecc48, 0x3eef20a6, 0x3ec165ac, 0x3f3a4501,
      0x3effdc42, 0x3e988ff0, 0x3f51f7da, 0x3f443ee8, 0x3e49edf8, 0x3f0ee752,
      0x3f60e541, 0x3dfeb8b8, 0x3e9d672a, 0x3f71a387, 0x3f49cd1f, 0x3e4b5e48,
      0x3f4407de, 0x3f1dcc19, 0x3f5db90d, 0x3df4e668, 0x3f2415ac, 0x3d2036c0,
      0x3f1937dc, 0x3f2e41e7, 0x3df0ea08, 0x3f207d8d, 0x3f54de5c, 0x3ebd6b04,
      0x3ccb1ce0, 0x3f3e12a0, 0x3f6466e2, 0x3d02c9a0, 0x3f049180, 0x3ef4e928,
      0x3f3ce2f4, 0x3ec7cccc, 0x3e43afbc, 0x3f248e42, 0x3f43918a, 0x3e2cd7d0,
      0x3e5667d8, 0x3f43b581, 0x3e14715c, 0x3f560b4c, 0x3e985292, 0x3f307ecf,
      0x3f1e8f76, 0x3f127f33, 0x3f4ca87d, 0x3f10d78a, 0x3f78e196, 0x3eae9436,
      0x3f091df0, 0x3f47b310, 0x3d3302c0, 0x3f4c97fb, 0x3f43dbe4, 0x3eafe3f0,
      0x3dc86608, 0x3f79ec4e, 0x3f00ae56, 0x3db23db8, 0x3b78b900, 0x3e750ee8,
      0x3f18c0bc, 0x3ee0f508, 0x3ed1941a, 0x3f252a81, 0x3e830b0c, 0x3e2a455c,
      0x3e3457a4, 0x3ece5dc0, 0x3f474bae, 0x3e79391c, 0x3e3ca940, 0x3f52c8c9,
      0x3e280e9c, 0x3f4bc808, 0x3eac37ea, 0x3f171c7f, 0x3e811d68, 0x3f628233,
      0x3f081aeb, 0x3e668b58, 0x3e524fb8, 0x3f060431, 0x3efe42f4, 0x3ee88ba8,
      0x3e904256, 0x3e97380a, 0x3f36e5fe, 0x3f52320d, 0x3f789972, 0x3e778920,
      0x3ec925de, 0x3f6b3040, 0x3f00a8c0, 0x3f048bd4, 0x3ea06fd8, 0x3ef9e334,
      0x3e7ef494, 0x3f5107ea, 0x3e78e178, 0x3f3a98cd, 0x3e966bf8, 0x3ecd02ee,
      0x3f700d02, 0x3f789ad5, 0x3ecbde24, 0x3f4ebdab, 0x3f1720d2, 0x3f3a08c4,
      0x3f3a7fc6, 0x3e2ffae4, 0x3e0faae8, 0x3ecc09e8, 0x3f1f00a4, 0x3ec6a7d6,
      0x3f762028, 0x3f4d490d, 0x3eed2a50, 0x3e0d1d84, 0x3ed2208e, 0x3e6a8dfc,
      0x3e506670, 0x3ebe807e, 0x3e226ae8, 0x3ef31cae, 0x3ea68cf8, 0x3f514123,
      0x3e9c3530, 0x3f744663, 0x3e9f6e0e, 0x3f34916e, 0x3ed4a298, 0x3eb41312,
      0x3ec7f226, 0x3f0fd8be, 0x3f680726, 0x3d34bb20, 0x3f109d2b, 0x3f5ee98a,
      0x3e5208a0, 0x3ecff9f0, 0x3f00bdd9, 0x3e64ad70, 0x3e1f3b94, 0x3f06c818,
      0x3d3448f0, 0x3f5c78e3, 0x3f01590a, 0x3f3f5d23, 0x3f76c802, 0x3f4a8229,
      0x3e2f6268, 0x3e5f4e80, 0x3da63b10, 0x3e218fbc, 0x3ee79b0a, 0x3ef12da4,
      0x3f0b1eb3, 0x3f5d3ef2, 0x3ece0f6e, 0x3f72eaea, 0x3f609c76, 0x3ec588a2,
      0x3f26821b, 0x3f34b1fb, 0x3e9d18d2, 0x3e36f0dc, 0x3eb39350, 0x3f04961a,
      0x3e89e9d4, 0x3cff9ae0, 0x3d65f5a0, 0x3f66ffb4, 0x3f06a561, 0x3edb3446,
      0x3ec6fc78, 0x3f3c4eb4, 0x3f4c77b3, 0x3deb7f70, 0x3f12c831, 0x3f0bad94,
      0x3e3476e4, 0x3ee84aac, 0x3e89db56, 0x3f0ae963, 0x3cff0160, 0x3f0f3758,
      0x3f46ce71, 0x3ebaa738, 0x3f65f39a, 0x3ef8b3fc, 0x3f16d8bf, 0x3f3a1f29,
      0x3f35b3c0, 0x3e8f1cfe, 0x3e51f31c, 0x3f31af9a, 0x3ed1fbfa, 0x3f027566,
      0x3f22276a, 0x3c632300, 0x3f06a19e, 0x3f11828f, 0x3f15517c, 0x3eec5686,
      0x3f37849b, 0x3ecff914, 0x3f288069, 0x3f6f67a9, 0x3f1c340a, 0x3f064ff6,
      0x3f358da9, 0x3f0db027, 0x3f080aae, 0x3ec92462, 0x3f4fa70a, 0x3f10f69f,
      0x3ee25440, 0x3eace126, 0x3f0fd537, 0x3f72877d, 0x3eb3cf50, 0x3f76d9f9,
      0x3f732b7a, 0x3e920596, 0x3ec8e89e, 0x3ebdd1e4, 0x3ea91346, 0x3f22981e,
      0x3d592dc0, 0x3ee045d6, 0x3f435a29, 0x3dc14358, 0x3f4e6f74, 0x3f70eeb9,
      0x3e5a1320, 0x3f0fb2e1, 0x3ef85dc6, 0x3f5e4519, 0x3f319171, 0x3f34a145,
      0x3f5e7569, 0x3eb26be2, 0x3f414f54, 0x3f1a466c, 0x3f21fc0c, 0x3f582f71,
      0x3f7fd18b, 0x3e99f12a, 0x3eaa84c0, 0x3f20f760, 0x3e347ea4, 0x3e1edd98,
      0x3dc48998, 0x3dd5ace0, 0x3e953030, 0x3f26d6dc, 0x3e7fd7f8, 0x3f5a4b96,
      0x3f387a47, 0x3f577696, 0x3f1e696f, 0x3f4d9809, 0x3df64db8, 0x3e6a57ac,
      0x3f504830, 0x3aa13e00, 0x3efcfd44, 0x3f407dab, 0x3f3849a3, 0x3f09fa5d,
      0x3ec16df8, 0x3f00d206, 0x3d8b01d8, 0x3f2b3041, 0x3e8363ee, 0x3f705db6,
      0x3e8febdc, 0x3f097505, 0x3f58532a, 0x3f12c068, 0x3f4edfcb, 0x3ef9bfd4,
      0x3ed74622, 0x3d4071a0, 0x3d135070, 0x3f0972a9, 0x3df15ad8, 0x3f4417ef,
      0x3e1c6988, 0x3ea42372, 0x3f645560, 0x3f45967c, 0x3d368900, 0x3ee80cfa,
      0x3e67671c, 0x3f48c9dd, 0x3dc8a3d0, 0x3f74e491, 0x3e78e600, 0x3e1e1fbc,
      0x3da61ec0, 0x3e6ab84c, 0x3d9620c0, 0x3eb9f948, 0x3f5c169c, 0x3f643138,
      0x3e088630, 0x3f723855, 0x3f4eff0f, 0x3d5799f0, 0x3f40e570, 0x3e62c9e4,
      0x3f5817d0, 0x3e9cb156, 0x3f3dca3d, 0x3f1e4567, 0x3e19d6c0, 0x3f7ceae9,
      0x3f276397, 0x3eca861a, 0x3f4f2fec, 0x3f429788, 0x3f1c976f, 0x3ed929ba,
      0x3f560095, 0x3ebc78e8, 0x3e8d7156, 0x3e19952c, 0x3db0afd0, 0x3e94a56e,
      0x3f051ecb, 0x3f43665f, 0x3eda474e, 0x3e2f0b2c, 0x3e043530, 0x3dab6028,
      0x3e9a58d4, 0x3d966070, 0x3f09c912, 0x3dd51d78, 0x3e8a96ea, 0x3f4bcc7c,
      0x3d9942f8, 0x3ebe3416, 0x3f309cd0, 0x3ef365a2, 0x3f43957d, 0x3dc081b8,
      0x3ecbb09a, 0x3f0535f7, 0x3ef6717e, 0x3e5845b0, 0x3f1072e5, 0x3edbc85a,
      0x3f71f39b, 0x3f30676e, 0x3c126b40, 0x3f27b5f7, 0x3ec6872e, 0x3eedc058,
      0x3f5469a4, 0x39b64000, 0x3ebce648, 0x3f3ea5e5, 0x3f283526, 0x3f32f4cc,
      0x3d0e8b50, 0x3e43fe94, 0x3f579a71, 0x3e8984e6, 0x3f18a322, 0x3e940f60,
      0x3d95f7e8, 0x3e7073c4, 0x3ef1eac6, 0x3e40a1d0, 0x3f47cb81, 0x3f1fa1ac,
      0x3f1579ce, 0x3f741f69, 0x3f017077, 0x3d032bd0, 0x3f33b776, 0x3e0b9120,
  };

  // 4,6,6,7
  uint32_t kernel_vals[] = {
      0xbd4a03c3, 0xbc5236e0, 0xbcae018c, 0x3bbe1100, 0xbd5142e3, 0x3d87edd1,
      0xbd47b618, 0x3b769320, 0x3d9ab045, 0xbd05cbfe, 0x3d61fe4a, 0x3c25a5c8,
      0xbd900d38, 0x3d62a71e, 0x3d9c1deb, 0xbcdfe380, 0xbcfb6226, 0x3d733b32,
      0xbccdcb56, 0xbd922153, 0x3d6b1ada, 0xbd6120bb, 0x3c675e68, 0x3d9ab0b5,
      0xbd665cdb, 0xbd954bc3, 0xbd8b4e1e, 0xbc8efd34, 0x3d761bda, 0x3cc9bd6c,
      0xbb01d160, 0xbd5238bb, 0xbd73ef38, 0xbb98c350, 0x3c010f40, 0x3b2d32a0,
      0xbc32d1c8, 0xbc4765c8, 0xbc0e9040, 0x3da601a9, 0x3d9fec2b, 0x3da50ef9,
      0x3cfd2260, 0x3d75a5aa, 0xbb2958e0, 0xbcf45a30, 0x3d5b947a, 0x3b4502a0,
      0xbd87a9ac, 0xbd30ab90, 0xbd7e999b, 0xbcdc1fe0, 0x3cf19a1c, 0x3cf83e74,
      0xbd6ce22e, 0x3c9412e0, 0xbd6314f8, 0xbd8989b6, 0xbb206ee0, 0xb94f9800,
      0xbb4cfd20, 0x3da94535, 0xbd91357b, 0xbd0a68a0, 0xbd84500b, 0xbda98373,
      0xbd90c926, 0x3cce5310, 0xbd666610, 0x3cf01390, 0xbce33ad6, 0xbaae3e40,
      0xbc9744a0, 0x3c36b518, 0xbc9bbcc4, 0xbd87c6ca, 0xbd829a38, 0xbd1270c6,
      0xbd2f3168, 0xbdaa5313, 0x3da73187, 0x3bd83540, 0xbd7bb888, 0xbd9c2266,
      0x3d385dba, 0xbd7461cb, 0xbd7f31ab, 0x3d800dc7, 0xbcbc873c, 0xbd667fb3,
      0xbd9abfd4, 0x3d22eaa6, 0x3c806804, 0xbcb97cc0, 0xbc88b354, 0xbda676d8,
      0xbd4ed83b, 0xbd1a0aae, 0xbd7773c8, 0x3bf2a210, 0x3ce93214, 0x3d9452fb,
      0x3c9b2c44, 0x3c990d8c, 0xbd964a23, 0x3d79671a, 0xbc6d6958, 0xbd69e633,
      0xbd138708, 0xbbc895d0, 0x3d2790f6, 0x3c61ca38, 0xbb4f0e20, 0xbd940883,
      0x3d7bfc2a, 0x3d6fdb12, 0x3d315be2, 0xbd2d4c3b, 0xbd0c25c0, 0x3d72df32,
      0x3c917af4, 0xbd8c49b7, 0xbd0b3336, 0xbd59673e, 0x3d84b6f9, 0x3d6b8ebe,
      0xbc50ffa0, 0x3da06631, 0x3c677ab8, 0x3da59d41, 0xbda8ae8a, 0x3daa0c4b,
      0x3c5d9af8, 0xbd87d57a, 0xbd2fb7e3, 0xbd5eb736, 0x3cc0fb3c, 0x3bec2f80,
      0x3d62a892, 0x3d01ef10, 0x3d5d3c0a, 0x3cacdaa0, 0xbcb23880, 0x3d6bfc32,
      0xbd8c4892, 0xbcc77b7c, 0xbd366458, 0xbc0f6448, 0x3b6d7660, 0x3d259518,
      0x3d917fed, 0x3c778448, 0xbd9d1dc6, 0x3daa79e5, 0x3c574c60, 0x3d8fcc0b,
      0xbd8a1148, 0x3ae3dc00, 0x3b02cd80, 0x3ca7b88c, 0xbce928d6, 0x3d6000fe,
      0x3d7cd93a, 0xbd0491f3, 0x3d939069, 0x3da781f5, 0x3cab431c, 0x3d707002,
      0xbd09fcfe, 0x3cf1dfc4, 0xbcc6e010, 0x3d9b0a09, 0xbc938a64, 0x3d934c85,
      0x3d044c48, 0x3d8bccb1, 0xbd7820d0, 0xbcd881ac, 0xbd0c0308, 0xbda88d38,
      0xbd1c52db, 0xbd090ae8, 0x3d478712, 0xbd3add30, 0x3d6eecaa, 0x3d436ff2,
      0x3c0c4e38, 0x3d57f8ba, 0x3d941841, 0x3d893f19, 0xbcbbe2bc, 0x3d5f09d6,
      0xbd7a70a8, 0xbc3c97c8, 0x3c817dc4, 0x3d4588a2, 0x3d522efa, 0xbd9de178,
      0xbc1b6868, 0x3ca2b0ac, 0x3da67919, 0xbd61e47b, 0x3d30937a, 0x3d20a766,
      0x3d83eabd, 0xbda09743, 0x3d1bcf6e, 0xbd0e940e, 0x3c1630e8, 0xbd95ac72,
      0xbcbe5580, 0xbd850cff, 0xbd0121cb, 0xbda5dcd3, 0x3c597e38, 0xbd048f08,
      0x3d7364e6, 0xbc5b3fe0, 0xbb9ecb50, 0xbcc07970, 0x3d495e42, 0xbd7244a8,
      0xbba458c0, 0xbd2938bb, 0xbc857b8c, 0x3d7c964a, 0xbc4a2920, 0x3c4ee508,
      0xbd8f9162, 0x3d6936f6, 0xbcf24430, 0x3c83eb40, 0x3c69dee0, 0xba93cac0,
      0xbd195598, 0x3d18eb9a, 0xbd646000, 0xbc436438, 0xbd198366, 0xbca107f4,
      0x3d87a943, 0xbd06fdc0, 0x3cb5c450, 0x3d386062, 0xbd2c5af0, 0xbbcde190,
      0x3da093c1, 0x3d3340ca, 0x3d7b3e6a, 0xbd884b9b, 0x3cc6e034, 0xbc27b048,
      0x3d685ba2, 0x3d47e7fa, 0xbd7c1920, 0x3d020fa8, 0x3cfede94, 0x3b942550,
      0x3d9e15cd, 0xbd87bafb, 0x3d1b78de, 0xbcc421a0, 0x3d2fbc62, 0xbc84d870,
      0x3d9fba41, 0x3d9d1cc9, 0x3d57c20e, 0xbc382738, 0x3d9a5495, 0xbcce81d6,
      0x3c6d1f18, 0x3cdc9670, 0x3da20251, 0xbda7c0eb, 0xbda8c18b, 0x3d23aef0,
      0xbd44bb1e, 0xbc7d4a58, 0xbc1d0c88, 0x3c8b2db4, 0xbc613720, 0x3d988a2d,
      0x3d9f800d, 0xbda3b07e, 0xbd560d4b, 0x3d84a039, 0xbd485743, 0x3c8ffad4,
      0x3ca4b1f4, 0xbd076a66, 0x3d9fe6cf, 0x3d1e1f82, 0xbd73ba73, 0xbd4d1ea3,
      0xbd1b65a8, 0x3cce6e20, 0x3d5626ce, 0x3d97d1af, 0xbd827e7a, 0xbd3a5a08,
      0x3ccbcf1c, 0x3cb14adc, 0x3d00c4e0, 0xbd824f02, 0xbcab27c6, 0x3d8a989b,
      0xbd982f57, 0xbd6df280, 0xbc9ad2bc, 0x3d8adfa9, 0x3d1af2f8, 0xbcb937e6,
      0xbd849250, 0xbd869830, 0xbb81f4b0, 0x3d80eaed, 0x3d659d6e, 0xbd6d9283,
      0xbd911c08, 0xb9dd0900, 0xbcddf680, 0x3d875dd1, 0xbce7e3e6, 0xbd261250,
      0x3d638eca, 0x3ce98ecc, 0xbd7a0fd6, 0x3d03a7fa, 0x3cdbc19c, 0xbd83005c,
      0x3ccd6be4, 0x3d743cca, 0xbcb62090, 0x3d8e6675, 0xbd069c18, 0xbd9910cc,
      0xbd40193b, 0xbc57c220, 0x3da191a9, 0xbd44f610, 0xbd30923b, 0x3ccfcfd0,
      0x3d99dd31, 0xbda5ff74, 0xbd6cf72e, 0xbc9be3c4, 0xbc9468f4, 0x3d8ea223,
      0x3d2b00c2, 0x3da1cfc5, 0x3d769002, 0xbc0ba500, 0x3ca23384, 0xbd455806,
      0xbda0bce0, 0x3c75e600, 0xbb8ed8d0, 0xbd3820d3, 0xbd59c020, 0x3d3931da,
      0xbd6e0228, 0xbcfc3f16, 0x3d424db2, 0x3ce49904, 0x3cb2f314, 0xbd09cc83,
      0x3da8afe5, 0x3c0677c8, 0xbd24a163, 0x3d370cd2, 0xbd925bee, 0x3cb6c29c,
      0x3c2a68c8, 0xbd340246, 0xbc050168, 0xbd216376, 0x3d340a32, 0xbd2faa58,
      0xbd83fb53, 0xbd78bfde, 0x3da08a5d, 0xbd36e5f0, 0xbb33c3e0, 0x3d3c6dbe,
      0xbc6b0880, 0x3d946a69, 0xbd733f56, 0x3d8743e3, 0xbda8f65f, 0x3d66b4d6,
      0xbc0493f8, 0xbcb2bde6, 0xbd5d4a2b, 0xbd09095b, 0xbb6b1e20, 0x3d8f511d,
      0x3d08ee3e, 0xbd1fc2ee, 0x3b389ba0, 0x3d5b997a, 0xbd4845c8, 0x3cc9af00,
      0x3c9d8610, 0x3d004b8e, 0x3c2ed268, 0x3da3da01, 0x3b3a9a20, 0xbcfd49f6,
      0xbd56ae76, 0xbd03c61b, 0xbd5726ee, 0xbd5a2928, 0x3d55484a, 0x3c9a634c,
      0xbd8770c6, 0x3d6d1f0a, 0x3d33e832, 0x3d694b3e, 0xbcc187fc, 0xbd242a76,
      0xbd13130e, 0x3da72273, 0x3d92b731, 0xbd4aebb6, 0xba1b6580, 0x3d72c8e6,
      0x3d4925e2, 0x3d236c5e, 0x3da15fff, 0xbaedc400, 0x3d4c3222, 0x3cbe0dac,
      0x3b0f3d80, 0xbcdc1c6c, 0xbc492478, 0xbb4e9ce0, 0xbc4403a0, 0xbca3c834,
      0xbd9505ff, 0xbd40e9c8, 0xbcafff9c, 0xbc7b90e0, 0x3cc9f06c, 0xbd4b6eab,
      0x3d7358ea, 0x3cbcc304, 0xbd7b946b, 0x3da83db9, 0x3da7ba81, 0x3d7273aa,
      0x3c9fb3f4, 0x3d35f28a, 0x3cb64c2c, 0x3d13ae6a, 0xbb8e24c0, 0xbd0a239b,
      0xbcac75bc, 0x3d37fdce, 0x3c8e076c, 0xbd19c92e, 0xbd8af5ca, 0xbce7a0e6,
      0xbd9bef43, 0xbd71ac18, 0xbc20bd68, 0xbd8713cb, 0xbd0ba7f6, 0x3d9bdc03,
      0xbd748a23, 0x3cf9f78c, 0x3d1957ce, 0xbcb48d90, 0x3d87a11d, 0xbd368300,
      0x3d71779a, 0x3d9f7779, 0x3d13d07e, 0x3d78d2a2, 0xbb27f100, 0xbc85c52c,
      0x3d71496a, 0x3d9dd073, 0xbd64cd66, 0xbd0dec4b, 0x3d5366f6, 0xbd140f3e,
      0xbccf4696, 0x3d890e07, 0xbd5fc223, 0xbd48bb08, 0xbb1e2120, 0xbcf35ff0,
      0xbb3fa780, 0x3d2367c0, 0xbcb8fd1c, 0x3c4230a0, 0x3c28b8e0, 0xbd843ad8,
      0x3d667bb2, 0x3d6ba9ba, 0xbd31a7c8, 0xbc91a29c, 0x3d1f525e, 0x3d56d02a,
      0x3c58f918, 0xbd4e7748, 0xbd069380, 0xbd868d93, 0x3c189918, 0x3c4db988,
      0xbd597f13, 0xbd66b160, 0xbc9c6190, 0x3c6cd880, 0x3beff080, 0xbd4fd32b,
      0xbda0b9f6, 0x3c616068, 0xbca808ac, 0x3c9ac354, 0xbd344e9b, 0xbb158160,
      0x3b365c60, 0xbd3e3f5e, 0xbc0d2708, 0x3d62fcc6, 0x3cda7c50, 0x3d153bc0,
      0x3d158ff8, 0x3d4ebf7a, 0xbbdae480, 0xbd6a50c6, 0xbd7e4803, 0xbc7792e8,
      0xbd12d0de, 0x3c343060, 0xbd7420e8, 0xbc5a75f8, 0x3ce2c7c0, 0xba3b0480,
      0x3ccd7200, 0x3c3f2c08, 0xbd46fff0, 0x3d4a83aa, 0xbd11f5d6, 0x3d3bee22,
      0x3d36d2b2, 0x3a294080, 0xbbed67d0, 0xbd8575c0, 0x3ccbd3dc, 0xbd7b58d6,
      0x3cfc2f3c, 0x3d10d15e, 0x3d9eac33, 0x3d63527e, 0xbd95759f, 0xbd817ba4,
      0x3d050c4e, 0x3d93373d, 0x3da49995, 0x3d90861d, 0x3d55d23a, 0x3d3b2d0a,
      0x3d9d3063, 0xbd723016, 0x3cc20d70, 0xb9c62100, 0xbd859a97, 0xbd0d1698,
      0xbda20baa, 0x3c74b958, 0x3d11e536, 0x3d9a2b91, 0x3d35b086, 0x3d7029e2,
      0xbd8368e6, 0x3d2295c8, 0x3d5e2072, 0xbc1980a8, 0xbd01fb43, 0x3d507efa,
      0x3d84132b, 0xbd69cbb8, 0xbd124ef8, 0x3d05e09a, 0xbd583096, 0xbd25dcb8,
      0xbc9b34f0, 0xbc012fe0, 0x3cf104c4, 0xbc939eec, 0xbd92f58e, 0xbda36403,
      0x3d4ac81a, 0xbd9225a3, 0xbd3a51c0, 0x3d98faeb, 0x3d6f850e, 0x3d4396c6,
      0x3d2b0e4a, 0x3da2cb89, 0x3d77ff42, 0x3d680d82, 0xb7820000, 0x3d04fd7a,
      0xbc040aa0, 0xbcd8f0f6, 0x3cf44ea0, 0xbd972e2a, 0x3da9c73b, 0xbd958eda,
      0xbc3de068, 0xbbf73cf0, 0xbcaed310, 0xbd9136ac, 0x3d3f64ca, 0xbd939ec8,
      0xbd5295ab, 0xbcf5ca66, 0xbd974058, 0xbd3b6956, 0x3c663100, 0xbd4bc706,
      0x3cd65f44, 0xbc8aa470, 0xbd77e1fb, 0x3d827151, 0x3d12755e, 0xbd62511e,
      0x3cc7c950, 0xbd526e3e, 0xbd26b046, 0xbd03d88b, 0xbdaaa8ee, 0x3c87e5bc,
      0x3c4d17b8, 0x3ab2c4c0, 0xbb761120, 0x3d96b75d, 0xbd8fadcf, 0xbc7c0238,
      0xbc6dd6a0, 0x3c67ac88, 0xbd686143, 0xbd8ab51f, 0xbd2b4853, 0x3d4d83ca,
      0x3d0652e6, 0x3da86b55, 0xba274280, 0x3d9f8853, 0x3d1f1aca, 0xbd1ff04e,
      0xbd87a7e4, 0xbbeb7310, 0xbd0e4be0, 0x3d82c1f5, 0x3b93f100, 0xbc48fda0,
      0xbca725a0, 0x3cc52edc, 0xbd0d028b, 0x3c064460, 0x3d108328, 0x3d445006,
      0x3ca2ebcc, 0x3cb4c6a0, 0xbd918164, 0xbce079b0, 0x3d468be6, 0x3bcf5b30,
      0x3c94a440, 0x3d57dd62, 0xbda84590, 0x3d4c6202, 0x3c5a8a08, 0x3d1c1212,
      0x3cb9ae74, 0x3d50fd96, 0x3d0b3670, 0xbd3f0786, 0x3d88c839, 0xbceb9f90,
      0x3c353208, 0xbd91131b, 0x3d61f992, 0x3d596c52, 0xbcb54dc0, 0xbd5eb8ce,
      0x3bf4c430, 0xbce49680, 0x3c96c7d0, 0x3da53077, 0x3d8b11d5, 0x3d007956,
      0xbd4240ce, 0xbd2dd5b0, 0xbd91414f, 0xbd9d1547, 0xba83de00, 0x3d2867f2,
      0x3d893173, 0x3d8460a1, 0x3d2e6bfa, 0xbd9f0580, 0x3d808d1d, 0xbce392b6,
      0xbbcea8c0, 0xbd4bbc6b, 0xbc0c4918, 0x3d4201fa, 0x3d42c78e, 0xbd1ed16e,
      0x3d447092, 0xbcb6e20c, 0xbca1eb10, 0x3c9982f0, 0x3c174de8, 0xbd4b2130,
      0x3d64072a, 0x3c003838, 0xbd98af67, 0x3cde6524, 0x3d9eac01, 0x3d3c4262,
      0x3d708e6a, 0xbd82ab1f, 0xbd6b1bf6, 0xbd1b98a0, 0xbcfccf66, 0xbd99004a,
      0x3d9f1509, 0xbd851e14, 0x3d8a5981, 0xbda1edb8, 0xbd828fc6, 0x3d17dfd6,
      0x3d3fb852, 0x3d85bafd, 0x3cbb2a14, 0xbc00aed8, 0x3d563aee, 0xbccda146,
      0x3daa4fc9, 0x3d009e36, 0x3d043db6, 0xbd59b186, 0xba87d740, 0xbcf34820,
      0xbd601c9b, 0x3d879839, 0xbc78e9e0, 0x3d522172, 0x3d843c15, 0xbcf89fd0,
      0x3d9de679, 0x3c76e580, 0x3da022d9, 0xbb2b0960, 0x3d8d958d, 0x3bb72830,
      0x3d47e44e, 0xbd861bcb, 0x3d841fbd, 0xbcb14396, 0xbd9e5a1f, 0xbd50064e,
      0xbd65881e, 0xbc2c1178, 0x3c9935f4, 0x3cd26b14, 0xbd964b93, 0x3d0ece78,
      0xbd3823e8, 0xbd390750, 0x3cc02404, 0x3ce14fc4, 0xbd468230, 0x3c71fc00,
      0xbbcb8d10, 0xbd65f760, 0x3d6b7c6e, 0x3cf6fde4, 0x3b848080, 0x3d8c8471,
      0x3d79d7da, 0x3bc5f110, 0x3d475ae6, 0x3cc6fd24, 0x3d216f98, 0x3d97f671,
      0x3cc07234, 0xbca16f40, 0x3d30e80a, 0xbd0c4656, 0x3d071596, 0x3c7ee198,
      0x3bd44630, 0xbc975adc, 0x3cf64664, 0xbd98952e, 0xbd658dce, 0xbd94b083,
      0x3d29ce46, 0xbc531258, 0x3d8a636b, 0xbc40ea08, 0x3da40a5d, 0xbd13582b,
      0x3d8f2b9d, 0xbd16a40e, 0xbda1e36a, 0xbd6b0edb, 0xbd3335bb, 0x3adacdc0,
      0x3bdebd00, 0xbd45d8b8, 0xbd6f4a10, 0xbc522c38, 0xbd65d418, 0xbda75ad6,
      0xbc027a68, 0x3da13ba5, 0x398f4500, 0xbd855b0c, 0xbaae74c0, 0x3c77ccf8,
      0x3cd991f0, 0x3c98cfcc, 0x3c67cfc0, 0x3c92a474, 0xbda73f5a, 0xbd47c613,
      0xbc87a984, 0xbd98ea20, 0xbd688dd6, 0xbd9e1cc8, 0xbd63b108, 0xbd42dc90,
      0xbd31977b, 0xbcc06b16, 0x3d748c1a, 0x3d081cd2, 0x3b634420, 0x3ce49d64,
      0x3cc3b2dc, 0xbd987c76, 0xbd08e1a6, 0xbc03aa00, 0xbda3ae33, 0x3c9f9694,
      0xbd6b8b8b, 0x3c19d538, 0x3c7765e0, 0x3d0880ce, 0xbd1a8f38, 0x3d3f94a2,
      0xbd72580e, 0x3d047fe6, 0xbb229f80, 0xbd289976, 0x3d882ef5, 0x3d83c213,
      0xbbffd530, 0xbd12bb66, 0x3b79b6a0, 0x3d320592, 0x3d41b30e, 0x3d0dfdb6,
      0xbb816370, 0xbc16cf78, 0xbd96a58f, 0xbbcbc7c0, 0x3cd957bc, 0x3d8f1b19,
      0xbd3e4b03, 0x3da8f2e7, 0xbd2e5dd3, 0x3be3d140, 0x3d1a5f92, 0x3cc65100,
      0xbcb5ecd6, 0xbd98aac0, 0x3d8f6871, 0x3d17c096, 0xbb20c5a0, 0x3ce1e18c,
      0xbd8fc1c0, 0xbbfcf430, 0x3ae3df00, 0xbd092c23, 0xbcece9ec, 0x3d30a792,
      0x3d31f3f6, 0x3d9680b1, 0x3d06d1da, 0xbc4dfd68, 0x3d9dd547, 0x3d85ae15,
      0x3d205320, 0x3d14a018, 0xbc8ad6fc, 0xbd42672e, 0xbd7779de, 0x3da53843,
      0xbd69a3d8, 0xbd3bbe16, 0xbd7468a6, 0x3cf76de0, 0x3d3af472, 0xbbc2b040,
      0xbd471f58, 0xbd9682bb, 0x3cb26d40, 0xbda47dee, 0x3d47fdba, 0x3d66c7e2,
      0xbcfb69ac, 0xbd9d1c80, 0xbc4a0e68, 0x3cf65330, 0xbd1b30ab, 0x3cd110a0,
      0xbc572bd8, 0x3d8fe485, 0x3abb1f40, 0x3d8baaed, 0x3c785ca8, 0xbda6b760,
      0x3c642688, 0xbd9156a0, 0xbb8e1b90, 0xbd393ae0, 0xbcba77bc, 0xbda80d80,
      0x3d89eb6d, 0xbcd71d4c, 0x3d5b3d2e, 0x3da096c1, 0x3d977241, 0xbc756640,
      0x3da514a5, 0x3d3e230a, 0x3ba671b0, 0xbda375c7, 0x3d8f01c3, 0xbd5e89ee,
      0x3d9913a5, 0x3ac94200, 0xbd46aec3, 0xbd85060f, 0x3a4ca080, 0x3d3901f2,
      0xbd2cc0d6, 0xbd38388b, 0xbd977c72, 0xbd68695e, 0xbd356b80, 0x3c396248,
      0x3d7d26e6, 0x3d9f7b59, 0x3d5170ce, 0xbd8fc497, 0x3b8a3e50, 0xbd60a916,
      0xbcf97180, 0x3ceb3d34, 0x3c90038c, 0xbd905acf, 0xbd447fa0, 0xbcb5506c,
      0x3ca3926c, 0x3ce23a2c, 0x3cc87fb0, 0xbc6f4800, 0x3cf6c04c, 0x3ceff3dc,
      0x3cdb344c, 0xbd747cd6, 0xbca6ca9c, 0x3da86511, 0x3d35aa3a, 0xbda53be8,
      0xbd4e883e, 0x3b353780, 0xbd67b886, 0xbd8006bf, 0x3a1be600, 0xbd16e450,
      0xbda47564, 0xbcaebbc0, 0xbd900413, 0xbd9261d2, 0x3d4f1bba, 0xbc26e7c0,
      0xbd4c509b, 0x3d925d1d, 0xbda9f75b, 0xbd958d2b, 0x3d8b3fb7, 0xbd0bd06b,
      0xbd08bf1b, 0x3d904c03, 0xbd321c06, 0xbd90b0ff, 0x3d5b7eaa, 0xbc8051d0,
  };

  // 7
  uint32_t bias_vals[] = {
      0x3d7895da, 0x3d66c0c2, 0x3da8f295, 0x3c8090d0,
      0x3b0969a0, 0x3da4fbad, 0xbd55808e,
  };

  // 4,5,3,7
  uint32_t output_exp_vals[] = {
      0x3e800009, 0x3e5cc2bb, 0x3dd8b7fc, 0xbd6f5e64, 0xbf0049b7, 0x3db2113b,
      0x3c2e3283, 0xbea7f82c, 0x3e190482, 0x3e964a40, 0xbdeea57b, 0xbe488eb1,
      0x3dbd218d, 0xbd843b27, 0x3c353168, 0x3dadf9bd, 0xbe95f953, 0xbe2ab356,
      0xbe95d365, 0xbe939e65, 0x3e89892d, 0x3e25c7d0, 0x3e9b0553, 0x3e39f0f1,
      0x3eca599b, 0xbe152c42, 0x3ea822df, 0xbe339202, 0x3e6d580b, 0x3e9794f4,
      0x3dde4c3a, 0x3d8b0621, 0xbf1a21a5, 0x3dfc1fe5, 0x3e49eed5, 0x3e5ebc15,
      0x3d19c5a0, 0x3d8d4f3d, 0xbea808ac, 0xbecf76da, 0xbee54599, 0x3ea8e7bd,
      0x3d4c47a1, 0xbcd8a9d4, 0x3e1d24d0, 0xbcd58c40, 0xbee6366e, 0x3e88acad,
      0xbbb28961, 0xbea28a89, 0x3ea61d80, 0x3ed57081, 0xbdf12a11, 0xbf1070d0,
      0xbeb9ecba, 0x3d948c77, 0xbe1630c7, 0xbd865d9a, 0x3d694f50, 0x3d618060,
      0xbe90c00a, 0xbc7090db, 0x3ec9335e, 0x3d69cdbb, 0x3d960288, 0x3edc27a8,
      0x3e078210, 0xbebb55cf, 0x3ddc602d, 0xbde874ef, 0x3db7ea35, 0x3e592130,
      0x3e2a8154, 0xbe215282, 0xbf16bf05, 0xbdadf37e, 0x3daf312c, 0xbe3b1d5e,
      0xbdf7a1ce, 0xbdbffeb8, 0xbe1678e4, 0xbe663303, 0xbe360a72, 0xbe2acad4,
      0x3e436c58, 0x3da03a17, 0xbcd28ce2, 0x3e86bb89, 0xbd8a93ca, 0x3e1ccc14,
      0xbcc44d16, 0x3e75ece0, 0x3e21a281, 0x3e9e283c, 0xbdc6f8fb, 0xbdd9e63e,
      0xbe41d862, 0x3e958b70, 0x3dd38536, 0x3d36528a, 0xbdae929f, 0xbce4e6a2,
      0xbd965d20, 0xbe1e6fc7, 0x3e596745, 0x3e6acaab, 0x3e0c2af5, 0x3ec92511,
      0x3d9b3198, 0xbe7669a4, 0x3e6d9559, 0xbe238994, 0xbd6f15a0, 0x3ecbbbb7,
      0x3dea562a, 0xbdcd3380, 0xbef6648c, 0xbe05855d, 0xbd10f5eb, 0x3ee80009,
      0x3db3b7f8, 0x3de7f23f, 0xbe15d9b3, 0xbe03a005, 0xbdbd7cce, 0x3e17e7f6,
      0x3e050054, 0x3dfdf63a, 0x3f0b55cd, 0x3dde99dd, 0xbe93ac48, 0x3d927932,
      0xbc006933, 0x3e995281, 0x3e9fa656, 0x3e0838d8, 0xbdb96f3f, 0xbe628743,
      0xbe9d09ac, 0xbcc5c04d, 0x3e2b9311, 0x3e544b71, 0xbe0b460d, 0xbeba0990,
      0xbe86fe5a, 0x3d86e498, 0x3e4f89b7, 0x3e85155f, 0x3e70834f, 0x3edb8026,
      0x3e106b80, 0xbeb7ecce, 0x3e6d4fd1, 0x3dcee4a3, 0x3e2cec70, 0x3e652ac0,
      0x3e3d0220, 0x3e802fb4, 0xbe94e2d7, 0xbe1a902e, 0x3dcf41ab, 0xbda36d77,
      0xbdd779ec, 0x3d22e8e3, 0xbe17c8d2, 0xbed6b5ab, 0xbd1cefdb, 0x3e7ebc8d,
      0x3cb6a703, 0x3dc6f159, 0x3e939f4c, 0x3cc6d6ae, 0xbea4d8c8, 0x3e5d0622,
      0xbd23eacd, 0xbda939c9, 0x3e1b1995, 0x3e2f815b, 0xbeac3cd3, 0xbf15d0da,
      0xbf03af7c, 0x3bc4a6d9, 0x3e8cf965, 0x3d1a27a2, 0x3eeaaa98, 0xbd7bdc52,
      0xbd4fb68e, 0xbe4063ea, 0x3d8d0cc5, 0x3e916e54, 0xbac6376b, 0x3dd4732e,
      0x3e7bfcc1, 0xbe81430c, 0x3e77d1b0, 0x3d80743a, 0x3c75e0f7, 0x3dc65b26,
      0x3cb872dc, 0x3d429fa2, 0xbe005912, 0xbe103baf, 0x3de03a8c, 0x3e851e20,
      0xbe285826, 0x3dbf8d53, 0xbe559c2a, 0xbe5b362e, 0x3d2ee6a4, 0x3e3f6f18,
      0x3cc667e5, 0x3be227ae, 0x3d016ffc, 0x3e20fbce, 0xbe9fe558, 0x3e10e7f8,
      0x3e4824d5, 0xbda27412, 0x3eb27828, 0x3e755759, 0xbe7c3e18, 0xbf2b8387,
      0xbe8a3ca3, 0xbde6d540, 0x3d44254c, 0x3c9c5216, 0xbdf7ec4c, 0xbe3e7179,
      0xbe8b1c89, 0xbe2ae027, 0xb94c3ecd, 0x3dee7547, 0xbd6cafec, 0x3e909c65,
      0xbe25799b, 0xbe957a87, 0x3dd1f2cf, 0x3d4fffba, 0x3e9ad1f3, 0x3ecf9869,
      0x3e851b77, 0xbef21b3c, 0xbf309a66, 0xbe9694e1, 0xbdd16fef, 0x3db6f5d4,
      0x3cd36c12, 0x3d88ba04, 0xbe8e4ee1, 0xbeac0437, 0xbde01afc, 0x3ed26389,
      0x3dfbaafb, 0x3b38fc61, 0x3e97a1f5, 0x3d6752c6, 0xbf07fb08, 0x3dcd47db,
      0xbe56aea5, 0x3e02e71c, 0x3f02f762, 0x3e4dc44e, 0xbd755279, 0xbedfb32f,
      0xbe522fa0, 0xbc4d388e, 0x3cecdfd6, 0x3da0a80e, 0x3bbac653, 0xbd82ef3d,
      0xbe60044e, 0xbe9ca11d, 0x3e631998, 0x3d3bb59f, 0x3e71dfde, 0x3d2286e5,
      0x3e98798d, 0xbf09b153, 0x3d6a9f74, 0x3e11df9c, 0x3e5de0f9, 0x3ea99bb4,
      0x3e43c72e, 0xbddb4395, 0xbe69c536, 0xbc0e0570, 0x3ea5b6ca, 0xbcf35b4f,
      0xbb791864, 0xbe413177, 0xbea6b9f5, 0xbddddcef, 0xbd76b1a3, 0xbc547e22,
      0x3e4ee61d, 0xbe2bf763, 0x3de683ed, 0x3dbff523, 0xbe3847ed, 0x3e775bb5,
      0xbd0b4a08, 0x3e59e88a, 0x3e26091d, 0xbdebc80d, 0x3dae7906, 0xbe19f53b,
      0xbd8380c1, 0xbc766243, 0x3d2245bc, 0x3c994d8d, 0x3e4c9d2a, 0x3e0e9a3a,
      0xbe13408d, 0x3da080bc, 0x3b439034, 0x3d52f123, 0x3dbe7f95, 0x3eeebcbb,
      0x3ea507c6, 0xbed81e08, 0x3e9538e6, 0x3d3f8f59, 0x3e91f5d3, 0x3e9249cf,
      0x3e83c73c, 0x3de597cb, 0xbee560c4, 0xbe1e8918, 0x3ca144b6, 0x3d726a89,
      0x3d846794, 0x3dd78c67, 0xbea7732f, 0xbe7bde9e, 0xbe6706c7, 0xbdf711c2,
      0x3e4bd329, 0x3d2304df, 0x3eabc3f0, 0x3d45e53a, 0xbedcc3e9, 0x3e8446f9,
      0xbe3fd995, 0x3e547af2, 0x3f0aa842, 0x3ec9a325, 0xbed2a254, 0xbf0f6b26,
      0xbde25b3f, 0x3dc8c105, 0x3d8ec9d0, 0x3d7fd7a1, 0x3e610f6a, 0xbdea6665,
      0xbead5c1f, 0xbe80e77d, 0x3e489cb7, 0x3ec65a7c, 0x3e4ffac6, 0x3d55f9a6,
      0x3e9ea98b, 0xbf0e625e, 0xbc8bde61, 0x3dd68ecb, 0x3bc64cbd, 0x3e86a09b,
      0x3d554cb4, 0xbeccd628, 0xbf035dfc, 0xbe6c27ea, 0x3d4483cf, 0x3e223af9,
      0xbe20cbee, 0x3dd718e7, 0xbe8fad2f, 0xbe938918, 0xbeb29804, 0x3e9142f0,
      0x3de3e136, 0x3e9a39ce, 0x3efa0610, 0x3d9406ea, 0xbee24e0f, 0x3e38404b,
      0xbe3ef7ee, 0xbc59ad65, 0x3e5f9e81, 0x3e23b360, 0xbea0121f, 0xbf0bfbbc,
      0xbec79dfc, 0xbd7102a6, 0x3eb5585c, 0x3dca0b72, 0x3cb1a8b7, 0xbe1a22e0,
      0xbe477474, 0xbe2e323b, 0x3e06e740, 0x3e6ec883, 0xbd822e0f, 0x3e84a3f8,
      0x3dbc54c1, 0xbe2a66fe, 0x3e5fe31b, 0x3d6cdf21, 0x3eed45e5, 0x3e49ef86,
      0x3e042067, 0xbd2b9ef1, 0x3d8982d5, 0x3cd070fb, 0x3ededc87, 0x3e0a8bfa,
      0xbb858c8b, 0x3d04acd8, 0xbe08ce60, 0xbe30cc81, 0x3d582e3d, 0x3db17c5b,
  };

  // 4,5,3,7
  uint32_t output_relu_exp_vals[] = {
      0x3e800009, 0x3e5cc2bb, 0x3dd8b7fc, 0x0,        0x0,        0x3db2113b,
      0x3c2e3283, 0x0,        0x3e190482, 0x3e964a40, 0x0,        0x0,
      0x3dbd218d, 0x0,        0x3c353168, 0x3dadf9bd, 0x0,        0x0,
      0x0,        0x0,        0x3e89892d, 0x3e25c7d0, 0x3e9b0553, 0x3e39f0f1,
      0x3eca599b, 0x0,        0x3ea822df, 0x0,        0x3e6d580b, 0x3e9794f4,
      0x3dde4c3a, 0x3d8b0621, 0x0,        0x3dfc1fe5, 0x3e49eed5, 0x3e5ebc15,
      0x3d19c5a0, 0x3d8d4f3d, 0x0,        0x0,        0x0,        0x3ea8e7bd,
      0x3d4c47a1, 0x0,        0x3e1d24d0, 0x0,        0x0,        0x3e88acad,
      0x0,        0x0,        0x3ea61d80, 0x3ed57081, 0x0,        0x0,
      0x0,        0x3d948c77, 0x0,        0x0,        0x3d694f50, 0x3d618060,
      0x0,        0x0,        0x3ec9335e, 0x3d69cdbb, 0x3d960288, 0x3edc27a8,
      0x3e078210, 0x0,        0x3ddc602d, 0x0,        0x3db7ea35, 0x3e592130,
      0x3e2a8154, 0x0,        0x0,        0x0,        0x3daf312c, 0x0,
      0x0,        0x0,        0x0,        0x0,        0x0,        0x0,
      0x3e436c58, 0x3da03a17, 0x0,        0x3e86bb89, 0x0,        0x3e1ccc14,
      0x0,        0x3e75ece0, 0x3e21a281, 0x3e9e283c, 0x0,        0x0,
      0x0,        0x3e958b70, 0x3dd38536, 0x3d36528a, 0x0,        0x0,
      0x0,        0x0,        0x3e596745, 0x3e6acaab, 0x3e0c2af5, 0x3ec92511,
      0x3d9b3198, 0x0,        0x3e6d9559, 0x0,        0x0,        0x3ecbbbb7,
      0x3dea562a, 0x0,        0x0,        0x0,        0x0,        0x3ee80009,
      0x3db3b7f8, 0x3de7f23f, 0x0,        0x0,        0x0,        0x3e17e7f6,
      0x3e050054, 0x3dfdf63a, 0x3f0b55cd, 0x3dde99dd, 0x0,        0x3d927932,
      0x0,        0x3e995281, 0x3e9fa656, 0x3e0838d8, 0x0,        0x0,
      0x0,        0x0,        0x3e2b9311, 0x3e544b71, 0x0,        0x0,
      0x0,        0x3d86e498, 0x3e4f89b7, 0x3e85155f, 0x3e70834f, 0x3edb8026,
      0x3e106b80, 0x0,        0x3e6d4fd1, 0x3dcee4a3, 0x3e2cec70, 0x3e652ac0,
      0x3e3d0220, 0x3e802fb4, 0x0,        0x0,        0x3dcf41ab, 0x0,
      0x0,        0x3d22e8e3, 0x0,        0x0,        0x0,        0x3e7ebc8d,
      0x3cb6a703, 0x3dc6f159, 0x3e939f4c, 0x3cc6d6ae, 0x0,        0x3e5d0622,
      0x0,        0x0,        0x3e1b1995, 0x3e2f815b, 0x0,        0x0,
      0x0,        0x3bc4a6d9, 0x3e8cf965, 0x3d1a27a2, 0x3eeaaa98, 0x0,
      0x0,        0x0,        0x3d8d0cc5, 0x3e916e54, 0x0,        0x3dd4732e,
      0x3e7bfcc1, 0x0,        0x3e77d1b0, 0x3d80743a, 0x3c75e0f7, 0x3dc65b26,
      0x3cb872dc, 0x3d429fa2, 0x0,        0x0,        0x3de03a8c, 0x3e851e20,
      0x0,        0x3dbf8d53, 0x0,        0x0,        0x3d2ee6a4, 0x3e3f6f18,
      0x3cc667e5, 0x3be227ae, 0x3d016ffc, 0x3e20fbce, 0x0,        0x3e10e7f8,
      0x3e4824d5, 0x0,        0x3eb27828, 0x3e755759, 0x0,        0x0,
      0x0,        0x0,        0x3d44254c, 0x3c9c5216, 0x0,        0x0,
      0x0,        0x0,        0x0,        0x3dee7547, 0x0,        0x3e909c65,
      0x0,        0x0,        0x3dd1f2cf, 0x3d4fffba, 0x3e9ad1f3, 0x3ecf9869,
      0x3e851b77, 0x0,        0x0,        0x0,        0x0,        0x3db6f5d4,
      0x3cd36c12, 0x3d88ba04, 0x0,        0x0,        0x0,        0x3ed26389,
      0x3dfbaafb, 0x3b38fc61, 0x3e97a1f5, 0x3d6752c6, 0x0,        0x3dcd47db,
      0x0,        0x3e02e71c, 0x3f02f762, 0x3e4dc44e, 0x0,        0x0,
      0x0,        0x0,        0x3cecdfd6, 0x3da0a80e, 0x3bbac653, 0x0,
      0x0,        0x0,        0x3e631998, 0x3d3bb59f, 0x3e71dfde, 0x3d2286e5,
      0x3e98798d, 0x0,        0x3d6a9f74, 0x3e11df9c, 0x3e5de0f9, 0x3ea99bb4,
      0x3e43c72e, 0x0,        0x0,        0x0,        0x3ea5b6ca, 0x0,
      0x0,        0x0,        0x0,        0x0,        0x0,        0x0,
      0x3e4ee61d, 0x0,        0x3de683ed, 0x3dbff523, 0x0,        0x3e775bb5,
      0x0,        0x3e59e88a, 0x3e26091d, 0x0,        0x3dae7906, 0x0,
      0x0,        0x0,        0x3d2245bc, 0x3c994d8d, 0x3e4c9d2a, 0x3e0e9a3a,
      0x0,        0x3da080bc, 0x3b439034, 0x3d52f123, 0x3dbe7f95, 0x3eeebcbb,
      0x3ea507c6, 0x0,        0x3e9538e6, 0x3d3f8f59, 0x3e91f5d3, 0x3e9249cf,
      0x3e83c73c, 0x3de597cb, 0x0,        0x0,        0x3ca144b6, 0x3d726a89,
      0x3d846794, 0x3dd78c67, 0x0,        0x0,        0x0,        0x0,
      0x3e4bd329, 0x3d2304df, 0x3eabc3f0, 0x3d45e53a, 0x0,        0x3e8446f9,
      0x0,        0x3e547af2, 0x3f0aa842, 0x3ec9a325, 0x0,        0x0,
      0x0,        0x3dc8c105, 0x3d8ec9d0, 0x3d7fd7a1, 0x3e610f6a, 0x0,
      0x0,        0x0,        0x3e489cb7, 0x3ec65a7c, 0x3e4ffac6, 0x3d55f9a6,
      0x3e9ea98b, 0x0,        0x0,        0x3dd68ecb, 0x3bc64cbd, 0x3e86a09b,
      0x3d554cb4, 0x0,        0x0,        0x0,        0x3d4483cf, 0x3e223af9,
      0x0,        0x3dd718e7, 0x0,        0x0,        0x0,        0x3e9142f0,
      0x3de3e136, 0x3e9a39ce, 0x3efa0610, 0x3d9406ea, 0x0,        0x3e38404b,
      0x0,        0x0,        0x3e5f9e81, 0x3e23b360, 0x0,        0x0,
      0x0,        0x0,        0x3eb5585c, 0x3dca0b72, 0x3cb1a8b7, 0x0,
      0x0,        0x0,        0x3e06e740, 0x3e6ec883, 0x0,        0x3e84a3f8,
      0x3dbc54c1, 0x0,        0x3e5fe31b, 0x3d6cdf21, 0x3eed45e5, 0x3e49ef86,
      0x3e042067, 0x0,        0x3d8982d5, 0x3cd070fb, 0x3ededc87, 0x3e0a8bfa,
      0x0,        0x3d04acd8, 0x0,        0x0,        0x3d582e3d, 0x3db17c5b,
  };

  test_conv2d(set, strides, input_vals, kernel_vals, bias_vals, output_exp_vals,
              output_relu_exp_vals, SAME_PADDING, NULL);
}

// wrong padding type = response code F000
void test_f000_fail() {

  input_set i = {1, 1, 1, 1, {1, 1}, 1};
  input_set *set = &i; // just so we can copy and paste code

  strides_input_set s = {1, 1};
  strides_input_set *strides = &s;

  zdnn_status status;

  zdnn_pool_padding padding = 0xFF;

  uint32_t input_dims[4] = {set->n, set->height_in, set->width_in,
                            set->channel_in};
  uint32_t kernel_dims[4] = {set->kernel_size[0], set->kernel_size[1],
                             set->channel_in, set->channel_out};
  uint32_t bias_dims[1] = {set->channel_out};
  uint32_t output_dims[4] = {set->n, 1, 1, set->channel_out};

  zdnn_ztensor *input_ztensor = alloc_ztensor_with_values(
      input_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *kernel_ztensor = alloc_ztensor_with_values(
      kernel_dims, ZDNN_HWCK, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *bias_ztensor = alloc_ztensor_with_values(
      bias_dims, ZDNN_1D, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *output_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  status = zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                       strides->height, strides->width, CONV2D_ACT_NONE, NULL,
                       output_ztensor);

  TEST_ASSERT_MESSAGE(status == ZDNN_FUNC_RC_F000,
                      "zdnn_conv2d(): status not ZDNN_FUNC_RC_F000");
}

// wrong activation function = response code F001
void test_f001_fail() {
  input_set i = {1, 1, 1, 1, {1, 1}, 1};
  input_set *set = &i; // just so we can copy and paste code

  strides_input_set s = {1, 1};
  strides_input_set *strides = &s;

  zdnn_status status;

  zdnn_pool_padding padding = VALID_PADDING;

  uint32_t input_dims[4] = {set->n, set->height_in, set->width_in,
                            set->channel_in};
  uint32_t kernel_dims[4] = {set->kernel_size[0], set->kernel_size[1],
                             set->channel_in, set->channel_out};
  uint32_t bias_dims[1] = {set->channel_out};
  uint32_t output_dims[4] = {set->n, 1, 1, set->channel_out};

  zdnn_ztensor *input_ztensor = alloc_ztensor_with_values(
      input_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *kernel_ztensor = alloc_ztensor_with_values(
      kernel_dims, ZDNN_HWCK, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *bias_ztensor = alloc_ztensor_with_values(
      bias_dims, ZDNN_1D, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *output_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  status =
      zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                  strides->height, strides->width, 0xFF, NULL, output_ztensor);

  TEST_ASSERT_MESSAGE(status == ZDNN_FUNC_RC_F001,
                      "zdnn_conv2d(): status not ZDNN_FUNC_RC_F001");
}

// both strides = 0, kernel height > 448
void test_f002_height_fail() {
  input_set i = {1, 1, 1, 1, {512, 1}, 1};
  input_set *set = &i; // just so we can copy and paste code

  strides_input_set s = {0, 0};
  strides_input_set *strides = &s;

  zdnn_status status;

  zdnn_pool_padding padding = VALID_PADDING;

  uint32_t input_dims[4] = {set->n, set->kernel_size[0], set->kernel_size[1],
                            set->channel_in};
  uint32_t kernel_dims[4] = {set->kernel_size[0], set->kernel_size[1],
                             set->channel_in, set->channel_out};
  uint32_t bias_dims[1] = {set->channel_out};
  uint32_t output_dims[4] = {set->n, 1, 1, set->channel_out};

  zdnn_ztensor *input_ztensor = alloc_ztensor_with_values(
      input_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *kernel_ztensor = alloc_ztensor_with_values(
      kernel_dims, ZDNN_HWCK, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *bias_ztensor = alloc_ztensor_with_values(
      bias_dims, ZDNN_1D, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *output_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  status = zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                       strides->height, strides->width, CONV2D_ACT_NONE, NULL,
                       output_ztensor);

  TEST_ASSERT_MESSAGE(status == ZDNN_FUNC_RC_F002,
                      "zdnn_conv2d(): status not ZDNN_FUNC_RC_F002");
}

// both strides = 0, kernel width > 448
void test_f002_width_fail() {
  input_set i = {1, 1, 1, 1, {1, 512}, 1};
  input_set *set = &i; // just so we can copy and paste code

  strides_input_set s = {0, 0};
  strides_input_set *strides = &s;

  zdnn_status status;

  zdnn_pool_padding padding = VALID_PADDING;

  uint32_t input_dims[4] = {set->n, set->kernel_size[0], set->kernel_size[1],
                            set->channel_in};
  uint32_t kernel_dims[4] = {set->kernel_size[0], set->kernel_size[1],
                             set->channel_in, set->channel_out};
  uint32_t bias_dims[1] = {set->channel_out};
  uint32_t output_dims[4] = {set->n, 1, 1, set->channel_out};

  zdnn_ztensor *input_ztensor = alloc_ztensor_with_values(
      input_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *kernel_ztensor = alloc_ztensor_with_values(
      kernel_dims, ZDNN_HWCK, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *bias_ztensor = alloc_ztensor_with_values(
      bias_dims, ZDNN_1D, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *output_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  status = zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                       strides->height, strides->width, CONV2D_ACT_NONE, NULL,
                       output_ztensor);

  TEST_ASSERT_MESSAGE(status == ZDNN_FUNC_RC_F002,
                      "zdnn_conv2d(): status not ZDNN_FUNC_RC_F002");
}

// both strides > 0, kernel height > 64
void test_f003_height_fail() {
  uint32_t bad_height = 70; // output height becomes 11

  // height_in must > kernel_height
  input_set i = {1, bad_height + 10, 1, 1, {bad_height, 1}, 1};
  input_set *set = &i; // just so we can copy and paste code

  strides_input_set s = {1, 1};
  strides_input_set *strides = &s;

  zdnn_status status;

  zdnn_pool_padding padding = VALID_PADDING;

  uint32_t input_dims[4] = {set->n, set->height_in, set->width_in,
                            set->channel_in};
  uint32_t kernel_dims[4] = {set->kernel_size[0], set->kernel_size[1],
                             set->channel_in, set->channel_out};
  uint32_t bias_dims[1] = {set->channel_out};
  uint32_t output_dims[4] = {set->n, 11, 1, set->channel_out}; //

  zdnn_ztensor *input_ztensor = alloc_ztensor_with_values(
      input_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *kernel_ztensor = alloc_ztensor_with_values(
      kernel_dims, ZDNN_HWCK, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *bias_ztensor = alloc_ztensor_with_values(
      bias_dims, ZDNN_1D, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *output_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  status = zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                       strides->height, strides->width, CONV2D_ACT_NONE, NULL,
                       output_ztensor);

  TEST_ASSERT_MESSAGE(status == ZDNN_FUNC_RC_F003,
                      "zdnn_conv2d(): status not ZDNN_FUNC_RC_F003");
}

// both strides > 0, kernel width > 64
void test_f003_width_fail() {
  uint32_t bad_width = 70; // output width becomes 11

  // width_in must > kernel_width
  input_set i = {1, 1, bad_width + 10, 1, {1, bad_width}, 1};
  input_set *set = &i; // just so we can copy and paste code

  strides_input_set s = {1, 1};
  strides_input_set *strides = &s;

  zdnn_status status;

  zdnn_pool_padding padding = VALID_PADDING;

  uint32_t input_dims[4] = {set->n, set->height_in, set->width_in,
                            set->channel_in};
  uint32_t kernel_dims[4] = {set->kernel_size[0], set->kernel_size[1],
                             set->channel_in, set->channel_out};
  uint32_t bias_dims[1] = {set->channel_out};
  uint32_t output_dims[4] = {set->n, 1, 11, set->channel_out};

  zdnn_ztensor *input_ztensor = alloc_ztensor_with_values(
      input_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *kernel_ztensor = alloc_ztensor_with_values(
      kernel_dims, ZDNN_HWCK, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *bias_ztensor = alloc_ztensor_with_values(
      bias_dims, ZDNN_1D, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *output_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  status = zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                       strides->height, strides->width, CONV2D_ACT_NONE, NULL,
                       output_ztensor);

  TEST_ASSERT_MESSAGE(status == ZDNN_FUNC_RC_F003,
                      "zdnn_conv2d(): status not ZDNN_FUNC_RC_F003");
}

// stride height > 13
void test_f004_stride_height_fail() {
  uint32_t bad_stride_height = 15;

  input_set i = {1, 2, 2, 1, {1, 1}, 1};
  input_set *set = &i; // just so we can copy and paste code

  strides_input_set s = {bad_stride_height, 1};
  strides_input_set *strides = &s;

  zdnn_status status;

  zdnn_pool_padding padding = VALID_PADDING;

  uint32_t input_dims[4] = {set->n, set->height_in, set->width_in,
                            set->channel_in};
  uint32_t kernel_dims[4] = {set->kernel_size[0], set->kernel_size[1],
                             set->channel_in, set->channel_out};
  uint32_t bias_dims[1] = {set->channel_out};
  uint32_t output_dims[4] = {set->n, 1, 2, set->channel_out};

  zdnn_ztensor *input_ztensor = alloc_ztensor_with_values(
      input_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *kernel_ztensor = alloc_ztensor_with_values(
      kernel_dims, ZDNN_HWCK, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *bias_ztensor = alloc_ztensor_with_values(
      bias_dims, ZDNN_1D, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *output_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  status = zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                       strides->height, strides->width, CONV2D_ACT_NONE, NULL,
                       output_ztensor);

  TEST_ASSERT_MESSAGE(status == ZDNN_FUNC_RC_F004,
                      "zdnn_conv2d(): status not ZDNN_FUNC_RC_F004");
}

// stride width > 13
void test_f004_stride_width_fail() {
  uint32_t bad_stride_width = 15;

  input_set i = {1, 2, 2, 1, {1, 1}, 1};
  input_set *set = &i; // just so we can copy and paste code

  strides_input_set s = {1, bad_stride_width};
  strides_input_set *strides = &s;

  zdnn_status status;

  zdnn_pool_padding padding = VALID_PADDING;

  uint32_t input_dims[4] = {set->n, set->height_in, set->width_in,
                            set->channel_in};
  uint32_t kernel_dims[4] = {set->kernel_size[0], set->kernel_size[1],
                             set->channel_in, set->channel_out};
  uint32_t bias_dims[1] = {set->channel_out};
  uint32_t output_dims[4] = {set->n, 2, 1, set->channel_out};

  zdnn_ztensor *input_ztensor = alloc_ztensor_with_values(
      input_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *kernel_ztensor = alloc_ztensor_with_values(
      kernel_dims, ZDNN_HWCK, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *bias_ztensor = alloc_ztensor_with_values(
      bias_dims, ZDNN_1D, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  zdnn_ztensor *output_ztensor = alloc_ztensor_with_values(
      output_dims, ZDNN_NHWC, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  status = zdnn_conv2d(input_ztensor, kernel_ztensor, bias_ztensor, padding,
                       strides->height, strides->width, CONV2D_ACT_NONE, NULL,
                       output_ztensor);

  TEST_ASSERT_MESSAGE(status == ZDNN_FUNC_RC_F004,
                      "zdnn_conv2d(): status not ZDNN_FUNC_RC_F004");
}

int main() {
  UNITY_BEGIN();
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(
      test_valid_padding_non_zero_strides_small);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(
      test_valid_padding_non_zero_strides_small_with_clip);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_valid_padding_zero_strides_small);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(
      test_same_padding_non_zero_strides_small);

  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(
      test_valid_padding_non_zero_strides_medium);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_valid_padding_zero_strides_medium);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(
      test_valid_padding_zero_strides_medium_with_clip);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(
      test_same_padding_non_zero_strides_medium);

  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(
      test_valid_padding_non_zero_strides_large);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_valid_padding_zero_strides_large);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(
      test_same_padding_non_zero_strides_large);

  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_f000_fail);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_f001_fail);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_f002_height_fail);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_f002_width_fail);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_f003_height_fail);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_f003_width_fail);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_f004_stride_height_fail);
  RUN_TEST_ALL_DLFLOAT16_PRE_DATATYPES(test_f004_stride_width_fail);
  return UNITY_END();
}
