/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <kstat.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/swap.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>

#include "utils.h"

static int populate_swaptable(uint16_t nswt_ent);

__thread int s_pagesize = 0;
__thread swaptbl_t* swaptable;
__thread size_t swaptable_size;
__thread uint16_t number_swaps;

int sa_open_swap() {
	swaptable = NULL;
	swaptable_size = 0;

	s_pagesize = sysconf(_SC_PAGESIZE);
	return 0;
}

int sa_close_swap() {
	if (swaptable != NULL) {
		int i;
		for (i = 0; i < swaptable_size / sizeof(struct swapent); ++i)
			free(swaptable->swt_ent[i].ste_path);
		free(swaptable);
	}
	return 0;
}

int sa_count_swaps(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	int ret = swapctl(SC_GETNSWP, NULL);
	if (ret == -1)
		return errno;

	*number = (uint16_t) ret;
	return 0;
}

int sa_reset_swaps() {
	int nswaps = swapctl(SC_GETNSWP, NULL);
	if (nswaps == -1)
		return errno;

	int ret = populate_swaptable(nswaps);
	if (ret != 0)
		return ret;

	number_swaps = nswaps;
	return 0;
}

int sa_get_swap(uint16_t index, struct sa_swap* dst) {
	if (dst == NULL)
		return EINVAL;

	if (index >= number_swaps)
		return ENODEV;

	strncpy(dst->name, swaptable->swt_ent[index].ste_path, sizeof(dst->name));
	dst->total = (uint64_t) swaptable->swt_ent[index].ste_pages * s_pagesize;
	dst->free = (uint64_t) swaptable->swt_ent[index].ste_free * s_pagesize;
	return 0;
}

int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	*written = 0;
	uint16_t i;
	for (i = 0; i < number_swaps; ++i) {
		if (i == dst_size)
			return ENOMEM;

		strncpy(dst[i].name, swaptable->swt_ent[i].ste_path, sizeof(dst->name));
		dst[i].total = (uint64_t) swaptable->swt_ent[i].ste_pages * s_pagesize;
		dst[i].free = (uint64_t) swaptable->swt_ent[i].ste_free * s_pagesize;
		++(*written);
	}
	return 0;
}

static int populate_swaptable(uint16_t nswt_ent) {
	size_t len = sizeof(int) + (nswt_ent * sizeof(struct swapent));
	if (len > swaptable_size) {
		safe_realloc((void*) &swaptable, &len);
		if (swaptable == NULL) {
			swaptable_size = len;
			return ENOMEM;
		}

		int i;
		for (i = swaptable_size / sizeof(struct swapent); i < nswt_ent; ++i) {
			swaptable->swt_ent[i].ste_path = (char*) malloc(MAXPATHLEN);
			if (swaptable->swt_ent[i].ste_path == NULL)
				return ENOMEM;
		}
		swaptable_size = len;
	}

	swaptable->swt_n = nswt_ent;
	if (swapctl(SC_LIST, swaptable) == -1)
		return errno;
	return 0;
}
