/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include <PNGHelper.h>
#include <iostream>
#include <cstdio>
#include <cstdlib>
#include <png.h>

namespace libgltf { namespace pnghelper
{

bool ReadPNGFromFile(const std::string& sFileName, char** pBuffer, int& nWidth, int& nHeight)
{
    unsigned char aHeader[8];    // 8 is the maximum size that can be checked

    /* open file and test for it being a png */
    FILE* pFile = fopen(sFileName.c_str(), "rb");
    if( !pFile )
    {
        std::cerr << "File " << sFileName.c_str() << " could not be opened for reading" << std::endl;
        return false;
    }

    fread(aHeader, 1, 8, pFile);
    if (png_sig_cmp(aHeader, 0, 8))
    {
        std::cerr << "File " << sFileName.c_str() << " is not recognized as a PNG file" << std::endl;
        fclose(pFile);
        return false;
    }


    /* initialization */
    png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);

    if (!png_ptr)
    {
        std::cerr << "png_create_read_struct failed" << std::endl;
        fclose(pFile);
        return false;
    }

    png_infop info_ptr = png_create_info_struct(png_ptr);
    if (!info_ptr)
    {
        std::cerr << "png_create_info_struct failed" << std::endl;
        fclose(pFile);
        return false;
    }

    if (setjmp(png_jmpbuf(png_ptr)))
    {
        std::cerr << "Error during init_io" << std::endl;
        fclose(pFile);
        return false;
    }

    png_init_io(png_ptr, pFile);
    png_set_sig_bytes(png_ptr, 8);

    png_read_info(png_ptr, info_ptr);

    nWidth = png_get_image_width(png_ptr, info_ptr);
    nHeight = png_get_image_height(png_ptr, info_ptr);

    png_read_update_info(png_ptr, info_ptr);

    /* read file */
    if (setjmp(png_jmpbuf(png_ptr)))
    {
        std::cerr << "Error during read_image" << std::endl;
        return false;
    }

    png_bytep* pRowPointers = (png_bytep*) malloc(sizeof(png_bytep) * nHeight);
    for (int i = 0; i < nHeight; ++i)
    {
        pRowPointers[i] = (png_byte*) malloc(png_get_rowbytes(png_ptr,info_ptr));
    }

    png_read_image(png_ptr, pRowPointers);

    int nColorType = png_get_color_type(png_ptr, info_ptr);

    if( nColorType != PNG_COLOR_TYPE_RGB && nColorType != PNG_COLOR_TYPE_RGBA )
    {
        std::cerr << "Input file must be PNG_COLOR_TYPE_RGB or PNG_COLOR_TYPE_RGBA" << std::endl;
        return false;
    }

    (*pBuffer) = new char[4*nWidth*nHeight];
    unsigned nIndex = 0;
    for (int i = 0; i < nHeight; ++i)
    {
        png_byte* pRow = pRowPointers[i];
        for ( int j = 0; j < nWidth; ++j)
        {
            png_byte* pPtr;
            if( nColorType == PNG_COLOR_TYPE_RGB )
            {
                pPtr = &(pRow[j*3]);
            }
            else
            {
                pPtr = &(pRow[j*4]);
            }
            (*pBuffer)[nIndex++] = pPtr[0];
            (*pBuffer)[nIndex++] = pPtr[1];
            (*pBuffer)[nIndex++] = pPtr[2];
            if( nColorType == PNG_COLOR_TYPE_RGB )
            {
                (*pBuffer)[nIndex++] = 127;
            }
            else
            {
                (*pBuffer)[nIndex++] = pPtr[3];
            }
        }
    }

    fclose(pFile);
    return true;
}

} // pnghelper
} // libgltf
