import errno
import json
import os
import sys
import time

import SecureOSFunctions

# Have a registry of all persistable components. Those might be
# happy to be invoked before python process is terminating.
persistableComponents=[]
def addPersistableComponent(component):
  persistableComponents.append(component)

def openPersistenceFile(fileName, flags):
  """This function opens the given persistence file. When O_CREAT
  was specified, the function will attempt to create the directories
  too."""
  try:
    fd=SecureOSFunctions.secureOpenFile(fileName, flags)
    return(fd)
  except OSError as openOsError:
    if ((flags&os.O_CREAT)==0) or (openOsError.errno!=errno.ENOENT):
      raise openOsError

# Find out, which directory is missing by stating our way up.
  dirNameLength=fileName.rfind('/')
  if(dirNameLength>0): os.makedirs(fileName[:dirNameLength])
  return(SecureOSFunctions.secureOpenFile(fileName, flags))

def createTemporaryPersistenceFile(fileName):
  """Create a temporary file within persistence directory to write
  new persistence data to it. Thus the old data is not modified,
  any error creating or writing the file will not harm the old
  state."""
  fd=None
  while True:
# FIXME: This should use O_TMPFILE, but not yet available. That would
# obsolete the loop also.
    fd=openPersistenceFile('%s.tmp-%f' % (fileName, time.time()), os.O_WRONLY|os.O_CREAT|os.O_EXCL)
    break
  return(fd)

noSecureLinkUnlinkAtWarnOnceFlag=True
def replacePersistenceFile(fileName, newFileHandle):
  """Replace the named file with the file refered by the handle."""
  global noSecureLinkUnlinkAtWarnOnceFlag
  if noSecureLinkUnlinkAtWarnOnceFlag:
    print >>sys.stderr, 'WARNING: SECURITY: unsafe unlink (unavailable unlinkat/linkat should be used, but not available in python)'
    noSecureLinkUnlinkAtWarnOnceFlag=False
  try:
    os.unlink(fileName)
  except OSError as openOsError:
    if openOsError.errno!=errno.ENOENT:
      raise openOsError

  tmpFileName=os.readlink('/proc/self/fd/%d' % newFileHandle)
  os.link(tmpFileName, fileName)
  os.unlink(tmpFileName)

def persistAll():
  for component in persistableComponents:
    component.doPersist()

def loadJson(fileName):
  """Load persistency data from file.
  @return None if file did not yet exist."""
  persistenceData=None
  try:
    persistenceFileHandle=openPersistenceFile(fileName,
        os.O_RDONLY|os.O_NOFOLLOW)
    persistenceData=os.read(persistenceFileHandle,
        os.fstat(persistenceFileHandle).st_size)
    os.close(persistenceFileHandle)
  except OSError as openOsError:
    if openOsError.errno!=errno.ENOENT:
      raise openOsError
    return(None)

  result=None
  try:
    result=json.loads(persistenceData)
  except ValueError as valueError:
    raise Exception('Corrupted data in %s' % fileName, valueError)

  return(result)


def storeJson(fileName, objectData):
  """Store persistency data to file."""
  persistenceData=json.dumps(objectData)
  fd=createTemporaryPersistenceFile(fileName)
  os.write(fd, persistenceData)
  replacePersistenceFile(fileName, fd)
  os.close(fd)
