// Copyright (c) 2015, Daniel Martí <mvdan@mvdan.cc>
// See LICENSE for licensing information

package main

import (
	"log"
	"os"
	"regexp"
	"strconv"
	"strings"
	"text/template"
	"unicode"
)

const path = "unicode.go"

var tmpl = template.Must(template.New("tlds").Parse(`// Generated by unicodegen

package xurls

const otherPuncMinusDoubleQuote = {{.}}
`))

func visit(rt *unicode.RangeTable, fn func(rune)) {
	for _, r16 := range rt.R16 {
		for r := rune(r16.Lo); r <= rune(r16.Hi); r += rune(r16.Stride) {
			fn(r)
		}
	}
	for _, r32 := range rt.R32 {
		for r := rune(r32.Lo); r <= rune(r32.Hi); r += rune(r32.Stride) {
			fn(r)
		}
	}
}

func writeUnicode() error {
	var b strings.Builder
	visit(unicode.Po, func(r rune) {
		if r != '"' {
			b.WriteRune(r)
		}
	})
	f, err := os.Create(path)
	if err != nil {
		return err
	}
	defer f.Close()
	return tmpl.Execute(f, strconv.Quote(regexp.QuoteMeta(b.String())))
}

func main() {
	log.Printf("Generating %s...", path)
	if err := writeUnicode(); err != nil {
		log.Fatalf("Could not write path: %v", err)
	}
}
