/*
 *  Copyright 2011  Integrated Computer Solutions - http://www.ics.com
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#ifndef MEDIA_H
#define MEDIA_H

#include <QtMultimedia/QMediaPlayer>
#include <QtCore>
#include <QAudioRecorder>
#include <QtMultimedia/QAudioEncoderSettings>

#include "../cplugin.h"

class Player;

class Media: public CPlugin {
    Q_OBJECT
public:
    explicit Media(Cordova *cordova): CPlugin(cordova) {
    }

    virtual const QString fullName() override {
        return Media::fullID();
    }

    virtual const QString shortName() override {
        return "Media";
    }

    static const QString fullID() {
        return "com.cordova.Media";
    }

    enum State {
        MEDIA_NONE = 0,
        MEDIA_STARTING = 1,
        MEDIA_RUNNING = 2,
        MEDIA_PAUSED = 3,
        MEDIA_STOPPED = 4
    };
    enum ErrorCode {
        MEDIA_ERR_NONE_ACTIVE = 0,
        MEDIA_ERR_ABORTED = 1,
        MEDIA_ERR_NETWORK = 2,
        MEDIA_ERR_DECODE = 3,
        MEDIA_ERR_NONE_SUPPORTED = 4
    };

public slots:
    void newPlayer(int scId, int ecId, const QString &src, int id);
    void relasePlayer(int scId, int ecId, QVariantMap p_options);
    void playerSetCallbacks(int successCb, int statusCb, QVariantMap p_options);

    void startRecording(int scId, int ecId, QVariantMap p_options);
    void stopRecording(int scId, int ecId, QVariantMap p_options);

    void play(int scId, int ecId, QVariantMap p_options);
    void pause(int scId, int ecId, QVariantMap p_options);
    void stop(int scId, int ecId, QVariantMap p_options);
    void getDuration(int scId, int ecId, QVariantMap p_options);
    void getCurrentPosition(int scId, int ecId, QVariantMap p_options);
    void seekTo(int scId, int ecId, QVariantMap p_options);
    void setVolume(int scId, int ecId, QVariantMap p_options);

private:
    QMap<int, QSharedPointer<Player> > m_id2Player;
};

class Player: public QObject {
    Q_OBJECT
public:
    Player(int ecId, QString src, CPlugin *plugin):
        m_state(Media::MEDIA_NONE),
        m_src(src),
        m_successCb(0),
        m_statusCb(0),
        m_errorCb(ecId),
        m_mode(MODE_NONE),
        m_plugin(plugin) {
        QUrl url(src, QUrl::TolerantMode);

        if (url.scheme().isEmpty()) {
            QAudioEncoderSettings audioSettings;

            m_recorder.setEncodingSettings(audioSettings);
            m_recorder.setOutputLocation(QFileInfo(src).absoluteFilePath());

            m_player.setMedia(QUrl::fromLocalFile(QFileInfo(src).absoluteFilePath()));
        } else {
            m_player.setMedia(url);
        }
        QObject::connect(&m_player, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)), this, SLOT(onMediaStatusChanged(QMediaPlayer::MediaStatus)));
        QObject::connect(&m_recorder, SIGNAL(error(QMediaRecorder::Error)), this, SLOT(onError(QMediaRecorder::Error)));
    }

    void setSuccessCb(int successCb) {
        m_successCb = successCb;
    }
    void setStatusCb(int statusCb) {
        m_statusCb = statusCb;
    }

    void startRecording() {
        if (recordMode() && m_state != Media::MEDIA_RUNNING) {
            m_recorder.record();
            setState(Media::MEDIA_RUNNING);
        }
    }
    void stopRecording() {
        if (recordMode() && m_state == Media::MEDIA_RUNNING) {
            m_recorder.stop();
            setState(Media::MEDIA_STOPPED);
        }
    }

    void setVolume(int volume) {
        m_player.setVolume(volume);
    }

    void play() {
        if (playMode() && m_state != Media::MEDIA_RUNNING) {
            m_player.play();
            setState(Media::MEDIA_RUNNING);
        }
    }
    void pause() {
        if (playMode() && m_state == Media::MEDIA_RUNNING) {
            m_player.pause();
            setState(Media::MEDIA_PAUSED);
        }
    }
    void stop() {
        if (playMode() && (m_state == Media::MEDIA_RUNNING || m_state == Media::MEDIA_PAUSED)) {
            m_player.stop();
            setState(Media::MEDIA_STOPPED);
        }
    }
    qint64 getDuration() {
        if (m_mode == MODE_NONE)
            return -1;
        if (m_mode != MODE_PLAY)
            return -2;
        return m_player.duration() / 1000;
    }
    qint64 getPosition() {
        if (m_mode != MODE_PLAY)
            return -1;
        return m_player.position() / 1000;
    }
    bool seekTo(qint64 position) {
        if (!m_player.isSeekable())
            return false;
        m_player.setPosition(position);
        return true;
    }
private slots:
    void onMediaStatusChanged(QMediaPlayer::MediaStatus status) {
        if (status == QMediaPlayer::InvalidMedia) {
            m_plugin->callbackWithoutRemove(m_errorCb, QString("new MediaError(%1, 'AudioPlayer Error: The current media cannot be played.')").arg(Media::MEDIA_ERR_ABORTED));
            setState(Media::MEDIA_STOPPED);
        }
        if (status == QMediaPlayer::EndOfMedia) {
            setState(Media::MEDIA_STOPPED);
            seekTo(0);
        }
    }
    void onError(QMediaRecorder::Error) {
        m_plugin->callbackWithoutRemove(m_errorCb, QString("new MediaError(%1, 'AudioPlayer Error: Device is not ready or not available.')").arg(Media::MEDIA_ERR_NONE_SUPPORTED));
        setState(Media::MEDIA_STOPPED);
    }

private:
    bool playMode() {
        switch (m_mode) {
        case Player::MODE_NONE:
            m_mode = MODE_PLAY;
            break;
        case Player::MODE_PLAY:
            break;
        case Player::MODE_RECORD:
            m_plugin->callbackWithoutRemove(m_errorCb, QString("new MediaError(%1, 'AudioPlayer Error: Can't play in record mode.')").arg(Media::MEDIA_ERR_NONE_SUPPORTED));
            return false;
            break;
        }
        return true;
    }

    bool recordMode() {
        switch (m_mode) {
        case Player::MODE_NONE:
            if (m_recorder.outputLocation().isEmpty()) {
                m_plugin->callbackWithoutRemove(m_errorCb, QString("new MediaError(%1, 'AudioPlayer Error: unsupported output location.')").arg(Media::MEDIA_ERR_NONE_SUPPORTED));
                return false;
            }
            m_mode = MODE_RECORD;
            break;
        case Player::MODE_PLAY:
            m_plugin->callbackWithoutRemove(m_errorCb, QString("new MediaError(%1, 'AudioPlayer Error: Can't play in play mode.')").arg(Media::MEDIA_ERR_NONE_SUPPORTED));
            return false;
            break;
        case Player::MODE_RECORD:
            break;
        }
        return true;
    }

    void setState(Media::State state) {
        m_state = state;
        m_plugin->callbackWithoutRemove(m_statusCb, QString("%1").arg(state));
    }

    QMediaPlayer m_player;

    QAudioRecorder m_recorder;

    Media::State m_state;
    QString m_src;
    int m_successCb, m_statusCb, m_errorCb;
    enum Mode {
        MODE_NONE,
        MODE_PLAY,
        MODE_RECORD
    };
    Mode m_mode;
    CPlugin *m_plugin;
};

#endif // MEDIA_H
