#!/bin/bash


function assert
{
    MESSAGE=$1
    shift
    eval $@ || echo $@ WARNING: $MESSAGE
}


HERE=$(basename $PWD|sed -e 's/-[0-9].*$//')
USERSCRIPT=$(echo *.user.js)
APPNAME=$(echo *.user.js|sed -e 's/\.user\.js//')
LOWERCASE=${APPNAME,,}
CHANGELOG=$(dpkg-parsechangelog|perl -ne 's/Source: // && print && exit')
SRCPKG=$(perl -ne 's/Source: // && print && exit' debian/control)
BINPKG=$(perl -ne 's/Package: // && print && exit' debian/control)
XBUNAME=$(perl -ne 's/XB-Ubuntu-Webapps-Name: // && print && exit' debian/control)


echo Linting $HERE...


#assert "Error Message" [ test case ]
assert "Missing." [ -s debian/rules ]
assert "Missing." [ -s debian/control ]
assert "Missing." [ -s manifest.json ]
assert "Missing." [ -s $APPNAME.test.js ]
assert "Missing." [ -s COPYING ]
assert "Missing." [ -s po/POTFILES.in ]
assert "Missing." [ -s po/$LOWERCASE.pot ]


assert "Missing build dependency on unity-webapps-dev."  \
    perl -e "'local$/;\$_=<>;m/Build-Depends:[^:]+unity-webapps-dev/||exit 1'"  \
    debian/control

assert "Missing dependency on unity-webapps-common."  \
    perl -e "'local$/;\$_=<>;m/\nDepends:[^:]+unity-webapps-common/||exit 1'"  \
    debian/control

assert "debian/compat isn't 9"  \
    diff debian/compat <(echo 9)


# There is a lot of tooling that makes assumptions about the names of things
# matching, so it's important to confirm that everything is consistent
assert "Source package and userscript names differ."  \
    [ \"$SRCPKG\" = \"unity-webapps-$LOWERCASE\" ]
assert "Binary package and userscript names differ."  \
    [ \"$BINPKG\" = \"unity-webapps-$LOWERCASE\" ]
assert "Source package and XB-Ubuntu-Webapps-Name differs."  \
    [ \"$SRCPKG\" = \"unity-webapps-${XBUNAME,,}\" ]
assert "Binary package and XB-Ubuntu-Webapps-Name differs."  \
    [ \"$BINPKG\" = \"unity-webapps-${XBUNAME,,}\" ]
assert "Project name doesn't match userscript name."  \
    [ \"$HERE\" = \"unity-webapps-$LOWERCASE\" ]
assert "Source and Binary package names differ."  \
    [ \"$SRCPKG\" = \"$BINPKG\" ]
assert "Source package and changelog names differ."  \
    [ \"$SRCPKG\" = \"$CHANGELOG\" ]
assert "Binary package and changelog names differ."  \
    [ \"$BINPKG\" = \"$CHANGELOG\" ]
assert "App name and XB-Ubuntu-Webapps-Name differs."  \
    [ \"$APPNAME\" = \"$XBUNAME\" ]


assert "Non-standard debian/rules"  \
    diff debian/rules - <<EOF
#!/usr/bin/make -f

# This is the standard debian/rules file common to all webapps.
include /usr/share/unity-webapps/rules.mk
EOF


assert "Test script is looking in the wrong place."  \
    grep -qs \"scriptName: \'$USERSCRIPT\'\" $APPNAME.test.js

assert "Wrong userscript in manifest.json."  \
    egrep -qs "scripts...\\\"$USERSCRIPT\\\"" manifest.json

assert "Wrong package-name in manifest.json."  \
    egrep -qs "package-name..\\\"$APPNAME\\\"" manifest.json

assert "POTFILES.in does not contain $USERSCRIPT"  \
    diff po/POTFILES.in <(echo $USERSCRIPT)

assert "Userscript is missing GETTEXT PLACEHOLDER"  \
    grep -qs \"%7B%22GETTEXT%22%3A%22PLACEHOLDER%22%7D\" $USERSCRIPT

ICONS_PRESENT=$(ls */*.png 2>/dev/null|wc -l)
ICONS_INSTALLED=$(grep -c png debian/install)
ICONS_COPYRIGHT=$(grep -c png debian/copyright)
assert "Found $ICONS_PRESENT icons but $ICONS_INSTALLED of them installed."  \
    [ $ICONS_PRESENT -le $ICONS_INSTALLED ]
if [ $ICONS_PRESENT -eq 0 ]; then
    assert "Nonexistent icons mentioned in debian/copyright"  \
        [ $ICONS_COPYRIGHT -eq 0 ]
else
    assert "Icons present but not mentioned in debian/copyright"  \
        [ $ICONS_COPYRIGHT -eq 1 ]
fi
