% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/naive-time.R
\name{naive_time_info}
\alias{naive_time_info}
\title{Info: naive-time}
\usage{
naive_time_info(x, zone)
}
\arguments{
\item{x}{\verb{[clock_naive_time]}

A naive-time.}

\item{zone}{\verb{[character]}

A valid time zone name.

Unlike most functions in clock, in \code{naive_time_info()} \code{zone} is vectorized
and is recycled against \code{x}.}
}
\value{
A data frame of low level information.
}
\description{
\code{naive_time_info()} retrieves a set of low-level information generally not
required for most date-time manipulations. It is used implicitly
by \code{as_zoned_time()} when converting from a naive-time.

It returns a data frame with the following columns:
\itemize{
\item \code{type}: A character vector containing one of:
\itemize{
\item \code{"unique"}: The naive-time maps uniquely to a zoned-time that can be
created with \code{zone}.
\item \code{"nonexistent"}: The naive-time does not exist as a zoned-time that can
be created with \code{zone}.
\item \code{"ambiguous"}: The naive-time exists twice as a zoned-time that can be
created with \code{zone}.
}
\item \code{first}: A \code{\link[=sys_time_info]{sys_time_info()}} data frame.
\item \code{second}: A \code{\link[=sys_time_info]{sys_time_info()}} data frame.
}
\subsection{type == "unique"}{
\itemize{
\item \code{first} will be filled out with sys-info representing daylight saving time
information for that time point in \code{zone}.
\item \code{second} will contain only \code{NA} values, as there is no ambiguity to
represent information for.
}
}

\subsection{type == "nonexistent"}{
\itemize{
\item \code{first} will be filled out with the sys-info that ends just prior to \code{x}.
\item \code{second} will be filled out with the sys-info that begins just after \code{x}.
}
}

\subsection{type == "ambiguous"}{
\itemize{
\item \code{first} will be filled out with the sys-info that ends just after \code{x}.
\item \code{second} will be filled out with the sys-info that starts just before \code{x}.
}
}
}
\details{
If the tibble package is installed, it is recommended to convert the output
to a tibble with \code{as_tibble()}, as that will print the df-cols much nicer.
}
\examples{
library(vctrs)

x <- year_month_day(1970, 04, 26, 02, 30, 00)
x <- as_naive_time(x)

# Maps uniquely to a time in London
naive_time_info(x, "Europe/London")

# This naive-time never existed in New York!
# A DST gap jumped the time from 01:59:59 -> 03:00:00,
# skipping the 2 o'clock hour
zone <- "America/New_York"
info <- naive_time_info(x, zone)
info

# You can recreate various `nonexistent` strategies with this info
as_zoned_time(x, zone, nonexistent = "roll-forward")
as_zoned_time(info$first$end, zone)

as_zoned_time(x, zone, nonexistent = "roll-backward")
as_zoned_time(info$first$end - 1, zone)

as_zoned_time(x, zone, nonexistent = "shift-forward")
as_zoned_time(as_sys_time(x) - info$first$offset, zone)

as_zoned_time(x, zone, nonexistent = "shift-backward")
as_zoned_time(as_sys_time(x) - info$second$offset, zone)

# ---------------------------------------------------------------------------
# Normalizing to UTC

# Imagine you had the following printed times, and knowledge that they
# are to be interpreted as in the corresponding time zones
df <- data_frame(
  x = c("2020-01-05 02:30:00", "2020-06-03 12:20:05"),
  zone = c("America/Los_Angeles", "Europe/London")
)

# The times are assumed to be naive-times, i.e. if you lived in the `zone`
# at the moment the time was recorded, then you would have seen that time
# printed on the clock. Currently, these are strings. To convert them to
# a time based type, you'll have to acknowledge that R only lets you have
# 1 time zone in a vector of date-times at a time. So you'll need to
# normalize these naive-times. The easiest thing to normalize them to
# is UTC.
df$naive <- naive_time_parse(df$x)

# Get info about the naive times using a vector of zones
info <- naive_time_info(df$naive, df$zone)
info

# We'll assume that some system generated these naive-times with no
# chance of them ever being nonexistent or ambiguous. So now all we have
# to do is use the offset to convert the naive-time to a sys-time. The
# relationship used is:
# offset = naive_time - sys_time
df$sys <- as_sys_time(df$naive) - info$first$offset
df

# At this point, both times are in UTC. From here, you can convert them
# both to either America/Los_Angeles or Europe/London as required.
as_zoned_time(df$sys, "America/Los_Angeles")
as_zoned_time(df$sys, "Europe/London")
}
