var Buffer = require('buffer').Buffer;
var Crypto = require('crypto');
var Auth   = exports;

function sha1(msg) {
  var hash = Crypto.createHash('sha1');
  hash.update(msg);
  // hash.digest() does not output buffers yet
  return hash.digest('binary');
};
Auth.sha1 = sha1;

function xor(a, b) {
  a = new Buffer(a, 'binary');
  b = new Buffer(b, 'binary');
  var result = new Buffer(a.length);
  for (var i = 0; i < a.length; i++) {
    result[i] = (a[i] ^ b[i]);
  }
  return result;
};
Auth.xor = xor;

Auth.token = function(password, scramble) {
  if (!password) {
    return new Buffer(0);
  }

  // password must be in binary format, not utf8
  var stage1 = sha1((new Buffer(password, "utf8")).toString("binary"));
  var stage2 = sha1(stage1);
  var stage3 = sha1(scramble.toString('binary') + stage2);
  return xor(stage3, stage1);
};

// This is a port of sql/password.c:hash_password which needs to be used for
// pre-4.1 passwords.
Auth.hashPassword = function(password) {
  var nr = [0x5030, 0x5735],
      add = 7,
      nr2 = [0x1234, 0x5671],
      result = new Buffer(8);

  if (typeof password == 'string'){
    password = new Buffer(password);
  }

  for (var i = 0; i < password.length; i++) {
    var c = password[i];
    if (c == 32 || c == 9) {
      // skip space in password
      continue;
    }

    // nr^= (((nr & 63)+add)*c)+ (nr << 8);
    // nr = xor(nr, add(mul(add(and(nr, 63), add), c), shl(nr, 8)))
    nr = this.xor32(nr, this.add32(this.mul32(this.add32(this.and32(nr, [0,63]), [0,add]), [0,c]), this.shl32(nr, 8)));

    // nr2+=(nr2 << 8) ^ nr;
    // nr2 = add(nr2, xor(shl(nr2, 8), nr))
    nr2 = this.add32(nr2, this.xor32(this.shl32(nr2, 8), nr));

    // add+=tmp;
    add += c;
  }

  this.int31Write(result, nr, 0);
  this.int31Write(result, nr2, 4);

  return result;
};

Auth.randomInit = function(seed1, seed2) {
  return {
    max_value: 0x3FFFFFFF,
    max_value_dbl: 0x3FFFFFFF,
    seed1: seed1 % 0x3FFFFFFF,
    seed2: seed2 % 0x3FFFFFFF
  };
};

Auth.myRnd = function(r){
  r.seed1 = (r.seed1 * 3 + r.seed2) % r.max_value;
  r.seed2 = (r.seed1 + r.seed2 + 33) % r.max_value;

  return r.seed1 / r.max_value_dbl;
};

Auth.scramble323 = function(message, password) {
  var to = new Buffer(8),
      hashPass = this.hashPassword(password),
      hashMessage = this.hashPassword(message.slice(0, 8)),
      seed1 = this.int32Read(hashPass, 0) ^ this.int32Read(hashMessage, 0),
      seed2 = this.int32Read(hashPass, 4) ^ this.int32Read(hashMessage, 4),
      r = this.randomInit(seed1, seed2);

  for (var i = 0; i < 8; i++){
    to[i] = Math.floor(this.myRnd(r) * 31) + 64;
  }
  var extra = (Math.floor(this.myRnd(r) * 31));

  for (var i = 0; i < 8; i++){
    to[i] ^= extra;
  }

  return to;
};

Auth.fmt32 = function(x){
  var a = x[0].toString(16),
      b = x[1].toString(16);

  if (a.length == 1) a = '000'+a;
  if (a.length == 2) a = '00'+a;
  if (a.length == 3) a = '0'+a;
  if (b.length == 1) b = '000'+b;
  if (b.length == 2) b = '00'+b;
  if (b.length == 3) b = '0'+b;
  return '' + a + '/' + b;
};

Auth.xor32 = function(a,b){
  return [a[0] ^ b[0], a[1] ^ b[1]];
};

Auth.add32 = function(a,b){
  var w1 = a[1] + b[1],
      w2 = a[0] + b[0] + ((w1 & 0xFFFF0000) >> 16);

  return [w2 & 0xFFFF, w1 & 0xFFFF];
};

Auth.mul32 = function(a,b){
  // based on this example of multiplying 32b ints using 16b
  // http://www.dsprelated.com/showmessage/89790/1.php
  var w1 = a[1] * b[1],
      w2 = (((a[1] * b[1]) >> 16) & 0xFFFF) + ((a[0] * b[1]) & 0xFFFF) + (a[1] * b[0] & 0xFFFF);

  return [w2 & 0xFFFF, w1 & 0xFFFF];
};

Auth.and32 = function(a,b){
  return [a[0] & b[0], a[1] & b[1]];
};

Auth.shl32 = function(a,b){
  // assume b is 16 or less
  var w1 = a[1] << b,
      w2 = (a[0] << b) | ((w1 & 0xFFFF0000) >> 16);

  return [w2 & 0xFFFF, w1 & 0xFFFF];
};

Auth.int31Write = function(buffer, number, offset) {
  buffer[offset] = (number[0] >> 8) & 0x7F;
  buffer[offset + 1] = (number[0]) & 0xFF;
  buffer[offset + 2] = (number[1] >> 8) & 0xFF;
  buffer[offset + 3] = (number[1]) & 0xFF;
};

Auth.int32Read = function(buffer, offset){
  return (buffer[offset] << 24)
       + (buffer[offset+1] << 16)
       + (buffer[offset+2] << 8)
       + (buffer[offset+3]);
};
