# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file provides access to "quilt" functionality required by "gquilt"

import os.path, os, re, pango

from gquilt_pkg import gquilt_tool
from gquilt_pkg import console, fsdb, ws_event, cmd_result
from gquilt_pkg import const
from gquilt_pkg import putils
from gquilt_pkg import utils

FSTATUS_MODIFIED = 'M'
FSTATUS_ADDED = 'A'
FSTATUS_REMOVED = 'R'
FSTATUS_CLEAN = 'C'
FSTATUS_MISSING = '!'
FSTATUS_NOT_TRACKED = '?'
FSTATUS_IGNORED = 'I'
FSTATUS_ORIGIN = ' '
FSTATUS_UNRESOLVED = 'U'

FSTATUS_MODIFIED_SET = set([FSTATUS_MODIFIED, FSTATUS_ADDED, FSTATUS_REMOVED,
                           FSTATUS_MISSING, FSTATUS_UNRESOLVED])

PatchStatusMap = putils.StatusMap(extant=FSTATUS_MODIFIED, added=FSTATUS_ADDED, deleted=FSTATUS_REMOVED)

class ScmDir(fsdb.GenDir):
    def __init__(self):
        fsdb.GenDir.__init__(self)
    def _new_dir(self):
        return ScmDir()
    def _update_own_status(self):
        if FSTATUS_UNRESOLVED in self.status_set:
            self.status = FSTATUS_UNRESOLVED
        elif self.status_set & FSTATUS_MODIFIED_SET:
            self.status = FSTATUS_MODIFIED
        elif self.status_set == set([FSTATUS_IGNORED]):
            self.status = FSTATUS_IGNORED
        elif self.status_set in [set([FSTATUS_NOT_TRACKED]), set([FSTATUS_NOT_TRACKED, FSTATUS_IGNORED])]:
            self.status = FSTATUS_NOT_TRACKED
    def _is_hidden_dir(self, dkey):
        status = self.subdirs[dkey].status
        if status not in [FSTATUS_UNRESOLVED, FSTATUS_MODIFIED]:
            return dkey[0] == '.' or status == FSTATUS_IGNORED
        return False
    def _is_hidden_file(self, fdata):
        if fdata.status not in FSTATUS_MODIFIED_SET:
            return fdata.name[0] == '.' or fdata.status == FSTATUS_IGNORED
        return False

class ScmFileDb(fsdb.GenFileDb):
    def __init__(self, file_list, unresolved_file_list=list()):
        fsdb.GenFileDb.__init__(self, ScmDir)
        lfile_list = len(file_list)
        index = 0
        while index < lfile_list:
            item = file_list[index]
            index += 1
            filename = item[2:]
            status = item[0]
            origin = None
            if status == FSTATUS_ADDED and index < lfile_list:
                if file_list[index][0] == FSTATUS_ORIGIN:
                    origin = file_list[index][2:]
                    index += 1
            elif filename in unresolved_file_list:
                status = FSTATUS_UNRESOLVED
            parts = filename.split(os.sep)
            self.base_dir.add_file(parts, status, origin)

def _hg_command_avail(cmd):
    res, _sout, _serr = utils.run_cmd("hg help %s" % cmd)
    return res == 0

def _convert_pop_res(res, sout, serr):
    if res != cmd_result.OK and re.search("local changes found, refresh first", sout + serr):
        res |= cmd_result.SUGGEST_REFRESH
    return cmd_result.Result(res, sout, serr)

def _convert_push_res(res, sout, serr):
    if res != cmd_result.OK and re.search("local changes found, refresh first", sout + serr):
        res |= cmd_result.SUGGEST_FORCE_OR_REFRESH
    return cmd_result.Result(res, sout, serr)

def _convert_copy_res(res, sout, serr):
    if re.search("not overwriting - file exists", sout + serr):
        res |= cmd_result.SUGGEST_FORCE
    return cmd_result.Result(res, sout, serr)

def _convert_import_res(res, sout, serr):
    if res != cmd_result.OK and re.search("already exists", sout + serr):
        res |= cmd_result.SUGGEST_FORCE_OR_RENAME
    return cmd_result.Result(res, sout, serr)

def _map_cmd_result(result, ignore_err_re=None, mapper=None):
    if result.eflags == 0:
        if _inotify_warning(result.stdout):
            return result
        result = cmd_result.map_cmd_result(result, ignore_err_re=ignore_err_re)
    elif mapper is None:
        flags = cmd_result.ERROR
        if result.serr.find('use -f to force') != -1:
            flags |= cmd_result.SUGGEST_FORCE
        if result.serr.find('refresh first') != -1:
            flags |= cmd_result.SUGGEST_REFRESH
        if result.serr.find('(revert --all, qpush to recover)') != -1:
            flags |= cmd_result.SUGGEST_RECOVER
        if result.stdout.find('errors during apply, please fix and refresh') != -1:
            flags = cmd_result.WARNING
        result = cmd_result.Result(flags, result.stdout, result.serr)
    if mapper is not None:
        return mapper(*result)
    return result

def _run_cmd_on_console(cmd, ignore_err_re=None, mapper=None):
    result = console.exec_console_cmd(cmd)
    return _map_cmd_result(result, ignore_err_re=ignore_err_re, mapper=mapper)

def _hg_dir(fdir=None):
    if fdir is None:
        fdir = os.getcwd()
    return os.path.join(fdir, ".hg")

def _patch_dir(fdir=None):
    if fdir is None:
        fdir = os.getcwd()
    return os.path.join(fdir, ".hg", "patches")
    
def _patch_file_name(patch):
    return os.path.join(_patch_dir(), patch)

def _numeric_str_sort(arg1, arg2):
    return cmp(int(arg1), int(arg2))

class MQError(Exception):
    """A Mercurial queues (mq) specific error"""

_qpop_re = re.compile("^(popping)\s.*\nnow at:.*$", re.M)
_qpush_re = re.compile("^(merging|applying)\s.*$", re.M)
_qguard_re = re.compile("^\s*([+-].*)\s*$")

def _extract_name_and_guards(string):
    name, gstring = string.split(':')
    match = _qguard_re.match(gstring)
    if match:
        return name, match.group(1).split()
    else:
        return name, []

def _inotify_warning(serr):
    return serr.strip() in ['(found dead inotify server socket; removing it)',
        '(inotify: received response from incompatible server version 1)']

# Now implement the tool interface for mq
class Interface(gquilt_tool.Interface):
    def __init__(self):
        gquilt_tool.Interface.__init__(self, "mq", cmd="hg ")
        self.status_deco_map = {
            None: gquilt_tool.Deco(pango.STYLE_NORMAL, "black"),
            FSTATUS_CLEAN: gquilt_tool.Deco(pango.STYLE_NORMAL, "black"),
            FSTATUS_MODIFIED: gquilt_tool.Deco(pango.STYLE_NORMAL, "blue"),
            FSTATUS_ADDED: gquilt_tool.Deco(pango.STYLE_NORMAL, "darkgreen"),
            FSTATUS_REMOVED: gquilt_tool.Deco(pango.STYLE_NORMAL, "red"),
            FSTATUS_UNRESOLVED: gquilt_tool.Deco(pango.STYLE_NORMAL, "magenta"),
            FSTATUS_MISSING: gquilt_tool.Deco(pango.STYLE_ITALIC, "pink"),
            FSTATUS_NOT_TRACKED: gquilt_tool.Deco(pango.STYLE_ITALIC, "cyan"),
            FSTATUS_IGNORED: gquilt_tool.Deco(pango.STYLE_ITALIC, "grey"),
        }
    def _get_qparent(self):
        cmd = 'hg log --template "{rev}" -r qparent'
        res, rev, serr = utils.run_cmd(cmd)
        if not res and rev:
            return rev
        return None
    def _no_patches_applied(self):
        return self.get_top_patch() == ""
    def _patch_most_recent_parent_rev(self, patch):
        res, plist, serr = self._patch_parent_revs(patch)
        if res != 0:
            return cmd_result.Result(res, plist, serr)
        return cmd_result.Result(res, plist[-1], serr)
    def _patch_parent_revs(self, patch):
        if not self.is_patch_applied(patch) and patch != "qbase":
            return cmd_result.Result(cmd_result.ERROR, "", "Patch %s is not applied." % patch)
        res, sout, serr = utils.run_cmd("hg parent --template \"{rev}" + os.linesep + "\" -r " + patch)
        if res != 0:
            return cmd_result.Result(res, sout, serr)
        elif sout == "":
            return cmd_result.Result(res, ["0"], serr)
        plist = sout.splitlines()
        plist.sort(_numeric_str_sort)
        return cmd_result.Result(res, plist, serr)
    def _unresolved_file_list(self,  fspath_list=None):
        cmd = 'hg resolve --list'
        if fspath_list:
            cmd += ' %s' % utils.file_list_to_string(fspath_list)
        res, sout, serr = utils.run_cmd(cmd)
        files = []
        if not res:
            for line in sout.splitlines():
                if line[0] == FSTATUS_UNRESOLVED:
                    files.append(line[2:])
        return files
    def _get_untracked_or_ignored_files(self, filelist):
        res, sout, serr = utils.run_cmd(" ".join(["hg status -uin"] + filelist))
        if not res:
            return cmd_result.Result(res, sout, serr)
        newfilelist = []
        for line in sout.splitlines(False):
            newfilelist.append(line)
        return cmd_result.Result(res, newfilelist, serr)
    def display_files_diff_in_viewer(self, viewer, files, patch=None):
        top = self.get_top_patch()
        if not top:
            return
        if not patch:
            patch = top
        elif not self.is_patch_applied(patch):
            return
        res, prev, serr = self._patch_most_recent_parent_rev(patch)
        if res != cmd_result.OK:
            raise MQError(serr)
        pats = (os.P_NOWAIT, "hg", "hg", "extdiff", "--program=" + viewer)
        if patch is top:
            pats += ("--rev=" + prev,)
        else:
            pats += ("--rev=" + prev, "--rev=" + patch)
        pats += tuple(files)
        pid = os.spawnlp(*pats)
    def do_add_files_to_patch(self, filelist):
        res, trimmedlist, serr = self._get_untracked_or_ignored_files(filelist)
        if res:
            if trimmedlist:
                ws_event.notify_events(ws_event.FILE_ADD)
            return cmd_result.Result(res, trimmedlist, serr)
        if not trimmedlist:
            return cmd_result.Result(cmd_result.OK, "", "")
        result = _run_cmd_on_console(" ".join(["hg add", trimmedlist]))
        ws_event.notify_events(ws_event.FILE_ADD)
        return result
    def do_copy_file(self, file_path, dest, force=False):
        cmd = "hg copy "
        if force:
            cmd += "-f "
        cmd += " ".join([file_path, dest])
        result = _run_cmd_on_console(cmd, mapper=_convert_copy_res)
        ws_event.notify_events(ws_event.FILE_ADD)
        return result
    def do_delete_patch(self, patch):
        result = _run_cmd_on_console(" ".join(["hg", "qdelete", patch]))
        ws_event.notify_events(ws_event.PATCH_DELETE)
        return result
    def do_exec_tool_cmd(self, cmd):
        result = _run_cmd_on_console(" ".join(["hg", cmd]))
        ws_event.notify_events(ws_event.ALL_EVENTS)
        return result
    def do_finish_patch(self, patch):
        result = _run_cmd_on_console(" ".join(["hg", "qfinish", patch]))
        ws_event.notify_events(ws_event.FILE_CHANGES|ws_event.PATCH_DELETE|ws_event.PATCH_POP)
        return result
    def do_fold_patch(self, patch):
        result = _run_cmd_on_console(" ".join(["hg", "qfold", patch]))
        ws_event.notify_events(ws_event.FILE_CHANGES|ws_event.PATCH_DELETE)
        return result
    def do_fold_patch_file(self, filename):
        res, sout, serr = _run_cmd_on_console(" < ".join(["patch -p1 --dry-run", filename]))
        if res != 0:
            return cmd_result.Result(cmd_result.ERROR, sout, serr)
        res, sout, serr = utils.run_cmd("hg status -u")
        old_unknowns = sout.splitlines()
        pres, pso, pse = _run_cmd_on_console(" < ".join(["patch -p1", filename]))
        if pres != 0:
            ws_event.notify_events(ws_event.FILE_CHANGES)
            return cmd_result.Result(pres, pso, pse)
        res, sout, serr = utils.run_cmd("hg status -u")
        for f in sout.splitlines():
            if f not in old_unknowns:
                console.LOG.exec_cmd_with_alert("hg add " + f[2:])
        ws_event.notify_events(ws_event.FILE_CHANGES)
        return cmd_result.Result(pres, pso, pse)
    def do_import_patch(self, filename, patchname=None, force=False):
        if patchname:
            cmd = "hg qimport -n %s " % patchname
        else:
            cmd = "hg qimport "
            patchname = os.path.basename(filename)
        if self.is_patch_applied(patchname):
            return cmd_result.Result(cmd_result.ERROR, "", "Patch %s is already applied!" % patchname)
        if force:
            cmd += "-f "
        result = _run_cmd_on_console(cmd + filename, mapper=_convert_import_res)
        ws_event.notify_events(ws_event.PATCH_CREATE)
        return result
    def do_move_file(self, file_path, dest, force=False):
        cmd = "hg rename "
        if force:
            cmd += "-f "
        cmd += " ".join([file_path, dest])
        result = _run_cmd_on_console(cmd, mapper=_convert_copy_res)
        ws_event.notify_events(ws_event.FILE_ADD|ws_event.FILE_DEL)
        return result
    def do_new_patch(self, name, force=False):
        cmd = "hg qnew"
        if force:
            cmd += " -f"
        result = _run_cmd_on_console(" ".join([cmd, name]), mapper=_convert_push_res)
        ws_event.notify_events(ws_event.PATCH_CREATE|ws_event.PATCH_PUSH)
        return result
    def do_pop_to(self, patch=None):
        cmd = "hg qpop"
        if patch is not None:
            if patch is "":
                cmd += " -a"
            else:
                cmd += " " + patch
        result = _run_cmd_on_console(cmd, ignore_err_re=_qpop_re, mapper=_convert_pop_res)
        events = ws_event.PATCH_POP
        if not self.get_in_progress():
            events |= ws_event.PMIC_CHANGE
        ws_event.notify_events(events)
        return result
    def do_push_to(self, patch=None, force=False, merge=False):
        in_charge = self.get_in_progress()
        if merge:
            cmd = 'hg -y qpush -m'
        else:
            cmd = 'hg qpush'
        if patch is None:
            if force:
                result = _run_cmd_on_console('%s -f' % cmd, ignore_err_re=_qpush_re)
            else:
                result = _run_cmd_on_console(cmd, ignore_err_re=_qpush_re)
        elif patch is "":
            if force:
                result = _run_cmd_on_console('%s -f -a' % cmd, ignore_err_re=_qpush_re)
            else:
                result = _run_cmd_on_console('%s -a' % cmd, ignore_err_re=_qpush_re)
        else:
            if force:
                result = _run_cmd_on_console('%s -f %s' % (cmd, patch), ignore_err_re=_qpush_re)
            else:
                result = _run_cmd_on_console('%s %s' % (cmd, patch), ignore_err_re=_qpush_re)
        if cmd_result.is_less_than_error(result[0]):
            events = ws_event.PATCH_PUSH
            if not in_charge:
                events |= ws_event.PMIC_CHANGE
            ws_event.notify_events(events)
        return result
    def do_refresh(self, patch=None, force=False, notify=True):
        cmd = "hg qrefresh"
        if force:
            return cmd_result.Result(cmd_result.ERROR, "", "mq has no concept of forced refresh")
        if patch is not None and patch != self.get_top_patch():
            return cmd_result.Result(cmd_result.ERROR, "", "mq will not refresh non top patches")
        result = _run_cmd_on_console(cmd)
        if notify:
            ws_event.notify_events(ws_event.PATCH_REFRESH)
        return result
    def do_remove_files_from_patch(self, filelist, patch=None):
        top = self.get_top_patch()
        if patch and patch != top:
            return cmd_result.Result(cmd_result.ERROR, "", "mq will not modify non top patches")
        if not top:
            return cmd_result.Result(cmd_result.ERROR, "", "No patches applied")
        res, prev, serr = self._patch_most_recent_parent_rev(top)
        if res != cmd_result.OK:
            return cmd_result.Result(res, prev, serr)
        files = " ".join(filelist)
        console.LOG.exec_cmd_with_alert("hg qrefresh " + files)
        cmd = " ".join(["hg revert --rev", prev, files])
        res, sout, serr = _run_cmd_on_console(cmd)
        ws_event.notify_events(ws_event.FILE_MOD|ws_event.FILE_ADD|ws_event.FILE_DEL)
        return cmd_result.Result(res, sout, serr)
    def do_rename_patch(self, patch, newname):
        cmd = "hg qrename "
        if patch is not None:
            cmd += patch + " "
        cmd += newname
        res, sout, serr = _run_cmd_on_console(cmd)
        ws_event.notify_events(ws_event.PATCH_CREATE|ws_event.PATCH_DELETE)
        if res != 0:
            return cmd_result.Result(cmd_result.ERROR, sout, serr)
        else:
            return cmd_result.Result(cmd_result.OK, sout, serr)
    def do_revert_files_in_patch(self, filelist, patch=None):
        top = self.get_top_patch()
        if patch and patch != top:
            return cmd_result.Result(cmd_result.ERROR, "", "mq will not modify non top patches")
        if not top:
            return cmd_result.Result(cmd_result.ERROR, "", "No patches applied")
        cmd = " ".join(["hg revert", " ".join(filelist)])
        res, sout, serr = _run_cmd_on_console(cmd)
        ws_event.notify_events(ws_event.FILE_MOD|ws_event.FILE_ADD|ws_event.FILE_DEL)
        return cmd_result.Result(res, sout, serr)
    def do_select_guards(self, guards):
        if not guards:
            guards = "--none"
        result = _run_cmd_on_console('hg qselect %s' % guards)
        ws_event.notify_events(ws_event.PATCH_MODIFY)
        return result
    def do_set_patch_guards(self, patch_name, guards):
        cmd = 'hg qguard '
        if not guards:
            cmd += "--none %s" % patch_name
        else:
            cmd += "-- %s %s" % (patch_name, guards)
        result = _run_cmd_on_console(cmd)
        ws_event.notify_events(ws_event.PATCH_MODIFY)
        return result
    def extdiff_and_full_patch_ok(self):
        return _hg_command_avail("extdiff")
    def get_all_patches_data(self):
        output = []
        res, sout, serr = utils.run_cmd('hg qguard -l')
        patch_plus_guard_list = sout.splitlines()
        applied_patches = self.get_applied_patches()
        if len(applied_patches) > 0:
            top_patch = applied_patches[-1]
            for ppg in patch_plus_guard_list:
                name, guards = _extract_name_and_guards(ppg)
                if name in applied_patches:
                    if name == top_patch:
                        state = const.TOP_PATCH
                    else:
                        state = const.APPLIED
                else:
                    state = const.NOT_APPLIED
                output.append(gquilt_tool.PatchData(name, state, guards))
        else:
            for ppg in patch_plus_guard_list:
                name, guards = _extract_name_and_guards(ppg)
                output.append(gquilt_tool.PatchData(name, const.NOT_APPLIED, guards))
        return output
    def get_applied_patches(self):
        res, sout, err = utils.run_cmd('hg qapplied')
        if res != 0:
            return []
        return sout.splitlines()
    def get_author_name_and_email(self):
        res, uiusername, serr = utils.run_cmd("hg showconfig ui.username")
        if res == 0 and uiusername:
            return uiusername.strip()
        else:
            return gquilt_tool.Interface.get_author_name_and_email(self)
    def get_combined_diff(self, start_patch=None, end_patch=None):
        if self._no_patches_applied():
            return cmd_result.Result(cmd_result.ERROR, "", "No patches applied")
        cmd = "hg diff --rev "
        if start_patch is None:
            res, prev, serr = self._patch_most_recent_parent_rev("qbase")
        else:
            res, prev, serr = self._patch_most_recent_parent_rev(start_patch)
        if res != cmd_result.OK:
            return cmd_result.Result(res, prev, serr)
        cmd += prev
        if end_patch is not None and end_patch != self.get_top_patch():
            cmd += " -rev " + end_patch
        res, sout, serr = utils.run_cmd(cmd)
        if res != 0:
            return cmd_result.Result(cmd_result.ERROR, sout, serr)
        return cmd_result.Result(res, sout, serr)
    def get_description_is_finish_ready(self, patch):
        pfn = self.get_patch_file_name(patch)
        res, pf_descr_lines = putils.get_patch_descr_lines(pfn)
        pf_descr = os.linesep.join(pf_descr_lines).strip()
        if not pf_descr:
            return False
        res, rep_descr, sout = utils.run_cmd('hg log --template "{desc}" --rev %s' % patch)
        if pf_descr != rep_descr.strip():
            top = self.get_top_patch()
            if top == patch:
                utils.run_cmd('hg qrefresh')
            else:
                # let's hope the user doesn't mind having the top patch refreshed
                utils.run_cmd('hg qrefresh')
                utils.run_cmd('hg qgoto %s' % patch)
                utils.run_cmd('hg qrefresh')
                utils.run_cmd('hg qgoto %s' % top)
        return True
    def get_diff(self, filelist=list(), patch=None):
        if not patch or patch == self.get_top_patch():
            cmd = "hg qdiff"
        elif not self.is_patch_applied(patch):
            res, diff = putils.get_patch_diff_lines(self.get_patch_file_name(patch))
            if res:
                return cmd_result.Result(cmd_result.OK, os.linesep.join(diff), [])
            return cmd_result.Result(cmd_result.ERROR, "", "Patch %s is not applied." % patch)
        else:
            res, prev, serr = self._patch_most_recent_parent_rev(patch)
            if res != cmd_result.OK:
                raise MQError(serr)
            cmd = "hg diff -r " + prev + " -r " + patch
        res, sout, serr = utils.run_cmd(" ".join([cmd, " ".join(filelist)]))
        if res != 0:
            return cmd_result.Result(cmd_result.ERROR, sout, serr)
        return cmd_result.Result(res, sout, serr)
    def get_diff_for_files(self, file_list=None, patch=None):
        if patch:
            parent = self.get_parent(patch)
            if not parent:
                # the patch is not applied
                pfn = self.get_patch_file_name(patch)
                result, diff = putils.get_patch_diff(pfn, file_list)
                if result:
                    return cmd_result.Result(cmd_result.OK, diff, '')
                else:
                    return cmd_result.Result(cmd_result.WARNING, '', diff)
        else:
            top = self.get_top_patch()
            if top:
                parent = self.get_parent(top)
            else:
                return cmd_result.Result(cmd_result.OK, '', '')
        cmd = 'hg diff --rev %s' % parent
        if patch:
            cmd += ' --rev %s' % patch
        if file_list:
            cmd += ' %s' % utils.file_list_to_string(file_list)
        res, sout, serr = utils.run_cmd(cmd)
        if res != 0:
            res = cmd_result.ERROR
        return cmd_result.Result(res, sout, serr)
    def get_in_progress(self):
        return self.get_top_patch() is not ''
    def get_next_patch(self):
        res, sout, _serr = utils.run_cmd("hg qnext")
        if res == 0:
            return sout.strip()
        else:
            return ""
    def get_base_patch(self):
        res, sout, serr = utils.run_cmd('hg qapplied')
        if res or not sout:
            return None
        else:
            return sout.splitlines()[0]
    def get_parent(self, patch):
        parent = 'qparent'
        for applied_patch in self.get_applied_patches():
            if patch == applied_patch:
                return parent
            else:
                parent = applied_patch
        return None
    def get_patch_file_db(self, patch=None):
        if patch and not self.is_patch_applied(patch):
            pfn = _patch_file_name(patch)
            return putils.get_patch_file_db(pfn, PatchStatusMap)
        top = self.get_top_patch()
        if not top:
            # either we're not in an mq playground or no patches are applied
            return ScmFileDb([])
        cmd = 'hg status -mardC'
        if patch:
            parent = self.get_parent(patch)
            cmd += ' --rev %s --rev %s' % (parent, patch)
        else:
            parent = self.get_parent(top)
            cmd += ' --rev %s' % parent
        res, sout, serr = utils.run_cmd(cmd)
        return ScmFileDb(sout.splitlines())
    def get_patch_file_name(self, patch):
        return _patch_file_name(patch)
    def get_patch_files(self, patch=None, withstatus=True):
        if self._no_patches_applied():
            return cmd_result.Result(cmd_result.OK, "", "")
        cmd = "hg status -m -a -d"
        if not withstatus:
            cmd += " -n"
        top = self.get_top_patch()
        if patch is None:
            patch = top
        res, prev, serr = self._patch_most_recent_parent_rev(patch)
        if res != cmd_result.OK:
            if patch and not self.is_patch_applied(patch):
                patchfile = self.get_patch_file_name(patch)
                is_ok, filelist = putils.get_patch_files(patchfile, withstatus)
                if is_ok:
                    return cmd_result.Result(cmd_result.OK, filelist, "")
                else:
                    return cmd_result.Result(cmd_result.ERROR, "", filelist)
            return cmd_result.Result(res, prev, serr)
        cmd += " --rev " + prev
        if patch != top:
            cmd += " --rev " + patch
        res, sout, serr = utils.run_cmd(cmd)
        if res != 0:
            return cmd_result.Result(cmd_result.ERROR, sout, serr)
        if withstatus:
            filelist = []
            for line in sout.splitlines():
                fobj = line[2:]
                if line[0] == "A":
                    filelist.append((fobj, const.ADDED))
                elif line[0] == "!":
                    filelist.append((fobj, const.DELETED))
                else:
                    filelist.append((fobj, const.EXTANT))
        else:
            filelist = sout.splitlines()
        return cmd_result.Result(res, filelist, serr)
    def get_patch_guards(self, patch):
        res, sout, err = utils.run_cmd('hg qguard ' + patch)
        if res != 0:
            return []
        name, guards = _extract_name_and_guards(sout)
        return guards
    def get_playground_root(self, fdir=None):
        if fdir:
            old_dir = os.getcwd()
            os.chdir(fdir)
        res, root, serr = utils.run_cmd("hg root")
        if fdir:
            os.chdir(old_dir)
        if res == 0:
            return root.strip()
        else:
            return None
    def get_selected_guards(self):
        res, sout, err = utils.run_cmd('hg qselect')
        if res != 0 or sout.strip() == "no active guards":
            return []
        return sout.split()
    def get_top_patch(self):
        res, sout, serr = utils.run_cmd("hg qtop")
        if res == 0:
            return sout.strip()
        else:
            return ""
    def get_ws_file_db(self):
        cmd = 'hg status -AC'
        qprev = self._get_qparent()
        if qprev is not None:
            cmd += ' --rev %s' % qprev
        res, sout, serr = utils.run_cmd(cmd)
        scm_file_db = ScmFileDb(sout.splitlines(), self._unresolved_file_list())
        scm_file_db.decorate_dirs()
        return scm_file_db
    def has_add_files(self):
        return False
    def has_finish_patch(self):
        return True
    def has_guards(self):
        # override in back end if the tool supports patch guards
        return True
    def has_refresh_non_top(self):
        return False
    def is_available(self):
        return _hg_command_avail("mq")
    def is_patch_applied(self, patch):
        res, sout, _serr = utils.run_cmd("hg qapplied")
        return res == 0 and patch in sout.splitlines()
    def is_playground(self, fdir=None):
        root = self.get_playground_root(fdir)
        if not root:
            return False
        pdir = _patch_dir(root)
        return os.path.exists(pdir) and os.path.isdir(pdir)
    def last_patch_in_series(self):
        res, sout, serr = utils.run_cmd("hg qseries")
        if res != 0:
            raise MQError(serr)
        return sout.splitlines()[-1]
    def new_playground(self, fdir=None):
        if os.path.exists(_patch_dir(fdir)):
            return cmd_result.Result(cmd_result.OK, "", "")
        if not os.path.exists(_hg_dir(fdir)):
            cmd = "hg init"
            if fdir is not None:
                cmd += " " + fdir
            res, sout, serr = utils.run_cmd(cmd)
            if res != cmd_result.OK:
                return cmd_result.Result(res, sout, serr)
        cmd = "hg qinit"
        if fdir is not None:
            olddir = os.getcwd()
            os.chdir(fdir)
        res, sout, serr = utils.run_cmd(cmd)
        if fdir is not None:
            os.chdir(olddir)
        return cmd_result.Result(res, sout, serr)
    def requires(self):
        return "\"mercurial\" with \"mq\" extension enabled <http://www.selenic.com/mercurial/>"
