#!PERL5
# Copyright (c) 1999                            RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#------------------------------------------------------------------------------
# Module Header
# Filename          :PathUtils.pm  
# Purpose           :PM to deal with different aspects of path manipulation
# Author            :Timur I. Bakeyev <timur@ripe.net>
# Date              :19990909
# Description       :Module deals with file path related things
# Language Version  :Perl 5.05003
# OSs Tested        :BSDI 3.1
# Command Line      : 
# Input Files       : NONE
# Output Files      : NONE   
# External Programs : 
# Problems          : 
# To Do             : 
# Comments          :
# Version           : $Id: PathUtils.pm,v 1.3 1999/10/13 13:45:14 timur Exp $
#------------------------------------------------------------------------------

package PathUtils;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;
require DynaLoader;

@ISA = qw(Exporter DynaLoader);
@EXPORT = qw(&cwd);
@EXPORT_OK = qw(&GetMountPoint &RealPath &AbsolutePath &SubPath &CanonDir);

$VERSION = '1.00';

bootstrap PathUtils $VERSION;

# Preloaded methods go here.
sub AbsolutePath
{
my($path) = @_;

my $rpath; # Real path

# Convert passed path to real path
if($rpath=RealPath($path))
    {
    return $rpath; # Just return the result
    }
# if passed path is absolute, we can try to do some magic
# stuff with mount pointer...
elsif($path =~ m%^/%)
    {
    my $mount; # Mount point name
    
    # Let's try to get mount point
    if($mount=GetMountPoint($path))
        {
        my $head;  #
        my $tail;  #
        
        do {
    	    # Compare pieces of mount point path with given path
    	    # If path is a part of mount point, then ve complete
    	    # the path by prepending unmatched prefix to the path
	    $head .= $1 if defined($1);
	    $tail = substr($mount, pos($mount)) if(pos($mount));
	    # If tail not empty and a complete match
	    if($tail && index($path, $tail) == 0)
	        {
	        $rpath = ${head} if defined($head);
	        $rpath .= ${path};
	        return $rpath;
	        }
	    } while($mount =~ m%(/?[^/]+)%g);
        # We failed to find a match, so just return original path
        return $path;
        }
    }
else # Relative path
    {
    if($rpath=cwd())
	{
	$rpath .= "/$path";
	return $rpath;
	}
    }
# We were unable to resolve path..
return $path;
}

sub SubPath
{
my($string, $substring) = @_;

if($substring && $string =~ m%^${substring}/?/?(.*)%)
    {
    return $1;
    }
else
    {
    return undef;
    }
}


sub CanonDir
{
my($path) = @_;

$path =~ s%(.*)/?/?%$1/%;

return $path;
}


1;

__END__

=head1 NAME

PathUtils - Perl extension for path manipulations

=head1 SYNOPSIS

  use PathUtils;

  $dir = cwd();
  
  $path = RealPath("../home");
  
  $mount_point = GetMountPoint(".");
  
  $relative = SubPath("/home/user/bin", "/home");
  
  $dir = CanonDir(cwd());


=head1 DESCRIPTION


The cwd() function just returns current working directory and corresponds
to system getcwd(3) function. In case of error returned value is underfined.

The RealPath() function is an interface to realpath(3) and calculates absolute
pathname from a given one, resolving all symbolic links, extra ``/'' characters 
and references to /./ and /../ in pathname. It'll fail, if one of the functions
chdir(2), close(2),  fchdir(2),  lstat(2),  open(2),  readlink(2) and getcwd(3),
that were called during resolving the path, failed.

GetMountPoint() takes arbitary existing path and returns it's mount point.
Function will fail if underlaying statfs(2) fails.

The AbsolutePath() function tries in several ways to determine real path of the 
filename. At first, it tries RealPath() to get it. If it failes then in case
of absolute path we try to get mount point and compare pieces of the path with
filename. If parts of the path are equal, we add as a prefix unmatched part of 
the mount dir. If filename is relative, we try to get current working directory 
via cwd() and prepend it to the filename. If all that methods failed, we return 
unmodified pathname. So, that function doesn't have any indication on bug..

Function SubPath() subtracts prefix from the path. If prefix isn't a part of the 
path, then function returns undef.

CanonDir() accepts directory name as parameter and returns it with the '/' at the 
end.

=head1 AUTHOR

Timur I. Bakeyev, RIPE NCC, <timur@ripe.net>

=head1 SEE ALSO

perl(1).

=cut
