/*
 * $Id: net.c,v 1.14 2006/04/20 11:20:55 lorenzo Exp $
 * 
 * Copyright (C) 2006 RIPE NCC
 * 
 * Original developer: Lorenzo Colitti <lorenzo@ripe.net>
 * Contributor(s):
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <errno.h>

#include <pthread.h>

#include "pinger.h"

#define BUFSIZE 1500

/** A pre-recorded DNS query. */
const char dnsquery[] = {
	0x01, 0x00, /* standard query */
	0x00, 0x01, /* 1 question */
	0x00, 0x00, /* 0 answer */
	0x00, 0x00, /* 0 authority */
	0x00, 0x00, /* 0 additional */
	0x00,       /* . */
	0x00, 0x02, /* NS */
	0x00, 0x01  /* IN */
};

/**
 * Sees if this is a response to one of our packets
 * @param packet IP packet to examine
 * @param len length of the IP packet in bytes
 * @param id the ICMP ID to match
 * @return a pointer to the ICMP header in the packet
 */
struct icmp *filterpacket(char *packet, int len, uint16_t id) {
	struct iphdr *iphdr = (struct iphdr *) packet;
	struct icmp *icmp;
	int hdrlen;

	if(!packet || !len)
		return NULL;

	if(len < sizeof(struct iphdr)) {
		fprintf(stderr, "Packet length (%d) < IP header!\n", hdrlen);
		return NULL;
	}

	/* Work out where ICMP header is */
	hdrlen = (iphdr->ihl * 4);
	icmp = (struct icmp *) (packet + hdrlen);

	/* Check length is exactly what we sent */
	if(len != hdrlen + sizeof(struct icmp) + sizeof(struct payload))
		return NULL;

	/* Check ID */
	if(ntohs(icmp->icmp_id) != id)
		return NULL;

	/* Check type */
	if( (icmp->icmp_type != ICMP_ECHOREPLY) || (icmp->icmp_code != 0) )
		return NULL;

	return icmp;
}

/**
 * Processes ICMP response
 * @param icmp a pointer to the ICMP header in the packet
 * @param reply (out) a |struct ping_reply| to fill in with the reply contents
 * @return a pointer to the target the ping was sent to
 */
struct target *processpacket(char *packet, struct icmp *icmp, struct ping_reply *reply, struct targetlist targetlist) {
	struct timeval tv, response_time;
	struct payload *payload = (struct payload *) (icmp + 1);
	int targetid;
	struct target *target;
	uint16_t seq;
	struct iphdr *iph;

	gettimeofday(&tv, NULL);

	targetid = payload->targetid;
	if( (targetid < 0) || (targetid >= targetlist.len) ) {
		fprintf(stderr, "Invalid target ID %d\n", targetid);
		return NULL;
	} else {
		target = targetlist.targets + targetid;
	}

	iph = (struct iphdr *) packet;
	if(memcmp(&iph->saddr, &target->addr.sin_addr, sizeof(iph->saddr))) {
		fprintf(stderr, "IP address mismatch: sent to %s, received from %s\n",
				inet_ntoa(target->addr.sin_addr),
				inet_ntoa(* (struct in_addr *) &iph->saddr));
		pthread_mutex_unlock(&target->lock);
		return NULL;
	}

	/* Work around broken implementations that mess up the sequence
	   number (e.g. 222.151.20.107, which responds with a seemingly random
	   seq if seq = 0) */
	pthread_mutex_lock(&target->lock);
	seq = ntohs(icmp->icmp_seq);
	if(seq > target->nsent) {
		fprintf(stderr, "Wrong icmp_seq %d from %s (sent %d)\n", seq, ip2str((struct sockaddr *) &target->addr), target->nsent);
		pthread_mutex_unlock(&target->lock);
		return NULL;
	}

	/* Ignore packets which take too long to come back */
	response_time = tv_subtract(tv, payload->tv);
	if(response_time.tv_sec > MAX_RESPONSE_TIME) {
		fprintf(stderr, "Reply from %s after %.2f ms, ignoring\n",
				ip2str((struct sockaddr *) &target->addr),
				tv_to_ms(response_time));
		pthread_mutex_unlock(&target->lock);
		return NULL;
	}

	target->nreceived++;

	pthread_mutex_unlock(&target->lock);

	reply->type = icmp->icmp_type;
	reply->code = icmp->icmp_code;
	reply->seq = seq;
	reply->response_time = response_time;

	return target;
}

/**
 * Receives one ICMP packet from the specified socket
 * @param socket socket to receive from
 * @param reply (out) a |struct ping_reply| to fill in with the reply contents
 * @param id the ICMP ID to filter on
 * @return a pointer to the target the ping was sent to, or NULL if this packet
   was not sent by us
 */
struct target *recvpacket(int sock, struct ping_reply *reply, uint16_t id, struct targetlist targetlist) {
	char buf[BUFSIZE];
	int len;
	struct icmp *icmp;

	len = recv(sock, buf, sizeof(buf), 0);
	if(len == -1)
		die("recv");

	icmp = filterpacket(buf, len, id);
	if(! icmp)
		return NULL;

	return processpacket(buf, icmp, reply, targetlist);
}

int sendudp(int sock, struct target *target, uint16_t id) {
	char packet[sizeof(dnsquery) + 2];
	uint16_t trans_id = ntohs(id);

	* (uint16_t *) packet = trans_id;
	if(!memcpy(packet + 2, dnsquery, sizeof(dnsquery)))
		die("memcpy");

	return sendto(sock, packet, sizeof(packet), 0, (struct sockaddr *) &target->addr, sizeof(target->addr));
}

/**
 * Sends a ping packet to the specified target
 * @param s the socket to send the packet to
 * @param seq the sequence number of the packet
 * @param target the target to send the ping to
 * @param id the ICMP id to send
 * @return the number of bytes sent
 */
int sendping(int sock, struct target *target, uint16_t seq, uint16_t id) {
	static unsigned char buf[sizeof(struct icmp) + sizeof(struct payload)];
	static struct icmp *icmp = NULL;
	static struct payload *payload = NULL;
	static pthread_mutex_t packet_lock = PTHREAD_MUTEX_INITIALIZER;
	int ret;

	/* One at a time please! */
	pthread_mutex_lock(&packet_lock);

	/* Set common fields first time only */
	if(icmp == NULL) {
		icmp = (struct icmp *) buf;

		icmp->icmp_type = ICMP_ECHO;
		icmp->icmp_code = 0;

		payload = (struct payload *) (buf + sizeof(struct icmp));
	}

	/* Set payload */
	payload->targetid = target->id;
	gettimeofday(&payload->tv, NULL);

	/* Calculate checksum. We can't use getpid() for the id because this thread
	   might have a different PID than the main thread */
	icmp->icmp_seq = htons(seq);
	icmp->icmp_id = htons(id);
	icmp->icmp_cksum = 0;
	icmp->icmp_cksum = in_cksum((uint8_t *)icmp, sizeof(buf));

	/* Send it out */
	ret = sendto(sock, buf, sizeof(buf), 0, (struct sockaddr *) &target->addr, sizeof(target->addr));
	pthread_mutex_unlock(&packet_lock);
	return ret;
}

/**
 * Calculates an IP header checksum. Code more or less lifted from Stevens.
 * @param buf buffer to calculate checksum on
 * @param len length of buffer
 * @return the checksum
 */
uint16_t in_cksum(uint8_t *buf, int len) {
	int i;
	uint16_t *w;
	uint32_t sum = 0;
	uint8_t oddbyte;

	for(i = 0; i < len; i += 2) {
		w = (uint16_t *) (buf + i);
		sum += *w;
	}

	if(len % 2) {
		oddbyte = buf[len - 1];
		sum += oddbyte;
	}

	sum = (sum >> 16) + (sum & 0xffff);
	sum += (sum >> 16);

	return ~sum;
}

/**
 * Converts a socket address structure to a string.
 * @param sa a |struct sockaddr_storage| containing the address
 * @return the string representation of the string address, or NULL if the
   address is not valid. Note that this string is statically allocated. 
 */
char *ip2str(struct sockaddr *sa) {
	static char host[255];

	if(getnameinfo((struct sockaddr *) sa, sizeof(*sa), (char *)host, sizeof(host), NULL, 0, NI_NUMERICHOST)) {
		return NULL;
	} else {
		return host;
	}
}

/**
 * Converts an IP address to a socket structure
 * @param ip the string representation of the IP address to convert
 * @param sa (out) the socket address structure to fill in
 * @param salen the length of the socket address structure to fill in
 * @return 0 on success, nonzero on failure
 */
int parse_ip(char *ip, uint16_t port, struct sockaddr *sa, int salen) {
	struct addrinfo *ai = NULL;
	struct addrinfo hints = {
		AI_NUMERICHOST, AF_INET, 0, 0, 0, NULL, NULL, NULL
	};
	int ret;
	char portstr[6];

	if(! ip) {
		return EAI_NONAME;
	}

	if(port == 0) {
		hints.ai_socktype = SOCK_RAW;
		hints.ai_protocol = IPPROTO_ICMP;
		ret = getaddrinfo(ip, NULL, &hints, &ai);
	} else {
		hints.ai_socktype = SOCK_DGRAM;
		hints.ai_protocol = IPPROTO_UDP;
		snprintf(portstr, sizeof(portstr), "%hu", port);
		ret = getaddrinfo(ip, portstr, &hints, &ai);
	}

	if(ret) {
		/* Bad IP address */
		if(ai) freeaddrinfo(ai);
		return ret;
	}

	if(ai->ai_addrlen > salen) {
		freeaddrinfo(ai);
		return ENOSPC;
	}

	memcpy(sa, ai->ai_addr, ai->ai_addrlen);
	freeaddrinfo(ai);

	return 0;
}

/**
 * Binds a socket to a specific ICMP ID
 * @param sock the socket to bind
 * @param id the ICMP ID
 * @return 0 on success, -1 on failure
 */
int bindsocket(int sock, uint16_t id) {
	struct sockaddr_storage recvaddr;
	socklen_t len = sizeof recvaddr;
	sa_family_t af;
	int ret;

	ret = getsockname(sock, (struct sockaddr *) &recvaddr, &len);
	if(ret)
		return ret;

	af = recvaddr.ss_family;
	memset(&recvaddr, 0, sizeof(recvaddr));

	switch(af) {
		case AF_INET:
			((struct sockaddr_in *) &recvaddr)->sin_port = id;
			break;
		case AF_INET6:
			((struct sockaddr_in6 *) &recvaddr)->sin6_port = id;
			break;
		default:
			errno = EAFNOSUPPORT;
			return EAFNOSUPPORT;
	}

	return bind(sock, (struct sockaddr *) &recvaddr, sizeof(recvaddr));
}
