# Copyright (c) 1996, 1997    The TERENA Association
# Copyright (c) 1998, 1999     RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#------------------------------------------------------------------------------
# Filename          :   ExternalMisc.pm
# Purpose           :   Miscellaneous emailrelated functions
# Author            :   Lee Wilmot
# Date              :   990109
# Language Version  :   Perl5, version 5.003_07
# OSs Tested        :   BSD
# Command Line      :   Nothing executable from command line.
# External Programs :   $RobotConfig::REG_COM
# Comments          :   The functions take a point to a hash with a field corresponding
#                       to each field in the email header. In addition, a 'body' field
#                       is expected containing (surprise!) the body of the email message.
# WARNING           :   THESE ROUTINES ARE PROBABLY USED ALSO BY EXTERNAL PROGRAMS.
#                       THAT'S WHY THIS FILE IS CALLED 'ExternalMisc.pm'.
#------------------------------------------------------------------------------

######################## PACKAGE INTERFACE #############################
#
# Please see the package concerned for descriptions of imported symbols.

package ExternalMisc;

use strict;

BEGIN {

    use vars qw( @ISA @EXPORT_OK );

    use Exporter ();

    @ISA = qw( Exporter );
    
    @EXPORT_OK = qw(
        &get_regids &get_ticketnumbers &get_message_id &get_email_format_date
        &remove_robot_addresses &removeRIPEAddresses
        &isContact &getRegInfo
    );
}

use RobotConfig qw(
    %F $REGID_REG $BAREREGID_REG $REG_COM $TICKNO_REG
    $DATE_COM $HOSTNAME_COM $UNIQUE_AFFIX
    $ROBOT_ADDRESS_REAL $ROBOT_ADDRESS_STANDALONE
    $DB_ROBOT_ADDRESS $INADDR_ROBOT_ADDRESS
    $DIST_ROBOT_ADDRESS
);

use Misc qw(
    &dprint
);

use RTT qw( 
    &ticket_exists
);


######################## PACKAGE INTERFACE #############################

#   Purpose         :   Look for and check regids in a message
#   In              :   $%: pointer to email header
#   Out             :   An array of valid regid's or undef
#   Comments        :   Expects a field called 'body', which is the rest of
#                       the message.
#                       Expects mail header field names to be as found in %F format,
#                       case senstive.
#                       Checks and returns them in the following order:
#                       - in x-ncc-regid field of mail header
#                       - in subject line with tag
#                       - in subject line without tag
#                       - in body with tag
#
sub get_regids {

    my $mailfields = shift @_;

    my @checkedRegids;

    ### Regid in email header field ###
    my $xnccRegid = checkRegid( $mailfields->{ $F{REGID} } );
    push @checkedRegids, $xnccRegid
       if ( defined $xnccRegid );

    ### Check in subject line if there is one ###

    my $subject = $mailfields->{ $F{EMAILSUBJECT} };

    if ( defined $subject ) {

        ### Regid in subject line with tag ###
        $subject =~ /$REGID_REG/i;

        my $subjectRegid = &checkRegid( $1 );

        push @checkedRegids, $subjectRegid
            if ( defined $subjectRegid );

        ### Bare regid in subject line ###

        $subject =~ /$BAREREGID_REG/i;

        my $subjectBareRegid = &checkRegid( $1 );

        push @checkedRegids, $subjectBareRegid
            if ( defined $subjectBareRegid );
    }

    ### Check for regid in email body ###

    my ( $isFax ) = 0;

	$isFax = ( $mailfields->{ $F{EMAILSUBJECT} } =~ /^FAX\s.*$/ )
		if ( defined $mailfields->{ $F{EMAILSUBJECT} } );

    # ---- get regid from body if not fax
    # will use the *first* one it finds
    my ( $firstBodyRegid );
    
    if ( !$isFax ) {
    
        # Mal: I changed this: doesnt have to be at start of line any more
        # The message could be a forwarded request.

        # Regid in body with tag: only the first one
        if ( $mailfields->{ $F{EMAILBODY} } =~ /$REGID_REG/i ) {

            my $bodyRegid = $1;
            $firstBodyRegid = &checkRegid( $bodyRegid );

            if ( defined $firstBodyRegid ) { 
                push @checkedRegids, $firstBodyRegid; 
            }
        }
    }

    return @checkedRegids;
}

#   Purpose         :   Check a regid is valid
#   In              :   $: regid to check
#   Out             :   $: the checked regid
#   Comments        :   
#
sub checkRegid {

    my ( $regid ) = @_;             # ARG: regid to check
    my ( $checkedRegid );           # RET: checked regid

    return undef 
        if ( ! defined $regid );

	$regid = lc $regid;

    my $escapedRegid = $regid;
    $escapedRegid =~ s/\./\\\./;    # used in match

    my $goodMatch = "^regid:\\s*($escapedRegid).*\$";

    my $checkCommand = "$REG_COM $regid 2>&1";

    if ( $regid && open ( CHECKREGID, "$checkCommand |" ) ) {

        while ( <CHECKREGID> ) {

            if ( $_ =~ /$goodMatch/ ) {         # exists
                 $checkedRegid = $1;
                 last;
            }
            elsif ( $_ =~ /no\smatch/ ) {
                last;
            }
        }

        close CHECKREGID;
    }

    return $checkedRegid;
}

#   Purpose         :   Look for and check existence of ticket numbers in a message
#   In              :   $%: pointer to email header
#   Out             :   An array of existing ticket number or undef
#   Comments        :   
#                       Expects a field called 'body', which is the rest of
#                       the message.
#                       Expects mail header field names to be as found in %F format,
#                       case sensitive.
#                       Checks and returns them in the following order:
#                       - in x-ncc-ticket field of mail header
#                       - in subject line of mail header
#                       - in body WITH TAG
#
sub get_ticketnumbers {

    my $mailfields = shift @_;

    my @existTicknos;

    my $subject = $mailfields->{$F{EMAILSUBJECT}} || "";
    my $ticket_field =  $mailfields->{$F{TICKET}} || "";
    my $body = $mailfields->{ $F{EMAILBODY} } || "";

    # Lee 19980805
    # Added line below. Some email software apparently corrupts
    # the # in NCC#00000000 to '=23'

    $subject =~ s/=23/#/;
    
    ### Check for tickno in explicit header field ###
    if ( $ticket_field =~ /($TICKNO_REG)/i ) {
    
        my $xnccTickno = $1;

        push @existTicknos, $xnccTickno
            if ( &ticket_exists($xnccTickno) );
    }

    ### Check for tickno in subject line ###        
    if ( $subject =~ /($TICKNO_REG)/i ) {

        my $subjectTickno = $1;

        push @existTicknos, $subjectTickno
            if ( &ticket_exists($subjectTickno) );
    }

    ### Check body if not got one in header ###

    my ( $isFax ) = ( $subject =~ /^FAX\s.*$/ );

    if ( @existTicknos == 0 && !$isFax ) {

        my $ticket_tag_reg = $F{TICKET};

        ### Tickno in body with tag: only the first one ###
        if ( $body =~ /$ticket_tag_reg:\s*($TICKNO_REG)/i ) {

            my $bodyTickno = $1;

            push @existTicknos, $bodyTickno
                if ( &ticket_exists($bodyTickno) );
        }
    }

    return @existTicknos;
}

#   Purpose :   Return the current date in a format suitable for 
#               inclusion in an email header
#   In      :   VOID
#   Out     :   $:  The date, or undef on error.
#
sub get_email_format_date {

    my ( $lines_found, $date_output ) = ( 0, "" );

    return undef
        if ( ! ( open DATE, "$DATE_COM 2>&1 |" ) );

    while (<DATE>) {

        $date_output .= $_;
        chop $date_output;

	&dprint( $date_output );

        $lines_found++;
    }

    close DATE;

    return undef
        if ( $? || $lines_found != 1 );
    
    return $date_output;
}

#   Purpose :   Return a nice unique string suitable for an email 
#               message id.
#   In      :   VOID
#   Out     :   $:  The message ID, or undef upon error.
#
sub get_message_id {

    my ( $lines_found, $hostname_output ) = ( 0, "" );

    return undef
        if ( ! ( open HOSTNAME, "$HOSTNAME_COM 2>&1 |" ) );

    while (<HOSTNAME>) {

        $hostname_output .= $_;
        chop $hostname_output;

        $lines_found++;
    }

    close HOSTNAME;

    return undef
        if ( $? || $lines_found != 1 );

    return join("", "<", time, "-", $UNIQUE_AFFIX, "\@", $hostname_output, ">");
}

#   Purpose :   Remove robot emails from a string
#   In      :   $$: a string to check
#   Out     :   VOID
#   Comments:   Potential problem with this particular replacement: 
#               the form "ADDRESS_X, ROBOT_ADDRESS" leads to "ADDRESS_X,"
#               as result, but mailer seems to handle this OK.
#
sub remove_robot_addresses {

    my $addresses = shift @_;

    # There are 4 robots: ip-request, distribution, db, inaddr
        
    $$addresses =~ s/[^,]*$ROBOT_ADDRESS_REAL[^,]*,{0,1}//ig;
    $$addresses =~ s/[^,]*$ROBOT_ADDRESS_STANDALONE[^,]*,{0,1}//ig;

    $$addresses =~ s/[^,]*$DIST_ROBOT_ADDRESS[^,]*,{0,1}//ig;
    $$addresses =~ s/[^,]*$DB_ROBOT_ADDRESS[^,]*,{0,1}//ig;
    $$addresses =~ s/[^,]*$INADDR_ROBOT_ADDRESS[^,]*,{0,1}//ig;
}

# Purpose           : Remove ripe.net addresses from a line
# Side Effects      :
# Comments          : ignores "*[\.@]ripe.net$" and "*[\.@]ripe.net\>$"
#                   : Credits and debits to Mal 19980507
sub removeRIPEAddresses {

    my ( $addressLine ) = @_;       # ARG cc line of mail
    my ( $fixedAddressLine );       # RET as above with ripe.net
    # addresses removed
    
    # might contain newlines
    $addressLine =~ s/\n//g;
    
    # make array of all addresses in line
    my ( @addresses ) = split ',', $addressLine;

    # array of checked addresses
    my @fixedAddresses;
    
    my $address;

    foreach $address ( @addresses ) {

        # If RIPE, ignore it

        if ( ( $address =~ /[\.@]ripe\.net\s*$/ )
            ||  ( $address =~ /[\.@]ripe\.net\s*\>\s*$/ ) ) {
            dprint( "ignoring RIPE address in Cc: \"$address\"\n" );
        }

        #otherwise, keep it

        else {
            
            push @fixedAddresses, $address;
        }
    }
    
    $fixedAddressLine = join ',', @fixedAddresses;
    
    return( $fixedAddressLine );
}


#------------------------------------------------------------------------------
# Purpose           : get registry service level and contacts
# Side Effects      :
# Comments          : this routine was adapted from Mal's routine
#                     in autohmdist.pl, so that it could be used
#                     by the robot too. It's main purpose here is
#                     to get the contacts for a registry, so that
#                     the robot can check whether the mail came
#                     from a registered contact.
#                     Lee 19980714
sub getRegInfo {

	my ( $regid ) = @_;			# ARGS: regid to read
	my ( $serviceLevel, @contacts );	# RET service level and contacts of registry

	# is correspondent listed as contact for reg ?
	if ( defined $regid ) {

		# command to get reg details
		my $lowerCaseRegid = lc $regid;
        	my $regCommand = "$REG_COM $lowerCaseRegid 2>&1 |";

                return undef
                    if ( ! ( open REGCALL, $regCommand ) );

        	# read reg output
        	my @regOutput;	# output of reg command
               	push @regOutput, $_
                    while ( <REGCALL> );

                close REGCALL;

        	return undef
                    if ( $? );

		# fields that we check 
		my @contactFields = ( $F{ADMINC}, $F{TECHC} );
		my $serviceLevelField = $F{REGFILESERVICELEVEL};

                # check each line of reg output

		my $line;
		foreach $line ( @regOutput ) {

			# contact line ?
			my $field; # label part of line
			foreach $field ( @contactFields ) {
				if ( $line =~ /^\s*$field:\s*(.*)\s*$/ ) {
					my $contact = $1; # value part of line
					push @contacts, $contact;
				}
			}

			# service level line ?
			if ( $line =~ /^\s*$serviceLevelField:\s*(\S+)\s+\((.*)\)/ ){
			        $serviceLevel = $1;
			        my $x_slhack = $2;
			} 
		}
	}

	return ( $serviceLevel, @contacts );	
}
	

#------------------------------------------------------------------------------
# Purpose           : is correspondent in reg's contacts list ?
# Side Effects      :
# Comments          : this routine was adapted from Mal's routine
#                     in autohmdist.pl, so that it could be used
#                     by the robot too. 
#                     Lee 19980714
#
sub isContact {

	my ( $correspondent, @contacts ) = @_;	# ARG - mail sender, array of contacts from reg
	my $isContact;				# RET - 1 or undef

	# get name part of email from/reply-to
	# crudely check contact names
        if ( $correspondent =~ /\"([^\"]*)\".*/ ) {
		$correspondent = $1;
	}
	elsif ( $correspondent =~ /^([^<]+)<[^>]+>$/ ) {
                $correspondent = $1;
        }

	# remove leading and trailing spaces from corresopndent
	$correspondent =~ s/^\s+//g;
	$correspondent =~ s/\s+$//g;

	# else use the whole thing

	# is correspondent listed as contact for reg ?
	if ( ( defined $correspondent ) && ( defined @contacts ) ) {
		# check each contact

		my $contact;
		foreach $contact ( @contacts ) {
			# match ?
			if ( lc ($contact) eq lc ($correspondent) ) {
				# matches
				$isContact = 1;
				last;
			}
			# else try removing nic handle
			elsif ( $contact =~ /^([^\(]*)\([^\)]*\).*$/ )	{
				$contact = $1;
				$contact =~ s/\s*$//;
				# case insensitive
                               	if ( lc ($contact) eq lc ($correspondent) ) {
					# matches
                                        $isContact = 1;
                                        last;
                                }
			}
		}
	}


	return $isContact;
}

1;
