package restservice

import (
	"context"
	"net/http"
	"testing"

	"github.com/stretchr/testify/require"
	dbmodel "isc.org/stork/server/database/model"
	dbtest "isc.org/stork/server/database/test"
	"isc.org/stork/server/gen/models"
	"isc.org/stork/server/gen/restapi/operations/users"
)

// Tests that create user account without necessary fields is rejected via REST API.
func TestCreateUserConflict(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, _ := NewRestAPI(dbSettings, db)

	// Both login and email missing here.
	su := dbmodel.SystemUser{
		Lastname: "Doe",
		Name:     "John",
	}

	// New user has empty email which conflicts with the default admin user empty email.
	params := users.CreateUserParams{
		Account: &models.UserAccount{
			User:     newRestUser(su),
			Password: models.Password("pass"),
		},
	}

	// Attempt to create the user and verify the response is negative.
	rsp := rapi.CreateUser(ctx, params)
	require.IsType(t, &users.CreateUserDefault{}, rsp)
	defaultRsp := rsp.(*users.CreateUserDefault)
	require.Equal(t, http.StatusConflict, getStatusCode(*defaultRsp))
	require.Equal(t, "User account with provided login/email already exists", *defaultRsp.Payload.Message)
}

// Tests that create user account with empty params is rejected via REST API.
func TestCreateUserEmptyParams(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, _ := NewRestAPI(dbSettings, db)

	// Try empty params - it should raise an error
	params := users.CreateUserParams{}
	rsp := rapi.CreateUser(ctx, params)
	require.IsType(t, &users.CreateUserDefault{}, rsp)
	defaultRsp := rsp.(*users.CreateUserDefault)
	require.Equal(t, http.StatusBadRequest, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to create new user account: missing data", *defaultRsp.Payload.Message)
}

// Tests that create user account with empty request is rejected via REST API.
func TestCreateUserEmptyRequest(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, _ := NewRestAPI(dbSettings, db)

	// Try empty request - it should raise an error
	params := users.CreateUserParams{
		Account: &models.UserAccount{},
	}
	rsp := rapi.CreateUser(ctx, params)
	require.IsType(t, &users.CreateUserDefault{}, rsp)
	defaultRsp := rsp.(*users.CreateUserDefault)
	require.Equal(t, http.StatusBadRequest, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to create new user account: missing data", *defaultRsp.Payload.Message)
}

// Tests that create user account with missing data is rejected via REST API.
func TestCreateUserMissingData(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, _ := NewRestAPI(dbSettings, db)

	// Try missing data - it should raise an error
	params := users.CreateUserParams{
		Account: &models.UserAccount{
			User: &models.User{},
		},
	}
	rsp := rapi.CreateUser(ctx, params)
	require.IsType(t, &users.CreateUserDefault{}, rsp)
	defaultRsp := rsp.(*users.CreateUserDefault)
	require.Equal(t, http.StatusBadRequest, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to create new user account: missing data", *defaultRsp.Payload.Message)
}

// Tests that user account can be created via REST API.
func TestCreateUser(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, _ := NewRestAPI(dbSettings, db)

	// Create the user and verify the response.
	su := dbmodel.SystemUser{
		Email:    "jb@example.org",
		Lastname: "Born",
		Login:    "jb",
		Name:     "John",
	}
	params := users.CreateUserParams{
		Account: &models.UserAccount{
			User:     newRestUser(su),
			Password: models.Password("pass"),
		},
	}

	rsp := rapi.CreateUser(ctx, params)
	require.IsType(t, &users.CreateUserOK{}, rsp)
	okRsp := rsp.(*users.CreateUserOK)
	require.Greater(t, *okRsp.Payload.ID, int64(0))
	require.Equal(t, *okRsp.Payload.Email, su.Email)
	require.Equal(t, *okRsp.Payload.Lastname, su.Lastname)
	require.Equal(t, *okRsp.Payload.Login, su.Login)
	require.Equal(t, *okRsp.Payload.Name, su.Name)

	// Also check that the user is indeed in the database.
	returned, err := dbmodel.GetUserByID(db, int(*okRsp.Payload.ID))
	require.NoError(t, err)
	require.Equal(t, su.Login, returned.Login)

	// An attempt to create the same user should fail with HTTP error 409.
	rsp = rapi.CreateUser(ctx, params)
	require.IsType(t, &users.CreateUserDefault{}, rsp)
	defaultRsp := rsp.(*users.CreateUserDefault)
	require.Equal(t, 409, getStatusCode(*defaultRsp))
}

// Tests that delete user account with empty params is rejected via REST API.
func TestDeleteUserEmptyParams(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create session manager.
	ctx, err = rapi.SessionManager.Load(ctx, "")
	require.NoError(t, err)

	// Create new super-admin user in the database.
	su := dbmodel.SystemUser{
		Email:    "john@example.org",
		Lastname: "White",
		Name:     "John",
		Password: "pass",
		Groups: []*dbmodel.SystemGroup{
			{
				ID: dbmodel.SuperAdminGroupID,
			},
		},
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	err = rapi.SessionManager.LoginHandler(ctx, &su)
	require.NoError(t, err)

	// Try empty params - it should raise an error
	params := users.DeleteUserParams{}
	rsp := rapi.DeleteUser(ctx, params)
	require.IsType(t, &users.DeleteUserDefault{}, rsp)
	defaultRsp := rsp.(*users.DeleteUserDefault)
	require.Equal(t, http.StatusNotFound, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to find user with ID 0 in the database", *defaultRsp.Payload.Message)
}

// Tests that delete user account with invalid user ID is rejected via REST API.
func TestDeleteUserInvalidUserID(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create session manager.
	ctx, err = rapi.SessionManager.Load(ctx, "")
	require.NoError(t, err)

	// Create new super-admin user in the database.
	su := dbmodel.SystemUser{
		Email:    "john@example.org",
		Lastname: "White",
		Name:     "John",
		Password: "pass",
		Groups: []*dbmodel.SystemGroup{
			{
				ID: dbmodel.SuperAdminGroupID,
			},
		},
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	err = rapi.SessionManager.LoginHandler(ctx, &su)
	require.NoError(t, err)

	// Try using invalid user ID - it should raise an error
	params := users.DeleteUserParams{
		ID: int64(su.ID + 1),
	}

	rsp := rapi.DeleteUser(ctx, params)
	require.IsType(t, &users.DeleteUserDefault{}, rsp)
	defaultRsp := rsp.(*users.DeleteUserDefault)
	require.Equal(t, http.StatusNotFound, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to find user with ID 3 in the database", *defaultRsp.Payload.Message)
}

// Tests that delete user account with same user ID as current session is rejected via REST API.
func TestDeleteUserSameUserAsSession(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create session manager.
	ctx, err = rapi.SessionManager.Load(ctx, "")
	require.NoError(t, err)

	// Create new super-admin user in the database.
	su := dbmodel.SystemUser{
		Email:    "john@example.org",
		Lastname: "White",
		Name:     "John",
		Password: "pass",
		Groups: []*dbmodel.SystemGroup{
			{
				ID: dbmodel.SuperAdminGroupID,
			},
		},
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	err = rapi.SessionManager.LoginHandler(ctx, &su)
	require.NoError(t, err)

	// Try same user ID as current session - it should raise an error - trying to delete itself
	params := users.DeleteUserParams{
		ID: int64(su.ID),
	}

	rsp := rapi.DeleteUser(ctx, params)
	require.IsType(t, &users.DeleteUserDefault{}, rsp)
	defaultRsp := rsp.(*users.DeleteUserDefault)
	require.Equal(t, http.StatusBadRequest, getStatusCode(*defaultRsp))
	require.Equal(t, "User account with provided login/email tries to delete itself", *defaultRsp.Payload.Message)
}

// Tests that user account can be deleted via REST API.
func TestDeleteUser(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create session manager.
	ctx, err = rapi.SessionManager.Load(ctx, "")
	require.NoError(t, err)

	// Create new super-admin user in the database.
	su := dbmodel.SystemUser{
		Email:    "john@example.org",
		Lastname: "White",
		Name:     "John",
		Password: "pass",
		Groups: []*dbmodel.SystemGroup{
			{
				ID: dbmodel.SuperAdminGroupID,
			},
		},
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	// Create new user in the database.
	su2 := dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err = dbmodel.CreateUser(db, &su2)
	require.False(t, con)
	require.NoError(t, err)

	err = rapi.SessionManager.LoginHandler(ctx, &su)
	require.NoError(t, err)
	params := users.DeleteUserParams{
		ID: int64(su2.ID),
	}

	rsp := rapi.DeleteUser(ctx, params)
	require.IsType(t, &users.DeleteUserOK{}, rsp)
	okRsp := rsp.(*users.DeleteUserOK)
	require.Greater(t, *okRsp.Payload.ID, int64(0))
	require.Equal(t, *okRsp.Payload.Email, su2.Email)
	require.Equal(t, *okRsp.Payload.Lastname, su2.Lastname)
	require.Equal(t, *okRsp.Payload.Login, su2.Login)
	require.Equal(t, *okRsp.Payload.Name, su2.Name)

	// Also check that the user is indeed not in the database.
	returned, err := dbmodel.GetUserByID(db, int(*okRsp.Payload.ID))
	require.NoError(t, err)
	require.Nil(t, returned)

	// An attempt to delete the same user should fail with HTTP error 409.
	rsp = rapi.DeleteUser(ctx, params)
	require.IsType(t, &users.DeleteUserDefault{}, rsp)
	defaultRsp := rsp.(*users.DeleteUserDefault)
	require.Equal(t, http.StatusNotFound, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to find user with ID 3 in the database", *defaultRsp.Payload.Message)
}

// Tests that super-admin user account can be deleted via REST API.
func TestDeleteUserInGroup(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create session manager.
	ctx, err = rapi.SessionManager.Load(ctx, "")
	require.NoError(t, err)

	// Create new super-admin user in the database.
	su := dbmodel.SystemUser{
		Email:    "john@example.org",
		Lastname: "White",
		Name:     "John",
		Password: "pass",
		Groups: []*dbmodel.SystemGroup{
			{
				ID: dbmodel.SuperAdminGroupID,
			},
		},
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	// Create new user in the database.
	su2 := dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
		Groups: []*dbmodel.SystemGroup{
			{
				ID: dbmodel.SuperAdminGroupID,
			},
		},
	}
	con, err = dbmodel.CreateUser(db, &su2)
	require.False(t, con)
	require.NoError(t, err)

	err = rapi.SessionManager.LoginHandler(ctx, &su)
	require.NoError(t, err)

	params := users.DeleteUserParams{
		ID: int64(su2.ID),
	}

	rsp := rapi.DeleteUser(ctx, params)
	require.IsType(t, &users.DeleteUserOK{}, rsp)
	okRsp := rsp.(*users.DeleteUserOK)
	require.Greater(t, *okRsp.Payload.ID, int64(0))
	require.Equal(t, *okRsp.Payload.Email, su2.Email)
	require.Equal(t, *okRsp.Payload.Lastname, su2.Lastname)
	require.Equal(t, *okRsp.Payload.Login, su2.Login)
	require.Equal(t, *okRsp.Payload.Name, su2.Name)

	// Also check that the user is indeed not in the database.
	returned, err := dbmodel.GetUserByID(db, int(*okRsp.Payload.ID))
	require.NoError(t, err)
	require.Nil(t, returned)

	// An attempt to delete the same user should fail with HTTP error 409.
	rsp = rapi.DeleteUser(ctx, params)
	require.IsType(t, &users.DeleteUserDefault{}, rsp)
	defaultRsp := rsp.(*users.DeleteUserDefault)
	require.Equal(t, http.StatusNotFound, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to find user with ID 3 in the database", *defaultRsp.Payload.Message)
}

// Tests that update user account with empty params is rejected via REST API.
func TestUpdateUserEmptyParams(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create new user in the database.
	su := dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	// Try empty params - it should raise an error
	params := users.UpdateUserParams{}
	rsp := rapi.UpdateUser(ctx, params)
	require.IsType(t, &users.UpdateUserDefault{}, rsp)
	defaultRsp := rsp.(*users.UpdateUserDefault)
	require.Equal(t, http.StatusBadRequest, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to update user account: missing data", *defaultRsp.Payload.Message)
}

// Tests that update user account with empty request is rejected via REST API.
func TestUpdateUserEmptyRequest(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create new user in the database.
	su := dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	// Try empty request - it should raise an error
	params := users.UpdateUserParams{
		Account: &models.UserAccount{},
	}
	rsp := rapi.UpdateUser(ctx, params)
	require.IsType(t, &users.UpdateUserDefault{}, rsp)
	defaultRsp := rsp.(*users.UpdateUserDefault)
	require.Equal(t, http.StatusBadRequest, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to update user account: missing data", *defaultRsp.Payload.Message)
}

// Tests that update user account with missing data is rejected via REST API.
func TestUpdateUserMissingData(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create new user in the database.
	su := dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	// Try missing data - it should raise an error
	params := users.UpdateUserParams{
		Account: &models.UserAccount{
			User: &models.User{},
		},
	}
	rsp := rapi.UpdateUser(ctx, params)
	require.IsType(t, &users.UpdateUserDefault{}, rsp)
	defaultRsp := rsp.(*users.UpdateUserDefault)
	require.Equal(t, http.StatusBadRequest, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to update user account: missing data", *defaultRsp.Payload.Message)
}

// Tests that update user account with invalid user ID is rejected via REST API.
func TestUpdateUserInvalidUserID(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create new user in the database.
	su := dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	// An attempt to update non-existing user (non macthing ID) should
	// result in an error 409.
	su.ID = 123
	params := users.UpdateUserParams{
		Account: &models.UserAccount{
			User:     newRestUser(su),
			Password: models.Password("pass"),
		},
	}
	rsp := rapi.UpdateUser(ctx, params)
	require.IsType(t, &users.UpdateUserDefault{}, rsp)
	defaultRsp := rsp.(*users.UpdateUserDefault)
	require.Equal(t, 409, getStatusCode(*defaultRsp))
}

// Tests that user account can be updated via REST API.
func TestUpdateUser(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()

	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create new user in the database.
	su := dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, &su)
	require.False(t, con)
	require.NoError(t, err)

	// Modify some values and update the user.
	su.Lastname = "Born"
	params := users.UpdateUserParams{
		Account: &models.UserAccount{
			User:     newRestUser(su),
			Password: models.Password("pass"),
		},
	}
	rsp := rapi.UpdateUser(ctx, params)
	require.IsType(t, &users.UpdateUserOK{}, rsp)

	// Also check that the user has been updated in the database.
	returned, err := dbmodel.GetUserByID(db, su.ID)
	require.NoError(t, err)
	require.Equal(t, su.Lastname, returned.Lastname)
}

// Tests that update user password with missing data is rejected via REST API.
func TestUpdateUserPasswordMissingData(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create new user in the database.
	user := &dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, user)
	require.False(t, con)
	require.NoError(t, err)

	// Try empty request - it should raise an error.
	params := users.UpdateUserPasswordParams{
		ID: int64(user.ID),
	}
	rsp := rapi.UpdateUserPassword(ctx, params)
	require.IsType(t, &users.UpdateUserPasswordDefault{}, rsp)
	defaultRsp := rsp.(*users.UpdateUserPasswordDefault)
	require.Equal(t, http.StatusBadRequest, getStatusCode(*defaultRsp))
	require.Equal(t, "Failed to update password for user: missing data", *defaultRsp.Payload.Message)
}

// Tests that user password can be updated via REST API.
func TestUpdateUserPassword(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create new user in the database.
	user := &dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, user)
	require.False(t, con)
	require.NoError(t, err)

	// Update user password via the API.
	params := users.UpdateUserPasswordParams{
		ID: int64(user.ID),
		Passwords: &models.PasswordChange{
			Newpassword: models.Password("updated"),
			Oldpassword: models.Password("pass"),
		},
	}
	rsp := rapi.UpdateUserPassword(ctx, params)
	require.IsType(t, &users.UpdateUserPasswordOK{}, rsp)

	// An attempt to update the password again should result in
	// HTTP error 400 (Bad Request) because the old password is
	// not matching anymore.
	params = users.UpdateUserPasswordParams{
		ID: int64(user.ID),
		Passwords: &models.PasswordChange{
			Newpassword: models.Password("updated"),
			Oldpassword: models.Password("pass"),
		},
	}
	rsp = rapi.UpdateUserPassword(ctx, params)
	require.IsType(t, &users.UpdateUserPasswordDefault{}, rsp)
	defaultRsp := rsp.(*users.UpdateUserPasswordDefault)
	require.Equal(t, http.StatusBadRequest, getStatusCode(*defaultRsp))
}

// Tests that multiple groups can be fetched from the database.
func TestGetGroups(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, _ := NewRestAPI(dbSettings, db)

	params := users.GetGroupsParams{}

	rsp := rapi.GetGroups(ctx, params)
	require.IsType(t, &users.GetGroupsOK{}, rsp)
	rspOK := rsp.(*users.GetGroupsOK)

	groups := rspOK.Payload
	require.NotNil(t, groups.Items)
	require.GreaterOrEqual(t, 2, len(groups.Items))
}

// Tests that user information can be retrieved via REST API.
func TestGetUsers(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create new user in the database.
	user := &dbmodel.SystemUser{
		Email:    "jd@example.org",
		Lastname: "Doe",
		Login:    "johndoe",
		Name:     "John",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, user)
	require.False(t, con)
	require.NoError(t, err)

	// Try empty request - it should work
	params := users.GetUsersParams{}

	rsp := rapi.GetUsers(ctx, params)
	require.IsType(t, &users.GetUsersOK{}, rsp)
	okRsp := rsp.(*users.GetUsersOK)
	require.EqualValues(t, 2, okRsp.Payload.Total) // we expect 2 users (admin and john doe)
	require.Len(t, okRsp.Payload.Items, 2)         // make sure there's entry with 2 users

	// Retrieve all users using the API
	var (
		limit int64 = 99999
		start int64
		text  string
	)

	params = users.GetUsersParams{
		Limit: &limit,
		Start: &start,
		Text:  &text,
	}

	rsp = rapi.GetUsers(ctx, params)
	require.IsType(t, &users.GetUsersOK{}, rsp)
	okRsp = rsp.(*users.GetUsersOK)
	require.EqualValues(t, 2, okRsp.Payload.Total) // we expect 2 users (admin and john doe)
	require.Len(t, okRsp.Payload.Items, 2)         // make sure there's entry with 2 users

	// Check the default user (admin)
	require.Equal(t, "admin", *okRsp.Payload.Items[0].Login)
	require.Equal(t, "admin", *okRsp.Payload.Items[0].Name)
	require.Equal(t, "admin", *okRsp.Payload.Items[0].Lastname)
	require.Equal(t, "", *okRsp.Payload.Items[0].Email)

	// Check the user we just added
	require.Equal(t, "johndoe", *okRsp.Payload.Items[1].Login)
	require.Equal(t, "John", *okRsp.Payload.Items[1].Name)
	require.Equal(t, "Doe", *okRsp.Payload.Items[1].Lastname)
	require.Equal(t, "jd@example.org", *okRsp.Payload.Items[1].Email)

	// Make sure the ID of the new user is different.
	// TODO: implement new test that add 100 users and verifies uniqueness of their IDs
	require.NotEqual(t, *okRsp.Payload.Items[0].ID, *okRsp.Payload.Items[1].ID)
}

// Tests that user information can be retrieved via REST API.
func TestGetUser(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, err := NewRestAPI(dbSettings, db)
	require.NoError(t, err)

	// Create new user in the database.
	user := &dbmodel.SystemUser{
		Email:    "jd@example.org",
		Lastname: "Doe",
		Login:    "johndoe",
		Name:     "John",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, user)
	require.False(t, con)
	require.NoError(t, err)

	// Retrieve the user with ID=2 using the API (that's the new user user we just added)
	var id int64 = 2
	params := users.GetUserParams{
		ID: id,
	}

	rsp := rapi.GetUser(ctx, params)
	require.IsType(t, &users.GetUserOK{}, rsp)
	okRsp := rsp.(*users.GetUserOK)
	require.Equal(t, user.Email, *okRsp.Payload.Email) // we expect 2 users (admin and john doe)
	require.Equal(t, id, *okRsp.Payload.ID)            // user ID
	require.Equal(t, user.Login, *okRsp.Payload.Login)
	require.Equal(t, user.Name, *okRsp.Payload.Name)
	require.Equal(t, user.Lastname, *okRsp.Payload.Lastname)

	// TODO: check that the new user belongs to a group
	// require.Len(t, okRsp.Payload.Groups, 1)
}

// Tests that new session can not be created using empty params.
func TestCreateSessionEmptyParams(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, _ := NewRestAPI(dbSettings, db)

	user := &dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, user)
	require.False(t, con)
	require.NoError(t, err)

	// try empty request - it should raise an error
	params := users.CreateSessionParams{}
	rsp := rapi.CreateSession(ctx, params)
	require.IsType(t, &users.CreateSessionBadRequest{}, rsp)
}

// Tests that new session can not be created using invalid credentials.
func TestCreateSessionInvalidCredentials(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, _ := NewRestAPI(dbSettings, db)

	user := &dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, user)
	require.False(t, con)
	require.NoError(t, err)

	// for next tests there is some data required in the context, let's prepare it
	ctx2, err := rapi.SessionManager.Load(ctx, "")
	require.NoError(t, err)

	params := users.CreateSessionParams{}
	// provide wrong credentials - it should raise an error
	params.Credentials.Useremail = &user.Email
	bad := "bad pass"
	params.Credentials.Userpassword = &bad
	rsp := rapi.CreateSession(ctx2, params)
	require.IsType(t, &users.CreateSessionBadRequest{}, rsp)
}

// Tests that new session can be created for a logged user.
func TestCreateSession(t *testing.T) {
	db, dbSettings, teardown := dbtest.SetupDatabaseTestCase(t)
	defer teardown()

	ctx := context.Background()
	rapi, _ := NewRestAPI(dbSettings, db)

	user := &dbmodel.SystemUser{
		Email:    "jan@example.org",
		Lastname: "Kowalski",
		Name:     "Jan",
		Password: "pass",
	}
	con, err := dbmodel.CreateUser(db, user)
	require.False(t, con)
	require.NoError(t, err)

	// for next tests there is some data required in the context, let's prepare it
	ctx2, err := rapi.SessionManager.Load(ctx, "")
	require.NoError(t, err)

	params := users.CreateSessionParams{}
	// provide correct credentials - it should create a new session
	params.Credentials.Useremail = &user.Email
	params.Credentials.Userpassword = &user.Password
	rsp := rapi.CreateSession(ctx2, params)
	require.IsType(t, &users.CreateSessionOK{}, rsp)
	okRsp := rsp.(*users.CreateSessionOK)
	require.Greater(t, *okRsp.Payload.ID, int64(0))

	delParams := users.DeleteSessionParams{}
	rsp = rapi.DeleteSession(ctx2, delParams)
	require.IsType(t, &users.DeleteSessionOK{}, rsp)
}
