package dbsession

import (
	"context"
	"database/sql"
	"net/http"
	"strconv"
	"strings"

	"github.com/alexedwards/scs/postgresstore"
	"github.com/alexedwards/scs/v2"

	// TODO: document why it is blank imported.
	_ "github.com/lib/pq"
	"github.com/pkg/errors"
	dbops "isc.org/stork/server/database"
	dbmodel "isc.org/stork/server/database/model"
)

// Provides session management mechanisms for Stork. It wraps the scs.SessionManager
// structure with Stork specific implementation of sessions.
type SessionMgr struct {
	scsSessionMgr *scs.SessionManager
}

// Creates new session manager instance. The new connection is created using the
// lib/pq driver via scs.SessionManager.
func NewSessionMgr(conn *dbops.BaseDatabaseSettings) (*SessionMgr, error) {
	connParams := conn.ConnectionParams()
	db, err := sql.Open("postgres", connParams)
	if err != nil {
		return nil, errors.Wrapf(err, "error connecting to the database for session management using credentials %s", connParams)
	}

	s := scs.New()
	s.Store = postgresstore.New(db)

	mgr := &SessionMgr{scsSessionMgr: s}

	return mgr, nil
}

// This function should be invoked upon successful authentication of the user which
// is logging in to the system. It renews or creates a new session token for the user.
// The user's login and identifier are stored in the session.
func (s *SessionMgr) LoginHandler(ctx context.Context, user *dbmodel.SystemUser) error {
	err := s.scsSessionMgr.RenewToken(ctx)
	if err != nil {
		return errors.Wrapf(err, "error while creating new session identifier")
	}

	s.scsSessionMgr.Put(ctx, "userID", user.ID)
	s.scsSessionMgr.Put(ctx, "userLogin", user.Login)
	s.scsSessionMgr.Put(ctx, "userEmail", user.Email)
	s.scsSessionMgr.Put(ctx, "userLastname", user.Lastname)
	s.scsSessionMgr.Put(ctx, "userName", user.Name)

	// If any user groups are associated with the user, store them
	// as a list of comma separated values.
	if len(user.Groups) > 0 {
		var groups string
		for i, g := range user.Groups {
			if i > 0 {
				groups += ","
			}
			groups += strconv.Itoa(g.ID)
		}
		s.scsSessionMgr.Put(ctx, "userGroupIds", groups)
	}

	return nil
}

// Destroys user session as a result of logout.
func (s *SessionMgr) LogoutHandler(ctx context.Context) error {
	err := s.scsSessionMgr.Destroy(ctx)
	if err != nil {
		return errors.Wrapf(err, "error while destroying a user session")
	}

	return nil
}

// Implements middleware which reads the session cookie, loads session data for the
// user and stores the token/ in the Cookie being sent to the user.
func (s *SessionMgr) SessionMiddleware(handler http.Handler) http.Handler {
	return s.scsSessionMgr.LoadAndSave(handler)
}

// Checks if the given session token exists in the database. This is typically used
// in unit testing to validate that the session data is persisted in the database.
func (s *SessionMgr) HasToken(token string) bool {
	_, exists, _ := s.scsSessionMgr.Store.Find(token)
	return exists
}

// Checks if the user is logged to the system. It is assumed that the session data
// is already fetched from the database and is stored in the request context.
// The returned values are: ok - if the user is logged, user identifier and user
// login.
func (s *SessionMgr) Logged(ctx context.Context) (ok bool, user *dbmodel.SystemUser) {
	id := s.scsSessionMgr.GetInt(ctx, "userID")
	// User has no session.
	if id == 0 {
		return false, nil
	}

	// User has a session.
	user = &dbmodel.SystemUser{ID: id}
	user.Login = s.scsSessionMgr.GetString(ctx, "userLogin")
	user.Email = s.scsSessionMgr.GetString(ctx, "userEmail")
	user.Lastname = s.scsSessionMgr.GetString(ctx, "userLastname")
	user.Name = s.scsSessionMgr.GetString(ctx, "userName")

	// Retrieve comma separated list of groups.
	userGroups := s.scsSessionMgr.GetString(ctx, "userGroupIds")

	if len(userGroups) > 0 {
		groups := strings.Split(userGroups, ",")
		for _, g := range groups {
			if id, err := strconv.Atoi(g); err == nil {
				user.Groups = append(user.Groups, &dbmodel.SystemGroup{ID: id})
			}
		}
	}

	return true, user
}

// This function is only for testing purposes to prepare request context.
func (s *SessionMgr) Load(ctx context.Context, token string) (context.Context, error) {
	ctx2, err := s.scsSessionMgr.Load(ctx, token)
	return ctx2, err
}
