/* loadtest.c 1.28.1 1995/07/19 18:31:09 davidsen Stable */
#include "inet.h"
#include <time.h>
#include <sys/times.h>
#include <string.h>
#include <signal.h>
#include <string.h>

/* EOS - end of string constant */
#define EOS             ((char) 0)
/* EOM - true if line is end of message */
#define EOM(str)        ((str)[0] == '.' && ((str)[1] == '\r' || (str)[1] == '\n'))
/* TRACE - print if verbose level N or greater */
#define TRACE(lvl,st)   if (lvl <= verbose) printf st
/* TRACE_TS - output trace with timestamp */
#define TRACE_TS(lvl,str) if (lvl <= verbose) Put_TS_Str(lvl, str)
/* typedef for article numbers */
typedef char ARTICLE[60];
/* retry socket open parameters */
#define MAX_SOCKET_RETRIES  5
#define OPEN_RETRY_TIME     3

/* some approximate sizes - dynamic in 1.30 and beyond */
#define GSIZE				80
#define NGROUPS				(20*1024)

/* load control values */
int NumToRead       = 30;       /* -n   number of articles per reader       */
int ThinkHead       = 0;        /* -T   time to scan the list of subjects   */
int ThinkRead       = 10;       /* -t   time to read on article             */
int Max_per_group   = 10;       /* -m   max arts in one groups              */
int NumProc         = 1;        /* -N   number of reader processes          */
int SpawnPause      = 3;        /* -p   pause between reader spawns         */
int Conserve        = 0;        /* -S   use socket conservation             */

/* some global values */
int TotalGroups;                /* set by runtime                           */
int sockfd;                     /* id of the open socket                    */
char MyPID[7] = "Master";       /* process ID (not pid)                     */
int verbose = 0;                /* flag for progress reporting              */
float ArtReadSec;               /* realtime to read articles                */
float TotalETsec;               /* total elapsed seconds                    */
float TotalCPUuser;             /* user CPU time                            */
float TotalCPUsystem;           /* system CPU time                          */
ARTICLE *ArtNums = NULL;        /* article numbers for read                 */
char IPaddress[16] =            /* server IP string                         */
    { SERV_HOST_ADDR };
time_t ARmin, ARmax;            /* article read min/max times               */
int CRcount = 0;                /* connect retry count                      */
int CRmax = 0;                  /* connect retry max                        */
    
/* room to store the group names */
static char *GroupNames[NGROUP];
static char  Groups[NGROUP * (GSIZE+1)];

/* last request send to the server */
char LastReq[MAXLINE];

/* socket i/o lines */
static char sendline[MAXLINE];
static char recvline[MAXLINE + 1];

/* line to build status messages */
static char statmsg[MAXLINE+1];

/* article reading counters */
int     ArtReadLines = 0;       /* total lines                              */
time_t  ArtReadTimes = 0;       /* total time to read (ticks)               */

/* prototypes */
static int BuildGNames(int);
static int putcs(const char *);
static int setgroup(int);
static int ReadOneArt(const char *);
static int ReadHeader(int);
static void ReadArts(void);
static void Usage(void);
static void TimeOut(void);
static void InitServer(void);
static void setalarm(int, const char *);
static void errexit(const char *, int);
static char *ArtRef(const char *);
static void Put_TS_Str(int, char *);

void 
main(int argc, char *argv[])
{
    int n, m;
    int wait4 = 0;
    int status;
    clock_t et1, et2;
    float et_sec;
    struct tms TimeDummy;
    struct hostent *HostInfo;

    /* initialize for any error messages */
    pname = argv[0];

    /* parse args */
    while ((n = getopt(argc, argv, "n:t:T:m:I:H:N:vp:S")) > 0) {
        switch (n) {
        case 'n':       /* articles to read, total      */
            NumToRead = atoi(optarg);
            break;
        case 't':       /* time to read one art         */
            ThinkRead = atoi(optarg);
            break;
        case 'T':       /* header think time            */
            ThinkHead = atoi(optarg);
            break;
        case 'm':       /* max arts/group               */
            Max_per_group = atoi(optarg);
            break;
        case 'I':       /* IP address                   */
            strcpy(IPaddress, optarg);
            break;
        case 'H':       /* host name */
            HostInfo = gethostbyname(optarg);
            if (HostInfo == NULL) {
                printf("Host %s not found\n", optarg);
                exit(1);
            }
            sprintf(IPaddress, "%d.%d.%d.%d",
                HostInfo->h_addr_list[0][0], HostInfo->h_addr_list[0][1],
                HostInfo->h_addr_list[0][2], HostInfo->h_addr_list[0][3]
            );
            break;
        case 'N':       /* # of readers                 */
            NumProc = atoi(optarg);
            break;
        case 'v':       /* be verbose                   */
            ++verbose;
            break;
        case 'p':       /* pause between forks()        */
            SpawnPause = atoi(optarg);
            if (SpawnPause < 1 || SpawnPause > 30) {
                errexit("-p val range 1..30", 1);
            }
            break;
        case 'S':       /* socket conservation on       */
            Conserve = 1;
            break;
        default:        /* something WRONG here...      */
            Usage();
            break;
        }
    }

    /* test for leftovers */
    if (optind < argc) {
        printf("[%s] unprocessed arguments (%d/%d)\n", MyPID, optind, argc);
        Usage();
    }

    /* give useful info */
    if (verbose) {
        printf(
            "NNTP Synthetic Load Generator v1.28.1 modified 1995/07/19 18:31:09\n"
            "%8d processes\n%8d articles/process\n"
            "%8d max articles in any one group\n"
            "%8d sec think time on titles (zero = 5 + 200ms/art)\n"
            "%8d sec to read each article\n"
            "Start a process every %d sec against server IP %s\n",
            NumProc, NumToRead, Max_per_group, ThinkHead, ThinkRead,
            SpawnPause, IPaddress
        );
        if (Conserve)
            printf("Reopen the socket for each article read\n"); else
            printf("One socket opened per process\n");
        fflush(stdout);
    }
    InitServer();

    /* build the list of groups */
    et1 = times(&TimeDummy);
    n = BuildGNames(sockfd);
    et1 = times(&TimeDummy) - et1;
    if (n < 0) {
        errexit("Can't build list of groups!", 1);
    }
    else
    {
        et_sec = (float)et1/CLK_TCK;
        printf("[%s] namebuild %6d groups in %6.3f sec (%3.0f g/s)\n",
            MyPID, n, et_sec, n/et_sec
        );
        TotalGroups = n;
    }

    /* get the date as seen by the server */
    if (putcs("date\r\n") < 0) {
        printf("[%s] Can't get date from server\n", MyPID);
        exit(1);
    }

    /* output the server's idea of the date */
    setalarm(5, "Server didn't respond to DATE");
    n = readline(sockfd, recvline, MAXLINE);
    alarm(0);
    recvline[n] = EOS;
    if (atoi(recvline) != 111) {
        printf("[%s] Can't get date from server\n", MyPID);
        exit(1);
    }
    sscanf(recvline+4, "%[0-9]*", sendline);
    sprintf(statmsg, "Server start date: %s\n", sendline);
    TRACE_TS(1, statmsg);

    /* here's where we really do the work */
    et1 = times(&TimeDummy);
    for (m = NumProc; m--; ) {
        if ((n = fork()) == 0) {
            /* light housekeeping to keep readers and parent separated */
            sprintf(MyPID, "S%05d", NumProc-m);
            wait4 = 0;
  
            /* gather wrapup statistics */
            et2 = times(&TimeDummy);
            ReadArts();
            TotalETsec = ((float)times(&TimeDummy) - et2)/CLK_TCK;
            TotalCPUuser = (TimeDummy.tms_utime + TimeDummy.tms_cutime)/(float)CLK_TCK;
            TotalCPUsystem = (TimeDummy.tms_stime + TimeDummy.tms_cstime)/(float)CLK_TCK;
            TRACE(1, (
                /* NOTE: multi-line format here... */
                "stats: %3d art  %6d lines  %8.3f ET(r)"
                "  %8.3f ET(t) %8.3fu %8.3fs\n"
                "min read %8.3f max read %8.3f %d retries, %d max\n",
                NumToRead, ArtReadLines,
                ArtReadSec, TotalETsec,
                TotalCPUuser, TotalCPUsystem,
                ARmin/(float)CLK_TCK, ARmax/(float)CLK_TCK,
                CRcount, CRmax
            ));
            TRACE_TS(1, "Process ends.\n");
            break;
        } else if (n > 0) {
            /* output trace and wait a bit to ease startup load */
            sprintf(statmsg, "Started process %5d\n", n);
            TRACE_TS(3, statmsg);
            ++wait4;
            sleep(SpawnPause);
        } else {
            sprintf(sendline, "[%s] fork() failed!", MyPID);
            perror(sendline);
            exit(1);
        }
    }

    /* wait for children if any */
    if (wait4) {
        for (m=wait4; m--; ) n = wait(&status);
        et_sec = ((float)times(&TimeDummy) - et1)/CLK_TCK;
        TRACE(1, ("[%s] %3d readers terminated, %7.3f sec\n", MyPID, wait4, et_sec));

        /* get the ending time */
        if (1 <= verbose) {
            if (putcs("date\r\n") < 0) {
                printf("[%s] can't send final DATE\n", MyPID);
                exit(1);
            }
            setalarm(5, "Can't read final date");
            n = readline(sockfd, recvline, MAXLINE);
            recvline[n] = EOS;
            alarm(0);
            sscanf(recvline, "%d %[0-9]", &n, sendline);
            if (n != 111) {
                printf("[%s] Can't read final DATE\n", MyPID);
                exit(1);
            }
            printf("[%s] server end date: %s\n", MyPID, sendline);
        }

        /* clean wrapup - send a nice quit */
        if ((n = putcs("quit\n")) < 0) {
            errexit("Can't send QUIT", 1);
        }
        n = readline(sockfd, recvline, MAXLINE);
        recvline[n] = EOS;
    }

    close(sockfd);

    exit(0);
}

/*****************************************************************
 |  BuildGNames - build list of group names
 |----------------------------------------------------------------
 |  saves the group names in the Groups array, with pointers in
 |  the GroupNames array.
 |
 |  Returns: # of groups or -1 for overflow
 ****************************************************************/

static int
BuildGNames(int sockfd)
{
    char *fillin = Groups;
    int n, ngroups = 0;

    /* go ask the server for some groups */
    n = putcs("list active\n");
    if (n < 1) {
        errexit("Can't ask for groups", 1);
    }

    /* wait for the reply saying it will happen */
    n = readline(sockfd, recvline, MAXLINE);
    recvline[n] = EOS;
    if (atoi(recvline) != 215) {
        sprintf(sendline,
            "LIST ACTIVE returned reject status:\n%12s%s", "", recvline);
        errexit(sendline, 1);
    }

    while (n = readline(sockfd, recvline, MAXLINE)) {
        recvline[n] = EOS;

        /* see if done */
        if (recvline[0] == '.') break;

        /* truncate the string at the first blank */
        *strchr(recvline, ' ') = EOS;

        /* if the array is full, stop trying */
        if (ngroups == NGROUP) {
            fprintf(stderr, "<W> too many groups > %d\n", NGROUP);
            return ngroups;
        }

        /* store the data in the array, save the pointer, bump the pointer */
        strcpy(fillin, recvline);
        GroupNames[ngroups++] = fillin;
        fillin += n;
    }

    return ngroups;
}

/*****************************************************************
 |  putcs - output a constant string to the socket
 |----------------------------------------------------------------
 |  Returns: chars sent
 ****************************************************************/

static int
putcs(const char *string)
{
    int n, len;

    /* save the command for future error messages */
    strcpy(LastReq, string);

    len = strlen(string);
    n = writen(sockfd, (char *)string, len);
    return (n == len ? n : -1);
}

/*****************************************************************
 |  ReadArts - read articles simulating delays
 |----------------------------------------------------------------
 |  Uses a separate process for each read, to allow seed
 ****************************************************************/

static void
ReadArts(void)
{
    unsigned short seeds[3];
    int count = NumToRead;
    int gcount = 0;
    int cur_group;
    int n;
    int avail = 0;

    /* seed the random generator */
    seeds[0] = 0;
    seeds[1] = getpid();
    seeds[2] = getppid();
    seed48(seeds);

    /* open the server just once if not conserving sockerts */
    if (!Conserve) InitServer();

    /* read articles */
    while (count) {
        /* see that some are available in this group */
        while (avail == 0) {
            avail = setgroup(cur_group = TotalGroups*drand48());
            TRACE(2,
                ("[%s] group %s, %d arts\n",
                MyPID, GroupNames[cur_group], avail)
            );
        }

        /* loop until you read one article */
        for (n = 0; n < avail && count; ++n) {
            if (ReadOneArt(ArtNums[n]) == 0) {
                --count;

                /* here's the user reading the article */
                if (ThinkRead) sleep(ThinkRead);
            }
        }
        avail = 0;
    }

    /* output the final results */
    ArtReadSec = (float)ArtReadTimes/CLK_TCK;

    /* close if still open */
    if (Conserve) close(sockfd);
}

/*****************************************************************
 |  ReadOneArt - read just one article
 |----------------------------------------------------------------
 |  asks for the head and then the body
 ****************************************************************/

static int
ReadOneArt(const char *artnum)
{
    int n;
    int rc = 0;

    /* timing stuff, line count */
    int nlines;
    time_t et;
    struct tms Dummy;

    /* start the clock */
    et = times(&Dummy);

    /* may have to do overhead */
    if (Conserve) InitServer();

    /* get the article */
    sprintf(sendline, "article %s\r\n", artnum);
    putcs(sendline);
    n = readline(sockfd, recvline, MAXLINE);
    recvline[n] = EOS;
    if ((n = atoi(recvline)) != 220) {
        /* didn't get data, check bad number */
        if (n == 430) {
            /* article not found, close socket, due to NOV? */
            TRACE(3, ("[%s] article %s not online\n", MyPID, artnum));
            if (Conserve) close(sockfd);
            return -1;
        }
        else
        {
            printf("[%s] article read failed (%s)\n\t%s\n",
                MyPID, sendline, recvline);
            exit(1);
        }
    }

    /* skip the text */
    do {
        n = readline(sockfd, recvline, MAXLINE);
        recvline[n] = 0;
        ++rc;
    } while (!EOM(recvline));
    ArtReadLines += rc - 1;

    /* stop the clock and add to the total */
    if (Conserve) close(sockfd);
    et = times(&Dummy) - et;
    ArtReadTimes += et;
    sprintf(statmsg, "got article, %d lines, %.3f sec\n",
        rc-1, (float)et/CLK_TCK);
    TRACE_TS(3, statmsg);
    if (et < ARmin || ARmin == 0) ARmin = et;
    if (et > ARmax) ARmax = et;

    return 0;
}

/*****************************************************************
 |  ReadHeader - read one header
 ****************************************************************/

static int
ReadHeader(int artnum)
{
    int n;
    int linecount = 0;

    /* ask for the article head */
    sprintf(sendline, "head %d\n", artnum);
    n = putcs(sendline);

    /* check read status */
    n = readline(sockfd, recvline, MAXLINE);
    recvline[n] = EOS;
    if ((n = atoi(recvline)) != 221) {
        /* bad status, see if invalid article */
        if (n == 423) return (-1);
        else {
            /* bogus answer, let's barf! */
            printf("[%s] <F> invalid response, not 221 or 423!\n\t%s",
                MyPID, recvline);
            printf("[%s] Last request: %s", MyPID, LastReq);
            exit(1);
        }
    }

    /* skip until end of text marker */
    strcpy(sendline, "Skipping header text lines");
    do {
        setalarm(10, sendline);
        n = readline(sockfd, recvline, MAXLINE);
        recvline[n] = EOS;
        ++linecount;
    } while (!EOM(recvline));
    alarm(0);

    return linecount;
}

/*****************************************************************
 |  setgroup - select a group
 |----------------------------------------------------------------
 |  Sets:       value of 1st available article
 |  Returns:    # of articles
 ****************************************************************/

static int
setgroup(int groupnum)
{
    int n, an;
    int status, numart, lo_art, hi_art;
    char *cur_gname;
    char *artstr;

    /* if the array id not allocated, do it now */
    if (ArtNums == NULL) {
        n = Max_per_group * sizeof(ARTICLE);
        ArtNums = (ARTICLE*) malloc(n);
        if (ArtNums == NULL) {
            printf("[%s] Can't allocate article list\n", MyPID);
            exit(1);
        }
    }

    /* open the server if needed */
    if (Conserve) InitServer();

    /* build the group command and do the set */
    cur_gname = GroupNames[groupnum];
    sprintf(sendline, "group %s\r\n", cur_gname);
    n = putcs(sendline);

    /* read and parse the response */
    n = readline(sockfd, recvline, MAXLINE);
    if (n < 0) errexit("Socket close while setting group", 1);
    recvline[n] = EOS;
    sscanf(recvline, "%d %d %d %d", &status, &numart, &lo_art, &hi_art);
    if (status != 211) {
        /* see if no group */
        if (status == 411) {
            sprintf(sendline, "No such group: %s", cur_gname);
            errexit(sendline, 1);
        } else {
            sprintf(sendline, "Unknown error: %s", recvline);
            errexit(sendline, 1);
        }
    }
    if (numart == 0) {
        /* close the socket and return zero count */
        if (Conserve) close(sockfd);
        return 0;
    }

    /* read the headers and wait for the user to think about them */
    sprintf(sendline, "xover  %d-%d\r\n", lo_art, hi_art);
    putcs(sendline);
    n = readline(sockfd, recvline, MAXLINE);
    recvline[n] = 0;
    if (atoi(recvline) != 224) {
        printf("[%s] didn't get data for xover command\n", MyPID);
        exit(1);
    }

    /* Now read some article id's */
    for (an = 0;;) {
        n = readline(sockfd, recvline, MAXLINE);
        recvline[n] = EOS;
        if (EOM(recvline)) break;
        artstr = ArtRef(recvline);
        if (an < Max_per_group && artstr != NULL)
            strcpy(ArtNums[an++], artstr);
    }

    /* "look at" the headers */
    if (Conserve) close(sockfd);
    sleep(ThinkHead ? ThinkHead : (5 + n/5));
    if (numart > Max_per_group) numart = Max_per_group;
    return numart;
}

/*****************************************************************
 |  Usage - show the command line options
 |----------------------------------------------------------------
 |  exits - no return
 ****************************************************************/

static void
Usage(void)
{
    puts("loadtest - NNTP synthetic load generator\n");
    puts("Syntax:");
    puts("  loadtest [ options ]");
    puts("Options:");
    puts("  -n   # of articles to read          def: 30");
    puts("  -t   time to read one art (sec)     def: 10");
    puts("  -m   max articles in one group,     def: 10");
    puts("  -T   time to read headers (sec)     def: 5+lines/5");
    puts("  -I   IP address of server (N.N.N.N) def: 127.0.0.1");
    puts("  -H   server host name (node.dom)    def: loopback");
    puts("  -N   # of readers                   def: 1");
    puts("  -v   verbose progress reports       def: off");
    puts("  -p   # sec between forks (1..30)    def: 3");
    puts("  -S   use socket conservation        def: off");

    exit(1);
}

/*****************************************************************
 |  Timeout handling on reads
 |----------------------------------------------------------------
 |  This code allows handling of timeouts for reads (or anything)
 |  by setting a timer and specifying the message to output if
 |  the timer times out.
 ****************************************************************/

static const char *TimeOutMsg;

static void
setalarm(int secs, const char *string)
{
    TimeOutMsg = string;
    signal(SIGALRM, (void(*)(int))TimeOut);
    alarm(secs);
}

static void
TimeOut(void)
{
    printf("[%s] Operation timeout: %s\n", MyPID, TimeOutMsg);
    printf("[%s] Last request: %s", MyPID, LastReq);
    exit(1);
}

/*****************************************************************
 |  InitServer - open a socket to the server, save the value
 |----------------------------------------------------------------
 |  exit if bad status
 ****************************************************************/

static void
InitServer(void)
{
    int n;
    int et;                     /* local et for socket open */
    struct tms Dummy;           /* dummy data structure to get clock */
    int RetryCount;
    struct sockaddr_in serv_addr;

    /* initialize for socket connection */
    bzero((char *) &serv_addr, sizeof(serv_addr));
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_addr.s_addr = inet_addr(IPaddress);
    serv_addr.sin_port = htons(SERV_TCP_PORT);

    for (RetryCount = 0; RetryCount < MAX_SOCKET_RETRIES; ++RetryCount ) {
        et = times(&Dummy);

        /* make the connect and error check */
        if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
            fprintf(stderr, "[%s] client: can't open stream socket\n", MyPID);
            exit(1);
        }

        if (connect(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0) {
            TRACE(3, ("[%s] client: can't connect to server\n", MyPID));
            close(sockfd);
            sleep(OPEN_RETRY_TIME);
            ++CRcount;
            continue;
        }
    
        /* read the HELO line from the server, print if found */
        n = readline(sockfd, recvline, MAXLINE);
        if (n < 0) {
            TRACE(3, ("[%s] Connection refused!\n", MyPID));
            close(sockfd);
            sleep(OPEN_RETRY_TIME);
            ++CRcount;
            continue;
        }
        /* connection is open */
        break;
    }

    /* see if any error */
    if (RetryCount) {
        /* see if we have given up */
        if (RetryCount == MAX_SOCKET_RETRIES) {
            sprintf(LastReq,
                "Can't connect after %d retries, give up!\n", RetryCount);
            TRACE_TS(0, LastReq);
            errexit(LastReq, 1);
        }
        else {
            /* see if retries current */
            if (RetryCount > CRmax) CRmax = RetryCount;
            sprintf(statmsg, "server open took %d retries\n", RetryCount);
            TRACE_TS(2, statmsg);
        }
    }
    et = times(&Dummy) - et;
    sprintf(statmsg, "socket open time %.3f sec.\n", et/(float)CLK_TCK);
    TRACE_TS(3, statmsg);

    recvline[n] = EOS;
    if (n == 0) {
        fprintf(stderr, "[%s] Empty HELO message read!\n", MyPID);
        exit(1);
    }
    if (atoi(recvline) != 200) {
        if (recvline[n-1] != '\n') {
            recvline[n] = '\n';
            recvline[++n] = EOS;
        }
        sprintf(statmsg, "HELO msg err, len %d\nWas: %.50s", n, recvline);
        TRACE_TS(0, statmsg);
        errexit("Bad HELO message", 1);
    }

    /* switch to reader service mode */
    if ((n = putcs("mode reader\r\n")) < 0) {
        errexit("Error sending mode switch!",1);
    } else {
        /* protect against loss of input */
        setalarm(20, "Wait for reply to \"mode reader\"");

        /* read the ack and display it */
        n = readline(sockfd, recvline, MAXLINE);
        alarm(0);
        if (n < 0) {
            errexit("No response to mode switch request!",1);
        }
        recvline[n] = EOS;
    }
}

/*****************************************************************
 |  SkipData - skip multiline data
 |----------------------------------------------------------------
 |  Checks that the first line matches the code of the argument,
 |  then reads lines until EOM
 ****************************************************************/

static int
SkipData(int TextType)
{
    int n;
    int linecount = -1;

    /* read the first line */
    sprintf(sendline, "Waiting for %d line", TextType);
    setalarm(10, sendline);
    n = readline(sockfd, recvline, MAXLINE);
    recvline[n] = EOS;
    alarm(0);

    /* check the type of the response */
    if (atoi(recvline) != TextType) {
        printf("[%s] <F> invalid response, expected %d!\n\t[%3d]=>%s\n",
            MyPID, TextType, strlen(recvline), recvline);
        printf("[%s] Last request: %s", MyPID, LastReq);
        exit(1);
    }

    /* skip until end of text marker */
    sprintf(sendline, "Skipping type %d text", TextType);
    do {
        setalarm(10, sendline);
        n = readline(sockfd, recvline, MAXLINE);
        recvline[n] = EOS;
        ++linecount;
    } while (!EOM(recvline));
    alarm(0);

    return linecount;
}

/*****************************************************************
 |  errexit - output a string and exit
 |----------------------------------------------------------------
 |  displays the string with process id, then exits status 1
 ****************************************************************/

static void
errexit(const char *emsg, int errstat)
{
    printf("[%s] <F> %s\n", MyPID, emsg);
    exit(errstat);
}

/*****************************************************************
 |  ArtRef - return the article reference string from an XOVER line
 |----------------------------------------------------------------
 |  Note: modifies the data in the buffer!
 ****************************************************************/

static char *
ArtRef(const char *str)
{
    int n;
    char *lastQ = NULL;
    static char ArtBuf[MAXLINE];

    /* do the sscanf */
    n = sscanf(str, "%*[^\t]\t%*[^\t]\t%*[^\t]\t%*[^\t]\t%[^>]", ArtBuf);
    if (n == 1) {
        strcat(ArtBuf, ">");
        lastQ = ArtBuf;
    }

    return lastQ;
}

/*****************************************************************
 | Put_TS_Str - output a trace message with timestamp
 ****************************************************************/

static void
Put_TS_Str(int lvl, char *string)
{
    char outbuf[2 * MAXLINE + 1];
    char *dest = outbuf, *dptr;
    char datestr[50];
    struct tm *ltime;
    time_t rightnow;
    struct tms Dummy;
    int ch;
    int needhdr = 1;            /* need a header for this line */
    float ftime;                /* fractional hours */

    /* build the datestring, once */
    rightnow = time((int *)NULL);
    ltime = localtime(&rightnow);
    ftime = (3600*ltime->tm_hour + 60*ltime->tm_min + ltime->tm_sec)/3600.0;
    sprintf(datestr, "[%s] %4d%03d%06.3f ", MyPID,
        ltime->tm_year + 1900, ltime->tm_yday, ftime
    );

    /* copy the input string, adding date for each newline */
    while (ch = *string++) {
        /* test for header needed */
        if (needhdr) {
            strcpy(dest, datestr);
            dest += strlen(datestr);
        }

        *dest++ = ch;
        needhdr = (ch == '\n');
    }

    /* now output the data */
    *dest = EOS;
    fputs(outbuf, stdout);
    fflush(stdout);

    /* if trace level is zero, fatal error, output to stderr, too */
    if (lvl == 0) fputs(outbuf, stderr);
}
