/*  $Revision: 1.15 $
**
**  Library routines to let other programs control innd.
*/
#include "configdata.h"
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#if	defined(DO_NEED_TIME)
#include <time.h>
#endif	/* defined(DO_NEED_TIME) */
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include "nntp.h"
#include "paths.h"
#include "libinn.h"
#include "clibrary.h"
#include "inndcomm.h"
#include "macros.h"

#define MIN_BUFFER_SIZE		4096

STATIC char			ICCsockname[sizeof _PATH_TEMPSOCK];
#if	defined(DO_HAVE_UNIX_DOMAIN)
STATIC struct sockaddr_un	ICCserv;
STATIC struct sockaddr_un	ICCclient;
#endif	/* defined(DO_HAVE_UNIX_DOMAIN) */
STATIC int			ICCfd;
STATIC int			ICCtimeout;


/*
**  Set the timeout.
*/
void
ICCsettimeout(i)
    int		i;
{
    ICCtimeout = i;
}


/*
**  Get ready to talk to the server.
*/
int
ICCopen()
{
    int		mask;

    /* Create a temporary name. */
    (void)strcpy(ICCsockname, _PATH_TEMPSOCK);
    (void)mktemp(ICCsockname);
    if (unlink(ICCsockname) < 0 && errno != ENOENT)
	return -1;

#if	defined(DO_HAVE_UNIX_DOMAIN)
    /* Make a socket and give it the name. */
    if ((ICCfd = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0)
	return -1;
    (void)memset((POINTER)&ICCclient, 0, sizeof ICCclient);
    ICCclient.sun_family = AF_UNIX;
    (void)strcpy(ICCclient.sun_path, ICCsockname);
    mask = umask(0);
    if (bind(ICCfd, (struct sockaddr *)&ICCclient,
		AF_UNIX_SOCKSIZE(ICCclient)) < 0) {
	(void)umask(mask);
	return -1;
    }
    (void)umask(mask);

    /* Name the server's socket. */
    (void)memset((POINTER)&ICCserv, 0, sizeof ICCserv);
    ICCserv.sun_family = AF_UNIX;
    (void)strcpy(ICCserv.sun_path, _PATH_NEWSCONTROL);
#else
    /* Make a named pipe and open it. */
    mask = umask(0);
    if (mkfifo(ICCsockname, 0666) < 0) {
	(void)umask(mask);
	return -1;
    }
    (void)umask(mask);
    if ((ICCfd = open(ICCsockname, O_RDWR)) < 0)
	return -1;
#endif	/* defined(DO_HAVE_UNIX_DOMAIN) */

    return 0;
}


/*
**  Close down.
*/
int
ICCclose()
{
    int		i;

    i = 0;
    if (close(ICCfd) < 0)
	i = -1;
    if (unlink(ICCsockname) < 0 && errno != ENOENT)
	i = -1;
    return i;
}


/*
**  See if the server is still there.  When in doubt, assume yes.
**  Cache the pid since a rebooted server won't know about our pending
**  message.
*/
STATIC BOOL
ICCserveralive()
{
    static PID_T	pid;
    FILE		*F;
    char		buff[SMBUF];

    if (pid == 0) {
	pid = 1;
	/* If we can't open the file, assume the server is there unless the
	 * file doesn't exist. */
	if ((F = fopen(_PATH_SERVERPID, "r")) == NULL)
	    return errno == ENOENT ? FALSE : TRUE;
	if (fgets(buff, sizeof buff, F) == NULL) {
	    (void)fclose(F);
	    return TRUE;
	}
	(void)fclose(F);
	pid = atoi(buff);
    }

    if (kill(pid, 0) > 0 || errno != ESRCH)
	return TRUE;
    return FALSE;
}


/*
**  Send an arbitrary command to the server.
*/
int
ICCcommand(cmd, argv, replyp)
    char		cmd;
    char		*argv[];
    char		**replyp;
{
    char		*buff;
    char		*p;
    char		*q;
    char		save;
    int			bufsiz;
    int			i;
#if	!defined(DO_HAVE_UNIX_DOMAIN)
    int			fd;
#endif	/* !defined(DO_HAVE_UNIX_DOMAIN) */
    int			len;
    FDSET		Rmask;
    struct timeval	T;

    /* Get the length of the buffer. */
    bufsiz = strlen(ICCsockname) + 1 + 1;
    for (i = 0; (p = argv[i]) != NULL; i++)
	bufsiz += 1 + strlen(p);
    if (bufsiz < MIN_BUFFER_SIZE)
	bufsiz = MIN_BUFFER_SIZE;
    buff = malloc((unsigned int)bufsiz);
    if (buff == NULL)
	return -1;
    if (replyp)
	*replyp = NULL;

    /* Format the message. */
    (void)sprintf(buff, "%s%c%c", ICCsockname, SC_SEP, cmd);
    for (p = buff + strlen(buff), i = 0; (q = argv[i]) != NULL; i++) {
	*p++ = SC_SEP;
	p += strlen(strcpy(p, q));
    }

    /* Send message. */
    len = p - buff;
#if	defined(DO_HAVE_UNIX_DOMAIN)
    if (sendto(ICCfd, buff, len, 0,
	    (struct sockaddr *)&ICCserv, AF_UNIX_SOCKSIZE(ICCserv)) < 0) {
	DISPOSE(buff);
	return -1;
    }
#else
    if ((fd = open(_PATH_NEWSCONTROL, O_WRONLY)) < 0) {
	DISPOSE(buff);
	return -1;
    }
    if (write(fd, buff, len) != len) {
	i = errno;
	DISPOSE(buff);
	(void)close(fd);
	errno = i;
	return -1;
    }
    (void)close(fd);
#endif	/* defined(DO_HAVE_UNIX_DOMAIN) */

    /* Possibly get a reply. */
    switch (cmd) {
    default:
	if (ICCtimeout >= 0)
	    break;
	/* FALLTHROUGH */
    case SC_SHUTDOWN:
    case SC_XABORT:
    case SC_XEXEC:
	DISPOSE(buff);
	return 0;
    }

    /* Wait for the reply. */
    for ( ; ; ) {
	FD_ZERO(&Rmask);
	FD_SET(ICCfd, &Rmask);
	T.tv_sec = ICCtimeout ? ICCtimeout : 120;
	T.tv_usec = 0;
	i = select(ICCfd + 1, &Rmask, (FDSET *)NULL, (FDSET *)NULL, &T);
	if (i < 0) {
	    DISPOSE(buff);
	    return -1;
	}
	if (i > 0 && FD_ISSET(ICCfd, &Rmask))
	    /* Server reply is there; go handle it. */
	    break;

	/* No data -- if we timed out, return. */
	if (ICCtimeout) {
	    DISPOSE(buff);
	    errno = ETIMEDOUT;
	    return -1;
	}

	if (!ICCserveralive()) {
	    DISPOSE(buff);
	    return -1;
	}
    }

    /* Read the reply. */
    i = RECVorREAD(ICCfd, buff, bufsiz - 1);
    if (i < 0) {
	DISPOSE(buff);
	return -1;
    }
    buff[i] = '\0';

    /* Parse the reply; expected to be like "<exitcode><space><text>" */
    i = 0;
    if (CTYPE(isdigit, buff[0])) {
	for (p = buff; *p && CTYPE(isdigit, *p); p++)
	    continue;
	if (*p) {
	    save = *p;
	    *p = '\0';
	    i = atoi(buff);
	    *p = save;
	}
    }
    if (replyp)
	*replyp = buff;
    else
	DISPOSE(buff);
    return i;
}


/*
**  Send a "cancel" command.
*/
int
ICCcancel(msgid)
    char	*msgid;
{
    char	*args[2];

    args[0] = msgid;
    args[1] = NULL;
    return ICCcommand(SC_CANCEL, args, (char **)NULL);
}


/*
**  Send a "go" command.
*/
int
ICCgo(why)
    char	*why;
{
    char	*args[2];

    args[0] = why;
    args[1] = NULL;
    return ICCcommand(SC_GO, args, (char **)NULL);
}


/*
**  Send a "pause" command.
*/
int
ICCpause(why)
    char	*why;
{
    char	*args[2];

    args[0] = why;
    args[1] = NULL;
    return ICCcommand(SC_PAUSE, args, (char **)NULL);
}


/*
**  Send a "reserve" command.
*/
int
ICCreserve(why)
    char	*why;
{
    char	*args[2];

    args[0] = why;
    args[1] = NULL;
    return ICCcommand(SC_RESERVE, args, (char **)NULL);
}
