/*
 *  Description:
 *      This file implements the Secure Hash Signature Standard
 *      algorithms as defined in the National Institute of Standards
 *      and Technology Federal Information Processing Standards
 *      Publication (FIPS PUB) 180-1 published on April 17, 1995, 180-2
 *      published on August 1, 2002, and the FIPS PUB 180-2 Change
 *      Notice published on February 28, 2004.
 *
 *      A combined document showing all algorithms is available at
 *              http://csrc.nist.gov/publications/fips/
 *              fips180-2/fips180-2withchangenotice.pdf
 *
 *      The SHA-1 algorithm produces a 160-bit message digest for a
 *      given data stream.  It should take about 2**n steps to find a
 *      message with the same digest as a given message and
 *      2**(n/2) to find any two messages with the same digest,
 *      when n is the digest size in bits.  Therefore, this
 *      algorithm can serve as a means of providing a
 *      "fingerprint" for a message.
 *
 *  Portability Issues:
 *      SHA-1 is defined in terms of 32-bit "words".  This code
 *      uses <stdint.h> (included via "sha.h") to define 32 and 8
 *      bit unsigned integer types.  If your C compiler does not
 *      support 32 bit unsigned integers, this code is not
 *      appropriate.
 *
 *  Caveats:
 *      SHA-1 is designed to work with messages less than 2^64 bits
 *      long. This implementation uses SHA1Input() to hash the bits
 *      that are a multiple of the size of an 8-bit character, and then
 *      uses SHA1FinalBits() to hash the final few bits of the input.
 *
 *  Authorship:
 *      This file is adapted from RFC 4634, by D. Eastlake et al.
 *      Copyright (C) The Internet Society (2006).
 *
 *      Permission is granted for all uses, commercial and non-commercial,
 *      of the sample code found in Section 8.  Royalty free license to
 *      use, copy, modify and distribute the software found in Section 8 is
 *      granted, provided that this document is identified in all material
 *      mentioning or referencing this software, and provided that
 *      redistributed derivative works do not contain misleading author or
 *      version information.
 *
 *      The authors make no representations concerning either the
 *      merchantability of this software or the suitability of this
 *      software for any particular purpose.  It is provided "as is"
 *      without express or implied warranty of any kind.
 *      
 */
#include <util/hash/sha1.h>

namespace isc {
namespace util {
namespace hash {

/* Local Function Prototyptes */
static void SHA1Finalize(SHA1Context *, uint8_t Pad_Byte);
static void SHA1PadMessage(SHA1Context *, uint8_t Pad_Byte);
static void SHA1ProcessMessageBlock(SHA1Context *);

/*
 * Define functions used by SHA1 hash
 */
static inline uint32_t
SHA_Ch(const uint32_t x, const uint32_t y, const uint32_t z) {
    return (((x) & ((y) ^ (z))) ^ (z));
}

static inline uint32_t
SHA_Maj(const uint32_t x, const uint32_t y, const uint32_t z) {
    return (((x) & ((y) | (z))) | ((y) & (z)));
}

static inline uint32_t
SHA_Parity(const uint32_t x, const uint32_t y, const uint32_t z) {
    return ((x) ^ (y) ^ (z));
}

static inline int 
SHA1CircularShift(uint8_t bits, uint32_t word) {
    return ((word << bits) | (word >> (32 - bits)));
}

static inline bool
SHA1AddLength(SHA1Context *context, uint32_t length) {
    uint32_t addTemp = context->Length_Low;
    context->Length_Low += length;
    if (context->Length_Low < addTemp && ++context->Length_High == 0) {
        return (true);
    } else {
        return (false);
    }
}

/*
 *  SHA1Reset
 *
 *  Description:
 *      This function will initialize the SHA1Context in preparation
 *      for computing a new SHA1 message digest.
 *
 *  Parameters:
 *      context: [in/out]
 *          The context to reset.
 *
 *  Returns:
 *      sha Error Code.
 *
 */
int
SHA1Reset(SHA1Context *context) {
    if (!context) {
        return (SHA_NULL);
    }

    context->Length_Low             = 0;
    context->Length_High            = 0;
    context->Message_Block_Index    = 0;

    context->Intermediate_Hash[0]   = 0x67452301;
    context->Intermediate_Hash[1]   = 0xEFCDAB89;
    context->Intermediate_Hash[2]   = 0x98BADCFE;
    context->Intermediate_Hash[3]   = 0x10325476;
    context->Intermediate_Hash[4]   = 0xC3D2E1F0;

    context->Computed   = 0;
    context->Corrupted  = 0;
    return (SHA_SUCCESS);
}


/*
 *  SHA1Input
 *
 *  Description:
 *      This function accepts an array of octets as the next portion
 *      of the message.
 *
 *  Parameters:
 *      context: [in/out]
 *          The SHA context to update
 *      message_array: [in]
 *          An array of characters representing the next portion of
 *          the message.
 *      length: [in]
 *          The length of the message in message_array
 *
 *  Returns:
 *      sha Error Code.
 *
 */
int
SHA1Input(SHA1Context *context, const uint8_t *message_array, unsigned length) {
    if (!length) {
        return (SHA_SUCCESS);
    }

    if (!context || !message_array) {
        return (SHA_NULL);
    }

    if (context->Computed) {
        context->Corrupted = SHA_STATEERROR;
        return (SHA_STATEERROR);
    }

    if (context->Corrupted) {
         return (context->Corrupted);
    }

    while(length-- && !context->Corrupted) {
        context->Message_Block[context->Message_Block_Index++] =
                                                    (*message_array & 0xFF);

        if (!SHA1AddLength(context, 8) &&
            (context->Message_Block_Index == SHA1_BLOCKSIZE))
        {
            SHA1ProcessMessageBlock(context);
        }

        message_array++;
    }

    return (SHA_SUCCESS);
}

/*
 * SHA1FinalBits
 *
 * Description:
 *   This function will add in any final bits of the message.
 *
 * Parameters:
 *   context: [in/out]
 *     The SHA context to update
 *   message_bits: [in]
 *     The final bits of the message, in the upper portion of the
 *     byte. (Use 0b###00000 instead of 0b00000### to input the
 *     three bits ###.)
 *   length: [in]
 *     The number of bits in message_bits, between 1 and 7.
 *
 * Returns:
 *   sha Error Code.
 */
int SHA1FinalBits(SHA1Context *context, const uint8_t message_bits,
                  unsigned int length)
{
    uint8_t masks[8] = {
        /* 0 0b00000000 */ 0x00,
        /* 1 0b10000000 */ 0x80,
        /* 2 0b11000000 */ 0xC0,
        /* 3 0b11100000 */ 0xE0,
        /* 4 0b11110000 */ 0xF0,
        /* 5 0b11111000 */ 0xF8,
        /* 6 0b11111100 */ 0xFC,
        /* 7 0b11111110 */ 0xFE
    };
    uint8_t markbit[8] = {
        /* 0 0b10000000 */ 0x80,
        /* 1 0b01000000 */ 0x40,
        /* 2 0b00100000 */ 0x20,
        /* 3 0b00010000 */ 0x10,
        /* 4 0b00001000 */ 0x08,
        /* 5 0b00000100 */ 0x04,
        /* 6 0b00000010 */ 0x02,
        /* 7 0b00000001 */ 0x01
    };

    if (!length) {
        return (SHA_SUCCESS);
    }

    if (!context) {
        return (SHA_NULL);
    }

    if (context->Computed || (length >= 8) || (length == 0)) {
        context->Corrupted = SHA_STATEERROR;
        return (SHA_STATEERROR);
    }

    if (context->Corrupted) {
        return (context->Corrupted);
    }

    SHA1AddLength(context, length);
    SHA1Finalize(context,
                 (uint8_t) ((message_bits & masks[length]) | markbit[length]));

    return (SHA_SUCCESS);
}

/*
 *  SHA1Result
 *
 *  Description:
 *      This function will return the 160-bit message digest into the
 *      Message_Digest array  provided by the caller.
 *      NOTE: The first octet of hash is stored in the 0th element,
 *            the last octet of hash in the 19th element.
 *
 *  Parameters:
 *      context: [in/out]
 *          The context to use to calculate the SHA-1 hash.
 *      Message_Digest: [out]
 *          Where the digest is returned.
 *
 *  Returns:
 *      sha Error Code.
 *
 */
int
SHA1Result(SHA1Context *context, uint8_t Message_Digest[SHA1_HASHSIZE]) {
    int i;

    if (!context || !Message_Digest) {
        return (SHA_NULL);
    }

    if (context->Corrupted) {
        return (context->Corrupted);
    }

    if (!context->Computed) {
        SHA1Finalize(context, 0x80);
    }

    for(i = 0; i < SHA1_HASHSIZE; ++i) {
        Message_Digest[i] = context->Intermediate_Hash[i>>2]
                            >> 8 * (3 - (i & 0x03));
    }

    return (SHA_SUCCESS);
}

/*
 * SHA1Finalize
 *
 * Description:
 *   This helper function finishes off the digest calculations.
 *
 * Parameters:
 *   context: [in/out]
 *     The SHA context to update
 *   Pad_Byte: [in]
 *     The last byte to add to the digest before the 0-padding
 *     and length. This will contain the last bits of the message
 *     followed by another single bit. If the message was an
 *     exact multiple of 8-bits long, Pad_Byte will be 0x80.
 *
 * Returns:
 *   sha Error Code.
 *
 */
static void SHA1Finalize(SHA1Context *context, uint8_t Pad_Byte) {
    int i;
    SHA1PadMessage(context, Pad_Byte);
    /* message may be sensitive, clear it out */
    for (i = 0; i < SHA1_BLOCKSIZE; ++i)
        context->Message_Block[i] = 0;
    context->Length_Low = 0;  /* and clear length */
    context->Length_High = 0;
    context->Computed = 1;
}

/*
 *  SHA1PadMessage
 *
 *  Description:
 *      According to the standard, the message must be padded to an even
 *      512 bits.  The first padding bit must be a '1'.  The last 64
 *      bits represent the length of the original message.  All bits in
 *      between should be 0.  This function will pad the message
 *      according to those rules by filling the Message_Block array
 *      accordingly.  It will also call the ProcessMessageBlock function
 *      provided appropriately.  When it returns, it can be assumed that
 *      the message digest has been computed.
 *
 *  Parameters:
 *      context: [in/out]
 *          The context to pad
 *      Pad_Byte: [in]
 *          The last byte to add to the digest before the 0-padding
 *          and length. This will contain the last bits of the message
 *          followed by another single bit. If the message was an
 *          exact multiple of 8-bits long, Pad_Byte will be 0x80.
 *
 *  Returns:
 *      Nothing.
 *
 */
static void SHA1PadMessage(SHA1Context *context, uint8_t Pad_Byte) {
    /*
     * Check to see if the current message block is too small to hold
     * the initial padding bits and length. If so, we will pad the
     * block, process it, and then continue padding into a second
     * block.
     */
    if (context->Message_Block_Index >= (SHA1_BLOCKSIZE - 8)) {
        context->Message_Block[context->Message_Block_Index++] = Pad_Byte;
        while (context->Message_Block_Index < SHA1_BLOCKSIZE) {
            context->Message_Block[context->Message_Block_Index++] = 0;
        }

        SHA1ProcessMessageBlock(context);
    } else
        context->Message_Block[context->Message_Block_Index++] = Pad_Byte;

    while (context->Message_Block_Index < (SHA1_BLOCKSIZE - 8))
        context->Message_Block[context->Message_Block_Index++] = 0;

    /*
     * Store the message length as the last 8 octets
     */
    context->Message_Block[56] = (uint8_t) (context->Length_High >> 24);
    context->Message_Block[57] = (uint8_t) (context->Length_High >> 16);
    context->Message_Block[58] = (uint8_t) (context->Length_High >> 8);
    context->Message_Block[59] = (uint8_t) (context->Length_High);
    context->Message_Block[60] = (uint8_t) (context->Length_Low >> 24);
    context->Message_Block[61] = (uint8_t) (context->Length_Low >> 16);
    context->Message_Block[62] = (uint8_t) (context->Length_Low >> 8);
    context->Message_Block[63] = (uint8_t) (context->Length_Low);

    SHA1ProcessMessageBlock(context);
}

/*
 *  SHA1ProcessMessageBlock
 *
 *  Description:
 *      This helper function will process the next 512 bits of the
 *      message stored in the Message_Block array.
 *
 *  Parameters:
 *      None.
 *
 *  Returns:
 *      Nothing.
 *
 *  Comments:
 *      Many of the variable names in this code, especially the
 *      single character names, were used because those were the
 *      names used in the publication.
 *
 *
 */
static void
SHA1ProcessMessageBlock(SHA1Context *context) {
    /* Constants defined in FIPS-180-2, section 4.2.1 */
    const uint32_t K[] = {
        0x5A827999,
        0x6ED9EBA1,
        0x8F1BBCDC,
        0xCA62C1D6
    };
    int           t;                 /* Loop counter                */
    uint32_t      temp;              /* Temporary word value        */
    uint32_t      W[80];             /* Word sequence               */
    uint32_t      A, B, C, D, E;     /* Word buffers                */

    /*
     * Initialize the first 16 words in the array W
     */
    for (t = 0; t < 16; t++) {
        W[t]  = ((uint32_t)context->Message_Block[t * 4]) << 24;
        W[t] |= ((uint32_t)context->Message_Block[t * 4 + 1]) << 16;
        W[t] |= ((uint32_t)context->Message_Block[t * 4 + 2]) << 8;
        W[t] |= ((uint32_t)context->Message_Block[t * 4 + 3]);
    }

    for (t = 16; t < 80; t++) {
        W[t] = SHA1CircularShift(1, W[t-3] ^ W[t-8] ^ W[t-14] ^ W[t-16]);
    }

    A = context->Intermediate_Hash[0];
    B = context->Intermediate_Hash[1];
    C = context->Intermediate_Hash[2];
    D = context->Intermediate_Hash[3];
    E = context->Intermediate_Hash[4];

    for (t = 0; t < 20; t++) {
        temp = SHA1CircularShift(5,A) + SHA_Ch(B, C, D) + E + W[t] + K[0];
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
        B = A;
        A = temp;
    }

    for (t = 20; t < 40; t++) {
        temp = SHA1CircularShift(5,A) + SHA_Parity(B, C, D) + E + W[t] + K[1];
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
        B = A;
        A = temp;
    }

    for (t = 40; t < 60; t++) {
        temp = SHA1CircularShift(5,A) + SHA_Maj(B, C, D) + E + W[t] + K[2];
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
        B = A;
        A = temp;
    }

    for (t = 60; t < 80; t++) {
        temp = SHA1CircularShift(5,A) + SHA_Parity(B, C, D) + E + W[t] + K[3];
        E = D;
        D = C;
        C = SHA1CircularShift(30,B);
        B = A;
        A = temp;
    }

    context->Intermediate_Hash[0] += A;
    context->Intermediate_Hash[1] += B;
    context->Intermediate_Hash[2] += C;
    context->Intermediate_Hash[3] += D;
    context->Intermediate_Hash[4] += E;

    context->Message_Block_Index = 0;
}

} // namespace hash
} // namespace util
} // namespace isc
