/*
 * Copyright (C) 2010-2011  Internet Systems Consortium, Inc. ("ISC")
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/* $Id: pcp.h 1253 2011-07-04 14:54:05Z fdupont $ */

/*
 * PCP (port-control-protocol) client library include
 *
 * Francis_Dupont@isc.org, December 2010
 *
 * a la miniUPnP libnatpmp
 *
 * types: pcp_t, pcp_request_t, pcp_response_t
 */

/* API version */

#define __PCP_API_VERSION__		0

#ifdef WIN32
#include <stdint.h>
#include <time.h>
#else
#include <sys/time.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif


/* Types */

#ifndef PCP_MAX_OPTIONS
#define PCP_MAX_OPTIONS	4
#endif
#define PCP_VECOPT_SIZE	(PCP_MAX_OPTIONS + 1)

typedef struct {			/* PCP handler (opaque) */
#ifdef WIN32
	SOCKET s;			/* server socket */
#else
	int s;				/* server socket */
#endif
	uint8_t usable;			/* better than s < 0 */
	uint8_t tries;			/* retry counter */
	uint8_t reset;			/* reset flag */
	struct timeval retry;		/* next retry date */
	uint16_t reqlen;		/* current request length */
	uint16_t resplen;		/* current response length */
	uint8_t *request;		/* current request */
#define PCP_RESPLENGTH	1040
	uint8_t *response;		/* current request */
} pcp_t;

typedef struct {			/* PCP option */
	uint8_t code;			/* option code */
	uint16_t length;		/* option data length (per 32 bits) */
	uint8_t *data;			/* option data */
} pcp_option_t;

typedef struct {			/* PCP element of request */
	uint8_t opcode;			/* opcode */
	uint8_t protocol;		/* protocol */
	uint8_t extaf;			/* external address family */
	union {
		uint32_t _ia4;		/* internal IPv4 address */
		uint8_t _ia6[16];	/* internal IPv6 address */
	} _ia;
#define intaddr4	_ia._ia4
#define intaddr6	_ia._ia6
	union {
		uint32_t _ea4;		/* external IPv4 address */
		uint8_t _ea6[16];	/* external IPv6 address */
	} _ea;
#define extaddr4	_ea._ea4
#define extaddr6	_ea._ea6
	union {
		uint32_t _rp4;		/* remote IPv4 peer */
		uint8_t _rp6[16];	/* remote IPv6 perr */
	} _rp;
#define remotepeer4	_rp._rp4
#define remotepeer6	_rp._rp6
	uint32_t lifetime;		/* lifetime */
	uint16_t intport;		/* internal port */
	uint16_t extport;		/* external port */
	uint16_t peerport;		/* remote peer port */
} pcp_request_t;

typedef struct {			/* PCP response */
	uint8_t result;			/* result-code */
	uint8_t optcnt;			/* option count */
	uint32_t epoch;			/* second since start of epoch */
	pcp_request_t assigned;		/* assigned request */
	pcp_option_t options[PCP_VECOPT_SIZE];	/* returned options */
	unsigned int optoffs[PCP_VECOPT_SIZE];	/* options offsets */
} pcp_response_t;

/* Routines */

extern int pcp_init(struct sockaddr *server,
		    struct sockaddr *source,
		    pcp_t *pcp);
extern int pcp_close(pcp_t *pcp);
extern int pcp_getsocket(pcp_t *pcp, int *sock);
extern int pcp_gettries(pcp_t *pcp, int *tries);
extern int pcp_gettimeout(pcp_t *pcp, struct timeval *timeout, int relative);
extern int pcp_setreset(pcp_t *pcp, int reset, int *old);
extern int pcp_third_party4(pcp_option_t ***options, uint32_t addr);
extern int pcp_third_party6(pcp_option_t ***options, uint8_t *addr);
extern int pcp_prefer_failure(pcp_option_t ***options);
extern int pcp_filter4(pcp_option_t ***options, uint32_t addr, uint16_t port);
extern int pcp_filter6(pcp_option_t ***options, uint8_t *addr, uint16_t port);
extern int pcp_makerequest(pcp_t *pcp,
			   pcp_request_t *request,
			   pcp_option_t **options);
extern int pcp_sendrequest(pcp_t *pcp);
extern int pcp_recvresponse(pcp_t *pcp, pcp_response_t *response);
extern const char *pcp_strerror(int error);
extern int pcp_fillintaddr(pcp_t *pcp, pcp_request_t *request);
extern int pcp_getmapping(pcp_t *pcp,
			  pcp_request_t *request,
			  pcp_option_t **options,
			  pcp_response_t *response);
extern int pcp_renewmapping(pcp_t *pcp, pcp_response_t *response);
extern int pcp_delmapping(pcp_t *pcp, pcp_response_t *response);
extern int _pcp_setrequest(pcp_t *pcp, uint8_t *request, uint16_t reqlen);
extern int _pcp_getrequest(pcp_t *pcp, uint8_t **request, uint16_t *reqlen);
extern int _pcp_setresponse(pcp_t *pcp, uint8_t *response, uint16_t resplen);
extern int _pcp_getresponse(pcp_t *pcp, uint8_t **response, uint16_t *resplen);

/* opcode values */
#define PCP_OPCODE_MAP4		1
#define PCP_OPCODE_MAP6		2
#define PCP_OPCODE_PEER4	3
#define PCP_OPCODE_PEER6	4
#define PCP_OPCODE_RESPONSE	128	/* response bit */

/* offsets in the common header */
#define PCP_VERSION		0
#define PCP_OPCODE		1
#define PCP_RESULT_CODE		3
#define PCP_LIFETIME		4
#define PCP_EPOCH		8
#define PCP_CLIENT_ADDR		12
#define PCP_OPCODE_HDR		28

/* offsets in the MAP header */
#define PCP_MAP_PROTOCOL	(PCP_OPCODE_HDR + 0)
#define PCP_MAP_INTERNAL_PORT	(PCP_OPCODE_HDR + 4)
#define PCP_MAP_EXTERNAL_PORT	(PCP_OPCODE_HDR + 6)
#define PCP_MAP_EXTERNAL_ADDR	(PCP_OPCODE_HDR + 8)

/* offsets in the PEER header */
#define PCP_PEER_PROTOCOL	(PCP_OPCODE_HDR + 0)
#define PCP_PEER_EXTERNAL_AF	(PCP_OPCODE_HDR + 1)
#define PCP_PEER_INTERNAL_PORT	(PCP_OPCODE_HDR + 4)
#define PCP_PEER_EXTERNAL_PORT	(PCP_OPCODE_HDR + 6)
#define PCP_PEER_REMOTE_PORT	(PCP_OPCODE_HDR + 8)
#define PCP_PEER_REMOTE_ADDR	(PCP_OPCODE_HDR + 12)
#define PCP_PEER_EXTERNAL_ADDR	(PCP_OPCODE_HDR + 16)

/* lengths */
#define PCP_LEN_MAP4		40
#define PCP_LEN_MAP6		52
#define PCP_LEN_PEER4		60
#define PCP_LEN_PEER6		72

/* external address families */

#define PCP_AF_IPv4		1
#define PCP_AF_IPv6		2

/* Errors */

/* OK */
#define PCP_OK			(0)	/* OK */
#define PCP_TRY_AGAIN		(1)	/* try again */
/* bad API use */
#define PCP_ERR_INVAL		(-1)	/* invalid arguments */
/* OS errors */
#define PCP_ERR_NOMEM		(-10)	/* malloc() failed */
#define PCP_ERR_SOCKET		(-11)	/* socket() syscall failed */
#define PCP_ERR_BIND		(-12)	/* bind() syscall failed */
#define PCP_ERR_CONNECT		(-13)	/* connect() syscall failed */
#define PCP_ERR_SEND		(-14)	/* send() syscall failed */
#define PCP_ERR_RECV		(-15)	/* recv() syscall failed */
#define PCP_ERR_SYSCALL		(-16)	/* miscellaneous syscall failed */
/* routine errors */
#define PCP_ERR_NOREQUEST	(-20)	/* no current request */
#define PCP_ERR_RECVBAD		(-21)	/* received a bad response */
#define PCP_ERR_TOOMANYOPTS	(-22)	/* too many options */
/* *mapping failure */
#define PCP_ERR_FAILURE		(-30)	/* *mapping() internal failure */
/* user application */
#define PCP_ERR_APP0		(-50)	/* user application error 0 */
#define PCP_ERR_APP1		(-51)	/* user application error 1 */
#define PCP_ERR_APP2		(-52)	/* user application error 2 */
/* protocol errors */
#define PCP_ERR_PROTOBASE	(-100)	/* base for protocol errors */
#define PCP_ERR_UNSUPVERSION	(-101)	/* unsupported version */
#define PCP_ERR_BADREQUEST	(-102)	/* malformed request */
#define PCP_ERR_UNSUPOPCODE	(-103)	/* unsupported opcode */
#define PCP_ERR_UNSUPOPTION	(-104)	/* unsupported option */
#define PCP_ERR_BADOPTION	(-105)	/* malformed option */
#define PCP_ERR_PROCERROR	(-106)	/* processing error */
#define PCP_ERR_SRVOVERLOAD	(-107)	/* overloaded server */
#define PCP_ERR_NETFAILURE	(-120)	/* network failure */
#define PCP_ERR_NORESOURCES	(-121)	/* out of resources */
#define PCP_ERR_UNSUPPROTO	(-122)	/* unsupported protocol */
#define PCP_ERR_NOTAUTH		(-123)	/* not authorized */
#define PCP_ERR_EXQUOTA		(-124)	/* user exceeded quota */
#define PCP_ERR_CANTPROVIDE	(-125)	/* cannot provide external port */
#define PCP_ERR_TOOMANYPEER	(-126)	/* excessive number of remote peers */
#define PCP_ERR_UNAUTH3PTY	(-151)	/* unauthorized third party */
#define PCP_ERR_IMPLICITMAP	(-228)	/* collides with implicit mapping */

/* Options */

#define PCP_OPTION_UNPROCESSED	0	/* options not processed by server */
#define PCP_OPTION_FILTER	2	/* remote peer filter */
#define PCP_OPTION_PREF_FAIL	3	/* prefer failure */
#define PCP_OPTION_THIRD_PARTY	4	/* third-party request */
#define PCP_OPTION_OPTIONAL	128	/* flag bit */

/* Others */

/* Default port of PCP service */

#define PCP_PORT		5351

/* Default retry timer (in seconds) */

#define PCP_RETRY_TIMER		2

/* Maximum number of tries */

#define PCP_MAX_TRIES		4

/* Default reset threshold (in seconds) */

#ifndef PCP_RESET_THRESHOLD
#define PCP_RESET_THRESHOLD		60
#endif

/* Tunables & co */

/* weak definition so you can overwrite them */

#ifdef WIN32
#define pcp_malloc(s)	malloc(s)
#define pcp_free(p)	free(p)
#else
extern void *pcp_malloc(size_t size);
extern void pcp_free(void *ptr);
#endif

/* inline utility to free allocated options */

static inline void pcp_freeoptions(pcp_option_t **options) {
	if (options != NULL) {
		unsigned int i;

		for (i = 0; options[i] != NULL; i++)
			pcp_free(options[i]);
		pcp_free(options);
		options = NULL;
	}
}

extern const char *err_app0_msg;
extern const char *err_app1_msg;
extern const char *err_app2_msg;

/* for bad response debugging */

#if defined(WIN32) && !defined(STATICLIB)
#ifdef PCP_EXPORTS
#define LIBSPEC __declspec(dllexport)
#else
#define LIBSPEC __declspec(dllimport)
#endif
#else
#define LIBSPEC
#endif

LIBSPEC int pcp_debug_offset;
LIBSPEC int pcp_debug_line;

#ifdef __cplusplus
}
#endif
