/*
 * Copyright (C) 2010-2011  Internet Systems Consortium, Inc. ("ISC")
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/* $Id: pcpd.c 786 2010-07-21 12:13:32Z fdupont $ */

/* PCP daemon (from xpmpd.c)
 *
 * Francis_Dupont@isc.org, December 2010
 *
 * Usage: [-c] [-d] [address]
 */

#define _GNU_SOURCE
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#ifdef PCP_ENCAP
#include <sys/un.h>
#endif

#include <ctype.h>
#include <db.h>
#include <errno.h>
#include <netdb.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

/* XXX should be made configurable */
#ifndef MAXPCP
#define MAXPCP		64
#endif
#ifndef MAXLIFE
#define MAXLIFE		7200
#endif

#define HOLD		120
#define HASHSIZE	128

/* PCP port number (== NAT-PMP port number) */
#define PORT		5351

/* offsets in the common header */
#define VERSION		0
#define OPCODE		1
#define RESULT_CODE	3
#define LIFETIME	4
#define EPOCH		8
#define CLIENT_ADDR	12
#define OPCODE_HDR	28
#define OPT_HDR_MAP	40
#define OPT_HDR_PEER	60

/* offsets in the MAP header */
#define MAP_PROTOCOL		(OPCODE_HDR + 0)
#define MAP_INTERNAL_PORT	(OPCODE_HDR + 4)
#define MAP_EXTERNAL_PORT	(OPCODE_HDR + 6)
#define MAP_EXTERNAL_ADDR	(OPCODE_HDR + 8)

/* offsets in the PEER header */
#define PEER_PROTOCOL		(OPCODE_HDR + 0)
#define PEER_EXTERNAL_AF	(OPCODE_HDR + 1)
#define PEER_INTERNAL_PORT	(OPCODE_HDR + 4)
#define PEER_EXTERNAL_PORT	(OPCODE_HDR + 6)
#define PEER_REMOTE_PORT	(OPCODE_HDR + 8)
#define PEER_REMOTE_ADDR	(OPCODE_HDR + 12)
#define PEER_EXTERNAL_ADDR	(OPCODE_HDR + 16)

/* opcode values */
#define MAP4		1
#define MAP6		2
#define PEER4		3
#define PEER6		4
#define RESPONSE	128	/* flag bit */

/* result codes */
#define SUCCESS			0
#define UNSUPP_VERSION		1
#define MALFORMED_REQUEST	2
#define UNSUPP_OPCODE		3
#define UNSUPP_OPTION		4
#define MALFORMED_OPTION	5
#define PROCESSING_ERROR	6
#define SERVER_OVERLOADED	7
#define ADDRESS_MISMATCH	8	/* XXX not defined in the spec */
#define NETWORK_FAILURE		20
#define NO_RESOURCES		21
#define UNSUPP_PROTOCOL		22
#define NOT_AUTHORIZED		23
#define USER_EX_QUOTA		24
#define CANNOT_PROVIDE_EXTERNAL_PORT	25
#define EXCESSIVE_REMOTE_PEERS	26
#define UNAUTH_THIRD_PARTY_INTERNAL_ADDRESS	51

/* option codes */
#define UNPROCESSED	0
#define FILTER		2
#define PREFER_FAILURE	3
#define THIRD_PARTY	4
#define OPTIONAL	128	/* flag bit */

#ifdef PCP_ENCAP
/* offsets and interesting constants for IPv4 and UDP headers */
#define IPLEN		2
#define IPPROTO		9
#define IPCKSUM		10
#define IPSRC		12
#define IPDST		16
#define UDPSPORT	20
#define UDPDPORT	22
#define UDPLEN		24
#define UDPCKSUM	26
#define IPHDRLEN	20
#define UDPHDRLEN	8
#endif

/* Local macros */

#define ISC_SLIST_HEAD(name, type)					\
	struct name {							\
		struct type *slh_first;					\
	}

#define ISC_SLIST_INIT(head)						\
	do {								\
		(head)->slh_first = NULL;				\
	} while (0)

#define ISC_SLIST_ENTRY(type)						\
	struct {							\
		struct type *sle_next;					\
	}

#define ISC_SLIST_EMPTY(head)						\
	((head)->slh_first == NULL)

#define ISC_SLIST_FIRST(head)						\
	((head)->slh_first)

#define ISC_SLIST_NEXT(elm, field)					\
	((elm)->field.sle_next)

#define ISC_SLIST_INSERT_HEAD(head, elm, field)				\
	do {								\
		(elm)->field.sle_next = (head)->slh_first;		\
		(head)->slh_first = (elm);				\
	} while (0)

#define ISC_SLIST_INSERT_AFTER(elm0, elm, field)			\
	do {								\
		(elm)->field.sle_next = (elm0)->field.sle_next;		\
		(elm0)->field.sle_next = (elm);				\
	} while (0)

#define ISC_SLIST_REMOVE_HEAD(head, field)				\
	do {								\
		(head)->slh_first = (head)->slh_first->field.sle_next;	\
	} while (0)

#define ISC_SLIST_REMOVE(head, elm, type, field)			\
	do {								\
		if ((head)->slh_first == (elm)) {			\
			ISC_SLIST_REMOVE_HEAD(head, field);		\
		} else {						\
			struct type *cur = (head)->slh_first;		\
			while (cur->field.sle_next != (elm))		\
				cur = cur->field.sle_next;		\
			cur->field.sle_next = (elm)->field.sle_next;	\
		}							\
	} while (0)

#define ISC_SLIST_FOREACH(var, head, field)				\
	for ((var) = (head)->slh_first;					\
	     (var) != NULL;						\
	     (var) = (var)->field.sle_next)

#define ISC_SLIST_FOREACH_SAFE(var, head, field, tvar)			\
	for ((var) = (head)->slh_first;					\
	     ((var) != NULL) &&	(((tvar) = (var)->field.sle_next), 1);	\
	     (var) = (tvar))

#define ISC_LIST_HEAD(name, type)					\
	struct name {							\
		struct type *lh_first;					\
	}

#define ISC_LIST_INIT(head)						\
	do {								\
		(head)->lh_first = NULL;				\
	} while (0)

#define ISC_LIST_ENTRY(type)						\
	struct {							\
		struct type *le_next;					\
		struct type **le_prev;					\
	}

#define ISC_LIST_EMPTY(head)						\
	((head)->lh_first == NULL)

#define ISC_LIST_FIRST(head)						\
	((head)->lh_first)

#define ISC_LIST_NEXT(elm, field)					\
	((elm)->field.le_next)

#define ISC_LIST_PREV(elm, field)					\
	((elm)->field.le_prev)

#define ISC_LIST_INSERT(head, elm, field)				\
	do {								\
		(elm)->field.le_next = (head)->lh_first;		\
		if ((head)->lh_first != NULL)				\
			(head)->lh_first->field.le_prev =		\
				&(elm)->field.le_next;			\
		(head)->lh_first = (elm);				\
		(elm)->field.le_prev = &(head)->lh_first;		\
	} while (0)

#define ISC_LIST_REMOVE(elm, field)					\
	do {								\
		if ((elm)->field.le_next != NULL)			\
			(elm)->field.le_next->field.le_prev =		\
				(elm)->field.le_prev;			\
		*(elm)->field.le_prev = (elm)->field.le_next;		\
		(elm)->field.le_next = NULL;				\
		(elm)->field.le_prev = NULL;				\
	} while (0)

#define ISC_LIST_FOREACH(var, head, field)				\
	for ((var) = (head)->lh_first;					\
	     (var) != NULL;						\
	     (var) = (var)->field.le_next)

#define ISC_LIST_FOREACH_SAFE(var, head, field, tvar)			\
	for ((var) = (head)->lh_first;					\
	     ((var) != NULL) &&	(((tvar) = (var)->field.le_next), 1);	\
	     (var) = (tvar))

#define Success		0
#define PortLimited	-1
#define OutofResources	-2
#define UnavailablePort	-3
#define Colliding	-4
#define SystemError	-5

#define PROTO2INDEX(x)	((x) == IPPROTO_TCP ? 0 : 1)

#define INDEX2PROTO(x)	((x) == 0 ? IPPROTO_TCP : IPPROTO_UDP)

/* heap */

u_int map_heap_size;
u_int map_heap_last;
struct mapping **map_heap;

/* port mapping */

struct mapping {
	ISC_LIST_ENTRY(mapping) chain;
	struct client *client;
	struct port *port;
	uint32_t src;
	u_int heap_index;
	time_t expire;
	uint16_t sport;
	uint8_t proto;
	uint8_t down;
};

/* client */

struct client {
	ISC_LIST_ENTRY(client) chain;
	struct in6_addr ipv6;
	struct natsrc *natsrc;
	time_t start;
	u_int mapcnt;
	ISC_LIST_HEAD(, mapping) mappings;
};

ISC_LIST_HEAD(bucket, client) clients[HASHSIZE];

/* per natted address object, clients per IPv6, IPv6 per port */

struct port {
	struct client *client;
	uint16_t port;
};

struct natsrc {
	ISC_SLIST_ENTRY(natsrc) chain;
	uint32_t addr;
	uint16_t minport[2], maxport[2];
	struct port *ports[2];
};

ISC_SLIST_HEAD(, natsrc) natsrcs;

void askdelsnat(struct mapping *mapping);
void dbaddsnat(struct mapping *mapping);
void dbdelsnat(struct mapping *mapping);

/* globals */

int dbclean = 0;
int transid;
char nextlinebuf[1024];
int nextlinebeg, nextlineend;
char pendingbuf[2048];
int pendingcnt;
int aftr;
int sock6;
#ifdef PCP_ENCAP
int sock4;
#define ENCAP_TO_PCPD "/var/run/aftr-pcpd.sock"
#define ENCAP_TO_AFTR "/var/run/pcpd-aftr.sock"
#endif
DB_ENV *dbenv;
DB *dbtun;
DB *dbsnat;

/* debug helper */

void
debug(const char *msg, ...)
{
	va_list va;

	va_start(va, msg);

	vsyslog(LOG_DEBUG, msg, va);
	va_end(va);
}

/* db_errcall helper */

void
db_errcall(const DB_ENV *_dbenv, const char *_pfx, const char *msg)
{
	_dbenv = _dbenv;
	_pfx = _pfx;
	syslog(LOG_ERR, "db %s", msg);
}

/* atexist helpers */

void
dbenv_close(void)
{
	(void) dbenv->close(dbenv, 0);
}

void
dbtun_close(void)
{
	(void) dbtun->close(dbtun, 0);
}

void
dbsnat_close(void)
{
	(void) dbsnat->close(dbsnat, 0);
}

/* Jenkins hash */

#define __jhash_mix(a, b, c) { \
	a -= b; a -= c; a ^= (c >> 13);	\
	b -= c; b -= a; b ^= (a << 8);	\
	c -= a; c -= b; c ^= (b >> 13);	\
	a -= b; a -= c; a ^= (c >> 12);	\
	b -= c; b -= a; b ^= (a << 16);	\
	c -= a; c -= b; c ^= (b >> 5);	\
	a -= b; a -= c; a ^= (c >> 3);	\
	b -= c; b -= a; b ^= (a << 10);	\
	c -= a; c -= b; c ^= (b >> 15);	}

#define JHASH_GOLDEN_RATIO	0x9e3779b9

inline u_short
jhash(struct in6_addr *ipv6)
{
	char *addr = (char *) ipv6;
	uint32_t a, b, c, k;

	memcpy(&a, addr, 4);
	a += JHASH_GOLDEN_RATIO;
	memcpy(&b, addr + 4, 4);
	b += JHASH_GOLDEN_RATIO;
	memcpy(&c, addr + 8, 4);

	__jhash_mix(a, b, c);

	memcpy(&k, addr + 12, 4);
	a += k;

	__jhash_mix(a, b, c);

	c = (c >> 16) + (c & 0xffff);
	c &= 0xffff;
	return (u_short) c;
}

/* mapping heap */

/* resize */

int
map_heap_resize(void)
{
	struct mapping **mh;
	u_int ms = map_heap_size + 1024;

	mh = (struct mapping **) malloc(ms * sizeof(*mh));
	if (mh == NULL) {
		syslog(LOG_ERR, "malloc(heap): %s\n", strerror(errno));
		return 0;
	}
	if (map_heap != NULL) {
		memcpy(mh, map_heap, map_heap_size * sizeof(*mh));
		free(map_heap);
	}
	map_heap = mh;
	map_heap_size = ms;
	return 1;
}

/* parent is i/2, left child is i*2 and right child is i*2+1 */

void
map_heap_float_up(u_int i, struct mapping *m)
{
	u_int p;

	for (p = i >> 1;
	     (i > 1) && (m->expire < map_heap[p]->expire);
	     i = p, p = i >> 1) {
		map_heap[i] = map_heap[p];
		map_heap[i]->heap_index = i;
	}
	map_heap[i] = m;
	map_heap[i]->heap_index = i;

	if ((i > 1) && (map_heap[i]->expire < map_heap[i >> 1]->expire))
		syslog(LOG_CRIT, "map_heap_float_up[%u]\n", i);
}

void
map_heap_sink_down(u_int i, struct mapping *m)
{
	u_int j, size, half;

	size = map_heap_last;
	half = size / 2;
	while (i <= half) {
		/* Find the smallest of the (at most) two children */
		j = i << 1;
		if ((j < size) &&
		    (map_heap[j + 1]->expire < map_heap[j]->expire))
			j++;
		if (m->expire < map_heap[j]->expire)
			break;
		map_heap[i] = map_heap[j];
		map_heap[i]->heap_index = i;
		i = j;
	}
	map_heap[i] = m;
	map_heap[i]->heap_index = i;

	if ((i > 1) && (map_heap[i]->expire < map_heap[i >> 1]->expire))
		syslog(LOG_CRIT, "map_heap_sink_down[%u]\n", i);
}

int
map_heap_insert(struct mapping *m)
{
	u_int new_last;

	new_last = map_heap_last + 1;
	if ((new_last >= map_heap_size) && !map_heap_resize())
		return 0;
	map_heap_last = new_last;
	map_heap_float_up(new_last, m);
	return 1;
}

void
map_heap_delete(u_int i)
{
	struct mapping *m;
	int less;

	if ((i < 1) || (i > map_heap_last)) {
		syslog(LOG_CRIT, "map_heap_delete[%u]\n", i);
		return;
	}

	if (i == map_heap_last) {
		map_heap[map_heap_last] = NULL;
		map_heap_last -= 1;
		return;
	}
	m = map_heap[map_heap_last];
	map_heap[map_heap_last] = NULL;
	map_heap_last -= 1;

	less = (m->expire < map_heap[i]->expire);
	map_heap[i] = m;
	if (less)
		map_heap_float_up(i, map_heap[i]);
	else
		map_heap_sink_down(i, map_heap[i]);
}

void
map_heap_increased(u_int i)
{
	if ((i < 1) || (i > map_heap_last)) {
		syslog(LOG_CRIT, "map_heap_increased[%u]\n", i);
		return;
	}

	map_heap_float_up(i, map_heap[i]);
}

void
map_heap_decreased(u_int i)
{
	if ((i < 1) || (i > map_heap_last)) {
		syslog(LOG_CRIT, "map_heap_decreased[%u]\n", i);
		return;
	}

	map_heap_sink_down(i, map_heap[i]);
}

struct mapping *
map_heap_element(u_int i)
{
	if (i < 1) {
		syslog(LOG_CRIT, "map_heap_element[%u]\n", i);
		return NULL;
	}

	if (i <= map_heap_last)
		return map_heap[i];
	return NULL;
}

/* create a new mapping */

struct mapping *
new_mapping(struct client *client, uint16_t nport, uint8_t proto)
{
	struct mapping *mapping;
	struct natsrc *natsrc;
	struct port *ports;

	mapping = (struct mapping *) malloc(sizeof(*mapping));
	if (mapping == NULL)
		return mapping;
	memset(mapping, 0, sizeof(*mapping));
	mapping->client = client;
	mapping->proto = proto;
	natsrc = client->natsrc;
	ports = natsrc->ports[PROTO2INDEX(proto)];
	mapping->port = ports + (nport - natsrc->minport[PROTO2INDEX(proto)]);
	if (mapping->port->port != nport)
		syslog(LOG_CRIT, "port mismatch");
	return mapping;
}

/* hold down a mapping */

void
hold(struct mapping *mapping)
{
	mapping->down = 1;
	mapping->expire = time(NULL) + HOLD;
	if (!map_heap_insert(mapping))
		syslog(LOG_CRIT, "map_heap_insert: %m");
	dbaddsnat(mapping);
}

/* create a new client */

struct client *
new_client(struct in6_addr *ipv6, struct natsrc *natsrc)
{
	struct client *client;

	client = (struct client *) malloc(sizeof(*client));
	if (client == NULL)
		return client;
	memset(client, 0, sizeof(*client));
	memcpy(&client->ipv6, ipv6, sizeof(*ipv6));
	client->natsrc = natsrc;
	client->start = time(NULL);
	ISC_LIST_INIT(&client->mappings);
	return client;
}

/* get a client with an IPv6 address */

struct client *
getclient(struct in6_addr *ipv6, int prompt)
{
	struct client *client;
	struct bucket *bucket;
	u_short hash;

	hash = jhash(ipv6) % HASHSIZE;
	bucket = &clients[hash];
	ISC_LIST_FOREACH(client, bucket, chain)
		if (memcmp(ipv6, &client->ipv6, sizeof(*ipv6)) == 0)
			break;
	if (prompt && (client != NULL)) {
		ISC_LIST_REMOVE(client, chain);
		ISC_LIST_INSERT(bucket, client, chain);
	}
	if ((client != NULL) && (client->start == 0))
		client->start = time(NULL);
	return client;
}

/* seconds since start of epoch */

uint32_t
sssoe(struct client *client)
{
	return (uint32_t)(time(NULL) - client->start);
}

/* search the mapping with nport */

struct mapping *
getmap(struct client *client, uint16_t nport, uint8_t proto)
{
	struct mapping *mapping;

	ISC_LIST_FOREACH(mapping, &client->mappings, chain)
		if ((mapping->port->port == nport) &&
		    (mapping->proto == proto))
			break;
	return mapping;
}

/* search the mapping with src/sport */

struct mapping *
getsrc(struct client *client, uint32_t src, uint16_t sport, uint8_t proto)
{
	struct mapping *mapping = NULL;

	ISC_LIST_FOREACH(mapping, &client->mappings, chain) {
		if ((mapping->src != src) || (mapping->proto != proto))
			continue;
		if (mapping->sport == sport)
			return mapping;
	}
	return NULL;
}

/* delete all mappings for a source or client */

void
delall(struct client *client, uint32_t src, uint8_t proto)
{
	struct mapping *mapping, *next;

	ISC_LIST_FOREACH_SAFE(mapping, &client->mappings, chain, next) {
		if ((proto != 0) && (mapping->proto != proto))
			continue;
		if (mapping->down)
			continue;
		if ((src != 0) && (mapping->src != src))
			continue;
		askdelsnat(mapping);
		if (mapping->expire != 0)
			map_heap_delete(mapping->heap_index);
		mapping->heap_index = 0;
		hold(mapping);
	}
}

/* create a new natsrc structure */

void
new_natsrc(uint32_t addr, uint16_t *minport, uint16_t *maxport)
{
	char host[INET_ADDRSTRLEN];
	struct natsrc *natsrc;
	struct port *port;
	size_t size[2];
	u_int i;

	(void) inet_ntop(AF_INET, &addr, host, INET_ADDRSTRLEN);
	debug("new_natsrc:\n\t%s tcp %d-%d\n\t%s udp %d-%d\n",
	      host, (int) minport[1], (int) maxport[1],
	      host, (int) minport[0], (int) maxport[0]);

	if ((minport[0] == 0) || (minport[1] == 0)) {
		fprintf(stderr,
			"new_natsrc: unsupported void pcp range for %s\n",
			host);
		return;
	}

	size[0] = (1 + maxport[0] - minport[0]) * sizeof(*port);
	size[1] = (1 + maxport[1] - minport[1]) * sizeof(*port);
	natsrc = (struct natsrc *) malloc(sizeof(*natsrc));
	if (natsrc == NULL) {
		fprintf(stderr, "new_natsrc failed: %s\n", strerror(errno));
		exit(1);
	}
	memset(natsrc, 0, sizeof(*natsrc));
	natsrc->addr = addr;
	natsrc->minport[0] = minport[0];
	natsrc->minport[1] = minport[1];
	natsrc->maxport[0] = maxport[0];
	natsrc->maxport[1] = maxport[1];
	port = (struct port *) malloc(size[0]);
	if (port == NULL) {
		fprintf(stderr, "new_natsrc failed0: %s\n", strerror(errno));
		exit(1);
	}
	memset(port, 0, size[0]);
	natsrc->ports[0] = port;
	for (i = minport[0]; i <= maxport[0]; i++, port++)
		port->port = i;
	port = (struct port *) malloc(size[1]);
	if (port == NULL) {
		fprintf(stderr, "new_natsrc failed1: %s\n", strerror(errno));
		exit(1);
	}
	memset(port, 0, size[1]);
	natsrc->ports[1] = port;
	for (i = minport[1]; i <= maxport[1]; i++, port++)
		port->port = i;
	ISC_SLIST_INSERT_HEAD(&natsrcs, natsrc, chain);
}

/* get first free port */

int
getfreeport(struct natsrc *natsrc, uint8_t proto)
{
	struct port *port;
	u_int i;

	port = natsrc->ports[PROTO2INDEX(proto)];
	for (i = natsrc->minport[PROTO2INDEX(proto)];
	     i <= natsrc->maxport[PROTO2INDEX(proto)];
	     i++, port++)
		if (port->client == NULL)
			return (int) i;
	syslog(LOG_WARNING, "can't get a free port");
	return OutofResources;
}

/* db tunnels: key is IPv6, data are natted IPv4 | start */

/* db tunnel add */

void
dbaddtunnel(struct client *client)
{
	DBT key, data;
	char kdata[8];

	memset(&key, 0, sizeof(key));
	memset(&data, 0, sizeof(data));
	key.size = 16;
	key.data = &client->ipv6;
	data.size = 8;
	data.data = kdata;
	memcpy(kdata, &client->natsrc->addr, 4);
	memcpy(kdata + 4, &client->start, 4);
	if (dbtun->put(dbtun, NULL, &key, &data, 0) != 0)
		debug("db->put(tun) failed");
}

/* db tunnel del */

void
dbdeltunnel(struct client *client)
{
	DBT key;

	memset(&key, 0, sizeof(key));
	key.size = 16;
	key.data = &client->ipv6;
	if (dbtun->del(dbtun, NULL, &key, 0) != 0)
		debug("db->del(tun) failed");
}

/* db snats:
 *   key is natted IPv4 | nport | proto,
 *   data are IPv6 | src IPv4 | sport | expire | down
 */

/* db snat add */

void
dbaddsnat(struct mapping *mapping)
{
	DBT key, data;
	char kbuf[7];
	char kdata[27];

	memset(&key, 0, sizeof(key));
	memset(&data, 0, sizeof(data));
	key.size = 7;
	key.data = kbuf;
	data.size = 27;
	data.data = kdata;
	memcpy(kbuf, &mapping->client->natsrc->addr, 4);
	memcpy(kbuf + 4, &mapping->port->port, 2);
	memcpy(kbuf + 6, &mapping->proto, 1);
	memcpy(kdata, &mapping->client->ipv6, 16);
	memcpy(kdata + 16, &mapping->src, 4);
	memcpy(kdata + 20, &mapping->sport, 2);
	memcpy(kdata + 22, &mapping->expire, 4);
	memcpy(kdata + 26, &mapping->down, 1);
	if (dbsnat->put(dbsnat, NULL, &key, &data, 0) != 0)
		debug("db->put(snat) failed");
}

/* db snat del */

void
dbdelsnat(struct mapping *mapping)
{
	DBT key;
	char kbuf[7];

	memset(&key, 0, sizeof(key));
	key.size = 7;
	key.data = kbuf;
	memcpy(kbuf, &mapping->client->natsrc->addr, 4);
	memcpy(kbuf + 4, &mapping->port->port, 2);
	memcpy(kbuf + 6, &mapping->proto, 1);
	if (dbsnat->del(dbsnat, NULL, &key, 0) != 0)
		debug("db->del(snat) failed");
}

/* send a command to AFTR daemon */

void
ask(char *cmd)
{
	char buf[128];

	(void) snprintf(buf, sizeof(buf), "%s\n", cmd);
	(void) send(aftr, buf, strlen(buf), 0);
	debug("send to AFTR: %s", cmd);
}

/* get next line from AFTR daemon */

char *
getnextline(int wait)
{
	fd_set set;
	struct timeval tv;
	char *ret, *eol;

	if (nextlinebeg >= nextlineend) {
		if (wait) {
			FD_ZERO(&set);
			FD_SET(aftr, &set);
			tv.tv_sec = 0;
			tv.tv_usec = 100000;
			select(aftr + 1, &set, NULL, NULL, &tv);
			if (!FD_ISSET(aftr, &set))
				return NULL;
		}
		memset(nextlinebuf, 0, sizeof(nextlinebuf));
		nextlinebeg = 0;
		nextlineend = recv(aftr, nextlinebuf, sizeof(nextlinebuf), 0);
	}
	eol = strchr(nextlinebuf + nextlinebeg, '\n');
	if (eol == NULL) {
		ret = nextlinebuf + nextlinebeg;
		nextlinebeg = nextlineend;
		return ret;
	}
	*eol++ = '\0';
	ret = nextlinebuf + nextlinebeg;
	nextlinebeg = eol - nextlinebuf;
	debug("received from AFTR: %s", ret);
	return ret;
}
		
/* expect a text from AFTR daemon */

char *
expect(const char *text, int *result)
{
	char *got, buf[64];
	int i, l, coll;

	transid++;
	*result = 0;
	coll = 0;
	(void) snprintf(buf, sizeof(buf), "echo %d\n", transid);
	(void) send(aftr, buf, strlen(buf), 0);
	for (i = 5; i != 0; i--) {
		got = getnextline(1);
		if (got == NULL)
			continue;
		if (strncmp(got, text, strlen(text)) == 0) {
			debug("got expected '%s' from AFTR", text);
			*result = 1;
			return got + strlen(text);
		}
		if (strcmp(got, "collision") == 0) {
			debug("got collision from AFTR");
			coll = 1;
		}
		if (strcmp(got, "command failed") == 0) {
			debug("got failure from AFTR");
			if (coll)
				*result = Colliding;
			return NULL;
		}
		l = strlen(got);
		if (strncmp(got, buf, l) == 0) {
			debug("got echo from AFTR");
			return NULL;
		}
		memcpy(pendingbuf + pendingcnt, got, l);
		pendingcnt += l + 1;
		pendingbuf[pendingcnt] = '\0';
	}
	return NULL;
}

/* ask list pcp */

void
asklistpcp(void)
{
	uint32_t addr, previous = 0;
	uint16_t minport[2], maxport[2];
	uint8_t proto;
	char *got, buf[64];
	int i, l;

	transid++;
	minport[0] = minport[1] = 0;
	maxport[0] = maxport[1] = 0;

	ask("list pcp");
	(void) snprintf(buf, sizeof(buf), "echo %d\n", transid);
	(void) send(aftr, buf, strlen(buf), 0);

	for (i = 5; i != 0; i--) {
		got = getnextline(1);
		if (got == NULL)
			continue;
		if (strncmp(got, "pcp ", 4) == 0) {
			char *line, *tok, *ptok;
			int minp, maxp;

			line = strdup(got);
			if (line == NULL) {
				fprintf(stderr,
					"asklistpcp: can't copy '%s': %s\n",
					got, strerror(errno));
				exit(1);
			}
			(void) strtok(line, " ");
			if ((tok = strtok(NULL, " ")) == NULL) {
				fprintf(stderr,
					"asklistpcp: can't get "
					"address in '%s'\n",
					got);
				exit(1);
			}
			if (inet_pton(AF_INET, tok, &addr) != 1) {
				fprintf(stderr,
					"asklistpcp: bad address in '%s'\n",
					got);
				exit(1);
			}
			if ((previous != 0) && (previous != addr)) {
				new_natsrc(previous, minport, maxport);
				minport[0] = minport[1] = 0;
				maxport[0] = maxport[1] = 0;
			}
			previous = addr;
			if ((tok = strtok(NULL, " ")) == NULL) {
				fprintf(stderr,
					"asklistpcp: can't get "
					"protocol in '%s'\n",
					got);
				exit(1);
			}
			if (strcmp(tok, "tcp") == 0)
				proto = IPPROTO_TCP;
			else if (strcmp(tok, "udp") == 0)
				proto = IPPROTO_UDP;
			else {
				fprintf(stderr,
					"asklistpcp: bad protocol in '%s'\n",
					got);
				exit(1);
			}
			if ((ptok = strtok(NULL, " ")) == NULL) {
				fprintf(stderr,
					"asklistpcp: can't get "
					"ports in '%s'\n",
					got);
				exit(1);
			}
			if (strtok(NULL, " ") != NULL) {
				fprintf(stderr,
					"asklistpcp: extra token in '%s'\n",
					got);
				exit(1);
			}
			if ((tok = strtok(ptok, "-")) == NULL) {
				fprintf(stderr,
					"asklistpcp: can't get "
					"minport in '%s'\n",
					got);
				exit(1);
			}
			minp = atoi(tok);
			if ((minp < 0) || (minp > 65535)) {
				fprintf(stderr,
					"asklistpcp: bad minport in '%s'\n",
					got);
				exit(1);
			}
			minport[PROTO2INDEX(proto)] = (uint16_t) minp;
			if ((tok = strtok(NULL, "\n")) == NULL) {
				fprintf(stderr,
					"asklistpcp: can't get "
					"maxport in '%s'\n",
					got);
				exit(1);
			}
			maxp = atoi(tok);
			if ((maxp < 0) || (maxp > 65535) ||
			    (minp > maxp) || ((maxp == 0) && (minp != 0))) {
				fprintf(stderr,
					"asklistpcp: bad maxport in '%s'\n",
					got);
				exit(1);
			}
			maxport[PROTO2INDEX(proto)] = (uint16_t) maxp;
			continue;
		}
		if (strcmp(got, "command failed") == 0) {
			debug("got failure from AFTR");
			return;
		}
		l = strlen(got);
		if (strncmp(got, buf, l) == 0) {
			if (previous != 0)
				new_natsrc(previous, minport, maxport);
			return;
		}
		memcpy(pendingbuf + pendingcnt, got, l);
		pendingcnt += l + 1;
		pendingbuf[pendingcnt] = '\0';
	}
	debug("timeout on 'list pcp'");
}

/* ask tunnel info */

uint32_t
asktunnel(struct in6_addr *addr)
{
	char host[INET6_ADDRSTRLEN];
	char cmd[128], resp[128];
	char *nsrc;
	int ret;
	uint32_t in;

	(void) inet_ntop(AF_INET6, addr, host, INET6_ADDRSTRLEN);
	(void) snprintf(cmd, sizeof(cmd), "try tunnel %s", host);
	(void) snprintf(resp, sizeof(resp), "tunnel %s ", host);
	ask(cmd);
	nsrc = expect(resp, &ret);
	if (ret) {
		if (inet_pton(AF_INET, nsrc, &in) <= 0) {
			debug("inet_pton(%s) failed?", nsrc);
			return 0;
		}
		debug("tunnel %s %s", host, nsrc);
		return in;
	} else {
		debug("tunnel %s failed?", host);
		return 0;
	}
}

/* ask tunnel add */

int
askaddtunnel(struct client *client)
{
	char host6[INET6_ADDRSTRLEN];
	char host4[INET_ADDRSTRLEN];
	char text[128];
	uint32_t addr;
	int ret;

	addr = client->natsrc->addr;
	(void) inet_ntop(AF_INET6, &client->ipv6, host6, INET6_ADDRSTRLEN);
	(void) inet_ntop(AF_INET, &addr, host4, INET_ADDRSTRLEN);
	(void) snprintf(text, sizeof(text), "try tunnel %s %s", host6, host4);
	ask(text);
	(void) expect(text + 4, &ret);
	if (ret)
		syslog(LOG_NOTICE, "%s", text + 4);
	else
		debug("add %s failed?", text + 4);
	return ret;
}

/* ask static NAT delete */

void
askdelsnat(struct mapping *mapping)
{
	struct client *client;
	char host6[INET6_ADDRSTRLEN];
	char host4[INET_ADDRSTRLEN];
	char cmd[256];
	char *proto;

	client = mapping->client;
	if (mapping->proto == IPPROTO_TCP)
		proto = " tcp ";
	else
		proto = " udp ";
	(void) inet_ntop(AF_INET6, &client->ipv6, host6, INET6_ADDRSTRLEN);
	(void) inet_ntop(AF_INET, &mapping->src, host4, INET_ADDRSTRLEN);
	(void) snprintf(cmd, sizeof(cmd), "delete nat %s%s%s %hu",
			host6, proto, host4, mapping->sport);
	ask(cmd);
	cmd[3] = 'd';
	cmd[4] = 'e';
	cmd[5] = 'l';
	syslog(LOG_NOTICE, "%s", cmd + 3);
}

/* ask static NAT add */

int
askaddsnat(struct mapping *mapping)
{
	struct client *client;
	char host6[INET6_ADDRSTRLEN];
	char src4[INET_ADDRSTRLEN];
	char dst4[INET_ADDRSTRLEN];
	char text[256];
	char *proto;
	int ret;

	client = mapping->client;
	if (mapping->proto == IPPROTO_TCP)
		proto = " tcp ";
	else
		proto = " udp ";
	(void) inet_ntop(AF_INET6, &client->ipv6, host6, INET6_ADDRSTRLEN);
	(void) inet_ntop(AF_INET, &mapping->src, src4, INET_ADDRSTRLEN);
	(void) inet_ntop(AF_INET, &client->natsrc->addr,
			 dst4, INET_ADDRSTRLEN);
	(void) snprintf(text, sizeof(text), "try nat %s%s%s %hu %s %hu",
			host6, proto, src4, mapping->sport, dst4,
			mapping->port->port);
	ask(text);
	(void) expect(text + 4, &ret);
	if (ret == 1) {
		text[0] = 'a';
		text[1] = 'd';
		text[2] = 'd';
		syslog(LOG_NOTICE, "%s", text);
	} else
		debug("add %s failed?", text + 4);
	return ret;
}

/* delete a client (direct removal of mappings) */

void
delclient(struct client *client)
{
	struct mapping *mapping;

	ISC_LIST_FOREACH(mapping, &client->mappings, chain) {
		if (mapping->expire != 0)
			map_heap_delete(mapping->heap_index);
		mapping->port->client = NULL;
		dbdelsnat(mapping);
		ISC_LIST_REMOVE(mapping, chain);
		free(mapping);
		client->mapcnt -= 1;
	}
	ISC_LIST_REMOVE(client, chain);
	dbdeltunnel(client);
	free(client);
}

/* get external address */

struct client *
getextaddr(struct in6_addr *ipv6)
{
	struct client *client;
	struct natsrc *natsrc;
	uint32_t nsrc;

	client = getclient(ipv6, 1);
	if (client != NULL)
		return client;
	natsrc = NULL;
	nsrc = asktunnel(ipv6);
	if (nsrc != 0) {
		ISC_SLIST_FOREACH(natsrc, &natsrcs, chain)
			if (natsrc->addr == nsrc)
				break;
	}
	if (natsrc != NULL) {
		struct bucket *bucket;
		u_short hash;

		client = new_client(ipv6, natsrc);
		if (client == NULL)
			return NULL;
		hash = jhash(ipv6) % HASHSIZE;
		bucket = &clients[hash];
		ISC_LIST_INSERT(bucket, client, chain);
		dbaddtunnel(client);
		return client;
	} else {
		char host[INET6_ADDRSTRLEN];

		(void) inet_ntop(AF_INET6, ipv6, host, INET6_ADDRSTRLEN);
		debug("can't get external address for %s", host);
		return NULL;
	}
}

/* delete a mapping */

void
delmapping(struct client *client, uint32_t src, uint16_t sport, uint8_t proto)
{
	struct mapping *mapping;

	mapping = getsrc(client, src, sport, proto);
	if (mapping == NULL) {
		debug("delmapping: no mapping");
		return;
	}
	if (mapping->down) {
		debug("delmapping: on hold");
		return;
	}
	askdelsnat(mapping);
	if (mapping->expire != 0)
		map_heap_delete(mapping->heap_index);
	mapping->heap_index = 0;
	hold(mapping);
	debug("delmapping: hold");
}

/* set a mapping */

int
setmapping(struct client *client, uint32_t src,
	   uint16_t sport, uint16_t *nport,
	   int locked, u_int life, uint8_t proto)
{
	struct client *holder;
	struct natsrc *natsrc;
	struct mapping *mapping;
	int outofrange, ret;
	time_t previous;

	natsrc = client->natsrc;
	mapping = getsrc(client, src, sport, proto);
	if ((mapping != NULL) && (mapping->port->port != *nport)) {
		if (locked)
			return UnavailablePort;
		*nport = mapping->port->port;
		return setmapping(client, src, sport, nport,
				  0, life, proto);
	}
	if (*nport == 0) {
		int candidate;

		if (locked)
			return UnavailablePort;
		candidate = getfreeport(natsrc, proto);
		if (candidate < 0)
			return candidate;
		*nport = (uint16_t) candidate;
		return setmapping(client, src, sport, nport,
				  0, life, proto);
	}
	holder = NULL;
	outofrange = 0;
	if ((*nport < natsrc->minport[PROTO2INDEX(proto)]) ||
	    (*nport > natsrc->maxport[PROTO2INDEX(proto)]))
		outofrange = 1;
	else {
		struct port *ports;

		ports = natsrc->ports[PROTO2INDEX(proto)];
		ports += *nport - natsrc->minport[PROTO2INDEX(proto)];
		holder = ports->client;
	}
	if (outofrange || ((holder != NULL) && (holder != client))) {
		*nport = 0;
		return setmapping(client, src, sport, nport,
				  locked, life, proto);
	}
		
	if (holder == NULL) {
		if (client->mapcnt >= MAXPCP)
			return PortLimited;
		mapping = new_mapping(client, *nport, proto);
		if (mapping == NULL)
			return OutofResources;
		mapping->src = src;
		mapping->sport = sport;
		ret = askaddsnat(mapping);
		if (ret == 1) {
			mapping->expire = time(NULL) + life;
			if (!map_heap_insert(mapping))
				syslog(LOG_CRIT, "map_heap_insert: %m");
			dbaddsnat(mapping);
			ISC_LIST_INSERT(&client->mappings, mapping, chain);
			mapping->port->client = client;
			client->mapcnt += 1;
			debug("setmapping: new");
			return Success;
		} else {
			debug("setmapping: failed (new)");
			free(mapping);
			if (ret == Colliding)
				return Colliding;
			else
				return SystemError;
		}
	}

	/* else (holder == client) */
	mapping = getmap(client, *nport, proto);
	if (mapping->src != src) {
		*nport = 0;
		return setmapping(client, src, sport, nport,
				  locked, life, proto);
	}
	if (mapping->down) {
		mapping->down = 0;
		mapping->sport = sport;
	} else if (mapping->sport != sport) {
		*nport = 0;
		return setmapping(client, src, sport, nport,
				  locked, life, proto);
	}
	ret = askaddsnat(mapping);
	if (ret != 1) {
		debug("setmapping: failed (new)");
		free(mapping);
		if (ret == Colliding)
			return Colliding;
		else
			return SystemError;
	}
	previous = mapping->expire;
	mapping->expire = time(NULL) + life;
	if (previous == 0) {
		if (!map_heap_insert(mapping))
			syslog(LOG_CRIT, "map_heap_insert: %m");
	} else if (previous > mapping->expire)
		map_heap_increased(mapping->heap_index);
	else
		map_heap_decreased(mapping->heap_index);
	dbaddsnat(mapping);
	debug("setmapping: renew");
	return Success;
}

/* send a mapping directly in hold down state */

int
setmappinghold(struct client *client, uint32_t src, uint16_t sport,
	       uint16_t nport, u_int expire, uint8_t proto)
{
	struct natsrc *natsrc;
	struct port *ports;
	struct mapping *mapping;

	natsrc = client->natsrc;
	if ((nport < natsrc->minport[PROTO2INDEX(proto)]) ||
	    (nport > natsrc->maxport[PROTO2INDEX(proto)]))
		return 0;
	ports = natsrc->ports[PROTO2INDEX(proto)];
	ports += nport - natsrc->minport[PROTO2INDEX(proto)];
	if (ports->client != NULL)
		return 0;
	if (client->mapcnt >= 64)
		return 0;
	mapping = new_mapping(client, nport, proto);
	if (mapping == NULL)
		return 0;
	mapping->down = 1;
	mapping->src = src;
	mapping->sport = sport;
	mapping->expire = expire;
	if (!map_heap_insert(mapping))
		syslog(LOG_CRIT, "map_heap_insert: %m");
	dbaddsnat(mapping);
	ISC_LIST_INSERT(&client->mappings, mapping, chain);
	mapping->port->client = client;
	client->mapcnt += 1;
	debug("setmappinghold: new");
	return 1;
}

/* set an error response code */

void
seterr(u_char *response, u_char code) {
	debug("error %d", code);
	response[RESULT_CODE] = code;
}

/* process a MAP4 request */

size_t
maprequest(struct in6_addr *src6, uint32_t src4, u_char *request, size_t len)
{
	struct client *client;
	u_char proto, option, *optdata;
	uint32_t ext, lifetime;
	uint16_t sport, nport, optlen;
	int result, have_3rdparty = 0, prefail = 0, unproc = 0;
	size_t off0, off;
	u_char unprocessed[128] = { 0 };

	if (len < OPT_HDR_MAP) {
		debug("underrun %d", len);
		seterr(request, MALFORMED_REQUEST);
		return len;
	}
	lifetime = ntohl(*(uint32_t *)(request + LIFETIME));
	proto = request[MAP_PROTOCOL];
	if ((proto != IPPROTO_TCP) && (proto != IPPROTO_UDP) &&
	    ((proto != 0) || (lifetime != 0))) {
		debug("bad protocol %d", proto);
		seterr(request, UNSUPP_PROTOCOL);
		return len;
	}
	sport = ntohs(*(uint16_t *)(request + MAP_INTERNAL_PORT));
	nport = ntohs(*(uint16_t *)(request + MAP_EXTERNAL_PORT));
	memcpy(&ext, request + MAP_EXTERNAL_ADDR, 4);

	/* process options */

	off = OPT_HDR_MAP;
	while (off < len) {
		if (off + 4 > len) {
			seterr(request, MALFORMED_OPTION);
			return len;
		}
		option = request[off];
		optlen = htons(*(uint16_t *)(request + off + 2));
		if (off + 4 + optlen > len) {
			seterr(request, MALFORMED_OPTION);
			return len;
		}
		optdata = request + off + 4;
		switch (option) {
		case THIRD_PARTY:
			if (optlen != 4 || have_3rdparty) {
		  		seterr(request, MALFORMED_OPTION);
				unprocessed[option] = 1;
				++unproc;
				break;
			}
			have_3rdparty = 1;
			memcpy(&src4, optdata, 4);
			/* aftr will validate addr */
			break;
		case PREFER_FAILURE:
			if (optlen != 0 || prefail || lifetime == 0) {
		  		seterr(request, MALFORMED_OPTION);
				unprocessed[option] = 1;
				++unproc;
				break;
			}
			prefail = 1;
			break;
		case FILTER:
#ifdef notyet	/* fall through to UNSUPP_OPTION */
			if (optlen != 8) {
		  		seterr(request, MALFORMED_OPTION);
				unprocessed[option] = 1;
				++unproc;
				break;
			}
			prefixlen = *(optdata + 1);
			rport = *(uint16_t *)(optdata + 2);
			raddr = *(uint32_t *)(optdata + 4);
			/* XXX send this to aftr */
#endif
		default:
			debug("unknown option %d", option);
			/* strip option out of response */
			memmove(request + off, request + off + 4 + optlen,
				4 + optlen);
			len -= 4 + optlen;
			/* if mandatory to process, add to UNPROCESSED list */
			if (option & OPTIONAL == 0) {
				seterr(request, UNSUPP_OPTION);
				unprocessed[option] = 1;
				++unproc;
				break;
			}
			/* skip the normal advancement of the off ptr */
			continue;
		}
		off += 4 + optlen;
	}
	if (unproc != 0) {
		int i;
		/* write out the UNPROCESSED option */
		request[len] = UNPROCESSED;
		request[len + 1] = 0;
		*(uint16_t *)(request + len + 2) = htons((unproc + 3) & ~3);
		len += 4;
		for (i = 0; i < sizeof(unprocessed); ++i) {
			if (unprocessed[i])
				request[len++] = i;
		}
		while (len & 3 != 0)
			request[len++] = 0;
		return len;
	}

	if (lifetime == 0) {
		/* delete */
		nport = 0;
		client = getclient(src6, 0);
		if (client != NULL) {
			*(uint32_t *)(request + EPOCH) = htonl(sssoe(client));
			if (sport != 0) {
				if (src4 == 0) {
					debug("no internal address");
					seterr(request, MALFORMED_REQUEST);
					return len;
				}
				delmapping(client, src4, sport, proto);
			} else {
				delall(client, src4, proto);
			}
		}
	} else {
		/* (lifetime > 0) */
		if (src4 == 0) {
			debug("no internal address");
			seterr(request, MALFORMED_REQUEST);
			return len;
		}
		if (sport == 0) {
			seterr(request, MALFORMED_REQUEST);
			return len;
		}
		client = getextaddr(src6);
		if (client == NULL) {
			char host[INET6_ADDRSTRLEN];
			(void) inet_ntop(AF_INET6, src6, host,
					 INET6_ADDRSTRLEN);
			debug("can't find client for setmapping: %s", host);
			seterr(request, NOT_AUTHORIZED); /* PROCESSING_ERROR */
			return len;
		}
		*(uint32_t *)(request + EPOCH) = htonl(sssoe(client));
		if ((ext != 0) &&
		    (memcmp(&ext, &client->natsrc->addr, 4) != 0)) {
			debug("external address mismatch");
			seterr(request, NOT_AUTHORIZED);
			return len;
		}
		if (lifetime > MAXLIFE)
			lifetime = MAXLIFE;
		result = setmapping(client, src4, sport, &nport,
				    prefail, lifetime, proto);
		switch (result) {
		case Success:
			break;
		case PortLimited:
			seterr(request, USER_EX_QUOTA);
			return len;
		case OutofResources:
			seterr(request, NO_RESOURCES);
			return len;
		case UnavailablePort:
			seterr(request, CANNOT_PROVIDE_EXTERNAL_PORT);
			return len;
		case Colliding:
		case SystemError:
			seterr(request, PROCESSING_ERROR);
			return len;
		default:
			seterr(request, NOT_AUTHORIZED);
                        return len;
		}
	}
	/* external address */
	if (client != NULL)
		memcpy(request + MAP_EXTERNAL_ADDR, &client->natsrc->addr, 4);
	else
		memset(request + MAP_EXTERNAL_ADDR, 0, 4);
	/* lifetime */
	lifetime = htonl(lifetime);
	memcpy(request + LIFETIME, &lifetime, 4);
	/* external port */
	nport = htons(nport);
	memcpy(request + MAP_EXTERNAL_PORT, &nport, 2);
	return len;
}

/* process a PEER4 request */

size_t
peerrequest(struct in6_addr *src6, uint32_t src4, u_char *request, size_t len)
{
	/* placeholder stub routine */
	seterr(request, UNSUPP_OPCODE);
	return len;
}

/* decode a request */

size_t
getrequest(struct in6_addr *src6, uint32_t src4, u_char *request, size_t len)
{
	u_char version, opcode;
	struct in6_addr client6;

	/* first sanity checks - drop packet if these occur */
	if (len < 4) {
		debug("underrun %d", len);
		return 0;
	}
	opcode = request[OPCODE];
	if (opcode & RESPONSE) {
		debug("response");
		return 0;
	}

	/* reuse the request buffer for the response */
	request[OPCODE] |= RESPONSE;
	request[RESULT_CODE] = SUCCESS;
	
	if (len > 1024) {
		debug("overrun %d", len);
		seterr(request, MALFORMED_REQUEST);
		return 1024;
	}
	if ((len & 3) != 0) {
		debug("bad align %d", len);
		seterr(request, MALFORMED_REQUEST);
		return (len + 3) & ~3;
	}
	version = request[VERSION];
	if (version != 1) {
		seterr(request, UNSUPP_VERSION);
		return len;
	}

	if (len < OPCODE_HDR) {
		debug("underrun %d", len);
		seterr(request, MALFORMED_REQUEST);
		return len;
	}
	memcpy(&client6, request + CLIENT_ADDR, 16);
	if (memcmp(&client6, src6, 16) != 0) {
		debug("client address mismatch");
		seterr(request, ADDRESS_MISMATCH);
		return len;
	}

	switch (opcode & ~RESPONSE) {
	case MAP4:
		return maprequest(src6, src4, request, len);
	case PEER4:
		return peerrequest(src6, src4, request, len);
	default:
		seterr(request, UNSUPP_OPCODE);
		return len;
	}
}

/* expire routine */

void
expire(void)
{
	struct mapping *mapping;
	time_t now;

	now = time(NULL);
	for (;;) {
		mapping = map_heap_element(1);
		if ((mapping == NULL) || (mapping->expire >= now))
			return;
		/* expire one mapping */
		map_heap_delete(mapping->heap_index);
		mapping->heap_index = 0;
		debug("expire");
		if (mapping->down) {
			/* final */
			dbdelsnat(mapping);
			ISC_LIST_REMOVE(mapping, chain);
			mapping->port->client = NULL;
			mapping->client->mapcnt -= 1;
			free(mapping);
		} else {
			/* hold it */
			askdelsnat(mapping);
			hold(mapping);
		}
	}
}

/* check for a 'tunnel del ' notification */

void
checkdeltunnel(const char *got)
{
	struct client *client;
	struct in6_addr ipv6;

	if (strncmp(got, "tunnel del ", 11) != 0)
		return;
	if (inet_pton(AF_INET6, got + 11, &ipv6) != 1)
		return;
	client = getclient(&ipv6, 0);
	if (client == NULL)
		return;
	syslog(LOG_NOTICE, "delete client %s", got + 11);
	delclient(client);
}

/* reload state databases */

void
reload(void)
{
	DBC *dbc;
	DBT key, data;
	char host6[INET6_ADDRSTRLEN];
	char host4[INET_ADDRSTRLEN];
	struct client *client;
	struct natsrc *natsrc;
	uint32_t addr;
	int ret;

	dbc = NULL;
	memset(&key, 0, sizeof(key));
	memset(&data, 0, sizeof(data));

	/* tunnels */
	if (dbtun->cursor(dbtun, NULL, &dbc, 0) != 0) {
		syslog(LOG_CRIT, "db->cursor(tun)");
		exit(1);
	}
	for (;;) {
		struct bucket *bucket;
		uint32_t start;
		u_short hash;

		ret = dbc->get(dbc, &key, &data, DB_NEXT);
		if (ret != 0) {
			if (ret != DB_NOTFOUND)
				syslog(LOG_ERR, "dbc->get(tun)");
			(void) dbc->close(dbc);
			break;
		}
		if ((key.size != 16) || (data.size != 8)) {
			syslog(LOG_ERR, "db(tunnel): bad");
			if (dbclean)
				(void) dbtun->del(dbtun, NULL, &key, 0);
			continue;
		}
		memcpy(&addr, (caddr_t) data.data, 4);
		memcpy(&start, (caddr_t) data.data + 4, 4);
		ISC_SLIST_FOREACH(natsrc, &natsrcs, chain)
			if (natsrc->addr == addr)
				break;
		if (natsrc == NULL) {
			(void) inet_ntop(AF_INET, &addr,
					 host4, INET_ADDRSTRLEN);
			debug("db(tunnel): can't find %s", host4);
			if (dbclean)
				(void) dbtun->del(dbtun, NULL, &key, 0);
			continue;
		}
		client = new_client((struct in6_addr *) key.data, natsrc);
		if (client == NULL) {
			debug("new_client failed");
			continue;
		}
		if (!askaddtunnel(client)) {
			free(client);
			continue;
		}
		hash = jhash((struct in6_addr *) key.data) % HASHSIZE;
		bucket = &clients[hash];
		ISC_LIST_INSERT(bucket, client, chain);
		client->start = start;
	}
	dbc = NULL;
	
	/* static nats */
	if (dbsnat->cursor(dbsnat, NULL, &dbc, 0) != 0) {
		syslog(LOG_CRIT, "db->cursor(snat)");
		exit(1);
	}
	for (;;) {
		uint32_t src, expire;
		uint16_t sport, nport;
		uint8_t proto, down;
		u_int life;
		int locked, result;
		time_t now;
		
		ret = dbc->get(dbc, &key, &data, DB_NEXT);
		if (ret != 0) {
			if (ret != DB_NOTFOUND)
				syslog(LOG_ERR, "dbc->get(snat)");
			(void) dbc->close(dbc);
			break;
		}
		if ((key.size != 7) || (data.size != 27)) {
			syslog(LOG_ERR, "db(snat): bad");
			if (dbclean)
				(void) dbsnat->del(dbsnat, NULL, &key, 0);
			continue;
		}
		memcpy(&addr, key.data, 4);
		memcpy(&nport, (caddr_t) key.data + 4, 2);
		memcpy(&proto, (caddr_t) key.data + 6, 1);
		memcpy(&src, (caddr_t) data.data + 16, 4);
		memcpy(&sport, (caddr_t) data.data + 20, 2);
		memcpy(&expire, (caddr_t) data.data + 22, 4);
		memcpy(&down, (caddr_t) data.data + 26, 1);
		if ((proto != IPPROTO_TCP) && (proto != IPPROTO_UDP)) {
			debug("db(snat): bad proto %d", (int) proto);
			if (dbclean)
				(void) dbsnat->del(dbsnat, NULL, &key, 0);
			continue;
		}
		now = time(NULL);
		if ((expire != 0) && (now >= (time_t) expire)) {
			if (dbclean)
				(void) dbsnat->del(dbsnat, NULL, &key, 0);
			continue;
		}
		client = getclient((struct in6_addr *) data.data, 0);
		if (client == NULL) {
			(void) inet_ntop(AF_INET6, data.data,
					 host6, INET6_ADDRSTRLEN);
			debug("db(snat): can't find %s", host6);
			if (dbclean)
				(void) dbsnat->del(dbsnat, NULL, &key, 0);
			continue;
		}
		natsrc = client->natsrc;
		if (natsrc->addr != addr) {
			(void) inet_ntop(AF_INET6, data.data,
					 host6, INET6_ADDRSTRLEN);
			debug("db(snat): address mismatch on %s", host6);
			if (dbclean)
				(void) dbsnat->del(dbsnat, NULL, &key, 0);
			continue;
		}
		if (down) {
			if (setmappinghold(client, src, sport,
					   nport, expire, proto))
				continue;
			debug("db(snat): setmappinghold failed");
			if (dbclean)
				(void) dbsnat->del(dbsnat, NULL, &key, 0);
			continue;
		}
		life = expire - now;
		locked = 1;
		result = setmapping(client, src, sport, &nport,
				    locked, life, proto);
		if (result == Success)
			continue;
		else
			debug("db(snat): setmapping failed %d", result);
		if (dbclean)
			(void) dbsnat->del(dbsnat, NULL, &key, 0);
	}
}

/* dump databases */

int
dump_client(FILE *ft, struct client *client)
{
	uint8_t *p;
	int i;

	/* key */
	fputc(' ', ft);
	p = (uint8_t *) &client->ipv6;
	for (i = 0; i < 16; i++)
		fprintf(ft, "%02x", (u_int) *p++);
	fputc('\n', ft);

	/* data */
	fputc(' ', ft);
	p = (uint8_t *) &client->natsrc->addr;
	for (i = 0; i < 4; i++)
		fprintf(ft, "%02x", (u_int) *p++);
	p = (uint8_t *) &client->start;
	for (i = 0; i < 4; i++)
		fprintf(ft, "%02x", (u_int) *p++);
	fputc('\n', ft);
	return -ferror(ft);
}

int
dump_mapping(FILE *fn, struct mapping *mapping)
{
	uint8_t *p;
	int i;

	/* key */
	fputc(' ', fn);
	p = (uint8_t *) &mapping->client->natsrc->addr;
	for (i = 0; i < 4; i++)
		fprintf(fn, "%02x", (u_int) *p++);
	p = (uint8_t *) &mapping->port->port;
	fprintf(fn, "%02x", (u_int) *p++);
	fprintf(fn, "%02x", (u_int) *p);
	fprintf(fn, "%02x\n", (u_int) mapping->proto);

	/* data */
	fputc(' ', fn);
	p = (uint8_t *) &mapping->client->ipv6;
	for (i = 0; i < 16; i++)
		fprintf(fn, "%02x", (u_int) *p++);
	p = (uint8_t *) &mapping->src;
	for (i = 0; i < 4; i++)
		fprintf(fn, "%02x", (u_int) *p++);
	p = (uint8_t *) &mapping->sport;
	fprintf(fn, "%02x", (u_int) *p++);
	fprintf(fn, "%02x", (u_int) *p);
	p = (uint8_t *) &mapping->expire;
	for (i = 0; i < 4; i++)
		fprintf(fn, "%02x", (u_int) *p++);
	fprintf(fn, "%02x\n", (u_int) mapping->down);
	return -ferror(fn);
}

void
dump(void)
{
	struct client *client;
	struct bucket *bucket;
	struct mapping *mapping;
	FILE *ft, *fn;
	int i, ret;

	ft = fopen("tunnels.dump", "w+");
	if (ft == NULL) {
		debug("dump(tunnel): can't open file: %s", strerror(errno));
		return;
	}
	fn = fopen("snats.dump", "w+");
	if (fn == NULL) {
		debug("dump(snat): can't open file: %s", strerror(errno));
		(void) fclose(ft);
		return;
	}

	ret = 0;
	for (i = 0; i < HASHSIZE; i++) {
		bucket = &clients[i];
		ISC_LIST_FOREACH(client, bucket, chain) {
			ret = dump_client(ft, client);
			if (ret < 0)
				break;
			ISC_LIST_FOREACH(mapping, &client->mappings, chain) {
				ret = dump_mapping(fn, mapping);
				if (ret < 0)
					break;
			}
			if (ret < 0)
				break;
		}
		if (ret < 0)
			break;
	}
	(void) fclose(ft);
	(void) fclose(fn);
}

/* signal handler (just set a flag) */

int todump;

void
handler(int dummy __attribute__((unused)))
{
	todump = 1;
}

#ifdef PCP_ENCAP
/* Compute checksum */

int
in_cksum(u_char *p, u_int l)
{
	int sum = 0;

	while (l > 1) {
		sum += *p++ << 8;
		sum += *p++;
		l -= 2;
	}
	if (l == 1)
		sum += *p << 8;
	sum = ((sum >> 16) & 0xffff) + (sum & 0xffff);
	sum += sum >> 16;
	return (0xffff & ~sum);
}

int
udp_cksum(u_char *pkt, u_int plen)
{
	int sum;
	u_char saved[4];

	memcpy(saved, pkt + 8, 4);
	pkt[8] = 0;
	memcpy(pkt + 10, pkt + UDPLEN, 2);
	sum = in_cksum(pkt + 8, plen - 8);
	memcpy(pkt + 8, saved, 4);
	return sum;
}
#endif

/* main loop */

void
mainloop(void)
{
	fd_set set;
	struct timeval tv;
	int maxfd, cc, plen;
	char *got, host[INET6_ADDRSTRLEN];
	struct sockaddr_in6 sin6;
	socklen_t sl;
	u_char packet[1080];

	for(;;) {
		expire();
		while (pendingcnt > 0) {
			int len;

			checkdeltunnel(pendingbuf);
			len = strlen(pendingbuf) + 1;
			memmove(pendingbuf, pendingbuf + len,
				pendingcnt - len);
			pendingcnt -= len;
		}
		pendingcnt = 0;
		if (todump)
			dump();
		todump = 0;

		FD_ZERO(&set);
		FD_SET(aftr, &set);
		maxfd = aftr;
		FD_SET(sock6, &set);
		if (sock6 > maxfd)
			maxfd = sock6;
#ifdef PCP_ENCAP
		FD_SET(sock4, &set);
		if (sock4 > maxfd)
			maxfd = sock4;
#endif
		tv.tv_sec = 1;
		tv.tv_usec = 0;
		cc = select(maxfd + 1, &set, NULL, NULL, &tv);
		if ((cc < 0) && (errno != EINTR)) {
			syslog(LOG_CRIT, "select fatal error: %m");
			exit(0);
		}
		if (cc <= 0)
			continue;

		if (FD_ISSET(aftr, &set)) {
			got = getnextline(0);
			if (got == NULL) {
				syslog(LOG_CRIT, 
				       "AFTR daemon connection closed");
				exit(0);
			}
			checkdeltunnel(got);
		}
		if (FD_ISSET(sock6, &set)) {
			memset(packet, 0, sizeof(packet));
			sl = sizeof(sin6);
			plen = recvfrom(sock6, packet, sizeof(packet), 0,
					(struct sockaddr *) &sin6, &sl);
			if (plen < 0) {
				syslog(LOG_CRIT, "PCP socket error: %m");
				continue;
			}
			(void) inet_ntop(AF_INET6, &sin6.sin6_addr,
					 host, INET6_ADDRSTRLEN);
			debug("got request from %s", host);
			plen = getrequest(&sin6.sin6_addr, 0, packet, plen);
			if (plen > 0) {
				debug("sendresp");
				(void) sendto(sock6, packet, plen, 0,
					      (struct sockaddr *) &sin6,
					      sizeof(sin6));
			}
		}
#ifdef PCP_ENCAP
		if (FD_ISSET(sock4, &set)) {
			struct msghdr msg;
			struct iovec iov[2];
			struct sockaddr_un aftr_addr;
			uint32_t src4;
			char host4[INET_ADDRSTRLEN];

			memset(&msg, 0, sizeof(msg));
			iov[0].iov_base = &sin6.sin6_addr;
			iov[0].iov_len = 16;
			memset(packet, 0, sizeof(packet));
			iov[1].iov_base = packet;
			iov[1].iov_len = sizeof(packet);
			msg.msg_iov = iov;
			msg.msg_iovlen = 2;
			memset(&aftr_addr, 0, sizeof(aftr_addr));
			msg.msg_name = (void *)&aftr_addr;
			msg.msg_namelen = sizeof(aftr_addr);
			plen = recvmsg(sock4, &msg, 0);
			if (plen < 0) {
				syslog(LOG_CRIT, "encap socket error: %m");
				continue;
			}
			memcpy(&src4, packet + IPSRC, 4);
			(void) inet_ntop(AF_INET, &src4,
					 host4, INET_ADDRSTRLEN);
			(void) inet_ntop(AF_INET6, &sin6.sin6_addr,
					 host, INET6_ADDRSTRLEN);
 			debug("got request from %s via %s", host4, host);
			plen = getrequest(&sin6.sin6_addr, src4,
					  packet + IPHDRLEN + UDPHDRLEN,
					  ntohs(*(uint16_t *)(packet + UDPLEN)));
			if (plen > 0) {
				debug("sendresp");
				/* adjust v4 headers, send back to aftr */
				memcpy(packet + IPSRC, packet + IPDST, 4);
				memcpy(packet + IPDST, &src4, 4);
				memcpy(packet + UDPDPORT, packet + UDPSPORT, 2);
				*(uint16_t *)(packet + UDPSPORT) = htons(PORT);
				*(uint16_t *)(packet + UDPLEN) = htons(plen);
				plen += IPHDRLEN + UDPHDRLEN;
				*(uint16_t *)(packet + IPLEN) = htons(plen);
				*(uint16_t *)(packet + IPCKSUM) = 0;
				*(uint16_t *)(packet + IPCKSUM) =
					htons(in_cksum(packet, IPHDRLEN));
				*(uint16_t *)(packet + UDPCKSUM) = 0;
				*(uint16_t *)(packet + UDPCKSUM) =
					htons(udp_cksum(packet, plen));
				iov[1].iov_len = plen;
				(void) sendmsg(sock4, &msg, 0);
			}
		}
#endif
	}
}
			
/* main */

int
main(int argc, char *argv[])
{
	int logopt = LOG_NDELAY;
	int i, opt, on, ret;
	char *address;
	struct sockaddr_in sa;
	struct sockaddr_in6 sa6;
	extern char *optarg;
	extern int optind;
#ifdef PCP_ENCAP
	struct sockaddr_un su;
#endif

	for (i = 0; i < HASHSIZE; i++)
		ISC_LIST_INIT(&clients[i]);
	ISC_SLIST_INIT(&natsrcs);

	while ((opt = getopt(argc, argv, "cd")) != -1)
		switch (opt) {
		case 'c':
			dbclean = 1;
			break;
		case 'd':
			logopt += LOG_PERROR;
			break;
		default:
		usage:
#define USAGE	"Usage: %s [-c] [-d] [address]\n"
			fprintf(stderr, USAGE, argv[0]);
			exit(1);
		}
	if (optind + 1 == argc)
		address = argv[optind];
	else if (optind == argc)
		address = NULL;
	else {
		fprintf(stderr, "extra arguments\n");
		goto usage;
	}

	openlog("pcpd", logopt, LOG_LOCAL6);

	/* connect to aftr control interface */
	aftr = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (aftr < 0) {
		syslog(LOG_ERR, "socket(inet): %m");
		exit(1);
	}
	memset(&sa, 0, sizeof(sa));
	sa.sin_family = AF_INET;
	(void) inet_pton(AF_INET, "127.0.0.1", &sa.sin_addr);
	sa.sin_port = htons(1015);
	if (connect(aftr, (struct sockaddr *) &sa, sizeof(sa)) < 0) {
		syslog(LOG_ERR, "connect(inet): %m");
		exit(1);
	}
	ask("session log off");
	ask("session notify on");
	ask("session config on");
	ask("session name pcpd");
	asklistpcp();

	if (ISC_SLIST_EMPTY(&natsrcs)) {
		fprintf(stderr, "list pcp is empty\n");
		exit(1);
	}

	/* create the listening sockets */
	sock6 = socket(PF_INET6, SOCK_DGRAM, IPPROTO_UDP);
	if (sock6 < 0) {
		syslog(LOG_ERR, "socket(inet6): %m");
		exit(1);
	}
	memset(&sa6, 0, sizeof(sa6));
	sa6.sin6_family = AF_INET6;
	if ((address != NULL) &&
	    (inet_pton(AF_INET6, address, &sa6.sin6_addr) <= 0)) {
		fprintf(stderr, "bad address \"%s\"\n", address);
		exit(1);
	}
	sa6.sin6_port = htons(PORT);
	on = 1;
	if (setsockopt(sock6, IPPROTO_IPV6, IPV6_V6ONLY,
		       &on, sizeof(on)) < 0)
		syslog(LOG_ERR, "setsockopt(v6only): %m");
	if (bind(sock6, (struct sockaddr *) &sa6, sizeof(sa6)) < 0) {
		syslog(LOG_ERR, "bind(inet6): %m");
		exit(1);
	}

#ifdef PCP_ENCAP
	sock4 = socket(PF_UNIX, SOCK_DGRAM, 0);
	if (sock4 < 0) {
		logerr("socket(unix): %s\n", strerror(errno));
		exit(-1);
	}
	memset(&su, 0, sizeof(su));
	su.sun_family = AF_UNIX;
	strcpy(su.sun_path, ENCAP_TO_PCPD);
	(void) unlink(ENCAP_TO_PCPD);
	if (bind(sock4, (struct sockaddr *) &su, sizeof(su)) < 0) {
		logerr("bind(unix): %s\n", strerror(errno));
		exit(1);
	}
#endif

	ret = db_env_create(&dbenv, 0);
	if (ret != 0) {
		syslog(LOG_ERR, "db_env_create: %s", db_strerror(ret));
		exit(1);
	}
	dbenv->set_errcall(dbenv, db_errcall);
#define DBENV_FLAGS	(DB_CREATE | DB_INIT_MPOOL | DB_INIT_TXN)
	ret = dbenv->open(dbenv, NULL, DBENV_FLAGS, 0);
	if (ret != 0) {
		syslog(LOG_ERR, "dbenv->open: %s", db_strerror(ret));
		exit(1);
	}
	(void) atexit(dbenv_close);
	ret = db_create(&dbtun, dbenv, 0);
	if (ret != 0) {
		syslog(LOG_ERR, "db_create(tun): %s", db_strerror(ret));
		exit(1);
	}
	ret = dbtun->open(dbtun, NULL, "tunnels.db", NULL,
			  DB_HASH, DB_CREATE | DB_AUTO_COMMIT, 0);
	if (ret != 0) {
		syslog(LOG_ERR, "db->open(tunnels.db): %s", db_strerror(ret));
		exit(1);
	}
	(void) atexit(dbtun_close);
	ret = db_create(&dbsnat, dbenv, 0);
	if (ret != 0) {
		syslog(LOG_ERR, "db_create(snat): %s", db_strerror(ret));
		exit(1);
	}
	ret = dbsnat->open(dbsnat, NULL, "snats.db", NULL,
			  DB_HASH, DB_CREATE | DB_AUTO_COMMIT, 0);
	if (ret != 0) {
		syslog(LOG_ERR, "db->open(snats.db): %s", db_strerror(ret));
		exit(1);
	}
	(void) atexit(dbsnat_close);
	reload();
	(void) signal(SIGUSR1, handler);

	mainloop();
	exit(1);
}
