/*
 * Copyright (C) 2010-2011  Internet Systems Consortium, Inc. ("ISC")
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Routines for Port Control Protocol packet disassembly.
 * draft-ietf-pcp-base-10.txt
 */

/* $Id: print-pcp.c 1164 2011-05-21 04:25:40Z pselkirk $ */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <tcpdump-stdinc.h>

#include <stdio.h>

#include "pcp.h"
#include "interface.h"
#include "addrtoname.h"
#include "extract.h"
`#include "ipproto.h"

const char *
pcp_errcode(u_int8_t errcode)
{
	static char buf[64];

	switch (errcode) {
	case PCP_RC_SUCCESS:
		return "success";
	case PCP_RC_UNSUPP_VERSION:
		return "unsupported version";
	case PCP_RC_MALFORMED_REQUEST:
		return "malformed request";
	case PCP_RC_UNSUPP_OPCODE:
		return "unsupported opcode";
	case PCP_RC_UNSUPP_OPTION:
		return "unsupported option";
	case PCP_RC_MALFORMED_OPTION:
		return "malformed option";
	case PCP_RC_PROCESSING_ERROR:
		return "processing error";
	case PCP_RC_SERVER_OVERLOADED:
		return "server overloaded";
	case PCP_RC_NETWORK_FAILURE:
		return "network failure";
	case PCP_RC_NO_RESOURCES:
		return "out of resources";
	case PCP_RC_UNSUPP_PROTOCOL:
		return "unsupported protocol";
	case PCP_RC_NOT_AUTHORIZED:
		return "not authorized";
	case PCP_RC_USER_EX_QUOTA:
		return "user exceeded quota";
	case PCP_RC_CANT_PROVIDE_EXT_PORT:
		return "cannot provide external port";
	case PCP_RC_EXCESSIVE_REMOTE_PEERS:
		return "excessive remote peers";
	case PCP_RC_UNAUTH_THIRD_PARTY:
		return "unauth third party internal address";
	case PCP_RC_ADDRESS_MISMATCH:
		return "address mismatch";
	default:
		(void) snprintf(buf, sizeof(buf),
				"unknown(%u)", (u_int) errcode);
		return (const char *)buf;
	}
}

void
pcp_print(const u_char *dat, u_int length)
{
	const u_char *ptr = dat;
	const u_char *endptr = dat + length;
	u_char *p;
	u_int8_t opcode, proto, extaf, opt;
	u_int16_t pport;
	u_int olen;

	TCHECK2(*ptr, 4);

	switch (*ptr) {
	case PCP_VERSION:
		printf("PCP");
		break;
	default:
		printf("PCP: unknown version(%u)", (u_int) *ptr);
		return;
	}
	ptr++;

	opcode = *ptr;
	if ((opcode & PCP_RESPONSE) == 0)
		printf(" request");
	else
		printf(" response");
	switch (opcode & ~PCP_RESPONSE) {
	case PCP_OC_MAP4:
		printf(" map4");
		break;
	case PCP_OC_MAP6:
		printf(" map6");
		break;
	case PCP_OC_PEER4:
		printf(" peer4");
		break;
	case PCP_OC_PEER6:
		printf(" peer6");
		break;
	default:
		printf(" unknown(%u)", (u_int) opcode);
		return;
	}
	ptr += 2;

	if ((opcode & PCP_RESPONSE) && (*ptr != PCP_RC_SUCCESS))
		printf(",error=%s", pcp_errcode(*ptr));
	ptr++;

	TCHECK2(*ptr, 4);
	printf(",lifetime=%u", EXTRACT_32BITS(ptr));
	ptr += 4;

	TCHECK2(*ptr, 4);
	if (opcode & PCP_RESPONSE)
		printf(",epoch=%u", EXTRACT_32BITS(ptr));
	ptr += 4;

	/* XXX can't print PCP Client's IP Address because we can't
	 * determine the address family of the enclosing IP packet */
	ptr += 16;

	switch (opcode & ~PCP_RESPONSE) {
	case PCP_OC_MAP4:
	case PCP_OC_MAP6:
		TCHECK(*ptr);
		proto = *ptr;
		if (proto == PCP_PROTO_ALL)
			printf(",proto=all");
		else
			printf(",proto=%s",
			       tok2str(ipproto_values, "unknown", (int) proto));
		ptr += 4;
		TCHECK2(*ptr, 4);
		printf(",priv=%u", EXTRACT_16BITS(ptr));
		ptr += 2;
		printf(",pub=%u", EXTRACT_16BITS(ptr));
		ptr += 2;
		if ((opcode & ~PCP_RESPONSE) == PCP_OC_MAP4) {
			TCHECK2(*ptr, 4);
			printf(",external=%s", ipaddr_string(ptr));
			ptr += 4;
		}
		else {
			TCHECK2(*ptr, 16);
			printf(",external=%s", ip6addr_string(ptr));
			ptr += 16;
		}
		break;
	case PCP_OC_PEER4:
	case PCP_OC_PEER6:
		TCHECK2(*ptr, 2);
		proto = *ptr;
		if (proto == PCP_PROTO_ALL)
			printf(",proto=all");
		else
			printf(",proto=%s",
			       tok2str(ipproto_values, "unknown", (int) proto));
		ptr++;
		extaf = *ptr;
		switch (extaf) {
		case PCP_AF_IPv4:
			printf(",extaf=IPv4");
			break;
		case PCP_AF_IPv6:
			printf(",extaf=IPv6");
			break;
		default:
			printf(",extaf=%d", (int) extaf);
			return;
		}
		ptr += 3;
		TCHECK2(*ptr, 2);
		printf(",priv=%u", EXTRACT_16BITS(ptr));
		ptr += 2;
		TCHECK2(*ptr, 2);
		printf(",pub=%u", EXTRACT_16BITS(ptr));
		ptr += 2;
		TCHECK2(*ptr, 2);
		pport = EXTRACT_16BITS(ptr);
		ptr += 4;
		if ((opcode & ~PCP_RESPONSE) == PCP_OC_PEER4) {
			TCHECK2(*ptr, 4);
			printf(",peer=%s.%u", ipaddr_string(ptr), pport);
			ptr += 4;
		}
		else {
			TCHECK2(*ptr, 16);
			printf(",peer=%s.%u", ip6addr_string(ptr), pport);
			ptr += 16;
		}
		if (extaf == PCP_AF_IPv4) {
			TCHECK2(*ptr, 4);
			printf(",external=%s", ipaddr_string(ptr));
			ptr += 4;
		}
		else {
			TCHECK2(*ptr, 16);
			printf(",external=%s", ip6addr_string(ptr));
			ptr += 16;
		}
		break;
	}

	while (ptr < endptr) {
		TCHECK2(*ptr, 4);
		opt = *ptr;
		ptr += 2;
		olen = EXTRACT_16BITS(ptr);
		ptr += 2;
		printf("\n\toption:");
		switch (opt) {
		case PCP_OPT_UNPROCESSED:
			printf(" unprocessed options");
			break;
		case PCP_OPT_FILTER:
			printf(" remote_peer_filter");
			break;
		case PCP_OPT_PREFER_FAILURE:
			printf(" prefer failure");
			break;
		case PCP_OPT_THIRD_PARTY:
			printf(" third party");
			break;
		default:
			printf(" unknown(%u)", (u_int) opt);
			break;
		}

		TCHECK2(*ptr, olen);
		switch (opt) {
		case PCP_OPT_UNPROCESSED:
			for (p = (u_char *) ptr; p < ptr + olen; ++p) {
				if (*p != 0)
					printf(" %u", (int) *p);
			}
			break;
		case PCP_OPT_FILTER:
			printf(" pfxlen=%u", (int) *(ptr + 1));
			printf(" port=%u", EXTRACT_16BITS(ptr + 2));
			if (olen == 8)
				printf(" addr=%s", ipaddr_string(ptr + 4));
			else if (olen == 20)
				printf(" addr=%s", ip6addr_string(ptr + 4));
			else
				printf(" data? %u", olen - 4);
			break;
		case PCP_OPT_PREFER_FAILURE:
			break;
		case PCP_OPT_THIRD_PARTY:
			if (olen == 4)
				printf(" addr=%s", ipaddr_string(ptr));
			else if (olen == 16)
				printf(" addr=%s", ip6addr_string(ptr));
			else
				printf(" data? %u", olen - 4);
			break;
		default:
			if (olen > 0)
				printf(" data? %u", olen);
			break;
		}
		ptr += olen;
	}
	return;

 trunc:
	printf("[|pcp]");
}
