/*
 * Copyright (C) 2010-2011  Internet Systems Consortium, Inc. ("ISC")
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/* $Id: pcp.c 1207 2011-06-29 17:13:22Z pselkirk $ */

/*
 * PCP (port-control-protocol) client library body
 *
 * Francis_Dupont@isc.org, December 2010
 *
 * a la miniUPnP libnatpmp
 */

#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "pcp.h"

/* Initialize a PCP handler */

int
pcp_init(struct sockaddr *server, struct sockaddr *source, pcp_t *pcp)
{
	struct sockaddr_in *server4 = (struct sockaddr_in *) server;
	struct sockaddr_in6 *server6 = (struct sockaddr_in6 *) server;
	int flags;

	if ((pcp == NULL) || (server == NULL) ||
	    ((source != NULL) && (server->sa_family != source->sa_family)))
		return PCP_ERR_INVAL;
	memset(pcp, 0, sizeof(pcp_t));
	pcp->s = socket(server->sa_family == AF_INET6 ? PF_INET6 : PF_INET,
			SOCK_DGRAM, IPPROTO_UDP);
	if (pcp->s < 0)
		return PCP_ERR_SOCKET;
	/* avoid s == 0 */
	if (pcp->s == 0) {
		int newfd;

		newfd = dup(pcp->s);
		(void) close(pcp->s);
		pcp->s = newfd;
		if (newfd < 0)
			return PCP_ERR_SYSCALL;
	}
	if (((flags = fcntl(pcp->s, F_GETFL, 0)) < 0) ||
	    (fcntl(pcp->s, F_SETFL, flags | O_NONBLOCK) < 0)) {
		(void) close(pcp->s);
		pcp->s = -1;
		return PCP_ERR_SYSCALL;
	}
	if (server->sa_family == AF_INET6) {
		if (server6->sin6_port == 0)
			server6->sin6_port = htons(PCP_PORT);
		if ((source != NULL) &&
		    (bind(pcp->s, source, sizeof(*server6)) < 0)) {
			(void) close(pcp->s);
			pcp->s = -1;
			return PCP_ERR_BIND;
		}
		if (connect(pcp->s, server, sizeof(*server6)) < 0) {
			(void) close(pcp->s);
			pcp->s = -1;
			return PCP_ERR_CONNECT;
		}
	} else {
		if (server4->sin_port == 0)
			server4->sin_port = htons(PCP_PORT);
		if ((source != NULL) &&
		    (bind(pcp->s, source, sizeof(*server4)) < 0)) {
			(void) close(pcp->s);
			pcp->s = -1;
			return PCP_ERR_BIND;
		}
		if (connect(pcp->s, server, sizeof(*server4)) < 0) {
			(void) close(pcp->s);
			pcp->s = -1;
			return PCP_ERR_CONNECT;
		}
	}
	return PCP_OK;
}

/* close a PCP handler */

int
pcp_close(pcp_t *pcp)
{
	if (pcp == NULL)
		return PCP_ERR_INVAL;
	if (pcp->request != NULL)
		pcp_free(pcp->request);
	pcp->reqlen = 0;
	pcp->request = NULL;
	if (pcp->s <= 0)
		return PCP_OK;
	if (close(pcp->s) < 0)
		return PCP_ERR_SYSCALL;
	pcp->s = -1;
	return PCP_OK;
}

/* Get the socket file descriptor */

int
pcp_getsocket(pcp_t *pcp, int *sock)
{
	if ((pcp == NULL) || (pcp->s <= 0) || (sock == NULL))
		return PCP_ERR_INVAL;
	*sock = pcp->s;
	return PCP_OK;
}

/* Get the number of tries */

int
pcp_gettries(pcp_t *pcp, int *tries)
{
	if ((pcp == NULL) || (tries == NULL))
		return PCP_ERR_INVAL;
	*tries = pcp->tries;
	return PCP_OK;
}

/* Get the timeout for the next retry */

int
pcp_gettimeout(pcp_t *pcp, struct timeval *timeout, int relative)
{
	struct timeval now;

	if ((pcp == NULL) || (timeout == NULL))
		return PCP_ERR_INVAL;
	if (pcp->request == NULL)
		return PCP_ERR_NOREQUEST;
	timeout->tv_sec = pcp->retry.tv_sec;
	timeout->tv_usec = pcp->retry.tv_usec;
	if (relative) {
		if (gettimeofday(&now, NULL) < 0)
			return PCP_ERR_SYSCALL;
		timeout->tv_sec -= now.tv_sec;
		timeout->tv_usec = now.tv_usec;
		if (timeout->tv_usec < 0) {
			timeout->tv_sec -= 1;
			timeout->tv_usec += 1000000;
		}
	}
	return PCP_OK;
}

/* Set the reset flag */

int
pcp_setreset(pcp_t *pcp, int reset, int *old)
{
	if (pcp == NULL)
		return PCP_ERR_INVAL;
	if (old != NULL)
		*old = (int) pcp->reset;
	if (reset)
		pcp->reset = 1;
	else
		pcp->reset = 0;
	return PCP_OK;
}

/* Stack HONOR_EXTERNAL_PORT option */

int
pcp_honor_external_port(pcp_option_t ***options)
{
	static pcp_option_t o;
	static pcp_option_t *t[PCP_MAX_OPTIONS];
	pcp_option_t **p;
	int i;

	memset(t, 0, sizeof(t));
	memset(&o, 0, sizeof(o));
	o.code = PCP_OPTION_HONOR;

	if (options == NULL)
		return PCP_ERR_INVAL;
	if (*options == NULL) {
		t[0] = &o;
		*options = t;
		return PCP_OK;
	}
	p = *options;
	i = 0;
	while (*p++ != NULL) {
		if (i + 2 >= PCP_MAX_OPTIONS)
			return PCP_ERR_TOOMANYOPTS;
		i++;
	}
	*p++ = &o;
	*p = NULL;
	return PCP_OK;
}

/* Stack REMOTE_PEER_FILTER IPv4 option */

int
pcp_remote_peer_filter4(pcp_option_t ***options, uint32_t addr, uint16_t port)
{
	static pcp_option_t o;
	static uint8_t c[8];
	static pcp_option_t *t[PCP_MAX_OPTIONS];
	pcp_option_t **p;
	uint16_t n16;
	int i;

	memset(t, 0, sizeof(t));
	memset(&o, 0, sizeof(o));
	memset(&c, 0, sizeof(c));
	o.code = PCP_OPTION_FILTER;
	o.length = 8;
	o.data = c;
	n16 = htons(port);
	memcpy(c + 2, &n16, 2);
	memcpy(c + 4, &addr, 4);

	if (options == NULL)
		return PCP_ERR_INVAL;
	if (*options == NULL) {
		t[0] = &o;
		*options = t;
		return PCP_OK;
	}
	p = *options;
	i = 0;
	while (*p++ != NULL) {
		if (i + 2 >= PCP_MAX_OPTIONS)
			return PCP_ERR_TOOMANYOPTS;
		i++;
	}
	*p++ = &o;
	*p = NULL;
	return PCP_OK;
}

/* Stack REMOTE_PEER_FILTER IPv6 option */

int
pcp_remote_peer_filter6(pcp_option_t ***options, uint8_t *addr, uint16_t port)
{
	static pcp_option_t o;
	static uint8_t c[20];
	static pcp_option_t *t[PCP_MAX_OPTIONS];
	pcp_option_t **p;
	uint16_t n16;
	int i;

	memset(t, 0, sizeof(t));
	memset(&o, 0, sizeof(o));
	memset(&c, 0, sizeof(c));
	o.code = PCP_OPTION_FILTER;
	o.length = 20;
	o.data = c;
	n16 = htons(port);
	memcpy(c + 2, &n16, 2);
	memcpy(c + 4, addr, 16);

	if (options == NULL)
		return PCP_ERR_INVAL;
	if (*options == NULL) {
		t[0] = &o;
		*options = t;
		return PCP_OK;
	}
	p = *options;
	i = 0;
	while (*p++ != NULL) {
		if (i + 2 >= PCP_MAX_OPTIONS)
			return PCP_ERR_TOOMANYOPTS;
		i++;
	}
	*p++ = &o;
	*p = NULL;
	return PCP_OK;
}

/* Stack THIRD_PARTY IPv4 option */

int
pcp_third_party4(pcp_option_t ***options, uint32_t addr)
{
	static pcp_option_t o;
	static uint8_t c[4];
	static pcp_option_t *t[PCP_MAX_OPTIONS];
	pcp_option_t **p;
	int i;

	memset(t, 0, sizeof(t));
	memset(&o, 0, sizeof(o));
	memset(&c, 0, sizeof(c));
	o.code = PCP_OPTION_THIRD_PARTY;
	o.length = 4;
	o.data = c;
	memcpy(c, &addr, 4);

	if (options == NULL)
		return PCP_ERR_INVAL;
	if (*options == NULL) {
		t[0] = &o;
		*options = t;
		return PCP_OK;
	}
	p = *options;
	i = 0;
	while (*p++ != NULL) {
		if (i + 2 >= PCP_MAX_OPTIONS)
			return PCP_ERR_TOOMANYOPTS;
		i++;
	}
	*p++ = &o;
	*p = NULL;
	return PCP_OK;
}

/* Stack THIRD_PARTY IPv6 option */

int
pcp_third_party6(pcp_option_t ***options, uint8_t *addr)
{
	static pcp_option_t o;
	static uint8_t c[16];
	static pcp_option_t *t[PCP_MAX_OPTIONS];
	pcp_option_t **p;
	int i;

	memset(t, 0, sizeof(t));
	memset(&o, 0, sizeof(o));
	memset(&c, 0, sizeof(c));
	o.code = PCP_OPTION_THIRD_PARTY;
	o.length = 16;
	o.data = c;
	memcpy(c, addr, 16);

	if (options == NULL)
		return PCP_ERR_INVAL;
	if (*options == NULL) {
		t[0] = &o;
		*options = t;
		return PCP_OK;
	}
	p = *options;
	i = 0;
	while (*p++ != NULL) {
		if (i + 2 >= PCP_MAX_OPTIONS)
			return PCP_ERR_TOOMANYOPTS;
		i++;
	}
	*p++ = &o;
	*p = NULL;
	return PCP_OK;
}

static int clearrequest(pcp_t *pcp);
static int createrequest(pcp_t *pcp,
			 pcp_request_t *request,
			 pcp_option_t **options);

/* clear/create a request */

int
pcp_makerequest(pcp_t *pcp, pcp_request_t *request, pcp_option_t **options)
{
	int result;

	if (pcp == NULL)
		return PCP_ERR_INVAL;
	result = clearrequest(pcp);
	if ((request == NULL) || (result != PCP_OK))
		return result;
	return createrequest(pcp, request, options);
}

/* Clear request */

static int
clearrequest(pcp_t *pcp)
{
	if (pcp->request != NULL)
		pcp_free(pcp->request);
	pcp->reqlen = 0;
	pcp->request = NULL;
	pcp->tries = 0;
	pcp->retry.tv_sec = 0;
	pcp->retry.tv_usec = 0;
	return PCP_OK;
}

/* Create request */

static int
createrequest(pcp_t *pcp, pcp_request_t *request, pcp_option_t **options)
{
	pcp_option_t **opts = options;
	unsigned int len, off;
	uint32_t n32;
	uint16_t n16;
	int fd, ret;
	struct sockaddr_storage ss;
	struct sockaddr_in *s4 = (struct sockaddr_in *) &ss;
	struct sockaddr_in6 *s6 = (struct sockaddr_in6 *) &ss;

	ret = pcp_getsocket(pcp, &fd);
	if (ret < PCP_OK)
		return ret;
	memset(&ss, 0, sizeof(ss));
#ifdef BSD
	ss.ss_len = sizeof(ss);
#endif
	len = sizeof(ss);
	if (getsockname(fd, (struct sockaddr *) &ss, &len) != 0)
		return PCP_ERR_SYSCALL;

	switch (request->opcode) {
	case PCP_OPCODE_MAP4:
		off = len = PCP_LEN_MAP4;
		break;
	case PCP_OPCODE_MAP6:
		off = len = PCP_LEN_MAP6;
		break;
	case PCP_OPCODE_PEER4:
		off = len = PCP_LEN_PEER4;
		break;
	case PCP_OPCODE_PEER6:
		off = len = PCP_LEN_PEER6;
		break;
	default:
		return PCP_ERR_INVAL;
	}

	if (options != NULL)
		while (*opts != NULL)
			len += 4 + ((*opts++)->length);
	if (len > 1024)
		return PCP_ERR_INVAL;
	pcp->request = (uint8_t *) pcp_malloc(len);
	if (pcp->request == NULL)
		return PCP_ERR_NOMEM;
	memset(pcp->request, 0, len);
	pcp->reqlen = len;

	pcp->request[PCP_VERSION] = 1;
	pcp->request[PCP_OPCODE] = request->opcode;
	n32 = htonl(request->lifetime);
	memcpy(pcp->request + PCP_LIFETIME, &n32, 4);
	switch (ss.ss_family) {
	case AF_INET:
		memcpy(pcp->request + PCP_CLIENT_ADDR, &s4->sin_addr, 4);
		break;
	case AF_INET6:
		memcpy(pcp->request + PCP_CLIENT_ADDR, &s6->sin6_addr, 16);
		break;
	default:
		return PCP_ERR_FAILURE;
	}

	switch (request->opcode) {
	case PCP_OPCODE_MAP4:
	case PCP_OPCODE_MAP6:
		pcp->request[PCP_MAP_PROTOCOL] = request->protocol;
		n16 = htons(request->intport);
		memcpy(pcp->request + PCP_MAP_INTERNAL_PORT, &n16, 2);
		n16 = htons(request->extport);
		memcpy(pcp->request + PCP_MAP_EXTERNAL_PORT, &n16, 2);
		if (request->opcode == PCP_OPCODE_MAP4)
			memcpy(pcp->request + PCP_MAP_EXTERNAL_ADDR,
			       &request->extaddr4, 4);
		else
			memcpy(pcp->request + PCP_MAP_EXTERNAL_ADDR,
			       request->extaddr6, 16);
		break;
	case PCP_OPCODE_PEER4:
	case PCP_OPCODE_PEER6:
		pcp->request[PCP_PEER_PROTOCOL] = request->protocol;
		pcp->request[PCP_PEER_EXTERNAL_AF] = request->extaf;
		n16 = htons(request->intport);
		memcpy(pcp->request + PCP_PEER_INTERNAL_PORT, &n16, 2);
		n16 = htons(request->extport);
		memcpy(pcp->request + PCP_PEER_EXTERNAL_PORT, &n16, 2);
		switch (request->opcode) {
		case PCP_OPCODE_PEER4:
			memcpy(pcp->request + PCP_PEER_REMOTE_ADDR,
			       &request->remotepeer4, 4);
			off = PCP_PEER_REMOTE_ADDR + 4;
			break;
		case PCP_OPCODE_PEER6:
			memcpy(pcp->request + PCP_PEER_REMOTE_ADDR,
			       request->remotepeer6, 16);
			off = PCP_PEER_REMOTE_ADDR + 16;
			break;
		}
		switch (request->extaf) {
		case PCP_AF_IPv4:
			memcpy(pcp->request + off, &request->extaddr4, 4);
			break;
		case PCP_AF_IPv6:
			memcpy(pcp->request + off, &request->extaddr6, 16);
			break;
		}
		off += 16;
	}

	if (options != NULL) {
		opts = options;
		while (*opts != NULL) {
			unsigned int olen = (*opts)->length;
			
			pcp->request[off] = (*opts)->code;
			n16 = htons((*opts)->length);
			off += 2;
			memcpy(pcp->request + off, &n16, 2);
			off += 2;
			memcpy(pcp->request + off, (*opts++)->data, olen);
			off += olen;
		}
	}
	return PCP_OK;
}

/* Send/resend the current request */

int
pcp_sendrequest(pcp_t *pcp)
{
	if ((pcp == NULL) || (pcp->s <= 0))
		return PCP_ERR_INVAL;
	if (pcp->request == NULL)
		return PCP_ERR_NOREQUEST;
	if (send(pcp->s, pcp->request, pcp->reqlen, 0) < 0)
		return PCP_ERR_SEND;
	if (gettimeofday(&pcp->retry, NULL) < 0)
		return PCP_ERR_SYSCALL;
	if (pcp->tries <= PCP_MAX_TRIES)
		pcp->retry.tv_sec += PCP_RETRY_TIMER << pcp->tries;
	else
		pcp->retry.tv_sec += PCP_RETRY_TIMER << PCP_MAX_TRIES;
	pcp->tries++;
	return PCP_OK;
}

#define BAD(off)					\
	do {						\
		pcp_debug_offset = off;			\
		pcp_debug_line = __LINE__;		\
		return PCP_ERR_RECVBAD;			\
	} while (0)

/* Receive the response to the current request */

int
pcp_recvresponse(pcp_t *pcp, pcp_response_t *response)
{
	uint32_t n32;
	uint16_t n16;
	unsigned int off = 0, optidx = 0;
	int cc;

	if ((pcp == NULL) || (pcp->s <= 0) || (response == NULL))
		return PCP_ERR_INVAL;
	memset(response, 0, sizeof(pcp_response_t));
	if (pcp->response == NULL) {
		pcp->response = (uint8_t *) pcp_malloc(PCP_RESPLENGTH);
		if (pcp->response == NULL)
			return PCP_ERR_NOMEM;
	}
	memset(pcp->response, 0, PCP_RESPLENGTH);
	cc = recv(pcp->s, pcp->response, PCP_RESPLENGTH, 0);
	if (cc < 0) {
		if (errno == EWOULDBLOCK)
			return PCP_TRY_AGAIN;
		return PCP_ERR_RECV;
	} else if (cc > 1024)
		BAD(cc);
	else if ((cc & 3) != 0)
		BAD(cc);
	pcp->resplen = cc;
	pcp_debug_offset = 0;
	pcp_debug_line = __LINE__;
	if (pcp->response[PCP_VERSION] != pcp->request[PCP_VERSION])
		BAD(PCP_VERSION);
	if (pcp->response[PCP_OPCODE] !=
	    (pcp->request[PCP_OPCODE] | PCP_OPCODE_RESPONSE))
		BAD(PCP_OPCODE);
	switch (pcp->request[PCP_OPCODE]) {
	case PCP_OPCODE_MAP4:
	case PCP_OPCODE_MAP6:
	case PCP_OPCODE_PEER4:
	case PCP_OPCODE_PEER6:
		break;
	default:
		BAD(PCP_OPCODE);
	}
	response->assigned.opcode = pcp->response[PCP_OPCODE];
	response->result = pcp->response[PCP_RESULT_CODE];
	memcpy(&n32, pcp->response + PCP_EPOCH, 4);
	response->epoch = ntohl(n32);
	memcpy(&n32, pcp->response + PCP_LIFETIME, 4);
	response->assigned.lifetime = ntohl(n32);
	switch (pcp->request[PCP_OPCODE]) {
	case PCP_OPCODE_MAP4:
	case PCP_OPCODE_MAP6:
		if (pcp->response[PCP_MAP_PROTOCOL] !=
		    pcp->request[PCP_MAP_PROTOCOL])
			BAD(PCP_MAP_PROTOCOL);
		response->assigned.protocol = pcp->response[PCP_MAP_PROTOCOL];
		if (memcmp(pcp->response + PCP_MAP_INTERNAL_PORT,
			   pcp->request + PCP_MAP_INTERNAL_PORT, 2) != 0)
			BAD(PCP_MAP_INTERNAL_PORT);
		memcpy(&n16, pcp->response + PCP_MAP_INTERNAL_PORT, 2);
		response->assigned.intport = ntohs(n16);
		switch (pcp->request[PCP_OPCODE]) {
		case PCP_OPCODE_MAP4:
			memcpy(&response->assigned.extaddr4,
			       pcp->response + PCP_MAP_EXTERNAL_ADDR, 4);
			off = PCP_MAP_EXTERNAL_ADDR + 4;
			break;
		case PCP_OPCODE_MAP6:
			memcpy(&response->assigned.extaddr6,
			       pcp->response + PCP_MAP_EXTERNAL_ADDR, 16);
			off = PCP_MAP_EXTERNAL_ADDR + 16;
			break;
		}
		break;
	case PCP_OPCODE_PEER4:
	case PCP_OPCODE_PEER6:
		if (pcp->response[PCP_PEER_PROTOCOL] !=
		    pcp->request[PCP_PEER_PROTOCOL])
			BAD(PCP_PEER_PROTOCOL);
		response->assigned.protocol = pcp->response[PCP_PEER_PROTOCOL];
		response->assigned.extaf = pcp->response[PCP_PEER_EXTERNAL_AF];
		if (memcmp(pcp->response + PCP_PEER_INTERNAL_PORT,
			   pcp->request + PCP_PEER_INTERNAL_PORT, 2) != 0)
			BAD(PCP_PEER_INTERNAL_PORT);
		memcpy(&n16, pcp->response + PCP_PEER_INTERNAL_PORT, 2);
		response->assigned.intport = ntohs(n16);
		memcpy(&n16, pcp->response + PCP_PEER_EXTERNAL_PORT, 2);
		response->assigned.extport = ntohs(n16);
		if (memcmp(pcp->response + PCP_PEER_REMOTE_PORT,
			   pcp->request + PCP_PEER_REMOTE_PORT, 2) != 0)
			BAD(PCP_PEER_REMOTE_PORT);
		memcpy(&n16, pcp->response + PCP_PEER_REMOTE_PORT, 2);
		response->assigned.peerport = ntohs(n16);
		switch (pcp->request[PCP_OPCODE]) {
		case PCP_OPCODE_PEER4:
			memcpy(&response->assigned.remotepeer4,
			       pcp->response + PCP_PEER_REMOTE_ADDR, 4);
			off = PCP_PEER_REMOTE_ADDR + 4;
			break;
		case PCP_OPCODE_PEER6:
			memcpy(&response->assigned.remotepeer6,
			       pcp->response + PCP_PEER_REMOTE_ADDR, 16);
			off = PCP_PEER_REMOTE_ADDR + 16;
			break;
		}
		switch (response->assigned.extaf) {
		case PCP_AF_IPv4:
			memcpy(&response->assigned.extaddr4,
			       pcp->request + off, 4);
			off += 4;
			break;
		case PCP_AF_IPv6:
			memcpy(&response->assigned.extaddr6,
			       pcp->request + off, 16);
			off += 16;
			break;
		}
	}

	while (cc != (int) off) {
		if ((int) off > cc)
			BAD(off);
		if (optidx >= PCP_MAX_OPTIONS)
			return PCP_ERR_TOOMANYOPTS;
		response->options[optidx].code = pcp->response[off];
		off += 2;
		memcpy(&n16, pcp->response + off, 2);
		response->options[optidx].length = ntohs(n16);
		off += 2;
		response->options[optidx].data = pcp->response + off;
		off += response->options[optidx].length;
		optidx++;
	}
	if (response->result == 0)
		return PCP_OK;
	if ((response->result & 128) || !pcp->reset)
		return PCP_ERR_PROTOBASE - response->result;
	/* transient error: resetting */
	if (gettimeofday(&pcp->retry, NULL) < 0)
		return PCP_ERR_SYSCALL;
	pcp->tries = 0;
	pcp->retry.tv_sec += PCP_RESET_TIMER;
	return PCP_TRY_AGAIN;
}

/* Translate an error to its human friendly description */

const char *
pcp_strerror(int error)
{
	switch (error) {
	case PCP_OK:
		return "ok";
	case PCP_TRY_AGAIN:
		return "try again";
	case PCP_ERR_INVAL:
		return "invalid arguments";
	case PCP_ERR_NOMEM:
		return "malloc() failed";
	case PCP_ERR_SOCKET:
		return "socket() syscall failed";
	case PCP_ERR_BIND:
		return "bind() syscall failed";
	case PCP_ERR_CONNECT:
		return "connect() syscall failed";
	case PCP_ERR_SEND:
		return "send() syscall failed";
	case PCP_ERR_RECV:
		return "recv() syscall failed";
	case PCP_ERR_SYSCALL:
		return "syscall failed";
	case PCP_ERR_NOREQUEST:
		return "no current request";
	case PCP_ERR_RECVBAD:
		return "bad response";
	case PCP_ERR_TOOMANYOPTS:
		return "too many options";
	case PCP_ERR_FAILURE:
		return "internal failure";
	case PCP_ERR_APP0:
		return err_app0_msg;
	case PCP_ERR_APP1:
		return err_app1_msg;
	case PCP_ERR_APP2:
		return err_app2_msg;
	case PCP_ERR_PROTOBASE:
		return "base for protocol errors (shouldn't happen)";
	case PCP_ERR_NETFAILURE:
		return "network failure";
	case PCP_ERR_NORESOURCES:
		return "out of resources";
	case PCP_ERR_IMPLICITMAP:
		return "collides with implicit mapping";
	case PCP_ERR_UNSUPVERSION:
		return "unsupported version";
	case PCP_ERR_UNSUPOPCODE:
		return "unsupported opcode";
	case PCP_ERR_UNSUPOPTION:
		return "unsupported option";
	case PCP_ERR_BADOPTION:
		return "malformed option";
	case PCP_ERR_UNSPECIFIED:
		return "unspecified error";
	case PCP_ERR_BADREQUEST:
		return "malformed request";
	case PCP_ERR_UNSUPPROTO:
		return "unsupported protocol";
	case PCP_ERR_NOTAUTH:
		return "not authorized";
	case PCP_ERR_EXQUOTA:
		return "user exceeded quota";
	case PCP_ERR_CANTHONOR:
		return "cannot honor external port";
	case PCP_ERR_CANTDELALL:
		return "unable to delete all";
	case PCP_ERR_CANTFWDPORT0:
		return "cannot forward port zero";
	case PCP_ERR_NOPEER:
		return "peer doesn't exist";
	default:
		if (error >= PCP_ERR_PROTOBASE)
			return "unknown protocol error";
		else
			return "unknown error";
	}
}

/* Synthetic getmapping() */

int
pcp_getmapping(pcp_t *pcp,
	       pcp_request_t *request,
	       pcp_option_t **options,
	       pcp_response_t *response)
{
	int ret, fd, tries;
	fd_set set;
	struct timeval tv;

	if ((pcp == NULL) || (response == NULL))
		return PCP_ERR_INVAL;
	ret = pcp_getsocket(pcp, &fd);
	if (ret < PCP_OK)
		return ret;
	if (request != NULL)
		ret = pcp_makerequest(pcp, request, options);
	if (ret < PCP_OK)
		return ret;
	ret = pcp_sendrequest(pcp);
	if (ret < PCP_OK)
		return ret;
	for (;;) {
		ret = pcp_gettries(pcp, &tries);
		if (ret < PCP_OK)
			return ret;
		if (tries > PCP_MAX_TRIES)
			return PCP_ERR_FAILURE;
		FD_ZERO(&set);
		FD_SET(fd, &set);
		ret = pcp_gettimeout(pcp, &tv, 1);
		if (ret < PCP_OK)
			return ret;
		ret = select(fd + 1, &set, NULL, NULL, &tv);
		if (ret < 0)
			return PCP_ERR_SYSCALL;
		if (FD_ISSET(fd, &set)) {
			memset(response, 0, sizeof(*response));
			ret = pcp_recvresponse(pcp, response);
			if (ret == PCP_TRY_AGAIN)
				continue;
			else
				return ret;
		}
		ret = pcp_sendrequest(pcp);
		if (ret < PCP_OK)
			return ret;
	}
}

/* Synthetix pcp_renewmapping() */

int
pcp_renewmapping(pcp_t *pcp, pcp_response_t *response)
{
	pcp_option_t **options;
	int ret;
	pcp_response_t r;

	if ((pcp == NULL) || (response == NULL))
		return PCP_ERR_INVAL;
	options = NULL;
	ret = pcp_honor_external_port(&options);
	if (ret != PCP_OK)
		return ret;
	memcpy(&r, response, sizeof(r));
	r.result = 0;
	r.epoch = 0;
	/* optimistic! */
	r.assigned.lifetime *= 2;
	ret = pcp_getmapping(pcp, &r.assigned, options, &r);
	if (ret == PCP_OK)
		memcpy(response, &r, sizeof(r));
	return ret;
}

/* Synthetic pcp_delmapping() */

int
pcp_delmapping(pcp_t *pcp, pcp_response_t *response)
{
	int ret;
	pcp_response_t r;

	if ((pcp == NULL) || (response == NULL))
		return PCP_ERR_INVAL;
	memcpy(&r, response, sizeof(r));
	r.result = 0;
	r.epoch = 0;
	r.assigned.lifetime = 0;
	r.assigned.extport = 0;
	ret = pcp_getmapping(pcp, &r.assigned, NULL, &r);
	if (ret == PCP_OK)
		memcpy(response, &r, sizeof(r));
	return ret;
}

/* Internal set request */

int
_pcp_setrequest(pcp_t *pcp, uint8_t *request, uint16_t reqlen)
{
	int result;

	if ((pcp == NULL) || ((request == NULL) && (reqlen != 0)))
		return PCP_ERR_INVAL;
	result = clearrequest(pcp);
	if ((request == NULL) || (result != PCP_OK))
		return result;
	pcp->request = (uint8_t *) pcp_malloc((size_t) reqlen);
	if (pcp->request == NULL)
		return PCP_ERR_NOMEM;
	memcpy(pcp->request, request, (size_t) reqlen);
	pcp->reqlen = reqlen;
	return PCP_OK;
}

/* Internal get request */

int
_pcp_getrequest(pcp_t *pcp, uint8_t **request, uint16_t *reqlen)
{
	if ((pcp == NULL) || (request == NULL) || (reqlen == NULL))
		return PCP_ERR_INVAL;
	*request = pcp->request;
	*reqlen = pcp->reqlen;
	return PCP_OK;
}

/* Internal set response */

int
_pcp_setresponse(pcp_t *pcp, uint8_t *response, uint16_t resplen)
{
	if ((pcp == NULL) || (resplen > PCP_RESPLENGTH) ||
	    ((response == NULL) && (resplen != 0)))
		return PCP_ERR_INVAL;
	pcp->resplen = resplen;
	if (resplen == 0)
		return PCP_OK;
	if (pcp->response == NULL) {
		pcp->response = (uint8_t *) pcp_malloc(PCP_RESPLENGTH);
		if (pcp->response == NULL) {
			pcp->resplen = 0;
			return PCP_ERR_NOMEM;
		}
	}
	memset(pcp->response, 0, PCP_RESPLENGTH);
	memcpy(pcp->response, response, resplen);
	return PCP_OK;
}

/* Internal get response */

int
_pcp_getresponse(pcp_t *pcp, uint8_t **response, uint16_t *resplen)
{
	if ((pcp == NULL) || (response == NULL) || (resplen == NULL))
		return PCP_ERR_INVAL;
	*response = pcp->response;
	*resplen = pcp->resplen;
	return PCP_OK;
}

/* Weak definition of pcp_malloc()/pcp_free() */

#pragma weak pcp_malloc

void * pcp_malloc(size_t size) { return malloc(size); }

#pragma weak pcp_free

void pcp_free(void *ptr) { free(ptr); }

/* Weak definition of err_app[012]_msg */

#pragma weak err_app0_msg

const char *err_app0_msg = "user application error 0";

#pragma weak err_app1_msg

const char *err_app1_msg = "user application error 1";

#pragma weak err_app2_msg

const char *err_app2_msg = "user application error 2";
