/*
 * Copyright (C) 2009  Internet Systems Consortium, Inc. ("ISC")
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/* $Id: cgn.c 220 2009-07-08 22:11:38Z fdupont $ */

/*
 * Dual-Stack Lite Carrier-Grade NAT
 *
 * Francis_Dupont@isc.org, November-December 2008
 *
 * Main structures are:
 *  - tunnels (fixed indexed vector, radix tree and hash table).
 *  - fragments (shared between IPv4/IPv6 and head/fragment,
 *   ordered by creation/expire time, heads in tailq, fragments in slist,
 *   IPv4 fragments are hashed). TODO: per tunnel reordering list +
 *   hash table dedicated to unknown tunnel (very low priority as
 *   fragments are supposed to be not common).
 *  - NAT entries (aka bindings) (red-black tree, per tunnel splay
 *   tree, expire heap and hash table) (support of static bindings with
 *   wildcard destination) (protocols: TCP, UDP and ICMP echo).
 *  - natted sources (available and reserved (to static bindings)).
 *
 * Routines:
 * (Utils)
 *  - log* and *2str routines
 *  - print_iots (print I/O timestamp in "tcpdump -tt" format).
 *  - tun_read/tun_write (FreeBSD compatibility routines).
 *  - tun_open (the only OS-dependent routine, use P2P basic tun).
 *  - trace routine (create/delete tunnel/NAT events)
 * (Data structures, basic)
 *  - Jenkins hash (specialized for per structure tables) with a random key.
 *   (from Linux but not under GPL)
 *  - red-black tree of NAT entries (from FreeBSD).
 *  - splay tree of per tunnel NAT entries (from FreeBSD).
 *  - heap for NAT entry expiration (from BIND9).
 *  - radix tree for tunnels (no support for prefixes aka descovery).
 *   (NOTE: internal nodes are tunnel structures with an impossible index).
 *   (from simplified Net-Patricia, GPL2, itself from BSD)
 * (Data structures, raw lookup/add/del/set...)
 *  - new_nat (tunnel index based available address and 256 nearest ports).
 * (Data structures, debug)
 * (Data structures, commands)
 * (Data structures, lists)
 * (Commands)
 *  - cmdline (config/command line)
 *  - load (load config).
 *  - commands (execute commands).
 * (Packet utils)
 *  - checksum (very basic).
 *  - defrag/defrag6 (from simplified FreeBSD) (NOTE: no overlapping!).
 *  - patch_tcpmss (both way, syn tcp only).
 *  - tcpstate_in (detect closing)
 *  - tcpstate_out (detect closing)
 * (IN: from tunnel)
 *  - filtericmpin (ICMP from tunnel to Internet, sanity only).
 *  - naticmpin (ICMPv4 from tunnel to Internet).
 *  - natin (from tunnel to Internet, lookup into the tunnel bindings,
 *   new_nat if not found).
 *  - prrin (strip down version of natin).
 *  - filterin (IPv4 packets from tunnel, IPv6 filtering is implicit in
 *   tunnel decapsulation, sanity checks only). Source is checked for
 *   RFC 1918+I-D else PRR/A+P. (from FreeBSD)
 *  - icmp6in (ICMPv6 from tunnel, basic translation).
 *  - decap (tunnel decapsulation, tunnel lookup and sanity checks only).
 *   (NOTE: only one interface/device is needed).
 * (OUT: from Internet)
 *  - toobigout (unfragmentable from Internet, emit ICMPv4 error).
 *   TODO: rename into errorout and extend.
 *  - natout (from Internet to tunnel, hashed cached then red-black tree
 *   lookup, reorder tunnel bindings).
 *  - naticmpout (ICMPv4 from Internet to tunnel, same than natout on
 *   the included IPv4 packet, no reordering).
 *  - filtericmpout (ICMPv4 from Internet, sanity checks including error
 *   types). (from FreeBSD)
 *  - filterout (IPv4 from Internet, sanity checks, check destination
 *   (so natted) address). (from FreeBSD)
 *  - encap (tunnel encapsulation, trivial).
 *  (MAIN)
 *  - loop (main loop, instantaneous select, gettimeofday, read, for
 *   IPv6: decap, filterin, natin, write, for IPv4: filterout, nat[icmp]out,
 *   encap, then expire bindings (using the heap) and fragments, then
 *   resize hash tables). TODO: loop on not-blocking read for some packets.
 *  - init_hashes (initialize hash tables).
 *  - setup_start (call ./cgn-start script, interface up, route all
 *   natted addresses to tun0, route the local IPv6 address to tun0).
 *   (NOTE: forget the point-to-point, in particular the shared peer address).
 *  - setup_stop (call ./cgn-stop script, interface down).
 *  - main (NOTE: no argument, never reach end).
 *
 * (other) TODO:
 *  - ECN copy. (NOTE: useful? mess checksum).
 *  - profiling with real traffic (ask Alain).
 *
 * At Comcast meeting TODOs:
 *   + counters (add dropped+reasons, frags, create/delete, etc)
 *   + rates (done for pps)
 *   + fragmentation (done)
 *   + pool restriction
 *   + port bucket
 *   + logs (done, look at previous item too)
 *   + IPv6 ACL
 *   + IPv4 ACL (done, should test again A+P?)
 *   + no-nat
 *   + show counters (alias of debug stat)
 *   + debug enable/disable (done)
 *   + quit command (done, including tests)
 *   - FTP ALG (retransmits, peer matching)
 *   - per protocol NAT trees
 *
 * Testing:
 *  - checksums: should be good when used
 *  - defrag: right on correct packets
 *  - nat: lookup/add_snat correct (port redirection)
 *  - nat heap: correct
 *  - nat tree: seems to work
 *  - filterin/natin/new_nat: works (note: beware of iptables: disable them)
 *  - filtericmpin/naticmpin/icmp6in: works
 *   (ICMP dropped by Linux kernel with spurious anti-spoof if the interface
 *    address is 10.0.100.1, works without problems with 10.0.100.2 or on
 *    FreeBSD)
 *  - toobigout: same partial issue than icmp6in
 *  - filterout/natout: works (both at natin reversal and port redirection)
 *  - filtericmpout/naticmpout: works
 *  - tunnel tree: seems to work
 *  - encap: works (including fragmentation)
 *  - decap: works (including reassembly)
 *  - load: works
 *  - hairpin (i.e., client to client): works (through double mapping)
 *  - tcpmss: patched (both ways)
 *  - commands: works (including extended debug, delete, list and trace)
 *  - a+p/prr: works (don't forget to add the address)
 *  - tcp closing: works
 *
 * Control flow:
 *
 *    loop()
 *     |          /-> commands()
 *     V         /                 /-> IPv6 read
 *    select() -+---> tun_read() -+
 *     |                           \-> IPv4 read
 *     V
 *    every second -> expire (NAT heap, IPv4 fragments, IPv6 fragments)
 *     |
 *     V
 *    every 256 seconds -> resize hash tables
 *                         (caches: fragments, NAT entries, tunnels)
 *
 *    IPv6 read:
 *     |
 *     V 
 *    decap() (sanity checks (IPv6), acl6(), icmp6in(), tunnel lookup/add,
 *     |       defrag6(), tel strip)
 *     |
 *     +---> icmp6in() (sanity check, classify, tunnel lookup)
 *     |      |
 *     |      +--> set_tunnel_mtu()
 *     |      |
 *     |      V
 *     |     naticmpin() (for translation to ICMPv4)
 *     V
 *    filterin() (sanity checks (IPv4), defrag(), classify)
 *     |
 *     | (ICMP not EchoRequest, RFC 1918+I-D source, default case)
 *     +---+--+--+
 *     |   |  |  |
 *     |   |  |  \----> natin()
 *     |   |  |            |
 *     |   |  \--> prrin() |
 *     |   V         |     |
 *     |  nonatin() -+-----+-> tun_write() (translated IPv4 standard packet)
 *     V
 *    naticmpin()
 *
 *    natin()
 *     |
 *     V
 *    NAT entry lookup/add
 *     |
 *     V
 *    translation (source <- mapped, fix checksums, for TCP patch MSS,
 *     |           FTP ALG and detect state needing shorter timeouts)
 *     V
 *    housekeeping (expiration heap, per tunnel NAT entry list reordering)
 *
 *    prrin()
 *     |
 *     V
 *    (PRR/A+P) NAT entry lookup (and NAT entry list reordering)
 *
 *    nonatin()
 *     |
 *     V
 *    NO-NAT (nearly nothing)
 *
 *    naticmpin()
 *     |
 *     V
 *    ICMPv4 header stripping
 *     |
 *     V
 *    filtericmpin() (sanity check, keep types 3/11/12)
 *     |
 *     V
 *    NAT entry lookup (on triggering packet)
 *     |
 *     V
 *    translate/build ICMP header
 *     |
 *     V
 *    translate triggering packet
 *     |
 *     V
 *    tun_write() (translated ICMPv4 error packet)
 *
 *    IPv4 read
 *     |
 *     V
 *    filterout()
 *     |
 *     +--> natout() -----+-> encap() (encapsulate/fragment)
 *     |                  |    |         
 *     +--> naticmpout() -+    V
 *     |                  |   tun_write() (encapsulating IPv6 packet/fragment)
 *     \--> nonatout() ---/
 *
 *    filterout()
 *     |
 *     V
 *    sanity checks (including on destination address), defrag()
 *     |
 *     V
 *    classify --> filtericmpout() (ICMP not EchoReply, error 3/11/12,
 *     |                            sanity checks on triggering packet)
 *     \
 *      \-> drop, standard or ICMP
 *
 *    natout()
 *     |
 *     V
 *    NAT entry lookup (including PRR/A+P case detection)
 *     |
 *     +--> toobigout() (DF packet > MTU, reflect ICMPv4 toobig)
 *     |
 *     V
 *    translation (destination <- original, fix checksums, for TCP patch MSS,
 *     |           FTP ALG and detect state needing shorter timeouts)
 *     V
 *    housekeeping (expiration heap, per tunnel NAT entry list reordering)
 *
 *    naticmpout()
 *     |
 *     V
 *    NAT entry lookup (including PRR/A+P case detection)
 *     |
 *     V
 *    translation (ICMP and triggering packet)
 *
 *    nonatout()
 *     |
 *     V
 *    NO-NAT (nearly nothing)
 */

#ifndef __linux__
#include <sys/types.h>
#include <sys/uio.h>
#endif
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/time.h>

#ifndef __linux__
#include <net/if.h>
#include <net/if_tun.h>
#else
#include <linux/if.h>
#include <linux/if_tun.h>
#ifndef ETH_P_IP
#include <linux/if_ether.h>
#endif
#endif

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdarg.h>
#include <netdb.h>

/* Offsets and interesting constants for IPv4, ICMPv4, UDP, TCP,
 * IPv6 and v6 Fragment structures */

#define IPTOS		1
#define IPLENH		2
#define IPLENL		3
#define IPID		4
#define IPOFFH		6
#define IPOFFL		7
#define IPTTL		8
#define IPPROTO		9
#define IPCKSUMH	10
#define IPCKSUML	11
#define IPSRC		12
#define IPDST		16
#define IPSPORT		20
#define IPDPORT		22

#define ICMPTYPE	20
#define ICMPCODE	21
#define ICMPCKSUMH	22
#define ICMPCKSUML	23
#define ICMPID		24
#define IP2		28
#define IP2LENH		(IP2 + IPLENH)
#define IP2LENL		(IP2 + IPLENL)
#define IP2OFFH		(IP2 + IPOFFH)
#define IP2OFFL		(IP2 + IPOFFL)
#define IP2PROTO	(IP2 + IPPROTO)
#define IP2CKSUMH	(IP2 + IPCKSUMH)
#define IP2CKSUML	(IP2 + IPCKSUML)
#define IP2SRC		(IP2 + IPSRC)
#define IP2DST		(IP2 + IPDST)
#define IP2SPORT	(IP2 + IPSPORT)
#define IP2DPORT	(IP2 + IPDPORT)

#define UDPLEN		(IPHDRLEN + 4)
#define UDPCKSUMH	(IPHDRLEN + 6)
#define UDPCKSUML	(IPHDRLEN + 7)

#define TCPSEQ		(IPHDRLEN + 4)
#define TCPACK		(IPHDRLEN + 8)
#define TCPOFF		(IPHDRLEN + 12)
#define TCPFLAGS	(IPHDRLEN + 13)
#define TCPCKSUMH	(IPHDRLEN + 16)
#define TCPCKSUML	(IPHDRLEN + 17)

#define IPVERMSK	0xf0
#define IP4V		0x40
#define IP4VNOOP	0x45
#define IPHDRLEN	20
#define IPMINLEN	28
#define IPMAXLEN	65535
#define IPDF		0x40
#define IPMF		0x20
#define IPOFFMSK	0x1f

#define IPICMP		1
#define IPTCP		6
#define IPUDP		17

#define PORTFTP		21

#define ICMPECHREP	0
#define ICMPECHREQ	8

#define TCPHDRLEN	20
#define TCPOFFMSK	0xf0
#define TCPFFIN		0x01
#define TCPFSYN		0x02
#define TCPFRST		0x04
#define TCPFACK		0x10
#define TCPOPTEOL	0
#define TCPOPTNOP	1
#define TCPOPTMSS	2
#define TCPOPTMSSLEN	4
#define TCPOPTMD5	19

#define IP6V		0x60
#define IP6LENH		4
#define IP6LENL		5
#define IP6PROTO	6
#define IP6TTL		7
#define IP6SRC		8
#define IP6DST		24
#define IP6HDRLEN	40

#define IP6FPROTO	40
#define IP6FOFFH	42
#define IP6FOFFL	43
#define IP6FID		44

#define IP6FLEN		(IP6HDRLEN + 8)
#define IP6FMF		0x01
#define IP6FMSK		0xf8

#define ICMP6TYPE	40
#define ICMP6CODE	41
#define ICMP6CKSUMH	42
#define ICMP6CKSUML	43
#define ICMP6PTR	44
#define IP62PROTO	54
#define IP62SRC		56
#define IP62DST		72
#define IP64		88
#define IP64PROTO	(IP64 + IPPROTO)

#define IP6OTEL		0x04

#define IP6IP4		4
#define IP6FRAG		44
#define IP6DSTOP	50
#define IP6ICMP		58

struct tunnel;

#define ICMPMAXLEN	200

/* file names */

#define SETUP_START	"./cgn-start"
#define SETUP_STOP	"./cgn-stop"
#define CGNCONFIG	"cgn-config"

/* exp(-1/{60,300,900}) for 1mn, 5mn, 15mn decays */
#ifndef DECAY1
#define DECAY1	.98347145382161748948
#endif
#ifndef DECAY5
#define DECAY5	.99667221605452332152
#endif
#ifndef DECAY15
#define DECAY15	.99888950594427931322
#endif

char tunname[64];
int tunfd;

FILE *ftrace;

#define DR_BAD6		0	/* bad IPv6 packet */
#define DR_ACL6		1	/* filtered by IPv6 ACL */
#define DR_NOTUN	2	/* no tunnel */
#define DR_ICMP6	3	/* bad/uninteresting ICMPv6 packet */
#define DR_BADIN	4	/* bad IPv4 in packet */
#define DR_NOPRR	5	/* bad PRR IPv4 in packet */
#define DR_NATCNT	6	/* too many NAT */
#define DR_NEWNAT	7	/* can't create a new NAT */
#define DR_ICMPIN	8	/* bad/uninteresting ICMPv4 in packet */
#define DR_BADOUT	9	/* bad IPv4 out packet */
#define DR_DSTOUT	10	/* bad destination address, out packet */
#define DR_ICMPOUT	11	/* bad/uninteresting ICMPv4 out packet */
#define DR_NATOUT	12	/* no NAT matching IPv4 out packet */
#define DR_TOOBIG	13	/* DF too big IPv4 out packet */
#define DR_F6CNT	14	/* too many IPv6 fragments */
#define DR_F6TCNT	15	/* too many IPv6 fragments per tunnel */
#define DR_BADF6	16	/* bad IPv6 fragment */
#define DR_F6TM		17	/* IPv6 fragment timeout */
#define DR_FINCNT	18	/* too many IPv4 in fragments */
#define DR_FINTCNT	19	/* too many IPv4 in fragments per tunnel */
#define DR_FOUTCNT	20	/* too many IPv4 out fragments */
#define DR_BADF4	21	/* bad IPv4 fragment */
#define DR_F4MEM	22	/* IPv4 fragment alloc pb */
#define DR_FINTM	23	/* IPv4 in fragment timeout */
#define DR_FOUTTM	24	/* IPv4 out fragment timeout */
#define DR_MAX		24	/* max dropped reason */

char *dropreason[DR_MAX + 1] = {
	"bad IPv6 packet",
	"filtered by IPv6 ACL",
	"no tunnel",
	"bad/uninteresting ICMPv6 packet",
	"bad IPv4 'in' packet",
	"bad PRR IPv4 'in' packet",
	"too many NAT entries for tunnel",
	"can't create a new NAT entry",
	"bad/uninteresting ICMPv4 'in' packet",
	"bad IPv4 'out' packet",
	"'out' packet with a bad destination address",
	"bad/uninteresting ICMPv4 'out' packet",
	"no NAT matching IPv4 'out' packet",
	"DF and too big IPv4 'out' packet",
	"too many IPv6 fragments",
	"too many IPv6 fragments per tunnel",
	"bad IPv6 fragment",
	"IPv6 fragment timeout",
	"too many IPv4 'in' fragments",
	"too many IPv4 'in' fragments per tunnel",
	"too many IPv4 'out' fragments",
	"bad IPv4 fragment",
	"IPv4 fragment allocation failure",
	"IPv4 'in' fragment timeout",
	"IPv4 'out' fragment timeout"
};

uint64_t statsrcv6, statsrcv4, statssent6, statssent4;
uint64_t statsfrgin6, statsfrgin, statsfrout, statsfrgout6;
uint64_t statsreas6, statsreasin, statsreasout;
uint64_t statsnatin, statsprrin, statsnonatin;
uint64_t statsnaticmpin6, statsnaticmpin4;
uint64_t statsnatout, statsprrout, statsnonatout, statsnaticmpout;
uint64_t statstcpmss, statsmsspatched, statstoobig;
uint64_t statsftpport, statsftpeprt, statsftp227, statsftp229;
uint64_t statscnat, statsdnat, statscbuck, statsdbuck;
uint64_t statsdropped[DR_MAX + 1];

uint64_t debugrcv6, debugrcv4, debugsent6, debugsent4;
uint64_t debugfrgin6, debugfrgin, debugfrgout6, debugreas6, debugreasin;
uint64_t debugnatin, debugprrin, debugnonatin;
uint64_t debugnaticmpin6, debugnaticmpin4;
uint64_t debugnatout, debugprrout, debugnonatout, debugnaticmpout;
uint64_t debugtcpmss, debugmsspatched, debugtoobig;
uint64_t debugftpport, debugftpeprt, debugftp227, debugftp229;
uint64_t debugcnat, debugdnat, debugcbuck, debugdbuck;
uint64_t debugdropped[DR_MAX + 1];

uint64_t lastrcv6, lastrcv4, lastsent6, lastsent4;
uint64_t lastcnat, lastdnat;
double ratercv6[3], ratercv4[3], ratesent6[3], ratesent4[3];
double ratecnat[3], ratednat[3];

u_char buf4[66000], buf6[66000], buf[1500];
u_int len;

time_t seconds, lastsecs, startsecs;

u_char icmpsrc[4];			/* interface address (for ICMP) */
u_char local6[16];			/* local IPv6 address */

int eqfrag = 0;
int use_autotunnel = 1;
int debuglevel = 0;

#define FRAG_LIFETIME	30
#ifndef FRAG6_MAXCNT
#define FRAG6_MAXCNT	1024
#endif
#ifndef FRAGIN_MAXCNT
#define FRAGIN_MAXCNT	1024
#endif
#ifndef FRAGOUT_MAXCNT
#define FRAGOUT_MAXCNT	1024
#endif
#ifndef FRAGTN6_MAXCNT
#define FRAGTN6_MAXCNT	16
#endif
#if (FRAGTN6_MAXCNT > 255)
#error "FRAGTN6_MAXCNT on 8 bits"
#endif
#ifndef FRAGTN4_MAXCNT
#define FRAGTN4_MAXCNT	64
#endif
#if (FRAGTN4_MAXCNT > 255)
#error "FRAGTN4_MAXCNT on 8 bits"
#endif

struct frag {				/* fragment entry */
	struct frag *next;		/* chaining (slist/tailq) */
	struct frag **prev;		/* tailq (first only) */
	struct frag *list;		/* in packet fragment (first only) */
	struct tunnel *tunnel;		/* reference to tunnel (or NULL) */
	u_char *buf;			/* buffer copy */
	u_int len;			/* buffer (full) length */
	u_int off;			/* fragment offset */
	time_t expire;			/* expiration date (first only) */
	u_short hash;			/* last hash (first only) */
	u_char more;			/* more flag */
};
struct frag *frags6_first;		/* IPv6 global fragment tailq */
struct frag **frags6_last = &frags6_first;
u_int frags6cnt;			/* IPv6 fragment count */
struct frag *fragsin_first;		/* IPv4 in global fragment tailq */
struct frag **fragsin_last = &fragsin_first;
u_int fragsincnt;			/* IPv4 in fragment count */
struct frag *fragsout_first;		/* IPv4 out global fragment tailq */
struct frag **fragsout_last = &fragsout_first;
u_int fragsoutcnt;			/* IPv4 out fragment count */

#define TCPPR	0
#define UDPPR	1
#define ICMPPR	2
#define PRCNT	3

#ifndef TCP_MINPORT
#define TCP_MINPORT	2048
#endif
#ifndef UDP_MINPORT
#define UDP_MINPORT	512
#endif
#ifndef ICMP_MINID
#define ICMP_MINID	0
#endif
#ifndef TCP_MAXPORT
#define TCP_MAXPORT	65535
#endif
#ifndef UDP_MAXPORT
#define UDP_MAXPORT	65535
#endif
#ifndef ICMP_MAXID
#define ICMP_MAXID	65535
#endif
#if (TCP_MINPORT <= 0) || (TCP_MAXPORT > 65535) || (TCP_MINPORT > TCP_MAXPORT)
#error "bad TCP_[MIN|MAX]PORT"
#endif
#if (UDP_MINPORT <= 0) || (UDP_MAXPORT > 65535) || (UDP_MINPORT > UDP_MAXPORT)
#error "bad UDP_[MIN|MAX]PORT"
#endif
#if (ICMP_MINID < 0) || (ICMP_MAXID > 65535) || (ICMP_MINID > ICMP_MAXID)
#error "bad ICMP_[MIN|MAX]ID"
#endif

struct natsrc {				/* IPv4 addresses for NAT */
	u_char addr[4];
	struct bucket *ffirst[PRCNT];	/* bucket free lists (first) */
	struct bucket **flast[PRCNT];	/* bucket free lists (tail) */
	u_short buckcnt[PRCNT];		/* inuse bucket counters */
	u_short natcnt[PRCNT];		/* inuse dynamic NAT entry counters */
	u_short minport[PRCNT];		/* min ports */
	u_short maxport[PRCNT];		/* max ports */
	u_short curport[PRCNT];		/* initial allocator */
};

struct natsrc **natsrcs;		/* NATted source addresses */
u_int natsrccnt;			/* NATted source address count */

#ifndef TCPBUCKSZ
#define TCPBUCKSZ	10
#endif
#ifndef UDPBUCKSZ
#define UDPBUCKSZ	8
#endif
#ifndef ICMPBUCKSZ
#define ICMPBUCKSZ	3
#endif
#if (TCPBUCKSZ <= 0) || (UDPBUCKSZ <= 0) || (ICMPBUCKSZ <= 0)
#error "bucket sizes must be > 0"
#endif
#if (TCPBUCKSZ > 255) || (UDPBUCKSZ > 255) || (ICMPBUCKSZ > 255)
#error "bucket sizes must be <= 255"
#endif

struct bucket {				/* port/id bucket */
	struct bucket *next;		/* bucket chaining */
	struct bucket **prev;
	u_int number;			/* number (== first port/id) */
	u_char proto;			/* protocol */
	u_char avail;			/* number of free NAT entries */
	u_char freebm[0];		/* free bitmap */
};

#define ALL_DST			1
#define PRR_NULL		2
#define MATCH_PORT		4
#define MATCH_ICMP		8
#define MATCH_ANY		12
#define ALL_FLAGS		15

#define TCP_LIFETIME		600
#define CLOSED_TCP_LIFETIME	120
#define UDP_LIFETIME		300
#define ICMP_LIFETIME		30
#define RETRANS_LIFETIME	10

#define TCP_DEFAULT		0
#define TCP_ACKED		1
#define TCP_CLOSED_IN		2
#define TCP_CLOSED_OUT		4
#define TCP_CLOSED_BOTH		6

#define RB_BLACK	0
#define RB_RED		1

struct nat {				/* NAT entry */
	struct nat *left;		/* NAT red-black tree */
	struct nat *right;
	struct nat *parent;
	struct nat *tleft;		/* per tunnel splay tree */
	struct nat *tright;
	struct nat *xfirst;		/* FTP ALG chain */
	struct nat *xnext;
	struct nat **xprev;
	struct bucket *bucket;		/* back pointer to bucket */
	struct tunnel *tunnel;		/* pointer to tunnel */
	time_t timeout;			/* timeout date (0 == infinity) */
	u_int lifetime;			/* extra lifetime on match */
	u_int heap_index;		/* index in heap */
					/* from tunnel to Internet */
	u_char src[4];			/* (original) source address */
	u_char nsrc[4];			/* (natted) source address */
	u_char dst[4];			/* destination address */
	u_char sport[2];		/* (original) source port */
	u_char nport[2];		/* (natted) source port */
	u_char dport[2];		/* destination port */
	u_char proto;			/* protocol TCP|UDP|ICMP */
	u_char flags;			/* flags */
	u_short hash;			/* last hash */
	u_char tcpst;			/* TCP state */
	u_char color;			/* RB_BLACK | RB_RED */
	uint32_t ftpseqd;		/* FTP TCP sequence delta */
};
struct nat *nat_tree;			/* red-black tree root */
u_int nat_heap_size;			/* current heap size */
u_int nat_heap_last;			/* heap last index */
struct nat **nat_heap;			/* heap array of NAT entry pointers */

u_int tnatcnt, unatcnt, onatcnt, snatcnt, prrcnt;	/* global counters */

#ifndef MAXNATCNT
#define MAXNATCNT	6000		/* maximum count of NAT entries */
#endif
#if (MAXNATCNT <= 0) || (MAXNATCNT > 65535)
#error "bad MAXNATCNT value"
#endif
#define MAXTUNBIT	128		/* 16*8 bits of key */
#define TUNDEFMTU	1500		/* default MTU */
#define TUNMINMTU	1280		/* minimal MTU */
#define TUNMSSFLG	0x01		/* patch TCP MSS flag */
#define TUNTBDROP	0x02		/* drop too big DF packets */
#define TUNTBICMP	0x04		/* send too big ICMPv4 */
#define TUNDEBUG	0x08		/* debug is enabled */
#define TUNNONAT	0x10		/* no-nat tunnel */
#define TUNGLUE		0x80		/* unused but still present in tree */

struct tunnel {				/* tunnel entry */
	struct tunnel *parent;		/* tunnel tree */
	struct tunnel *left;
	struct tunnel *right;
	u_char remote[16];		/* remote IPv6 address */
#define key	remote
	u_int bit;			/* internal node branch bit */
	u_short srcidx;			/* index into natsrcs[] */
	u_char frg6cnt;			/* IPv6 fragment count */
	u_char frg4cnt;			/* IPv4 in fragment count */
	union {
		struct bucket *_blist;	/* bucket list */
		struct tunnel *_nnext;	/* no-nat chain */
	} _u1;
#define blist		_u1._blist
#define nnext		_u1._nnext
	union {
		struct nat *_tnat_root;	/* NAT entries */
		u_char _nnaddr[4];	/* no-nat prefix */
	} _u2;
#define tnat_root	_u2._tnat_root
#define nnaddr		_u2._nnaddr
	u_short buckcnt;		/* bucket count */
#define nnplen		buckcnt
	u_short natcnt;			/* NAT entry count */
	u_short hash;			/* last hash */
	u_short mtu;			/* tunnel MTU */
	u_char flags;			/* flags */
};
struct tunnel *tunnel_tree;		/* tunnel tree */
struct tunnel *tunnel_debugged;		/* the tunnel under debug */
struct tunnel *nonat_first;		/* no-nat head */
struct tunnel **nonat_last = &nonat_first;
int tunnel_configured;			/* for before any tunnel configs */
u_short tundefmtu = TUNDEFMTU;		/* tunnel default MTU (configurable) */
int enable_msspatch = 0;		/* whether to patch MSS (def. cfg.) */
int default_toobig = TUNTBICMP;		/* default policy about too big */

u_int tuncnt;				/* global counter */

struct acl6 {
	struct acl6 *next;		/* chain */
	u_char addr[16];		/* prefix */
	u_char mask[16];		/* mask */
};
struct acl6 *acl6_first;		/* (stailq) acl6 list */
struct acl6 **acl6_last = &acl6_first;

#define MAXFRAGHASH	256		/* max v4 fragment hash table size */
#define MINFRAGHASH	64		/* min v4 fragment hash table size */

struct frag **fraghash;			/* v4 fragment hash table */
u_int fraghashsz;
uint32_t fraghashrnd;
uint64_t tfhlookups, tfhhits;		/* total lookups/hits counters */
uint64_t pfhlookups, pfhhits;		/* period lookups/hits counters */

#define MAXNATHASH	65536		/* max NAT hash table size */
#define MINNATHASH	1024		/* min NAT hash table size */

struct nat **nathash;			/* NAT hash table */
u_int nathashsz;
uint32_t nathashrnd;
uint64_t tnhlookups, tnhhits;		/* total lookups/hits counters */
uint64_t pnhlookups, pnhhits;		/* period lookups/hits counters */

#define MAXTUNHASH	16384		/* max tunnel hash table size */
#define MINTUNHASH	16		/* min tunnel hash table size */

struct tunnel **tunhash;		/* tunnel hash table */
u_int tunhashsz;
uint32_t tunhashrnd;
uint64_t tthlookups, tthhits;		/* total lookups/hits counters */
uint64_t pthlookups, pthhits;		/* period lookups/hits counters */

int tunnel_configured;

struct cmd {
	char *name;
	u_int len;			/* could be populated at startup */
	int required_args;
	int (*func)(char *line, char *usage);
	char *usage;
};

int cmd_acl6(char *line, char *usage);
int cmd_address(char *line, char *usage);
int cmd_autotunnel(char *line, char *usage);
int cmd_debug(char *line, char *usage);
int cmd_delete(char *line, char *usage);
int cmd_defmtu(char *line, char *usage);
int cmd_eqfrag(char *line, char *usage);
int cmd_help(char *line, char *usage);
int cmd_list(char *line, char *usage);
int cmd_mtu(char *line, char *usage);
int cmd_mss(char *line, char *usage);
int cmd_nat(char *line, char *usage);
int cmd_nonat(char *line, char *usage);
int cmd_noop(char *line, char *usage);
int cmd_port(char *line, char *usage);
int cmd_prr(char *line, char *usage);
int cmd_quit(char *line, char *usage);
int cmd_show(char *line, char *usage);
int cmd_toobig(char *line, char *usage);
int cmd_tunnel(char *line, char *usage);
int cmd_trace(char *line, char *usage);

struct cmd cmd[] = {
	/* MUST REMAIN IN ALPHABETICAL ORDER */
	{ "acl6",	4, 1, cmd_acl6,
	  "<IPv6>/<prefix_length>" },
	{ "address",	7, 1, cmd_address,
	  "<IPv4>|interface <IPv4> <IPv6>" },
	{ "autotunnel",10, 1, cmd_autotunnel,
	  "on|off" },
	{ "debug",	5, 1, cmd_debug,
	  "set|address|disable|dropped|enable|fragment|hash|nat|stat|tunnel" },
	{ "delete",	6, 1, cmd_delete,
	  "nat|prr|tunnel" },
	{ "defmtu",	6, 1, cmd_defmtu,
	  "<mtu>" },
	{ "eqfrag",	6, 1, cmd_eqfrag,
	  "on|off" },
	{ "help",	4, 0, cmd_help,
	  "" },
	{ "list",	4, 1, cmd_list,
	  "nat|tunnel" },
	{ "mtu",	3, 1, cmd_mtu,
	  "<IPv6> <mtu>" },
	{ "mss",	3, 1, cmd_mss,
	  "[IPv6] on|off" },
	{ "nat",	3, 1, cmd_nat,
	  "<IPv6> tcp|udp <IPv4_src> <port_src> <IPv4_new> <port_new>" },
	{ "nonat",	5, 1, cmd_nonat,
	  "<IPv6> <IPv4>/<prefix_length>" },
	{ "noop",	4, 0, cmd_noop,
	  "" },
	{ "port",	4, 1, cmd_port,
	  "<IPv4> tcp|udp|echo <min>-<max>" },
	{ "prr",	3, 1, cmd_prr,
	  "<IPv6> tcp|udp <IPv4> <port>" },
	{ "quit",	4, 0, cmd_quit,
	  "" },
	{ "show",	4, 1, cmd_show,
	  "dropped|stat" },
	{ "toobig",	6, 1, cmd_toobig,
	  "[<IPv6>] on|off|strict" },
	{ "tunnel",	6, 1, cmd_tunnel,
	  "<IPv6>" },
	{ "trace",	5, 1, cmd_trace,
	  "<filename>" },
	{ NULL,		0, 0, NULL,
	  "" }
};

int cmd_debug_address(char *line, char *usage);
int cmd_debug_bucket(char *line, char *usage);
int cmd_debug_disable(char *line, char *usage);
int cmd_debug_dropped(char *line, char *usage);
int cmd_debug_enable(char *line, char *usage);
int cmd_debug_fragment(char *line, char *usage);
int cmd_debug_hash(char *line, char *usage);
int cmd_debug_nat(char *line, char *usage);
int cmd_debug_set(char *line, char *usage);
int cmd_debug_stat(char *line, char *usage);
int cmd_debug_tunnel(char *line, char *usage);

struct cmd debugcmd[] = {
	/* MUST REMAIN IN ALPHABETICAL ORDER */
	{ "address",	7, 0, cmd_debug_address,  "" },
	{ "bucket",	6, 1, cmd_debug_bucket,   "<addr>" },
	{ "disable",	7, 0, cmd_debug_disable,  "[clear]" },
	{ "dropped",	7, 0, cmd_debug_dropped,  "" },
	{ "enable",	6, 1, cmd_debug_enable,   "<addr>" },
	{ "fragment",	8, 1, cmd_debug_fragment, "IPv6|in|out|<addr>" },
	{ "hash",	4, 0, cmd_debug_hash,	  "" },
	{ "nat",	3, 0, cmd_debug_nat,	  "[<addr>]" },
	{ "set",	3, 0, cmd_debug_set,	  "[<level>]" },
	{ "stat",	4, 0, cmd_debug_stat,	  "" },
	{ "tunnel",	6, 0, cmd_debug_tunnel,	  "[<IPv6>]" },
	{ NULL,		0, 0, NULL,		  "" }
};

int cmd_delete_acl6(char *line, char *usage);
int cmd_delete_nat(char *line, char *usage);
int cmd_delete_nonat(char *line, char *usage);
int cmd_delete_prr(char *line, char *usage);
int cmd_delete_tunnel(char *line, char *usage);

struct cmd deletecmd[] = {
	/* MUST REMAIN IN ALPHABETICAL ORDER */
	{ "acl6",	4, 1, cmd_delete_acl6,
	  "<IPv6>" },
	{ "nat",	3, 1, cmd_delete_nat,
	  "<IPv6> tcp|udp <IPv4> <port>" },
	{ "nonat",	5, 1, cmd_delete_nonat,
	  "<IPv6>" },
	{ "prr",	3, 1, cmd_delete_prr,
	  "<IPv6> tcp|udp <IPv4> <port>" },
	{ "tunnel",	6, 1, cmd_delete_tunnel,
	  "<IPv6>" },
	{ NULL,		0, 0, NULL,
	  "" }
};

int cmd_list_acl6(char *line, char *usage);
int cmd_list_address(char *line, char *usage);
int cmd_list_nat(char *line, char *usage);
int cmd_list_nonat(char *line, char *usage);
int cmd_list_tunnel(char *line, char *usage);

struct cmd listcmd[] = {
	/* MUST REMAIN IN ALPHABETICAL ORDER */
	{ "acl6",	4, 0, cmd_list_acl6,
	  "" },
	{ "address",	7, 0, cmd_list_address,
	  "" },
	{ "nat",	3, 0, cmd_list_nat,
	  "[conf|static|prr|dynamic|all]" },
	{ "nonat",	5, 0, cmd_list_nonat,
	  "" },
	{ "tunnel",	6, 0, cmd_list_tunnel,
	  "" },
	{ NULL,		0, 0, NULL,
	  "" }
};

int cmd_show_dropped(char *line, char *usage);
int cmd_show_stat(char *line, char *usage);

struct cmd showcmd[] = {
	/* MUST REMAIN IN ALPHABETICAL ORDER */
	{ "dropped",	7, 0, cmd_debug_dropped,  "" },
	{ "stat",	4, 0, cmd_debug_stat,	  "" },
	{ NULL,		0, 0, NULL,		  "" }
};

const u_char mask4[33][4] = {
	{ 0x00, 0x00, 0x00, 0x00 },
	{ 0x80, 0x00, 0x00, 0x00 },
	{ 0xc0, 0x00, 0x00, 0x00 },
	{ 0xe0, 0x00, 0x00, 0x00 },
	{ 0xf0, 0x00, 0x00, 0x00 },
	{ 0xf8, 0x00, 0x00, 0x00 },
	{ 0xfc, 0x00, 0x00, 0x00 },
	{ 0xfe, 0x00, 0x00, 0x00 },
	{ 0xff, 0x00, 0x00, 0x00 },
	{ 0xff, 0x80, 0x00, 0x00 },
	{ 0xff, 0xc0, 0x00, 0x00 },
	{ 0xff, 0xe0, 0x00, 0x00 },
	{ 0xff, 0xf0, 0x00, 0x00 },
	{ 0xff, 0xf8, 0x00, 0x00 },
	{ 0xff, 0xfc, 0x00, 0x00 },
	{ 0xff, 0xfe, 0x00, 0x00 },
	{ 0xff, 0xff, 0x00, 0x00 },
	{ 0xff, 0xff, 0x80, 0x00 },
	{ 0xff, 0xff, 0xc0, 0x00 },
	{ 0xff, 0xff, 0xe0, 0x00 },
	{ 0xff, 0xff, 0xf0, 0x00 },
	{ 0xff, 0xff, 0xf8, 0x00 },
	{ 0xff, 0xff, 0xfc, 0x00 },
	{ 0xff, 0xff, 0xfe, 0x00 },
	{ 0xff, 0xff, 0xff, 0x00 },
	{ 0xff, 0xff, 0xff, 0x80 },
	{ 0xff, 0xff, 0xff, 0xc0 },
	{ 0xff, 0xff, 0xff, 0xe0 },
	{ 0xff, 0xff, 0xff, 0xf0 },
	{ 0xff, 0xff, 0xff, 0xf8 },
	{ 0xff, 0xff, 0xff, 0xfc },
	{ 0xff, 0xff, 0xff, 0xfe },
	{ 0xff, 0xff, 0xff, 0xff }
};

const u_char mask6[129][16] = {
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe },
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }
};

/*
 * Utils
 */

/* logging, mainly for debug */

void
logit(int loglvl, char *msg, ...)
{
	va_list va;

	if (loglvl <= debuglevel) {
		va_start(va, msg);
		fprintf(stderr, "LOG: ");
		vfprintf(stderr, msg, va);
		fprintf(stderr, "\n");
		va_end(va);
	}
}

/* replacements for 'fprintf(stderr, ...\n)' */

void
logerr(char *msg, ...)
{
	va_list va;

	va_start(va, msg);
	vfprintf(stderr, msg, va);
	va_end(va);
}

void
loginfo(char *msg, ...)
{
	va_list va;

	va_start(va, msg);
	vfprintf(stderr, msg, va);
	va_end(va);
}

/* convert to string (print helpers) */

char *
addr2str(int family, u_char *addr)
{
	static char addrbuf[8][NI_MAXHOST]; /* XXX ugly thread-unsafe hack */
	static int round = 0;
	char *cp;

	round = (round + 1) & 7;
	cp = addrbuf[round];

	/* XXX: assume this succeeds */
	inet_ntop(family, addr, cp, NI_MAXHOST);

	return (cp);
}

char *
proto2str(u_int proto)
{
	static char buf[16];

	/* sort by likely frequency, not by absolute value */
	switch (proto) {
	case IPTCP:
		return "tcp";
	case IPUDP:
		return "udp";
	case IPICMP:
		return "icmp";
	default:
		sprintf(buf, "%u", proto);
		return buf;
	}
}

char *
toobig2str(struct tunnel *t)
{
	switch (t->flags & (TUNTBDROP | TUNTBICMP)) {
	case 0:
		return "off";
	case TUNTBICMP:
		return "on";
	case (TUNTBDROP | TUNTBICMP):
		return "strict";
	default:
		return "drop-only";
	}
}

/* Print tunnel interface/device I/O time stamps (tcpdump -tt format) */

void
print_iots(char *way, ssize_t cc)
{
	struct timeval tv;

	(void) gettimeofday(&tv, NULL);
	loginfo("%u.%06u %s %u\n",
		(unsigned) tv.tv_sec, (unsigned) tv.tv_usec,
		way, (unsigned) cc);
}

/* For FreeBSD compatibility */

ssize_t
tun_read(void *b, size_t c)
{
#ifndef __linux__
	struct iovec iov[2];
	uint32_t ifh;
	ssize_t cc;

	ifh = 0;
	iov[0].iov_base = &ifh;
	iov[0].iov_len = sizeof(ifh);
	iov[1].iov_base = b;
	iov[1].iov_len = c;
	cc = readv(tunfd, iov, 2);
	if (cc < 0)
		return cc;
	if ((size_t) cc < sizeof(ifh))
		return 0;
	cc -= sizeof(ifh);
	if (debuglevel > 10)
		print_iots("in", cc);
	return cc;
#else
#ifdef USE_TUN_PI
	struct iovec iov[2];
	struct tun_pi pi;
	ssize_t cc;

	pi.flags = 0;
	pi.proto = 0;
	iov[0].iov_base = &pi;
	iov[0].iov_len = sizeof(pi);
	iov[1].iov_base = b;
	iov[1].iov_len = c;
	cc = readv(tunfd, iov, 2);
	if (cc < 0)
		return cc;
	if ((size_t) cc < sizeof(pi))
		return 0;
	cc -= sizeof(pi);
	if (debuglevel > 10)
		print_iots("in", cc);
	return cc;
#else
	ssize_t cc;

	cc = read(tunfd, b, c);
	if ((cc > 0) && (debuglevel > 10))
		print_iots("in", cc);
	return cc;
#endif
#endif
}

ssize_t
tun_write(int af, void *b, size_t c)
{
#ifndef __linux__
	struct iovec iov[2];
	uint32_t ifh;
	ssize_t cc;

	ifh = htonl((uint32_t) af);
	iov[0].iov_base = &ifh;
	iov[0].iov_len = sizeof(ifh);
	iov[1].iov_base = b;
	iov[1].iov_len = c;
	cc = writev(tunfd, iov, 2);
	if (cc < 0)
		return cc;
	if ((size_t) cc < sizeof(ifh))
		return 0;
	cc -= sizeof(ifh);
	if (debuglevel > 10)
		print_iots("out", cc);
	return cc;
#else
#ifdef USE_TUN_PI
	struct iovec iov[2];
	struct tun_pi pi;
	ssize_t cc;

	pi.flags = 0;
	if (af == AF_INET6)
		pi.proto = htons(ETH_P_IPV6);
	else
		pi.proto = htons(ETH_P_IP);
	iov[0].iov_base = &pi;
	iov[0].iov_len = sizeof(pi);
	iov[1].iov_base = b;
	iov[1].iov_len = c;
	cc = writev(tunfd, iov, 2);
	if (cc < 0)
		return cc;
	if ((size_t) cc < sizeof(pi))
		return 0;
	cc -= sizeof(pi);
	if (debuglevel > 10)
		print_iots("out", cc);
	return cc;
#else
	ssize_t cc;

	(void) af;
	cc = write(tunfd, b, c);
	if ((cc > 0) && (debuglevel > 10))
		print_iots("out", cc);
        return cc;
#endif
#endif
}

/* Open tun interface/device (OS dependent) */

int
tun_open(void)
{
	int fd = -1;
#ifndef __linux__
	int i;

	for (i = 0; i <= 255; i++) {
		snprintf(tunname, sizeof(tunname), "/dev/tun%d", i);
		fd = open(tunname, O_RDWR);
		if ((fd >= 0) || (errno == ENOENT))
			break;
	}
	if (fd >= 0) {
		i = 0;
		ioctl(fd, TUNSLMODE, &i);
		i = 1;
		ioctl(fd, TUNSIFHEAD, &i);
		i = IFF_POINTOPOINT;
		ioctl(fd, TUNSIFMODE, &i);
	}
#else
	struct ifreq ifr;

	fd = open("/dev/net/tun", O_RDWR);
	if (fd < 0) {
		logerr("open: %s\n", strerror(errno));
		return -1;
	}

	memset(&ifr, 0, sizeof(ifr));
	strncpy(ifr.ifr_name, "tun0", IFNAMSIZ);
#ifdef USE_TUN_PI
	ifr.ifr_flags = IFF_TUN;
#else
	ifr.ifr_flags = IFF_TUN | IFF_NO_PI;
#endif
       
	if (ioctl(fd, TUNSETIFF, &ifr) < 0) {
		logerr("ioctl: %s\n", strerror(errno));
		close(fd);
		return -1;
	}
#ifdef notyet
	ioctl(fd, TUNSETNOCSUM, 1);
#endif
#endif
	return fd;
}

/* Tracing */

void
trace_open(char *fname)
{
	if (ftrace != NULL)
		(void) fclose(ftrace);
	ftrace = fopen(fname, "a");
	if (ftrace == NULL) {
		logerr("fopen(trace): %s\n", strerror(errno));
		return;
	}
	setlinebuf(ftrace);
}

void
trace_tunnel(struct tunnel *t, char *action)
{
	if (ftrace == NULL)
		return;
	fprintf(ftrace, "%ld tunnel %s %s\n",
		(long) seconds, action, addr2str(AF_INET6, t->remote));
}

void
trace_nat(struct nat *n, char *action)
{
	u_int p;

	if (ftrace == NULL)
		return;
#ifdef NOPRIVACY
	fprintf(ftrace, "%ld nat %s %s %s",
		(long) seconds, action,
		addr2str(AF_INET6, n->tunnel->remote),
		n->proto == IPTCP ? "tcp" : "udp");
	p = n->sport[0] << 8;
	p |= n->sport[1];
	fprintf(ftrace, " %s %u", addr2str(AF_INET, n->src), p);
	p = n->nport[0] << 8;
	p |= n->nport[1];
	fprintf(ftrace, " %s %u", addr2str(AF_INET, n->nsrc), p);
	p = n->dport[0] << 8;
	p |= n->dport[1];
	fprintf(ftrace, " %s %u\n", addr2str(AF_INET, n->dst), p);
#else
	fprintf(ftrace, "%ld nat %s %s %s",
		(long) seconds, action,
		addr2str(AF_INET6, n->tunnel->remote),
		n->proto == IPTCP ? "tcp" : "udp");
	p = n->nport[0] << 8;
	p |= n->nport[1];
	fprintf(ftrace, " %s %u\n", addr2str(AF_INET, n->nsrc), p);
#endif
}		

void
trace_bucket(struct bucket *b, struct tunnel *t, char *action)
{
	if (ftrace == NULL)
		return;
	fprintf(ftrace, "%ld bucket %s %s %s",
		(long) seconds, action,
		addr2str(AF_INET6, t->remote),
		b->proto == IPTCP ? "tcp" : "udp");
	fprintf(ftrace, " %s #%u\n",
		addr2str(AF_INET, natsrcs[t->srcidx]->addr),
		b->number);
}		

/*
 * Data structures
 *	basic
 */

/* Jenkins hash */

#define __jhash_mix(a, b, c) { \
	a -= b; a -= c; a ^= (c >> 13);	\
	b -= c; b -= a; b ^= (a << 8);	\
	c -= a; c -= b; c ^= (b >> 13);	\
	a -= b; a -= c; a ^= (c >> 12);	\
	b -= c; b -= a; b ^= (a << 16);	\
	c -= a; c -= b; c ^= (b >> 5);	\
	a -= b; a -= c; a ^= (c >> 3);	\
	b -= c; b -= a; b ^= (a << 10);	\
	c -= a; c -= b; c ^= (b >> 15);	}

#define JHASH_GOLDEN_RATIO	0x9e3779b9

inline u_short
jhash_frag(void)
{
	uint32_t a, b, c;

	memcpy(&a, buf4 + IPSRC, 4);
	a += JHASH_GOLDEN_RATIO;
	memcpy(&b, buf4 + IPDST, 4);
	b += JHASH_GOLDEN_RATIO;
	memcpy(&c, &buf4[IPID], 2);
	memcpy((u_char *)&c + 2, &buf4[IPID], 2);
	c += fraghashrnd;

	__jhash_mix(a, b, c);

	c = (c >> 16) + (c & 0xffff);
	c &= 0xffff;
	return (u_short) c;
}

inline u_short
jhash_nat(struct nat *n)
{
	uint32_t a, b, c;

	memcpy(&a, n->nsrc, 4);
	a += JHASH_GOLDEN_RATIO;
	memcpy(&b, n->nport, 2);
	memcpy((u_char *)&b + 2, n->nport, 2);
	b += JHASH_GOLDEN_RATIO;
	c = n->proto + nathashrnd;

	__jhash_mix(a, b, c);

	c = (c >> 16) + (c & 0xffff);
	c &= 0xffff;
	return (u_short) c;
}

inline u_short
jhash_tunnel(u_char *addr)
{
	uint32_t a, b, c, k;

	memcpy(&a, addr, 4);
	a += JHASH_GOLDEN_RATIO;
	memcpy(&b, addr + 4, 4);
	b += JHASH_GOLDEN_RATIO;
	memcpy(&c, addr + 8, 4);
	c += tunhashrnd;

	__jhash_mix(a, b, c);

	memcpy(&k, addr + 12, 4);
	a += k;

	__jhash_mix(a, b, c);

	c = (c >> 16) + (c & 0xffff);
	c &= 0xffff;
	return (u_short) c;
}

/* Nat entry compare routines */

int
nat_tree_compare_abs(struct nat *n1, struct nat *n2)
{
	if (n1->proto != n2->proto)
		return (int)n1->proto - (int)n2->proto;
	if (n1->nsrc[0] != n2->nsrc[0])
		return (int)n1->nsrc[0] - (int)n2->nsrc[0];
	if (n1->nsrc[1] != n2->nsrc[1])
		return (int)n1->nsrc[1] - (int)n2->nsrc[1];
	if (n1->nsrc[2] != n2->nsrc[2])
		return (int)n1->nsrc[2] - (int)n2->nsrc[2];
	if (n1->nsrc[3] != n2->nsrc[3])
		return (int)n1->nsrc[3] - (int)n2->nsrc[3];
	if (n1->nport[0] != n2->nport[0])
		return (int)n1->nport[0] - (int)n2->nport[0];
	if (n1->nport[1] != n2->nport[1])
		return (int)n1->nport[1] - (int)n2->nport[1];
	if (n1->flags & ALL_DST) {
		if (n2->flags & ALL_DST) {
			if (n1->flags & PRR_NULL) {
				if (n2->flags & PRR_NULL)
					return 0;
				else
					return -1;
			} else if (n2->flags & PRR_NULL)
				return 1;
			else
				return 0;
		} else
			return -1;
	}
	if (n1->dst[0] != n2->dst[0])
		return (int)n1->dst[0] - (int)n2->dst[0];
	if (n1->dst[1] != n2->dst[1])
		return (int)n1->dst[1] - (int)n2->dst[1];
	if (n1->dst[2] != n2->dst[2])
		return (int)n1->dst[2] - (int)n2->dst[2];
	if (n1->dst[3] != n2->dst[3])
		return (int)n1->dst[3] - (int)n2->dst[3];
	if ((n1->flags & MATCH_ANY) == 0)
		return 0;
	if (n1->dport[0] != n2->dport[0])
		return (int)n1->dport[0] - (int)n2->dport[0];
	if (n1->dport[1] != n2->dport[1])
		return (int)n1->dport[1] - (int)n2->dport[1];
	return 0;
}

int
nat_tree_compare_wild(struct nat *n1, struct nat *n2)
{
	if (n1->proto != n2->proto)
		return (int)n1->proto - (int)n2->proto;
	if (n1->nsrc[0] != n2->nsrc[0])
		return (int)n1->nsrc[0] - (int)n2->nsrc[0];
	if (n1->nsrc[1] != n2->nsrc[1])
		return (int)n1->nsrc[1] - (int)n2->nsrc[1];
	if (n1->nsrc[2] != n2->nsrc[2])
		return (int)n1->nsrc[2] - (int)n2->nsrc[2];
	if (n1->nsrc[3] != n2->nsrc[3])
		return (int)n1->nsrc[3] - (int)n2->nsrc[3];
	if (n1->nport[0] != n2->nport[0])
		return (int)n1->nport[0] - (int)n2->nport[0];
	if (n1->nport[1] != n2->nport[1])
		return (int)n1->nport[1] - (int)n2->nport[1];
	if ((n1->flags | n2->flags) & ALL_DST)
		return 0;
	if (n1->dst[0] != n2->dst[0])
		return (int)n1->dst[0] - (int)n2->dst[0];
	if (n1->dst[1] != n2->dst[1])
		return (int)n1->dst[1] - (int)n2->dst[1];
	if (n1->dst[2] != n2->dst[2])
		return (int)n1->dst[2] - (int)n2->dst[2];
	if (n1->dst[3] != n2->dst[3])
		return (int)n1->dst[3] - (int)n2->dst[3];
	if ((n1->flags & MATCH_ANY) == 0)
		return 0;
	if (n1->dport[0] != n2->dport[0])
		return (int)n1->dport[0] - (int)n2->dport[0];
	if (n1->dport[1] != n2->dport[1])
		return (int)n1->dport[1] - (int)n2->dport[1];
	return 0;
}

int
nat_splay_compare_abs(struct nat *n1, struct nat *n2)
{
	if (n1->tunnel != n2->tunnel)
		return (int)(n1->tunnel - n2->tunnel);
	if (n1->proto != n2->proto)
		return (int)n1->proto - (int)n2->proto;
	if (n1->src[0] != n2->src[0])
		return (int)n1->src[0] - (int)n2->src[0];
	if (n1->src[1] != n2->src[1])
		return (int)n1->src[1] - (int)n2->src[1];
	if (n1->src[2] != n2->src[2])
		return (int)n1->src[2] - (int)n2->src[2];
	if (n1->src[3] != n2->src[3])
		return (int)n1->src[3] - (int)n2->src[3];
	if (n1->sport[0] != n2->sport[0])
		return (int)n1->sport[0] - (int)n2->sport[0];
	if (n1->sport[1] != n2->sport[1])
		return (int)n1->sport[1] - (int)n2->sport[1];
	if (n1->flags & ALL_DST) {
		if (n2->flags & ALL_DST) {
			if (n1->flags & PRR_NULL) {
				if (n2->flags & PRR_NULL)
					return 0;
				else
					return -1;
			} else if (n2->flags & PRR_NULL)
				return 1;
			else
				return 0;
		} else
			return -1;
	}
	if (n1->dst[0] != n2->dst[0])
		return (int)n1->dst[0] - (int)n2->dst[0];
	if (n1->dst[1] != n2->dst[1])
		return (int)n1->dst[1] - (int)n2->dst[1];
	if (n1->dst[2] != n2->dst[2])
		return (int)n1->dst[2] - (int)n2->dst[2];
	if (n1->dst[3] != n2->dst[3])
		return (int)n1->dst[3] - (int)n2->dst[3];
	if ((n1->flags & MATCH_ANY) == 0)
		return 0;
	if (n1->dport[0] != n2->dport[0])
		return (int)n1->dport[0] - (int)n2->dport[0];
	if (n1->dport[1] != n2->dport[1])
		return (int)n1->dport[1] - (int)n2->dport[1];
	return 0;
}

int
nat_splay_compare_wild(struct nat *n1, struct nat *n2)
{
	if (n1->tunnel != n2->tunnel)
		return (int)(n1->tunnel - n2->tunnel);
	if (n1->proto != n2->proto)
		return (int)n1->proto - (int)n2->proto;
	if (n1->src[0] != n2->src[0])
		return (int)n1->src[0] - (int)n2->src[0];
	if (n1->src[1] != n2->src[1])
		return (int)n1->src[1] - (int)n2->src[1];
	if (n1->src[2] != n2->src[2])
		return (int)n1->src[2] - (int)n2->src[2];
	if (n1->src[3] != n2->src[3])
		return (int)n1->src[3] - (int)n2->src[3];
	if (n1->sport[0] != n2->sport[0])
		return (int)n1->sport[0] - (int)n2->sport[0];
	if (n1->sport[1] != n2->sport[1])
		return (int)n1->sport[1] - (int)n2->sport[1];
	if ((n1->flags | n2->flags) & ALL_DST)
		return 0;
	if (n1->dst[0] != n2->dst[0])
		return (int)n1->dst[0] - (int)n2->dst[0];
	if (n1->dst[1] != n2->dst[1])
		return (int)n1->dst[1] - (int)n2->dst[1];
	if (n1->dst[2] != n2->dst[2])
		return (int)n1->dst[2] - (int)n2->dst[2];
	if (n1->dst[3] != n2->dst[3])
		return (int)n1->dst[3] - (int)n2->dst[3];
	if ((n1->flags & MATCH_ANY) == 0)
		return 0;
	if (n1->dport[0] != n2->dport[0])
		return (int)n1->dport[0] - (int)n2->dport[0];
	if (n1->dport[1] != n2->dport[1])
		return (int)n1->dport[1] - (int)n2->dport[1];
	return 0;
}

/* Red-Black NAT tree routines */

void
nat_tree_rotate_left(struct nat *n, struct nat *tmp)
{
	tmp = n->right;
	if ((n->right = tmp->left) != NULL)
		tmp->left->parent = n;
	if ((tmp->parent = n->parent) != NULL) {
		if (n == n->parent->left)
			n->parent->left = tmp;
		else
			n->parent->right = tmp;
	} else
		nat_tree = tmp;
	tmp->left = n;
	n->parent = tmp;
}

void
nat_tree_rotate_right(struct nat *n, struct nat *tmp)
{
	tmp = n->left;
	if ((n->left = tmp->right) != NULL)
		tmp->right->parent = n;
	if ((tmp->parent = n->parent) != NULL) {
		if (n == n->parent->left)
			n->parent->left = tmp;
		else
			n->parent->right = tmp;
	} else
		nat_tree = tmp;
	tmp->right = n;
	n->parent = tmp;
}

void
nat_tree_insert_color(struct nat *n)
{
	struct nat *parent, *gparent, *tmp;

	while (((parent = n->parent) != NULL) &&
	       (parent->color == RB_RED)) {
		gparent = parent->parent;
		if (parent == gparent->left) {
			tmp = gparent->right;
			if ((tmp != NULL) && (tmp->color == RB_RED)) {
				tmp->color = RB_BLACK;
				parent->color = RB_BLACK;
				gparent->color = RB_RED;
				n = gparent;
				continue;
			}
			if (parent->right == n) {
				nat_tree_rotate_left(parent, tmp);
				tmp = parent;
				parent = n;
				n = tmp;
			}
			parent->color = RB_BLACK;
			gparent->color = RB_RED;
			nat_tree_rotate_right(gparent, tmp);
		} else {
			tmp = gparent->left;
			if ((tmp != NULL) && (tmp->color == RB_RED)) {
				tmp->color = RB_BLACK;
				parent->color = RB_BLACK;
				gparent->color = RB_RED;
				n = gparent;
				continue;
			}
			if (parent->left == n) {
				nat_tree_rotate_right(parent, tmp);
				tmp = parent;
				parent= n;
				n = tmp;
			}
			parent->color = RB_BLACK;
			gparent->color = RB_RED;
			nat_tree_rotate_left(gparent, tmp);
		}
	}
	nat_tree->color = RB_BLACK;
}

void
nat_tree_remove_color(struct nat *parent, struct nat *n)
{
	struct nat *tmp;

	while (((n == NULL) || (n->color == RB_BLACK)) &&
	       (n != nat_tree)) {
		if (parent->left == n) {
			tmp = parent->right;
			if (tmp->color == RB_RED) {
				tmp->color = RB_BLACK;
				parent->color = RB_RED;
				nat_tree_rotate_left(parent, tmp);
				tmp = parent->right;
			}
			if (((tmp->left == NULL) ||
			     (tmp->left->color == RB_BLACK)) &&
			    ((tmp->right == NULL) ||
			     (tmp->right->color == RB_BLACK))) {
				tmp->color = RB_RED;
				n = parent;
				parent = n->parent;
			} else {
				if ((tmp->right == NULL) ||
				    (tmp->right->color == RB_BLACK)) {
					struct nat *oleft = tmp->left;

					if (oleft != NULL)
						oleft->color = RB_BLACK;
					tmp->color = RB_RED;
					nat_tree_rotate_right(tmp, oleft);
					tmp = parent->right;
				}
				tmp->color = parent->color;
				parent->color = RB_BLACK;
				if (tmp->right != NULL)
					tmp->right->color = RB_BLACK;
				nat_tree_rotate_left(parent, tmp);
				n = nat_tree;
				break;
			}
		} else {
			tmp = parent->left;
			if (tmp->color == RB_RED) {
				tmp->color = RB_BLACK;
				parent->color = RB_RED;
				nat_tree_rotate_right(parent, tmp);
				tmp = parent->left;
			}
			if (((tmp->left == NULL) ||
			     (tmp->left->color == RB_BLACK)) &&
			    ((tmp->right == NULL) ||
			     (tmp->right->color== RB_BLACK))) {
				tmp->color = RB_RED;
				n = parent;
				parent = n->parent;
			} else {
				if ((tmp->left == NULL) ||
				    (tmp->left->color == RB_BLACK)) {
					struct nat *oright = tmp->right;

					if (oright != NULL)
						oright->color = RB_BLACK;
					tmp->color = RB_RED;
					nat_tree_rotate_left(tmp, oright);
					tmp = parent->left;
				}
				tmp->color = parent->color;
				parent->color = RB_BLACK;
				if (tmp->left != NULL)
					tmp->left->color = RB_BLACK;
				nat_tree_rotate_right(parent, tmp);
				n = nat_tree;
				break;
			}
		}
	}
	if (n != NULL)
		n->color = RB_BLACK;
}

struct nat *
nat_tree_remove(struct nat *n)
{
	struct nat *child, *parent, *old = n;
	u_char color;

	if (n->left == NULL)
		child = n->right;
	else if (n->right == NULL)
		child = n->left;
	else {
		struct nat *left;

		n = n->right;
		while ((left = n->left) != NULL)
			n = left;
		child = n->right;
		parent = n->parent;
		color = n->color;
		if (child != NULL)
			child->parent = parent;
		if (parent != NULL) {
			if (parent->left == n)
				parent->left = child;
			else
				parent->right = child;
		} else
			nat_tree = child;
		if (n->parent == old)
			parent = n;
		n->left = old->left;
		n->right = old->right;
		n->parent = old->parent;
		n->color = old->color;
		if (old->parent != NULL) {
			if (old->parent->left == old)
				old->parent->left = n;
			else
				old->parent->right = n;
		} else
			nat_tree = n;
		old->left->parent = n;
		if (old->right != NULL)
			old->right->parent = n;
		if (parent != NULL) {
			left = parent;
			do { /* void */
			} while ((left = left->parent) != NULL);
		}
		goto color;
	}
	parent = n->parent;
	color = n->color;
	if (child != NULL)
		child->parent = parent;
	if (parent != NULL) {
		if (parent->left == n)
			parent->left = child;
		else
			parent->right = child;
	} else
		nat_tree = child;
    color:
	if (color == RB_BLACK)
		nat_tree_remove_color(parent, child);
	return old;
}

struct nat *
nat_tree_insert(struct nat *n)
{
	struct nat *tmp = nat_tree, *parent = NULL;
	int comp = 0;

	while (tmp != NULL) {
		parent = tmp;
		comp = nat_tree_compare_abs(n, parent);
		if (comp < 0)
			tmp = tmp->left;
		else if (comp > 0)
			tmp = tmp->right;
		else
			return tmp;
	}
	n->parent = parent;
	n->left = n->right = NULL;
	n->color = RB_RED;
	if (parent != NULL) {
		if (comp < 0)
			parent->left = n;
		else
			parent->right = n;
	} else
		nat_tree = n;
	nat_tree_insert_color(n);
	return NULL;
}

struct nat *
nat_tree_find(struct nat *n)
{
	struct nat *tmp = nat_tree;
	int comp;

	while (tmp != NULL) {
		comp = nat_tree_compare_wild(n, tmp);
		if (comp < 0)
			tmp = tmp->left;
		else if (comp > 0)
			tmp = tmp->right;
		else
			return tmp;
	}
	return NULL;
}

struct nat *
nat_tree_find_abs(struct nat *n)
{
	struct nat *tmp = nat_tree;
	int comp;

	while (tmp != NULL) {
		comp = nat_tree_compare_abs(n, tmp);
		if (comp < 0)
			tmp = tmp->left;
		else if (comp > 0)
			tmp = tmp->right;
		else
			return tmp;
	}
	return NULL;
}

struct nat *
nat_tree_find_next(struct nat *n)
{
	struct nat *tmp = nat_tree, *res = NULL;
	int comp;

	while (tmp != NULL) {
		comp = nat_tree_compare_abs(n, tmp);
		if (comp < 0) {
			res = tmp;
			tmp = tmp->left;
		} else if (comp > 0)
			tmp = tmp->right;
		else
			return tmp;
	}
	return res;
}

struct nat *
nat_tree_next(struct nat *n)
{
	if (n->right != NULL) {
		n = n->right;
		while (n->left != NULL)
			n = n->left;
	} else {
		if ((n->parent != NULL) && (n == n->parent->left))
			n = n->parent;
		else {
			while ((n->parent!= NULL) && (n == n->parent->right))
				n = n->parent;
			n = n->parent;
		}
	}
	return n;
}

struct nat *
nat_tree_minmax(int val)
{
	struct nat *tmp = nat_tree, *parent = NULL;

	while (tmp != NULL) {
		parent = tmp;
		if (val < 0)
			tmp = tmp->left;
		else
			tmp = tmp->right;
	}
	return parent;
}

struct nat *
nat_tree_min(void)
{
	return nat_tree_minmax(-1);
}

struct nat *
nat_tree_max(void)
{
	return nat_tree_minmax(1);
}

/* foreach == _min,,_next */

/* Per tunnel splay tree */

inline void
nat_splay_rotate_right(struct nat *n)
{
	struct tunnel *t = n->tunnel;

	t->tnat_root->tleft = n->tright;
	n->tright = t->tnat_root;
	t->tnat_root = n;
}

inline void
nat_splay_rotate_left(struct nat *n)
{
	struct tunnel *t = n->tunnel;

	t->tnat_root->tright = n->tleft;
	n->tleft = t->tnat_root;
	t->tnat_root = n;
}

inline struct nat *
nat_splay_link_right(struct nat *n)
{
	struct tunnel *t = n->tunnel;

	n->tright = t->tnat_root;
	n = t->tnat_root;
	t->tnat_root = t->tnat_root->tright;
	return n;
}

inline struct nat *
nat_splay_link_left(struct nat *n)
{
	struct tunnel *t = n->tunnel;

	n->tleft = t->tnat_root;
	n = t->tnat_root;
	t->tnat_root = t->tnat_root->tleft;
	return n;
}

inline void
nat_splay_assemble(struct nat *n, struct nat *left, struct nat *right)
{
	struct tunnel *t = n->tunnel;

	left->tright = t->tnat_root->tleft;
	right->tleft = t->tnat_root->tright;
	t->tnat_root->tleft = n->tright;
	t->tnat_root->tright = n->tleft;
}

void
nat_splay_splay(struct nat *n)
{
	struct tunnel *t = n->tunnel;
	struct nat node, *left, *right, *tmp;
	int comp;

	if (t->tnat_root == NULL)
		return;
	node.tunnel = n->tunnel;
	node.tleft = node.tright = NULL;
	left = right = &node;
	while ((comp = nat_splay_compare_abs(n, t->tnat_root)) != 0) {
		if (comp < 0) {
			tmp = t->tnat_root->tleft;
			if (tmp == NULL)
				break;
			if (nat_splay_compare_abs(n, tmp) < 0) {
				nat_splay_rotate_right(tmp);
				if (t->tnat_root->tleft == NULL)
					break;
			}
			right = nat_splay_link_left(right);
		} else if (comp > 0) {
			tmp = t->tnat_root->tright;
			if (tmp == NULL)
				break;
			if (nat_splay_compare_abs(n, tmp) > 0) {
				nat_splay_rotate_left(tmp);
				if (t->tnat_root->tright == NULL)
					break;
			}
			left = nat_splay_link_right(left);
		}
	}
	nat_splay_assemble(&node, left, right);
}

void
nat_splay_splay_wild(struct nat *n)
{
	struct tunnel *t = n->tunnel;
	struct nat node, *left, *right, *tmp;
	int comp;

	if (t->tnat_root == NULL)
		return;
	node.tunnel = n->tunnel;
	node.tleft = node.tright = NULL;
	left = right = &node;
	while ((comp = nat_splay_compare_wild(n, t->tnat_root)) != 0) {
		if (comp < 0) {
			tmp = t->tnat_root->tleft;
			if (tmp == NULL)
				break;
			if (nat_splay_compare_wild(n, tmp) < 0) {
				nat_splay_rotate_right(tmp);
				if (t->tnat_root->tleft == NULL)
					break;
			}
			right = nat_splay_link_left(right);
		} else if (comp > 0) {
			tmp = t->tnat_root->tright;
			if (tmp == NULL)
				break;
			if (nat_splay_compare_wild(n, tmp) > 0) {
				nat_splay_rotate_left(tmp);
				if (t->tnat_root->tright == NULL)
					break;
			}
			left = nat_splay_link_right(left);
		}
	}
	nat_splay_assemble(&node, left, right);
}

struct nat *
nat_splay_insert(struct nat *n)
{
	struct tunnel *t = n->tunnel;

	if (t->tnat_root == NULL)
		n->tleft = n->tright = NULL;
	else {
		int comp;

		nat_splay_splay(n);
		comp = nat_splay_compare_abs(n, t->tnat_root);
		if (comp < 0) {
			n->tleft = t->tnat_root->tleft;
			n->tright = t->tnat_root;
			t->tnat_root->tleft = NULL;
		} else if (comp > 0) {
			n->tright = t->tnat_root->tright;
			n->tleft = t->tnat_root;
			t->tnat_root->tright = NULL;
		} else
			return t->tnat_root;
	}
	t->tnat_root = n;
	return NULL;
}

struct nat *
nat_splay_remove(struct nat *n)
{
	struct tunnel *t = n->tunnel;
	
	if (t->tnat_root == NULL)
		return NULL;
	nat_splay_splay(n);
	if (nat_splay_compare_abs(n, t->tnat_root) == 0) {
		if (t->tnat_root->tleft == NULL)
			t->tnat_root = t->tnat_root->tright;
		else {
			struct nat *tmp;

			tmp = t->tnat_root->tright;
			t->tnat_root = t->tnat_root->tleft;
			nat_splay_splay(n);
			t->tnat_root->tright = tmp;
		}
		return n;
	}
	return NULL;
}

/* TODO: revisit nat_splay_compare_wild() to make partial match simpler */

struct nat *
nat_splay_find(struct nat *n, int exact)
{
	struct tunnel *t = n->tunnel;

	if (t->tnat_root == NULL)
		return NULL;
	if (exact) {
		nat_splay_splay(n);
		if (nat_splay_compare_abs(n, t->tnat_root) == 0)
			return t->tnat_root;

	} else {
		nat_splay_splay_wild(n);
		if (nat_splay_compare_wild(n, t->tnat_root) == 0)
			return t->tnat_root;
	}
	return NULL;
}

/* min/max/next... */

/* NAT heap */

/* resize */

int
nat_heap_resize(void)
{
	struct nat **nh;
	u_int ns = nat_heap_size + 1024;

	nh = (struct nat **) malloc(ns * sizeof(*nh));
	if (nh == NULL) {
		logerr("malloc(heap): %s\n", strerror(errno));
		return 0;
	}
	if (nat_heap != NULL) {
		memcpy(nh, nat_heap, nat_heap_size * sizeof(*nh));
		free(nat_heap);
	}
	nat_heap = nh;
	nat_heap_size = ns;
	return 1;
}

/* parent is i/2, left child is i*2 and right child is i*2+1 */

void
nat_heap_float_up(u_int i, struct nat *n)
{
	u_int p;

	for (p = i >> 1;
	     (i > 1) && (n->timeout < nat_heap[p]->timeout);
	     i = p, p = i >> 1) {
		nat_heap[i] = nat_heap[p];
		nat_heap[i]->heap_index = i;
	}
	nat_heap[i] = n;
	nat_heap[i]->heap_index = i;

	if ((i > 1) && (nat_heap[i]->timeout < nat_heap[i >> 1]->timeout))
		logerr("BUG:nat_heap_float_up[%u]\n", i);
}

void
nat_heap_sink_down(u_int i, struct nat *n)
{
	u_int j, size, half;

	size = nat_heap_last;
	half = size / 2;
	while (i <= half) {
		/* Find the smallest of the (at most) two children */
		j = i << 1;
		if ((j < size) &&
		    (nat_heap[j + 1]->timeout < nat_heap[j]->timeout))
			j++;
		if (n->timeout < nat_heap[j]->timeout)
			break;
		nat_heap[i] = nat_heap[j];
		nat_heap[i]->heap_index = i;
		i = j;
	}
	nat_heap[i] = n;
	nat_heap[i]->heap_index = i;

	if ((i > 1) && (nat_heap[i]->timeout < nat_heap[i >> 1]->timeout))
		logerr("BUG:nat_heap_sink_down[%u]\n", i);
}

int
nat_heap_insert(struct nat *n)
{
	u_int i;

	i = ++nat_heap_last;
	if ((nat_heap_last >= nat_heap_size) && !nat_heap_resize())
		return 0;
	nat_heap_float_up(i, n);
	return 1;
}

void
nat_heap_delete(u_int i)
{
	struct nat *n;
	int less;

	if ((i < 1) || (i > nat_heap_last)) {
		logerr("BUG:nat_heap_delete[%u]\n", i);
		return;
	}

	if (i == nat_heap_last) {
		nat_heap[nat_heap_last] = NULL;
		nat_heap_last--;
		return;
	}
	n = nat_heap[nat_heap_last];
	nat_heap[nat_heap_last] = NULL;
	nat_heap_last--;

	less = (n->timeout < nat_heap[i]->timeout);
	nat_heap[i] = n;
	if (less)
		nat_heap_float_up(i, nat_heap[i]);
	else
		nat_heap_sink_down(i, nat_heap[i]);
}

void
nat_heap_increased(u_int i)
{
	if ((i < 1) || (i > nat_heap_last)) {
		logerr("BUG:nat_heap_increased[%u]\n", i);
		return;
	}

	nat_heap_float_up(i, nat_heap[i]);
}

void
nat_heap_decreased(u_int i)
{
	if ((i < 1) || (i > nat_heap_last)) {
		logerr("BUG:nat_heap_decreased[%u]\n", i);
		return;
	}

	nat_heap_sink_down(i, nat_heap[i]);
}

struct nat *
nat_heap_element(u_int i)
{
	if (i < 1) {
		logerr("BUG:nat_heap_element[%u]\n", i);
		return NULL;
	}

	if (i <= nat_heap_last)
		return nat_heap[i];
	return NULL;
}

/* Tunnel radix (patricia) tree routines */

struct tunnel *
tunnel_tree_find(struct tunnel *t)
{
	struct tunnel *node = tunnel_tree;

	if (tunnel_tree == NULL)
		return NULL;
	while (node->bit < MAXTUNBIT) {
		if (t->key[node->bit >> 3] & (0x80 >> (node->bit & 7)))
			node = node->right;
		else
			node = node->left;
		if (node == NULL)
			return NULL;
	}
	if ((node->bit > MAXTUNBIT) || ((node->flags & TUNGLUE) != 0))
		return NULL;
	if (node->bit != MAXTUNBIT)
		logerr("PATRICIA0\n");
	if (memcmp(node->key, t->key, sizeof(t->key)) == 0)
		return node;
	return NULL;
}

struct tunnel *
tunnel_tree_insert(struct tunnel *t, int *exposing)
{
	struct tunnel *node, *parent, *glue;
	u_int i, j, r, check, differ;

	*exposing = 0;
	if (tunnel_tree == NULL) {
		t->parent = NULL;
		t->left = t->right = NULL;
		t->bit = MAXTUNBIT;
		tunnel_tree = t;
		return NULL;
	}
	node = tunnel_tree;
	while ((node->bit < MAXTUNBIT) || ((node->flags & TUNGLUE) != 0)) {
		if (t->key[node->bit >> 3] & (0x80 >> (node->bit & 7))) {
			if (node->right == NULL)
				break;
			node = node->right;
		} else {
			if (node->left == NULL)
				break;
			node = node->left;
		}
		if (node == NULL)
			logerr("PATRICIA1\n");
	}
	if ((node->flags & TUNGLUE) != 0)
		logerr("PATRICIA2\n");
	if (node->bit < MAXTUNBIT)
		check = node->bit;
	else
		check = MAXTUNBIT;
	differ = 0;
	for (i = 0; (i << 3) < check; i++) {
		if ((r = (t->key[i] ^ node->key[i])) == 0) {
			differ = (i + 1) << 3;
			continue;
		}
		for (j = 0; j < 8; j++)
			if (r & (0x80 >> j))
				break;
		if (j >= 8)
			logerr("PATRICIA3\n");
		differ = (i << 3) + j;
		break;
	}
	if (differ > check)
		differ = check;
	parent = node->parent;
	while ((parent != NULL) && (parent->bit >= differ)) {
		node = parent;
		parent = node->parent;
	}
	if ((differ == MAXTUNBIT) && (node->bit == MAXTUNBIT)) {
		if ((node->flags & TUNGLUE) == 0)
			return node;
		memcpy(node->key, t->key, sizeof(t->key));
		node->flags = 0;
		*exposing = 1;
		return node;
	}
	t->bit = MAXTUNBIT;
	if (node->bit == differ) {
		t->parent = node;
		if ((node->bit < MAXTUNBIT) &&
		    (t->key[node->bit >> 3] & (0x80 >> (node->bit & 7)))) {
			if (node->right != NULL)
				logerr("PATRICIA4\n");
			node->right = t;
		} else {
			if (node->left != NULL)
				logerr("PATRICIA5\n");
			node->left = t;
		}
		return NULL;
	}
	if (differ == MAXTUNBIT) {
		t->left = node;
		t->parent = node->parent;
		if (node->parent == NULL) {
			if (tunnel_tree != node)
				logerr("PATRICIA6\n");
			tunnel_tree = t;
		} else if (node->parent->right == node)
			node->parent->right = t;
		else
			node->parent->left = t;
		node->parent = t;
		return NULL;
	}
	glue = (struct tunnel *) malloc(sizeof(*t));
	if (glue == NULL) {
		logerr("malloc(tunnel_tree_insert): %s\n", strerror(errno));
		return t;
	}
	memset(glue, 0, sizeof(*t));
	glue->flags |= TUNGLUE;
	glue->parent = node->parent;
	glue->bit = differ;
	if ((differ < MAXTUNBIT) &&
	    (t->key[differ >> 3] & (0x80 >> (differ & 7)))) {
		glue->right = t;
		glue->left = node;
	} else {
		glue->right = node;
		glue->left = t;
	}
	t->parent = glue;
	if (node->parent == NULL) {
		if (tunnel_tree != node)
			logerr("PATRICIA7\n");
		tunnel_tree = glue;
	} else if (node->parent->right == node)
		node->parent->right = glue;
	else
		node->parent->left = glue;
	node->parent = glue;
	return NULL;
}

void
tunnel_tree_remove(struct tunnel *t)
{
	struct tunnel *parent, *child;

	if ((t->right != NULL) && (t->left != NULL)) {
		t->flags |= TUNGLUE;
		return;
	}
	if ((t->right == NULL) && (t->left == NULL)) {
		parent = t->parent;
		t->flags |= TUNGLUE;
		free(t);
		if (parent == NULL) {
			if (tunnel_tree != t)
				logerr("PATRICIA8\n");
			tunnel_tree = NULL;
			return;
		}
		if (parent->right == t) {
			parent->right = NULL;
			child = parent->left;
		} else {
			parent->left = NULL;
			child = parent->right;
		}
		if ((parent->flags & TUNGLUE) == 0)
			return;
		if (parent->parent == NULL) {
			if (tunnel_tree != parent)
				logerr("PATRICIA9\n");
			tunnel_tree = child;
		} else if (parent->parent->right == parent)
			parent->parent->right = child;
		else {
			if (parent->parent->left != parent)
				logerr("PATRICIAa\n");
			parent->parent->left = child;
		}
		child->parent = parent->parent;
		free(parent);
		return;
	}
	if (t->right != NULL)
		child = t->right;
	else {
		if (t->left == NULL)
			logerr("PATRICIAb\n");
		child = t->left;
	}
	parent = t->parent;
	child->parent = parent;
	t->flags |= TUNGLUE;
	free(t);
	if (parent == NULL) {
		if (tunnel_tree != t)
			logerr("PATRICIAc\n");
		tunnel_tree = child;
		return;
	}
	if (parent->right == t)
		parent->right = child;
	else {
		if (parent->left != t)
			logerr("PATRICIAd\n");
		parent->left = child;
	}
}

/*
 * Data structures
 *	lookup/add/del/set...
 */

/* Lookup matching NAT */

struct nat *
nat_lookup(struct nat *nat0)
{
	struct nat *n;
	u_short hash;

	tnhlookups++, pnhlookups++;
	hash = jhash_nat(nat0);
	hash &= nathashsz - 1;
	n = nathash[hash];
	if ((n != NULL) && (nat_tree_compare_wild(nat0, n) == 0)) {
		tnhhits++, pnhhits++;
		return n;
	}
	n = nat_tree_find(nat0);
	if (n == NULL)
		return NULL;
	n->hash = hash;
	nathash[hash] = n;
	return n;
}

/* lookup a matching tunnel */

struct tunnel *
tunnel_lookup(u_char *peer)
{
	struct tunnel *t, tunnel0;
	u_short hash;

	tthlookups++, pthlookups++;
	hash = jhash_tunnel(peer);
	hash &= tunhashsz - 1;
	t = tunhash[hash];
	if ((t != NULL) && (memcmp(peer, t->remote, 16) == 0)) {
		tthhits++, pthhits++;
		return t;
	}
	memcpy(tunnel0.remote, peer, 16);
	t = tunnel_tree_find(&tunnel0);
	if (t == NULL)
		return NULL;
	t->hash = hash;
	tunhash[hash] = t;
	return t;
}

/* Create a new NAT entry (returns 1 if successful, 0 otherwise) */

int
new_nat(struct nat *n)
{
	struct tunnel *t = n->tunnel;
	struct natsrc *ns = natsrcs[t->srcidx];
	struct bucket *b;
	u_int i;
	u_char sz, pr;

	logit(10, "new_nat");

	switch (n->proto) {
	case IPTCP:
		sz = TCPBUCKSZ;
		pr = TCPPR;
		break;
	case IPUDP:
		sz = UDPBUCKSZ;
		pr = UDPPR;
		break;
	default:
		sz = ICMPBUCKSZ;
		pr = ICMPPR;
		break;
	}
	for (b = t->blist; b != NULL; b = b->next) {
		if (b->avail == 0)
			continue;
		if (b->proto != n->proto)
			continue;
		for (i = 0; i < sz; i++)
			if (b->freebm[i / 8] & (1 << (i % 8)))
				goto found;
	}
	/* get a new bucket */
	if (ns->curport[pr] < ns->maxport[pr]) {
		b = (struct bucket *) malloc(sizeof(*b) + (sz + 7) / 8);
		if (b == NULL) {
			logerr("malloc(bucket): %s\n", strerror(errno));
			free(n);
			return 0;
		}
		memset(b, 0, sizeof(*b));
		b->number = ns->curport[pr];
		b->proto = n->proto;
		if ((u_int) ns->curport[pr] + sz > (u_int) ns->maxport[pr])
			sz = ns->maxport[pr] - ns->curport[pr];
		ns->curport[pr] += sz;
		b->avail = sz;
		for (i = 0; i < (sz / 8); i++)
			b->freebm[i] = 0xff;
		for (i = 0; i < (sz % 8); i++)
			b->freebm[sz / 8] |= 1 << i;
		logit(10, "create new bucket");
	} else {
		b = ns->ffirst[pr];
		if (b == NULL) {
			logerr("no free bucket for %s\n",
			       addr2str(AF_INET, ns->addr));
			free(n);
			return 0;
		}
		if (b->next != NULL)
			b->next->prev = b->prev;
		else
			ns->flast[pr] = b->prev;
		*b->prev = b->next;
		logit(10, "get bucket from free list");
	}
	b->next = t->blist;
	t->blist = b;
	b->prev = &t->blist;
	i = 0;
	ns->buckcnt[pr] += 1;
	t->buckcnt++;
	statscbuck++;
	if (t->flags & TUNDEBUG)
		debugcbuck++;
	if (n->proto != IPICMP)
		trace_bucket(b, t, "add");

    found:
	b->avail--;
	b->freebm[i / 8] &= ~(1 << (i % 8));
	i += b->number;
	n->nport[0] = i >> 8;
	n->nport[1] = i & 0xff;
	n->bucket = b;

	if (!nat_heap_insert(n)) {
		free(n);
		return 0;
	}
	if (nat_tree_insert(n) != NULL) {
		logerr("SEVERE:new_nat rb collision\n");
		nat_heap_delete(n->heap_index);
		free(n);
		return 0;
	}
	if (nat_splay_insert(n) != NULL) {
		logerr("SEVERE:new_nat splay collision\n");
		(void) nat_tree_remove(n);
		nat_heap_delete(n->heap_index);
		free(n);
		return 0;
	}
	if (n->flags & ALL_DST)
		snatcnt++;
	else
		switch (n->proto) {
		case IPTCP:
			tnatcnt++;
			break;
		case IPUDP:
			unatcnt++;
			break;
		default:
			onatcnt++;
			break;
		}
	ns->natcnt[pr] += 1;
	t->natcnt++;
	statscnat++;
	if (t->flags & TUNDEBUG)
		debugcnat++;
	logit(10, "create nat entry");
	if (n->proto != IPICMP)
		trace_nat(n, "add");
	return 1;
}

/* Add a tunnel entry (returns a pointer to the new tunnel,
   or NULL on failure */

struct tunnel *
add_tunnel(u_char *peer)
{
	struct tunnel *t, *e;
	int exposing = 0;
	static u_int srcidx = 0;

	logit(10, "add_tunnel");

	tunnel_configured = 1;

	if (tunnel_lookup(peer) != NULL) {
		logerr("already bound[%s]\n", addr2str(AF_INET6, peer));
		return NULL;
	}
	t = (struct tunnel *) malloc(sizeof(*t));
	if (t == NULL) {
		logerr("malloc(tunnel): %s\n", strerror(errno));
		return NULL;
	}
	memset(t, 0, sizeof(*t));
	memcpy(t->remote, peer, 16);
	e = tunnel_tree_insert(t, &exposing);
	if (e != NULL) {
		if (!exposing) {
			logerr("already exist (add_tunnel)\n");
			free(t);
			return NULL;
		}
		free(t);
		t = e;
	}
	t->srcidx = srcidx;
	if (++srcidx == natsrccnt)
		srcidx = 0;
	t->mtu = tundefmtu;
	if (enable_msspatch)
		t->flags |= TUNMSSFLG;
	t->flags |= default_toobig;

	logit(1, "create tunnel: remote=%s, mtu=%u, mss=%s toobig=%s",
	      addr2str(AF_INET6, peer), t->mtu,
	      (t->flags & TUNMSSFLG) != 0 ? "on" : "off",
	      toobig2str(t));
	tuncnt++;
	trace_tunnel(t, "add");
	return t;
}

/* Delete a fragment chain */

void
del_frag4(struct frag *f)
{
	struct frag *p, *q;

	for (p = f->list; p != NULL; p = q) {
		q = p->next;
		free(p->buf);
		free(p);
	}
	free(f->buf);
	if (f->next != NULL)
		f->next->prev = f->prev;
	else if (f->tunnel == NULL)
		fragsout_last = f->prev;
	else
		fragsin_last = f->prev;
	*f->prev = f->next;
	if ((f->hash < fraghashsz) && (fraghash[f->hash] == f))
		fraghash[f->hash] = NULL;
	if (f->tunnel != NULL) {
		fragsincnt--;
		f->tunnel->frg4cnt--;
	} else
		fragsoutcnt--;
	free(f);
}

void
del_frag6(struct frag *f)
{
	struct frag *p, *q;

	for (p = f->list; p != NULL; p = q) {
		q = p->next;
		free(p->buf);
		free(p);
	}
	free(f->buf);
	if (f->next != NULL)
		f->next->prev = f->prev;
	else
		frags6_last = f->prev;
	*f->prev = f->next;
	frags6cnt--;
	f->tunnel->frg6cnt--;
	free(f);
}

/* Delete a bucket */

void
del_bucket(struct bucket *b, struct tunnel *t)
{
	struct natsrc *ns = natsrcs[t->srcidx];
	u_char pr;

	if (b->next != NULL)
		b->next->prev = b->prev;
	*b->prev = b->next;
	switch (b->proto) {
	case IPTCP:
		pr = TCPPR;
		break;
	case IPUDP:
		pr = UDPPR;
		break;
	default:
		pr = ICMPPR;
		break;
	}
	b->next = NULL;
	b->prev = ns->flast[pr];
	*ns->flast[pr] = b;
	ns->flast[pr] = &b->next;
	ns->buckcnt[pr] -= 1;
	t->buckcnt--;
	statsdbuck++;
	if (t->flags & TUNDEBUG)
		debugdbuck++;
	if (b->proto != IPICMP)
		trace_bucket(b, t, "del");
}

/* Delete a NAT entry */

void
del_nat(struct nat *n)
{
	struct tunnel *t = n->tunnel;
	struct bucket *b = n->bucket;
	struct natsrc *ns;
	struct nat *f;
	u_int i;
	u_char sz, pr;

	if (n->flags & ALL_DST) {
		if (n->flags & PRR_NULL)
			prrcnt--;
		else
			snatcnt--;
	} else switch (n->proto) {
		case IPTCP:
			tnatcnt--;
			break;
		case IPUDP:
			unatcnt--;
			break;
		default:
			onatcnt--;
			break;
		}
	statsdnat++;
	if (t->flags & TUNDEBUG)
		debugdnat++;
	if (n->proto != IPICMP)
		trace_nat(n, "del");
	logit(10, "del_nat");
	if (n->heap_index)
		nat_heap_delete(n->heap_index);
	n->heap_index = 0;
	while ((f = n->xfirst) != NULL) {
		if (f->xnext != NULL)
			f->xnext->xprev = f->xprev;
		*f->xprev = f->xnext;
		f->xnext = NULL;
		f->xprev = NULL;
	}
	if (n->xprev != NULL) {
		if (n->xnext != NULL)
			n->xnext->xprev = n->xprev;
		*n->xprev = n->xnext;
	}

	if (nat_tree_remove(n) == NULL)
		logerr("rb not found(del_nat)\n");
	if (nat_splay_remove(n) == NULL)
		logerr("splay not found(del_nat)\n");
	t->natcnt--;

	if ((n->hash < nathashsz) && (nathash[n->hash] == n))
		nathash[n->hash] = NULL;
	i = (n->nport[0] << 8) | n->nport[1];
	free(n);
	if (b == NULL)
		return;

	i -= b->number;
	switch (b->proto) {
	case IPTCP:
		sz = TCPBUCKSZ;
		pr = TCPPR;
		break;
	case IPUDP:
		sz = UDPBUCKSZ;
		pr = UDPPR;
		break;
	default:
		sz = ICMPBUCKSZ;
		pr = ICMPPR;
		break;
	}
	ns = natsrcs[t->srcidx];
	ns->natcnt[pr] -= 1;
	b->freebm[i / 8] |= 1 << (i % 8);
	b->avail++;
	if (b->avail >= sz)
		del_bucket(b, t);
}

/* Delete a tunnel entry */

void
del_tunnel(u_char *peer)
{
	struct tunnel *t;
	struct nat *n;

	logit(10, "del_tunnel");

	t = tunnel_lookup(peer);
	if (t == NULL) {
		logerr("already unbound[%s]\n", addr2str(AF_INET6, peer));
		return;
	} else if (t->flags & TUNNONAT) {
		logerr("nonat tunnel[%s]\n", addr2str(AF_INET6, peer));
		return;
	}
	while ((n = t->tnat_root) != NULL)
		del_nat(n);
	if (t == tunnel_debugged)
		tunnel_debugged = NULL;
	t->flags &= ~TUNDEBUG;
	tuncnt--;
	trace_tunnel(t, "del");
	if ((t->hash < tunhashsz) && (tunhash[t->hash] == t))
		tunhash[t->hash] = NULL;
	tunnel_tree_remove(t);
	/* t freed by tunnel_tree_remove() */
}

/* Set tunnel MTU */

void
set_tunnel_mtu(u_char *remote, u_int mtu, u_int force)
{
	struct tunnel *t;

	logit(10, "set_tunnel_mtu");

	t = tunnel_lookup(remote);
	if (t == NULL)
		return;
	if (mtu > IPMAXLEN)
		mtu = IPMAXLEN;
	else if (mtu < TUNMINMTU)
		mtu = TUNMINMTU;
	if (!force && (t->mtu < (u_short) mtu))
		return;
	t->mtu = (u_short) mtu;
}

/* Set tunnel flag */

void
set_tunnel_flag(u_char *remote, u_char flags, u_char mask)
{
	struct tunnel *t;
	u_char oflags;

	t = tunnel_lookup(remote);
	if (t == NULL)
		return;
	oflags = t->flags;
	t->flags = (oflags & ~mask) | (flags & mask);
}

/*
 * Data structures
 *	debug
 */

void
print_hashes(FILE *s)
{
	u_int n, cnt;

	if (s == NULL)
		s = stdout;
	fprintf(s, "fragment: %d <= %u <= %d\n",
		MINFRAGHASH, fraghashsz, MAXFRAGHASH);
	cnt = 0;
	for (n = 0; n < fraghashsz; n++)
		if (fraghash[n] != NULL)
			cnt++;
	fprintf(s,
#if ULONG_MAX > 4294967295UL
		"\tfill %u hits %lu/%lu lookups %lu/%lu\n",
#else
		"\tfill %u hits %llu/%llu lookups %llu/%llu\n",
#endif
		cnt, pfhhits, tfhhits, pfhlookups, tfhlookups);
	fprintf(s, "nat: %d <= %u <= %d\n",
		MINNATHASH, nathashsz, MAXNATHASH);
	cnt = 0;
	for (n = 0; n < nathashsz; n++)
		if (nathash[n] != NULL)
			cnt++;
	fprintf(s,
#if ULONG_MAX > 4294967295UL
		"\tfill %u hits %lu/%lu lookups %lu/%lu\n",
#else
		"\tfill %u hits %llu/%llu lookups %llu/%llu\n",
#endif
		cnt, pnhhits, tnhhits, pnhlookups, tnhlookups);
	fprintf(s, "tunnel: %d <= %u <= %d\n",
		MINTUNHASH, tunhashsz, MAXTUNHASH);
	cnt = 0;
	for (n = 0; n < tunhashsz; n++)
		if (tunhash[n] != NULL)
			cnt++;
	fprintf(s,
#if ULONG_MAX > 4294967295UL
		"\tfill %u hits %lu/%lu lookups %lu/%lu\n",
#else
		"\tfill %u hits %llu/%llu lookups %llu/%llu\n",
#endif
		cnt, pthhits, tthhits, pthlookups, tthlookups);
}

/* statistics */

uint64_t
dropped(int what, int ipv)
{
	uint64_t accu = 0;

	if (ipv == 0) {
		if (what == 0) {
			accu += statsdropped[DR_BAD6];
			accu += statsdropped[DR_ACL6];
			accu += statsdropped[DR_NOTUN];
			accu += statsdropped[DR_ICMP6];
			accu += statsdropped[DR_F6CNT];
			accu += statsdropped[DR_F6TCNT];
			accu += statsdropped[DR_BADF6];
			accu += statsdropped[DR_F6TM];
		} else {
			accu += debugdropped[DR_BAD6];
			accu += debugdropped[DR_ACL6];
			accu += debugdropped[DR_NOTUN];
			accu += debugdropped[DR_ICMP6];
			accu += debugdropped[DR_F6CNT];
			accu += debugdropped[DR_F6TCNT];
			accu += debugdropped[DR_BADF6];
			accu += debugdropped[DR_F6TM];
		}
	} else {
		if (what == 0) {
			accu += statsdropped[DR_BADIN];
			accu += statsdropped[DR_NOPRR];
			accu += statsdropped[DR_NATCNT];
			accu += statsdropped[DR_NEWNAT];
			accu += statsdropped[DR_ICMPIN];
			accu += statsdropped[DR_BADOUT];
			accu += statsdropped[DR_DSTOUT];
			accu += statsdropped[DR_ICMPOUT];
			accu += statsdropped[DR_NATOUT];
			accu += statsdropped[DR_TOOBIG];
			accu += statsdropped[DR_FINCNT];
			accu += statsdropped[DR_FINTCNT];
			accu += statsdropped[DR_FOUTCNT];
			accu += statsdropped[DR_BADF4];
			accu += statsdropped[DR_F4MEM];
			accu += statsdropped[DR_FINTM];
			accu += statsdropped[DR_FOUTTM];
		} else {
			accu += debugdropped[DR_BADIN];
			accu += debugdropped[DR_NOPRR];
			accu += debugdropped[DR_NATCNT];
			accu += debugdropped[DR_NEWNAT];
			accu += debugdropped[DR_ICMPIN];
			accu += debugdropped[DR_BADOUT];
			accu += debugdropped[DR_DSTOUT];
			accu += debugdropped[DR_ICMPOUT];
			accu += debugdropped[DR_NATOUT];
			accu += debugdropped[DR_TOOBIG];
			accu += debugdropped[DR_FINCNT];
			accu += debugdropped[DR_FINTCNT];
			accu += debugdropped[DR_FOUTCNT];
			accu += debugdropped[DR_BADF4];
			accu += debugdropped[DR_F4MEM];
			accu += debugdropped[DR_FINTM];
			accu += debugdropped[DR_FOUTTM];
		}
	}
	return accu;
}

void
print_stats(FILE *s)
{
	if (s == NULL)
		s = stdout;
	fprintf(s, "tun=%u tcp=%u/%u udp=%u/%u other=%u/%u stat=%u prr=%u\n",
		tuncnt, tnatcnt, tnatcnt / natsrccnt,
		unatcnt, unatcnt / natsrccnt,
		onatcnt , onatcnt / natsrccnt,
		snatcnt, prrcnt);
#if ULONG_MAX > 4294967295UL
	fprintf(s, "received: v6=%lu v4=%lu\n", statsrcv6, statsrcv4);
	fprintf(s, "sent: v6=%lu v4=%lu\n", statssent6, statssent4);
	fprintf(s, "dropped: v6=%lu v4=%lu\n",
		dropped(0, 0), dropped(0, 1));
	fprintf(s, "frag: in6[%u]=%lu/%lu in[%u]=%lu/%lu ",
		frags6cnt, statsfrgin6, statsreas6,
		fragsincnt, statsfrgin, statsreasin);
	fprintf(s, "frag: out[%u]=%lu/%lu out6=%lu\n",
		fragsoutcnt, statsfrout, statsreasout, statsfrgout6);
	fprintf(s, "in: nat=%lu prr=%lu nonat=%lu icmp6=%lu icmp4=%lu\n",
		statsnatin, statsprrin, statsnonatin,
		statsnaticmpin6, statsnaticmpin4);
	fprintf(s, "out: nat=%lu prr=%lu nonat=%lu icmp=%lu\n",
		statsnatout, statsprrout, statsnonatout, statsnaticmpout);
	fprintf(s, "tcpmss: seen=%lu patched=%lu toobig=%lu\n",
		statstcpmss, statsmsspatched, statstoobig);
	fprintf(s, "ftpalg: port=%lu eprt=%lu 227=%lu 229=%lu\n",
		statsftpport, statsftpeprt, statsftp227, statsftp229);
	fprintf(s, "nat: created=%lu deleted=%lu\n",
		statscnat, statsdnat);
	fprintf(s, "bucket: created=%lu deleted=%lu\n",
		statscbuck, statsdbuck);
	if (tunnel_debugged == NULL)
		goto rates;
	fprintf(s, "debug for %s\n",
		addr2str(AF_INET6, tunnel_debugged->remote));
	fprintf(s, " received: v6=%lu v4=%lu\n", debugrcv6, debugrcv4);
	fprintf(s, " sent: v6=%lu v4=%lu\n", debugsent6, debugsent4);
	fprintf(s, " dropped: v6=%lu v4=%lu\n",
		dropped(1, 0), dropped(1, 1));
	fprintf(s, " frag: in6[%u]=%lu/%lu in[%u]=%lu/%lu out6=%lu\n",
		tunnel_debugged->frg6cnt, debugfrgin6, debugreas6,
		tunnel_debugged->frg4cnt, debugfrgin, debugreasin,
		debugfrgout6);
	fprintf(s, " in: nat=%lu prr=%lu nonat=%lu icmp6=%lu icmp4=%lu\n",
		debugnatin, debugprrin, debugnonatin,
		debugnaticmpin6, debugnaticmpin4);
	fprintf(s, " out: nat=%lu prr=%lu nonat=%lu icmp=%lu\n",
		debugnatout, debugprrout, debugnonatout, debugnaticmpout);
	fprintf(s, " tcpmss: seen=%lu patched=%lu toobig=%lu\n",
		debugtcpmss, debugmsspatched, debugtoobig);
	fprintf(s, " ftpalg: port=%lu eprt=%lu 227=%lu 229=%lu\n",
		debugftpport, debugftpeprt, debugftp227, debugftp229);
	fprintf(s, " nat[%u]: created=%lu deleted=%lu\n",
		tunnel_debugged->natcnt, debugcnat, debugdnat);
	fprintf(s, " bucket[%u]: created=%lu deleted=%lu\n",
		tunnel_debugged->buckcnt, debugcbuck, debugdbuck);
#else
	fprintf(s, "received: v6=%llu v4=%llu\n", statsrcv6, statsrcv4);
	fprintf(s, "sent: v6=%llu v4=%llu\n", statssent6, statssent4);
	fprintf(s, "dropped: v6=%llu v4=%llu\n",
		dropped(0, 0), dropped(0, 1));
	fprintf(s, "frag: in6[%u]=%llu/%llu in[%u]=%llu/%llu ",
		frags6cnt, statsfrgin6, statsreas6,
		fragsincnt, statsfrgin, statsreasin);
	fprintf(s, "frag: out[%u]=%llu/%llu out6=%llu\n",
		fragsoutcnt, statsfrout, statsreasout, statsfrgout6);
	fprintf(s, "in: nat=%llu prr=%llu nonat=%llu icmp6=%llu icmp4=%llu\n",
		statsnatin, statsprrin, statsnonatin,
		statsnaticmpin6, statsnaticmpin4);
	fprintf(s, "out: nat=%llu prr=%llu nonat=%llu icmp=%llu\n",
		statsnatout, statsprrout, statsnonatout, statsnaticmpout);
	fprintf(s, "tcpmss: seen=%llu patched=%llu toobig=%llu\n",
		statstcpmss, statsmsspatched, statstoobig);
	fprintf(s, "ftpalg: port=%llu eprt=%llu 227=%llu 229=%llu\n",
		statsftpport, statsftpeprt, statsftp227, statsftp229);
	fprintf(s, "nat: created=%llu deleted=%llu\n",
		statscnat, statsdnat);
	fprintf(s, "bucket: created=%llu deleted=%llu\n",
		statscbuck, statsdbuck);
	if (tunnel_debugged == NULL)
		goto rates;
	fprintf(s, "debug for %s\n",
		addr2str(AF_INET6, tunnel_debugged->remote));
	fprintf(s, " received: v6=%llu v4=%llu\n", debugrcv6, debugrcv4);
	fprintf(s, " sent: v6=%llu v4=%llu\n", debugsent6, debugsent4);
	fprintf(s, " dropped: v6=%llu v4=%llu\n",
		dropped(1, 0), dropped(1, 1));
	fprintf(s, " frag: in6[%u]=%llu/%llu in[%u]=%llu/%llu out6=%llu\n",
		tunnel_debugged->frg6cnt, debugfrgin6, debugreas6,
		tunnel_debugged->frg4cnt, debugfrgin, debugreasin,
		debugfrgout6);
	fprintf(s, " in: nat=%llu prr=%llu nonat=%llu icmp6=%llu icmp4=%llu\n",
		debugnatin, debugprrin, debugnonatin,
		debugnaticmpin6, debugnaticmpin4);
	fprintf(s, " out: nat=%llu prr=%llu nonat=%llu icmp=%llu\n",
		debugnatout, debugprrout, debugnonatout, debugnaticmpout);
	fprintf(s, " tcpmss: seen=%llu patched=%llu toobig=%llu\n",
		debugtcpmss, debugmsspatched, debugtoobig);
	fprintf(s, " ftpalg: port=%llu eprt=%llu 227=%llu 229=%llu\n",
		debugftpport, debugftpeprt, debugftp227, debugftp229);
	fprintf(s, " nat[%u]: created=%llu deleted=%llu\n",
		tunnel_debugged->natcnt, debugcnat, debugdnat);
	fprintf(s, " bucket[%u]: created=%llu deleted=%llu\n",
		tunnel_debugged->buckcnt, debugcbuck, debugdbuck);
#endif
    rates:
	fprintf(s, "rates\n");
	fprintf(s, "received v6=%f %f %f\n",
		ratercv6[0], ratercv6[1], ratercv6[2]);
	fprintf(s, "received v4=%f %f %f\n",
		ratercv4[0], ratercv4[1], ratercv4[2]);
	fprintf(s, "sent v6=%f %f %f\n",
		ratesent6[0], ratesent6[1], ratesent6[2]);
	fprintf(s, "sent v4=%f %f %f\n",
		ratesent4[0], ratesent4[1], ratesent4[2]);
	fprintf(s, "nat created=%f %f %f\n",
		ratecnat[0], ratecnat[1], ratecnat[2]);
	fprintf(s, "nat deleted=%f %f %f\n",
		ratednat[0], ratednat[1], ratednat[2]);
}

void
print_dropped(FILE *s)
{
	int i;
	if (s == NULL)
		s = stdout;
#if ULONG_MAX > 4294967295UL
	fprintf(s, "summary: v6=%lu(%lu) v4=%lu(%lu)\n",
		dropped(0, 0), dropped(1, 0), dropped(0, 1), dropped(1, 1));
	for (i = 0; i <= DR_MAX; i++) {
		if ((statsdropped[i] != 0) || (debugdropped[i] != 0))
			fprintf(s, "\t%lu(%lu):\t%s\n",
				statsdropped[i], debugdropped[i],
				dropreason[i]);
	}
#else
	fprintf(s, "summary: v6=%llu(%llu) v4=%llu(%llu)\n",
		dropped(0, 0), dropped(1, 0), dropped(0, 1), dropped(1, 1));
	for (i = 0; i <= DR_MAX; i++) {
		if ((statsdropped[i] != 0) || (debugdropped[i] != 0))
			fprintf(s, "\t%llu(%llu):\t%s\n",
				statsdropped[i], debugdropped[i],
				dropreason[i]);
	}
#endif
}

/* fragments */

void
print_frag_elm(FILE *s, struct frag *p)
{
	if (s == NULL)
		s = stdout;
	fprintf(s, "addr=%lx next=%lx ", (u_long) p, (u_long) p->next);
	fprintf(s, "buf=%lx len=%u off=%u\n",
		(u_long) p->buf, p->len, p->off);
}

void
print_frag(FILE *s, struct frag *f)
{
	struct frag *p;

	if (s == NULL)
		s = stdout;
	fprintf(s, "addr=%lx next=%lx ", (u_long) f, (u_long) f->next);
	if (f->tunnel != NULL)
		fprintf(s, "tunnel=%s ",
			addr2str(AF_INET6, f->tunnel->remote));
	fprintf(s, "buf=%lx len=%u off=%u expire=%ld\n",
		(u_long) f->buf, f->len, f->off,
		(long) (f->expire - seconds));

	if (f->list == NULL)
		return;
	fprintf(s, "Chain:\n");
	for (p = f->list; p != NULL; p = p->next) {
		fprintf(s, "\t");
		print_frag_elm(s, p);
	}
}

void
print_frags6(FILE *s)
{
	struct frag *f;
	int i = 0;

	if (s == NULL)
		s = stdout;
	for (f = frags6_first; f != NULL; f = f->next, i++)
		fprintf(s, "%s%lx%s",
			(i & 3) != 0 ? " " : "",
			(u_long) f,
			(i & 3) == 3 ? "\n" : "");
	if ((i & 3) != 3)
		fprintf(s, "\n");
}

void
print_fragsin(FILE *s)
{
	struct frag *f;
	int i = 0;

	if (s == NULL)
		s = stdout;
	for (f = fragsin_first; f != NULL; f = f->next, i++)
		fprintf(s, "%s%lx%s",
			(i & 3) != 0 ? " " : "",
			(u_long) f,
			(i & 3) == 3 ? "\n" : "");
	if ((i & 3) != 3)
		fprintf(s, "\n");
}

void
print_fragsout(FILE *s)
{
	struct frag *f;
	int i = 0;

	if (s == NULL)
		s = stdout;
	for (f = fragsout_first; f != NULL; f = f->next, i++)
		fprintf(s, "%s%lx%s",
			(i & 3) != 0 ? " " : "",
			(u_long) f,
			(i & 3) == 3 ? "\n" : "");
	if ((i & 3) != 3)
		fprintf(s, "\n");
}

/* NAT sources */

void
print_natsrcs(FILE *s)
{
	struct natsrc *ns;
	u_int i;

	if (s == NULL)
		s = stdout;
	fprintf(s, "natsrccnt=%u\tbucket size: tcp=%u udp=%u echo=%u\n",
		natsrccnt, TCPBUCKSZ, UDPBUCKSZ, ICMPBUCKSZ);
	for (i = 0; i < natsrccnt; i++) {
		fprintf(s, "natsrc[%u]: ", i);
		ns = natsrcs[i];
		if (ns == NULL) {
			fprintf(s, "NULL???\n");
			continue;
		}
		fprintf(s,
			"%s\n tcp=%u[%u/%u] udp=%u[%u/%u] echo=%u[%u/%u]\n",
			addr2str(AF_INET, ns->addr),
			ns->natcnt[0], ns->buckcnt[0],
			(ns->maxport[0] - ns->minport[0]) / TCPBUCKSZ,
			ns->natcnt[1], ns->buckcnt[1],
			(ns->maxport[1] - ns->minport[1]) / UDPBUCKSZ,
			ns->natcnt[2], ns->buckcnt[2],
			(ns->maxport[2] - ns->minport[2]) / ICMPBUCKSZ);
	}
}

/* buckets */

void
print_bucket(FILE *s, struct bucket *b)
{
	if (s == NULL)
		s = stdout;
	fprintf(s, "addr=%lx ", (u_long) b);
	fprintf(s, "next=%lx ", (u_long) b->next);
	fprintf(s, "number=%u proto=%s avail=%u\n",
		b->number, proto2str(b->proto), b->avail);
}

/* NAT entries */

void
print_nat(FILE *s, struct nat *n)
{
	u_int p;

	if (s == NULL)
		s = stdout;
	fprintf(s, "addr=%lx ", (u_long) n);
	if (n->tunnel != NULL) {
		fprintf(s, "tunnel=%s ",
			addr2str(AF_INET6, n->tunnel->remote));
		fprintf(s, "tleft=%lx ", (u_long) n->tleft);
		fprintf(s, "tright=%lx ", (u_long) n->tright);
	}
	if (n->timeout != 0) {
		fprintf(s, "timeout=%ld ", (long) (n->timeout - seconds));
		fprintf(s, "lifetime=%u ", n->lifetime);
		fprintf(s, "heap=%u ", n->heap_index);
	}
	fprintf(s, "\n");
	p = n->sport[0] << 8;
	p |= n->sport[1];
	fprintf(s, "%d src=%s/%u -> ", (int) n->proto,
		addr2str(AF_INET, n->src), p);
	p = n->nport[0] << 8;
	p |= n->nport[1];
	fprintf(s, "%s/%u dst=", addr2str(AF_INET, n->nsrc), p);
	if (n->flags & ALL_DST) {
		if (n->flags & PRR_NULL)
			fprintf(s, "any PRR\n");
		else
			fprintf(s, "*\n");
	} else {
		p = n->dport[0] << 8;
		p |= n->dport[1];
		fprintf(s, "%s/%u\n", addr2str(AF_INET, n->dst), p);
	}
	switch (n->proto) {
	case IPICMP:
		if (n->flags != MATCH_ICMP)
			fprintf(s, "flags=%d\n", (int) n->flags);
		break;
	case IPTCP:
		if (n->tcpst != TCP_DEFAULT)
			fprintf(s, "tcp status=%d\n", (int) n->tcpst);
		if (n->flags & ~ALL_FLAGS)
			fprintf(s, "flags=%d\n", (int) n->flags);
		break;
	case IPUDP:
		if (n->flags & ~ALL_FLAGS)
			fprintf(s, "flags=%d\n", (int) n->flags);
		break;
	default:
		fprintf(s, "flags=%d\n", (int) n->flags);
		break;
	}
	/* TODO: print FTP ALG stuff */
}

void
print_nat_tree_elm(FILE *s, struct nat *n)
{
	if (s == NULL)
		s = stdout;
	if (n == NULL) {
		fprintf(s, "root=%lx\n", (u_long) nat_tree);
		return;
	}
	fprintf(s, "%c ", n->color == RB_BLACK ? 'B' : 'R');
	fprintf(s, "%lx ", (u_long) n);
	if (n->left != NULL)
		fprintf(s, "left=%lx ", (u_long) n->left);
	if (n->right != NULL)
		fprintf(s, "right=%lx ", (u_long) n->right);
	if (n->parent != NULL)
		fprintf(s, "parent=%lx\n", (u_long) n->parent);
}

void
print_nat_tree_branch(FILE *s, struct nat *n)
{
	if (n == NULL)
		return;
	if (n->left != NULL) {
		if ((n->left->left != NULL) || (n->left->right != NULL)) {
			fprintf(s, "<");
			print_nat_tree_branch(s, n->left);
		}
		fprintf(s, "<");
		print_nat_tree_elm(s, n->left);
	}
	if (n->right != NULL) {
		if ((n->right->left != NULL) || (n->right->right != NULL)) {
			fprintf(s, ">");
		print_nat_tree_branch(s, n->right);
		}
		fprintf(s, ">");
		print_nat_tree_elm(s, n->right );
	}
}

void
print_nat_tree(FILE *s)
{
	if (s == NULL)
		s = stdout;
	fprintf(s, "nat root at %lx\n", (u_long) nat_tree);
	print_nat_tree_branch(s, nat_tree);
	if (nat_tree != NULL) {
		print_nat_tree_elm(s, nat_tree);
		fprintf(s, "\n");
	}
}

void
print_nat_splay_elm(FILE *s, struct nat *n)
{
	if (s == NULL)
		s = stdout;
	fprintf(s, "%lx", (u_long) n);
	if (n->tleft != NULL)
		fprintf(s, " tleft=%lx", (u_long) n->tleft);
	if (n->tright != NULL)
		fprintf(s, " tright=%lx", (u_long) n->tright);
	fprintf(s, "\n");
}

void
print_nat_splay_branch(FILE *s, struct nat *n)
{
	if (n == NULL)
		return;
	if (n->tleft != NULL) {
		fprintf(s, "<");
		print_nat_splay_branch(s, n->tleft);
	}
	print_nat_splay_elm(s, n);
	if (n->tright != NULL) {
		fprintf(s, ">");
		print_nat_splay_branch(s, n->tright);
	}
}

void
print_nat_splay(FILE *s, u_char *remote)
{
	struct tunnel *t;

	if (s == NULL)
		s = stdout;
	t = tunnel_lookup(remote);
	if ((t == NULL) || (t->flags & TUNNONAT))
		return;
	if (t->tnat_root != NULL) {
		fprintf(s, "nat splay root at %lx\n", (u_long) t->tnat_root);
		print_nat_splay_branch(s, t->tnat_root);
	}
}

void
print_nat_heap(FILE *s, u_int idx)
{
	if (s == NULL)
		s = stdout;
	if ((idx == 0) || (idx > nat_heap_last))
		fprintf(s, "size=%u last=%u array=%lx\n",
			nat_heap_size, nat_heap_last, (u_long) nat_heap);
	else
		fprintf(s, "heap[%u]=%lx\n", idx, (u_long) nat_heap[idx]);
}

/* tunnels */

void
print_tunnel(FILE *s, u_char *remote)
{
	struct tunnel *t;

	if (s == NULL)
		s = stdout;
	t = tunnel_lookup(remote);
	if (t == NULL) {
		fprintf(s, "no tunnel at %s\n", addr2str(AF_INET6, remote));
		return;
	}
	if (t->flags & TUNNONAT) {
		fprintf(s, "nonat %s/%u (next at %lx)\n",
			addr2str(AF_INET, t->nnaddr), t->nnplen,
			(u_long) t->nnext);
		goto common;
	}
	if (t->buckcnt != 0)
		fprintf(s, "%u bucket ", t->buckcnt);
	if (t->blist != NULL)
		fprintf(s, "first bucket at %lx", (u_long) t->blist);
	if ((t->buckcnt != 0) || (t->blist != NULL))
		fprintf(s, "\n");
	if (t->natcnt != 0)
		fprintf(s, "%u nat ", t->natcnt);
	if (t->tnat_root != NULL)
		fprintf(s, "root nat at %lx", (u_long) t->tnat_root);
	if ((t->natcnt != 0) || (t->tnat_root != NULL))
		fprintf(s, "\n");
    common:
	if (t->mtu != tundefmtu)
		fprintf(s, "mtu=%u ", (u_int) t->mtu);
	fprintf(s, "mss=%s toobig=%s%s\n",
		(t->flags & TUNMSSFLG) != 0 ? "on" : "off",
		toobig2str(t),
		(t->flags & TUNDEBUG) != 0 ? " debug=on" : "");
}

void
print_tunnel_tree_elm(FILE *s, struct tunnel *t)
{
	if (s == NULL)
		s = stdout;
	if (t == NULL) {
		fprintf(s, "root=%lx\n", (u_long) tunnel_tree);
		return;
	}
	fprintf(s, "%lx ", (u_long) t);
	if ((t->flags & TUNGLUE) != 0) {
		fprintf(s, "bit=%u ", t->bit);
		fprintf(s, "parent=%lx ", (u_long) t->parent);
		if (t->left != NULL)
			fprintf(s, "left=%lx ", (u_long) t->left);
		if (t->right != NULL)
			fprintf(s, "right=%lx", (u_long) t->right);
		fprintf(s, "\n");
	} else {
		fprintf(s, "%s\n", addr2str(AF_INET6, t->remote));
		fprintf(s, "parent=%lx\n", (u_long) t->parent);
		if (t->left != NULL)
			fprintf(s, "left?=%lx\n", (u_long) t->left);
		if (t->right != NULL)
			fprintf(s, "right?=%lx\n", (u_long) t->right);
	}
}

void
print_tunnel_tree_branch(FILE *s, struct tunnel *t)
{
	if (t == NULL)
		return;
	if (t->left != NULL) {
		if ((t->left->left != NULL) || (t->left->right != NULL)) {
			fprintf(s, "<");
			print_tunnel_tree_branch(s, t->left);
		}
		fprintf(s, "<");
		print_tunnel_tree_elm(s, t->left);
	}
	if (t->right != NULL) {
		if ((t->right->left != NULL) || (t->right->right != NULL)) {
			fprintf(s, ">");
			print_tunnel_tree_branch(s, t->right);
		}
		fprintf(s, ">");
		print_tunnel_tree_elm(s, t->right);
	}
}

void
print_tunnel_tree(FILE *s)
{
	if (s == NULL)
		s = stdout;
	fprintf(s, "tunnel root at %lx\n", (u_long) tunnel_tree);
	if (tunnel_debugged != NULL)
		fprintf(s, "debugged tunnel at %lx\n",
			(u_long) tunnel_debugged);
	print_tunnel_tree_branch(s, tunnel_tree);
	if (tunnel_tree != NULL)
		print_tunnel_tree_elm(s, tunnel_tree);
}

/*
 * Data structures
 *	commands
 */

/* Add an IPv6 ACL entry */

void
add_acl6(u_char *addr, u_int plen)
{
	struct acl6 *a;

	a = (struct acl6 *) malloc(sizeof(*a));
	if (a == NULL) {
		logerr("malloc(acl6): %s\n", strerror(errno));
		return;
	}
	memset(a, 0, sizeof(*a));
	memcpy(a->addr, addr, 16);
	memcpy(a->mask, mask6[plen], 16);
	*acl6_last = a;
	acl6_last = &a->next;
}

/* Add an IPv4 source for NAT */

void
add_natsrc(u_char *addr)
{
	struct natsrc **nv, *ns;

	nv = (struct natsrc **) malloc((natsrccnt + 1) * sizeof(*nv));
	if (nv == NULL) {
		logerr("malloc(natsrcs): %s\n", strerror(errno));
		return;
	}
	if (natsrcs != NULL) {
		memcpy(nv, natsrcs, natsrccnt * sizeof(*nv));
		free(natsrcs);
	}
	natsrcs = nv;
	ns = (struct natsrc *) malloc(sizeof(*ns));
	if (ns == NULL) {
		logerr("malloc(natsrc): %s\n", strerror(errno));
		return;
	}
	nv[natsrccnt++] = ns;

	memset(ns, 0, sizeof(*ns));
	memcpy(ns->addr, addr, 4);
	ns->flast[0] = &ns->ffirst[0];
	ns->flast[1] = &ns->ffirst[1];
	ns->flast[2] = &ns->ffirst[2];
	ns->minport[0] = TCP_MINPORT;
	ns->minport[1] = UDP_MINPORT;
	ns->minport[2] = ICMP_MINID;
	ns->maxport[0] = TCP_MAXPORT;
	ns->maxport[1] = UDP_MAXPORT;
	ns->maxport[2] = ICMP_MAXID;

	ns->curport[0] = ns->minport[0];
	ns->curport[1] = ns->minport[1];
	ns->curport[2] = ns->minport[2];
	ns->maxport[0] = ns->minport[0] +
		((ns->maxport[0] - ns->minport[0]) / TCPBUCKSZ) * TCPBUCKSZ;
	ns->maxport[1] = ns->minport[1] +
		((ns->maxport[1] - ns->minport[1]) / UDPBUCKSZ) * UDPBUCKSZ;
	ns->maxport[2] = ns->minport[2] +
		((ns->maxport[2] - ns->minport[2]) / ICMPBUCKSZ) * ICMPBUCKSZ;
}

/* Limit port range in an IPv4 source for NAT */

void
limit_port(u_char *addr, u_int proto, u_int minport, u_int maxport)
{
	struct natsrc *ns;
	u_int i;
	u_char sz, pr;

	for (i = 0; i < natsrccnt; i++) {
		ns = natsrcs[i];
		if (ns == NULL)
			continue;
		if (memcmp(ns->addr, addr, 4) == 0)
			goto found;
	}
	logerr("no natsrc for %s\n", addr2str(AF_INET, addr));
	return;

    found:
	switch (proto) {
	case IPTCP:
		sz = TCPBUCKSZ;
		pr = TCPPR;
		break;
	case IPUDP:
		sz = UDPBUCKSZ;
		pr = UDPPR;
		break;
	default:
		sz = ICMPBUCKSZ;
		pr = ICMPPR;
		break;
	}
	ns->curport[pr] = ns->minport[pr] = minport;
	ns->maxport[pr] = minport + ((maxport - minport) / sz) * sz;
}

/* Add a static NAT entry */

void
add_snat(u_char *src6, u_int proto,
	 u_char *src, u_char *sport,
	 u_char *nsrc, u_char *nport)
{
	struct nat *n;
	struct tunnel *t;

	t = tunnel_lookup(src6);
	if (t == NULL) {
		t = add_tunnel(src6);
		if (t == NULL)
			return;
	}
	n = (struct nat *) malloc(sizeof(*n));
	if (n == NULL) {
		logerr("malloc(snat): %s\n", strerror(errno));
		return;
	}
	memset(n, 0, sizeof(*n));
	n->tunnel = t;
	n->proto = proto;
	n->flags = ALL_DST | MATCH_PORT;
	memcpy(n->src, src, 4);
	memcpy(n->nsrc, nsrc, 4);
	memcpy(n->sport, sport, 2);
	memcpy(n->nport, nport, 2);
	if (nat_tree_insert(n) != NULL) {
		logerr("rb-collision(snat)\n");
		free(n);
		return;
	}
	if (nat_splay_insert(n) != NULL) {
		logerr("splay-collision(snat)\n");
		(void) nat_tree_remove(n);
		free(n);
		return;
	}
	snatcnt++;
	t->natcnt++;
	statscnat++;
	if (t->flags & TUNDEBUG)
		debugcnat++;
	trace_nat(n, "add");
}

/* Add a A+P/PRR null NAT entry */

void
add_prr(u_char *src6, u_int proto, u_char *src, u_char *sport)
{
	struct nat *n;
	struct tunnel *t;

	t = tunnel_lookup(src6);
	if (t == NULL) {
		t = add_tunnel(src6);
		if (t == NULL)
			return;
	}
	n = (struct nat *) malloc(sizeof(*n));
	if (n == NULL) {
		logerr("malloc(prr): %s\n", strerror(errno));
		return;
	}
	memset(n, 0, sizeof(*n));
	n->tunnel = t;
	n->proto = proto;
	n->flags = ALL_DST | PRR_NULL | MATCH_PORT;
	memcpy(n->src, src, 4);
	memcpy(n->nsrc, src, 4);
	memcpy(n->sport, sport, 2);
	memcpy(n->nport, sport, 2);
	if (nat_tree_insert(n) != NULL) {
		logerr("rb-collision(prr)\n");
		free(n);
		return;
	}
	if (nat_splay_insert(n) != NULL) {
		logerr("splay-collision(prr)\n");
		(void) nat_tree_remove(n);
		free(n);
		return;
	}
	prrcnt++;
	t->natcnt++;
	statscnat++;
	if (t->flags & TUNDEBUG)
		debugcnat++;
	trace_nat(n, "add");
}

/* Add a no-nat entry */

void
add_nonat(u_char *peer, u_char *addr, u_int plen)
{
	struct tunnel *t;

	t = add_tunnel(peer);
	if (t == NULL)
		return;
	t->flags |= TUNNONAT;
	memcpy(t->nnaddr, addr, 4);
	t->nnplen = plen;
	*nonat_last = t;
	nonat_last = &t->nnext;
}

/* Delete an IPv6 ACL entry */

void
del_acl6(u_char *addr)
{
	struct acl6 *a = acl6_first;

	if (a == NULL) {
		logerr("IPv6 ACL list is empty\n");
		return;
	}
	if (memcmp(a->addr, addr, 16) == 0) {
		acl6_first = a->next;
		if (a->next == NULL)
			acl6_last = &acl6_first;
	} else if (a->next == NULL)
		goto bad;
	else {
		while (memcmp(a->next->addr, addr, 16) != 0) {
			a = a->next;
			if (a->next == NULL)
				goto bad;
		}
		a->next = a->next->next;
		if (a->next == NULL)
			acl6_last = &a->next;
	}
	free(a);
	return;

    bad:
	logerr("can't find IPv6 ACL for %s\n", addr2str(AF_INET6, addr));
}

/* Delete static NAT entry */

void
del_snat(u_char *src6, u_int proto, u_char *src, u_char *sport)
{
	struct nat nat0, *n;
	struct tunnel *t;

	t = tunnel_lookup(src6);
	if (t == NULL) {
		logerr("no tunnel %s\n", addr2str(AF_INET6, src6));
		return;
	}

	memset(&nat0, 0, sizeof(nat0));
	nat0.tunnel = t;
	nat0.proto = proto;
	nat0.flags = ALL_DST;
	memcpy(&nat0.src, src, 4);
	memcpy(&nat0.sport, sport, 2);
	n = nat_splay_find(&nat0, 1);
	if (n == NULL) {
		logerr("not found\n");
		return;
	}

	del_nat(n);
}

/* Delete a A+P/PRR null NAT entry */

void
del_prr(u_char *src6, u_int proto, u_char *src, u_char *sport)
{
	struct nat nat0, *n;
	struct tunnel *t;

	t = tunnel_lookup(src6);
	if (t == NULL) {
		logerr("no tunnel %s\n", addr2str(AF_INET6, src6));
		return;
	}

	memset(&nat0, 0, sizeof(nat0));
	nat0.tunnel = t;
	nat0.proto = proto;
	nat0.flags = ALL_DST | PRR_NULL;
	memcpy(&nat0.src, src, 4);
	memcpy(&nat0.sport, sport, 2);
	n = nat_splay_find(&nat0, 1);
	if (n == NULL) {
		logerr("not found\n");
		return;
	}

	del_nat(n);
}

/* Delete a no-nat entry (from del_tunnel()) */

void
del_nonat(u_char *peer)
{
	struct tunnel *t, *x;

	t = tunnel_lookup(peer);
	if (t == NULL) {
		logerr("already unbound[%s]\n", addr2str(AF_INET6, peer));
		return;
	} else if ((t->flags & TUNNONAT) == 0) {
		logerr("CRIT:not a nonat tunnel[%s]\n",
		       addr2str(AF_INET6, peer));
		return;
	}
	if (t == nonat_first) {
		nonat_first = t->nnext;
		if (t->nnext == NULL)
			nonat_last = &nonat_first;
	} else {
		for (x = nonat_first; ; x = x->nnext)
			if (x == NULL) {
				logerr("CRIT: can't find nonat[%s]\n",
				       addr2str(AF_INET6, peer));
				return;
			} else if (t == x->nnext)
				break;
		x->nnext = t->nnext;
		if (t->nnext == NULL)
			nonat_last = &x->nnext;
	}
	if (t == tunnel_debugged)
		tunnel_debugged = NULL;
	t->flags &= ~(TUNDEBUG | TUNNONAT);
	tuncnt--;
	trace_tunnel(t, "del");
	if ((t->hash < tunhashsz) && (tunhash[t->hash] == t))
		tunhash[t->hash] = NULL;
	tunnel_tree_remove(t);
	/* t freed by tunnel_tree_remove() */
}


/*
 * Data structures
 *	lists
 */

/* IPv6 ACL */

void
list_acl6(FILE *s)
{
	struct acl6 *a;
	u_int plen;
	
	if (s == NULL)
		s = stdout;
	for (a = acl6_first; a != NULL; a = a->next) {
		for (plen = 0; plen < 129; plen++)
			if (memcmp(a->mask, mask6[plen], 16) == 0)
				break;
		if (plen > 128) {
			logerr("illegal acl6 %s\n",
			       addr2str(AF_INET6, a->addr));
			continue;
		}
		fprintf(s, "acl6 %s/%u\n",
			addr2str(AF_INET6, a->addr), plen);
	}
}

/* natsrc/addresses */

void
list_natsrcs(FILE *s)
{
	struct natsrc *ns;
	u_int i;

	if (s == NULL)
		s = stdout;
	for (i = 0; i < natsrccnt; i++) {
		ns = natsrcs[i];
		if (ns == NULL)
			continue;
		fprintf(s, "address %s\n", addr2str(AF_INET, ns->addr));
		fprintf(s, "port %s tcp %u-%u\n",
			addr2str(AF_INET, ns->addr),
			ns->minport[0], ns->maxport[0]);
		fprintf(s, "port %s udp %u-%u\n",
			addr2str(AF_INET, ns->addr),
			ns->minport[1], ns->maxport[1]);
		fprintf(s, "port %s echo %u-%u\n",
			addr2str(AF_INET, ns->addr),
			ns->minport[2], ns->maxport[2]);
	}
}

/* nats */

void
list_nat_elm(FILE *s, struct nat *n, int lstatic, int lprr, int ldynamic)
{
	u_int p;

	if (n->tunnel == NULL) {
		logerr("nat without tunnel?\n");
		return;
	}
	if (n->flags & ALL_DST) {
		if (n->flags & PRR_NULL) {
			if (!lprr)
				return;
			if ((n->proto != IPTCP) && (n->proto != IPUDP)) {
				logerr("prr with illegal protocol\n");
				return;
			}
			p = n->sport[0] << 8;
			p |= n->sport[1];
			fprintf(s, "prr %s %s %s %u\n",
				addr2str(AF_INET6, n->tunnel->remote),
				proto2str(n->proto),
				addr2str(AF_INET, n->src), p);
		} else {
			if (!lstatic)
				return;
			if ((n->proto != IPTCP) && (n->proto != IPUDP)) {
				logerr("static nat with illegal protocol\n");
				return;
			}
			p = n->sport[0] << 8;
			p |= n->sport[1];
			fprintf(s, "nat %s %s %s %u",
				addr2str(AF_INET6, n->tunnel->remote),
				proto2str(n->proto),
				addr2str(AF_INET, n->src), p);
			p = n->nport[0] << 8;
			p |= n->nport[1];
			fprintf(s, " %s %u\n", addr2str(AF_INET, n->nsrc), p);
		}
	} else {
		if (!ldynamic)
			return;
		p = n->sport[0] << 8;
		p |= n->sport[1];
		fprintf(s, "<> %s %s %s %u",
			addr2str(AF_INET6, n->tunnel->remote),
			proto2str(n->proto),
			addr2str(AF_INET, n->src), p);
		p = n->nport[0] << 8;
		p |= n->nport[1];
		fprintf(s, " %s %u", addr2str(AF_INET, n->nsrc), p);
		p = n->dport[0] << 8;
		p |= n->dport[1];
		fprintf(s, " %s %u\n", addr2str(AF_INET, n->dst), p);
	}
}	

void
list_nat_branch(FILE *s, struct nat *n, int lstatic, int lprr, int ldynamic)
{
	if (n == NULL)
		return;
	list_nat_branch(s, n->left, lstatic, lprr, ldynamic);
	list_nat_elm(s, n, lstatic, lprr, ldynamic);
	list_nat_branch(s, n->right, lstatic, lprr, ldynamic);
}

void
list_nat(FILE *s, int lstatic, int lprr, int ldynamic)
{
	if (nat_tree == NULL)
		return;
	list_nat_branch(s, nat_tree->left, lstatic, lprr, ldynamic);
	list_nat_elm(s, nat_tree, lstatic, lprr, ldynamic);
	list_nat_branch(s, nat_tree->right, lstatic, lprr, ldynamic);
}

void
list_snat(FILE *s)
{
	if (s == NULL)
		s = stdout;
	list_nat(s, 1, 0, 0);
}

void
list_prr(FILE *s)
{
	if (s == NULL)
		s = stdout;
	list_nat(s, 0, 1, 0);
}

void
list_dynnat(FILE *s)
{
	if (s == NULL)
		s = stdout;
	list_nat(s, 0, 0, 1);
}

void
list_confnat(FILE *s)
{
	if (s == NULL)
		s = stdout;
	list_nat(s, 1, 1, 0);
}

void
list_allnat(FILE *s)
{
	if (s == NULL)
		s = stdout;
	list_nat(s, 1, 1, 1);
}

/* no-nat */

void
list_nonat(FILE *s)
{
	struct tunnel *t;

	if (s == NULL)
		s = stdout;
	for (t = nonat_first; t != NULL; t = t->nnext) {
		if ((t->flags & TUNNONAT) == 0) {
			logerr("a not no-nat in no-nat list for %s",
			       addr2str(AF_INET6, t->remote));
			continue;
		}
		fprintf(s, "nonat %s %s/%u\n",
			addr2str(AF_INET6, t->remote),
			addr2str(AF_INET, t->nnaddr), t->nnplen);
	}
}

/* tunnels */

void
list_tunnel_elm(FILE *s, struct tunnel *t)
{
	if ((t->flags & TUNGLUE) != 0)
		return;
	fprintf(s, "tunnel %s\n", addr2str(AF_INET6, t->remote));
	if (t->mtu != tundefmtu)
		fprintf(s, "mtu %u\n", (u_int) t->mtu);
}

void
list_tunnel_branch(FILE *s, struct tunnel *t)
{
	if (t == NULL)
		return;
	list_tunnel_branch(s, t->left);
	list_tunnel_elm(s, t);
	list_tunnel_branch(s, t->right);
}

void
list_tunnel_tree(FILE *s)
{
	if (s == NULL)
		s = stdout;
	if (tunnel_tree == NULL)
		return;
	list_tunnel_branch(s, tunnel_tree->left);
	list_tunnel_elm(s, tunnel_tree);
	list_tunnel_branch(s, tunnel_tree->right);
}

/*
 * Commands
 */

/* Debug commands */

int
cmd_debug_address(char *l, char *usage)
{
	l = l;
	usage = usage;
	print_natsrcs(NULL);
	return 0;
}

int
cmd_debug_bucket(char *l, char *usage)
{
	char *ep = NULL;
	long addr;

	addr = strtol(l - 1, &ep, 16);
	if (*ep != '\0') {
		logerr(usage);
		return -1;
	}
	print_bucket(NULL, (struct bucket *) addr);
	return 0;
}

int
cmd_debug_disable(char *l, char *usage)
{
	if (tunnel_debugged != NULL) {
		tunnel_debugged->flags &= ~TUNDEBUG;
		tunnel_debugged = NULL;
	}
	if (*l == '\0')
		return 0;
	if (strncasecmp(l, "clear", 5) != 0) {
		logerr(usage);
		return -1;
	}
	debugrcv6 = debugrcv4 = debugsent6 = debugsent4 = 0;
	debugnatin = debugprrin = debugnaticmpin6 = debugnaticmpin4 = 0;
	debugnatout = debugnaticmpout = 0;
	debugtcpmss = debugmsspatched = 0;
	return 0;
}

int
cmd_debug_dropped(char *l, char *usage)
{
	l = l;
	usage = usage;
	print_dropped(NULL);
	return 0;
}

int
cmd_debug_enable(char *l, char *usage)
{
	struct tunnel *t;
	u_char peer[16];

	if (inet_pton(AF_INET6, l, peer) != 1) {
		logerr(usage);
		return -1;
	}
	if (tunnel_debugged != NULL) {
		/* don't reset the flags for multiple tunnel debug?*/
#ifdef notyet
		tunnel_debugged->flags &= ~TUNDEBUG;
#endif
		tunnel_debugged = NULL;
	}
	t = tunnel_lookup(peer);
	if (t == NULL) {
		logerr("can't find tunnel[%s]\n", addr2str(AF_INET6, peer));
		return -1;
	}
	t->flags |= TUNDEBUG;
	tunnel_debugged = t;
	return 0;
}

int
cmd_debug_fragment(char *l, char *usage)
{
	char *ep = NULL;
	long addr;

	if (strncasecmp(l, "IPv6", 4) == 0) {
		print_frags6(NULL);
		return 0;
	}
	if (strncasecmp(l, "in", 2) == 0) {
		print_fragsin(NULL);
		return 0;
	}
	if (strncasecmp(l, "out", 3) == 0) {
		print_fragsout(NULL);
		return 0;
	}
	addr = strtol(l - 1, &ep, 16);
	if (*ep != '\0') {
		logerr(usage);
		return -1;
	}
	print_frag(NULL, (struct frag *) addr);
	return 0;
}

int
cmd_debug_hash(char *l, char *usage)
{
	l = l;
	usage = usage;
	print_hashes(NULL);
	return 0;
}

int
cmd_debug_nat(char *l, char *usage)
{
	char *ep = NULL;
	long addr;

	if (*l == '\0') {
		print_nat_heap(NULL, 0);
		print_nat_tree(NULL);
		return 0;
	}
	addr = strtol(l - 1, &ep, 16);
	if (*ep != '\0') {
		logerr(usage);
		return -1;
	}
	print_nat(NULL, (struct nat *) addr);
	return 0;
}

int
cmd_debug_set(char *l, char *usage)
{
	usage = usage;
	if (*l != '\0')
		debuglevel = atoi(l);
	else
		debuglevel = 1;
	return 0;
}

int
cmd_debug_stat(char *l, char *usage)
{
	l = l;
	usage = usage;
	print_stats(NULL);
	return 0;
}

int
cmd_debug_tunnel(char *l, char *usage)
{
	u_char peer[16];

	if (*l == '\0')
		print_tunnel_tree(NULL);
	else {
		if (inet_pton(AF_INET6, l, peer) != 1) {
			logerr(usage);
			return -1;
		}
		print_tunnel(NULL, peer);
	}
	return 0;
}

/* Delete commands */

int
cmd_delete_acl6(char *l, char *usage)
{
	u_char addr[16];

	if (inet_pton(AF_INET6, l, addr) != 1) {
		logerr(usage);
		return -1;
	}
	del_acl6(addr);
	return 0;
}

int
cmd_delete_nonat(char *l, char *usage)
{
	u_char peer[16];

	if (inet_pton(AF_INET6, l, peer) != 1) {
		logerr(usage);
		return -1;
	}
	del_nonat(peer);
	return 0;
}

int
cmd_delete_nat(char *l, char *usage)
{
	char *n;
	u_int proto, sp;
	u_char peer[16], src[4], sport[2];

	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	if (inet_pton(AF_INET6, l, peer) != 1)
		goto bad;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	proto = (strcasecmp(l, "tcp") == 0) ? IPTCP : IPUDP;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	if (inet_pton(AF_INET, l, src) != 1)
		goto bad;
	sp = atoi(n);
	sport[0] = sp >> 8;
	sport[1] = sp & 0xff;
	del_snat(peer, proto, src, sport);
	return 0;

    bad:
	logerr(usage);
	return -1;
}

int
cmd_delete_prr(char *l, char *usage)
{
	char *n;
	u_int proto, sp;
	u_char peer[16], src[4], sport[2];

	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	if (inet_pton(AF_INET6, l, peer) != 1)
		goto bad;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	proto = (strcasecmp(l, "tcp") == 0) ? IPTCP : IPUDP;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	if (inet_pton(AF_INET, l, src) != 1)
		goto bad;
	sp = atoi(n);
	sport[0] = sp >> 8;
	sport[1] = sp & 0xff;
	del_prr(peer, proto, src, sport);
	return 0;

    bad:
	logerr(usage);
	return -1;
}

int
cmd_delete_tunnel(char *l, char *usage)
{
	u_char peer[16];

	if (inet_pton(AF_INET6, l, peer) != 1) {
		logerr(usage);
		return -1;
	}
	del_tunnel(peer);
	return 0;
}

/* List commands */

int
cmd_list_acl6(char *l, char *usage)
{
	l = l;
	usage = usage;
	list_acl6(NULL);
	return 0;
}

int
cmd_list_address(char *l, char *usage)
{
	l = l;
	usage = usage;
	list_natsrcs(NULL);
	return 0;
}

int
cmd_list_nat(char *l, char *usage)
{
	usage = usage;
	if ((*l == '\0') || (strncasecmp(l, "conf", 3) == 0))
		list_confnat(NULL);
	else if (strncasecmp(l, "all", 3) == 0)
		list_allnat(NULL);
	else if (strncasecmp(l, "dyn", 3) == 0)
		list_dynnat(NULL);
	else if ((strncasecmp(l, "prr", 3) == 0) ||
		 (strncasecmp(l, "a+p", 3) == 0))
		list_prr(NULL);
	else
		list_snat(NULL);
	return 0;
}

int
cmd_list_nonat(char *l, char *usage)
{
	l = l;
	usage = usage;
	list_nonat(NULL);
	return 0;
}

int
cmd_list_tunnel(char *l, char *usage)
{
	l = l;
	usage = usage;
	list_tunnel_tree(NULL);
	return 0;
}

/* show aliases */

int
cmd_show_dropped(char *l, char *usage)
{
	l = l;
	usage = usage;
	print_dropped(NULL);
	return 0;
}

int
cmd_show_stat(char *l, char *usage)
{
	l = l;
	usage = usage;
	print_stats(NULL);
	return 0;
}

/* Generic command dispatcher */

int
cmd_dispatch(char *l, struct cmd *cmd, char *usage_prefix)
{
	int c;
	char usage[80];

	c = tolower(*l);

	for ( ; cmd->name != NULL && cmd->name[0] < c; cmd++)
		/* match first letter of command */;

	for ( ; cmd->name != NULL && cmd->name[0] == c; cmd++) {
		if (strncasecmp(l, cmd->name, cmd->len) == 0) {
			sprintf(usage, "%s %s %s\n",
				usage_prefix, cmd->name, cmd->usage);
			l += cmd->len;
			if (((*l == '\0') || (*l++ != ' ')) &&
			    cmd->required_args) {
				logerr(usage);
				return -1;
			}
			return cmd->func(l, usage);
		}
	}

	logerr("unknown command: %s\n", l);
	return -1;
}

/* decode one line of config/command */

int
cmdline(char *line)
{
	char *l;

	l = line;
	if (line[strlen(l) - 1] == '\n')
		line[strlen(l) - 1] = '\0';

	switch (*l) {
	case '#':
	case '\0':
		return 0;

	case '?':
		return cmd_help(l, "");

	default:
		return cmd_dispatch(l, cmd, "usage:");
	}
}

/* Top-level commands */

int
cmd_debug(char *l, char *usage)
{
	usage = usage;
	return cmd_dispatch(l, debugcmd, "usage: debug");
}

int
cmd_delete(char *l, char *usage)
{
	usage = usage;
	return cmd_dispatch(l, deletecmd, "usage: delete");
}

int
cmd_list(char *l, char *usage)
{
	usage = usage;
	return cmd_dispatch(l, listcmd, "usage: list");
}

int
cmd_show(char *l, char *usage)
{
	usage = usage;
	return cmd_dispatch(l, showcmd, "usage: show");
}

int
cmd_acl6(char *l, char *usage)
{
	char *n;
	u_int plen;
	u_char addr[16];

	n = strchr(l, '/');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	if (inet_pton(AF_INET6, l, addr) != 1)
		goto bad;
	plen = atoi(n);
	if (plen > 128)
		goto bad;
	add_acl6(addr, plen);
	return 0;

    bad:
	logerr(usage);
	return -1;
}

int
cmd_address(char *l, char *usage)
{
	char *n;
	u_char src[4];

	n = strchr(l, ' ');
	if (n != NULL)
		*n++ = '\0';
	if (strcasecmp(l, "interface") == 0) {
		l = n;
		if (n == NULL)
			goto usage_interface;
		n = strchr(l, ' ');
		if (n == NULL)
			goto usage_interface;
		*n++ = '\0';
		if (inet_pton(AF_INET, l, icmpsrc) != 1)
			goto usage_interface;
		if (inet_pton(AF_INET6, n, local6) != 1)
			goto usage_interface;
		return 0;
	}
	if (inet_pton(AF_INET, l, src) != 1) {
		logerr(usage);
		return -1;
	}
	add_natsrc(src);
	return 0;

    usage_interface:
	logerr("usage: address interface <IPv4> <IPv6>\n");
	return -1;
}

int
cmd_autotunnel(char *l, char *usage)
{
	if (strcasecmp(l, "on") == 0)
		use_autotunnel = 1;
	else if (strcasecmp(l, "off") == 0)
		use_autotunnel = 0;
	else {
		logerr(usage);
		return -1;
	}
	return 0;
}

int
cmd_defmtu(char *l, char *usage)
{
	usage = usage;
	if (tunnel_configured)
		logerr("defmtu must be defined before any 'tunnel' config\n");
	tundefmtu = atoi(l);
	if (tundefmtu < TUNMINMTU) {
		logerr("illegal value for defmtu, reset to %d\n", TUNMINMTU);
		tundefmtu = TUNMINMTU;
	}
	return 0;
}

int
cmd_eqfrag(char *l, char *usage)
{
	if (strcasecmp(l, "on") == 0)
		eqfrag = 1;
	else if (strcasecmp(l, "off") == 0)
		eqfrag = 0;
	else {
		logerr(usage);
		return -1;
	}
	return 0;
}

int
cmd_help(char *l, char *usage)
{
	struct cmd *c;

	l = l;
	usage = usage;
	logerr("available commands:\n");
	for (c = cmd; c->name != NULL; c++)
		logerr(" %s\n", c->name);
	return 0;
}

int
cmd_mtu(char *l, char *usage)
{
	char *n;
	u_int mtu;
	u_char peer[16];

	n = strchr(l, ' ');
	if (n == NULL) {
		logerr(usage);
		return -1;
	}
	*n++ = '\0';
	if (inet_pton(AF_INET6, l, peer) != 1) {
		logerr(usage);
		return -1;
	}
	mtu = atoi(n);
	set_tunnel_mtu(peer, mtu, 1);
	return 0;
}

int
cmd_mss(char *l, char *usage)
{
	char *n;
	u_char peer[16];

	if (strcasecmp(l, "on") == 0)
		enable_msspatch = 1;
	else if (strcasecmp(l, "off") == 0)
		enable_msspatch = 0;
	else {
		n = strchr(l, ' ');
		if (n == NULL)
			goto usage;
		*n++ = '\0';
		if (inet_pton(AF_INET6, l, peer) != 1)
			goto usage;
		l = n;
		if (strcasecmp(l, "on") == 0)
			set_tunnel_flag(peer, TUNMSSFLG, TUNMSSFLG);
		else if (strcasecmp(l, "off") == 0)
			set_tunnel_flag(peer, 0, TUNMSSFLG);
		else
			goto usage;
	}
	return 0;

    usage:
	logerr(usage);
	return -1;
}

int
cmd_nonat(char *l, char *usage)
{
	char *n;
	u_int plen;
	u_char peer[16], addr[4];

	n = strchr(l, ' ');
	if (n == NULL)
		goto usage;
	*n++ = '\0';
	if (inet_pton(AF_INET6, l, peer) != 1)
		goto usage;
	l = n;
	n = strchr(l, '/');
	if (n == NULL)
		goto usage;
	*n++ = '\0';
	if (inet_pton(AF_INET, l, addr) != 1)
		goto usage;
	plen = atoi(n);
	if (plen > 32)
		goto usage;
	add_nonat(peer, addr, plen);
	return 0;

    usage:
	logerr(usage);
	return -1;
}	

int
cmd_nat(char *l, char *usage)
{
	char *n;
	u_int proto, sp, np;
	u_char peer[16], src[4], nsrc[4], sport[2], nport[2];

	n = strchr(l, ' ');
	if (n == NULL)
		goto usage;
	*n++ = '\0';
	if (inet_pton(AF_INET6, l, peer) != 1)
		goto usage;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto usage;
	*n++ = '\0';
	proto = (strcasecmp(l, "tcp") == 0) ? IPTCP : IPUDP;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto usage;
	*n++ = '\0';
	if (inet_pton(AF_INET, l, src) != 1)
		goto usage;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto usage;
	*n++ = '\0';
	sp = atoi(l);
	sport[0] = sp >> 8;
	sport[1] = sp & 0xff;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto usage;
	*n++ = '\0';
	if (inet_pton(AF_INET, l, nsrc) != 1)
		goto usage;
	np = atoi(n);
	nport[0] = np >> 8;
	nport[1] = np & 0xff;
	add_snat(peer, proto, src, sport, nsrc, nport);
	return 0;

    usage:
	logerr(usage);
	return -1;
}

int
cmd_noop(char *l, char *usage)
{
	l = l;
	usage = usage;
	fprintf(stderr, "alive\n");
	return 0;
}

int
cmd_port(char *l, char *usage)
{
	char *n;
	u_char src[4];
	u_int proto, minport, maxport;

	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	if (inet_pton(AF_INET, l, src) != 1)
		goto bad;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	if (strcasecmp(l, "tcp") == 0)
		proto = IPTCP;
	else if (strcasecmp(l, "udp") == 0)
		proto = IPUDP;
	else if (strcasecmp(l, "echo") == 0)
		proto = IPICMP;
	else
		goto bad;
	l = n;
	n = strchr(l, '-');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	minport = atoi(l);
	maxport = atoi(n);
	if ((minport <= 0) || (maxport > 65535) || (minport > maxport))
		goto bad;
	limit_port(src, proto, minport, maxport);
	return 0;

    bad:
	logerr(usage);
	return -1;
}

int
cmd_prr(char *l, char *usage)
{
	char *n;
	u_int proto, sp;
	u_char peer[16], src[4], sport[2];

	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	if (inet_pton(AF_INET6, l, peer) != 1)
		goto bad;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	proto = (strcasecmp(l, "tcp") == 0) ? IPTCP : IPUDP;
	l = n;
	n = strchr(l, ' ');
	if (n == NULL)
		goto bad;
	*n++ = '\0';
	if (inet_pton(AF_INET, l, src) != 1)
		goto bad;
	sp = atoi(n);
	sport[0] = sp >> 8;
	sport[1] = sp & 0xff;
	add_prr(peer, proto, src, sport);
	return 0;

    bad:
	logerr(usage);
	return -1;
}

int
cmd_quit(char *l, char *usage)
{
	l = l;
	usage = usage;
	fprintf(stderr, "quitting\n");
	return 1;
}

int
cmd_toobig(char *l, char *usage)
{
	char *n;
	u_char flags;
	u_char peer[16];

	if (strcasecmp(l, "on") == 0) {
		if (tunnel_configured)
			goto err_tunnel_configured;
		default_toobig = TUNTBICMP;
	} else if (strcasecmp(l, "off") == 0) {
		if (tunnel_configured)
			goto err_tunnel_configured;
		default_toobig = 0;
	} else if (strcasecmp(l, "strict") == 0) {
		if (tunnel_configured)
			goto err_tunnel_configured;
		default_toobig = TUNTBDROP | TUNTBICMP;
	} else {
		n = strchr(l, ' ');
		if (n == NULL)
			goto usage;
		*n++ = '\0';
		if (inet_pton(AF_INET6, l, peer) != 1)
			goto usage;
		l = n;
		if (strcasecmp(l, "on") == 0)
			flags = TUNTBICMP;
		else if (strcasecmp(l, "off") == 0)
			flags = 0;
		else if (strcasecmp(l, "strict") == 0)
			flags = TUNTBDROP | TUNTBICMP;
		else
			goto usage;
		set_tunnel_flag(peer, flags, TUNTBDROP | TUNTBICMP);
	}
	return 0;

    err_tunnel_configured:
	logerr("toobig policy must be specified before any 'tunnel' config\n");
	return -1;

    usage:
	logerr(usage);
	return -1;
}

int
cmd_tunnel(char *l, char *usage)
{
	u_char peer[16];

	tunnel_configured = 1;
	if (inet_pton(AF_INET6, l, peer) != 1) {
		logerr(usage);
		return -1;
	}
	if (add_tunnel(peer) == NULL)
		return -1;
	return 0;
}

int
cmd_trace(char *l, char *usage)
{
	usage = usage;
	trace_open(l);
	return 0;
}

/* load config */

int
load(void)
{
	FILE *f;
	char *l, line[200];

	f = fopen(CGNCONFIG, "r");
	if (f == NULL) {
		logerr("fopen(config): %s\n", strerror(errno));
		return -1;
	}
	line[sizeof(line) - 1] = '\0';
	for (;;) {
		l = fgets(line, sizeof(line) - 1, f);
		if (l == NULL) {
			fclose(f);
			return 0;
		}
		if (cmdline(line) < 0)
			return -1;
	}
	logerr("unreachable\n");
	return -1;
}

/* read commands */

int
commands(int fd)
{
	static char cbuf[400];
	static size_t cpos = 0;
	char *l, line[200];
	int cc;
	size_t len;

	line[sizeof(line) - 1] = '\0';
	for (;;) {
		if (cpos == 0)
			memset(cbuf, 0, sizeof(cbuf));
		if (cpos > 0) {
			l = strchr(cbuf, '\n');
			if (l != NULL) {
				*l++ = '\0';
				len = strlen(cbuf) + 1;
				if (len > sizeof(line)) {
					memmove(cbuf,
						cbuf + len,
						sizeof(cbuf) - len);
					cpos -= len;
					continue;
				}
				memcpy(line, cbuf, len);
				memmove(cbuf, cbuf + len, sizeof(cbuf) - len);
				cpos -= len;
				if (cmdline(line) == 1)
					return 1;
				continue;
			}
		}
		if (ioctl(fd, FIONREAD, &cc) < 0) {
			logerr("stdin FIONREAD: %s\n", strerror(errno));
			exit(-1);
		}
		if (cc == 0)
			return 0;
		cc = read(fd, cbuf + cpos, sizeof(cbuf) - cpos - 1);
		if (cc < 0) {
			logerr("stdin read: %s\n", strerror(errno));
			exit(-1);
		}
		if (cc == 0)
			return 0;
		cpos += cc;
	}
}

/*
 * Packet utils
 */

/* Compute checksum */

int
in_cksum(u_char *p, u_int l)
{
	int sum = 0;

	while (l > 1) {
		sum += *p++ << 8;
		sum += *p++;
		l -= 2;
	}
	if (l == 1)
		sum += *p << 8;
	sum = ((sum >> 16) & 0xffff) + (sum & 0xffff);
	sum += sum >> 16;
	return (0xffff & ~sum);
}

int
pseudo_cksum(u_char *p, u_char *pl, u_int l)
{
	int sum;
	u_char saved[4];

	memcpy(saved, p + 8, 4);
	p[8] = 0;
	p[10] = pl[0];
	p[11] = pl[1];
	sum = in_cksum(p + 8, l - 8);
	memcpy(p + 8, saved, 4);
	return sum;
}

int
pseudo6_cksum(void)
{
	int sum;
	u_int l;
	u_char saved[8];

	memcpy(saved, buf6, 8);
	l = len - IP6HDRLEN;
	buf6[0] = l >> 24;
	buf6[1] = (l >> 16) & 0xff;
	buf6[2] = (l >> 8) & 0xff;
	buf6[3] = l & 0xff;
	buf6[4] = buf6[5] = buf6[6] = 0;
	buf6[7] = saved[IP6PROTO];
	sum = in_cksum(buf6, len);
	memcpy(buf6, saved, 8);
	return sum;
}

/* Apply RFC 1624 HC' = ~(~HC + ~m + m') */

void
fix_cksum(u_char *psum,
	  u_char *oaddr, u_char *naddr,
	  u_char *oport, u_char *nport)
{
	int sum;
	int m;

	sum = psum[0] << 8;
	sum |= psum[1];
	sum = ~sum & 0xffff;
	if ((oaddr != NULL) && (memcmp(oaddr, naddr, 4) != 0)) {
		m = oaddr[0] << 8;
		m |= oaddr[1];
		m += oaddr[2] << 8;
		m += oaddr[3];
		m = (m >> 16) + (m & 0xffff);
		sum += m ^ 0xffff;
		m = naddr[0] << 8;
		m |= naddr[1] & 0xff;
		m += naddr[2] << 8;
		m += naddr[3] & 0xff;
		sum += m;
	}
	if ((oport != NULL) && (memcmp(oport, nport, 2) != 0)) {
		m = oport[0] << 8;
		m |= oport[1];
		sum += m ^ 0xffff;
		m = nport[0] << 8;
		m |= nport[1];
		sum += m;
	}
	sum = (sum >> 16) + (sum & 0xffff);
	sum += sum >> 16;
	sum = ~sum & 0xffff;
	psum[0] = sum >> 8;
	psum[1] = sum & 0xff;
}

void
fix_msscksum(u_char *psum, u_char *pmss, u_short newmss)
{
	int sum;
	int m;

	logit(10, "fix_msscksum");

	sum = psum[0] << 8;
	sum |= psum[1];
	sum = ~sum & 0xffff;
	m = pmss[0] << 8;
	m |= pmss[1];
	m = (m >> 16) + (m & 0xffff);
	sum += m ^ 0xffff;
	pmss[0] = newmss >> 8;
	pmss[1] = newmss;
	m = pmss[0] << 8;
	m |= pmss[1];
	sum += m;
	sum = (sum >> 16) + (sum & 0xffff);
	sum += sum >> 16;
	sum = ~sum & 0xffff;
	psum[0] = sum >> 8;
	psum[1] = sum & 0xff;
}

/* TODO: replace seqd by a structure, add seq/ack flag, patch after position */

void
fix_ftpseqcksum(u_char *psum, u_char *pseq, int32_t seqd)
{
	int sum;
	int32_t s0, s;

	logit(10, "fix_ftpseqcksum");

	sum = psum[0] << 8;
	sum |= psum[1];
	sum = ~sum & 0xffff;
	s0 = pseq[0] << 24;
	s0 |= pseq[1] << 16;
	s0 |= pseq[2] << 8;
	s0 |= pseq[3];
	s = ((s0 >> 16) & 0xffff) + (s0 & 0xffff);
	s = (s >> 16) + (s & 0xffff);
	sum += s ^ 0xffff;
	s0 += seqd;
	pseq[0] = (s0 >> 24) & 0xff ;
	pseq[1] = (s0 >> 16) & 0xff;
	pseq[2] = (s0 >> 8) & 0xff;
	pseq[3] = s0 & 0xff;
	s = ((s0 >> 16) & 0xffff) + (s0 & 0xffff);
	sum += s;
	sum = (sum >> 16) + (sum & 0xffff);
	sum += sum >> 16;
	sum = ~sum & 0xffff;
	psum[0] = sum >> 8;
	psum[1] = sum & 0xff;
}	

/* Defrag IPv4 (from tunnel or Internet)
 * This is called on receipt of a fragment.  It reassembles the packet if
 * possible, or stores the fragment for later reassembly.
 * Returns 1 if packet reassembled (in buf4), 0 otherwise.
 */

int
defrag(struct tunnel *t)
{
	struct frag **head, ***last, *f, *p, *q;
	u_int off;
	u_char more = 0;
	u_short hash;
	int cksum;

	if (t == NULL) {
		logit(10, "defrag(out)");
		statsfrout++;

		if (fragsoutcnt >= FRAGOUT_MAXCNT) {
			logit(10, "too many IPv4 out fragments");
			statsdropped[DR_FOUTCNT]++;
			return 0;
		}
		head = &fragsout_first;
		last = &fragsout_last;
	} else {
		logit(10, "defrag(in)");
		statsfrgin++;
		if (t->flags & TUNDEBUG)
			debugfrgin++;

		if (fragsincnt >= FRAGIN_MAXCNT) {
			logit(10, "too many IPv4 in fragments");
			statsdropped[DR_FINCNT]++;
			if (t->flags & TUNDEBUG)
				debugdropped[DR_FINCNT]++;
			return 0;
		} else if (t->frg4cnt >= FRAGTN4_MAXCNT) {
			logit(10, "too many IPv4 in fragments for tunnel %s",
			      addr2str(AF_INET6, t->remote));
			statsdropped[DR_FINTCNT]++;
			if (t->flags & TUNDEBUG)
				debugdropped[DR_FINTCNT]++;
			return 0;
		}
		head = &fragsin_first;
		last = &fragsin_last;
	}

	if (buf4[IPOFFH] & IPMF)
		more = 1;
	len -= IPHDRLEN;
	if (more && (len & 0x7)) {
		logerr("badfrag4\n");
		statsdropped[DR_BADF4]++;
		if ((t != NULL) && (t->flags & TUNDEBUG))
			debugdropped[DR_BADF4]++;
		return 0;
	}
	off = (buf4[IPOFFH] & IPOFFMSK) << 8;
	off |= buf4[IPOFFL];
	off <<= 3;

	tfhlookups++, pfhlookups++;
	hash = jhash_frag();
	hash &= fraghashsz - 1;
	f = fraghash[hash];
	if ((f != NULL) &&
	    (f->tunnel == t) &&
	    (f->buf[IPPROTO] == buf4[IPPROTO]) &&
	    (memcmp(f->buf + IPID, buf4 + IPID, 2) == 0) &&
	    (memcmp(f->buf + IPSRC, buf4 + IPSRC, 8) == 0)) {
		tfhhits++, pfhhits++;
		goto found;
	}
	for (f = *head; f != NULL; f = f->next) {
		if (f->tunnel != t)
			continue;
		if (f->buf[IPPROTO] != buf4[IPPROTO])
			continue;
		if (memcmp(f->buf + IPID, buf4 + IPID, 2) != 0)
			continue;
		if (memcmp(f->buf + IPSRC, buf4 + IPSRC, 8) == 0)
			break;
	}
	if (f == NULL) {
		f = (struct frag *) malloc(sizeof(*f));
		if (f == NULL) {
			statsdropped[DR_F4MEM]++;
			if ((t != NULL) && (t->flags & TUNDEBUG))
				debugdropped[DR_F4MEM]++;
			return 0;
		}
		memset(f, 0, sizeof(*f));
		f->buf = (u_char *) malloc(IPHDRLEN);
		if (f->buf == NULL) {
			statsdropped[DR_F4MEM]++;
			if ((t != NULL) && (t->flags & TUNDEBUG))
				debugdropped[DR_F4MEM]++;
			free(f);
			return 0;
		}
		memcpy(f->buf, buf4, IPHDRLEN);
		f->len = IPHDRLEN;
		f->off = off;
		f->more = more;
		f->expire = seconds + FRAG_LIFETIME;
		f->tunnel = t;
		if ((f->next = *head) != NULL)
			(*head)->prev = &f->next;
		else
			*last = &f->next;
		*head = f;
		f->prev = head;
		if (t != NULL) {
			fragsincnt++;
			t->frg4cnt++;
		} else
			fragsoutcnt++;
	}
	/* don't promote to head because of timeouts */
	f->hash = hash;
	fraghash[hash] = f;
    found:
	for (p = NULL, q = f->list; q != NULL; p = q, q = q->next) {
		if (q->off == off)
			return 0;
		if (q->off > off)
			break;
		if (!q->more)
			return 0;
	}
	if (((q != NULL) && (off + len > q->off)) ||
	    ((q != NULL) && !more) ||
	    ((p != NULL) && (p->off + p->len > off)))
		return 0;
	q = (struct frag *) malloc(sizeof(*q));
	if (q == NULL) {
		statsdropped[DR_F4MEM]++;
		if ((t != NULL) && (t->flags & TUNDEBUG))
			debugdropped[DR_F4MEM]++;
		return 0;
	}
	memset(q, 0, sizeof(*q));
	q->buf = (u_char *) malloc(len);
	if (q->buf == NULL) {
		statsdropped[DR_F4MEM]++;
		if ((t != NULL) && (t->flags & TUNDEBUG))
			debugdropped[DR_F4MEM]++;
		free(q);
		return 0;
	}
	memcpy(q->buf, buf4 + IPHDRLEN, len);
	q->len = len;
	q->off = off;
	q->more = more;
	if (p != NULL) {
		q->next = p->next;
		p->next = q;
	} else {
		if ((off == 0) && (f->off != 0)) {
			memcpy(f->buf, buf4, IPHDRLEN);
			f->off = off;
			f->more = more;
		}
		q->next = f->list;
		f->list = q;
	}

	if (f->off != 0)
		return 0;
	off = 0;
	for (p = NULL, q = f->list; q != NULL; p = q, q = q->next) {
		if (q->off != off)
			return 0;
		off += q->len;
	}
	if ((p == NULL) || p->more)
		return 0;
	len = off + IPHDRLEN;
	if (len > IPMAXLEN) {
		statsdropped[DR_BADF4]++;
		if ((t != NULL) && (t->flags & TUNDEBUG))
			debugdropped[DR_BADF4]++;
		del_frag4(f);
		return 0;
	}

	memcpy(buf4, f->buf, IPHDRLEN);
	buf4[IPOFFH] &= ~(IPDF|IPMF);
	for (p = f->list; p != NULL; p = p->next)
		memcpy(buf4 + IPHDRLEN + p->off, p->buf, p->len);
	buf4[IPLENH] = len >> 8;
	buf4[IPLENL] = len & 0xff;
	buf4[IPCKSUMH] = buf4[IPCKSUML] = 0;
	cksum = in_cksum(buf4, IPHDRLEN);
	buf4[IPCKSUMH] = cksum >> 8;
	buf4[IPCKSUML] = cksum & 0xff;
	del_frag4(f);

	logit(10, "reassembled IPv4 packet");
	if (t != NULL) {
		statsreasin++;
		if (t->flags & TUNDEBUG)
			debugreasin++;
	} else
		statsreasout++;

	return 1;
}

/* Defrag IPv6 (from tunnel)
 * This is called on receipt of a fragment.  It reassembles the packet if
 * possible, or stores the fragment for later reassembly.
 * Returns 1 if packet reassembled (in buf6), 0 otherwise.
 */

int
defrag6(struct tunnel *t)
{
	struct frag *f, *p, *q;
	u_int off;
	u_char more = 0;

	logit(10, "defrag6");
	statsfrgin6++;
	if (t->flags & TUNDEBUG)
		debugfrgin6++;

	if (frags6cnt >= FRAG6_MAXCNT) {
		logit(10, "too many IPv6 fragments");
		statsdropped[DR_F6CNT]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_F6CNT]++;
		return 0;
	} else if (t->frg6cnt >= FRAGTN6_MAXCNT) {
		logit(10, "too many IPv6 fragments for tunnel %s",
		      addr2str(AF_INET6, t->remote));
		statsdropped[DR_F6TCNT]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_F6TCNT]++;
		return 0;
	}

	if (buf6[IP6FOFFL] & IP6FMF)
		more = 1;
	len -= IP6FLEN;
	if (more && (len & 0x7)) {
		logerr("badfrag6\n");
		statsdropped[DR_BADF6]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BADF6]++;
		return 0;
	}
	off = buf6[IP6FOFFH] << 8;
	off |= buf6[IP6FOFFL] & IP6FMSK;

	for (f = frags6_first; f != NULL; f = f->next) {
		if (f->tunnel != t)
			continue;
		if (f->buf[IP6FPROTO] != buf6[IP6FPROTO])
			continue;
		/* matching tunnel == matching addresses */
		if (memcmp(f->buf + IP6FID, buf6 + IP6FID, 4) == 0)
			break;
	}
	if (f == NULL) {
		f = (struct frag *) malloc(sizeof(*f));
		if (f == NULL) {
			statsdropped[DR_F6CNT]++;
			if (t->flags & TUNDEBUG)
				debugdropped[DR_F6CNT]++;
			return 0;
		}
		memset(f, 0, sizeof(*f));
		f->buf = (u_char *) malloc(IP6FLEN);
		if (f->buf == NULL) {
			statsdropped[DR_F6CNT]++;
			if (t->flags & TUNDEBUG)
				debugdropped[DR_F6CNT]++;
			free(f);
			return 0;
		}
		memcpy(f->buf, buf6, IP6FLEN);
		f->len = IP6FLEN;
		f->off = off;
		f->more = more;
		f->expire = seconds + FRAG_LIFETIME;
		f->tunnel = t;
		if ((f->next = frags6_first) != NULL)
			frags6_first->prev = &f->next;
		else
			frags6_last = &f->next;
		frags6_first = f;
		f->prev = &frags6_first;
		frags6cnt++;
		t->frg6cnt++;
	}
	/* don't promote to head because of timeouts */
	for (p = NULL, q = f->list; q != NULL; p = q, q = q->next) {
		if (q->off == off)
			return 0;
		if (q->off > off)
			break;
		if (!q->more)
			return 0;
	}
	if (((q != NULL) && (off + len > q->off)) ||
	    ((q != NULL) && !more) ||
	    ((p != NULL) && (p->off + p->len > off)))
		return 0;
	q = (struct frag *) malloc(sizeof(*q));
	if (q == NULL) {
		statsdropped[DR_F6CNT]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_F6CNT]++;
		return 0;
	}
	memset(q, 0, sizeof(*q));
	q->buf = (u_char *) malloc(len);
	if (q->buf == NULL) {
		statsdropped[DR_F6CNT]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_F6CNT]++;
		free(q);
		return 0;
	}
	memcpy(q->buf, buf6 + IP6FLEN, len);
	q->len = len;
	q->off = off;
	q->more = more;
	if (p != NULL) {
		q->next = p->next;
		p->next = q;
	} else {
		if ((off == 0) && (f->off != 0)) {
			memcpy(f->buf, buf6, IP6FLEN);
			f->off = off;
			f->more = more;
		}
		q->next = f->list;
		f->list = q;
	}

	if (f->off != 0)
		return 0;
	off = 0;
	for (p = NULL, q = f->list; q != NULL; p = q, q = q->next) {
		if (q->off != off)
			return 0;
		off += q->len;
	}
	if ((p == NULL) || p->more)
		return 0;
	len = off;
	if (len > IPMAXLEN) {
		del_frag6(f);
		statsdropped[DR_BADF6]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BADF6]++;
		return 0;
	}

	memcpy(buf6, f->buf, IP6HDRLEN);
	for (p = f->list; p != NULL; p = p->next)
		memcpy(buf6 + IP6HDRLEN + p->off, p->buf, p->len);
	buf6[IP6LENH] = len >> 8;
	buf6[IP6LENL] = len & 0xff;
	len += IP6HDRLEN;
	del_frag6(f);

	logit(10, "reassembled IPv6 packet");
	statsreas6++;
	if (t->flags & TUNDEBUG)
		debugreas6++;

	return 1;
}

/* patch TCP MSS */

void
patch_tcpmss(struct tunnel *t)
{
	u_int hl, i, found = 0;
	u_short mss;

	/* is patching enabled? */
	if ((t->flags & TUNMSSFLG) == 0)
		return;
	/* need SYN flag */
	if ((buf4[TCPFLAGS] & TCPFSYN) == 0)
		return;
	hl = (buf4[TCPOFF] & TCPOFFMSK) >> 2;
	/* no data */
	if (hl + IPHDRLEN != len)
		return;
	/* but some options */
	if (hl <= TCPHDRLEN)
		return;
	/* scan option */
	i = IPHDRLEN + TCPHDRLEN;
	while (i < len) {
		if (buf4[i] == TCPOPTEOL) {
			if (found == 0)
				loginfo("no TCP MSS\n");
			break;
		}
		if (buf4[i] == TCPOPTNOP) {
			i++;
			continue;
		}
		if (i + 2 > len) {
			logerr("TCP options overrun0\n");
			return;
		}
		if (buf4[i + 1] < 2) {
			logerr("bad TCP option length\n");
			return;
		}
		if (i + buf4[i + 1] > len) {
			logerr("TCP option overrun\n");
			return;
		}
		if (buf4[i] == TCPOPTMD5) {
			loginfo("TCP MD5\n");
			return;
		}
		if (buf4[i] == TCPOPTMSS) {
			if (found != 0)
				logerr("duplicate TCP MSS\n");
			else
				found = i;
		}
		i += buf4[i + 1];
	}
	if (found == 0) {
		loginfo("no TCP MSS (after scan)\n");
		return;
	}
	i = found;
	if (buf4[i + 1] != TCPOPTMSSLEN) {
		logerr("bad TCP MSS option length\n");
		return;
	}
	i += 2;
	mss = buf4[i] << 8;
	mss |= buf4[i + 1];
	statstcpmss++;
	if (t->flags & TUNDEBUG)
		debugtcpmss++;
	/* no patch needed */
	if ((mss + IPHDRLEN + TCPHDRLEN) <= (t->mtu - IP6HDRLEN))
		return;
	fix_msscksum(buf4 + TCPCKSUMH, buf4 + i,
		     t->mtu - (IP6HDRLEN + IPHDRLEN + TCPHDRLEN));
	statsmsspatched++;
	if (t->flags & TUNDEBUG)
		debugmsspatched++;
}

/* detect TCP closing (from tunnel) */

int
tcpstate_in(struct nat *n)
{
	if (n->timeout == 0)
		return 0;
	if ((n->tcpst == TCP_DEFAULT) && (buf4[TCPFLAGS] & TCPFACK))
		n->tcpst = TCP_ACKED;
	if ((buf4[TCPFLAGS] & (TCPFFIN|TCPFRST)) == 0)
		return 0;
	if (buf4[TCPFLAGS] & TCPFRST)
		n->tcpst |= TCP_CLOSED_BOTH;
	else if (n->tcpst & TCP_CLOSED_IN)
		return 0;
	else
		n->tcpst |= TCP_CLOSED_IN;
	if ((n->tcpst & TCP_CLOSED_OUT) == 0)
		return 0;
	logit(10, "tcpstate_in changes lifetime");
	if (buf4[TCPFLAGS] & TCPFRST) {
		if (n->tcpst & TCP_ACKED)
			n->lifetime = ICMP_LIFETIME;
		else
			n->lifetime = RETRANS_LIFETIME;
	} else
		n->lifetime = CLOSED_TCP_LIFETIME;
	if (n->timeout <= seconds + (time_t) n->lifetime)
		return 0;
	n->timeout = seconds + n->lifetime;
	nat_heap_increased(n->heap_index);
	return 1;
}

/* detect TCP closing (from Internet) */

int
tcpstate_out(struct nat *n)
{
	if (n->timeout == 0)
		return 0;
	if ((n->tcpst == TCP_DEFAULT) && (buf4[TCPFLAGS] & TCPFACK))
		n->tcpst = TCP_ACKED;
	if ((buf4[TCPFLAGS] & (TCPFFIN|TCPFRST)) == 0)
		return 0;
	if (buf4[TCPFLAGS] & TCPFRST)
		n->tcpst |= TCP_CLOSED_BOTH;
	else if (n->tcpst & TCP_CLOSED_OUT)
		return 0;
	else
		n->tcpst |= TCP_CLOSED_OUT;
	if ((n->tcpst & TCP_CLOSED_IN) == 0)
		return 0;
	logit(10, "tcpstate_out changes lifetime");
	if (buf4[TCPFLAGS] & TCPFRST) {
		if (n->tcpst & TCP_ACKED)
			n->lifetime = ICMP_LIFETIME;
		else
			n->lifetime = RETRANS_LIFETIME;
	} else
		n->lifetime = CLOSED_TCP_LIFETIME;
	if (n->timeout <= seconds + (time_t) n->lifetime)
		return 0;
	n->timeout = seconds + n->lifetime;
	nat_heap_increased(n->heap_index);
	return 1;
}

#ifdef FTPALG
/* Get the NAT entry for FTP DATA */

struct nat *
get_ftptempdata(struct nat *n0, u_char *np)
{
	struct nat *n;

	logit(10, "get_ftptempdata");

	for (n = n0->xfirst; n != NULL; n = n->xnext)
		if ((n->sport[0] == np[0]) && (n->sport[1] == np[1]))
			break;
	if (n != NULL)
		return n;

	if (n0->tunnel->natcnt >= MAXNATCNT) {
		logit(10, "too many nat entries(get_ftptempdata)");
		statsdropped[DR_NATCNT]++;
		if (n0->tunnel->flags & TUNDEBUG)
			debugdropped[DR_NATCNT]++;
		return NULL;
	}

	n = (struct nat *) malloc(sizeof(*n));
	if (n == NULL) {
		logerr("malloc(get_ftptempdata): %s\n", strerror(errno));
		statsdropped[DR_NEWNAT]++;
		if (n0->tunnel->flags & TUNDEBUG)
			debugdropped[DR_NEWNAT]++;
		return NULL;
	}
	memset(n, 0, sizeof(*n));
	n->tunnel = n0->tunnel;
	n->proto = n0->proto;
	memcpy(n->src, n0->src, 4);
	memcpy(n->nsrc, n0->nsrc, 4);
	memcpy(n->sport, np, 2);
	n->flags = ALL_DST | MATCH_PORT;
	n->timeout = seconds + CLOSED_TCP_LIFETIME;
	n->lifetime = TCP_LIFETIME;
	if (!new_nat(n)) {
		statsdropped[DR_NEWNAT]++;
		if (n0->tunnel->flags & TUNDEBUG)
			debugdropped[DR_NEWNAT]++;
		return NULL;
	}
	n->xnext = n0->xfirst;
	if (n->xnext != NULL)
		n0->xfirst->xprev = &n->xnext;
	n0->xfirst = n;
	n->xprev = &n0->xfirst;
	logit(10, "get_ftptempdata got %lx", (u_long) n);
	return n;
}

/* Patch a FTP command in a packet */

struct nat *
patch_ftpcmd(struct nat *n, char *cmd)
{
	u_int off, l;
	int delta, cksum;
	u_char tlen[2];

	/* remove checksum check when done before */
	tlen[0] = (len - IPHDRLEN) >> 8;
	tlen[1] = (len - IPHDRLEN) & 0xff;
	if (pseudo_cksum(buf4, tlen, len) != 0) {
		logerr("checksum(patch_ftpcmd)\n");
		return NULL;
	}

	off = IPHDRLEN + ((buf4[TCPOFF] & TCPOFFMSK) >> 2);
	l = strlen(cmd);
	delta = l - (len - off);
	memcpy(buf4 + off, cmd, l);
	len += delta;
	buf4[IPLENH] = len >> 8;
	buf4[IPLENL] = len & 0xff;
	buf4[IPCKSUMH] = buf4[IPCKSUML] = 0;
	cksum = in_cksum(buf4, IPHDRLEN);
	buf4[IPCKSUMH] = cksum >> 8;
	buf4[IPCKSUML] = cksum & 0xff;
	buf4[TCPCKSUMH] = buf4[TCPCKSUML] = 0;
	tlen[0] = (len - IPHDRLEN) >> 8;
	tlen[1] = (len - IPHDRLEN) & 0xff;
	cksum = pseudo_cksum(buf4, tlen, len);
	buf4[TCPCKSUMH] = cksum >> 8;
	buf4[TCPCKSUML] = cksum & 0xff;
	n->ftpseqd += delta;
	return n;
}

/* Patch FTP packet from tunnel to Internet */

struct nat *
patch_ftpin(struct nat *n0, int way)
{
	struct nat *td, *n = n0;
	char pat[128];
	u_int off;
	u_short ps;
	u_char p[2];

	logit(10, "patch_ftpin%d %lx", way, (u_long) n);

	off = IPHDRLEN + ((buf4[TCPOFF] & TCPOFFMSK) >> 2);
	buf4[len] = '\0';
	memset(pat, 0, 128);
	if (way == 0) {
		if (n->ftpseqd != 0)
			fix_ftpseqcksum(buf4 + TCPCKSUMH,
					buf4 + TCPSEQ,
					n->ftpseqd);
		/* TODO: improve scan... */
		snprintf(pat, 128,
			 "PORT %u,%u,%u,%u,%%hhu,%%hhu\r\n",
			 n->src[0], n->src[1], n->src[2], n->src[3]);
		if (sscanf((char *)buf4 + off, pat, p, p + 1) == 2) {
			statsftpport++;
			if (n->tunnel->flags & TUNDEBUG)
				debugftpport++;
			td = get_ftptempdata(n, p);
			if (td == NULL)
				return NULL;
			snprintf(pat, 128,
				 "PORT %u,%u,%u,%u,%u,%u\r\n",
				 td->nsrc[0], td->nsrc[1],
				 td->nsrc[2], td->nsrc[3],
				 td->nport[0], td->nport[1]);
			n = patch_ftpcmd(n, pat);
			return n;
		}
		snprintf(pat, 128,
			 "EPRT |1|%u.%u.%u.%u|%%hu|\r\n",
			 n->src[0], n->src[1], n->src[2], n->src[3]);
		if (sscanf((char *)buf4 + off, pat, &ps) == 1) {
			statsftpeprt++;
			if (n->tunnel->flags & TUNDEBUG)
				debugftpeprt++;
			td = get_ftptempdata(n, p);
			if (td == NULL)
				return NULL;
			ps = td->nport[0] << 8;
			ps |= td->nport[1];
			snprintf(pat, 128,
				 "EPRT |1||%u.%u.%u.%u|%u|\r\n",
				 td->nsrc[0], td->nsrc[1],
				 td->nsrc[2], td->nsrc[3], ps);
			n = patch_ftpcmd(n, pat);
			return n;
		}
	} else {
		if (n0->flags & ALL_DST) {
			for (n = n0->xfirst; n != NULL; n = n->xnext) {
				if ((memcmp(buf4 + IPDST, n->dst, 4) == 0) &&
				    (memcmp(buf4 + IPDPORT, n->dport, 2) == 0))
					break;
			}
			if (n == NULL) {
				logerr("orphan FTP server packet\n");
				return NULL;
			}
		}
		if (n->ftpseqd != 0)
			fix_ftpseqcksum(buf4 + TCPCKSUMH,
					buf4 + TCPSEQ,
					n->ftpseqd);
		/* TODO: less strict pattern */
		snprintf(pat, 128,
			 "227 Entering Passive Mode "
			 "(%u,%u,%u,%u,%%hhu,%%hhu)\r\n",
			 n->src[0], n->src[1], n->src[2], n->src[3]);
		if (sscanf((char *)buf4 + off, pat, p, p + 1) == 2) {
			statsftp227++;
			if (n->tunnel->flags & TUNDEBUG)
				debugftp227++;
			td = get_ftptempdata(n, p);
			if (td == NULL)
				return NULL;
			snprintf(pat, 128,
				 "227 Entering Passive Mode "
				 "(%u,%u,%u,%u,%u,%u)\r\n",
				 td->nsrc[0], td->nsrc[1],
				 td->nsrc[2], td->nsrc[3],
				 td->nport[0], td->nport[1]);
			n = patch_ftpcmd(n, pat);
			return n;
		}
		/* TODO: less strict pattern */
		strcpy(pat, "229 Entering Extended Passive Mode (|||%hu|\r\n");
		if (sscanf((char *)buf4 + off, pat, &ps) == 1) {
			statsftp229++;
			if (n->tunnel->flags & TUNDEBUG)
				debugftp229++;
			td = get_ftptempdata(n, p);
			if (td == NULL)
				return NULL;
			ps = td->nport[0] << 8;
			ps |= td->nport[1];
			snprintf(pat, 128,
				 "229 Entering Extended Passive "
				 "Mode (|||%u|)\r\n", ps);
			n = patch_ftpcmd(n, pat);
			return n;
		}
	}
	return n;
}
#endif

/*
 * IN: from tunnel to Internet
 */

/* Filter ICMPv4 packets from tunnel */

int
filtericmpin(u_char type4)
{
	u_int l;

	logit(10, "filtericmpin");

	if (len < IPMINLEN)
		return 0;
	if (len > ICMPMAXLEN)
		len = ICMPMAXLEN;
	switch (type4) {
	case 3:		/* unreachable */
	case 11:	/* time exceeded */
	case 12:	/* parameter problem */
		break;

	default:
		logit(10, "unhandled icmp type %d", (int)type4);
		return 0;
	}
	if (buf4[0] != IP4VNOOP) {
		logerr("byte0(filtericmpin)\n");
		return 0;
	}
	if ((buf4[IPSRC] == 127) || (buf4[IPDST] == 127)) {
		logerr("localnet(filtericmpin)\n");
		return 0;
	}
	if ((buf4[IPSRC] >= 224) || (buf4[IPDST] >= 224)) {
		logerr("multicast(filtericmpin)\n");
		return 0;
	}
	if ((buf4[IPOFFH] & IPOFFMSK) || (buf4[IPOFFL] != 0)) {
		logerr("fragment(filtericmpin)\n");
		return 0;
	}
	if ((buf4[IPPROTO] != IPTCP) && (buf4[IPPROTO] != IPUDP)) {
		logerr("protocol(filtericmpin)\n");
		return 0;
	}
	l = buf4[IPLENH] << 8;
	l |= buf4[IPLENL];
	if (l < IPMINLEN) {
		logerr("short(filtericmpin)\n");
		return 0;
	}
	if (in_cksum(buf4, IPHDRLEN) != 0) {
		logerr("checksum(filtericmpin)\n");
		return 0;
	}
	logit(10, "accepted");
	return 1;
}

/* ICMPv4 from tunnel to Internet */

void
naticmpin(struct tunnel *t,
	  u_char type4,
	  u_char code4,
	  uint32_t mtu4,
	  int from6)
{
	struct nat nat0, *n;
	int cksum, cc;
	u_short id;

	if (!from6) {
		/* real ICMPv4 */
		logit(10, "naticmpin(v4)");
		if ((len < IP2 + IPMINLEN) ||
		    (memcmp(buf4 + IPDST, buf4 + IP2SRC, 4) != 0)) {
			statsdropped[DR_ICMPIN]++;
			if (t->flags & TUNDEBUG)
				debugdropped[DR_ICMPIN]++;
			return;
		}
		type4 = buf4[ICMPTYPE];
		code4 = buf4[ICMPCODE];
		mtu4 = buf4[ICMPID + 2] << 8;
		mtu4 |= buf4[ICMPID + 3];
		len -= IP2;
		/* save headers */
		memcpy(buf, buf4, IP2);
		memmove(buf4, buf4 + IP2, len);
	} else
		logit(10, "naticmpin(v6)");
	if (!filtericmpin(type4)) {
		statsdropped[DR_ICMPIN]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_ICMPIN]++;
		return;
	}
	memset(&nat0, 0, sizeof(nat0));
	nat0.tunnel = t;
	nat0.proto = buf4[IPPROTO];
	memcpy(&nat0.src, buf4 + IPDST, 4);
	memcpy(&nat0.sport, buf4 + IPDPORT, 2);
	n = nat_splay_find(&nat0, 0);
	if (n == NULL) {
		logit(10, "no nat entry for naticmpin");
		statsdropped[DR_ICMPIN]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_ICMPIN]++;
		return;
	}
	memmove(buf4 + IP2, buf4, len);
	if (!from6 && (memcmp(n->src, buf + IPSRC, 4) == 0)) {
		/* end-to-end ICMPv4 */
		memcpy(buf4, buf, IP2);
		len += IP2;
		memcpy(buf4 + IPSRC, n->nsrc, 4);
		fix_cksum(buf4 + IPCKSUMH, n->src, n->nsrc, NULL, NULL);
	} else {
		/* transit ICMPv4: build headers */
		memset(buf4, 0, IP2);
		len += IP2;
		buf4[0] = IP4VNOOP;
		buf4[IPLENH] = len >> 8;
		buf4[IPLENL] = len & 0xff;
		id = random() & 0xffff;
		memcpy(buf4 + IPID, &id, 2);
		buf4[IPTTL] = 64;
		buf4[IPPROTO] = IPICMP;
		memcpy(buf4 + IPSRC, icmpsrc, 4);
		memcpy(buf4 + IPDST, buf4 + IP2SRC, 4);
		cksum = in_cksum(buf4, IPHDRLEN);
		buf4[IPCKSUMH] = cksum >> 8;
		buf4[IPCKSUML] = cksum & 0xff;
		buf4[ICMPTYPE] = type4;
		buf4[ICMPCODE] = code4;
		if ((type4 == 3) && (code4 == 4)) {
			buf4[ICMPID] = mtu4 >> 24;
			buf4[ICMPID + 1] = (mtu4 >> 16) & 0xff;
			buf4[ICMPID + 2] = (mtu4 >> 8) & 0xff;
			buf4[ICMPID + 3] = mtu4 & 0xff;
		}
	}
	memcpy(buf4 + IP2DST, n->nsrc, 4);
	fix_cksum(buf4 + IP2CKSUMH, n->src, n->nsrc, NULL, NULL);
	if (n->flags & MATCH_PORT) {
		memcpy(buf4 + IP2DPORT, n->nport, 2);
		if (n->proto == IPTCP)
			fix_cksum(buf4 + IP2 + TCPCKSUMH,
				  n->src, n->nsrc,
				  n->sport, n->nport);
		else if ((n->proto == IPUDP) &&
			 ((buf4[IP2 + UDPCKSUMH] != 0) ||
			  (buf4[IP2 + UDPCKSUML] != 0)))
			fix_cksum(buf4 + IP2 + UDPCKSUMH,
				  n->src, n->nsrc,
				  n->sport, n->nport);
	}
	cksum = in_cksum(buf4 + IPHDRLEN, len - IPHDRLEN);
	buf4[ICMPCKSUMH] = cksum >> 8;
	buf4[ICMPCKSUML] = cksum & 0xff;
	/* don't bump timeout or reorder */
	cc = tun_write(AF_INET, buf4, len);
	if (cc < 0)
		logerr("write(icmpin): %s\n", strerror(errno));
	else if (cc != (int) len)
		logerr("short(icmpin)\n");
	else {
		if (from6) {
			statsnaticmpin6++;
			if (t->flags & TUNDEBUG)
				debugnaticmpin6++;
		} else {
			statsnaticmpin4++;
			if (t->flags & TUNDEBUG)
				debugnaticmpin4++;
		}
		statssent4++;
		if (t->flags & TUNDEBUG)
			debugsent4++;
	}
}

/* From tunnel to Internet (source NAT case)
 * Returns 1 if packet successfully translated, 0 otherwise
 */

int
natin(struct tunnel *t)
{
	struct nat nat0, *n;
	int increased = 0;
	u_int sport, dport;

	logit(10, "natin");

	/* find an existing nat binding that matches the 5-tuple
	 * { proto, src addr, src port, dst addr, dst port }
	 */
	memset(&nat0, 0, sizeof(nat0));
	nat0.tunnel = t;
	nat0.proto = buf4[IPPROTO];
	memcpy(&nat0.src, buf4 + IPSRC, 4);
	memcpy(&nat0.dst, buf4 + IPDST, 4);
	switch (buf4[IPPROTO]) {
	case IPICMP:
		memcpy(&nat0.sport, buf4 + ICMPID, 2);
		nat0.flags = MATCH_ICMP;
		break;
	case IPTCP:
	case IPUDP:
		memcpy(&nat0.sport, buf4 + IPSPORT, 2);
		memcpy(&nat0.dport, buf4 + IPDPORT, 2);
		nat0.flags = MATCH_PORT;
		break;
	}
	n = nat_splay_find(&nat0, 0);
    got:
	if (n != NULL) {
		/* rewrite the IPv4 header with nat src addr/port */
		memcpy(buf4 + IPSRC, n->nsrc, 4);
		fix_cksum(buf4 + IPCKSUMH, n->src, n->nsrc, NULL, NULL);
		if (n->flags & MATCH_PORT)
			memcpy(buf4 + IPSPORT, n->nport, 2);
		else if (n->flags & MATCH_ICMP)
			memcpy(buf4 + ICMPID, n->nport, 2);
		switch (n->proto) {
		case IPTCP:
			fix_cksum(buf4 + TCPCKSUMH,
				  n->src, n->nsrc,
				  n->sport, n->nport);
			patch_tcpmss(t);
#ifdef FTPALG
			if ((buf4[IPDPORT] == 0) &&
			    (buf4[IPDPORT + 1] == PORTFTP))
				n = patch_ftpin(n, 0);
			else if ((n->sport[0] == 0) &&
				 (n->sport[1] == PORTFTP))
				n = patch_ftpin(n, 1);
			if (n == NULL) {
				statsdropped[DR_BADIN]++;
				if (t->flags & TUNDEBUG)
					debugdropped[DR_BADIN]++;
				return 0;
			}
#endif
			increased = tcpstate_in(n);
			break;

		case IPUDP:
			if ((buf4[UDPCKSUMH] != 0) || (buf4[UDPCKSUML] != 0)) {
				fix_cksum(buf4 + UDPCKSUMH,
					  n->src, n->nsrc,
					  n->sport, n->nport);
			}
			break;
		case IPICMP:
			fix_cksum(buf4 + ICMPCKSUMH,
				  NULL, NULL,
				  n->sport, n->nport);
			break;
		}
		if (n->timeout && !increased &&
		    (seconds + RETRANS_LIFETIME > n->timeout)) {
			n->timeout = seconds + RETRANS_LIFETIME;
			nat_heap_decreased(n->heap_index);
		}
		if (debuglevel >= 10) {
			sport = (n->sport[0] << 8) | n->sport[1];
			dport = (n->dport[0] << 8) | n->dport[1];
			logit(10, "%s %s/%u -> %s/%u",
			      proto2str(n->proto),
			      addr2str(AF_INET, n->src), sport,
			      addr2str(AF_INET, n->dst), dport);
		}
		statsnatin++;
		if (t->flags & TUNDEBUG)
			debugnatin++;
		return 1;
	} else if (t->natcnt >= MAXNATCNT) {
		logit(10, "too many nat entries");
		statsdropped[DR_NATCNT]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_NATCNT]++;
		return 0;
	}

	/* no matching nat binding found, try to create new one */
	n = (struct nat *) malloc(sizeof(*n));
	if (n == NULL) {
		logerr("malloc(nat): %s\n", strerror(errno));
		statsdropped[DR_NEWNAT]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_NEWNAT]++;
		return 0;
	}
	memset(n, 0, sizeof(*n));
	n->tunnel = t;
	n->timeout = seconds + RETRANS_LIFETIME;
	memcpy(n->src, buf4 + IPSRC, 4);
	memcpy(n->dst, buf4 + IPDST, 4);
	memcpy(n->nsrc, natsrcs[t->srcidx]->addr, 4);
	n->proto = buf4[IPPROTO];
	switch (buf4[IPPROTO]) {
	case IPICMP:
		n->flags = MATCH_ICMP;
		memcpy(n->sport, buf4 + ICMPID, 2);
		n->lifetime = ICMP_LIFETIME;
		break;

	case IPTCP:
		n->flags = MATCH_PORT;
		memcpy(n->sport, buf4 + IPSPORT, 2);
		memcpy(n->dport, buf4 + IPDPORT, 2);
		n->lifetime = TCP_LIFETIME;
		break;

	case IPUDP:
		n->flags = MATCH_PORT;
		memcpy(n->sport, buf4 + IPSPORT, 2);
		memcpy(n->dport, buf4 + IPDPORT, 2);
		n->lifetime = UDP_LIFETIME;
		break;
	}
	if (new_nat(n))
		goto got;
	statsdropped[DR_NEWNAT]++;
	if (t->flags &TUNDEBUG)
		debugdropped[DR_NEWNAT]++;
	return 0;
}

/* From tunnel to Internet (PRR/A+P case)
 * This checks the packet against the (statically configured) list of
 * PRR bindings, but does not alter the packet at the exception of TCP MSS.
 * Returns 1 if packet is okay to send, 0 otherwise
 */

int
prrin(struct tunnel *t)
{
	struct nat nat0, *n;
	u_int sport;

	logit(10, "prrin");

	/* find an existing nat binding that matches the 5-tuple
	 * { proto, src addr, src port, dst addr, dst port }
	 */
	memset(&nat0, 0, sizeof(nat0));
	nat0.tunnel = t;
	nat0.proto = buf4[IPPROTO];
	nat0.flags = ALL_DST | PRR_NULL;
	memcpy(&nat0.src, buf4 + IPSRC, 4);
	switch (buf4[IPPROTO]) {
	case IPICMP:
		memcpy(&nat0.sport, buf4 + ICMPID, 2);
		break;
	case IPTCP:
	case IPUDP:
		memcpy(&nat0.sport, buf4 + IPSPORT, 2);
		break;
	}
	n = nat_splay_find(&nat0, 1);
	if (n != NULL) {
		if (buf4[IPPROTO] == IPTCP)
			patch_tcpmss(t);
		if (debuglevel >= 10) {
			sport = (n->sport[0] << 8) | n->sport[1];
			logit(10, "%s %s/%u",
			      proto2str(n->proto),
			      addr2str(AF_INET, n->src), sport);
		}
		statsprrin++;
		if (t->flags & TUNDEBUG)
			debugprrin++;
		return 1;
	}
	logit(10, "no nat entry for prrin");
	statsdropped[DR_NOPRR]++;
	if (t->flags & TUNDEBUG)
		debugdropped[DR_NOPRR]++;
	return 0;
}

/* From tunnel to Internet (NO-NAT case)
 * This does not alter the packet at the exception of TCP MSS.
 */

void
nonatin(struct tunnel *t)
{
	logit(10, "nonatin");

	if (buf4[IPPROTO] == IPTCP)
		patch_tcpmss(t);
	statsnonatin++;
	if (t->flags & TUNDEBUG)
		debugnonatin++;
}

/* Filter IPv4 packets from tunnel
 * Returns:
 * 0: drop packet
 * 1: perform NAT on packet
 * 2: perform PRR on packet
 * 3: perform NONAT on packet
 */

int
filterin(struct tunnel *t)
{
	u_int l;
#ifdef notyet
	u_char tlen[2];
#endif

    again:
	logit(10, "filterin");

	/* sanity checking */
	if (len < IPMINLEN) {
		logerr("length(filterin)\n");
		statsdropped[DR_BADIN]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BADIN]++;
		return 0;
	}
	if (buf4[0] != IP4VNOOP) {
		logerr("byte0(filterin)\n");
		statsdropped[DR_BADIN]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BADIN]++;
		return 0;
	}
	if ((buf4[IPSRC] == 127) || (buf4[IPDST] == 127)) {
		logerr("localnet(filterin)\n");
		statsdropped[DR_BADIN]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BADIN]++;
		return 0;
	}
	if ((buf4[IPSRC] >= 224) || (buf4[IPDST] >= 224)) {
		logerr("multicast(filterin)\n");
		statsdropped[DR_BADIN]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BADIN]++;
		return 0;
	}
	if ((buf4[IPPROTO] != IPTCP) &&
	    (buf4[IPPROTO] != IPUDP) &&
	    (buf4[IPPROTO] != IPICMP)) {
		logerr("protocol(filterin)\n");
		statsdropped[DR_BADIN]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BADIN]++;
		return 0;
	}
	l = buf4[IPLENH] << 8;
	l |= buf4[IPLENL];
	if ((l < IPMINLEN) || (l > len)) {
		logerr("short(filterin)\n");
		statsdropped[DR_BADIN]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BADIN]++;
		return 0;
	}
	len = l;
	if (in_cksum(buf4, IPHDRLEN) != 0) {
		logerr("checksum(filterin)\n");
		statsdropped[DR_BADIN]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BADIN]++;
		return 0;
	}

	/* IPv4 fragment */
	if ((buf4[IPOFFH] & (IPMF|IPOFFMSK)) || (buf4[IPOFFL] != 0)) {
		/* if packet can be successfully reassembled,
		 * process it as a new packet
		 */
		if (defrag(t))
			goto again;
		return 0;
	}

	switch (buf4[IPPROTO]) {
	case IPICMP:
		if (in_cksum(buf4 + IPHDRLEN, len - IPHDRLEN) != 0) {
			logerr("checksum(ICMP,filterin)\n");
			statsdropped[DR_BADIN]++;
			if (t->flags & TUNDEBUG)
				debugdropped[DR_BADIN]++;
			return 0;
		}
		if (t->flags & TUNNONAT)
			return 3;
		if (buf4[ICMPTYPE] != ICMPECHREQ) {
			naticmpin(t, (u_char) 0, (u_char) 0, 0, 0);
			return 0;
		}
		break;

	case IPTCP:
#ifdef notyet
		tlen[0] = (len - IPHDRLEN) >> 8;
		tlen[1] = (len - IPHDRLEN) & 0xff;
		if (pseudo_cksum(buf4, tlen, len) != 0) {
			logerr("checksum(TCP,filterin)\n");
			statsdropped[DR_BADIN]++;
			if (t->flags & TUNDEBUG)
				debugdropped[DR_BADIN]++;
			return 0;
		}
#endif
		break;

	case IPUDP:
#ifdef notyet
		if (((buf4[UDPCKSUMH] != 0) || (buf4[UDPCKSUML] != 0)) &&
		    (pseudo_cksum(buf4, buf4 + UDPLEN, len) != 0)) {
			logerr("checksum(UDP,filterin)\n");
			statsdropped[DR_BADIN]++;
			if (t->flags & TUNDEBUG)
				debugdropped[DR_BADIN]++;
			return 0;
		}
#endif
		break;
	}

	if (t->flags & TUNNONAT)
		return 3;

	/* RFC 1918 or unpublished I-D address -> NAT */
	if (buf4[IPSRC] == 10)
		return 1;
	if ((buf4[IPSRC] == 172) && ((buf4[IPSRC + 1] & 0xf0) == 16))
		return 1;
	if ((buf4[IPSRC] == 192) && (buf4[IPSRC + 1] == 168))
		return 1;
	if ((buf4[IPSRC] == 192) &&
	    (buf4[IPSRC + 1] == 0) && (buf4[IPSRC + 2] == 0) &&
	    ((buf4[IPSRC + 3] & 0xf8) == 0))
		return 1;

	/* global IPv4 address -> A+P/PRR */
	return 2;
}

/* Deal with ICMPv6 from tunnel side */

void
icmp6in(void)
{
	struct tunnel *t;
	u_char code4 = 1;
	u_int mtu = 0;

	logit(10, "icmp6in");

	if (len < IP6HDRLEN + 8 + IP6HDRLEN + 8) {
		logerr("short(icmp6in)\n");
		statsdropped[DR_BAD6]++;
		return;
	}
	switch (buf6[ICMP6TYPE]) {
	case 1:		/* destination unreachable */
		if (buf6[ICMP6CODE] > 3) {	/* address unreachable */
			statsdropped[DR_ICMP6]++;
			return;
		}
		break;

	case 2:		/* packet too big */
		if (buf6[ICMP6CODE] != 0) {
			statsdropped[DR_ICMP6]++;
			return;
		}
		mtu = buf6[ICMP6PTR] << 24;
		mtu |= buf6[ICMP6PTR + 1] << 16;
		mtu |= buf6[ICMP6PTR + 2] << 8;
		mtu |= buf6[ICMP6PTR + 3];
		if (mtu == 0) {
			logerr("mtu(icmp6in)\n");
			statsdropped[DR_ICMP6]++;
			return;
		}
		break;

	case 3:		/* time exceeded */
		if (buf6[ICMP6CODE] != 0) {
			statsdropped[DR_ICMP6]++;
			return;
		}
		break;

	case 4:		/* parameter problem */
		break;

	default:
		logit(10, "unhandled icmp6 type %d", (int)buf6[ICMP6TYPE]);
		statsdropped[DR_ICMP6]++;
		return;
	}
	if (memcmp(buf6 + IP6DST, buf6 + IP62SRC, 16) != 0) {
		logerr("dest(icmp6in)\n");
		statsdropped[DR_ICMP6]++;
		return;
	}
	if ((buf6[IP62PROTO] != IP6IP4) || (len < IP64 + IPMINLEN)) {
		logerr("length(icmp6in)\n");
		statsdropped[DR_ICMP6]++;
		return;
	}
	if ((buf6[IP64PROTO] != IPTCP) && (buf6[IP64PROTO] != IPUDP)) {
		logerr("protocol(icmp6in)\n");
		statsdropped[DR_ICMP6]++;
		return;
	}
	if (pseudo6_cksum() != 0) {
		logerr("checksum(icmp6in)\n");
		statsdropped[DR_ICMP6]++;
		return;
	}
	t = tunnel_lookup(buf6 + IP62DST);
	if (t == NULL) {
		logerr("icmp6in: no tunnel found for %s\n",
		       addr2str(AF_INET6, buf6 + IP62DST));
		statsdropped[DR_NOTUN]++;
		return;
	}
	if (mtu != 0) {
		logit(10, "icmp6in: set mtu to %u", mtu);
		set_tunnel_mtu(t->remote, mtu, 0);
		return;
	}

	/* decapsulate the icmp packet and send it on */
	len -= IP64;
	memcpy(buf4, buf6 + IP64, len);
	naticmpin(t, (u_char) 3, code4, 0, 1);
	return;
}

/* IPv6 ACL filtering (return 0 for drop, 1 for accept) */

int
acl6(u_char *src)
{
	struct acl6 *a;
	u_int i;

	for (a = acl6_first; a != NULL; a = a->next) {
		for (i = 0; i < 16; i++)
			if ((src[i] & a->mask[i]) != a->addr[i])
				break;
		if (i == 16)
			return 1;
	}
	logit(1, "%s dropped ACL6", addr2str(AF_INET6, src));
	return 0;
}

/* Decapsulate IPv4 packets from IPv6
 *
 * This copies the encapsulated IPv4 packet from the global IPv6 packet
 * buffer buf6[] to the global IPv4 packet buffer buf4[], and returns the
 * tunnel that the packet belongs to (determined from the source IPv6
 * address).
 */

struct tunnel *
decap(void)
{
	struct tunnel *t = NULL;
	u_int l;
	u_char tos;

    again:
	logit(10, "decap");

	/* sanity checks */
	/* version check is also done in loop(), but keep it here
	 * for the "again" cases
	 */
	if ((buf6[0] & IPVERMSK) != IP6V) {
		logerr("version(decap)\n");
		statsdropped[DR_BAD6]++;
		return NULL;
	}
	if (len < IP6HDRLEN + 8 + 8) {
		logerr("short(decap)\n");
		statsdropped[DR_BAD6]++;
		return NULL;
	}

	/* deal with icmp packets separately */
	if (buf6[IP6PROTO] == IP6ICMP) {
		if (acl6(buf6 + IP6SRC))
			icmp6in();
		return NULL;
	}

	/* find (or create) the tunnel this belongs to */
	if (t == NULL)
		t = tunnel_lookup(buf6 + IP6SRC);
	if (t == NULL) {
		if (use_autotunnel) {
			if (!acl6(buf6 + IP6SRC)) {
				statsdropped[DR_ACL6]++;
				return NULL;
			}
			t = add_tunnel(buf6 + IP6SRC);
			if (t == NULL) {
				logerr("failed to create tunnel for %s\n",
				       addr2str(AF_INET6, buf6 + IP6SRC));
				statsdropped[DR_NOTUN]++;
				return NULL;
			}
		} else {
			logerr("no tunnel for %s, and autotunnel disabled\n",
			       addr2str(AF_INET6, buf6 + IP6SRC));
			statsdropped[DR_NOTUN]++;
			return NULL;
		}
	}

	logit(3, "decap: got packet, length=%u", len);

	/* sanity checks */
	len -= IP6HDRLEN;
	l = buf6[IP6LENH] << 8;
	l |= buf6[IP6LENL];
	if (l > len) {
		logerr("length (decap)\n");
		statsdropped[DR_BAD6]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BAD6]++;
		return NULL;
	} else if (l < len) {
		if (l < 8 + 8) {
			logerr("short (decap)\n");
			statsdropped[DR_BAD6]++;
			if (t->flags & TUNDEBUG)
				debugdropped[DR_BAD6]++;
			return NULL;
		}
		len = l;
	}

	if (buf6[IP6PROTO] != IP6IP4) {
		/* not an encapsulated IPv4 packet */
		if ((buf6[IP6PROTO] == IP6FRAG) &&
		    (buf6[IP6FPROTO] == IP6IP4)) {
			/* IPv6 fragment of an encapsulated packet */
			len += IP6HDRLEN;
			if (defrag6(t))
				/* if packet can be successfully reassembled,
				 * process it as a new packet
				 */
				goto again;
			return NULL;
		}

		if ((buf6[IP6PROTO] == IP6DSTOP) &&
		    (buf6[IP6FPROTO] == IP6IP4) &&
		    (buf6[IP6FPROTO + 1] == 0)) {
			/*
			 * Destination option header:
			 * likely a Tunnel-Encapsulation-Limit, strip it!
			 */
			l -= 8;
			len -= 8;
			buf6[IP6LENH] = l >> 8;
			buf6[IP6LENL] = l & 0xff;
			buf6[IP6PROTO] = buf6[IP6FPROTO];
			memmove(buf6 + IP6HDRLEN, buf6 + IP6HDRLEN + 8, len);
			len += IP6HDRLEN;
			goto again;
		}

		/* some other IPv6 packet */
		logerr("header6 (decap)\n");
		statsdropped[DR_BAD6]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_BAD6]++;
		return NULL;
	}

	/* copy the payload to the IPv4 packet buffer */
	tos = (buf6[0] & 0x0f) << 4;
	tos |= buf6[1] >> 4;
	memcpy(buf4, buf6 + IP6HDRLEN, len);

#ifdef notyet
	/* CE and inner != not-ECT and inner != CE -> CE */
	if (((tos & 3) == 3) &&
	    ((buf4[IPTOS] & 3) != 0) &&
	    ((buf4[IPTOS] & 3) != 3)) {
		int cksum;

		if (in_cksum(buf4, IPHDRLEN) != 0)
			return NULL;
		buf4[IPTOS] |= tos & 3;
		buf4[ICMPCKSUMH] = buf4[ICMPCKSUML] = 0;
		cksum = in_cksum(buf4, IPHDRLEN);
		buf4[ICMPCKSUMH] = cksum >> 8;
		buf4[ICMPCKSUML] = cksum & 0xff;
	}
#endif

	return t;
}

/*
 * OUT: from Internet to tunnel
 */

/* Clone a FTP server static binding at the first match */

#ifdef FTPALG
struct nat *
get_ftptempsrv(struct nat *n0)
{
	struct nat *n;

	logit(10, "get_ftptempsrv");

	if (n0->tunnel->natcnt >= MAXNATCNT) {
		logit(10, "too many nat entries(get_ftptempsrv)");
		statsdropped[DR_NATCNT]++;
		if (n0->tunnel->flags & TUNDEBUG)
			debugdropped[DR_NATCNT]++;
		return NULL;
	}

	n = (struct nat *) malloc(sizeof(*n));
	if (n == NULL) {
		logerr("malloc(get_ftptempsrv): %s\n", strerror(errno));
		statsdropped[DR_NEWNAT]++;
		if (n0->tunnel->flags & TUNDEBUG)
			debugdropped[DR_NEWNAT]++;
		return NULL;
	}
	memset(n, 0, sizeof(*n));
	n->tunnel = n0->tunnel;
	n->proto = n0->proto;
	memcpy(n->src, n0->src, 4);
	memcpy(n->nsrc, n0->nsrc, 4);
	memcpy(n->sport, n0->sport, 2);
	memcpy(n->nport, n0->nport, 2);
	memcpy(n->dst, buf4 + IPSRC, 4);
	memcpy(n->dport, buf4 + IPSPORT, 2);
	n->flags = MATCH_PORT;
	n->timeout = seconds + CLOSED_TCP_LIFETIME;
	n->lifetime = TCP_LIFETIME;

	if (!nat_heap_insert(n)) {
		free(n);
		return NULL;
	}
	if (nat_tree_insert(n) != NULL) {
		logerr("SEVERE: rb collision(get_ftptempsrv)\n");
		nat_heap_delete(n->heap_index);
		free(n);
		return NULL;
	}
	if (nat_splay_insert(n) != NULL) {
		logerr("SEVERE: splay collision(get_ftptempsrv)\n");
		(void) nat_tree_remove(n);
		nat_heap_delete(n->heap_index);
		free(n);
		return NULL;
	}
	n->xnext = n0->xfirst;
	if (n->xnext != NULL)
		n0->xfirst->xprev = &n->xnext;
	n0->xfirst = n;
	n->xprev = &n0->xfirst;

	tnatcnt++;
	n->tunnel->natcnt++;
	statscnat++;
	if (n->tunnel->flags & TUNDEBUG)
		debugcnat++;
	trace_nat(n, "add");
	logit(10, "get_ftptempsrv got %lx", (u_long) n);
	return n;
}

/* Patch FTP packet from Internet to tunnel */

struct nat *
patch_ftpout(struct nat *n0, int way)
{
	struct nat *n = n0;

	logit(10, "patch_ftpout%d %lx", way, (u_long) n);

	if ((way == 1) && (n0->flags & ALL_DST)) {
		for (n = n0->xfirst; n != NULL; n = n->xnext) {
			if ((memcmp(buf4 + IPSRC, n->dst, 4) == 0) &&
			    (memcmp(buf4 + IPSPORT, n->dport, 2) == 0))
				break;
		}
		if (n == NULL)
			n = get_ftptempsrv(n0);
		if (n == NULL)
			return NULL;
	}
	if (n->ftpseqd != 0)
		fix_ftpseqcksum(buf4 + TCPCKSUMH, buf4 + TCPACK, -n->ftpseqd);
	return n;
}
#endif

/* Too big error on unfragmentable packet from Internet */

void
toobigout(struct nat *n)
{
	int cksum, cc;
	u_short id, mtu;

	logit(10, "toobigout");

	if (len > ICMPMAXLEN)
		len = ICMPMAXLEN;
	memmove(buf4 + IP2, buf4, len);
	memset(buf4, 0, IP2);
	len += IP2;
	buf4[0] = IP4VNOOP;
	buf4[IPLENH] = len >> 8;
	buf4[IPLENL] = len & 0xff;
	id = random() & 0xffff;
	memcpy(buf4 + IPID, &id, 2);
	buf4[IPTTL] = 64;
	buf4[IPPROTO] = IPICMP;
	memcpy(buf4 + IPSRC, icmpsrc, 4);
	memcpy(buf4 + IPDST, buf4 + IP2SRC, 4);
	cksum = in_cksum(buf4, IPHDRLEN);
	buf4[IPCKSUMH] = cksum >> 8;
	buf4[IPCKSUML] = cksum & 0xff;
	buf4[ICMPTYPE] = 3;
	buf4[ICMPCODE] = 4;
	mtu = n->tunnel->mtu - IP6HDRLEN;
	buf4[ICMPID + 2] = mtu >> 8;
	buf4[ICMPID + 3] = mtu & 0xff;
	cksum = in_cksum(buf4 + IPHDRLEN, len - IPHDRLEN);
	buf4[ICMPCKSUMH] = cksum >> 8;
	buf4[ICMPCKSUML] = cksum & 0xff;

	if (n->timeout && (seconds + (time_t) n->lifetime > n->timeout)) {
		n->timeout = seconds + n->lifetime;
		nat_heap_decreased(n->heap_index);
	}

	cc = tun_write(AF_INET, buf4, len);
	if (cc < 0)
		logerr("write(toobigout): %s\n", strerror(errno));
	else if (cc != (int) len)
		logerr("short(toobigout)\n");
	else {
		statstoobig++;
		if (n->tunnel->flags & TUNDEBUG)
			debugtoobig++;
	}
}

/* From Internet to tunnel */

struct tunnel *
natout(void)
{
	struct nat *n, nat0;
#ifdef FTPALG
	struct nat *nn;
#endif
	int increased = 0;

	logit(10, "natout");

	memset(&nat0, 0, sizeof(nat0));
	nat0.proto = buf4[IPPROTO];
	memcpy(nat0.nsrc, buf4 + IPDST, 4);
	memcpy(nat0.dst, buf4 + IPSRC, 4);
	switch (nat0.proto) {
	case IPICMP:
		memcpy(nat0.nport, buf4 + ICMPID, 2);
		nat0.flags = MATCH_ICMP;
		break;
	case IPTCP:
	case IPUDP:
		memcpy(nat0.nport, buf4 + IPDPORT, 2);
		memcpy(nat0.dport, buf4 + IPSPORT, 2);
		nat0.flags = MATCH_PORT;
		break;
	}
	n = nat_lookup(&nat0);
	if (n == NULL) {
		logit(10, "no nat entry for natout");
		statsdropped[DR_NATOUT]++;
		return NULL;
	}
	if ((buf4[IPOFFH] & IPDF) &&
	    (buf4[IPPROTO] != IPICMP) &&
	    ((buf4[IPOFFH] & IPOFFMSK) == 0) &&
	    (buf4[IPOFFL] == 0) &&
	    (len + IP6HDRLEN > (u_int) n->tunnel->mtu)) {
		u_int saved = len;
		u_char tbpol = n->tunnel->flags & (TUNTBDROP | TUNTBICMP);

		if (tbpol == TUNTBICMP)
			memcpy(buf, buf4, len);
		if ((tbpol & TUNTBICMP) != 0)
			toobigout(n);
		if (tbpol == TUNTBICMP)
			memcpy(buf4, buf, saved);
		if ((tbpol & TUNTBDROP) != 0) {
			logit(10, "too big dropped");
			statsdropped[DR_TOOBIG]++;
			if (n->tunnel->flags & TUNDEBUG)
				debugdropped[DR_TOOBIG]++;
			return NULL;
		}
	}
	if (n->flags & PRR_NULL) {
		if (buf4[IPPROTO] == IPTCP)
			patch_tcpmss(n->tunnel);
		if (n != n->tunnel->tnat_root)
			nat_splay_splay(n);
		statsprrout++;
		if (n->tunnel->flags & TUNDEBUG)
			debugprrout++;
		return n->tunnel;
	}
	memcpy(buf4 + IPDST, n->src, 4);
	fix_cksum(buf4 + IPCKSUMH, n->nsrc, n->src, NULL, NULL);
	if (n->flags & MATCH_PORT)
		memcpy(buf4 + IPDPORT, n->sport, 2);
	else if (n->flags & MATCH_ICMP)
		memcpy(buf4 + ICMPID, n->sport, 2);
	switch (n->proto) {
	case IPTCP:
		fix_cksum(buf4 + TCPCKSUMH,
			  n->nsrc, n->src,
			  n->nport, n->sport);
		patch_tcpmss(n->tunnel);
#ifdef FTPALG
		if ((buf4[IPSPORT] == 0) && (buf4[IPSPORT+ 1] == PORTFTP))
			nn = patch_ftpout(n, 0);
		else if ((n->sport[0] == 0) && (n->sport[1] == PORTFTP))
			nn = patch_ftpout(n, 1);
		else
			nn = n;
		if (nn == NULL) {
			statsdropped[DR_NATOUT]++;
			if (n->tunnel->flags & TUNDEBUG)
				debugdropped[DR_NATOUT]++;
			return NULL;
		} else
			n = nn;
#endif
		increased = tcpstate_out(n);
		break;

	case IPUDP:
		if ((buf4[UDPCKSUMH] != 0) || (buf4[UDPCKSUML] != 0)) {
			fix_cksum(buf4 + UDPCKSUMH,
				  n->nsrc, n->src,
				  n->nport, n->sport);
		}
		break;
	case IPICMP:
		fix_cksum(buf4 + ICMPCKSUMH,
			  NULL, NULL,
			  n->nport, n->sport);
		break;
	}
	if (n->timeout && !increased &&
	    (seconds + (time_t) n->lifetime > n->timeout)) {
		n->timeout = seconds + n->lifetime;
		nat_heap_decreased(n->heap_index);
	}
	if (n != n->tunnel->tnat_root)
		nat_splay_splay(n);
	statsnatout++;
	if (n->tunnel->flags & TUNDEBUG)
		debugnatout++;
	return n->tunnel;
}

/* ICMP from Internet to tunnel */

struct tunnel *
naticmpout(void)
{
	struct nat *n, nat0;
	int cksum;

	logit(10, "naticmpout");

	nat0.proto = buf4[IP2PROTO];
	nat0.flags = MATCH_PORT;
	memcpy(nat0.nsrc, buf4 + IPDST, 4);
	memcpy(nat0.dst, buf4 + IP2DST, 4);
	memcpy(nat0.nport, buf4 + IP2SPORT, 2);
	memcpy(nat0.dport, buf4 + IP2DPORT, 2);
	n = nat_lookup(&nat0);
	if (n == NULL) {
		logit(10, "no nat entry for naticmpout");
		statsdropped[DR_NATOUT]++;
		return NULL;
	}
	if ((buf4[IPOFFH] & IPDF) &&
	    (len + IP6HDRLEN > (u_int) n->tunnel->mtu)) {
		logit(10, "too large with DF (naticmpout)");
		statsdropped[DR_TOOBIG]++;
		if (n->tunnel->flags & TUNDEBUG)
			debugdropped[DR_TOOBIG]++;
		return NULL;
	}
	if (n->flags & PRR_NULL)
		goto prr_short_cut;
	memcpy(buf4 + IPDST, n->src, 4);
	fix_cksum(buf4 + IPCKSUMH, n->nsrc, n->src, NULL, NULL);
	memcpy(buf4 + IP2SRC, n->src, 4);
	memcpy(buf4 + IP2SPORT, n->sport, 2);
	fix_cksum(buf4 + IP2CKSUMH, n->nsrc, n->src, NULL, NULL);
	buf4[ICMPCKSUMH] = buf4[ICMPCKSUML] = 0;
	cksum = in_cksum(buf4 + IPHDRLEN, len - IPHDRLEN);
	buf4[ICMPCKSUMH] = cksum >> 8;
	buf4[ICMPCKSUML] = cksum & 0xff;
	if (n->timeout && (seconds + ICMP_LIFETIME > n->timeout)) {
		n->timeout = seconds + ICMP_LIFETIME;
		nat_heap_decreased(n->heap_index);
	}
    prr_short_cut:
	/* don't promote to head */
	statsnaticmpout++;
	if (n->tunnel->flags & TUNDEBUG)
		debugnaticmpout++;
	return n->tunnel;
}

/* NO-NAT from Internet to tunnel */

struct tunnel *
nonatout(void)
{
	struct tunnel *t = NULL;

	logit(10, "nonatout");

	for (t = nonat_first; t != NULL; t = t->nnext) {
		const u_char *m = mask4[t->nnplen];

		if ((t->flags & TUNNONAT) == 0) {
			logerr("a not no-nat in no-nat list[%s]\n",
			       addr2str(AF_INET6, t->remote));
			continue;
		}
		if (((buf4[IPDST] & m[0]) == t->nnaddr[0]) &&
		    ((buf4[IPDST + 1] & m[1]) == t->nnaddr[1]) &&
		    ((buf4[IPDST + 2] & m[2]) == t->nnaddr[2]) &&
		    ((buf4[IPDST + 3] & m[3]) == t->nnaddr[3]))
			break;
	}
	if (t == NULL) {
		logerr("can't refind the no-nat entry for %s\n",
		       addr2str(AF_INET, buf4 + IPDST));
		return NULL;
	}

	if ((buf4[IPOFFH] & IPDF) &&
	    (buf4[IPPROTO] != IPICMP) &&
	    ((buf4[IPOFFH] & IPOFFMSK) == 0) &&
	    (buf4[IPOFFL] == 0) &&
	    (len + IP6HDRLEN > (u_int) t->mtu) &&
	    (t->flags & TUNTBDROP)) {
		logit(10, "too big dropped");
		statsdropped[DR_TOOBIG]++;
		if (t->flags & TUNDEBUG)
			debugdropped[DR_TOOBIG]++;
		return NULL;
	}
	if (buf4[IPPROTO] == IPTCP)
		patch_tcpmss(t);
	statsnonatout++;
	if (t->flags & TUNDEBUG)
		debugnonatout++;
	return t;
}

/* Filter ICMPv4 packets from Internet
 * Returns 1 if packet is okay to send, 0 otherwise
 */

int
filtericmpout(void)
{
	u_int l, i;

	logit(10, "filtericmpout");

	/* sanity checks */
	if (len < IP2 + IPMINLEN) {
		logerr("short(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	switch (buf4[ICMPTYPE]) {
	case 3:		/* unreachable */
	case 11:	/* time exceeded */
	case 12:	/* parameter problem */
		break;
	default:
		logit(10, "unhandled icmp type %d", (int)buf4[ICMPTYPE]);
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}

	/* sanity checks on referenced IPv4 header */
	if (buf4[IP2] != IP4VNOOP) {
		logerr("byte0(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	if ((buf4[IP2SRC] == 127) || (buf4[IP2DST] == 127)) {
		logerr("localnet(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	if ((buf4[IP2SRC] >= 224) || (buf4[IP2DST] >= 224)) {
		logerr("multicast(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	if ((buf4[IP2PROTO] != IPTCP) && (buf4[IP2PROTO] != IPUDP)) {
		logerr("protocol(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	if (memcmp(buf4 + IPDST, buf4 + IP2SRC, 4) != 0) {
		logerr("destination(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	l = buf4[IP2LENH] << 8;
	l |= buf4[IP2LENL];
	if (l < IPMINLEN) {
		logerr("short(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	if (in_cksum(buf4 + IP2, IPHDRLEN) != 0) {
		logerr("checksum(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	if ((buf4[IP2OFFH] & IPOFFMSK) || (buf4[IP2OFFL] != 0)) {
		logerr("fragment(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	/* naticmpout() will recompute it from scratch */
	if (in_cksum(buf4 + IPHDRLEN, len - IPHDRLEN) != 0) {
		logerr("checksum2(filtericmpout)\n");
		statsdropped[DR_ICMPOUT]++;
		return 0;
	}
	for (i = 0; i < natsrccnt; i++) {
		if (memcmp(buf4 + IPDST, natsrcs[i]->addr, 4) == 0)
			return 1;
	}
	statsdropped[DR_DSTOUT]++;
	return 0;
}

/* Filter IPv4 packets from Internet
 * Returns:
 * 0: drop packet
 * 1: perform inbound nat on packet
 * 2: perform inbound nat on icmp packet 
 * 3: perform inbound no-nat on packet
 */

int
filterout(void)
{
	struct tunnel *t;
	u_int l, i;
#ifdef notyet
	u_char tlen[2];
#endif
	char src[16], dst[16];
	int ret = -1;

    again:
	logit(10, "filterout");

	/* sanity checking */
	if (len < IPMINLEN) {
		logerr("length(filterout)\n");
		statsdropped[DR_BADOUT]++;
		return 0;
	}
	if (buf4[0] != IP4VNOOP) {
		logerr("byte0(filterout)\n");
		statsdropped[DR_BADOUT]++;
		return 0;
	}
	if ((buf4[IPSRC] == 127) || (buf4[IPDST] == 127)) {
		logerr("localnet(filterout)\n");
		statsdropped[DR_BADOUT]++;
		return 0;
	}
	if ((buf4[IPSRC] >= 224) || (buf4[IPDST] >= 224)) {
		logerr("multicast(filterout)\n");
		statsdropped[DR_BADOUT]++;
		return 0;
	}
	l = buf4[IPLENH] << 8;
	l |= buf4[IPLENL];
	if ((l < IPMINLEN) || (l > len)) {
		logerr("short(filterout)\n");
		statsdropped[DR_BADOUT]++;
		return 0;
	}
	len = l;
	if (in_cksum(buf4, IPHDRLEN) != 0) {
		logerr("checksum(filterout)\n");
		statsdropped[DR_BADOUT]++;
		return 0;
	}

	/* shortcut destination address control if already done */
	if (ret >= 0)
		goto done;

	/* no-nat matching is here */
	for (t = nonat_first; t != NULL; t = t->nnext) {
		const u_char *m = mask4[t->nnplen];

#ifdef notyet
		/* both expensive and what to do?! */
		if ((t->flags & TUNNONAT) == 0) {
			logerr("a not no-nat in no-nat list\n");
			continue;
		}
#endif
		if (((buf4[IPDST] & m[0]) == t->nnaddr[0]) &&
		    ((buf4[IPDST + 1] & m[1]) == t->nnaddr[1]) &&
		    ((buf4[IPDST + 2] & m[2]) == t->nnaddr[2]) &&
		    ((buf4[IPDST + 3] & m[3]) == t->nnaddr[3])) {
			ret = 3;
			goto done;
		}
	}

	/* match dst to a public NAT addr */
	for (i = 0; i < natsrccnt; i++) {
		if (memcmp(buf4 + IPDST, natsrcs[i]->addr, 4) == 0) {
			ret = 0;
			goto done;
		}
	}
	logerr("dest(filterout): no nat binding for %s from %s\n", 
	       inet_ntop(AF_INET, buf4 + IPDST, dst, 16),
	       inet_ntop(AF_INET, buf4 + IPSRC, src, 16));
	statsdropped[DR_DSTOUT]++;
	return 0;

    done:
	if ((buf4[IPPROTO] != IPTCP) &&
	    (buf4[IPPROTO] != IPUDP) &&
	    (buf4[IPPROTO] != IPICMP)) {
		logerr("proto(filterout)\n");
		statsdropped[DR_BADOUT]++;
		return 0;
	}
	if ((buf4[IPOFFH] & (IPMF|IPOFFMSK)) || (buf4[IPOFFL] != 0)) {
		/* IPv4 fragment, try to reassemble */
		if (defrag(NULL))
			/* if packet can be successfully reassembled,
			 * process it as a new packet
			 */
			goto again;
		return 0;
	}
	switch (buf4[IPPROTO]) {
	case IPICMP:
		if (ret == 3)
			break;
		if (buf4[ICMPTYPE] == ICMPECHREP)
			/* ping reply doesn't include referenced IPv4 header,
			 * so don't do the extra validation of filtericmpout
			 */
			ret = 1;
		else if (filtericmpout())
			ret = 2;
		break;

	case IPTCP:
		if (len < IPHDRLEN + 20) {
			logerr("short(TCP,filterout)\n");
			statsdropped[DR_BADOUT]++;
			return 0;
		}
		if (ret != 3)
			ret = 1;
		break;

	case IPUDP:
		if (ret != 3)
			ret = 1;
		break;

	default:
		logerr("proto(filterout)\n");
		statsdropped[DR_BADOUT]++;
		return 0;
	}
#ifdef notyet
	switch (buf4[IPPROTO]) {
	case IPICMP:
		if (in_cksum(buf4 + IPHDRLEN, len - IPHDRLEN) != 0) {
			logerr("checksum(ICMP,filterout)\n");
			statsdropped[DR_BADOUT]++;
			return 0;
		}
		break;

	case IPTCP:
		tlen[0] = (len - IPHDRLEN) >> 8;
		tlen[1] = (len - IPHDRLEN) & 0xff;
		if (pseudo_cksum(buf4, tlen, len) != 0) {
			logerr("checksum(TCP,filterout)\n");
			statsdropped[DR_BADOUT]++;
			return 0;
		}
		break;

	case IPUDP:
		if (((buf4[UDPCKSUMH] != 0) || (buf4[UDPCKSUML] != 0)) &&
		    (pseudo_cksum(buf4, buf4 + UDPLEN, len) != 0)) {
			logerr("checksum(UDP,filterout)\n");
			statsdropped[DR_BADOUT]++;
			return 0;
		}
		break;
	}
#endif
	return ret;
}

/* Encapsulate IPv4 packets into IPv6, and send the packet */

void
encap(struct tunnel *t)
{
	u_int off, flen, mtu;
	u_int fmtu, nfrag, flen0, fleno;
	uint32_t id;
	u_char tos;
	int cc, fcc;

	logit(10, "encap: tunnel %s", addr2str(AF_INET6, t->remote));

	mtu = (u_int) t->mtu;
	if (len + IP6HDRLEN <= mtu) {
		/* simple case: packet fits in the tunnel mtu */
		memset(buf6, 0, IP6HDRLEN);
		buf6[0] = IP6V;
		tos = buf4[IPTOS];
		if ((tos & 3) == 3)
			tos &= ~0x02;		/* CE -> ECT(0) */
#ifdef notyet
		buf6[0] |= tos >> 4;
		buf6[1] = tos << 4;
#endif
		buf6[IP6LENH] = len >> 8;
		buf6[IP6LENL] = len & 0xff;
		buf6[IP6PROTO] = IP6IP4;
		buf6[IP6TTL] = 64;		/* default TTL */
		memcpy(buf6 + IP6SRC, local6, 16);
		memcpy(buf6 + IP6DST, t->remote, 16);
		memcpy(buf6 + IP6HDRLEN, buf4, len);
		len += IP6HDRLEN;

		cc = tun_write(AF_INET6, buf6, len);
		if (cc < 0)
			logerr("write(encap): %s\n", strerror(errno));
		else if (cc != (int) len)
			logerr("short(encap)\n");
		else {
			statssent6++;
			if (t->flags & TUNDEBUG)
				debugsent6++;
		}
		return;
	}

	/* fragment the packet */
	logit(10, "encap: len %u > mtu % u, fragmenting",
	      len + IP6HDRLEN, mtu);
	memset(buf6, 0, IP6FLEN);
	buf6[0] = IP6V;
	tos = buf4[IPTOS];
	if ((tos & 3) == 3)
		tos &= ~0x02;		/* CE -> ECT(0) */
#ifdef notyet
	buf6[0] |= tos >> 4;
	buf6[1] = tos << 4;
#endif
	buf6[IP6PROTO] = IP6FRAG;
	buf6[IP6TTL] = 64;		/* default TTL */
	memcpy(buf6 + IP6SRC, local6, 16);
	memcpy(buf6 + IP6DST, t->remote, 16);
	buf6[IP6FPROTO] = IP6IP4;
	id = (uint32_t) random();
	memcpy(buf6 + IP6FID, &id, 4);

	flen0 = 0;		/* XXX: silence compiler */
	fleno = 0;
	if (eqfrag) {
		fmtu = (mtu - IP6FLEN) & ~7;
		nfrag = len / fmtu;			/* always >= 1 */
		if (len % fmtu != 0)
			nfrag++;
		/* len = fmtu * (nfrag - 1) + rem; 0 <= rem < fmtu */
		fleno = (len / nfrag) & ~7;		/* lower medium size */
		flen0 = len - fleno * (nfrag - 1);	/* remainder */
		flen0 = (flen0 + 7) & ~7;
		if (flen0 > fmtu) {
			/* too much remainder, switch to bigger medium size,
			   but still <= fmtu */
			fleno += 8;
			/* recompute remainder (shall be this time <= fmtu) */
			flen0 = len - (fleno * (nfrag - 1));
			flen0 = (flen0 + 7) & ~7;
		}
		/* biggest should be first, smallest last */
		if (flen0 < fleno)
			flen0 = fleno;
	}
	for (off = 0; off < len; off += flen) {
		flen = len - off;
		if (flen > mtu - IP6FLEN) {
			if (eqfrag) {
				if (off == 0) /* first fragment */
					flen = flen0;
				else /* intermediate fragment */
					flen = fleno;
			} else {
				flen = mtu - IP6FLEN;
				flen &= ~7;
			}
		}
		buf6[IP6FOFFH] = off >> 8;
		buf6[IP6FOFFL] = off & IP6FMSK;
		if (flen + off < len)
			buf6[IP6FOFFL] |= IP6FMF;
		buf6[IP6LENH] = (flen + 8) >> 8;
		buf6[IP6LENL] = (flen + 8) & 0xff;
		memcpy(buf6 + IP6FLEN, buf4 + off, flen);
		fcc = (int) flen + IP6FLEN;

		cc = tun_write(AF_INET6, buf6, fcc);
		if (cc < 0) {
			logerr("write(encap): %s\n", strerror(errno));
			return;
		} else if (cc != fcc) {
			logerr("short(encap)\n");
			return;
		}
		statsfrgout6++;
		if (t->flags & TUNDEBUG)
			debugfrgout6++;
	}
	statssent6++;		
	if (t->flags & TUNDEBUG)
		debugsent6++;
}

/*
 * Main...
 */

/* Main loop */

int
loop(void)
{
	struct tunnel *t;
	struct nat *n;
	struct frag *f;
	fd_set set;
	struct timeval tv;
	int infd, maxfd, cc;
	double delta;

	FD_ZERO(&set);
	infd = fileno(stdin);
	FD_SET(infd, &set);
	FD_SET(tunfd, &set);
	maxfd = tunfd > infd ? tunfd : infd;
	tv.tv_sec = 1;
	tv.tv_usec = 0;

	cc = select(maxfd + 1, &set, NULL, NULL, &tv);
	if ((cc < 0) && (errno != EINTR)) {
		logerr("select: %s\n", strerror(errno));
		exit(-1);
	}
	seconds = time(NULL);

	/* deal with interactive commands */
	if (FD_ISSET(infd, &set) && (commands(infd) == 1))
		return 1;

	/* deal with packets on the tunnel interface */
	if (FD_ISSET(tunfd, &set)) {
		cc = tun_read(buf, sizeof(buf));
		if (cc < 0) {
			logerr("read: %s\n", strerror(errno));
			goto next;
		}
		if (cc == 0) {
			logerr("read zero bytes\n");
			goto next;
		}

		/* IPv6 packet: probably tunneled IPv4 from CGN client */
		if ((buf[0] & IPVERMSK) == IP6V) {
			statsrcv6++;
			len = cc;
			if (len < IP6HDRLEN + 8 + 8) {
				logerr("short IPv6 packet\n");
				statsdropped[DR_BAD6]++;
				goto next;
			}
			memcpy(buf6, buf, len);
			t = decap();
			if (t == NULL)
				goto next;
			if (t->flags & TUNDEBUG)
				debugrcv6++;
			switch (filterin(t)) {
			case 1:
				if (natin(t))
					break;
				else
					goto next;
			case 2:
				if (prrin(t))
					break;
				else
					goto next;
			case 3:
				nonatin(t);
				break;
			default:
				goto next;
			}
			/* IPv4 packet has been validated, rewritten
			 * in the NAT case; now send it
			 */
			cc = tun_write(AF_INET, buf4, len);
			if (cc < 0)
				logerr("write: %s\n", strerror(errno));
			else if (cc != (int) len)
				logerr("write returned %d, expected %d\n",
				       cc, len);
			else {
				logit(3, "wrote %d bytes", cc);
				statssent4++;
				if (t->flags & TUNDEBUG)
					debugsent4++;
			}
			goto next;
		}

		/* IPv4 packet: probably NATed packet for CGN client */
		if ((buf[0] & IPVERMSK) == IP4V) {
			statsrcv4++;
			len = cc;
			if (len < IPMINLEN) {
				logerr("short IPv4 packet\n");
				statsdropped[DR_BADOUT]++;
				goto next;
			}
			memcpy(buf4, buf, len);
			switch (filterout()) {
			case 1:
				t = natout();
				if (t == NULL)
					goto next;
				if (t->flags & TUNDEBUG)
					debugrcv4++;
				encap(t);
				break;

			case 2:
				t = naticmpout();
				if (t == NULL)
					goto next;
				if (t->flags & TUNDEBUG)
					debugrcv4++;
				encap(t);
				break;

			case 3:
				t = nonatout();
				if (t == NULL)
					goto next;
				if (t->flags & TUNDEBUG)
					debugrcv4++;
				encap(t);
				break;

			default:
				goto next;
			}
			goto next;
		}
		logerr("unexpected IP version number %d\n",
		       (buf[0] & IPVERMSK) >> 4);
	}

    next:
	/* every second: decay rates, expire nats, fragments */
	if (seconds != lastsecs) {
		while (seconds != lastsecs) {
			ratercv6[0] *= DECAY1;
			ratercv6[1] *= DECAY5;
			ratercv6[2] *= DECAY15;
			ratercv4[0] *= DECAY1;
			ratercv4[1] *= DECAY5;
			ratercv4[2] *= DECAY15;
			ratesent6[0] *= DECAY1;
			ratesent6[1] *= DECAY5;
			ratesent6[2] *= DECAY15;
			ratesent4[0] *= DECAY1;
			ratesent4[1] *= DECAY5;
			ratesent4[2] *= DECAY15;
			ratecnat[0] *= DECAY1;
			ratecnat[1] *= DECAY5;
			ratecnat[2] *= DECAY15;
			ratednat[0] *= DECAY1;
			ratednat[1] *= DECAY5;
			ratednat[2] *= DECAY15;
			lastsecs++;
		}
		delta = (double) (statsrcv6 - lastrcv6);
		lastrcv6 = statsrcv6;
		ratercv6[0] += (1.0 - DECAY1) * delta;
		ratercv6[1] += (1.0 - DECAY5) * delta;
		ratercv6[2] += (1.0 - DECAY15) *delta;
		delta = (double) (statsrcv4 - lastrcv4);
		lastrcv4 = statsrcv4;
		ratercv4[0] += (1.0 - DECAY1) * delta;
		ratercv4[1] += (1.0 - DECAY5) * delta;
		ratercv4[2] += (1.0 - DECAY15) * delta;
		delta = (double) (statssent6 - lastsent6);
		lastsent6 = statssent6;
		ratesent6[0] += (1.0 - DECAY1) * delta;
		ratesent6[1] += (1.0 - DECAY5) * delta;
		ratesent6[2] += (1.0 - DECAY15) * delta;
		delta = (double) (statssent4 - lastsent4);
		lastsent4 = statssent4;
		ratesent4[0] += (1.0 - DECAY1) * delta;
		ratesent4[1] += (1.0 - DECAY5) * delta;
		ratesent4[2] += (1.0 - DECAY15) * delta;
		delta = (double) (statscnat - lastcnat);
		lastcnat = statscnat;
		ratecnat[0] += (1.0 - DECAY1) * delta;
		ratecnat[1] += (1.0 - DECAY5) * delta;
		ratecnat[2] += (1.0 - DECAY15) * delta;
		delta = (double) (statsdnat - lastdnat);
		lastdnat = statsdnat;
		ratednat[0] += (1.0 - DECAY1) * delta;
		ratednat[1] += (1.0 - DECAY5) * delta;
		ratednat[2] += (1.0 - DECAY15) * delta;

		for (;;) {
			n = nat_heap_element(1);
			if ((n == NULL) || (n->timeout >= seconds))
				break;
			del_nat(n);
		}
		for (;;) {
			/* next must be the first field */
			f = *frags6_last;
			if ((f == NULL) || (f->expire >= seconds))
				break;
			statsdropped[DR_F6TM]++;
			if (f->tunnel->flags & TUNDEBUG)
				debugdropped[DR_F6TM]++;
			del_frag6(f);
		}
		for (;;) {
			/* next must be the first field */
			f = *fragsin_last;
			if ((f == NULL) || (f->expire >= seconds))
				break;
			statsdropped[DR_FINTM]++;
			if (f->tunnel->flags & TUNDEBUG)
				debugdropped[DR_FINTM]++;
			del_frag4(f);
		}
		for (;;) {
			/* next must be the first field */
			f = *fragsout_last;
			if ((f == NULL) || (f->expire >= seconds))
				break;
			statsdropped[DR_FOUTTM]++;
			del_frag4(f);
		}
	}

	/* every 256 seconds: resize hash tables */
	if (((seconds - startsecs) & 0xff) == 0) {
		if (pfhlookups / 4 > pfhhits) {
			u_int newhashsz;
			struct frag **newhash;

			newhashsz = fraghashsz * 2;
			if (newhashsz > MAXFRAGHASH) {
				logerr("fragment trashing\n");
				goto natsz;
			}
			newhash = (struct frag **)
				malloc(newhashsz * sizeof(*newhash));
			if (newhash == NULL) {
				logerr("malloc(newfraghash): %s\n",
				       strerror(errno));
				goto natsz;
			}
			memset(newhash, 0, newhashsz * sizeof(*newhash));
			fraghashsz = newhashsz;
			free(fraghash);
			fraghash = newhash;
			fraghashrnd = (uint32_t) random();
			pfhhits = newhashsz;
			loginfo("upsize fragment hash\n");
		} else if ((9 * pfhlookups) / 10 < pfhhits) {
			u_int newhashsz;
			struct frag **newhash;

			newhashsz = fraghashsz / 2;
			if (newhashsz < MINFRAGHASH)
				goto natsz;
			newhash = (struct frag **)
				malloc(newhashsz * sizeof(*newhash));
			if (newhash == NULL) {
				logerr("malloc(newfraghash): %s\n",
				       strerror(errno));
				goto natsz;
			}
			memset(newhash, 0, newhashsz * sizeof(*newhash));
			fraghashsz = newhashsz;
			free(fraghash);
			fraghash = newhash;
			fraghashrnd = (uint32_t) random();
			pfhhits = newhashsz;
			loginfo("downsize fragment hash\n");
		}
	    natsz:
		if (pnhlookups / 4 > pnhhits) {
			u_int newhashsz;
			struct nat **newhash;

			newhashsz = nathashsz * 2;
			if (newhashsz > MAXNATHASH) {
				logerr("NAT trashing\n");
				goto tunsz;
			}
			newhash = (struct nat **)
				malloc(newhashsz * sizeof(*newhash));
			if (newhash == NULL) {
				logerr("malloc(newnathash): %s\n",
				       strerror(errno));
				goto tunsz;
			}
			memset(newhash, 0, newhashsz * sizeof(*newhash));
			nathashsz = newhashsz;
			free(nathash);
			nathash = newhash;
			nathashrnd = (uint32_t) random();
			pnhhits = newhashsz;
			loginfo("upsize NAT hash\n");
		} else if ((9 * pnhlookups) / 10 < pnhhits) {
			u_int newhashsz;
			struct nat **newhash;

			newhashsz = nathashsz / 2;
			if (newhashsz < MINNATHASH)
				goto tunsz;
			newhash = (struct nat **)
				malloc(newhashsz * sizeof(*newhash));
			if (newhash == NULL) {
				logerr("malloc(newnathash): %s\n",
				       strerror(errno));
				goto tunsz;
			}
			memset(newhash, 0, newhashsz * sizeof(*newhash));
			nathashsz = newhashsz;
			free(nathash);
			nathash = newhash;
			nathashrnd = (uint32_t) random();
			pnhhits = newhashsz;
			loginfo("downsize NAT hash\n");
		}
	    tunsz:
		if (pthlookups / 4 > pthhits) {
			u_int newhashsz;
			struct tunnel **newhash;

			newhashsz = tunhashsz * 2;
			if (newhashsz > MAXTUNHASH) {
				logerr("tunnel trashing\n");
				return 0;
			}
			newhash = (struct tunnel **)
				malloc(newhashsz * sizeof(*newhash));
			if (newhash == NULL) {
				logerr("malloc(newtunhash): %s\n",
				       strerror(errno));
				return 0;
			}
			memset(newhash, 0, newhashsz * sizeof(*newhash));
			tunhashsz = newhashsz;
			free(tunhash);
			tunhash = newhash;
			tunhashrnd = (uint32_t) random();
			pthhits = newhashsz;
			loginfo("upsize tunnel hash\n");
		} else if ((9 * pthlookups) / 10 < pthhits) {
			u_int newhashsz;
			struct tunnel **newhash;

			newhashsz = tunhashsz / 2;
			if (newhashsz < MINTUNHASH)
				return 0;
			newhash = (struct tunnel **)
				malloc(newhashsz * sizeof(*newhash));
			if (newhash == NULL) {
				logerr("malloc(newtunhash): %s\n",
				       strerror(errno));
				return 0;
			}
			memset(newhash, 0, newhashsz * sizeof(*newhash));
			tunhashsz = newhashsz;
			free(tunhash);
			tunhash = newhash;
			tunhashrnd = (uint32_t) random();
			pthhits = newhashsz;
			loginfo("downsize tunnel hash\n");
		}
	}
	return 0;
}

/* Initialize hash tables */

void
init_hashes(void)
{
	lastsecs = startsecs = time(NULL);

	fraghashsz = MINFRAGHASH;
	fraghashrnd = (uint32_t) random();
	fraghash = (struct frag **) malloc(fraghashsz * sizeof(*fraghash));
	if (fraghash == NULL) {
		logerr("malloc(fraghash): %s\n", strerror(errno));
		exit(-1);
	}
	memset(fraghash, 0, fraghashsz * sizeof(*fraghash));
	pfhhits = fraghashsz;

	nathashsz = MINNATHASH;
	nathashrnd = (uint32_t) random();
	nathash = (struct nat **) malloc(nathashsz * sizeof(*nathash));
	if (nathash == NULL) {
		logerr("malloc(nathash): %s\n", strerror(errno));
		exit(-1);
	}
	memset(nathash, 0, nathashsz * sizeof(*nathash));

	tunhashsz = MINTUNHASH;
	tunhashrnd = (uint32_t) random();
	tunhash = (struct tunnel **) malloc(tunhashsz * sizeof(*tunhash));
	if (tunhash == NULL) {
		logerr("malloc(tunhash): %s\n", strerror(errno));
		exit(-1);
	}
	memset(tunhash, 0, tunhashsz * sizeof(*tunhash));
}

/* Run setup start script */

void
setup_start(void)
{
	int fd;

	fd = tun_open();
	if (fd <= 0) {
		logerr("tun_open() failed\n");
		exit(-1);
	}
	if (system(SETUP_START) < 0) {
		logerr("system() failed\n");
		(void) system(SETUP_STOP);
		close(fd);
		exit(-1);
	}
	tunfd = fd;
}

/* Run setup stop script */

void
setup_stop(void)
{
	(void) system(SETUP_STOP);
	close(tunfd);
	tunfd = -1;
}

/* main */

int
main( /* int argc, char **argv */ void)
{
	/* amd64: 80, 24, 128, 64
	   i386:  56, 16, 80, 36
	*/
#ifdef SIZES
	printf("tunnel=%d, bucket=%d nat=%d, frag=%d\n",
	       (int) sizeof(struct tunnel),
	       (int) sizeof(struct bucket),
	       (int) sizeof(struct nat),
	       (int) sizeof(struct frag));
#endif
	init_hashes();
	setup_start();
	if (load() != 0)
		goto stop;
	if (natsrccnt == 0) {
		logerr("no natted address?\n");
		goto stop;
	}
	if (icmpsrc[0] == 0) {
		logerr("no interface address?\n");
		goto stop;
	}
	if (acl6_first == NULL) {
		logerr("no acl6 entry?\n");
		goto stop;
	}
	while (loop() != 1)
		/* continue */;
  stop:
	setup_stop();
	return 0;
}
