// $Id: whois.cc 1.6.1.2 Wed, 15 Oct 1997 11:28:18 -0700 wlee $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu <cengiz@isi.edu>

#include "config.hh"
#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <cstdarg>
#include <iostream.h>


extern "C" {
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>

extern int connect(...);
extern int socket(...);
extern struct hostent *gethostbyname(...);

#ifdef DEFINE_HTONS_WITH_ELLIPSIS
extern u_short htons(...);
#else  // DEFINE_HTONS_WITH_ELLIPSIS
#ifdef DEFINE_HTONS_WITH_USHORT
extern u_short htons(unsigned short);
#endif // DEFINE_HTONS_WITH_USHORT
#endif // DEFINE_HTONS_WITH_ELLIPSIS

#ifndef HAVE_MEMCPY
#  define memcpy(d, s, n) bcopy ((s), (d), (n))
#endif
}

#include "whois.hh"
#include "trace.hh"
#include "_SetOfPix.hh"
#include "Map.hh"

WhoisParser whois;

char const *Whois::dflt_host      = "whois.ra.net";
char const *Whois::dflt_sources   = "";
int   Whois::dflt_port      = 43;
int   Whois::ignore_errors  = No;

int Whois::ArgvHost(char *dst, char *key, const char *_nextArg) {
   if (_nextArg)
      {
      SetDefaultHost(_nextArg);
      return 1; // return 1 to signify nextArg is used by us
      }
   return 0;
}

int Whois::ArgvSources(char *dst, char *key, const char *_nextArg) {
   if (_nextArg)
      {
      SetDefaultSources(_nextArg);
      return 1; // return 1 to signify nextArg is used by us
      }
   return 0;
}

int Whois::ArgvPort(char *dst, char *key, const char *_nextArg) {
   if (_nextArg)
      {
      SetDefaultPort(atoi(_nextArg));
      return 1; // return 1 to signify nextArg is used by us
      }
   return 0;
}
 
int Whois::ReportErrors(char *dst, char *key, const char *_nextArg) {
   ignore_errors = No;
   return 0; // return 0 to signify nextArg is not used by us
}

int Whois::IgnoreErrors(char *dst, char *key, const char *_nextArg) {
   ignore_errors = Yes;
   return 0; // return 0 to signify nextArg is not used by us
}

void Whois::Open(const char *_host, const int _port, const char *_sources) {
   if (_is_open)
      Close();

   // see if we should ignore error messages from IRR
   error.ignore(ignore_errors);

   struct sockaddr_in server_sockaddr;
   struct hostent *hp;
   int sock;

   Trace(TR_WHOIS_QUERY) << "Whois: Open " 
			 << _host << ", " << _port << ", " << _sources << endl;

   strcpy(host, _host);
   strcpy(sources, _sources);
   port = _port;

   hp = gethostbyname(host);
   if (!hp)
      error.Die("Error: gethostbyname(%s) failed.\n", host);

   server_sockaddr.sin_family = AF_INET;
   memcpy((char *) &(server_sockaddr.sin_addr.s_addr), hp->h_addr, hp->h_length);
   server_sockaddr.sin_port = htons((u_short) port);
   
   sock = socket(AF_INET, SOCK_STREAM, 0);
/*
   // Set keep alive option for the socket -- wlee@isi.edu
   int on = 1;
   if (setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (void *)&on, sizeof(on)) < 0)
     error.warning("Warning: setsockopt SO_KEEPALIVE failed!\n");
*/
   if (sock < 0)
      error.Die("Error: socket() failed.\n");

   if (connect(sock, (struct sockaddr *) &server_sockaddr, 
	       sizeof(server_sockaddr)) < 0)
      error.Die("Error: connect() failed.\n");


   in  = fdopen(sock, "r");
   out = fdopen(sock, "w");
   _is_open = 1;

   fwrite("!!\n", 1, 3, out);  // keep the connection open
   fflush(out);

   SetSources(sources);
}

void Whois::Close() {
   if (!_is_open)
      return;

   Trace(TR_WHOIS_QUERY) << "Whois: Close" << endl;

   fwrite("q\n", 1, 2, out);  // quit
   fflush(out);

   fclose(in);
   fclose(out);
   _is_open = 0;
}

void Whois::SetSources(const char *_sources) {
   int err = 0;

   if (! _is_open)
      Open();

   if (*_sources)
      strcpy(sources, _sources);
   else
      strcpy(sources, "-*");

   Trace(TR_WHOIS_QUERY) << "Whois: SetSources " << sources << endl;

   if (! QueryKillResponse("!s%s", sources)) {
      err = 1;
      error.error("Error: setting source to %s failed.\n", sources);
   }

   if (current_sources) 
      delete [] current_sources;
   QueryResponse(current_sources, "!s-lc");
   current_sources[strlen(current_sources) - 1] = 0; // chop \n

   if (err)
      error.error("Error: current source setting is %s.\n", sources);
}

char *Whois::GetSources(void) {
   if (! _is_open)
      Open();
   return current_sources;
}

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

int Whois::PendingData() {
   int status;
   int answer;

   if ((status = fcntl(fileno(in), F_GETFL, 0)) < 0)
      return TRUE;

   if (fcntl(fileno(in), F_SETFL, O_NDELAY) < 0)
      return TRUE;

   char c = fgetc(in);
   if (c == EOF) {
      clearerr(in);
      answer = FALSE;
   } else {
      ungetc(c, in);
      answer = TRUE;
   }

   fcntl(fileno(in), F_SETFL, status);

   Trace(TR_WHOIS_QUERY) << "Whois: PendingData " << answer << endl;

   return answer;
}

int Whois::Response(char *&response) { 
   if (!_is_open)
      Open();

   char buffer[1024]; 

   // Read the "A<byte-count>" line
   if (! fgets(buffer, sizeof(buffer), in)) {
      error.fatal("Error: fgets() failed.\n");
      return 0;
   }

   Trace(TR_WHOIS_RESPONSE) << "Whois: Response " << buffer << endl;

   if (*buffer == 'D') { // key not found error
      error.warning("Warning: key not found error for query %s.\n", last_query);
      return 0;
   }
   if (*buffer == 'C') { // returned success
      response = new char[1];
      *response = 0;
      return 1;
   }
   if (*buffer != 'A') { // we are expecting a byte-count line
      error.warning("Warning: no byte count error for query %s.\n", last_query);
      return 0;
   }

   int count = atoi(buffer + 1);
   response = new char[count + 1];

   if (fread(response, 1, count, in) != count)
      error.fatal("Error: fread() failed.\n");
   response[count] = 0;

   Trace(TR_WHOIS_RESPONSE) << "Whois: Response\n" << response << endl;

   // Read the return code line
   if (!fgets(buffer, sizeof(buffer), in)) {
      error.fatal("Error: fgets() failed.\n");
      return 0;
   }

   Trace(TR_WHOIS_RESPONSE) << "Whois: Response " << buffer << endl;

   if (*buffer != 'C') {
      error.warning("Warning: no end of data line error for query %s.\n", 
		    last_query);
      /* following is commented which will cause a leak. but o/w it may cause other run time errors */
      /*      delete [] response;
      response = NULL; */
      return 0;
   }

   return 1;
}

// rusty - this function was added to read an irrserver response, 
// including the initial 'A' and following 'C' lines.  used by relayd
// to cache responses from the the server.
int Whois::TotalResponse(char *&response) { 
   if (!_is_open)
      Open();
   if (error())
       return 0;

   char buffer[1024]; 

   // Read the "A<byte-count>" line
   if (!fgets(buffer, sizeof(buffer), in)) {
       error ("fgets() failed.");
       return 0;
   }

   if (*buffer == 'D') { // key not found error
       response = new char[strlen(buffer)];
       strcpy (response, buffer);
   } else if (*buffer == 'C') { // returned success
       response = new char[strlen (buffer) + 1];
       strcpy (response, buffer);
   } else if (*buffer != 'A') { // we are expecting a byte-count line
       response = new char[strlen (buffer) + 1];
       strcpy (response, buffer);
   }

   if (*buffer != 'A')
       return (strlen (response));

   // AXXX + body + return_code & extra, if any
   int len = strlen (buffer);
   int count = atoi(buffer + 1);
   response = new char[len + count + 80];

   strcpy (response, buffer);
   if (fread((char *) &response[len], 1, count, in) != count) {
       error ("fread() failed.");
       return 0;
   }

   len += count;
   response[len] = '\0';

   // Read the return code line
   if (!fgets(buffer, sizeof(buffer), in)) {
      error ("fgets() failed.");
      return 0; 
   }

   strcpy ((char *) &response[len], buffer);
   len += strlen (buffer);
   response[len] = '\0';

   return (error()) ? 0 : len;	// rusty: 0 or count
}

void Whois::WriteQuery(const char *format, va_list ap) {
   vsprintf(last_query, format, ap);

   int length = strlen(last_query);

   if (last_query[length - 1] != '\n') {
      last_query[length] = '\n';
      length++;
      last_query[length] = 0;
   }

   //   if (! fwrite(last_query, 1, length, out))
   if (fwrite(last_query, 1, length, out) != length)
      error.fatal("Error: fwrite() failed.\n");

   fflush(out);

   last_query[length - 1] = 0;
   Trace(TR_WHOIS_QUERY) << "Whois: WriteQuery " << last_query << endl;
}

void Whois::Query(const char *format, ...) {
   if (!_is_open)
      Open();

   va_list ap;
   va_start(ap, format);

   WriteQuery(format, ap);

   va_end(args);
}

int Whois::QueryResponse(char *&response, const char *format, ...) { 
   if (!_is_open)
      Open();

   va_list ap;
   va_start(ap, format);

   WriteQuery(format, ap);

   va_end(args);

   return Response(response);
}

int Whois::QueryKillResponse(const char *format, ...) { 
   char *response = NULL;
   
   if (!_is_open)
      Open();

   va_list ap;
   va_start(ap, format);

   WriteQuery(format, ap);

   va_end(args);

   int code = Response(response);

   if (response)
      delete [] response;
   return code;
}


//////////////////////////////////////////////////////////////////////

int WhoisParser::ParseAutNum(const char *query, AutNum &as) {
   Query(query);

   extern AutNum *irr_parser_as;
   irr_parser_as = &as;
   int code = ParseObject();
   irr_parser_as = NULL;
   return code;
}

int WhoisParser::ParseRoute(const char *query, Route &r) {
   Query(query);

   extern Route *irr_parser_rt;
   irr_parser_rt = &r;
   int code = ParseObject();
   irr_parser_rt = NULL;
   return code;
}

int WhoisParser::ParseObject() {
   char buffer[1024];

   // Read the "A<byte-count>" line
   if (!fgets(buffer, sizeof(buffer), in)) {
      error.fatal("Error: fgets() failed.\n");
      return 0;
   }

   Trace(TR_WHOIS_RESPONSE) << "Whois: Response " << buffer << endl;

   if (*buffer == 'D') { // key not found error
      error.warning("Warning: key not found error for query %s.\n", last_query);
      return 0;
   }
   if (*buffer == 'C') { // returned success
      return 1;
   }
   if (*buffer != 'A') { // we are expecting a byte-count line
      error.warning("Warning: no byte count error for query %s.\n", last_query);
      return 0;
   }

   extern int irr_lexer_input_size;
   extern FILE *irrin;
   extern int irrparse();

   irr_lexer_input_size = atoi(buffer + 1);
   irrin = in;
   irrparse();

   // Read the return code line
   if (!fgets(buffer, sizeof(buffer), in)) {
      error.fatal("Error: fgets() failed.\n");
      return 0;
   }

   Trace(TR_WHOIS_RESPONSE) << "Whois: Response " << buffer << endl;

   if (*buffer != 'C') {
      error.warning("Warning: no end of data line error for query %s.\n", 
		    last_query);
      return 0;
   }
   
   if (buffer[1] != '\n') {
      buffer[strlen(buffer) - 1] = 0;
      extern char *my_strchr(char *s, int c);
      char *s = my_strchr(buffer + 1, ','); // skip the source count
      if (s)
	 s++;
      else
	 s = buffer + 1;
      error.warning("Warning: Conflicting objects for query %s in sources %s.\n", 
		    last_query, s);   
   }

   return 1;
}

int WhoisParser::ParseExpansion(const char *query, const char *key, 
				_SetOfPix& result, 
				const CharPtr2Pix c2pix) {

   Query(query);

   char buffer[1024]; 

   // Read the "A<byte-count>" line
   if (!fgets(buffer, sizeof(buffer), in)) {
      error.fatal("Error: fgets() failed.\n");
      return 0;
   }

   Trace(TR_WHOIS_RESPONSE) << "Whois: Response " << buffer << endl;

   if (*buffer == 'D') { // key not found error
      error.warning("Warning: key not found error for query %s.\n", last_query);
      return 0;
   }
   if (*buffer == 'C') { // returned success
      return 1;
   }
   if (*buffer != 'A') { // we are expecting a byte-count line
      error.warning("Warning: no byte count error for query %s.\n", last_query);
      return 0;
   }

   int count = atoi(buffer + 1);

   char *response = (char *) STAllocator.allocate(count + 1);

   if (fread(response, 1, count, in) != count) {
      error.fatal("Error: fread() failed.\n");
      return 0;
   }

   Trace(TR_WHOIS_RESPONSE) << "Whois: Response " << response << endl;

   response[count] = 0;

   char *word;
   for (word = strtok(response, " \t\n"); word; word = strtok(NULL, " \t\n")) {
      result.add((*c2pix)(word));
   } 

   // Read the return code line
   if (! fgets(buffer, sizeof(buffer), in)) {
      error.fatal("Error: fgets() failed.\n");
      return 0;
   }

   Trace(TR_WHOIS_RESPONSE) << "Whois: Response " << buffer << endl;

   if (*buffer != 'C') {
      error.warning("Warning: no end of data line error for query %s.\n", 
		    last_query);
      return 0;
   }

   return 1;
}
