// $Id: regexp.hh 1.5 Fri, 18 Jul 1997 16:00:26 -0700 wlee $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary regexp_empty_strfor lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu <cengiz@isi.edu>

#ifndef REGEXP_H
#define REGEXP_H

#include "config.hh"
#include <iostream.h>
#include "_SetOfPix.hh"
#include "List.hh"
#include "Route.hh" // needed for regexp::match
#include "asno.hh"

extern "C" {
#include "re2dfa.h"
}

enum RegexpSymbol_t {
   REGEXP_SYMBOL_AS_NO, REGEXP_SYMBOL_AS_MACRO, REGEXP_SYMBOL_ANY
};

enum Regexp_t {
   REGEXP_UNKNOWN,
   REGEXP_EMPTY_SET,
   REGEXP_EMPTY_STR,
   REGEXP_SYMBOL,
   REGEXP_STAR,
   REGEXP_QUESTION,
   REGEXP_PLUS,
   REGEXP_CAT,
   REGEXP_OR,
   REGEXP_EOL,
   REGEXP_BOL,
   REGEXP_NF
};

class regexp {
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);
   friend ostream& operator<<(ostream& os, const regexp& r);

public:
   Regexp_t regexp_type;
   rd_fm* m;

   regexp() { 
      regexp_type = REGEXP_UNKNOWN;
      m = NULL;
   }
   regexp(const regexp& r) { 
      regexp_type = r.regexp_type; 
      m = NULL;
   }

   virtual ~regexp() {}

   virtual regexp* dup() const = 0;

   int operator==(regexp& b);
   int is_universal();
   int is_empty();

   regexp* cat(regexp *l, regexp *r) const;
   regexp* or(regexp *l, regexp *r) const;
   regexp* star(regexp *l) const;
   regexp* question(regexp *l) const;
  
   static void expand_AS_macros () {
      expand_as_macros = 1;
   }

   // private:
   virtual rd_fm* re2nfa() const = 0;
   regexp* construct(rd_fm* m) const;
   int match(ListHead<ListNodePix>& path);

   rd_fm* dfa();

   static int expand_as_macros;
};

class regexp_empty_set : public regexp {
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

public:
   regexp_empty_set() { 
      regexp_type = REGEXP_EMPTY_SET; 
   }
   regexp_empty_set(const regexp_empty_set& r) {
      regexp_type = REGEXP_EMPTY_SET;
   }
   virtual ~regexp_empty_set() {}
   virtual regexp* dup() const;

private:
   virtual rd_fm* re2nfa() const;
};

class regexp_bol : public regexp { // bol is beginning-of-line ^
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

public:
   regexp_bol() { 
      regexp_type = REGEXP_BOL; 
   }
   regexp_bol(const regexp_bol& r) {
      regexp_type = REGEXP_BOL;
   }
   virtual ~regexp_bol() {}
   virtual regexp* dup() const;

private:
   virtual rd_fm*  re2nfa() const;
};

class regexp_eol : public regexp { // eol is end-of-line $
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

public:
   regexp_eol() { 
      regexp_type = REGEXP_EOL; 
   }
   regexp_eol(const regexp_eol& r) {
      regexp_type = REGEXP_EOL;
   }
   virtual ~regexp_eol() {}

   virtual regexp* dup() const;

private:
   virtual rd_fm*  re2nfa() const;
};

class regexp_empty_str : public regexp {
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

public:
   regexp_empty_str () { 
      regexp_type = REGEXP_EMPTY_STR; 
   }
   regexp_empty_str (const regexp_empty_str& r)  {
      regexp_type = REGEXP_EMPTY_STR;
   }
   virtual ~regexp_empty_str () {}

   virtual regexp* dup() const;

private:
   virtual rd_fm*  re2nfa() const;
};

class regexp_symbol : public regexp {
   friend class regexp;
   friend ostream& operator<<(ostream& os, regexp_symbol& r);
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

private:
   re_asno_t asnumbers;
   _SetOfPix asmacros;
   int complemented;

public:
   regexp_symbol () { 
      regexp_type = REGEXP_SYMBOL; 
      complemented = 0;
   }
   // ~regexp_symbol() {   } // compiler is OK
   // regexp_symbol(const regexp_symbol& s) {ASSERT(0);} // compiler is OK

   regexp_symbol(RegexpSymbol_t type, Pix p = NULL) {
      regexp_type = REGEXP_SYMBOL; 
      complemented = 0;
      add(type, p);
   }

   void add(RegexpSymbol_t type, Pix p1 = NULL, Pix p2 = NULL);

   void add(int as1, int as2) {
      asnumbers.add(as1, as2);
   }

   void complement() {
      complemented = ! complemented;
   }

   virtual regexp* dup() const;

private:
   virtual rd_fm*  re2nfa() const;
};

class regexp_cat : public regexp {
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

private:
   regexp*	left;
   regexp*	right;

public:
   regexp_cat () {
      regexp_type = REGEXP_CAT; 
      left = (regexp*) NULL; 
      right = (regexp*) NULL; 
   }

   regexp_cat(const regexp_cat& r) : left(r.left->dup()),right(r.right->dup()) {
      regexp_type = REGEXP_CAT;
   }

   regexp_cat(regexp *l, regexp *r) {
      left = l;
      right = r;
      regexp_type = REGEXP_CAT; 
   }

   virtual ~regexp_cat() { 
      if (left)
	 delete left;
      if (right)
	 delete right;
   }

   virtual regexp* dup() const;

private:
   virtual rd_fm*  re2nfa() const;
};

class regexp_or : public regexp {
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

private:
   regexp*	left;
   regexp*	right;

public:
   regexp_or() {
      regexp_type = REGEXP_OR; 
      left = (regexp*) NULL; 
      right = (regexp*) NULL; 
   }

   regexp_or(const regexp_or& s) : left(s.left->dup()),right(s.right->dup()) {
      regexp_type = REGEXP_OR;
   }

   regexp_or(regexp *l, regexp *r) {
      left = l;
      right = r;
      regexp_type = REGEXP_OR; 
   }

   virtual ~regexp_or() {
      if (left)
	 delete left;
      if (right)
	 delete right;
   }

   virtual regexp* dup() const;

private:
   virtual rd_fm*  re2nfa() const;
};

class regexp_star : public regexp {
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

private:
   regexp*	left;

public:
   regexp_star(){
      regexp_type = REGEXP_STAR; 
      left = (regexp*) NULL; 
   }
   regexp_star(const regexp_star& s) : left(s.left->dup()) {
      regexp_type = REGEXP_STAR;
   }
   regexp_star(regexp *l) {
      left = l;
      regexp_type = REGEXP_STAR; 
   }
   
   virtual ~regexp_star() {
      if (left)
	 delete left;
   }

   virtual regexp* dup() const;

private:
   virtual rd_fm* re2nfa() const;
};

class regexp_question : public regexp {
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

private:
   regexp*	left;

public:
   regexp_question(){
      regexp_type = REGEXP_QUESTION; 
      left = (regexp*) NULL; 
   }
   regexp_question(const regexp_question& s) : left(s.left->dup()) {
      regexp_type = REGEXP_QUESTION;
   }
   regexp_question(regexp *l) {
      left = l;
      regexp_type = REGEXP_QUESTION; 
   }
   virtual ~regexp_question() {
      if (left)
	 delete left;
   }
   virtual regexp* dup() const;

private:
   virtual rd_fm*  re2nfa() const;
};

class regexp_plus : public regexp {
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);

private:
   regexp*	left;

public:
   regexp_plus(){
      regexp_type = REGEXP_PLUS; 
      left = (regexp*) NULL; 
   }
   regexp_plus(const regexp_plus& s) : left(s.left->dup()) {
      regexp_type = REGEXP_PLUS;
   }
   regexp_plus(regexp *l) {
      left = l;
      regexp_type = REGEXP_PLUS; 
   }

   virtual ~regexp_plus() {
      if (left)
	 delete left;
   }
   virtual regexp* dup() const;

private:
   virtual rd_fm*  re2nfa() const;
};

class regexp_nf : public regexp {
   friend class regexp;
   friend ostream& operator<<(ostream& os, const regexp& r);
   friend ostream& operator<<(ostream& os, const regexp_nf& r);
   friend int cisco_print_re_(ostream& os, const regexp& r);
   friend regexp *rsd_convert_re_bol(regexp *r, int asno);
   friend int cisco_print_as_path(regexp_nf&, int&, int, int, int, int);

private:
   class RegexpConjunct {
   public:
      class ReInt {
      public:
	 regexp *re;
	 int    negated;
	 ListNode regexs;
	 ReInt() : regexs(this) {
	    re = NULL;
	    negated = 0;
	 }
	 ReInt(const ReInt& b) : regexs(this) {
	    re = b.re->dup();
	    negated = b.negated;
	 }
	// Added by wlee@isi.edu
	~ReInt(void) { 
	  if (re) delete re;
	}
      };
      int    mark;
      ListHead<ReInt> regexs;
      ListNode        rclist;
      RegexpConjunct() : rclist(this) {
	 mark    = 0;
      }
      RegexpConjunct(const RegexpConjunct &s);
   };
   ListHead<RegexpConjunct> rclist;

public:
   regexp_nf() {
      become_empty();
   }

   regexp_nf(const regexp_nf& s);

   regexp_nf(regexp* re) {
      regexp_type = REGEXP_NF;
      RegexpConjunct *rc = new RegexpConjunct;
      RegexpConjunct::ReInt *ri = new RegexpConjunct::ReInt;
      ri->re = re;
      ri->negated = 0;
      rc->regexs.append(ri->regexs);
      rclist.append(rc->rclist);
      m = re->dfa();
  }

  // Added by wlee@isi.edu
  ~regexp_nf(void) {
     if (m) rd_free_dfa(m);
  }

   virtual regexp* dup() const;

   regexp_nf* dup_nf() const { 
      return new regexp_nf(*this);
   }

   void become_universal();
   void become_empty();

   void do_not();
   void do_or(regexp_nf &b);	// destroys b
   void do_and(regexp_nf &b);	// destroys b

private:
   virtual rd_fm* re2nfa() const;
   void do_and_terms(regexp_nf &b);	// destroys b's terms
};



#endif   // REGEXP_H
