#!/usr/bin/perl -w

#=============================================================================
#
#  Copyright 2006  Etienne URBAH for the EGEE project
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details at
#  http://www.gnu.org/licenses/gpl.html
#
#  This script parses the log file generated by lfc-perf-symlinks.sh to
#  generate a file in TSV format (tab separated values).
#
#=============================================================================

use strict ;

#-----------------------------------------------------------------------------
#  Constants
#-----------------------------------------------------------------------------
my @uniqueCommands   = ( 'lcg-cr', 'lfc-mkdir', 'lfc-rm', 'lcg-del' ) ;
my $uniquePattern    = '^(' . join('|', @uniqueCommands) . ')' ;

my @multipleCommands = ( 'create symlinks:',
                         'lfc-ls:         ',
                         'list   symlinks:',
                         'delete symlinks:' ) ;
my $multiplePattern  = '^(?:(python3?2? +lfc-create-symlinks\\.py) +[^ ]+|'.
                           '(lfc-ls)|'.
                           '(python3?2? +lfc-readdirg\\.py)|'.
                           '(python3?2? +lfc-rm-symlinks-of-dir)\\.py)' ;

my $timestamp = ' ' x 19 ;
my $hostname  = ' ' x 30 ;
my $version   = ' ' x  7 ;
my $se_host   = ' ' x 30 ;
my $ce_host   = ' ' x 30 ;
my $log_host  = ' ' x 30 ;

#-----------------------------------------------------------------------------
#  Variables
#-----------------------------------------------------------------------------
my $lfc_host ;
my @hosts ;
my $command ;
my $commandLine ;
my $b_unique_OK ;
my $index ;
my $number ;
my %commands ;
my %results ;
my %uniqueErrors ;
my %multipleErrors ;
my $time ;
my %times ;
my %nb ;
my %total ;

#-----------------------------------------------------------------------------
#  Loop on the lines of the log file
#-----------------------------------------------------------------------------
while  ( <> )
{
  #---------------------------------------------------------------------------
  #  Environment
  #---------------------------------------------------------------------------
  if    ( m/^TIMESTAMP=(.*)/ )
        { $timestamp = $1 }

  elsif ( m/^HOSTNAME=(.*)/ )
        { $hostname = sprintf('%-30s', $1) }

  elsif ( m/^GLITE_VERSION=(.*)/ )
        { $version = sprintf('%-7s', $1) }

  elsif ( m/^SE_HOST=(.*)/ )
        { $se_host = sprintf('%-30s', $1) }

  elsif ( m/^GlueCEUniqueID=([^:]*):/ )
        { $ce_host = sprintf('%-30s', $1) }

  elsif ( m/^GLITE_WMS_LOG_DESTINATION=(.*)/ )
        { $log_host = sprintf('%-30s', $1) }

  elsif ( m/^LFC_HOST=([^#\n].*)/ )
        {
          $lfc_host = sprintf('%-30s', $1) ;
          if  ( not defined($nb{$lfc_host}) )
              {
                push(@hosts, $lfc_host) ;
                foreach  $command  ( @uniqueCommands )
                {
                  $nb{$lfc_host}{$command}    = 0 ;
                  $total{$lfc_host}{$command} = 0 ;
                }
              }
        }
  
  #---------------------------------------------------------------------------
  #  RTT  (Round Trip Time)
  #---------------------------------------------------------------------------
  elsif ( m|^(rtt)\s+.*\s+=\s+[0-9\.]+/([0-9\.]+)/|  )
        { $times{$lfc_host}{$1} = sprintf("%9s\tms", $2) }
  
  #---------------------------------------------------------------------------
  #  Commands on an unique folder or file
  #---------------------------------------------------------------------------
  elsif ( m/$uniquePattern /  )
        {
          $command     = $1 ;
          $commandLine = $_ ;
          $b_unique_OK = 1 ;
        }
  
  #---------------------------------------------------------------------------
  #  Commands on multiples file
  #---------------------------------------------------------------------------
  elsif ( m/^$multiplePattern +_folder_([0-9]+)_files/ )
        {
          $index   = (grep(defined($-[$_]), 1..$#-))[0] - 1 ;
          $command = $multipleCommands[$index] ;
          $number  = sprintf('%9s', $+) ;
          push(@{$commands{$lfc_host}}, [$command, $number]) ;
          $results{$lfc_host}{$command}{$number} = '' ;
        }
  
  #---------------------------------------------------------------------------
  #  Error messages
  #---------------------------------------------------------------------------
  elsif ( m/^Error|^send2nsd:|error:/i and
          defined($lfc_host) and defined($command) )
        {
          if  ( defined($number) )
              { push(@{$multipleErrors{$lfc_host}{$command}{$number}}, $_) }
          else
              {
                if  ( $b_unique_OK )
                    { push(@{$uniqueErrors{$lfc_host}{$command}},
                           'LFC_HOST=', $lfc_host, "\n", $commandLine) }
                push(@{$uniqueErrors{$lfc_host}{$command}}, $_) ;
                $b_unique_OK = 0 ;
              }
        }

  #---------------------------------------------------------------------------
  #  Result messages
  #---------------------------------------------------------------------------
  elsif ( m/^ *([0-9]+)$/ or
          m/_folder_[0-9]+_files *: *([0-9]+)( +symbolic +links +(created|deleted)|$)/ )
        { $results{$lfc_host}{$command}{$number} = sprintf("%9s", $1) }
  
  #---------------------------------------------------------------------------
  #  Elapsed time
  #---------------------------------------------------------------------------
  elsif ( m/^real\s+(.*)/ )
        {
          $time = $1 ;
          
          if    ( $time =~ m/^([0-9\.]+) *s/ )
                { $time = $1 }
          elsif ( $time =~ m/^([0-9]+)m([0-9\.]+)s/ )
                { $time = (60 * $1) + $2 }
          
          if    ( defined($number) )
                { $times{$lfc_host}{$command}{$number} =
                      sprintf("%9s\ts ", $time) }
          elsif ( $b_unique_OK )
                {
                  $nb{$lfc_host}{$command} ++ ;
                  $total{$lfc_host}{$command} += $time ;
                }
          
          undef $number ;
          undef $command ;
        }
}

#-----------------------------------------------------------------------------
#  Print header
#-----------------------------------------------------------------------------
if  ( $ce_host eq (' ' x 30) )
    { $ce_host = $log_host }

printf "%-16s\t%9s\t%9s\t%9s\t  \t%-30s\t%-30s\t%-30s\t%-30s\t%-7s\t%-19s\n",
       'Command', 'Number', 'Result', 'Time', 'LFC_host', 'CE_host',
       'WN_host', 'SE_host', 'Version', 'Timestamp' ;

#-----------------------------------------------------------------------------
#  Loop on the LFC hosts to print lines
#-----------------------------------------------------------------------------
my $b_OK = 1 ;
my $result ;
my $errorsRef ;

foreach  $lfc_host  ( @hosts )
{
  #---------------------------------------------------------------------------
  #  RTT  (Round Trip Time)
  #---------------------------------------------------------------------------
  if  ( defined($times{$lfc_host}{'rtt'}) )
      { print "rtt:   ", "         \t" x 3, $times{$lfc_host}{'rtt'},
              "\t", $lfc_host, "\t", $ce_host, "\t", $hostname, "\t",
              $se_host, "\t", $version, "\t", $timestamp, "\n" }
  
  #---------------------------------------------------------------------------
  #  Commands on an unique folder or file
  #---------------------------------------------------------------------------
  foreach  $command  ( @uniqueCommands )
  {
    if  ( defined($total{$lfc_host}{$command}) and
          ($total{$lfc_host}{$command} > 0) )
        { printf "%-16s\t         \t         \t%9.3f\ts \t".
                 "%s\t%s\t%s\t%s\t%s\t%s\n",
                 $command.':',
                 $total{$lfc_host}{$command} / $nb{$lfc_host}{$command},
                 $lfc_host, $ce_host, $hostname, $se_host, $version,
                 $timestamp }
    
    $errorsRef = $uniqueErrors{$lfc_host}{$command} ;
    if  ( defined($errorsRef) )
        { print STDERR @{$errorsRef}, "\n" }
  }
  
  #---------------------------------------------------------------------------
  #  Commands on multiples file
  #---------------------------------------------------------------------------
  foreach  my $commandAndNumber  ( @{$commands{$lfc_host}} )
  {
    ($command, $number) = @{$commandAndNumber} ;
    $result = $results{$lfc_host}{$command}{$number} ;
    print $command, "\t", $number, "\t", $result, "\t",
          $times{$lfc_host}{$command}{$number}, "\t",
          $lfc_host, "\t", $ce_host, "\t", $hostname, "\t",
          $se_host, "\t", $version, "\t", $timestamp, "\n" ;
    
    #-------------------------------------------------------------------------
    #  Error messages and Erroneous results 
    #-------------------------------------------------------------------------
    $errorsRef = $multipleErrors{$lfc_host}{$command}{$number} ;
    if  ( defined($errorsRef) or ($result ne $number) )
        {
          if  ( $b_OK )
              {
                printf STDERR "%-16s\t%9s\t%9s\t%9s\t  \t".
                              "%-30s\t%-30s\t%-30s\t%-30s\t%-7s\t%-19s\n\n",
                              'Command',  'Number',  'Result',  'Time',
                              'LFC_host', 'CE_host', 'WN_host', 'SE_host',
                              'Version', 'Timestamp' ;
                $b_OK = 0 ;
              }
          if  ( $result ne $number )
              { print STDERR $command, "\t", $number, "\t", $result, "\t",
                             $times{$lfc_host}{$command}{$number}, "\t",
                             $lfc_host, "\t", $ce_host, "\t", $hostname, "\t",
                             $se_host, "\t", $version, "\t", $timestamp,
                             "\n" }
          if  ( defined($errorsRef) )
              { print STDERR @{$errorsRef}, "\n" }
        }
  }
}
