/*
 * YH - Console Chinese Environment -
 * Copyright (C) 1999 Red Flag Linux (office@sonata.iscas.ac.cn)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */


/*****************---vgadisp.c----*********************/


#include <stdio.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/errno.h>
#include <sys/types.h>

#ifdef OPENSERVER
#include <sys/vtkd.h>
#include <sys/console.h>
#else
#include <sys/vt.h>
#include <sys/kd.h>
#endif

#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <string.h>
#include <stdlib.h>

#include "display.h"
#include "incode.h"

#include <vga.h>
#include <sys/io.h>

#define SIG_REL 	SIGUSR1
#define SIG_ACQ 	SIGUSR2


#define TXTSCRSIZE	6100	/* save screen text size */

/********* Attribute values *********/

#define ATTR_UNDERL	0x01	/* Underline bit for B&W card */
#define ATTR_INTENSE	0x08

#define	ATTR_BLINK	0x80
#define	ATTR_BOLDBKG	0x80
#define ATTR_BOLDF	0x08

#define BL_ENB	0x20	/* High - Selects Blinking attribute in ALPHA */

#define A(x)    		((struct vga_a *) ((x)->adp_area))

struct vga_a 
{
	u_char a_battr;        /* Bold & Blink attributes      */
	u_char a_cattr;        /* Color/Reverse attributes     */
	u_char a_attrs;        /* OR battr with cattr          */
	u_char a_modereg;      /* contents of 6845 mode reg    */
	ushort  a_csr;         /* cursor shape                 */
	ushort  a_scroff;      /* offset of screen in video    */
	u_char a_dspmode;      /* mode of the display          */
	u_char a_colorreg;     /* contents of the color reg    */
	u_char a_fonttype;     /* font type ie 8x8 or 8x14     */
	ushort  a_VgaDisp;      /* Display type                */
};

extern int	fdVGA;

int	isdisplayed;
int	in_ansi = 0;
int should_acq_screen = 0;
int should_rel_screen = 0;

static char *Screenmem;

static void vga1_scroll();	
static void vga1_copy();
static void vga1_clear();
static void vga1_pchar();
void vga1_scurs(int echo_on);	
static void vga1_sgr();
static int vga1_adapctl();

#ifdef linux
extern void save_register(void);
extern void restore_register(void);
static void init_register(void);
#endif

int 	incode;
extern unsigned short int big5toucs[];
extern unsigned short int ucstogbk[];
static void Big5StrToUcs();
static void UcsStrToGbk();
static int codeflag=0;
unsigned char tmp;

u_char *glyphbuf;
u_char gbk[2];

#include "screen.h"
static struct port_io_struct pis;
                
struct adapter cn_adapsw[]=	
{
   	vga1_scroll,		
	vga1_copy,
	vga1_clear,
	vga1_pchar,
	vga1_scurs,
	vga1_sgr,
	vga1_adapctl
};

static char	szGBK16FontFile[] = "/usr/local/yh/lib/gbk16.fnt";

#define GLYPH_BUF_SIZE			32
#define GB_SECTION_GLYPH_BUF_SIZE	(32 * 94)
#define GB_FONT_BUF_SIZE		(GB_SECTION_GLYPH_BUF_SIZE * 94)

typedef unsigned char	BYTE;

static char	*readfntfile(int ascii, int *plen);
static int	is_gbk(BYTE b1, BYTE b2);
static int	is_gb(BYTE b1, BYTE b2);
static BYTE 	*read_gbk16glyph(BYTE gbk_hz[2]);
static void	read_gb16font(BYTE gbfontbuf[GB_FONT_BUF_SIZE]);

static int graf();                     /* Set everything up        */
static void grafend();                 /* Restore user's text mode */
static void grafquit();                /* Clean-up and exit        */
//static void rel_screen();
//static void acq_screen();

static void vga1_grafmode();
static void vga1_restore();
static void grafmode();

void	     RedrawScreen(struct screen *scrn);
void 	     SetColor(int plane);
void 	     BitMask(int mask);
void 	     SetWriteMode(unsigned mode);
unsigned     ReadRegister(unsigned outport, unsigned regist, unsigned mode);
void	     WriteRegister(unsigned indexport,unsigned index, unsigned data );

/*void  clear_screen(struct screen *);*/

static void	m6845sgr(struct screen *scrn, int gr);
static void     m6845adapctl(struct screen *msp, int cmd, int arg);
static int      vgaac_cmdtobgcl(struct screen *msp,int cmd); 


/*************************functions*******************************/

/*
 * This route open the needed font files and initiate the multiscreen
 */
int	initdisplay()
{
	static char	mv_savscrn[TXTSCRSIZE];
	struct screen	*scrn = &g_scrn;
        
	int len;
        
	if( (scrn->mv_font_text = readfntfile(1, &len)) == NULL )
		return	-1;
	if (graf() < 0)
		return	-1;
	scrn->mv_row = scrn->mv_col = 0;
	scrn->mv_top=0;	
	scrn->mv_rsz = 25;
	scrn->mv_csz = 80;
	scrn->mv_wrap=0;   
	scrn->mv_tabstop[0]=0x01010100;	
	scrn->mv_tabstop[1]=
	scrn->mv_tabstop[2]=
	scrn->mv_tabstop[3]=
	scrn->mv_tabstop[4]=0x01010101;
	scrn->mv_csstate = 0;
	scrn->mv_curflg = 0;
	scrn->mb_time = 2;
	scrn->mb_freq = 0x600;
	scrn->mv_font = 0;
	scrn->mv_norm.fore = WHITE;
	scrn->mv_norm.back = BLACK;
	scrn->mv_rev.fore = BLACK;
	scrn->mv_rev.back = WHITE;
	scrn->mv_grfc.fore = WHITE;
	scrn->mv_grfc.back = BLACK;
	scrn->mv_ovscan = BLACK;
	scrn->mf_status |= MFS_TEXTMODE;
	scrn->mv_savsz = 80*25;
	scrn->mv_adapter = cn_adapsw;	
	scrn->mv_savscrn = mv_savscrn;
	memset(scrn->mv_savscrn,0,TXTSCRSIZE);
	cn_adapsw[0].v_videoram = Screenmem;
	cn_adapsw[0].v_vidoff = 0L;

	A(scrn)->a_battr=0;
	A(scrn)->a_cattr=scrn->mv_norm.back<<4|scrn->mv_norm.fore;
	A(scrn)->a_attrs=scrn->mv_norm.back<<4|scrn->mv_norm.fore;
	A(scrn)->a_csr=0x1000|0x012;
	A(scrn)->a_scroff=0;
	A(scrn)->a_fonttype=0;
	A(scrn)->a_VgaDisp=VGA;
	Writebackground(scrn);
	PutCC16(scrn,0,25,7,"");
	return	0;
}


/*
 * Set up the graphics multiscreen stuff and call another
 * routine to set up card.
 */
static int	graf()
{
	struct vt_mode smode;
	int adapter, privlcmd;
	struct sigaction	act;
	struct sembuf	sops[1];

	act.sa_flags = 0;
	memset(&act.sa_mask, 0, sizeof(act.sa_mask));

	act.sa_handler = rel_screen;
#if (defined(OPENSERVER) || defined(GEMINI))
	act.sa_sigaction = rel_screen;
#endif
	sigaction(SIG_REL, &act, NULL);
	act.sa_handler = acq_screen;
#if (defined(OPENSERVER) || defined(GEMINI))
	act.sa_sigaction = acq_screen;
#endif
	sigaction(SIG_ACQ, &act, NULL);
#ifdef UNIXWARE
	smode.mode = VT_PROCESS;
#else
	smode.mode = VT_PROCESS;
#endif
	smode.waitv = 0;        /* not implemented, reserved */
        smode.relsig = SIG_REL;
	smode.acqsig = SIG_ACQ;
	smode.frsig  = SIGINT;  /* not implemented, reserved */
	if(-1 == ioctl(fdVGA, VT_SETMODE, &smode))
		return	-1;
        privlcmd=KDENABIO;
	if(-1 == ioctl(fdVGA, privlcmd, 1))
		return	-1;
        ioperm(0x3c0,26,1); 	
        if (-1==vga_setmode(G640x480x16))
           return -1;
   	Screenmem=(char *)vga_getgraphmem();
	if (-1 == (int) Screenmem)
		return	-1;
	isdisplayed = 1;
	return	0;
}


void	rel_screen()
{
#ifdef UNIXWARE
	u_char	*msg = (u_char *) "\n\n\033[1m˳׻ƽ̨"
			"лͼλ밴Enter\033[0m\n";
	ansi(msg, strlen((char *) msg));
#else
	struct sembuf	sops[1];
	if (in_ansi)
	{
		should_rel_screen = 1;
		return;
	}
	isdisplayed = 0;
	killtimer();
#ifdef OPENSERVER
	ioctl(fdVGA, VT_RELDISP, VT_TRUE);
#else
	ioctl(fdVGA, VT_RELDISP, 1);
#endif
#endif
}


void	acq_screen()
{
	struct sembuf	sops[1];

	if (in_ansi)
	{
		should_acq_screen = 1;
		return;
	}

	vga1_restore();
	ioctl(fdVGA, VT_RELDISP, VT_ACKACQ);
	settimer(333);
	g_scrn.mv_curflg=0;
	isdisplayed = 1;
}


static void	vga1_restore()
{
	int 		x,y,i;
	unsigned char   oldclockmode;
	char 		*Screenmem;

#if 0	
        ioctl(fdVGA,MODESWITCH | SW_VGA12,0);
        Screenmem = (char *)ioctl(fdVGA, MAPCONS, (char *)0);
#else
        vga_setmode(4);
   	Screenmem=(char *)vga_getgraphmem();
#endif
	cn_adapsw[0].v_videoram = Screenmem;
	cn_adapsw[0].v_vidoff = 0L;
	(&g_scrn)->mv_adapter->v_videoram = Screenmem;
	(&g_scrn)->mv_adapter->v_vidoff = 0L;
	oldclockmode=ReadRegister(0x3c4,0x3c5,0x01);
	WriteRegister(0x3c4,0x01,oldclockmode|0x20);
	if(bottom_disable!=1)
	{
		Writebackground(&g_scrn);
		redrawbottom(&g_scrn,0,25,7);
	}

	RedrawScreen(&g_scrn);
	WriteRegister(0x3c4,0x01,oldclockmode);
}


void	redrawbottom(struct screen *scrn,int x,int y,int color)
{
	unsigned Zcode,Bcode;
	int i , rec ;
	faddr_t pdest,vidram,vidram1,vidram_g;
	unsigned char oldlogicmode,oldwritemode;

        Writebackground(scrn);
        
        SetColor(color);
	BitMask(0xff);

	vidram=&scrn->mv_savscrn[6000];
	vidram1=&scrn->mv_savscrn[6000+1];
	vidram_g = scrn->mv_adapter->v_videoram;
	oldlogicmode=ReadRegister(0x3ce,0x3cf,0x03);
	oldwritemode=ReadRegister(0x3ce,0x3cf,0x05);

	SetWriteMode(0);
	WriteRegister(0x3ce,0x03,00);

	while(*vidram)
	{
		pdest = &vidram_g[((10+y*18)*80+x)&0xFFFF];
		if((*vidram&0x80)&&(*vidram1&0x80))
		{
			gbk[0] = *vidram;
			gbk[1] = *vidram1;
			glyphbuf = read_gbk16glyph(gbk);

			for(i=0;i<16;i++)
			{
				pdest = &vidram_g[(x+(10+y*18+i)*scrn->mv_csz)&0xFFFF];
				*pdest=~glyphbuf[0+2*i];
				pdest = &vidram_g[(x+1+(10+y*18+i)*scrn->mv_csz)&0xFFFF];
				*pdest=~glyphbuf[0+2*i+1];
			}
			x+=2 ;
			if(x>scrn->mv_csz)
				return;
			vidram=&scrn->mv_savscrn[(6000+x)%TXTSCRSIZE];
			vidram1=&scrn->mv_savscrn[(6000+x+1)%TXTSCRSIZE];
		}
		else
		{
			for( i = 0; i < 16; i++)
			{
				pdest = &vidram_g[(x+(10+y*18+i)*scrn->mv_csz)&0xFFFF];
				*pdest = ~scrn->mv_font_text[(*vidram) * 16 + i];
			}
			x+=1;
			if(x>scrn->mv_csz)
				return;
			vidram=&scrn->mv_savscrn[(6000+x)%TXTSCRSIZE];
			vidram1=&scrn->mv_savscrn[(6000+x+1)%TXTSCRSIZE];
		}
	}
	SetWriteMode(oldwritemode);
	WriteRegister(0x3ce,0x03,oldlogicmode);
}


void	ClearBottom(struct screen *scrn)
{
	int i;
	faddr_t  vidram_g;      /* The whole video buffer */
	register int offset;
	register faddr_t Screen;
	unsigned char oldlogicmode,oldwritemode;
	if(!isdisplayed)
		return;

	oldlogicmode=ReadRegister(0x3ce,0x3cf,0x03);
	oldwritemode=ReadRegister(0x3ce,0x3cf,0x05);

	SetWriteMode(0);
	WriteRegister(0x3ce,0x03,00);

	SetColor(0x0f);
	BitMask(0xff);

	vidram_g = scrn->mv_adapter->v_videoram;
	offset = scrn->mv_adapter->v_vidoff+452*scrn->mv_csz;

	for( i = 0; i < 27*scrn->mv_csz; i++)
	{
		Screen = &vidram_g[(offset ++ )&0xFFFF];
		*Screen = 0;
	}

	SetColor(A(scrn)->a_attrs&0x0f);
}


void	Writebackground(scrn)
struct screen *scrn;
{
	register int i;
        char color[641];
        
        ClearBottom(scrn);
        
        color[640]='\0';
        for(i=0;i<640;i++)
           color[i]=7;
        vga_drawscanline(452,color);
        vga_drawscanline(454,color);
        for(i=457;i<479;i++)
           vga_drawscanline(i,color);
	SetColor(A(scrn)->a_attrs&0x0f);
}


static char	*readfntfile(ascii, plen)
int		ascii;
int		*plen;
{
	FILE			*pf;
	char			*buf;

	*plen = ascii ? 4096 : GB_FONT_BUF_SIZE;
	buf = (char *) malloc(*plen);
	if (ascii)
	{
		pf = fopen("/usr/local/yh/lib/8x16.fnt", "r");
		fread(buf, 1, *plen, pf);
		fclose(pf);
	}
	else
		read_gb16font((BYTE *) buf);
	return	buf;
}


/* 
 * put hanzi into the bottom of screen
 */
void	PutCC16(struct screen *scrn,int x,int y,int color, u_char *str)
{
	unsigned Zcode,Bcode;
	int i , rec ;
	faddr_t pdest,vidram,vidram_g;
	FILE *fp;
	int incode;

	SetWriteMode(0);
	SetColor(color);
	BitMask(0xff);
	vidram=&scrn->mv_savscrn[6000];
	vidram_g = scrn->mv_adapter->v_videoram;

	if(!isdisplayed)
		return ;

	WriteRegister(0x3ce,0x03,0);

	fp=fopen("/usr/local/yh/lib/incode.txt","r");
	fscanf(fp,"%d",&incode);
	fclose(fp);

	while(*str)
	{
		pdest = &vidram_g[((10+y*18)*80+x)&0xFFFF];
		vidram=&scrn->mv_savscrn[(6000+x)%TXTSCRSIZE];
		if(*str>0x80&&*str<0xff)
		{
		   if(!incode)
		   {
			unsigned char tmp1[5],tmp2[5];
			int num1,num2;
			tmp1[0]=str[0];
			tmp1[1]=str[1];
			if(str[0]==0xa1 && str[1]==0xe3)
			{
				str[0]=0xa1;
				str[1]=0xab;
				goto gbk;
			}
			Big5StrToUcs(tmp1,tmp2,2,4,&num1,&num2);
			UcsStrToGbk(tmp2,tmp1,num2,num2,&num1,&num2);
			str[0]=tmp1[0];
			str[1]=tmp1[1];
		   }
gbk:			*vidram=*str;
			vidram=&scrn->mv_savscrn[(6000+x+1)%TXTSCRSIZE];
			*vidram=str[1];
			gbk[0] = *str;
			gbk[1] = str[1];
			glyphbuf = read_gbk16glyph(gbk);

			for(i=0;i<16;i++)
			{
				pdest = &vidram_g[(x+(10+y*18+i)*scrn->mv_csz)&0xFFFF];
				*pdest=~glyphbuf[2*i];
				pdest = &vidram_g[(x+1+(10+y*18+i)*scrn->mv_csz)&0xFFFF];
				*pdest=~glyphbuf[2*i+1];
			}
			x+=2 ;
			str+=2;
			if(x>=scrn->mv_csz)
				break;
		}
		else
		{
			*vidram=*str;
			for( i = 0; i < 16; i++)
			{
				pdest = &vidram_g[(x+(10+y*18+i)*scrn->mv_csz)&0xFFFF];
				*pdest = ~scrn->mv_font_text[(*str) * 16 + i];
			}
			x+=1;
			str+=1;
			if(x>=scrn->mv_csz)
				break;
		}
	}
}


void	RedrawScreen(scrn)
struct screen *scrn;
{
	unsigned char  word[3];
	faddr_t vidram;
	int i,k,j;
	faddr_t  vidram_g;      /* The whole video buffer */
	register int offset;
	register int lastoffset;
	register int curoffset;
	int lastcol,lastrow,curcol,currow;
	char *Screen;
	char old;
	int row,col;
	int tabzode=9+0xa0;

	unsigned Zcode,Bcode;
	int  rec ;
	static unsigned char firstchar,secondchar;

	int IsHanZiFlag;

	unsigned char oldlogicmode,oldwritemode;

	row=col=0;
	vidram=&scrn->mv_savscrn[0];
	vidram_g = scrn->mv_adapter->v_videoram;

	save_register();
	init_register();

	IsHanZiFlag=0;
	for(k = 0 ; k < scrn->mv_savsz ; k++)
	{
		
		word[0] = *vidram++;
		word[1] = *vidram++;
		word[2] = *vidram++;
		offset =  row*18*scrn->mv_csz + col + scrn->mv_csz;

                if(((word[2]==3) || (IsHanZiFlag)) && (word[0]>=0x40))
                {
			lastcol = curcol ;
			lastrow = currow ;
			curcol = col ;
			currow = row ;
			col = lastcol;
			row = lastrow;
			lastoffset =  row*18*scrn->mv_csz + col + scrn->mv_csz;
			col = curcol;
			row = currow;
			curoffset =  row*18*scrn->mv_csz + col + scrn->mv_csz;
				
			if(IsHanZiFlag)
			{
				outb(0,0x3ce);
				outb((word[1]>>4)&0x0f,0x3cf);
				outb(8,0x3ce);
				outb(0xff,0x3cf);
				for(i=0;i<18;i++)
				{
		Screen=&vidram_g[(lastoffset+(i-1)*scrn->mv_csz)&0xFFFF];
				old=*Screen;
				*Screen=0;
				old=*Screen;
		Screen=&vidram_g[(curoffset+(i-1)*scrn->mv_csz)&0xFFFF];
				old=*Screen;
				*Screen=0;
				}
				{
				u_char col[4];
				for(i=0;i<1;i++)
					col[i]=*Screen;
				}
					
				secondchar=word[0];
				gbk[1] = secondchar;
				gbk[0] = firstchar;
				glyphbuf = read_gbk16glyph(gbk);
				IsHanZiFlag=0;

				/*WriteRegister(0x3ce,0x03,0x00);
				SetColor(word[1]&0x0f);*/
				WriteRegister(0x3ce,0x0,word[1]&0x0f);
				outb(8,0x3ce);
				for(i=0;i<16;i++)
				{
		Screen = &vidram_g[(lastoffset+i * scrn->mv_csz)&0xFFFF];
				outb(glyphbuf[2*i],0x3cf);
				old=*Screen;
				*Screen = 0;
		Screen = &vidram_g[(curoffset + i * scrn->mv_csz)&0xFFFF];
				outb(glyphbuf[2*i+1],0x3cf);
				old=*Screen;
				*Screen=0;
				}

				if (firstchar==tabzode)
				{
					if (glyphbuf[0+2*15]!=0)
					{
						Screen=&vidram_g[(lastoffset+16*scrn->mv_csz)&0xFFFF];
						outb(glyphbuf[2*15],0x3cf);
						old=*Screen;
						*Screen = 0;
					}

					if (glyphbuf[0+2*15+1]!=0)
					{
						Screen=&vidram_g[(curoffset+16*scrn->mv_csz)&0xFFFF];
						outb(glyphbuf[2*15+1],0x3cf);
						old=*Screen;
						*Screen = 0;
					}

					if (glyphbuf[0]!=0)
					{
						Screen=&vidram_g[(lastoffset-scrn->mv_csz)&0xFFFF];
						outb(glyphbuf[0],0x3cf);
						old=*Screen;
						*Screen = 0;
					}

					if (glyphbuf[0+1]!=0)
					{
						Screen=&vidram_g[(curoffset-scrn->mv_csz)&0xFFFF];
						outb(glyphbuf[1],0x3cf);
						old=*Screen;
						*Screen = 0;
					}
				}
			}
			else 
			{
				IsHanZiFlag=1;
				firstchar=word[0];
			}
                }
		else if ((word[2]==1) || (word[2]==2) || (word[2]==4))
		{
			outb(0,0x3ce);
			outb((word[1]>>4)&0x0f,0x3cf);
			outb(8,0x3ce);
			outb(0xff,0x3cf);	
			for(i=0;i<18;i++)
			{
			Screen=&vidram_g[(offset+(i-1)*scrn->mv_csz)&0xFFFF];
			old=*Screen;
			*Screen=0;
			old=*Screen;
			}
	
			{
			u_char col[4];
			int i;
			outb(4,0x3ce);
			for(i=0;i<4;i++)
			  {
				outb(i,0x3cf);
				col[i]=*Screen;
			  }
			}

			IsHanZiFlag = 0 ;
			/*WriteRegister(0x3ce,0x03,0x00);
			SetColor(word[1]&0x0f);*/

			if(word[2]==4)
				offset -= scrn->mv_csz;

			outb(0,0x3ce);
			outb((word[1]&0x0f),0x3cf);
			outb(8,0x3ce);
			for( i = 0; i < 16; i++)
			{
			Screen = &vidram_g[(offset + i * scrn->mv_csz)&0xFFFF];
			outb(scrn->mv_font_text[word[0]*16+i],0x3cf);
			old=*Screen;
			*Screen=0;
			}

			if(word[2]==4)
			{
				if(scrn->mv_font_text[word[0]*16+15]!=0)
				{
					for(i=16;i<18;i++)
					{
			Screen = &vidram_g[(offset+i*scrn->mv_csz)&0xFFFF];
			outb(scrn->mv_font_text[word[0]*16+15],0x3cf);
					old = *Screen;
					*Screen = 0;
					}
				}
			}
                }
		else
                {
			outb(0,0x3ce);
			outb((word[1]>>4)&0x0f,0x3cf);
			outb(8,0x3ce);
			outb(0xff,0x3cf);
			for(i=0;i<18;i++)
			{
			Screen=&vidram_g[(offset+(i-1)*scrn->mv_csz)&0xFFFF];
			old=*Screen;
			*Screen=0;
			}			
		}

finish:
		col++;
		if(col==80)
		{
			row++;
			col=0;
		}
	}
	SetWriteMode(oldwritemode);
	restore_register();
}


/*
 * This function will output ascii ,extended ascii and chinese HANZI 
 */
static void vga1_pchar(scrn, bp, n)
struct screen *scrn;
char *bp;
int n;
{
	int 			no;
	faddr_t 		vid;	  	/* The whole video text buffer	*/
	int 			col,row;
	unsigned char 		oldwritemode,oldlogicmode;
	register int 		i;
	register int 		offset;
	register int 		lastoffset;
	register int 		curoffset;
	char 			*Screen;
	char 			*cp;
	char 			old;
	u_char 			ch;
	faddr_t 		vidram;	  	/* The whole video text buffer	*/
	faddr_t  		vidram_g;	/* The whole video buffer 	*/
	int 			tabzode=9+0xa0;
	unsigned 		Zcode,Bcode;
	int  			rec ;
	static int 		IsHanZiFlag;
	static unsigned char 	firstchar,secondchar;
	static int 		lastcol,curcol,lastrow,currow;
	static u_char 		first;
	int 			num;
	u_char 			c;

	{
		FILE *fp;
		fp=fopen("/usr/local/yh/lib/incode.txt","r");
		fscanf(fp,"%d",&incode);
		fclose(fp);
	}

	if((n>0) && (!incode))
	{
		unsigned char *tmpbuf1,*tmpbuf2;
		int num1,num2,i;
		if(codeflag)
		{
			if(scrn->mv_col==0)
			{
				scrn->mv_row--;
				scrn->mv_col=79;
			}
			else
				scrn->mv_col--;
			tmpbuf1=malloc(n+1);
			tmpbuf1[0]=tmp;
			for(i=0;i<n;i++) tmpbuf1[i+1]=bp[i];
			n=n+1;
		}
		else	
		{
			tmpbuf1=malloc(n);
			for(i=0;i<n;i++) tmpbuf1[i]=bp[i];
		}

		if ((tmpbuf2=malloc(2*n))==NULL) exit(1);
		Big5StrToUcs(tmpbuf1,tmpbuf2,n,2*n,&num1,&num2);

		n=num2;
		free(tmpbuf1);
		if ((tmpbuf1=malloc(n))==NULL) exit(1);
		UcsStrToGbk(tmpbuf2,tmpbuf1,n,n,&num1,&num2);

		n=num2;
		for(i=0;i<n;i++)   bp[i]=tmpbuf1[i];
		free(tmpbuf1);
		free(tmpbuf2);
	}


	/*if(!isdisplayed)
	{*/
		while(n-- > 0)
		{
			ch = *bp++;
			vidram = &scrn->mv_savscrn[(3*CURSOR(scrn))%TXTSCRSIZE];
			*vidram = ch;
			vidram = &scrn->mv_savscrn[(3*CURSOR(scrn)+1)%TXTSCRSIZE];
			*vidram = A(scrn)->a_attrs;

			if(scrn->mv_font==0)
			{
				if(((ch>=129)&&(ch<=254))|| IsHanZiFlag)
				{
					vidram = &scrn->mv_savscrn[(3*CURSOR(scrn)+2)%TXTSCRSIZE];
					*vidram = 3;
				}
				else if(isprint(ch)) /* write ascii  */
				{
					vidram = &scrn->mv_savscrn[(3*CURSOR(scrn)+2)%TXTSCRSIZE];
					*vidram = 1;
				}
				
			}
			else if(scrn->mv_font==2)
			{
				if(ch<128)
				{
					ch ^= 0x80;
					if((ch>=161)&&(ch<=254))
					{
						vidram=&scrn->mv_savscrn[(3*CURSOR(scrn)+2)%TXTSCRSIZE];
						*vidram = 3;
					}
				}
				else if(ch>=128)
				{
					vidram = &scrn->mv_savscrn[(3*CURSOR(scrn)+2)%TXTSCRSIZE];
					*vidram = 4;
				}
			}
			scrn->mv_col++;
		}
		return;
	}



static void  vga1_scroll(scrn, lines)   
register struct screen 	*scrn;
int lines;
{
	int count;
	unsigned char oldwritemode;
	faddr_t pdest;
	int i;
	faddr_t vidram;
	faddr_t dest_vidram,sour_vidram;
        char *cp;
        
	vidram=&scrn->mv_savscrn[0];
	count=lines*18*scrn->mv_csz;

	/*if(!isdisplayed)
	{*/
		if(lines>=0)
		{
			dest_vidram=&vidram[(3*scrn->mv_csz*scrn->mv_top)%TXTSCRSIZE];
			sour_vidram=&vidram[(3*scrn->mv_csz*(scrn->mv_top+lines))%TXTSCRSIZE];
			memmove(dest_vidram,sour_vidram,
			    3*scrn->mv_csz*(scrn->mv_rsz-scrn->mv_top-lines));

			for(i=0;i<lines*scrn->mv_csz;i++)
			{
				vidram[(3*i+3*scrn->mv_savsz-lines*3*scrn->mv_csz)\
					%TXTSCRSIZE]=0;
				vidram[(3*i+1+3*scrn->mv_savsz-lines*3*scrn->mv_csz)\
					%TXTSCRSIZE]= A(scrn)->a_attrs;
				vidram[(3*i+2+3*scrn->mv_savsz-lines*3*scrn->mv_csz)\
					%TXTSCRSIZE]=0;
			}
		}
		else 
		{
			dest_vidram=&vidram[(3*scrn->mv_csz*(scrn->mv_top-lines))%TXTSCRSIZE];
			sour_vidram=&vidram[(3*scrn->mv_csz*scrn->mv_top)%TXTSCRSIZE];
			memmove(dest_vidram,sour_vidram,
			    3*scrn->mv_csz*(scrn->mv_rsz+lines-scrn->mv_top));

			for(i=0;i<(-lines*scrn->mv_csz);i++)
			{
				vidram[(3*i+scrn->mv_top)%TXTSCRSIZE]=0;
				vidram[(3*i+1+scrn->mv_top)%TXTSCRSIZE]=A(scrn)->a_attrs;
				vidram[(3*i+2+scrn->mv_top)%TXTSCRSIZE]=0;
			}
		}
		return;
	}


static void   vga1_clear(scrn, row, col, count)  
struct screen 	*scrn;
unsigned short 	row, col, count;
{
	int	 	scrnsiz;
	int 	 	cursor;
	faddr_t  	vidram;       	/* The whole video text buffer       */
	faddr_t  	vid;          	/* The whole video text buffer       */
	faddr_t  	vidptr;       	/* pointer into video text buffer    */
	faddr_t  	vidram_g;       /* The whole video buffer	     */
	register 	faddr_t Screen;
	register 	int i;
	unsigned 	oldlogicmode,oldwritemode;
	int 	 	col1,row1,no;

	if(!isdisplayed)
		return;

	scrnsiz = SCREENSZ(scrn);
	cursor = (row * scrn->mv_csz) + col;
	if(count>(scrnsiz-cursor))
		return;

	/*
	 * Clear text buffer.
	 */
	vidram = scrn->mv_savscrn;
	vidptr = &vidram[(scrn->mv_adapter->v_vidoff + 3*cursor)%TXTSCRSIZE];

	for( i = 0; i < count;  i++)
	{
		vidptr[(3*i)%TXTSCRSIZE]=0;
		vidptr[(3*i+1)%TXTSCRSIZE]=A(scrn)->a_attrs;
		vidptr[(3*i+2)%TXTSCRSIZE]=0;
	}

	oldlogicmode=ReadRegister(0x3ce,0x3cf,0x03);
	oldwritemode=ReadRegister(0x3ce,0x3cf,0x05);
	SetWriteMode(0);
	WriteRegister(0x3ce,0x03,0x00);

	if(cursor == 0 && count == scrnsiz)
	{
		SetColor(~(A(scrn)->a_attrs>>4&0x0f));
		A(scrn)->a_scroff = 0;
		scrn->mv_adapter->v_vidoff = 0;
		memset( scrn->mv_adapter->v_videoram ,0 ,scrn->mv_rsz*scrn->mv_csz*18);
		if((A(scrn)->a_attrs>>4&0x0f)!=0)
		{

			SetColor(A(scrn)->a_attrs>>4&0x0f);
			memset(scrn->mv_adapter->v_videoram,0xff,
			    scrn->mv_rsz*scrn->mv_csz*18);
		}
		return;

	}

	/*
	 * Clear video buffer.
	 */
	vidram_g = scrn->mv_adapter->v_videoram;

	while ( count-- )
	{
		SetColor(~(A(scrn)->a_attrs>>4&0x0f));

		for(i = 0; i < 18; i++)
		{
			Screen=&vidram_g[(scrn->mv_csz*18*row+col+i*scrn->mv_csz)&0xFFFF];
			*Screen = 0;
		}

		if((A(scrn)->a_attrs>>4&0x0f)!=0)
		{
			SetColor(A(scrn)->a_attrs>>4&0x0f);

			for(i = 0; i < 18; i++)
			{
				Screen=&vidram_g[(scrn->mv_csz*18*row+col+\
					i*scrn->mv_csz)&0xFFFF];
				*Screen = 0xff;
			}
		}

		if(col<(scrn->mv_csz-1))
		{
			col++;
		}
		else
		{
			col=0;
			row++;
		}

	}

	WriteRegister(0x3ce,0x03,oldlogicmode);
	SetWriteMode(oldwritemode);
}


void	vga1_scurs(int echo_on)	
{	
        static int 		echo_tag=0;
        unsigned char 		attribute;
        unsigned char 		* cp;
	
        struct screen 		*scrn = &g_scrn;
	faddr_t 		vidram_g;       
	register faddr_t	Screen;
	register int 		i;
	int 			offset;
	char 			old;
	faddr_t 		vidram;
	register faddr_t 	vidptr ;
	u_char 			ch;
	unsigned char  		oldlogicmode;
        
	if (! isdisplayed)
		return;

	if (scrn->mv_curflg & CUR_LOCK) return;
	if (scrn->mv_curflg & CUR_HIDDEN) return;
	scrn->mv_curflg |= CUR_LOCK;

	vidram_g = scrn->mv_adapter->v_videoram;
	offset = scrn->mv_adapter->v_vidoff + CURSOR_G(scrn);
   	
        cp=&scrn->mv_savscrn[(3*CURSOR(scrn))%TXTSCRSIZE];
        if(*cp==0)
           attribute=A(scrn)->a_attrs;
        else
           attribute=*(cp+1);
        
        save_register();
        init_register();
     
        outb(8,0x3ce);
        outb(0xff,0x3cf);
        
        if(!echo_on || echo_tag==0)
        {
              outb(0,0x3ce);
              outb(attribute>>4&0x0f,0x3cf);
              echo_tag=1;
        
        }
        else
        {           	
              outb(0,0x3ce);
              outb(attribute&0x0f,0x3cf);
              echo_tag=0;
        }
        
	for( i = ansirange(A(scrn)->a_csr>>8&0xff,18);i< ansirange((A(scrn)->a_csr&0xff),18); i++)
	{
		Screen = &vidram_g[(offset + i * scrn->mv_csz)&0xFFFF];
		old=*Screen;
		*Screen = 0;
	}
                
   	restore_register();
        scrn->mv_curflg ^= CUR_ON;
	scrn->mv_curflg &= ~CUR_LOCK;
}


static void   vga1_copy(scrn, drow, dcol, srow, scol, count) 
struct screen	*scrn;
int 		drow, dcol, srow, scol, count;
{
	int src_csr, des_csr;
	int dir;
	faddr_t vidram;          /* The whole text buffer        */
	faddr_t src, des;        /* pointers into text buffer    */
	faddr_t vidram_g;       /* The whole video buffer       */
	faddr_t src_g, des_g;   /* pointers into video buffer   */
	int 	i;
	unsigned char oldwritemode;

	if(!isdisplayed)
		return;

	dir = 1;
	src_csr = A(scrn)->a_scroff + (scrn->mv_csz * srow) + scol;
	des_csr = A(scrn)->a_scroff + (scrn->mv_csz * drow) + dcol;
	if(src_csr < des_csr)
	{
		dir = -1;
		src_csr += count - 1;
		des_csr += count - 1;
	}

	/*
	 * Copy text in text buffer.
	 */
        vidram = scrn->mv_savscrn;
        src = &vidram[(3*(scol+scrn->mv_csz*srow))%TXTSCRSIZE];
        des = &vidram[(3*(dcol+scrn->mv_csz*drow))%TXTSCRSIZE];
        memmove(des,src,3*count);
}

/*
 * Set Graphical Rendition (set text attributes)
 * Call the common 6845 attibute code for most of
 * the stuff.  Take care of the special cases for EGA.
 */
static void   vga1_sgr(scrn, gr)  
struct screen *scrn;
int gr;
{

	switch(gr)
	{
	/*
	 * warning:
	 * Our use of a_modereg is as a logical place to store
	 * the color mode blink enable bit (like in the CGA)
	 * It is not the real value of attr_controller reg # 0x10.
	 */
	default:

		m6845sgr(scrn, gr);
		break;
	}

	A(scrn)->a_attrs = A(scrn)->a_cattr | A(scrn)->a_battr;

	if (mono)
	{
		if (A(scrn)->a_attrs > 0x10)
#if 0
			A(scrn)->a_attrs = ((mono & 0x07) << 4) | (mono >> 4);
#else
			A(scrn)->a_attrs = mono;
#endif
		else
			A(scrn)->a_attrs = mono;
	}
}


static int vga1_adapctl(msp, cmd, arg)  
struct screen *msp;
{
	int retcode = 1;

	switch(cmd)
	{
	default:
		m6845adapctl(msp, cmd, arg);
		break;
	case AC_ONSCREEN:
		{
			int i;
			i = ReadRegister(0x3c4,0x3c5,1);
			i &=~(0x20);
			WriteRegister(0x3c4,1,i);
		}
		break;
	case AC_OFFSCREEN:
		{
			int i;
			i=ReadRegister(0x3c4,0x3c5,1);
			i |=0x20;
			WriteRegister(0x3c4,1,i);
		}
		break;
	case AC_SENDSCRN:
		break;

	case AC_SETOS:
#ifdef NO       /* figure out vga overscan later */
		msp->mv_ovscan = A(msp)->a_colorreg = arg;
		break;
#endif
		break;
	case AC_ONN:
	case AC_ONR:
	case AC_ONG:
		m6845adapctl(msp, cmd, arg);

		if(vgaac_cmdtobgcl(msp, cmd) & 0x80)
			A(msp)->a_modereg &= ~BL_ENB;
		break;

	case AC_FONTCHAR:
	case AC_DEFCSR:
		m6845adapctl(msp, cmd, arg);
		break;

	case AC_PRIMODE:
		break;

	case AC_SAVSZQRY:
		break;

	case AC_USERFONT:
		break;

	case AC_SAVSCRN:
	case AC_RESSCRN:
		break;

	case AC_CSRCTL:
		if(arg)
			msp->mv_curflg=CUR_ON;
		else
			msp->mv_curflg=CUR_HIDDEN;

		return(retcode);

	case AC_IOPRIVL:
		break;

	case AC_SOFTRESET:
		break;
	}

	A(msp)->a_attrs = A(msp)->a_cattr | A(msp)->a_battr;

	if (mono)
	{
		if (A(msp)->a_attrs > 0x10)
#if 0
			A(msp)->a_attrs = ((mono & 0x07) << 4) | (mono >> 4);
#else
			A(msp)->a_attrs = mono;
#endif
		else
			A(msp)->a_attrs = mono;
	}

	return(retcode);
}


void	bell(bfreq, btime)
unsigned bfreq;
int btime;
{
	register int bt;
	int arg;

	bt = btime * 100;
	if (bt == 0)
		return;

	/* 
	 * bfreq is the period in microseconds (1/1000000s), but ioctl routine need 
	 * the  frequency, so we must convert the period to the frequency
	 */
	arg = (bt<<16)|(1000000/bfreq);
	ioctl(fdVGA, KDMKTONE,  arg);
}


/*
 * Programming the attributes for characters displayed
 * in text mode controlled by a 6845 Video Controller.
 */
static void  m6845sgr(scrn, gr)
struct screen *scrn;
int gr;
{
	int cl;
	
        switch(gr)
	{ 
	case SGR_NORMAL:
                scrn->mv_norm.fore=7;
                scrn->mv_norm.back=0;
                A(scrn)->a_cattr  = scrn->mv_norm.fore;
		A(scrn)->a_cattr |= scrn->mv_norm.back << 4;
                scrn->mf_status &= ~MFS_TMODES;
                scrn->mf_status &= ~MFS_BOLD;
                break;
	case SGR_BOLD:
                scrn->mf_status |= MFS_BOLD;
                A(scrn)->a_battr |= ATTR_INTENSE;
		break;
	case SGR_UNDERL:
		break;
	case SGR_BLINK:
		A(scrn)->a_battr |= ATTR_BLINK;
		break;
	case SGR_REVERSE:
		A(scrn)->a_cattr = scrn->mv_rev.fore;
		A(scrn)->a_cattr |= scrn->mv_rev.back << 4;
		scrn->mf_status &= ~MFS_TMODES;
		scrn->mf_status |=  MFS_REVSTEXT;
		break;
        case SGR_0REVERSE:
		A(scrn)->a_cattr  = scrn->mv_norm.fore;
		A(scrn)->a_cattr |= scrn->mv_norm.back << 4;
		break;
	case SGR_FORECOLOR+0:
	case SGR_FORECOLOR+1:
	case SGR_FORECOLOR+2:
	case SGR_FORECOLOR+3:
	case SGR_FORECOLOR+4:
	case SGR_FORECOLOR+5:
	case SGR_FORECOLOR+6:
	case SGR_FORECOLOR+7:
		scrn->mv_norm.fore = ansicolormap[gr-SGR_FORECOLOR];

		/*
 		 *      This sets reverse and graphics characters
		 *      which was a design decision on my part.
		 *      it may not be well liked.
		 *      brianm
		 */

		scrn->mv_rev.back = scrn->mv_norm.fore;
		scrn->mv_rev.fore = scrn->mv_norm.back;

		scrn->mv_grfc = scrn->mv_norm;

		if((scrn->mf_status&MFS_TMODES)==MFS_REVSTEXT)
		{
			A(scrn)->a_cattr = scrn->mv_rev.fore;
			A(scrn)->a_cattr |= scrn->mv_rev.back << 4;
                        if( scrn->mf_status & MFS_BOLD)
                            A(scrn)->a_battr |= ATTR_INTENSE;

		}
		else
		{
			A(scrn)->a_cattr &= ~0x07;
			A(scrn)->a_cattr |= ansicolormap[gr-SGR_FORECOLOR];
                        if( scrn->mf_status & MFS_BOLD)
                            A(scrn)->a_battr |= ATTR_INTENSE;

		}
		break;
	case SGR_BACKCOLOR+0:
	case SGR_BACKCOLOR+1:
	case SGR_BACKCOLOR+2:
	case SGR_BACKCOLOR+3:
	case SGR_BACKCOLOR+4:
	case SGR_BACKCOLOR+5:
	case SGR_BACKCOLOR+6:
	case SGR_BACKCOLOR+7:
		scrn->mv_norm.back = ansicolormap[gr-SGR_BACKCOLOR];

		/*
		 *      This sets reverse and graphics characters
		 *      which was a design decision on my part.
		 *      it may not be well liked.
		 *      brianm
		 */

		scrn->mv_rev.back = scrn->mv_norm.fore;
		scrn->mv_rev.fore = scrn->mv_norm.back;
		scrn->mv_grfc = scrn->mv_norm;

		if((scrn->mf_status&MFS_TMODES)==MFS_REVSTEXT)
		{
			A(scrn)->a_cattr = scrn->mv_rev.fore;
			A(scrn)->a_cattr |= scrn->mv_rev.back << 4;
                        if( scrn->mf_status & MFS_BOLD)
                            A(scrn)->a_battr |= ATTR_INTENSE;

		}
		else
		{
			A(scrn)->a_cattr &= ~0x70;
			A(scrn)->a_cattr |= ansicolormap[gr-SGR_BACKCOLOR] << 4;
                         if( scrn->mf_status & MFS_BOLD)
                            A(scrn)->a_battr |= ATTR_INTENSE;
		}
		break;
	default:
		return;
	}
	A(scrn)->a_attrs = A(scrn)->a_cattr | A(scrn)->a_battr;
}


static void m6845adapctl(msp, cmd, arg)
struct screen *msp;
{
	switch(cmd)
	{
	case AC_BLINKB:
		if(arg)
			A(msp)->a_modereg |= BL_ENB;
		else
			A(msp)->a_modereg &= ~BL_ENB;
		break;
	case AC_BOLDBKG:
		if(arg)
			A(msp)->a_cattr |= ATTR_BOLDBKG;
		else
			A(msp)->a_cattr &= ~ATTR_BOLDBKG;
		break;
	case AC_BOLDF:
		if(arg)
			A(msp)->a_cattr |= ATTR_BOLDF;
		else
			A(msp)->a_cattr &= ~ATTR_BOLDF;
		break;
        case AC_FONTCHAR:
		(*msp->mv_adapter->v_pchar)(msp, &arg, 1);
		break;
	case AC_DEFCSR:
		/* to fuck vi */
		if(arg==(0x0a00|0x0c))
			arg=0x0f00|0x011;
		A(msp)->a_csr = arg;
		break;
	case AC_DEFNF:
		if(A(msp)->a_cattr & ATTR_BOLDF)
                   msp->mv_norm.fore = arg+8;
                else
                   msp->mv_norm.fore = arg;
		break;
	case AC_DEFNB:
	        msp->mv_norm.back = arg;
		break;
	case AC_DEFRF:
		msp->mv_rev.fore = arg;
		break;
	case AC_DEFRB:
		msp->mv_rev.back = arg;
		break;
	case AC_DEFGF:
		msp->mv_grfc.fore = arg;
		break;
	case AC_DEFGB:
		msp->mv_grfc.back = arg;
		break;
	case AC_ONN:
		A(msp)->a_cattr = msp->mv_norm.back << 4 | msp->mv_norm.fore;
		break;
	case AC_ONR:
		A(msp)->a_cattr = msp->mv_rev.back << 4 | msp->mv_rev.fore;
		break;
	case AC_ONG:
		A(msp)->a_cattr = msp->mv_grfc.back << 4 | msp->mv_grfc.fore;
		break;
	case AC_SETOS:
		msp->mv_ovscan = arg;
		break;
	}
	A(msp)->a_attrs = A(msp)->a_cattr | A(msp)->a_battr;
}


static int vgaac_cmdtobgcl(msp, cmd)
struct screen *msp;
{
	switch(cmd)
	{
	case AC_ONN:
		return(msp->mv_norm.back);
	case AC_ONR:
		return(msp->mv_rev.back);
	case AC_ONG:
		return(msp->mv_grfc.back);
	}
}


/*
 * determine which color planes can be wrote
 * the value of the argu "plane" means, for example:
 *  0x0f	all four color planes,
 *  that is bit0,bit1,bit2,bit3,relate plane0,plane 1,plane2,plane3.
 *  Register S2 is Map Mask Register
 */
void SetColor(int plane)
{
#ifdef OPENSERVER
	/*set to write register S2 */
	pis.dir = OUT_ON_PORT;
	pis.port = 0x3c4;
	pis.data = 2;
	ioctl(fdVGA,VGAIO,&pis);

	/*write the prefer color plane */
	pis.port = 0x3c5;
	pis.data = plane;
	ioctl(fdVGA,VGAIO,&pis);
#endif
#ifdef linux
        outb(2,0x3c4);
        outb(plane,0x3c5);         
#else
   	outw(0x3c4, (plane << 8) | 2);
#endif
}


/*
 * determine which bits can be wrote
 */
void BitMask(int mask)
{
#ifdef OPENSERVER
	/* set to write register G8 */
	pis.dir = OUT_ON_PORT;
	pis.port = 0x3ce;
	pis.data = 0x08;
	ioctl(fdVGA,VGAIO,&pis);

	/* write the mask to G8 */
	pis.port = 0x3cf;
	pis.data = mask;
	ioctl(fdVGA,VGAIO,&pis);
#endif
#ifdef linux
        outb(8,0x3ce);
        outb(mask,0x3cf);         
#else
	outw(0x3ce, (mask << 8) | 0x08);
#endif
}


/*
 * set up write mode.
 *the value of the argu "mode" means:
 * 0x00  means mode 0
 * 0x01  means mode 1
 * 0x02  means mode 2------we use this mode
 * 0x03  means mode 3
 */
void SetWriteMode(unsigned mode)
{
	u_char old;
#ifdef OPENSERVER
	/* set to write register G5 */
	pis.dir = OUT_ON_PORT;
	pis.port = 0x3ce;
	pis.data = 0x05;
	ioctl(fdVGA,VGAIO,&pis);

	/* read G5 value into pis.data */
	pis.dir = IN_ON_PORT;
	pis.port = 0x3cf;
	ioctl(fdVGA,VGAIO,&pis);

	old =pis.data;
	/*clean the bit 0 and bit 1 */
	old &= 0xfc;
	/* Do XOR between old value and new value,and then transmit it to pis.data for write */
	pis.data = old | mode;
	pis.dir = OUT_ON_PORT;
	pis.port = 0x3cf;
	ioctl(fdVGA,VGAIO,&pis);
#endif
#ifdef linux
        outb(5,0x3ce);
        old = inb(0x3cf);
        old&=0xfc;
        outb(old | mode,0x3cf); 
        if (mode==0)
        { outb(1,0x3ce);
          outb(0,0x3cf);
        }        
#else
	outb(0x3ce, 0x05);
	old = inb(0x3cf);
	old &= 0xfc;
	outb(0x3cf, old | mode);
#endif
}


#ifdef linux
#define REG_NUM 10
static char old_register[REG_NUM];
void save_register(void)
{  
   int i,j;
        
   for(i=0;i<=8;i++)
   {
      outb(i,0x3ce);
      old_register[i]=inb(0x3cf);
   }
   for(j=2;j<=3;j++)
   {
      outb(j,0x3c4);
      old_register[i++]=inb(0x3c5);
   }
}


void restore_register(void)
{  
   int i,j;

   for(i=0;i<=8;i++)
   {
      outb(i,0x3ce);
      outb(old_register[i],0x3cf);
   }
   for(j=2;j<=3;j++)
   {
      outb(j,0x3c4);
      outb(old_register[i++],0x3c5);
   }
}


void init_register(void)
{  
   u_char old;
        
        /*write mode 0*/
   outb(5,0x3ce);
   old = inb(0x3cf);
   outb(old&0xfc,0x3cf); 
      /*permit register*/
   outb(1,0x3ce);
   outb(0x0f,0x3cf);
      /*page mask register*/
   outb(2,0x3c4);
   outb(0x0f,0x3c5);
   /*yiwei register*/
   outb(3,0x3ce);
   outb(0,0x3cf);
}
#endif  

         
unsigned ReadRegister(unsigned outport, unsigned regist, unsigned mode)
{
#ifdef OPENSERVER
	pis.dir = OUT_ON_PORT;
	pis.port = outport;
	pis.data = mode;
	ioctl(fdVGA,VGAIO,&pis);

	pis.dir = IN_ON_PORT;
	pis.port = regist;
	ioctl(fdVGA,VGAIO,&pis);
	return(pis.data);
#endif
#ifdef linux
        outb(mode,outport);
#else
	outb(outport, mode);
	return inb(regist);
#endif
}


void	WriteRegister(unsigned indexport,unsigned index, unsigned data )
{
#ifdef OPENSERVER
	pis.dir=OUT_ON_PORT;
	pis.port=indexport;
	pis.data=index;
	ioctl(fdVGA,VGAIO,&pis);

	pis.port=indexport+1;
	pis.data=data;
	ioctl(fdVGA,VGAIO,&pis);
#endif
#ifdef linux
        outb(index,indexport);
        outb(data,indexport+1);         
#else
	outw(indexport, (data << 8) | index);
#endif
}


static int   is_gbk(BYTE b1, BYTE b2)
{
	if ((b1 < 0x81) || (b1 > 0xfe))
		return 0;
	if ((b2 < 0x40) || (b2 == 0x7f) || (b2 > 0xfe))
		return 0;
	return	1;
}


static int   is_gb(BYTE b1, BYTE b2)
{
	if ((b1 < 0xa1) || (b1 > 0xfe))
		return 0;
	if ((b2 < 0xa1) || (b2 > 0xfe))
		return 0;
	return	1;
}


static BYTE *read_gbk16glyph(BYTE gbk_hz[2])
{
	FILE	*pf;
	int		idx;
	static int	should_read_gbfont = 1;
	static BYTE	gbfontbuf[GB_FONT_BUF_SIZE];
	static BYTE	glyphbuf[GLYPH_BUF_SIZE];

	if (should_read_gbfont)
	{
		should_read_gbfont = 0;
		read_gb16font(gbfontbuf);
	}

	if (! is_gbk(gbk_hz[0], gbk_hz[1]))
		goto	error;

	if (is_gb(gbk_hz[0], gbk_hz[1]))
	{
		idx = (gbk_hz[0] - 0xa1) * 94 + (gbk_hz[1] - 0xa1);
		return	&gbfontbuf[idx * GLYPH_BUF_SIZE];
	}

	/* non-GB GBK */
	idx = (gbk_hz[0] - 0x81) * 190 + (gbk_hz[1] - 0x40);
	if (gbk_hz[1] > 0x7f)
		idx--;

	pf = fopen(szGBK16FontFile, "rb");
	fseek(pf, GLYPH_BUF_SIZE * idx, SEEK_SET);
	fread(glyphbuf, 1, GLYPH_BUF_SIZE, pf);
	fclose(pf);
	return glyphbuf;

error:
	memset(glyphbuf, 0xff, GLYPH_BUF_SIZE); 
	return glyphbuf;
}


static void	read_gb16font(BYTE gbfontbuf[GB_FONT_BUF_SIZE])
{
	FILE	*pf;
	int	section;
	int	idx;

	pf = fopen(szGBK16FontFile, "rb");
	for (section = 0xa1; section <= 0xfe; section++)
	{
		idx = (section - 0x81) * 190 + (0xa1 - 0x40) - 1;
		fseek(pf, GLYPH_BUF_SIZE * idx, SEEK_SET);
		fread(gbfontbuf, 1, GB_SECTION_GLYPH_BUF_SIZE, pf);
		gbfontbuf += GB_SECTION_GLYPH_BUF_SIZE;
	}

	fclose(pf);
}


/*Note: if nToConvert == -1, we assume that the input string was ended by a zero (0) */
static void Big5StrToUcs(
	unsigned char *lpInBig5Str, /*Big5 string*/
	unsigned char *lpOutUcsStr, /*Ucs string*/
	int nToConvert,  /*number of bytes of Big5 string to be converted*/
	int nSizeUcsBuff, /*size of Ucs buffer lpOutUcsStr (in bytes)*/
	int *lpnBig5Converted, /*bytes of Big5 string being converted*/
	int *lpnUcsConverted) /*bytes of Ucs string being converted*/
{
	register int nIn, nOut;
	unsigned int ord;
	register unsigned char byte1,byte2;
	unsigned short int code;

	if (nToConvert == -1)  nToConvert=strlen(lpInBig5Str);
	if (nSizeUcsBuff == -1) nSizeUcsBuff=nToConvert;

	nIn=0; nOut=0;

	while ((nIn<nToConvert-1) && (nOut<nSizeUcsBuff-1))
	{
		byte1=lpInBig5Str[nIn++];
		if (byte1<=0x7f)
		{
			/*ascii*/
			lpOutUcsStr[nOut++]=0;
			lpOutUcsStr[nOut++]=byte1;
			continue;
		}
		if ((byte1<0xa1) || (byte1>0xf9))
		{
			/*unknown or single byte char*/
			lpOutUcsStr[nOut++]=BAD_UCS1;
			lpOutUcsStr[nOut++]=BAD_UCS2;
			continue;
		}
		/*double byte char, processed here*/
		byte2=lpInBig5Str[nIn++];
		if ((byte2>=0x40) && (byte2<=0x7e))
		{	
			ord=(byte1-0xa1)*157+(byte2-0x40);
			code=big5toucs[ord];
		}
		else if ((byte2>=0xa1) && (byte2<=0xfe))
		{
			ord=(byte1-0xa1)*157+(byte2-0xa1)+63;
			code=big5toucs[ord];
		}
		else
		{
			/*unknown or bad double byte char*/
			code=UNKNOWN_UCS;
		}
		lpOutUcsStr[nOut++]=(code>>8)&0xff;
		lpOutUcsStr[nOut++]=code&0xff;
	}
	if (nIn==nToConvert-1 && nOut<nSizeUcsBuff-1)
	/*let's process the last char*/
	{
		byte1=lpInBig5Str[nIn++];
		if(byte1<=0x7f)
		{ /*the last char was not Hanzi character,just copy it*/
			lpOutUcsStr[nOut++]=0;
			lpOutUcsStr[nOut++]=byte1;
			codeflag=0;
		}
		else
		{
			codeflag=1;
			tmp=byte1;
			nIn--;
		}
	}
	else
		codeflag=0;
	if (lpnBig5Converted) *lpnBig5Converted = nIn;
	if (lpnUcsConverted) *lpnUcsConverted = nOut;
	return;
}


/*Note: if nToConvert == -1, we assume that the input string was ended by a zero (0) */
static void UcsStrToGbk(
	unsigned char *lpInUcsStr,  /*UCS string*/
	unsigned char *lpOutGbkStr, /*Gbk string*/
	int nToConvert,   /*number of bytes of UCS string to be converted*/
	int nSizeGbkBuff, /*size of GBK buffer lpOutGbkStr (in bytes)*/
	int *lpnUcsConverted, /*bytes of Ucs string being converted*/
	int *lpnGbkConverted) /*bytes of Gbk string being converted*/
{		
	register int nIn,nOut;
	unsigned int ord;
	register unsigned char byte1,byte2;
	unsigned short int code;
	
	if (nToConvert == -1) nToConvert=strlen(lpInUcsStr);
	if (nSizeGbkBuff == -1) nSizeGbkBuff=nToConvert;
 
	nIn=0; nOut=0;

	while ((nIn<nToConvert-1) && (nOut<nSizeGbkBuff-1))
	{
		byte1=lpInUcsStr[nIn++];
		byte2=lpInUcsStr[nIn++];
		if (byte1 == 0 && (byte2 & 0x80) ==0)
		{	/*ascii*/
			lpOutGbkStr[nOut++]=byte2;
			continue;
		}
		ord=byte1*256+byte2;
		code=ucstogbk[ord];
		lpOutGbkStr[nOut++]=(code>>8) & 0xff;
		lpOutGbkStr[nOut++]=code & 0xff;
	}
	if (lpnUcsConverted) *lpnUcsConverted=nIn;
	if (lpnGbkConverted) *lpnGbkConverted=nOut;
	return;
}
