/* Copyright 2001 Matt Flax <flatmax@ieee.org>
   This file is part of MFFM Time Scale Modification for Audio.

   MFFM Time Scale Modification for Audio is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   MFFM Time Scale Modification for Audio is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You have received a copy of the GNU General Public License
   along with MFFM Time Scale Modification for Audio
 */
#ifndef WSOLA_STREAM_H_
#define WSOLA_STREAM_H_

/** This class is intended to simplify the interfacing of WSOLA to
    different media handling libraries.
    It allocates a memory buffer to load input audio data into and then works
    the WSOLA algorithm from this buffer.


    The paradigm is simple ... start by loading the buffer full.
    Each time the buffer enters the second half, load the first half.
    Each time the buffer enters the first half, load the second half.
*/

#include "WSOLA.H"

#define MAX_TAU 5.0
#define MIN_TAU 1e-2

class WSOLAStream : public WSOLA {
  ///This template is the cyclic buffer
  TIMECODETYPE_W buffer;
public:
  ///Open with specific hanning/sample rate and channels
  WSOLAStream(int hl=HANNING_LENGTH(SAMPLE_FREQUENCY), int sf=SAMPLE_FREQUENCY, int ch=1) : WSOLA(hl, sf, ch) {
    //init the buffer
    buffer.init(0,2*(MAX_TAU*hl+hl/2)); ///the buffer is more then 2 times max
    buffer.window=2*(MAX_TAU*hl+hl/2); ///This actually forces buffer malloc
  }

  ///Returns the number of bytes to read from file
  int getByteCount(void){return buffer.getEnd()*sizeof(ATYPE)/2;}

  /** Loads the data to the buffer.
      you supply the audioData and if you can't read a full half
      buffer, then specify the byte count read ... this must mean that
      there is a disk error OR you have reached the end of the audio stream.
      See getByteCount() to work out how many bytes to read.
  */
  int loadData(ATYPE *audioData, int cnt=getByteCount()){
    if (buffer>=getByteCount())//In last half, so load the first half
      memcpy(&buffer.window[0],audioData,getByteCount());
    else //In the first half so load the last half
      memcpy(&buffer.window[getByteCount()],audioData,getByteCount());
  }

  /** nextBlock will process the next frame and return 1 if a new
      load is required by the buffer. (use loadData to load it)
      
  */
  int nextBlock(double tau){
    //Try to find half frame transitions
    //remember the location of the current audio data pointer
    int thisLoc=buffer.getCount();

    //process WSOLA one frame

    //shift to the file location
    int smallesLoc;
    if (getDesiredIndex()>getSourceIndex()){
      smallestLoc=getSourceIndex();
    } else {
      smallestLoc=getDesiredIndex();
    }

    //Check for transition to the second half of the buffer
    if (smallestLoc>=getByteCount() && thisLoc<getByteCount())
      return 1;
    //Check for transition to the first half of the buffer
    if (smallestLoc<getByteCount() && thisLoc>=getByteCount())
      return 1;
    
    //If we make it here, then there was no transition
    return 0;
  }
};

#endif //WSOLA_STREAM_H_
