#!/usr/bin/env python
#
# WAJIG - Debian Command Line System Administrator
#
# Copyright (c) Graham.Williams@csiro.au
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

#------------------------------------------------------------------------
#
# Standard python modules
#
#------------------------------------------------------------------------
import getopt
import string
import sys
import re

#------------------------------------------------------------------------
#
# Wajig modules
#
#------------------------------------------------------------------------
import documentation
import commands
import changes
import perform

#------------------------------------------------------------------------
#
# Global Variables
#
#------------------------------------------------------------------------
progname = "wajig"

#------------------------------------------------------------------------
#
# SUPPORT METHODS
#
#------------------------------------------------------------------------

def requires_no_args(command, args):
    if len(args) > 1: 
        print progname + ": Error: " + command +\
	      " requires no further arguments"
	sys.exit(1)

def requires_one_arg(command, args, message):
    if len(args) != 2:
        print progname + ": Error: " + command +\
	      " requires " + message
	sys.exit(1)

def requires_opt_arg(command, args, message):
    if len(args) > 2:
        print progname + ": Error: " + command +\
	      " requires " + message
	sys.exit(1)

def requires_args(command, args, type):
    if len(args) == 1: 
        print progname + ": Error: " + command +\
	      " requires " + type
	sys.exit(1)

#------------------------------------------------------------------------
#
# MAIN PROGRAM
#
#------------------------------------------------------------------------

def main():
    #
    # Process any command line options
    #
    try:
	#
	# Parse the command line arguments
	#
	sopts = "hqtv"
	lopts = ["help", "quiet", "teaching", "verbose=", "version"]
        opts, args = getopt.getopt(sys.argv[1:], sopts, lopts)
    except getopt.error, e:
	#
        # On error print help information and exit:
	#
	print e
        documentation.usage()
        sys.exit(2)
    #
    # Initialise variables
    #
    teaching = 0
    verbose  = 0
    #
    # Action the command line options
    #
    for o, a in opts:
        if o in ("-h", "--help"):
            documentation.usage()
            sys.exit()
        elif o in ("-q", "--quiet"):
	    commands.set_quiet()
        elif o in ("-t", "--teaching"):
	    teaching = teaching + 1
            perform.set_teaching_level(teaching)
        elif o == "-v":
	    verbose = verbose + 1 
            commands.set_verbosity_level(verbose)
        elif o == "--verbose":
            verbose = string.atoi(a)
            commands.set_verbosity_level(verbose)
        elif o == "--version":
	    documentation.version()
 	    sys.exit()

    #
    # If no command then print usage
    #
    if len(args) == 0:
      documentation.usage()
      sys.exit()

    #
    # Process the command. Lowercase it so that we allow any case
    # for commands and allow hyphens and underscores.
    #
    command = re.sub('-|_', '', string.lower(args[0]))
    if   command == "doc":
        requires_no_args(command, args)
	verbose=2
	documentation.help(verbose)
	sys.exit(0)
    elif command == "help":
        requires_no_args(command, args)
	documentation.help(verbose)
	sys.exit(0)

    #
    # Before we do any other command make sure the right files exist.
    #
    changes.ensure_initialised()

    if   command == "autoclean":
	requires_no_args(command, args)
        commands.do_command("apt-get autoclean",root=1)

    elif command == "bug":
        requires_one_arg(command, args, "a single named package")
        commands.do_command("reportbug " + args[1])

    elif command == "build":
        requires_args(command, args, "a list of package names")
	commands.do_command("apt-get source -b " + perform.concat(args[1:]),
			     root=1)

    elif command == "clean":
	requires_no_args(command, args)
        commands.do_command("apt-get clean",root=1)

    elif command == "dailyupgrade":
        requires_no_args(command, args)
        commands.do_update()
        commands.do_command("apt-get -u dist-upgrade",root=1,update=1)

    elif command == "describe":
        requires_args(command, args, "a list of packages")
	commands.do_describe(args[1:])

    elif command == "describenew" or command == "newdescribe":
        requires_no_args(command, args)
	commands.do_describe_new()

    elif command == "detail":
        requires_args(command, args, "a list of packages")
        verbose = 2
        commands.set_verbosity_level(verbose)
	commands.do_describe(args[1:])

    elif command == "detailnew" or command == "newdetail":
        requires_no_args(command, args)
        verbose = 2
        commands.set_verbosity_level(verbose)
	commands.do_describe_new()

    elif command == "distupgrade":
        requires_no_args(command, args)
        commands.do_command("apt-get -u dist-upgrade", root=1, update=1)

    elif command == "download":
        requires_args(command, args, "a list of packages")
	commands.do_command("apt-get --download-only install " \
                            + perform.concat(args[1:]),
                            root=1, update=1)

    elif command == "edit":
	requires_one_arg(command, args, "file to edit")
        commands.do_command("sensible-editor " + args[1], root=1)

    elif command == "editsources":
	requires_no_args(command, args)
        commands.do_command("sensible-editor /etc/apt/sources.list",root=1)

    elif command == "filedownload":
        requires_one_arg(command, args, "a file name containing list of packages")
	commands.do_command("apt-get --download-only install $(cat " + args[1] + ")",
                            root=1,update=1)

    elif command == "fileinstall":
        requires_one_arg(command, args, "a file name containing list of packages")
        commands.do_command("apt-get install $(cat " + args[1] + ")",root=1,update=1)

    elif command == "findfile":
        requires_one_arg(command, args, "a file name")
        commands.do_command("dpkg --search " + args[1])

    elif command == "force":
        requires_args(command, args, "a package name")
        commands.do_force(args[1:])

    elif command == "geturl":
        requires_one_arg(command, args, "a package name")
        # Not yet quite working
	commands.do_command("/usr/lib/apt-move/fetch -t " + args[1], root=1)

    elif command == "hold":
        requires_args(command, args, "a list of packages to place on hold")
	commands.do_hold(args[1:])
	# Perhaps I can use map to do_command over each package TODO

    elif command == "init":
        requires_no_args(command, args)
        changes.reset_files()

    elif command == "install":
        requires_args(command, args, "a list of packages, .deb files, or url")
	commands.do_install(args[1:])

    elif command == "integrity":
        requires_no_args(command, args)
	commands.do_command("debsums -s -a")

    elif command == "lastupdate":
        requires_no_args(command, args)
	commands.do_command("/bin/ls -l --full-time " +\
			    changes.available_file +\
			    "|awk '{printf \"Last update was " +\
		            "%s %s %s %s %s\\n\"" +\
			    ", $6, $8, $7, $10, $9}'")

    elif command == "list":
        requires_no_args(command, args)
	commands.do_command("dpkg --list '*'")

    elif command == "listalts":
        requires_no_args(command, args)
	commands.do_command("ls /etc/alternatives/ | " +\
			    "egrep -v '(\.1|\.1\.gz)$'")

    elif command == "listfiles":
        requires_one_arg(command, args, "the name of a single Debian package")
	commands.do_command("dpkg --listfiles " + args[1])

    elif command == "listinstalled":
	requires_opt_arg(command, args, "at most one argument")
        commands.do_listinstalled(args[1:])
        
    elif command == "listnames":
	requires_opt_arg(command, args, "at most one argument")
	commands.do_listnames(args[1:])

    elif command == "move":
        requires_no_args(command, args)
	commands.do_command("apt-move update", root=1)

    elif command == "new":
        requires_no_args(command, args)
	commands.do_describe_new()

    elif command == "newupgrades":
        requires_no_args(command, args)
	commands.do_newupgrades()

    elif command == "orphans":
        requires_no_args(command, args)
	commands.do_command("deborphan")

    elif command == "policy":
        requires_args(command, args, "a list of packages")
	commands.do_command("apt-cache policy " + perform.concat(args[1:]))

    elif command == "purge":
        requires_args(command, args, "a list of packages")
	commands.do_command("apt-get --purge remove " + perform.concat(args[1:]),
			     root=1, update=1)

    elif command == "reconfigure":
        requires_args(command, args, "a list of packages")
	commands.do_command("dpkg-reconfigure " + perform.concat(args[1:]), 
			     root=1)

    elif command == "reinstall":
        requires_args(command, args, "a list of packages")
	commands.do_command("apt-get --reinstall install " +\
		   	     perform.concat(args[1:]), root=1)

    elif command == "reload":
        requires_one_arg(command, args, "name of service to " + command)
	commands.do_command("/etc/init.d/" + args[1] + " " + command, root=1)

    elif command == "remove":
        requires_args(command, args, "a list of packages")
	commands.do_command("apt-get remove " + perform.concat(args[1:]), 
			     root=1, update=1)

    elif command == "repackage":
        requires_one_arg(command, args, "name of an installed package")
	commands.do_command("dpkg-repack " + args[1])

    elif command == "reset":
        requires_no_args(command, args)
	changes.reset_files()

    elif command == "restart":
        requires_one_arg(command, args, "name of service to " + command)
	commands.do_command("/etc/init.d/" + args[1] + " " + command, root=1)

    elif command == "rpminstall":
        requires_one_arg(command, args, "a Red Hat package file name (.rpm)")
	commands.do_command("alien --to-deb --install " + args[1], root=1)
        changes.update_installed()
        
    elif command == "rpmtodeb" or command == "rpm2deb":
        requires_one_arg(command, args, "a Red Hat package file name (.rpm)")
	commands.do_command("alien -d " + args[1], root=1)

    elif command == "search":
        requires_args(command, args, "a list of words to search for")
	commands.do_command("apt-cache search " + perform.concat(args[1:]), 
			     root=1)

    elif command == "source":
	# TODO Determine what needs super user access and only do that as su
        requires_args(command, args, "a list of package names")
	commands.do_command("apt-get source " + perform.concat(args[1:]), root=1)

    elif command == "start":
        requires_one_arg(command, args, "name of service to " + command)
	commands.do_command("/etc/init.d/" + args[1] + " " + command, root=1)

    elif command == "status":
	commands.do_status(args[1:])

    elif command == "stop":
        requires_one_arg(command, args, "name of service to " + command)
	commands.do_command("/etc/init.d/" + args[1] + " " + command, root=1)

    elif command == "toupgrade":
        requires_no_args(command, args)
        commands.do_toupgrade()

    elif command == "update":
        requires_no_args(command, args)
        commands.do_update()

    elif command == "updatealts":
        requires_one_arg(command, args, "name of alternative to update")
        commands.do_command("update-alternatives --config " + args[1], root=1)

    elif command == "upgrade":
	if len(args) > 1:
	    commands.do_command("apt-get install " + perform.concat(args[1:]),
				root=1)
        else:
            commands.do_command("apt-get -u upgrade", root=1)

    elif command == "version":
        documentation.version()

    elif command == "whatis":
        requires_args(command, args, "a list of package names")
	commands.do_describe(args[1:])

    elif command == "whichpkg":
        requires_one_arg(command, args, "a file name (possibly with a path)")
	commands.do_whichpkg(args[1])

    else:
      print "The command `" + command + "' (" + args[0] + \
            ") was not recognised."
      print "Perhaps it is not yet implemented."
      print "Try `wajig help' for further information."
      print "Try `wajig -v help' for list of all commands."

#------------------------------------------------------------------------
#
# Start it all
#
#------------------------------------------------------------------------
if __name__ == '__main__':
    main()

