// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


//
// gtkviewer.h - wrapper class for the GTK GUI.
//

#ifndef _GTK_VIEWER_H_
#define _GTK_VIEWER_H_

#include <gtk/gtk.h>
#include <cstring>
#include <cstdio>
#include <cmath>
#include <cctype>
#include <cstdlib>
using namespace std;

#include "../classes/viewer.h"    // class prototype
#include "../classes/greek.h"     // for Greek letters
#include "../classes/specclass.h" // for colors

class GTKViewer : public StarViewer {

 private:
  GdkPixmap *pm;
  GdkGC     *gc; /* the `graphics context' has fg and bg colors, etc. */
  gint filled;
  unsigned long drawing_color;
  
  static GdkColor GTKColorVal(unsigned long color) 
    {
      GdkColor c = { 
	gdk_rgb_xpixel_from_rgb(color),
	256 * RED(color),
	256 * GREEN(color),
	256 * BLUE(color) 
      };
      return c;
    }

  // These functions are for drawing Greek letters:
  void draw_greek_letter(int x, int y, unsigned int letter)
    {
      for (int yy = 0; yy < 10; yy++)
	for (unsigned int xx = 0; xx < Greek[letter].pixelwidth
	       && xx < strlen(Greek[letter].pixellines[yy]); xx++)
	  if (! isspace(Greek[letter].pixellines[yy][xx]))
	    gdk_draw_point (pm, gc, x + xx, y + yy - 8);
      return;
    }

  void draw_text_with_greek(GdkFont *font, int x, int y, char *text,
		  	    unsigned int pointsize)
    {
      unsigned int i = 1;
      while (i < NUM_GREEK_LETTERS 
	     && strncasecmp(text, Greek[i].name, Greek[i].namelen) != 0)
	i++;
      if (i < NUM_GREEK_LETTERS) {
	draw_greek_letter(x, y, i);
	text += Greek[i].namelen;
	x += (Greek[i].pixelwidth + 2);
	
	// now check for superscript
	if (text[0] == '(' && isdigit(text[1]) && text[2] == ')') {
	  text++;
	  text[1] = 0;
	  gdk_draw_text (pm, font, gc, x, y - (pointsize/2), text, 1);
	  x += 5;
	  text += 2;
	}
      }
      gdk_draw_text (pm, font, gc, x, y, text, strlen(text));
      return;
    }


 public:
  GTKViewer(GdkPixmap *PM) 
    { 
      pm = PM; gc = gdk_gc_new (pm); filled = TRUE; 
      gdk_rgb_init();
    }

  ~GTKViewer() { gdk_gc_unref(gc); }

  unsigned int width() const
    { 
      gint width, height;
      gdk_window_get_size(pm, &width, &height);
      return width;
    }

  unsigned int height() const 
    {
      gint width, height;
      gdk_window_get_size(pm, &width, &height);
      return height;
    }

  void setcolor(unsigned long color) 
    { 
      GdkColor g = GTKColorVal(color);
      gdk_gc_set_foreground(gc, &g);
      gdk_gc_set_background(gc, &g);
      drawing_color = color;
    }

  void setfill(bool fillmode)
    { filled = fillmode; }

  void fill(unsigned long color)
    {
      setcolor(color);
      gdk_draw_rectangle (pm, gc, /*filled*/ TRUE,
			  0, 0, width(), height());
    }

  void drawline(unsigned int x1, unsigned int y1,
		unsigned int x2, unsigned int y2)
    { gdk_draw_line (pm, gc, x1, y1, x2, y2); }

  void drawbox(unsigned int xul, unsigned int yul,
	       unsigned int width, unsigned int height)
    { gdk_draw_rectangle (pm, gc, filled, xul, yul, width, height); }

  /* GDK uses units of (1/64) degree for angles.  God only knows why. */

  /* gdk_draw_arc has the neat design feature that if you pass it negative
     dimensions, it makes a hyperbola instead of an ellipse.  However, I don't
     think the user would expect a "drawellipse" function to produce a
     hyperbola, so I've used abs() to ensure that all of gdk_draw_arc's args
     are positive.
  */

  void drawcircle(unsigned int xc, unsigned int yc, int radius)
    { gdk_draw_arc (pm, gc, filled, xc - abs(radius), yc - abs(radius),
		    abs(radius*2), abs(radius*2),
		    /*angle1*/ 0, /*angle2*/ 360*64); }

  void drawstar(unsigned int xc, unsigned int yc, int radius)
    {
      // GDK's small circles are very ugly, hence this function will make
      // its own for stars with radius < 6 pixels.
      if (radius <= 0) return;
      else {
	static GdkGC *border_gc = gdk_gc_new (pm);
	GdkColor g = GTKColorVal((drawing_color & 0xFEFEFE) / 2);
	gdk_gc_set_foreground(border_gc, &g);
	gdk_gc_set_background(border_gc, &g);

	switch (radius) {
	  case 1:
	    gdk_draw_rectangle (pm, border_gc, TRUE, xc - 1, yc - 1, 3, 3);
	    gdk_draw_line (pm, gc, xc, yc - 1, xc, yc + 1);
	    gdk_draw_line (pm, gc, xc - 1, yc, xc + 1, yc);
	    break;
	  case 2:
	    gdk_draw_rectangle (pm, border_gc, TRUE, xc - 2, yc - 1, 5, 3);
	    gdk_draw_rectangle (pm, border_gc, TRUE, xc - 1, yc - 2, 3, 5);
	    gdk_draw_rectangle (pm, gc, TRUE, xc - 1, yc - 1, 3, 3);
	    break;
	  case 3:
	    gdk_draw_rectangle (pm, border_gc, TRUE, xc - 3, yc - 1, 7, 3);
	    gdk_draw_rectangle (pm, border_gc, TRUE, xc - 1, yc - 3, 3, 7);
	    gdk_draw_rectangle (pm, border_gc, TRUE, xc - 2, yc - 2, 5, 5);
	    gdk_draw_rectangle (pm, gc, TRUE, xc - 2, yc - 1, 5, 3);
	    gdk_draw_rectangle (pm, gc, TRUE, xc - 1, yc - 2, 3, 5);
	    break;
	  default:
	    gdk_draw_arc (pm, gc, TRUE, xc - radius, yc - radius,
			  radius * 2, radius * 2,
			  /*angle1*/ 0, /*angle2*/ 360*64);
    	    gdk_draw_arc (pm, border_gc, FALSE, xc - radius, yc - radius,
		          radius*2, radius*2,
			  /*angle1*/ 0, /*angle2*/ 360*64);
        }
      }
    }

  void drawellipse(unsigned int xc, unsigned int yc,
		   int semiwidth, int semiheight)
    { gdk_draw_arc (pm, gc, filled, xc - abs(semiwidth), yc - abs(semiheight),
		    abs(semiwidth * 2), abs(semiheight * 2), 0, 360 * 64); }

  void drawarc(unsigned int xc, unsigned int yc, // arc center
	       int semiwidth, int semiheight,    
	       double startangle, double endangle) // angles in radians
    { gdk_draw_arc (pm, gc, FALSE, xc - abs(semiwidth), yc - abs(semiheight),
		    abs(semiwidth * 2), abs(semiheight * 2),
		    (int)(startangle * 64 / M_PI_180),
		    (int)((endangle - startangle) * 64 / M_PI_180)); }

  void drawtext(const char *text, unsigned int x, unsigned int y,
		unsigned int pointsize)
    { 
      char fontbegin[] = "-adobe-helvetica-medium-r-normal--";
      char fontend[] = "-100-*-*-*-*-*-*";
      char fontname[200];
      snprintf(fontname, 200, "%s%d%s", fontbegin, pointsize, fontend);
      fontname[199] = 0;
      GdkFont *font = gdk_font_load(fontname);
      
      char *textbuf = new char[strlen(text) + 1];
      strcpy(textbuf, text);
      
      draw_text_with_greek (font, x, y, textbuf, pointsize);
      delete [] textbuf;
    }

  void drawtext(char text, unsigned int x, unsigned int y,
		unsigned int pointsize)
    {
      char textstring[2];
      textstring[0] = text; textstring[1] = 0;
      drawtext(textstring, x, y, pointsize);
    }

} ;

#endif // _GTK_VIEWER_H_
