/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2012 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.texture;

import com.jogamp.opengl.util.texture.TextureData;
import org.scilab.forge.scirenderer.SciRendererException;
import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLCanvas;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLDrawingTools;
import org.scilab.forge.scirenderer.texture.AbstractTexture;
import org.scilab.forge.scirenderer.texture.AnchorPosition;
import org.scilab.forge.scirenderer.texture.Texture;
import org.scilab.forge.scirenderer.texture.TextureManager;
import org.scilab.forge.scirenderer.tranformations.Transformation;
import org.scilab.forge.scirenderer.tranformations.TransformationFactory;
import org.scilab.forge.scirenderer.tranformations.TransformationManager;
import org.scilab.forge.scirenderer.tranformations.Vector3d;

import javax.media.opengl.GL2;
import java.awt.Dimension;
import java.nio.FloatBuffer;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * @author Pierre Lando
 */
public class JoGLTextureManager implements TextureManager {
    private final Set<JoGLTexture> allTextures = new HashSet<JoGLTexture>();

    public JoGLTextureManager(JoGLCanvas canvas) {
    }

    /**
     * Texture binder.
     * Bind the given texture to the given OpenGl context.
     * @param drawingTools drawing tools.
     * @param texture given texture.
     * @throws org.scilab.forge.scirenderer.SciRendererException if the texture can't be bind.
     */
    public void bind(JoGLDrawingTools drawingTools, Texture texture) throws SciRendererException {
        if ((texture instanceof JoGLTexture) && (allTextures.contains((JoGLTexture) texture))) {
            ((JoGLTexture) texture).bind(drawingTools);
        }
    }

    /**
     * Draw the given texture.
     * @param drawingTools used drawing tools.
     * @param texture the texture too drawn.
     * @throws org.scilab.forge.scirenderer.SciRendererException if the texture is invalid.
     */
    public void draw(JoGLDrawingTools drawingTools, Texture texture) throws SciRendererException {
        if ((texture instanceof JoGLTexture) && (allTextures.contains((JoGLTexture) texture))) {
            ((JoGLTexture) texture).draw(drawingTools);
        }
    }

    public void draw(JoGLDrawingTools drawingTools, Texture texture, AnchorPosition anchor, ElementsBuffer positions, double rotationAngle) throws SciRendererException {
        if ((texture instanceof JoGLTexture) && (allTextures.contains((JoGLTexture) texture))) {
            if (positions != null) {
                FloatBuffer data = positions.getData();
                if (data != null) {
                    data.rewind();
                    float[] position = {0, 0, 0, 1};
                    while (data.remaining() >= 4) {
                        data.get(position);
                        ((JoGLTexture) texture).draw(drawingTools, anchor, new Vector3d(position), rotationAngle);
                    }
                }
            }
        }
    }

    public void draw(JoGLDrawingTools drawingTools, Texture texture, AnchorPosition anchor, Vector3d position, double rotationAngle) throws SciRendererException {
        if ((texture instanceof JoGLTexture) && (allTextures.contains((JoGLTexture) texture))) {
            ((JoGLTexture) texture).draw(drawingTools, anchor, position, rotationAngle);
        }
    }

    /** Called when gl context is gone. */
    public void glReload() {
        for (JoGLTexture texture : allTextures) {
            texture.glReload();
        }
    }

    @Override
    public Texture createTexture() {
        JoGLTexture texture = new JoGLTexture();
        allTextures.add(texture);
        return texture;
    }

    @Override
    public void dispose(Collection<Texture> textures) {
        for (Texture texture : textures) {
            dispose(texture);
        }
    }

    @Override
    public void dispose(Texture texture) {
        if ((texture instanceof JoGLTexture) && (allTextures.contains((JoGLTexture) texture))) {
            //((JoGLTexture) texture).dispose(); TODO.
        }
    }

    /**
     * Inner class for {@link Texture} implementation.
     */
    private class JoGLTexture extends AbstractTexture implements Texture {
        private com.jogamp.opengl.util.texture.Texture[] textures;
        private int wCuts;
        private int hCuts;

        /**
         * Default constructor.
         */
        public JoGLTexture() {
        }

        /**
         * Bind the texture in the OpenGl context.
         * @param drawingTools current drawing tools.
         * @throws SciRendererException if the texture is invalid.
         */
        public void bind(JoGLDrawingTools drawingTools) throws SciRendererException {
            GL2 gl = drawingTools.getGl().getGL2();
            if (isValid()) {
                checkData(drawingTools);
                if (textures.length == 1) {
                    gl.glEnable(GL2.GL_TEXTURE_2D);
                    textures[0].setTexParameteri(gl, GL2.GL_TEXTURE_MAG_FILTER, getAsGLFilter(getMagnificationFilter(), false));
                    textures[0].setTexParameteri(gl, GL2.GL_TEXTURE_MIN_FILTER, getAsGLFilter(getMinifyingFilter(), true));
                    textures[0].setTexParameteri(gl, GL2.GL_TEXTURE_WRAP_S, getAsGLWrappingMode(getSWrappingMode()));
                    textures[0].setTexParameteri(gl, GL2.GL_TEXTURE_WRAP_T, getAsGLWrappingMode(getTWrappingMode()));
                    gl.glTexEnvi(GL2.GL_TEXTURE_ENV, GL2.GL_TEXTURE_ENV_MODE, GL2.GL_REPLACE);
                    textures[0].bind(gl);
                } else {
                    throw new SciRendererException("Texture is too large");
                }
            } else {
                throw new SciRendererException("Texture have no data.");
            }
        }

        /**
         * Check if the texture data are up to date.
         * @param drawingTools the drawing tools.
         * @throws SciRendererException if the texture is too big.
         */
        private void checkData(JoGLDrawingTools drawingTools) throws SciRendererException {
            if (isValid() && !upToDate) {
                upToDate = true;

		GL2 gl = drawingTools.getGl().getGL2();
                releaseTextures(gl);

                Dimension textureSize = getDataProvider().getTextureSize();
                int maxSize = drawingTools.getGLCapacity().getMaximumTextureSize();
                wCuts = (int) Math.ceil(textureSize.getWidth() / maxSize);
                hCuts = (int) Math.ceil(textureSize.getHeight() / maxSize);

                textures = new com.jogamp.opengl.util.texture.Texture[wCuts * hCuts];
                int k = 0;
                for (int i = 0; i < wCuts; i++) {
                    for (int j = 0; j < hCuts; j++) {
                        textures[k] = com.jogamp.opengl.util.texture.TextureIO.newTexture(GL2.GL_TEXTURE_2D);
                        int x = i * maxSize;
                        int y = j * maxSize;
                        int width = getSubTextureSize((i == (wCuts - 1)), textureSize.width, maxSize);
                        int height = getSubTextureSize((j == (hCuts - 1)), textureSize.height, maxSize);

                        TextureData data = new TextureData(
			        gl.getGLProfile(),
                                GL2.GL_RGBA,
                                width,
                                height,
                                0, // no border
                                GL2.GL_RGBA, // pixel format.
                                GL2.GL_UNSIGNED_BYTE, // pixel type.
                                true, // mipmap
                                false, // no compression
                                false, // no vertical flip needed TODO as an option from data provider.
                                getDataProvider().getSubData(x, y, width, height),
                                null   // no flusher
                        );
                        textures[k].updateImage(gl, data);
                        k++;
                    }
                }
            }
        }

        private void releaseTextures(GL2 gl) {
            if (textures != null) {
                for (com.jogamp.opengl.util.texture.Texture texture : textures) {
                    texture.destroy(gl);
                }
                textures = null;
            }
        }

        public void draw(JoGLDrawingTools drawingTools, AnchorPosition anchor, Vector3d position, double rotationAngle) throws SciRendererException {
            TransformationManager transformationManager = drawingTools.getTransformationManager();
            Transformation canvasProjection = transformationManager.getCanvasProjection();

            boolean sceneCoordinate = drawingTools.getTransformationManager().isUsingSceneCoordinate();
            if (!sceneCoordinate) {
                drawingTools.getTransformationManager().useSceneCoordinate();
            }

            Vector3d projected;
            if (sceneCoordinate) {
                projected = canvasProjection.project(position);
            } else {
                projected = position;
            }

            Dimension dimension = drawingTools.getCanvas().getDimension();

            Vector3d s = new Vector3d(2 / dimension.getWidth(), 2 / dimension.getHeight(), 1);
            Vector3d t = new Vector3d(
                    (projected.getX() * 2) / dimension.getWidth() - 1,
                    (projected.getY() * 2) / dimension.getHeight() - 1,
                    projected.getZ()
            );

            Transformation translation = TransformationFactory.getTranslateTransformation(getAnchorDeltaX(anchor), getAnchorDeltaY(anchor), 0);

            transformationManager.getProjectionStack().push(TransformationFactory.getIdentity());
            transformationManager.getModelViewStack().push(TransformationFactory.getAffineTransformation(s, t));
            transformationManager.getModelViewStack().pushRightMultiply(TransformationFactory.getRotationTransformation(-rotationAngle, 0, 0, 1));
            transformationManager.getModelViewStack().pushRightMultiply(translation);
            draw(drawingTools);
            transformationManager.getModelViewStack().pop();
            transformationManager.getModelViewStack().pop();
            transformationManager.getModelViewStack().pop();
            transformationManager.getProjectionStack().pop();

            if (!sceneCoordinate) {
                drawingTools.getTransformationManager().useWindowCoordinate();
            }
        }

        /**
         * Draw the texture in XY plane.
         * @param drawingTools the drawing tools.
         * @throws SciRendererException if the texture is invalid.
         */
        public void draw(JoGLDrawingTools drawingTools) throws SciRendererException {
            checkData(drawingTools);
            final int maxSize = drawingTools.getGLCapacity().getMaximumTextureSize();
            final Dimension textureSize = getDataProvider().getTextureSize();
            final GL2 gl = drawingTools.getGl().getGL2();
            gl.glEnable(GL2.GL_TEXTURE_2D);

            gl.glEnable(GL2.GL_BLEND);
            gl.glBlendFunc(GL2.GL_SRC_ALPHA, GL2.GL_ONE_MINUS_SRC_ALPHA);

            gl.glEnable(GL2.GL_ALPHA_TEST);
            gl.glAlphaFunc(GL2.GL_GREATER, 0.0f);

            int k = 0;
            for (int i = 0; i < wCuts; i++) {
                for (int j = 0; j < hCuts; j++) {
                    int x = i * maxSize;
                    int y = j * maxSize;
                    int width = getSubTextureSize((i == (wCuts - 1)), textureSize.width, maxSize);
                    int height = getSubTextureSize((j == (hCuts - 1)), textureSize.height, maxSize);

                    textures[k].setTexParameteri(gl, GL2.GL_TEXTURE_MAG_FILTER, getAsGLFilter(getMagnificationFilter(), false));
                    textures[k].setTexParameteri(gl, GL2.GL_TEXTURE_MIN_FILTER, getAsGLFilter(getMinifyingFilter(), true));
                    textures[k].setTexParameteri(gl, GL2.GL_TEXTURE_WRAP_S, getAsGLWrappingMode(getSWrappingMode()));
                    textures[k].setTexParameteri(gl, GL2.GL_TEXTURE_WRAP_T, getAsGLWrappingMode(getTWrappingMode()));
                    gl.glTexEnvi(GL2.GL_TEXTURE_ENV, GL2.GL_TEXTURE_ENV_MODE, GL2.GL_REPLACE);
                    textures[k].bind(gl);

                    gl.glBegin(GL2.GL_QUADS);
                    gl.glTexCoord2d(0, 1);
                    gl.glVertex2f(x, y);

                    gl.glTexCoord2d(1, 1);
                    gl.glVertex2f(x + width, y);

                    gl.glTexCoord2d(1, 0);
                    gl.glVertex2f(x + width, y + height);

                    gl.glTexCoord2d(0, 0);
                    gl.glVertex2f(x, y + height);
                    gl.glEnd();

                    k++;
                }
            }
            gl.glDisable(GL2.GL_ALPHA_TEST);
            gl.glDisable(GL2.GL_TEXTURE_2D);
            gl.glDisable(GL2.GL_BLEND);
        }

        /**
         * Compute the sub texture size.
         * @param lastPart true if this is the last part.
         * @param textureSize the global texture size.
         * @param maxSize the maximum sub-texture size.
         * @return the sub texture size.
         */
        private int getSubTextureSize(boolean lastPart, int textureSize, int maxSize) {
            if (lastPart) {
                int lastSize = textureSize % maxSize;
                if (lastSize == 0) {
                    return maxSize;
                } else {
                    return lastSize;
                }
            } else {
                return maxSize;
            }
        }

        private int getAsGLWrappingMode(Texture.Wrap wrappingMode) {
            switch (wrappingMode) {
                case CLAMP:
                    return GL2.GL_CLAMP;
                case REPEAT:
                    return GL2.GL_REPEAT;
                default:
                    return GL2.GL_REPEAT;
            }
        }

        private int getAsGLFilter(Texture.Filter filter, boolean mipmap) {
            int returnedValue;
            if (mipmap) {
                switch (filter) {
                    case LINEAR:
                        returnedValue = GL2.GL_LINEAR_MIPMAP_LINEAR;
                        break;
                    case NEAREST:
                        returnedValue = GL2.GL_NEAREST_MIPMAP_NEAREST;
                        break;
                    default:
                        returnedValue = GL2.GL_NEAREST;
                        break;
                }
            } else {
                switch (filter) {
                    case LINEAR:
                        returnedValue = GL2.GL_LINEAR;
                        break;
                    case NEAREST:
                        returnedValue = GL2.GL_NEAREST;
                        break;
                    default:
                        returnedValue = GL2.GL_NEAREST;
                        break;
                }
            }
            return returnedValue;
        }

        /** Called when gl context is gone. */
        public void glReload() {
            textures = null;
            upToDate = false;
        }

        /**
         * Return the deltaX to apply to the sprite in regards to the given anchor.
         * @param anchor the given anchor.
         * @return the deltaX to apply to the sprite in regards to the given anchor.
         */
        protected float getAnchorDeltaX(AnchorPosition anchor) {
            int spriteWidth = getDataProvider().getTextureSize().width;
            switch (anchor) {
                case LEFT:
                case LOWER_LEFT:
                case UPPER_LEFT:
                    return 0;
                case UP:
                case CENTER:
                case DOWN:
                    return -spriteWidth / 2f;
                case RIGHT:
                case LOWER_RIGHT:
                case UPPER_RIGHT:
                    return -spriteWidth;
                default:
                    return 0;
            }
        }

        /**
         * Return the deltaY to apply to the sprite in regards to the given anchor.
         * @param anchor the given anchor.
         * @return the deltaY to apply to the sprite in regards to the given anchor.
         */
        protected float getAnchorDeltaY(AnchorPosition anchor) {
            int spriteHeight = getDataProvider().getTextureSize().height;
            switch (anchor) {
                case UPPER_LEFT:
                case UP:
                case UPPER_RIGHT:
                    return -spriteHeight;
                case LEFT:
                case CENTER:
                case RIGHT:
                    return -spriteHeight / 2f;
                case LOWER_LEFT:
                case DOWN:
                case LOWER_RIGHT:
                    return 0;
                default:
                    return 0;
            }
        }
    }
}
