
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#ifndef __RBHASH_H__
#define __RBHASH_H__

/* Documentation!
 *
 * In order to use this hash table implementation, you must create
 * your own structure to be used as an entry in the hash table.
 * This structure *MUST* start with the first entry as a
 * 'RBHASH_HEADER;', and the second entry MUST be RBuf *key.
 *
 * You may use the supplied RBHashEntry for a generic data item
 * (void *), or define your own entry structure, so long as it
 * starts with RBHASH_HEADER, and has a RBuf *key as the 
 * second entry.
 *
 * For Example:
 *
 * struct _MyHashEntry {
 *	    RBHASH_HEADER;
 *	    RBuf *key;
 *	    RBuf *data;
 *	    ...
 * }
 *
 * You can create a toplevel RBHash structure by using rbhash_new,
 * or you can init an already allocated RBHash struct using
 * RBHASH_INIT(hash).
 *
 * In all cases, the entire hash entry is passed around where applicable.
 * This lets you do whatever you want in terms of the
 * data stored in the struct etc.
 */


typedef struct _RBHashHeader	RBHashHeader;
typedef struct _RBHashEntry	RBHashEntry;
typedef struct _RBHash		RBHash;

struct _RBHashHeader {
    RBHashEntry **prev_next_ptr; /* >: ) */
    RBHashEntry *next;
    unsigned long hashval;
    RBuf *key;
};
#define RBHASH_HEADER RBHashHeader rbhash_header

struct _RBHashEntry {
    RBHASH_HEADER;
    void *data;
};

struct _RBHash {
    RBHashEntry **entries;
    unsigned int num_entries;
    unsigned int table_size;
};

#define rbhash_entry_getkey(hash_entry) \
    (hash_entry)->rbhash_header.key

#define rbhash_entry_setkey(hash_entry,rbhash_entry_key) \
    (hash_entry)->rbhash_header.key = rbhash_entry_key


/* Initialize a newly created list structure */
#define RBHASH_INIT(hash) \
    (hash)->num_entries = 0; \
    (hash)->table_size = 0; \
    (hash)->entries = NULL

/* Cleans out a hash struct.  This frees the entries table,
 * but does not remove the entries from the hash table itself.  I
 * recommend doing that with a RBHASH_FOREACH and rbhash_remove_entry,
 * freeing the appropriate items in the structure, and the
 * structure itself, then call this to clean up the Hash. (if you
 * didn't use rbhash_new() that is. */
#define RBHASH_DESTRUCT(hash) \
    if ((hash)->entries) \
        rmem_free ((hash)->entries); \
    (hash)->entries = NULL; \
    (hash)->num_entries = 0; \
    (hash)->table_size = 0


/* This will foreach through all the entries in the hash table, assigning each
 * one to the 'variable' argument.  Unfortunately, we require the extra 'counter'
 * argument, which should be an integer for use in the for loop */
#define RBHASH_FOREACH(hash, variable) \
do { \
    RBHashEntry *tmp_rbhash_next_entry; \
        unsigned int tmp_rbhash_counter; \
        for (tmp_rbhash_counter = 0; tmp_rbhash_counter < hash->table_size; tmp_rbhash_counter++) \
            for (variable = (void *) hash->entries[tmp_rbhash_counter], \
                tmp_rbhash_next_entry = (void *) variable ? variable->rbhash_header.next : NULL; \
                variable != NULL; \
                variable = (void *) tmp_rbhash_next_entry, \
                tmp_rbhash_next_entry = tmp_rbhash_next_entry ? tmp_rbhash_next_entry->rbhash_header.next : NULL)



/* This just unlinks a hash entry from the hash table, and may be done at any time
 * from anywhere */
#define rbhash_remove_entry(hash, entry) \
    do { \
        RBHash *roy_hash_tmp = (hash); \
        RBHashEntry *roy_hash_entry_tmp = (void *) (entry); \
        \
        if (!roy_hash_entry_tmp) \
            break; \
        if (roy_hash_entry_tmp->rbhash_header.next) { \
            roy_hash_entry_tmp->rbhash_header.next->rbhash_header.prev_next_ptr = \
                                 roy_hash_entry_tmp->rbhash_header.prev_next_ptr; \
        } \
        *roy_hash_entry_tmp->rbhash_header.prev_next_ptr = \
                   roy_hash_entry_tmp->rbhash_header.next; \
        roy_hash_entry_tmp->rbhash_header.next = NULL; \
        roy_hash_entry_tmp->rbhash_header.prev_next_ptr = NULL; \
        roy_hash_tmp->num_entries--; \
    } while (0)
/* (You get all that ?  Do I know how to have fun or what ? :) ) */


/* Create a newly allocated hash table structure */
RBHash *
rbhash_new (void);

/* Free a hash table, as created with rbhash_new.  Again, this does not free
 * the entries in the table. */
void
rbhash_free (RBHash *hash);

/* Lookup an entry given a key.  The return value is a
 * pointer to the entry you provided for insert, or NULL
 * if it was not found */
void *
rbhash_lookup (RBHash *hash, RBuf *key);

/* Insert a hash entry (as described above).  The key should already
 * be set for the entry using rbhash_entry_setkey(). */
void
rbhash_insert (RBHash *hash, void *entry, RBuf *key);

/* Remove an entry from the hash table.  This only
 * unlinks it from the table, and does not free
 * the key, or any data in your entry struct */
void *
rbhash_remove (RBHash *hash, RBuf *key);

/* Return out the size of the rhash. */
#define rbhash_len(hash) (hash->num_entries)


#endif /* __RBHASH_H__ */



