/*
 * rtchar.h
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** \file
  * Contains the class Char, which defines static helper functions to determine 
  * and convert the type of a character.
  */
#ifndef __LRT_CHAR__
#define __LRT_CHAR__

namespace lrt {

/** Defines static helper functions to determine and convert the type of a character.
  * Note that in libRT, every character is supposed to have the Latin-1 charset.
  */
class Char  
{
public:
	/** The maximum radix supported for integer <-> String conversions. (36) */
	static const int MAX_RADIX;

	/** Detects if a character is whitespace. 
	  * Every spacing and control character is defined as whitespace.
	  */
	static inline bool isSpace(char ch);

	/** Detects if a character is an upper-case letter. */
	static inline bool isUpperCase(char ch);

	/** Detects if a character is a lower-case letter. */
	static inline bool isLowerCase(char ch);

	/** Detects if a character is a digit for decimal numbers. */
	static inline bool isDigit(char ch);

	/** Detects if a character is a digit for numbers represented with radix <tt>radix</tt>. */
	static bool isDigit(char ch, int radix);

	/** Detects if a character is a letter.
	  * This function is equivalent to <tt>isLowerCase || isUpperCase</tt>.
	  */
	static inline bool isLetter(char ch);

	/** Returns the lower-case equivalent of a character, if it is an upper-case letter.
	  * If the given character is not uppercase, it is left unchanged. 
	  */
	static inline char lowerCase(char ch);

	/** Returns the upper-case equivalent of a character, if it is a lower-case letter.
	  * If the given character is not lowercase, it is left unchanged. 
	  */
	static inline char upperCase(char ch);

	/** Returns the numeric value of a character.
	  * The returned value is in range 0..MAX_RADIX, if the character is an English letter or digit.
	  * If the character is not numerically representable, results are undefined.
	  */
	static inline int  numericValue(char ch);

	/** Returns the character representing the given number.
	  * Note that the given number is not positive and smaller than MAX_RADIX, results are undefined.
	  */
	static inline char charValue(int num); 

private:
	Char();
	~Char();
};

} // namespace

#include "rtchar.inline.cpp"

#endif

