#!/usr/bin/python

"""
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
"""

__version__ = "$Revision: 1.15 $"
# $Source: /cvsroot/quantlib/QuantLib-Python/QuantLib/test/european_option.py,v $

import QuantLib
from QuantLib import MarketElementHandle, SimpleMarketElement
from QuantLib import TermStructureHandle, FlatForward
from QuantLib import Date, Calendar, DayCounter
from QuantLib import PlainOption, EuropeanEngine

import unittest
from math import exp

def relErr(x1, x2, reference):
    if reference != 0.0:
        return abs(x1-x2)/reference
    else:
        return 10e10

def pricer(type,underlying,strike,divCurve,rfCurve,exDate,volatility):
    return PlainOption(type,underlying,strike,divCurve,rfCurve,
                       exDate,volatility,EuropeanEngine())


class EuropeanOptionTest(unittest.TestCase):
    def runTest(self):
        "Testing European option pricer"

        err_delta = 5e-5
        err_gamma = 5e-5
        err_theta = 6e-3
        err_rho   = 5e-5
        err_divRho= 5e-5
        err_vega  = 5e-5

        for under in [100]:
          dS = under/10000.0
          underlying = MarketElementHandle(SimpleMarketElement(under))
          underlyingP = MarketElementHandle(SimpleMarketElement(under+dS))
          underlyingM = MarketElementHandle(SimpleMarketElement(under-dS))
          for Qrate in [0.04, 0.05, 0.06]:
            dQ = Qrate/10000.0
            divCurve = TermStructureHandle(FlatForward('EUR',
                DayCounter('act/360'), Date(12,10,2001),
                Calendar('TARGET'), 2, Qrate))
            divCurveP = TermStructureHandle(FlatForward('EUR',
                DayCounter('act/360'), Date(12,10,2001),
                Calendar('TARGET'), 2, Qrate+dQ))
            divCurveM = TermStructureHandle(FlatForward('EUR',
                DayCounter('act/360'), Date(12,10,2001),
                Calendar('TARGET'), 2, Qrate-dQ))
            for exDate in [Date(16,10,2002)]:
              for Rrate in [0.01, 0.05, 0.15]:
                dR = Rrate/10000.0
                rfCurve = TermStructureHandle(FlatForward('EUR',
                    DayCounter('act/360'), Date(12,10,2001),
                    Calendar('TARGET'), 2, Rrate))
                rfCurveP = TermStructureHandle(FlatForward('EUR',
                    DayCounter('act/360'), Date(12,10,2001),
                    Calendar('TARGET'), 2, Rrate+dR))
                rfCurveM = TermStructureHandle(FlatForward('EUR',
                    DayCounter('act/360'), Date(12,10,2001),
                    Calendar('TARGET'), 2, Rrate-dR))
                for vol in [0.11, 0.5, 1.2]:
                  dVol = vol/10000.0
                  volatility = MarketElementHandle(SimpleMarketElement(vol))
                  volatilityP = MarketElementHandle(
                    SimpleMarketElement(vol+dVol))
                  volatilityM = MarketElementHandle(
                    SimpleMarketElement(vol-dVol))
                  for strike in [50, 99.5, 100, 100.5, 150]:
                    for type in ['Call','Put','Straddle']:

                      #Check Greeks
                      dT = 1
                      opt = pricer(type,underlying,strike,
                                   divCurve,rfCurve,exDate,volatility)
                      opt_val = opt.NPV()
                      if opt_val>0.00001*under:
                        optPs = pricer(type, underlyingP, strike, divCurve,
                          rfCurve,  exDate ,   volatility)
                        optMs = pricer(type, underlyingM, strike, divCurve,
                          rfCurve,  exDate ,   volatility)
                        optPt = pricer(type, underlying , strike, divCurve,
                          rfCurve,  exDate+dT,  volatility)
                        optMt = pricer(type, underlying , strike, divCurve,
                          rfCurve,  exDate-dT,  volatility)
                        optPr = pricer(type, underlying , strike, divCurve,
                          rfCurveP, exDate   , volatility)
                        optMr = pricer(type, underlying , strike, divCurve,
                          rfCurveM, exDate   , volatility)
                        optPq = pricer(type, underlying , strike, divCurveP,
                          rfCurve,  exDate   , volatility)
                        optMq = pricer(type, underlying , strike, divCurveM,
                          rfCurve,  exDate   , volatility)
                        optPv = pricer(type, underlying , strike, divCurve,
                          rfCurve,  exDate   , volatilityP)
                        optMv = pricer(type, underlying , strike, divCurve,
                          rfCurve,  exDate   , volatilityM)

                        deltaNum = (optPs.NPV()-optMs.NPV())/(2*dS)
                        gammaNum = (optPs.delta()-optMs.delta())/(2*dS)
                        thetaNum =-(optPt.NPV()-optMt.NPV())/(2*dT/365.0)
                        rhoNum   = (optPr.NPV()-optMr.NPV())/(2*dR)
                        divRhoNum= (optPq.NPV()-optMq.NPV())/(2*dQ)
                        vegaNum  = (optPv.NPV()-optMv.NPV())/(2*dVol)

                        delta  = opt.delta()
                        gamma  = opt.gamma()
                        theta  = opt.theta()
                        rho    = opt.rho()
                        divRho = opt.dividendRho()
                        vega   = opt.vega()

                        if not (relErr(delta,deltaNum,under)<=err_delta
                                and relErr(gamma,gammaNum,under)<=err_gamma
                                and relErr(theta,thetaNum,under)<=err_theta
                                and relErr(rho,  rhoNum,  under)<=err_rho
                                and relErr(divRho, divRhoNum, under) <= err_divRho
                                and relErr(vega, vegaNum, under)<=err_vega):
                            self.fail("""
Option details: %(type)s %(under)f %(strike)f %(Qrate)f %(Rrate)f %(exDate)s %(vol)f
    value  = %(opt_val)+9.5f
    delta  = %(delta)+9.5f, deltaNum  = %(deltaNum)+9.5f
    gamma  = %(gamma)+9.5f, gammaNum  = %(gammaNum)+9.5f
    theta  = %(theta)+9.5f, thetaNum  = %(thetaNum)+9.5f
    rho    = %(rho)+9.5f, rhoNum    = %(rhoNum)+9.5f
    divRho = %(divRho)+9.5f, divRhoNum = %(divRhoNum)+9.5f
    vega   = %(vega)+9.5f, vegaNum   = %(vegaNum)+9.5f
	                                  """ % locals())


if __name__ == '__main__':
    print 'testing QuantLib', QuantLib.__version__, QuantLib.QuantLibc.__file__, QuantLib.__file__
    import sys
    suite = unittest.TestSuite()
    suite.addTest(EuropeanOptionTest())
    if sys.hexversion >= 0x020100f0:
        unittest.TextTestRunner(verbosity=2).run(suite)
    else:
        unittest.TextTestRunner().run(suite)

