"""
Configuration library for managing 4Suite RDF Drivers.  Only those known to 
work are registered, the others can be added easily.
The neccessary configuration parameters for each driver are listed below :

MySQL Configuration:
driver: mysql
dbHandle: rdf:userName/password@hostName:port:dbName
model: <modelName>

Postgres Configuration:
driver: postgres
dbHandle: rdf:userName/password@hostName:port:dbName
model: <modelName>

Dbm Configuration:
driver: dbm
dbHandle: <dbName>
model: <modelName>

Memory Configuration:
driver: memory
dbHandle: <dbName>
model: <modelName>

Bsd Db Configuration:
driver: bsddb
model: <modelName>

modelName - The name of the RDF Model
dbName    - The name of the database

The parameters are passed to the Configuration Manager's constructor in the following format:
driver=bsddb|mysql|postgres|dbm|memory,model=modelName,dbHandle=connString|dbName

for example:

driver=mysql,model=system,dbHandle=rdf:root/toor@localhost:3306:ft__xmlserver


Copyright 2005 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

def configureMySQL(modelName,connectString,create):
    """
    Utility function for creating a MySQL driver instance
    """
    try:
        from Ft.Rdf.Drivers import MySQL
        MySQL.InitializeModule()
    except:
        raise Exception("Unable to setup driver: MySQL")
    if create:
        if MySQL.ExistsDb(connectString,modelName) in [1,0]:
            MySQL.DestroyDb(connectString,modelName)
            return MySQL.CreateDb(connectString, modelName)
        else:
            return MySQL.CreateDb(connectString, modelName)
    else:
        return MySQL.DbAdapter(connectString,modelName)

def configurePostgres(modelName,connectString,create):
    """
    Utility function for creating a Postgres driver instance
    """
    try:
        from Ft.Rdf.Drivers import Postgres
        Postgres.InitializeModule()
    except:
        raise Exception("Unable to setup driver: Postgres")
    if create:
        if Postgres.ExistsDb(connectString,modelName) in [1,0]:
            Postgres.DestroyDb(connectString,modelName)
            return Postgres.CreateDb(connectString, modelName)
        else:
            return Postgres.CreateDb(connectString, modelName)
    else:
        return Postgres.DbAdapter(connectString,modelName)        

def configureBsdDb(modelName,dbHandle,create):
    """
    Utility function for creating a BsdDb driver instance
    """
    try:
        from Ft.Rdf.Drivers import BerkeleyDB
        BerkeleyDB.InitializeModule()
    except:
        raise Exception("Unable to setup driver: BerkeleyDB")
    if create:
        if BerkeleyDB.ExistsDb(modelName) in [1,0]:
            BerkeleyDB.DestroyDb(modelName)
            return BerkeleyDB.CreateDb(modelName)
        else:
            return BerkeleyDB.CreateDb(modelName)
    else:
        return BerkeleyDB.DbAdapter(modelName)
        
def configureDbm(modelName,dbName,create):
    """
    Utility function for creating a Dbm driver instance
    """
    try:
        from Ft.Rdf.Drivers import Dbm
        Dbm.InitializeModule()
    except:
        raise Exception("Unable to setup driver: Dbm")
    if create:
        if Dbm.ExistsDb(dbName,modelName) in [1,0]:
            Dbm.DestroyDb(dbName,modelName)
            return Dbm.CreateDb(dbName,modelName)
        else:
            return Dbm.CreateDb(dbName,modelName)
    else:
        return Dbm.DbAdapter(dbName,modelName)

def configureMemory(modelName,dbName,create):
    """
    Utility function for creating a Memory driver instance
    """
    try:
        from Ft.Rdf.Drivers import Memory
        Memory.InitializeModule()
    except:
        raise Exception("Unable to setup driver: Dbm")
    if create:
        if Memory.ExistsDb(dbName,modelName) in [1,0]:
            Memory.DestroyDb(dbName,modelName)
            return Memory.CreateDb(dbName,modelName)
        else:
            return Memory.CreateDb(dbName,modelName)
    else:
        return Memory.DbAdapter(dbName,modelName)

class FtRDFConfigurationManager(object):
    """
    The Configuration Manager takes a configuration string which provides the neccessary parameters
    for connecting to a particular driver.  The create argument is a boolean which specifies whether
    or not to create a new database (destroying any existing database in the process).
    """
    def __init__(self,configuration,create=False):
        kvDict = dict([(part.split('=')[0],part.split('=')[-1]) for part in configuration.split(',')])
        for k in ['driver','model','dbHandle']:
            if k not in kvDict:
                raise Exception("Missing required parameter (%s) in configuration string"%(k))
        self._model = kvDict['model']
        self._driverName = kvDict['driver']
        self._dbHandle = kvDict['dbHandle']
        self._driverFactoryFunc=DRIVER_CONFIGURATION_HANDLERS[kvDict['driver']]
        self._create = create        
    
    def createDriver(self):
        """
        Creates an instance of the driver specified by the configuration string and returns it.
        Note: The returned driver isn't live and it's begin() function needs to be invokeds
        """
        driver=self._driverFactoryFunc(self._model,self._dbHandle,self._create)                
        return driver
            
DRIVER_CONFIGURATION_HANDLERS = {
    'mysql':configureMySQL,
    'dbm':configureDbm,
    'memory':configureMemory,
    'bsddb':configureBsdDb,
    'postgres':configurePostgres,
}            
            
if __name__ == '__main__':
    #ftRdfConf = FtRDFConfigurationManager('driver=mysql,model=system,dbHandle=rdf:root/1618@localhost:3306:xmlserver')
    ftRdfConf = FtRDFConfigurationManager('driver=memory,model=test,dbHandle=test')
    from Ft.Rdf.Model import Model
    db=ftRdfConf.createDriver()
    m=Model(db)
    print m.size()
    db.rollback()
