/*
   Copyright (C) 1994-2001 Digitool, Inc
   This file is part of Opensourced MCL.

   Opensourced MCL is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   Opensourced MCL is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "lisp.h"
#include "lisp_globals.h"


#ifdef LINUX
#include <pthread.h>
#include <semaphore.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>


typedef pthread_mutex_t* MUTEX;
typedef sem_t* COUNTING_SEMAPHORE;

#define LOCK_MUTEX(m) pthread_mutex_lock(m)
#define UNLOCK_MUTEX(m) pthread_mutex_unlock(m)
#define DESTROY_MUTEX(m) pthread_mutex_destroy(m)
#define DESTROY_COUNTING_SEMAPHORE(c) sem_destroy(c)

#define INCREMENT_COUNTING_SEMAPHORE(s) sem_post(s)
#define WAIT_ON_COUNTING_SEMAPHORE(s) sem_wait(s)

#define CURRENT_THREAD_YIELD() sched_yield()

#if 0
#define SUSPEND_PROXY(p) pthread_kill(p->task_id, SIGSTOP)
#define RESUME_PROXY(p) pthread_kill(p->task_id, SIGCONT)
#else
#define SUSPEND_PROXY(p)
#define RESUME_PROXY(p)
#endif

pthread_attr_t _default_proxy_attr, *default_proxy_attr = &_default_proxy_attr;
#endif

#ifdef VXWORKS
#include <vxWorks.h>
#include <semLib.h>
#include <taskLib.h>
#include <sys/times.h>
#include <stdio.h>

#define getpid() taskIdSelf()
typedef SEM_ID MUTEX;
typedef SEM_ID COUNTING_SEMAPHORE;

#define LOCK_MUTEX(m) semTake(m, WAIT_FOREVER)
#define UNLOCK_MUTEX(m) semGive(m)
#define DESTROY_MUTEX(m) semDelete(m)
#define DESTROY_COUNTING_SEMAPHORE(c) semDelete(c)

#define INCREMENT_COUNTING_SEMAPHORE(s) semGive(s)
#define WAIT_ON_COUNTING_SEMAPHORE(s) semTake(s, WAIT_FOREVER)

#define CURRENT_THREAD_YIELD() taskDelay(0)

#if 0
#else
#define SUSPEND_PROXY(p) taskSuspend(p->task_id)
#define RESUME_PROXY(p) taskResume(p->task_id)
#endif

#endif

typedef enum {
  WAIT_NOTHING, 
  WAIT_DIE,
  WAIT_MUTEX,
  WAIT_SELECT_INPUT,
  WAIT_COUNTING_SEMAPHORE,
  WAIT_TIMEOUT,
  WAIT_DEQUEUE=9999
} wait_reason;


#define PUPPET_DEBUG 0
#if PUPPET_DEBUG
MUTEX puppet_debug_lock;

#define PUPPET_PRINTF(fmt, arg) _puppet_printf(fmt, arg);

void
_puppet_printf(char *fmt, int arg)
{
  LOCK_MUTEX(puppet_debug_lock);
  fprintf(stderr, "task_id = %05d : ", getpid());
  fprintf(stderr, fmt, arg);
  fprintf(stderr, "\n");
  fflush(stderr);
  UNLOCK_MUTEX(puppet_debug_lock);
}

#else
#define PUPPET_PRINTF(fmt, arg) ((void *) 0)
#endif    
    
typedef struct _dll_node {
  struct _dll_node *pred;
  struct _dll_node *succ;
} dll_node;

#define DLL_SUCC(node) (((dll_node *)node)->succ)
#define DLL_PRED(node) (((dll_node *)node)->pred)

/* A header is just a distinguished node */
#define HEADER_FIRST(header) (DLL_SUCC(header))
#define HEADER_LAST(header) (DLL_PRED(header))

/* When a node isn't on a linked list, its prev and next fields
   will be NULL. Or, when it is they won't be ...*/

#define NODE_ENQUEUED_P(node) (DLL_SUCC(node) != NULL)

/* A lockable DLL Queue header */
typedef struct _dllQ {
  dll_node node;
  MUTEX lock;
  COUNTING_SEMAPHORE avail;
} dllQ;

typedef struct _proxy {
  dll_node node;
  MUTEX runnable;
#ifdef LINUX
  pthread_t task_id;		/* pthread/task id */
  pid_t pid;			/* for Linux's clone()ed threads */
#endif
#ifdef VXWORKS
  int task_id;			/* OS task id */
#endif
  int vxpriority;		/* vxworks priority  */
  int nativepriority;		/* if not running vxworks */
  wait_reason what_to_block_for;/* nothing, select(fd), semaphore */
  void *block_arg;		/* an fd, a semaphore, ... */
  int proxy_id;			/* for debugging */
} proxy;
  

#ifdef LINUX
pthread_t lisp_task;
pid_t lisp_task_pid;
#endif

#ifdef VXWORKS
int lisp_task, lisp_timer_task;
#endif

dllQ *lisp_proxy_Q = NULL;

typedef enum {
  PROXY_EVENT_ENQUEUED,
  PROXY_EVENT_DEQUEUED,
  PROXY_EVENT_THREAD_READY
} proxy_event_code;
    
    
typedef struct _proxy_event {
  proxy_event_code code;
  proxy *p;
  struct timeval when;
} proxy_event;

proxy_event *proxy_event_head = NULL, 
  *proxy_event_ptr = NULL,
  *proxy_event_tail = NULL;


MUTEX proxy_event_lock = (MUTEX) NULL;


    
MUTEX
create_mutex()
{
#ifdef LINUX
  MUTEX m = (MUTEX) allocate(sizeof(pthread_mutex_t));
  pthread_mutexattr_t m_attr = {PTHREAD_MUTEX_ERRORCHECK_NP};
  pthread_mutex_init(m, &m_attr);
#endif
#ifdef VXWORKS
  MUTEX m = semMCreate(SEM_Q_PRIORITY);
#endif
  return m;
}

COUNTING_SEMAPHORE
create_counting_semaphore(int initialvalue)
{
#ifdef LINUX
  COUNTING_SEMAPHORE s = (COUNTING_SEMAPHORE) allocate(sizeof(sem_t));
  sem_init(s, 0, initialvalue);
#endif
#ifdef VXWORKS
  COUNTING_SEMAPHORE s = semCCreate(SEM_Q_PRIORITY, initialvalue);
#endif
  return s;
}


void
record_proxy_event(proxy_event_code what, proxy *which)
{
  if (proxy_event_head) {
    LOCK_MUTEX(proxy_event_lock);
    if (proxy_event_ptr != proxy_event_tail) {
      proxy_event_ptr->code = what;
      proxy_event_ptr->p = which;
      gettimeofday(&(proxy_event_ptr->when), NULL);
      ++proxy_event_ptr;
    }
    UNLOCK_MUTEX(proxy_event_lock);
  }
}

void
allocate_proxy_events(int n)
{
  void *p;

  LOCK_MUTEX(proxy_event_lock);
  if (proxy_event_head == NULL) {
    p = (void *) allocate((n+1)*sizeof(proxy_event));
    if (p != NULL) {
      proxy_event_head = p;
      proxy_event_ptr = p;
      proxy_event_tail = proxy_event_head+n;
    }
  }
  UNLOCK_MUTEX(proxy_event_lock);
}

/* Caller must deallocate the event buffer.  Once this returns,
   it'd be OK to allocate another one ... */
proxy_event *
retrieve_proxy_events(int *count)
{
  proxy_event *p;

  LOCK_MUTEX(proxy_event_lock);
  p = proxy_event_head;
  *count = proxy_event_ptr - proxy_event_head;
  proxy_event_head = proxy_event_tail = proxy_event_ptr = NULL;
  UNLOCK_MUTEX(proxy_event_lock);
  return p;
}
     
      

int
can_change_priority = 0;

void
set_lisp_task_priority(int nativepriority)
{
#ifdef LINUX
  if (can_change_priority) {
    setpriority(PRIO_PROCESS, lisp_task_pid, nativepriority);
  }
#endif
#ifdef VXWORKS
  int timer_priority = nativepriority-1;
  taskPrioritySet(lisp_timer_task, timer_priority);
  taskPrioritySet(lisp_task, nativepriority);
#endif
}

void
ensure_lisp_task_priority_at_least(proxy *p)
{
  int 
#ifdef LINUX
    lispnative = getpriority(PRIO_PROCESS, lisp_task_pid),
#endif
#ifdef VXWORKS
    lispnative,
#endif
    pnative = p->nativepriority;
#ifdef VXWORKS
  taskPriorityGet(lisp_task, &lispnative);
#endif
  if (pnative < lispnative) {
    if (can_change_priority) {
      set_lisp_task_priority(pnative);
    }
  }
}
    

void
puppet_init()
{
#ifdef LINUX
  can_change_priority = (0 == geteuid());
  lisp_task = pthread_self();
  lisp_task_pid = getpid();
#endif
#ifdef VXWORKS
  can_change_priority = 1;
  lisp_task = taskIdSelf();
#endif
  lisp_proxy_Q = (dllQ *) allocate(sizeof(dllQ));
  DLL_SUCC(lisp_proxy_Q) = (dll_node *) lisp_proxy_Q;
  DLL_PRED(lisp_proxy_Q) = (dll_node *) lisp_proxy_Q;
  lisp_proxy_Q->avail = create_counting_semaphore(0);
  lisp_proxy_Q->lock = create_mutex();

#ifdef LINUX
  pthread_attr_init(default_proxy_attr);
  pthread_attr_setdetachstate(default_proxy_attr, PTHREAD_CREATE_DETACHED);
  pthread_attr_setstacksize(default_proxy_attr, PTHREAD_STACK_MIN);
#endif
#if PUPPET_DEBUG
  puppet_debug_lock = create_mutex();
#endif
  proxy_event_lock = create_mutex();
}

#ifdef LINUX
/* VxWorks task priorities range from 0 (most important) to 255 (least).
   Linux process priorities for non-privileged processes range from
   -20 (most important) to 20 (least important).  Map back and forth.
   In VxWorks, the default priority of a task spawned in the shell
   is 100.  In Linux, it's 0. 
 */

double vx_over_native = 255.0/40.0;
double native_over_vx = 40.0/255.0;


int
vx_pri2native_pri(int vxpri)
{
  if (vxpri <= 100) {
    return -((100 - vxpri) / 5);
  }
  return ((255 - vxpri) / 7);
}

int
native_pri2vx_pri(int nativepri)
{
  if (nativepri <= 20) {
    return (nativepri + 20) * 5;
  }
  return (nativepri + 20) * 7;
}

#endif

#ifdef VXWORKS
#define vx_pri2native_pri(v) (v)
#define native_pri2vx_pri(n) (n)
#endif

void
force_lisp_preemption()
{
  struct lispsymbol *interrupt_level_sym = &nrs_INTERRUPT_LEVEL;
  int interrupt_level = (int) interrupt_level_sym->vcell;
  if (interrupt_level != (-2<<fixnumshift)) {
    /* Does this need to be thread-safe ?  At the moment, I
       don't think so.  Lisp may need to bind *interrupt-level*
       to -2 when it's unwinding the current thread's specials */
    PUPPET_PRINTF("trying to force preemption", 0);
    preemption_handler(2); 
  }
}

void
set_lowest_priority(proxy *p)
{
#ifdef LINUX
  if (can_change_priority) {
    setpriority(PRIO_PROCESS, p->pid, vx_pri2native_pri(255));
  }
#endif
#ifdef VXWORKS
  taskPrioritySet(p->task_id, 255);
#endif
}

void
set_nominal_priority(proxy *p)
{
#ifdef LINUX
  if (can_change_priority) {
    setpriority(PRIO_PROCESS, p->pid, p->nativepriority);
  }
#endif
#ifdef VXWORKS
  taskPrioritySet(p->task_id, p->nativepriority);
#endif
}

/* The header containing this node is locked by the caller, so this
   operation is atomic */

dll_node *
remove_dll_node(dll_node *node)
{
  dll_node *prev = DLL_PRED(node), *next = DLL_SUCC(node);
  DLL_SUCC(prev) = next;
  DLL_PRED(next) = prev;
  DLL_SUCC(node) = NULL;
  DLL_PRED(node) = NULL;
  return node;
}

/* Likewise, the dllQ we're inserting into is locked by the caller.
   The caller should unlock the dllQ, then increment the "avail"
   counting semaphore */
dll_node *
insert_dll_node_before(dll_node *node, dll_node *before)
{
  dll_node *before_before = DLL_PRED(before);

  if ((node == before_before) ||
      (node == before) ||
      (DLL_SUCC(node) != NULL) ||
      (DLL_PRED(node) != NULL)) {
    Bug(NULL, "Insert: bad q/node!");
  }

  DLL_SUCC(before_before) = node;
  DLL_PRED(node) = before_before;
  DLL_SUCC(node) = before;
  DLL_PRED(before) = node;
  return node;
}


void
check_dllQ(dllQ *q)
{
  dll_node *prev, *end = (dll_node *)q, *next;
  LOCK_MUTEX(q->lock);
  
  for (prev = end, next = DLL_SUCC(prev); 
       next != end; 
       prev = next, next = DLL_SUCC(prev)) {
    if ((next == NULL) ||
	(DLL_PRED(next) != prev)) {
      Bug(NULL, "Bad DllQ!");
    }
  }
  UNLOCK_MUTEX(q->lock);
}

/* Insert proxy p in dllQ q, before the first element of 
   arithmetically greater vxpriority. (In VxWorks, "0" is
   the highest priority.) Put it at the end of the queue
   if no such element exists */

void
insert_proxy_in_q(proxy *p, dllQ *q)
{
  LOCK_MUTEX(q->lock);
  {
    int priority = p->vxpriority;
    dll_node *endnode = (dll_node *)q;
    dll_node *next = HEADER_FIRST(endnode);

    p->what_to_block_for = WAIT_DEQUEUE;
    PUPPET_PRINTF("adding proxy %x", p->proxy_id);
    record_proxy_event(PROXY_EVENT_ENQUEUED, p);
    while (next != endnode) {
      if ((((proxy *)next)->vxpriority) > priority) break;
      next = DLL_SUCC(next);
    }
    check_dllQ(q);
    insert_dll_node_before((dll_node *)p, next);
    check_dllQ(q);
  /* If what we just inserted was inserted at the front of the queue,
     ensure that the lisp task is running at at least as logically
     high priority */
    if (DLL_PRED(p) == endnode) {
      ensure_lisp_task_priority_at_least(p);
    }
  }
  INCREMENT_COUNTING_SEMAPHORE(q->avail);
  UNLOCK_MUTEX(q->lock);
  /* If the lisp task is waiting on the queue, it sets ccl::*interrupt-level*
     to -2; this should be the only time that variable has that value.
     If lisp is waiting on the queue, bumping the "avail" semaphore
     will wake it up.  If lisp's not already waiting on the queue, we'll
     have to encourage it to do so at its next convenient opportunity. */
  force_lisp_preemption();
  set_lowest_priority(p);
}


dll_node *
pop_Q(dllQ *q)
{
  dll_node *p;
  p = HEADER_FIRST(q);
  if (p == (dll_node *) q) {
    p = NULL;
  }
  return p;
}

/*
  Wait until at least one proxy is available, then pop the
  first (logically highest priority) entry from the queue
  and return it. */
proxy *
pop_proxy_q()
{
  dllQ *q = lisp_proxy_Q;
  proxy *p = NULL, *next_node = NULL;
  dll_node *node;
  struct lispsymbol *interrupt_level_sym = &nrs_INTERRUPT_LEVEL;
  int old_interrupt_level = (int) interrupt_level_sym->vcell;

  interrupt_level_sym->vcell = (LispObj)(-2<<fixnumshift);
  do {
    WAIT_ON_COUNTING_SEMAPHORE(q->avail);
    LOCK_MUTEX(q->lock);
    node = HEADER_FIRST(q);
    /* There should always be something on the queue at 
       this point; it "can't happen" that the queue is
       empty, but if it does, keep waiting */
    if (node != ((dll_node *) q)) {
      p = (proxy *) node;
      next_node = (proxy *) (DLL_SUCC(p));
      /* Take the proxy's "runnable" semaphore */
      LOCK_MUTEX(p->runnable);
      check_dllQ(q);
      remove_dll_node(node);
      check_dllQ(q);
      if (next_node != (proxy *) q) {
	set_nominal_priority(next_node);
      }
      
      /* ???      p->what_to_block_for = 0; */
      PUPPET_PRINTF( "popped proxy %x", p->proxy_id);
      record_proxy_event(PROXY_EVENT_DEQUEUED, p);
      RESUME_PROXY(p);
    }
    UNLOCK_MUTEX(q->lock);
  } while (NULL == p);
  interrupt_level_sym->vcell = (LispObj)old_interrupt_level;
  /* We'll want to run the proxy's thread at its priority */
  set_lisp_task_priority(p->nativepriority);
  return p;
}

void *
proxy_function(void *arg)
{
  proxy *p = (proxy *) arg;
  dllQ *q = lisp_proxy_Q;
  int wait_for, fd, do_insert;

#ifdef LINUX
  p->pid = getpid();
#endif
  set_nominal_priority(p);
#ifdef VXWORKS
  ioTaskStdSet(0, 0, ioTaskStdGet(lisp_task, 0));
  ioTaskStdSet(0, 1, ioTaskStdGet(lisp_task, 1));
  ioTaskStdSet(0, 2, ioTaskStdGet(lisp_task, 2));
#endif
  while(1) {

    PUPPET_PRINTF("proxy %x: waiting for runnable\n", p->proxy_id);
    /* Block while our lisp thread is running. */
    LOCK_MUTEX(p->runnable);
    PUPPET_PRINTF("proxy %x: have runnable\n", p->proxy_id);
    /* Might as well unlock it now */
    UNLOCK_MUTEX(p->runnable);
    
    /* See if we have something to wait for */
    wait_for = p->what_to_block_for;
    p->what_to_block_for = WAIT_DEQUEUE;
    do_insert = 1;
    switch (wait_for) {
    case WAIT_DEQUEUE:
      /* We've told the lisp task that we want our thread to run (and 
	 for it to grab our "runnable" lock, putting us to sleep), but
	 it wasn't listening (busy with GC or some other atomic operation.)
	 All we can do is to go back to sleep in that case.

	 HOWEVER, if we're on the front of the queue we need to ensure
	 that the lisp task will wake up.
      */
      do_insert = 0;
      if ((dllQ *)(DLL_PRED(p)) == q) {
	force_lisp_preemption();
      }
      CURRENT_THREAD_YIELD();
      break;
      
    case WAIT_MUTEX:
      LOCK_MUTEX((MUTEX)(p->block_arg));
      break;
    case WAIT_SELECT_INPUT:
      {
	fd_set 
	  waitfds, 
#ifdef LINUX
	  *errfds = &waitfds
#endif
#ifdef VXWORKS
	  *errfds = NULL
#endif
	  ;
	int fd = (int) p->block_arg;
	
	FD_ZERO(&waitfds);
	FD_SET(fd, &waitfds);
	select(1+fd,&waitfds, NULL, errfds, NULL);
      }
      break;
    case WAIT_COUNTING_SEMAPHORE:
      WAIT_ON_COUNTING_SEMAPHORE((COUNTING_SEMAPHORE)(p->block_arg));
      break;
    case WAIT_TIMEOUT:
      {
	struct timeval tv;
	unsigned microseconds = (unsigned) p->block_arg;
	
	tv.tv_sec = microseconds / 1000000;
	tv.tv_usec = microseconds % 1000000;
	select(0, NULL, NULL, NULL, &tv);
      }
      break;
    case WAIT_DIE:
      {
	COUNTING_SEMAPHORE dying = (COUNTING_SEMAPHORE) p->block_arg;
	DESTROY_MUTEX(p->runnable);
#ifdef LINUX
	deallocate(p->runnable);
#endif
	deallocate(p);
	INCREMENT_COUNTING_SEMAPHORE(dying);
	return NULL;
      }
    case WAIT_NOTHING:
    default:
      break;
    }
    if (do_insert) {
      if ((DLL_SUCC(p) != NULL) ||
	  (DLL_PRED(p) != NULL)) {
	Bug(NULL, "Inserting node again!");
      }
      insert_proxy_in_q(p, lisp_proxy_Q);
    } 
    else {
      PUPPET_PRINTF("proxy %x: sleeping", p->proxy_id);
      SUSPEND_PROXY(p);
      PUPPET_PRINTF("proxy %x: awakened", p->proxy_id);
    } 

  }
}

/* lisp thread calls this indirectly to let the proxy
   wait on its "runnable" lock (and maybe something else ) */
void
_proxy_wait(proxy *p, wait_reason why, void *wait_what)
{
  if ((DLL_SUCC(p) != NULL) ||
      (DLL_PRED(p) != NULL)) {
    char buf[48];
    sprintf(buf, "Waiting on enqueued proxy %d!", p->proxy_id);
    Bug(NULL, buf);
  }
  p->what_to_block_for = why;
  p->block_arg = wait_what;
  set_nominal_priority(p);
  PUPPET_PRINTF("yielding runnable lock for %x", p->proxy_id);
  PUPPET_PRINTF("... (Waiting for %d)", why);
  UNLOCK_MUTEX(p->runnable);
}

/* I'm not sure that lisp needs all of these entry points - they're
   just wrappers around _proxy_wait() - but they may make things
   a little clearer */

/* The proxy task is runnable; it has nothing in particular to wait
   for */

/* Lisp calls this when the current task is preempted by something
   of higher priority */
void
proxy_wait(proxy *p)
{
  PUPPET_PRINTF("Proxy %x waiting", p->proxy_id);
  _proxy_wait(p, WAIT_NOTHING, NULL);
}

/* Wait until the proxy can lock mutex m */
proxy *
proxy_wait_mutex(proxy *p, MUTEX m)
{
  _proxy_wait(p, WAIT_MUTEX, (void *) m);
  return pop_proxy_q();
}

/* Wait until input is available on fd */
proxy *
proxy_wait_select_input(proxy *p, int fd)
{
  PUPPET_PRINTF("proxy waiting for input: %x", p->proxy_id);
  _proxy_wait(p, WAIT_SELECT_INPUT, (void *) fd);
  return pop_proxy_q();
}

/* Wait on a counting semaphore */
proxy *
proxy_wait_counting_semaphore(proxy *p, COUNTING_SEMAPHORE s)
{
  _proxy_wait(p, WAIT_COUNTING_SEMAPHORE, (void *) s);
  return pop_proxy_q();
}

/* Wait for n milliseconds.  There's a large upper bound on N.
   This isn't going to be too exact: there's some latency here */
proxy *
proxy_wait_timeout(proxy *p, unsigned n)
{
  _proxy_wait(p, WAIT_TIMEOUT, (void *) n);
  return pop_proxy_q();
}

void
kill_proxy(proxy *p)
{ 
  COUNTING_SEMAPHORE is_dead = create_counting_semaphore(0);
  
  _proxy_wait(p, WAIT_DIE, is_dead);
  WAIT_ON_COUNTING_SEMAPHORE(is_dead);
  DESTROY_COUNTING_SEMAPHORE(is_dead);
#ifdef LINUX
  deallocate(is_dead);
#endif
}

/* Create a new task and a proxy record for it.  The caller owns the
   task's "runnable" lock. */

proxy *
create_proxy_task(int vxpriority)
{
  proxy *p = (proxy *)allocate(sizeof(proxy));
  int nativepriority = vx_pri2native_pri(vxpriority);
  static int task_counter = 0;
  MUTEX r = create_mutex();
#ifdef LINUX
  pthread_t native_task;
#endif
#ifdef VXWORKS
  int native_task;
  char name_buf[32];
#endif

  
  LOCK_MUTEX(r);
  p->runnable = r;
  DLL_SUCC(p) = NULL;
  DLL_PRED(p) = NULL;
  p->vxpriority = vxpriority;
  p->nativepriority = nativepriority;
  p->what_to_block_for = WAIT_NOTHING;
  p->block_arg = NULL;
  p->proxy_id = task_counter;
  PUPPET_PRINTF("new proxy: %x", p->proxy_id);
  PUPPET_PRINTF(" ... priority = %d", vxpriority);
#ifdef LINUX
  pthread_create(&(p->task_id), default_proxy_attr, proxy_function, p);
#endif
#ifdef VXWORKS
  sprintf(name_buf, "lisp_proxy_task%d", task_counter);
  p->task_id = taskSpawn(name_buf, 
			 vxpriority,
			 VX_FP_TASK, 
			 4*(1<<12), 
			 (FUNCPTR)proxy_function,
			 (int) p, 2, 3, 4, 5, 6, 7, 8, 9, 10);
#endif
  task_counter++;
  return p;
}

int
current_task_priority()
{
#ifdef LINUX
  return native_pri2vx_pri(getpriority(PRIO_PROCESS, getpid()));
#endif

#ifdef VXWORKS
  int pri;
  taskPriorityGet(0, &pri);
  return pri;
#endif
}

int
proxy_interrupt(proxy *p)
{
  return 0;
}

int
proxy_restore(proxy *p)
{
  return 0;
}
