/*
   Copyright (C) 1994-2001 Digitool, Inc
   This file is part of Opensourced MCL.

   Opensourced MCL is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   Opensourced MCL is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "lisp.h"
#include "lisp_globals.h"
#include "gc.h"
#include "area.h"
#include <stdlib.h>
#include <string.h>
#include "lisp-exceptions.h"
#include "loader.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#ifdef LINUX
#include <mcheck.h>
#include <dirent.h>
#include <dlfcn.h>
#include <sys/time.h>
#endif

#ifdef DARWIN
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#endif

#ifdef VXWORKS
#include <limits.h>
#include <sys/stat.h>
#include <vxWorks.h>
#include <time.h>
#include <nfsLib.h>
#include <ioLib.h>
#include <taskLib.h>
#include <arch/ppc/vxPpcLib.h>
#endif
#include <ctype.h>
#ifndef VXWORKS
#include <sys/select.h>
#endif


/* This is kept outside of the kernel_globals structure
   'cause it's referenced so often. */

LispObj lisp_nil = (LispObj) 0;


/* These are all "persistent" : they're initialized when
   subprims are first loaded and should never change. */
extern LispObj (*start_lisp)(LispObj, LispObj);
extern LispObj subprims_base;
extern LispObj ret1valn;
extern LispObj lexpr_return;
extern LispObj lexpr_return1v;
LispObj real_subprims_base = 0;
LispObj text_start = 0;

/* A pointer to some of the kernel's own data; also persistent. */

extern LispObj import_ptrs_base;


unsigned
align_to_power_of_2(unsigned n, unsigned power)
{
  unsigned align = (1<<power) -1;

  return (n+align) & ~align;
}


void
MakeDataExecutable(void *, unsigned);

void
make_dynamic_heap_executable(LispObj *p, LispObj *q)
{
  void * cache_start = (void *) p;
  unsigned ncacheflush = (unsigned) q - (unsigned) p;

  MakeDataExecutable(cache_start, ncacheflush);  
}
      

#ifdef DARWIN
LispObj
darwin_remap_subprims(void *curloc)
{
  void *desired = (void *) (1<<20);

  if (curloc != desired) {
    void *ret = mmap(desired, 
                    1<<12,
                    PROT_READ|PROT_WRITE,
                    MAP_FIXED|MAP_ANON,
                    -1, 
                    0);

    if (ret != desired) {
      fprintf (stderr, "can't remap subprims \n");
      exit(1);
    } else {
      int i, disp;
      pc jtab = (pc) curloc, dest = (pc)ret;
      LispObj instr, target;
      
      for (i = 0; i < 256; i++, jtab++) {
       instr = *jtab;
       if (instr == 0) break;
       disp = instr & (~3 & ((1<<26)-1));
       target = (LispObj)jtab+disp;
       *dest++ = (BA_VAL) | target;
      }
      
      MakeDataExecutable(ret, 1024);
    }
  }
  return (LispObj)desired;
}

#endif

/* This should write-protect the bottom of the stack.
   Doing so reliably involves ensuring that everything's unprotected on exit.
*/

static BytePtr
allocate_lisp_stack(unsigned useable,
                    unsigned softsize,
                    unsigned hardsize,
                    lisp_protection_kind softkind,
                    lisp_protection_kind hardkind,
                    Ptr *h_p,
                    BytePtr *base_p,
                    protected_area_ptr *softp,
                    protected_area_ptr *hardp)
{
  unsigned size = useable+softsize+hardsize;
  unsigned overhead;
  BytePtr base, softlimit, hardlimit;
  OSErr err;
  Ptr h = allocate(size+4095);
  protected_area_ptr hprotp = NULL, sprotp;

  if (h == NULL) {
    return NULL;
  }
  if (h_p) *h_p = h;
  base = (BytePtr) align_to_power_of_2((unsigned) h, 12);
  hardlimit = (BytePtr) (base+hardsize);
  softlimit = hardlimit+softsize;

  overhead = (base - (BytePtr) h);
  if (hardsize) {
    hprotp = new_protected_area((BytePtr)base,hardlimit,hardkind, hardsize, false);
    if (hprotp == NULL) {
      if (base_p) *base_p = NULL;
      if (h_p) *h_p = NULL;
      deallocate(h);
      return NULL;
    }
    if (hardp) *hardp = hprotp;
  }
  if (softsize) {
    sprotp = new_protected_area(hardlimit,softlimit, softkind, softsize, true);
    if (sprotp == NULL) {
      if (base_p) *base_p = NULL;
      if (h_p) *h_p = NULL;
      if (hardp) *hardp = NULL;
      if (hprotp) delete_protected_area(hprotp);
      deallocate(h);
      return NULL;
    }
    if (softp) *softp = sprotp;
  }
  if (base_p) *base_p = base;
  return (BytePtr) ((unsigned)(base+size));
}

/* This'll allocate a tstack or a vstack, but the thread
   mangler won't let us allocate or reliably protect
   a control stack.
*/
area *
allocate_lisp_stack_area(area_code stack_type,
                         unsigned useable, 
                         unsigned softsize, 
                         unsigned hardsize, 
                         lisp_protection_kind softkind, 
                         lisp_protection_kind hardkind)

{
  BytePtr base, bottom;
  Ptr h;
  area *a = NULL;
  protected_area_ptr soft_area=NULL, hard_area=NULL;

  bottom = allocate_lisp_stack(useable, 
                               softsize, 
                               hardsize, 
                               softkind, 
                               hardkind, 
                               &h, 
                               &base,
                               &soft_area, 
                               &hard_area);

  if (bottom) {
    a = new_area(base, bottom, stack_type);
    a->hardlimit = base+hardsize;
    a->softlimit = base+hardsize+softsize;
    a->h = h;
    a->softprot = soft_area;
    a->hardprot = hard_area;
    add_area(a);
  }
  return a;
}

area*
register_cstack(BytePtr bottom, unsigned size)
{
  BytePtr lowlimit = (BytePtr) (((((unsigned)bottom)-size)+4095)&~4095);
  area *a = new_area((BytePtr) bottom-size, bottom, AREA_CSTACK);

  a->hardlimit = lowlimit+4096;
  a->softlimit = lowlimit+8192;
  add_area(a);
  return a;
}
  
area*
allocate_vstack(unsigned useable)
{
  return allocate_lisp_stack_area(AREA_VSTACK, 
                                  useable, 
                                  4096,
                                  0,
                                  kVSPsoftguard,
                                  kVSPhardguard);
}

area *
allocate_tstack(unsigned useable)
{
  return allocate_lisp_stack_area(AREA_TSTACK, 
                                  useable, 
                                  4096,
                                  0,
                                  kTSPsoftguard,
                                  kTSPhardguard);
}

typedef struct LSIZ_resource LSIZ_resource, *LSIZ_ptr;

#pragma options align=mac68k
struct LSIZ_resource {
  long mac_heap_minimum;
  long mac_heap_maximum;
  short mac_heap_percentage;
  long low_memory_threshold;
  long copying_gc_threshold;
  long stack_minimum;
  long stack_maximum;
  short stack_percentage;
};
#pragma options align=reset

struct LSIZ_resource fake_LSIZ = {
  100 << 10,
  400 << 10,
  5,
  (64 << 10),
  0x7fffffff,
  (32 << 10),
  (180 << 10),
  6
};

/* It's hard to believe that max & min don't exist already */
unsigned unsigned_min(unsigned x, unsigned y)
{
  if (x <= y) {
    return x;
  } else {
    return y;
  }
}

unsigned unsigned_max(unsigned x, unsigned y)
{
  if (x >= y) {
    return x;
  } else {
    return y;
  }
}





#ifdef VXWORKS
int
lisp_partition_exists = 0;

Ptr
lisp_heap_partition = NULL;
#endif

int
reserved_area_size = (1<<30);

void
create_kernel_globals()
{
  KG = (kernel_globals *)allocate(sizeof(kernel_globals));
  if (KG != NULL) {
    bzero((void *)KG, sizeof(kernel_globals));
    KG->cache_block_size = 32;	/* CPU-specific */
  } else {
    fprintf(stderr, "Can't create kernel globals structure\n");
    fflush(stderr);
    lisp_exit(-1);
  }
}

#ifdef VXWORKS
Ptr
create_memory_partition(unsigned totalsize)
{
  lisp_heap_partition = malloc(totalsize);
  if (lisp_heap_partition != NULL) {
    dl_malloc_init(lisp_heap_partition, totalsize);
    lisp_partition_exists = 1;
  } else {
    lisp_exit(-1);
  }
  return lisp_heap_partition;
}

#endif

#define DEFAULT_LISP_HEAP_GC_THRESHOLD (4<<20)

unsigned
lisp_heap_gc_threshold = DEFAULT_LISP_HEAP_GC_THRESHOLD;

/*
  'start' should be on a segment boundary; 'len' should be
  an integral number of segments.
  remap the entire range, a segment at a time.
*/

void 
uncommit_pages(void *start, unsigned len)
{
  BytePtr p;

  /* LOCK_MMAP_LOCK(); */

  madvise(start, len, MADV_DONTNEED);
  
  for (p = start; len; len -= heap_segment_size, p += heap_segment_size) {
    munmap(p, heap_segment_size);
    if (p != mmap(p,
		  heap_segment_size,
		  PROT_NONE,
		  MAP_PRIVATE | MAP_FIXED | MAP_ANON,
		  -1,
		  0)) {
      Fatal("mmap error", "");
    }
  }
  /* UNLOCK_MMAP_LOCK(); */
}



area *
create_reserved_area(unsigned totalsize)
{
  OSErr err;
  Ptr h;
  unsigned base, n;
  BytePtr  end, lastbyte, start, protstart, p;
  area *reserved;
  bitvector markbits;



  totalsize = align_to_power_of_2(totalsize, log2_heap_segment_size);
  start = mmap(NULL,
	       totalsize+heap_segment_size,
	       PROT_NONE,
	       MAP_PRIVATE | MAP_ANON,
	       -1,
	       0);
  if (start == MAP_FAILED) {
    perror("Initial mmap");
    return NULL;
  }
  munmap(start, totalsize+heap_segment_size);
  start =  (void *)((((unsigned)start)+heap_segment_size) & ~(heap_segment_size-1));
  for (p = start , n = totalsize; 
       n ; 
       n -= heap_segment_size, p += heap_segment_size) {
    if(mmap(p, heap_segment_size, PROT_NONE, MAP_PRIVATE | MAP_ANON | MAP_FIXED, -1, 0) != p) {
      return NULL;
    }
    mprotect(p, heap_segment_size, PROT_NONE);
  }

  h = (Ptr) start;
  base = (unsigned) start;
  lastbyte = (BytePtr) (start+totalsize);
  /*
    Allocate mark bits here.  They need to be 1/64 the size of the
     maximum useable area of the heap (+ 3 words for the EGC.)
  */
  end = lastbyte;
  end = (BytePtr) ((unsigned)((((unsigned)end) - ((totalsize+63)>>6)) & ~4095));

  markbits = (bitvector)end;
  end = (BytePtr) ((unsigned)((((unsigned)end) - ((totalsize+63) >> 6)) & ~4095));
  global_reloctab = (LispObj *) end;
  reserved = new_area(start, end, AREA_VOID);
  /* The root of all evil is initially linked to itself. */
  reserved->pred = reserved->succ = reserved;
  KG->all_areas = reserved;
  reserved->markbits = markbits;
  return reserved;
}

void *
allocate_from_reserved_area(unsigned size)
{
  area *reserved = reserved_area;
  BytePtr low = reserved->low, high = reserved->high;
  unsigned avail = high-low;
  size = align_to_power_of_2(size, log2_heap_segment_size);

  if (size > avail) {
    return NULL;
  }
  reserved->low += size;
  reserved->active = reserved->low;
  reserved->ndwords -= (size>>3);
  return low;
}


#define FILE_MAP_FROM_RESERVED_AREA 0

void *
file_map_reserved_pages(unsigned len, int prot, int fd, unsigned offset)
{
  void *start;
  unsigned 
    offset_of_page = offset & ~((1<<12)-1), 
    offset_in_page = offset - offset_of_page,
    segment_len = align_to_power_of_2((offset+len)-offset_of_page, 
				      log2_heap_segment_size);
  
  /* LOCK_MMAP_LOCK(); */
#if FILE_MAP_FROM_RESERVED_AREA
  start = allocate_from_reserved_area(segment_len);
  if (start == NULL) {
    return start;
  }
#endif
#if FILE_MAP_FROM_RESERVED_AREA
  if (start != mmap(start,
		    segment_len,
		    prot,
		    MAP_PRIVATE | MAP_FIXED,
		    fd,
		    offset_of_page)) {
    return NULL;
  }
#else
  if ((start = mmap(NULL,
		    segment_len,
		    prot,
		    MAP_PRIVATE,
		    fd,
		    offset_of_page)) == (void *)-1) {
    return NULL;
  }
#endif
  /* UNLOCK_MMAP_LOCK(); */
  return (void *) (((unsigned)start) + offset_in_page);
}

void
ensure_gc_structures_writable()
{
  area *a = active_dynamic_area;
  unsigned 
    ndwords = a->ndwords,
    markbits_size = 12+((a->ndwords+7)>>3),
    reloctab_size = (sizeof(LispObj)*(((ndwords+31)>>5)+1));

  UnProtectMemory(global_reloctab, reloctab_size);
  UnProtectMemory(a->markbits, markbits_size);

}

area *
allocate_dynamic_area(unsigned initsize)
{
  unsigned totalsize = align_to_power_of_2(initsize + (heap_segment_size * 2)
					   , log2_heap_segment_size);
  BytePtr start, end, p, q;
  protected_area_ptr hardp, softp;
  area *a;

  start = allocate_from_reserved_area(totalsize);
  if (start == NULL) {
    return NULL;
  }
  end = start + totalsize;
  p = end - heap_segment_size;
  a = new_area(start, p, AREA_DYNAMIC);
  a->active = start+initsize;
  add_area(a);
  a->markbits = reserved_area->markbits;
  reserved_area->markbits = NULL;
  hardp = new_protected_area(p, end, kHEAPhard, heap_segment_size, false);
  q = p- (((p - a->active)>>log2_heap_segment_size)<<log2_heap_segment_size);
  softp = new_protected_area(q, p, kHEAPsoft, heap_segment_size, true);
  UnProtectMemory(start, q-start);
  a->h = start;
  a->softprot = softp;
  a->hardprot = hardp;
  a->hardlimit = p;
  ensure_gc_structures_writable();
  return a;
}


/*
  The dynamic area's hard protected_area should be unprotected when
  this is called (it'll move.)  

  The caller should decide whether or not this is really a good idea.
*/

Boolean
grow_dynamic_area(unsigned delta)
{
  area *a = active_dynamic_area, *reserved = reserved_area;
  protected_area_ptr hardp = a->hardprot, softp = a->softprot;
  unsigned avail = reserved->high - reserved->low;
  
  delta = align_to_power_of_2(delta, log2_heap_segment_size);
  if (delta > avail) {
    delta = avail;
  }
  if (!allocate_from_reserved_area(delta)) {
    return false;
  }
  a->high += delta;
  a->ndwords = area_dword(a->high, a->low);
  hardp->start += delta;
  hardp->end += delta;
  softp->start += delta;
  softp->end += delta;
  a->hardlimit = hardp->start;
  lisp_global(HEAP_END) += delta;
  ensure_gc_structures_writable();
  return true;
}

/*
  As above.  Pages that're returned to the reserved_area are
  "condemned" (e.g, we try to convince the OS that they never
  existed ...)
*/
Boolean
shrink_dynamic_area(unsigned delta)
{
  area *a = active_dynamic_area, *reserved = reserved_area;
  protected_area_ptr hardp = a->hardprot, softp = a->softprot;
  
  delta = align_to_power_of_2(delta, log2_heap_segment_size);

  a->high -= delta;
  a->ndwords = area_dword(a->high, a->low);
  hardp->start -= delta;
  hardp->end -= delta;
  softp->start -= delta;
  softp->end -= delta;
  a->hardlimit = hardp->start;
  uncommit_pages(hardp->end, delta);
  reserved->low -= delta;
  reserved->ndwords += (delta>>3);
  lisp_global(HEAP_END) -= delta;
  return true;
}


/* 
 *interrupt-level* is >= 0 when interrupts are enabled and < 0
 during without-interrupts. Normally, it is 0. When this timer
 goes off, it sets it to 1 if it's 0, or if it's negative,
 walks up the special binding list looking for a previous
 value of 0 to set to 1. 
*/
void
preemption_handler(int level)
{
  struct lispsymbol *interrupt_level_sym = &nrs_INTERRUPT_LEVEL;
  int interrupt_level = (int) interrupt_level_sym->vcell;
  LispObj tagged_il_sym = ((LispObj) interrupt_level_sym) + fulltag_misc;

  lisp_global(TICKS) += (1<<fixnumshift); /* Screw: handle overflow ? */

  if (interrupt_level == 0) {
    nrs_INTERRUPT_LEVEL.vcell = level<<fixnumshift;
  }
  else if ((tag_of(interrupt_level) == tag_fixnum) && /* This test may not be necessary */
           (interrupt_level < 0)) {
    struct special_binding *b;
    LispObj b_value;
    for (b = (special_binding *) lisp_global(DB_LINK); b != 0; b = b->link) {
      if (((b->sym) == (lispsymbol *) tagged_il_sym)) {
        if ((b_value = (b->value)) == 0) {
          b->value = level<<fixnumshift;
          break;
        }
        else if (b_value > 0) {
          break;
        }
      }
    }
  }
}

#ifndef VXWORKS
typedef struct {
  int total_hits;
  int lisp_hits;
  int active;
  int interval;
} metering_info;

metering_info
lisp_metering =
{
  0, 
  0, 
  0, 
  0
  };

void
metering_proc(int signum, struct sigcontext *context)
{
  lisp_metering.total_hits++;
#ifndef DARWIN
  if (xpGPR(context,rnil) == lisp_nil) {
    unsigned current_lisp = lisp_metering.lisp_hits, element;
    LispObj 
      rpc = (LispObj) xpPC(context),
      rfn = xpGPR(context, fn),
      rnfn = xpGPR(context, nfn),
      reg,
      v =  nrs_ALLMETEREDFUNS.vcell;

    if (area_containing((BytePtr)rfn) == NULL) {
      rfn = (LispObj) 0;
    }
    if (area_containing((BytePtr)rnfn) == NULL) {
      rnfn = (LispObj) 0;
    }

    if (tag_of(rpc) == tag_fixnum) {
      if (register_codevector_contains_pc(rfn, rpc)) {
	reg = rfn;
      } else if (register_codevector_contains_pc(rnfn, rpc)) {
	reg = rnfn;
      } else {
	reg = rpc;
      }
      element = current_lisp % lisp_metering.active;
      lisp_metering.lisp_hits++;
      deref(v,element+1) = reg; /* NOT memoized */
    }
  }
#endif
}
#endif

#if defined(LINUX) || defined(DARWIN)
#define WHAT_ITIMER ITIMER_REAL
#define WHAT_TIMER_SIGNAL SIGALRM
#endif


void
alarm_handler (int signum, struct sigcontext *context)
{
  if (signum == WHAT_TIMER_SIGNAL) {
    preemption_handler(1);
    if (lisp_metering.active) {
      metering_proc(signum, context);
    }
  }
  if (signum == SIGINT) {
    lisp_global(INTFLAG) = (1 << fixnumshift);
  }
}




/* Start up the VBL task that frobs *interrupt-level* to cause an interrupt. */
#if defined(LINUX) || defined(DARWIN)
void
start_vbl()
{
  static struct itimerval vbl_timer = {
    {0, 0},
    {0, 0}};
  int 
    ticks_per_second = sysconf(_SC_CLK_TCK), 
    usec_per_tick = 1000000 / ticks_per_second;
  vbl_timer.it_interval.tv_usec = usec_per_tick;
  vbl_timer.it_value.tv_usec = usec_per_tick;
  install_signal_handler(WHAT_TIMER_SIGNAL, (__sighandler_t)alarm_handler);
  setitimer(WHAT_ITIMER, &vbl_timer, NULL);
  signal(SIGINT, (__sighandler_t)alarm_handler);

}
#endif

#ifdef VXWORKS
int vxworks_timer_enabled = 1;

void
vxworks_timer_proc()
{
  while(vxworks_timer_enabled) {
    taskDelay(1);
    preemption_handler(1);
  }
  exit(0);
}

void
start_vbl()
{
  WIND_TCB *tcb = taskTcb(0);
  int priority = tcb->priority - 1;
  extern int lisp_timer_task;

  if (priority < 0) {
    priority = 0;
  }
  lisp_timer_task = taskSpawn("ppccl_timer", priority, 0, 8<<10, (FUNCPTR) vxworks_timer_proc, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10);
  signal(SIGINT, alarm_handler);
}

#endif

extern BytePtr
current_stack_pointer(void);



  
Ptr fatal_spare_ptr = NULL;

void
prepare_for_the_worst()
{
  /* I guess that CouldDialog is no more */
  /* CouldDialog(666); */
}

void
Fatal(StringPtr param0, StringPtr param1)
{

  if (fatal_spare_ptr) {
    deallocate(fatal_spare_ptr);
    fatal_spare_ptr = NULL;
  }
  fprintf(stderr, "Fatal error: %s\n%s\n", param0, param1);
  lisp_exit(-1);
}

OSErr application_load_err = noErr;
extern Boolean load_subprims(char *);
#ifdef VXWORKS
extern Boolean load_vxlow(char *);
#endif

area *
set_nil(LispObj);

#ifdef VXWORKS
extern char *
strdup(char *);
char * default_lisp_image_name = "./VXPPCCL";
#endif

#ifdef DARWIN
/* 
   The underlying file system may be case-insensitive (e.g., HFS),
   so we can't just case-invert the kernel's name.
   Tack ".image" onto the end of the kernel's name.  Much better ...
*/
char *
default_image_name(char *orig)
{
  int len = strlen(orig) + strlen(".image") + 1;
  char *copy = (char *) malloc(len);

  if (copy) {
    strcat(copy, orig);
    strcat(copy, ".image");
  }
  return copy;
}

#else
char *
default_image_name(char *orig)
{
  char *copy = strdup(orig), *base = copy, *work = copy, c;
  if (copy == NULL) {
    return NULL;
  }
  while(*work) {
    if (*work++ == '/') {
      base = work;
    }
  }
  work = base;
  while (c = *work) {
    if (islower(c)) {
      *work++ = toupper(c);
    } else {
      *work++ = tolower(c);
    }
  }
  return copy;
}
#endif

/*
  Cleanup everything so that we can run some lisp image again.
  Free everything allocated in the lisp_heap_partition and
  reset the (private) malloc that we use.
  */

void
image_memory_cleanup()
{
#ifdef VXWORKS
  KG = NULL;
  free(lisp_heap_partition);
  lisp_partition_exists = 0;
  zone_malloc_reset();
  lisp_nil = (LispObj)NULL;
#endif
}

char *program_name = NULL;

void
usage_exit(char *herald, int exit_status, char* other_args)
{
  if (herald && *herald) {
    fprintf(stderr, "%s\n", herald);
  }
  fprintf(stderr, "usage: %s <options>\n", program_name);
  fprintf(stderr, "\t or %s <image-name>\n", program_name);
  fprintf(stderr, "\t where <options> are one or more of:\n");
  if (other_args && *other_args) {
    fputs(other_args, stderr);
  }
  fprintf(stderr, "\t-R, --heap-reserve <n>: reserve <n> (default: %d)\n",
	  reserved_area_size);
  fprintf(stderr, "\t\t bytes for heap expansion\n");
  fprintf(stderr, "\t-b, --batch: exit when EOF on *STANDARD-INPUT*\n");
  fprintf(stderr, "\t--no-sigtrap : obscure option for running under GDB\n");
  fprintf(stderr, "\t-I, --image-name <image-name>\n");
  fprintf(stderr, "\t and <image-name> defaults to %s\n", 
	  default_image_name(program_name));
  fprintf(stderr, "\n");
  exit(exit_status);
}

int no_sigtrap = 0;
char *image_name = NULL;
int batch_flag = 0;

#ifndef VXWORKS

/* 
   The set of arguments recognized by the kernel is
   likely to remain pretty small and pretty simple.
   This removes everything it recognizes from argv;
   remaining args will be processed by lisp code.
*/

void
process_options(int argc, char *argv[])
{
  int i, j, k, num_elide, flag, arg_error;
  char *arg, *val;

  for (i = 1; i < argc;) {
    arg = argv[i];
    arg_error = 0;
    if (*arg != '-') {
      i++;
    } else {
      num_elide = 0;
      val = NULL;
      if ((flag = (strncmp(arg, "-I", 2) == 0)) ||
	  (strcmp (arg, "--image-name") == 0)) {
	if (flag && arg[2]) {
	  val = arg+2;
	  num_elide = 1;
	} else {
	  if ((i+1) < argc) {
	    val = argv[i+1];
	    num_elide = 2;
	  } else {
	    arg_error = 1;
	  }
	}
	if (val) {
	  image_name = val;
	}
      } else if ((flag = (strncmp(arg, "-R", 2) == 0)) ||
	   (strcmp(arg, "--heap-reserve") == 0)) {
	unsigned reserved_size;
	char *tail;

	if (flag && arg[2]) {
	  val = arg+2;
	  num_elide = 1;
	} else {
	  if ((i+1) < argc) {
	    val = argv[i+1];
	    num_elide = 2;
	  } else {
	    arg_error = 1;
	  }
	}

	if (val) {
	  reserved_size = strtoul(val, &tail, 0);
	  switch(*tail) {
	  case '\0':
	    break;

	  case 'M':
	  case 'm':
	    reserved_size = reserved_size << 20;
	    break;

	  case 'K':
	  case 'k':
	    reserved_size = reserved_size << 10;
	    break;

	  case 'G':
	  case 'g':
	    reserved_size = reserved_size << 30;
	    break;

	  default:
	    fprintf(stderr, "couldn't parse heap-reserve argument ~s", optarg);
	    reserved_size = reserved_area_size;
	    break;
	  }
	}

	if (reserved_size <= (1<< 30)) {
	  reserved_area_size = reserved_size;
	}

      } else if (strcmp(arg, "--no-sigtrap") == 0) {
	no_sigtrap = 1;
	num_elide = 1;
      } else if ((strcmp(arg, "-b") == 0) ||
		 (strcmp(arg, "--batch") == 0)) {
	batch_flag = 1;
	num_elide = 1;
      } else {
	i++;
      }
      if (arg_error) {
	usage_exit("error in program arguments", 1, "");
      }
      if (num_elide) {
	for (j = i+num_elide, k=i; j < argc; j++, k++) {
	  argv[k] = argv[j];
	}
	argc -= num_elide;
	argv[argc] = NULL;
      }
    }
  }
}



main(int argc, char *argv[])
#endif
#ifdef VXWORKS
vxppcclMain()
#endif
{
#ifdef VXWORKS
  int argc = 1;
  char *argv[] = {"vxppccl", 0};
#endif
  extern  set_fpscr(unsigned);
  extern void altivec_probe(void);
  extern int altivec_present;
  extern LispObj load_image(char *);
  long resp;
  BytePtr heap_start, heap_end, stack_end;
  area *a;
  BytePtr stack_base;

  setr2(0);
#if defined(LINUX) || defined(DARWIN)

  program_name = argv[0];
  if ((argc == 2) && (*argv[1] != '-')) {
    image_name = argv[1];
    argv[1] = NULL;
  } else {
    process_options(argc,argv);
  }
  if (image_name == NULL) {
    image_name = default_image_name(argv[0]);
  }

#endif
#ifdef VXWORKS
#if 0
    ioTaskStdSet(0, 2, ioTaskStdGet(0, 1));
#endif
    image_name = default_lisp_image_name;
#endif

  create_kernel_globals();
#if 0
  fcntl(0, F_SETFL, O_NONBLOCK |  fcntl(0, F_GETFL));
#endif

  prepare_for_the_worst();

  if (!load_subprims("./lisp_subprims.o")) {
    fprintf(stderr, "Couldn't load subprims.\n");
    lisp_exit(-1);
  }

  real_subprims_base = subprims_base;
#ifdef DARWIN
  real_subprims_base = darwin_remap_subprims((void *)subprims_base);
#endif
#ifdef VXWORKS
  if (!load_vxlow("./vxlow.o")) {
    fprintf(stderr, "Couldn't load vxlow module.\n");
    lisp_exit(-1);
  }
#endif
  create_reserved_area(reserved_area_size);
  set_nil(load_image(image_name));
  exception_init();
#ifdef PROXY_SCHEDULER
  puppet_init();
#endif

  if (lisp_global(SUBPRIMS_BASE) == 0) {
    Fatal(": Couldn't load subprims library.", "");
  }
  
  if (application_load_err != noErr) {
    Fatal(": Error loading application data.", (load_error_message != NULL) ? load_error_message : (StringPtr) "Unknown.");
  }

  set_fpscr(0xD0);

  lisp_global(IMAGE_NAME) = (LispObj) image_name;
  lisp_global(ARGV) = (LispObj) argv;
  lisp_global(KERNEL_IMPORTS) = (LispObj) import_ptrs_base;
  a = active_dynamic_area;
  lisp_global(SAVE_FREEPTR) = (LispObj) (a->active);
  heap_start = a->low;
  heap_end = a->high;

  lisp_global(METERING_INFO) = (LispObj) &lisp_metering;

  *(double *) &(lisp_global(DOUBLE_FLOAT_ONE)) = (double) 1.0;
  lisp_global(EXCEPTION_SAVED_REGISTERS) = (LispObj) 0;

  
  
  lisp_global(HEAP_START) = (LispObj) heap_start;
  lisp_global(HEAP_END) = (LispObj) heap_end;

  lisp_global(ARGV) = (LispObj) argv;
  lisp_global(ERRNO) = (LispObj) &errno;
  lisp_global(HOST_PLATFORM) = (LispObj)
#ifdef LINUX
    1
#endif
#ifdef VXWORKS
    2
#endif
#ifdef DARWIN
    3
#endif
    /*We'll get a syntax error here if nothing's defined. */
    << fixnumshift;


#ifdef LINUX
  lisp_global(BATCH_FLAG) = (batch_flag << fixnumshift);
#endif
  if (KG->nilreg_area != NULL) {
    BytePtr lowptr = (BytePtr) a->low;

    a = active_dynamic_area;
    /* Create these areas as AREA_STATIC, change them to AREA_DYNAMIC */
    KG->g1_area = new_area(lowptr, lowptr, AREA_STATIC);
    KG->g2_area = new_area(lowptr, lowptr, AREA_STATIC);
    KG->tenured_area = new_area(lowptr, lowptr, AREA_STATIC);
    add_area(KG->tenured_area);
    add_area(KG->g2_area);
    add_area(KG->g1_area);

    KG->g1_area->code = AREA_DYNAMIC;
    KG->g2_area->code = AREA_DYNAMIC;
    KG->tenured_area->code = AREA_DYNAMIC;

/*    a->older = g1_area; */ /* Not yet: this is what "enabling the EGC" does. */
    KG->g1_area->younger = a;
    KG->g1_area->older = KG->g2_area;
    KG->g2_area->younger = KG->g1_area;
    KG->g2_area->older = KG->tenured_area;
    KG->tenured_area->younger = KG->g2_area;
    KG->tenured_area->refbits = a->markbits;
    lisp_global(TENURED_AREA) = (LispObj)(KG->tenured_area);
    KG->g2_area->threshold = (1<<18); /* 256K */
    KG->g1_area->threshold = (1<<17); /* 128K */
    a->threshold = (1<<16);     /* 64K */
  }

  memo_base =
    (BytePtr) allocate_lisp_stack(1<<15, 1<<12, 0, kMEMOprotect, kNotProtected, NULL, NULL, NULL, NULL);
  lisp_global(SAVE_MEMO) = (LispObj) memo_base;
      
  a = allocate_vstack(STACK_SEGMENT_SIZE);
  stack_end = a->high;
  lisp_global(SAVE_VSP) = (LispObj) stack_end;
  lisp_global(CURRENT_VS) = (LispObj) a;

  a = allocate_tstack(STACK_SEGMENT_SIZE);
  stack_end = a->high;
  lisp_global(SAVE_TSP) = (LispObj) stack_end;
  lisp_global(CURRENT_TS) = (LispObj) a;

  stack_base = current_stack_pointer()-StackSpace();
  a = register_cstack(current_stack_pointer(), StackSpace());
#ifndef MACOS
  init_mac_threads((void *)(stack_base));
#endif
  lisp_global(CURRENT_CS) = (LispObj) a;
  lisp_global(CS_OVERFLOW_LIMIT) = (LispObj) (a->softlimit);

  start_vbl();

  altivec_present = 1;		/* assume no trap */
  altivec_probe();		/* try to use vector unit */
  lisp_global(ALTIVEC_PRESENT) = altivec_present << fixnumshift;


  start_lisp(lisp_nil, 0);
  lisp_exit(0);
}

int
lisp_exit(int status)
{
#ifdef VXWORKS
  vxworks_timer_enabled = 0;
#endif
  exception_cleanup();
  image_memory_cleanup();
  exit(status);
  return status;
}


mutable_data_section_header **next_libP = NULL;
unsigned next_libnum = 0;


area *
set_nil(LispObj r)
{
  static_header *static_headerP;
  area *root, *a, *primary;
  mutable_data_section_header *libP;
  BytePtr low, high;
  unsigned libnum = 2;

  if (lisp_nil == (LispObj)NULL) {

    lisp_nil = r;
    lisp_global(SUBPRIMS_BASE) = subprims_base;
    lisp_global(RET1VALN) = ret1valn;
    lisp_global(LEXPR_RETURN) = lexpr_return;
    lisp_global(LEXPR_RETURN1V) = lexpr_return1v;

    static_headerP = (static_header *) (&(lisp_global(STATIC_HEAP_START)));
    next_libP = &(static_headerP->next);

    low = static_headerP->low;
    high = static_headerP->high;
    
    primary = new_area(low, high, AREA_STATIC);
    add_area(primary);

    make_dynamic_heap_executable((LispObj *) (low),
				 (LispObj *) (high));

    for (libP = static_headerP->next; libP; libP = libP->next) {
      BytePtr *lowP;
      
      /*      load_library_or_die(libP); */
      low = libP->mutable_low;
      a = new_area(low, libP->mutable_high, AREA_STATIC);
      a->owner = libnum;
      add_area(a);

      low = libP->immutable_low;
      lowP = (BytePtr *) low;
      a = new_area(low, libP->immutable_high, AREA_STATICLIB);
      a->owner = libnum;
      add_area(a);

      a = new_area(lowP[0], lowP[1], AREA_READONLY);
      a->owner = libnum;
      add_area(a);


      libnum++;
      next_libP = &(libP->next);
    }

    next_libnum = libnum;

    static_headerP = (static_header *) (&(lisp_global(READONLY_SECTION_START)));
    if (static_headerP->low) {
      add_area(new_area(static_headerP->low, static_headerP->high, AREA_READONLY));
    }

    lisp_global(ALL_AREAS) = (LispObj) (KG->all_areas);
    return primary;
  }
  return NULL;
}


LispObj init_code_vector = (LispObj)NULL;


OSErr
application_loader(int fd, long pos, LispObj r)
{
  area *a;
  init_code_vector = deref(r,0);
  deref(r, 0) = (LispObj)NULL;

  KG->nilreg_area = set_nil(r);


  a = load_application_data(&application_load_err, fd, pos);
  return noErr;
}



#ifdef VXWORKS
char *strdup(char *src)
{
  char *dest = (char *) allocate(strlen(src)+1);
  if (dest != NULL) {
    strcpy(dest, src);
  }
  return dest;
}
#endif

      
#ifndef MACOS
void
MakeDataExecutable(void *start, unsigned nbytes)
{
  extern void flush_cache_lines();
  unsigned ustart = (unsigned) start, base, end;
  
  base = (ustart) & ~(KG->cache_block_size-1);
  end = (ustart + nbytes + KG->cache_block_size - 1) & ~(KG->cache_block_size-1);
  flush_cache_lines(base, (end-base)/KG->cache_block_size, KG->cache_block_size);
}

int
StackSpace()
{
#if defined(LINUX) || defined(DARWIN)
  return STACK_SEGMENT_SIZE;
#endif
#ifdef VXWORKS
  WIND_TCB *tcb = taskTcb(0);
  char here;
  return &here - tcb->pStackLimit;
#endif
}

void *
GetSharedLibrary(char *path, int mode)
{
#ifdef VXWORKS
  return NULL;
#else
#ifdef DARWIN
  return NULL;
#else
  return dlopen(path, mode);
#endif
#endif
}








int
metering_control(int interval)
{
#ifdef DARWIN
  return -1;
#else
  if (interval) {
    if (! lisp_metering.active) {
      LispObj amf = nrs_ALLMETEREDFUNS.vcell;
      if (fulltag_of(amf) == fulltag_misc) {
        unsigned header = header_of(amf);

        if (header_subtag(header) == subtag_simple_vector) {

          lisp_metering.interval = interval;
          lisp_metering.total_hits = 0;
          lisp_metering.lisp_hits = 0;
          lisp_metering.active = header_element_count(header);
          return 0;
        }
      }
    }
    return -1;
  }  else {
    if (lisp_metering.active) {
      lisp_metering.active = 0;
      return 0;
    } else {
      return -1;
    }
  }
#endif
}



void
set_nil_and_start()
{
}

int
ioctl_1(int fd, int selector, int *arg)
{
#ifdef VXWORKS
  switch(selector) {
  case 0x4004667f:
    selector = FIONREAD;
    break;
  }
#endif
  return ioctl(fd, selector, (int)arg);
}

int
xftruncate(int fd, int len)
{
#ifndef VXWORKS
  return ftruncate(fd, len);
#endif
#ifdef VXWORKS
  return -1;
#endif
}


char *
xgetcwd(char *buf, int len)
{
  char *p = getcwd(buf, len);
#ifdef VXWORKS
  if (*p != '/') {
    char *q = strchr(p, ':');
    
    if ((q != NULL) &&
	(*(++q) == '/')) {
      strcpy(p, q);
    }
  }
#endif
  return p;
}

#ifdef VXWORKS
#include <sys/stat.h>

struct linux_stat {
  int st_dev_high;
  int st_dev;
  unsigned __pad1;
  unsigned st_ino;
  unsigned st_mode;
  unsigned st_nlink;
  unsigned st_uid;
  unsigned st_gid;
  unsigned st_rdev_high;
  unsigned st_rdev;
  unsigned __pad2;
  long st_size;
  unsigned st_blksize;
  long st_blocks;
  unsigned st_atime;
  unsigned __unused1;
  unsigned st_mtime;
  unsigned __unused2;
  unsigned st_ctime;
  unsigned __unused3;
  unsigned __unused4;
  unsigned __unused5;
};

void
copy_stat_results(struct linux_stat *l, struct stat *s)
{
  l->st_dev = s->st_dev;
  l->st_ino = s->st_dev;
  l->st_mode = s->st_mode;
  l->st_nlink = s->st_nlink;
  l->st_uid = s->st_uid;
  l->st_gid = s->st_gid;
  l->st_rdev = s->st_rdev;
  l->st_size = s->st_size;
  l->st_atime = s->st_atime;
  l->st_mtime = s->st_mtime;
  l->st_ctime = s->st_ctime;
  l->st_blksize = s->st_blksize;
  l->st_blocks = s->st_blocks;
}
    
int
__xstat(unsigned long version, char *name, struct linux_stat *l)
{
  struct stat s_, *s = &s_;
  int result = stat(name, s);
  if (result == 0) {
    copy_stat_results(l,s);
  }
  return result;
}

int
__lxstat(unsigned long version, char *name, struct linux_stat *l)
{
  /* VxWorks doesn't seem to have an "lstat". */
  return __xstat(version,name,l);
}

void
report_unimplemented(char *name)
{
  char buf[128];
  sprintf(buf, "call to unimplemented function %s\n", name);
  Bug(NULL,buf);
}

int
gettimeofday(struct timespec *tm, void *tz)
{
  clock_gettime(CLOCK_REALTIME, tm);
  tm->tv_nsec /= 1000;
  return 0;
}



struct rusage
{
  /* Total amount of user time used.  */
  struct timespec ru_utime;
  /* Total amount of system time used.  */
  struct timespec ru_stime;
  /* Maximum resident set size (in kilobytes).  */
  long int ru_maxrss;
  /* Amount of sharing of text segment memory
     with other processes (kilobyte-seconds).  */
  long int ru_ixrss;
  /* Amount of data segment memory used (kilobyte-seconds).  */
  long int ru_idrss;
  /* Amount of stack memory used (kilobyte-seconds).  */
  long int ru_isrss;
  /* Number of soft page faults (i.e. those serviced by reclaiming
     a page from the list of pages awaiting reallocation.  */
  long int ru_minflt;
  /* Number of hard page faults (i.e. those that required I/O).  */
  long int ru_majflt;
  /* Number of times a process was swapped out of physical memory.  */
  long int ru_nswap;
  /* Number of input operations via the file system.  Note: This
     and `ru_oublock' do not include operations with the cache.  */
  long int ru_inblock;
  /* Number of output operations via the file system.  */
  long int ru_oublock;
  /* Number of IPC messages sent.  */
  long int ru_msgsnd;
  /* Number of IPC messages received.  */
  long int ru_msgrcv;
  /* Number of signals delivered.  */
  long int ru_nsignals;
  /* Number of voluntary context switches, i.e. because the process
     gave up the process before it had to (usually to wait for some
     resource to be available).  */
  long int ru_nvcsw;
  /* Number of involuntary context switches, i.e. a higher priority process
     became runnable or the current process used up its time slice.  */
  long int ru_nivcsw;
};

int 
getrusage(int who, struct rusage *usage)
{
  gettimeofday(&usage->ru_utime, NULL);
  usage->ru_stime.tv_sec = 0;
  usage->ru_stime.tv_nsec = 0;
  return 0;
}

#define Undefined(name) void name() {report_unimplemented(#name);}
Undefined(__tcgetattr)
Undefined(acosh)
Undefined(atanh)
Undefined(asinh)

struct passwd {
  char    *pw_name;       /* user name */
  char    *pw_passwd;     /* user password */
  int   pw_uid;         /* user id */
  int   pw_gid;         /* group id */
  char    *pw_gecos;      /* real name */
  char    *pw_dir;        /* home directory */
  char    *pw_shell;      /* shell program */
};

struct passwd
fmh = {
  "byers",
  "no",
  4954,
  4023,
  "guess",
  "/home/byers",
  "sh"
};

struct passwd *
getpwuid(int uid)
{
  if (uid = fmh.pw_uid) {
    return &fmh;
  }
  return NULL;
}

int
getuid()
{
  char host[512];
  int uid, gid, ngids = 0, gidbuf[100];

  nfsAuthUnixGet(host, &uid, &gid, &ngids, gidbuf);
  return uid;
}




static char *
canonicalize (const char *name, char *resolved)
{
  char *rpath, *dest, *extra_buf = NULL;
  const char *start, *end, *rpath_limit;
  long int path_max;
  int num_links = 0;

  if (name == NULL)
    {
      /* As per Single Unix Specification V2 we must return an error if
	 either parameter is a null pointer.  We extend this to allow
	 the RESOLVED parameter be NULL in case the we are expected to
	 allocate the room for the return value.  */
      errnoSet (EINVAL);
      return NULL;
    }

  if (name[0] == '\0')
    {
      /* As per Single Unix Specification V2 we must return an error if
	 the name argument points to an empty string.  */
      errnoSet (ENOENT);
      return NULL;
    }

  path_max = PATH_MAX;

  rpath = resolved ? alloca (path_max) : malloc (path_max);
  rpath_limit = rpath + path_max;

  if (name[0] != '/')
    {
      if (!xgetcwd (rpath, path_max))
	goto error;
      dest = strchr (rpath, '\0');
    }
  else
    {
      rpath[0] = '/';
      dest = rpath + 1;
    }

  for (start = end = name; *start; start = end)
    {
      struct stat st;
      int n;

      /* Skip sequence of multiple path-separators.  */
      while (*start == '/')
	++start;

      /* Find end of path component.  */
      for (end = start; *end && *end != '/'; ++end)
	/* Nothing.  */;

      if (end - start == 0)
	break;
      else if (end - start == 1 && start[0] == '.')
	/* nothing */;
      else if (end - start == 2 && start[0] == '.' && start[1] == '.')
	{
	  /* Back up to previous component, ignore if at root already.  */
	  if (dest > rpath + 1)
	    while ((--dest)[-1] != '/');
	}
      else
	{
	  size_t new_size;

	  if (dest[-1] != '/')
	    *dest++ = '/';

	  if (dest + (end - start) >= rpath_limit)
	    {
	      ptrdiff_t dest_offset = dest - rpath;

	      if (resolved)
		{
		  errnoSet (ENAMETOOLONG);
		  goto error;
		}
	      new_size = rpath_limit - rpath;
	      if (end - start + 1 > path_max)
		new_size += end - start + 1;
	      else
		new_size += path_max;
	      rpath = realloc (rpath, new_size);
	      rpath_limit = rpath + new_size;
	      if (rpath == NULL)
		return NULL;

	      dest = rpath + dest_offset;
	    }

	  dest = memcpy (dest, start, end - start);
	  dest += (end-start);
	  *dest = '\0';

	  if (stat (rpath, &st) < 0)
	    goto error;

#ifdef VXWORKS_RESOLVES_LINKS
	  if (S_ISLNK (st.st_mode))
	    {
	      char *buf = __alloca (path_max);
	      size_t len;

	      if (++num_links > MAXSYMLINKS)
		{
		  errnoSet (ELOOP);
		  goto error;
		}

	      n = __readlink (rpath, buf, path_max);
	      if (n < 0)
		goto error;
	      buf[n] = '\0';

	      if (!extra_buf)
		extra_buf = __alloca (path_max);

	      len = strlen (end);
	      if ((long int) (n + len) >= path_max)
		{
		  errnoSet (ENAMETOOLONG);
		  goto error;
		}

	      /* Careful here, end may be a pointer into extra_buf... */
	      memmove (&extra_buf[n], end, len + 1);
	      name = end = memcpy (extra_buf, buf, n);

	      if (buf[0] == '/')
		dest = rpath + 1;	/* It's an absolute symlink */
	      else
		/* Back up to previous component, ignore if at root already: */
		if (dest > rpath + 1)
		  while ((--dest)[-1] != '/');
	    }
#endif
	}
    }
  if (dest > rpath + 1 && dest[-1] == '/')
    --dest;
  *dest = '\0';

  return resolved ? memcpy (resolved, rpath, dest - rpath + 1) : rpath;

error:
  if (resolved)
    strcpy (resolved, rpath);
  else
    free (rpath);
  return NULL;
}

/* We don't seem to have any way of reading links */
char *
realpath(char *pathname, char *resolvedname)
{
  if (resolvedname == NULL)
    {
      errnoSet (EINVAL);
      return NULL;
    }

  return canonicalize (pathname, resolvedname);
}
#endif
#endif



int
fd_setsize_bytes()
{
  return FD_SETSIZE/8;
}

void
do_fd_set(int fd, fd_set *fdsetp)
{
  FD_SET(fd, fdsetp);
}

void
do_fd_clr(int fd, fd_set *fdsetp)
{
  FD_CLR(fd, fdsetp);
}

int
do_fd_is_set(int fd, fd_set *fdsetp)
{
  return FD_ISSET(fd,fdsetp);
}

void
do_fd_zero(fd_set *fdsetp)
{
  FD_ZERO(fdsetp);
}
