;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 2001 Clozure Associates
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;



#+linux-target
(defloadvar *db-lib*
    (ignore-errors (open-shared-library "libdb1.so")))


(defstruct ffi-type
  (ordinal nil)
  (defined nil)
  (string)
  (name)                                ; a keyword, uppercased or NIL
)

(defmethod print-object ((x ffi-type) out)
  (print-unreadable-object (x out :type t :identity t)
    (format out "~a" (ffi-type-string x))))

(defvar *ffi-prefix* "")


(defstruct (ffi-union (:include ffi-type))
  fields
  (anon-global-id (unless name (concatenate 'string *ffi-prefix* "-" string))))

(defstruct (ffi-struct (:include ffi-type))
  fields
  (anon-global-id (unless name (concatenate 'string *ffi-prefix* "-" string))))

(defstruct (ffi-typedef (:include ffi-type))
  (type))

(defun ffi-struct-reference (s)
  (or (ffi-struct-name s) (ffi-struct-anon-global-id s)))

(defun ffi-union-reference (u)
  (or (ffi-union-name u) (ffi-union-anon-global-id u)))

(defstruct (ffi-function (:include ffi-type))
  arglist
  return-value)





    
(def-foreign-type nil
  (:struct dbm-constant
   (:class (:unsigned 32))
   (:value
    (:union nil
      (:s32 (:signed 32))
      (:u32 (:unsigned 32))
      (:single-float :float)
      (:double-float :double)))))

(defconstant db-string-constant 0)
(defconstant db-read-string-constant 1)
(defconstant db-s32-constant 2)
(defconstant db-u32-constant 3)
(defconstant db-float-constant 4)
(defconstant db-double-constant 5)
(defconstant db-char-constant 6)

(defparameter *arg-spec-encoding*
  '((#\Space . :void)
    (#\a . :address)
    (#\F . :signed-fullword)
    (#\f . :unsigned-fullword)
    (#\H . :signed-halfword)
    (#\h . :unsigned-halfword)
    (#\B . :signed-byte)
    (#\b . :unsigned-byte)
    (#\s . :single-float)
    (#\d . :double-float)
    (#\L . :signed-doubleword)
    (#\l . :unsigned-doubleword)))

(defun encode-arguments (args result string)
  (let* ((i 1))
    (flet ((encoded-arg (thing)
             (if thing
               (or (car (rassoc thing *arg-spec-encoding*))
                   (return-from encode-arguments nil))
               #\Space)))
      (setf (schar string 0) (encoded-arg result))
      (dolist (arg args t)
        (setf (schar string i) (encoded-arg arg)
              i (1+ i))))))

(defun decode-arguments (string)
  (let* ((result (cdr (assoc (schar string 0) *arg-spec-encoding*)))
         (args ()))
    (do* ((i 1 (1+ i)))
         ((= i (length string)) (values (nreverse args) result))
      (push (cdr (assoc (schar string i) *arg-spec-encoding*))
            args))))

;;; encoded external function looks like:
;;; byte min-args
;;; byte name-length
;;; name-length bytes of name
;;; result+arg specs

(defun extract-db-function (dbt)
  (let* ((val nil)
         (dsize (pref dbt :DBT.size)))
    (with-macptrs ((dptr))
      (%setf-macptr dptr (pref dbt :DBT.data))
      (unless (%null-ptr-p dptr)
	(let* ((min-args (%get-byte dptr))
	       (name-len (%get-byte dptr 1))
	       (external-name (%str-from-ptr (%inc-ptr dptr 2) name-len))
	       (encoding-len (- dsize (+ 2 name-len)))
	       (encoding (make-string encoding-len)))
	  (declare (dynamic-extent encoding))
	  (%copy-ptr-to-ivector dptr (+ 2 name-len) encoding 0 encoding-len)
	  (multiple-value-bind (args result)
	      (decode-arguments encoding)
	    (setq val (make-external-function-definition
		       :entry-name external-name
		       :arg-specs args
		       :result-spec result
		       :min-args min-args))))))
    val))

(defun db-lookup-function (db-file name)
  (rletZ ((value :DBT)
	  (key :DBT))
    (with-cstrs ((keyname (string name)))
      (setf (pref key :DBT.data) keyname
            (pref key :DBT.size) (length (string name))
            (pref value :DBT.data) (%null-ptr)
            (pref value :DBT.size) 0)
      (if (eql 0 (ff-call (pref db-file :DB.get)
			  :address db-file
			  :address key
			  :address value
			  :unsigned-fullword 0
			  :signed-fullword))
	(extract-db-function value)))))


(defun save-db-function (db-file fname def)
  (let* ((external-name (efd-entry-name def))
         (args (efd-arg-specs def))
         (result (efd-result-spec def))
         (min-args (efd-min-args def))
         (namelen (length external-name))
         (enclen (1+ (length args)))
         (buflen (+ 2 namelen enclen))
         (encoding (make-string enclen)))
    (declare (dynamic-extent encoding))
    (when (encode-arguments args result encoding)
      (%stack-block ((buf buflen))
        (setf (%get-byte buf) min-args
              (%get-byte buf 1) namelen)
        (%copy-ivector-to-ptr external-name 0 buf 2 namelen)
        (%copy-ivector-to-ptr encoding 0 buf (+ 2 namelen) enclen)
        (rletZ ((contents :DBT)
		(key :DBT))
          (setf (pref contents DBT.data) buf
                (pref contents :DBT.size) buflen)
          (with-cstrs ((keyname (string fname)))
            (setf (pref key DBT.data) keyname
                  (pref key :DBT.size) (length (string fname)))
	    (ff-call (pref db-file :DB.put)
		     :address db-file
		     :address key
		     :address contents
		     :unsigned-fullword 0
		     :signed-fullword)))))))
        
         
(defun extract-db-constant-value (dbt)
  (let* ((val nil)
         (dsize (pref dbt :DBT.size)))
    (with-macptrs ((dptr))
      (%setf-macptr dptr (pref dbt DBT.data))
      (unless (%null-ptr-p dptr)
	(let* ((class (pref dptr :dbm-constant.class)))
	  (setq val
		(ecase class
                  ((#.db-string-constant #.db-read-string-constant)
                   (let* ((str (%str-from-ptr (%inc-ptr dptr 4) (- dsize 4))))
                     (if (eql class db-read-string-constant)
                       (read-from-string str)
                       str)))
                  (#.db-s32-constant (pref dptr :dbm-constant.value.s32))
                  (#.db-u32-constant (pref dptr :dbm-constant.value.u32))
                  (#.db-float-constant (pref dptr :dbm-constant.value.single-float))
                  (#.db-double-constant (pref dptr :dbm-constant.value.double-float))
                  (#.db-char-constant (code-char (pref dptr :dbm-constant.value.u32))))))))
    val))



(defun db-lookup-constant (db-file name)
  (rletZ ((value :DBT)
	  (key :DBT))
    (with-cstrs ((keyname (string name)))
      (setf (pref key :DBT.data) keyname
            (pref key :DBT.size) (length (string name))
            (pref value :DBT.data) (%null-ptr)
            (pref value :DBT.size) 0)
      (if (eql 0 (ff-call (pref db-file :DB.get)
			  :address db-file
			  :address key
			  :address value
			  :unsigned-fullword 0
			  :signed-fullword))
	(extract-db-constant-value value)))))
    


(defun %db-open (namestring flags mode &optional (db-type #$DB_BTREE))
  (with-cstrs ((db-name namestring))
    (let* ((result (#_dbopen db-name flags mode db-type (%null-ptr))))
      (unless (%null-ptr-p result)
	result))))

(defun create-db-file (pathname)
  (or (%db-open (namestring (translate-logical-pathname pathname))
		(logior #$O_RDWR #$O_CREAT #$O_TRUNC)
		#o666)
      (error "Couldn't create db file ~s" pathname)))

(defun open-input-db-file (pathname)
  (or
   (%db-open (namestring (translate-logical-pathname pathname)) #$O_RDONLY  #o666)
   (error "Couldn't open database file ~s" pathname )))

(defun %db-close (db-file)
  (ff-call (pref db-file :DB.close)
	   :address db-file
	   :signed-fullword))


(defun db-define-string-constant (db-file name val &optional (class db-string-constant))
  (let* ((dsize (+ 4 (length val))))
    (%stack-block ((valbuf dsize))
      (%copy-ivector-to-ptr val 0 valbuf 4 (length val))
      (setf (%get-long valbuf) class)
      (rletZ ((content :DBT)
	      (key :DBT))
        (setf (pref content :DBT.size) dsize
              (pref content DBT.data) valbuf)
        (with-cstrs ((keyname (string name)))
          (setf (pref key :DBT.size) (length (string name))
                (pref key DBT.data) keyname)
	    (ff-call (pref db-file :DB.put)
		     :address db-file
		     :address key
		     :address content
		     :unsigned-fullword 0
		     :signed-fullword))))))
      
(defun db-define-constant (db-file name val)
  (typecase val
    (string (db-define-string-constant db-file name val))
    ((or (unsigned-byte 32)
         (signed-byte 32)
         short-float
         double-float
         character)
     (rletZ ((constant :dbm-constant)
	     (content :DBT)
	     (key :DBT))
       (etypecase val
         ((signed-byte 32)
          (setf (pref constant :dbm-constant.value.s32) val)
          (setf (pref constant :dbm-constant.class) db-s32-constant))
         ((unsigned-byte 32)
          (setf (pref constant :dbm-constant.value.u32) val)
          (setf (pref constant :dbm-constant.class) db-u32-constant))
         (short-float
          (setf (pref constant :dbm-constant.value.single-float) val)
          (setf (pref constant :dbm-constant.class) db-float-constant))
         (double-float
          (setf (pref constant :dbm-constant.value.double-float) val)
          (setf (pref constant :dbm-constant.class) db-double-constant))
         (character
          (setf (pref constant :dbm-constant.value.u32) (char-code val))
          (setf (pref constant :dbm-constant.class) db-char-constant)))
       (setf (pref content DBT.data) constant
             (pref content :DBT.size) (record-length :dbm-constant))
       (with-cstrs ((keyname (string name)))
         (setf (pref key DBT.data) keyname
               (pref key :DBT.size) (length (string name)))
	    (ff-call (pref db-file :DB.put)
		     :address db-file
		     :address key
		     :address content
		     :unsigned-fullword 0
		     :signed-fullword))))
    (t (db-define-string-constant db-file name (format nil "~a" val) db-read-string-constant))))

(defmacro with-new-db-file ((var pathname) &body body)
  (let* ((db (gensym)))
    `(let* (,db)
      (unwind-protect
           (let* ((,var (setq ,db (create-db-file ,pathname))))
             ,@body)
        (when ,db (%db-close ,db))))))

(defun compact-db-file (inpath outpath)
  (with-new-db-file (outfile outpath)
    (let* ((infile (open-input-db-file inpath)))
      (unwind-protect
	   (rletZ ((key :DBT)
		   (value :DBT))
	     (loop
		 (let* ((result (ff-call (pref infile :DB.seq)
					 :address infile
					 :address key
					 :address value
					 :unsigned-fullword #$R_NEXT
					 :signed-fullword)))
		   (if (eql 0 result)
		     (ff-call (pref outfile :DB.put)
			      :address outfile
			      :address key
			      :address value
			      :unsigned-fullword 0
			      :signed-fullword)
		     (return)))))
	(%db-close infile)))))

(defun interface-db-pathname (name d &optional (ftd *target-ftd*))
  (merge-pathnames name
		   (merge-pathnames (interface-dir-subdir d)
				    (ftd-interface-db-directory ftd))))

(def-ccl-pointers reset-db-files ()
  (do-interface-dirs (d)
    (setf (interface-dir-constants-interface-db-file d) nil
	  (interface-dir-functions-interface-db-file d) nil
	  (interface-dir-records-interface-db-file d) nil
	  (interface-dir-types-interface-db-file d) nil)))

(defun db-constants (dir)
  (or (interface-dir-constants-interface-db-file dir)
      (setf (interface-dir-constants-interface-db-file dir)
	    (open-input-db-file (interface-db-pathname "constants.db" dir)))))

(defun db-types (dir)
  (or (interface-dir-types-interface-db-file dir)
      (setf (interface-dir-types-interface-db-file dir)
	    (open-input-db-file (interface-db-pathname "types.db" dir)))))

(defun db-records (dir)
  (or (interface-dir-records-interface-db-file dir)
      (setf (interface-dir-records-interface-db-file dir)
	    (open-input-db-file (interface-db-pathname "records.db" dir)))))

(defun db-functions (dir)
  (or (interface-dir-functions-interface-db-file dir)
      (setf (interface-dir-functions-interface-db-file dir)
	    (open-input-db-file (interface-db-pathname "functions.db" dir)))))

(defun load-os-constant (sym &optional reader-stream)
  (declare (ignore reader-stream))
  (let* ((val (or (do-interface-dirs (d)
		    (let* ((v (db-lookup-constant (db-constants d) sym)))
		      (when v (return v))))
                  (error "Constant not found: ~s" sym))))
    (let* ((*record-source-file* nil))
      (%defconstant sym val)
      val)))


(defun load-external-function (sym &optional reader-stream)
  (declare (ignore reader-stream))
  (let* ((def (or (do-interface-dirs (d)
		    (let* ((f (db-lookup-function (db-functions d) sym)))
		      (when f (return f))))
                  (error "Foreign function not found: ~s" sym))))
    (setf (gethash sym (ftd-external-function-definitions
			*target-ftd*)) def)
    (setf (macro-function sym) #'%external-call-expander)
    sym))
  
(set-dispatch-macro-character 
 #\# #\$
 (qlfun |#$-reader| (stream char arg)
   (declare (ignore char))
   (let* ((sym
	   (let* ((*package*
		   (find-package (ftd-interface-package-name *target-ftd*))))
	     (read stream t nil t))))
     (unless *read-suppress*
       (unless arg (setq arg 0))
       (ecase arg
         (0
          (unless (boundp sym)
            (load-os-constant sym stream)))
         (1 (makunbound sym) (load-os-constant sym stream)))
       sym))))

(set-dispatch-macro-character #\# #\_
  (qlfun |#_-reader| (stream char arg)
    (when arg (unless *read-suppress* (%err-disp $xrdnoarg char)))
    (let* ((sym
	    (let* ((*package*
		    (find-package (ftd-interface-package-name *target-ftd*))))
	      (read stream t nil t))))
      (unless *read-suppress*
        (unless (and sym (symbolp sym)) (report-bad-arg sym 'symbol))
        (load-external-function sym stream)))))



(eval-when (:compile-toplevel :execute)
  (defconstant encoded-type-void 0)
  (defconstant encoded-type-signed-32 1)
  (defconstant encoded-type-unsigned-32 2)
  (defconstant encoded-type-signed-8 3)
  (defconstant encoded-type-unsigned-8 4)
  (defconstant encoded-type-signed-16 5)
  (defconstant encoded-type-unsigned-16 6)
  (defconstant encoded-type-signed-n 7) ;N
  (defconstant encoded-type-unsigned-n 8) ;N
  (defconstant encoded-type-single-float 9)
  (defconstant encoded-type-double-float 10)
  (defconstant encoded-type-pointer 11) ; <type>
  (defconstant encoded-type-array 12) ; <size> <type>
  (defconstant encoded-type-named-struct-ref 13); <tag>
  (defconstant encoded-type-named-union-ref 14) ;<tag>
  (defconstant encoded-type-named-type-ref 15) ; <name>
  (defconstant encoded-type-anon-struct-ref 16) ; <tag>
  (defconstant encoded-type-anon-union-ref 17) ; <tag>
  )


(defun encode-name (name)
  (if (null name)
    '(0)
    (let* ((string (string name))
           (length (length string)))
      (cons length (map 'list #'char-code string)))))

(defun encode-ffi-field (field)
  `(,@(encode-name (car field)) ,@(encode-ffi-type (cadr field))))

(defun encode-ffi-field-list (fields)
  (let* ((len (length fields)))
    (labels ((encode-fields (fields)
               (if fields
                 `(,@(encode-ffi-field (car fields)) ,@(encode-fields (cdr fields))))))
      `(,len ,@(encode-fields fields)))))

(defun encode-ffi-union (u)
  (let* ((name (ffi-union-name u)))
    (if name
      `(,encoded-type-named-union-ref 
        ,@(encode-name name)
        ,@(encode-ffi-field-list (ffi-union-fields u)))
      `(,encoded-type-anon-union-ref
        ,@(encode-ffi-field-list (ffi-union-fields u))))))

(defun encode-ffi-struct (s)
  (let* ((name (ffi-struct-name s)))
    (if name
      `(,encoded-type-named-struct-ref 
        ,@(encode-name (ffi-struct-name s))
        ,@(encode-ffi-field-list (ffi-struct-fields s)))
      `(,encoded-type-anon-struct-ref
        ,@(encode-ffi-field-list (ffi-struct-fields s))))))

(defun encode-u32 (val)
  `(,(ldb (byte 8 24) val)
    ,(ldb (byte 8 16) val)
    ,(ldb (byte 8 8) val)
    ,(ldb (byte 8 0) val)))


(defun encode-ffi-type (spec)
  (case (car spec)
    (:primitive
     (let ((primtype (cadr spec)))
       (if (atom primtype)
         (case primtype
           (:float `(,encoded-type-single-float))
           (:double `(,encoded-type-double-float))
           (:void `(,encoded-type-void))
           (:signed `(,encoded-type-signed-32))
           (:unsigned `(,encoded-type-unsigned-32))
           ((:long-double :complex-int
                        :complex-float :complex-double :complex-long-double)
            (encode-ffi-type `(:struct ,primtype))))
         (ecase (car primtype)
           (* `(,encoded-type-pointer ,@(encode-ffi-type
                                           (if (eq (cadr primtype) t)
                                             `(:primitive :void)
                                             (cadr primtype)))))
           (:signed
            (case (cadr primtype)
              (32 `(,encoded-type-signed-32))
              (16 `(,encoded-type-signed-16))
              (8 `(,encoded-type-signed-8))
              (t `(,encoded-type-signed-n ,(cadr primtype)))))
           (:unsigned
            (case (cadr primtype)
              (32 `(,encoded-type-unsigned-32))
              (16 `(,encoded-type-unsigned-16))
              (8 `(,encoded-type-unsigned-8))
              (t `(,encoded-type-unsigned-n ,(cadr primtype)))))))))
     (:struct
      (let* ((s (cadr spec))
             (name (ffi-struct-name s)))
      `(,(if name
             encoded-type-named-struct-ref
             encoded-type-anon-struct-ref)
        ,@(encode-name (ffi-struct-reference s)))))
     (:union
      (let* ((u (cadr spec))
             (name (ffi-union-name u)))
      `(,(if name
             encoded-type-named-union-ref
             encoded-type-anon-union-ref)
        ,@(encode-name (ffi-union-reference u)))))
     (:typedef
      `(,encoded-type-named-type-ref ,@(encode-name (ffi-typedef-name (cadr spec)))))
     (:pointer
      `(,encoded-type-pointer ,@(encode-ffi-type
                                   (if (eq (cadr spec) t)
                                     '(:primitive :void)
                                     (cadr spec)))))
     (:array
      `(,encoded-type-array ,@(encode-u32 (cadr spec)) ,@(encode-ffi-type (caddr spec))))
     (t
      (break "Type spec = ~s" spec))))

(defun encode-ffi-arg-type (spec)
  (case (car spec)
    (:primitive
     (let ((primtype (cadr spec)))
       (if (atom primtype)
         (case primtype
           (:float `(#\s))
           (:double `(#\d))
           (:void `(#\Space))
           (:signed `(#\F))
           (:unsigned `(f))
           ((:long-double :complex-int
			  :complex-float :complex-double :complex-long-double)            
            #|(encode-ffi-arg-type `(:struct ,primtype))|#
            `(#\?)))
         (ecase (car primtype)
           (* `(#\a))
           (:signed
            (let* ((nbits (cadr primtype)))
              (if (<= nbits 8)
                '(#\B)
                (if (<= nbits 16)
                  '(#\H)
                  (if (<= nbits 32)
                    '(#\F)
		    (if (<= nbits 64)
		      `(#\L)
		      '(#\?)))))))
           (:unsigned
            (let* ((nbits (cadr primtype)))
              (if (<= nbits 8)
                '(#\b)
                (if (<= nbits 16)
                  '(#\h)
                  (if (<= nbits 32)
                    '(#\f)
		    (if (<= nbits 64)
		      `(#\l)
		      '(#\?)))))))))))
    ((:struct :union) `(#\a))
    (:typedef
      (encode-ffi-arg-type (ffi-typedef-type (cadr spec))))
    (:pointer
      `(#\a))
    (:array
      `(#\?))))

(defun encode-ffi-arg-list (args)
  (if args
    `(,@(encode-ffi-arg-type (car args)) ,@(encode-ffi-arg-list (cdr args)))))

(defvar *prepend-underscores-to-ffi-function-names* nil)

(defun encode-ffi-function (f)
  (let* ((args (ffi-function-arglist f))
	 (string (ffi-function-string f))
	 (name (if *prepend-underscores-to-ffi-function-names*
		 (concatenate 'string "_" string)
		 string))
         (min-args (length args))
         (result (ffi-function-return-value f)))
    `(,min-args
      ,@(encode-name name)
      ,@(encode-ffi-arg-type result)
      ,@(encode-ffi-arg-list args))))
    
(defun save-byte-list (ptr l)
  (do* ((l l (cdr l))
        (i 0 (1+ i)))
       ((null l))
    (let* ((b (car l)))
      (if (typep b 'character)
        (setq b (char-code b)))
      (setf (%get-unsigned-byte ptr i) b))))

(defun db-write-byte-list (db-file keyname bytes)
  (let* ((len (length bytes)))
    (%stack-block ((p len))
      (save-byte-list p bytes)
      (rletZ ((contents :dbt)
	      (key :dbt))
	(with-cstrs ((keystring (string keyname)))
	  (setf (pref contents DBT.data) p
		(pref contents :DBT.size) len
		(pref key DBT.data) keystring
		(pref key :DBT.size) (length (string keyname)))
	  (ff-call (pref db-file :DB.put)
		   :address db-file
		   :address key
		   :address contents
		   :unsigned-fullword 0
		   :signed-fullword))))))

(defun save-ffi-function (db-file fun)
  (let* ((encoding (encode-ffi-function fun)))
    (db-write-byte-list db-file
			(string-upcase (ffi-function-string fun))
			encoding)))

(defun save-ffi-typedef (db-file def)
  (db-write-byte-list db-file
                       (ffi-typedef-name def)
                       (encode-ffi-type (ffi-typedef-type def))))

(defun save-ffi-struct (db-file s)
  (db-write-byte-list db-file (ffi-struct-reference s) (encode-ffi-struct s)))

(defun save-ffi-union (db-file u)
  (db-write-byte-list db-file (ffi-union-reference u) (encode-ffi-union u)))

(defun %decode-name (buf p)
  (declare (type macptr buf) (fixnum p))
  (let* ((n (%get-unsigned-byte buf p)))
    (declare (fixnum n))
    (if (zerop n)
      (values nil 1)
      (let* ((pname (make-string n)))
        (%copy-ptr-to-ivector buf (1+ p) pname 0 n)
        (values (intern pname *keyword-package*)
                (+ p (1+ n)))))))

(defun %decode-u32 (buf p)
  (declare (fixnum p) (type macptr buf))
  (values (dpb
           (%get-unsigned-byte buf p)
           (byte 8 24)
           (dpb
            (%get-unsigned-byte buf (+ p 1))
            (byte 8 16)
            (dpb
             (%get-unsigned-byte buf (+ p 2))
             (byte 8 8)
             (%get-unsigned-byte buf (+ p 3)))))
          (+ p 4)))
  
;; Should return a FOREIGN-TYPE structure.
(defun %decode-type (buf p)
  (declare (type macptr buf) (fixnum p))
  (let* ((q (1+ p)))
    (ecase (%get-unsigned-byte buf p)
      (#.encoded-type-void (values (parse-foreign-type :void) q))
      (#.encoded-type-signed-32 (values (svref *signed-integer-types* 32) q))
      (#.encoded-type-unsigned-32 (values (svref *unsigned-integer-types* 32) q))
      (#.encoded-type-signed-8 (values (svref *signed-integer-types* 8) q))
      (#.encoded-type-unsigned-8 (values (svref *unsigned-integer-types* 8) q))
      (#.encoded-type-signed-16 (values (svref *signed-integer-types* 16) q))
      (#.encoded-type-unsigned-16 (values (svref *unsigned-integer-types* 16) q))
      (#.encoded-type-signed-n (values (make-foreign-integer-type
                                          :signed t
                                          :bits (%get-unsigned-byte buf q))
                                         (1+ q)))
      (#.encoded-type-unsigned-n (values (make-foreign-integer-type
                                            :signed nil
                                            :bits (%get-unsigned-byte buf q))
                                           (1+ q)))
      (#.encoded-type-single-float (values (parse-foreign-type :float) q))
      (#.encoded-type-double-float (values (parse-foreign-type :double) q))
      (#.encoded-type-pointer (multiple-value-bind (target qq)
                                    (if (eql (%get-unsigned-byte buf q)
                                             encoded-type-void)
                                      (values nil (1+ q))
                                      (%decode-type buf q))
                                  (values (make-foreign-pointer-type :to target)
                                          qq)))
      (#.encoded-type-array
       (multiple-value-bind (size qq) (%decode-u32 buf q)
         (multiple-value-bind (target qqq) (%decode-type buf qq)
           (let* ((type-alignment (foreign-type-alignment target))
                  (type-bits (foreign-type-bits target)))
             (values (make-foreign-array-type
                      :element-type target
                      :dimensions (list size)
                      :alignment type-alignment
                      :bits (if type-bits
                              (* (align-offset type-bits type-alignment) size)))
                     qqq)))))
      (#.encoded-type-named-type-ref
       (multiple-value-bind (name qq) (%decode-name buf q)
         (values (%parse-foreign-type name) qq)))
      (#.encoded-type-named-struct-ref
       (multiple-value-bind (name qq) (%decode-name buf q)
         (values (or (info-foreign-type-struct name)
                     (setf (info-foreign-type-struct name)
                           (make-foreign-record-type :kind :struct
                                                     :name name)))
                 qq)))
      (#.encoded-type-named-union-ref
       (multiple-value-bind (name qq) (%decode-name buf q)
         (values (or (info-foreign-type-union name)
                     (setf (info-foreign-type-union name)
                           (make-foreign-record-type :kind :union
                                                     :name name)))
                 qq)))
      ((#.encoded-type-anon-struct-ref #.encoded-type-anon-union-ref)
       (multiple-value-bind (tag qq) (%decode-name buf q)
         (values (load-record tag) qq))))))


(defun extract-db-type (dbt)
  (with-macptrs ((dptr))
    (%setf-macptr dptr (pref dbt DBT.data))
    (unless (%null-ptr-p dptr)
      (%decode-type dptr 0))))

(defun %load-foreign-type (db-file name)
  (with-cstrs ((string (string name)))
    (rletZ ((contents :DBT)
	    (key :DBT))
      (setf (pref key :DBT.size) (length (string name))
            (pref key DBT.data) string
            (pref contents DBT.data) (%null-ptr)
            (pref contents :DBT.size) 0)
      (if (eql 0 (ff-call (pref db-file :DB.get)
			  :address db-file
			  :address key
			  :address contents
			  :unsigned-fullword 0
			  :signed-fullword))
	(let* ((type (extract-db-type contents)))
	  (if type
	    (%def-foreign-type name type)))))))

(defun load-foreign-type (name)
  (do-interface-dirs (d)
    (let* ((type (%load-foreign-type (db-types d) name)))
      (when type (return type)))))

(defun %decode-field (buf p)
  (declare (type macptr buf) (fixnum p))
  (multiple-value-bind (name q) (%decode-name buf p)
    (multiple-value-bind (type r) (%decode-type buf q)
      (values (make-foreign-record-field :type type :name name) r))))

(defun %decode-field-list (buf p)
  (declare (type macptr buf) (fixnum p))
  (let* ((n (%get-unsigned-byte buf p))
         (fields nil))
    (incf p)
    (dotimes (i n (values (nreverse fields) p))
      (multiple-value-bind (field q) (%decode-field buf p)
        (push field fields)
        (setq p q)))))

(defun %determine-record-attributes (rtype parsed-fields)
  (let* ((total-bits 0)
         (overall-alignment 1)
         (kind (foreign-record-type-kind rtype)))
    (dolist (field parsed-fields)
      (let* ((field-type (foreign-record-field-type field))
             (bits (ensure-foreign-type-bits field-type))
             (alignment (foreign-type-alignment field-type)))
        (unless bits
          (error "Unknown size: ~S"
                 (unparse-foreign-type field-type)))
        (unless alignment
          (error "Unknown alignment: ~S"
                 (unparse-foreign-type field-type)))
        (setq overall-alignment (max overall-alignment alignment))
        (ecase kind
          (:struct (let* ((offset (align-offset total-bits alignment)))
                     (setf (foreign-record-field-offset field) offset)
                     (setq total-bits (+ offset bits))))
          (:union (setq total-bits (max total-bits bits))))))
    (setf (foreign-record-type-fields rtype) parsed-fields
          (foreign-record-type-alignment rtype) overall-alignment
          (foreign-record-type-bits rtype) (align-offset total-bits overall-alignment))
    rtype))

(defun %decode-record-type (buf p)
  (declare (type macptr buf) (fixnum p))
  (let* ((rcode (%get-unsigned-byte buf p)))
    (multiple-value-bind (name q)
        (case rcode
          ((#.encoded-type-anon-struct-ref #.encoded-type-anon-union-ref)
           (values nil (1+ p)))
          (t
           (%decode-name buf (1+ p))))
      (%determine-record-attributes
       (if name
         (if (eql rcode encoded-type-named-struct-ref)
           (or (info-foreign-type-struct name)
               (setf (info-foreign-type-struct name)
                     (make-foreign-record-type :kind :struct :name name)))
           (or (info-foreign-type-union name)
               (setf (info-foreign-type-union name)
                     (make-foreign-record-type :kind :union :name name))))
         (make-foreign-record-type
          :kind (if (eql rcode encoded-type-anon-struct-ref)
                  :struct
                  :union)
          :name name))
       (%decode-field-list buf q)))))

(defun extract-db-record (dbt)
  (with-macptrs ((dptr))
    (%setf-macptr dptr (pref dbt DBT.data))
    (unless (%null-ptr-p dptr)
      (%decode-record-type dptr 0))))

(defun %load-foreign-record (db-file name)
  (with-cstrs ((string (string name)))
    (rlet ((contents :dbt)
           (key :dbt))
      (setf (pref key :DBT.size) (length (string name))
            (pref key DBT.data) string
            (pref contents DBT.data) (%null-ptr)
            (pref contents :DBT.size) 0)
      (if (eql 0 (ff-call (pref db-file :DB.get)
			  :address db-file
			  :address key
			  :address contents
			  :unsigned-fullword 0
			  :signed-fullword))      
	(extract-db-record contents)))))

(defun load-record (name)
  (do-interface-dirs (d)
    (let* ((r (%load-foreign-record (db-records d) name)))
      (when r (return r)))))


