;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;

;; L1-stack-groups.lisp

(cl:in-package "CCL")

(defvar *bind-io-control-vars-per-process* nil
  "If true, bind I/O control variables per process")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Stack groups
;;;

; A stack group is tagged as a function.
; It's code vector trampolines to code that switches stack groups
(def-accessors (stack-group) %svref
  sg.code-vector                        ; The trampoline code
  sg.%funcall-stack-group               ; #'%funcall-stack-group
  ;-- Above here fixed due to code requirements
  sg.name                               ; a string
  sg.resumer                            ; my resumer, another stack group
  sg.cs-area                            ; control stack area
  sg.vs-area                            ; value stack area
  sg.ts-area                            ; temp stack area
  sg.cs-overflow-limit                  ; control stack overflow limit
  sg.threadID                           ; thread manager ID or #$kNoThreadID
  sg.cs-size                            ; initial control stack size
  sg.vs-size                            ; initial value stack size
  sg.ts-size                            ; initial temp stack size
  sg.maxsize                            ; unused, allow for stack segmentation
  sg.interrupt-function.args            ; for stack-group-interrupt
  sg.initial-function.args              ; so we can startup after save-application
  ;-- sg.lfun-bits must be last
  sg.lfun-bits                          ; lfun-bits
  sg.element-count
  )

; Set to true to make the Lisp startup with a single process.
; That process will be both event processor and the initial listener.
(defparameter *intentional-single-process-p*  t)

; This is true if *intentional-single-process-p* is true or the thread manager is not present.
; It is initialized by the def-ccl-pointers below.
(defvar *single-process-p*)

; It already has a value of NIL when this code runs.
(defvar *thread-manager-present-p* nil)

; Allocate a tstack area with at least useable-bytes
; Returns a fixnum encoding the address of an area structure.
(defun allocate-tstack (useable-bytes)
  (with-macptrs ((tstack (ff-call (%kernel-import arch::kernel-import-allocate_tstack)
                                      :unsigned-fullword (logand (+ useable-bytes 4095) -4096)
                                      :address)))
    (when (%null-ptr-p tstack)
      (error "Can't allocate tstack"))
    (%fixnum-from-macptr tstack)))



; Allocate a vstack area with at least useable-bytes
; Returns a fixnum encoding the address of an area structure.
(defun allocate-vstack (useable-bytes)
  (with-macptrs ((vstack (ff-call (%kernel-import arch::kernel-import-allocate_vstack)
                                      :unsigned-fullword (logand (+ useable-bytes 4095) -4096)
                                      :address)))
    (when (%null-ptr-p vstack)
      (error "Can't allocate vstack"))
    (%fixnum-from-macptr vstack)))



; Create a new, empty control stack area
; Returns a fixnum encoding the address of an area structure.
(defun new-cstack-area ()
  (with-macptrs ((cstack (ff-call (%kernel-import arch::kernel-import-register_cstack)
                                      :unsigned-fullword 0   ; address
                                      :unsigned-fullword 0   ; size
                                      :address)))
    (when (%null-ptr-p cstack)
      (error "Can't allocate cstack"))
    ; Prevent stack overflow of infant stack group
    ; (Actually, I don't think this is necessary)
    (setf (pref cstack :gc-area.softlimit) (%null-ptr)
          (pref cstack :gc-area.hardlimit) (%null-ptr))
    (%fixnum-from-macptr cstack)))


; Free the result of allocate-tstack, allocate-vstack, or register-cstack
(defun free-stack-area (stack-area)
  (with-macptrs ((area-ptr (%null-ptr)))
    (%setf-macptr-to-object area-ptr stack-area)
    (ff-call (%kernel-import arch::kernel-import-condemn-area)
                 :address area-ptr
                 :void))
  nil)

(def-ccl-pointers *thread-manager-present-p* ()
  (setq *thread-manager-present-p*
        (not (zerop (the integer (%kernel-import arch::kernel-import-newthread)))))
  (setq *single-process-p*
        (or *intentional-single-process-p*
            (not *thread-manager-present-p*))))

(defun %kernel-global-offset (name-or-offset)
  (if (fixnump name-or-offset)
    name-or-offset
    (arch::%kernel-global name-or-offset)))

(defun %kernel-global-offset-form (name-or-offset-form)
  (cond ((and (listp name-or-offset-form)
              (eq 'quote (car name-or-offset-form))
              (listp (cdr name-or-offset-form))
              (symbolp (cadr name-or-offset-form))
              (null (cddr name-or-offset-form)))
         (arch::%kernel-global (cadr name-or-offset-form)))
        ((fixnump name-or-offset-form)
         name-or-offset-form)
        (t `(%kernel-global-offset ,name-or-offset-form))))

; This behaves like a function, but looks up the kernel global
; at compile time if possible. Probably should be done as a function
; and a compiler macro, but we can't define compiler macros yet,
; and I don't want to add it to "ccl:compiler;optimizers.lisp"
(defmacro %get-kernel-global (name-or-offset)
  `(%get-kernel-global-from-offset ,(%kernel-global-offset-form name-or-offset)))

(defmacro %get-kernel-global-ptr (name-or-offset dest)
  `(%get-kernel-global-ptr-from-offset ,(%kernel-global-offset-form name-or-offset) ,dest))

(defmacro %set-kernel-global (name-or-offset new-value)
  `(%set-kernel-global-from-offset
    ,(%kernel-global-offset-form name-or-offset)
    ,new-value))



; If you change this, you need to change the arch::%sg.funcall-stack-group
; slot in all stack groups, or they won't get the new definition.
(defun %funcall-stack-group (stack-group arg)
  (setf (sg.resumer stack-group) *current-stack-group*)
  (stack-group-resume stack-group arg))

(setf (sg.%funcall-stack-group #'%stack-group-trampoline) #'%funcall-stack-group)

(defvar *stack-group-code-vector*
  (%svref #'%stack-group-trampoline 0))



(defun %cons-stack-group (name)
  (let ((sg (make-uvector sg.element-count arch::subtag-function
                          :initial-element nil)))
    (setf (sg.code-vector sg) *stack-group-code-vector*
          (sg.%funcall-stack-group sg) #'%funcall-stack-group
          (sg.name sg) name
          (sg.threadID sg) kNoThreadID
          (sg.lfun-bits sg) (logior (ash 1 $lfbits-trampoline-bit)
                                         (dpb 1 $lfbits-numreq 0)))
    sg))



(defun stack-group-p (sg)
  (and (functionp sg)
       (eq (%svref sg 0) *stack-group-code-vector*)))



(setf (type-predicate 'stack-group) 'stack-group-p)

(defmethod print-object ((sg stack-group) stream)
  (print-stack-group sg stream nil))



(defun print-stack-group (sg stream suppress-address)
  (print-unreadable-object (sg stream :type t :identity (not suppress-address))
    (format stream "~s" (sg.name sg))))

; The number of bytes in a consing (or stack) area
(defun %area-size (area)
  (ash (- (%fixnum-ref area arch::area.high)
          (%fixnum-ref area arch::area.low))
       2))



(defvar *current-stack-group*)

(unless *current-stack-group*
  (setq *current-stack-group* (%cons-stack-group "Initial")))

(defvar *initial-stack-group* *current-stack-group*)



; The stack group that will be active on "return" from #_YieldToThread
(defvar *next-stack-group* nil)

; The argument passed to stack-group-resume
(defvar *resume-stack-group-arg* nil)



; 32K fails to compile this file
(defparameter *min-sp-stack-size* (ash 16 10))
(defparameter *min-vsp-stack-size* (ash 16 10))
(defparameter *min-tsp-stack-size* (ash 32 10))

(defvar *stack-group-population*
  (%cons-population (list *initial-stack-group*) $population_weak-list t))



(defun terminate-stack-groups ()
  (let ((population *stack-group-population*)
        list)
    (loop
      (without-interrupts
       (setq list (population-termination-list population))
       (unless list (return))
       (setf (population-termination-list population) (cdr list)
             (cdr list) nil))
      (kill-stack-group (car list)))
    (delete-unused-stack-areas)))


; This needs to be in LAP so that it won't vpush anything
(defun %db-link-chain-in-area-p (area &optional (sg *current-stack-group*))
  (let ((db (db-link sg))
        (high (%fixnum-ref area arch::area.high))
        (low (%fixnum-ref area arch::area.low)))
    (declare (fixnum db high low))
    (loop
      (when (eql 0 db) (return nil))
      (when (and (<= low db) (< db high))
        (return t))
      (setq db (%fixnum-ref db)))))


; Don't free stack oreas that contain part of the db_link chain.
(defun delete-unused-stack-areas ()
  (without-interrupts
   (do-unexhausted-stack-groups (sg)
     (macrolet ((do-area (sg.area &optional check-db-link)
                  `(let* ((current-p (eq sg *current-stack-group*))
                          area younger ,@(and check-db-link '(a)))
                     ; It's important that if sg is the current stack group,
                     ; then this code does no vsp or tsp pushes until the free-stack-area call.
                     (when current-p
                       (%normalize-areas))
                     (setq area (,sg.area sg)
                           younger (%fixnum-ref area arch::area.younger))
                     (unless (eql younger 0)
                       (unless ,(when check-db-link
                                  `(progn
                                     (setq a younger)
                                     (loop
                                       (when (if current-p
                                               (%db-link-chain-in-current-sg-area a)
                                               (%db-link-chain-in-area-p a sg))
                                         (return t))
                                       (setq a (%fixnum-ref a arch::area.younger))
                                       (when (eql a 0)
                                         (return nil)))))
                         (%fixnum-set area arch::area.younger 0)
                         (%fixnum-set younger arch::area.older 0)
                         (free-stack-area younger))))))
       (do-area sg.ts-area)
       (do-area sg.vs-area t)
       (%free-younger-cs-areas (sg.cs-area sg))
       ))))


; This makes a stack group only.
; Allocating the stacks and creating the thread is done by stack-group-preset
; now we have 3 optional args stack-size args
(defun make-stack-group (name &optional (stack-size 16384)(vstack-size stack-size)
                              (tstack-size stack-size))       ; default is 16K for each stack
  (setq name (require-type name 'string))
  (unless (and (fixnump stack-size) (> (the fixnum stack-size) 0))
    (setq stack-size (require-type stack-size '(and fixnum (integer 0 *)))))
  (unless *thread-manager-present-p*
    (error "The thread manager is not present"))
  (let* ((sg (%cons-stack-group name)))
         ;(each-stack-size (ceiling stack-size 3)))    
    ; listener total is 184k vs 128k in 3.0
    ; min process is 32 + 32 vs 32 in 3.0 and 32 + 32 in current 4.0 
    (setf (sg.cs-size sg) (max *min-sp-stack-size* stack-size)
          (sg.vs-size sg) (max *min-vsp-stack-size* vstack-size)
          (sg.ts-size sg) (max *min-tsp-stack-size* tstack-size)
          (sg.maxsize sg) (+ (sg.cs-size sg) (sg.vs-size sg) (sg.ts-size sg)))
    (push sg (population-data *stack-group-population*))
    sg))



; This is mostly for compatibility, but is a useful way to type-check a stack-group arg
(defun sg-buffer (sg)
  (unless (stack-group-p sg)
    (setq sg (require-type sg 'stack-group)))
  sg)



; Maybe this should return the real size of the areas, not the
; allocation request size (which gets rounded up to the real size)
(defun stack-group-size (&optional (stack-group *current-stack-group*))
  (sg-buffer stack-group)               ; type check
  (if (%stack-group-exhausted-p stack-group)
    (sg.maxsize stack-group)
    (multiple-value-call '+ (%stack-group-stack-space *current-stack-group*))))


(defun stack-group-maximum-size (&optional (sg *current-stack-group*))
  (sg-buffer sg)                        ; type check
  (sg.maxsize sg))



(defun (setf stack-group-maximum-size) (value &optional (sg *current-stack-group*))
  (sg-buffer sg)                        ; type check
  (setq value (require-type value 'fixnum))
  (setf (sg.maxsize sg) value))



(defparameter *initial-tsp-stack-segment-size* (* 64 1024))
(defparameter *initial-vsp-stack-segment-size* (* 64 1024))

(defmacro with-area-macptr ((var area) &body body)
  `(with-macptrs (,var)
     (%setf-macptr-to-object ,var ,area)
     ,@body))



; Store the threadid in the threshold slot of the area structure.
(defun gc-area.threadid (area)
  (with-area-macptr (p area)
    (%get-signed-long p arch::area.threshold)))


(defun (setf gc-area.threadid) (threadid area)
  (with-area-macptr (p area)
    (setf (%get-signed-long p arch::area.threshold)
          threadid)))


(defun gc-area.return-sp (area)
  (%fixnum-ref area arch::area.gc-count))


(defun (setf gc-area.return-sp) (return-sp area)
  (setf (%fixnum-ref area arch::area.gc-count) return-sp))



(defun stack-group-preset (sg function &rest args)
  (declare (dynamic-extent args))
  (sg-buffer sg)                        ; type check
  (when (eq sg *initial-stack-group*)
    (error "Can't preset initial stack group"))
  (unless (eq sg *current-stack-group*)
    (setq function (require-type function 'function))
    (without-interrupts
      (let ((top-listener nil))
        (unless (eql kNoThreadID (sg.threadID sg))
          (setq top-listener (symbol-value-in-stack-group '*top-listener* sg))
          (%kill-stack-group sg))       ; easier than trying to munge the active state
        (let* ((vs-area nil)
               (ts-area nil)
               (cs-area nil)
               (thread-id nil)
               (fn.args (cheap-cons function (cheap-copy-list args))))
          (unwind-protect
            (progn
              (setq vs-area (allocate-vstack *initial-vsp-stack-segment-size*))
              (setq ts-area (allocate-tstack *initial-tsp-stack-segment-size*))
              (setq cs-area (new-cstack-area))
              (setq thread-id (new-stack-group-thread (sg.cs-size sg)))
              (setf (gc-area.threadid cs-area) thread-id)
              (setf (sg.threadID sg) thread-id
                    (sg.vs-area sg) vs-area
                    (sg.ts-area sg) ts-area
                    (sg.cs-area sg) cs-area
                    (sg.cs-overflow-limit sg) 0
                    (sg.initial-function.args sg) fn.args))
            (unless thread-id
              (when vs-area
                (free-stack-area vs-area))
              (when ts-area
                (free-stack-area ts-area))
              (when cs-area
                (free-stack-area cs-area))))
          ; Start it up. This runs the threadEntry function.
          (funcall sg (cheap-cons top-listener fn.args))
          (setf (sg.resumer sg) nil))))))



(defun kill-stack-group (stack-group)
  (when (eq stack-group *current-stack-group*)
    (error "Attempt to kill *current-stack-group*"))
  (when (eq stack-group *initial-stack-group*)
    (error "Attempt to kill *initial-stack-group*"))
  (unless (%stack-group-exhausted-p stack-group)
    (%kill-stack-group stack-group)))

    

; Kill a stack group.
; Kill its thread.
; Unlink it from the *current-stack-group* chain.
; Free its stack areas.
; Only called on a stack group with an active thread.
; Assumes sg is not *current-stack-group*
(defun %kill-stack-group (sg &optional shutdown-p)
  (without-interrupts
   (let ((ts-area (sg.ts-area sg))
         (vs-area (sg.vs-area sg))
         (cs-area (sg.cs-area sg))
         (fn.args (sg.initial-function.args sg)))
     ; Clear state
     (setf (sg.ts-area sg) nil
           (sg.vs-area sg) nil
           (sg.cs-area sg) nil
           (sg.threadID sg) kNoThreadID
           (sg.resumer sg) nil)
     (unless shutdown-p
       (setf (sg.initial-function.args sg) nil)
       (when fn.args
         (cheap-free-list fn.args)))
     (free-stack-area ts-area)
     (free-stack-area vs-area)
     (let (older-cs-area)
       (loop
         (setq older-cs-area (%fixnum-ref cs-area arch::area.older))
         (when (eql 0 older-cs-area)
           (return))
         (setq cs-area older-cs-area)))
     (%free-younger-cs-areas cs-area t))))


(defun %free-younger-cs-areas (cs-area &optional (free-cs-area-too nil))
  (let (younger-cs-area)
    (loop
      (setf younger-cs-area (%fixnum-ref cs-area arch::area.younger)
            (%fixnum-ref cs-area arch::area.younger) 0)
      (when free-cs-area-too
        (progn (ff-call
                 (%kernel-import arch::kernel-import-disposethread)
                 :unsigned-fullword (gc-area.threadID cs-area)
                 :address (%null-ptr)
                 :unsigned-fullword 0
                 :signed-halfword))
        (setf (%fixnum-ref cs-area arch::area.older) 0)          ; free-stack-area frees the whole younger/older chain
        (free-stack-area cs-area))
      (when (eql 0 younger-cs-area) (return))
      (setq cs-area younger-cs-area)
      (setq free-cs-area-too t))))



(defun shutdown-stack-groups ()
  (dolist (sg (population-data *stack-group-population*))
    (unless (or (eq sg *initial-stack-group*)
                (eq sg *current-stack-group*)
                (%stack-group-exhausted-p sg))
      (%kill-stack-group sg t))))

(defun %current-xp ()
  (let ((xframe (%get-kernel-global 'xframe)))
    (when (eql xframe 0)
      (error "No current exception frame"))
    (%fixnum-ref xframe
                 (get-field-offset :xframe-list.this))))

(defun vsp-mismatch-error (sb was)
  (error "VSP mismatch after control stack overflow~%SB: #x~x, was: #x~x"
         (ash sb arch::fixnum-shift)
         (ash was arch::fixnum-shift)))

; the size of the active portion of a control stack segment
(defparameter *cs-segment-size* (* 64 1024))



; the size of the active portion of a value stack segment
(defparameter *vs-segment-size* (* 64 1024))



; the size of the active portion of a temp stack segment
(defparameter *ts-segment-size* (* 64 1024))



; How much space to leave for overflow on the control stack
(defparameter *cs-hard-overflow-size* 4096)
(defparameter *cs-soft-overflow-size* 4096)


(declaim (fixnum *cs-segment-size* *cs-hard-overflow-size* 
                 *cs-soft-overflow-size* *vs-soft-overflow-size* *ts-soft-overflow-size*))

(declaim (fixnum *cs-segment-size* *cs-hard-overflow-size* 
                 *cs-soft-overflow-size* *vs-soft-overflow-size* *ts-soft-overflow-size*))

; Create a new Thread Manager thread
(defun new-stack-group-thread (cs-size)
  (let ((stack-size (+ (logand (+ cs-size 4095) -4096)          ; round up to 4K multiple
                       *cs-hard-overflow-size*
                       *cs-soft-overflow-size*)))
    (rlet ((threadMade :long))
      (ff-call
       (%kernel-import arch::kernel-import-newthread)
       :unsigned-fullword kCooperativeThread             ; threadStyle
       :address *stack-group-startup-function*   ; threadEntry
       :address (%null-ptr)                      ; threadParam
       :unsigned-fullword stack-size                       ; stackSize
       :unsigned-fullword kCreateIfNeeded                ; options
       :address (%null-ptr)                      ; threadResult
       :address threadMade                       ; threadMade
       :signed-halfword
       )
      (%get-long threadMade))))



(defmacro with-self-bound-io-control-vars (&body body)
  `(let (; from CLtL2, table 22-7:
         (*package* *package*)
         (*print-array* *print-array*)
         (*print-base* *print-base*)
         (*print-case* *print-case*)
         (*print-circle* *print-circle*)
         (*print-escape* *print-escape*)
         (*print-gensym* *print-gensym*)
         (*print-length* *print-length*)
         (*print-level* *print-level*)
         (*print-lines* *print-lines*)
         (*print-miser-width* *print-miser-width*)
         (*print-pprint-dispatch* *print-pprint-dispatch*)
         (*print-pretty* *print-pretty*)
         (*print-radix* *print-radix*)
         (*print-readably* *print-readably*)
         (*print-right-margin* *print-right-margin*)
         (*read-base* *read-base*)
         (*read-default-float-format* *read-default-float-format*)
         (*read-eval* *read-eval*)
         (*read-suppress* *read-suppress*)
         (*readtable* *readtable*))
     ,@body))

; Enter here from the LAP startup code to apply the user function
; to the user args specified in stack-group-preset.
(defun %run-stack-group-function (sg sp)
  ; First we need to fix up the sg.cs-area and set the global cs-overflow-limit.
  (initialize-sg-cs-area sg sp)
  (let ((*top-listener* nil))
    (flet ((call-initial-stack-group-function ()
             (let* ((top-listener.fn.args *resume-stack-group-arg*)
                    (fn (cadr top-listener.fn.args))
                    (args (cddr top-listener.fn.args)))
               (setq *resume-stack-group-arg* nil)
               (setq sg nil)            ; So this stack group can be garbage collected
               (setq *top-listener* (car top-listener.fn.args))
               (free-cons top-listener.fn.args)
               (setq top-listener.fn.args nil)
               (stack-group-return nil)         ; return to stack-group-preset
               (setq *interrupt-level* 0)       ; was negative from stack-group-preset
               (handle-stack-group-interrupts)  ; may get interrupts before user startup
               (apply fn args))))
      (let* ((value (if *bind-io-control-vars-per-process*
                      (with-self-bound-io-control-vars
                        (call-initial-stack-group-function))
                      (call-initial-stack-group-function))))
        ; This stack group is exhausted.
        ; Let the initial stack group deallocate everything, and return the value to our resumer.
        (without-interrupts
         (stack-group-interrupt *initial-stack-group* nil
                                #'%stack-group-exit *current-stack-group* value)
         ; pass value here in case the resumer is the initial stack group
         (stack-group-resume *initial-stack-group* value)
         (error "This stack group should have been killed before it got here"))))))



; This runs in the *initial-stack-group* when a stack group is dying.
; It is responsible for killing the stack group and returning
; the value to its resumer.
(defun %stack-group-exit (sg value)
  (let ((resumer (sg.resumer sg)))
    (%kill-stack-group sg)
    (when resumer
      (stack-group-resume resumer value))))




(defun initialize-sg-cs-area (sg sp)
  (declare (fixnum sp))
  (rlet ((stack-space-ptr :long))
    (ff-call
     (%kernel-import arch::kernel-import-threadcurrentstackspace)
     :unsigned-fullword kCurrentThreadID
     :address stack-space-ptr
     :signed-halfword)
    (setf (%get-byte stack-space-ptr 3)
          (logand #xfc (%get-byte stack-space-ptr)))    ; ensure tagged as fixnum
    (let* ((stack-space (%get-object stack-space-ptr 0))
           (limit (- sp stack-space))
           (hard-limit (+ limit (the fixnum (ash *cs-hard-overflow-size* -2))))
           (soft-limit (+ hard-limit (the fixnum (ash *cs-soft-overflow-size* -2))))
           (cs-area (sg.cs-area sg))
           (ndwords (ash stack-space -1)))      ; longwords -> doublewords
      (declare (fixnum stack-space limit hard-limit soft-limit))
      (setf (%fixnum-ref cs-area arch::area.low) limit)
      (setf (%fixnum-ref cs-area arch::area.high) sp)
      (setf (%fixnum-ref cs-area arch::area.softlimit) soft-limit)
      (setf (%fixnum-ref cs-area arch::area.hardlimit) hard-limit)
      (with-macptrs (cs-area-ptr)
        (%setf-macptr-to-object cs-area-ptr cs-area)
        (setf (%get-long cs-area-ptr arch::area.ndwords) ndwords))
      (%set-kernel-global 'arch::cs-overflow-limit soft-limit))))


; This is the ONLY way to switch stack groups.
; If anyone else calls #_YieldToThread, things will break.
; C code can have its own threads, but it must call back
; to MCL from the same thread that called it, and it must
; not switch to an MCL thread.
(defun stack-group-resume (stack-group arg)
  (sg-buffer stack-group)               ; type check
  (when (eq stack-group *current-stack-group*)
    (return-from stack-group-resume arg))
  (when (%stack-group-exhausted-p stack-group)
    (error "Attempt to resume exhausted ~s" stack-group))
  (let ((YieldToThread (%kernel-import arch::kernel-import-yieldtothread))
        result)
    (without-interrupts
     (progn
       (catch '*inactive-stack-group-catch*
         (%ensure-vsp-stack-space)      ; make sure there's room for ff-call to save the saved registers
         (setq *next-stack-group* stack-group
               *resume-stack-group-arg* arg
               arg nil)                        ; allow GC
         (%reverse-special-bindings nil)
         (%normalize-areas)
         (%save-stack-group-context *current-stack-group*)
         ; (#_YieldToThread (sg.threadID stack-group))
         ; Do it explicitly to be doubly sure that it's resolved
         ; Calling back to resolve-slep-address in the current context is a bad idea
         (ff-call YieldToThread :signed-fullword (sg.threadID stack-group) :signed-halfword)
         (%restore-stack-group-context *next-stack-group*)
         (%reverse-special-bindings t))
       (setq result *resume-stack-group-arg*
             *current-stack-group* *next-stack-group*
             *next-stack-group* nil
             *resume-stack-group-arg* nil)))
      (multiple-value-bind (value value-p) (handle-stack-group-interrupts)
        (when value-p
          (setq result value)))
      result))

(defun handle-stack-group-interrupts ()
  (let (interrupt-function.args
        (current *current-stack-group*)
        result result-p)
    (loop
      (without-interrupts               ; paranoia
       (setq interrupt-function.args (sg.interrupt-function.args current))
       (unless interrupt-function.args
         (return))
       (setf (sg.interrupt-function.args current) nil))
      (let* ((resumer.err (car interrupt-function.args))
             (fn (cadr interrupt-function.args))
             (args (cddr interrupt-function.args))
             (done? (not resumer.err))          ; non-immediate interrupts can throw through here
             (value (block nil
                      (unwind-protect
                        (prog1
                          (apply fn args)
                          (setq done? t))
                        (unless done?
                          (return (cdr resumer.err)))))))
        (when resumer.err
          (progn
            (setq result (stack-group-resume (car resumer.err) value)
                  result-p t)
            (free-cons resumer.err))))
      (cheap-free-list interrupt-function.args))
    (values result result-p)))

(defun stack-group-return (arg)
  (let ((resumer (sg.resumer *current-stack-group*)))
    (unless resumer
      (error "~s has no resumer" *current-stack-group*))
    (stack-group-resume resumer arg)))


; If now-p is true, calls the function right away and returns
; what it returns. It will explicitly resume the caller.
; If an error occurs, it will be signalled and any attempt to
; throw out will cause now-p to be returned to the calling stack group.
; If now-p is false, queues up the function for running
; next time the scheduler schedules its process.
; In neither case should the function explicitly switch stack
; groups.
(defun stack-group-interrupt (sg now-p function &rest args)
  (declare (dynamic-extent args))
  (sg-buffer sg)                        ; type check
  (unless (functionp function)
    (setq function (require-type function 'function)))
  (if (eq sg *current-stack-group*)
    (apply function args)
    (without-interrupts
     (let-globally ((*in-scheduler* now-p))
       (let* ((old-fn.args (sg.interrupt-function.args sg))
              (resumer.err (and now-p (cheap-cons *current-stack-group* now-p)))
              (new-fn.args (cheap-cons resumer.err (cheap-cons function (cheap-copy-list args)))))
         (when old-fn.args
           (setq new-fn.args
                 (cheap-list resumer.err #'sg-nested-interrupt sg new-fn.args old-fn.args)))
         (setf (sg.interrupt-function.args sg) new-fn.args)
         (when now-p
           (stack-group-resume sg nil)))))))

(defun sg-nested-interrupt (sg new-fn.args old-fn.args)
  (let* ((fn (cadr new-fn.args))
         (args (cddr new-fn.args)))
    (prog1
      (apply fn args)
      (setf (sg.interrupt-function.args sg) old-fn.args)
      (cheap-free-list new-fn.args))))

(def-ccl-pointers *initial-stack-group* ()
  (setq *stack-group-startup-function*  threadEntry)
  (let ((sg *initial-stack-group*)
        (current-cs (%get-kernel-global 'current-cs))
        (current-vs (%get-kernel-global 'current-vs))
        (current-ts (%get-kernel-global 'current-ts)))
    (setf (sg.resumer sg) sg
          (sg.cs-area sg) current-cs
          (sg.vs-area sg) current-vs
          (sg.ts-area sg) current-ts
          (sg.cs-overflow-limit sg) (%get-kernel-global 'cs-overflow-limit)
          (sg.threadID sg) kApplicationThreadID
          (gc-area.threadid current-cs) kApplicationThreadID
          (sg.cs-size sg) (%area-size current-cs)
          (sg.vs-size sg) (max *min-vsp-stack-size* (%area-size current-vs))
          (sg.ts-size sg) (max *min-tsp-stack-size* (%area-size current-ts))
          (sg.maxsize sg) (+ (sg.cs-size sg) (sg.vs-size sg) (sg.ts-size sg)))))



(defun %stack-group-exhausted-p (sg)
  (eql (sg.threadID (sg-buffer sg)) kNoThreadID))

(defvar *canonical-error-value*
  '(*canonical-error-value*))



(defun %funcall-in-stack-group (sg thunk)
  (sg-buffer sg)                        ; type check
  (if (eq sg *current-stack-group*)
    (funcall thunk)
    (without-interrupts
     (let* ((sg-thunk #'(lambda (thunk)
                          (let ((values (multiple-value-list (funcall thunk))))
                            (declare (dynamic-extent values))
                            (cheap-copy-list values))))
            (values (let-globally ((*in-scheduler* t))
                      (stack-group-interrupt
                       sg *canonical-error-value* sg-thunk thunk))))
       (if (eq values *canonical-error-value*)
         (error "Error in stack-group-interrupt on ~s" sg)
         (multiple-value-prog1
           (apply 'values values)
           (cheap-free-list values)))))))

(defun symbol-value-in-stack-group (sym sg)
  (let ((loc (%symbol-value-locative-in-stack-group sym sg)))
    (if (null loc)
      (symbol-value sym)
      (let ((val (%fixnum-ref loc)))
        (when (eq val (%unbound-marker-8))
          (error "~s is unbound in ~s" sym sg))
        val))))

(defun (setf symbol-value-in-stack-group) (value sym sg)
  (let ((loc (%symbol-value-locative-in-stack-group sym sg)))
    (if (null loc)
      (setf (symbol-value sym) value)
      (setf (%fixnum-ref loc) value))))

(defun %symbol-value-locative-in-stack-group (sym sg)
  (sg-buffer sg)                        ; type check
  (if (eq sg *current-stack-group*)
    nil
    (or (%last-symbol-value-locative-in-db-chain
         sym (db-link sg))
        (%last-symbol-value-locative-in-db-chain
         sym (db-link)))))

(defun %last-symbol-value-locative-in-db-chain (sym db)
  (let ((last-found nil))
    (loop
      (when (eql 0 db) (return))
      (when (eq sym (%fixnum-ref db 4))
        (setq last-found db))
      (setq db (%fixnum-ref db 0)))
    (and last-found (%i+ last-found 2))))

; energized is the opposite of exhausted, right?
(defmacro do-unexhausted-stack-groups ((s) &body body)
  `(dolist (,s (population-data *stack-group-population*))
     (unless (%stack-group-exhausted-p ,s)
       ,@body)))



(defmacro do-inactive-stack-groups ((s) &body body)
  (let* ((current (gensym)))
    `(let ((,current *current-stack-group*))
       (do-unexhausted-stack-groups (,s)
         (unless (eq ,s ,current)
           ,@body)))))

; Start up the stack groups that were active at save-application time.
(def-ccl-pointers stack-groups ()
  (do-inactive-stack-groups (s)
    (unless (eq s *initial-stack-group*)
      (let* ((fn.args (sg.initial-function.args s)))
        (when (and (listp fn.args) (functionp (car fn.args)))
          (apply 'stack-group-preset s (car fn.args) (cdr fn.args)))))))

;;; Backtrace support
;;;

; Linked list of fake stack frames.
; %frame-backlink looks here
(defvar *fake-stack-frames* nil)

(def-accessors (fake-stack-frame) %svref
  nil                           ; 'fake-stack-frame
  %fake-stack-frame.sp          ; fixnum. The stack pointer where this frame "should" be
  %fake-stack-frame.next-sp     ; Either sp or another fake-stack-frame
  %fake-stack-frame.fn          ; The current function
  %fake-stack-frame.lr          ; fixnum offset from fn (nil if fn is not functionp)
  %fake-stack-frame.vsp         ; The value stack pointer
  %fake-stack-frame.link        ; next in *fake-stack-frames* list
  )
  
(defmacro %cons-fake-stack-frame (&optional sp next-sp fn lr vsp link)
  `(%istruct 'fake-stack-frame ,sp ,next-sp ,fn ,lr ,vsp ,link))



(defun fake-stack-frame-p (x)
  (istruct-typep x 'fake-stack-frame))

(set-type-predicate 'fake-stack-frame 'fake-stack-frame-p)

(defmacro do-db-links ((db-link &optional var value) &body body)
  (let ((thunk (gensym))
        (var-var (or var (gensym)))
        (value-var (or value (gensym))))
    `(block nil
       (let ((,thunk #'(lambda (,db-link ,var-var ,value-var)
                         (declare (ignorable ,db-link))
                         ,@(unless var (list `(declare (ignore ,var-var))))
                         ,@(unless value (list `(declare (ignore ,value-var))))
                         ,@body)))
         (declare (dynamic-extent ,thunk))
         (map-db-links ,thunk)))))

(defun map-db-links (f)
  (without-interrupts
   (let ((db-link (%fixnum-ref (%get-kernel-global 'db-link))))         ; skip the without-interrupts binding
     (loop
       (when (eql 0 db-link) (return))
       (funcall f db-link (%fixnum-ref db-link 4) (%fixnum-ref db-link 8))
       (setq db-link (%fixnum-ref db-link))))))

(defun %get-frame-ptr (&optional (stack-group *current-stack-group*))
  (sg-buffer stack-group)               ; type check
  (if (eq stack-group *current-stack-group*)
    (%current-frame-ptr)
    (%fixnum-ref (sg.cs-area stack-group) arch::area.active)))




(defun %stack< (index1 index2 &optional (sg *current-stack-group*))
  (cond ((fake-stack-frame-p index1)
         (let ((sp1 (%fake-stack-frame.sp index1)))
           (declare (fixnum sp1))
           (if (fake-stack-frame-p index2)
             (or (%stack< sp1 (%fake-stack-frame.sp index2) sg)
                 (eq index2 (%fake-stack-frame.next-sp index1)))
             (%stack< sp1 (%i+ index2 1) sg))))
        ((fake-stack-frame-p index2)
         (%stack< index1 (%fake-stack-frame.sp index2)))
        (t (let* ((cs-area (sg.cs-area sg)))
             (loop
               (when (%ptr-in-area-p index1 cs-area)
                 (return))
               (setq cs-area (%fixnum-ref cs-area arch::area.older))
               (when (eql 0 cs-area)
                 ; Should we signal an error here?
                 (return-from %stack< nil)))
             (if (%ptr-in-area-p index2 cs-area)
               (%i< index1 index2)
               (loop
                 (setq cs-area (%fixnum-ref cs-area arch::area.older))
                 (when (eql 0 cs-area)
                   (return nil))
                 (when (%ptr-in-area-p index2 cs-area)
                   (return t))))))))

(defun %frame-savefn (p)
  (if (fake-stack-frame-p p)
    (%fake-stack-frame.fn p)
    (%%frame-savefn p)))

(defun %frame-savevsp (p)
  (if (fake-stack-frame-p p)
    (%fake-stack-frame.vsp p)
    (%%frame-savevsp p)))

(defun frame-vsp (frame &optional parent-vsp sg)
  (declare (ignore parent-vsp sg))
  (%frame-savevsp frame))

(defun bottom-of-stack-p (p sg)
  (and (fixnump p)
       (locally (declare (fixnum p))
	 (let ((cs-area (sg.cs-area sg)))
	   (loop
	       (when (%ptr-in-area-p p cs-area)
		 (return nil))
	       (setq cs-area (%fixnum-ref cs-area arch::area.older))
	     (when (eql 0 cs-area)
	       (return t)))))))

(defun next-catch (catch)
  (let ((next-catch (uvref catch arch::catch-frame.link-cell)))
    (unless (eql next-catch 0) next-catch)))

(defun catch-frame-sp (catch)
  (uvref catch arch::catch-frame.csp-cell))

(defun catch-csp-p (p stack-group)
  (let ((catch (%catch-top stack-group)))
    (loop
      (when (null catch) (return nil))
      (let ((sp (catch-frame-sp catch)))
        (when (eql sp p)
          (return t)))
      (setq catch (next-catch catch)))))

; @@@ this needs to load early so errors can work
(defun next-lisp-frame (p stack-group)
  (let ((frame p))
    (loop
      (let ((parent (%frame-backlink frame stack-group)))
        (multiple-value-bind (lisp-frame-p bos-p) (lisp-frame-p parent stack-group)
          (if lisp-frame-p
            (return parent)
            (if bos-p
              (return nil))))
        (setq frame parent)))))

(defun parent-frame (p stack-group)
  (loop
    (let ((parent (next-lisp-frame p stack-group)))
      (when (or (null parent)
                (not (catch-csp-p parent stack-group)))
        (return parent))
      (setq p parent))))


; @@@ this needs to load early so errors can work
(defun cfp-lfun (p stack-group &optional child)
  (declare (ignore stack-group child))
  (if (fake-stack-frame-p p)
    (let ((fn (%fake-stack-frame.fn p))
          (lr (%fake-stack-frame.lr p)))
      (when (and (functionp fn) (fixnump lr))
        (values fn (%fake-stack-frame.lr p))))
    (without-interrupts                   ; Can't GC while we have lr in our hand
     (let ((fn (%frame-savefn p))
           (lr (%frame-savelr p)))
       (declare (fixnum lr))
       (when (functionp fn)
         (let* ((function-vector (%svref fn 0))
                (pc-words (- lr (the fixnum (%uvector-data-fixnum function-vector)))))
           (declare (fixnum pc-words))
           (when (and (>= pc-words 0) (< pc-words (uvsize function-vector)))
             (values fn
                     (the fixnum (ash pc-words arch::fixnum-shift))))))))))

(defun last-frame-ptr (&optional (stack-group *current-stack-group*))
  (let* ((current (%get-frame-ptr stack-group))
         (last current))
    (loop
      (setq current (parent-frame current stack-group))
      (if current
        (setq last current)
        (return last)))))



(defun child-frame (p sg)
  (let* ((current (%get-frame-ptr sg))
         (last nil))
    (loop
      (when (null current)
        (return nil))
      (when (eq current p) (return last))
      (setq last current
            current (parent-frame current sg)))))



; Used for printing only.
(defun index->address (p)
  (when (fake-stack-frame-p p)
    (setq p (%fake-stack-frame.sp p)))
  (ldb (byte 32 0)  (ash p arch::fixnumshift)))

; This returns the current head of the db-link chain.
; The db-link chain is reversed for other than the *current-stack-group*
(defun db-link (&optional (stack-group *current-stack-group*))
  (sg-buffer stack-group)               ; type check
  (if (eq stack-group *current-stack-group*)
    (%get-kernel-global 'db-link)
    (progn
      (when (%stack-group-exhausted-p stack-group)
        (error "~s is exhausted" stack-group))
      (%svref (%catch-top stack-group) arch::catch-frame.db-link-cell))))



(defun previous-db-link (db-link start &optional (stack-group *current-stack-group*))
  (declare (fixnum db-link start))
  (if (eq stack-group *current-stack-group*)
    (let ((prev nil))
      (loop
        (when (or (eql db-link start) (eql 0 start))
          (return prev))
        (setq prev start
              start (%fixnum-ref start 0))))
    (let ((prev (%fixnum-ref db-link)))
      (unless (eql prev 0) prev))))

(defun count-db-links-in-frame (vsp parent-vsp &optional (stack-group *current-stack-group*))
  (declare (fixnum vsp parent-vsp))
  (let ((db (db-link stack-group))
        (count 0)
        (first nil)
        (last nil)
        (current? (eq stack-group *current-stack-group*)))
    (declare (fixnum db count))
    (loop
      (cond ((eql db 0)
             (unless current?
               (rotatef first last))
             (return (values count (or first 0) (or last 0))))
            ((and (>= db vsp) (< db parent-vsp))
             (unless first (setq first db))
             (setq last db)
             (incf count)))
      (setq db (%fixnum-ref db)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; bogus-thing-p support
;;;

(defun %ptr-in-area-p (ptr area)
  (declare (fixnum ptr area))
  (and (<= (the fixnum (%fixnum-ref area arch::area.low)) ptr)
       (>= (the fixnum (%fixnum-ref area arch::area.high)) ptr)))

(defun %active-area (area active)
  (or (do ((a area (%fixnum-ref a arch::area.older)))
          ((eql a 0))
        (when (%ptr-in-area-p active a)
          (return a)))
      (do ((a (%fixnum-ref area arch::area.younger) (%fixnum-ref a arch::area.younger)))
          ((eql a 0))
        (when (%ptr-in-area-p active a)
          (return a)))))

(defun %ptr-to-vstack-p (sg idx)
  (declare (fixnum idx))
  (sg-buffer sg)                        ; type check
  (let* ((vs-area (%active-area (sg.vs-area sg) idx)))
    (when vs-area
      (let ((active (if (and (eq sg *current-stack-group*)
                             (%ptr-in-area-p (%current-vsp) vs-area))
                      (%current-vsp)
                      (%fixnum-ref vs-area arch::area.active)))
            (high (%fixnum-ref vs-area arch::area.high)))
        (declare (fixnum active high))
        (and (< active idx)
             (< idx high))))))

(defun %on-tsp-stack (sg object)
  (declare (fixnum object))             ; lie
  (sg-buffer sg)                        ; type check
  (let* ((ts-area (%active-area (sg.ts-area sg) object)))
    (when ts-area
      (let ((active (if (and (eq sg *current-stack-group*)
                             (%ptr-in-area-p (%current-tsp) ts-area))
                      (%current-tsp)
                      (%fixnum-ref ts-area arch::area.active)))
            (high (%fixnum-ref ts-area arch::area.high)))
        (declare (fixnum active high))
        (and (< active object)
             (< object high))))))

(defun on-any-tsp-stack (object)
  (or (%on-tsp-stack *current-stack-group* object)
      (do-inactive-stack-groups (sg)
        (when (%on-tsp-stack sg object)
          (return t)))))

(defun on-any-vstack (idx)
  (or (%ptr-to-vstack-p *current-stack-group* idx)
      (do-inactive-stack-groups (sg)
        (when (%ptr-to-vstack-p sg idx)
          (return t)))))

; This MUST return either T or NIL.
(defun temporary-cons-p (x)
  (and (consp x)
       (not (null (or (on-any-vstack x)
                      (on-any-tsp-stack x))))))

(defmacro do-gc-areas ((area) &body body)
  (let ((initial-area (gensym)))
    `(let* ((,initial-area (%get-kernel-global 'all-areas))
            (,area ,initial-area))
       (declare (fixnum ,initial-area ,area))
       (loop
         (setq ,area (%fixnum-ref ,area arch::area.succ))
         (when (eql ,area ,initial-area)
           (return))
         ,@body))))

(defmacro do-consing-areas ((area) &body body)
  (let ((code (gensym)))
  `(do-gc-areas (,area)
     (let ((,code (%fixnum-ref ,area arch::area.code)))
       (when (or (eql ,code arch::area-readonly)
                 (eql ,code arch::area-staticlib)
                 (eql ,code arch::area-static)
                 (eql ,code arch::area-dynamic))
         ,@body)))))



(defun %value-cell-header-at-p (cur-vsp)
  (eql arch::value-cell-header (%fixnum-address-of (%fixnum-ref cur-vsp))))

(defun count-stack-consed-value-cells-in-frame (vsp parent-vsp sg)
  (declare (ignore sg))                 ; use when we have multiple stack segments
  (let ((cur-vsp vsp)
        (count 0))
    (declare (fixnum cur-vsp count))
    (loop
      (when (>= cur-vsp parent-vsp) (return))
      (when (and (evenp cur-vsp) (%value-cell-header-at-p cur-vsp))
        (incf count)
        (incf cur-vsp))                 ; don't need to check value after header
      (incf cur-vsp))
    count))

; stack consed value cells are one of two forms:
;
; nil             ; n-4
; header          ; n = even address (multiple of 8)
; value           ; n+4
;
; header          ; n = even address (multiple of 8)
; value           ; n+4
; nil             ; n+8

(defun in-stack-consed-value-cell-p (arg-vsp vsp parent-vsp sg)
  (declare (ignore sg))                 ; use when we have multiple stack segments
  (declare (fixnum arg-vsp vsp parent-vsp))
  (if (evenp arg-vsp)
    (%value-cell-header-at-p arg-vsp)
    (or (and (> arg-vsp vsp)
             (%value-cell-header-at-p (the fixnum (1- arg-vsp))))
        (let ((next-vsp (1+ arg-vsp)))
          (declare (fixnum next-vsp))
          (and (< next-vsp parent-vsp)
               (%value-cell-header-at-p next-vsp))))))

; Return two values: the vsp of p and the vsp of p's "parent" frame.
; The "parent" frame vsp might actually be the end of p's segment,
; if the real "parent" frame vsp is in another segment.
(defun vsp-limits (p stack-group)
  (let* ((vsp (%frame-savevsp p))
         parent)
    (when (eql vsp 0)
      ; This frame is where the code continues after an unwind-protect cleanup form
      (setq vsp (%frame-savevsp (child-frame p stack-group))))
    (flet ((grand-parent (frame)
             (let ((parent (parent-frame frame stack-group)))
               (when (and parent (eq parent (%frame-backlink frame)))
                 (let ((grand-parent (parent-frame parent stack-group)))
                   (when (and grand-parent (eq grand-parent (%frame-backlink parent)))
                     grand-parent))))))
      (declare (dynamic-extent #'grand-parent))
      (let* ((frame p)
             grand-parent)
        (loop
          (setq grand-parent (grand-parent frame))
          (when (or (null grand-parent) (not (eql 0 (%frame-savevsp grand-parent))))
            (return))
          (setq frame grand-parent))
        (setq parent (parent-frame frame stack-group)))
      (let ((parent-vsp (if parent (%frame-savevsp parent) vsp))
            (vsp-area (%active-area (sg.vs-area stack-group) vsp)))
        (if (eql 0 parent-vsp)
          (values vsp vsp)              ; p is the kernel frame pushed by an unwind-protect cleanup form
          (progn
            (unless vsp-area
              (error "~s is not a stack frame pointer for ~s" p stack-group))
            (unless (%ptr-in-area-p parent-vsp vsp-area)
              (setq parent-vsp (%fixnum-ref vsp-area arch::area.high)))
            (values vsp parent-vsp)))))))

(defun count-values-in-frame (p stack-group &optional child)
  (declare (ignore child))
  (multiple-value-bind (vsp parent-vsp) (vsp-limits p stack-group)
    (values
     (- parent-vsp 
        vsp
        (* 2 (count-db-links-in-frame vsp parent-vsp stack-group))
        (* 3 (count-stack-consed-value-cells-in-frame vsp parent-vsp stack-group))))))

(defun nth-value-in-frame-loc (sp n sg lfun pc child-frame vsp parent-vsp)
  (declare (ignore child-frame))        ; no ppc function info yet
  (declare (fixnum sp))
  (setq n (require-type n 'fixnum))
  (unless (or (null vsp) (fixnump vsp))
    (setq vsp (require-type vsp '(or null fixnum))))
  (unless (or (null parent-vsp) (fixnump parent-vsp))
    (setq parent-vsp (require-type parent-vsp '(or null fixnum))))
  (unless (and vsp parent-vsp)
    (multiple-value-setq (vsp parent-vsp) (vsp-limits sp sg)))
  (locally (declare (fixnum n vsp parent-vsp))
    (multiple-value-bind (db-count first-db last-db)
                         (count-db-links-in-frame vsp parent-vsp sg)
      (declare (ignore db-count))
      (declare (fixnum first-db last-db))
      (let ((arg-vsp (1- parent-vsp))
            (cnt n)
            (phys-cell 0)
            db-link-p)
        (declare (fixnum arg-vsp cnt phys-cell))
        (loop
          (if (eql (the fixnum (- arg-vsp 2)) last-db)
            (setq db-link-p t
                  arg-vsp last-db
                  last-db (previous-db-link last-db first-db sg)
                  phys-cell (+ phys-cell 2))
            (setq db-link-p nil))
          (unless (in-stack-consed-value-cell-p arg-vsp vsp parent-vsp sg)
            (when (< (decf cnt) 0)
              (return
               (if db-link-p
                 ; Really ought to find the next binding if not the current sg, but
                 ; noone has complained about this bug before, so why fix it?
                 (values (+ 2 arg-vsp)
                         :saved-special
                         (%fixnum-ref (1+ arg-vsp)))
                 (multiple-value-bind (type name) (find-local-name phys-cell lfun pc)
                   (values arg-vsp type name))))))
          (incf phys-cell)
          (when (< (decf arg-vsp) vsp)
            (error "n out of range")))))))

(defun nth-value-in-frame (sp n sg &optional lfun pc child-frame vsp parent-vsp)
  (multiple-value-bind (loc type name)
                       (nth-value-in-frame-loc sp n sg lfun pc child-frame vsp parent-vsp)
    (values (%fixnum-ref loc) type name)))

(defun set-nth-value-in-frame (sp n sg new-value &optional child-frame vsp parent-vsp)
  (let ((loc (nth-value-in-frame-loc sp n sg nil nil child-frame vsp parent-vsp)))
    (setf (%fixnum-ref loc) new-value)))

; True if the object is in one of the heap areas
(defun %in-consing-area-p (x area)
  (declare (fixnum x))                  ; lie
  (let* ((low (%fixnum-ref area arch::area.low))
         (high (%fixnum-ref area arch::area.high))
         (active (%fixnum-ref area arch::area.active))
         (freeptr (%get-freeptr))
         (curptr (if (and (<= low freeptr)
                          (<= freeptr high))
                   freeptr
                   active)))
    (declare (fixnum low high active freeptr curptr))
    (and (<= low x) (< x curptr))))



(defun in-any-consing-area-p (x)
  (do-consing-areas (area)
    (when (%in-consing-area-p x area)
      (return t))))

(defun valid-subtag-p (subtag)
  (declare (fixnum subtag))
  (let* ((tagval (ldb (byte (- arch::num-subtag-bits arch::ntagbits) arch::ntagbits) subtag)))
    (declare (fixnum tagval))
    (case (logand subtag arch::fulltagmask)
      (#. arch::fulltag-immheader (not (eq (%svref *immheader-types* tagval) 'bogus)))
      (#. arch::fulltag-nodeheader (not (eq (%svref *nodeheader-types* tagval) 'bogus)))
      (t nil))))



(defun valid-header-p (thing)
  (let* ((fulltag (fulltag thing)))
    (declare (fixnum fulltag))
    (case fulltag
      (#.arch::fulltag-misc (valid-subtag-p (typecode thing)))
      ((#.arch::fulltag-immheader #.arch::fulltag-nodeheader) nil)
      (t t))))




(defun bogus-thing-p (x)
  (when x
    (or (not (valid-header-p x))
        (let ((tag (lisptag x)))
          (unless (or (eql tag arch::tag-fixnum)
                      (eql tag arch::tag-imm)
                      (in-any-consing-area-p x))
            ; Might be in Multifinder heap.  Hard to tell.
            ; Make sure it's in RAM somewhere, more or less.
            (or (< (the fixnum x) 0)
                ; This is terribly complicated, should probably write some LAP

                (let ((typecode (typecode x)))
                  (not (or (memq x *heap-ivectors*)
                           (case typecode
                             (#.arch::tag-list
                              (temporary-cons-p x))
                             ((#.arch::subtag-symbol #.arch::subtag-code-vector)
                              t)              ; no stack-consed symbols or code vectors
                             (#.arch::subtag-value-cell
                              (on-any-vstack x))
                             (t
                              (on-any-tsp-stack x))))))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; terminate-when-unreachable
;;;

#|
Message-Id: <v02130502ad3e6a2f1542@[205.231.144.48]>
Mime-Version: 1.0
Content-Type: text/plain; charset="us-ascii"
Date: Wed, 7 Feb 1996 10:32:55 -0500
To: pmcldev@digitool.com
From: bitCraft@taconic.net (Bill St. Clair)
Subject: terminate-when-unreachable

I propose that we add a general termination mechanism to PPC MCL.
We need it to properly terminate stack groups, it would be
a nicer way to do the termination for macptrs than the current
ad-hoc mechanism (which BTW is not yet part of PPC MCL), and
it is a nice addition to MCL. I don't think it's hard to make
the garbage collector support this, and I volunteer to do the
work unless Gary really wants to.

I see two ways to support termination:

1) Do termination for hash tables. This was our plan for
   2.0, but Gary got confused about how to mark the objects at
   the right time (or so I remember).

2) Resurrect weak alists (they're not part of the PPC garbage
   collector) and add a termination bit to the population type.
   This allows for termination of weak lists and weak alists,
   though the termination mechanism really only needs termination
   for a single weak alist.

I prefer option 2, weak alists, since it avoids the overhead
necessary to grow and rehash a hash table. It also uses less space,
since a finalizeable hash table needs to allocate two cons cells
for each entry so that the finalization code has some place to
put the deleted entry.

I propose the following interface (slightly modified from what
Apple Dylan provides):

terminate-when-unreachable object &optional (function 'terminate)
  When OBJECT becomes unreachable, funcall FUNCTION with OBJECT
  as a single argument. Each call of terminate-when-unreachable
  on a single (EQ) object registers a new termination function.
  All will be called when the object becomes unreachable.

terminate object                                         [generic function]
  The default termination function

terminate (object t)                                     [method]
  The default method. Ignores object. Returns nil.

drain-termination-queue                                  [function]
  Drain the termination queue. I.e. call the termination function
  for every object that has become unreachable.

*enable-automatic-termination*                           [variable]
  If true, the default, drain-termination-queue will be automatically
  called on the first event check after the garbage collector runs.
  If you set this to false, you are responsible for calling
  drain-termination-queue.

cancel-terminate-when-unreachable object &optional function
  Removes the effect of the last call to terminate-when-unreachable
  for OBJECT & FUNCTION (both tested with EQ). Returns true if
  it found a match (which it won't if the object has been moved
  to the termination queue since terminate-when-unreachable was called).
  If FUNCTION is NIL or unspecified, then it will not be used; the
  last call to terminate-when-unreachable with the given OBJECT will
  be undone.

termination-function object
  Return the function passed to the last call of terminate-when-unreachable
  for OBJECT. Will be NIL if the object has been put in the
  termination queue since terminate-when-unreachable was called.

|#


(defvar *termination-population*
  (%cons-terminatable-alist))

(defvar *enable-automatic-termination* t)

(defun terminate-when-unreachable (object &optional (function 'terminate))
  (let ((new-cell (list (cons object function)))
        (population *termination-population*))
    (without-interrupts
     (setf (cdr new-cell) (population-data population)
           (population-data population) new-cell))
    function))

(defmethod terminate ((object t))
  nil)

(defun drain-termination-queue ()
  (let ((cell nil)
        (population *termination-population*))
    (loop
      (without-interrupts
       (let ((list (population-termination-list population)))
         (unless list (return))
         (setf cell (car list)
               (population-termination-list population) (cdr list))))
      (funcall (cdr cell) (car cell)))))

(defun cancel-terminate-when-unreachable (object &optional (function nil function-p))
  (let ((found-it? nil))
    (flet ((test (object cell)
             (and (eq object (car cell))
                  (or (not function-p)
                      (eq function (cdr cell)))
                  (setq found-it? t))))
      (declare (dynamic-extent #'test))
      (without-interrupts
       (setf (population-data *termination-population*)
             (delete object (population-data *termination-population*)
                     :test #'test
                     :count 1)))
      found-it?)))

(defun termination-function (object)
  (cdr (assq object (population-data *termination-population*))))

(defun do-automatic-termination ()
  (when *enable-automatic-termination*
    (drain-termination-queue)))

(queue-fixup
 (add-gc-hook 'do-automatic-termination :post-gc)
 (add-gc-hook 'terminate-stack-groups :post-gc))

