// Copyright (C) 2005 Shai Ayal <shaiay@users.sourceforge.net>
// Copyright (C) 2007 Kai Habel <kai7000@gmx.net>
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  

#include <FL/gl.h>
#include <GL/glu.h>
#include "patch.h"
#include "axes.h"
#include "figure.h"
#include "line_plotter.h"
#include "mathutils.h"
#include <iostream>
#include "glutesscb.h"

#define OBJ_NAME "patch"

// callback routines for GLU Tesselator
void CALLBACK beginCallback(GLenum which)
{
  glBegin(which);
}

void CALLBACK endCallback(void)
{
  glEnd();
}

void CALLBACK edgeCallback(GLboolean flag)
{
  // empty
}

void CALLBACK errorCallback(GLenum errorCode)
{
  const GLubyte *estring;
  estring = gluErrorString(errorCode);
  std::cerr <<  "Tessellation Error: " << estring << std::endl;;
  exit (0);
}

void CALLBACK vertexCallback(GLvoid *vertex)
{
  const GLdouble *pointer;
  pointer = (GLdouble *) vertex;
  glColor4dv(pointer+3);
  glVertex3dv(pointer);
}

void CALLBACK combineCallback(GLdouble coords[3], GLdouble *d[4],
                     GLfloat w[4], GLdouble **dataOut )
{
  GLdouble *v = new GLdouble[7];
  // reset color
  for (unsigned int i = 3; i < 7; i++)
    v[i] = 0.0;
  // set xyz value
  for (unsigned int i = 0; i < 3; i++)
    v[i] = coords[i];
  for (unsigned int i = 3; i < 7; i++)
    for (unsigned int n = 0; n < 4; n++)
      // FIXME
      // the next 'if' should not be needed here, since the d-pointer
      // is according to the docs always valid, ...
      // but otherwise it dies for me,
      // [Kai]
      if (w[n] > 0.0)
        v[i] += w[n]*d[n][i];
  glColor4d(v[3],v[4],v[5],v[6]);
  *dataOut = v;
}

Patch::Patch(ocpl::Handle Parent, Matrix& XData, Matrix& YData, Matrix& CData) : Object(Parent)
{
  Properties["XData"]     = new Matrix(XData,1);
  Properties["YData"]     = new Matrix(YData,1);
  Properties["CData"]     = new Matrix(CData,1);

  SET_TYPE;
  COPY_DEFAULT(FaceColor,ColorRadio);
  COPY_DEFAULT(EdgeColor,ColorNone);
  COPY_DEFAULT(LineWidth,Scalar);
  COPY_DEFAULT(Clipping,Radio);
  COPY_DEFAULT(LineStyle,Radio);
  COPY_DEFAULT(Marker,Radio);
  COPY_DEFAULT(MarkerFaceColor,Color);
  COPY_DEFAULT(MarkerEdgeColor,Color);
  COPY_DEFAULT(MarkerSize,Scalar);
  COPY_DEFAULT(FaceAlpha,Scalar);
 /*
  //! 2D for now
  min[2] = ocpl::nan;
  max[2] = ocpl::nan;
  lmin[2] = ocpl::nan;
  lmax[2] = ocpl::nan;
*/
  FindMinMax(XData,min[0],max[0],lmin[0],lmax[0],YData.len());
  FindMinMax(YData,min[1],max[1],lmin[1],lmax[1]);
  FindMinMax(CData,min[2],max[2],lmin[2],lmax[2]);
}

void Patch::draw()
{
  IS_VISIBLE;

  MAKE_REF(xdata,Matrix);
  MAKE_REF(ydata,Matrix);
  MAKE_REF(cdata,Matrix);
  MAKE_REF(clipping,Radio);
  MAKE_REF(linestyle,Radio);
  MAKE_REF(edgecolor,ColorNone);
  MAKE_REF(facecolor,ColorRadio);
  MAKE_REF(linewidth,Scalar);
  MAKE_REF(markerfacecolor,Color);
  MAKE_REF(markeredgecolor,Color);
  MAKE_REF(marker,Radio);
  MAKE_REF(markersize,Scalar);
  MAKE_REF(facealpha,Scalar);

  FindMinMax(cdata,min[2],max[2],lmin[2],lmax[2]);

  Radio None("|{none}|");
  Figure* fig = dynamic_cast<Figure*>(FindParentOfType("Figure"));
  Axes* axes = dynamic_cast<Axes*>(FindParentOfType("axes"));
  Radio& xscale = ::Get<Radio>(axes,"xscale");
  Radio& yscale = ::Get<Radio>(axes,"yscale");
  Matrix& cmap  = ::Get<Matrix>(fig,"ColorMap");
  Matrix& clim  = ::Get<Matrix>(axes,"clim");

  SET_CLIPPING;

  GLUtesselator* tobj;
  tobj = gluNewTess();

  gluTessCallback(tobj, GLU_TESS_VERTEX,
		  (GLUTesselatorFunction)vertexCallback);
  gluTessCallback(tobj, GLU_TESS_BEGIN,
		  (GLUTesselatorFunction) beginCallback);
  gluTessCallback(tobj, GLU_TESS_END,
		  (GLUTesselatorFunction) endCallback);
  gluTessCallback(tobj, GLU_TESS_ERROR,
                  (GLUTesselatorFunction) errorCallback);
  gluTessCallback(tobj, GLU_TESS_COMBINE,
		  (GLUTesselatorFunction) combineCallback);
  gluTessCallback(tobj, GLU_TESS_EDGE_FLAG,
		  (GLUTesselatorFunction) edgeCallback);
  gluTessProperty(tobj, GLU_TESS_WINDING_RULE,
                     GLU_TESS_WINDING_ODD);
  gluTessProperty(tobj, GLU_TESS_TOLERANCE, 1e-20);

  GLuint startList = glGenLists(1);
  GLdouble *pt;
  double cv[4]={1.0,1.0,1.0,1.0};
  double face_alpha_val = facealpha.GetVal();
  const bool face_color_interp = (facecolor.val()=="interp");
  const bool face_color_flat = (facecolor.val()=="flat");
  const bool face_color_none = (facecolor.val()=="none");
  const bool interp_facet = !facecolor.IsColor()&&face_color_interp; 

  if (face_color_interp)
    glShadeModel(GL_SMOOTH);
  else
    glShadeModel(GL_FLAT);
  if (!face_color_none) {
    glNewList(startList, GL_COMPILE);
    facecolor.SetAlpha(face_alpha_val);
    if (facecolor.IsColor())
      facecolor.GetRGBA(cv);
    else if (face_color_flat) {
      InterpColor(cmap, clim, cdata(0), cv);
    }

    gluTessBeginPolygon(tobj, NULL);
      gluTessBeginContour(tobj);
        for(int i=0;i<ydata.nr()*ydata.nc();i++)
        {
          pt = new GLdouble[7]; // for xyz- and rgba- data
          pt[0] = LOGIT(xdata(i),xscale()=="log");
          pt[1] = LOGIT(ydata(i),yscale()=="log");
          pt[2] = 0.0; //z-value set to zero for now;
          if (interp_facet)
            InterpColor(cmap, clim, cdata(i), cv);
          for (unsigned int i = 0; i < 4; i++)
            pt[i+3] = cv[i];
          gluTessVertex(tobj, pt, pt);
        }
      gluTessEndContour(tobj);
    gluTessEndPolygon(tobj);
    glEndList();
    glCallList(startList);
    glFlush();
  }
  gluDeleteTess(tobj);

  if (edgecolor.IsColor()) {
    NEXT_LAYER;
    line_plotter::Instance().plot(xdata,
				ydata,
				linestyle,
				&edgecolor,
				linewidth(),
				markerfacecolor,
				markeredgecolor,
				marker,
				markersize(),
				axes,
				None,
				0,0,0,
				printing,
				true);
  }
  UNSET_CLIPPING;
}

