//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: waveview.cpp,v 1.1 2002/01/30 14:10:09 muse Exp $
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <stdio.h>
#include <values.h>

#include <qpainter.h>
#include "../wave.h"
#include "waveview.h"
#include "song.h"
#include "event.h"
#include "../sf/sndfile.h"
#include "../midieditor.h"

//---------------------------------------------------------
//   WaveView
//---------------------------------------------------------

WaveView::WaveView(MidiEditor* pr, QWidget* parent, int xscale, int yscale)
   : View(parent, xscale, 1)
      {
      editor = pr;
      setVirt(true);
      pos[0] = int(tempomap.tick2time(song->cpos()) * sampleRate);
      pos[1] = int(tempomap.tick2time(song->lpos()) * sampleRate);
      pos[2] = int(tempomap.tick2time(song->rpos()) * sampleRate);
      yScale = yscale;

      setMouseTracking(true);
      setBg(white);

      if (editor->parts()->empty()) {
            curPart = 0;
            }
      else {
            curPart   = (WavePart*)(editor->parts()->begin()->second);
            curPartId = curPart->sn();
            }
      connect(song, SIGNAL(posChanged(int,int,bool)), SLOT(setPos(int,int,bool)));
      }

//---------------------------------------------------------
//   setYScale
//---------------------------------------------------------

void WaveView::setYScale(int val)
      {
      yScale = val;
      redraw();
      }

//---------------------------------------------------------
//   draw
//---------------------------------------------------------

void WaveView::pdraw(QPainter& p, const QRect& rr)
      {
      int x1 = rr.x();
      int x2 = rr.right() + 1;
      if (x1 < 0)
            x1 = 0;
      if (x2 > width())
            x2 = width();
      int hh = height();
      int h  = hh/2;
      int y  = rr.y() + h;

      for (iPart ip = editor->parts()->begin(); ip != editor->parts()->end(); ++ip) {
            WavePart* wp = (WavePart*)(ip->second);
            EventList* el = wp->events();
            for (iEvent e = el->begin(); e != el->end(); ++e) {
                  WaveEvent* event  = (WaveEvent*)e->second;
// printf("draw part %p event %p\n", curPart, event);
                  Clip* clip        = event->clip();
                  if (!clip)
                        continue;
                  const SndFile* f  = clip->file();
                  if (!f)
                        continue;
                  int xScale = xmag;
                  if (xScale < 0)
                        xScale = -xScale;

                  int sx  = (event->posSample() + xScale/2 - startSample) / xScale;
                  int ex  = (event->posSample() + event->lenSample() + xScale/2 - startSample) / xScale;
// printf("----%d-%d  xpos %d  %d-%d\n", sx,ex, xpos, x1,x2);

                  sx -= xpos;
                  ex -= xpos;
                  if (sx < x1)
                        sx = x1;
                  if (ex > x2)
                        ex = x2;

                  int pos = (xpos + sx) * xScale + clip->spos() - event->posSample();
                  h      = hh / (f->channels() * 2);
                  int cc = hh % (f->channels() * 2) ? 0 : 1;

// printf("draw %d-%d  pos %d  Event: pos %d len:%d, clip pos %d\n",
//         sx, ex, event->posSample(), pos, event->lenSample(), clip->spos());

                  for (int i = sx; i < ex; i++) {
                        y  = rr.y() + h;
                        SampleV sa[f->channels()];
                        const_cast<SndFile*>(f)->read(sa, xScale, pos);
                        pos += xScale;
                        for (unsigned k = 0; k < f->channels(); ++k) {
                              int peak = (sa[k].peak * (h - 1)) / yScale;
                              int rms  = (sa[k].rms  * (h - 1)) / yScale;
                              if (peak > h)
                                    peak = h;
                              if (rms > h)
                                    rms = h;
                              p.setPen(QColor(darkGray));
                              p.drawLine(i, y - peak - cc, i, y + peak);
                              p.setPen(QColor(black));
                              p.drawLine(i, y - rms - cc, i, y + rms);
                              y  += 2 * h;
                              }
                        }
                  }
            }
      View::pdraw(p, rr);
      }

//---------------------------------------------------------
//   draw
//---------------------------------------------------------

void WaveView::draw(QPainter& p, const QRect& r)
      {
      int x = r.x();
      int y = r.y();
      int w = r.width();
      int h = r.height();
      if (x < 0)
            x = 0;
      if (y < 0)
            y = 0;

      int x2 = x + w;
      int y2 = y + h;

      int h2 = height()/2;
      int h4 = h2/2;
      int center1 = h4;
      int center2 = h2+h4;
      //
      //    draw marker & centerline
      //
      p.setPen(red);
      if (pos[0] >= x && pos[0] < x2) {
            p.drawLine(pos[0], y, pos[0], y2);
            }
      p.setPen(blue);
      if (pos[1] >= x && pos[1] < x2) {
            p.drawLine(pos[1], y, pos[1], y2);
            }
      if (pos[2] >= x && pos[2] < x2)
            p.drawLine(pos[2], y, pos[2], y2);
      p.setPen(QColor(blue));
      p.drawLine(x, center1, x2, center1);
      p.setPen(QColor(red));
      p.drawLine(x, center2, x2, center2);
      p.setPen(QColor(black));
      p.drawLine(x, h2, x2, h2);
      }

//---------------------------------------------------------
//   getCaption
//---------------------------------------------------------

QString WaveView::getCaption() const
      {
      return QString("Part ") + curPart->name();
      }

//---------------------------------------------------------
//   songChanged
//---------------------------------------------------------

void WaveView::songChanged(int flags)
      {
      if (flags & ~SC_SELECTION) {
            startSample  = MAXINT;
            endSample    = 0;
            curPart      = 0;
            for (iPart p = editor->parts()->begin(); p != editor->parts()->end(); ++p) {
                  WavePart* part = (WavePart*)(p->second);
                  if (part->sn() == curPartId)
                        curPart = part;
                  int ssample = part->posSample();
                  int esample = ssample + part->lenSample();
                  if (ssample < startSample)
                        startSample = ssample;
                  if (esample > endSample)
                        endSample = esample;
                  }
            }
      redraw();
      }

//---------------------------------------------------------
//   setPos
//    set one of three markers
//    idx   - 0-cpos  1-lpos  2-rpos
//    flag  - emit followEvent()
//---------------------------------------------------------

void WaveView::setPos(int idx, int val, bool adjustScrollbar)
      {
      val = int(tempomap.tick2time(val) * sampleRate);
      if (!isVisible()) {
            pos[idx] = val;
            return;
            }
      if (pos[idx] == val)
            return;
      int opos = mapx(pos[idx]);
      int npos = mapx(val);

      if (adjustScrollbar && idx == 0) {
            switch (song->follow()) {
                  case  Song::NO:
                        break;
                  case Song::JUMP:
                        if (npos >= width()) {
                              pos[0] = val;
                              emit followEvent(val);
                              return;
                              }
                        break;
                  case Song::CONTINUOUS:
                        {
                        int w2 = width()/2;
                        if (npos > w2) {
                              int ppos =  pos[0] - rmapxDev(w2);
                              if (ppos < 0)
                                    ppos = 0;
                              emit followEvent(ppos);
                              opos = mapx(pos[0]);
                              npos = mapx(val);
                              }
                        }
                        break;
                  }
            }
#if 0
      int x;
      int w = 1;
      if (opos > npos) {
            w += opos - npos;
            x = npos;
            }
      else {
            w += npos - opos;
            x = opos;
            }
#endif
      pos[idx] = val;
//      redraw(QRect(x, 0, w, height()));
      redraw(QRect(opos, 0, 1, height()));
      redraw(QRect(npos, 0, 1, height()));
      }

//---------------------------------------------------------
//   viewMousePressEvent
//---------------------------------------------------------

void WaveView::viewMousePressEvent(QMouseEvent* event)
      {
      button = event->button();
      viewMouseMoveEvent(event);
      }

void WaveView::viewMouseReleaseEvent(QMouseEvent*)
      {
      button = QMouseEvent::NoButton;
      }

//---------------------------------------------------------
//   viewMouseMoveEvent
//---------------------------------------------------------

void WaveView::viewMouseMoveEvent(QMouseEvent* event)
      {
      int x = event->x();
      int i;
      emit timeChanged(x);

      switch (button) {
            case QMouseEvent::LeftButton:
                  i = 0;
                  break;
            case QMouseEvent::MidButton:
                  i = 1;
                  break;
            case QMouseEvent::RightButton:
                  i = 2;
                  break;
            default:
                  return;
            }
      x = tempomap.time2tick(double(x) / double(sampleRate));
      song->setPos(i, x);
      }

//---------------------------------------------------------
//   range
//    returns range in samples
//---------------------------------------------------------

void WaveView::range(int* s, int *e)
      {
      *s = curPart->posSample();
      int etick = curPart->posTick() + curPart->lenTick();
      *e = tempomap.tick2samples(etick);
      }

