//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: synth.cpp,v 1.4 2002/02/11 10:10:57 muse Exp $
//
//  This file is derived from IIWU Synth and modified
//    for MusE.
//  Parts of IIWU are derived from Smurf Sound Font Editor.
//  Parts of Smurf Sound Font Editor are derived from
//    awesfx utilities
//  Smurf:  Copyright (C) 1999-2000 Josh Green
//  IIWU:   Copyright (C) 2001 Peter Hanappe
//  MusE:   Copyright (C) 2001 Werner Schweer
//  awesfx: Copyright (C) 1996-1999 Takashi Iwai
//=========================================================

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#include <math.h>
#include <sys/mman.h>
#include <poll.h>
#include <sys/time.h>

#include "sfont.h"
#include "synth.h"

int _sampleRate = 44100;      // TODO

enum iiwu_status {
      IIWU_OK = 0,
      IIWU_FAILED = 1
      };

// #define DEBUG

#ifdef DEBUG
#define IIWU_LOG(format, args...) printf("%s:%d:" format "\n", __FILE__, __LINE__, ##args);
#else
#define IIWU_LOG(format, args...)
#endif

enum iiwu_midi_event_type {
      NOTE_OFF = 0x80,
      NOTE_ON  = 0x90,
      };

enum iiwu_driver_status {
      IIWU_MIDI_READY,
      IIWU_MIDI_LISTENING,
      IIWU_MIDI_DONE
      };

bool ISynth::initialized = false;  // has the synth module been initialized?

/* conversion tables */
static double iiwu_ct2hz_tab[IIWU_CENTS_HZ_SIZE];
static double iiwu_vel2cb_tab[IIWU_VEL_CB_SIZE];
static double iiwu_cb2amp_tab[IIWU_CB_AMP_SIZE];
static double iiwu_concave_tab[128];
static double iiwu_convex_tab[128];
static double iiwu_pan_tab[IIWU_PAN_SIZE];
static iiwu_interp_coeff_t interp_coeff[IIWU_INTERP_MAX];

/* default modulators */
static Mod default_pan_mod;
static Mod default_att_mod;
static Mod default_pitch_bend_mod;

// static int modRes = 128;     // modulator resolution in samples

//---------------------------------------------------------
//   Sample
//---------------------------------------------------------

Sample::Sample()
      {
      _name[0]   = 0;
      sampletype = 0;
      cents      = 0;
      next       = 0;
      _data      = 0;
      start      = 0;
      end        = 0;
      loopstart  = 0;
      loopend    = 0;
      samplerate = 0;
      origpitch  = 0;
      pitchadj   = 0;
      }

Sample::~Sample()
      {
      }

#ifdef RTCAP
//---------------------------------------------------------
//   getCapabilities
//---------------------------------------------------------

static void getCapabilities()
      {
#ifdef __linux__
      const char* napp = getenv("GIVERTCAP");
      system(napp ? napp : "givertcap");
#endif // __linux__
      }
#endif

//---------------------------------------------------------
//   init
//    Does all the initialization for this module.
//---------------------------------------------------------

void ISynth::initSynth()
      {
      initialized = true;

      for (int i = 0; i < IIWU_CENTS_HZ_SIZE; i++) {
            iiwu_ct2hz_tab[i] = pow(2.0, double(i) / 1200.0);
            }

      for (int i = 0; i < IIWU_VEL_CB_SIZE; i++) {
            double x = (127.0 - i) / 127.0;
            iiwu_vel2cb_tab[i] = 360.0 * x;
            }

      /* centibels to amplitude conversion */
      for (int i = 0; i < IIWU_CB_AMP_SIZE; i++) {
            iiwu_cb2amp_tab[i] = pow(10.0, double(i) / -200.0);
            }

      /* initialize the conversion tables */

      /* concave transform curve */
      iiwu_concave_tab[0] = 0.0;
      iiwu_concave_tab[127] = 1.0;
      double x = log10(128.0 / 127.0);
      for (int i = 1; i < 127; i++) {
            iiwu_concave_tab[i] = (double) (1.0 - x / log10(127.0 / (127.0 - i)));
            }

      /* convex transform curve */
      iiwu_convex_tab[0] = 0;
      iiwu_convex_tab[127] = 1.0;
      x = log10(128.0 / 127.0);
      for (int i = 1; i < 127; i++) {
            iiwu_convex_tab[i] = (double) (x / log10(127.0 / i));
            }

      /* initialize the pan conversion table */
      x = PI / 2.0 / (IIWU_PAN_SIZE - 1.0);
      for (int i = 0; i < IIWU_PAN_SIZE; i++) {
            iiwu_pan_tab[i] = (double) sin(i * x);
            }

#if defined(USE_TRUNCATION)
      /* this is a stupid */
      for (int i = 0; i < IIWU_INTERP_MAX; i++) {
            interp_coeff[i] = 0;
            }
#elif defined(USE_LINEAR_INTERPOLATION)
      /* this is a bit stupid */
      for (int i = 0; i < IIWU_INTERP_MAX; i++) {
            x = (double) i / (double) IIWU_INTERP_MAX;
            interp_coeff[i].a0 = (double) (1.0 - x);
            interp_coeff[i].a1 = (double) x;
            }
#elif defined(USE_CUBIC_INTERPOLATION)
      /* this is cool */
      /* Initialize the coefficients for the interpolation. The math comes
       * from a mail, posted by Olli Niemitalo to the music-dsp mailing
       * list (I found it in the music-dsp archives
       * http://www.smartelectronix.com/musicdsp/).  */

      for (int i = 0; i < IIWU_INTERP_MAX; i++) {
            x = double(i) / double(IIWU_INTERP_MAX);
            interp_coeff[i].a0 = x * (-0.5 + x * (1 - 0.5 * x));
            interp_coeff[i].a1 = 1.0 + x * x * (1.5 * x - 2.5);
            interp_coeff[i].a2 = x * (0.5 + x * (2.0 - 1.5 * x));
            interp_coeff[i].a3 = 0.5 * x * x * (x - 1.0);
            }
#endif

      /* initialize the default modulators */
      /* pan */
      default_pan_mod.set_source1(10, IIWU_MOD_CC | IIWU_MOD_LINEAR | IIWU_MOD_BIPOLAR | IIWU_MOD_POSITIVE);
      default_pan_mod.set_source2(0, 0);
      default_pan_mod.set_dest(GEN_PAN);
      default_pan_mod.set_amount(1000.0);

      /* initial attenuation */
      default_att_mod.set_source1(7, IIWU_MOD_CC | IIWU_MOD_CONCAVE | IIWU_MOD_UNIPOLAR | IIWU_MOD_NEGATIVE);
      default_att_mod.set_source2(0, 0);
      default_att_mod.set_dest(GEN_ATTENUATION);
      default_att_mod.set_amount(960.0);

      /* pitch bend */
      default_pitch_bend_mod.set_source1(IIWU_MOD_PITCHWHEEL,
		       IIWU_MOD_GC | IIWU_MOD_LINEAR | IIWU_MOD_BIPOLAR | IIWU_MOD_POSITIVE);
      default_pitch_bend_mod.set_source2(IIWU_MOD_PITCHWHEELSENS,
		       IIWU_MOD_GC | IIWU_MOD_LINEAR | IIWU_MOD_UNIPOLAR | IIWU_MOD_POSITIVE);
      default_pitch_bend_mod.set_dest(GEN_PITCH);
      default_pitch_bend_mod.set_amount(12700.0);
      }

//---------------------------------------------------------
//   iiwu_ct2hz
//---------------------------------------------------------

static double iiwu_ct2hz(double dcents)
      {
      double tab[11] = {
            13.75, 27.5, 55.0, 110.0, 220.0,
            440.0, 880.0, 1760.0, 3520.0,
            7040.0, 14080.0
            };
      int cents = int(dcents) - 900;
      int idx   = cents/1200;
      if (idx > 10)
            return 1.0;
      double val = iiwu_ct2hz_tab[cents%1200];
      return tab[idx] * val;
      }

//---------------------------------------------------------
//   iiwu_vel2cb
//---------------------------------------------------------

static double iiwu_vel2cb(double vel)
      {
      return iiwu_vel2cb_tab[int(vel)];
      }

//---------------------------------------------------------
//   iiwu_cb2amp
//    convert centibel to amplification (0 - 96dB)
//---------------------------------------------------------

static double iiwu_cb2amp(double cb)
      {
      if (cb < 0) {
            return 1.0;
            }
      if (cb > 960) {
            return 0.0;
            }
      return iiwu_cb2amp_tab[int(cb)];
      }

//---------------------------------------------------------
//   iiwu_tc2sec
//    time cent to second
//---------------------------------------------------------

static double iiwu_tc2sec(double tc)
      {
      return (tc == -32768.0) ? 0.0 : pow(2.0, tc / 1200.0);
      }

//---------------------------------------------------------
//   iiwu_act2hz
//    Convert from absolute cents to Hertz
//---------------------------------------------------------

static double iiwu_act2hz(double c)
      {
      return (8.176 * pow(2.0, c / 1200.0));
      }

//---------------------------------------------------------
//   iiwu_hz2ct
//    Convert from Hertz to cents
//---------------------------------------------------------

static double iiwu_hz2ct(double f)
      {
      return (6900 + 1200 * log(f / 440.0) / log(2.0));
      }

//---------------------------------------------------------
//   iiwu_pan
//---------------------------------------------------------

static double iiwu_pan(double c, bool left)
      {
      if (left) {
            c = -c;
            }
      if (c < -500) {
            return (double) 0.0;
            }
      else if (c > 500) {
            return (double) 1.0;
            }
      else {
            return iiwu_pan_tab[int(c) + 500];
            }
      }

//---------------------------------------------------------
//   iiwu_concave
//---------------------------------------------------------

static double iiwu_concave(double val)
      {
      if (val < 0) {
            return 0;
            }
      else if (val > 127) {
            return 1;
            }
      return iiwu_concave_tab[int(val)];
      }

//---------------------------------------------------------
//   iiwu_convex
//---------------------------------------------------------

static double iiwu_convex(double val)
      {
      if (val < 0) {
            return 0;
            }
      else if (val > 127.0) {
            return 1;
            }
      return iiwu_convex_tab[int(val)];
      }

//---------------------------------------------------------
//   ISynth
//---------------------------------------------------------

ISynth::ISynth(const char* cn)
   : Mess(cn, 2)
      {
      startFrame = 0.0;
      _gmMode    = false;     // General Midi Mode
      _masterVol = 0x3fff;    // max vol
      sfont      = 0;         // the loaded soundfont
      nextSFont  = 0;         // for getNextParam
      spFree     = 0;
      spBusy     = 0;

      /* initialize all the conversion tables and other stuff */
      if (!initialized)
            initSynth();

      for (int i = 0; i < IIWU_NUM_CHANNELS; i++)
            channel[i].setChannum(i);

      /* allocate all synthesis processes */
      for (int i = 0; i < 128; ++i) {
            SynthProc* sp = new SynthProc(this);
            sp->next = spFree;
            spFree   = sp;
            }

      /* allocate the sample buffers */
      maxBufferSize = 512;
      mono_buf      = new float[maxBufferSize];
      }

//---------------------------------------------------------
//   processEvent
//---------------------------------------------------------

void ISynth::processEvent(MEvent* ev)
      {
      switch(ev->type()) {
            case SND_SEQ_EVENT_NOTEON:
            case SND_SEQ_EVENT_KEYPRESS:
                  noteon(ev->chan(), ev->dataA(), ev->dataB());
                  break;

            case SND_SEQ_EVENT_NOTEOFF:
                  noteoff(ev->chan(), ev->dataA());
                  break;

            case SND_SEQ_EVENT_PGMCHANGE:
                  program_change(ev->chan(), ev->dataB());
                  break;

            case SND_SEQ_EVENT_PITCHBEND:
                  pitch_bend(ev->chan(), (ev->dataA()<<7) | ev->dataB());
                  break;

            case SND_SEQ_EVENT_CONTROL14:
            case SND_SEQ_EVENT_NONREGPARAM:
            case SND_SEQ_EVENT_REGPARAM:
            case SND_SEQ_EVENT_CONTROLLER:
                  setCc(ev->chan(), ev->dataA(), ev->dataB());
                  break;

            case SND_SEQ_EVENT_SYSEX:
                  sysex(ev->data(), ev->dataLen());
                  break;
            case SND_SEQ_EVENT_CHANPRESS:
                  break;
            default:
                  printf("processEvent: unknown event\n");
            }
      }

//---------------------------------------------------------
//   sysex
//---------------------------------------------------------

void ISynth::sysex(const unsigned char* data, int len)
      {
      if (len >= 6 && data[0] == 0xf0 && data[len-1] == 0xf7) {

            //---------------------------------------------
            //  Universal Non Realtime
            //---------------------------------------------

            if (data[1] == 0x7e) {
                  if (data[2] == 0x7f) {  // device Id
                        if (data[3] == 0x9) {   // GM
                              if (data[4] == 0x1) {
                                    gmOn(true);
                                    return;
                                    }
                              else if (data[4] == 0x2) {
                                    gmOn(false);
                                    return;
                                    }
                              }
                        }
                  }

            //---------------------------------------------
            //  Universal Realtime
            //---------------------------------------------

            else if (data[1] == 0x7f) {
                  if (data[2] == 0x7f) {  // device Id
                        if ((data[3] == 0x4) && (data[4] == 0x1)) {
                              setMasterVol(data[6]*128 + data[5]);
                              return;
                              }
                        }
                  }

            //---------------------------------------------
            //  MusE Soft Synth
            //---------------------------------------------

            else if (data[1] == 0x7c) {
                  int n = len - 5;
                  if (n < 1) {
                        printf("iiwu: bad sysEx:\n");
                        return;
                        }
                  char buffer[n+1];
                  memcpy(buffer, (char*)data+4, n);
                  buffer[n] = 0;
                  if (data[2] == 0) {     // iiwu
                        if (data[3] == 1) {  // load sound font
                              sysexSoundFont(SF_REPLACE, buffer);
                              return;
                              }
                        else if (data[3] == 2) {  // load sound font
                              sysexSoundFont(SF_ADD, buffer);
                              return;
                              }
                        else if (data[3] == 3) {  // load sound font
                              sysexSoundFont(SF_REMOVE, buffer);
                              return;
                              }
                        }
                  }
            else if (data[1] == 0x41) {   // roland
                  if (data[2] == 0x10 && data[3] == 0x42 && data[4] == 0x12
                     && data[5] == 0x40 && data[6] == 00 && data[7] == 0x7f
                     && data[8] == 0x41) {
                        // gs on
                        gmOn(true);
                        }
                  }
            }
      printf("iiwu: unknown sysex received, len %d:\n", len);
      for (int i = 0; i < len; ++i)
            printf("%02x ", data[i]);
      printf("\n");
      }

//---------------------------------------------------------
//   gmOn
//---------------------------------------------------------

void ISynth::gmOn(bool flag)
      {
      _gmMode = flag;
      allNotesOff();
      if (flag) {
            for (int i = 0; i < 16; ++i) {
                  resetAllController(i);
                  channel[i].setCc(7, 100);
                  channel[i].setCc(10, 64);
                  }
            channel[9].setHbank(1);  // reset drum channel
            channel[9].setLbank(0);
            channel[9].setProg(0);
            }
      }

//---------------------------------------------------------
//   deleteSFonts
//---------------------------------------------------------

void ISynth::deleteSFonts()
      {
      SFont* sf = sfont;
      while (sf) {
            SFont* f = sf->next;
            delete sf;
            sf = f;
            }
      sfont = 0;
      }

//---------------------------------------------------------
//   allNotesOff
//    stop all notes
//---------------------------------------------------------

void ISynth::allNotesOff()
      {
      SynthProc* sp;
      for (sp = spBusy; sp && sp->next; sp = sp->next)
            ;
      if (sp) {
            sp->next = spFree;
            spFree   = spBusy;
            spBusy   = 0;
            }
      }

//---------------------------------------------------------
//   sysexSoftfont
//---------------------------------------------------------

void ISynth::sysexSoundFont(SfOp op, const char* data)
      {
      for (SFont* sf = sfont; sf; sf = sf->next)
            if (strcmp(sf->filePath(), data) == 0) {
                  return;
                  }

      allNotesOff();

      switch(op) {
            case SF_REMOVE:
                  break;
            case SF_REPLACE:
                  deleteSFonts();
            case SF_ADD:
                  sfload(data);
                  break;
            }
      }

//---------------------------------------------------------
//   init
//    return true on error
//---------------------------------------------------------

bool ISynth::init()
      {
      //
      //  get some default sound font:
      //
      char* museLib = INSTDIR;
      char* sfont = getenv("DEFAULT_SOUNDFONT");
      if (sfont == 0)
            sfont = "MiniPiano.SF2";
      char* sfontPath;
      char buffer[strlen(museLib) + strlen(sfont) + strlen("soundfonts") + 2];
      if (*sfont == '/')
            sfontPath = sfont;
      else {
            sprintf(buffer, "%s/soundfonts/%s", museLib, sfont);
            sfontPath = buffer;
            }
      if (sfload(sfontPath))
            printf("iiwu: Failed to load default soundfont <%s>\n", sfontPath);
      return false;
      }

//---------------------------------------------------------
//   ~ISynth
//---------------------------------------------------------

ISynth::~ISynth()
      {
      pthread_cancel(midiThread);

      if (pthread_join(midiThread, 0)) {
            fprintf(stderr, "Failed to join the midi thread\n");
            }

      deleteSFonts();
      SynthProc* sp = spFree;
      while (sp) {
            SynthProc* nsp = sp->next;
            delete sp;
            sp = nsp;
            }
      sp = spBusy;
      while (sp) {
            SynthProc* nsp = sp->next;
            delete sp;
            sp = nsp;
            }
      delete mono_buf;
      }

//---------------------------------------------------------
//   noteon
//---------------------------------------------------------

void ISynth::noteon(int chan, int key, int vel)
      {
      int i = 0;
      for (SynthProc* sp = spBusy; sp && sp->next; sp = sp->next)
            ++i;
// printf("%3d noteOn %d %x %x\n", i, chan, key, vel);

      /* notes with velocity equal zero go to noteoff  */
      if (vel == 0) {
            noteoff(chan, key);
            return;
            }

      /* let the channel handle the noteon */
      channel[chan].noteon(this, key, vel);
      }

//---------------------------------------------------------
//   noteoff
//    search all synthesis processes with this channel
//    and key pair
//---------------------------------------------------------

void ISynth::noteoff(int chan, int key)
      {
      if (chan == 9)    // dont stop drums
            return;
      for (SynthProc* sp = spBusy; sp; sp = sp->next) {
            if ((sp->chan() == chan) && (sp->key == key)) {
                  sp->noteoff();
                  }
            }
      }

//---------------------------------------------------------
//   resetAllController
//---------------------------------------------------------

void ISynth::resetAllController(int chan)
      {
      channel[chan].setCc(1, 0);
      channel[chan].setCc(11, 127);
      channel[chan].setCc(64, 0);
      channel[chan].setCc(65, 0);
      channel[chan].setCc(66, 0);
      channel[chan].setCc(67, 0);
      // RPN NRPN 0
      // pitch 64,0
      // channel pressure 0
      }

//---------------------------------------------------------
//   cc
//---------------------------------------------------------

void ISynth::setCc(int chan, int num, int val)
      {
      // set the controller value in the channel
      channel[chan].setCc(num, val);

// printf("setCC %d %p %d %d\n", chan, &channel[chan], num, val);

      switch (num) {
            case 0:     // hbank
                  if (!_gmMode)
                        hbank_select(chan, val);
                  return;
            case 0x20:  // lbank
                  if (!_gmMode)
                        lbank_select(chan, val);
                  return;
            case 0x40:        // sustain
                  if (val < 0x40) {
                        //
                        // on sustain release switch off all sprocs in
                        // sustain mode:
                        //
// printf("sustain release\n");
                        for (SynthProc* sp = spBusy; sp; sp = sp->next) {
                              if (sp->chan() == chan && sp->sustained()) {
                                    sp->setSustained(false);
                                    sp->noteoff();
                                    }
                              }
                        }
                  return;
            case 120:   // all sounds off
            case 123:   // all notes off
                  allNotesOff();
                  return;
            case 121:   // reset all controllers
                  resetAllController(chan);
                  break;
            }

      //  tell all busy synthesis processes on this channel to
      //  update their synthesis parameters

      for (SynthProc* sp = spBusy; sp; sp = sp->next) {
            if (sp->chan() == chan)
                  sp->modulate(1, num, val);
            }
      }

//---------------------------------------------------------
//   pitch_bend
//---------------------------------------------------------

void ISynth::pitch_bend(int chan, int val)
      {
      /* set the pitch-bend value in the channel */
      channel[chan].setPitchBend(val);

      //  tell all busy synthesis processes on this channel to
      //  update their synthesis parameters
// printf("pitch chan %d val %d\n", chan, val);
      for (SynthProc* sp = spBusy; sp; sp = sp->next) {
            if (sp->chan() == chan)
                  sp->modulate(0, IIWU_MOD_PITCHWHEEL, val);
            }
      }

//---------------------------------------------------------
//   program_change
//---------------------------------------------------------

void ISynth::program_change(int chan, char prog)
      {
      if (sfont == 0)
            return;

      if (_gmMode && chan == 9)  // not for drum channel in GM mode
            return;

      /* inform the channel of the new program number */
      channel[chan].setProg(prog);

      int hbank =  channel[chan].hbank();
      int lbank =  channel[chan].lbank();
      IIWU_LOG("program change: chan=%d, hbank=%d, lbank=%d, prog=%d",
         chan, hbank, lbank, prog);

      Preset* preset = sfont->get_preset(hbank, lbank, prog);
      channel[chan].setPreset(preset);
      }

//---------------------------------------------------------
//   bank_select
//---------------------------------------------------------

void ISynth::hbank_select(int chan, char bank)
      {
      if ((chan >= 0) && (chan < IIWU_NUM_CHANNELS))
            channel[chan].setHbank(bank);
      }

//---------------------------------------------------------
//   lbank_select
//---------------------------------------------------------

void ISynth::lbank_select(int chan, char bank)
      {
      if ((chan >= 0) && (chan < IIWU_NUM_CHANNELS))
            channel[chan].setLbank(bank);
      }

//---------------------------------------------------------
//   program_reset
//    Resend a bank select and a program change for every
//    channel. This function is called mainly after a
//    soundfont has been loaded, unloaded or reloaded.
//---------------------------------------------------------

void ISynth::program_reset()
      {
      for (int i = 0; i < 16; i++) {
            program_change(i, channel[i].prog());
            }
      }

//---------------------------------------------------------
//   write
//---------------------------------------------------------

void ISynth::write(int n, float** ports, int offset)
      {
      for (SynthProc* sp = spBusy; sp;) {
            SynthProc* nsp = sp->next;
            sp->write(n, mono_buf+offset, ports[0]+offset, ports[1]+offset);
            sp = nsp;
            }
      }

//---------------------------------------------------------
//   stop
//---------------------------------------------------------

void ISynth::stop(SynthProc* _sp)
      {
// printf("STOP=================\n");
      SynthProc* psp = 0;
      for (SynthProc* csp = spBusy; csp; psp = csp, csp = csp->next) {
            if (csp == _sp) {
                  if (psp)
                        psp->next = csp->next;
                  else
                        spBusy = csp->next;
                  csp->next = spFree;
                  spFree = csp;
                  return;
                  }
            }
      printf("STOP SP: not found\n");
      exit(-1);
      }

//---------------------------------------------------------
//   start
//---------------------------------------------------------

void ISynth::start(SynthProc* sp)
      {
      sp->next = spBusy;
      spBusy   = sp;
      }

//---------------------------------------------------------
//   iiwu_synth_alloc_sp
//---------------------------------------------------------

SynthProc* ISynth::alloc_sp(Channel* /*chan*/, int /*key*/, int /*vel*/)
      {
      if (spFree) {                  // allocate next free sp
            SynthProc* sp = spFree;
            spFree = spFree->next;
            return sp;
            }
//DEBUG:
      printf("max polyphony reached:all sps used\n");

      // No synthesis process available: steal one:
      for (int k = 0; k < IIWU_NUM_CHANNELS; k++) {
            /* We steal a voice according to the priorities of the
               channels. (DLS specifications) */

            // the (inversed) priority in which oscillators are stolen
            static unsigned char chan_priority[] = {
                  15, 14, 13, 12, 11, 10, 8, 7, 6, 5, 4, 3, 2, 1, 0, 9
                  };

            unsigned char ch = chan_priority[k];

            /* first, try to find a key that's been released */

            for (SynthProc* sp = spBusy; sp; sp = sp->next) {
                  if ((sp->chan() == ch) && sp->released()) {
                        stop(sp);
                        SynthProc* sp = spFree;
                        spFree = spFree->next;
                        return sp;
                        }
                  }
            /* second, go for any key on this channel */
            for (SynthProc* sp = spBusy; sp; sp = sp->next) {
                  if (sp->chan() == ch) {
                        stop(sp);
                        SynthProc* sp = spFree;
                        spFree = spFree->next;
                        return sp;
                        }
                  }
            }
      return 0;
      }

//---------------------------------------------------------
//   sfload
//---------------------------------------------------------

int ISynth::sfload(const char* filename)
      {
      if (filename == 0)
            return IIWU_FAILED;

      SFont* sf = new SFont;

      printf("load soundfont <%s>\n", filename);
      if (sf->load(filename)) {
            printf("load soundfont <%s> failed\n", filename);
            delete sf;
            return IIWU_FAILED;
            }

      /* insert the sfont as the first on the list */
      sf->next = sfont;
      sfont    = sf;

      /* reset the presets for all channels */
      program_reset();

//    dump();
      return IIWU_OK;
      }

//---------------------------------------------------------
//   Channel
//---------------------------------------------------------

Channel::Channel()
      {
      for (int i = 0; i < 128; i++)
            _cc[i] = 0;
      _cc[7]                  = 127;      // channel volume
      _cc[10]                 = 64;       // pan controller
      _cc[11]                 = 127;      // modulation
      _preset                 = 0;
      key_pressure            = 0;
      channel_pressure        = 0;
      _pitch_bend             = 0;
      pitch_wheel_sensitivity = 2;        // two semi-tones
      }

//---------------------------------------------------------
//   setChannum
//---------------------------------------------------------

void Channel::setChannum(int num)
      {
      _chan  = num;
      _prog  = 0;
      _lbank = 0;
      _hbank = (num == 9) ? 1 : 0;     // default drum
      }

//---------------------------------------------------------
//   noteon
//---------------------------------------------------------

void Channel::noteon(ISynth* synth, int key, int vel)
      {
      if (preset() == 0)    // make sure this channel has a preset
            return;

      Zone* global_preset_zone = 0;
      Zone* pz = preset()->zone;
      if (pz && pz->inst == 0) {
            global_preset_zone = pz;
            pz = pz->next;
            }
      for (; pz; pz = pz->next) {
            if (!pz->inside_range(key, vel))
                  continue;
            Inst* inst = pz->inst;

            Zone* global_inst_zone = 0;
            Zone* inst_zone = inst->zone;
            if (inst_zone && inst_zone->getSample() == 0) {
                  global_inst_zone = inst_zone;
                  inst_zone = inst_zone->next;
                  }

            /* run thru all the zones of this instrument */
            for (; inst_zone; inst_zone = inst_zone->next) {
                  Sample* sample = inst_zone->getSample();

                  if (sample == 0 || sample->in_rom())
                        continue;
                  if (!inst_zone->inside_range(key, vel))
                        continue;

                  /* this is a good zone. allocate a new synthesis process and
                     initialize it */
                  SynthProc* sp = synth->alloc_sp(this, key, vel);
                  if (sp == 0)
                        return;

// printf("alloc synthProc %p %d 0x%x\n", sp, _chan, key);
                  sp->key = key;
                  sp->vel = vel;
                  sp->init(this, sample);

                  // set the generators of the synthesis process. the
                  // generators of the global instrument zone are taken as
                  // default values.

                  if (global_inst_zone) {
//printf("global instrument zone\n");
                        for (int i = 0; i < GEN_LAST; i++) {
                              if (global_inst_zone->gen[i].flags) {
                                    sp->gen[i].val   = global_inst_zone->gen[i].val;
                                    sp->gen[i].flags = GEN_SET;
                                    }
                              }
                        }
                  // the generators of the instrument zone have authority and
                  // erase the default values.

                  for (int i = 0; i < GEN_LAST; i++) {
                        if (inst_zone->gen[i].flags) {
                              sp->gen[i].val   = inst_zone->gen[i].val;
                              sp->gen[i].flags = GEN_SET;
                              }
                        }
//printf("generator VOLENVDECAY: %f %d\n",
//        sp->gen[GEN_VOLENVDECAY].val, inst_zone->gen[GEN_VOLENVDECAY].flags);

//printf("generator VOLENVRELEASE: %f %d\n",
//        sp->gen[GEN_VOLENVRELEASE].val, inst_zone->gen[GEN_VOLENVRELEASE].flags);

                  // the generators of the global preset zone add to the
                  // existing values.

                  if (global_preset_zone) {
//printf("global preset zone\n");
                        for (int i = 0; i < GEN_LAST; i++) {
                              /* FIXEM: some generators should not be tested at the
                              preset level, like the "override root key" */
                              if (global_preset_zone->gen[i].flags) {
                                    sp->gen[i].val += global_preset_zone->gen[i].val;
                                    sp->gen[i].flags = GEN_SET;
                                    }
                              }
                        }

                  /* the generators of the preset zone add to the existing
                     values. */
                  for (int i = 0; i < GEN_LAST; i++) {
                        if (pz->gen[i].flags) {
                              sp->gen[i].val += pz->gen[i].val;
                              sp->gen[i].flags = GEN_SET;
                              }
                        }
//printf("generator+preset VOLENVDECAY: %f(+%f) %d\n",
//  sp->gen[GEN_VOLENVDECAY].val,
//  pz->gen[GEN_VOLENVDECAY].val,
//  pz->gen[GEN_VOLENVDECAY].flags);

                  /* add the default modulators to the synthesis process. */
                  sp->add_mod(&default_pan_mod);
                  sp->add_mod(&default_att_mod);
                  sp->add_mod(&default_pitch_bend_mod);
                  /* calculate all the runtime synthesis parameters. */
                  sp->optimize();

                  /* add the synthesis process to the synthesis loop. */
                  synth->start(sp);
                  }
            }
      }

//---------------------------------------------------------
//   SynthProc
//---------------------------------------------------------

SynthProc::SynthProc(ISynth* s)
      {
      synth   = s;
      }

//---------------------------------------------------------
//   init
//    Initialize the synthesis process and sets all its
//    generators to their initial value.
//---------------------------------------------------------

void SynthProc::init(Channel* ch, Sample* s)
      {
      _sustained        = false;
      attenuation       = 0;
      _pitch            = 6900.0;
      root_pitch        = 440.0;
      channel           = ch;
      mod_count         = 0;
      _sample           = s;
      ticks             = 0;

      phase.setInt(_sample->start);

      /* sample and loop start and end points */
      start_offset      = 0;
      end_offset        = 0;
      loop_start_offset = 0;
      loop_end_offset   = 0;

      /* vol env */
      volenv_state      = ENV_SKIP;
      volenv_val        = 0.0;
      volenv_delay      = 0;
      volenv_attack     = 0.0;
      volenv_hold       = 0.0;
      volenv_decay      = 0.0;
      volenv_sustain    = 0.0;
      volenv_release    = 0.0;

      /* mod env */
      modenv_state      = ENV_SKIP;
      modenv_val        = 0.0f;
      modenv_delay      = 0;
      modenv_attack     = 0;
      modenv_hold       = 0;
      modenv_decay      = 0;
      modenv_sustain    = 0;
      modenv_release    = 0.0;
      modenv_to_fc      = 0;
      modenv_to_pitch   = 0;

      /* mod lfo */
      modlfo_val        = 0.0;
      modlfo_delay      = 0;
      modlfo_to_fc      = 0;
      modlfo_to_pitch   = 0;
      modlfo_to_vol     = 0;

      /* vib lfo */
      viblfo_val        = 0.0f;
      viblfo_to_pitch   = 0;
      viblfo_delay      = 0;

      /* filter */
      fres              = 20001.0f;
      q                 = 0.0f;
      w1                = 0.0f;
      w2                = 0.0f;

      pan               = 0;

      /* set all the generators to their initial value */
      gen_init_array(gen);
      }

#if defined(USE_TRUNCATION)
#define INTERPOL(_o,_a,_b,_i,_c)   _o=_a*_b[_i]
#elif defined(USE_LINEAR_INTERPOLATION)
#define INTERPOL(_o,_a,_b,_i,_c)      { \
  _o = _a * (_c->a0 * _b[_i] + _c->a1 * _b[_i+1]); \
}
#elif defined(USE_CUBIC_INTERPOLATION)
#define INTERPOL(_o,_a,_b,_i,_c) { \
  _o = _a * (_c->a0 * _b[_i-1] + _c->a1 * _b[_i] + _c->a2 * _b[_i+1] + _c->a3 * _b[_i+2]); \
}
#else
#error No interpolation defined
#endif

//---------------------------------------------------------
//  write
//    generate len samples
//---------------------------------------------------------

void SynthProc::write(int len, float* mono, float* left, float* right)
      {
      bool switchOff = false;

// printf("write %d state %d env %f\n", len, volenv_state, volenv_val);

      switch (volenv_state) {
            case ENV_SKIP:
                  volenv_val = 1.0;
                  break;

            case ENV_SUSTAIN:
                  volenv_val = volenv_sustain;
                  break;

            case ENV_RELEASE:
#if 0
printf("write %d RELEASE env %f += %f\n",
    len, volenv_val, volenv_release > 0.0 ? double(-len)/volenv_release : -1.0);
#endif
                  volenv_val += volenv_release > 0.0 ? double(-len)/volenv_release : -1.0;
                  if (volenv_val < 0.0001) {
                        synth->stop(this);
                        return;
                        }
                  break;

            case ENV_DELAY:
                  if (ticks < volenv_delay)
                        break;
                  volenv_state = ENV_ATTACK;

            case ENV_ATTACK:
                  if (volenv_attack > 0.0) {
                        volenv_val += len / volenv_attack;
                        if (volenv_val < 1.0)
                              break;
                        }
                  volenv_val   = 1.0;
                  volenv_state = ENV_HOLD;

            case ENV_HOLD:
                  if (ticks < volenv_hold)
                        break;
                  volenv_state = ENV_DECAY;

            case ENV_DECAY:
#if 0
printf("write %d DECAY env %f += %f\n",
    len, volenv_val, volenv_decay > 0.0 ? double(-len)/volenv_decay : -1.0);
#endif
                  volenv_val += volenv_decay > 0.0 ? double(-len)/volenv_decay : -1.0;
                  if (volenv_val <= volenv_sustain) {
                        volenv_val = volenv_sustain;
                        volenv_state = ENV_SUSTAIN;
                        }
            case ENV_OFF:
                  break;
            }

      // no need to go further if the final amplitude is zero
      if (volenv_val < 0.0001) {
            if (volenv_state != ENV_DELAY)
                  synth->stop(this);
            ticks += len;
            return;
            }

//printf(" mod %d env %f\n", modenv_state, modenv_val);

      switch (modenv_state) {
            case ENV_SKIP:
                  modenv_val = 1.0;
                  break;
            case ENV_SUSTAIN:
                  modenv_val = modenv_sustain;
                  break;
            case ENV_RELEASE:
                  modenv_val += modenv_release > 0.0 ? double(-len)/modenv_release : -1.0;
                  if (modenv_val <= 0.0) {
                        modenv_val = 0.0;
                        modenv_state = ENV_OFF;
                        }
                  break;
            case ENV_DELAY:
                  if (ticks < modenv_delay)
                        break;
                  modenv_state = ENV_ATTACK;

            case ENV_ATTACK:
                  modenv_val += modenv_attack > 0.0 ? double(len)/modenv_attack : 1.0;
                  if (modenv_val < 1.0)
                        break;
                  modenv_val = 1.0;
                  modenv_state = ENV_HOLD;

            case ENV_HOLD:
                  if (ticks < modenv_hold)
                        break;
                  modenv_state = ENV_DECAY;

            case ENV_DECAY:
                  modenv_val += modenv_decay > 0 ? double(-len)/modenv_decay : -1.0;
                  if (modenv_val <= modenv_sustain) {
                        modenv_val = modenv_sustain;
                        modenv_state = ENV_SUSTAIN;
                        }
                  break;
            case ENV_OFF:
                  break;
            }


      /* mod lfo */
      if (ticks >= modlfo_delay) {
            modlfo_val += (modlfo_incr*len);
            if (modlfo_val > 1.0) {
                  modlfo_incr = - modlfo_incr;
                  modlfo_val = (double) 2.0 - modlfo_val;
                  }
            else if (modlfo_val < -1.0) {
                  modlfo_incr = -modlfo_incr;
                  modlfo_val = (double) -2.0 - modlfo_val;
                  }
            }

      /* vib lfo */
      if (ticks >= viblfo_delay) {
            viblfo_val += (viblfo_incr*len);
            if (viblfo_val > (double) 1.0) {
                  viblfo_incr = -viblfo_incr;
                  viblfo_val = (double) 2.0 - viblfo_val;
                  }
            else if (viblfo_val < -1.0) {
                  viblfo_incr = -viblfo_incr;
                  viblfo_val = (double) -2.0 - viblfo_val;
                  }
            }

      /* calculate final amplitude
       * - initial gain
       * - amplitude envelope
       */
      double amp;
      double at = attenuation + modlfo_val * modlfo_to_vol;
      if (volenv_state == ENV_ATTACK) {
            amp = iiwu_cb2amp(int(at)) * volenv_val;
            }
      else {
            amp = iiwu_cb2amp(int(at + 960 * (1.0 - volenv_val)));
            }

      if (amp < 0.0001)             // CHECK
            synth->stop(this);

//      printf("amp %f = 2amp(at:%f + %f)\n", amp, attenuation, 960 * (1.0 - volenv_val));

      // cc7 - channel volume; cc11 - modulation
      double masterVol = pow(10, 2*log10(synth->masterVol()/16383.0));
      double chanVol   = pow(10, 2*log10(channel->cc(7)*channel->cc(11)/(127.0*127.0)));

      // dividing by 32768 to scale samples down to [-1, 1] interval
      amp = amp * masterVol * chanVol / 32768;
//      printf("amp %f %f %f (%d*%d)\n",
//         amp, masterVol, chanVol, channel->cc(7), channel->cc(11));

      /* calculate final pitch, or phase increment */
      double incr = iiwu_ct2hz(_pitch
		    + modlfo_val * modlfo_to_pitch
		    + viblfo_val * viblfo_to_pitch
		    + modenv_val * modenv_to_pitch) / root_pitch;

      Phase phase_incr(incr);
      Phase end_phase(incr * len);

      end_phase += phase;

      //--------------------------------------------
      // to loop or not to loop
      Phase end(end_offset);
      Phase loopstart(loop_start_offset);
      Phase loopend(loop_end_offset);

      const short* data = _sample->data();
      iiwu_interp_coeff_t* coeff = 0;

      if ((_SAMPLEMODE() == IIWU_LOOP_DURING_RELEASE) || ((_SAMPLEMODE() == IIWU_LOOP) && (volenv_state < ENV_RELEASE))) {
            if (end_phase < loopend) {
                  /* we're not close to the loop point, so don't check */
                  for (int i = 0; i < len; i++) {
                        int _index = phase.index();
#ifndef USE_TRUNCATION
                        coeff = &interp_coeff[phase.interp_index()];
#endif
                        INTERPOL(mono[i], amp, data, _index, coeff);
                        phase += phase_incr;
                        }
                  }
            else {

                  /* might be looping, so check */
                  for (int i = 0; i < len; i++) {
                        int _index = phase.index();
#ifndef USE_TRUNCATION
                        coeff = &interp_coeff[phase.interp_index()];
#endif
                        INTERPOL(mono[i], amp, data, _index, coeff);
                        phase += phase_incr;

                        /* if we passed the loopend, wrap around as
                           phase += loopstart - loopend */
                        if (phase > loopend) {
                              phase -= loopend;
                              phase += loopstart;
                              }
                        }
                  }
            }
      else {
            if (end_phase < end) {
                  /* no fear of end of buffer */
                  for (int i = 0; i < len; i++) {
                        int _index = phase.index();
#ifndef USE_TRUNCATION
                        coeff = &interp_coeff[phase.interp_index()];
#endif
                        INTERPOL(mono[i], amp, data, _index, coeff);
                        phase += phase_incr;
                        }
                  }
            else {
                  /* check for end of buffer */
                  for (int i = 0; i < len; i++) {
                        if (phase < end) {
                              int _index = phase.index();
#ifndef USE_TRUNCATION
                              coeff = &interp_coeff[phase.interp_index()];
#endif
                              INTERPOL(mono[i], amp, data, _index, coeff);
                              phase += phase_incr;
                              }
                        else {
                              // if it's passed the end, turn the synthesis process off
                              if (!switchOff) {
                                    volenv_state = ENV_OFF;
                                    switchOff = true;
                                    }
                              mono[i] = 0;
                              }
                        }
                  }
            }

      /* pass the sound thru the resonant filter */
      double _fres = iiwu_ct2hz(fres
		    + modlfo_val * modlfo_to_fc
		    + modenv_val * modenv_to_fc);

      /* I'm using 19900 as upper frequency because the default 13500
         cents returns 19912 Hz */
      if ((_fres <= 19900.0) && (q > 0.0)) {
            /* calculate filter coefficients
             *
             * FIXME: 1) i've picked up an old algorithm i've been using for 8
             * (?)  years. i'm not sure that the q-factor in soundfont units
             * corresponds to what this calculation is expecting. i'll use it
             * till i have something better. 2) the DLS specification is a
             * little more precise about the filter characteristics but they
             * use a not-so-standard-to-me definition of fc.
             * */
            double x, kb;
            double w, y, b1, b2, gain;
            x = PI * _fres / _sampleRate;
            kb = cos(x) / sin(x);   /* is there a way to avoid the cos and sin functions? */
            x = 1.0 + kb * kb + kb / q;
            b2 = (double) ((1.0 + kb * kb - kb / q) / x);
            b1 = (double) (2.0 * (1.0 - kb * kb) / x);
            gain = (double) (0.25 * (1.0 + b1 + b2));
            /* do the actual filtering */
            for (int i = 0; i < len; i++) {
                  w = gain * mono[i] - b1 * w1 - b2 * w2;
                  y = w + w1 + w1 + w2;
                  w2 = w1;
                  w1 = w;
                  mono[i] = y;
                  }
            }
      /* copy the final samples in the left and right buffers */

      /* panning */
      float amp_left  = iiwu_pan(pan, 1) * masterVol;
      float amp_right = iiwu_pan(pan, 0) * masterVol;
      for (int i = 0; i < len; i++) {
            left[i]  += amp_left  * mono[i];
            right[i] += amp_right * mono[i];
            }
      ticks += len;
      if (switchOff)
            synth->stop(this);
      }

//---------------------------------------------------------
//   optimize
//    in this function we calculate the values of all the
//    parameters. The parameters are converted to their most
//    useful unit for the DSP algorithm, for exemple, number
//    of samples instead of timecents. Some parameters keep
//    their "perceptual" unit and conversion will be done in
//    the DSP function. This is the case, for exemple, for
//    the pitch since it is modulated by the controllers in
//    cents.
//---------------------------------------------------------

void SynthProc::optimize()
      {
      float x;

      /* volume envelope
       *
       * - delay and hold times are converted to absolute number of samples
       * - sustain is converted to its absolute value
       * - attack, decay and release are converted to their increment per sample
       */
      volenv_sustain = iiwu_cb2amp(int(gen[GEN_VOLENVSUSTAIN].val));

      /* first, calculate relative sample times */
      volenv_delay   = int(_sampleRate * iiwu_tc2sec(gen[GEN_VOLENVDELAY].val));
      volenv_attack  = _sampleRate * iiwu_tc2sec(gen[GEN_VOLENVATTACK].val);
      volenv_hold    = _sampleRate * iiwu_tc2sec(gen[GEN_VOLENVHOLD].val);
      volenv_decay   = _sampleRate * iiwu_tc2sec(gen[GEN_VOLENVDECAY].val);
#if 0
printf("volenv_decay %f(%f) %f %d\n",
      gen[GEN_VOLENVDECAY].val,
      iiwu_tc2sec(gen[GEN_VOLENVDECAY].val),
      volenv_decay, _sampleRate);
#endif

      volenv_release = _sampleRate * iiwu_tc2sec(gen[GEN_VOLENVRELEASE].val);

      /* calc absolute time for hold */
      volenv_hold += volenv_delay + volenv_attack;

// printf("optimize delay %f  attack %f  hold %f decay %f release %f\n",
//   volenv_delay/44100.0, volenv_attack/44100.0, volenv_hold/44100.0,
//    volenv_decay/44100.0, volenv_release/44100.0);

      /* do we have to calculate an envelope? */
      volenv_state = (gen[GEN_VOLENVDELAY].flags ||
		      gen[GEN_VOLENVATTACK].flags ||
		      gen[GEN_VOLENVHOLD].flags ||
		      gen[GEN_VOLENVDECAY].flags ||
		      gen[GEN_VOLENVSUSTAIN].flags ||
		      gen[GEN_VOLENVRELEASE].flags) ? ENV_DELAY : ENV_SKIP;

      /* modulation envelope
       *
       * - delay and hold times are converted to absolute number of samples
       * - sustain is converted to its absolute value
       * - attack, decay and release are converted to their increment per sample
       */
      modenv_sustain = 1.0 - 0.001 * gen[GEN_MODENVSUSTAIN].val;
      if (modenv_sustain < 0) {
            modenv_sustain = 0;
            }
      else if (modenv_sustain > 1) {
            modenv_sustain = 1;
            }

      /* first, calculate relative sample times */
      modenv_delay   = int(double(_sampleRate) * iiwu_tc2sec(gen[GEN_MODENVDELAY].val));
      modenv_attack  = _sampleRate * iiwu_tc2sec(gen[GEN_MODENVATTACK].val);
      modenv_hold    = _sampleRate * iiwu_tc2sec(gen[GEN_MODENVHOLD].val);
      modenv_decay   = _sampleRate * iiwu_tc2sec(gen[GEN_MODENVDECAY].val);
      modenv_release = _sampleRate * iiwu_tc2sec(gen[GEN_MODENVRELEASE].val);

      /* calc absolute time for hold */
      modenv_hold += modenv_delay + modenv_attack;

      /* do we have to calculate an envelope? */
      modenv_state = (gen[GEN_MODENVDELAY].flags ||
		      gen[GEN_MODENVATTACK].flags ||
		      gen[GEN_MODENVHOLD].flags ||
		      gen[GEN_MODENVDECAY].flags ||
		      gen[GEN_MODENVSUSTAIN].flags ||
		      gen[GEN_MODENVRELEASE].flags) ? ENV_DELAY : ENV_SKIP;
      modenv_to_pitch = gen[GEN_MODENVTOPITCH].val;
      modenv_to_fc = gen[GEN_MODENVTOFILTERFC].val;

      /* pitch */
      /* the GEN_PITCH is a hack to fit the pitch bend controller into the
         modulator paradigm */
      gen[GEN_PITCH].val = 100 * key;
      _pitch = _GEN(GEN_PITCH) + 100 * _GEN(GEN_COARSETUNE) + _GEN(GEN_FINETUNE);

      /* root key */
      if (gen[GEN_OVERRIDEROOTKEY].val > -1) {
            root_pitch = 100.0 * gen[GEN_OVERRIDEROOTKEY].val;
            }
      else {
            root_pitch = (double)(_sample->origpitch) * 100.0 + _sample->pitchadj;
            }
      root_pitch = iiwu_ct2hz(root_pitch);
      root_pitch *= (double) _sampleRate / (double) _sample->samplerate;
      x = iiwu_ct2hz(_pitch) / root_pitch;

//DEBUG
//      printf("key=%x, coarse tune=%f, fine tune=%f, pitch=%f, root=%f, freq=%f\n",
//  	 key, _GEN(GEN_COARSETUNE), _GEN(GEN_FINETUNE), pitch(), root_pitch, x);

      /* sample start and ends points */
      start_offset = int((_sample->start
		+ gen[GEN_STARTADDROFS].val
		+ 32768 * gen[GEN_STARTADDRCOARSEOFS].val));

      end_offset = int((_sample->end
	      + gen[GEN_ENDADDROFS].val
	      + int(32768 * gen[GEN_ENDADDRCOARSEOFS].val)));

      loop_start_offset = int((_sample->loopstart
		     + gen[GEN_STARTLOOPADDROFS].val
		     + 32768 * gen[GEN_STARTLOOPADDRCOARSEOFS].val));

      loop_end_offset = int((_sample->loopend
		   + gen[GEN_ENDLOOPADDROFS].val
		   + 32768 * gen[GEN_ENDLOOPADDRCOARSEOFS].val));

      /* attenuation */
      if (gen[GEN_VELOCITY].flags) {
            attenuation = iiwu_vel2cb((int) gen[GEN_VELOCITY].val);
            }
      else {
            attenuation = iiwu_vel2cb(vel);
            if (gen[GEN_ATTENUATION].flags) {
                  attenuation += gen[GEN_ATTENUATION].val;
                  }
            }

      /* pan */
      pan = gen[GEN_PAN].val;

      /* mod lfo
       *
       * - the frequency is converted into a delta value, per sample
       * - the delay into a sample delay
       */
      modlfo_incr = 4.0 * iiwu_act2hz(gen[GEN_MODLFOFREQ].val) / _sampleRate;
      modlfo_delay = int(_sampleRate * iiwu_tc2sec(gen[GEN_MODLFODELAY].val));
      modlfo_to_pitch = gen[GEN_MODLFOTOPITCH].val;
      modlfo_to_vol = gen[GEN_MODLFOTOVOL].val;
      modlfo_to_fc = gen[GEN_MODLFOTOFILTERFC].val;

      /* vib lfo
       *
       * - the frequency is converted into a delta value, per samples
       * - the delay into a sample delay
       */
      viblfo_incr = 4.0 * iiwu_act2hz(gen[GEN_VIBLFOFREQ].val) / _sampleRate;
      viblfo_delay = int(_sampleRate * iiwu_tc2sec(gen[GEN_VIBLFODELAY].val));
      viblfo_to_pitch = gen[GEN_VIBLFOTOPITCH].val;

      /* filter
       *
       * - the resonance frequency is converted from absolute cents to midicents
       */
      fres = iiwu_hz2ct(iiwu_act2hz(gen[GEN_FILTERFC].val));
      q    = gen[GEN_FILTERQ].val / 10.0;
      }

//---------------------------------------------------------
//   update_param
//---------------------------------------------------------

void SynthProc::update_param(int _gen)
      {
      switch (_gen) {
            case GEN_PAN:
                  pan = _GEN(GEN_PAN);
                  break;
            case GEN_ATTENUATION:
                  if (gen[GEN_VELOCITY].flags) {
                        attenuation = iiwu_vel2cb((int) gen[GEN_VELOCITY].val);
                        }
                  else {
                        attenuation = iiwu_vel2cb(vel);
                        if (gen[GEN_ATTENUATION].flags) {
                              attenuation += gen[GEN_ATTENUATION].val;
                              }
                        }
                  break;
            case GEN_PITCH:
                  _pitch = _GEN(GEN_PITCH) + 100 * _GEN(GEN_COARSETUNE) + _GEN(GEN_FINETUNE);
                  break;
            }
      }

//---------------------------------------------------------
//   modulate
//    In this implementation, I want to make sure that all controllers
//    are event based: the parameter values of the DSP algorithm should
//    only be updates when a controller event arrived and not at every
//    iteration of the audio cycle.
//
//    The update is done in three steps:
//
//    - first, we look for all the modulators that have the changed
//    controller as a source. This will yield a list of generators that
//    will be changed because of the controller event.
//
//    - For every changed generator, calculate its new value. This is the
//    sum of its original value plus the values of al the attached
//    modulators.
//
//    - For every changed generator, convert its value to the correct
//    unit of the corresponding DSP parameter
//---------------------------------------------------------

void SynthProc::modulate(int cc, int ctrl, int /*val*/)
      {
/*    printf("Chan=%d, CC=%d, Src=%d, Val=%d\n", channel->channum, cc, ctrl, val); */

      for (int i = 0; i < mod_count; i++) {
            Mod* _mod = &mod[i];
            /* step 1: find all the modules that have the changed controller
             * as input source. */
            if (iiwu_mod_has_source(_mod, cc, ctrl)) {
                  int _gen      = _mod->dest;
                  double modval = 0.0;

                  /* step 2: for every changed modulator, calculate the modulation
                   * value of its associated generator */
                  for (int k = 0; k < mod_count; k++) {
                        if (_mod[k].dest ==  _gen) {
                              modval += _mod[k].get_value(channel, this);
                              }
                        }
                  gen[_gen].mod = modval;

                  /* step 3: now that we have the new value of the generator,
                   * recalculate the parameter values that are derived from the
                   * generator */
                  update_param(_gen);
                  }
            }
      }

//---------------------------------------------------------
//   noteoff
//---------------------------------------------------------

void SynthProc::noteoff()
      {
      bool sustain = channel->cc(0x40) >= 0x40;

//printf("note off-- %x %x sustain %d\n", key, vel, sustain);
      if (sustain) {
            _sustained = true;
            }
      else {
            volenv_state = ENV_RELEASE;
            modenv_state = ENV_RELEASE;
            }
      }

//---------------------------------------------------------
//   add_mod
//---------------------------------------------------------

void SynthProc::add_mod(Mod* _mod)
      {
      if (mod_count < IIWU_NUM_MOD) {
            mod[mod_count++].clone(_mod);
            }
      }

//---------------------------------------------------------
//   Mod::clone
//---------------------------------------------------------

void Mod::clone(Mod* src)
      {
      dest   = src->dest;
      src1   = src->src1;
      flags1 = src->flags1;
      src2   = src->src2;
      flags2 = src->flags2;
      amount = src->amount;
      }

//---------------------------------------------------------
//   Mod::set_source1
//---------------------------------------------------------

void Mod::set_source1(int src, int flags)
      {
      src1   = src;
      flags1 = flags;
      }

//---------------------------------------------------------
//   Mod::set_source2
//---------------------------------------------------------

void Mod::set_source2(int src, int flags)
      {
      src2   = src;
      flags2 = flags;
      }

//---------------------------------------------------------
//   get_value
//---------------------------------------------------------

double Mod::get_value(Channel* chan, SynthProc* sp)
      {
      double v1 = 0.0, v2 = 1.0;
      double range1 = 128.0, range2 = 128.0;

      /* get the initial value of the first source */
      if (src1 > 0) {
            if (flags1 & IIWU_MOD_CC) {
                  v1 = chan->cc(src1);
                  }
            else {  /* source 1 is one of the direct controllers */
                  switch (src1) {
                        case IIWU_MOD_VELOCITY:
                              v1 = sp->vel;
                              break;
                        case IIWU_MOD_KEY:
                              v1 = sp->key;
                              break;
                        case IIWU_MOD_KEYPRESSURE:
                              v1 = chan->key_pressure;
                              break;
                        case IIWU_MOD_CHANNELPRESSURE:
                              v1 = chan->channel_pressure;
                              break;
                        case IIWU_MOD_PITCHWHEEL:
                              v1 = chan->pitchBend();
                              range1 = 0x4000;
                              break;
                        case IIWU_MOD_PITCHWHEELSENS:
                              v1 = chan->pitch_wheel_sensitivity;
                              break;
                        default:
                              v1 = 0.0; 	
                        }
                  }

            /* transform the input value */
            switch (flags1 & 0x0f) {
                  case 0: /* linear, unipolar, positive */
                        v1 /= range1;
                        break;
                  case 1: /* linear, unipolar, negative */
                        v1 = 1.0 - v1 / range1;	  	
                        break;
                  case 2: /* linear, bipolar, positive */
                        v1 = -1.0 + 2.0 * v1 / range1;
                        break;
                  case 3: /* linear, bipolar, negative */
                        v1 = -1.0 + 2.0 * v1 / range1;
                        break;
                  case 4: /* concave, unipolar, positive */
                        v1 = iiwu_concave(v1);
                        break;
                  case 5: /* concave, unipolar, negative */
                        v1 = iiwu_concave(128 - v1);
                        break;
                  case 6: /* concave, bipolar, positive */
                        v1 = (v1 > 64)? iiwu_concave(2 * (v1 - 64)) : -iiwu_concave(2 * (64 - v1));
                        break;
                  case 7: /* concave, bipolar, negative */
                        v1 = (v1 > 64)? -iiwu_concave(2 * (v1 - 64)) : iiwu_concave(2 * (64 - v1));
                        break;
                  case 8: /* convex, unipolar, positive */
                        v1 = iiwu_convex(v1);
                        break;
                  case 9: /* convex, unipolar, negative */
                        v1 = 1.0 - iiwu_convex(v1);
                        break;
                  case 10: /* convex, bipolar, positive */
                        v1 = (v1 > 64)? -iiwu_convex(2 * (int(v1) - 64)) : iiwu_convex(2 * (64 - int(v1)));
                        break;
                  case 11: /* convex, bipolar, negative */
                        v1 = (v1 > 64)? -iiwu_convex(2 * (v1 - 64)) : iiwu_convex(2 * (64 - v1));
                        break;
                  case 12: /* switch, unipolar, positive */
                        v1 = (v1 >= 64)? 1.0 : 0.0;
                        break;
                  case 13: /* switch, unipolar, negative */
                        v1 = (v1 >= 64)? 0.0 : 1.0;
                        break;
                  case 14: /* switch, bipolar, positive */
                        v1 = (v1 >= 64)? 1.0 : -1.0;
                        break;
                  case 15: /* switch, bipolar, negative */
                        v1 = (v1 >= 64)? -1.0 : 1.0;
                        break;
                  }
            }
      else {
            return 0.0;
            }

      /* no need to go further */
      if (v1 == 0.0) {
            return 0.0;
            }

      /* get the second input source */
      if (src2 > 0) {
            if (flags2 & IIWU_MOD_CC) {
                  v2 = chan->cc(src2);
                  }
            else {
                  switch (src2) {
                        case IIWU_MOD_VELOCITY:
                              v2 = sp->vel;
                              break;
                        case IIWU_MOD_KEY:
                              v2 = sp->key;
                              break;
                        case IIWU_MOD_KEYPRESSURE:
                              v2 = chan->key_pressure;
                              break;
                        case IIWU_MOD_CHANNELPRESSURE:
                              v2 = chan->channel_pressure;
                              break;
                        case IIWU_MOD_PITCHWHEEL:
                              v2 = chan->pitchBend();
                              break;
                        case IIWU_MOD_PITCHWHEELSENS:
                              v2 = chan->pitch_wheel_sensitivity;
                              break;
                        default:
                              v1 = 0.0; 	
                        }
                  }

            /* transform the second input value */
            switch (flags2 & 0x0f) {
                  case 0: /* linear, unipolar, positive */
                        v2 /= range2;
                        break;
                  case 1: /* linear, unipolar, negative */
                        v2 = 1.0 - v2 / range2;	  	
                        break;
                  case 2: /* linear, bipolar, positive */
                        v2 = -1.0 + 2.0 * v2 / range2;
                        break;
                  case 3: /* linear, bipolar, negative */
                        v2 = -1.0 + 2.0 * v2 / range2;
                        break;
                  case 4: /* concave, unipolar, positive */
                        v2 = iiwu_concave(v2);
                        break;
                  case 5: /* concave, unipolar, negative */
                        v2 = iiwu_concave(128 - v2);
                        break;
                  case 6: /* concave, bipolar, positive */
                        v2 = (v2 > 64)? iiwu_concave(2 * (v2 - 64)) : -iiwu_concave(2 * (64 - v2));
                        break;
                  case 7: /* concave, bipolar, negative */
                        v2 = (v2 > 64)? -iiwu_concave(2 * (v2 - 64)) : iiwu_concave(2 * (64 - v2));
                        break;
                  case 8: /* convex, unipolar, positive */
                        v2 = iiwu_convex(v2);
                        break;
                  case 9: /* convex, unipolar, negative */
                        v2 = 1.0 - iiwu_convex(v2);
                        break;
                  case 10: /* convex, bipolar, positive */
                        v2 = (v2 > 64)? -iiwu_convex(2 * (v2 - 64)) : iiwu_convex(2 * (64 - v2));
                        break;
                  case 11: /* convex, bipolar, negative */
                        v2 = (v2 > 64)? -iiwu_convex(2 * (v2 - 64)) : iiwu_convex(2 * (64 - v2));
                        break;
                  case 12: /* switch, unipolar, positive */
                        v2 = (v2 >= 64)? 1.0 : 0.0;
                        break;
                  case 13: /* switch, unipolar, negative */
                        v2 = (v2 >= 64)? 0.0 : 1.0;
                        break;
                  case 14: /* switch, bipolar, positive */
                        v2 = (v2 >= 64)? 1.0 : -1.0;
                        break;
                  case 15: /* switch, bipolar, negative */
                        v2 = (v2 >= 64)? -1.0 : 1.0;
                        break;
                  }
            }
      else {
            v2 = 1.0;
            }
      /* it's as simple as that: */
      return amount * v1 * v2;
      }

//=========================================================
//    MESSS interface
//=========================================================

//---------------------------------------------------------
//   getPatchName
//---------------------------------------------------------

const char* ISynth::getPatchName(int ch, int /*hbank*/, int lbank, int prog, MType /*type*/)
      {
      if (ch == 9 && _gmMode) {
            const MidiPatch* patch = getFirstPatch(ch);
            return patch ? patch->name : 0;
            }
      for (SFont* sf = sfont; sf; sf = sf->next) {
            for (Preset* pr = sf->preset; pr; pr = pr->next) {
                  if (!((ch == 9) ^ (pr->hbank == 1)) && (lbank == -1 || int(pr->lbank) == lbank) && prog == int(pr->prog))
                        return pr->name;
                  }
            }
      return "---";
      }

//---------------------------------------------------------
//   getFirstPatch
//---------------------------------------------------------

const MidiPatch* ISynth::getFirstPatch(int ch) const
      {
      if (ch == 9 && _gmMode) {
            for (SFont* sf = sfont; sf; sf = sf->next) {
                  for (Preset* pr = sf->preset; pr; pr = pr->next) {
                        if (pr->hbank == 1)
                              return pr;
                        }
                  }
            return 0;
            }
      for (SFont* sf = sfont; sf; sf = sf->next) {
            for (Preset* pr = sf->preset; pr; pr = pr->next) {
                  if (!((ch == 9) ^ (pr->hbank == 1)))
                        return pr;
                  }
            }
      return 0;
      }

//---------------------------------------------------------
//   getNextPatch
//---------------------------------------------------------

const MidiPatch* ISynth::getNextPatch(int ch, const MidiPatch* mp) const
      {
      if (mp == 0)
            return getFirstPatch(ch);
      if (ch == 9 && _gmMode)
            return 0;
      Preset* spres = (Preset*) mp;
      SFont* sf = sfont;
      for (; sf; sf = sf->next) {
            Preset* pr = sf->preset;
            for (; pr; pr = pr->next) {
                  if (pr == spres) {
                        pr = pr->next;
                        for (;;) {
                              if (pr == 0) {
                                    sf = sf->next;
                                    if (sf == 0)
                                          return 0;
                                    pr = sf->preset;
                                    }
                              else {
                                    if (!((ch == 9) ^ (pr->hbank == 1)))
                                          return pr;
                                    pr = pr->next;
                                    }
                              }
                        }
                  }
            }
      return 0;
      }

//---------------------------------------------------------
//   getFirstParameter
//---------------------------------------------------------

bool ISynth::getFirstParameter(const char*& s, const char*& value) const
      {
      static char* pname = "fsoundfont";
      if (sfont) {
            s  = pname;
            value = sfont->filePath();
            nextSFont = sfont->next;
            return true;
            }
      return false;
      }

//---------------------------------------------------------
//   getNextParameter
//---------------------------------------------------------

bool ISynth::getNextParameter(const char*& s, const char*& value) const
      {
      if (nextSFont) {
            static char* pname = "soundfont";
            s  = pname;
            value = nextSFont->filePath();
            nextSFont = nextSFont->next;
            return true;
            }
      return false;
      }

//---------------------------------------------------------
//   setParameter
//---------------------------------------------------------

void ISynth::setParameter(const char* p, const char* value)
      {
      if (strcmp(p, "soundfont") == 0) {
            sysexSoundFont(SF_ADD, value);
            }
      else if (strcmp(p, "fsoundfont") == 0) {
            sysexSoundFont(SF_REPLACE, value);
            }
      else
            fprintf(stderr, "iiwu: setParameter(%s,%s): unknown param\n",
               p, value);
      }

