//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: seqmsg.cpp,v 1.2 2002/02/13 11:42:56 muse Exp $
//
//  (C) Copyright 1999/2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <stdio.h>
#include "song.h"
#include "seq.h"
#include "audioport.h"
#include "midiport.h"
#include "minstrument.h"
#include "sf/sndfile.h"
#include "app.h"
#include "amixer.h"
#include "tempo.h"
#include "sig.h"
#include "audiothread.h"

#include "mididev.h"
#include "audiodev.h"
#include "driver/alsamidi.h"
#include "midithread.h"

//---------------------------------------------------------
//   sendMessage
//    send request from gui to sequencer
//    wait until request is processed
//---------------------------------------------------------

bool MidiThread::sendMessage(const MidiMsg* m, bool doUndo)
      {
      if (doUndo)
           song->startUndo();
      sendMsg(m);
      if (doUndo)
            song->endMsgCmd();
      return false;
      }

//---------------------------------------------------------
//   msgSeek
//---------------------------------------------------------

void MidiThread::msgSeek(int val)
      {
      MidiMsg msg;
      msg.id = SEQM_SEEK;
      msg.a  = val;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgPlay
//---------------------------------------------------------

void MidiThread::msgPlay(bool val)
      {
      MidiMsg msg;
      msg.id = SEQM_PLAY;
      msg.a  = val;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgAddSynthi
//---------------------------------------------------------

void MidiThread::msgAddSynthI(SynthI* /*synth*/)
      {
      MidiMsg msg;
      msg.id = MIDI_ADD_SYNTHI;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   msgShowInstrumentGui
//---------------------------------------------------------

void MidiThread::msgShowInstrumentGui(MidiInstrument* instr, bool val)
      {
      MidiMsg msg;
      msg.id = MIDI_SHOW_INSTR_GUI;
      msg.p1 = instr;
      msg.a  = val;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   addTrack
//---------------------------------------------------------

void MidiThread::msgAddTrack(Track* track, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_ADD_TRACK;
      msg.p1 = track;
      sendMessage(&msg, doUndoFlag);
      if (track->type() == Track::WAVE) {
            if (audioMixer)
                  audioMixer->updateMixer();
            audioThread->msgAddRoute((WaveTrack*)track, song->master());
            }
      }

//---------------------------------------------------------
//   msgRemoveTracks
//---------------------------------------------------------

void MidiThread::msgRemoveTracks(bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_REMOVE_TRACKS;
      sendMessage(&msg, doUndoFlag);
      if (audioMixer)
            audioMixer->updateMixer();
      }

//---------------------------------------------------------
//   msgChangeTrack
//---------------------------------------------------------

void MidiThread::msgChangeTrack(Track* oldTrack, Track* newTrack, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_CHANGE_TRACK;
      msg.p1 = oldTrack;
      msg.p2 = newTrack;
      sendMessage(&msg, doUndoFlag);
      if (newTrack->type() == Track::WAVE) {
            if (audioMixer)
                  audioMixer->updateMixer();
            audioThread->msgAddRoute((WaveTrack*)newTrack, song->master());
            }
      }

//---------------------------------------------------------
//   msgMoveTrack
//    move track idx1 to slot idx2
//---------------------------------------------------------

void MidiThread::msgMoveTrack(int idx1, int idx2, bool doUndoFlag = true)
      {
      int n = song->tracks()->size();
      if (idx1 >= n || idx2 >= n)   // sanity check
            return;
      MidiMsg msg;
      msg.id = SEQM_MOVE_TRACK;
      msg.a = idx1;
      msg.b = idx2;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgAddPart
//---------------------------------------------------------

void MidiThread::msgAddPart(Part* part, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_ADD_PART;
      msg.p1 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgRemovePart
//---------------------------------------------------------

void MidiThread::msgRemovePart(Part* part, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_REMOVE_PART;
      msg.p1 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgChangePart
//---------------------------------------------------------

void MidiThread::msgChangePart(Part* oldPart, Part* newPart,
   bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_CHANGE_PART;
      msg.p1 = oldPart;
      msg.p2 = newPart;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgAddEvent
//---------------------------------------------------------

void MidiThread::msgAddEvent(Event* event, Part* part, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_ADD_EVENT;
      msg.p1 = event;
      msg.p2 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgDeleteEvent
//---------------------------------------------------------

void MidiThread::msgDeleteEvent(Event* event, Part* part, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_REMOVE_EVENT;
      msg.p1 = event;
      msg.p2 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgChangeEvent
//---------------------------------------------------------

void MidiThread::msgChangeEvent(Event* oe, Event* ne, Part* part,
   bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_CHANGE_EVENT;
      msg.p1 = oe;
      msg.p2 = ne;
      msg.p3 = part;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgAddTempo
//---------------------------------------------------------

void MidiThread::msgAddTempo(int tick, int tempo, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_ADD_TEMPO;
      msg.a = tick;
      msg.b = tempo;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgSetTempo
//---------------------------------------------------------

void MidiThread::msgSetTempo(int tick, int tempo, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_SET_TEMPO;
      msg.a = tick;
      msg.b = tempo;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgDeleteTempo
//---------------------------------------------------------

void MidiThread::msgDeleteTempo(int tick, int tempo, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_REMOVE_TEMPO;
      msg.a = tick;
      msg.b = tempo;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgAddSig
//---------------------------------------------------------

void MidiThread::msgAddSig(int tick, int z, int n, bool doUndoFlag = true)
      {
      MidiMsg msg;
      msg.id = SEQM_ADD_SIG;
      msg.a = tick;
      msg.b = z;
      msg.c = n;
      sendMessage(&msg, doUndoFlag);
      }

//---------------------------------------------------------
//   msgScanAlsaMidiPorts
//---------------------------------------------------------

void MidiThread::msgScanAlsaMidiPorts()
      {
      MidiMsg msg;
      msg.id = SEQM_SCAN_ALSA_MIDI_PORTS;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   undo
//---------------------------------------------------------

void MidiThread::undo()
      {
      MidiMsg msg;
      msg.id = SEQM_UNDO;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   redo
//---------------------------------------------------------

void MidiThread::redo()
      {
      MidiMsg msg;
      msg.id = SEQM_REDO;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   resetDevices
//---------------------------------------------------------

void MidiThread::resetDevices()
      {
      MidiMsg msg;
      msg.id = SEQM_RESET_DEVICES;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   initDevices
//---------------------------------------------------------

void MidiThread::msgInitDevices()
      {
printf("MidiThread::msgInitDevices\n");
      MidiMsg msg;
      msg.id = SEQM_INIT_DEVICES;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   localOff
//---------------------------------------------------------

void MidiThread::localOff()
      {
      MidiMsg msg;
      msg.id = SEQM_MIDI_LOCAL_OFF;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   ctrlChanged
//---------------------------------------------------------

void MidiThread::ctrlChanged(int port, int channel, int ctrl, int val)
      {
      MidiMsg msg;
      msg.id = SEQM_MIDI_CTRL;
      msg.port = port;
      msg.channel = channel;
      msg.ctrl = ctrl;
      msg.a = val;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   masterVolChanged
//---------------------------------------------------------

void MidiThread::masterVolChanged(int val)
      {
      MidiMsg msg;
      msg.id = SEQM_MIDI_MASTER_VOLUME;
      msg.b = val;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   setMidiDevice
//---------------------------------------------------------

void MidiThread::setMidiDevice(MidiPort* port, MidiDevice* device)
      {
      MidiMsg msg;
      msg.id = SEQM_SET_MIDI_DEVICE;
      msg.p1 = port;
      msg.p2 = device;
      sendMessage(&msg, false);
      }

//---------------------------------------------------------
//   playMidiEvent
//---------------------------------------------------------

void MidiThread::playMidiEvent(MidiEvent* event)
      {
      MidiMsg msg;
      msg.id = SEQM_PLAY_MIDI_EVENT;
      msg.p1 = event;
      sendMessage(&msg, false);
      }
