//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: seq.cpp,v 1.4 2002/02/27 15:53:45 muse Exp $
//
//  (C) Copyright 1999/2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <stdio.h>
#include <errno.h>

#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <assert.h>

#include <qtimer.h>

#include "seq.h"
#include <qstring.h>
#include <fcntl.h>
#include "xml.h"
#include "song.h"
#include "midi.h"
#include "device.h"
#include "globals.h"
#include "plugins/plugin.h"
#include "midictrl.h"
#include "midiport.h"
#include "audioport.h"
#include "driver/mididev.h"
#include "driver/audiodev.h"
#include "event.h"
#include "minstrument.h"
#include "audiomix.h"
#include "sf/sndfile.h"
#include "driver/alsamidi.h"
#include "synth.h"
#include "audiothread.h"
#include "audioprefetch.h"
#include "midithread.h"

//---------------------------------------------------------
//   Sequencer
//---------------------------------------------------------

Sequencer::Sequencer()
      {
      if (realTimePriority < sched_get_priority_min(SCHED_RR))
            realTimePriority = sched_get_priority_min(SCHED_RR);
      else if (realTimePriority > sched_get_priority_max(SCHED_RR))
            realTimePriority = sched_get_priority_max(SCHED_RR);

      midiThread = new MidiThread(realTimeScheduling,
         realTimePriority, lockMemory);
      audioThread = new AudioThread(realTimeScheduling,
         realTimePriority/3, lockMemory);
      audioPrefetch = new AudioPrefetch(realTimeScheduling,
         realTimePriority/4, lockMemory);
      }

//---------------------------------------------------------
//   start
//---------------------------------------------------------

void Sequencer::startThread()
      {
      if (!noAudio) {
            audioThread->start();
            audioPrefetch->start();
            }
      midiThread->start();
      }

//---------------------------------------------------------
//   stop
//---------------------------------------------------------

void Sequencer::stopThread()
      {
      midiThread->stop();
      if (!noAudio) {
            audioPrefetch->stop();
            audioThread->stop();
            }
      }

//---------------------------------------------------------
//   Sequencer
//---------------------------------------------------------

Sequencer::~Sequencer()
      {
      delete audioPrefetch;
      delete audioThread;
      delete midiThread;
      }

//---------------------------------------------------------
//   writeConfiguration
//---------------------------------------------------------

void Sequencer::writeConfiguration(int level, Xml& xml)
      {
      xml.tag(level++, "sequencer");

      xml.tag(level++, "metronom");
      xml.intTag(level, "premeasures", preMeasures);
      xml.intTag(level, "measurepitch", measureClickNote);
      xml.intTag(level, "measurevelo", measureClickVelo);
      xml.intTag(level, "beatpitch", beatClickNote);
      xml.intTag(level, "beatvelo", beatClickVelo);
      xml.intTag(level, "channel", clickChan);
      xml.intTag(level, "port", clickPort);

      xml.intTag(level, "precountEnable", precountEnableFlag);
      xml.intTag(level, "fromMastertrack", precountFromMastertrackFlag);
      xml.intTag(level, "signatureZ", precountSigZ);
      xml.intTag(level, "signatureN", precountSigN);
      xml.intTag(level, "prerecord", precountPrerecord);
      xml.intTag(level, "preroll", precountPreroll);
      xml.intTag(level, "midiClickEnable", midiClickFlag);
      xml.intTag(level, "audioClickEnable", audioClickFlag);
      xml.tag(level--, "/metronom");

      xml.intTag(level, "rcEnable",   rcEnable);
      xml.intTag(level, "rcStop",     rcStopNote);
      xml.intTag(level, "rcRecord",   rcRecordNote);
      xml.intTag(level, "rcGotoLeft", rcGotoLeftMarkNote);
      xml.intTag(level, "rcPlay",     rcPlayNote);

      SndFile* master = audioThread->masterMix();
      if (master) {
            SndFile* master = audioThread->masterMix();
            xml.tag(level++, "mixdown");
            xml.intTag(level, "record",     audioThread->mixdownRecord());
            xml.strTag(level, "path",       master->path().latin1());
            xml.intTag(level, "channels",   master->channels());
            xml.intTag(level, "format",     master->format());
            xml.intTag(level, "samplebits", master->sampleBits());
            xml.etag(level--, "mixdown");
            }

      for (int i = 0; i < MIDI_PORTS; ++i) {
            MidiPort* mport = &midiPorts[i];
            MidiDevice* dev = mport->device();
            if (dev) {
                  xml.tag(level++, "midiport idx=\"%d\"", i);
                  xml.strTag(level, "instrument", mport->instrument()->iname());
                  xml.strTag(level, "name",   dev->name());
                  xml.intTag(level, "record", dev->rwFlags() & 0x2 ? 1 : 0);
                  xml.etag(level--, "midiport");
                  }
            }
      AudioPort* port  = &audioPort;
      AudioDevice* dev = port->device();
      if (dev) {
            xml.tag(level++, "audioport idx=\"%d\"", 0);
            xml.strTag(level, "name",   dev->name().latin1());
            xml.intTag(level, "record", port->rwFlags() & 0x2 ? 1 : 0);
            xml.intTag(level, "play",   port->rwFlags() & 0x1);
            QString s;
            port->writeConfiguration(level, xml);
            xml.tag(level--, "/audioport");
            }

      for (int i = 0; i < AUDIO_GROUPS; ++i) {
            xml.tag(level++, "audiogroup idx=\"%d\"", i);
            song->group(i)->writeConfiguration(level, xml);
            xml.tag(level--, "/audiogroup");
            }

      xml.tag(level++, "audiomaster");
      song->master()->writeConfiguration(level, xml);
      xml.tag(level--, "/audiomaster");

      xml.tag(level, "/sequencer");
      }

//---------------------------------------------------------
//   loadConfigMetronom
//---------------------------------------------------------

void Sequencer::loadConfigMetronom(Xml& xml)
      {
      for (;;) {
            Xml::Token token = xml.parse();
            if (token == Xml::Error || token == Xml::End)
                  break;
            QString tag = xml.s1();
            switch (token) {
                  case Xml::TagStart:
                        if (tag == "premeasures")
                              preMeasures = xml.parseInt();
                        else if (tag == "measurepitch")
                              measureClickNote = xml.parseInt();
                        else if (tag == "measurevelo")
                              measureClickVelo = xml.parseInt();
                        else if (tag == "beatpitch")
                              beatClickNote = xml.parseInt();
                        else if (tag == "beatvelo")
                              beatClickVelo = xml.parseInt();
                        else if (tag == "channel")
                              clickChan = xml.parseInt();
                        else if (tag == "port")
                              clickPort = xml.parseInt();
                        else if (tag == "precountEnable")
                              precountEnableFlag = xml.parseInt();
                        else if (tag == "fromMastertrack")
                              precountFromMastertrackFlag = xml.parseInt();
                        else if (tag == "signatureZ")
                              precountSigZ = xml.parseInt();
                        else if (tag == "signatureN")
                              precountSigN = xml.parseInt();
                        else if (tag == "prerecord")
                              precountPrerecord = xml.parseInt();
                        else if (tag == "preroll")
                              precountPreroll = xml.parseInt();
                        else if (tag == "midiClickEnable")
                              midiClickFlag = xml.parseInt();
                        else if (tag == "audioClickEnable")
                              audioClickFlag = xml.parseInt();
                        else
                              xml.unknown("Metronome");
                        break;
                  case Xml::TagEnd:
                        if (tag == "metronom")
                              return;
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   readConfiguration
//---------------------------------------------------------

void Sequencer::readConfiguration(Xml& xml)
      {
      for (;;) {
            Xml::Token token = xml.parse();
            if (token == Xml::Error || token == Xml::End)
                  break;
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::TagStart:
                        if (tag == "metronom")
                              loadConfigMetronom(xml);
                        else if (tag == "midiport")
                              readConfigMidiPort(xml);
                        else if (tag == "audioport")
                              readConfigAudioPort(xml);
                        else if (tag == "audiogroup")
                              readConfigAudioGroup(xml);
                        else if (tag == "audiomaster")
                              readConfigAudioMaster(xml);
                        else if (tag == "rcStop")
                              rcStopNote = xml.parseInt();
                        else if (tag == "rcEnable")
                              rcEnable = xml.parseInt();
                        else if (tag == "rcRecord")
                              rcRecordNote = xml.parseInt();
                        else if (tag == "rcGotoLeft")
                              rcGotoLeftMarkNote = xml.parseInt();
                        else if (tag == "rcPlay")
                              rcPlayNote = xml.parseInt();
                        else if (tag == "mixdown")
                              readMixdown(xml);
                        else
                              xml.unknown("Seq");
                        break;
                  case Xml::TagEnd:
                        if (tag == "sequencer") {
                              return;
                              }
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   readMixdown
//---------------------------------------------------------

void Sequencer::readMixdown(Xml& xml)
      {
      QString path;
      int channels   = 2;
      int format     = SF_FORMAT_WAV | SF_FORMAT_PCM;
      int samplebits = 16;

      for (;;) {
            Xml::Token token = xml.parse();
            if (token == Xml::Error || token == Xml::End)
                  break;
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::TagStart:
                        if (tag == "path")
                              path = xml.parse1();
                        else if (tag == "channels")
                              channels = xml.parseInt();
                        else if (tag == "format")
                              format = xml.parseInt();
                        else if (tag == "samplebits")
                              samplebits = xml.parseInt();
                        else if (tag == "record") {
                              bool record = xml.parseInt();
                              if (record)
                                    audioThread->msgOpenMixdownRecord();
                              }
                        else
                              xml.unknown("mixdown");
                        break;
                  case Xml::TagEnd:
                        if (tag == "mixdown") {
                              SndFile* sf = new SndFile(path);
                              sf->setFormat(format, channels, sampleRate, samplebits);
                              audioThread->msgSetMixdownFile(sf);
                              return;
                              }
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   readConfigMidiPort
//---------------------------------------------------------

void Sequencer::readConfigMidiPort(Xml& xml)
      {
      int idx = 0;
      QString device;
      QString instrument;
      bool recFlag = false;
      bool thruFlag = false;

      for (;;) {
            Xml::Token token = xml.parse();
            if (token == Xml::Error || token == Xml::End)
                  break;
            QString tag = xml.s1();
            switch (token) {
                  case Xml::TagStart:
                        if (tag == "name")
                              device = xml.parse1();
                        else if (tag == "record")
                              recFlag = xml.parseInt();
                        else if (tag == "instrument")
                              instrument = xml.parse1();
                        else if (tag == "midithru")
                              thruFlag = xml.parseInt(); // obsolete
                        else
                              xml.unknown("MidiDevice");
                        break;
                  case Xml::Attribut:
                        if (tag == "idx") {
                              idx = xml.s2().toInt();
                              }
                        break;
                  case Xml::TagEnd:
                        if (tag == "midiport") {
                              if (idx > MIDI_PORTS) {
                                    fprintf(stderr, "bad midi port %d (>%d)\n",
                                       idx, MIDI_PORTS);
                                    idx = 0;
                                    }
                              MidiDevice* dev = midiDevices.find(device);
                              MidiPort* mp = &midiPorts[idx];
                              if (dev) {
                                    mp->setrwFlags(recFlag ? 3 : 1);
                                    midiThread->setMidiDevice(mp, dev);
                                    }
                              midiPorts[idx].setInstrument(
                                 registerMidiInstrument(instrument)
                                 );
                              return;
                              }
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   readConfigAudioPort
//---------------------------------------------------------

void Sequencer::readConfigAudioPort(Xml& xml)
      {
      int idx;
      QString device;
      AudioDevice* dev;
      AudioPort* port = 0;
      int rwFlags = 0;

      for (;;) {
            Xml::Token token = xml.parse();
            if (token == Xml::Error || token == Xml::End)
                  break;
            QString tag = xml.s1();
            switch (token) {
                  case Xml::TagStart:
                        if (tag == "name")
                              device = xml.parse1();
                        else if (tag == "record")
                              rwFlags |= xml.parseInt() ? 2 : 0;
                        else if (tag == "play")
                              rwFlags |= xml.parseInt() ? 1 : 0;
                        else if (tag == "audionode")
                              port->readConfiguration(xml);
                        else
                              xml.unknown("AudioDevice");
                        break;
                  case Xml::Attribut:
                        if (tag == "idx") {
                              idx = xml.s2().toInt();
                              idx = 0;
                              port = &audioPort;
                              }
                        break;
                  case Xml::TagEnd:
                        if (tag == "audioport") {
                              if (!device.isEmpty()) {
                                    dev = audioDevices.find(device);
                                    if (dev) {
                                          port->setrwFlags(rwFlags);
                                          audioThread->msgSetAudioDevice(port, dev);
                                          }
                                    audioThread->msgAddRoute(port, port->route());
                                    }
                              return;
                              }
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   readConfigAudioGroup
//---------------------------------------------------------

void Sequencer::readConfigAudioGroup(Xml& xml)
      {
      AudioMixer* mixer = 0;
      for (;;) {
            Xml::Token token = xml.parse();
            if (token == Xml::Error || token == Xml::End)
                  break;
            QString tag = xml.s1();
            switch (token) {
                  case Xml::TagStart:
                        if (tag == "audionode")
                              mixer->readConfiguration(xml);
                        else
                              xml.unknown("AudioGroup");
                        break;
                  case Xml::Attribut:
                        if (tag == "idx") {
                              int idx = xml.s2().toInt();
                              if (idx > AUDIO_GROUPS) {
                                    fprintf(stderr, "bad audio group %d (>%d)\n",
                                       idx, AUDIO_GROUPS);
                                    idx = 0;
                                    }
                              mixer = song->group(idx);
                              }
                        break;
                  case Xml::TagEnd:
                        if (tag == "audiogroup") {
                              audioThread->msgAddRoute(mixer, mixer->route());
                              return;
                              }
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   readConfigAudioMaster
//---------------------------------------------------------

void Sequencer::readConfigAudioMaster(Xml& xml)
      {
      AudioMixer* mixer = song->master();
      for (;;) {
            Xml::Token token = xml.parse();
            if (token == Xml::Error || token == Xml::End)
                  break;
            QString tag = xml.s1();
            switch (token) {
                  case Xml::TagStart:
                        if (tag == "audionode")
                              mixer->readConfiguration(xml);
                        else
                              xml.unknown("AudioMaster");
                        break;
                  case Xml::TagEnd:
                        if (tag == "audiomaster") {
//                              song->msgAddRoute(*mixer, mixer->route());
//                              mixer.setRoute(audioPort); // override settings
                              return;
                              }
                  default:
                        break;
                  }
            }
      }

