//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: plugin.cpp,v 1.2 2002/02/08 09:56:29 muse Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <stdio.h>
#include <dlfcn.h>

#include <qwidget.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qsignalmapper.h>
#include <qpushbutton.h>
#include <qlistview.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qwhatsthis.h>

#include "filedialog.h"
#include "slider.h"
#include "plugin.h"
#include "xml.h"
#include <qdir.h>
#include "icons.h"

PluginList plugins;

static const char* preset_file_pattern[] = {
      "presets (*.pre *.pre.gz *.pre.bz2)",
      "All Files (*)",
      0
      };

//---------------------------------------------------------
//   Plugin
//---------------------------------------------------------

Plugin::Plugin(const QString& n,
   LADSPA_Descriptor_Function df, const LADSPA_Descriptor* d)
   : libname(n), ladspa(df), plugin(d)
      {
      _inports = 0;
      _outports = 0;
      for (unsigned k = 0; k < d->PortCount; ++k) {
            LADSPA_PortDescriptor pd = d->PortDescriptors[k];
            if (pd &  LADSPA_PORT_CONTROL)
                  continue;
            if (pd &  LADSPA_PORT_INPUT)
                  ++_inports;
            else
                  ++_outports;
            }
      }

//---------------------------------------------------------
//   loadPluginLib
//---------------------------------------------------------

static void loadPluginLib(QFileInfo* fi)
      {
//      void* handle = dlopen(fi->filePath().data(), RTLD_LAZY);
      void* handle = dlopen(fi->filePath().data(), RTLD_NOW);
      if (handle == 0) {
            fprintf(stderr, "dlopen(%s) failed: %s\n",
              fi->filePath().data(), dlerror());
            return;
            }
      LADSPA_Descriptor_Function ladspa = (LADSPA_Descriptor_Function)dlsym(handle, "ladspa_descriptor");

      if (!ladspa) {
            const char *txt = dlerror();
            if (txt) {
                  fprintf(stderr,
                        "Unable to find ladspa_descriptor() function in plugin "
                        "library file \"%s\": %s.\n"
                        "Are you sure this is a LADSPA plugin file?\n",
                        fi->filePath().data(),
                        txt);
                  exit(1);
                  }
            }
      const LADSPA_Descriptor* descr;
      for (int i = 0;; ++i) {
            descr = ladspa(i);
            if (descr == NULL)
                  break;
            LADSPA_Properties properties = descr->Properties;
            int ai = 0;
            int ao = 0;
            for (unsigned k = 0; k < descr->PortCount; ++k) {
                  LADSPA_PortDescriptor pd = descr->PortDescriptors[k];
                  if (pd &  LADSPA_PORT_CONTROL)
                        continue;
                  if (pd &  LADSPA_PORT_INPUT)
                        ++ai;
                  else
                        ++ao;
                  }
            if (LADSPA_IS_INPLACE_BROKEN(properties)) {
                  if (debugMsg) {
      	            fprintf(stderr,
	      	         "Plugin \"%s\" is not capable of in-place processing and "
		               "therefore cannot be used by this program.\n",
		               descr->Name);
                        }
                  }
            else if (ai != ao) {
                  if (debugMsg)
	                  fprintf(stderr,
                           "Plugin \"%s\" cannot be used: inports(%d) != outports(%d).\n",
		               descr->Name, ai, ao);
                  }
            else
                  plugins.add(fi->baseName(), ladspa, descr);
            }
      }

//---------------------------------------------------------
//   loadPluginDir
//---------------------------------------------------------

static void loadPluginDir(const QString& s)
      {
      printf("load plugin dir <%s>\n", s.latin1());
      QDir pluginDir(s, "*.so", QDir::Files);
      if (pluginDir.exists()) {
            const QFileInfoList* list = pluginDir.entryInfoList();
            QFileInfoListIterator it(*list);
            QFileInfo* fi;
            while((fi = it.current())) {
                  loadPluginLib(fi);
                  ++it;
                  }
            }
      }

//---------------------------------------------------------
//   initPlugins
//---------------------------------------------------------

void initPlugins()
      {
      loadPluginDir(museGlobalLib + "/plugins");

      char* ladspaPath = getenv("LADSPA_PATH");
      if (ladspaPath == 0)
            return;

      char* p = ladspaPath;
      while (*p != '\0') {
            char* pe = p;
            while (*pe != ':' && *pe != '\0')
                  pe++;

            int n = pe - p;
            if (n) {
                  char* buffer = new char[n + 1];
                  strncpy(buffer, p, n);
                  buffer[n] = '\0';
                  loadPluginDir(QString(buffer));
                  delete buffer;
                  }
            p = pe;
            if (*p == ':')
                  p++;
            }
      }

//---------------------------------------------------------
//   find
//---------------------------------------------------------

Plugin* PluginList::find(const QString& file, const QString& name)
      {
      for (iPlugin i = begin(); i != end(); ++i) {
            if ((file == i->lib()) && (name == i->label()))
                  return &*i;
            }
      printf("Plugin <%s> not found\n", name.data());
      return 0;
      }

//---------------------------------------------------------
//   find
//---------------------------------------------------------

PluginI* Pipeline::find(int idx) const
      {
      int i = 0;
      for (ciPluginI pi = begin(); pi != end(); ++pi, ++i) {
            if (i == idx)
                  return *pi;
            }
      return 0;
      }

//---------------------------------------------------------
//   insert
//---------------------------------------------------------

void Pipeline::insert(PluginI* plugin, int index)
      {
      if (plugin == 0) {
            remove(index);
            return;
            }
      int i = 0;
      for (iPluginI pi = begin(); pi != end(); ++pi, ++i) {
            if (i == index) {
                  *pi = plugin;
                  return;
                  }
            }
      for (; i < index; ++i)
            push_back(0);
      push_back(plugin);
      }

//---------------------------------------------------------
//   remove
//---------------------------------------------------------

void Pipeline::remove(int index)
      {
      int i = 0;
      for (iPluginI pi = begin(); pi != end(); ++pi, ++i) {
            if (i == index) {
                  delete *pi;
                  *pi = 0;
                  break;
                  }
            }
      }

//---------------------------------------------------------
//   isOn
//---------------------------------------------------------

bool Pipeline::isOn(int idx) const
      {
      PluginI* p = find(idx);
      if (p)
            return p->on();
      return false;
      }

//---------------------------------------------------------
//   setOn
//---------------------------------------------------------

void Pipeline::setOn(int idx, bool flag)
      {
      PluginI* p = find(idx);
      if (p) {
            p->setOn(flag);
            }
      }

//---------------------------------------------------------
//   label
//---------------------------------------------------------

QString Pipeline::label(int idx) const
      {
      PluginI* p = find(idx);
      if (p)
            return p->label();
      return QString("");
      }

//---------------------------------------------------------
//   name
//---------------------------------------------------------

QString Pipeline::name(int idx) const
      {
      PluginI* p = find(idx);
      if (p)
            return p->name();
      return QString("");
      }

//---------------------------------------------------------
//   showGui
//---------------------------------------------------------

void Pipeline::showGui(int idx, bool flag)
      {
      PluginI* p = find(idx);
      if (p) {
            printf("show gui %d\n", flag);
            p->showGui(flag);
            }
      }

//---------------------------------------------------------
//   guiVisible
//---------------------------------------------------------

bool Pipeline::guiVisible(int idx)
      {
      PluginI* p = find(idx);
      if (p)
            return p->guiVisible();
      return false;
      }

//---------------------------------------------------------
//   apply
//---------------------------------------------------------

void Pipeline::apply(float* buffer, int segmentSize, int /* channel */)
      {
      for (iPluginI ip = begin(); ip != end(); ++ip) {
            PluginI* p = *ip;
            if (p && p->on()) {
                  p->connect(buffer, segmentSize);
                  p->apply(segmentSize);
                  }
            }
      }

//---------------------------------------------------------
//   PluginI
//---------------------------------------------------------

void PluginI::init()
      {
      plugin            = 0;
      instances         = 0;
      handle            = 0;
      controls          = 0;
      controlPorts      = 0;
      gui               = 0;
      _on               = true;
      initControlValues = false;
      }

PluginI::PluginI()
      {
      init();
      }

//---------------------------------------------------------
//   PluginI
//---------------------------------------------------------

PluginI::~PluginI()
      {
      if (gui)
            delete gui;
      if (controls)
            delete controls;
      if (handle)
            delete handle;
      }

//---------------------------------------------------------
//   initPluginInstance
//    return true on error
//---------------------------------------------------------

bool PluginI::initPluginInstance(Plugin* plug, int channel)
      {
      if (plug == 0) {
            printf("initPluginInstance: zero plugin\n");
            return true;
            }
      plugin = plug;

      if (plug->inports() != plug->outports()) {
            printf("initPluginInstance: inports != outports\n");
            return true;
            }
      if (channel % plug->inports()) {
            printf("initPluginInstance: cannot map %d ports to %d channel\n",
               plug->inports(), channel);
            return true;
            }

//      instances = (channel+plug->inports()-1)/plug->inports();
      instances = channel/plug->inports();
      handle    = new LADSPA_Handle[instances];
      for (int i = 0; i < instances; ++i) {
            handle[i] = plugin->instantiate();
            if (handle[i] == 0)
                  return true;
            }

      controlPorts = 0;
      int ports    = plugin->ports();

      for (int k = 0; k < ports; ++k) {
            LADSPA_PortDescriptor pd = plugin->portd(k);
            if (pd & LADSPA_PORT_CONTROL)
                  ++controlPorts;
            }
      controls = new Port[controlPorts];
      for (int i = 0; i < controlPorts; ++i) {
            controls[i].val    = 1.0;
            controls[i].tmpVal = 1.0;
            }
      int curPort = 0;
      for (int k = 0; k < ports; ++k) {
            LADSPA_PortDescriptor pd = plugin->portd(k);
            if (pd & LADSPA_PORT_CONTROL) {
                  for (int i = 0; i < instances; ++i)
                        plugin->connectPort(handle[i], k, &controls[curPort].val);
                  controls[curPort].idx = k;
                  ++curPort;
                  }
            }
      return false;
      }

//---------------------------------------------------------
//   connect
//---------------------------------------------------------

void PluginI::connect(float* l, int segmentSize)
      {
      float* ip = l;
      float* op = l;
      for (int i = 0; i < instances; ++i) {
            for (int k = 0; k < plugin->ports(); ++k) {
                  if (isAudioIn(k)) {
                        plugin->connectPort(handle[i], k, ip);
                        ip += segmentSize;
                        }
                  if (isAudioOut(k)) {
                        plugin->connectPort(handle[i], k, op);
                        op += segmentSize;
                        }
                  }
            }
      }

//---------------------------------------------------------
//   setControl
//---------------------------------------------------------

bool PluginI::setControl(const QString& s, double val)
      {
      for (int i = 0; i < controlPorts; ++i) {
            if (plugin->portName(controls[i].idx) == s) {
                  controls[i].val = controls[i].tmpVal = val;
                  return false;
                  }
            }
      return true;
      }

//---------------------------------------------------------
//   saveConfiguration
//---------------------------------------------------------

void PluginI::writeConfiguration(int level, Xml& xml)
      {
      xml.tag(level++, "plugin file=\"%s\" label=\"%s\" channel=\"%d\"",
         plugin->lib().latin1(), plugin->label().latin1(), instances * plugin->inports());
      for (int i = 0; i < controlPorts; ++i) {
            int idx = controls[i].idx;
            xml.tag(level, "control name=\"%s\" val=\"%f\" /",
               plugin->portName(idx), controls[i].tmpVal);
            }
      if (_on == false)
            xml.intTag(level, "on", _on);
      xml.tag(level--, "/plugin");
      }

//---------------------------------------------------------
//   loadControl
//---------------------------------------------------------

bool PluginI::loadControl(Xml& xml)
      {
      QString file;
      QString label;
      QString name="mops";
      double val = 0.0;

      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();

            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return true;
                  case Xml::TagStart:
                        xml.unknown("PluginI-Control");
                        break;
                  case Xml::Attribut:
                        if (tag == "name")
                              name = xml.s2();
                        else if (tag == "val")
                              val = xml.s2().toDouble();
                        break;
                  case Xml::TagEnd:
                        if (tag == "control") {
                              if (setControl(name, val)) {
                                    printf("config plugin: val <%s>=%f not found\n",
                                       name.latin1(), val);
                                    return false;
                                    }
                              initControlValues = true;
                              }
                        return true;
                  default:
                        break;
                  }
            }
      return true;
      }

//---------------------------------------------------------
//   readConfiguration
//    return true on error
//---------------------------------------------------------

bool PluginI::readConfiguration(Xml& xml)
      {
      QString file;
      QString label;
      instances = 1;

      for (;;) {
            Xml::Token token(xml.parse());
            const QString& tag(xml.s1());
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return true;
                  case Xml::TagStart:
                        if (plugin == 0) {
                              plugin = plugins.find(file, label);
                              if (plugin && initPluginInstance(plugin, instances)) {
                                    plugin = 0;
                                    xml.parse1();
                                    break;
                                    }
                              }
                        if (tag == "control")
                              loadControl(xml);
                        else if (tag == "on")
                              _on = xml.parseInt();
                        else
                              xml.unknown("PluginI");
                        break;
                  case Xml::Attribut:
                        if (tag == "file")
                              file = xml.s2();
                        else if (tag == "label")
                              label = xml.s2();
                        else if (tag == "channel")
                              instances = xml.s2().toInt();
                        break;
                  case Xml::TagEnd:
                        if (tag == "plugin") {
                              if (plugin == 0) {
                                    plugin = plugins.find(file, label);
                                    if (plugin && initPluginInstance(plugin, instances))
                                          return true;
                                    }
                              return false;
                              }
                        return true;
                  default:
                        break;
                  }
            }
      return true;
      }

//---------------------------------------------------------
//   showGui
//---------------------------------------------------------

void PluginI::showGui()
      {
      if (gui == 0)
            makeGui();
      if (gui->isVisible())
            gui->hide();
      else
            gui->show();
      }

void PluginI::showGui(bool flag)
      {
      if (flag) {
            if (gui == 0)
                  makeGui();
            gui->show();
            }
      else {
            if (gui)
                  gui->hide();
            }
      }

//---------------------------------------------------------
//   guiVisible
//---------------------------------------------------------

bool PluginI::guiVisible()
      {
      return gui != 0 && gui->isVisible();
      }


//---------------------------------------------------------
//   makeGui
//---------------------------------------------------------

void PluginI::makeGui()
      {
      gui = new PluginGui(this);
      }

//---------------------------------------------------------
//   apply
//---------------------------------------------------------

void PluginI::apply(int n)
      {
      for (int i = 0; i < controlPorts; ++i)
            controls[i].val = controls[i].tmpVal;
      for (int i = 0; i < instances; ++i)
            plugin->apply(handle[i], n);
      }

//---------------------------------------------------------
//   PluginDialog
//    select Plugin dialog
//---------------------------------------------------------

PluginDialog::PluginDialog(QWidget* parent=0, const char* name=0, bool modal=true)
  : QDialog(parent, name, modal)
      {
      setCaption("MusE: select plugin");
      QVBoxLayout* layout = new QVBoxLayout(this);

      pList  = new QListView(this);
      pList->setSorting(-1);
      pList->setAllColumnsShowFocus(true);
      pList->addColumn("Lib",   110);
      pList->addColumn("Label", 110);
      pList->addColumn("AI",    30);
      pList->addColumn("AO",    30);
      pList->addColumn("CI",    30);
      pList->addColumn("CO",    30);
      pList->setColumnWidthMode(1, QListView::Maximum);

      for (iPlugin i = plugins.begin(); i != plugins.end(); ++i) {
            int ai = 0;
            int ao = 0;
            int ci = 0;
            int co = 0;
            for (int k = 0; k < i->ports(); ++k) {
                  LADSPA_PortDescriptor pd = i->portd(k);
                  if (pd &  LADSPA_PORT_CONTROL) {
                        if (pd &  LADSPA_PORT_INPUT)
                              ++ci;
                        else
                              ++co;
                        }
                  else {
                        if (pd &  LADSPA_PORT_INPUT)
                              ++ai;
                        else
                              ++ao;
                        }
                  }
            new QListViewItem(pList, i->lib(),
               i->label(),
               QString().setNum(ai),
               QString().setNum(ao),
               QString().setNum(ci),
               QString().setNum(co));
            }

      layout->addWidget(pList);

      //---------------------------------------------------
      //  Ok/Cancel Buttons
      //---------------------------------------------------

      QBoxLayout* w5 = new QHBoxLayout;
      layout->addLayout(w5);

      QPushButton* okB     = new QPushButton(tr("Ok"), this);
      okB->setDefault(true);
      QPushButton* cancelB = new QPushButton(tr("Cancel"), this);
      okB->setFixedWidth(80);
      cancelB->setFixedWidth(80);
      w5->addWidget(okB);
      w5->addSpacing(12);
      w5->addWidget(cancelB);
      w5->addStretch(1);

      connect(cancelB, SIGNAL(clicked()), SLOT(reject()));
      connect(okB,     SIGNAL(clicked()), SLOT(accept()));
      }

//---------------------------------------------------------
//   value
//---------------------------------------------------------

Plugin* PluginDialog::value()
      {
      QListViewItem* item = pList->selectedItem();
      if (item)
            return plugins.find(item->text(0), item->text(1));
      return 0;
      }
#if 0
PluginI* PluginDialog::value()
      {
      QListViewItem* item = pList->selectedItem();
      if (item) {
            Plugin* plugin = plugins.find(item->text(0), item->text(1));
            if (plugin) {
                  PluginI* plugi = new PluginI();
                  if (plugi->initPluginInstance(plugin)) {
                        printf("cannot intantiate plugin <%s/%s>\n",
                           item->text(0).latin1(), item->text(1).latin1());
                        delete plugi;
                        plugi = 0;
                        }
                  else
                        printf("Plugin <%s/%s>\n",
                           plugi->lib().latin1(),
                           plugi->label().latin1());
                  return plugi;
                  }
            else
                  printf("plugin <%s/%s> not found\n",
                     item->text(0).latin1(), item->text(1).latin1());
            }
      return 0;
      }
#endif

//---------------------------------------------------------
//   getPlugin
//---------------------------------------------------------

Plugin* PluginDialog::getPlugin(QWidget* parent)
      {
      PluginDialog* dialog = new PluginDialog(parent);
      if (dialog->exec())
            return dialog->value();
      return 0;
      }

//---------------------------------------------------------
//   activate
//---------------------------------------------------------

void PluginI::activate()
      {
      for (int i = 0; i < instances; ++i)
            plugin->activate(handle[i]);
      if (initControlValues) {
            for (int i = 0; i < controlPorts; ++i) {
                  controls[i].val = controls[i].tmpVal;
                  }
            }
      else {
            //
            // get initial control values from plugin
            //
            for (int i = 0; i < controlPorts; ++i) {
                  controls[i].tmpVal = controls[i].val;
                  }
            }
      }

const char* presetOpenText = "<img source=\"fileopen\"> "
      "Click this button to load a saved <em>preset</em>.";
const char* presetSaveText = "Click this button to save cuurent parameter "
      "settings as a <em>preset</em>.  You will be prompted for a file name.";

//---------------------------------------------------------
//   PluginGui
//---------------------------------------------------------

PluginGui::PluginGui(PluginI* p)
   : MainWindow(0)
      {
      plugin = p;
      setCaption(plugin->name());

      QToolBar* tools = new QToolBar("File Buttons", this);
      QToolButton * fileOpen = new QToolButton(*openIcon, "Load Preset",
					    QString::null, this, SLOT(load()),
					    tools, "open new song" );

      QToolButton * fileSave = new QToolButton(*saveIcon, "Save Preset",
					    QString::null,
					    this, SLOT(save()),
					    tools, "save current song" );

      QWhatsThis::whatsThisButton(tools);
      QWhatsThis::add(fileOpen, presetOpenText);
      QMimeSourceFactory::defaultFactory()->setPixmap("fileopen", *openIcon );
      QWhatsThis::add(fileSave, presetSaveText);

      QWidget* mw = new QWidget(this);
      setCentralWidget(mw);
      QGridLayout* grid = new QGridLayout(mw);
      grid->setSpacing(3);

      QSignalMapper* mapper = new QSignalMapper(mw);
      connect(mapper, SIGNAL(mapped(int)), SLOT(parameterChanged(int)));

      int n = plugin->parameters();
      slider   = new Slider*[n];
      valLabel = new QLabel*[n];
      int style = Slider::BgTrough | Slider::BgSlot;
      for (int i = 0; i < n; ++i) {
            LADSPA_PortRangeHint range = plugin->range(i);
            QLabel* label = new QLabel(plugin->paramName(i), mw);
            label->setFixedHeight(20);
            double lower = 0.0;     // default values
            double upper = 1.0;
// #define LADSPA_IS_HINT_TOGGLED(x)
// #define LADSPA_IS_HINT_LOGARITHMIC(x)
// #define LADSPA_IS_HINT_INTEGER(x)

// printf("LADSPA port %d hint %x  %f-%f\n",
//   i, range.HintDescriptor, range.LowerBound, range.UpperBound);

            if (LADSPA_IS_HINT_BOUNDED_BELOW(range.HintDescriptor))
                  lower = range.LowerBound;
            if (LADSPA_IS_HINT_BOUNDED_ABOVE(range.HintDescriptor))
                  upper = range.UpperBound;
            if (LADSPA_IS_HINT_SAMPLE_RATE(range.HintDescriptor)) {
                  lower *= sampleRate;
                  upper *= sampleRate;
                  }
            valLabel[i]   = new QLabel(mw);
            valLabel[i]->setFixedWidth(40);
            valLabel[i]->setMargin(2);
            valLabel[i]->setBackgroundMode(PaletteBase);
            valLabel[i]->setFrameStyle(QFrame::Panel | QFrame::Sunken);
            valLabel[i]->setLineWidth(2);
            valLabel[i]->setFixedHeight(20);
            slider[i] = new Slider(mw, "param", Slider::Horizontal,
               Slider::None, style);
            valLabel[i]->setNum(plugin->param(i));
            slider[i]->setValue(plugin->param(i));
            slider[i]->setFixedHeight(20);

            slider[i]->setRange(lower, upper);
//  printf("RANGE %f - %f\n", lower, upper);

            if (LADSPA_IS_HINT_LOGARITHMIC(range.HintDescriptor)) {
                  ;
                  }

            grid->addWidget(label,         i, 0);
            grid->addWidget(valLabel[i],   i, 1);
            grid->addWidget(slider[i],     i, 2);

            mapper->setMapping(slider[i], i);
            connect(slider[i], SIGNAL(valueChanged(double)), mapper, SLOT(map()));
            }
//      slider[1]->setEnabled(false);
      grid->setColStretch(2, 10);
      }

//---------------------------------------------------------
//   parameterChanged
//---------------------------------------------------------

void PluginGui::parameterChanged(int param)
      {
      double val = slider[param]->value();
      plugin->setParam(param, val);
      valLabel[param]->setNum(val);
      }

//---------------------------------------------------------
//   load
//---------------------------------------------------------

void PluginGui::load()
      {
      QString fn = getOpenFileName("presets/plugins", preset_file_pattern, this);
      if (fn.isEmpty())
            return;
      bool popenFlag;
      FILE* f = fileOpen(this, fn, ".pre", "r", popenFlag, true);
      if (f == 0)
            return;

      Xml xml(f);
      int mode = 0;
      for (;;) {
            Xml::Token token = xml.parse();
            QString tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (mode == 0 && tag == "muse")
                              mode = 1;
                        else if (mode == 1 && tag == "plugin") {
                              plugin->readConfiguration(xml);
                              mode = 0;
                              }
                        else
                              xml.unknown("PluginGui");
                        break;
                  case Xml::Attribut:
                        break;
                  case Xml::TagEnd:
                        if (!mode && tag == "muse")
                              goto ende;
                  default:
                        break;
                  }
            }
ende:
      if (popenFlag)
            pclose(f);
      else
            fclose(f);

      int n = plugin->parameters();
      for (int i = 0; i < n; ++i) {
            valLabel[i]->setNum(plugin->param(i));
            slider[i]->setValue(plugin->param(i));
            }
      }

//---------------------------------------------------------
//   save
//---------------------------------------------------------

void PluginGui::save()
      {
      QString s("presets/plugins/");
      s += plugin->label();

      QString fn = getSaveFileName(s, preset_file_pattern, this);
      if (fn.isEmpty())
            return;
      bool popenFlag;
      FILE* f = fileOpen(this, fn, ".pre", "w", popenFlag, false, true);
      if (f == 0)
            return;
      Xml xml(f);
      xml.header();
      xml.tag(0, "muse version=\"1.0\"");
      plugin->writeConfiguration(1, xml);
      xml.tag(1, "/muse");

      if (popenFlag)
            pclose(f);
      else
            fclose(f);
      }
