//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: ossaudio.cpp,v 1.3 2002/02/27 11:52:58 muse Exp $
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include "config.h"

#ifdef AUDIO
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <qstring.h>
#include <linux/soundcard.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include "globals.h"
#include "audiodev.h"

static int samplesize = 16;   // 8 - 16 - 24 - 32
static int channels   = 2;

struct OssDevice {
      const char* name;
      const char* path;
      };

OssDevice ossDevices[] = {
      { "OssDsp1", "/dev/dsp" },
      { "OssDsp1", "/dev/sound/dsp" },
      { "OssDsp2", "/dev/dsp1" },
      { "OssDsp2", "/dev/sound/dsp1" },
      { "OssDsp3", "/dev/dsp2" },
      { "OssDsp3", "/dev/sound/dsp2" },
      { "OssDsp4", "/dev/dsp3" },
      { "OssDsp4", "/dev/sound/dsp3" },
      };

//---------------------------------------------------------
//   OssAudioDevice
//---------------------------------------------------------

class OssAudioDevice : public AudioDevice {
      int openFlags;
      const char* path;
      int fd;
      int bufferSize;
      dev_t dev;
      ino_t ino;

   public:
      OssAudioDevice(const char* p, const QString& name, dev_t d, ino_t i)
         : AudioDevice(name) {
            path = p;
            fd   = -1;
            _rwFlags = 0x3;
            dev = d;
            ino = i;
            }
      virtual QString open(int);
      virtual void close();
      virtual int read(unsigned char* p, int n);
      virtual int write(const unsigned char* p, int n);
      virtual int selectrfd() const { return fd; }
      virtual int selectwfd() const { return fd; }
      virtual void start() const;
      virtual void stop () const;
      bool compare(dev_t d, ino_t i) { return d == dev && i == ino; }
      };

/**
 *  iiwu_oss_set_queue_size
 *
 *  Set the internal buffersize of the output device.
 *
 *  @param ss Sample size in bits
 *  @param ch Number of channels
 *  @param qs The queue size in frames
 *  @param bs The synthesis buffer size in frames
 */
int oss_set_queue_size(int fd, int ss, int ch, int qs, int bs)
      {
      unsigned int fragmentSize;
      unsigned int fragSizePower;
      unsigned int fragments;
      unsigned int fragmentsPower;
      fragmentSize = (unsigned int) (bs * ch * ss / 8);
      fragSizePower = 0;
      while (0 < fragmentSize) {
            fragmentSize = (fragmentSize >> 1);
            fragSizePower++;
            }
      fragSizePower--;
      fragments = (unsigned int) (qs / bs);
      if (fragments < 2) {
            fragments = 2;
            }
      /* make sure fragments is a power of 2 */
      fragmentsPower = 0;
      while (0 < fragments) {
            fragments = (fragments >> 1);
            fragmentsPower++;
            }
      fragmentsPower--;
      fragments = (1 << fragmentsPower);
      fragments = (fragments << 16) + fragSizePower;
      return ioctl(fd, SNDCTL_DSP_SETFRAGMENT, &fragments);
      }

//---------------------------------------------------------
//   open
//---------------------------------------------------------

QString OssAudioDevice::open(int rwFlags)
      {
      if (fd == -1) {
            int flags = (rwFlags & 2) ? O_RDWR : O_WRONLY;
            fd = ::open(path, flags | O_NDELAY, 0);
            }
      if (fd == -1)
            return QString(strerror(errno));

      // ss Sample size in bits
      // ch Number of channels
      // qs The queue size in frames
      // bs The synthesis buffer size in frames
      if (oss_set_queue_size(fd, 16, 2, segmentSize, segmentSize*2) == -1)
            fprintf(stderr,"set segmentSize failed: %s\n", strerror(errno));

      if (ioctl(fd, SNDCTL_DSP_SAMPLESIZE, &samplesize) == -1)
            fprintf(stderr,"set samplesize failed: %s\n", strerror(errno));
      if (ioctl(fd, SNDCTL_DSP_CHANNELS, &channels) == -1)
            fprintf(stderr,"set channels failed: %s\n", strerror(errno));

      int samplerate = sampleRate;
      if (ioctl(fd, SNDCTL_DSP_SPEED, &samplerate) == -1)
            fprintf(stderr,"set samplerate failed: %s\n", strerror(errno));
      if (samplerate != sampleRate) {
            printf("cannot set sampleRate to %d, set to %d\n", sampleRate, samplerate);
            sampleRate = samplerate;
            }

      if (ioctl(fd, SNDCTL_DSP_GETBLKSIZE, &bufferSize) == -1)
            fprintf(stderr, "get blocksize failed: %s\n", strerror(errno));
      bufferSize /= 8;
      if (bufferSize != segmentSize) {
            printf("cannot set segemntSize to %d, set to %d\n", segmentSize, bufferSize);
            segmentSize = bufferSize;
            }
      if (ioctl(fd, SNDCTL_DSP_RESET, 0) == -1)
            fprintf(stderr, "reset dsp failed: %s\n", strerror(errno));
      return QString("OK");
      }

//---------------------------------------------------------
//   read
//---------------------------------------------------------

int OssAudioDevice::read(unsigned char* buffer, int n)
      {
      if (fd == -1)
            return n;
      return ::read(fd, buffer, n);
      }

//---------------------------------------------------------
//   write
//---------------------------------------------------------

int OssAudioDevice::write(const unsigned char* buffer, int n)
      {
      if (fd == -1)
            return n;
      int rv = ::write(fd, buffer, n * 4);
      if (rv != n) {
            if (rv > 0)
                  return rv;
            if (rv == -1 && errno == EAGAIN)
                  return 0;
            printf("oss device <%s> write failed: %d %s\n",
               path, errno, strerror(errno));
            return 0;
            }
      return rv * 4;
      }

//---------------------------------------------------------
//   close
//---------------------------------------------------------

void OssAudioDevice::close()
      {
      if (fd != -1)
            ::close(fd);
      fd = -1;
      }

//---------------------------------------------------------
//   start
//---------------------------------------------------------

void OssAudioDevice::start() const
      {
      }

//---------------------------------------------------------
//   stop
//---------------------------------------------------------

void OssAudioDevice::stop() const
      {
      }

//---------------------------------------------------------
//   initOssAudio
//    return true if no soundcard found
//---------------------------------------------------------

bool initOssAudio()
      {
      bool notFound = true;
      for (unsigned i = 0; i < sizeof(ossDevices)/sizeof(*ossDevices); ++i) {
            int fd = open(ossDevices[i].path, O_RDONLY | O_NDELAY, 0);
            if (fd != -1) {
                  struct stat buf;
                  if (fstat(fd, &buf) == -1) {
                        fprintf(stderr, "initOssAudio: stat() failed: %s", strerror(errno));
                        close(fd);
                        continue;
                        }
                  close(fd);
                  if (!S_ISCHR(buf.st_mode)) {
                        continue;
                        }
                  for (iAudioDevice k = audioDevices.begin();
                     k != audioDevices.end(); ++k) {
                        OssAudioDevice* dev = dynamic_cast<OssAudioDevice*>(*k);
                        if (dev == 0)
                              continue;
                        if (dev->compare(buf.st_dev, buf.st_ino)) {
                              fd = 0;
                              break;
                              }
                        }
                  if (fd == 0)
                        continue;
                  AudioDevice* d = new OssAudioDevice(
                     ossDevices[i].path, ossDevices[i].name,
                     buf.st_dev, buf.st_ino);
                  audioDevices.add(d);
                  notFound = false;
                  }
            }
      return notFound;
      }
#endif

