//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: alsa9midi.cpp,v 1.5 2002/02/27 08:48:09 muse Exp $
//  (C) Copyright 2000-2001 Werner Schweer (ws@seh.de)
//=========================================================

#include "alsamidi.h"
#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==9)
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <qstring.h>
#include <glob.h>
#include "globals.h"
#include "mididev.h"
#include "midictrl.h"
#include "seq.h"
#include "../midiport.h"
#include <list>
#include "event.h"

static int alsaSeqFdi = -1;
static int alsaSeqFdo = -1;

//---------------------------------------------------------
//   MidiAlsaDevice
//---------------------------------------------------------

MidiAlsaDevice::MidiAlsaDevice(const snd_seq_addr_t& a, const QString& n)
   : MidiDevice(n)
      {
      adr = a;
      init();
      }

int MidiAlsaDevice::selectWfd()
      {
      return alsaSeqFdo;
      }

//---------------------------------------------------------
//   open
//---------------------------------------------------------

QString MidiAlsaDevice::open(int rw)
      {
      rwFlag = rw;
      snd_seq_port_subscribe_t* subs;

#ifdef snd_seq_port_subscribe_alloca
      snd_seq_port_subscribe_alloca(&subs);
#else
      subs = (snd_seq_port_subscribe_t*)alloca(sizeof(*subs));
      memset(subs, 0, sizeof(*subs));
#endif

      // subscribe for writing
      if (rwFlag & 1) {
#ifdef snd_seq_port_subscribe_alloca
            snd_seq_port_subscribe_set_sender(subs, &musePort);
            snd_seq_port_subscribe_set_dest(subs, &adr);
#else
            subs->sender = musePort;
            subs->dest   = adr;
#endif
            int error = snd_seq_subscribe_port(alsaSeq, subs);
            if (error < 0)
                  return QString("Play: ")+QString(snd_strerror(error));
            }

      // subscribe for reading
      if (rwFlag & 2) {
#ifdef snd_seq_port_subscribe_alloca
            snd_seq_port_subscribe_set_dest(subs, &musePort);
	      snd_seq_port_subscribe_set_sender(subs, &adr);
#else
	      subs->dest   = musePort;
	      subs->sender = adr;
#endif
            int error = snd_seq_subscribe_port(alsaSeq, subs);
            if (error < 0)
                  return QString("Rec: ") + QString(snd_strerror(error));
            }
      return QString("OK");
      }

//---------------------------------------------------------
//   close
//---------------------------------------------------------

void MidiAlsaDevice::close()
      {
      snd_seq_port_subscribe_t* subs;

#ifdef snd_seq_port_subscribe_alloca
      snd_seq_port_subscribe_alloca(&subs);
#else
      subs = (snd_seq_port_subscribe_t*)alloca(sizeof(*subs));
      memset(subs, 0, sizeof(*subs));
#endif
      if (rwFlag & 1) {
#ifdef snd_seq_port_subscribe_alloca
            snd_seq_port_subscribe_set_sender(subs, &musePort);
            snd_seq_port_subscribe_set_dest(subs, &adr);
#else
            subs->sender = musePort;
            subs->dest   = adr;
#endif
            snd_seq_unsubscribe_port(alsaSeq, subs);
            }

      if (rwFlag & 2) {
#ifdef snd_seq_port_subscribe_alloca
	      snd_seq_port_subscribe_set_dest(subs, &musePort);
	      snd_seq_port_subscribe_set_sender(subs, &adr);
#else
	      subs->dest = musePort;
	      subs->sender = adr;
#endif
            snd_seq_unsubscribe_port(alsaSeq, subs);
            }
      }
#define snd_seq_ev_set_nrpn(ev,ch,cc,val) \
      ((ev)->type = SND_SEQ_EVENT_NONREGPARAM,\
       snd_seq_ev_set_fixed(ev),\
       (ev)->data.control.channel = (ch),\
       (ev)->data.control.param = (cc),\
       (ev)->data.control.value = (val))

//---------------------------------------------------------
//   putEvent
//---------------------------------------------------------

void MidiAlsaDevice::putEvent(const MidiEvent* e)
      {
      if (midiOutputTrace) {
            printf("midiAlsa: ");
            e->dump();
            }
      int chn = e->channel();
      int a = e->dataA();
      int b = e->dataB();
      int c = e->dataC();

      snd_seq_event_t event;
      prepareEvent(&event);
      switch(e->type()) {
            case MidiEvent::Note:
                  snd_seq_ev_set_noteon(&event, chn, a, b);
                  break;
            case MidiEvent::NoteOff:
                  // snd_seq_ev_set_noteoff(&event, chn, a, b);
                  snd_seq_ev_set_noteoff(&event, chn, a, 0);
                  break;
            case MidiEvent::Program:
                  snd_seq_ev_set_pgmchange(&event, chn, a);
                  break;
            case MidiEvent::RPN:
                  snd_seq_ev_set_controller(&event, chn, CTRL_LDATA, b & 0x7f);
                  snd_seq_event_output_direct(alsaSeq, &event);
                  snd_seq_ev_set_controller(&event, chn, CTRL_HDATA, b >> 7);
                  snd_seq_event_output_direct(alsaSeq, &event);
                  snd_seq_ev_set_controller(&event, chn, CTRL_LRPN, a & 0x7f);
                  snd_seq_event_output_direct(alsaSeq, &event);
                  snd_seq_ev_set_controller(&event, chn, CTRL_HRPN, a >> 7);
                  break;
            case MidiEvent::NRPN:
                  snd_seq_ev_set_controller(&event, chn, CTRL_LDATA, b & 0x7f);
                  snd_seq_event_output_direct(alsaSeq, &event);
                  snd_seq_ev_set_controller(&event, chn, CTRL_HDATA, b >> 7);
                  snd_seq_event_output_direct(alsaSeq, &event);
                  snd_seq_ev_set_controller(&event, chn, CTRL_LNRPN, a & 0x7f);
                  snd_seq_event_output_direct(alsaSeq, &event);
                  snd_seq_ev_set_controller(&event, chn, CTRL_HNRPN, a >> 7);
                  break;
            case MidiEvent::Ctrl7:
                  snd_seq_ev_set_controller(&event, chn, a, b);
                  break;
            case MidiEvent::Ctrl14:
                  snd_seq_ev_set_controller(&event, chn, a & 0x7f, b & 0x7f);
                  snd_seq_event_output_direct(alsaSeq, &event);
                  snd_seq_ev_set_controller(&event, chn, a >> 7, b >> 7);
                  break;
            case MidiEvent::Pitch:
                  snd_seq_ev_set_pitchbend(&event, chn, a);
                  break;
            case MidiEvent::PAfter:
                  // chnEvent2(chn, 0xa0, a, b);
                  break;
            case MidiEvent::CAfter:
                  snd_seq_ev_set_chanpress(&event, chn, a);
                  break;
            case MidiEvent::Sysex:
                  {
                  const unsigned char* p = e->data();
                  int n     = e->dataLen();
                  int len   = n + sizeof(event) + 2;
                  char* buf = new char[len];
                  event.type          = SND_SEQ_EVENT_SYSEX;
                  event.flags         = SND_SEQ_EVENT_LENGTH_VARIABLE;
                  event.data.ext.len  = n + 2;
                  event.data.ext.ptr  = (void*)(buf + sizeof(event));
                  memcpy(buf, &event, sizeof(event));
                  char* pp = buf + sizeof(event);
                  *pp++ = 0xf0;
                  memcpy(pp, p, n);
                  pp += n;
                  *pp = 0xf7;
                  int error = snd_seq_event_output_direct(alsaSeq, &event);
                  len       = snd_seq_event_length(&event);
                  if (error != len)
                        printf("MidiAlsaDevice::putEvent(): midi write error, returns %d expected %d: %s\n",
                           error, len, snd_strerror(error));
                  delete buf;
                  }
                  return;

            default:
                  printf("MidiAlsaDevice::putEvent(): event type %d not implemented\n",
                     e->type());
                  return;
            }
      int error = snd_seq_event_output_direct(alsaSeq, &event);
      int len   = snd_seq_event_length(&event);
      if (error != len)
             printf("MidiAlsaDevice::putEvent(): midi write error, returns %d expected %d: %s\n",
                error, len, snd_strerror(error));
      }

//---------------------------------------------------------
//   putClock
//---------------------------------------------------------

void MidiAlsaDevice::putClock()
      {
      if (midiOutputTrace)
            printf("midiAlsa: clock\n");
      snd_seq_event_t event;
      prepareEvent(&event);
      event.type    = SND_SEQ_EVENT_CLOCK;
      int error = snd_seq_event_output_direct(alsaSeq, &event);
      if (error < 0)
            printf("MidiAlsaDevice::putClock(): alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   putStart
//---------------------------------------------------------

void MidiAlsaDevice::putStart()
      {
      if (midiOutputTrace)
            printf("midiAlsa: start\n");
      snd_seq_event_t event;
      prepareEvent(&event);
      event.type    = SND_SEQ_EVENT_START;
      int error = snd_seq_event_output_direct(alsaSeq, &event);
      if (error < 0)
            printf("putStart: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   putStop
//---------------------------------------------------------

void MidiAlsaDevice::putStop()
      {
      if (midiOutputTrace)
            printf("midiAlsa: stop\n");
      snd_seq_event_t event;
      prepareEvent(&event);
      event.type    = SND_SEQ_EVENT_STOP;
      int error = snd_seq_event_output_direct(alsaSeq, &event);
      if (error < 0)
            printf("putStop: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   putContinue
//---------------------------------------------------------

void MidiAlsaDevice::putContinue()
      {
      if (midiOutputTrace)
            printf("midiAlsa: continue\n");
      snd_seq_event_t event;
      prepareEvent(&event);
      event.type    = SND_SEQ_EVENT_CONTINUE;
      int error = snd_seq_event_output_direct(alsaSeq, &event);
      if (error < 0)
            printf("putContinue: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   putSongpos
//---------------------------------------------------------

void MidiAlsaDevice::putSongpos(int beat)
      {
      if (midiOutputTrace)
            printf("midiAlsa: songpos %d\n", beat);
      snd_seq_event_t event;
      prepareEvent(&event);
      event.data.control.value = beat;
      event.type    = SND_SEQ_EVENT_SONGPOS;
      int error = snd_seq_event_output_direct(alsaSeq, &event);
      if (error < 0)
            printf("putContinue: alsa midi write error: %s\n", snd_strerror(error));
      }

//---------------------------------------------------------
//   initMidiAlsa
//    return true on error
//---------------------------------------------------------

bool initMidiAlsa()
      {
      int error = snd_seq_open(&alsaSeq, "hw", SND_SEQ_OPEN_DUPLEX, SND_SEQ_NONBLOCK);
      if (error < 0) {
            fprintf(stderr, "Could not open ALSA sequencer: %s\n",
               snd_strerror(error));
            alsaFound = false;
            return true;
            }
      const int inCap  = SND_SEQ_PORT_CAP_SUBS_READ;
      const int outCap = SND_SEQ_PORT_CAP_SUBS_WRITE;

      alsaFound = true;
      snd_seq_client_info_t *cinfo;
#ifdef snd_seq_client_info_alloca
      snd_seq_client_info_alloca(&cinfo);
      snd_seq_client_info_set_client(cinfo, -1);
#else
      cinfo = (snd_seq_client_info_t*)alloca(sizeof(*cinfo));
      memset(cinfo, 0, sizeof(*cinfo));
      cinfo->client = -1;
#endif

      while (snd_seq_query_next_client(alsaSeq, cinfo) >= 0) {
            snd_seq_port_info_t *pinfo;
#ifdef snd_seq_port_info_alloca
            snd_seq_port_info_alloca(&pinfo);
            snd_seq_port_info_set_client(pinfo, snd_seq_client_info_get_client(cinfo));
            snd_seq_port_info_set_port(pinfo, -1);
#else
	    pinfo = (snd_seq_port_info_t*)alloca(sizeof(*pinfo));
	    memset(pinfo, 0, sizeof(*pinfo));
	    pinfo->client = cinfo->client;
	    pinfo->port = -1;
#endif

            while (snd_seq_query_next_port(alsaSeq, pinfo) >= 0) {
#ifdef snd_seq_port_info_alloca
		      unsigned int capability = snd_seq_port_info_get_capability(pinfo);
#else
		      unsigned int capability = pinfo->capability;
#endif
                  if ((capability & outCap) == 0)
                        continue;
#ifdef snd_seq_port_info_alloca
                  snd_seq_addr_t adr = *snd_seq_port_info_get_addr(pinfo);
                  MidiAlsaDevice* dev = new MidiAlsaDevice(adr, snd_seq_port_info_get_name(pinfo));
#else
                  snd_seq_addr_t adr;
                  adr.client = pinfo->client;
                  adr.port = pinfo->port;
                  MidiAlsaDevice* dev = new MidiAlsaDevice(adr, pinfo->name);
#endif
                  int flags = 0;
                  if (capability & outCap)
                        flags |= 1;
                  if (capability & inCap)
                        flags |= 2;
                  dev->setrwFlags(flags);
//  printf("ALSA port add: <%s>, flags %d 0x%0x\n",
//              pinfo->name, flags, capability);
                  midiDevices.add(dev);
                  }
            }
      snd_seq_set_client_name(alsaSeq, "MusE Sequencer");
      int ci = snd_seq_poll_descriptors_count(alsaSeq, POLLIN);
      int co = snd_seq_poll_descriptors_count(alsaSeq, POLLOUT);

      if (ci > 1 || co > 1) {
            printf("ALSA midi: cannot handle more than one poll fd\n");
            abort();
            }

      struct pollfd pfdi[ci];
      struct pollfd pfdo[co];
      snd_seq_poll_descriptors(alsaSeq, pfdi, ci, POLLIN);
      snd_seq_poll_descriptors(alsaSeq, pfdo, co, POLLOUT);
      alsaSeqFdo = pfdo[0].fd;
      alsaSeqFdi = pfdi[0].fd;

      int port  = snd_seq_create_simple_port(alsaSeq, "MusE Port 0",
         inCap | outCap | SND_SEQ_PORT_CAP_READ | SND_SEQ_PORT_CAP_WRITE,
         SND_SEQ_PORT_TYPE_APPLICATION);
      if (port < 0) {
            perror("create port");
            exit(1);
            }
      musePort.port   = port;
      musePort.client = snd_seq_client_id(alsaSeq);

      //-----------------------------------------
      //    subscribe to "Announce"
      //    this enables callbacks for any
      //    alsa port changes
      //-----------------------------------------

      snd_seq_addr_t aadr;
      aadr.client = SND_SEQ_CLIENT_SYSTEM;
      aadr.port   = SND_SEQ_PORT_SYSTEM_ANNOUNCE;

      snd_seq_port_subscribe_t* subs;
#ifdef snd_seq_port_subscribe_alloca
      snd_seq_port_subscribe_alloca(&subs);
      snd_seq_port_subscribe_set_dest(subs, &musePort);
      snd_seq_port_subscribe_set_sender(subs, &aadr);
#else
      subs = (snd_seq_port_subscribe_t*)alloca(sizeof(*subs));
      memset(subs, 0, sizeof(*subs));
      subs->dest = musePort;
      subs->sender = aadr;
#endif
      error = snd_seq_subscribe_port(alsaSeq, subs);
      if (error < 0) {
            printf("Alsa: Subscribe System failed: %s", snd_strerror(error));
            return true;
            }
      return false;
      }

struct AlsaPort {
      snd_seq_addr_t adr;
      char* name;
      int flags;
      AlsaPort(snd_seq_addr_t a, const char* s, int f) {
            adr = a;
            name = strdup(s);
            flags = f;
            }
      };

static std::list<AlsaPort> portList;

//---------------------------------------------------------
//   alsaScanMidiPorts
//---------------------------------------------------------

void alsaScanMidiPorts()
      {
      if (!alsaFound)
            return;
      const int inCap  = SND_SEQ_PORT_CAP_SUBS_READ;
      const int outCap = SND_SEQ_PORT_CAP_SUBS_WRITE;

      portList.clear();

      snd_seq_client_info_t* cinfo;
#ifdef snd_seq_client_info_alloca
      snd_seq_client_info_alloca(&cinfo);
      snd_seq_client_info_set_client(cinfo, 0);
#else
      cinfo = (snd_seq_client_info_t*)alloca(sizeof(*cinfo));
      memset(cinfo, 0, sizeof(*cinfo));
      cinfo->client = 0;
#endif

      while (snd_seq_query_next_client(alsaSeq, cinfo) >= 0) {
#ifdef SND_SEQ_GROUP_SYSTEM
            if (strcmp(cinfo->group, SND_SEQ_GROUP_SYSTEM) == 0)
                  continue;
#endif
            snd_seq_port_info_t *pinfo;
#ifdef snd_seq_port_info_alloca
            snd_seq_port_info_alloca(&pinfo);
            snd_seq_port_info_set_client(pinfo, snd_seq_client_info_get_client(cinfo));
            snd_seq_port_info_set_port(pinfo, -1);
#else
            pinfo = (snd_seq_port_info_t*)alloca(sizeof(*pinfo));
            memset(pinfo, 0, sizeof(*pinfo));
            pinfo->client = cinfo->client;
            pinfo->port = -1;
#endif
            while (snd_seq_query_next_port(alsaSeq, pinfo) >= 0) {
#ifdef snd_seq_port_info_alloca
		      unsigned int capability = snd_seq_port_info_get_capability(pinfo);
#else
		      unsigned int capability = pinfo->capability;
#endif
                  if (((capability & outCap) == 0)
                     && ((capability & inCap) == 0))
                        continue;
                  snd_seq_addr_t adr;
                  const char* name;
#ifdef snd_seq_port_info_alloca
                  adr  = *snd_seq_port_info_get_addr(pinfo);
                  name = snd_seq_port_info_get_name(pinfo);
#else
                  adr.client = pinfo->client;
                  adr.port   = pinfo->port;
                  name       = pinfo->name;
#endif
                  if (adr.client == musePort.client && adr.port == musePort.port)
                        continue;
                  int flags = 0;
                  if (capability & outCap)
                        flags |= 1;
                  if (capability & inCap)
                        flags |= 2;
// printf("ALSA port add: <%s>, flags %d\n", name, flags);
                  portList.push_back(AlsaPort(adr, name, flags));
                  }
            }
      //
      //  check for devices to delete
      //
      for (iMidiDevice i = midiDevices.begin(); i != midiDevices.end();) {
            MidiAlsaDevice* d = dynamic_cast<MidiAlsaDevice*>(*i);
            if (d == 0) {
                  ++i;
                  continue;
                  }
            std::list<AlsaPort>::iterator k = portList.begin();
            for (; k != portList.end(); ++k) {
                  if (k->adr.client == d->adr.client
                     && k->adr.port == d->adr.port) {
                        break;
                        }
                  }
            if (k == portList.end()) {
                  if (d->port() != -1)
                        midiPorts[d->port()].setMidiDevice(0);
                  iMidiDevice k = i;
// printf("erase device\n");
                  ++i;
                  midiDevices.erase(k);
                  }
            else {
                  ++i;
                  }
            }
      //
      //  check for devices to add
      //
      for (std::list<AlsaPort>::iterator k = portList.begin(); k != portList.end(); ++k) {
            iMidiDevice i = midiDevices.begin();
// printf("ALSA port: <%s>\n", k->name);
            for (;i != midiDevices.end(); ++i) {
                  MidiAlsaDevice* d = dynamic_cast<MidiAlsaDevice*>(*i);
                  if (d == 0)
                        continue;
                  if ((k->adr.client == d->adr.client) && (k->adr.port == d->adr.port)) {
                        break;
                        }
                  }
            if (i == midiDevices.end()) {
                  // add device
                  MidiAlsaDevice* dev = new MidiAlsaDevice(k->adr,
                     k->name);
                  dev->setrwFlags(k->flags);
                  midiDevices.add(dev);
// printf("add device\n");
                  }
            }
      }

//---------------------------------------------------------
//   alsaSelectRfd
//---------------------------------------------------------

int alsaSelectRfd()
      {
      return alsaSeqFdi;
      }

//---------------------------------------------------------
//   alsaSelectWfd
//---------------------------------------------------------

int alsaSelectWfd()
      {
      return alsaSeqFdo;
      }

#endif
