## This code is written by Davide Albanese, <albanese@fbk.eu>
## (C) 2010 Fondazione Bruno Kessler - Via Santa Croce 77, 38100 Trento, ITALY.

## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

__all__ = ["GradientDescent"]


import spectralreg as sr
import numpy as np

class GradientDescent(object):
    """Gradient Descent Method
    """

    def __init__(self, kernel, t, stepsize):
        """Initialization.

        :Parameters:
          kernel: kernel object
            kernel
          t : int (> 0)
            number of iterations
          stepsize: float
            step size
        """

        self.t = t
        self.stepsize = stepsize
        self.kernel = kernel
        self.__x = None
        self.__c = None
        
        
    def learn(self, x, y):
        """Compute the regression coefficients.

        :Parameters:
          x : numpy 2d array (n x p)
            matrix of regressors
          y : numpy 1d array (n)
            response
        """
        

        if not isinstance(x, np.ndarray):
            raise ValueError("x must be an numpy 2d array")

        if not isinstance(y, np.ndarray):
            raise ValueError("y must be an numpy 1d array")

        if x.ndim > 2:
            raise ValueError("x must be an 2d array")
        
        if x.shape[0] != y.shape[0]:
            raise ValueError("x and y are not aligned")

        c = np.zeros(x.shape[0])
        k = self.kernel.matrix(x)
        self.__c = sr.gradient_descent_steps(c, k, y, self.stepsize, self.t)
        self.__x = x.copy()

        print self.__c

    def pred(self, x):
        """Compute the predicted response.
        
        :Parameters:
          x : numpy 2d array (n x p)
            matrix of regressors
        
        :Returns:
          yp : 1d ndarray
             predicted response
        """


        if not isinstance(x, np.ndarray):
            raise ValueError("x must be an numpy 2d array")

        if x.ndim > 2:
            raise ValueError("x must be an 2d array")
       
        if x.shape[1] != self.__x.shape[1]:
            raise ValueError("x is not aligned")

        y = np.empty(x.shape[0])
        
        for i in range(x.shape[0]):
            k =  self.kernel.vector(x[i], self.__x)
            y[i] = np.sum(self.__c * k)
            
        return y

    
        
        
        
